from __future__ import annotations
from typing import Dict, Any, Optional, Union, TYPE_CHECKING
import os
import logging
from urllib.parse import urlparse, urlunparse
from .utils import lazy
from .utils.typing import Meta
from .render import RenderedString
import jinja2

if TYPE_CHECKING:
    from .site import Site
    from .file import File
    from .contents import Dir

log = logging.getLogger("page")


class PageNotFoundError(Exception):
    pass


class PageValidationError(Exception):
    def __init__(self, page: "Page", msg: str):
        self.page = page
        self.msg = msg


class PageMissesFieldError(PageValidationError):
    def __init__(self, page: "Page", field: str):
        super().__init__(page, f"missing required field meta.{field}")


class Page:
    """
    A source page in the site.

    This can be a static asset, a file to be rendered, a taxonomy, a
    directory listing, or anything else.
    """
    # Page type
    TYPE: str

    def __init__(
            self,
            site: Site,
            src: Optional[File],
            meta: Meta,
            dir: Optional[Dir] = None):
        # Site for this page
        self.site = site
        # contents.Dir which loaded this page, set at load time for non-autogenerated pages
        self.dir: Optional["contents.Dir"] = dir
        # File object for this page on disk, or None if this is an autogenerated page
        self.src = src
        # A dictionary with the page metadata. See the README for documentation
        # about its contents.
        self.meta: Meta = meta

    def validate(self):
        """
        Enforce common meta invariants.

        Performs validation and completion of metadata.

        Raises PageValidationError or one of its subclasses of the page should
        not be added to the site.
        """
        # Run metadata on load functions
        for f in self.site.metadata_on_load_functions:
            f(self)

        # TODO: move more of this to on_load functions

        # template must exist, and defaults to page.html
        self.meta.setdefault("template", "page.html")

        # Render the metadata entres generated that are templates for other
        # entries
        self.site.theme.render_metadata_templates(self)

        # title must exist
        if "title" not in self.meta:
            self.meta["title"] = self.meta["site_name"]

        # Check the existence of other mandatory fields
        if "site_url" not in self.meta:
            raise PageMissesFieldError(self, "site_url")

        # Make sure site_path exists and is relative
        site_path = self.meta.get("site_path")
        if site_path is None:
            raise PageMissesFieldError(self, "site_path")
        if site_path.startswith("/"):
            self.meta["site_path"] = site_path.lstrip("/")

        # Make sure build_path exists and is relative
        build_path = self.meta.get("site_path")
        if build_path is None:
            raise PageMissesFieldError(self, "build_path")
        if build_path.startswith("/"):
            self.meta["build_path"] = build_path.lstrip("/")

    @lazy
    def page_template(self):
        template = self.meta["template"]
        if isinstance(template, jinja2.Template):
            return template
        return self.site.theme.jinja2.get_template(template)

    @lazy
    def redirect_template(self):
        return self.site.theme.jinja2.get_template("redirect.html")

    @property
    def date_as_iso8601(self):
        from dateutil.tz import tzlocal
        ts = self.meta.get("date", None)
        if ts is None:
            return None
        # TODO: Take timezone from config instead of tzlocal()
        tz = tzlocal()
        ts = ts.astimezone(tz)
        offset = tz.utcoffset(ts)
        offset_sec = (offset.days * 24 * 3600 + offset.seconds)
        offset_hrs = offset_sec // 3600
        offset_min = offset_sec % 3600
        if offset:
            tz_str = '{0:+03d}:{1:02d}'.format(offset_hrs, offset_min // 60)
        else:
            tz_str = 'Z'
        return ts.strftime("%Y-%m-%d %H:%M:%S") + tz_str

    def resolve_path(self, target: str) -> "Page":
        """
        Return a Page from the site, given a source or site path relative to
        this page.

        The path is resolved relative to this page, and if not found, relative
        to the parent page, and so on until the top.
        """
        # Absolute URLs are resolved as is
        if target.startswith("/"):
            if target == "/":
                target_relpath = ""
            else:
                target_relpath = os.path.normpath(target.lstrip("/"))

            # Try by source path
            res = self.site.pages_by_src_relpath.get(target_relpath)
            if res is not None:
                return res

            # Try by site path
            res = self.site.pages.get(target_relpath)
            if res is not None:
                return res

            # Try adding STATIC_PATH as a compatibility with old links
            target_relpath = os.path.join(self.site.STATIC_PATH, target_relpath)

            # Try by source path
            res = self.site.pages_by_src_relpath.get(target_relpath)
            if res is not None:
                log.warn("%s: please use %s instead of %s", self, target_relpath, target)
                return res

            raise PageNotFoundError(f"cannot resolve absolute path {target}")

        # Relative urls are tried based on all path components of this page,
        # from the bottom up

        # First using the source paths
        if self.src is not None:
            root = os.path.dirname(self.src.relpath)
            while True:
                target_relpath = os.path.normpath(os.path.join(root, target))
                if target_relpath == ".":
                    target_relpath = ""

                res = self.site.pages_by_src_relpath.get(target_relpath)
                if res is not None:
                    return res

                if not root:
                    break

                root = os.path.dirname(root)

        # The using the site paths
        root = self.meta["site_path"]
        while True:
            target_relpath = os.path.normpath(os.path.join(root, target))
            if target_relpath == ".":
                target_relpath = ""

            res = self.site.pages.get(target_relpath)
            if res is not None:
                return res

            if not root:
                break

            root = os.path.dirname(root)

        raise PageNotFoundError(f"cannot resolve `{target!r}` relative to `{self!r}`")

    def resolve_url(self, url: str) -> str:
        """
        Resolve internal URLs.

        Returns the argument itself if the URL does not need changing, else
        returns the new URL.

        To check for a noop, check like ``if page.resolve_url(url) is url``

        This is used by url resolver postprocessors, like in markdown or
        restructured text pages.

        For resolving urls in templates, see Theme.jinja2_url_for().
        """
        parsed = urlparse(url)
        if parsed.scheme or parsed.netloc:
            return url
        if not parsed.path:
            return url

        try:
            dest = self.url_for(parsed.path)
        except PageNotFoundError as e:
            log.warn("%s", e)
            return url

        dest = urlparse(dest)

        return urlunparse(
            (dest.scheme, dest.netloc, dest.path,
             parsed.params, parsed.query, parsed.fragment)
        )

    def url_for(self, arg: Union[str, "Page"], absolute=False) -> str:
        """
        Generate a URL for a page, specified by path or with the page itself
        """
        page: "Page"

        if isinstance(arg, str):
            page = self.resolve_path(arg)
        else:
            page = arg

        # If the destination has a different site_url, generate an absolute url
        if self.meta["site_url"] != page.meta["site_url"]:
            absolute = True

        if absolute:
            site_url = page.meta["site_url"].rstrip("/")
            return f"{site_url}/{page.meta['site_path']}"
        else:
            return "/" + page.meta["site_path"]

    def check(self, checker):
        pass

    def target_relpaths(self):
        res = [self.meta["build_path"]]
        for relpath in self.meta.get("aliases", ()):
            res.append(os.path.join(relpath, "index.html"))
        return res

    def __str__(self):
        return self.meta["site_path"]

    def __repr__(self):
        return "{}:{}".format(self.TYPE, self.src.relpath)

    @lazy
    def content(self):
        """
        Return only the rendered content of the page, without headers, footers,
        and navigation.
        """
        template = self.page_template
        template_content = template.blocks.get("page_content")
        block_name = "page_content"
        if template_content is None:
            template_content = template.blocks.get("content")
            block_name = "content"
            if template_content is None:
                log.warn("%s: `page_content` and `content` not found in template %s", self, template.name)
                return ""

        try:
            return jinja2.Markup("".join(template_content(template.new_context({"page": self}))))
        except jinja2.TemplateError as e:
            log.error("%s: failed to render %s.%s: %s", template.filename, self.src.relpath, block_name, e)
            log.debug("%s: failed to render %s.%s: %s",
                      template.filename, self.src.relpath, block_name, e, exc_info=True)
            # TODO: return a "render error" page? But that risks silent errors
            return ""

    def to_dict(self):
        from .utils import dump_meta
        res = {
            "src": {
                "relpath": str(self.src.relpath),
                "abspath": str(self.src.abspath),
            },
            "meta": dump_meta(self.meta),
        }
        return res

    def render(self):
        res = {
            self.meta["build_path"]: RenderedString(self.render_template(self.page_template)),
        }

        aliases = self.meta.get("aliases", ())
        if aliases:
            for relpath in aliases:
                html = self.render_template(self.redirect_template)
                res[os.path.join(relpath, "index.html")] = RenderedString(html)

        return res

    def render_template(self, template: jinja2.Template, template_args: Dict[Any, Any] = None) -> str:
        """
        Render a jinja2 template, logging things if something goes wrong
        """
        if template_args is None:
            template_args = {}
        template_args.setdefault("page", self)
        try:
            return template.render(**template_args)
        except jinja2.TemplateError as e:
            log.error("%s: failed to render %s: %s", template.filename, self.src.relpath, e)
            log.debug("%s: failed to render %s: %s", template.filename, self.src.relpath, e, exc_info=True)
            # TODO: return a "render error" page? But that risks silent errors
            return None
