package net.sf.statcvs.pages;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import net.sf.statcvs.Messages;
import net.sf.statcvs.charts.ChartImage;
import net.sf.statcvs.charts.DirectoryPieChartMaker.CodeDistributionChartMaker;
import net.sf.statcvs.charts.TimeBarChartMaker.HourBarChartMaker;
import net.sf.statcvs.charts.TimeBarChartMaker.WeekdayBarChartMaker;
import net.sf.statcvs.model.Author;
import net.sf.statcvs.model.Commit;
import net.sf.statcvs.model.Repository;
import net.sf.statcvs.model.Revision;
import net.sf.statcvs.output.ReportConfig;
import net.sf.statcvs.reports.DirectoriesForAuthorTableReport;

/**
 * @author anja
 * @author Richard Cyganiak (richard@cyganiak.de)
 * @version $Id: DeveloperPageMaker.java,v 1.11 2008/04/02 11:22:14 benoitx Exp $
 */
public class DeveloperPageMaker {
    private final static int RECENT_COMMITS_LENGTH = 20;
    private final static NumberFormat PERCENT_FORMAT = NumberFormat.getPercentInstance();

    static {
        PERCENT_FORMAT.setMinimumFractionDigits(1);
        PERCENT_FORMAT.setMaximumFractionDigits(1);
    }

    public static String getURL(final Author developer) {
        return getFilename(developer) + ".html";
    }

    private static String getFilename(final Author developer) {
        return "user_" + HTML.escapeAuthorName(developer.getName());
    }

    private final ReportConfig config;
    private final Author developer;
    private final Repository repository;

    public DeveloperPageMaker(final ReportConfig config, final Author developer) {
        this.config = config;
        this.developer = developer;
        this.repository = config.getRepository();
    }

    public Page toFile() {
        final ChartImage hourChart = new HourBarChartMaker(this.config, this.developer.getRevisions(), Messages.getString("ACTIVITY_TIME_FOR_AUTHOR_TITLE")
                + " " + this.developer.getRealName(), "activity_time_" + HTML.escapeAuthorName(this.developer.getName()) + ".png").toFile();
        final ChartImage weekdayChart = new WeekdayBarChartMaker(this.config, this.developer.getRevisions(), Messages
                .getString("ACTIVITY_DAY_FOR_AUTHOR_TITLE")
                + " " + this.developer.getRealName(), "activity_day_" + HTML.escapeAuthorName(this.developer.getName()) + ".png").toFile();
        final ChartImage codeDistributionChart = new CodeDistributionChartMaker(this.config, this.developer).toFile();

        String title;
        if (this.config.isDeveloper(this.developer)) {
            title = this.config.getProjectName() + " Developers: " + this.developer.getRealName();
        } else {
            title = "Non-developer Login: " + this.developer.getRealName();
        }
        final Page page = this.config.createPage(getFilename(this.developer), this.developer.getRealName(), title);
        page.addAttribute("Login name", this.developer.getName());
        if (this.developer.getRealName() != null && !this.developer.getRealName().equals(this.developer.getName())) {
            page.addAttribute("Real name", this.developer.getRealName());
        }
        if (this.developer.getEmail() != null && this.developer.getEmail().length() > 0) {
            page.addRawAttribute("Email", "<a href=\"mailto:" + this.developer.getEmail() + "\">" + this.developer.getEmail() + "</a>");
        }
        if (this.developer.getHomePageUrl() != null && this.developer.getHomePageUrl().length() > 0) {
            page.addRawAttribute("Home Page", "<a href=\"" + this.developer.getHomePageUrl() + "\">" + this.developer.getHomePageUrl() + "</a>");
        }
        if (this.developer.getImageUrl() != null && this.developer.getImageUrl().length() > 0) {
            page.addRawAttribute("Image", "<img src=\"" + this.developer.getImageUrl() + "\" alt=\"" + this.developer.getRealName() + "\"/>");
        }
        page.addAttribute("Total Commits", getNumberAndPercentage(this.developer.getRevisions().size(), this.repository.getRevisions().size()));
        page.addAttribute("Lines of Code", getNumberAndPercentage(countContributedLines(this.developer.getRevisions()), countContributedLines(this.repository
                .getRevisions())));
        page.addAttribute("Most Recent Commit", ((Revision) this.developer.getRevisions().last()).getDate());
        page.addSection(Messages.getString("ACTIVITY_TITLE"));
        page.add(hourChart);
        page.add(weekdayChart);
        page.addSection("Activity in Directories");
        page.add(new DirectoriesForAuthorTableReport(this.config, this.developer));
        if (codeDistributionChart != null) {
            page.add(codeDistributionChart);
        }
        page.addSection(Messages.getString("MOST_RECENT_COMMITS"));
        page.addRawContent(new CommitListFormatter(this.config, getRecentCommits(), Collections.EMPTY_LIST, RECENT_COMMITS_LENGTH, false).render());
        return page;
    }

    private List getRecentCommits() {
        final List results = new ArrayList();
        final Iterator it = this.repository.getCommits().iterator();
        while (it.hasNext()) {
            final Commit commit = (Commit) it.next();
            if (this.developer.equals(commit.getAuthor())) {
                results.add(commit);
            }
        }
        return results;
    }

    private int countContributedLines(final Collection revisions) {
        int result = 0;
        final Iterator it = revisions.iterator();
        while (it.hasNext()) {
            final Revision element = (Revision) it.next();
            result += element.getNewLines();
        }
        return result;
    }

    /**
     * returns the percentage of a given total count and the count.
     * This will work, because division by zero is not a problem with doubles:
     * you get NaN (and the formatter will format that too).
     * @author Jan Dockx
     * @param value
     * @param total
     * @return String percentage string
     */
    private String getNumberAndPercentage(final int value, final int total) {
        final double factor = (double) value / (double) total;
        return value + " (" + PERCENT_FORMAT.format(factor) + ")";
    }
}
