/* $Id: stsettings.c,v 400.1 2002/07/25 08:43:16 sgifford Exp $ */

#include "stsettings.h"
#include "startalk.h"
#include "stdebug.h"
#include "settings.h"

#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

struct setmap {
  unsigned char whichsetting;
  unsigned char type;
  unsigned char whichbuf;
  unsigned char whichbyte;
  unsigned char shift, mask;
};

struct setmap map[] = {
  { STARTALK_SET_RING            , STARTALK_SET_TYPE_BOOL,  0,  2, 6,  1 },
  { STARTALK_SET_VIBRATE         , STARTALK_SET_TYPE_BOOL,  0,  1, 2,  1 },
  { STARTALK_SET_MSG_RMD_TONE    , STARTALK_SET_TYPE_BOOL,  0,  1, 5,  1 },
  { STARTALK_SET_DTMF_IS_LONG    , STARTALK_SET_TYPE_BOOL,  0,  0, 1,  1 },
  { STARTALK_SET_DTMF_ON         , STARTALK_SET_TYPE_BOOL,  0,  2, 7,  1 },
  { STARTALK_SET_SVC_TONE        , STARTALK_SET_TYPE_BOOL,  0,  1, 3,  1 },
  { STARTALK_SET_SILENT_MODE     , STARTALK_SET_TYPE_BOOL,  0,  0, 0,  1 },
  { STARTALK_SET_ROAM_RING       , STARTALK_SET_TYPE_BOOL,  0,  0, 7,  1 },
  { STARTALK_SET_CALL_RMD_TONE   , STARTALK_SET_TYPE_BOOL,  0,  0, 4,  1 }, 
  { STARTALK_SET_KEYPAD_TONE     , STARTALK_SET_TYPE_BOOL,  0,  3, 7,  1 },
  { STARTALK_SET_INSTREDIAL      , STARTALK_SET_TYPE_BOOL,  0,  1, 4,  1 },
  { STARTALK_SET_MULTKEYANS      , STARTALK_SET_TYPE_BOOL,  0,  2, 2,  1 },
  { STARTALK_SET_CALLGUARD       , STARTALK_SET_TYPE_BOOL,  0,  0, 6,  1 },
  { STARTALK_SET_ANSWEROPEN      , STARTALK_SET_TYPE_BOOL,  0,  0, 2,  1 },
  { STARTALK_SET_AUTOANSWER      , STARTALK_SET_TYPE_BOOL,  0,  3, 1,  1 },

  { STARTALK_SET_QUICKMENU       , STARTALK_SET_TYPE_BOOL,  0,  2, 0,  1 },
  { STARTALK_SET_AUTOHYPHEN      , STARTALK_SET_TYPE_BOOL,  0,  3, 4,  1 },
  { STARTALK_SET_24HRCLOCK       , STARTALK_SET_TYPE_BOOL,  0,  1, 6,  1 },
  { STARTALK_SET_SVCLIGHT        , STARTALK_SET_TYPE_BOOL,  0,  0, 3,  1 },
  { STARTALK_SET_CONFIRMBROWSE   , STARTALK_SET_TYPE_BOOL,  0,  0, 5,  1 },
  { STARTALK_SET_ANALOGONLY      , STARTALK_SET_TYPE_BOOL,  0,  3, 0,  1 },
  { STARTALK_SET_INDIVTIMER      , STARTALK_SET_TYPE_BOOL,  0,  3, 2,  1 },
  { STARTALK_SET_1MINTIMER       , STARTALK_SET_TYPE_BOOL,  0,  2, 4,  1 },
  { STARTALK_SET_USE_REPTIMER    , STARTALK_SET_TYPE_BOOL,  0,  7, 0,  1 },
  { STARTALK_SET_USE_SNGLTIMER   , STARTALK_SET_TYPE_BOOL,  0, 10, 0,  1 },
  { STARTALK_SET_AUTOLOCK        , STARTALK_SET_TYPE_BOOL,  0,  3, 5,  1 },
  { STARTALK_SET_PRIOCALL        , STARTALK_SET_TYPE_BOOL,  0,  3, 6,  1 },

  { STARTALK_SET_RINGER_STYLE    , STARTALK_SET_TYPE_UCHAR, 0,  4, 0, 15 },
  { STARTALK_SET_SVCLEVEL        , STARTALK_SET_TYPE_UCHAR, 0,  6, 0, 15 },
  { STARTALK_SET_BKLIGHT_MODE    , STARTALK_SET_TYPE_UCHAR, 2,  1, 0, 15 },

  { STARTALK_SET_REPTIMER_DUR    , STARTALK_SET_TYPE_UINT,  0,  8, 0,  0 },
  { STARTALK_SET_SINGTIMER_DUR   , STARTALK_SET_TYPE_UINT,  0, 11, 0,  0 },
  { STARTALK_SET_LOCKCODE        , STARTALK_SET_TYPE_UINT,  1, 42, 0,  0 },

  { STARTALK_SET_SECURECODE      , STARTALK_SET_TYPE_SUINT, 1, 39, 0,  0 },
  
  { STARTALK_SET_LANG            , STARTALK_SET_TYPE_LANG,  0,  5, 0,  7 },
};

/* Private prototypes */
static int startalk_fill_settings_buf(int serialfd, struct startalk_settings *se);
static struct startalk_settings *startalk_parse_settings(struct startalk_settings *se);
static int validate_settings(struct startalk_settings *n);


void
startalk_dump_settings_offsets(void)
{
  debugf(3,"Settings Offsets:\n");
}

int
startalk_initialize_stsettings(int serialfd)
{
  return 0;
}

struct startalk_settings *
startalk_get_settings(int serialfd)
{
  struct startalk_settings *se;

  se = startalk_new_settings();
  if (!se)
    return (void *) reterr(0,"Error allocating new settings entry");
  if (startalk_fill_settings_buf(serialfd,se) != 0)
    return (void *) reterr(0,"Error getting settings from phone");

  return startalk_parse_settings(se);
}

static struct startalk_settings *
startalk_parse_settings(struct startalk_settings *se)
{
  int mapsize, i;
  int val;
  long lval;
  unsigned char databyte, databyte2, databyte3;

  /* OK, big, serious fun. */
  mapsize = sizeof(map) / sizeof(map[0]);
  for(i=0;i<mapsize;i++)
  {
    debugf(3,"Looking at map entry %d: setting=%d, type=%d, buf=%d, byte=%d, shift=%d, mask=%d\n",i, map[i].whichsetting, map[i].type, map[i].whichbuf, map[i].whichbyte, map[i].shift, map[i].mask);
    if ( (map[i].type == STARTALK_SET_TYPE_BOOL) ||
         (map[i].type == STARTALK_SET_TYPE_UCHAR) ||
         (map[i].type == STARTALK_SET_TYPE_LANG) )
    {
      databyte = se->databuf[map[i].whichbuf][map[i].whichbyte];
      debugf(4,"Byte %d of buffer %d is: %hd (%02hx)\n",map[i].whichbyte, map[i].whichbuf, databyte, databyte);
      debugf(4,"Shifted right %d bits is %hd (%02hx)\n",map[i].shift, databyte >> map[i].shift, databyte >> map[i].shift);
      val = (databyte >> map[i].shift) & map[i].mask;
      debugf(2,"Final value for map entry %d: %d\n",i,val);
      if (map[i].type == STARTALK_SET_TYPE_BOOL)
        startalk_set_setting_bool(se, map[i].whichsetting, val);
      else if (map[i].type == STARTALK_SET_TYPE_UCHAR)
        startalk_set_setting_uchar(se, map[i].whichsetting, val);
      else if (map[i].type == STARTALK_SET_TYPE_LANG)
        startalk_set_setting_lang(se, map[i].whichsetting, val);
    }
    else if (map[i].type == STARTALK_SET_TYPE_UINT)
    {
      databyte  = se->databuf[map[i].whichbuf][map[i].whichbyte];
      databyte2 = se->databuf[map[i].whichbuf][map[i].whichbyte+1];
      debugf(4,"Byte %d-%d of buffer %d are is: %hd %hd (%02hx %02hx)\n",map[i].whichbyte, map[i].whichbyte+1, map[i].whichbuf, databyte, databyte2, databyte, databyte2);
      lval = databyte * 256 + databyte2;
      debugf(2,"Final value for map entry %d: %ld\n",i,lval);
      startalk_set_setting_uint(se, map[i].whichsetting, lval);
    }
    else if (map[i].type == STARTALK_SET_TYPE_SUINT)
    {
      databyte  = se->databuf[map[i].whichbuf][map[i].whichbyte];
      databyte2 = se->databuf[map[i].whichbuf][map[i].whichbyte+1];
      databyte3 = se->databuf[map[i].whichbuf][map[i].whichbyte+2];
      debugf(4,"Byte %d-%d of buffer %d are : %hd %hd %hd (%02hx %02hx %02hx)\n",map[i].whichbyte,map[i].whichbyte+2,map[i].whichbuf, databyte, databyte2, databyte3, databyte, databyte2, databyte3);
      lval = databyte * 65536L + databyte2 * 256 + databyte3;
      debugf(2,"Final value for map entry %d: %ld\n",i,lval);
      startalk_set_setting_suint(se, map[i].whichsetting, lval);
    }
    else
    {
      die("Error in setting map, item %d: unknown type %d\n",i,map[i].type);
    }
  }
  /* Quick hack */
  se->ss_string[0] = se->ss_greeting_buf;
  
  return se;
}

static int
startalk_fill_settings_buf(int serialfd, struct startalk_settings *se)
{
  extern int testing;
  unsigned char cmdbuf1[] = { 0xcf, 0x07, 0x01 };
  unsigned char cmdbuf2[] = { 0x60, 0x01 };
  unsigned char cmdbuf3[] = { 0xcf, 0x07, 0x03 };
  unsigned char cmdgreet[] = { 0xc7, 0x0d };

  unsigned char cmdbuf4[] = { 0x96 };
  unsigned char cmdbuf5[] = { 0xcf, 0x01, 0x00 };
  unsigned char cmdbuf6[] = { 0xcf, 0x07, 0x00 };
  
  unsigned char buf[8192];
  int msgsize;

  debugf(2,"Getting settings from phone\n");

  if (se->gotbuf)
    return 0;
  
  debugf(2,"Start Settings block 1\n");

  msgsize = 8192;
  if (startalk_run_cmd(serialfd, (char *)cmdbuf1, sizeof(cmdbuf1), (char *)buf, &msgsize) != 0)
    return reterr(-1,"retrieving settings buffer 1");
  if ((msgsize-8) != 17)
    return reterr(-1,"Unrecognized format for settings entry 1 (%d bytes)",msgsize);
  memcpy(se->buf1,buf+6,msgsize-8);
  debugf(2,"End Settings block 1\n");

  debugf(2,"Start Settings block 2\n");
  msgsize = 8192;
  if (startalk_run_cmd(serialfd, (char *)cmdbuf2, sizeof(cmdbuf2),(char *)buf,&msgsize) != 0)
    return reterr(-1,"retrieving settings buffer 2");
  if ((msgsize-8) != 66)
    return reterr(-1,"Unrecognized format for settings entry 2 (%d bytes)",msgsize);
  memcpy(se->buf2,buf+6,msgsize-8);
  debugf(2,"End Settings block 2\n");

  /* If they've already set the greeting, don't overwrite it. */
  /* This is all very ugly, but it's not worth writing better code
   * for just one string setting...
   */
  if (!se->ss_string[0])
  {
    debugf(2,"Start Settings block greeting\n");
    if (startalk_writemsg(serialfd, (char *)cmdgreet, sizeof(cmdgreet)) != 0)
      return reterr(-1,"requesting greeting");
    if (startalk_readmsg(serialfd, (char *)buf, 8192, &msgsize) != 0)
      return reterr(-1,"getting greeting from phone");
    if ((msgsize-8) != 13)
      return reterr(-1,"Unrecognized format for greeting entry (%d bytes)",msgsize);
    memcpy(se->ss_greeting_buf,buf+7,12);
    se->ss_greeting_buf[12] = '\0';
    debugf(2,"End Settings block greeting\n");
  }

  debugf(2,"Start Settings block 3\n");
  msgsize = 8192;
  if (startalk_run_cmd(serialfd, (char *)cmdbuf3, sizeof(cmdbuf3),(char *)buf,&msgsize) != 0)
    return reterr(-1,"retrieving settings buffer 3");
    if ((msgsize-8) != 1)
      return reterr(-1,"Unrecognized format for settings entry 3 (%d bytes)",msgsize);
    memcpy(se->buf3,buf+6,msgsize-8);
  debugf(2,"End Settings block 3\n");

  if (testing)
  {
    debugf(2,"Start Settings block 4\n");
    msgsize = 8192;
    if (startalk_run_cmd(serialfd, (char *)cmdbuf4, sizeof(cmdbuf4),(char *)buf,&msgsize) != 0)
      return reterr(-1,"retrieving settings buffer 4");
/*      if ((msgsize-8) != 66) */
/*        return reterr(-1,"Unrecognized format for settings entry 4 (%d bytes)",msgsize); */
/*      memcpy(se->buf2,buf+6,msgsize-8); */
    debugf(2,"End Settings block 4\n");

    debugf(2,"Start Settings block 5\n");
    msgsize = 8192;
    if (startalk_run_cmd(serialfd, (char *)cmdbuf5, sizeof(cmdbuf5),(char *)buf,&msgsize) != 0)
      return reterr(-1,"retrieving settings buffer 5");
/*      if ((msgsize-8) != 66) */
/*        return reterr(-1,"Unrecognized format for settings entry 5 (%d bytes)",msgsize); */
/*      memcpy(se->buf2,buf+6,msgsize-8); */
    debugf(2,"End Settings block 5\n");


    debugf(2,"Start Settings block 6\n");
    msgsize = 8192;
    if (startalk_run_cmd(serialfd, (char *)cmdbuf6, sizeof(cmdbuf6),(char *)buf,&msgsize) != 0)
      return reterr(-1,"retrieving settings buffer 4");
/*      if ((msgsize-8) != 66) */
/*        return reterr(-1,"Unrecognized format for settings entry 4 (%d bytes)",msgsize); */
/*      memcpy(se->buf2,buf+6,msgsize-8); */
    debugf(2,"End Settings block 6\n");
  }
  debugf(2,"Done getting settings from phone\n");
  return 0;
}

int
startalk_put_settings(int serialfd, struct startalk_settings *se)
{
  unsigned char sendbuf[256];
  unsigned char inbuf[256];
  int i;
  int mapsize;
  int setbyte, setmask;
  int bufchanged[sizeof(se->databuf)];
  long lsetbyte;
  
  if (startalk_fill_settings_buf(serialfd, se) != 0)
    return reterr(0,"Error getting current settings from phone");
  
  mapsize = sizeof(map)/sizeof(map[0]);
  for(i=0;i<mapsize;i++)
  {
    debugf(3,"Looking at map entry %d: setting=%d, type=%d, buf=%d, byte=%d, shift=%d, mask=%d\n",i, map[i].whichsetting, map[i].type, map[i].whichbuf, map[i].whichbyte, map[i].shift, map[i].mask);
    if (map[i].type == STARTALK_SET_TYPE_BOOL)
    {
      if ((setbyte=startalk_get_setting_bool(se, map[i].whichsetting)) == -1)
        continue;
      if ( (setbyte != 1) && (setbyte != 0) )
      {
        warn("Boolean value for map entry %d wasn't zero or 1!\n",map[i].whichsetting);
        continue;
      }
      debugf(4,"Value from settings struct is %d (%x)\n",setbyte,setbyte);
      setbyte &= map[i].mask;    /* Unnecessary. */
      setbyte <<= (map[i].shift);
      setmask = map[i].mask << map[i].shift;
      debugf(4,"setbyte is %d\n",setbyte);
      debugf(4,"Old byte was %d (%x)\n",se->databuf[map[i].whichbuf][map[i].whichbyte],se->databuf[map[i].whichbuf][map[i].whichbyte]);
      se->databuf[map[i].whichbuf][map[i].whichbyte] = (se->databuf[map[i].whichbuf][map[i].whichbyte] & (~setmask)) | setbyte;
      debugf(4,"New byte is %d (%x)\n",se->databuf[map[i].whichbuf][map[i].whichbyte],se->databuf[map[i].whichbuf][map[i].whichbyte]);

      bufchanged[map[i].whichbuf] = 1;
    }
    else if (map[i].type == STARTALK_SET_TYPE_UCHAR)
    {
      if ((setbyte=startalk_get_setting_uchar(se, map[i].whichsetting)) == -1)
        continue;
      if ( (setbyte < 0) || (setbyte > 255) )
      {
        warn("UCHAR value for map entry %d wasn't between 0 and 255!\n",map[i].whichsetting);
        continue;
      }
      debugf(4,"Value from settings struct is %d (%x)\n",setbyte,setbyte);
      setbyte &= map[i].mask;
      setbyte <<= (map[i].shift);
      setmask = map[i].mask << map[i].shift;
      debugf(4,"setbyte is %d\n",setbyte);
      debugf(4,"Old byte was %d (%x)\n",se->databuf[map[i].whichbuf][map[i].whichbyte],se->databuf[map[i].whichbuf][map[i].whichbyte]);
      se->databuf[map[i].whichbuf][map[i].whichbyte] = (se->databuf[map[i].whichbuf][map[i].whichbyte] & (~setmask)) | setbyte;
      debugf(4,"New byte is %d (%x)\n",se->databuf[map[i].whichbuf][map[i].whichbyte],se->databuf[map[i].whichbuf][map[i].whichbyte]);
      bufchanged[map[i].whichbuf] = 1;
    }
    else if (map[i].type == STARTALK_SET_TYPE_UINT)
    {
      if ( (lsetbyte = startalk_get_setting_uint(se, map[i].whichsetting)) == -1)
        continue;
      if ( (lsetbyte < 0) || (lsetbyte > 65535) )
      {
        warn("UINT value for map entry %d wasn't between 0 and 65535!\n",map[i].whichsetting);
        continue;
      }
      debugf(4,"Value from settings struct is %ld (0x%lx)\n",lsetbyte,lsetbyte);
      se->databuf[map[i].whichbuf][map[i].whichbyte] = lsetbyte / 256;
      se->databuf[map[i].whichbuf][map[i].whichbyte+1] = lsetbyte % 256;

      bufchanged[map[i].whichbuf] = 1;
    }
    else if (map[i].type == STARTALK_SET_TYPE_SUINT)
    {
      if ( (lsetbyte = startalk_get_setting_suint(se, map[i].whichsetting)) == -1)
        continue;
      if ( (lsetbyte < 0) || (lsetbyte > 16777216) )
      {
        warn("SUINT value for map entry %d wasn't between 0 and 16777216!\n",map[i].whichsetting);
        continue;
      }
      debugf(4,"Value from settings struct is %ld (0x%lx)\n",lsetbyte,lsetbyte);
      se->databuf[map[i].whichbuf][map[i].whichbyte] = lsetbyte / 65536;
      se->databuf[map[i].whichbuf][map[i].whichbyte+1] = (lsetbyte % 65536) / 256;
      se->databuf[map[i].whichbuf][map[i].whichbyte+2] = lsetbyte % 256;

      bufchanged[map[i].whichbuf] = 1;
    }
    else if (map[i].type == STARTALK_SET_TYPE_LANG)
    {
      if ((setbyte=startalk_get_setting_lang(se, map[i].whichsetting)) == -1)
        continue;
      if ( (setbyte < 0) || (setbyte > 6) )
      {
        warn("LANG value for map entry %d wasn't between 0 and 6!\n",map[i].whichsetting);
        continue;
      }
      debugf(4,"Value from settings struct is %d (%x)\n",setbyte,setbyte);
      setbyte &= map[i].mask;
      setbyte <<= (map[i].shift);
      setmask = map[i].mask << map[i].shift;
      debugf(4,"setbyte is %d\n",setbyte);
      debugf(4,"Old byte was %d (%x)\n",se->databuf[map[i].whichbuf][map[i].whichbyte],se->databuf[map[i].whichbuf][map[i].whichbyte]);
      se->databuf[map[i].whichbuf][map[i].whichbyte] = (se->databuf[map[i].whichbuf][map[i].whichbyte] & (~setmask)) | setbyte;
      debugf(4,"New byte is %d (%x)\n",se->databuf[map[i].whichbuf][map[i].whichbyte],se->databuf[map[i].whichbuf][map[i].whichbyte]);

      bufchanged[map[i].whichbuf] = 1;
    }
  }

  /* Make sure all of these settings are valid. */
  if (!(validate_settings(se)))
    return errorf("Invalid settings encountered, not setting.");
        
  if (bufchanged[0])
  {
    debugf(2,"Changing settings in buf1\n");
    /* OK, let's build this request packet. */
    /* Header */
    /* Bytes 0-2 */
    sendbuf[0] = 0xcf;
    sendbuf[1] = 0x07;
    sendbuf[2] = 0x02;
    memcpy(sendbuf + 3, se->buf1, 17); /* FIX constant value. */
    
    debugf(3,"OK, sending completed packet to phone\n");
    if (startalk_writemsg(serialfd, (char *)sendbuf, 17 + 3) != 0)
      return errorf("writing settings");
    debugf(3,"Sent completed packet succesfully!\n");
    
    /* Make sure it succeeded. */
    debugf(3,"Reading response...\n");
    if (startalk_readmsg(serialfd, (char *)inbuf, 255, &i) != 0)
      return errorf("getting response from writing settings entry");
    
    if (i<8)
    {
      return errorf("phone returned error while writing settings: response too short (%d bytes)",i);
    }
    else if (inbuf[6] != 0xac)
    {
      return errorf("phone returned error while writing settings: response code 0x%02hx",inbuf[6]);
    }
    
    debugf(3,"Response read succesfully!\n");
  }
  else
  {
    debugf(2,"No changed for buf1, so no packet sent.\n");
  }

  if (bufchanged[1])
  {
    debugf(2, "Changing settings in buf2\n");
    /* OK, let's build this request packet. */
    /* Header */
    /* Bytes 0-2 */
/*      sendbuf[0] = 0x60; */
/*      sendbuf[1] = 0x02; */
    sendbuf[0] = 0x5e;
    sendbuf[1] = 0x03;
    memcpy(sendbuf+2, se->buf2, 64);  /* FIX constant value */

    debugf(3,"OK, sending completed packet to phone\n");
    if (startalk_writemsg(serialfd, (char *)sendbuf, 64 + 2) != 0)
      return errorf("writing settings buffer 2");

    /* Make sure it succeeded */
    debugf(3,"Reading response...\n");
    if (startalk_readmsg(serialfd, (char *)inbuf, 255, &i) != 0)
      return errorf("Gettinr response from writing settings buffer 2");

    if (i < 8)
    {
      return errorf("phone returned error while writing settings buffer 2: response too short (%d bytes)",i);
    }
    else if (inbuf[6] != 0xac)
    {
      return errorf("phone returned error while writing settings buffer 2: response code 0x%02hx",inbuf[6]);
    }
    debugf(3,"Response read succesfully!\n");
  }
  else
  {
    debugf(2,"No change for buf2, so no packet sent.\n");
  }
  
  if (se->ss_string[0])
  {
    debugf(2,"Changing greeting\n");
    /* Greeting changed */
    sendbuf[0] = 0xc6;
    sendbuf[1] = 0x0d;
    sendbuf[2] = 0x00;

    memcpy(sendbuf + 3, se->ss_greeting_buf, 12); /* FIX constant value */
    debugf(3,"OK, sending completed packet to phone\n");
    if (startalk_writemsg(serialfd, (char *)sendbuf, 12 + 3) != 0)
      return errorf("writing greeting");
    debugf(3,"Sent completed packet succesfully\n");

    /* Make sure it succeeded. */
    debugf(3,"Reading response...\n");
    if (startalk_readmsg(serialfd, (char *)inbuf, 255, &i) != 0)
      return errorf("getting response from writing greeting");
    if (i < 8)
    {
      return errorf("phone returned error while writing greeting: response too short (%d bytes)",i);
    }
    else if (inbuf[6] != 0x00)
    {
      return errorf("phone returned error while writing greeting: response code 0x%02x",inbuf[6]);
    }

    debugf(3,"Response read succesfully!\n");
  }
  else
  {
    debugf(2,"No change for greeting, so no packet sent.\n");
  }
      
      
  return 0;
}

int
startalk_get_greeting(int serialfd, char *greeting, int bufsiz)
{
  char buf[8192];

  if ((bufsiz != 0) && (bufsiz < 13))
    return errorf("buffer size too small (need 13 bytes)");
  
  if (startalk_writemsg(serialfd,"\xC7\x0D",2) != 0)
    return errorf("requesting greeting");
  if (startalk_readmsg(serialfd,buf,8192,NULL) != 0)
    return errorf("getting response from greeting");
  if (greeting && bufsiz)
  {
    strncpy(greeting,&buf[7],12);
    greeting[12]=0;
  }
  return 0;
}

static int
validate_settings(struct startalk_settings *n)
{
  int warnings = 0;
  struct startalk_settings *testse;
  char *failreason;
  unsigned long val;
  char *sval;
  int i;
  
  if (!(testse = malloc(sizeof(struct startalk_settings))))
    return errorf("malloc error getting memory for testse:");
  memcpy(testse, n, sizeof(struct startalk_settings));
  if (!startalk_parse_settings(testse))
    return errorf("error parsing settings");
  
  /* Now, copy changed settings into the test structure. */
  for(i=0;i<STARTALK_SET_BOOL_LAST;i++)
  {
    if ( (val=startalk_get_setting_bool(n, i)) != -1)
      startalk_set_setting_bool(testse,i,val);
  }
  for(i=0;i<STARTALK_SET_UCHAR_LAST;i++)
  {
    if ( (val=startalk_get_setting_uchar(n, i)) != -1)
      startalk_set_setting_uchar(testse,i,val);
  }
  for(i=0;i<STARTALK_SET_STRING_LAST;i++)
  {
    if ( (sval=startalk_get_setting_string(n, i)) != NULL)
      startalk_set_setting_string(testse,i,sval);
  }
  for(i=0;i<STARTALK_SET_UINT_LAST;i++)
  {
    if ( (val=startalk_get_setting_uint(n, i)) != -1)
      startalk_set_setting_uint(testse,i,val);
  }
  for(i=0;i<STARTALK_SET_LANG_LAST;i++)
  {
    if ( (val=startalk_get_setting_lang(n, i)) != -1)
      startalk_set_setting_lang(testse,i,val);
  }
  for(i=0;i<STARTALK_SET_SUINT_LAST;i++)
  {
    if ( (val=startalk_get_setting_suint(n, i)) != -1)
      startalk_set_setting_suint(testse,i,val);
  }

  /* And test each value which has changed. */
  for(i=0;i<STARTALK_SET_BOOL_LAST;i++)
  {
    if ( (val=startalk_get_setting_bool(n, i)) != -1)
    {
      if ( (failreason=startalk_validate_setting_bool(testse,i,val)) != NULL )
      {
        warn("Invalid setting: %s\n",failreason);
        warnings++;
      }
    }
  }
  for(i=0;i<STARTALK_SET_UCHAR_LAST;i++)
  {
    if ( (val=startalk_get_setting_uchar(n, i)) != -1)
    {
      if ( (failreason=startalk_validate_setting_uchar(testse,i,val)) != NULL )
      {
        warn("Invalid setting: %s\n",failreason);
        warnings++;
      }
    }
  }
  for(i=0;i<STARTALK_SET_STRING_LAST;i++)
  {
    if ( (sval=startalk_get_setting_string(n, i)) != NULL)
    {
      if ( (failreason=startalk_validate_setting_string(testse,i,sval)) != NULL)
      {
        warn("Invalid setting: %s\n",failreason);
        warnings++;
      }
    }
  }
  for(i=0;i<STARTALK_SET_UINT_LAST;i++)
  {
    if ( (val=startalk_get_setting_uint(n, i)) != -1)
    {
      if ( (failreason=startalk_validate_setting_uint(testse,i,val)) != NULL)
      {
        warn("Invalid setting: %s\n",failreason);
        warnings++;
      }
    }
  }
  for(i=0;i<STARTALK_SET_LANG_LAST;i++)
  {
    if ( (val=startalk_get_setting_lang(n, i)) != -1)
    {
      if ( (failreason=startalk_validate_setting_lang(testse,i,val)) != NULL)
      {
        warn("Invalid setting: %s\n",failreason);
        warnings++;
      }
    }
  }
  for(i=0;i<STARTALK_SET_SUINT_LAST;i++)
  {
    if ( (val=startalk_get_setting_suint(n, i)) != -1)
    {
      if ( (failreason=startalk_validate_setting_suint(testse,i,val)) != NULL)
      {
        warn("Invalid setting: %s\n",failreason);
        warnings++;
      }
    }
  }
  if (warnings)
    return reterr(0,"%d invalid settings were encountered (see warnings)",warnings);
  else
    return 1;
}
