#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# StarPU --- Runtime system for heterogeneous multicore architectures.
#
# Copyright (C) 2020-2022  Université de Bordeaux, CNRS (LaBRI UMR 5800), Inria
#
# StarPU is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation; either version 2.1 of the License, or (at
# your option) any later version.
#
# StarPU is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# See the GNU Lesser General Public License in COPYING.LGPL for more details.
#

import sys

"""
Convert event keys into event names
Running starpu_fxt_tool with the option -number-events produces a file number_events.data
This file contains the number of events for each event type.
Events are represented with their key.
To convert event keys to event names, call starpu_fxt_number_events_to_names.py
"""

# STARPU_FXT_EVENT_DEFINES is generated by configure and is the output of
# the following command:
# grep -E "#define\s+_STARPU_(MPI_)?FUT_" src/common/fxt.h mpi/src/starpu_mpi_fxt.h | grep 0x | grep -v 0x1 |cut -d : -f 2

fxt_codes_raw = """
#define _STARPU_FUT_WORKER_INIT_START	0x5100
#define _STARPU_FUT_WORKER_INIT_END	0x5101
#define	_STARPU_FUT_START_CODELET_BODY	0x5102
#define	_STARPU_FUT_END_CODELET_BODY	0x5103
#define _STARPU_FUT_JOB_PUSH		0x5104
#define _STARPU_FUT_JOB_POP		0x5105
#define _STARPU_FUT_UPDATE_TASK_CNT	0x5106
#define _STARPU_FUT_START_FETCH_INPUT_ON_TID	0x5107
#define _STARPU_FUT_END_FETCH_INPUT_ON_TID	0x5108
#define _STARPU_FUT_START_PUSH_OUTPUT_ON_TID	0x5109
#define _STARPU_FUT_END_PUSH_OUTPUT_ON_TID	0x5110
#define _STARPU_FUT_TAG		0x5111
#define _STARPU_FUT_TAG_DEPS	0x5112
#define _STARPU_FUT_TASK_DEPS		0x5113
#define _STARPU_FUT_DATA_COPY		0x5114
#define _STARPU_FUT_WORK_STEALING	0x5115
#define _STARPU_FUT_WORKER_DEINIT_START	0x5116
#define _STARPU_FUT_WORKER_DEINIT_END	0x5117
#define _STARPU_FUT_WORKER_SLEEP_START	0x5118
#define _STARPU_FUT_WORKER_SLEEP_END	0x5119
#define _STARPU_FUT_TASK_SUBMIT		0x511a
#define _STARPU_FUT_CODELET_DATA_HANDLE	0x511b
#define _STARPU_FUT_MODEL_NAME		0x511c
#define _STARPU_FUT_DATA_NAME		0x511d
#define _STARPU_FUT_DATA_COORDINATES	0x511e
#define _STARPU_FUT_HANDLE_DATA_UNREGISTER	0x511f
#define _STARPU_FUT_CODELET_DATA_HANDLE_NUMA_ACCESS	0x5120
#define	_STARPU_FUT_NEW_MEM_NODE	0x5122
#define	_STARPU_FUT_START_CALLBACK	0x5123
#define	_STARPU_FUT_END_CALLBACK	0x5124
#define	_STARPU_FUT_TASK_DONE		0x5125
#define	_STARPU_FUT_TAG_DONE		0x5126
#define	_STARPU_FUT_START_ALLOC		0x5127
#define	_STARPU_FUT_END_ALLOC		0x5128
#define	_STARPU_FUT_START_ALLOC_REUSE	0x5129
#define	_STARPU_FUT_END_ALLOC_REUSE	0x5130
#define	_STARPU_FUT_USED_MEM	0x512a
#define _STARPU_FUT_TASK_NAME	0x512b
#define _STARPU_FUT_DATA_WONT_USE	0x512c
#define _STARPU_FUT_TASK_COLOR	0x512d
#define _STARPU_FUT_DATA_DOING_WONT_USE	0x512e
#define _STARPU_FUT_TASK_LINE	0x512f
#define	_STARPU_FUT_START_MEMRECLAIM	0x5131
#define	_STARPU_FUT_END_MEMRECLAIM	0x5132
#define	_STARPU_FUT_START_DRIVER_COPY	0x5133
#define	_STARPU_FUT_END_DRIVER_COPY	0x5134
#define	_STARPU_FUT_START_DRIVER_COPY_ASYNC	0x5135
#define	_STARPU_FUT_END_DRIVER_COPY_ASYNC	0x5136
#define	_STARPU_FUT_START_PROGRESS_ON_TID	0x5137
#define	_STARPU_FUT_END_PROGRESS_ON_TID		0x5138
#define _STARPU_FUT_USER_EVENT		0x5139
#define _STARPU_FUT_SET_PROFILING	0x513a
#define _STARPU_FUT_TASK_WAIT_FOR_ALL	0x513b
#define _STARPU_FUT_EVENT		0x513c
#define _STARPU_FUT_THREAD_EVENT	0x513d
#define	_STARPU_FUT_CODELET_DETAILS	0x513e
#define	_STARPU_FUT_CODELET_DATA	0x513f
#define _STARPU_FUT_LOCKING_MUTEX	0x5140
#define _STARPU_FUT_MUTEX_LOCKED	0x5141
#define _STARPU_FUT_UNLOCKING_MUTEX	0x5142
#define _STARPU_FUT_MUTEX_UNLOCKED	0x5143
#define _STARPU_FUT_TRYLOCK_MUTEX	0x5144
#define _STARPU_FUT_RDLOCKING_RWLOCK	0x5145
#define _STARPU_FUT_RWLOCK_RDLOCKED	0x5146
#define _STARPU_FUT_WRLOCKING_RWLOCK	0x5147
#define _STARPU_FUT_RWLOCK_WRLOCKED	0x5148
#define _STARPU_FUT_UNLOCKING_RWLOCK	0x5149
#define _STARPU_FUT_RWLOCK_UNLOCKED	0x514a
#define _STARPU_FUT_LOCKING_SPINLOCK	0x514b
#define _STARPU_FUT_SPINLOCK_LOCKED	0x514c
#define _STARPU_FUT_UNLOCKING_SPINLOCK	0x514d
#define _STARPU_FUT_SPINLOCK_UNLOCKED	0x514e
#define _STARPU_FUT_TRYLOCK_SPINLOCK	0x514f
#define _STARPU_FUT_COND_WAIT_BEGIN	0x5150
#define _STARPU_FUT_COND_WAIT_END	0x5151
#define _STARPU_FUT_MEMORY_FULL		0x5152
#define _STARPU_FUT_DATA_LOAD 		0x5153
#define _STARPU_FUT_START_UNPARTITION_ON_TID 0x5154
#define _STARPU_FUT_END_UNPARTITION_ON_TID 0x5155
#define	_STARPU_FUT_START_FREE		0x5156
#define	_STARPU_FUT_END_FREE		0x5157
#define	_STARPU_FUT_START_WRITEBACK	0x5158
#define	_STARPU_FUT_END_WRITEBACK	0x5159
#define _STARPU_FUT_SCHED_COMPONENT_PUSH_PRIO 	0x515a
#define _STARPU_FUT_SCHED_COMPONENT_POP_PRIO 	0x515b
#define	_STARPU_FUT_START_WRITEBACK_ASYNC	0x515c
#define	_STARPU_FUT_END_WRITEBACK_ASYNC		0x515d
#define	_STARPU_FUT_HYPERVISOR_BEGIN    0x5160
#define	_STARPU_FUT_HYPERVISOR_END	0x5161
#define _STARPU_FUT_BARRIER_WAIT_BEGIN		0x5162
#define _STARPU_FUT_BARRIER_WAIT_END		0x5163
#define _STARPU_FUT_WORKER_SCHEDULING_START	0x5164
#define _STARPU_FUT_WORKER_SCHEDULING_END	0x5165
#define _STARPU_FUT_WORKER_SCHEDULING_PUSH	0x5166
#define _STARPU_FUT_WORKER_SCHEDULING_POP	0x5167
#define	_STARPU_FUT_START_EXECUTING	0x5168
#define	_STARPU_FUT_END_EXECUTING	0x5169
#define _STARPU_FUT_SCHED_COMPONENT_NEW		0x516a
#define _STARPU_FUT_SCHED_COMPONENT_CONNECT	0x516b
#define _STARPU_FUT_SCHED_COMPONENT_PUSH	0x516c
#define _STARPU_FUT_SCHED_COMPONENT_PULL	0x516d
#define _STARPU_FUT_TASK_SUBMIT_START	0x516e
#define _STARPU_FUT_TASK_SUBMIT_END	0x516f
#define _STARPU_FUT_TASK_BUILD_START	0x5170
#define _STARPU_FUT_TASK_BUILD_END	0x5171
#define _STARPU_FUT_TASK_MPI_DECODE_START	0x5172
#define _STARPU_FUT_TASK_MPI_DECODE_END		0x5173
#define _STARPU_FUT_TASK_MPI_PRE_START	0x5174
#define _STARPU_FUT_TASK_MPI_PRE_END	0x5175
#define _STARPU_FUT_TASK_MPI_POST_START	0x5176
#define _STARPU_FUT_TASK_MPI_POST_END	0x5177
#define _STARPU_FUT_TASK_WAIT_START	0x5178
#define _STARPU_FUT_TASK_WAIT_END	0x5179
#define _STARPU_FUT_TASK_WAIT_FOR_ALL_START	0x517a
#define _STARPU_FUT_TASK_WAIT_FOR_ALL_END	0x517b
#define _STARPU_FUT_HANDLE_DATA_REGISTER	0x517c
#define _STARPU_FUT_START_FETCH_INPUT	0x517e
#define _STARPU_FUT_END_FETCH_INPUT	0x517f
#define _STARPU_FUT_TASK_THROTTLE_START	0x5180
#define _STARPU_FUT_TASK_THROTTLE_END	0x5181
#define _STARPU_FUT_DATA_STATE_INVALID 0x5182
#define _STARPU_FUT_DATA_STATE_OWNER      0x5183
#define _STARPU_FUT_DATA_STATE_SHARED     0x5184
#define _STARPU_FUT_DATA_REQUEST_CREATED   0x5185
#define _STARPU_FUT_PAPI_TASK_EVENT_VALUE   0x5186
#define _STARPU_FUT_TASK_EXCLUDE_FROM_DAG   0x5187
#define _STARPU_FUT_TASK_END_DEP	0x5188
#define _STARPU_FUT_TASK_BUBBLE		0x5189
#define _STARPU_MPI_FUT_START				0x5201
#define _STARPU_MPI_FUT_STOP				0x5202
#define _STARPU_MPI_FUT_BARRIER				0x5203
#define _STARPU_MPI_FUT_ISEND_SUBMIT_BEGIN		0x5204
#define _STARPU_MPI_FUT_ISEND_SUBMIT_END		0x5205
#define _STARPU_MPI_FUT_IRECV_SUBMIT_BEGIN		0x5206
#define _STARPU_MPI_FUT_IRECV_SUBMIT_END		0x5207
#define _STARPU_MPI_FUT_ISEND_COMPLETE_BEGIN		0x5208
#define _STARPU_MPI_FUT_ISEND_COMPLETE_END		0x5209
#define _STARPU_MPI_FUT_DATA_SET_RANK			0x521a
#define _STARPU_MPI_FUT_IRECV_TERMINATED		0x521b
#define _STARPU_MPI_FUT_ISEND_TERMINATED		0x521c
#define _STARPU_MPI_FUT_TESTING_DETACHED_BEGIN		0x521d
#define _STARPU_MPI_FUT_TESTING_DETACHED_END		0x521e
#define _STARPU_MPI_FUT_TEST_BEGIN			0x521f
#define _STARPU_MPI_FUT_TEST_END			0x5220
#define _STARPU_MPI_FUT_IRECV_COMPLETE_BEGIN		0x520a
#define _STARPU_MPI_FUT_IRECV_COMPLETE_END		0x520b
#define _STARPU_MPI_FUT_SLEEP_BEGIN			0x520c
#define _STARPU_MPI_FUT_SLEEP_END			0x520d
#define _STARPU_MPI_FUT_DTESTING_BEGIN			0x520e
#define _STARPU_MPI_FUT_DTESTING_END			0x520f
#define _STARPU_MPI_FUT_UTESTING_BEGIN			0x5210
#define _STARPU_MPI_FUT_UTESTING_END			0x5211
#define _STARPU_MPI_FUT_UWAIT_BEGIN			0x5212
#define _STARPU_MPI_FUT_UWAIT_END			0x5213
#define _STARPU_MPI_FUT_POLLING_BEGIN			0x5214
#define _STARPU_MPI_FUT_POLLING_END			0x5215
#define _STARPU_MPI_FUT_DRIVER_RUN_BEGIN		0x5216
#define _STARPU_MPI_FUT_DRIVER_RUN_END			0x5217
#define _STARPU_MPI_FUT_DATA_SET_TAG			0x5218
#define _STARPU_MPI_FUT_IRECV_NUMA_NODE			0x5219
#define _STARPU_MPI_FUT_ISEND_NUMA_NODE			0x5221
#define _STARPU_MPI_FUT_CHECKPOINT_BEGIN            	0x5222
#define _STARPU_MPI_FUT_CHECKPOINT_END              	0x5223
"""

PROGNAME=sys.argv[0]

number_events_path = None

def usage():
    print("Convert event keys in number_events.data to event names")
    print("")
    print("Usage: %s <number_events.data path>" % PROGNAME)
    print("")
    print("Options:")
    print("	-h, --help          display this help and exit")
    print("	-v, --version       output version information and exit")
    print("")
    print("Report bugs to <starpu-devel@inria.fr>")
    sys.exit(1)

if len(sys.argv) == 2:
    if sys.argv[1] == '-v' or sys.argv[1] == '--version':
        print("%s (StarPU) 1.4.1" % PROGNAME)
        sys.exit(0)
    elif sys.argv[1] == '-h' or sys.argv[1] == '--help':
        usage()
    else:
        number_events_path = sys.argv[1]
else:
    usage()

# Process fxt_code_raw content to ease the conversion:
fxt_codes = dict()
for line in fxt_codes_raw.split("\n"):
    elements = line.split()

    if len(elements) == 3:
        key = int(elements[2][2:], 16)
        assert key not in fxt_codes

        fxt_codes[key] = elements[1]


# Convert content of the file:
nb_events = 0

with open(number_events_path, 'r') as f:
    for line in f:
        elements = line.split()
        if len(elements) == 2:
            key = int(elements[0][2:], 16)
            nb = int(elements[1])
            nb_events += nb
            if key in fxt_codes:
                print("%12d    %s" % (nb, fxt_codes[key]))
            else:
                print("%12d    %s" % (nb, elements[0]))

print("       TOTAL:   %d" % nb_events)
