// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  starplot.cc
  This is the file which includes the StarPlot main() function.
  However, this does little besides set up the main program window -
  most of the program functionality is defined in the other .cc files
  in this directory.
*/

#include <gtk/gtk.h>
#include "starplot.h"
#include "starplot.xpm"

#define FLIP(boolean) ((boolean) = !(boolean))

// initialize globals

#define SET(type, var)		type var = defaults::var
namespace globals {
  SET(Rules, chart_rules);
  SET(double, hr_mag_bright);
  SET(double, hr_mag_dim);
  SET(std::string, program_help_browser);
  SET(std::string, display_fontname);

  StarArray	    *chart_stararray	= NULL;
  GTKViewer	    *program_viewer	= NULL;
  GtkWidget	    *program_canvas	= NULL;
  GtkWidget	    *program_status	= NULL;
  GTKViewer         *program_hr_viewer  = NULL;
  GtkWidget         *program_hr_canvas  = NULL;
  GdkPixbuf	    *program_icon	= NULL;
  GdkPixmap	    *program_pixmap	= NULL;
}; // end "globals" namespace
  

void redraw_display(GtkWidget *widget)
{
  if (globals::program_viewer && globals::chart_stararray) {
    globals::chart_stararray->Display(globals::program_viewer);
    gdk_draw_pixmap(widget->window,
		    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		    globals::program_pixmap,
		    0, 0, 0, 0,
		    widget->allocation.width, widget->allocation.height);
  }
}


/* Create a new backing pixmap of the appropriate size */
static gint configure_event( GtkWidget         *widget,
			     GdkEventConfigure *event )
{
  if (globals::program_pixmap) {
    delete globals::program_viewer;
    gdk_pixmap_unref (globals::program_pixmap);
  }
  globals::program_pixmap = gdk_pixmap_new(widget->window,
					   widget->allocation.width,
					   widget->allocation.height,
					   -1);
  globals::program_viewer =
    new GTKViewer(globals::program_pixmap, widget);
  redraw_display(widget);
  return true;
}


/* Redraw the screen from the backing pixmap */
static gint expose_event( GtkWidget      *widget,
			  GdkEventExpose *event )
{
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  globals::program_pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);
  return false;
}


void redraw_all(star_changetype_t changetype)
{
  globals::chart_stararray->SetRules(globals::chart_rules, changetype);
  redraw_display(globals::program_canvas);
  if (globals::program_hr_viewer)
    hr_redraw_display(globals::program_hr_canvas);
  if (globals::program_treeview)
    update_info(globals::program_treeview);

  std::string statustext;
  if (globals::chart_rules.ChartFileNames.size() == 1)
    statustext = starstrings::ssprintf(_("Viewing %d of %d stars of %s"),
	globals::chart_stararray->size(),
	globals::chart_stararray->totalstars(),
	globals::chart_rules.ChartFileNames[0].c_str());
  else
    statustext = starstrings::ssprintf(_("Viewing %d of %d stars in %d files"),
	globals::chart_stararray->size(),
	globals::chart_stararray->totalstars(),
	globals::chart_rules.ChartFileNames.size());
  gtk_label_set_text (GTK_LABEL (globals::program_status), statustext.c_str());
}


/* Function to respond to mouse clicks on the star chart.  A left-click
 * will recenter the chart to the clicked star.  A right-click will pop up
 * a mini-dialog box with some information about the star.  A left-click
 * on a chart legend item will toggle the visibility of that spectral 
 * class.  If no star or legend item is near the mouse click location,
 * nothing will happen.
 */
static gint button_press_event( GtkWidget      *widget,
				GdkEventButton *event )
{
  bool found = false;
  int i;
  gdouble X = event->x, Y = event->y;

  if (! (globals::program_pixmap && globals::chart_stararray)) return false;

  /* check for star legend item */
  if (event->button == 1 && globals::chart_rules.ChartLegend) {
    int xbase = (globals::program_viewer->width() > 100) ?
		globals::program_viewer->width() - 95 : 5;

    if (pow(X - (xbase + LEGEND_OFFSETS[10][0]), 2)
    	+ pow(Y - LEGEND_OFFSETS[10][1], 2) < 25) {
      FLIP(globals::chart_rules.StarClasses[0]);
      redraw_all(FILTER_CHANGE);
      return true;
    }

    for (unsigned int i = 0; i < 10; i++) {
      if (pow(X - (xbase + LEGEND_OFFSETS[i][0]), 2) 
          + pow(Y - LEGEND_OFFSETS[i][1], 2) < 25) {
	FLIP(globals::chart_rules.StarClasses[i]);
	redraw_all(FILTER_CHANGE);
	return true;
      }
    }
  }

  /* if not a legend item, find the star that was clicked on */
  for (i = globals::chart_stararray->size() - 1; i >= 0; i--) {
    int x = (*globals::chart_stararray)[i].GetXPixel();
    int y = (*globals::chart_stararray)[i].GetYPixel();
    int r = (*globals::chart_stararray)[i].GetRPixel();

    // give the hotspot two pixels extra in radius to make it easier to
    // click on stars --------------------------v
    if (pow(X - x, 2) + pow(Y - y, 2) < pow(r + 2, 2)) {
      found = true;
      break;
    }
  }
    
  if (found) {
    if (event->button == 1) {
      /* recenter the chart on the clicked star */
      globals::chart_rules.ChartLocation =
	(*globals::chart_stararray)[i].GetStarXYZ();
      redraw_all(LOCATION_CHANGE);
    }
    else if (event->button == 3 || event->button == 2) {
      /* display information about the star */
      my_gtk_star_popup((*globals::chart_stararray)[i]);
    }
  }

  return true;
}


gint delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  starplot_quit();
  return false;
}


int main( int argc, char *argv[] )
{
  GtkWidget *window;
  GtkWidget *mainmenu;
  GtkWidget *mainbox, *hbox, *statusbar;
  GtkWidget *buttonbar;
  
  /* extract GTK command-line flags and set LC_ALL */
  gtk_init(&argc, &argv);

  // Get starplot-specific command-line flags
  // XXX: to do

  // set up i18n
  
  // This prevents the problem of having atof not deal well with en_US
  // style decimal numbers in the data files.  Of course then all numbers
  // are displayed in en_US style as well, so this isn't perfect...
  setlocale(LC_NUMERIC, "C");  
  bindtextdomain (PACKAGE, LOCALEDIR);
  // GTK+ 2 gets very angry if UTF-8 isn't used:
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  textdomain (PACKAGE);

  // Set up the window icon
  globals::program_icon = gdk_pixbuf_new_from_xpm_data (starplot_xpm);
  
  /* This sets the window title and border width: */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "StarPlot " STARPLOT_VERSION);
  gtk_window_set_icon (GTK_WINDOW (window), globals::program_icon);
  gtk_container_set_border_width (GTK_CONTAINER (window), 0);
  my_gtk_main_menu (window, &mainmenu);
  my_gtk_push_buttons (window, &buttonbar);

  /* Set a delete_event handler for ending the program */
  g_signal_connect (G_OBJECT (window), "delete_event",
		      G_CALLBACK (delete_event), NULL);

  /* Set up the drawing area */
  globals::program_canvas = gtk_drawing_area_new();
  gtk_drawing_area_size (GTK_DRAWING_AREA (globals::program_canvas), 
		          defaults::program_width,
			  defaults::program_height);
  gtk_widget_show (globals::program_canvas);

  /* Signals used to handle backing pixmap */
  g_signal_connect (G_OBJECT (globals::program_canvas), "expose_event",
		      G_CALLBACK (expose_event), NULL);
  g_signal_connect (G_OBJECT (globals::program_canvas),"configure_event",
		      G_CALLBACK (configure_event), NULL);
  
  /* Event signals */
  g_signal_connect (G_OBJECT (globals::program_canvas), "button_press_event",
		      G_CALLBACK (button_press_event), NULL);
  gtk_widget_set_events (globals::program_canvas, GDK_EXPOSURE_MASK
			 | GDK_LEAVE_NOTIFY_MASK
			 | GDK_BUTTON_PRESS_MASK
			 | GDK_POINTER_MOTION_MASK
			 | GDK_POINTER_MOTION_HINT_MASK);

  hbox = gtk_hbox_new (false, 0);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (hbox), buttonbar, false, true, 0);
  gtk_box_pack_start (GTK_BOX (hbox), globals::program_canvas, true, true, 0);

  globals::program_status = gtk_label_new ("");
  gtk_misc_set_alignment (GTK_MISC (globals::program_status), 0.0F, 0.0F);
  gtk_widget_show (globals::program_status);
  statusbar = gtk_frame_new (0);
  gtk_frame_set_shadow_type (GTK_FRAME (statusbar), GTK_SHADOW_OUT);
  gtk_widget_show (statusbar);
  gtk_container_add (GTK_CONTAINER (statusbar), globals::program_status);

  mainbox = gtk_vbox_new (false, 0);
  gtk_container_border_width (GTK_CONTAINER (mainbox), 1);
  gtk_container_add (GTK_CONTAINER (window), mainbox);
  gtk_box_pack_start (GTK_BOX (mainbox), mainmenu, false, true, 0);
  gtk_box_pack_start (GTK_BOX (mainbox), hbox, true, true, 0);
  gtk_box_pack_start (GTK_BOX (mainbox), statusbar, false, true, 0);

  globals::chart_stararray = new StarArray();
  gtk_widget_realize (globals::program_canvas);

  /* load settings from starplot RC file, if it exists */
  read_rc_file();

  /* assume any command-line arguments are data files to use */
  if (argc > 1) {
    globals::chart_rules.ChartFileNames = StringList(1, argv[1]);
    for (int i = 2; i < argc; i++)
      globals::chart_rules.ChartFileNames.push_back(argv[i]);
  }

  gtk_widget_modify_font (GTK_WIDGET (globals::program_canvas),
    pango_font_description_from_string (globals::display_fontname.c_str()));
  redraw_all(FILE_CHANGE);

  gtk_widget_show (globals::program_canvas);
  gtk_widget_show (mainmenu);
  gtk_widget_show (buttonbar);
  gtk_widget_show (mainbox);
  gtk_widget_show (window);

  gtk_main ();
  return 0;
}

