// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

 
/*
  menuops.cc
  This file contains code to set up the StarPlot menu bar and to deal with
  the toggle and radio button options in the StarPlot Options menu.
*/

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <string.h> // for strdup()
#include <cstdlib>  // for free()
#include <map>

#include "starplot.h"
#include "xpmdata.h"

GtkItemFactoryEntry menu_items[] = {
  FILE_MENU, CHART_MENU, OPTIONS_CUSTOM_MENU, OPTIONS_TOGGLE_MENU,
  OPTIONS_LABEL_MENU, OPTIONS_DIAMETER_MENU, OPTIONS_COORD_MENU, STARS_MENU,
  HELP_MENU
};

GtkWidget *options_toggle_widgets[OPTIONS_TOGGLE_MENU_SIZE];
char options_toggle_names[OPTIONS_TOGGLE_MENU_SIZE][100];
GtkWidget *options_label_widgets[OPTIONS_LABEL_MENU_SIZE];
char options_label_names[OPTIONS_LABEL_MENU_SIZE][100];
GtkWidget *options_diameter_widgets[OPTIONS_DIAMETER_MENU_SIZE];
char options_diameter_names[OPTIONS_DIAMETER_MENU_SIZE][100];
GtkWidget *options_coord_widgets[OPTIONS_COORD_MENU_SIZE];
char options_coord_names[OPTIONS_COORD_MENU_SIZE][100];


/* set_menu_globals(): A boring function to set all the arrays of menu item
 *  widgets to be globally available. */

void set_menu_globals (GtkItemFactory *item_factory)
{
  unsigned int i;
  GtkWidget *item = 0;

  strcpy(options_toggle_names[TOGGLE_BARS], _("/Options/Toggle Bars"));
  strcpy(options_toggle_names[TOGGLE_GRID], _("/Options/Toggle Grid"));
  strcpy(options_toggle_names[TOGGLE_LEGEND], _("/Options/Toggle Legend"));

  strcpy(options_label_names[(int)LANDMARK_LABEL],
	 _("/Options/Star Labels/Landmark Stars"));
  strcpy(options_label_names[(int)STRING_LABEL],
	 _("/Options/Star Labels/All Star Names"));
  strcpy(options_label_names[(int)NUMBER_LABEL],
	 _("/Options/Star Labels/Numerical Index"));
  strcpy(options_label_names[(int)NO_LABEL],
	 _("/Options/Star Labels/Labels Off"));

  strcpy(options_diameter_names[(int)MAGNITUDE_DIAMETERS],
	 _("/Options/Star Diameters/By Magnitude"));
  strcpy(options_diameter_names[(int)MK_DIAMETERS],
	 _("/Options/Star Diameters/By MK Class"));

  strcpy(options_coord_names[CELESTIAL], 
	 _("/Options/Coordinate System/Celestial"));
  strcpy(options_coord_names[GALACTIC],
	 _("/Options/Coordinate System/Galactic"));

  /* The following code sets the check buttons in the Options menu to be
     toggled ON by default, matching the initial state of the program.

     According to the GTK documentation, I should not use the "active" field
     of GtkCheckMenuItem directly, so below I should replace each line
       (GTK_CHECK_MENU_ITEM (item))->active = true;
     with the line
       gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item), true);
     However, when I do this, StarPlot segfaults.  Fixes welcomed.
  */
  for (i = 0; i < OPTIONS_TOGGLE_MENU_SIZE; i++) {
    item = gtk_item_factory_get_widget (item_factory, options_toggle_names[i]);
    (GTK_CHECK_MENU_ITEM (item))->active = true;
    // gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item), true);
    options_toggle_widgets[i] = item;
  }

  for (i = 0; i < OPTIONS_LABEL_MENU_SIZE; i++)
    options_label_widgets[i] =
      gtk_item_factory_get_widget (item_factory, options_label_names[i]);
  for (i = 0; i < OPTIONS_DIAMETER_MENU_SIZE; i++)
    options_diameter_widgets[i] =
      gtk_item_factory_get_widget (item_factory, options_diameter_names[i]);
  for (i = 0; i < OPTIONS_COORD_MENU_SIZE; i++)
    options_coord_widgets[i] =
      gtk_item_factory_get_widget (item_factory, options_coord_names[i]);

  return;
}


/* my_gtk_main_menu(): This function prepares the StarPlot menu bar. */

void my_gtk_main_menu (GtkWidget *window, GtkWidget **menubar)
{
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  int nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
 
  /* Gettextize the menu item names */
  for (int i = 0; i < nmenu_items; i++) {
    menu_items[i].path = strdup(gettext(menu_items[i].path));
    if (menu_items[i].item_type[0] != 0 && menu_items[i].item_type[0] != '<')
      menu_items[i].item_type = strdup(gettext(menu_items[i].item_type));
  }
  
  accel_group = gtk_accel_group_new ();
       
  /* This function initializes the item factory.
     Param 1: The type of menu - can be GTK_TYPE_MENU_BAR, GTK_TYPE_MENU,
     or GTK_TYPE_OPTION_MENU.
     Param 2: The path of the menu.
     Param 3: A pointer to a gtk_accel_group.  The item factory sets up
     the accelerator table while generating menus.
  */

  item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", 
				       accel_group);

  /* This function generates the menu items. Pass the item factory,
     the number of items in the array, the array itself, and any
     callback data for the the menu items. */
  gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);

  /* Attach the new accelerator group to the window. */
  gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);

  /* Set menu widgets to be globally available */
  set_menu_globals(item_factory);

  if (menubar)
    /* Finally, return the actual menu bar created by the item factory. */ 
    *menubar = gtk_item_factory_get_widget (item_factory, "<main>");
}


/* my_gtk_button_bar(): This function creates a button bar with
   OK, Defaults and Cancel buttons, and packs it into
   a vbox or hbox given as an argument.  It will NOT, however, set
   up the callback functions. */

void my_gtk_button_bar(GtkWidget **OK_btn, GtkWidget **defaults_btn,
		       GtkWidget **cancel_btn, GtkWidget *box, int spacing)
{
  GtkWidget *buttonbox = gtk_hbutton_box_new ();
  gtk_button_box_set_layout (GTK_BUTTON_BOX (buttonbox), GTK_BUTTONBOX_SPREAD);
  gtk_box_set_spacing (GTK_BOX (buttonbox), spacing);
  gtk_widget_show (buttonbox);
  gtk_box_pack_start (GTK_BOX (box), buttonbox, false, false, 0);

  *OK_btn = gtk_button_new_from_stock (GTK_STOCK_OK);
  gtk_container_add (GTK_CONTAINER (buttonbox), *OK_btn);
  gtk_widget_show(*OK_btn);
  
  *defaults_btn = gtk_button_new_from_stock (GTK_STOCK_REVERT_TO_SAVED);
  gtk_container_add (GTK_CONTAINER (buttonbox), *defaults_btn);
  gtk_widget_show(*defaults_btn);
  
  *cancel_btn = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
  gtk_container_add (GTK_CONTAINER (buttonbox), *cancel_btn);
  gtk_widget_show(*cancel_btn);
  
  return;
}

/* my_gtk_buttons_connect_destroy(): Connects all the buttons supplied
   as arguments to the gtk_destroy_widget() function.  Should be called
   after any other signal connections.
*/
void my_gtk_buttons_connect_destroy(GtkWidget *ok, GtkWidget *defaults,
				    GtkWidget *cancel, GtkWidget *window)
{
  g_signal_connect_swapped (G_OBJECT (ok), "clicked",
    G_CALLBACK (gtk_widget_destroy), G_OBJECT (window));
  g_signal_connect_swapped (G_OBJECT (defaults), "clicked",
    G_CALLBACK (gtk_widget_destroy), G_OBJECT (window));
  g_signal_connect_swapped (G_OBJECT (cancel), "clicked",
    G_CALLBACK (gtk_widget_destroy), G_OBJECT (window));
  return;
}

/* my_gtk_popup_button(): creates a button for closing popup. */
void my_gtk_popup_button(GtkWidget **close, GtkWidget *box, GtkWidget *window)
{
  GtkWidget *buttonbox = gtk_hbutton_box_new ();
  gtk_button_box_set_layout (GTK_BUTTON_BOX (buttonbox), GTK_BUTTONBOX_SPREAD);
  gtk_box_set_spacing (GTK_BOX (buttonbox), 0);
  gtk_widget_show (buttonbox);
  gtk_box_pack_start (GTK_BOX (box), buttonbox, false, false, 0);

  *close = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
  gtk_container_add (GTK_CONTAINER (buttonbox), *close);
  g_signal_connect_swapped (G_OBJECT (*close), "clicked",
    G_CALLBACK (gtk_widget_destroy), G_OBJECT (window));
  gtk_widget_show(*close);
}

/* set_item(): This function is a general function which can be used to
 *  set the value of any item, and if desired, change the state of the
 *  applicable GtkMenuItem in a radio-button/check-button menu.
 *  "itemptr" is a pointer to the value to change; "value" is the new value
 *  desired; "menuptr" is the GtkWidget associated with the value; "redraw"
 *  is true if updating the StarPlot main display is desired, and "c"
 *  is the type of display update to perform. */

template<class T> void
set_item(T *itemptr, T value, GtkWidget *menuptr, bool redraw,
	 star_changetype_t c)
{
  // this keeps track of whether or not a particular item should be changed
  static std::map<T *, bool> enabled;

  // will only insert a "true" value for given item pointer if that item
  //  has not yet been used:
  std::pair<T *, bool> currenttype = std::pair<T *, bool>(itemptr, true);
  enabled.insert(currenttype);

  // test for whether itemptr is enabled, to ensure that (a) this function
  //  doesn't cause itself to be called again by playing with radio button
  //  toggle states; (b) this is not being called as a result of a _different_
  //  radio button which controls the same value being toggled ON, thus
  //  toggling this one OFF.

  if (enabled[itemptr]) {
    enabled[itemptr] = false;
    {
      *itemptr = value;
      if (menuptr) {
	if (GTK_OBJECT_TYPE(menuptr) == gtk_type_from_name("GtkRadioMenuItem"))
	  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuptr), true);
	else if (gtk_type_is_a (GTK_OBJECT_TYPE(menuptr), 
				gtk_type_from_name("GtkCheckMenuItem")))
	  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuptr),
					  value ? true : false);
      }
      if (redraw)
	redraw_all (c);
    }
    enabled[itemptr] = true;
  }
}


void flip_toggle_item(bool *itemptr, int number, bool redraw)
{ set_item(itemptr, ! *itemptr, options_toggle_widgets[number], redraw); }

void set_toggle_item(bool *itemptr, bool value, int number, bool redraw)
{
  if (*itemptr == value) return;
  else flip_toggle_item(itemptr, number, redraw);
}

void toggle_bars()  
{ flip_toggle_item(&globals::chart_rules.StarBars, TOGGLE_BARS, true); }
void toggle_grid() 
{ flip_toggle_item(&globals::chart_rules.ChartGrid, TOGGLE_GRID, true); }
void toggle_legend() 
{ flip_toggle_item(&globals::chart_rules.ChartLegend, TOGGLE_LEGEND, true); }

void set_label_item(star_label_t l, bool redraw)
{
  if (globals::chart_rules.StarLabels == l) return;

  if (globals::program_hr_viewer) {
    set_item(&globals::chart_rules.StarLabels, l,
	     hr_options_label_widgets[l], false);
  }
  set_item(&globals::chart_rules.StarLabels, l,
	   options_label_widgets[l], redraw);
}

void labels_all()       { set_label_item(STRING_LABEL, true); }
void labels_landmk()    { set_label_item(LANDMARK_LABEL, true); }
void labels_numerical() { set_label_item(NUMBER_LABEL, true); }
void labels_off()       { set_label_item(NO_LABEL, true); }

void set_diameter_item(star_diameter_t d, bool redraw)
{
  if (globals::chart_rules.StarDiameters == d) return;

  if (globals::program_hr_viewer) {
    set_item(&globals::chart_rules.StarDiameters, d,
	     hr_options_diameter_widgets[d], false);
  }
  set_item(&globals::chart_rules.StarDiameters, d,
	   options_diameter_widgets[d], redraw);
}

void mk_diameters()  { set_diameter_item(MK_DIAMETERS, true); }
void mag_diameters() { set_diameter_item(MAGNITUDE_DIAMETERS, true); }

void set_coord_item(bool coords, bool redraw)
{
  static bool enabled = true;
  if (globals::chart_rules.CelestialCoords == coords) return;

  if (enabled) {      // wrapper to keep the checkmenuitem toggle from
    enabled = false;  // causing endless recursive calls of this function
    {
      globals::chart_rules.CelestialCoords = coords;
      if (redraw) {
	if (coords == (bool)CELESTIAL)
	  globals::chart_rules.ChartLocation = globals::chart_rules.ChartLocation.toCelestial();
	else
	  globals::chart_rules.ChartLocation = globals::chart_rules.ChartLocation.toGalactic();
	redraw_all(COORDINATE_CHANGE);
      }
      gtk_check_menu_item_set_active 
	(GTK_CHECK_MENU_ITEM (options_coord_widgets[coords]), true);
    }
    enabled = true;
  }
}

void cel_coords() { set_coord_item(CELESTIAL, true); }
void gal_coords() { set_coord_item(GALACTIC, true); }


/* These callbacks are for the button bar immediately underneath the menu. */

void button_zoom(GtkWidget *w, gpointer zoomfactor)
{
  double zf = *(double *)zoomfactor;

  // zoomfactor is > 1.0 to zoom in, < 1.0 to zoom out
  globals::chart_rules.ChartRadius /= zf;
  if (globals::chart_rules.ChartAutosetDimmestMagnitude && zf != 1.0)
    globals::chart_rules.ChartDimmestMagnitude = automagnitude(globals::chart_rules.ChartRadius);
  redraw_all(RADIUS_CHANGE);
  return;
}

void button_rotate(GtkWidget *w, gpointer angle)
{
  double a = *(double *)angle;
  globals::chart_rules.ChartOrientation
    = SolidAngle(globals::chart_rules.ChartOrientation.getPhi() + a,
		 globals::chart_rules.ChartOrientation.getTheta());
  redraw_all(ORIENTATION_CHANGE);
  return;
}

void button_tilt(GtkWidget *w, gpointer angle)
{
  double a = *(double *)angle;
  globals::chart_rules.ChartOrientation
    = SolidAngle(globals::chart_rules.ChartOrientation.getPhi(),
		 globals::chart_rules.ChartOrientation.getTheta() + a);
  redraw_all(ORIENTATION_CHANGE);
  return;
}

void button_magchange(GtkWidget *w, gpointer magchange)
{
  double m = *(double *)magchange;
  globals::chart_rules.ChartDimmestMagnitude += m;
  redraw_all(FILTER_CHANGE);
  return;
}


/* Define the button bar: */

void my_gtk_push_buttons (GtkWidget *window, GtkWidget **buttonbar)
{
  GtkWidget *vbox, *buttons[9], *pixmapw;
  GtkTooltips *tooltips;
  GtkStyle *style = gtk_widget_get_style (window);
  GtkStyle *blackbackground = gtk_style_copy (style);
  GdkPixmap *pixmap, *mask;

  static double zoominfactor = 2.0, zoomoutfactor = 1.0 / zoominfactor;
  static double leftangle = 0.5 HOURS, rightangle = -leftangle;
  static double northangle = 5.0 DEGREES, southangle = -northangle;
  static double magup = -0.5, magdown = -magup;

  vbox = gtk_vbox_new (false, 0);
  gtk_widget_show (vbox);

  *buttonbar = gtk_event_box_new();
  gtk_container_add (GTK_CONTAINER (*buttonbar), vbox);
  gtk_widget_show (*buttonbar);

  blackbackground->bg[GTK_STATE_NORMAL] = blackbackground->black;
  gtk_widget_set_style (*buttonbar, blackbackground);

  for (unsigned int i = 0; i < 9; i++) {
    // this creates the button icons; they are defined in the static
    //  array `xpmdata' in the xpmdata.h header file.
    pixmap = gdk_pixmap_colormap_create_from_xpm_d 
      (0, gtk_widget_get_colormap(window), &mask,
       &style->bg[GTK_STATE_NORMAL], (gchar **)xpmdata[i]);
    pixmapw = gtk_pixmap_new (pixmap, mask);
    gdk_pixmap_unref(pixmap);
    gdk_pixmap_unref(mask);

    // create the buttons themselves
    gtk_widget_show (pixmapw);
    buttons[i] = gtk_button_new();
    gtk_container_add (GTK_CONTAINER(buttons[i]), pixmapw);
    gtk_widget_show (buttons[i]);
    gtk_box_pack_start (GTK_BOX (vbox), buttons[i], false, true, 0);
  }

  // connect the callback functions to the buttons
  g_signal_connect (G_OBJECT (buttons[0]), "clicked",
		      G_CALLBACK (file_open), 0);
  g_signal_connect (G_OBJECT (buttons[1]), "clicked",
		      G_CALLBACK (button_zoom), &zoominfactor);
  g_signal_connect (G_OBJECT (buttons[2]), "clicked",
		      G_CALLBACK (button_zoom), &zoomoutfactor);
  g_signal_connect (G_OBJECT (buttons[3]), "clicked",
		      G_CALLBACK (button_rotate), &leftangle);
  g_signal_connect (G_OBJECT (buttons[4]), "clicked",
		      G_CALLBACK (button_rotate), &rightangle);
  g_signal_connect (G_OBJECT (buttons[5]), "clicked",
		      G_CALLBACK (button_tilt), &northangle);
  g_signal_connect (G_OBJECT (buttons[6]), "clicked",
		      G_CALLBACK (button_tilt), &southangle);
  g_signal_connect (G_OBJECT (buttons[7]), "clicked",
		      G_CALLBACK (button_magchange), &magup);
  g_signal_connect (G_OBJECT (buttons[8]), "clicked",
		      G_CALLBACK (button_magchange), &magdown);

  // and set some popup help tips
  tooltips = gtk_tooltips_new ();
  gtk_tooltips_set_tip (tooltips, buttons[0], _("Open star database..."), 0);
  gtk_tooltips_set_tip (tooltips, buttons[1], _("Zoom in"), 0);
  gtk_tooltips_set_tip (tooltips, buttons[2], _("Zoom out"), 0);
  gtk_tooltips_set_tip (tooltips, buttons[3],
			_("Rotate chart clockwise about its axis"), 0);
  gtk_tooltips_set_tip (tooltips, buttons[4], 
			_("Rotate chart counterclockwise about its axis"), 0);
  gtk_tooltips_set_tip (tooltips, buttons[5], 
			_("Tilt chart north pole towards you"), 0);
  gtk_tooltips_set_tip (tooltips, buttons[6], 
			_("Tilt chart south pole towards you"), 0);
  gtk_tooltips_set_tip (tooltips, buttons[7], 
			_("Decrease magnitude limit (Show fewer stars)"), 0);
  gtk_tooltips_set_tip (tooltips, buttons[8], 
			_("Increase magnitude limit (Show more stars)"), 0);
  return;
}


/* Silly little "about" dialog */
void abouthelp()
{
  GtkWidget *popup, *OK, *topbox, *textbox, *frame;
  GtkTextBuffer *text;
  
  popup = gtk_dialog_new ();
  gtk_window_set_policy (GTK_WINDOW (popup), false, false, true);
  gtk_window_set_title (GTK_WINDOW (popup), _("About StarPlot"));
  
  topbox = gtk_vbox_new (true, 0);
  gtk_container_set_border_width (GTK_CONTAINER (topbox), 10);
  gtk_widget_show (topbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (popup)->vbox), topbox,
		      true, true, 0);
 
  frame = gtk_frame_new (0);
  gtk_widget_show (frame);
  
  string textstr = string(_("StarPlot version %s\n\n"));
  textstr += string(_("Copyright %s 2000-2004 Kevin B. McCarty\n"));
  textstr += string(_("Licensed under the GNU General Public License\n\n"));
  textstr += string(_(
"For help, please select Help->Documentation from the menu, or \
see the HTML documents in %s\n\n"));
  textstr += string(_("For additional data files, see the StarPlot web page:\n"));
  textstr += string("http://starplot.org/");
  textstr = starstrings::ssprintf(textstr.c_str(),
	    STARPLOT_VERSION, COPYRIGHT_UTF8, DOCDIR "/html");
  
  text = gtk_text_buffer_new (0);
  gtk_text_buffer_set_text (text, textstr.c_str(), textstr.size());
  textbox = gtk_text_view_new_with_buffer (text);
  gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (textbox), GTK_WRAP_WORD);
  gtk_text_view_set_editable (GTK_TEXT_VIEW (textbox), false);
  gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (textbox), false);
  gtk_widget_set_usize (textbox, 350, 200);
  gtk_widget_show (textbox);

  gtk_container_add (GTK_CONTAINER (frame), textbox);
  gtk_box_pack_start (GTK_BOX (topbox), frame, true, true, 0);

  my_gtk_popup_button (&OK,  GTK_DIALOG (popup)->action_area, popup);
  gtk_window_set_focus (GTK_WINDOW (popup), OK);
  gtk_widget_show (popup);
}


/* Note: help_select() function is defined in filedialogs.cc since it goes
   with the other file selection dialogs. */


static void openurl(char * url);

/* helpdocs(): Read help documentation */
void helpdocs()
{ openurl("file://" DOCDIR "/html/index.html"); }

void webpage()
{ openurl("http://www.starplot.org/"); }

void openurl(char * url)
{
  pid_t pid;
  if (starstrings::isempty(globals::program_help_browser)) {
    help_select_and_open();
    return;
  }

  char * temp = new char[globals::program_help_browser.size() + 1];
  std::strcpy(temp, globals::program_help_browser.c_str());
  char * const command[3] = { temp, url, NULL };

  pid = fork();

  if (pid < 0) /* problem */ return;
  else if (pid == 0) {
    /* avoid zombie processes by fork()ing twice */
    /* This code comes from _Advanced Programming in the UNIX Environment_,
     *  W. Richard Stevens, 1993 (program #8.5) */
    pid = fork();
    if (pid < 0) /* problem */ _exit(EXIT_FAILURE);
    else if (pid == 0) { /* second child: execvp() the help browser */
      if (execvp(command[0], command) < 0) _exit(EXIT_FAILURE);
    }
    else /* first child */ _exit(EXIT_SUCCESS);
  }

  /* parent waits for first child */
  delete [] temp;
  if (waitpid(pid, NULL, 0) != pid) return;
  return;
}
