/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/
#include <klib/container.h>
#include <klib/log.h>
#include <klib/out.h>
#include <kapp/main.h>
#include <kfs/directory.h>
#include <sra/types.h>
#include <os-native.h>
#include <sysalloc.h>

#include "debug.h"
#include "core.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* ### READ_FILTER splitter/filter ##################################################### */

typedef struct ReadFilterSplitter_struct {
    const SRAColumn* col_rdf;
    const SRAColumn* col_nrd;
    SRAReadFilter read_filter;
} ReadFilterSplitter;

static
rc_t ReadFilterSplitter_GetKeySet(const SRASplitter* cself, SRASplitter_Keys* key, int* keys, spotid_t spot, uint32_t readmask)
{
    rc_t rc = 0;
    ReadFilterSplitter* self = (ReadFilterSplitter*)cself;

    if( self == NULL || key == NULL ) {
        rc = RC(rcSRA, rcNode, rcExecuting, rcParam, rcNull);
    } else {
        const SRAReadFilter* rdf;
        const uint8_t* nreads;
        bitsz_t o = 0, sz = 0, sz2 = 0;

        *keys = 0;
        if( self->col_rdf != NULL && self->col_nrd != NULL ) {
            if( (rc = SRAColumnRead(self->col_rdf, spot, (const void **)&rdf, &o, &sz)) == 0 &&
                (rc = SRAColumnRead(self->col_nrd, spot, (const void **)&nreads, &o, &sz2)) == 0 ) {
                sz /= 8;
                if( sz != sizeof(*rdf) * (*nreads) ) {
                    rc = RC(rcSRA, rcNode, rcExecuting, rcData, rcCorrupt);
                } else {
                    uint8_t i = *nreads;
                    *keys = *nreads;
                    while( i > 0 ) {
                        i--;
                        if( self->read_filter != 0xFF && self->read_filter != rdf[i] ) {
                            /* skip */
                            key[i].key = NULL;
                        } else if( rdf[i] == SRA_READ_FILTER_PASS ) {
                            key[i].key = "pass";
                        } else if( rdf[i] == SRA_READ_FILTER_REJECT ) {
                            key[i].key = "reject";
                        } else if( rdf[i] == SRA_READ_FILTER_CRITERIA ) {
                            key[i].key = "criteria";
                        } else if( rdf[i] == SRA_READ_FILTER_REDACTED ) {
                            key[i].key = "redacted";
                        } else {
                            rc = RC(rcSRA, rcNode, rcExecuting, rcData, rcUnknown);
                        }
                        key[i].readmask = (1 << i);
                    }
                }
            }
        }
    }
    return rc;
}

typedef struct ReadFilterSplitterFactory_struct {
    const SRATable* table;
    const SRAColumn* col_rdf;
    const SRAColumn* col_nrd;
    SRAReadFilter read_filter;
} ReadFilterSplitterFactory;

static
rc_t ReadFilterSplitterFactory_Init(const SRASplitterFactory* cself)
{
    rc_t rc = 0;
    ReadFilterSplitterFactory* self = (ReadFilterSplitterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcSRA, rcType, rcConstructing, rcParam, rcNull);
    } else {
        if( (rc = SRATableOpenColumnRead(self->table, &self->col_rdf, "READ_FILTER", sra_read_filter_t)) != 0 ) {
            if( GetRCState(rc) == rcNotFound ) {
                LOGMSG(klogWarn, "Column READ_FILTER was not found, param ignored");
                rc = 0;
            } else if( GetRCState(rc) == rcExists ) {
                rc = 0;
            }
        }
        if( rc == 0 && (rc = SRATableOpenColumnRead(self->table, &self->col_nrd, "NREADS", vdb_uint8_t)) != 0 ) {
            if( GetRCState(rc) == rcNotFound ) {
                LOGMSG(klogWarn, "Column NREADS was not found, param ignored");
                rc = 0;
            } else if( GetRCState(rc) == rcExists ) {
                rc = 0;
            }
        }
    }
    return rc;
}

static
rc_t ReadFilterSplitterFactory_NewObj(const SRASplitterFactory* cself, const SRASplitter** splitter)
{
    rc_t rc = 0;
    ReadFilterSplitterFactory* self = (ReadFilterSplitterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcSRA, rcType, rcExecuting, rcParam, rcNull);
    } else {
        if( (rc = SRASplitter_Make(splitter, sizeof(ReadFilterSplitter), NULL, ReadFilterSplitter_GetKeySet, NULL, NULL)) == 0 ) {
            ((ReadFilterSplitter*)(*splitter))->col_rdf = self->col_rdf;
            ((ReadFilterSplitter*)(*splitter))->col_nrd = self->col_nrd;
            ((ReadFilterSplitter*)(*splitter))->read_filter = self->read_filter;
        }
    }
    return rc;
}

static
void ReadFilterSplitterFactory_Release(const SRASplitterFactory* cself)
{
    if( cself != NULL ) {
        ReadFilterSplitterFactory* self = (ReadFilterSplitterFactory*)cself;
        SRAColumnRelease(self->col_rdf);
        SRAColumnRelease(self->col_nrd);
    }
}

static
rc_t ReadFilterSplitterFactory_Make(const SRASplitterFactory** cself, const SRATable* table, SRAReadFilter read_filter)
{
    rc_t rc = 0;
    ReadFilterSplitterFactory* obj = NULL;

    if( cself == NULL || table == NULL ) {
        rc = RC(rcSRA, rcType, rcAllocating, rcParam, rcNull);
    } else if( (rc = SRASplitterFactory_Make(cself, eSplitterRead, sizeof(*obj),
                                             ReadFilterSplitterFactory_Init,
                                             ReadFilterSplitterFactory_NewObj,
                                             ReadFilterSplitterFactory_Release)) == 0 ) {
        obj = (ReadFilterSplitterFactory*)*cself;
        obj->table = table;
        obj->read_filter = read_filter;
    }
    return rc;
}

/* ### SPOT_GROUP splitter/filter ##################################################### */

typedef struct SpotGroupSplitter_struct {
    char cur_key[256];
    const SRAColumn* col;
    bool split;
    char* const* spot_group;
} SpotGroupSplitter;

static
rc_t SpotGroupSplitter_GetKey(const SRASplitter* cself, const char** key, spotid_t spot, uint32_t* readmask)
{
    rc_t rc = 0;
    SpotGroupSplitter* self = (SpotGroupSplitter*)cself;

    if( self == NULL || key == NULL ) {
        rc = RC(rcSRA, rcNode, rcExecuting, rcParam, rcNull);
    } else {
        const char* g = NULL;
        bitsz_t o = 0, sz = 0;

        *key = self->cur_key;
        if( self->col != NULL ) {
            if( (rc = SRAColumnRead(self->col, spot, (const void **)&g, &o, &sz)) == 0 ) {
                sz /= 8;
                /* truncate trailing \0 */
                while( sz > 0 && g[sz - 1] == '\0' ) {
                    sz--;
                }
                if( sz > sizeof(self->cur_key) - 1 ) {
                    rc = RC(rcSRA, rcNode, rcExecuting, rcBuffer, rcInsufficient);
                } else {
                    int i;
                    bool found = false;
                    memcpy(self->cur_key, g, sz);
                    self->cur_key[sz] = '\0';
                    for(i = 0; self->spot_group[i] != NULL; i++) {
                        if( strcmp(self->cur_key, self->spot_group[i]) == 0 ) {
                            found = true;
                            break;
                        }
                    }
                    if( self->spot_group[0] != NULL && !found ) {
                        /* list not empty and not in list -> skip */
                        *key = NULL;
                    } else if( !self->split ) {
                        *key = "";
                    }
                }
            }
        }
    }
    return rc;
}

typedef struct SpotGroupSplitterFactory_struct {
    const SRATable* table;
    const SRAColumn* col;
    bool split;
    char* const* spot_group;
} SpotGroupSplitterFactory;

static
rc_t SpotGroupSplitterFactory_Init(const SRASplitterFactory* cself)
{
    rc_t rc = 0;
    SpotGroupSplitterFactory* self = (SpotGroupSplitterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcSRA, rcType, rcConstructing, rcParam, rcNull);
    } else {
        if( (rc = SRATableOpenColumnRead(self->table, &self->col, "SPOT_GROUP", vdb_ascii_t)) != 0 ) {
            if( GetRCState(rc) == rcNotFound ) {
                LOGMSG(klogWarn, "Column SPOT_GROUP was not found, param ignored");
                rc = 0;
            } else if( GetRCState(rc) == rcExists ) {
                rc = 0;
            }
        }
    }
    return rc;
}

static
rc_t SpotGroupSplitterFactory_NewObj(const SRASplitterFactory* cself, const SRASplitter** splitter)
{
    rc_t rc = 0;
    SpotGroupSplitterFactory* self = (SpotGroupSplitterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcSRA, rcType, rcExecuting, rcParam, rcNull);
    } else {
        if( (rc = SRASplitter_Make(splitter, sizeof(SpotGroupSplitter), SpotGroupSplitter_GetKey, NULL, NULL, NULL)) == 0 ) {
            ((SpotGroupSplitter*)(*splitter))->col = self->col;
            ((SpotGroupSplitter*)(*splitter))->split = self->split;
            ((SpotGroupSplitter*)(*splitter))->spot_group = self->spot_group;
        }
    }
    return rc;
}

static
void SpotGroupSplitterFactory_Release(const SRASplitterFactory* cself)
{
    if( cself != NULL ) {
        SpotGroupSplitterFactory* self = (SpotGroupSplitterFactory*)cself;
        SRAColumnRelease(self->col);
    }
}

static
rc_t SpotGroupSplitterFactory_Make(const SRASplitterFactory** cself, const SRATable* table, bool split, char* const spot_group[])
{
    rc_t rc = 0;
    SpotGroupSplitterFactory* obj = NULL;

    if( cself == NULL || table == NULL ) {
        rc = RC(rcSRA, rcType, rcAllocating, rcParam, rcNull);
    } else if( (rc = SRASplitterFactory_Make(cself, eSplitterSpot, sizeof(*obj),
                                             SpotGroupSplitterFactory_Init,
                                             SpotGroupSplitterFactory_NewObj,
                                             SpotGroupSplitterFactory_Release)) == 0 ) {
        obj = (SpotGroupSplitterFactory*)*cself;
        obj->table = table;
        obj->split = split;
        obj->spot_group = spot_group;
    }
    return rc;
}

/* ### Common dumper code ##################################################### */

static
rc_t SRADumper_DumpRun(const SRATable* table, spotid_t minSpotId, spotid_t maxSpotId, const SRASplitterFactory* factories)
{
    rc_t rc = 0, rcr = 0;
    spotid_t spot = 0;
    uint32_t readmask;
    const SRASplitter* root_splitter = NULL;

    rc = SRASplitterFactory_NewObj(factories, &root_splitter);

    for(spot = minSpotId; rc == 0 && spot <= maxSpotId; spot++ ) {
        readmask = ~0;
        rc = SRASplitter_AddSpot(root_splitter, spot, &readmask);
        rc = rc ? rc : Quitting();
    }
    rcr = SRASplitter_Release(root_splitter);

    return rc ? rc : rcr;
}

typedef struct UsageData_struct {
    const char* prog;
    const SRADumperFmt_Arg* args[2];
} UsageData;

static const SRADumperFmt_Arg KMainArgs[] =
    {
        {"A", "accession", "Accession", NULL},
        {"O", "outdir", "Output directory", "'.'"},
        {"N", "minSpotId", "Minimum spot id to output", NULL},
        {"X", "maxSpotId", "Maximum spot id to output", NULL},
        {"G", "spot-group", "Split into files by SPOT_GROUP (member name)", NULL},
        {"GL", "spot-group-list [<list>]", "Filter by SPOT_GROUP (member): name[,...]", NULL},
        {"R", "read-filter [<filter>]", "Split into files by READ_FILTER value,\n"
                                        "\t\t\t\t\toptional filter by a value: pass|reject|criteria|redacted", NULL},
        {"T", "group-in-dirs", "Split into subdirectories instead of files", NULL},
        {"K", "keep-empty-files", "Do not delete empty files", NULL},
        {"L", "log-level <level>", "Logging level: fatal|sys|int|err|warn|info", "err"},
#if _DEBUGGING
        {"+", "debug <flag>", "debug: APP|KFS|XML|VDB|SRA|REF|LEGREF", NULL},
#endif
        {"H", "help", "Prints this message", NULL},
        {NULL, NULL, NULL, NULL} /* terminator */
    };

static UsageData usage_data;

rc_t CC Usage ( const Args * args )
{
    const char * progname;
    const char * fullname;
    rc_t rc;
    ver_t version;

    fullname = "core-dumper";
    progname = fullname;
    version = KAppVersion();
    rc = ArgsProgram (args, &fullname, &progname);

    OUTMSG (( "\nUsage:\n" ));
    return 0;
}

static
const char* CoreUsage( void * data, int full )
{
    int k, i;
    UsageData* d = (UsageData*)data;

    printf( "\n"
            "Usage:\n"
            "  %s [options] [ -A ] <accession>\n"
            "  %s [options] <path>\n"
            "\n", d -> prog, d -> prog);

    if ( full )
    {
        for( k = 0; k < 2; k++) {
            for( i = 0; d->args[k] != NULL && d->args[k][i].abbr != NULL; i++) {
                if( k > 0 && i == 0 ) {
                    printf("\nFormat options:\n\n");
                }
                printf("  -%s", d->args[k][i].abbr);
                if( d->args[k][i].full != NULL ) {
                    printf(", --%-30s", d->args[k][i].full);
                } else {
                    printf("%-*s", (int)(35 - strlen(d->args[k][i].abbr)), "");
                }
                if( d->args[k][i].descr != NULL ) {
                    printf(" %s%c", d->args[k][i].descr, d->args[k][i].descr[strlen(d->args[k][i].descr) - 1] == '\n' ? '\0' : '.');
                }
                if( d->args[k][i].dflt != NULL ) {
                    printf(" Default: %s", d->args[k][i].dflt);
                }
                printf("\n");
                if( k == 0 && i == 0 ) {
                    printf("\nOptions:\n\n");
                }
            }
        }
    }
    else
    {
        printf( "Use option --help for more information\n" );
    }

    printf("\nVersion: %u.%u.%u\n\n", KAppVersion() >> 24, (KAppVersion() >> 16) & 0x00FF, KAppVersion() & 0x0000FFFF );
    exit(rcArgv);
    return NULL;
}

static
rc_t SRADumper_ArgsValidate(const char* prog, const SRADumperFmt_Arg* args)
{
    rc_t rc = 0;
    int k, i;

    usage_data.prog = prog;
    usage_data.args[0]= KMainArgs;
    usage_data.args[1]= args;

    for( i = 0; KMainArgs[i].abbr != NULL; i++) {
        if( strcmp(KMainArgs[i].abbr, "L") == 0  && KMainArgs[i].dflt != NULL ) {
            /* set default log level */
            if( (rc = LogLevelSet(KMainArgs[i].dflt)) != 0 ) {
                PLOGERR(klogErr, (klogErr, rc, "default log level $(lvl)", PLOG_S(lvl), KMainArgs[i].dflt));
                CoreUsage(&usage_data, 0);
            }
#if _DEBUGGING
        } else if( strcmp(KMainArgs[i].abbr, "+") == 0 && KMainArgs[i].dflt != NULL) {
            /* set default log level */
            if( (rc = KDbgSetString(KMainArgs[i].dflt)) != 0 ) {
                PLOGERR(klogErr, (klogErr, rc, "default debug level $(lvl)", PLOG_S(lvl), KMainArgs[i].dflt));
                CoreUsage(&usage_data, 0);
            }
#endif
        }
        for( k = 0; args != NULL && args[k].abbr != NULL; k++) {
            if( strcmp(args[k].abbr, KMainArgs[i].abbr) == 0 ||
                (args[k].full != NULL && strcmp(args[k].full, KMainArgs[i].full) == 0) ) {
                rc = RC(rcExe, rcArgv, rcValidating, rcParam, rcDuplicate);
            }
        }
    }
    return rc;
}

bool SRADumper_GetArg(char const* const abbr, char const* const full, int* i, int argc, char *argv[], const char** value)
{
    rc_t rc = 0;
    const char* arg = argv[*i];
    while(*arg == '-' && *arg != '\0') {
        arg++;
    }
    if( abbr != NULL && strcmp(arg, abbr) == 0 ) {
/*        SRA_DUMP_DBG(9, ("GetArg key: '%s'\n", arg)); */
        arg = arg + strlen(abbr);
        if( value != NULL && arg[0] == '\0' && (*i + 1) < argc ) {
            if( argv[*i + 1][0] != '-' ) {
                /* advance only if next is not an option with '-' */
                *i = *i + 1;
                arg = argv[*i];
            }
        } else {
            arg = NULL;
        }
    } else if( full != NULL && strcmp(arg, full) == 0 ) {
/*        SRA_DUMP_DBG(9, ("GetArg key: '%s'\n", arg)); */
        if( value != NULL && (*i + 1) < argc ) {
            if( argv[*i + 1][0] != '-' ) {
                /* advance only if next is not an option with '-' */
                *i = *i + 1;
                arg = argv[*i];
            }
        } else {
            arg = NULL;
        }
    } else {
        return false;
    }

/*    SRA_DUMP_DBG(9, ("GetArg val: '%s'\n", arg)); */
    if( value == NULL && arg != NULL ) {
        rc = RC(rcApp, rcArgv, rcAccessing, rcParam, rcUnexpected);
    } else if( value != NULL ) {
        if( arg == NULL && *value == '\0' ) {
            rc = RC(rcApp, rcArgv, rcAccessing, rcParam, rcNotFound);
        } else if( arg != NULL && arg[0] != '\0' ) {
            *value = arg;
        }
    }
    if( rc != 0 ) {
        PLOGERR(klogErr, (klogErr, rc, "$(a), $(f): $(v)",
            PLOG_3(PLOG_S(a),PLOG_S(f),PLOG_S(v)), abbr, full, arg));
        CoreUsage(&usage_data, 0);
    }
    return rc == 0 ? true : false;
}

/*******************************************************************************
 * KMain - defined for use with kapp library
 *******************************************************************************/
rc_t CC KMain ( int argc, char* argv[] )
{
    rc_t rc = 0;
    int i = 0;
    const char* arg, *option;
    const SRAMgr* sraMGR = NULL;
    SRADumperFmt fmt;
    char const* outdir = NULL;
    spotid_t minSpotId = 1;
    spotid_t maxSpotId = ~0;
    bool sub_dir = false;
    bool keep_empty = false;
    const char *table_path = NULL;
    char const *P_option = NULL;
    char const *D_option = NULL;
    char buffer[4096];
    const SRASplitterFactory* fact_head = NULL;
    
    bool spot_group_on = false;
    int spot_groups = 0;
    char* spot_group[128] = {NULL};
    bool read_filter_on = false;
    SRAReadFilter read_filter = 0xFF;

    memset(&fmt, 0, sizeof(fmt));
    if( (rc = SRADumper_Init(&fmt)) != 0 ) {
        LOGERR(klogErr, rc, "formatter initialization");
        return 100;
    } else if( fmt.get_factory == NULL ) {
        rc = RC(rcExe, rcFormatter, rcValidating, rcInterface, rcNull);
        LOGERR(klogErr, rc, "formatter factory");
        return 101;
    } else if( (rc = SRADumper_ArgsValidate(argv[0], fmt.arg_desc)) != 0 ) {
        LOGERR(klogErr, rc, "formatter args list");
        return 102;
    }
    if ( argc < 2 )
    {
        CoreUsage(&usage_data, 0);
        return 0;
    }

    for(i = 1; i < argc; i++)
    {
        arg = argv[i];
        if( arg[0] != '-' ) {
            if( table_path != NULL && table_path != fmt.accession ) {
                rc = RC ( rcExe, rcArgv, rcReading, rcParam, rcUnrecognized );
                PLOGERR(klogErr, (klogErr, rc, "unrecognized param: '$(p)'", PLOG_S(p), argv[i]));
                goto Catch;
            }
            table_path = arg;
            continue;
        }

        arg = NULL;
        option = "-";
        if( SRADumper_GetArg("L", "log-level", &i, argc, argv, &arg) ) {
            if( (rc = LogLevelSet(arg)) != 0 ) {
                PLOGERR(klogErr, (klogErr, rc, "log level $(lvl)", PLOG_S(lvl), arg));
                goto Catch;
            }
        } else if( SRADumper_GetArg("+", "debug", &i, argc, argv, &arg) ) {
#if _DEBUGGING
            if( (rc = KDbgSetString(arg)) != 0 ) {
                PLOGERR(klogErr, (klogErr, rc, "debug level $(lvl)", PLOG_S(lvl), arg));
                goto Catch;
            }
#endif
        } else if(SRADumper_GetArg("H", "help", &i, argc, argv, NULL) ||
                  SRADumper_GetArg("?", "h", &i, argc, argv, NULL) ) {
            CoreUsage(&usage_data, 1);

        } else if( SRADumper_GetArg("D", "table-path", &i, argc, argv, &D_option) ) {
            LOGMSG(klogErr, "Key -D is deprecated, see --help");
        } else if( SRADumper_GetArg("P", "path", &i, argc, argv, &P_option) ) {
            LOGMSG(klogErr, "Key -P is deprecated, see --help");

        } else if( SRADumper_GetArg("A", "accession", &i, argc, argv, (const char**)&fmt.accession) ) {
            if( table_path == NULL ) {
                table_path = fmt.accession;
            }
        } else if( SRADumper_GetArg("O", "outdir", &i, argc, argv, &outdir) ) {
        } else if( SRADumper_GetArg("N", "minSpotId", &i, argc, argv, &arg) ) {
            minSpotId = AsciiToU32(arg, NULL, NULL);
        } else if( SRADumper_GetArg("X", "maxSpotId", &i, argc, argv, &arg) ) {
            maxSpotId = AsciiToU32(arg, NULL, NULL);
        } else if( SRADumper_GetArg("G", "spot-group", &i, argc, argv, NULL) ) {
            spot_group_on = true;
        } else if( SRADumper_GetArg("GL", "spot-group-list", &i, argc, argv, NULL) ) {
            if( i + 1 < argc && argv[i + 1][0] != '-' ) {
                int f = 0, t = 0;
                i++;
                while( argv[i][t] != '\0' ) {
                    if( argv[i][t] == ',' ) {
                        if( t - f > 0 ) {
                            spot_group[spot_groups++] = strndup(&argv[i][f], t - f);
                        }
                        f = t + 1;
                    }
                    t++;
                }
                if( t - f > 0 ) {
                    spot_group[spot_groups++] = strndup(&argv[i][f], t - f);
                }
                spot_group[spot_groups] = NULL;
            }
        } else if( SRADumper_GetArg("R", "read-filter", &i, argc, argv, NULL) ) {
            read_filter_on = true;
            if( i + 1 < argc && argv[i + 1][0] != '-' ) {
                i++;
                if( read_filter != 0xFF ) {
                    rc = RC(rcApp, rcArgv, rcReading, rcParam, rcDuplicate);
                    PLOGERR(klogErr, (klogErr, rc, "$(p): $(o)", PLOG_2(PLOG_S(p),PLOG_S(o)), argv[i - 1], argv[i]));
                    CoreUsage(&usage_data, 1);
                }
                if( strcasecmp(argv[i], "pass") == 0 ) {
                    read_filter = SRA_READ_FILTER_PASS;
                } else if( strcasecmp(argv[i], "reject") == 0 ) {
                    read_filter = SRA_READ_FILTER_REJECT;
                } else if( strcasecmp(argv[i], "criteria") == 0 ) {
                    read_filter = SRA_READ_FILTER_CRITERIA;
                } else if( strcasecmp(argv[i], "redacted") == 0 ) {
                    read_filter = SRA_READ_FILTER_REDACTED;
                } else {
                    /* must be accession */
                    i--;
                }
            }
        } else if( SRADumper_GetArg("T", "group-in-dirs", &i, argc, argv, NULL) ) {
            sub_dir = true;
        } else if( SRADumper_GetArg("K", "keep-empty-files", &i, argc, argv, NULL) ) {
            keep_empty = true;
        } else if( fmt.add_arg && fmt.add_arg(&fmt, SRADumper_GetArg, &i, argc, argv) ) {
        } else {
            rc = RC(rcApp, rcArgv, rcReading, rcParam, rcIncorrect);
            PLOGERR(klogErr, (klogErr, rc, "$(p)", PLOG_S(p), argv[i]));
            CoreUsage(&usage_data, 1);
        }
    }
    if( table_path == NULL ) {
        if( D_option != NULL ) {
            /* support deprecated '-D' option */
            table_path = D_option;
        } else if( fmt.accession == NULL || fmt.accession[0] == '\0' ) {
            /* must have accession to proceed */
            rc = RC(rcExe, rcArgv, rcValidating, rcParam, rcEmpty);
            LOGERR(klogErr, rc, "Expected accession");
            goto Catch;
        } else if( P_option != NULL && P_option[0] != '\0' ) {
            /* support deprecated '-P' option */
            i = snprintf(buffer, sizeof(buffer), "%s/%s", P_option, fmt.accession);
            if( i < 0 || i >= sizeof(buffer) ) {
                rc = RC(rcExe, rcArgv, rcValidating, rcParam, rcExcessive);
                LOGERR(klogErr, rc, "Path too long");
                goto Catch;
            }
            table_path = buffer;
        }
    }
    SRA_DUMP_DBG(5, ("table path '%s'\n", table_path));
    if( (rc = SRAMgrMakeRead(&sraMGR)) != 0 ) {
        LOGERR(klogInt, rc, "failed to open SRA manager");
        goto Catch;
    }
    if( (rc = SRAMgrOpenTableRead(sraMGR, &fmt.table, table_path)) != 0 ) {
        PLOGERR(klogErr, (klogErr, rc, "failed to open '$(path)'", "path=%s", table_path));
        goto Catch;
    }
    /* infer accession from table_path if missing */
    if( fmt.accession == NULL || fmt.accession[0] == 0 ) {
        char *ext;
        size_t l;

        strcpy(buffer, table_path);
        l = strlen(buffer);
        while( strchr("\\/", buffer[l - 1]) != NULL ) {
            buffer[--l] = '\0';
        }
        fmt.accession = strrchr(buffer, '/');
        if( fmt.accession++ == NULL ) {
            fmt.accession = buffer;
        }
        /* cut of [.lite].sra if any */
        ext = strrchr(fmt.accession, '.');
        if( ext != NULL && strcasecmp(ext, ".sra") == 0 ) {
            *ext = '\0';
            ext = strrchr(fmt.accession, '.');
            if( ext != NULL && strcasecmp(ext, ".lite") == 0 ) {
                *ext = '\0';
            }
        }
    }
    SRA_DUMP_DBG(5, ("accession: '%s'\n", fmt.accession));

    while( rc == 0 ) {
        spotid_t smax = 0, smin = 0;

        if( (rc = fmt.get_factory(&fmt, &fact_head)) != 0 ) {
            break;
        }
        if( fact_head == NULL ) {
            rc = RC(rcExe, rcFormatter, rcResolving, rcInterface, rcNull);
            break;
        }
        if( (rc = SRATableMaxSpotId(fmt.table, &smax)) != 0 ||
            (rc = SRATableMinSpotId(fmt.table, &smin)) != 0 ) {
            break;
        }
        
        /* test if we have to dump anything... */
        if ( smax < minSpotId )
        {
            rc = RC( rcExe, rcFormatter, rcValidating, rcRow, rcTooBig );
            break;
        }

        if( maxSpotId > smax ) {
            maxSpotId = smax;
        }
        if( minSpotId < smin ) {
            minSpotId = smin;
        }
        if( minSpotId > maxSpotId ) {
            spotid_t temp = maxSpotId;
            maxSpotId = minSpotId;
            minSpotId = temp;
        }

        if( spot_group_on || spot_groups > 0 ) {
            const SRASplitterFactory* f = NULL;
            if( (rc = SpotGroupSplitterFactory_Make(&f, fmt.table, spot_group_on, spot_group)) == 0 ) {
                if( (rc = SRASplitterFactory_AddNext(f, fact_head)) == 0 ) {
                    fact_head = f;
                } else {
                    SRASplitterFactory_Release(f);
                }
            }
        }
        if( read_filter_on ) {
            const SRASplitterFactory* f = NULL;
            if( (rc = ReadFilterSplitterFactory_Make(&f, fmt.table, read_filter)) == 0 ) {
                if( (rc = SRASplitterFactory_AddNext(f, fact_head)) == 0 ) {
                    fact_head = f;
                } else {
                    SRASplitterFactory_Release(f);
                }
            }
        }
        if( (rc = SRASplitterFactory_InitFiler(fmt.accession, sub_dir, keep_empty, outdir)) == 0 &&
            (rc = SRASplitterFactory_Init(fact_head)) == 0 ) { 
            rc = SRADumper_DumpRun(fmt.table, minSpotId, maxSpotId, fact_head);
            break;
        }
    }
    if( fmt.release ) {
        rc_t rr = fmt.release(&fmt);
        if( rr != 0 ) {
            SRA_DUMP_DBG(1, ("formatter release error %R\n", rr));
        }
    }
    for(i = 0; i < spot_groups; i++) {
        free(spot_group[i]);
    }
    if( rc == 0 ) {
        PLOGMSG(klogInfo, (klogInfo, "$(a) $(n) spots", PLOG_2(PLOG_S(a),PLOG_U32(n)), fmt.accession, maxSpotId - minSpotId + 1));
    } else {
        PLOGERR(klogErr, (klogErr, rc, "Failed $(a)", PLOG_S(a), fmt.accession));
    }
Catch:
    SRASplitterFactory_Release(fact_head);
    SRATableRelease(fmt.table);
    SRAMgrRelease(sraMGR);
    return rc;
}
