/*===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 */

#include <kapp/main.h>
#include <klib/log.h>
#include <klib/rc.h>
#include <klib/namelist.h>
#include <klib/container.h>
#include <klib/debug.h>
#include <kdb/manager.h>
#include <kdb/table.h>
#include <kdb/namelist.h>
#include <kdb/consistency-check.h>
#include <vdb/manager.h>
#include <vdb/table.h>
#include <kfs/kfs-priv.h>
#include <sra/srapath.h>
#include <sysalloc.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sra-dbcc.vers.h"

rc_t CC Usage ( struct Args const * args ) { return 0; }

static rc_t report_index(const CCReportInfoBlock *what, void *ctx)
{
    rc_t *rc2 = ctx;
    
    switch (what->type) {
    case ccrpt_Done:
        if (what->info.done.rc) {
            (void)PLOGERR(klogErr, (klogErr, what->info.done.rc, "Index '$(index)': $(mesg)", "index=%s,mesg=%s", what->objName, what->info.done.mesg));
            if (*rc2 == 0)
                *rc2 = what->info.done.rc;
        }
        return 0; /* continue with check */
    case ccrpt_Index:
        return 0; /* continue with check */
    case ccrpt_MD5:
        if (what->info.MD5.rc) {
            (void)PLOGERR(klogErr, (klogErr, what->info.MD5.rc, "File '$(file)' of index '$(index)' failed MD5 check", "file=%s,index=%s", what->info.MD5.file, what->objName));
            if (*rc2 == 0)
                *rc2 = what->info.MD5.rc;
        }
        return 0; /* continue with check */
    default:
        return RC(rcExe, rcTable, rcVisiting, rcParam, rcUnexpected);
    }
}

static rc_t report_column(const CCReportInfoBlock *what, void *ctx)
{
    rc_t *rc2 = ctx;
    
    switch (what->type) {
    case ccrpt_Done:
        if (what->info.done.rc) {
            (void)PLOGERR(klogErr, (klogErr, what->info.done.rc, "Column '$(column)': $(mesg)", "column=%s,mesg=%s", what->objName, what->info.done.mesg));
            if (*rc2 == 0)
                *rc2 = what->info.done.rc;
        }
        return 0; /* continue with check */
    case ccrpt_Blob:
        return 0; /* continue with check */
    case ccrpt_MD5:
        if (what->info.MD5.rc) {
            (void)PLOGERR(klogErr, (klogErr, what->info.MD5.rc, "File '$(file)' of column '$(column)' failed MD5 check", "file=%s,column=%s", what->info.MD5.file, what->objName));
            if (*rc2 == 0)
                *rc2 = what->info.MD5.rc;
        }
        return 0; /* continue with check */
    default:
        return RC(rcExe, rcTable, rcVisiting, rcParam, rcUnexpected);
    }
}

static rc_t report_table(const CCReportInfoBlock *what, void *ctx)
{
    rc_t *rc2 = ctx;
    
    switch (what->type) {
    case ccrpt_Done:
        if (what->info.done.rc) {
            (void)PLOGERR(klogErr, (klogErr, what->info.done.rc, "Table '$(table)': $(mesg)", "table=%s,mesg=%s", what->objName, what->info.done.mesg));
            if (*rc2 == 0)
                *rc2 = what->info.done.rc;
        }
        return 0; /* continue with check */
    case ccrpt_MD5:
        if (what->info.MD5.rc) {
            (void)PLOGERR(klogErr, (klogErr, what->info.MD5.rc, "File '$(file)' of table '$(table)' failed MD5 check", "file=%s,table=%s", what->info.MD5.file, what->objName));
            if (*rc2 == 0)
                *rc2 = what->info.MD5.rc;
        }
        return 0; /* continue with check */
    default:
        return RC(rcExe, rcTable, rcVisiting, rcParam, rcUnexpected);
    }
}

static rc_t CC report(const CCReportInfoBlock *what, void *ctx)
{
    rc_t rc = Quitting();
    
    if (rc)
        return rc;
    switch (what->objType) {
    case kptTable:
        return report_table(what, ctx);
    case kptColumn:
        return report_column(what, ctx);
    case kptIndex:
        return report_index(what, ctx);
    default:
        return RC(rcExe, rcTable, rcVisiting, rcParam, rcUnexpected);
    }
}

static
rc_t kdbcc(const KDirectory *dir, const char name[], int mode)
{
    rc_t rc2 = 0;
    const KDBManager *mgr;
    rc_t rc = KDBManagerMakeRead(&mgr, dir);
    if (rc == 0) {
        const KTable *tbl;
        
        rc = KDBManagerOpenTableRead(mgr, &tbl, name);
        if (rc == 0)
            rc = KTableConsistencyCheck(tbl, (mode & 4) ? 3 : (mode & 2) ? 1 : 0, report, &rc2);
        KDBManagerRelease(mgr);
    }
    return rc ? rc : rc2;
}

static
rc_t vdbcc(const KDirectory *dir, const char name[], int mode)
{
#if 0
    if (mode & 8) {
        const VDBManager *mgr;
        rc_t rc = VDBManagerMakeRead(&mgr, dir);
        
        if (rc == 0) {
            const VTable *tbl;
            
            rc = VDBManagerOpenTableRead(mgr, &tbl, NULL, name);
            if (rc == 0)
                rc = VTableConsistencyCheck(tbl, 2);
        }
        return rc;
    }
#endif
    return 0;
}

static
rc_t dbcc(const KDirectory *dir, const char name[], int mode)
{
    rc_t rc = kdbcc(dir, name, mode);
    
    if (rc)
        return rc;
    return vdbcc(dir, name, mode);
}

static char const* const defaultLogLevel = 
#if _DEBUGGING
"debug5";
#else
"info";
#endif

/*******************************************************************************
 * Usage
 *******************************************************************************/
static void usage(const char *progName)
{
    const char* p = strrchr(progName, '/');
    p = p ? p + 1 : progName;
    printf("\nUsage: %s [options] table\n\n", p);
    printf(
           "    -5|--md5        Check components md5s if present, fail otherwise,\n"
           "                    unless other checks are requested (default: yes)\n"
           "    -b|--blob-crc   Check blobs CRC32 (default: no)\n"
#if 0
           "    -i|--index      Check 'skey' index (default: no)\n"
#endif
           "\n"
           "    -h|--help       This help\n"
           "    -V|--version    Program version\n"
           "\n");
}

uint32_t CC KAppVersion(void)
{
    return SRA_DBCC_VERS;
}

rc_t CC KMain ( int argc, char *argv [] )
{
    rc_t rc = 0;
    int i;
    const char *arg;
    char *src_path = NULL;
    char *tbl_name = NULL;
    bool md5_chk = true, md5_chk_explicit = false;
    bool blob_crc = false;
    bool index_chk = false;
    const KDirectory *src_dir = NULL;

    char path_buffer [ 4096 ];

    KLogLevelSet(klogInfo);
    
    if (argc < 2) {
        usage(argv[0]);
        rc = RC(rcExe, rcArgv, rcReading, rcParam, rcUnknown);
    }
    else {
        for (i = 1; rc == 0 && i < argc; i++)
        {
            if (argv[i][0] != '-' && src_path == NULL)
            {
                KDirectory *dir;
                rc = KDirectoryNativeDir(&dir);
                if (rc == 0)
                {
                    src_path = argv[i];
                    tbl_name = strrchr(src_path, '/');
                    if (tbl_name != NULL)
                    {
                        if (tbl_name == src_path)
                            src_path = "/";
                        *tbl_name++ = '\0';
                    }
                    else
                    {
                        SRAPath *pmgr;

                        /* check for accession */
                        rc = SRAPathMake ( & pmgr, dir );
                        if ( rc == 0 )
                        {
                            rc = SRAPathFind ( pmgr, src_path, path_buffer, sizeof path_buffer );
                            if ( rc == 0 )
                            {
                                /* use mapped path */
                                src_path = path_buffer;
                                tbl_name = strrchr(src_path, '/');
                                if ( tbl_name == NULL )
                                    tbl_name = src_path;
                                else
                                {
                                    if ( tbl_name == src_path )
                                        src_path = "/";
                                    * tbl_name ++ = '\0';
                                }
                            }

                            (void)SRAPathRelease ( pmgr );
                        }

                        if ( rc != 0 )
                        {
                            /* appears to be a simple name */
                            tbl_name = src_path;
                            src_path = ".";
                        }
                    }

                    rc = KDirectoryOpenDirRead(dir, &src_dir, false, src_path);
                    KDirectoryRelease(dir);
                }
            }
            else if (strcmp(argv[i], "-5") == 0 || strcmp(argv[i], "--md5") == 0 ) {
                md5_chk_explicit = true;
            }
            else if (strcmp(argv[i], "-b") == 0 || strcmp(argv[i], "--blob-crc") == 0 ) {
                blob_crc = true;
            }
#if 1
            else if (strcmp(argv[i], "-i") == 0 || strcmp(argv[i], "--index") == 0 ) {
                index_chk = true;
            }
#endif
            else if (strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "--help") == 0 ) {
                usage(argv[0]);
                return 0;
            }
            else if (strcmp(argv[i], "-V") == 0 || strcmp(argv[i], "--version") == 0 ) {
                printf("Version: %u.%u.%u\n", KAppVersion() >> 24, (KAppVersion() >> 16) & 0xFF, KAppVersion() & 0xFFFF);
                return 0;
            }
            else {
                usage(argv[0]);
                rc = RC(rcExe, rcArgv, rcReading, rcParam, rcUnknown);
            }
        }
        if (rc == 0) {
            if (blob_crc || index_chk) {
                /* if blob or index is requested, md5 is off unless explicitly requested */
                md5_chk = md5_chk_explicit;
            }
            if (src_path == NULL)
                rc = RC(rcExe, rcArgv, rcReading, rcParam, rcNotFound);
            
            if (rc == 0)
                rc = dbcc(src_dir, tbl_name, (md5_chk ? 1 : 0)|(blob_crc ? 2 : 0)|(index_chk ? 4 : 0));
        }
    }
    KDirectoryRelease(src_dir);
    if (rc == 0) {
        (void)PLOGMSG(klogInfo, (klogInfo, "Table '$(table)' ok", PLOG_S(table), tbl_name));
    }
    else {
        (void)PLOGERR(klogErr, (klogErr, rc, "Table '$(table)' check failed", PLOG_S(table), tbl_name));
    }
    return rc;
}
