/*===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 */
#ifndef IRZIP_VERSION
#define IRZIP_VERSION 2
#endif

#define DELTA_UND  0
#if _ARCH_BITS == 32
#define DELTA_POS  0x7ffffffffffffff0LL
#define DELTA_NEG  0x7ffffffffffffff1LL
#define DELTA_BOTH 0x7ffffffffffffff2LL
#else
#define DELTA_POS  0x7ffffffffffffff0
#define DELTA_NEG  0x7ffffffffffffff1
#define DELTA_BOTH 0x7ffffffffffffff2
#endif

#if ENCODING
static rc_t ENCODE(uint8_t dst[], size_t dsize, size_t *used, int64_t *Min, int64_t *Slope, uint8_t *planes, const STYPE Y[], unsigned N)
{
    unsigned i;
    unsigned k;
    rc_t rc=0;
    STYPE min,a0;
    STYPE y_cur;
    int64_t 	slope, delta_type = DELTA_UND;
    uint64_t    sum_a0,sum_min,sum_i,sum_delta;
    uint8_t	*scratch=NULL;
    bool	no_delta_both=false;

    if( N < 10 ) /*** no reason to encode **/
		return RC(rcXF, rcFunction, rcExecuting, rcBuffer, rcInsufficient);

    scratch=malloc(N*sizeof(STYPE));
    if(scratch==NULL) return RC(rcXF, rcFunction, rcExecuting, rcBuffer, rcInsufficient);
   
    /*** the code tries to best-fit one of 3 cases ****/
    /*** 1. delta - for monotonous data **/
    /*** 2. horizontal line through the minimum ****/
    /*** 3. sloped line from the first element y[i] = y[0] + i * slope ***/
    /*** all algorithms keep delta positive ****/
    /*** 1. is preffered ***/
    /*** the best between 2 and 3 is choised by the sum of all deltas ****/
    /*** floating point calculations are avoided ***/
	a0 =  Y[0];
	y_cur = Y[1];
	slope = (int64_t)Y[1] - Y[0];
	if(slope < 0 ) {
		min = Y[1]; 
		sum_min = -slope;
		delta_type = DELTA_NEG;
	} else {
		min = Y[0];
		sum_min = slope;
		if( slope > 0 ) delta_type = DELTA_POS;
	}
	sum_a0 = 0;
	sum_delta = 0;
	sum_i = 1;
	for (i = 2; i < N; ++i) {
		int64_t  ad;
		int64_t  md = (int64_t)Y[i] - min;
		int64_t diff =  (int64_t)Y[i] - Y[i-1];
		int64_t abs_diff = labs(diff);

		sum_delta += abs_diff;
		if(abs_diff >= ((uint64_t)1)<<(sizeof(STYPE)*8-1)) no_delta_both=true;/* otherwise we loose a bit ***/
		if( delta_type != DELTA_BOTH){
			if ( diff > 0){
				if(delta_type == DELTA_NEG ) delta_type =DELTA_BOTH;
				else if (delta_type == DELTA_UND) delta_type = DELTA_POS;
			} else if ( diff < 0 ){
				if(delta_type == DELTA_POS ) delta_type =DELTA_BOTH;
				else if (delta_type == DELTA_UND) delta_type = DELTA_NEG;
				
			}
		}

		sum_i += i;
		if ( md > 0) {
			sum_min+= md;
		} else if ( md < 0 ) {
			min = Y[i];
			sum_min += -md * (i - 1 ); /** running sum difference increases by this amount ***/
		}
		y_cur += slope;
		ad = (int64_t)Y[i] - y_cur;
		if ( ad > 0 ) {
			sum_a0 += ad;
		} else if (ad < 0 ){ /** need to adjust the slope ***/
			int64_t sd = -ad / i;
			if (sd*i < -ad) sd++; /** to adjust for rounding above **/
			
			y_cur -= sd*i;
			slope -= sd;
			sum_a0 = sum_a0 + ad + sum_i*sd;

		}
	}

    /*** define which delta is the smallest **/
    /*** do DELTA_BOTH only for 64 bit ***/
    if( (delta_type == DELTA_POS || delta_type == DELTA_NEG || (delta_type == DELTA_BOTH &&  !no_delta_both))
           && sum_delta <= sum_min && sum_delta <= sum_a0){
	a0=Y[0];
	slope = delta_type;
    } else if(sum_a0 > sum_min){
	a0 = min;
	slope = 0;
    }

    *planes = 0;
    *used = 0;
    *Min= a0;
    *Slope = slope;

    /*** rotate the array ***/
    for( i = 0; i != N; ++i){
	STYPE val;
        if(slope == DELTA_POS){
		val= Y[i] - a0;
		a0 =  Y[i];
	} else if( slope == DELTA_NEG){
		val = a0 - Y[i];
		a0 =  Y[i];
	} else if( slope == DELTA_BOTH){
		if( Y[i] >= a0 ){ /** move sign bit into the lowest bit ***/
			val = ( Y[i] - a0 ) << 1;
		} else {
			val = ( a0 - Y[i]) * 2 + 1;
		}
		a0 =  Y[i];
	} else {
	   val = Y[i] - a0 - slope*i;
	}
	for(k=0;k<sizeof(STYPE);k++){
		if ((scratch[i+k*N] = (uint8_t)(val & 0xff)) != 0){
			*planes |= (1<<k);
		}
		val >>= 8;
	}
    }

    /*** record the arrays ***/
    for(k=0;k<sizeof(STYPE) && rc == 0; k++){
	if (*planes & (1<<k)) {
		szbuf s2;
		s2.used = 0;
		s2.size = dsize - *used;
		s2.buf  = dst + *used;
		rc = zlib_compress(&s2, scratch+k*N, N,Z_RLE, Z_BEST_SPEED);
		if ( rc == 0 ) {
			*used += s2.used;
			if (s2.used == 0) /*** skip zipping **/
				 rc=RC(rcXF, rcFunction, rcExecuting, rcBuffer, rcInsufficient);
		}
	}
    }
    if(scratch) free(scratch);
    return rc;
}

#endif

#if DECODING
static rc_t DECODE(STYPE Y[], unsigned N, int64_t min, int64_t slope, uint8_t planes, const uint8_t src[], size_t ssize)
{
    unsigned k;
    size_t j;
    unsigned m;
    unsigned i;
    uint8_t *scratch=NULL;
    rc_t rc=0;

#if _ARCH_BITS == 32
    uint64_t mask = (1ULL <<(sizeof(STYPE)*8-1))-1;
#else
    uint64_t mask = (1UL <<(sizeof(STYPE)*8-1))-1;
#endif
    
    memset(Y, 0, sizeof(Y[0]) * N);
    for (j = k = 0, m = 1; m < 0x100; m <<= 1, k += 8) {
        size_t n;
        
        if ((planes & m) == 0)
            continue;
        
        n = 0;
	if( !scratch ) scratch = malloc(N);
	
        rc = zlib_decompress(scratch, N, &n, src + j, ssize - j);
        if (rc) goto DONE;
        j += n;
        
        for (i = 0; i != N; ++i)
            Y[i] |= ((STYPE)scratch[i]) << k;
    }
    if(min != 0 || slope !=0 ) {
	for (i = 0; i != N; ++i){
		if ( slope == DELTA_POS ){
			min += Y[i];
			Y[i] = min;
		} else if (slope == DELTA_NEG ) {
			min -= Y[i];
			Y[i] = min;
		} else if (slope == DELTA_BOTH) {
			if(Y[i] & 1){
				min -= (Y[i]>>1) & mask;
			} else {
				min += (Y[i]>>1) & mask;
			}
			Y[i] = min;
		} else {
			Y[i] += min + slope*i;
		}
	}
    }
DONE:
    free(scratch);
    return rc;
}
#endif
