/*===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 * ===========================================================================
 *
 */

#include <sra/extern.h>
#include <sra/wsradb.h>
#include <sra/types.h>
#include <vdb/schema.h>
#include <vdb/table.h>
#include <vdb/cursor.h>
#include <vdb/vdb-priv.h>
#include <kdb/meta.h>
#include <klib/refcount.h>
#include <klib/log.h>
#include <klib/debug.h>
#include <klib/rc.h>
#include <sysalloc.h>

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "sra-debug.h"

#define KONST
#include "sra-priv.h"

#define RC_MODULE (rcSRA)
#define RC_TARGET (rcTable)
#define CLASS "SRATable"

/* Whack
 */

rc_t SRATableWhack ( SRATable *self )
{
    SRATableDestroy( self );
    return 0;
}

/* Create
 *  creates a new table
 *
 *  "tbl" [ OUT ] - return parameter for table
 *
 *  "path" [ IN ] - NUL terminated table name
 */
LIB_EXPORT rc_t CC SRAMgrCreateTable(SRAMgr *self, SRATable **tbl, const char *typespec, const char *path, ...) {
    va_list va;
    rc_t rc;
    
    va_start(va, path);
    rc = SRAMgrVCreateTable(self, tbl, typespec, path, va);
    va_end(va);
    return rc;
}

LIB_EXPORT rc_t CC SRAMgrVCreateTable(SRAMgr *mgr, SRATable **rslt, const char *typespec, const char *spec, va_list args) {
    SRATable *self;
    char path[4096];
    rc_t rc;
    
    if (mgr == NULL)
        return RC(RC_MODULE, RC_TARGET, rcConstructing, rcSelf, rcNull);
    if (spec == NULL || rslt == NULL)
        return RC(RC_MODULE, RC_TARGET, rcConstructing, rcParam, rcNull);
    
    *rslt = NULL;
    
    rc = ResolveTablePath(mgr, path, sizeof(path), spec, args);
    if (rc)
        return rc;
    
    self = calloc(1, sizeof(*self));
    if (self == NULL) {
        return RC(RC_MODULE, RC_TARGET, rcConstructing, rcMemory, rcExhausted);
    }
    
    if( (rc = VDBManagerCreateTable(mgr->vmgr, &self->vtbl, mgr->schema, typespec, (mgr->mode & kcmBitMask) | kcmCreate, path)) == 0 &&
        (rc = VTableSetUserData(self->vtbl, &self->stats, NULL)) == 0 &&
        (rc = VTableOpenMetadataUpdate(self->vtbl, &self->meta)) == 0 &&
        (rc = KMetadataVersion(self->meta, &self->metavers)) == 0 &&
        (rc = VTableCreateCursorWrite(self->vtbl, &self->curs, kcmInsert)) == 0 )
    {
        self->mgr = SRAMgrAttach(mgr);
        self->mode = mgr->mode;
        self->read_only = false;
        KRefcountInit(&self->refcount, 1, "SRATable", "OpenTableUpdate", path);
        VectorInit(&self->wcol, 0, 16);
        *rslt = self;
        return 0;
    }
    SRATableWhack(self);
    return rc;
}


/* OpenUpdate
 *  open an existing table
 *
 *  "run" [ OUT ] - return parameter for table
 *
 *  "path" [ IN ] - NUL terminated table name
 */
LIB_EXPORT rc_t CC SRAMgrOpenTableUpdate(SRAMgr *self, SRATable **tbl, const char *path, ...) {
    va_list va;
    rc_t rc;
    
    va_start(va, path);
    rc = SRAMgrVOpenTableUpdate(self, tbl, path, va);
    va_end(va);
    return rc;
}

LIB_EXPORT rc_t CC SRAMgrVOpenTableUpdate(SRAMgr *mgr, SRATable **rslt, const char *spec, va_list args) {
    SRATable *self;
    char path[4096];
    rc_t rc;
    
    if (mgr == NULL)
        return RC(RC_MODULE, RC_TARGET, rcConstructing, rcSelf, rcNull);
    if (spec == NULL || rslt == NULL)
        return RC(RC_MODULE, RC_TARGET, rcConstructing, rcParam, rcNull);
    
    *rslt = NULL;
    
    rc = ResolveTablePath(mgr, path, sizeof(path), spec, args);
    if (rc)
        return rc;
    
    self = calloc(1, sizeof(*self));
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcConstructing, rcMemory, rcExhausted);
    
    rc = VDBManagerOpenTableUpdate(mgr->vmgr, &self->vtbl, mgr->schema, path);
    if (rc == 0) {
        rc = VTableOpenMetadataUpdate(self->vtbl, &self->meta);
        if (rc == 0) {
            rc = KMetadataVersion(self->meta, &self->metavers);
            if (rc == 0) {
                rc = VTableCreateCursorWrite(self->vtbl, &self->curs, kcmInsert);
                if (rc == 0) {
                    self->mgr = SRAMgrAttach(mgr);
                    self->mode = mgr->mode;
                    self->read_only = false;
                    KRefcountInit(&self->refcount, 1, "SRATable", "OpenTableUpdate", path);
                    
                    SRATableLoadMetadata(self); /* fallible */
                    VectorInit(&self->wcol, 0, 16);
                    
                    *rslt = self;
                    return 0;
                }
            }
        }
    }
    SRATableWhack(self);
    return rc;
}

/* NewSpot
 *  creates a new spot record,
 *  returning spot id.
 *
 *  "id" [ OUT ] - return parameter for id of newly created spot
 */
LIB_EXPORT rc_t CC SRATableNewSpot( SRATable *self, spotid_t *id ) {
    rc_t rc;
    
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcInserting, rcSelf, rcNull);
    if (id == NULL)
        return RC(RC_MODULE, RC_TARGET, rcInserting, rcParam, rcNull);
    
    if (self->curs_open == false) {
#if _DEBUGGING
        fprintf(stderr, "opening cursor\n");
#endif
        rc = VCursorOpen(self->curs);
        if (rc)
            return rc;
        self->curs_open = true;
    }
    
    rc = VCursorOpenRow(self->curs);
    if (rc == 0) {
        int64_t rowid;
        if( (rc = VCursorRowId(self->curs, &rowid)) == 0 ) {
            *id = rowid;
        }
    }
    return rc;
}

/* OpenSpot
 *  opens an existing spot record from id
 *
 *  "id" [ IN ] - 1-based spot id
 */
LIB_EXPORT rc_t CC SRATableOpenSpot( SRATable *self, spotid_t id ) {
    rc_t rc;
    
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcOpening, rcSelf, rcNull);
#if 0
    /* TODO: translate spot id to row id */
    rc = VCursorSetRowId(self->curs, id);
    if (rc == 0) {
        rc = VCursorOpenRow(self->curs);
    }
#else
    rc = RC(RC_MODULE, RC_TARGET, rcOpening, rcFunction, rcUnsupported);
#endif
    return rc;
}


/* CloseSpot
 *  closes a spot opened with either NewSpot or OpenSpot
 */
LIB_EXPORT rc_t CC SRATableCloseSpot( SRATable *self ) {
    rc_t rc;
    
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcClosing, rcSelf, rcNull);
    
    rc = VCursorCommitRow(self->curs);
    if (rc == 0)
        return VCursorCloseRow(self->curs);
    VCursorCloseRow(self->curs);
    return rc;
}


/* Commit
 *  commit all changes
 */
LIB_EXPORT rc_t CC SRATableCommit( SRATable *self ) {
    rc_t rc;
    
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcCommitting, rcSelf, rcNull);
    rc = VCursorCommit(self->curs);
    VCursorRelease(self->curs);
    self->curs = NULL;
    if (rc == 0)
        rc = VTableReindex(self->vtbl);
    return rc;
}

static int CC cmp_index( const void *A, const void *B ) {
    return *(const uint32_t *)A - ((const SRAColumn *)B)->idx;
}

static bool find_by_index(const Vector *vec, uint32_t idx, uint32_t *cndx) {
    uint32_t found;
    
    if ( VectorFind(vec, &idx, &found, cmp_index ) != NULL ) {
        *cndx = found;
        return true;
    }
    return false;
}

/* OpenColumnWrite
 *  open a column for write
 *
 *  "idx" [ OUT ] - return parameter for 1-based column index.
 *
 *  "col" [ OUT, NULL OKAY ] - optional return parameter for
 *  newly opened column.
 *
 *  "name" [ IN ] - NUL terminated string in UTF-8 giving column name
 *
 *  "datatype" [ IN ] - NUL terminated string in ASCII
 *   describing fully qualified column data type
 */
LIB_EXPORT rc_t CC SRATableOpenColumnWrite ( SRATable *self,
        uint32_t *idx, SRAColumn **col, const char *name, const char *datatype) {
    SRAColumn *rslt;
    uint32_t ndx;
    uint32_t cndx;
    rc_t rc;
    
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcOpening, rcSelf, rcNull);
    
    if (name == NULL || idx == NULL)
        return RC(RC_MODULE, RC_TARGET, rcOpening, rcParam, rcNull);
    
    *idx = 0;
    
    if (datatype && datatype[0]) {
#if _DEBUGGING
        fprintf(stderr, "adding column (%s)%s\n", datatype, name);
#endif
        rc = VCursorAddColumn(self->curs, &ndx, "(%s)%s", datatype, name);
    }
    else {
#if _DEBUGGING
        fprintf(stderr, "adding column %s\n", name);
#endif
        rc = VCursorAddColumn(self->curs, &ndx, name);
    }
    
    if (rc != 0) {
        if (GetRCState ( rc ) != rcExists)
            return rc;
        
        find_by_index(&self->wcol, ndx, &cndx);
        rslt = VectorGet(&self->wcol, cndx);
    }
    else {
        VTypedecl type;
        VTypedesc desc;
        
        rslt = malloc(sizeof(*rslt));
        if (rslt == NULL)
            return RC(RC_MODULE, rcColumn, rcConstructing, rcMemory, rcExhausted);
        
        rc = VCursorDatatype(self->curs, ndx, &type, &desc);
        if (rc == 0) {
            KRefcountInit(&rslt->refcount, 1, "SRAColumn", "OpenColumnWrite", name);

            /* if the column remains fully contained in the array and has no
             * references passed out (and this is the only place that a writable
             * column could be passed out) then the Attach leads to a vicious
             * embrace with the table
             */
            rslt->tbl = col ? SRATableAttach(self) : NULL;
            rslt->idx = ndx;
            rslt->read_only = false;
            rslt->elem_bits = VTypedescSizeof(&desc);
            
            rc = VectorAppend(&self->wcol, &cndx, rslt);
        }
        if (rc != 0) {
            free(rslt);
            return rc;
        }
    }
    if (col) {
        *col = rslt;
        SRAColumnAddRef(rslt);
    }
    *idx = cndx + 1;
    return rc;
}

static
rc_t lookup_and_validate(SRATable *self, const SRAColumn **rslt, uint32_t idx, const void *base, bitsz_t offset, bitsz_t size) {
    const SRAColumn *col;
    
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcWriting, rcSelf, rcNull);
    
    col = VectorGet(&self->wcol, idx - 1);
    if (col == NULL)
        return RC(RC_MODULE, RC_TARGET, rcWriting, rcParam, rcInvalid);
    
    if (size % col->elem_bits != 0)
        return RC(RC_MODULE, RC_TARGET, rcWriting, rcParam, rcInvalid);
    
    if (offset % col->elem_bits != 0)
        return RC(RC_MODULE, RC_TARGET, rcWriting, rcParam, rcInvalid);
    
    *rslt = col;
    return 0;
}

/* SetIdxColumnDefault
 *  give a default value for column
 *
 *  if no value gets written to a column within an open spot,
 *  this value is substituted.
 *
 *  "idx" [ IN ] - 1-based column index
 *
 *  "base" [ IN ] and "offset" [ IN ] - pointer and bit offset
 *  to start of row data
 *
 *  "size" [ IN ] - size in bits of row data
 */
LIB_EXPORT rc_t CC SRATableSetIdxColumnDefault ( SRATable *self,
        uint32_t idx, const void *base, bitsz_t offset, bitsz_t size ) {
    const SRAColumn *col;
    rc_t rc;
    
    rc = lookup_and_validate(self, &col, idx, base, offset, size);
    if (rc == 0)
        rc = VCursorDefault(self->curs, col->idx, col->elem_bits, base, offset / col->elem_bits, size / col->elem_bits);
    return rc;
}


/* WriteIdxColumn
 *  write row data to an indexed column
 *
 *  "idx" [ IN ] - 1-based column index
 *
 *  "base" [ IN ] and "offset" [ IN ] - pointer and bit offset
 *  to start of row data
 *
 *  "size" [ IN ] - size in bits of row data
 */
LIB_EXPORT rc_t CC SRATableWriteIdxColumn ( SRATable *self,
        uint32_t idx, const void *base, bitsz_t offset, bitsz_t size ) {
    const SRAColumn *col;
    rc_t rc;
    
    rc = lookup_and_validate(self, &col, idx, base, offset, size);
    if (rc == 0)
        rc = VCursorWrite(self->curs, col->idx, col->elem_bits, base, offset / col->elem_bits, size / col->elem_bits);
    return rc;
}


/* MetaFreeze
 *  freezes current metadata revision
 *  further modification will begin on a copy
 */
LIB_EXPORT rc_t CC SRATableMetaFreeze ( SRATable *self ) {
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcOpening, rcSelf, rcNull);
    
    return KMetadataFreeze(self->meta);
}


/* OpenMDataNode
 *  open a metadata node
 */
LIB_EXPORT rc_t CC SRATableOpenMDataNodeUpdate ( SRATable *self, struct KMDataNode **node, const char *path, ... ) {
    va_list va;
    rc_t rc;
    
    va_start(va, path);
    rc = SRATableVOpenMDataNodeUpdate(self, node, path, va);
    va_end(va);
    return rc;
}

LIB_EXPORT rc_t CC SRATableVOpenMDataNodeUpdate ( SRATable *self, struct KMDataNode **node, const char *spec, va_list args ) {
    rc_t rc;
    char path[4096];
    int n;
    
    if (self == NULL)
        return RC(RC_MODULE, RC_TARGET, rcOpening, rcSelf, rcNull);
    if (spec == NULL)
        return RC(RC_MODULE, RC_TARGET, rcOpening, rcParam, rcNull);
    
    n = vsnprintf(path, sizeof(path), spec, args);
    if (n >= sizeof(path))
        return RC(RC_MODULE, RC_TARGET, rcOpening, rcName, rcTooLong);
    
    rc = KMetadataOpenNodeUpdate(self->meta, node, path);
    if (rc)
    {
        SRADBG(("failed to open table metadata node '%s' %R", path, rc));
    }
    return rc;
}
