/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include <klib/extern.h>
#include <klib/out.h>
#include <klib/status.h>
#include <klib/log.h>
#include <klib/debug.h>
#include <klib/text.h>
#include <klib/rc.h>
#include <klib/sort.h>
#include <sysalloc.h>
#include "writer-priv.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include <os-native.h>

static char wrt_app[32];
static size_t wrt_app_length;
static char wrt_vers[8];
static size_t wrt_vers_length;

void* KWrt_DefaultWriterDataStdOut = NULL;
void* KWrt_DefaultWriterDataStdErr = NULL;

/*
 *  "appname" [ IN ] - identity of executable, usually argv[0]
 *
 *  "vers" [ IN ] - 4-part version code: 0xMMmmrrrr, where
 *      MM = major release
 *      mm = minor release
 *    rrrr = bug-fix release
 */
LIB_EXPORT rc_t CC KWrtInit( const char* appname, uint32_t vers )
{
    rc_t rc;

    if ( appname == NULL )
        return RC ( rcRuntime, rcLog, rcConstructing, rcString, rcNull );
    if ( appname [ 0 ] == 0 )
        return RC ( rcRuntime, rcLog, rcConstructing, rcString, rcEmpty );

    do
    {
        const char* progname;
        const char* ext;
        size_t progname_z;

        /* find whichever is last \ or / */
        string_measure(appname, &progname_z);
        progname = string_rchr(appname, progname_z, '/');
        if( progname == NULL ) {
            progname = appname;
        } else {
            progname++;
            string_measure(progname, &progname_z);
        }
        appname = string_rchr(progname, progname_z, '\\');
        if( appname == NULL ) {
            appname = progname;
        } else {
            appname++;
        }
        string_measure(appname, &progname_z);

        ext = string_chr(appname, progname_z, '.');

        if( appname != NULL ) {
            wrt_app_length = ext - appname;
        } else {
            wrt_app_length = progname_z;
        }
        if ( wrt_app_length >= sizeof(wrt_app) ) {
            wrt_app_length = sizeof(wrt_app) - 1;
        }
        memcpy(wrt_app, appname, wrt_app_length);
        wrt_app[wrt_app_length] = '\0';

        wrt_vers_length = knprintf(wrt_vers, sizeof(wrt_vers), "%V", vers);

        rc = KWrtSysInit(&KWrt_DefaultWriterDataStdOut, &KWrt_DefaultWriterDataStdErr);
        if (rc) break;

        rc = KOutInit();
        if (rc) break;

        rc = KLogInit();
        if (rc) break;

        rc = KStsInit();
        if (rc) break;

        rc = KDbgInit();
    } while (0);

    return rc;
}

/*--------------------------------------------------------------------------
 * nvp - name/value pair
 */
static
int CC wrt_nvp_cmp_func(const void *a, const void *b, void * ignored)
{
    int i = 0;
    const char *key = a;
    const char *name = ( ( const wrt_nvp_t* ) b ) -> name;

    while(key[i] == name[i]) {
        if( key[i] == '\0' || name[i] == '\0' ) {
            break;
        }
        i++;
    }
    /* treat \0 or right-paren as key terminator */
    if( key[i] != 0 && key[i] != ')' ) {
        return key[i] - name[i];
    }
    return 0 - name[i];
}

static
int CC wrt_nvp_sort_func(const void *a, const void *b, void * ignored)
{
    const wrt_nvp_t *left = a;
    const wrt_nvp_t *right = b;
    return strcmp ( left -> name, right -> name );
}

LIB_EXPORT void CC wrt_nvp_sort( size_t argc, wrt_nvp_t argv[])
{
    if( argc > 1 ) {
        ksort(argv, argc, sizeof(argv[0]), wrt_nvp_sort_func, NULL);
    }
}

LIB_EXPORT const wrt_nvp_t* CC wrt_nvp_find( size_t argc, const wrt_nvp_t argv[], const char* key )
{
    if( argc > 0 ) {
        return kbsearch(key, argv, argc, sizeof(argv[0]), wrt_nvp_cmp_func, NULL);
    }
    return NULL;
}

LIB_EXPORT const char* CC wrt_nvp_find_value( size_t argc, const wrt_nvp_t argv[], const char* key )
{
    if( argc > 0 ) {
        const wrt_nvp_t* n = (const wrt_nvp_t*)kbsearch(key, argv, argc, sizeof(argv[0]), wrt_nvp_cmp_func, NULL);
        if( n != NULL ) {
            return n->value;
        }
    }
    return NULL;
}

static
rc_t RCLiteral ( rc_t rc, char *buffer, size_t bsize, size_t *num_writ )
{
    size_t len = knprintf(buffer, bsize, "rc = %u.%u.%u.%u.%u",
                ( unsigned int ) GetRCModule ( rc ), ( unsigned int ) GetRCTarget ( rc ),
                ( unsigned int ) GetRCContext ( rc ), ( unsigned int ) GetRCObject ( rc ),
                ( unsigned int ) GetRCState ( rc ));

    *num_writ = len;
    if( len >= bsize ) {
        return RC ( rcRuntime, rcLog, rcLogging, rcBuffer, rcInsufficient );
    }
    return 0;
}

LIB_EXPORT rc_t CC RCExplain ( rc_t rc, char *buffer, size_t bsize, size_t *num_writ )
{
    return RCExplain2
        ( rc, buffer, bsize, num_writ, eRCExOpt_CompleteMsg );
}

LIB_EXPORT rc_t CC RCExplain2 ( rc_t rc, char *buffer, size_t bsize, size_t *num_writ,
                                enum ERCExplain2Options options )
{
    bool noMessageIfNoError =
        (options == eRCExOpt_NoMessageIfNoError || 
         options == eRCExOpt_ObjAndStateOnlyIfError);
    int len;
    size_t total = 0;

    const char *mod = GetRCModuleText ( GetRCModule ( rc ) );
    const char *targ = GetRCTargetText ( GetRCTarget ( rc ) );
    const char *ctx = GetRCContextText ( GetRCContext ( rc ) );
    const char *obj = GetRCObjectText ( GetRCObject ( rc ) );
    const char *state = GetRCStateText ( GetRCState ( rc ) );

    assert( buffer && num_writ );

    *num_writ = 0;
    if( rc == 0 && noMessageIfNoError ) {
        buffer[0] = '\0';
        return 0;
    }

    /* English'ish formatting */
    if( obj != NULL ) {
        len = snprintf(buffer + total, bsize - total, "%s", obj);
        if( len < 0 || ( total + len ) >= bsize ) {
            return RCLiteral ( rc, buffer, bsize, num_writ );
        }
        total += len;
    }
    if( state != NULL ) {
        len = snprintf(buffer + total, bsize - total, "%s%s", total ? " " : "", state);
        if( len < 0 || ( total + len ) >= bsize ) {
            return RCLiteral ( rc, buffer, bsize, num_writ );
        }
        total += len;
    }
    if( rc != 0 && options == eRCExOpt_CompleteMsg ) {
        if( ctx != NULL ) {
            len = snprintf ( buffer + total, bsize - total, "%swhile %s", total ? " " : "", ctx );
            if ( len < 0 || ( total + len ) >= bsize ) {
                return RCLiteral ( rc, buffer, bsize, num_writ );
            }
            total += len;
            if( targ != NULL ) {
                len = snprintf ( buffer + total, bsize - total, "%s%s", total ? " " : "", targ );
                if( len < 0 || ( total + len ) >= bsize ) {
                    return RCLiteral ( rc, buffer, bsize, num_writ );
                }
                total += len;
            }
        } else if( targ != NULL ) {
            len = snprintf ( buffer + total,
                bsize - total, "%swhile acting upon %s", total ? " " : "", targ );
            if( len < 0 || ( total + len ) >= bsize ) {
                return RCLiteral ( rc, buffer, bsize, num_writ );
            }
            total += len;
        }
    }
    if( mod != NULL && options == eRCExOpt_CompleteMsg ) {
        len = snprintf(buffer + total, bsize - total, "%swithin %s module", total ? " " : "", mod);
        if( len < 0 || ( total + len ) >= bsize ) {
            return RCLiteral ( rc, buffer, bsize, num_writ );
        }
        total += len;
    }
    *num_writ = total;
    return 0;
}

/*--------------------------------------------------------------------------
 * RC
 */


/* GetRCModuleText
 */
const char *GetRCModuleText ( enum RCModule mod )
{
    if ( ( int ) mod < 0 || ( int ) mod >= ( int ) rcLastModule_v1_1 )
        return "<INVALID-MODULE>";
    return gRCModule_str [ ( int ) mod ];
}

/* GetRCModuleIdxText
 */
const char *GetRCModuleIdxText ( enum RCModule mod )
{
    if ( ( int ) mod < 0 || ( int ) mod >= ( int ) rcLastModule_v1_1 )
        return "<INVALID-MODULE>";
    return gRCModuleIdx_str [ ( int ) mod ];
}

/* GetRCTargetText
 */
const char *GetRCTargetText ( enum RCTarget targ )
{
    if ( ( int ) targ < 0 || ( int ) targ >= ( int ) rcLastTarget_v1_1 )
        return "<INVALID-TARGET>";
    return gRCTarget_str [ ( int ) targ ];
}

/* GetRCTargetIdxText
 */
const char *GetRCTargetIdxText ( enum RCTarget targ )
{
    if ( ( int ) targ < 0 || ( int ) targ >= ( int ) rcLastTarget_v1_1 )
        return "<INVALID-TARGET>";
    return gRCTargetIdx_str [ ( int ) targ ];
}

/* GetRCContextText
 */
const char *GetRCContextText ( enum RCContext ctx )
{
    if ( ( int ) ctx < 0 || ( int ) ctx >= ( int ) rcLastContext_v1_1 )
        return "<INVALID-CONTEXT>";
    return gRCContext_str [ ( int ) ctx ];
}

/* GetRCContextIdxText
 */
const char *GetRCContextIdxText ( enum RCContext ctx )
{
    if ( ( int ) ctx < 0 || ( int ) ctx >= ( int ) rcLastContext_v1_1 )
        return "<INVALID-CONTEXT>";
    return gRCContextIdx_str [ ( int ) ctx ];
}

/* GetRCObjectText
 */
const char *GetRCObjectText ( int obj )
{
    if ( ( int ) obj < 0 || ( int ) obj >= ( int ) rcLastObject_v1_1 )
        return "<INVALID-OBJECT>";
    if ( ( int ) obj < ( int ) rcLastTarget_v1_1 )
        return gRCTarget_str [ ( int ) obj ];
    return gRCObject_str [ ( int ) obj - ( int ) ( rcLastTarget_v1_1 - 1 ) ];
}

/* GetRCObjectIdxText
 */
const char *GetRCObjectIdxText ( int obj )
{
    if ( ( int ) obj < 0 || ( int ) obj >= ( int ) rcLastObject_v1_1 )
        return "<INVALID-OBJECT>";
    if ( ( int ) obj < ( int ) rcLastTarget_v1_1 )
        return gRCTargetIdx_str [ ( int ) obj ];
    return gRCObjectIdx_str [ ( int ) obj - ( int ) ( rcLastTarget_v1_1 - 1 ) ];
}

/* GetRCStateText
 */
const char *GetRCStateText ( enum RCState state )
{
    if ( ( int ) state < 0 || ( int ) state >= ( int ) rcLastState_v1_1 )
        return "<INVALID-STATE>";
    return gRCState_str [ ( int ) state ];
}

/* GetRCStateIdxText
 */
const char *GetRCStateIdxText ( enum RCState state )
{
    if ( ( int ) state < 0 || ( int ) state >= ( int ) rcLastState_v1_1 )
        return "<INVALID-STATE>";
    return gRCStateIdx_str [ ( int ) state ];
}

/* InsertSpace
 *  inserts a division after current text
 *
 *  "spacer" [ IN, NULL OKAY ] - optional characters to insert
 */
LIB_EXPORT rc_t CC LogInsertSpace(const char *spacer, char *buffer, size_t bsize, size_t *num_writ)
{
    int len;

    if ( spacer == NULL )
    {
        if ( bsize < 2 )
            return RC ( rcRuntime, rcLog, rcLogging, rcBuffer, rcInsufficient );
        buffer [ 0 ] = ' ';
        buffer [ 1 ] = 0;
        * num_writ = 1;
        return 0;
    }

    len = snprintf ( buffer, bsize, spacer );

    * num_writ = len;

    if ( len < 0 || (size_t)len >= bsize )
    {
        if ( len < 0 )
            * num_writ = 0;
        return RC ( rcRuntime, rcLog, rcLogging, rcBuffer, rcInsufficient );
    }

    return 0;
}

LIB_EXPORT rc_t CC LogAppName(char *buffer, size_t bsize, size_t *num_writ)
{
    if( wrt_app_length > bsize ) {
        return RC(rcRuntime, rcLog, rcLogging, rcBuffer, rcInsufficient);
    }
    memcpy(buffer, wrt_app, wrt_app_length);
    *num_writ = wrt_app_length;
    return 0;
}

LIB_EXPORT rc_t CC LogAppVersion(char *buffer, size_t bsize, size_t *num_writ)
{
    if( wrt_vers_length > bsize ) {
        return RC(rcRuntime, rcLog, rcLogging, rcBuffer, rcInsufficient);
    }
    memcpy(buffer, wrt_vers, wrt_vers_length);
    *num_writ = wrt_vers_length;
    return 0;
}

LIB_EXPORT rc_t CC LogFlush(KWrtHandler* handler, const char *buffer, const size_t bsize)
{
    rc_t rc = 0;
    size_t num_written;
    size_t remaining;

    assert(handler != NULL);
    assert(buffer != NULL);

    for(remaining = bsize; rc == 0 && remaining > 0; remaining -= num_written, buffer += num_written) {
        rc = handler->writer(handler->data, buffer, remaining, &num_written);
    }
    return rc;
}
