/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

/*--------------------------------------------------------------------------
 * forwards
 */
struct KSysFile;
#define KFILE_IMPL struct KSysFile

#include <kfs/extern.h>
#include "sysfile-priv.h"
#include <kfs/kfs-priv.h>
#include <klib/rc.h>
#include <sysalloc.h>

#ifndef __USE_UNIX98
#define __USE_UNIX98 1
#endif
#include <unistd.h>

#include <limits.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <assert.h>


/*--------------------------------------------------------------------------
 * KSysFile
 *  a Unix file
 */

/* Destroy
 */
static
rc_t KSysFileDestroy ( KSysFile *self )
{
    if ( close ( self -> fd ) != 0 ) switch ( errno )
    {
    case EBADF:
        break;
    case EINTR:
        return RC ( rcFS, rcFile, rcDestroying, rcFunction, rcIncomplete );
    default:
        return RC ( rcFS, rcFile, rcDestroying, rcNoObj, rcUnknown );
    }
    
	free ( self );
    return 0;
}

/* GetSysFile
 *  returns an underlying system file object
 *  and starting offset to contiguous region
 *  suitable for memory mapping, or NULL if
 *  no such file is available.
 */
static
KSysFile *KSysFileGetSysFile ( const KSysFile *self, uint64_t *offset )
{
    * offset = 0;
    return ( KSysFile* ) self;
}

/* RandomAccess
 *  ALMOST by definition, the file is random access
 *  certain file types ( notably compressors ) will refuse random access
 *
 *  returns 0 if random access, error code otherwise
 */
static
rc_t KSysFileRandomAccess ( const KSysFile *self )
{
    struct stat st;

    if ( fstat ( self -> fd, & st ) != 0 ) switch ( errno )
    {
    case EBADF:
        return RC ( rcFS, rcFile, rcAccessing, rcFileDesc, rcInvalid );
    default:
        return RC ( rcFS, rcFile, rcAccessing, rcNoObj, rcUnknown );
    }

    if ( ! S_ISREG ( st . st_mode ) )
/*         return RC ( rcFS, rcFile, rcAccessing, rcFileDesc, rcIncorrect ); */
        return RC ( rcFS, rcFile, rcAccessing, rcFunction, rcUnsupported );
    return 0;
}


/* Type
 *  returns a KFileDesc
 *  not intended to be a content type,
 *  but rather an implementation class
 */
static
uint32_t KSysFileType ( const KSysFile *self )
{
    struct stat st;

    if ( fstat ( self -> fd, & st ) != 0 )
        return kfdInvalid;

    if ( ! S_ISREG ( st . st_mode ) )
    {
        if ( S_ISCHR ( st . st_mode ) )
            return kfdCharDev;
        if ( S_ISBLK ( st . st_mode ) )
            return kfdBlockDev;
        if ( S_ISFIFO ( st . st_mode ) )
            return kfdFIFO;
        if ( S_ISSOCK ( st . st_mode ) )
            return kfdSocket;
    }

    return kfdFile;
}


/* Size
 *  returns size in bytes of file
 *
 *  "size" [ OUT ] - return parameter for file size
 */
static
rc_t KSysFileSize ( const KSysFile *self, uint64_t *size )
{
    struct stat st;

    if ( fstat ( self -> fd, & st ) != 0 ) switch ( errno )
    {
    case EBADF:
        return RC ( rcFS, rcFile, rcAccessing, rcFileDesc, rcInvalid );
    default:
        return RC ( rcFS, rcFile, rcAccessing, rcNoObj, rcUnknown );
    }

    if ( S_ISDIR ( st . st_mode ) )
/*         return RC ( rcFS, rcFile, rcAccessing, rcFileDesc, rcIncorrect ); */
        return RC ( rcFS, rcFile, rcAccessing, rcFunction, rcUnsupported );

    * size = st . st_size;

    return 0;
}

/* SetSize
 *  sets size in bytes of file
 *
 *  "size" [ IN ] - new file size
 */
static
rc_t KSysFileSetSize ( KSysFile *self, uint64_t size )
{
    if ( ftruncate ( self -> fd, size ) != 0 ) switch ( errno )
    {
    case EBADF:
        return RC ( rcFS, rcFile, rcUpdating, rcFileDesc, rcInvalid );
    default:
        return RC ( rcFS, rcFile, rcUpdating, rcNoObj, rcUnknown );
    }

    return 0;
}

/* Read
 *  read file from known position
 *
 *  "pos" [ IN ] - starting position within file
 *
 *  "buffer" [ OUT ] and "bsize" [ IN ] - return buffer for read
 *
 *  "num_read" [ OUT, NULL OKAY ] - optional return parameter
 *  giving number of bytes actually read
 */
static
rc_t KSysFileRead ( const KSysFile *self, uint64_t pos,
    void *buffer, size_t bsize, size_t *num_read )
{
    assert ( self != NULL );
    while ( 1 )
    {
        ssize_t count = pread ( self -> fd, buffer, bsize, pos );

        if ( count < 0 ) switch ( errno )
        {
        case EINTR:
            continue;
        case EIO:
            return RC ( rcFS, rcFile, rcReading, rcTransfer, rcUnknown );
        case EBADF:
            return RC ( rcFS, rcFile, rcReading, rcFileDesc, rcInvalid );
        case EISDIR:
            return RC ( rcFS, rcFile, rcReading, rcFileDesc, rcIncorrect );
        case EINVAL:
            return RC ( rcFS, rcFile, rcReading, rcParam, rcInvalid );
        default:
            return RC ( rcFS, rcFile, rcReading, rcNoObj, rcUnknown );
        }

        assert ( num_read != NULL );
        * num_read = count;
        break;
    }

    return 0;
}

/* Write
 *  write file at known position
 *
 *  "pos" [ IN ] - starting position within file
 *
 *  "buffer" [ IN ] and "size" [ IN ] - data to be written
 *
 *  "num_writ" [ OUT, NULL OKAY ] - optional return parameter
 *  giving number of bytes actually written
 */
static
rc_t KSysFileWrite ( KSysFile *self, uint64_t pos,
    const void *buffer, size_t size, size_t *num_writ)
{
    assert ( self != NULL );
    while ( 1 )
    {
        ssize_t count = pwrite ( self -> fd, buffer, size, pos );

        if ( count < 0 ) switch ( errno )
        {
        case ENOSPC:
            return RC ( rcFS, rcFile, rcWriting, rcStorage, rcExhausted );
        case EINTR:
            continue;
        case EFBIG:
            return RC ( rcFS, rcFile, rcWriting, rcFile, rcExcessive );
        case EIO:
            return RC ( rcFS, rcFile, rcWriting, rcTransfer, rcUnknown );
        case EBADF:
            return RC ( rcFS, rcFile, rcWriting, rcFileDesc, rcInvalid );
        case EINVAL:
            return RC ( rcFS, rcFile, rcWriting, rcParam, rcInvalid );
        default:
            return RC ( rcFS, rcFile, rcWriting, rcNoObj, rcUnknown );
        }

        assert ( num_writ != NULL );
        * num_writ = count;
        break;
    }

    return 0;
}


/* Make
 *  create a new file object
 *  from file descriptor
 */
static KFile_vt_v1 vtKSysFile =
{
    /* version 1.1 */
    1, 1,

    /* start minor version 0 methods */
    KSysFileDestroy,
    KSysFileGetSysFile,
    KSysFileRandomAccess,
    KSysFileSize,
    KSysFileSetSize,
    KSysFileRead,
    KSysFileWrite,
    /* end minor version 0 methods */

    /* start minor version == 1 */
    KSysFileType
    /* end minor version == 1 */
};

static
rc_t KSysFileMakeVT ( KSysFile **fp, int fd, const KFile_vt *vt,
    bool read_enabled, bool write_enabled )
{
    rc_t rc;
    KSysFile *f;

    if ( fd < 0 )
        return RC ( rcFS, rcFile, rcConstructing, rcFileDesc, rcInvalid );

    f = malloc ( sizeof *f );
    if ( f == NULL )
        rc = RC ( rcFS, rcFile, rcConstructing, rcMemory, rcExhausted );
    else
    {
        rc = KFileInit ( & f -> dad, vt, read_enabled, write_enabled );
        if ( rc == 0 )
        {
            f -> fd = fd;
            * fp = f;
            return 0;
        }

        free ( f );
    }
    return rc;
}

LIB_EXPORT rc_t CC KSysFileMake ( KSysFile **fp, int fd, bool read_enabled, bool write_enabled )
{
    return KSysFileMakeVT ( fp, fd, ( const KFile_vt* ) & vtKSysFile,
        read_enabled, write_enabled );
}

/*--------------------------------------------------------------------------
 * KFile
 *  Unix-specific standard i/o interfaces
 */

typedef struct KStdIOFile KStdIOFile;
struct KStdIOFile
{
    KSysFile dad;
    uint64_t pos;
};


/* Destroy
 *  does not close fd
 */
static
rc_t KStdIOFileDestroy ( KSysFile *self )
{
	free ( self );
    return 0;
}

static KFile_vt_v1 vtKStdIOFile =
{
    /* version 1.1 */
    1, 1,

    /* start minor version 0 methods */
    KStdIOFileDestroy,
    KSysFileGetSysFile,
    KSysFileRandomAccess,
    KSysFileSize,
    KSysFileSetSize,
    KSysFileRead,
    KSysFileWrite,
    /* end minor version 0 methods */

    /* start minor version == 1 */
    KSysFileType
    /* end minor version == 1 */
};

/* RandomAccess
 */
static
rc_t KStdIOFileRandomAccess ( const KSysFile *self )
{
/*     return RC ( rcFS, rcFile, rcAccessing, rcFileDesc, rcIncorrect ); */
        return RC ( rcFS, rcFile, rcAccessing, rcFunction, rcUnsupported );
}


/* Size
 */
static
rc_t KStdIOFileSize ( const KSysFile *self, uint64_t *size )
{
    * size = 0;

/*     return RC ( rcFS, rcFile, rcAccessing, rcFileDesc, rcIncorrect ); */
    return RC ( rcFS, rcFile, rcAccessing, rcFunction, rcUnsupported );
}

/* SetSize
 */
static
rc_t KStdIOFileSetSize ( KSysFile *self, uint64_t size )
{
/*     return RC ( rcFS, rcFile, rcUpdating, rcFileDesc, rcIncorrect ); */
        return RC ( rcFS, rcFile, rcAccessing, rcFunction, rcUnsupported );
}

/* Read
 *  read file from known position
 *
 *  "pos" [ IN ] - starting position within file
 *
 *  "buffer" [ OUT ] and "bsize" [ IN ] - return buffer for read
 *
 *  "num_read" [ OUT, NULL OKAY ] - optional return parameter
 *  giving number of bytes actually read
 */
static
rc_t KStdIOFileRead ( const KSysFile *dad, uint64_t pos,
    void *buffer, size_t bsize, size_t *num_read )
{
    KStdIOFile *self = ( KStdIOFile* ) dad;
    assert ( self != NULL );

    if ( self -> pos != pos )
        return RC ( rcFS, rcFile, rcReading, rcParam, rcInvalid );

    while ( 1 )
    {
        ssize_t count = read ( self -> dad . fd, buffer, bsize );

        if ( count < 0 ) switch ( errno )
        {
        case EINTR:
            continue;
        case EIO:
            return RC ( rcFS, rcFile, rcReading, rcTransfer, rcUnknown );
        case EBADF:
            return RC ( rcFS, rcFile, rcReading, rcFileDesc, rcInvalid );
        case EISDIR:
            return RC ( rcFS, rcFile, rcReading, rcFileDesc, rcIncorrect );
        case EINVAL:
            return RC ( rcFS, rcFile, rcReading, rcParam, rcInvalid );
        default:
            return RC ( rcFS, rcFile, rcReading, rcNoObj, rcUnknown );
        }

        assert ( num_read != NULL );
        * num_read = count;
        self -> pos += count;
        break;
    }

    return 0;
}

/* Write
 *  write file at known position
 *
 *  "pos" [ IN ] - starting position within file
 *
 *  "buffer" [ IN ] and "size" [ IN ] - data to be written
 *
 *  "num_writ" [ OUT, NULL OKAY ] - optional return parameter
 *  giving number of bytes actually written
 */
static
rc_t KStdIOFileWrite ( KSysFile *dad, uint64_t pos,
    const void *buffer, size_t size, size_t *num_writ)
{
    KStdIOFile *self = ( KStdIOFile* ) dad;
    assert ( self != NULL );

    if ( self -> pos != pos )
        return RC ( rcFS, rcFile, rcWriting, rcParam, rcInvalid );

    while ( 1 )
    {
        ssize_t count = write ( self -> dad . fd, buffer, size );

        if ( count < 0 ) switch ( errno )
        {
        case ENOSPC:
            return RC ( rcFS, rcFile, rcWriting, rcStorage, rcExhausted );
        case EINTR:
            continue;
        case EFBIG:
            return RC ( rcFS, rcFile, rcWriting, rcFile, rcExcessive );
        case EIO:
            return RC ( rcFS, rcFile, rcWriting, rcTransfer, rcUnknown );
        case EBADF:
            return RC ( rcFS, rcFile, rcWriting, rcFileDesc, rcInvalid );
        case EINVAL:
            return RC ( rcFS, rcFile, rcWriting, rcParam, rcInvalid );
        default:
            return RC ( rcFS, rcFile, rcWriting, rcNoObj, rcUnknown );
        }

        assert ( num_writ != NULL );
        * num_writ = count;
        self -> pos += count;
        break;
    }

    return 0;
}

static KFile_vt_v1 vtKStdIOStream =
{
    /* version 1.1 */
    1, 1,

    /* start minor version 0 methods */
    KStdIOFileDestroy,
    KSysFileGetSysFile,
    KStdIOFileRandomAccess,
    KStdIOFileSize,
    KStdIOFileSetSize,
    KStdIOFileRead,
    KStdIOFileWrite,
    /* end minor version 0 methods */

    /* start minor version == 1 */
    KSysFileType
    /* end minor version == 1 */
};

static
rc_t KStdIOFileTest ( KFile **rp, int fd, bool *seekable )
{
    struct stat st;

    if ( rp == NULL )
        return RC ( rcFS, rcFile, rcCreating, rcParam, rcNull );

    if ( fstat ( fd, & st ) == 0 )
    {
        if ( S_ISREG ( st . st_mode ) )
            * seekable = true;
        else
            * seekable = false;

        return 0;
    }

    * rp = NULL;

    if ( errno == EBADF )
        return RC ( rcFS, rcFile, rcCreating, rcFileDesc, rcInvalid );

    return RC ( rcFS, rcFile, rcCreating, rcNoObj, rcUnknown );
}

static
rc_t KStdIOFileMake ( KFile **fp, int fd,
    bool seekable, bool read_enabled, bool write_enabled )
{
    rc_t rc;
    KStdIOFile *f;

    if ( seekable )
    {
        return KSysFileMakeVT ( ( KSysFile** ) fp, fd,
            ( const KFile_vt* ) & vtKStdIOFile, read_enabled, write_enabled );
    }

    if ( fd < 0 )
        return RC ( rcFS, rcFile, rcConstructing, rcFileDesc, rcInvalid );

    f = malloc ( sizeof *f );
    if ( f == NULL )
        rc = RC ( rcFS, rcFile, rcConstructing, rcMemory, rcExhausted );
    else
    {
        rc = KFileInit ( & f -> dad . dad,
           ( const KFile_vt* ) & vtKStdIOStream, read_enabled, write_enabled );
        if ( rc == 0 )
        {
            f -> dad . fd = fd;
            f -> pos = 0;
            * fp = & f -> dad . dad;
            return 0;
        }

        free ( f );
    }
    return rc;
}

/* MakeStdIn
 *  creates a read-only file on stdin
 */
LIB_EXPORT rc_t CC KFileMakeStdIn ( const KFile **std_in )
{
    bool seekable;
    rc_t rc = KStdIOFileTest ( ( KFile** ) std_in, 0, & seekable );
    if ( rc != 0 )
        return rc;

    return KStdIOFileMake ( ( KFile** ) std_in, 0, seekable, true, false );
}

/* MakeStdOut
 * MakeStdErr
 *  creates a write-only file on stdout or stderr
 */
LIB_EXPORT rc_t CC KFileMakeStdOut ( KFile **std_out )
{
    bool seekable;
    rc_t rc = KStdIOFileTest ( std_out, 1, & seekable );
    if ( rc != 0 )
        return rc;

    return KStdIOFileMake ( std_out, 1, seekable, false, true );
}

LIB_EXPORT rc_t CC KFileMakeStdErr ( KFile **std_err )
{
    bool seekable;
    rc_t rc = KStdIOFileTest ( std_err, 2, & seekable );
    if ( rc != 0 )
        return rc;

    return KStdIOFileMake ( std_err, 2, seekable, false, true );
}

/* MakeFDFile
 *  creates a file from a file-descriptor
 *  not supported under Windows
 */
LIB_EXPORT rc_t CC KFileMakeFDFileRead ( const KFile **f, int fd )
{
    bool seekable;
    rc_t rc = KStdIOFileTest ( ( KFile** ) f, fd, & seekable );
    if ( rc != 0 )
        return rc;

    return KStdIOFileMake ( ( KFile** ) f, fd, seekable, true, false );
}

LIB_EXPORT rc_t CC KFileMakeFDFileWrite ( KFile **f, bool update, int fd )
{
    bool seekable;
    rc_t rc = KStdIOFileTest ( f, fd, & seekable );
    if ( rc != 0 )
        return rc;

    return KStdIOFileMake ( f, fd, seekable, update, true );
}
