/*=======================================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include <kfs/extern.h>

struct KBZipFile;
#define KFILE_IMPL struct KBZipFile

#include <kfs/impl.h>
#include <kfs/bzip.h>
#include <klib/rc.h>
#include <sysalloc.h>

#include <bzlib.h>      /* bz_stream */
#include <assert.h>
#include <stdlib.h>     /* malloc */

/***************************************************************************************/
/* bzip2 File                                                                    */
/***************************************************************************************/

#define BZ2CHUNK 0x20000    /* 128K */
/** bzip2 KFile structure */
struct KBZipFile {
    KFile dad;
    KFile *file; /* inderlying KFile */
    uint64_t filePosition;
    uint64_t myPosition;
    int BZ2_bzDecompressResult;
    bz_stream strm;
    char buff[BZ2CHUNK]; /* buffer to cache KFile data */
    bool completed;
};
typedef struct KBZipFile KBZipFile;

/* virtual functions declarations (definitions for unsupported ) ***********************/

static struct KSysFile *CC s_GetSysFile(const KBZipFile *self,
    uint64_t *offset)
{ return NULL; }

static rc_t CC s_FileRandomAccess(const KBZipFile *self)
{ return RC ( rcFS, rcFile, rcAccessing, rcFunction, rcUnsupported ); }

static uint32_t CC s_FileType ( const KBZipFile *self )
{ return KFileType ( self -> file ); }

static rc_t CC s_FileSize(const KBZipFile *self, uint64_t *size)
{ return RC ( rcFS, rcFile, rcAccessing, rcFunction, rcUnsupported ); }

static rc_t CC s_FileSetSize(KBZipFile *self,
    uint64_t size)
{ return RC ( rcFS, rcFile, rcUpdating, rcFunction, rcUnsupported ); }

/* read-only methods *******************************************************************/

static rc_t CC KBZipFile_InDestroy(KBZipFile *self);

static rc_t CC KBZipFile_InRead(const KBZipFile *cself,
    uint64_t pos,
    void *buffer,
    size_t bsize,
    size_t *num_read);

static rc_t CC KBZipFile_InWrite(KBZipFile *self,
    uint64_t pos,
    const void *buffer,
    size_t size,
    size_t *num_writ)
{ return RC ( rcFS, rcFile, rcWriting, rcFunction, rcUnsupported ); }

/** virtual table **********************************************************************/
static KFile_vt_v1 s_vtKFile_InBzip2 = {
    /* version */
    1, 1,

    /* 1.0 */
    KBZipFile_InDestroy,
    s_GetSysFile,
    s_FileRandomAccess,
    s_FileSize,
    s_FileSetSize,
    KBZipFile_InRead,
    KBZipFile_InWrite,

    /* 1.1 */
    s_FileType
};

/** Factory method definition **********************************************************/

LIB_EXPORT rc_t CC KFileMakeBzip2ForRead( const struct KFile **result,
    const struct KFile *file )
{
    rc_t rc;
    bz_stream* strm;
    KBZipFile *obj;

    if ( result == NULL || file == NULL )
        return RC ( rcFS, rcFile, rcConstructing, rcParam, rcNull );

    obj = (KBZipFile*) malloc(sizeof(KBZipFile));
    if (!obj)
        return RC ( rcFS, rcFile, rcConstructing, rcMemory, rcExhausted );

    rc = KFileInit(&obj->dad, (const KFile_vt*) &s_vtKFile_InBzip2, true, false);
    if (rc != 0) {
        free(obj);
        return rc;
    }

    strm = &obj->strm;
    strm->bzalloc  = 0;
    strm->bzfree   = 0;
    strm->opaque   = 0;
    strm->avail_in = 0;
    strm->next_in  = 0;

    /* TBD - should switch on possible return codes */
    if (BZ2_bzDecompressInit(strm, 1, /* verbosity */
			     0) /* small */
        != BZ_OK)
    {
        free(obj);
        return RC ( rcFS, rcFile, rcConstructing, rcNoObj, rcUnknown );
    }

        obj->myPosition   = 0;
        obj->filePosition = 0;
        obj->BZ2_bzDecompressResult = BZ_OK;

    rc = KFileAddRef(file);
    if ( rc != 0 )
    {
        obj -> file = NULL;
        KBZipFile_InDestroy ( obj );
    }
    else
    {
        obj->file = (KFile*) file;
        *result = &obj->dad;
    }

    return rc;
}

/* private functions declarations ******************************************************/

static int s_read(KBZipFile *self,
    char *buffer,
    size_t bsize,
    size_t *num_read,
    rc_t *rc);

static rc_t z_read (KBZipFile * self, void * buffer, size_t bsize, size_t * num_read );
static rc_t z_skip (KBZipFile *self, uint64_t pos);

/* virtual functions definitions *******************************************************/

static rc_t CC KBZipFile_InDestroy(KBZipFile *self) {
    rc_t rc = KFileRelease(self->file);
    if (rc == 0) {
        BZ2_bzDecompressEnd(&self->strm);
        free(self);
    }

    return rc;
}

static rc_t CC KBZipFile_InRead(const KBZipFile *cself,
    uint64_t pos,
    void *buffer,
    size_t bsize,
    size_t *num_read)
{
    rc_t rc = 0;
    KBZipFile *self = (KBZipFile*) cself;

    size_t numRead = 0, ignore;
    if (!num_read)
    {   num_read = &ignore; }

    *num_read = 0;

    if (!cself || !buffer)
        return RC ( rcFS, rcFile, rcReading, rcParam, rcNull );

    if (!bsize)
    {   return 0; }

    if (pos < self->myPosition)
    {
	return RC ( rcFS, rcFile, rcReading, rcParam, rcInvalid );
    }

    if (pos > self->myPosition)
    {
	rc =  z_skip (self, pos);
	if (rc)
	    return rc;
	if (pos != self->myPosition)
	    return 0;
    }

    rc = z_read ( self, buffer, bsize, &numRead );
    if (rc)
	return rc;

    *num_read = numRead;

    self->myPosition += numRead;

    return 0;
}

/* private functions definitions *******************************************************/

static int s_read(KBZipFile *self,
    char *buffer,
    size_t bsize,
    size_t *num_read,
    rc_t *rc)
{
    bz_stream* strm;
    int ret;

    assert(self && buffer && bsize && num_read);

    strm = &self->strm;

    ret = 0;
    while (!*num_read) {
        strm->next_out  = buffer;
        strm->avail_out = bsize;
        ret = BZ2_bzDecompress(strm);
        assert(ret == BZ_OK || ret == BZ_STREAM_END);  /* state not clobbered */
        *num_read = bsize - strm->avail_out;
        if (strm->avail_out > 0) {
            size_t src_read;
            * rc = KFileRead
                (self->file, self->filePosition, self->buff, sizeof(self->buff), &src_read);
            if (*rc != 0)
            {   return -70; }
            strm->avail_in = src_read;
            self->filePosition += src_read;
            strm->next_in = self->buff;
        }
        if (!strm->avail_in)
        {   break; }
    }
    return ret;
}

static rc_t z_read ( KBZipFile * self, void * buffer, size_t bsize, size_t * num_read )
{
    rc_t rc = 0;
    size_t numRead = 0;

    do
    {
        int ret;
        size_t have = 0;

        if (self->BZ2_bzDecompressResult == BZ_STREAM_END)
            break;
        ret = s_read(self, (char*)buffer + numRead, bsize - numRead, &have, &rc);
        if ( ret == -70 ) /* rc hack - known to not collide with bzlib errors */
            return rc;
        self->BZ2_bzDecompressResult = ret;
        if (!have)
            break;
        numRead += have;
    } while (numRead != bsize);

    *num_read = numRead;
    return rc;
}

static rc_t z_skip (KBZipFile *self, uint64_t pos)
{
    rc_t rc = 0;
    size_t num_read = 0;
    size_t to_read;
    uint8_t buff [ 32 * 1024 ];

    for ( to_read = sizeof buff; self -> myPosition < pos; self -> myPosition += num_read )
    {
	if (self->myPosition + sizeof buff > pos)
	    to_read = pos - self->myPosition;

	rc = z_read ( self, buff, to_read, &num_read );

	if ( rc )
	    break;

	if (num_read == 0)
	    break;

    }
    return rc;
}

/***************************************************************************************/
/* Gzip Output File                                                                    */
/***************************************************************************************/

/* virtual functions declarations (definitions for unsupported ) ***********************/

static rc_t CC KBZipFile_OutDestroy(KBZipFile *self);

static rc_t CC KBZipFile_OutRead(const KBZipFile *cself,
    uint64_t pos,
    void *buffer,
    size_t bsize,
    size_t *num_read)
{ return RC ( rcFS, rcFile, rcReading, rcFunction, rcUnsupported ); }

static rc_t CC KBZipFile_OutWrite(struct KBZipFile *self,
    uint64_t pos,
    const void *buffer,
    size_t bsize,
    size_t *num_writ);

/** virtual table **********************************************************************/
static KFile_vt_v1 s_vtKFile_OutBzip2 = {
    /* version */
    1, 1,

    /* 1.0 */
    KBZipFile_OutDestroy,
    s_GetSysFile,
    s_FileRandomAccess,
    s_FileSize,
    s_FileSetSize,
    KBZipFile_OutRead,
    KBZipFile_OutWrite,

    /* 1.1 */
    s_FileType
};

/** Factory method definition **********************************************************/
LIB_EXPORT rc_t CC KFileMakeBzip2ForWrite( struct KFile **result,
    struct KFile *file )
{
    rc_t rc;
    bz_stream* strm;
    KBZipFile *obj;

    if ( result == NULL || file == NULL )
        return RC ( rcFS, rcFile, rcConstructing, rcParam, rcNull );

    obj = (KBZipFile*) malloc(sizeof(KBZipFile));
    if (!obj)
        return RC ( rcFS, rcFile, rcConstructing, rcMemory, rcExhausted );

    rc = KFileInit(&obj->dad, (const KFile_vt*) &s_vtKFile_OutBzip2, false, true);
    if (rc != 0) {
        free(obj);
        return rc;
    }

    strm = &obj->strm;
    strm->bzalloc = 0;
    strm->bzfree  = 0;
    strm->opaque  = 0;

    /* TBD - should switch on return codes */
    if (BZ2_bzCompressInit(strm, 9, /* blockSize100k */
				 1, /* verbosity */
				 30) /* workFactor */
        != BZ_OK)
    {
        free(obj);
        return RC ( rcFS, rcFile, rcConstructing, rcNoObj, rcUnknown );
    }

        obj->myPosition   = 0;
        obj->filePosition = 0;
        obj->completed    = 0;

    rc = KFileAddRef(file);
    if ( rc != 0 )
    {
        obj -> file = NULL;
        KBZipFile_OutDestroy ( obj );
    }
    else
    {
        obj->file = file;
        *result = &obj->dad;
    }

    return rc;
}

/* private functions declarations ******************************************************/

static int s_Bzip2AndWrite(KBZipFile *self,
    int action,
    size_t *num_writ,
    rc_t *status);

/* virtual functions definitions *******************************************************/

static rc_t CC KBZipFile_OutDestroy(KBZipFile *self) {
    rc_t rc;
    if (!self->completed) {
        int ret;
        size_t wrtn;
        bz_stream* strm = &self->strm;
        strm->avail_in = 0;
        strm->next_in = 0;
        wrtn = 0;
        rc = 0;
        ret = s_Bzip2AndWrite(self, BZ_FINISH, &wrtn, &rc);
        assert(ret == BZ_STREAM_END);        /* stream will be complete */

        BZ2_bzCompressEnd(strm);   /* clean up */

        /* this doesn't work - once the bz stream has
           been destroyed, it can't be reused */
        if ( rc != 0 ) return rc;
        self->completed = true;
    }

    rc = KFileRelease(self->file);
    if (rc == 0)
    {   free(self); }

    return rc;
}

static rc_t CC KBZipFile_OutWrite(struct KBZipFile *self,
    uint64_t pos,
    const void *buffer,
    size_t bsize,
    size_t *num_writ)
{
    int ret;
    rc_t rc;
    bz_stream* strm;
    size_t ignore;
    if (!num_writ)
    {   num_writ = &ignore; }

    *num_writ = 0;

    if (pos != self->myPosition)
    {   return RC ( rcFS, rcFile, rcWriting, rcParam, rcInvalid ); }

    strm = &self->strm;
    strm->next_in  = (char*) buffer;
    strm->avail_in = bsize;

    rc = 0;
    ret = s_Bzip2AndWrite(self, BZ_RUN, num_writ, &rc);
    if ( rc != 0 )
        return rc;
    assert(ret != BZ_STREAM_END);        /* stream will be complete */

    self->myPosition += * num_writ;

    return 0;
}

/* private functions definitions *******************************************************/

static int s_Bzip2AndWrite(KBZipFile *self,
    int action,
    size_t *num_writ,
    rc_t *rc)
{
    bz_stream *strm;
    unsigned avail_in;
    int ret;

    assert(self && num_writ && rc);

    *num_writ = 0;
    strm = &self->strm;
    avail_in = strm->avail_in;
     ret = 0;
    /* run deflate() on input until output buffer not full, finish
       compression if all of source has been read in */
    do {
        uint32_t have;
        size_t written;
        strm->avail_out = sizeof(self->buff);
        strm->next_out = self->buff;
        ret = BZ2_bzCompress(strm, action);    /* no bad return value */

        /* state not clobbered */
        assert(ret == BZ_OK || ret == BZ_RUN_OK
            || ret == BZ_FINISH_OK || ret == BZ_STREAM_END);

        have = sizeof(self->buff) - strm->avail_out;
        written = 0;
        *rc = KFileWrite(self->file, self->filePosition, self->buff, have, &written);
        /* this is wrong - BZ_IO_ERROR could be returned by the library
           and if we were interpreting it, we would be confused. we need
           to use some other error code [ e.g. the -70 hack above ] */
        if (*rc)
        {   return BZ_IO_ERROR; }
        self->filePosition += written;
        *num_writ = avail_in - strm->avail_in;
    } while (strm->avail_out == 0);
    assert(strm->avail_in == 0);     /* all input will be used */
    return ret;
}

/* EOF */
