/*===========================================================================
 *
 *                            PUBLIC DOMAIN NOTICE
 *               National Center for Biotechnology Information
 *
 *  This software/database is a "United States Government Work" under the
 *  terms of the United States Copyright Act.  It was written as part of
 *  the author's official duties as a United States Government employee and
 *  thus cannot be copyrighted.  This software/database is freely available
 *  to the public for use. The National Library of Medicine and the U.S.
 *  Government have not placed any restriction on its use or reproduction.
 *
 *  Although all reasonable efforts have been taken to ensure the accuracy
 *  and reliability of the software and data, the NLM and the U.S.
 *  Government do not and cannot warrant the performance or results that
 *  may be obtained by using this software or data. The NLM and the U.S.
 *  Government disclaim all warranties, express or implied, including
 *  warranties of performance, merchantability or fitness for any particular
 *  purpose.
 *
 *  Please cite the author in any work or product based on this material.
 *
 *============================================================================
 *
 */

#include <kdb/extern.h>

#define KONST const
#include "index-priv.h"
#include "dbmgr-priv.h"
#include "database-priv.h"
#include "table-priv.h"
#include "kdb-priv.h"
#include "column-priv.h"
#undef KONST

#include "cc-priv.h"

#include <kdb/index.h>

#include <kfs/file.h>
#include <kfs/md5.h>
#include <klib/refcount.h>
#include <klib/rc.h>
#include <klib/namelist.h>
#include <kdb/namelist.h>
#include <os-native.h>
#include <sysalloc.h>

#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <errno.h>

#undef index

static
rc_t KTableCheckMD5(const KTable *self, CCReportFunc report, void *ctx)
{
    CCReportInfoBlock nfo;
    char *sep;

    memset(&nfo, 0, sizeof(nfo));
    nfo.objType = kptTable;
    nfo.objName = self->path;
    sep = strrchr(self->path, '/');
    if (sep) {
        nfo.objName = sep + 1;
    }
    
    return DirectoryCheckMD5(self->dir, "md5", &nfo, report, ctx);
}

static
rc_t KTableCheckColumns(const KTable *self, int level, CCReportFunc report, void *ctx)
{
    uint32_t n;
    KNamelist *list;
    rc_t rc = KTableListCol(self, &list);
    
    if (rc)
        return rc;

    rc = KNamelistCount(list, &n);
    if (rc == 0) {
        CCReportInfoBlock nfo;
        
        memset(&nfo, 0, sizeof(nfo));
        nfo.objType = kptColumn;
        
        for (nfo.objId = 0; nfo.objId != n && rc == 0; ++nfo.objId) {
            rc = KNamelistGet(list, nfo.objId, &nfo.objName);
            if (rc == 0) {
                const KColumn *col;
                
                rc = KTableOpenColumnRead(self, &col, nfo.objName);
                if (rc == 0) {
                    rc = KColumnConsistencyCheck(col, level, &nfo, report, ctx);
                    KColumnRelease(col);
                }
            }
        }
    }
    
    KNamelistRelease(list);
    return rc;
}

static
rc_t KTableCheckIndexMD5(const KDirectory *dir,
                         CCReportInfoBlock *nfo,
                         CCReportFunc report, void *ctx)
{
    char md5[4096];

    snprintf(md5, sizeof(md5), "%s.md5", nfo->objName);
    return DirectoryCheckMD5(dir, md5, nfo, report, ctx);
}

static const KDirectory *KTableFindIndexDir(const KTable *self)
{
    const KDirectory *idxDir;
    
    rc_t rc = KDirectoryOpenDirRead(self->dir, &idxDir, false, "idx");
    if (rc) {
        KDirectoryAddRef(self->dir);
        return self->dir;
    }
    return idxDir;
}

static
rc_t KTableCheckIndices(const KTable *self, int level, CCReportFunc report, void *ctx)
{
    uint32_t n;
    KNamelist *list;
    rc_t rc = KTableListIdx(self, &list);
    
    if (rc)
        return rc;
    rc = KNamelistCount(list, &n);
    if (rc == 0) {
        CCReportInfoBlock nfo;
        
        memset(&nfo, 0, sizeof(nfo));
        nfo.objType = kptIndex;
        for (nfo.objId = 0; rc == 0 && nfo.objId != (int)n; ++nfo.objId) {
            const KDirectory *idxDir = KTableFindIndexDir(self);

            rc = KNamelistGet(list, nfo.objId, &nfo.objName);
            if (rc)
                break;
            
            rc = KTableCheckIndexMD5(idxDir, &nfo, report, ctx);
            if (rc == 0 && level > 0) {
                const KIndex *idx;
                
                rc = KTableOpenIndexRead(self, &idx, nfo.objName);
                if (rc) {
                    nfo.type = ccrpt_Done;
                    nfo.info.done.rc = rc;
                    nfo.info.done.mesg = "could not be opened";
                    rc = report(&nfo, ctx);
                }
                else {
                    nfo.type = ccrpt_Index;
                    rc = KIndexConsistencyCheck(idx, level < 3 ? 1 : 3,
                                                &nfo.info.index.start_id,
                                                &nfo.info.index.id_range,
                                                &nfo.info.index.num_keys,
                                                &nfo.info.index.num_rows,
                                                &nfo.info.index.num_holes);
                    KIndexRelease(idx);
                    if (rc) {
                        nfo.type = ccrpt_Done;
                        nfo.info.done.rc = rc;
                        nfo.info.done.mesg = "could not be validated";
                    }
                    rc = report(&nfo, ctx);
                }
            }
            KDirectoryRelease(idxDir);
        }
    }
    KNamelistRelease(list);
    return rc;
}

LIB_EXPORT
rc_t CC KTableConsistencyCheck(const KTable *self, uint32_t level,
                               CCReportFunc report, void *ctx)
{
    rc_t rc;
    
    if (self == NULL)
        return RC(rcDB, rcTable, rcValidating, rcSelf, rcNull);
    
    rc = KTableCheckMD5(self, report, ctx);
    if (rc)
        return rc;
    
    rc = KTableCheckColumns(self, level, report, ctx);
    if (rc)
        return rc;
    
    rc = KTableCheckIndices(self, level, report, ctx);
    if (rc)
        return rc;
        
    return 0;
}
