/* specter
 *
 * userspace logging daemon for netfilter ULOG target
 * of the linux 2.4/2.6 netfilter subsystem.
 *
 * (C) 2000-2003 by Harald Welte <laforge@gnumonks.org>
 * (C) 2004 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 *
 * This code is released under the terms of GNU GPL.
 */

#ifndef _SPECTER_H
#define _SPECTER_H

#include <sys/socket.h>
#include <net/if.h>
#include <linux/netfilter_ipv4/ipt_ULOG.h>
#include <conffile/conffile.h>


/***********************************************************************
 *   OUTPUT/INPUT PLUGIN STRUCTURES
 ***********************************************************************/
#define SPECTER_PLUGIN_NAME_LEN 32
typedef struct specter_output {
	char name[SPECTER_PLUGIN_NAME_LEN];
	char path[PATH_MAX];
	config_entry_t *ce_base;
	int ce_num;
	void *(*init)(config_entry_t *ce);
	void (*fini)(config_entry_t *ce, void *data);
	int (*output)(config_entry_t *ce, void *data);
	int (*signal)(config_entry_t *ce, void *data, int signal);
} specter_output_t;

typedef struct specter_input {
	char name[SPECTER_PLUGIN_NAME_LEN];
	char path[PATH_MAX];
	int (*input)(ulog_packet_msg_t *pkt);
} specter_input_t;


/***********************************************************************
 *   SPECTER_IRET_T STRUCTURE
 *      informations gathered by input plugins are stored here
 ***********************************************************************/
enum {
	SPECTER_IRET_NONE,

	SPECTER_IRET_INT8,
	SPECTER_IRET_INT16,
	SPECTER_IRET_INT32,
	SPECTER_IRET_INT64,

	SPECTER_IRET_UINT8,
	SPECTER_IRET_UINT16,
	SPECTER_IRET_UINT32,
	SPECTER_IRET_UINT64,

	SPECTER_IRET_BOOL,

	SPECTER_IRET_IPADDR,

	SPECTER_IRET_STRING,
	SPECTER_IRET_RAW
};

enum {
	SPECTER_RETF_VALID	= 0x0001, /* contains valid result */
	SPECTER_RETF_FREE	= 0x0002  /* ptr needs to be free()d */
};

#define SPECTER_IRET_NAME_LEN 32
typedef struct specter_iret {
	struct specter_iret *next;
	specter_input_t *plugin;
	int len;
	u_int16_t type; /* see above (SPECTER_IRET_...) */
	u_int16_t flags; /* see above (SPECTER_RETF_... */
	char name[SPECTER_IRET_NAME_LEN];
	union {
		u_int8_t	b;
		u_int8_t	ui8;
		u_int16_t	ui16;
		u_int32_t	ui32;
		u_int64_t	ui64;
		int8_t		i8;
		int16_t		i16;
		int32_t		i32;
		int64_t		i64;
		void		*ptr;
	} value;
} specter_iret_t;


/***********************************************************************
 *   PUBLIC INTERFACE
 ***********************************************************************/
int register_input(specter_input_t *plugin, specter_iret_t *iret, int num, int flags);
int register_output(specter_output_t *plugin, int flags);
specter_iret_t *find_iret(char *name);
config_entry_t *GET_CE(config_entry_t *ce, int x);


/***********************************************************************
 *   LOGGING FACILITY
 ***********************************************************************/
enum {
	SPECTER_DEBUG	= 1,	/* debugging information */
	SPECTER_INFO	= 3,
	SPECTER_NOTICE	= 5,	/* abnormal/unexpected condition */
	SPECTER_ERROR	= 7,	/* error condition, requires user action */
	SPECTER_FATAL	= 8	/* fatal, program aborted */
};

void __specter_log(int level, const char *file, const char *format, ...);
#define specter_log(level, format, args...) \
	__specter_log(level, __FILE__, format, ## args)


#endif /* specter.h */

