/*
/ Objects.cpp
/ miscellaneous ancillary classes implementation
/
/ version 1.2, 2008 October 9
/
/ Author: Sandro Furieri a-furieri@lqt.it
/
/ Copyright (C) 2008  Alessandro Furieri
/
/    This program is free software: you can redistribute it and/or modify
/    it under the terms of the GNU General Public License as published by
/    the Free Software Foundation, either version 3 of the License, or
/    (at your option) any later version.
/
/    This program is distributed in the hope that it will be useful,
/    but WITHOUT ANY WARRANTY; without even the implied warranty of
/    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/    GNU General Public License for more details.
/
/    You should have received a copy of the GNU General Public License
/    along with this program.  If not, see <http://www.gnu.org/licenses/>.
/
*/

#include "Classdef.h"

MyObject::MyObject(int type, wxString & name)
{
//
// constructor - TreeItemData
//
  Type = type;
  Name = name;
  Column = wxT("");
}

MyObject::MyObject(int type, wxString & name, wxString & column)
{
//
// constructor - TreeItemData
//
  Type = type;
  Name = name;
  Column = column;
}

void MyVariant::Copy(MyVariant * other)
{
// 
// transfers a BLOB value
//
  if (other->Type != MY_BLOB_VARIANT)
    return;
  if (!(other->Blob))
    return;
  if (Blob)
    delete[]Blob;
  Type = MY_BLOB_VARIANT;
  BlobSize = other->BlobSize;
  Blob = other->Blob;
  other->Type = MY_NULL_VARIANT;
  other->Blob = NULL;
  other->BlobSize = 0;
}

void MyVariant::Set(const unsigned char *text)
{
//
// sets a String value for this Variant value
//
  Type = MY_TXT_VARIANT;
  TxtValue = wxString::FromUTF8((const char *) text);
}

void MyVariant::Set(const void *blob, int size)
{
//
// sets a BLOB value for this Variant value
//
  if (size <= 0)
    return;
  Type = MY_BLOB_VARIANT;
  BlobSize = size;
  Blob = new unsigned char[size];
  memcpy(Blob, blob, size);
}

void MyRowVariant::Create(int cols)
{
//
// creating the row variant
//
  if (ColumnArray)
    delete[]ColumnArray;
  NumCols = cols;
  ColumnArray = new MyVariant[NumCols];
}

void MyRowVariant::Set(int col, sqlite3_int64 value)
{
//
// setting an Integer value for the Nth column
//
  MyVariant *var;
  if (col < 0 || col >= NumCols)
    return;
  var = ColumnArray + col;
  var->Set(value);
}

void MyRowVariant::Set(int col, double value)
{
//
// setting a Double value for the Nth column
//
  MyVariant *var;
  if (col < 0 || col >= NumCols)
    return;
  var = ColumnArray + col;
  var->Set(value);
}

void MyRowVariant::Set(int col, const unsigned char *value)
{
//
// setting a String value for the Nth column
//
  MyVariant *var;
  if (col < 0 || col >= NumCols)
    return;
  var = ColumnArray + col;
  var->Set(value);
}

void MyRowVariant::Set(int col, const void *blob, int size)
{
//
// setting a BLOB value for the Nth column
//
  MyVariant *var;
  if (col < 0 || col >= NumCols)
    return;
  var = ColumnArray + col;
  var->Set(blob, size);
}

MyVariant *MyRowVariant::GetColumn(int col)
{
//
// returns a Column Variant Value
//
  MyVariant *var;
  if (col < 0 || col >= NumCols)
    return NULL;
  var = ColumnArray + col;
  return var;
}

MyVariantList::MyVariantList()
{
//
// constructor - result set container
//
  NumCols = 0;
  ColumnName = NULL;
  First = NULL;
  Last = NULL;
}

MyVariantList::~MyVariantList()
{
//
// destructor - result set container
//
  MyRowVariant *el;
  MyRowVariant *elN;
  if (ColumnName)
    delete[]ColumnName;
  el = First;
  while (el)
    {
      elN = el->GetNext();
      delete el;
      el = elN;
    }
}

int MyVariantList::GetRows()
{
//
// counting how many rows are there
//
  int cnt = 0;
  MyRowVariant *el = First;
  while (el)
    {
      cnt++;
      el = el->GetNext();
    }
  return cnt;
}

MyRowVariant *MyVariantList::Add(int columns)
{
//
// adds a row into the result set
//
  if (!NumCols)
    {
      NumCols = columns;
      ColumnName = new wxString[NumCols];
    }
  MyRowVariant *el = new MyRowVariant(columns);
  if (!First)
    First = el;
  if (Last)
    Last->SetNext(el);
  Last = el;
  return el;
}

void MyVariantList::SetColumnName(int col, const char *name)
{
//
// storing a column name
//
  if (col < 0 || col >= NumCols)
    return;
  ColumnName[col] = wxString::FromUTF8(name);
}

wxString & MyVariantList::GetColumnName(int col)
{
//
// retrieving a column name
//
  return ColumnName[col];
}

MyBlobs::MyBlobs(int rows, int cols)
{
//
// constructor - a BLOB matrix
//
  int r;
  MyRowVariant *rowVar;
  NumRows = 0;
  NumCols = 0;
  Rows = NULL;
  if (rows < 1 || cols < 1)
    return;
  NumRows = rows;
  NumCols = cols;
  Rows = new MyRowVariant[rows];
  for (r = 0; r < rows; r++)
    {
      rowVar = Rows + r;
      rowVar->Create(cols);
    }
}

MyBlobs::~MyBlobs()
{
//
// destructor - a BLOB matrix
//
  if (Rows)
    delete[]Rows;
}

void MyBlobs::SetBlob(int row, int col, MyVariant * org)
{
//
// setting a BLOB value
//
  MyRowVariant *rowVar;
  MyVariant *dest;
  if (row < 0 || row >= NumRows)
    return;
  if (col < 0 || col >= NumCols)
    return;
  rowVar = Rows + row;
  if (!rowVar)
    return;
  dest = rowVar->GetColumn(col);
  if (!dest)
    return;
  dest->Copy(org);
}

MyVariant *MyBlobs::GetBlob(int row, int col)
{
//
// return a BLOB value
//
  MyRowVariant *rowVar;
  MyVariant *dest;
  if (row < 0 || row >= NumRows)
    return NULL;
  if (col < 0 || col >= NumCols)
    return NULL;
  rowVar = Rows + row;
  if (!rowVar)
    return NULL;
  dest = rowVar->GetColumn(col);
  if (!dest)
    return NULL;
  if (dest->GetType() != MY_BLOB_VARIANT)
    return NULL;
  return dest;
}

MyValues::MyValues(int rows, int cols)
{
//
// constructor - a generic values matrix
//
  int r;
  MyRowVariant *rowVar;
  NumRows = 0;
  NumCols = 0;
  Rows = NULL;
  if (rows < 1 || cols < 1)
    return;
  NumRows = rows;
  NumCols = cols;
  Rows = new MyRowVariant[rows];
  for (r = 0; r < rows; r++)
    {
      rowVar = Rows + r;
      rowVar->Create(cols);
    }
}

MyValues::~MyValues()
{
//
// destructor - a generic values matrix
//
  if (Rows)
    delete[]Rows;
}

void MyValues::SetValue(int row, int col, sqlite3_int64 value)
{
//
// setting an integer value
//
  MyRowVariant *rowVar;
  MyVariant *dest;
  if (row < 0 || row >= NumRows)
    return;
  if (col < 0 || col >= NumCols)
    return;
  rowVar = Rows + row;
  if (!rowVar)
    return;
  dest = rowVar->GetColumn(col);
  if (!dest)
    return;
  dest->Set(value);
}

void MyValues::SetValue(int row, int col, double value)
{
//
// setting a double value
//
  MyRowVariant *rowVar;
  MyVariant *dest;
  if (row < 0 || row >= NumRows)
    return;
  if (col < 0 || col >= NumCols)
    return;
  rowVar = Rows + row;
  if (!rowVar)
    return;
  dest = rowVar->GetColumn(col);
  if (!dest)
    return;
  dest->Set(value);
}

void MyValues::SetValue(int row, int col, wxString & value)
{
//
// setting a string value
//
  MyRowVariant *rowVar;
  MyVariant *dest;
  if (row < 0 || row >= NumRows)
    return;
  if (col < 0 || col >= NumCols)
    return;
  rowVar = Rows + row;
  if (!rowVar)
    return;
  dest = rowVar->GetColumn(col);
  if (!dest)
    return;
  dest->Set(value);
}

MyVariant *MyValues::GetValue(int row, int col)
{
//
// return a generic value
//
  MyRowVariant *rowVar;
  MyVariant *dest;
  if (row < 0 || row >= NumRows)
    return NULL;
  if (col < 0 || col >= NumCols)
    return NULL;
  rowVar = Rows + row;
  if (!rowVar)
    return NULL;
  dest = rowVar->GetColumn(col);
  if (!dest)
    return NULL;
  return dest;
}

MyRowVariant *MyValues::GetRow(int row)
{
//
// return a row of generic values
//
  MyRowVariant *rowVar;
  if (row < 0 || row >= NumRows)
    return NULL;
  rowVar = Rows + row;
  return rowVar;
}

MyColumnInfo::MyColumnInfo(wxString & name, bool pkey)
{
//
// constructor - a table column object
//
  Name = name;
  PrimaryKey = pkey;
  Geometry = false;
  GeometryIndex = false;
  MbrCache = false;
  Next = NULL;
}

MyIndexInfo::MyIndexInfo(wxString & name)
{
//
// constructor - a table index object
//
  Name = name;
  Next = NULL;
}

MyTriggerInfo::MyTriggerInfo(wxString & name)
{
//
// constructor - a table trigger object
//
  Name = name;
  Next = NULL;
}

MyTableInfo::~MyTableInfo()
{
//
// destructor - a table columns collection
//
  MyColumnInfo *elc;
  MyColumnInfo *elcN;
  MyIndexInfo *eli;
  MyIndexInfo *eliN;
  MyTriggerInfo *elt;
  MyTriggerInfo *eltN;
  elc = FirstColumn;
  while (elc)
    {
      elcN = elc->GetNext();
      delete elc;
      elc = elcN;
    }
  eli = FirstIndex;
  while (eli)
    {
      eliN = eli->GetNext();
      delete eli;
      eli = eliN;
    }
  elt = FirstTrigger;
  while (elt)
    {
      eltN = elt->GetNext();
      delete elt;
      elt = eltN;
    }
}

void MyTableInfo::AddColumn(wxString & name, bool pkey)
{
//
// inserting a column def into a table
//
  MyColumnInfo *el = new MyColumnInfo(name, pkey);
  if (!FirstColumn)
    FirstColumn = el;
  if (LastColumn)
    LastColumn->SetNext(el);
  LastColumn = el;
}

void MyTableInfo::SetGeometry(wxString & name, bool index, bool cached)
{
//
//setting a geometry column
//
  MyColumnInfo *elc;
  elc = FirstColumn;
  while (elc)
    {
      if (name == elc->GetName())
        {
          elc->SetGeometry();
          if (index == true)
            elc->SetGeometryIndex();
          if (cached == true)
            elc->SetMbrCache();
        }
      elc = elc->GetNext();
    }
}

void MyTableInfo::AddIndex(wxString & name)
{
//
// inserting an index def into a table
//
  MyIndexInfo *el = new MyIndexInfo(name);
  if (!FirstIndex)
    FirstIndex = el;
  if (LastIndex)
    LastIndex->SetNext(el);
  LastIndex = el;
}

void MyTableInfo::AddTrigger(wxString & name)
{
//
// inserting a trigger def into a table
//
  MyTriggerInfo *el = new MyTriggerInfo(name);
  if (!FirstTrigger)
    FirstTrigger = el;
  if (LastTrigger)
    LastTrigger->SetNext(el);
  LastTrigger = el;
}

MyViewInfo::~MyViewInfo()
{
//
// destructor - a view columns collection
//
  MyColumnInfo *elc;
  MyColumnInfo *elcN;
  elc = First;
  while (elc)
    {
      elcN = elc->GetNext();
      delete elc;
      elc = elcN;
    }
}

void MyViewInfo::AddColumn(wxString & name)
{
//
// inserting a column def into a view
//
  MyColumnInfo *el = new MyColumnInfo(name, false);
  if (!First)
    First = el;
  if (Last)
    Last->SetNext(el);
  Last = el;
}

MySqlHistory::~MySqlHistory()
{
//
// destructor - the SQL queries history
//
  MySqlQuery *elq;
  MySqlQuery *elqN;
  elq = First;
  while (elq)
    {
      elqN = elq->GetNext();
      delete elq;
      elq = elqN;
    }
}

void MySqlHistory::Add(wxString & sql)
{
//
// inserting an SQL query into the history
//
  if (Last)
    {
      // avoiding stupid duplicates
      if (Last->GetSql() == sql)
        return;
    }
  MySqlQuery *el = new MySqlQuery(sql);
  if (!First)
    First = el;
  el->SetPrev(Last);
  if (Last)
    Last->SetNext(el);
  Last = el;
  Current = el;
}

MySqlQuery *MySqlHistory::GetNext()
{
//
// return the next SQL query
//
  if (Current)
    {
      if (Current->GetNext())
        {
          Current = Current->GetNext();
          return Current;
      } else
        return NULL;
    }
  return NULL;
}

MySqlQuery *MySqlHistory::GetPrev()
{
//
// return the previous SQL query
//
  if (Current)
    {
      if (Current->GetPrev())
        {
          Current = Current->GetPrev();
          return Current;
      } else
        return NULL;
    }
  return NULL;
}

bool MySqlHistory::TestNext()
{
//
// tests if the next SQL query exists
//
  if (Current)
    {
      if (Current->GetNext())
        return true;
      else
        return false;
    }
  return false;
}

bool MySqlHistory::TestPrev()
{
//
// tests if the previous SQL query exists
//
  if (Current)
    {
      if (Current->GetPrev())
        return true;
      else
        return false;
    }
  return false;
}

AutoFDOTables::~AutoFDOTables()
{
//
// destructor - auto FDO-OGR wrapper linked list
//
  AutoFDOTable *el;
  AutoFDOTable *elN;
  el = First;
  while (el)
    {
      elN = el->GetNext();
      delete el;
      el = elN;
    }
}

void AutoFDOTables::Add(const char *name, const int len)
{
//
// adding a table name to the auto FDO-OGR wrapper linked list
//
  AutoFDOTable *el = new AutoFDOTable(name, len);
  if (!First)
    First = el;
  if (Last)
    Last->SetNext(el);
  Last = el;
}
