/*
/ AuxCurl.cpp
/ checking and notifying updated versions
/
/ version 2.1, 2018 August 1
/
/ Author: Sandro Furieri a.furieri@lqt.it
/
/ Copyright (C) 2018  Alessandro Furieri
/
/    This program is free software: you can redistribute it and/or modify
/    it under the terms of the GNU General Public License as published by
/    the Free Software Foundation, either version 3 of the License, or
/    (at your option) any later version.
/
/    This program is distributed in the hope that it will be useful,
/    but WITHOUT ANY WARRANTY; without even the implied warranty of
/    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/    GNU General Public License for more details.
/
/    You should have received a copy of the GNU General Public License
/    along with this program.  If not, see <http://www.gnu.org/licenses/>.
/
*/

#include <stdlib.h>
#include <memory.h>
#include <string.h>

#include <curl/curl.h>

#include "config.h"

#ifdef SPATIALITE_AMALGAMATION
#include <spatialite/sqlite3.h>
#else
#include <sqlite3.h>
#endif

#include <spatialite.h>

#include "AuxCurl.h"

typedef struct curlMemBufferStruct
{
/* a struct handling a dynamically growing output buffer */
  unsigned char *Buffer;
  size_t WriteOffset;
  size_t BufferSize;
  int Error;
} curlMemBuffer;
typedef curlMemBuffer *curlMemBufferPtr;

static void curlMemBufferInitialize(curlMemBufferPtr buf)
{
// initializing a dynamically growing output buffer 
  buf->Buffer = NULL;
  buf->WriteOffset = 0;
  buf->BufferSize = 0;
  buf->Error = 0;
}

static void curlMemBufferReset(curlMemBufferPtr buf)
{
// cleaning a dynamically growing output buffer
  if (buf->Buffer)
    free(buf->Buffer);
  buf->Buffer = NULL;
  buf->WriteOffset = 0;
  buf->BufferSize = 0;
  buf->Error = 0;
}

static void
curlMemBufferAppend(curlMemBufferPtr buf, const unsigned char *payload,
                    size_t size)
{
// appending into the buffer
  size_t free_size = buf->BufferSize - buf->WriteOffset;
  if (size > free_size)
    {
      // we must allocate a bigger buffer
      size_t new_size;
      unsigned char *new_buf;
      if (buf->BufferSize == 0)
        new_size = size + 1024;
      else if (buf->BufferSize <= 4196)
        new_size = buf->BufferSize + size + 4196;
      else if (buf->BufferSize <= 65536)
        new_size = buf->BufferSize + size + 65536;
      else
        new_size = buf->BufferSize + size + (1024 * 1024);
      new_buf = (unsigned char *) malloc(new_size);
      if (!new_buf)
        {
          buf->Error = 1;
          return;
        }
      if (buf->Buffer)
        {
          memcpy(new_buf, buf->Buffer, buf->WriteOffset);
          free(buf->Buffer);
        }
      buf->Buffer = new_buf;
      buf->BufferSize = new_size;
    }
  memcpy(buf->Buffer + buf->WriteOffset, payload, size);
  buf->WriteOffset += size;
}

static size_t
curlStoreData(char *ptr, size_t size, size_t nmemb, void *userdata)
{
// updating the dynamic buffer
  size_t total = size * nmemb;
  curlMemBufferAppend((curlMemBufferPtr) userdata, (unsigned char *) ptr,
                      total);
  return total;
}

static void
curlCheckHttpHeader(curlMemBufferPtr buf, int *http_status, char **http_code)
{
// checking the HTTP header 
  unsigned char *p_in;
  unsigned char *base_status;
  unsigned char *base_code;
  int size_status = 0;
  int size_code = 0;
  char *tmp;

  *http_status = -1;
  *http_code = NULL;
  if (buf->Buffer == NULL)
    return;
  if (buf->WriteOffset < 10)
    return;
  if (memcmp(buf->Buffer, "HTTP/1.1 ", 9) != 0
      && memcmp(buf->Buffer, "HTTP/1.0 ", 9) != 0)
    return;

// attempting to retrieve the HTTP status 
  p_in = buf->Buffer + 9;
  base_status = p_in;
  while ((size_t) (p_in - buf->Buffer) < buf->WriteOffset)
    {
      if (*p_in == ' ')
        break;
      size_status++;
      p_in++;
    }
  if (size_status <= 0)
    return;
  tmp = (char *) malloc(size_status + 1);
  memcpy(tmp, base_status, size_status);
  *(tmp + size_status) = '\0';
  *http_status = atoi(tmp);
  free(tmp);

// attempting to retrieve the HTTP code 
  p_in = buf->Buffer + 10 + size_status;
  base_code = p_in;
  while ((size_t) (p_in - buf->Buffer) < buf->WriteOffset)
    {
      if (*p_in == '\r')
        break;
      size_code++;
      p_in++;
    }
  if (size_code <= 0)
    return;
  tmp = (char *) malloc(size_code + 1);
  memcpy(tmp, base_code, size_code);
  *(tmp + size_code) = '\0';
  *http_code = tmp;
}

extern char *GetUpdateVersion()
{
//
// checking if there is an updated version
//
  CURL *curl = NULL;
  CURLcode res;
  int http_status;
  char *http_code;
  char *text = NULL;
  curlMemBuffer headerBuf;
  curlMemBuffer bodyBuf;
  char *request;
  int windows;
#ifdef _WIN32
  windows = 1;
#else
  windows = 0;
#endif
  request =
    sqlite3_mprintf
    ("http://www.gaia-gis.it/cgi-bin/splitegui_update?windows=%d&cpu=%s&version=%s",
     windows, spatialite_target_cpu(), VERSION);
  curl = curl_easy_init();

  if (curl)
    {
      /* setting the URL */
      curl_easy_setopt(curl, CURLOPT_URL, request);

      /* no progress meter please */
      curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1L);
      /* setting the output callback function */
      curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curlStoreData);

      /* initializes the buffers */
      curlMemBufferInitialize(&headerBuf);
      curlMemBufferInitialize(&bodyBuf);
      curl_easy_setopt(curl, CURLOPT_WRITEHEADER, &headerBuf);
      curl_easy_setopt(curl, CURLOPT_WRITEDATA, &bodyBuf);

      /* Perform the request, res will get the return code */
      res = curl_easy_perform(curl);
      /* Check for errors */
      if (res != CURLE_OK)
        {
          fprintf(stderr, "CURL error: %s\n", curl_easy_strerror(res));
          goto stop;
        }

      /* verifying the HTTP status code */
      curlCheckHttpHeader(&headerBuf, &http_status, &http_code);
      if (http_status != 200)
        {
          fprintf(stderr, "Invalid HTTP status code: %d %s\n",
                  http_status, http_code);
          if (http_code != NULL)
            free(http_code);
          goto stop;
        }
      if (http_code != NULL)
        free(http_code);

      curlMemBufferReset(&headerBuf);
      text = (char *) malloc(bodyBuf.WriteOffset + 1);
      strcpy(text, (const char *) (bodyBuf.Buffer));

    stop:
      curlMemBufferReset(&headerBuf);
      curlMemBufferReset(&bodyBuf);
      curl_easy_cleanup(curl);
    }

  sqlite3_free(request);
  return text;
}


extern bool DoDownloadUpdatedPackage(const char *download_url,
                                     unsigned char **data, int *data_len)
{
//
// downloading the updated version package
//
  CURL *curl = NULL;
  CURLcode res;
  int http_status;
  char *http_code;
  curlMemBuffer headerBuf;
  curlMemBuffer bodyBuf;
  bool retcode = false;

  *data = NULL;
  *data_len = 0;
  curl = curl_easy_init();

  if (curl)
    {
      /* setting the URL */
      curl_easy_setopt(curl, CURLOPT_URL, download_url);

      /* no progress meter please */
      curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1L);
      /* setting the output callback function */
      curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curlStoreData);

      /* initializes the buffers */
      curlMemBufferInitialize(&headerBuf);
      curlMemBufferInitialize(&bodyBuf);
      curl_easy_setopt(curl, CURLOPT_WRITEHEADER, &headerBuf);
      curl_easy_setopt(curl, CURLOPT_WRITEDATA, &bodyBuf);

      /* Perform the request, res will get the return code */
      res = curl_easy_perform(curl);
      /* Check for errors */
      if (res != CURLE_OK)
        {
          fprintf(stderr, "CURL error: %s\n", curl_easy_strerror(res));
          goto stop;
        }

      /* verifying the HTTP status code */
      curlCheckHttpHeader(&headerBuf, &http_status, &http_code);
      if (http_status != 200)
        {
          fprintf(stderr, "Invalid HTTP status code: %d %s\n",
                  http_status, http_code);
          if (http_code != NULL)
            free(http_code);
          goto stop;
        }
      if (http_code != NULL)
        free(http_code);

      curlMemBufferReset(&headerBuf);
      *data_len = bodyBuf.WriteOffset;
      *data = (unsigned char *) malloc(*data_len);
      memcpy(*data, (const unsigned char *) (bodyBuf.Buffer), *data_len);
      retcode = true;

    stop:
      curlMemBufferReset(&headerBuf);
      curlMemBufferReset(&bodyBuf);
      curl_easy_cleanup(curl);
    }

  return retcode;
}
