#   SparkleShare, an instant update workflow to Git.
#   Copyright (C) 2010  Hylke Bons <hylkebons@gmail.com>
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import shutil
import time
import urllib

# http://projects.gnome.org/nautilus-python/documentation/html/
from gi.repository import Nautilus, GObject, Gtk, Gdk

SPARKLESHARE_PATH = os.path.join (os.path.expanduser ('~'), "SparkleShare")

import gettext
gettext.bindtextdomain('sparkleshare', '/usr/local/share/locale')
gettext.textdomain('sparkleshare')
_ = gettext.gettext

class SparkleShareExtension (GObject.GObject, Nautilus.MenuProvider):


    def __init__ (self):
        pass

    def checkout_version (self, menu, file_path, commit_hash, username, timestamp):

        file_name = os.path.basename (file_path)
        tmp_file_path = os.path.join (SPARKLESHARE_PATH, ".tmp", file_name)

        # Move the current version to a temporary path
        shutil.move (file_path, tmp_file_path)

        # Check out the earlier version
        os.chdir (os.path.dirname (file_path))
        os.popen ("git checkout " + commit_hash + " '" + file_name + "'")

        new_tmp_file_name = file_name + " (" + username + ", "
        new_tmp_file_name += time.strftime ("%H:%M %d %b %Y", timestamp).replace (" 0", " ") + ") "

        # Rename the checked out file
        shutil.move (file_name, new_tmp_file_name)

        # Move the original file back
        shutil.move (tmp_file_path, file_path)

        return True

    def format_web_link (self, path):
        # Get the remote url used for the repo
        self.chdir_to_repo_base(path)
        url_command = os.popen ("git config --get remote.origin.url")
        origin_url = url_command.readline ().strip ()
        if not origin_url:
            return
        
        # Get components
        # TODO use regex here or something not so ugly
        protocol, remaining = origin_url.split ("://", 1)
        host, origin_path = remaining.split("@")[1].split("/", 1);
        # Format the right web url depending on the service
        repo_base = self.get_repo_base_path(path)
        relative_path = path.split(repo_base, 1)[1].lstrip("/")

        #url = url.rstrip (".git")
        if "gitorious.org" in host:
            # ssh://git@gitorious.org/gnome-design/gnome-design.git
            # http://gitorious.org/gnome-design/gnome-design/blobs/raw/master/COPYING
            url = "http://" + host + "/" + urllib.quote(origin_path.rstrip(".git")) + "/blobs/master/" + urllib.quote(relative_path)
        elif "github.com" in host:
            # ssh://git@github.com/hbons/SparkleShare.git
            # https://raw.github.com/hbons/SparkleShare/master/README
            url = "http://raw.github.com/" + urllib.quote(origin_path.rstrip(".git")) + "/raw/master/" + urllib.quote(relative_path)
        else:
            # https://git.one-gear.com/?p=thansen/Public.git;a=blob;f=SparkleShare.txt;hb=HEAD
            url = "http://" + host + "/?p=" + urllib.quote(origin_path) +";a=blob;f=" + urllib.quote(relative_path) + ";hb=HEAD"

        return url

    def copy_web_link (self, menu, path):
        url = self.format_web_link(path)
        clipboard = Gtk.Clipboard.get (Gdk.Atom.intern ("CLIPBOARD", False))
        clipboard.set_text (url, -1)
        clipboard.store ()

        return
   
    def chdir_to_repo_base(self, file_path):
        base_path = self.get_repo_base_path(file_path)
        os.chdir(base_path)
        
    def get_repo_base_path(self, path):
        path = os.path.abspath(path)
        parts = path.split(SPARKLESHARE_PATH, 1)[1].split("/")
        if len(parts) > 1:
            sub_folder = parts[1]
        else:
            sub_folder = parts[0]
        return SPARKLESHARE_PATH + "/" + sub_folder

    def get_file_items (self, window, files):

		# Only work if one file is selected
        if len (files) != 1:
            return

        file_reference = files [0]

		# Only work if we're in a SparkleShare repository folder
        if file_reference.is_directory ():
            return
        if not (file_reference.get_parent_uri ().startswith ('file://' + SPARKLESHARE_PATH)):
            return
        if file_reference.get_parent_uri () == 'file://' + SPARKLESHARE_PATH:
            return

        file_path = urllib.unquote ('/' + file_reference.get_uri ().lstrip('file:/'))
        url = self.format_web_link (file_path)
        parent_path = os.path.dirname (os.path.abspath (file_path))
        
        top_menuitem = Nautilus.MenuItem (name="Nautilus::SparkleShare",
                                            label="SparkleShare")
        
        top_submenu = Nautilus.Menu ()
        top_menuitem.set_submenu (top_submenu)
        
        web_link_menu_item_copy = Nautilus.MenuItem (name="Nautilus::CopyWebLink",
                                                label=_("Copy Web Link"),
                                                tip=_("Copy the web address of this file to the clipboard"))

        web_link_menu_item_copy.connect ("activate", self.copy_web_link, file_path)
        

        epochs        = ["", "", "", "", "", "", "", "", "", ""]
        commit_hashes = ["", "", "", "", "", "", "", "", "", ""]


        time_command   = os.popen ("git log -10 --format='%at' '" + file_path + "'")

        author_command = os.popen ("git log -10 --format='%an' '" + file_path + "'")

        hash_command = os.popen ("git log -10 --format='%H' '" + file_path + "'")

        i = 0
        for line in time_command.readlines ():
            epochs [i] = line.strip ("\n")
            i += 1
        

        # Only work if there is history
        if i < 2:
            top_submenu.append_item (web_link_menu_item_copy)
            return [top_menuitem]

        i = 0
        for line in hash_command.readlines ():
            commit_hashes [i] = line.strip ("\n")
            i += 1

        earlier_version_menu_item = Nautilus.MenuItem (name="Nautilus::OpenOlderVersion",
                                                    label=_("Get Earlier Version"),
                                                    tip=_("Make a copy of an earlier version in this folder"))
        version_submenu = Nautilus.Menu ()

        i = 0
        for line in author_command.readlines ():

            if i > 0:

                timestamp = time.strftime ("%d %b\t%H:%M", time.localtime (float (epochs [i])))
                username = line.strip ("\n")

                menu_item = Nautilus.MenuItem (name="Nautilus::Version" + epochs [i],
                                           label=timestamp + "\t" + username,
                                           tip=_("Select to get a copy of this version"))

                menu_item.connect ("activate", self.checkout_version, file_path, commit_hashes [i],
                                   username, time.localtime (float (epochs [i])))
                version_submenu.append_item (menu_item)

            i += 1

        earlier_version_menu_item.set_submenu (version_submenu)
        top_submenu.append_item (earlier_version_menu_item)
        top_submenu.append_item (web_link_menu_item_copy)

        return [top_menuitem]
