-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Exceptions;
with SPARK.Ada.Strings.Unbounded.Not_SPARK;
with SPARK.Ada.Text_IO.Not_SPARK;

package body SPARK.Ada.Text_IO.Unbounded_String is
   --# hide SPARK.Ada.Text_IO.Unbounded_String;

   pragma Warnings ("Y"); -- Turn off warnings for Ada 2005 features

   procedure Create
     (File :    out Text_IO.File_Type;
      Mode : in     Text_IO.File_Mode;
      Name : in     Strings.Unbounded.Unbounded_String;
      Form : in     Strings.Unbounded.Unbounded_String) is
   begin
      Standard.Ada.Text_IO.Create
        (File => File.The_File_Type,
         Mode => SPARK.Ada.Text_IO.Not_SPARK.To_File_Mode (From => Mode),
         Name => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Name),
         Form => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Form));
      File.The_Exception := No_Exception;
   exception
      when Event : others =>
         File.The_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if File.The_Exception = No_Exception then
            raise;
         end if;
   end Create;

   procedure Open
     (File :    out Text_IO.File_Type;
      Mode : in     Text_IO.File_Mode;
      Name : in     Strings.Unbounded.Unbounded_String;
      Form : in     Strings.Unbounded.Unbounded_String) is
   begin
      Standard.Ada.Text_IO.Open
        (File => File.The_File_Type,
         Mode => SPARK.Ada.Text_IO.Not_SPARK.To_File_Mode (From => Mode),
         Name => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Name),
         Form => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Form));
      File.The_Exception := No_Exception;
   exception
      when Event : others =>
         File.The_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if File.The_Exception = No_Exception then
            raise;
         end if;
   end Open;

   function Name
     (File : Text_IO.File_Type) return Strings.Unbounded.Unbounded_String is
   begin
      return SPARK.Ada.Strings.Unbounded.String_To_Unbounded_String
        (Source => Standard.Ada.Text_IO.Name
           (File => File.The_File_Type));
   end Name;

   function Form
     (File : Text_IO.File_Type) return Strings.Unbounded.Unbounded_String is
   begin
      return SPARK.Ada.Strings.Unbounded.String_To_Unbounded_String
        (Source => Standard.Ada.Text_IO.Form
           (File => File.The_File_Type));
   end Form;

   procedure Put_File
     (File : in out Text_IO.File_Type;
      Item : in     Strings.Unbounded.Unbounded_String) is
   begin
      Standard.Ada.Text_IO.Put
        (File => File.The_File_Type,
         Item => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Item));
      File.The_Exception := No_Exception;
   exception
      when Event : others =>
         File.The_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if File.The_Exception = No_Exception then
            raise;
         end if;
   end Put_File;

   procedure Put_Output (Item : in Strings.Unbounded.Unbounded_String)
   is
   begin
      Standard.Ada.Text_IO.Put
        (File => Standard.Ada.Text_IO.Standard_Output,
         Item => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Item));
      The_Standard_Output_Exception := No_Exception;
   exception
      when Event : others =>
         The_Standard_Output_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if The_Standard_Output_Exception = No_Exception then
            raise;
         end if;
   end Put_Output;

   procedure Put_Error (Item : in Strings.Unbounded.Unbounded_String)
   is
   begin
      Standard.Ada.Text_IO.Put
        (File => Standard.Ada.Text_IO.Standard_Error,
         Item => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Item));
      The_Standard_Error_Exception := No_Exception;
   exception
      when Event : others =>
         The_Standard_Error_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if The_Standard_Error_Exception = No_Exception then
            raise;
         end if;
   end Put_Error;

   procedure Procedure_Get_Line_File
     (File     : in out Text_IO.File_Type;
      Item     :    out Strings.Unbounded.Unbounded_String;
      Arg_Last :    out Natural) is
      Tmp_Item : constant String := Standard.Ada.Text_IO.Get_Line
        (File => File.The_File_Type);
   begin
      Item     := SPARK.Ada.Strings.Unbounded.String_To_Unbounded_String
        (Source => Tmp_Item);
      Arg_Last := SPARK.Ada.Strings.Unbounded.Get_Length (Source => Item);
      File.The_Exception := No_Exception;
   exception
      when Event : others =>
         File.The_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if File.The_Exception = No_Exception then
            raise;
         end if;
   end Procedure_Get_Line_File;

   procedure Procedure_Get_Line_Input
     (Item     : out Strings.Unbounded.Unbounded_String;
      Arg_Last : out Natural) is
      Tmp_Item : constant String := Standard.Ada.Text_IO.Get_Line
        (File => Standard.Ada.Text_IO.Standard_Input);
   begin
      Item     := SPARK.Ada.Strings.Unbounded.String_To_Unbounded_String
        (Source => Tmp_Item);
      Arg_Last := SPARK.Ada.Strings.Unbounded.Get_Length (Source => Item);
      The_Standard_Input_Exception := No_Exception;
   exception
      when Event : others =>
         The_Standard_Input_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if The_Standard_Input_Exception = No_Exception then
            raise;
         end if;
   end Procedure_Get_Line_Input;

   function Function_Get_Line_File
     (File : Text_IO.File_Type) return Strings.Unbounded.Unbounded_String is
   begin
      return SPARK.Ada.Strings.Unbounded.String_To_Unbounded_String
        (Source => Standard.Ada.Text_IO.Get_Line
           (File => File.The_File_Type));
   end Function_Get_Line_File;

   function Function_Get_Line_Input
     return Strings.Unbounded.Unbounded_String is
   begin
      return SPARK.Ada.Strings.Unbounded.String_To_Unbounded_String
        (Source => Standard.Ada.Text_IO.Get_Line
           (File => Standard.Ada.Text_IO.Standard_Input));
   end Function_Get_Line_Input;

   procedure Put_Line_File (File : in out Text_IO.File_Type;
                            Item : in     Strings.Unbounded.Unbounded_String)
   is
   begin
      Standard.Ada.Text_IO.Put_Line
        (File => File.The_File_Type,
         Item => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Item));
      File.The_Exception := No_Exception;
   exception
      when Event : others =>
         File.The_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if File.The_Exception = No_Exception then
            raise;
         end if;
   end Put_Line_File;

   procedure Put_Line_Output (Item : in Strings.Unbounded.Unbounded_String)
   is
   begin
      Standard.Ada.Text_IO.Put_Line
        (File => Standard.Ada.Text_IO.Standard_Output,
         Item => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Item));
      The_Standard_Output_Exception := No_Exception;
   exception
      when Event : others =>
         The_Standard_Output_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if The_Standard_Output_Exception = No_Exception then
            raise;
         end if;
   end Put_Line_Output;

   procedure Put_Line_Error (Item : in Strings.Unbounded.Unbounded_String)
   is
   begin
      Standard.Ada.Text_IO.Put_Line
        (File => Standard.Ada.Text_IO.Standard_Error,
         Item => SPARK.Ada.Strings.Unbounded.Not_SPARK.To_String
           (Source => Item));
      The_Standard_Error_Exception := No_Exception;
   exception
      when Event : others =>
         The_Standard_Error_Exception := Get_Exception_T
           (The_Exception_Name => Standard.Ada.Exceptions.Exception_Name (X => Event));
         if The_Standard_Error_Exception = No_Exception then
            raise;
         end if;
   end Put_Line_Error;

end SPARK.Ada.Text_IO.Unbounded_String;
