-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Generic_Subprogram_Instantiation (Node  : in STree.SyntaxNode;
                                               Scope : in Dictionary.Scopes) is

   type Expected_Generic_Kind is (Generic_Procedure, Generic_Function);

   Next_Node : STree.SyntaxNode;

   procedure Check_Valid_Instantiation_Ident (Ident_Node : in     STree.SyntaxNode;
                                              Scope      : in     Dictionary.Scopes;
                                              Ok         :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Ok                         from Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   is
      Sym       : Dictionary.Symbol;
      Ident_Str : LexTokenManager.Lex_String;
   begin
      -- check that name is not already in use
      Ident_Str := Node_Lex_String (Node => Ident_Node);
      Sym       :=
        Dictionary.LookupItem (Name              => Ident_Str,
                               Scope             => Scope,
                               Context           => Dictionary.ProofContext,
                               Full_Package_Name => False);
      if Sym = Dictionary.NullSymbol then
         Ok := True;
      else
         Ok := False;
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
      end if;
   end Check_Valid_Instantiation_Ident;

   ----------------------------------------------------------

   procedure Check_Generic
     (Ident_Node  : in     STree.SyntaxNode;
      Scope       : in     Dictionary.Scopes;
      Kind        : in     Expected_Generic_Kind;
      Generic_Sym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Generic_Sym,
   --#         STree.Table                from Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         Kind,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table;
   --# pre Syntax_Node_Type (Ident_Node, STree.Table) = SP_Symbols.identifier;
   --# post STree.Table = STree.Table~;
   is
      -- Checks that the identifier after "is new" represents a visible generic unit of the
      -- appropriate kind.  Returns symbol of this generic unit if legal or a null symbol
      -- otherwise.
      Sym : Dictionary.Symbol;
   begin
      Sym :=
        Dictionary.LookupItem
        (Name              => Node_Lex_String (Node => Ident_Node),
         Scope             => Scope,
         Context           => Dictionary.ProgramContext,
         Full_Package_Name => False);
      -- Kludge Alert
      -- If I am honest, I don't really understand why this is needed.  We have looked
      -- up the generic in program context so I would expect to get the Ada function
      -- symbol.  A test case showed otherwise, hence this is in for now.  PNA 13/1/4
      if Dictionary.IsProofFunction (Sym) then
         Sym := Dictionary.GetAdaFunction (Sym);
      end if;

      -- Validate generic unit symbol
      if Sym = Dictionary.NullSymbol then
         ErrorHandler.Semantic_Error
           (Err_Num   => 1,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Node_Lex_String (Node => Ident_Node));
      elsif Dictionary.Is_Generic_Subprogram (The_Symbol => Sym) then
         if Kind = Generic_Procedure and then Dictionary.IsFunction (Sym) then
            -- wrong sort of subprogram
            ErrorHandler.Semantic_Error
              (Err_Num   => 631,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => LexTokenManager.Null_String);
            Sym := Dictionary.NullSymbol;
         elsif Kind = Generic_Function and then Dictionary.IsProcedure (Sym) then
            -- wrong sort of subprogram
            ErrorHandler.Semantic_Error
              (Err_Num   => 632,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => LexTokenManager.Null_String);
            Sym := Dictionary.NullSymbol;
         elsif not Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract, Sym) then
            -- right sort of subprogram, but generic declaration had errors
            ErrorHandler.Semantic_Warning
              (Err_Num  => 390,
               Position => Node_Position (Node => Ident_Node),
               Id_Str   => LexTokenManager.Null_String);
            Sym := Dictionary.NullSymbol;
         else
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => Ident_Node);
         end if;
      else -- not a generic subprgoram at all
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 630,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Sym       => Sym,
            Scope     => Scope);
         Sym := Dictionary.NullSymbol;
      end if;
      Generic_Sym := Sym;
   end Check_Generic;

   ----------------------------------------------------------

   procedure Process_Generic_Procedure_Instantiation (Node  : in STree.SyntaxNode;
                                                      Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_procedure_instantiation;
   --# post STree.Table = STree.Table~;
   is
      Generic_Sym, Instantiation_Sym : Dictionary.Symbol;
      Generic_Node                   : STree.SyntaxNode;
      Instantiation_Ident_Node       : STree.SyntaxNode;
      Errors_Found                   : Boolean;
      Ok                             : Boolean;
   begin
      Instantiation_Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
      -- ASSUME Instantiation_Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Instantiation_Ident_Node) = SP_Symbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Instantiation_Ident_Node = identifier in Process_Generic_Procedure_Instantiation");
      Check_Valid_Instantiation_Ident (Ident_Node => Instantiation_Ident_Node,
                                       Scope      => Scope,
                                       Ok         => Ok);
      if Ok then
         Generic_Node := Next_Sibling (Current_Node => Next_Sibling (Current_Node => Child_Node (Current_Node => Node)));
         -- ASSUME Generic_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Generic_Node) = SP_Symbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Generic_Node = identifier in Process_Generic_Procedure_Instantiation");
         Check_Generic (Ident_Node  => Generic_Node,
                        Scope       => Scope,
                        Kind        => Generic_Procedure,
                        Generic_Sym => Generic_Sym);

         if Generic_Sym /= Dictionary.NullSymbol then
            -- check parameters etc.
            -- add the instantiation
            Dictionary.AddSubprogramInstantiation
              (Name          => Node_Lex_String (Node => Instantiation_Ident_Node),
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Declaration   => Dictionary.Location'(Start_Position => Node_Position (Node => Instantiation_Ident_Node),
                                                     End_Position   => Node_Position (Node => Instantiation_Ident_Node)),
               TheGeneric    => Generic_Sym,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                     End_Position   => Node_Position (Node => Node)),
               Scope         => Scope,
               Context       => Dictionary.ProgramContext,
               Subprogram    => Instantiation_Sym);

            -- check parameters etc.
            Wf_Generic_Actual_Part
              (Generic_Node      => Generic_Node,
               Generic_Sym       => Generic_Sym,
               Instantiation_Sym => Instantiation_Sym,
               Scope             => Scope,
               Error_Found       => Errors_Found);
            if Errors_Found then
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Instantiation_Sym);
            else
               -- do formal/actual substitutions
               Dictionary.InstantiateSubprogramParameters
                 (GenericSubprogramSym => Generic_Sym,
                  ActualSubprogramSym  => Instantiation_Sym,
                  Comp_Unit            => ContextManager.Ops.Current_Unit,
                  Declaration          => Dictionary.Location'(Start_Position => Node_Position (Node => Instantiation_Ident_Node),
                                                               End_Position   => Node_Position (Node => Instantiation_Ident_Node)));
            end if;
         end if;
      end if;
   end Process_Generic_Procedure_Instantiation;

   ----------------------------------------------------

   procedure Process_Generic_Function_Instantiation (Node  : in STree.SyntaxNode;
                                                     Scope : in Dictionary.Scopes)

   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_function_instantiation;
   --# post STree.Table = STree.Table~;
   is
      Generic_Sym, Instantiation_Sym : Dictionary.Symbol;
      Instantiation_Ident_Node       : STree.SyntaxNode;
      Generic_Node                   : STree.SyntaxNode;
      Errors_Found                   : Boolean;
      Ok                             : Boolean;
   begin
      Instantiation_Ident_Node := Child_Node (Current_Node => Node);
      -- ASSUME Instantiation_Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Instantiation_Ident_Node) = SP_Symbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Instantiation_Ident_Node = identifier in Process_Generic_Function_Instantiation");
      Check_Valid_Instantiation_Ident (Ident_Node => Instantiation_Ident_Node,
                                       Scope      => Scope,
                                       Ok         => Ok);
      if Ok then
         Generic_Node := Next_Sibling (Current_Node => Instantiation_Ident_Node);
         -- ASSUME Generic_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Generic_Node) = SP_Symbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Generic_Node = identifier in Process_Generic_Function_Instantiation");
         Check_Generic (Ident_Node  => Generic_Node,
                        Scope       => Scope,
                        Kind        => Generic_Function,
                        Generic_Sym => Generic_Sym);
         if Generic_Sym /= Dictionary.NullSymbol then
            -- add the instantiation
            Dictionary.AddSubprogramInstantiation
              (Name          => Node_Lex_String (Node => Instantiation_Ident_Node),
               Comp_Unit     => ContextManager.Ops.Current_Unit,
               Declaration   => Dictionary.Location'(Start_Position => Node_Position (Node => Instantiation_Ident_Node),
                                                     End_Position   => Node_Position (Node => Instantiation_Ident_Node)),
               TheGeneric    => Generic_Sym,
               Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                     End_Position   => Node_Position (Node => Node)),
               Scope         => Scope,
               Context       => Dictionary.ProgramContext,
               Subprogram    => Instantiation_Sym);

            -- check parameters etc.
            Wf_Generic_Actual_Part
              (Generic_Node      => Generic_Node,
               Generic_Sym       => Generic_Sym,
               Instantiation_Sym => Instantiation_Sym,
               Scope             => Scope,
               Error_Found       => Errors_Found);

            if Errors_Found then
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Instantiation_Sym);
            else
               Dictionary.InstantiateSubprogramParameters
                 (GenericSubprogramSym => Generic_Sym,
                  ActualSubprogramSym  => Instantiation_Sym,
                  Comp_Unit            => ContextManager.Ops.Current_Unit,
                  Declaration          => Dictionary.Location'(Start_Position => Node_Position (Node => Instantiation_Ident_Node),
                                                               End_Position   => Node_Position (Node => Instantiation_Ident_Node)));
            end if;
         end if;
      end if;
   end Process_Generic_Function_Instantiation;

   ----------------------------------------------------

   procedure Process_Generic_Child_Function_Instantiation (Node  : in STree.SyntaxNode;
                                                           Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Aggregate_Stack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Aggregate_Stack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   --# pre Syntax_Node_Type (Node, STree.Table) = SP_Symbols.generic_child_function_instantiation;
   --# post STree.Table = STree.Table~;
   is
      Prefix_Sym, Generic_Sym, Instantiation_Sym : Dictionary.Symbol;
      Prefix_Node, Generic_Node                  : STree.SyntaxNode;
      Instantiation_Ident_Node                   : STree.SyntaxNode;
      Errors_Found                               : Boolean;
      Ok                                         : Boolean;
   begin
      Instantiation_Ident_Node := Child_Node (Current_Node => Node);
      -- ASSUME Instantiation_Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Instantiation_Ident_Node) = SP_Symbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Instantiation_Ident_Node = identifier in Process_Generic_Child_Function_Instantiation");
      Check_Valid_Instantiation_Ident (Ident_Node => Instantiation_Ident_Node,
                                       Scope      => Scope,
                                       Ok         => Ok);
      if Ok then
         -- check prefix, in practice the only thing acceptable here will be package Ada
         Prefix_Node := Next_Sibling (Current_Node => Instantiation_Ident_Node);
         -- ASSUME Prefix_Node = identifier
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Prefix_Node) = SP_Symbols.identifier,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Prefix_Node = identifier in Process_Generic_Child_Function_Instantiation");
         Prefix_Sym :=
           Dictionary.LookupItem
           (Name              => Node_Lex_String (Node => Prefix_Node),
            Scope             => Scope,
            Context           => Dictionary.ProgramContext,
            Full_Package_Name => False);
         if Prefix_Sym = Dictionary.NullSymbol then
            ErrorHandler.Semantic_Error
              (Err_Num   => 1,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Prefix_Node),
               Id_Str    => Node_Lex_String (Node => Prefix_Node));
         elsif not Dictionary.IsPackage (Prefix_Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 18,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Prefix_Node),
               Id_Str    => Node_Lex_String (Node => Prefix_Node));
         else -- potentially valid prefix
            Generic_Node := Next_Sibling (Current_Node => Prefix_Node);
            -- ASSUME Generic_Node = identifier
            SystemErrors.RT_Assert
              (C       => Syntax_Node_Type (Node => Generic_Node) = SP_Symbols.identifier,
               Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Generic_Node = identifier in Process_Generic_Child_Function_Instantiation");
            Check_Generic
              (Ident_Node  => Generic_Node,
               Scope       => Dictionary.VisibleScope (Prefix_Sym),
               Kind        => Generic_Function,
               Generic_Sym => Generic_Sym);
            if Generic_Sym /= Dictionary.NullSymbol then
               -- add the instantiation
               Dictionary.AddSubprogramInstantiation
                 (Name          => Node_Lex_String (Node => Instantiation_Ident_Node),
                  Comp_Unit     => ContextManager.Ops.Current_Unit,
                  Declaration   => Dictionary.Location'(Start_Position => Node_Position (Node => Instantiation_Ident_Node),
                                                        End_Position   => Node_Position (Node => Instantiation_Ident_Node)),
                  TheGeneric    => Generic_Sym,
                  Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                        End_Position   => Node_Position (Node => Node)),
                  Scope         => Scope,
                  Context       => Dictionary.ProgramContext,
                  Subprogram    => Instantiation_Sym);

               -- check parameters etc.
               Wf_Generic_Actual_Part
                 (Generic_Node      => Generic_Node,
                  Generic_Sym       => Generic_Sym,
                  Instantiation_Sym => Instantiation_Sym,
                  Scope             => Scope,
                  Error_Found       => Errors_Found);
               if Errors_Found then
                  Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract, Instantiation_Sym);
               else
                  STree.Set_Node_Lex_String (Sym  => Prefix_Sym,
                                             Node => Prefix_Node);
                  -- do formal/actual substitutions
                  Dictionary.InstantiateSubprogramParameters
                    (GenericSubprogramSym => Generic_Sym,
                     ActualSubprogramSym  => Instantiation_Sym,
                     Comp_Unit            => ContextManager.Ops.Current_Unit,
                     Declaration          =>
                       Dictionary.Location'(Start_Position => Node_Position (Node => Instantiation_Ident_Node),
                                            End_Position   => Node_Position (Node => Instantiation_Ident_Node)));
               end if;
            end if;
         end if;
      end if;
   end Process_Generic_Child_Function_Instantiation;

begin -- Wf_Generic_Subprogram_Instantiation
   Next_Node := Child_Node (Current_Node => Node);
   -- ASSUME Next_Node = generic_procedure_instantiation OR generic_function_instantiation OR
   --                    generic_child_function_instantiation
   if Syntax_Node_Type (Node => Next_Node) = SP_Symbols.generic_procedure_instantiation then
      -- ASSUME Next_Node = generic_procedure_instantiation
      Process_Generic_Procedure_Instantiation (Node  => Next_Node,
                                               Scope => Scope);
   elsif Syntax_Node_Type (Node => Next_Node) = SP_Symbols.generic_function_instantiation then
      -- ASSUME Next_Node = generic_function_instantiation
      Process_Generic_Function_Instantiation (Node  => Next_Node,
                                              Scope => Scope);
   elsif Syntax_Node_Type (Node => Next_Node) = SP_Symbols.generic_child_function_instantiation then
      -- ASSUME Next_Node = generic_child_function_instantiation
      Process_Generic_Child_Function_Instantiation (Node  => Next_Node,
                                                    Scope => Scope);
   else
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Next_Node = generic_procedure_instantiation OR generic_function_instantiation OR " &
           "generic_child_function_instantiation in Wf_Generic_Subprogram_Instantiation");
   end if;
end Wf_Generic_Subprogram_Instantiation;
