-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (SparkLex)
package body LineManager is

   procedure Clear_Line is
   begin
      Curr_Line :=
        Program_Line'
        (SparkLex.In_Ada,
         SparkLex.Start_Annotation,
         0,
         1,
         1,
         1,
         1,
         SparkLex.Line'(SparkLex.Line_Lengths => SparkLex.End_Of_Line));
   end Clear_Line;

   procedure Copy_Out_Line (Line : out Program_Line) is
   begin
      Line :=
        Program_Line'
        (Context        => SparkLex.In_Ada,
         Anno_Context   => SparkLex.Start_Annotation,
         Line_Length    => Curr_Line.Line_Length,
         Line_No        => Curr_Line.Line_No,
         Last_Token_Pos => Curr_Line.Last_Token_Pos,
         Curr_Pos       => Curr_Line.Last_Token_Pos,
         Lookahead_Pos  => Curr_Line.Lookahead_Pos,
         Conts          => Curr_Line.Conts);
   end Copy_Out_Line;

   procedure Copy_In_Line (Line : in Program_Line) is
   begin
      Curr_Line := Line;
   end Copy_In_Line;

   procedure Record_Curr_Pos is
   begin
      Curr_Line.Last_Token_Pos := Curr_Line.Curr_Pos;
   end Record_Curr_Pos;

   procedure Reset_Curr_Pos is
   begin
      Curr_Line.Curr_Pos      := Curr_Line.Last_Token_Pos;
      Curr_Line.Lookahead_Pos := Curr_Line.Last_Token_Pos;
   end Reset_Curr_Pos;

   procedure Inspect_Char (Ch : out Character) is
   begin
      Ch := Curr_Line.Conts (Curr_Line.Curr_Pos);
   end Inspect_Char;

   procedure Accept_Char is
   begin
      case Curr_Line.Conts (Curr_Line.Curr_Pos) is
         when SparkLex.End_Of_Text | SparkLex.End_Of_Line =>
            null;
         when others =>
            Curr_Line.Curr_Pos      := Curr_Line.Curr_Pos + 1;
            Curr_Line.Lookahead_Pos := Curr_Line.Curr_Pos;
      end case;
   end Accept_Char;

   procedure Lookahead_Char (Ch : out Character) is
   begin
      case Curr_Line.Conts (Curr_Line.Lookahead_Pos) is
         when SparkLex.End_Of_Text =>
            Ch := SparkLex.End_Of_Text;
         when SparkLex.End_Of_Line =>
            Ch := SparkLex.End_Of_Line;
         when others =>
            Curr_Line.Lookahead_Pos := Curr_Line.Lookahead_Pos + 1;
            Ch                      := Curr_Line.Conts (Curr_Line.Lookahead_Pos);
      end case;
   end Lookahead_Char;

   procedure Accept_Lookahead is
   begin
      case Curr_Line.Conts (Curr_Line.Lookahead_Pos) is
         when SparkLex.End_Of_Text | SparkLex.End_Of_Line =>
            null;
         when others =>
            Curr_Line.Lookahead_Pos := Curr_Line.Lookahead_Pos + 1;
      end case;
      Curr_Line.Curr_Pos := Curr_Line.Lookahead_Pos;
   end Accept_Lookahead;

   procedure Reject_Lookahead is
   begin
      Curr_Line.Lookahead_Pos := Curr_Line.Curr_Pos;
   end Reject_Lookahead;

   procedure Next_Sig_Char (Prog_Text : in SPARK_IO.File_Type) is
      Loc_Pos        : E_Strings.Lengths;
      Separator_Char : Boolean;

      procedure Get_Next_Line (Prog_Text : in SPARK_IO.File_Type)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out Curr_Line;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --# derives Curr_Line,
      --#         ErrorHandler.Error_Context,
      --#         LexTokenManager.State,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Curr_Line,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Prog_Text,
      --#                                         SPARK_IO.File_Sys;
      is
         Pos                : E_Strings.Lengths;
         End_Pos            : E_Strings.Lengths;
         Line               : E_Strings.T;
         Line_No_Before_Get : Positive;

         function Separator (Ch : Character) return Boolean is
            Result : Boolean;
         begin
            case Ch is
               when ' '                       |
                 Ada.Characters.Latin_1.HT |
                 Ada.Characters.Latin_1.VT |
                 Ada.Characters.Latin_1.CR |
                 Ada.Characters.Latin_1.LF |
                 Ada.Characters.Latin_1.FF =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
            return Result;
         end Separator;

         procedure Inc_Line_Number
         --# global in out Curr_Line;
         --# derives Curr_Line from *;
         is
         begin
            if Curr_Line.Line_No = LexTokenManager.Line_Numbers'Last then
               SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Too_Many_File_Lines,
                                         Msg     => "in SparkLex.LineManager");
            end if;
            Curr_Line.Line_No := Curr_Line.Line_No + 1;
         end Inc_Line_Number;

      begin
         End_Pos             := 0;
         Curr_Line.Conts (0) := ' ';
         loop
            if SPARK_IO.End_Of_Line (Prog_Text) then
               -- Skip empty line
               if SPARK_IO.End_Of_File (Prog_Text) then
                  End_Pos                   := 1;
                  Curr_Line.Conts (End_Pos) := SparkLex.End_Of_Text;
               else
                  SPARK_IO.Skip_Line (Prog_Text, 1);
                  Inc_Line_Number;
               end if;
            else
               -- Attempt to read the line
               Line_No_Before_Get := SPARK_IO.Line (Prog_Text);
               E_Strings.Get_Line (File  => Prog_Text,
                                   E_Str => Line);
               End_Pos := E_Strings.Get_Length (E_Str => Line);
               Inc_Line_Number;
               if End_Pos < 1 then -- Examiner bug - OK but not acccepted

                  -- Unable to read line, eight-bit character?
                  ErrorHandler.Lex_Error
                    (Error_Message    => "Line contains illegal character(s)",
                     Recovery_Message => "Ignored",
                     Error_Item       => LexTokenManager.Lex_Value'(Position  => LexTokenManager.Token_Position'(Start_Line_No => Curr_Line.Line_No,
                                                                                                                 Start_Pos     => 2),
                                                                    Token_Str => LexTokenManager.Null_String));
                  SPARK_IO.Skip_Line (Prog_Text, 1);
                  if Line_No_Before_Get = SPARK_IO.Line (Prog_Text) then
                     End_Pos                   := 1;
                     Curr_Line.Conts (End_Pos) := SparkLex.End_Of_Text;
                  end if;
               else
                  -- got a line!
                  for I in E_Strings.Positions range 1 .. End_Pos loop
                     Curr_Line.Conts (I) := E_Strings.Get_Element (E_Str => Line,
                                                                   Pos   => I);
                  end loop;
                  if End_Pos = E_Strings.Lengths'Last then
                     -- Line too long
                     if Line_No_Before_Get = SPARK_IO.Line (Prog_Text) then
                        -- skip unread bit of line.
                        SPARK_IO.Skip_Line (Prog_Text, 1);
                     end if;
                     End_Pos                   := End_Pos - 1;
                     Curr_Line.Conts (End_Pos) := SparkLex.Line_Too_Long;
                  end if;
               end if;
            end if;

            Curr_Line.Conts (End_Pos + 1) := SparkLex.End_Of_Line;
            -- Skip over leading seperators.
            Pos := 0;
            loop
               exit when not Separator (Ch => Curr_Line.Conts (Pos));
               exit when Pos = End_Pos;
               Pos := Pos + 1;
            end loop;

            exit when not Separator (Ch => Curr_Line.Conts (Pos));
            exit when Curr_Line.Conts (End_Pos) = SparkLex.End_Of_Text;
            exit when Curr_Line.Conts (End_Pos) = SparkLex.Line_Too_Long;
         end loop;

         -- Context sensitive annotation continuation check
         if Curr_Line.Context = SparkLex.In_Annotation then
            if End_Pos - Pos >= 2
              and then Curr_Line.Conts (Pos) = '-'
              and then Curr_Line.Conts (Pos + 1) = '-'
              and then Curr_Line.Conts (Pos + 2) = CommandLineData.Content.Anno_Char then
               null;
            else
               -- Insert end of annotation
               Pos                   := 0;
               Curr_Line.Conts (Pos) := SparkLex.End_Of_Annotation;
            end if;
         end if;

         Curr_Line.Line_Length   := End_Pos;
         Curr_Line.Curr_Pos      := Pos;
         Curr_Line.Lookahead_Pos := Pos;
      end Get_Next_Line;

   begin -- Next_Sig_Char
      Separator_Char := True;
      Loc_Pos        := Curr_Line.Curr_Pos;
      while Separator_Char loop
         case Curr_Line.Conts (Loc_Pos) is
            when SparkLex.End_Of_Line =>
               Get_Next_Line (Prog_Text => Prog_Text);
               Loc_Pos := Curr_Line.Curr_Pos;
            when ' '                       |
              Ada.Characters.Latin_1.HT |
              Ada.Characters.Latin_1.VT |
              Ada.Characters.Latin_1.LF |
              Ada.Characters.Latin_1.FF =>
               Loc_Pos := Loc_Pos + 1;  -- Skip separator
            when others =>
               Separator_Char := False;
         end case;
      end loop;
      Curr_Line.Last_Token_Pos := Loc_Pos;
      Curr_Line.Curr_Pos       := Loc_Pos;
      Curr_Line.Lookahead_Pos  := Loc_Pos;
   end Next_Sig_Char;

   procedure Set_Context (New_Context : in SparkLex.Program_Context) is
   begin
      Curr_Line.Context := New_Context;
   end Set_Context;

   procedure Set_Anno_Context (New_Context : in SparkLex.Annotation_Context) is
   begin
      Curr_Line.Anno_Context := New_Context;
   end Set_Anno_Context;

end LineManager;
