-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with ContextManager.Ops;
with VCG;

separate (Sem)
procedure CompUnit (Top_Node : in STree.SyntaxNode;
                    Do_VCG   : in Boolean) is
   type Annotation_Symbol_Table is array (Boolean) of SPSymbols.SPSymbol;

   type Visibility_Error_Hint is (No_Hint, In_Global_List, In_Derives_Import_List, In_Derives_Export_List, In_Suspends_List);

   type Tilde_Context is (Code, Precondition, Postcondition, FunctionReturn);
   subtype Anno_Tilde_Context is Tilde_Context range Precondition .. FunctionReturn;

   type Hidden_Class is (Not_Hidden, Handler_Hidden, All_Hidden);

   -------------------------------------------------------
   --# inherit CommandLineData,
   --#         ExaminerConstants,
   --#         Sem,
   --#         SystemErrors;
   package ExpStack is

      type ExpStackType is private;

      procedure Init (Stack : out ExpStackType);
      --# derives Stack from ;

      procedure Push (X     : in     Sem.Exp_Record;
                      Stack : in out ExpStackType);
      --# global in CommandLineData.Content;
      --# derives Stack from *,
      --#                    X &
      --#         null  from CommandLineData.Content;

      procedure Pop (Item  :    out Sem.Exp_Record;
                     Stack : in out ExpStackType);
      --# global in CommandLineData.Content;
      --# derives Item,
      --#         Stack from Stack &
      --#         null  from CommandLineData.Content;

      function Top (Stack : ExpStackType) return Sem.Exp_Record;

      function HasOneEntry (Stack : ExpStackType) return Boolean;

      function IsEmpty (Stack : ExpStackType) return Boolean;

   private

      subtype IndexRange is Integer range 1 .. ExaminerConstants.WalkExpStackMax;
      type StackArray is array (IndexRange) of Sem.Exp_Record;
      subtype TopRange is Integer range 0 .. ExaminerConstants.WalkExpStackMax;

      type ExpStackType is record
         S      : StackArray;
         TopPtr : TopRange;
      end record;

   end ExpStack;

   -------------------------------------------------------
   --# inherit Dictionary,
   --#         ExaminerConstants,
   --#         SPARK_IO,
   --#         SystemErrors;
   package TypeContextStack is

      type TStackType is private;

      procedure Init (Stack : out TStackType);
      --# derives Stack from ;

      procedure Push (X     : in     Dictionary.Symbol;
                      Stack : in out TStackType);
      --# derives Stack from *,
      --#                    X;

      procedure Pop (Stack : in out TStackType);
      --# derives Stack from *;

      function Top (Stack : in TStackType) return Dictionary.Symbol;

      function HasOneEntry (Stack : in TStackType) return Boolean;

   private

      subtype IndexRange is Integer range 1 .. ExaminerConstants.WalkExpStackMax;
      type StackArray is array (IndexRange) of Dictionary.Symbol;
      subtype TopRange is Integer range 0 .. ExaminerConstants.WalkExpStackMax;

      type TStackType is record
         S      : StackArray;
         TopPtr : TopRange;
      end record;

   end TypeContextStack;

   --------------------------------------------

   --# inherit CompleteCheck,
   --#         Dictionary,
   --#         ExaminerConstants,
   --#         Sem,
   --#         SPARK_IO,
   --#         SystemErrors;
   package AggregateStack
   --# own State;
   is
      procedure Init;
      --# global out State;
      --# derives State from ;

      procedure Push
        (TypeSym     : in Dictionary.Symbol;
         LowerBound  : in Sem.Typ_Type_Bound;
         UpperBound  : in Sem.Typ_Type_Bound;
         AggFlags    : in Sem.Typ_Agg_Flags;
         Counter     : in Natural;
         CompleteRec : in CompleteCheck.T);
      --# global in out State;
      --# derives State from *,
      --#                    AggFlags,
      --#                    CompleteRec,
      --#                    Counter,
      --#                    LowerBound,
      --#                    TypeSym,
      --#                    UpperBound;

      procedure Pop
        (TypeSym     : out Dictionary.Symbol;
         LowerBound  : out Sem.Typ_Type_Bound;
         UpperBound  : out Sem.Typ_Type_Bound;
         AggFlags    : out Sem.Typ_Agg_Flags;
         Counter     : out Natural;
         CompleteRec : out CompleteCheck.T);
      --# global in out State;
      --# derives AggFlags,
      --#         CompleteRec,
      --#         Counter,
      --#         LowerBound,
      --#         State,
      --#         TypeSym,
      --#         UpperBound  from State;

      -- In the analysis of array aggregates, the expression walker
      -- needs to know the expected type of the array index for the
      -- current aggregate.  To make this work for multi-dimensional
      -- aggregates, this has to be stored in the Aggregate Stack, and
      -- is accessed with the following function.
      function TopTypeSym return  Dictionary.Symbol;
      --# global in State;
   end AggregateStack;

   -----------------------------------------------------------------

   --# inherit ExaminerConstants,
   --#         SPARK_IO,
   --#         SystemErrors;
   package Stack
   --# own State;
   is

      procedure Init;
      --# global out State;
      --# derives State from ;

      procedure Push (X : in Boolean);
      --# global in out State;
      --# derives State from *,
      --#                    X;

      procedure Pop; -- (Item : out Boolean);
                     --# global in out State;
                     --# derives State from *;

      -- Note: this is not a proper function as it has a hidden
      --       side effect if a system error is raised
      function Top return Boolean;
      --# global in State;
   end Stack;

   --------------- Procedure variables -----------------------------

   Semantic_Error_In_Subprogram_Body  : Boolean;
   Data_Flow_Error_In_Subprogram_Body : Boolean;
   Unused_Data_Flow_Error_Flag        : Boolean;

   Stmt_Node, Last_Node, Next_Node : STree.SyntaxNode;
   NodeType                        : SPSymbols.SPSymbol;
   ErrStatus                       : ErrorHandler.Error_Level;
   Current_Scope                   : Dictionary.Scopes;
   TheHeap                         : Heap.HeapRecord;
   NodeTable                       : RefList.HashTable;
   GlobalComponentData             : ComponentManager.ComponentData;
   Unused                          : Dictionary.Symbol;

   -------------- Package bodies ------------------------------

   package body Stack is separate;

   package body ExpStack is separate;

   package body TypeContextStack is separate;

   package body AggregateStack is separate;

   --------------------------------------------
   --Routines for handling list of names
   --------------------------------------------

   procedure CreateNameList (List      :    out Lists.List;
                             HeapParam : in out Lists.List_Heap)
   --# derives HeapParam,
   --#         List      from HeapParam;
   is
      Ok : Boolean;

   begin
      Lists.New_List (Heap     => HeapParam,
                      The_List => List,
                      OK       => Ok);
      if not Ok then
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.List_Overflow_In_Expression,
                                   Msg     => "in CreateNameList");
      end if;
   end CreateNameList;

   --------------------------------------------

   procedure DisposeOfNameList (List      : in out Lists.List;
                                HeapParam : in out Lists.List_Heap)
   --# global in LexTokenManager.State;
   --# derives HeapParam from *,
   --#                        LexTokenManager.State,
   --#                        List &
   --#         List      from ;
   is
   begin
      Lists.Delete_List (Heap     => HeapParam,
                         The_List => List);
   end DisposeOfNameList;

   --------------------------------------------

   procedure AddName
     (Name      : in     LexTokenManager.Lex_String;
      List      : in     Lists.List;
      HeapParam : in out Lists.List_Heap;
      Present   :    out Boolean)
   --# global in LexTokenManager.State;
   --# derives HeapParam,
   --#         Present   from HeapParam,
   --#                        LexTokenManager.State,
   --#                        List,
   --#                        Name;
   is
      Ok : Boolean;
   begin
      Lists.Add_Name (Heap            => HeapParam,
                      The_List        => List,
                      Name            => Name,
                      Already_Present => Present,
                      Ok              => Ok);
      if not Ok then
         SystemErrors.Fatal_Error (Sys_Err => SystemErrors.List_Overflow_In_Expression,
                                   Msg     => "in AddName");
      end if;
   end AddName;

   ----------------------------------------------------------------------
   ---------------- Embedded subprograms ----------------------
   ----------------------------------------------------------------------

   -- The following procedures are used to handle view changes and other
   -- things associated with tagged types and inherited subprogram calls.
   -- They are declared here because they are used in wf_proc_call,
   -- wf_positional_argument_association, wf_named_argument_association,
   -- and wf_primary.

   function ConvertTaggedActual (Actual, TaggedParameterSym : Dictionary.Symbol) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   --  Function is only called if we know substitution is required
   is
      ActualLocal : Dictionary.Symbol;
   begin
      ActualLocal := Actual;
      loop
         -- normal exit: successful conversion
         exit when Dictionary.GetType (ActualLocal) = TaggedParameterSym;
         -- abnormal exit: can't follow inherited field pointers because type is hidden
         exit when Dictionary.GetFirstRecordSubcomponent (ActualLocal) = Dictionary.NullSymbol;
         ActualLocal := Dictionary.GetFirstRecordSubcomponent (ActualLocal);
      end loop;
      return ActualLocal;
   end ConvertTaggedActual;

   ---------------------------------------------------------------

   procedure AddRecordSubComponents
     (RecordVarSym  : in     Dictionary.Symbol;
      RecordTypeSym : in     Dictionary.Symbol;
      ComponentData : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ComponentData,
   --#         Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    ComponentData,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    RecordTypeSym,
   --#                                    RecordVarSym,
   --#                                    TheHeap &
   --#         null                  from CommandLineData.Content;
   is
      TypeComponent : Dictionary.Symbol;
      VarComponent  : Dictionary.Symbol;

      PrefixComponentNode : ComponentManager.Component;
      CurrentRecordVar    : Dictionary.Symbol;
      CurrentRecordType   : Dictionary.Symbol;

   begin
      CurrentRecordVar := RecordVarSym;

      CurrentRecordType := RecordTypeSym;
      if Dictionary.IsSubtype (CurrentRecordType) then
         CurrentRecordType := Dictionary.GetRootType (CurrentRecordType);
      end if;

      PrefixComponentNode := ComponentManager.GetComponentNode (ComponentData, CurrentRecordVar);
      -- for non tagged records the outer loop only runs once.  For extended tagged
      -- records we loop through all the inherited fields and add them too

      -- A private extended tagged type where the private part is hidden
      -- might result in PrefixComponentNode being Null here, in which
      -- case nothing can be done.
      loop
         for I in Natural range 1 .. Dictionary.GetNumberOfNonExtendedComponents (CurrentRecordType) loop
            TypeComponent := Dictionary.GetNonExtendedRecordComponent (CurrentRecordType, I);
            Dictionary.AddRecordSubcomponent
              (Prefix       => CurrentRecordVar,
               Component    => TypeComponent,
               Comp_Unit    => ContextManager.Ops.Current_Unit,
               Subcomponent => VarComponent);
            PrefixComponentNode := ComponentManager.GetComponentNode (ComponentData, CurrentRecordVar);
            ComponentManager.AddNextChild (ComponentData, TheHeap, PrefixComponentNode, VarComponent);
         end loop;
         CurrentRecordType := Dictionary.GetRootOfExtendedType (CurrentRecordType);
         -- stop when there are no more inherited fields
         exit when CurrentRecordType = Dictionary.NullSymbol;
         -- if the record has no components (perhaps it is a private type and the private
         -- part is hidden) then we cannot add any subcomponents
         exit when Dictionary.GetNumberOfNonExtendedComponents (CurrentRecordType) = 0;

         -- Simillarly, if the PrefixComponentNode is null, then we
         -- can't find its FirstChild, so we must be done.
         exit when ComponentManager.IsNullComponent (PrefixComponentNode);
         CurrentRecordVar :=
           ComponentManager.GetName (ComponentData, ComponentManager.GetFirstChild (ComponentData, PrefixComponentNode));
      end loop;

      ComponentManager.Dump_All_Component_Trees (Data => ComponentData);
   end AddRecordSubComponents;

   ---------------------------------------------------------------

   procedure TaggedActualMustBeObjectCheck
     (NodePos         : in     LexTokenManager.Token_Position;
      FormalType      : in     Dictionary.Symbol;
      ActualType      : in     Dictionary.Symbol;
      ControllingType : in     Dictionary.Symbol;
      IsAVariable     : in     Boolean;
      IsAConstant     : in     Boolean;
      ErrorFound      : in out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         ActualType,
   --#                                         ControllingType,
   --#                                         FormalType,
   --#                                         IsAConstant,
   --#                                         IsAVariable &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ActualType,
   --#                                         CommandLineData.Content,
   --#                                         ControllingType,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FormalType,
   --#                                         IsAConstant,
   --#                                         IsAVariable,
   --#                                         LexTokenManager.State,
   --#                                         NodePos,
   --#                                         SPARK_IO.File_Sys;
   is
   begin
      if (ControllingType /= Dictionary.NullSymbol)
        and then (FormalType = ControllingType)
        and then (ActualType /= ControllingType)
        and then -- but must be an extension of it (earlier check)
        not (IsAVariable or IsAConstant) then
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 827,
            Reference => ErrorHandler.No_Reference,
            Position  => NodePos,
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end TaggedActualMustBeObjectCheck;

   -----------------------------------------------------------------------------------

   -- Create_Implicit_Positive_Subtype extracted from PlantConstrainingType so that it can be
   -- called directly from wf_constant_declaration where it is used to created an implcit
   -- subtype for string constants.
   procedure Create_Implicit_Positive_Subtype
     (String_Length    : in     Maths.Value;
      Location         : in     Dictionary.Location;
      Index_Constraint :    out Dictionary.Symbol)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length &
   --#         Index_Constraint      from ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    String_Length;
   is
      Constraint_Str         : E_Strings.T;
      Constraint_Lex_Str     : LexTokenManager.Lex_String;
      Upper_Bound_Lex_Str    : LexTokenManager.Lex_String;
      Index_Constraint_Local : Dictionary.Symbol;
   begin
      -- Create an ExaminerString of the form "Positive__n" where n is the string length
      Constraint_Str := E_Strings.Copy_String (Str => "Positive__");
      -- The value of "n" will not exceed a size that can be printed within an ExaminerString
      -- so the conversion will not truncate here.
      E_Strings.Append_Examiner_String (E_Str1 => Constraint_Str,
                                        E_Str2 => Maths.ValueToString (String_Length));
      -- Insert this name into the string table; either we add it an get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      LexTokenManager.Insert_Examiner_String (Str     => Constraint_Str,
                                              Lex_Str => Constraint_Lex_Str);
      -- Look up type in Dictionary in case it has previously been added
      -- Note that we put these implicit subtype in Standard (where Positive itself lives)
      -- and that we declare them in proof rather than Ada context
      Index_Constraint_Local :=
        Dictionary.LookupItem
        (Name              => Constraint_Lex_Str,
         Scope             => Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard),
         Context           => Dictionary.ProofContext,
         Full_Package_Name => False);
      if Index_Constraint_Local = Dictionary.NullSymbol then
         -- not previously added, so we add a new subtype here
         Maths.StorageRep (String_Length, Upper_Bound_Lex_Str);
         Dictionary.AddIntegerSubtype
           (Name            => Constraint_Lex_Str,
            Static          => True,
            Parent          => Dictionary.GetPredefinedIntegerType,
            ParentReference => Location,
            Lower           => LexTokenManager.One_Value,
            Upper           => Upper_Bound_Lex_Str,
            Comp_Unit       => ContextManager.Ops.Current_Unit,
            Declaration     => Location,
            Scope           => Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard),
            Context         => Dictionary.ProofContext,
            Subtype_Symbol  => Index_Constraint_Local);
      end if;
      Index_Constraint := Index_Constraint_Local;
   end Create_Implicit_Positive_Subtype;

   -----------------------------------------------------------------------------------

   procedure Create_Implicit_String_Subtype
     (String_Length      : in     Maths.Value;
      Location           : in     Dictionary.Location;
      The_String_Subtype :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.File_Sys     from *,
   --#                                    CommandLineData.Content,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    String_Length &
   --#         The_String_Subtype    from ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    Location,
   --#                                    String_Length;
   is
      String_Subtype_Str       : E_Strings.T;
      String_Subtype_Lex_Str   : LexTokenManager.Lex_String;
      Index_Constraint         : Dictionary.Symbol;
      The_String_Subtype_Local : Dictionary.Symbol;
   begin
      -- First create an index subtype of the form positive__n
      Create_Implicit_Positive_Subtype
        (String_Length    => String_Length,
         Location         => Location,
         Index_Constraint => Index_Constraint);
      -- Create an ExaminerString of the form "String__n" where n is the string length
      String_Subtype_Str := E_Strings.Copy_String (Str => "String__");
      -- The value of "n" will not exceed a size that can be printed within an ExaminerString
      -- so the conversion will not truncate here.
      E_Strings.Append_Examiner_String (E_Str1 => String_Subtype_Str,
                                        E_Str2 => Maths.ValueToString (String_Length));
      -- Insert this name into the string table; either we add it an get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      -- Insert this name into the string table; either we add it and get the LexStr back or,
      -- if it is already there, we get the existing LexStr back
      LexTokenManager.Insert_Examiner_String (Str     => String_Subtype_Str,
                                              Lex_Str => String_Subtype_Lex_Str);
      -- Look up type in Dictionary in case it has previously been added
      -- Note that we put these implicit subtype in Standard (where Positive and String themselves live)
      -- and that we declare them in proof rather than Ada context
      The_String_Subtype_Local :=
        Dictionary.LookupItem
        (Name              => String_Subtype_Lex_Str,
         Scope             => Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard),
         Context           => Dictionary.ProofContext,
         Full_Package_Name => False);
      if The_String_Subtype_Local = Dictionary.NullSymbol then
         -- not previously added, so we add a new subtype here
         Dictionary.AddArraySubtype
           (Name            => String_Subtype_Lex_Str,
            Parent          => Dictionary.GetPredefinedStringType,
            ParentReference => Location,
            Comp_Unit       => ContextManager.Ops.Current_Unit,
            Declaration     => Location,
            Scope           => Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard),
            Context         => Dictionary.ProofContext,
            Static          => CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83,
            ArraySubtype    => The_String_Subtype_Local);
         Dictionary.AddArrayIndex
           (TheArrayType       => The_String_Subtype_Local,
            TheIndexType       => Index_Constraint,
            Comp_Unit          => ContextManager.Ops.Current_Unit,
            IndexTypeReference => Location);

      end if;
      The_String_Subtype := The_String_Subtype_Local;
   end Create_Implicit_String_Subtype;

   -----------------------------------------------------------------------------------

   -- PlantConstrainingType declared here because it used by wf_proc_call,
   -- wf_position_argument_association and wf_named_argument_association

   procedure PlantConstrainingType
     (ExpressionType : in Dictionary.Symbol;
      String_Length  : in Maths.Value;
      ActualNode     : in STree.SyntaxNode)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         SPARK_IO.File_Sys,
   --#         STree.Table           from *,
   --#                                    ActualNode,
   --#                                    ContextManager.Ops.Unit_Stack,
   --#                                    Dictionary.Dict,
   --#                                    ExpressionType,
   --#                                    LexTokenManager.State,
   --#                                    STree.Table,
   --#                                    String_Length &
   --#         LexTokenManager.State from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpressionType,
   --#                                    String_Length;
   is
      -- This procedure is used to plant a type symbol in the syntax tree, at the location of
      -- an actual parameter, for use by the VCG.  IFF the ExpressionType is String
      -- (indicating that the actual parameter is a string literal) then the parameter String_Length
      -- is used to create (or obtain if it's already declared) a subtype of Positive called Positive__n
      -- where n is the string length; this implicitly-declared subtype is then used as the constraint.
      -- For anything other than a string literal actual parameter we
      -- plant the symbol of the constraining array subtype.  The array subtype effectively passes
      -- the symbols of /all/ the constraining indexes however many dimensions the array has.

      TypeToPlant : Dictionary.Symbol := Dictionary.NullSymbol;

   begin -- PlantConstrainingType
      if Dictionary.IsUnconstrainedArrayType (Dictionary.GetRootType (ExpressionType))
        and then not Dictionary.IsUnconstrainedArrayType (ExpressionType) then

         -- Not a string, so plant the array type
         TypeToPlant := ExpressionType;
      elsif Dictionary.IsPredefinedStringType (ExpressionType) and String_Length /= Maths.NoValue then

         -- If the actual is a String Literal like "Hello World", then the ExpressionType
         -- will be Predefined String and String_Length will have a well-defined value.
         -- In this case, we create an implicit constraining subtype.
         Create_Implicit_Positive_Subtype
           (String_Length    => String_Length,
            Location         => Dictionary.Location'(Start_Position => Node_Position (Node => ActualNode),
                                                     End_Position   => Node_Position (Node => ActualNode)),
            Index_Constraint => TypeToPlant);
      end if;
      STree.AddNodeSymbol (ActualNode, TypeToPlant);
   end PlantConstrainingType;

   ----------------------------------------------------------------------
   -- FindPreviousPackage
   --
   -- Given a initial_declarative_item_rep node, we sometimes need to find
   -- if there's a embedded package declaration in the tree rooted at that
   -- node.  This function returns the LexString of that package if it
   -- can be found, or NullString if it can't.
   --
   -- This function is used by wf_renaming_declararation and
   -- wf_use_type_declaration, both of which need to locate such
   -- packages, so this function appears here so it can be called
   -- by both subunits.
   ----------------------------------------------------------------------

   function FindPreviousPackage (Node : STree.SyntaxNode) return LexTokenManager.Lex_String
   --# global in STree.Table;
   is
      Last_Node : STree.SyntaxNode;
      PackIdent : LexTokenManager.Lex_String;
   begin
      -- Phase 1 - search down the tree rooted at Node for a
      -- basic_declarative_item node or a package_declaration node
      Last_Node := Child_Node (Current_Node => Node);
      loop

         exit when Syntax_Node_Type (Node => Last_Node) /= SPSymbols.initial_declarative_item_rep;

         if Syntax_Node_Type (Node => Next_Sibling (Current_Node => Last_Node)) = SPSymbols.basic_declarative_item
           or else Syntax_Node_Type (Node => Next_Sibling (Current_Node => Last_Node)) = SPSymbols.package_declaration then
            -- If there's a basic_declarative_item or a package_declaration to the right
            -- of Last_Node, then set Last_Node to that node and exit.
            Last_Node := Next_Sibling (Current_Node => Last_Node);
         else
            -- No?  Then go down the tree and try again
            Last_Node := Child_Node (Current_Node => Last_Node);
         end if;
      end loop;

      -- Phase 2
      -- Last_Node should be a basic_declarative_item or a package_declaration
      case Syntax_Node_Type (Node => Last_Node) is
         when SPSymbols.basic_declarative_item =>
            -- No previous package, so return NullString
            PackIdent := LexTokenManager.Null_String;
         when SPSymbols.package_declaration =>

            -- If this package has an inherit clause, then skip over it
            Last_Node := Child_Node (Current_Node => Last_Node);
            if Syntax_Node_Type (Node => Last_Node) = SPSymbols.inherit_clause then
               Last_Node := Next_Sibling (Current_Node => Last_Node);
            end if;

            -- Find and return the package's identifier node
            Last_Node := Last_Child_Of (Start_Node => Last_Node);
            PackIdent := Node_Lex_String (Node => Last_Node);
         when others =>
            PackIdent := LexTokenManager.Null_String; -- to avoid flow error
            SystemErrors.Fatal_Error (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                                      Msg     => "in WFF.CU.FindPreviousPackage");
      end case;
      return PackIdent;
   end FindPreviousPackage;

   ----------------------------------------------------------------------

   -- Returns whether a subprogram_implementation node has a fully hidden body,
   -- a hidden handler part, or no hiding at all.  This code depends on the grammar
   -- in SPARK.LLA section 6.3
   function Body_Hidden_Class (Node : in STree.SyntaxNode) return Hidden_Class
   --# global in STree.Table;
   is
      Result          : Hidden_Class;
      Pragma_Rep_Node : STree.SyntaxNode;
      End_Node        : STree.SyntaxNode;
   begin
      -- ASSUME Node = subprogram_implementation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.subprogram_implementation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = subprogram_implementation in Body_Hidden_Class");
      Pragma_Rep_Node := Child_Node (Current_Node => Node);
      -- ASSUME Pragma_Rep_Node = pragma_rep
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Pragma_Rep_Node) = SPSymbols.pragma_rep,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Pragma_Rep_Node = pragma_rep in Body_Hidden_Class");
      -- if the Pragma_Rep is immediately followed by hidden part, then the whole
      -- body must be hidden.  If the PragmaRep is followed by something else, then
      -- we _might_ have a hidden handler part at End_Node.
      if Syntax_Node_Type (Node => Next_Sibling (Current_Node => Pragma_Rep_Node)) = SPSymbols.hidden_part then
         Result := All_Hidden;
      else
         End_Node := Last_Sibling_Of (Start_Node => Pragma_Rep_Node);
         -- ASSUME End_Node = designator OR hidden_part
         if Syntax_Node_Type (Node => End_Node) = SPSymbols.hidden_part then
            -- ASSUME End_Node = hidden_part
            Result := Handler_Hidden;
         elsif Syntax_Node_Type (Node => End_Node) = SPSymbols.designator then
            Result := Not_Hidden;
         else
            Result := Not_Hidden;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect End_Node = designator OR hidden_part in Body_Hidden_Class");
         end if;
      end if;
      return Result;
   end Body_Hidden_Class;

   ----------------------------------------------------------------------

   procedure wf_external_interface
     (Pragma_Node : in     STree.SyntaxNode;
      Entity_Sym  : in     Dictionary.Symbol;
      Error_Found :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------------------

   procedure wf_elaborate_body (PragmaNode : in STree.SyntaxNode;
                                PackSym    : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         PackSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         PackSym,
   --#                                         PragmaNode,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         PackSym,
   --#                                         PragmaNode;
      is separate;

   -----------------------------------------------------------------------------

   -- Checks Val against Typ'First .. Typ'Last.
   -- NewVal = Val if all OK or if arithmetic overflow raised a warning
   -- NewVal = Maths.NoValue if check performed and failed
   procedure ConstraintCheck
     (Val          : in     Maths.Value;
      NewVal       :    out Maths.Value;
      IsAnnotation : in     Boolean;
      Typ          : in     Dictionary.Symbol;
      Position     : in     LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Typ,
   --#                                         Val &
   --#         NewVal                     from Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Typ,
   --#                                         Val;
      is separate;

   --------------------------------------------------------------------------

   function Illegal_Unconstrained (LeftType, RightType : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;

   --------------------------------------------------------------------------

   function UnknownTypeRecord return Exp_Record
   --returns a "safe" exp result designed to minimise consequential errs
   --# global in Dictionary.Dict;
   is
   begin
      return Exp_Record'
        (Type_Symbol             => Dictionary.GetUnknownTypeMark,
         Other_Symbol            => Dictionary.NullSymbol,
         Stream_Symbol           => Dictionary.NullSymbol,
         Tagged_Parameter_Symbol => Dictionary.NullSymbol,
         Variable_Symbol         => Dictionary.NullSymbol,
         Param_Count             => 0,
         Param_List              => Lists.Null_List,
         Sort                    => Type_Result,
         Arg_List_Found          => False,
         Is_AVariable            => False,
         Is_An_Entire_Variable   => False,
         Errors_In_Expression    => True,
         Has_Operators           => False,
         Is_Static               => True,
         Is_Constant             => True,
         Is_ARange               => False,
         String_Value            => LexTokenManager.Null_String,
         Value                   => Maths.NoValue,
         Range_RHS               => Maths.NoValue);
   end UnknownTypeRecord;

   -------------------------------------------------------------------------

   procedure AssignmentCheck
     (Position   : in     LexTokenManager.Token_Position;
      Scope      : in     Dictionary.Scopes;
      TargetType : in     Dictionary.Symbol;
      ExpResult  : in out Exp_Record)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ExpResult,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         TargetType &
   --#         ExpResult                  from *,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         TargetType &
   --#         LexTokenManager.State      from *,
   --#                                         Dictionary.Dict,
   --#                                         ExpResult,
   --#                                         TargetType;
      is separate;

   --------------------------------------------------------------------------

   function IsEnclosingPackage (OuterPack : Dictionary.Symbol;
                                Scope     : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
      ScopeChain, OuterScope : Dictionary.Scopes;
      Result                 : Boolean;
   begin
      if OuterPack = Dictionary.GetRegion (Scope) then
         Result := True;
      else
         Result     := False;
         OuterScope := Dictionary.LocalScope (OuterPack);
         ScopeChain := Scope;
         while not Dictionary.IsGlobalScope (ScopeChain) loop
            ScopeChain := Dictionary.GetEnclosingScope (ScopeChain);
            if ScopeChain = OuterScope then
               Result := True;
               exit;
            end if;
         end loop;
      end if;

      return Result;
   end IsEnclosingPackage;

   -----------------------------------------------------------------------------------------

   procedure AddChild
     (RootIdNode : in     STree.SyntaxNode;
      IsPrivate  : in     Boolean;
      Scope      : in     Dictionary.Scopes;
      ChildSym   :    out Dictionary.Symbol;
      ChildStr   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ChildStr                   from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         RootIdNode,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ChildSym,
   --#         Dictionary.Dict            from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsPrivate,
   --#                                         LexTokenManager.State,
   --#                                         RootIdNode,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IsPrivate,
   --#                                         LexTokenManager.State,
   --#                                         RootIdNode,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsPrivate,
   --#                                         LexTokenManager.State,
   --#                                         RootIdNode,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IsPrivate,
   --#                                         LexTokenManager.State,
   --#                                         RootIdNode,
   --#                                         Scope;
   is
      CurrNode  : STree.SyntaxNode;
      CurrStr   : LexTokenManager.Lex_String;
      CurrSym   : Dictionary.Symbol;
      ParentSym : Dictionary.Symbol;
      ChildSort : Dictionary.PackageSort;
   begin
      CurrNode := RootIdNode;
      CurrStr  := Node_Lex_String (Node => CurrNode);
      CurrSym  := Dictionary.LookupItem (Name             => CurrStr,
                                         Scope            => Scope,
                                         Context          => Dictionary.ProofContext,
                                        Full_Package_Name => False);
      loop -- to handle multiple prefixes
         if CurrSym = Dictionary.NullSymbol then
            -- not declared or visible
            ParentSym := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 140,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => CurrNode),
               Id_Str    => CurrStr);
            exit;
         end if;

         if not Dictionary.IsPackage (CurrSym) then
            --can't be a parent
            CurrSym   := Dictionary.NullSymbol;
            ParentSym := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 18,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => CurrNode),
               Id_Str    => CurrStr);
            exit;
         end if;

         -- CurrStr (CurrSym) is visible and its a package
         STree.Set_Node_Lex_String (Sym  => CurrSym,
                                    Node => CurrNode);
         ParentSym := CurrSym;
         CurrNode  := Next_Sibling (Parent_Node (Current_Node => CurrNode));
         CurrStr   := Node_Lex_String (Node => CurrNode);
         CurrSym   :=
           Dictionary.LookupSelectedItem
           (Prefix   => CurrSym,
            Selector => CurrStr,
            Scope    => Scope,
            Context  => Dictionary.ProofContext);
         exit when Syntax_Node_Type (Node => Next_Sibling (Parent_Node (Current_Node => CurrNode))) /= SPSymbols.identifier;
         -- when no more identifier (s) to right
      end loop;

      if CurrSym /= Dictionary.NullSymbol then
         -- child already declared
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => CurrNode),
            Id_Str    => CurrStr);
      elsif ParentSym /= Dictionary.NullSymbol then
         -- check that CurrStr has not been declared as a body stub
         CurrSym :=
           Dictionary.LookupImmediateScope
           (Name    => CurrStr,
            Scope   => Dictionary.LocalScope (ParentSym),
            Context => Dictionary.ProgramContext);
         if CurrSym /= Dictionary.NullSymbol and then Dictionary.HasBodyStub (CurrSym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => CurrNode),
               Id_Str    => CurrStr);
         elsif CurrSym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => CurrSym,
                                       Node => CurrNode);
         end if;

         --check that CurrStr not inherited by parent of private child:
         CurrSym :=
           Dictionary.LookupItem
           (Name              => CurrStr,
            Scope             => Dictionary.VisibleScope (ParentSym),
            Context           => Dictionary.ProofContext,
            Full_Package_Name => False);
         if IsPrivate and then CurrSym /= Dictionary.NullSymbol then
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => CurrNode),
               Id_Str    => CurrStr);
         elsif CurrSym /= Dictionary.NullSymbol then
            STree.Set_Node_Lex_String (Sym  => CurrSym,
                                       Node => CurrNode);
         end if;
      end if;

      if ParentSym /= Dictionary.NullSymbol then
         if IsPrivate then
            ChildSort := Dictionary.PrivateChild;
         else
            ChildSort := Dictionary.Public;
         end if;
         Dictionary.AddChildPackage
           (TheParent     => ParentSym,
            Sort          => ChildSort,
            Name          => CurrStr,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            Specification => Dictionary.Location'(Start_Position => Node_Position (Node => RootIdNode),
                                                  End_Position   => Node_Position (Node => RootIdNode)),
            Scope         => Scope,
            ThePackage    => ChildSym);
      else
         -- Parent is not valid (i.e. undeclared or not a package) so we can't do any more.
         -- Signal failure back to caller with null child symbol
         ChildSym := Dictionary.NullSymbol;
         CurrStr  := LexTokenManager.Null_String;
      end if;
      ChildStr := CurrStr;
   end AddChild;

   -----------------------------------------------------------------------------------------

   procedure GetLiteralValue (Node : in     STree.SyntaxNode;
                              Val  :    out Maths.Value)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Val                        from LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;
   --------------------------------------------
   procedure GetCharacterLiteral (Node : in     STree.SyntaxNode;
                                  Val  :    out Maths.Value)
   --# global in LexTokenManager.State;
   --#        in STree.Table;
   --# derives Val from LexTokenManager.State,
   --#                  Node,
   --#                  STree.Table;
      is separate;
   --------------------------------------------

   procedure GetStringLiteralLength (Node : in     STree.SyntaxNode;
                                     Val  :    out Maths.Value)
   --# global in LexTokenManager.State;
   --#        in STree.Table;
   --# derives Val from LexTokenManager.State,
   --#                  Node,
   --#                  STree.Table;
      is separate;
   --------------------------------------------
   function GetObjectValue (Sym : Dictionary.Symbol) return Maths.Value
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;
   --------------------------------------------
   function GetEnumLitValue (Sym : Dictionary.Symbol) return Maths.Value
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;
   --------------------------------------------
   procedure CalcBinaryOperator
     (Node_Pos            : in     LexTokenManager.Token_Position;
      Operator            : in     SPSymbols.SPSymbol;
      Left_Val, Right_Val : in     Maths.Value;
      Is_Annotation       : in     Boolean;
      Result              : in out Exp_Record)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Is_Annotation,
   --#                                         Left_Val,
   --#                                         LexTokenManager.State,
   --#                                         Node_Pos,
   --#                                         Operator,
   --#                                         Result,
   --#                                         Right_Val,
   --#                                         SPARK_IO.File_Sys &
   --#         Result                     from *,
   --#                                         Dictionary.Dict,
   --#                                         Left_Val,
   --#                                         LexTokenManager.State,
   --#                                         Operator,
   --#                                         Right_Val;
      is separate;
   --------------------------------------------
   procedure CalcAttribute
     (Node         : in     STree.SyntaxNode;
      AttribName   : in     LexTokenManager.Lex_String;
      PrefixKind   : in     Dictionary.PrefixSort;
      Prefix       : in     Dictionary.Symbol;
      BaseFound    : in     Boolean;
      IsAnnotation : in     Boolean;
      Argument     : in out Maths.Value;
      RHSofRange   :    out Maths.Value)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Argument,
   --#         RHSofRange                 from Argument,
   --#                                         AttribName,
   --#                                         BaseFound,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Prefix &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Argument,
   --#                                         AttribName,
   --#                                         BaseFound,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Prefix,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         null                       from PrefixKind;
      is separate;

   --------------------------------------------------------------------

   function UnknownSymbolRecord return Exp_Record
   --# global in Dictionary.Dict;
      is separate;

   --------------------------------------------------------------------

   function SubstituteProtectedTypeSelfReference (Sym, PrefixSymbol : Dictionary.Symbol) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Symbol;
   begin
      -- if Sym is the implicitly-declared own variable of a protected type
      -- then we must replace it with the "current instance of the protected object"
      -- before checking whether it is visible.
      -- Background: given protected type PT its operations will globally reference and
      -- derive PT meaning, in this case, "myself".
      -- If an object PO of type PT (or a subtype of PT) is declared then calls to its
      -- operations will take the form PO.Op and the calling environment will be annotated
      -- in terms of PO.  Therefore, when checking that the globals necessary for the call
      -- PO.Op are visible (for example), we need to replace all references to PT into
      -- references to PO before making the check.  The Prefix Symbol of the call is the
      -- symbol we need to substitute in.
      Result := Sym;
      if Dictionary.IsOwnVariable (Sym) and then Dictionary.IsProtectedType (Dictionary.GetOwner (Sym)) then
         Result := PrefixSymbol;
      end if;
      return Result;
   end SubstituteProtectedTypeSelfReference;

   ---------------------------------------------------------------

   -- this function checks if the symbol passed is a stream variable or
   -- a function which globally accesses a stream variable.  If it is
   -- it returns the symbol otherwise it returns NullSymbol

   function StreamReferencesBy (Sym   : Dictionary.Symbol;
                                Scope : Dictionary.Scopes) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Symbol := Dictionary.NullSymbol;
      It     : Dictionary.Iterator;
   begin -- StreamReferencesBy
      if Dictionary.IsAdaFunction (Sym) then --IsAdaFunction used to block proof functions
         It := Dictionary.FirstGlobalVariable (Dictionary.GetAbstraction (Sym, Scope), Sym);
         while not Dictionary.IsNullIterator (It) loop
            if Dictionary.IsOwnVariableOrConstituentWithMode (Dictionary.CurrentSymbol (It)) then
               Result := Sym;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      else -- check for stream variable case
         if Dictionary.IsOwnVariableOrConstituentWithMode (Sym) then
            Result := Sym;
         end if;
      end if;
      return Result;
   end StreamReferencesBy;

   --------------------------------------------------------------------

   -- this function checks if the symbol passed is a protected variable or
   -- a function which globally accesses a protected variable.  If it is and
   -- the protected variable is not owned by the region we are calling from
   -- it returns the symbol otherwise it returns NullSymbol

   function ProtectedReferencesBy (Sym   : Dictionary.Symbol;
                                   Scope : Dictionary.Scopes) return Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result     : Dictionary.Symbol := Dictionary.NullSymbol;
      CurrentVar : Dictionary.Symbol;
      It         : Dictionary.Iterator;

      function IsLocalProtectedState (VarSym : Dictionary.Symbol;
                                      Scope  : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      -- pre IsOwnVariable (VarSym) and GetOwnVariableProtected (VarSym)
      -- returns True if the owner of VarSym is the protected type which also encloses the
      -- scope we are making the function call in
      is
         CompilationUnit : Dictionary.Symbol;

         function GetEnclosingPackageOrProtectedType (Scope : Dictionary.Scopes) return Dictionary.Symbol
         --# global in Dictionary.Dict;
         is
            Current : Dictionary.Scopes;
            Region  : Dictionary.Symbol;
         begin -- GetEnclosingPackageOrProtectedType
            Current := Scope;
            loop
               Region := Dictionary.GetRegion (Current);
               exit when Dictionary.IsPackage (Region);
               exit when Dictionary.IsProtectedType (Region);
               Current := Dictionary.GetEnclosingScope (Current);
            end loop;
            return Region;
         end GetEnclosingPackageOrProtectedType;

      begin -- IsLocalProtectedState
         CompilationUnit := GetEnclosingPackageOrProtectedType (Scope);
         return Dictionary.IsProtectedType (CompilationUnit) and then Dictionary.GetOwner (VarSym) = CompilationUnit;
      end IsLocalProtectedState;

   begin -- ProtectedReferencesBy
      if Dictionary.IsAdaFunction (Sym) then --IsAdaFunction used to block proof functions
         It := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, Sym);
         while not Dictionary.IsNullIterator (It) loop
            CurrentVar := Dictionary.CurrentSymbol (It);
            if Dictionary.IsOwnVariable (CurrentVar)
              and then Dictionary.GetOwnVariableProtected (CurrentVar)
              and then not Dictionary.IsOwnVariableOrConstituentWithMode (CurrentVar)
              and then not IsLocalProtectedState (CurrentVar, Scope) then
               Result := Sym;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      else -- check for pragma atomic protected variable case
         if Dictionary.IsOwnVariable (Sym)
           and then Dictionary.GetOwnVariableProtected (Sym)
           and then not Dictionary.IsOwnVariableOrConstituentWithMode (Sym) then
            Result := Sym;
         end if;
      end if;
      return Result;
   end ProtectedReferencesBy;

   --------------------------------------------------------------------

   procedure StackIdentifier
     (Sym          : in     Dictionary.Symbol;
      IdStr        : in     LexTokenManager.Lex_String;
      Node         : in     STree.SyntaxNode;
      Prefix       : in     Dictionary.Symbol;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      RefVar       : in     SeqAlgebra.Seq;
      Dotted       : in     Boolean;
      Context      : in     Tilde_Context;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         EStack                     from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Dotted,
   --#                                         IdStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Prefix,
   --#                                         Scope,
   --#                                         Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         Dictionary.Dict,
   --#                                         Dotted,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IdStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Prefix,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Dotted,
   --#                                         IdStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Prefix,
   --#                                         RefVar,
   --#                                         Scope,
   --#                                         Sym,
   --#                                         TheHeap &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IdStr,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Prefix,
   --#                                         Scope,
   --#                                         Sym;
      is separate;

   procedure wf_tilde
     (Node    : in     STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      EStack  : in out ExpStack.ExpStackType;
      Context : in     Anno_Tilde_Context)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Context,
   --#                                         Dictionary.Dict,
   --#                                         Scope;
      is separate;
   --------------------------------------------
   procedure wf_percent
     (Node   : in     STree.SyntaxNode;
      Scope  : in     Dictionary.Scopes;
      EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         Scope;
      is separate;
   --------------------------------------------
   procedure wf_identifier
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      RefVar       : in     SeqAlgebra.Seq;
      Context      : in     Tilde_Context;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         CommandLineData.Content,
   --#                                         Context,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         RefVar,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;
   --------------------------------------
   procedure Wf_Selected_Component
     (Node           : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      E_Stack        : in out ExpStack.ExpStackType;
      Ref_Var        : in     SeqAlgebra.Seq;
      Component_Data : in out ComponentManager.ComponentData;
      Context        : in     Tilde_Context)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Component_Data             from *,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         E_Stack                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Ref_Var,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -----------------------------------------------------------------------

   procedure wf_primary
     (Node          : in     STree.SyntaxNode;
      Scope         : in     Dictionary.Scopes;
      EStack        : in out ExpStack.ExpStackType;
      IsAnnotation  : in     Boolean;
      RefVar        : in     SeqAlgebra.Seq;
      ComponentData : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ComponentData,
   --#         Dictionary.Dict            from ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         RefVar,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope;
      is separate;

   -----------------------------------------------------------------------

   procedure wf_attribute (EStack : in out ExpStack.ExpStackType)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --# derives EStack from *,
   --#                     Dictionary.Dict &
   --#         null   from CommandLineData.Content;
      is separate;

   -----------------------------------------------------------------------

   procedure wf_attribute_designator
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean;
      RefVar       : in     SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         RefVar,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -----------------------------------------------------------------------

   procedure down_wf_name_argument_list
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      Next_Node    :    out STree.SyntaxNode;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         HeapParam                  from *,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         STree.Table &
   --#         Next_Node                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------------

   procedure up_wf_name_argument_list
     (Node         : in     STree.SyntaxNode;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         HeapParam                  from *,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   ----------------------------------------------------------------------

   procedure wf_positional_argument_association
     (Node          : in out STree.SyntaxNode;
      Scope         : in     Dictionary.Scopes;
      EStack        : in out ExpStack.ExpStackType;
      IsAnnotation  : in     Boolean;
      RefVar        : in     SeqAlgebra.Seq;
      ComponentData : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives ComponentData              from *,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         CommandLineData.Content,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         TheHeap &
   --#         LexTokenManager.State      from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Scope &
   --#         Node                       from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         RefVar,
   --#                                         TheHeap &
   --#         STree.Table                from *,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope;
      is separate;

   ----------------------------------------------------------------------

   -- This procedure is used in wf_named_argument_association and
   -- expressiontypefromcontext to find a named argument association
   -- parameter.  If NameIsParameterName returns True, then the
   -- identifier simple_name (below Node) is a legal parameter name
   -- for the given subprogram, and ParamSym denotes that parameter name.
   -- If NameIsParameterName return False, then the identifier is not
   -- a legal parameter names, and ParamSym is NullSymbol.

   procedure FindNamedArgumentAssociationParameter
     (Node                : in     STree.SyntaxNode;
      SubprogSym          : in     Dictionary.Symbol;
      IsAnnotation        : in     Boolean;
      NameIsParameterName :    out Boolean;
      ParamSym            :    out Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out STree.Table;
   --# derives NameIsParameterName,
   --#         ParamSym,
   --#         STree.Table         from Dictionary.Dict,
   --#                                  IsAnnotation,
   --#                                  LexTokenManager.State,
   --#                                  Node,
   --#                                  STree.Table,
   --#                                  SubprogSym;
   ---- pre Syntax_Node_Type (Node => Node) = SPSymbols.named_argument_association
   is
      IdentNode : STree.SyntaxNode;
      IdentStr  : LexTokenManager.Lex_String;
      It        : Dictionary.Iterator;
      Sym       : Dictionary.Symbol;

      function FindIdentifier (Node : STree.SyntaxNode) return STree.SyntaxNode
      --# global in IsAnnotation;
      --#        in STree.Table;
      is
         SIMPLE_LOOKUP : constant Annotation_Symbol_Table :=
           Annotation_Symbol_Table'(False => SPSymbols.simple_name,
                                    True  => SPSymbols.annotation_simple_name);
         R             : STree.SyntaxNode;
      begin
         if Syntax_Node_Type (Node => Child_Node (Node)) = SIMPLE_LOOKUP (IsAnnotation) then
            R := Child_Node (Child_Node (Node));
         else
            R := Child_Node (Next_Sibling (Child_Node (Node)));
         end if;
         return R;
      end FindIdentifier;

   begin
      IdentNode := FindIdentifier (Node);
      IdentStr  := Node_Lex_String (Node => IdentNode);

      NameIsParameterName := False;
      ParamSym            := Dictionary.NullSymbol;

      It := Dictionary.FirstSubprogramParameter (SubprogSym);
      while not Dictionary.IsNullIterator (It) loop
         Sym := Dictionary.CurrentSymbol (It);
         if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Dictionary.GetSimpleName (Sym),
                                                                 Lex_Str2 => IdentStr) =
           LexTokenManager.Str_Eq then
            NameIsParameterName := True;
            ParamSym            := Sym;
            STree.Set_Node_Lex_String (Sym  => Sym,
                                       Node => IdentNode);
            exit;
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;

   end FindNamedArgumentAssociationParameter;

   ----------------------------------------------------------------------

   procedure wf_named_argument_association
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         LexTokenManager.State      from Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         HeapParam                  from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure HomoImplTypeConv
     (Operator              : in     SPSymbols.SPSymbol;
      Left_Type, Right_Type : in out Dictionary.Symbol;
      Left_Val              : in     Maths.Value;
      Right_Val             : in     Maths.Value;
      Left_Has_Operators    : in     Boolean;
      Right_Has_Operators   : in     Boolean;
      Left_Pos              : in     LexTokenManager.Token_Position;
      Right_Pos             : in     LexTokenManager.Token_Position;
      Is_Annotation         : in     Boolean;
      T_Stack               : in     TypeContextStack.TStackType;
      Scope                 : in     Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Is_Annotation,
   --#                                         Left_Has_Operators,
   --#                                         Left_Pos,
   --#                                         Left_Type,
   --#                                         Left_Val,
   --#                                         LexTokenManager.State,
   --#                                         Operator,
   --#                                         Right_Has_Operators,
   --#                                         Right_Pos,
   --#                                         Right_Type,
   --#                                         Right_Val,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys &
   --#         Left_Type                  from *,
   --#                                         Dictionary.Dict,
   --#                                         Left_Has_Operators,
   --#                                         Operator,
   --#                                         Right_Type,
   --#                                         Scope,
   --#                                         T_Stack &
   --#         Right_Type                 from *,
   --#                                         Dictionary.Dict,
   --#                                         Left_Type,
   --#                                         Operator,
   --#                                         Right_Has_Operators,
   --#                                         Scope,
   --#                                         T_Stack;
      is separate;

   -----------------------------------------------------------------

   procedure CheckBinaryOperator
     (Operator      : in     SPSymbols.SPSymbol;
      Left          : in     Exp_Record;
      Right         : in     Exp_Record;
      Scope         : in     Dictionary.Scopes;
      T_Stack       : in     TypeContextStack.TStackType;
      Op_Pos        : in     LexTokenManager.Token_Position;
      Left_Pos      : in     LexTokenManager.Token_Position;
      Right_Pos     : in     LexTokenManager.Token_Position;
      Convert       : in     Boolean;
      Is_Annotation : in     Boolean;
      Result        : in out Exp_Record)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Convert,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Is_Annotation,
   --#                                         Left,
   --#                                         Left_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Operator,
   --#                                         Op_Pos,
   --#                                         Right,
   --#                                         Right_Pos,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         T_Stack &
   --#         Result                     from *,
   --#                                         CommandLineData.Content,
   --#                                         Convert,
   --#                                         Dictionary.Dict,
   --#                                         Is_Annotation,
   --#                                         Left,
   --#                                         Operator,
   --#                                         Right,
   --#                                         Scope,
   --#                                         T_Stack;
      is separate;

   -----------------------------------------------------------------

   function NullTypeRecord return Exp_Record
   --# global in Dictionary.Dict;
      is separate;

   -----------------------------------------------------------------

   function Ops_Are_Same_And_Commutative (Op1, Op2 : in SPSymbols.SPSymbol) return Boolean is
   begin
      return Op1 = Op2 and then (Op1 = SPSymbols.plus or else Op1 = SPSymbols.multiply);
   end Ops_Are_Same_And_Commutative;

   -----------------------------------------------------------------

   procedure Wf_Simple_Expression
     (Node                    : in     STree.SyntaxNode;
      Scope                   : in     Dictionary.Scopes;
      E_Stack                 : in out ExpStack.ExpStackType;
      T_Stack                 : in     TypeContextStack.TStackType;
      Context_Requires_Static : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context_Requires_Static,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         E_Stack                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         LexTokenManager.State      from *,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         Node,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         Node,
   --#                                         Scope,
   --#                                         T_Stack;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Simple_Expression_Opt
     (Node    : in     STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      E_Stack : in out ExpStack.ExpStackType;
      T_Stack : in     TypeContextStack.TStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         E_Stack                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         Node,
   --#                                         T_Stack;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Expression
     (Node    : in     STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      E_Stack : in out ExpStack.ExpStackType;
      T_Stack : in     TypeContextStack.TStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         E_Stack,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         T_Stack;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Term
     (Node                    : in     STree.SyntaxNode;
      Scope                   : in     Dictionary.Scopes;
      E_Stack                 : in out ExpStack.ExpStackType;
      T_Stack                 : in     TypeContextStack.TStackType;
      Context_Requires_Static : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context_Requires_Static,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         E_Stack                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         Node,
   --#                                         Scope,
   --#                                         T_Stack;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Factor
     (Node    : in     STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      E_Stack : in out ExpStack.ExpStackType;
      T_Stack : in     TypeContextStack.TStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         E_Stack                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         Node,
   --#                                         Scope,
   --#                                         T_Stack;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Relation
     (Node    : in     STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      E_Stack : in out ExpStack.ExpStackType;
      T_Stack : in     TypeContextStack.TStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         E_Stack                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         T_Stack &
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope;
      is separate;

   -----------------------------------------------------------------

   procedure wf_arange
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure down_wf_aggregate
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      Next_Node    :    out STree.SyntaxNode;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives AggregateStack.State       from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         HeapParam                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Next_Node                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure wf_ancestor_part
     (Node         : in out STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         HeapParam                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Node                       from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         EStack,
   --#                                         Node;
      is separate;

   -----------------------------------------------------------------

   function NullParameterRecord return Exp_Record
   --# global in Dictionary.Dict;
      is separate;

   -----------------------------------------------------------------

   procedure down_wf_aggregate_choice_rep
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      IsAnnotation : in     Boolean;
      Next_Node    :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         HeapParam,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Next_Node                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Record_Component_Selector_Name
     (Node       : in     STree.SyntaxNode;
      Scope      : in     Dictionary.Scopes;
      E_Stack    : in out ExpStack.ExpStackType;
      Heap_Param : in out Lists.List_Heap)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         Heap_Param,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         E_Stack,
   --#         Heap_Param                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         Heap_Param,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope;
      is separate;

   -----------------------------------------------------------------

   procedure up_wf_aggregate_choice
     (Node   : in     STree.SyntaxNode;
      Scope  : in     Dictionary.Scopes;
      EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives AggregateStack.State       from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from AggregateStack.State,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure up_wf_named_association_rep
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      HeapParam    : in out Lists.List_Heap;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         HeapParam                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         LexTokenManager.State,
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure up_wf_named_record_component_association
     (Node      : in     STree.SyntaxNode;
      Scope     : in     Dictionary.Scopes;
      EStack    : in out ExpStack.ExpStackType;
      HeapParam : in out Lists.List_Heap)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         HeapParam                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         HeapParam,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         LexTokenManager.State      from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack &
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         Node;
      is separate;

   -----------------------------------------------------------------

   procedure up_wf_positional_association (Node   : in     STree.SyntaxNode;
                                           EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure up_wf_aggregate_or_expression
     (Node         : in out STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         Node,
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure up_wf_component_association
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         LexTokenManager.State,
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure wf_positional_record_component_association
     (Node         : in out STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope &
   --#         LexTokenManager.State      from *,
   --#                                         Dictionary.Dict,
   --#                                         EStack &
   --#         Node,
   --#         STree.Table                from Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure up_wf_aggregate
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out AggregateStack.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives AggregateStack.State,
   --#         EStack                     from AggregateStack.State,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from AggregateStack.State,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Qualified_Expression
     (Node    : in     STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      E_Stack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         E_Stack,
   --#         LexTokenManager.State      from Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -------------------------------------------------------------------

   procedure down_wf_aggregate_or_expression
     (Node         : in     STree.SyntaxNode;
      EStack       : in out ExpStack.ExpStackType;
      IsAnnotation : in     Boolean;
      Next_Node    :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         Next_Node                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         IsAnnotation,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -------------------------------------------------------------------

   ---------------------------------------------------------------------------------
   -- The type context needed for an attribute with arguments (i.e. 'Val, 'Max etc)
   -- depends on the Prefix and the attribute identifier.  Some attributes have two
   -- arguments, so we need a special handler here, since an attribute_designator
   -- node is the closent common parent node of one or both arguments.
   ---------------------------------------------------------------------------------

   function AttributeDesignatorTypeFromContext
     (ExpNode      : in STree.SyntaxNode;
      IsAnnotation : in Boolean;
      EStack       : in ExpStack.ExpStackType;
      TStack       : in TypeContextStack.TStackType)
     return         Dictionary.Symbol
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
      is separate;

   ---------------------------------------------------------------------------------
   -- Returns the type needed for an expression from its context
   ---------------------------------------------------------------------------------

   procedure ExpressionTypeFromContext
     (Exp_Node         : in     STree.SyntaxNode;
      E_Stack          : in     ExpStack.ExpStackType;
      T_Stack          : in     TypeContextStack.TStackType;
      New_Context_Type :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out STree.Table;
   --# derives New_Context_Type from CommandLineData.Content,
   --#                               Dictionary.Dict,
   --#                               Exp_Node,
   --#                               E_Stack,
   --#                               LexTokenManager.State,
   --#                               STree.Table,
   --#                               T_Stack &
   --#         STree.Table      from *,
   --#                               CommandLineData.Content,
   --#                               Dictionary.Dict,
   --#                               Exp_Node,
   --#                               E_Stack,
   --#                               LexTokenManager.State;
      is separate;

   ---------------------------------------------------------------------------------
   -- Primary grammer productions nearly always have the same type context
   -- as their parent nodes, but there is a single exception - when a primary
   -- node is the right-hand operand of an exponentiation operator, where the
   -- context is always Integer.
   ---------------------------------------------------------------------------------

   function PrimaryTypeFromContext
     (Node   : in STree.SyntaxNode;
      TStack : in TypeContextStack.TStackType)
     return   Dictionary.Symbol
   --# global in Dictionary.Dict;
   --#        in STree.Table;
      is separate;

   ---------------------------------------------------------------------------------
   -- range constraints appear below aggregate_choice, case_choice, and below
   -- relational operators, and so can have a chage of type context.
   ---------------------------------------------------------------------------------

   function RangeConstraintTypeFromContext
     (ExpNode : in STree.SyntaxNode;
      EStack  : in ExpStack.ExpStackType;
      TStack  : in TypeContextStack.TStackType)
     return    Dictionary.Symbol
   --# global in STree.Table;
      is separate;

   ---------------------------------------------------------------------------------
   -- Simple expressions appear below aggregate_choice, case_choice, and below
   -- relational operators, and so can have a chage of type context.
   ---------------------------------------------------------------------------------

   function SimpleExpressionTypeFromContext
     (ExpNode : in STree.SyntaxNode;
      TStack  : in TypeContextStack.TStackType)
     return    Dictionary.Symbol
   --# global in AggregateStack.State;
   --#        in Dictionary.Dict;
   --#        in STree.Table;
      is separate;

   -------------------------------------------------------------------

   procedure WalkExpression
     (Exp_Node                : in     STree.SyntaxNode;
      Scope                   : in     Dictionary.Scopes;
      Type_Context            : in     Dictionary.Symbol;
      Context_Requires_Static : in     Boolean;
      Result                  :    out Exp_Record;
      Ref_Var                 : in     SeqAlgebra.Seq;
      Component_Data          : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Result,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         Type_Context &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Context_Requires_Static,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         Type_Context &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         Type_Context;
      is separate;

   --------------------------------------------------------------------------------------
   -- This procedure has essentially the functionality of WalkExpression
   -- but is used to walk expressions which the language rules require to be a
   -- "name".  In addition to the things returned by WalkExpression, a flag is
   -- returned indicating whether the expression was indeed a name.
   -- If the expression is not a name the expression is not traversed at all
   -- and Result is set to the UnknownTypeRecord
   --
   -- After the name node has been found it is traversed by WalkExpression and a subset
   -- of the checks usually done by wf_primary are acarried out on the result.  More
   -- checks are done here than are necessary for this application (e.g. getting bounds
   -- of a type mark) but they have been left in to make Walk_Name more widely applicable;
   -- it could be moved to Wellformed if ever needed elsewhere.

   procedure Walk_Name
     (Exp_Node       : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Component_Data : in out ComponentManager.ComponentData;
      Result         :    out Exp_Record;
      Is_A_Name      :    out Boolean;
      Ref_Var_Param  : in     SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var_Param,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var_Param,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Is_A_Name                  from Exp_Node,
   --#                                         STree.Table &
   --#         Result                     from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Ref_Var_Param,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      Node : STree.SyntaxNode;
      Sym  : Dictionary.Symbol;
   begin
      -- ASSUME Exp_Node = expression
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Exp_Node) = SPSymbols.expression,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Exp_Node = expression in Walk_Name");

      -- preset result for all non-name cases advance to relation node;
      Result := UnknownTypeRecord;
      Node   := Child_Node (Current_Node => Exp_Node);
      -- ASSUME Node = relation
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.relation,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = relation in Walk_Name");
      if Next_Sibling (Current_Node => Node) = STree.NullNode then
         -- ASSUME Next_Sibling (Current_Node => Node) = NULL
         -- advance to simple_expression;
         Node := Child_Node (Current_Node => Node);
         -- ASSUME Node = simple_expression
         SystemErrors.RT_Assert
           (C       => Syntax_Node_Type (Node => Node) = SPSymbols.simple_expression,
            Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = simple_expression in Walk_Name");
         if Next_Sibling (Current_Node => Node) = STree.NullNode then
            -- ASSUME Next_Sibling (Current_Node => Node) = NULL
            -- advance to simple_expression_opt;
            Node := Child_Node (Current_Node => Node);
            -- ASSUME Node = simple_expression OR simple_expression_opt
            if Syntax_Node_Type (Node => Node) = SPSymbols.simple_expression_opt then
               -- Node = simple_expression_opt
               -- advance to term;
               Node := Child_Node (Current_Node => Node);
               -- ASSUME Node = unary_adding_operator OR term
               if Syntax_Node_Type (Node => Node) = SPSymbols.term then
                  -- ASSUME Node = term
                  -- advance to factor;
                  Node := Child_Node (Current_Node => Node);
                  -- ASSUME Node = term OR factor
                  if Syntax_Node_Type (Node => Node) = SPSymbols.factor then
                     -- ASSUME Node = factor
                     -- advance to primary;
                     Node := Child_Node (Current_Node => Node);
                     -- ASSUME Node = primary OR RWabs OR RWnot
                     if Syntax_Node_Type (Node => Node) = SPSymbols.primary then
                        -- ASSUME Node = primary
                        if Next_Sibling (Current_Node => Node) = STree.NullNode then
                           -- ASSUME Next_Sibling (Current_Node => Node) = NULL
                           -- advance to name;
                           Node := Child_Node (Current_Node => Node);
                           -- ASSUME Node = numeric_literal OR character_literal OR string_literal OR name OR
                           --               qualified_expression OR expression OR attribute
                           if Syntax_Node_Type (Node => Node) = SPSymbols.name then
                              -- ASSUME Node = name
                              Is_A_Name := True;
                              WalkExpression
                                (Exp_Node                => Node,
                                 Scope                   => Scope,
                                 Type_Context            => Dictionary.GetUnknownTypeMark,
                                 Context_Requires_Static => False,
                                 Result                  => Result,
                                 Ref_Var                 => Ref_Var_Param,
                                 Component_Data          => Component_Data);
                              -- now perform some checks on the result akin to those
                              -- of wf_primary
                              Sym := Result.Other_Symbol;
                              case Result.Sort is
                                 when Is_Unknown =>
                                    Result := UnknownTypeRecord;
                                 when Type_Result =>
                                    null;
                                 when Is_Package =>
                                    Result := UnknownTypeRecord;
                                    ErrorHandler.Semantic_Error
                                      (Err_Num   => 5,
                                       Reference => ErrorHandler.No_Reference,
                                       Position  => Node_Position (Node => Node),
                                       Id_Str    => Dictionary.GetSimpleName (Sym));
                                 when Is_Function =>
                                    ErrorHandler.Semantic_Error
                                      (Err_Num   => 3,
                                       Reference => ErrorHandler.No_Reference,
                                       Position  => Node_Position (Node => Node),
                                       Id_Str    => Dictionary.GetSimpleName (Sym));
                                    Result.Is_Static   := False;
                                    Result.Is_Constant := False;
                                    Result.Is_ARange   := False;
                                 when Is_Object =>
                                    Result.Is_ARange := False;
                                    if Dictionary.IsVariableOrSubcomponent (Sym) then
                                       SeqAlgebra.AddMember (TheHeap, Ref_Var_Param, Natural (Dictionary.SymbolRef (Sym)));
                                    end if;
                                 when Is_Type_Mark =>
                                    Result.Is_Static   := Dictionary.IsStatic (Sym, Scope);
                                    Result.Is_Constant := True;
                                    if Dictionary.IsScalarTypeMark (Sym, Scope) then -- we can get bounds of range and so on
                                       Result.Is_ARange := True;
                                       -- get upper and lower bounds of type from dictionary
                                       Result.Value     :=
                                         Maths.ValueRep
                                         (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                                              LexTokenManager.First_Token, Sym));
                                       Result.Range_RHS :=
                                         Maths.ValueRep
                                         (Dictionary.GetScalarAttributeValue (False, -- don't want base type
                                                                              LexTokenManager.Last_Token, Sym));
                                    else -- not scalar so we don't do bounds and its not a range
                                       Result.Is_ARange := False;
                                    end if;
                                 when Is_Parameter_Name =>
                                    null; -- should never occur
                              end case;
                              Result.Param_Count := 0;
                              Result.Param_List  := Lists.Null_List;
                           elsif Syntax_Node_Type (Node => Node) = SPSymbols.numeric_literal
                             or else Syntax_Node_Type (Node => Node) = SPSymbols.character_literal
                             or else Syntax_Node_Type (Node => Node) = SPSymbols.string_literal
                             or else Syntax_Node_Type (Node => Node) = SPSymbols.qualified_expression
                             or else Syntax_Node_Type (Node => Node) = SPSymbols.expression
                             or else Syntax_Node_Type (Node => Node) = SPSymbols.attribute then
                              -- ASSUME Node = numeric_literal OR character_literal OR string_literal OR
                              --               qualified_expression OR expression OR attribute
                              Is_A_Name := False;
                           else
                              Is_A_Name := False;
                              SystemErrors.Fatal_Error
                                (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                                 Msg     => "Expect Node = numeric_literal OR character_literal OR string_literal OR name OR qualified_expression OR expression OR attribute in Walk_Name");
                           end if;
                        elsif Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.double_star then
                           -- ASSUME Next_Sibling (Current_Node => Node) = double_star
                           Is_A_Name := False;
                        else
                           Is_A_Name := False;
                           SystemErrors.Fatal_Error
                             (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                              Msg     => "Expect Next_Sibling (Current_Node => Node) = double_star OR NULL in Walk_Name");
                        end if;
                     elsif Syntax_Node_Type (Node => Node) = SPSymbols.RWabs
                       or else Syntax_Node_Type (Node => Node) = SPSymbols.RWnot then
                        -- ASSUME Node = RWabs OR RWnot
                        Is_A_Name := False;
                     else
                        Is_A_Name := False;
                        SystemErrors.Fatal_Error
                          (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                           Msg     => "Expect Node = primary OR RWabs OR RWnot in Walk_Name");
                     end if;
                  elsif Syntax_Node_Type (Node => Node) = SPSymbols.term then
                     -- ASSUME Node = term
                     Is_A_Name := False;
                  else
                     Is_A_Name := False;
                     SystemErrors.Fatal_Error
                       (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                        Msg     => "Expect Node = term OR factor in Walk_Name");
                  end if;
               elsif Syntax_Node_Type (Node => Node) = SPSymbols.unary_adding_operator then
                  -- ASSUME Node = unary_adding_operator
                  Is_A_Name := False;
               else
                  Is_A_Name := False;
                  SystemErrors.Fatal_Error
                    (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                     Msg     => "Expect Node = unary_adding_operator OR term in Walk_Name");
               end if;
            elsif Syntax_Node_Type (Node => Node) = SPSymbols.simple_expression then
               -- ASSUME Node = simple_expression
               Is_A_Name := False;
            else
               Is_A_Name := False;
               SystemErrors.Fatal_Error
                 (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
                  Msg     => "Expect Node = simple_expression OR simple_expression_opt in Walk_Name");
            end if;
         elsif Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.relational_operator
           or else Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.inside
           or else Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.outside then
            -- ASSUME Next_Sibling (Current_Node => Node) = relational_operator OR inside OR outside
            Is_A_Name := False;
         else
            Is_A_Name := False;
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Next_Sibling (Current_Node => Node) = relational_operator OR inside OR outside OR NULL in Walk_Name");
         end if;
      elsif Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.expression_rep1
        or else Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.expression_rep2
        or else Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.expression_rep3
        or else Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.expression_rep4
        or else Syntax_Node_Type (Node => Next_Sibling (Current_Node => Node)) = SPSymbols.expression_rep5 then
         -- ASSUME Next_Sibling (Current_Node => Node) = expression_rep1 OR expression_rep2 OR expression_rep3 OR
         --                                              expression_rep4 OR expression_rep5
         Is_A_Name := False;
      else
         Is_A_Name := False;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Next_Sibling (Current_Node => Node) = expression_rep1 OR expression_rep2 OR expression_rep3 OR expression_rep4 OR expression_rep5 OR NULL in Walk_Name");
      end if;
   end Walk_Name;

   ---------------------------------------------------------------------

   procedure CheckPriorityRange
     (Error_Sym   : in     Dictionary.Symbol;
      Scope       : in     Dictionary.Scopes;
      Pragma_Kind : in     Dictionary.RavenscarPragmas;
      Err_Pos     : in     LexTokenManager.Token_Position;
      Value       : in     Maths.Value;
      Value_Rep   :    out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Error_Sym,
   --#                                         Err_Pos,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Kind,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Value &
   --#         LexTokenManager.State,
   --#         Value_Rep                  from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Kind,
   --#                                         Value;
      is separate;

   ---------------------------------------------------------------------

   procedure CheckPriorityPropertyConsistency
     (Sym                   : in     Dictionary.Symbol;
      TypeSym               : in     Dictionary.Symbol;
      PriorityPropertyValue : in     LexTokenManager.Lex_String;
      ErrorNode             : in     STree.SyntaxNode;
      Consistent            :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         PriorityPropertyValue,
   --#                                         Sym,
   --#                                         TypeSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         LexTokenManager.State,
   --#                                         PriorityPropertyValue,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym,
   --#                                         TypeSym;
      is separate;

   ---------------------------------------------------------------------

   procedure CheckInterruptPropertyConsistency
     (HasInterruptProperty : in     Boolean;
      Sym                  : in     Dictionary.Symbol;
      TypeSym              : in     Dictionary.Symbol;
      ErrorNode            : in     STree.SyntaxNode;
      Consistent           :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         HasInterruptProperty,
   --#                                         TypeSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         HasInterruptProperty,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym,
   --#                                         TypeSym;
      is separate;

   ---------------------------------------------------------------------

   procedure CreateInterruptStreamVariable
     (ForPO             : in Dictionary.Symbol;
      TheHandler        : in LexTokenManager.Lex_String;
      TheStreamVariable : in LexTokenManager.Lex_String;
      ErrorNode         : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         ForPO,
   --#                                         LexTokenManager.State,
   --#                                         TheHandler,
   --#                                         TheStreamVariable &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         ForPO,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHandler,
   --#                                         TheStreamVariable;
      is separate;

   ---------------------------------------------------------------------

   procedure CreateInterruptStreamVariables (ForPO     : in Dictionary.Symbol;
                                             ErrorNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         ForPO,
   --#                                         LexTokenManager.State &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         ForPO,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ---------------------------------------------------------------------

   procedure CheckSuspendablePropertyConsistency
     (Sym              : in     Dictionary.Symbol;
      TypeSym          : in     Dictionary.Symbol;
      IsInSuspendsList : in     Boolean;
      ErrorNode        : in     STree.SyntaxNode;
      Consistent       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         IsInSuspendsList,
   --#                                         Sym,
   --#                                         TypeSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         IsInSuspendsList,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym,
   --#                                         TypeSym;
      is separate;

   ---------------------------------------------------------------------

   procedure CheckCeilingPriority
     (Sym              : in Dictionary.Symbol;
      Scope            : in Dictionary.Scopes;
      CheckList        : in Dictionary.Iterator;
      PriorityLexValue : in LexTokenManager.Lex_String;
      ErrorNode        : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CheckList,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         LexTokenManager.State,
   --#                                         PriorityLexValue,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym;
      is separate;

   ---------------------------------------------------------------------

   function UnexpectedInitialization (Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return -- not an initialized own variable
        ((Dictionary.IsOwnVariable (Sym) and then not Dictionary.OwnVariableIsInitialized (Sym))
         or else
           -- not an initialized constituent
           (Dictionary.IsConstituent (Sym) and then not Dictionary.OwnVariableIsInitialized (Dictionary.GetSubject (Sym))))
        and then
        -- not moded (this last limb to suppress error 333 for stream vars
        (Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.DefaultMode);
   end UnexpectedInitialization;

   ------------------------------------------------------------------------

   procedure wf_pragma (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         LexTokenManager.State,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   ---------------------------------------------------------------------

   procedure down_wf_store
     (Node      : in     STree.SyntaxNode;
      Scope     : in     Dictionary.Scopes;
      EStack    : in out ExpStack.ExpStackType;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         Scope &
   --#         Next_Node                  from Dictionary.Dict,
   --#                                         EStack,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;
   --------------------------------------------
   procedure Wf_Store_List
     (Node    : in out STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      E_Stack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         E_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         E_Stack                    from *,
   --#                                         Dictionary.Dict,
   --#                                         Scope &
   --#         Node                       from *,
   --#                                         Dictionary.Dict,
   --#                                         E_Stack,
   --#                                         STree.Table;
      is separate;
   ---------------------------------------------------------
   procedure down_wf_store_list
     (Node      : in     STree.SyntaxNode;
      Scope     : in     Dictionary.Scopes;
      EStack    : in out ExpStack.ExpStackType;
      Next_Node :    out STree.SyntaxNode)
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in STree.Table;
   --# derives EStack    from * &
   --#         Next_Node from Dictionary.Dict,
   --#                        EStack,
   --#                        Node,
   --#                        Scope,
   --#                        STree.Table &
   --#         null      from CommandLineData.Content;
      is separate;
   --------------------------------------------
   procedure up_wf_store
     (Node   : in     STree.SyntaxNode;
      Scope  : in     Dictionary.Scopes;
      EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         EStack,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         EStack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Type_Mark
     (Node          : in     STree.SyntaxNode;
      Current_Scope : in     Dictionary.Scopes;
      Context       : in     Dictionary.Contexts;
      Type_Sym      :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         STree.Table,
   --#         Type_Sym                   from CommandLineData.Content,
   --#                                         Context,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure down_wf_quantifier
     (Node      : in     STree.SyntaxNode;
      Estack    : in out ExpStack.ExpStackType;
      Scope     : in out Dictionary.Scopes;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Scope,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Estack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Next_Node                  from Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure up_wf_quantifier
     (Node   : in     STree.SyntaxNode;
      Estack : in out ExpStack.ExpStackType;
      Scope  : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Estack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Estack                     from *,
   --#                                         Dictionary.Dict,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict;
      is separate;
   ------------------------------------------------------------
   procedure WalkAnnotationExpression
     (Exp_Node       : in     STree.SyntaxNode;
      Scope          : in     Dictionary.Scopes;
      Type_Context   : in     Dictionary.Symbol;
      Context        : in     Anno_Tilde_Context;
      Result         :    out Exp_Record;
      Component_Data : in out ComponentManager.ComponentData)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Component_Data,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Result,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         Type_Context &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         Type_Context &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         Component_Data,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         Type_Context;
      is separate; -- AggregateStack.State will appear RSN

   ---------------------------------------------------------------------

   procedure wf_priority_value
     (Node       : in     STree.SyntaxNode;
      PragmaKind : in     Dictionary.RavenscarPragmas;
      Context    : in     Dictionary.Contexts;
      ErrorSym   : in     Dictionary.Symbol;
      Scope      : in     Dictionary.Scopes;
      ValueRep   :    out LexTokenManager.Lex_String;
      Compatible :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Compatible,
   --#         Dictionary.Dict,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorSym,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         PragmaKind,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         LexTokenManager.State,
   --#         ValueRep                   from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         PragmaKind,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Priority_Pragma (Node  : in STree.SyntaxNode;
                                 Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Renaming_Declaration (Node  : in STree.SyntaxNode;
                                      Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Use_Type_Clause (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------

   procedure wf_proof_renaming_declaration (Node  : in STree.SyntaxNode;
                                            Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   --------------------------------------------------------------------

   procedure wf_predicate
     (Node        : in     STree.SyntaxNode;
      Scope       : in     Dictionary.Scopes;
      Context     : in     Anno_Tilde_Context;
      ErrorsFound :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         ErrorsFound,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure CheckNamedAssociation
     (TheFormals             : in Dictionary.Symbol;
      Scope                  : in Dictionary.Scopes;
      NamedArgumentAssocNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         NamedArgumentAssocNode,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheFormals;
      is separate;

   ------------------------------------------------------------------------

   function FindActualNode
     (For_Formal                : in Dictionary.Symbol;
      Named_Argument_Assoc_Node : in STree.SyntaxNode)
     return                      STree.SyntaxNode
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --#        in STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Justification_Statement (Node  : in STree.SyntaxNode;
                                         Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ------------------------------------------------------------------------

   procedure WalkStatements
     (SeqNode : in     STree.SyntaxNode;
      Scope   : in     Dictionary.Scopes;
      Table   : in out RefList.HashTable)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out GlobalComponentData;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SeqNode,
   --#                                         STree.Table,
   --#                                         Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SeqNode,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Table,
   --#                                         TheHeap &
   --#         GlobalComponentData        from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SeqNode,
   --#                                         STree.Table,
   --#                                         Table,
   --#                                         TheHeap;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Subtype_Declaration (Node  : in STree.SyntaxNode;
                                     Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   --------------------------------------------------------------------

   procedure CheckTaskModifierConsistency
     (TheOwnVarType  : in     Dictionary.Symbol;
      TheVarType     : in     Dictionary.Symbol;
      ModifierIsTask : in     Boolean;
      ErrorNode      : in     STree.SyntaxNode;
      Consistent     :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ModifierIsTask,
   --#                                         TheOwnVarType,
   --#                                         TheVarType &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         LexTokenManager.State,
   --#                                         ModifierIsTask,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheOwnVarType,
   --#                                         TheVarType;
      is separate;

   --------------------------------------------------------------------

   procedure CheckProtectedModifierConsistency
     (TheType             : in     Dictionary.Symbol;
      ModifierIsProtected : in     Boolean;
      ErrorNode           : in     STree.SyntaxNode;
      Consistent          :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Consistent                 from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ModifierIsProtected,
   --#                                         TheType &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorNode,
   --#                                         LexTokenManager.State,
   --#                                         ModifierIsProtected,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheType;
      is separate;

   -----------------------------------------------------------------------------
   -- wf_variable_declaration
   --
   -- Note: Enclosing_Unit_Scope represents the scope of the enclosing program
   -- unit, whilst Declaration_Scope represents the scope of the current
   -- declaration. These are normally the same. The exception is the special
   -- case of a declaration within a protected type, where they may differ.
   -- For example, the declaration could be in the private part of the PT but
   -- the PT itself could be in the visible part of the enclosing unit.
   -----------------------------------------------------------------------------

   procedure Wf_Variable_Declaration
     (Node                 : in STree.SyntaxNode;
      Enclosing_Unit_Scope : in Dictionary.Scopes;
      Declaration_Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Declaration_Scope,
   --#                                         Dictionary.Dict,
   --#                                         Enclosing_Unit_Scope,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Declaration_Scope,
   --#                                         Dictionary.Dict,
   --#                                         Enclosing_Unit_Scope,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Declaration_Scope,
   --#                                         Dictionary.Dict,
   --#                                         Enclosing_Unit_Scope,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------

   procedure Wf_Constant_Declaration (Node          : in STree.SyntaxNode;
                                      Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   ---------------------------------------------------------------------

   procedure wf_entire_variable
     (Node         : in     STree.SyntaxNode;
      CurrentScope : in     Dictionary.Scopes;
      ErrorHint    : in     Visibility_Error_Hint;
      VarSym       :    out Dictionary.Symbol;
      Dotted       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dotted,
   --#         STree.Table,
   --#         VarSym                     from CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorHint,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ----------------------------------------------------------------------------

   procedure CheckDerivesConsistency
     (SubprogSym : in Dictionary.Symbol;
      Position   : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Position,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         Dictionary.Dict,
   --#                                         SubprogSym,
   --#                                         TheHeap;
      is separate;

   --------------------------------------------------------------------------

   -- procedure for creating a "full" dependency relation

   procedure CreateFullDependency
     (Node                   : in STree.SyntaxNode;
      SubprogSym             : in Dictionary.Symbol;
      Abstraction            : in Dictionary.Abstractions;
      ImportList, ExportList : in SeqAlgebra.Seq)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in     TheHeap;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                ExportList,
   --#                                ImportList,
   --#                                Node,
   --#                                STree.Table,
   --#                                SubprogSym,
   --#                                TheHeap &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Dictionary.Dict,
   --#                                ExportList,
   --#                                ImportList,
   --#                                LexTokenManager.State,
   --#                                Node,
   --#                                STree.Table,
   --#                                SubprogSym,
   --#                                TheHeap;
      is separate;

   -----------------------------------------------------------------------

   procedure AddDerivesStreamEffects
     (Node        : in STree.SyntaxNode;
      Subprog_Sym : in Dictionary.Symbol;
      Abstraction : in Dictionary.Abstractions)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Node,
   --#                                STree.Table,
   --#                                Subprog_Sym &
   --#         SPARK_IO.File_Sys from *,
   --#                                Abstraction,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.State,
   --#                                Node,
   --#                                STree.Table,
   --#                                Subprog_Sym;
      is separate;

   ---------------------------------------------------------------------

   procedure CreateFullSubProgDependency
     (Node        : in STree.SyntaxNode;
      SubprogSym  : in Dictionary.Symbol;
      Abstraction : in Dictionary.Abstractions)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         Abstraction,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap;
      is separate;

   ---------------------------------------------------------------------

   procedure wf_global_definition
     (Node         : in     STree.SyntaxNode;
      CurrentScope : in     Dictionary.Scopes;
      SubprogSym   : in     Dictionary.Symbol;
      FirstSeen    : in     Boolean;
      SemErrFound  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         SemErrFound,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         STree.Table,
   --#                                         SubprogSym;
      is separate;

   ----------------------------------------------------------------------

   procedure wf_dependency_relation
     (Node         : in STree.SyntaxNode;
      CurrentScope : in Dictionary.Scopes;
      SubprogSym   : in Dictionary.Symbol;
      FirstSeen    : in Boolean;
      GlobDefErr   : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         GlobDefErr,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         GlobDefErr,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         GlobDefErr,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         GlobDefErr,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         STree.Table,
   --#                                         SubprogSym,
   --#                                         TheHeap &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SubprogSym,
   --#                                         TheHeap;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Function_Annotation
     (Node          : in STree.SyntaxNode;
      Current_Scope : in Dictionary.Scopes;
      Subprog_Sym   : in Dictionary.Symbol;
      First_Seen    : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         STree.Table,
   --#                                         Subprog_Sym;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Property_List
     (Node          : in STree.SyntaxNode;
      Type_Node_Pos : in LexTokenManager.Token_Position;
      Scope         : in Dictionary.Scopes;
      The_Owner     : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         The_Owner &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap,
   --#                                         The_Owner,
   --#                                         Type_Node_Pos;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Declare_Annotation
     (Node         : in STree.SyntaxNode;
      Scope        : in Dictionary.Scopes;
      Task_Or_Proc : in Dictionary.Symbol;
      First_Seen   : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         Task_Or_Proc,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Task_Or_Proc,
   --#                                         TheHeap;
      is separate;

   --------------------------------------------------------------------------------

   procedure Wf_Procedure_Annotation
     (Node          : in STree.SyntaxNode;
      Current_Scope : in Dictionary.Scopes;
      Subprog_Sym   : in Dictionary.Symbol;
      First_Seen    : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Seen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym,
   --#                                         TheHeap;
      is separate;

   ----------------------------------------------------------------------------

   procedure wf_procedure_constraint
     (Node      : in STree.SyntaxNode;
      Scope     : in Dictionary.Scopes;
      FirstSeen : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------------------------------------------------

   procedure wf_function_constraint
     (Node      : in STree.SyntaxNode;
      Scope     : in Dictionary.Scopes;
      FirstSeen : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------------------------------------------------

   procedure Wf_Inherit_Clause
     (Node     : in STree.SyntaxNode;
      Comp_Sym : in Dictionary.Symbol;
      Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Context_Clause
     (Node     : in STree.SyntaxNode;
      Comp_Sym : in Dictionary.Symbol;
      Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Comp_Sym,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Formal_Part
     (Node             : in STree.SyntaxNode;
      Current_Scope    : in Dictionary.Scopes;
      Subprog_Sym      : in Dictionary.Symbol;
      First_Occurrence : in Boolean;
      Context          : in Dictionary.Contexts)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         First_Occurrence,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         Subprog_Sym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Context,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         First_Occurrence,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Subprog_Sym;
      is separate;

   ------------------------------------------------------------------------

   -- procedure to check whether a properly-defined subprogram or stub creates
   -- overloading by re-using the name of a potentially inheritable root
   -- operation

   procedure CheckNoOverloadingFromTaggedOps
     (SpecNode      : in STree.SyntaxNode;
      SubprogSym    : in Dictionary.Symbol;
      Scope         : in Dictionary.Scopes;
      Abstraction   : in Dictionary.Abstractions;
      Is_Overriding : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict            from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SpecNode,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Is_Overriding,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SpecNode,
   --#                                         STree.Table,
   --#                                         SubprogSym;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Function_Specification
     (Node        : in     STree.SyntaxNode;
      Hidden      : in     Boolean;
      Scope       : in out Dictionary.Scopes;
      Subprog_Sym :    out Dictionary.Symbol;
      First_Seen  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         Hidden,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Hidden,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         First_Seen                 from Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Scope,
   --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -------------------------------------------------------------------------

   procedure Wf_Procedure_Specification
     (Node        : in     STree.SyntaxNode;
      Hidden      : in     Boolean;
      Scope       : in out Dictionary.Scopes;
      Subprog_Sym :    out Dictionary.Symbol;
      First_Seen  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Hidden,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         First_Seen,
   --#         STree.Table                from Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         Scope,
   --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Hidden,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   ----------------------------------------------------------------------------

   -- function used by wf_subprogram_body and wf_body_stub

   function HasParameterGlobalOrReturnOfLocalPrivateType (SubprogSym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   -- returns true if the subprogram has a parameter or global which is of a private type
   -- but which is not private when viewd from the subprogram
   is
      Result  : Boolean := False;
      It      : Dictionary.Iterator;
      TypeSym : Dictionary.Symbol;

      function IsPrivateButNotPrivateHere (TypeSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      --#        in SubprogSym;
      is
      begin -- IsPrivateButNotPrivateHere
         return Dictionary.TypeIsPrivate (TypeSym)
           and then not Dictionary.IsPrivateType (TypeSym, Dictionary.LocalScope (SubprogSym));
      end IsPrivateButNotPrivateHere;

   begin -- HasParameterGlobalOrReturnOfLocalPrivateType

      -- iterate through parameters
      It := Dictionary.FirstSubprogramParameter (SubprogSym);
      while It /= Dictionary.NullIterator loop
         TypeSym := Dictionary.GetType (Dictionary.CurrentSymbol (It));
         Result  := IsPrivateButNotPrivateHere (TypeSym);
         exit when Result;

         It := Dictionary.NextSymbol (It);
      end loop;

      if not Result then -- no parameters were private, so check globals
         It := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, SubprogSym);
         while It /= Dictionary.NullIterator loop
            TypeSym := Dictionary.GetType (Dictionary.CurrentSymbol (It));
            Result  := IsPrivateButNotPrivateHere (TypeSym);
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
      end if;

      -- no parameters or globals were private, so check for function return type.
      if not Result then
         if Dictionary.IsFunction (SubprogSym) then
            TypeSym := Dictionary.GetType (SubprogSym);
            Result  := IsPrivateButNotPrivateHere (TypeSym);
         end if;
      end if;

      return Result;
   end HasParameterGlobalOrReturnOfLocalPrivateType;

   ----------------------------------------------------------------------------

   procedure Wf_Subprogram_Body
     (Node      : in     STree.SyntaxNode;
      Scope     : in out Dictionary.Scopes;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Next_Node,
   --#         Scope                      from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure wf_entry_body
     (Node      : in     STree.SyntaxNode;
      Scope     : in out Dictionary.Scopes;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Next_Node,
   --#         Scope                      from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Proof_Function_Declaration
     (Node     : in     STree.SyntaxNode;
      Scope    : in     Dictionary.Scopes;
      Func_Sym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Func_Sym                   from ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table;
      is separate;

   -----------------------------------------------------------------------

   procedure Wf_Body_Stub (Node  : in STree.SyntaxNode;
                           Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -----------------------------------------------------------------------

   procedure Wf_Subunit (Node  : in     STree.SyntaxNode;
                         Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope,
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   --------------------------------------------------------------------

   procedure CheckEmbedBodies (CompSym : in Dictionary.Symbol;
                               Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         CompSym,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ---------------------------------------------------------------------

   procedure CheckSuspendsListAccountedFor (ProcOrTask : in Dictionary.Symbol;
                                            Node       : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         ProcOrTask,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ---------------------------------------------------------------------

   procedure up_wf_subprogram_body (Node  : in     STree.SyntaxNode;
                                    Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict;
      is separate;

   ---------------------------------------------------------------------

   -- this subprogram used to a subunit of wf_package_declaration.wf_visible; it has
   -- been moved up to this level to make it available to wf_protected_type.wf_protected_operation

   procedure Wf_Subprogram_Declaration
     (Node            : in     STree.SyntaxNode;
      Current_Scope   : in     Dictionary.Scopes;
      The_Subprog_Sym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         The_Subprog_Sym            from ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
      is separate;

   ---------------------------------------------------------------------

   procedure Wf_Known_Discriminant_Part
     (Node               : in STree.SyntaxNode;
      Protected_Type_Sym : in Dictionary.Symbol;
      Scope              : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Protected_Type_Sym,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Protected_Type_Sym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;
   ----------
   procedure Wf_Array_Type_Definition
     (Node         : in     STree.SyntaxNode;
      Scope        : in     Dictionary.Scopes;
      Ident_Node   : in     STree.SyntaxNode;
      Dec_Loc      : in     LexTokenManager.Token_Position;
      Is_Generic   : in     Boolean;
      Errors_Found :    out Boolean;
      The_Array    :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Errors_Found,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         Is_Generic,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         Is_Generic,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         Is_Generic,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         The_Array                  from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;
   ----------
   procedure Wf_Full_Type_Declaration (Node  : in STree.SyntaxNode;
                                       Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------------------------------------------

   procedure Wf_Basic_Declarative_Item (Node          : in STree.SyntaxNode;
                                        Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure CheckAnnouncedTypesDeclared
     (PackSym : in Dictionary.Symbol;
      Scope   : in Dictionary.Scopes;
      Node    : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         PackSym,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   --------------------------------------------------------------------

   procedure wf_generic_actual_part
     (GenericNode      : in     STree.SyntaxNode;
      GenericSym       : in     Dictionary.Symbol;
      InstantiationSym : in     Dictionary.Symbol;
      Scope            : in     Dictionary.Scopes;
      ErrorFound       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GenericNode,
   --#                                         GenericSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorFound                 from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GenericNode,
   --#                                         GenericSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GenericNode,
   --#                                         GenericSym,
   --#                                         GlobalComponentData,
   --#                                         InstantiationSym,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure wf_generic_subprogram_instantiation (Node  : in STree.SyntaxNode;
                                                  Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ----------------------------------------------------------------------

   -- Factored out of wf_package_declaration to allow reuse in generic package declaration

   procedure Wf_Package_Specification
     (Node          : in STree.SyntaxNode;
      Ident_Str     : in LexTokenManager.Lex_String;
      Pack_Sym      : in Dictionary.Symbol;
      Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         Ident_Str,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Pack_Sym,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Package_Declaration (Node          : in STree.SyntaxNode;
                                     Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ----------------------------------------------------------------------

   -- PNA temporary reduced annotation for phase 1 generics cfr 1340

   procedure wf_generic_package_instantiation (Node  : in STree.SyntaxNode;
                                               Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ----------------------------------------------------------------------

   procedure Wf_Generic_Declaration (Node          : in STree.SyntaxNode;
                                     Current_Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   ----------------------------------------------------------------------

   procedure CheckPackageNeedsBody (Node    : in STree.SyntaxNode;
                                    PackSym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         PackSym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ----------------------------------------------------------------------

   procedure wf_package_body (Node  : in     STree.SyntaxNode;
                              Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Scope,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ----------------------------------------------------------------------

   procedure up_wf_package_body (Node  : in     STree.SyntaxNode;
                                 Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         STree.Table;
      is separate;

   ----------------------------------------------------------------------

   procedure wf_package_initialization (Node  : in STree.SyntaxNode;
                                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Protected_Body
     (Node      : in     STree.SyntaxNode;
      Scope     : in out Dictionary.Scopes;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         Next_Node,
   --#         Scope,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure up_wf_protected_body (Node  : in     STree.SyntaxNode;
                                   Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict;
      is separate;

   ------------------------------------------------------------------------

   procedure Wf_Task_Body
     (Node      : in     STree.SyntaxNode;
      Scope     : in out Dictionary.Scopes;
      Next_Node :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Next_Node,
   --#         Scope                      from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   ------------------------------------------------------------------------

   procedure up_wf_task_body (Node  : in     STree.SyntaxNode;
                              Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Scope                      from *,
   --#                                         Dictionary.Dict;
      is separate;

   ------------------------------------------------------------------------

   procedure wf_machine_code_insertion (Node  : in STree.SyntaxNode;
                                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

begin -- CompUnit
   Next_Node := Top_Node;
   Stack.Init;
   Heap.Initialize (TheHeap);
   RefList.Init (NodeTable);
   --# accept Flow, 10, AggregateStack.State, "Ineffective assignment expected";
   AggregateStack.Init;

   ComponentManager.Initialise (GlobalComponentData);

   Current_Scope := Dictionary.GlobalScope;

   while Next_Node /= STree.NullNode loop
      Last_Node := Next_Node;
      NodeType  := Syntax_Node_Type (Node => Last_Node);
      case NodeType is
         when SPSymbols.package_declaration | SPSymbols.private_package_declaration =>
            -- ASSUME Last_Node = package_declaration OR private_package_declaration
            ErrorHandler.Start_Unit;
            Wf_Package_Declaration (Node          => Last_Node,
                                    Current_Scope => Current_Scope);
            ErrorHandler.End_Unit;
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.generic_declaration | SPSymbols.private_generic_package_declaration =>
            -- ASSUME Last_Node = generic_declaration OR private_generic_package_declaration
            ErrorHandler.Start_Unit;
            Wf_Generic_Declaration (Node          => Last_Node,
                                    Current_Scope => Current_Scope);
            ErrorHandler.End_Unit;
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.generic_package_instantiation =>
            -- ASSUME Last_Node = generic_package_instantiation
            wf_generic_package_instantiation (Last_Node, Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.generic_subprogram_instantiation =>
            -- ASSUME Last_Node = generic_subprogram_instantiation
            wf_generic_subprogram_instantiation (Last_Node, Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.package_body =>
            -- ASSUME Last_Node = package_body
            ErrorHandler.Start_Unit;
            wf_package_body (Last_Node, Current_Scope);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SPSymbols.subprogram_body | SPSymbols.not_overriding_subprogram_body =>
            -- ASSUME Last_Node = subprogram_body OR not_overriding_subprogram_body
            ErrorHandler.Error_Reset;
            ErrorHandler.Start_Unit;
            Wf_Subprogram_Body (Node      => Last_Node,
                                Scope     => Current_Scope,
                                Next_Node => Next_Node);
            ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
            if ErrStatus = ErrorHandler.No_Error then
               Stack.Push (False);
            else
               Stack.Push (True);
            end if;
         when SPSymbols.entry_body =>
            -- ASSUME Last_Node = entry_body
            ErrorHandler.Error_Reset;
            ErrorHandler.Start_Unit;
            wf_entry_body (Node      => Last_Node,
                           Scope     => Current_Scope,
                           Next_Node => Next_Node);
            ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
            if ErrStatus = ErrorHandler.No_Error then
               Stack.Push (False);
            else
               Stack.Push (True);
            end if;
         when SPSymbols.task_body =>
            -- ASSUME Last_Node = task_body
            --# accept Flow, 41, "Expected stable expression";
            if CommandLineData.Ravenscar_Selected then
               --# end accept;
               ErrorHandler.Error_Reset;
               ErrorHandler.Start_Unit;
               Wf_Task_Body (Node      => Last_Node,
                             Scope     => Current_Scope,
                             Next_Node => Next_Node);
               ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
               if ErrStatus = ErrorHandler.No_Error then
                  Stack.Push (False);
               else
                  Stack.Push (True);
               end if;
            else -- can't use task except in Ravenscar
               ErrorHandler.Semantic_Error
                 (Err_Num   => 850,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Last_Node),
                  Id_Str    => LexTokenManager.Null_String);
               -- prune tree walk, we don't want to enter this production when it's illegal
               Next_Node := STree.NullNode;
            end if;
         when SPSymbols.protected_body =>
            -- ASSUME Last_Node = protected_body
            --# accept Flow, 41, "Expected stable expression";
            if CommandLineData.Ravenscar_Selected then
               --# end accept;
               ErrorHandler.Start_Unit;
               Wf_Protected_Body (Node      => Last_Node,
                                  Scope     => Current_Scope,
                                  Next_Node => Next_Node);
            else -- can't use protected type except in Ravenscar
               ErrorHandler.Semantic_Error
                 (Err_Num   => 850,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Last_Node),
                  Id_Str    => LexTokenManager.Null_String);
               -- prune tree walk, we don't want to enter this production when it's illegal
               Next_Node := STree.NullNode;
            end if;
         when SPSymbols.body_stub =>
            -- ASSUME Last_Node = body_stub
            Wf_Body_Stub (Node  => Last_Node,
                          Scope => Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.subunit =>
            -- ASSUME Last_Node = subunit
            Wf_Subunit (Node  => Last_Node,
                        Scope => Current_Scope);
            Next_Node := Child_Node (Current_Node => Last_Node);
         when SPSymbols.proof_function_declaration =>
            -- ASSUME Last_Node = proof_function_declaration
            --# accept Flow, 10, Unused, "Expected ineffective assignment to Unused";
            Wf_Proof_Function_Declaration (Node     => Last_Node,
                                           Scope    => Current_Scope,
                                           Func_Sym => Unused);
            --# end accept;
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.basic_declarative_item =>
            -- ASSUME Last_Node = basic_declarative_item
            Wf_Basic_Declarative_Item (Node          => Last_Node,
                                       Current_Scope => Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.renaming_declaration =>
            -- ASSUME Last_Node = renaming_declaration
            Wf_Renaming_Declaration (Node  => Last_Node,
                                     Scope => Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.proof_renaming_declaration =>
            -- ASSUME Last_Node = proof_renaming_declaration
            wf_proof_renaming_declaration (Last_Node, Current_Scope);
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.use_type_clause =>
            -- ASSUME Last_Node = use_type_clause
            Wf_Use_Type_Clause (Last_Node);
            Next_Node := STree.NullNode; -- nothing below here used
         when SPSymbols.sequence_of_statements =>
            -- ASSUME Last_Node = sequence_of_statements
            Heap.ReportUsage (TheHeap);
            Heap.Initialize (TheHeap);
            RefList.Init (NodeTable);
            WalkStatements (SeqNode => Last_Node,
                            Scope   => Current_Scope,
                            Table   => NodeTable);
            ComponentManager.ReportUsage (GlobalComponentData);
            Next_Node := STree.NullNode;
         when SPSymbols.package_initialization =>
            -- ASSUME Last_Node = package_initialization
            wf_package_initialization (Last_Node, Current_Scope);
            ErrorHandler.Error_Reset;
            Next_Node := Child_Node (Last_Node);
         when SPSymbols.apragma =>
            -- ASSUME Last_Node = apragma
            wf_pragma (Last_Node, Current_Scope);
            Next_Node := STree.NullNode;
         when SPSymbols.code_insertion =>
            -- ASSUME Last_Node = code_insertion
            wf_machine_code_insertion (Last_Node, Current_Scope);
            Next_Node := STree.NullNode;
            -- following series of cases are places where we prune tree walk
            -- because there is nothing below node of use to us
         when SPSymbols.with_clause | SPSymbols.code_statement | -- probably can't reach this one any more
           SPSymbols.context_clause_rep | SPSymbols.refinement_definition =>
            -- ASSUME Last_Node = with_clause OR code_statement OR context_clause_rep OR refinement_definition
            Next_Node := STree.NullNode;
            -- this tests for whether down loop end has been reached at a terminal
         when others =>
            if NodeType in SPSymbols.SPNonTerminal then
               Next_Node := Child_Node (Current_Node => Last_Node);
            else
               Next_Node := STree.NullNode;
            end if;
      end case;

      if Next_Node = STree.NullNode then ------up loop----------
         loop
            Next_Node := Next_Sibling (Current_Node => Last_Node);
            exit when Next_Node /= STree.NullNode;

            Next_Node := Parent_Node (Current_Node => Last_Node);
            exit when Next_Node = STree.NullNode;

            Last_Node := Next_Node;
            case Syntax_Node_Type (Node => Last_Node) is
               when SPSymbols.package_body =>
                  -- ASSUME Last_Node = package_body
                  up_wf_package_body (Last_Node,
                                      -- using and to get
                                      Current_Scope);
                  ErrorHandler.End_Unit;
               when SPSymbols.protected_body =>
                  -- ASSUME Last_Node = protected_body
                  up_wf_protected_body (Last_Node,
                                        -- using and to get
                                        Current_Scope);
                  ErrorHandler.End_Unit;
               when SPSymbols.task_body =>
                  -- ASSUME Last_Node = task_body
                  up_wf_task_body (Last_Node,
                                   -- using and to get
                                   Current_Scope);
                  Stack.Pop;
                  ErrorHandler.Error_Reset;
                  ErrorHandler.End_Unit;
               when SPSymbols.entry_body =>
                  -- ASSUME Last_Node = entry_body
                  Current_Scope := Dictionary.GetEnclosingScope (Current_Scope);
                  Stack.Pop;
                  ErrorHandler.Error_Reset;
                  ErrorHandler.End_Unit;
               when SPSymbols.package_initialization =>
                  -- ASSUME Last_Node = package_initialization
                  Stmt_Node := Child_Node (Current_Node => Last_Node);
                  ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
                  if Syntax_Node_Type (Node => Stmt_Node) /= SPSymbols.hidden_part
                    and then ErrStatus = ErrorHandler.No_Error then
                     --# accept Flow, 10, Unused_Data_Flow_Error_Flag,
                     --#        "Expected ineffective assignment to Unused_Data_Flow_Error_Flag";
                     FlowAnalyser.FlowAnalyse
                       (Dictionary.GetRegion (Current_Scope),
                        Stmt_Node,
                        Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                        GlobalComponentData,
                        TheHeap,
                        NodeTable,
                        Unused_Data_Flow_Error_Flag);
                     --# end accept;
                     ErrorHandler.Get_Error_Severity (Severity => ErrStatus);

                     if ErrStatus = ErrorHandler.No_Error then
                        ErrorHandler.Report_Success
                          (Position    => Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                           Subprog_Str => LexTokenManager.Null_String);
                     end if;
                  end if;

               when SPSymbols.subprogram_implementation =>
                  -- ASSUME Last_Node = subprogram_implementation
                  Stmt_Node := Child_Node (Current_Node => Last_Node);
                  while Stmt_Node /= STree.NullNode
                    and then Syntax_Node_Type (Node => Stmt_Node) /= SPSymbols.sequence_of_statements loop
                     Stmt_Node := Next_Sibling (Current_Node => Stmt_Node);
                  end loop;
                  if Stmt_Node /= STree.NullNode then
                     ErrorHandler.Get_Error_Severity (Severity => ErrStatus);
                     Semantic_Error_In_Subprogram_Body  := not (ErrStatus = ErrorHandler.No_Error and then Stack.Top = False);
                     Data_Flow_Error_In_Subprogram_Body := False; -- default in case flow analyser not called below
                     if not Semantic_Error_In_Subprogram_Body then
                        -- don't do flow analysis in presence of semantic errors
                        FlowAnalyser.FlowAnalyse
                          (Dictionary.GetRegion (Current_Scope),
                           Stmt_Node,
                           Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                           GlobalComponentData,
                           TheHeap,
                           NodeTable,
                           -- to get
                           Data_Flow_Error_In_Subprogram_Body);
                        ErrorHandler.Get_Error_Severity (Severity => ErrStatus);

                        if ErrStatus = ErrorHandler.No_Error then
                           ErrorHandler.Report_Success
                             (Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                              Dictionary.GetSimpleName (Dictionary.GetRegion (Current_Scope)));
                        end if;
                        ErrorHandler.Flush_Echo_Messages;
                     end if;
                     -- Call VCG regardless of error status (used to be inside above IF statement).  Instead, we pass
                     -- the error status to the VCG which generates a false VC (or null PF) for subprograms
                     -- containing static semantic errors
                     VCG.Generate_VCs
                       (Start_Node                    => Stmt_Node,
                        Scope                         => Current_Scope,
                        Do_VCG                        => Do_VCG,
                        End_Position                  => Node_Position (Node => Next_Sibling (Current_Node => Stmt_Node)),
                        Flow_Heap                     => TheHeap,
                        Semantic_Error_In_Subprogram  => Semantic_Error_In_Subprogram_Body,
                        Data_Flow_Error_In_Subprogram => Data_Flow_Error_In_Subprogram_Body,
                        Type_Check_Exports            => ErrorHandler.Possibly_Invalid_Values);
                  end if;

               when SPSymbols.subprogram_body | SPSymbols.not_overriding_subprogram_body =>
                  -- ASSUME Last_Node = subprogram_body OR not_overriding_subprogram_body
                  up_wf_subprogram_body (Last_Node,
                                         -- using and to get
                                         Current_Scope);
                  Stack.Pop;
                  ErrorHandler.Error_Reset;
                  ErrorHandler.End_Unit;
               when SPSymbols.main_program_declaration =>
                  -- ASSUME Last_Node = main_program_declaration
                  if Dictionary.GetThePartition /= Dictionary.NullSymbol
                    and then Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract, Dictionary.GetThePartition) then
                     FlowAnalyser.FlowAnalysePartition (Last_Node, TheHeap);
                  end if;
               when others =>
                  null;
            end case;
         end loop;
      end if;
   end loop;

   Heap.ReportUsage (TheHeap);

   --# accept Flow, 33, Unused, "Expected Unused to be neither referenced nor exported" &
   --#        Flow, 33, Unused_Data_Flow_Error_Flag, "Expected Unused_Data_Flow_Error_Flag to be neither referenced nor exported";
end CompUnit;
