-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

-- Overview: Called to check validity of a
-- simple_expression node.  Replaces calls to StaticSimpleExpression,
-- BaseTypeSimpleExpression and CheckTypeSimpleExpression
----------------------------------------------------------------------------

separate (Sem.CompUnit)
procedure Wf_Simple_Expression
  (Node                    : in     STree.SyntaxNode;
   Scope                   : in     Dictionary.Scopes;
   E_Stack                 : in out ExpStack.ExpStackType;
   T_Stack                 : in     TypeContextStack.TStackType;
   Context_Requires_Static : in     Boolean) is

   Left, Right, Result   : Exp_Record;
   Op_Node               : STree.SyntaxNode;
   Operator              : SPSymbols.SPSymbol;
   Left_Base, Right_Base : Dictionary.Symbol;
   Result_Str            : E_Strings.T;

   function Length_Sum (Left, Right : Exp_Record) return Maths.Value
   --# global in Dictionary.Dict;
   is
      Unused : Maths.ErrorCode;
      Sum    : Maths.Value;

      function Length_Of (S : in Exp_Record) return Maths.Value
      --# global in Dictionary.Dict;
      is
         Result : Maths.Value;
      begin
         if Dictionary.IsPredefinedCharacterType (Dictionary.GetRootType (S.Type_Symbol)) then
            Result := Maths.OneInteger;
         else
            Result := S.Range_RHS;
         end if;
         return Result;
      end Length_Of;

   begin -- Length_Sum

      --# accept Flow, 10, Unused, "Expected ineffective assignment";
      Maths.Add (Length_Of (S => Left), Length_Of (S => Right), Sum, Unused);
      --# end accept;
      --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
      return Sum;
   end Length_Sum;

   -- Returns either the Character or String Literal associated with Exp
   -- as a string literal. A character literal 'a' is returned as the
   -- string literal "a" for example. The resulting string does NOT
   -- include opening or closing quotations.
   function Get_String (Exp      : Exp_Record;
                        Exp_Type : Dictionary.Symbol) return E_Strings.T
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
   --# pre Dictionary.IsPredefinedCharacterType (Exp_Type, Dictionary.Dict) or
   --#   Dictionary.IsPredefinedStringType (Exp_Type, Dictionary.Dict);
   is
      Str_Length : E_Strings.Lengths;
      Char_Code  : Integer;
      Unused     : Maths.ErrorCode;
      Str        : E_Strings.T;
   begin
      if Dictionary.IsPredefinedCharacterType (Exp_Type) then
         --# accept F, 10, Unused, "Ineffective assignment OK here";
         Maths.ValueToInteger (Exp.Value, Char_Code, Unused);
         --# end accept;
         SystemErrors.RT_Assert
           (C       => Char_Code >= Character'Pos (Character'First) and then Char_Code <= Character'Pos (Character'Last),
            Sys_Err => SystemErrors.Expression_Stack_Corrupt,
            Msg     => "Wf_Simple_Expression.Get_String - value of character literal not in Character");
         Str := E_Strings.Empty_String;
         E_Strings.Append_Char (Str, Character'Val (Char_Code));
      else
         -- Get the String representation of Exp. This will include a preceding
         -- and a trailing " character at this point
         Str        := LexTokenManager.Lex_String_To_String (Exp.String_Value);
         Str_Length := E_Strings.Get_Length (Str); -- including two " characters
         if Str_Length > 2 then
            -- Must contain at least one character, but we still want
            -- to get rid of the outer " characters.
            Str := E_Strings.Section (Str, 2, Str_Length - 2);
         else
            -- Must be a null String "", so
            Str := E_Strings.Empty_String;
         end if;
      end if;
      --# accept F, 33, Unused, "Unused not referenced OK";
      return Str;
   end Get_String;

begin -- Wf_Simple_Expression

   -- ASSUME Node = simple_expression OR annotation_simple_expression
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.simple_expression
        or else Syntax_Node_Type (Node => Node) = SPSymbols.annotation_simple_expression,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = simple_expression OR annotation_simple_expression in Wf_Simple_Expression");

   Op_Node := Child_Node (Current_Node => Node);
   -- ASSUME Op_Node = simple_expression OR simple_expression_opt OR
   --                  annotation_simple_expression OR annotation_simple_expression_opt
   if Syntax_Node_Type (Node => Op_Node) = SPSymbols.simple_expression
     or else Syntax_Node_Type (Node => Op_Node) = SPSymbols.annotation_simple_expression then
      -- ASSUME Op_Node = simple_expression OR annotation_simple_expression
      Op_Node := Child_Node (Current_Node => Next_Sibling (Current_Node => Op_Node));
      -- ASSUME Op_Node = plus OR minus OR ampersand
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Op_Node) = SPSymbols.plus
           or else Syntax_Node_Type (Node => Op_Node) = SPSymbols.minus
           or else Syntax_Node_Type (Node => Op_Node) = SPSymbols.ampersand,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Op_Node = plus OR minus OR ampersand in Wf_Simple_Expression");
      -- binary_adding_operator exists
      Operator := Syntax_Node_Type (Node => Op_Node);
      ExpStack.Pop (Right, E_Stack);
      ExpStack.Pop (Left, E_Stack);
      Result := NullTypeRecord; -- safety: we may not set all fields below

      -- do static checks first
      Result.Is_Constant := Left.Is_Constant and then Right.Is_Constant;
      -- Ada95 and Ada2005 say that & can be a static function.
      -- LRM83(4.9) says that & is never static, so...
      case CommandLineData.Content.Language_Profile is
         when CommandLineData.SPARK83 =>
            Result.Is_Static := Left.Is_Static and then Right.Is_Static and then Operator /= SPSymbols.ampersand;
         when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>
            Result.Is_Static := Left.Is_Static and then Right.Is_Static;
      end case;
      Result.Has_Operators := True;

      if Left.Is_ARange or else Right.Is_ARange then
         Result := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => 90,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Op_Node),
            Id_Str    => LexTokenManager.Null_String);
      else -- neither are ranges

         -- now do type compat and operator visibility checks
         if Operator = SPSymbols.ampersand then
            Left_Base  := Dictionary.GetRootType (Left.Type_Symbol);
            Right_Base := Dictionary.GetRootType (Right.Type_Symbol);
            if (Dictionary.IsPredefinedCharacterType (Left_Base) or else Dictionary.IsPredefinedStringType (Left_Base))
              and then (Dictionary.IsPredefinedCharacterType (Right_Base) or else Dictionary.IsPredefinedStringType (Right_Base))
            then
               -- "&" expressions in SPARK are always expected to be constant (SR 4.5.3)
               -- As such, we should be able to compute the length and value of the result.
               if Result.Is_Constant then
                  Result.Range_RHS   := Length_Sum (Left  => Left,
                                                    Right => Right);
                  Result.Type_Symbol := Dictionary.GetPredefinedStringType;

                  -- Now build the result, from a starting ", then Left, then Right,
                  -- then a closing "
                  Result_Str := E_Strings.Empty_String;
                  E_Strings.Append_Char (Result_Str, '"');
                  E_Strings.Append_Examiner_String (Result_Str, Get_String (Left, Left_Base));
                  E_Strings.Append_Examiner_String (Result_Str, Get_String (Right, Right_Base));
                  E_Strings.Append_Char (Result_Str, '"');

                  LexTokenManager.Insert_Examiner_String (Result_Str, Result.String_Value);
               else
                  Result := UnknownTypeRecord;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 37,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Node),
                     Id_Str    => LexTokenManager.Null_String);
               end if;
            else
               -- Types of Left and/or Right are wrong...
               Result := UnknownTypeRecord;
               ErrorHandler.Semantic_Error_Sym2
                 (Err_Num   => 35,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Op_Node),
                  Sym       => Left_Base,
                  Sym2      => Right_Base,
                  Scope     => Scope);
            end if;
         else -- its plus or minus
            CheckBinaryOperator
              (Operator      => Operator,
               Left          => Left,
               Right         => Right,
               Scope         => Scope,
               T_Stack       => T_Stack,
               Op_Pos        => Node_Position (Node => Op_Node),
               Left_Pos      => Node_Position (Node => Child_Node (Current_Node => Node)),
               Right_Pos     => Node_Position
                 (Node => Next_Sibling (Current_Node => Next_Sibling (Current_Node => Child_Node (Current_Node => Node)))),
               Convert       => True,
               Is_Annotation => Syntax_Node_Type (Node => Node) = SPSymbols.annotation_simple_expression,
               Result        => Result);

            -- Seed Op_Node with type to aid selection of operator in VCG
            STree.AddNodeSymbol (Op_Node, Result.Type_Symbol);

            CalcBinaryOperator
              (Node_Pos      => Node_Position (Node => Node),
               Operator      => Operator,
               Left_Val      => Left.Value,
               Right_Val     => Right.Value,
               Is_Annotation => Syntax_Node_Type (Node => Node) = SPSymbols.annotation_simple_expression,
               Result        => Result);

            if Syntax_Node_Type (Node => Parent_Node (Current_Node => Node)) = SPSymbols.simple_expression then
               -- ASSUME Parent_Node (Current_Node => Node) = simple_expression
               ----------------------------------------------------------------
               -- If the parent is also a simple_expression, then we must have
               -- an unparenthesized expression with two adding operators, such
               -- as A + B + C
               --
               -- Here, we issue warning 302 to warn of potential evaluation
               -- order dependency.
               --
               -- We can reduce false-alarm rate here by suppressing the
               -- warning in two specific cases:
               --  a) If the sub-expression under consideration is static
               --     AND the expression as a whole appears in a context
               --     that requires a static expression.  Example: a type
               --     declaration such as
               --       type T is range B + 2 - 3 .. 10;
               --   or
               --  b) A modular-typed expression where the two operators
               --     under consideration are both the same and
               --     commutative.  For example:
               --       A := A + B + C;
               --     where A, B, and C are all of the same modular
               --     (sub-)type.
               --
               -- The same logic is used in wf_term for multiplying
               -- operators.
               ----------------------------------------------------------------
               if (Context_Requires_Static and then Result.Is_Static)
                 or else (Dictionary.TypeIsModular (Result.Type_Symbol)
                            and then Ops_Are_Same_And_Commutative
                            (Operator,
                             Syntax_Node_Type (Node => Child_Node (Current_Node => Next_Sibling (Current_Node => Node)))))
               then
                  null;
               else
                  ErrorHandler.Semantic_Warning
                    (Err_Num  => 302,
                     Position => Node_Position (Node => Node),
                     Id_Str   => LexTokenManager.Null_String);
               end if;
            end if;
         end if;
      end if;
      Result.Errors_In_Expression := Result.Errors_In_Expression
        or else Left.Errors_In_Expression
        or else Right.Errors_In_Expression;

      -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
      -- This symbol is used (by Wf_Assign) to convery information to the VCG to supress
      -- checks when an unchecked_conversion is assigned to something of the same subtype.
      -- We do not want this mechanism if the unchecked_conversion is sued in any other context
      -- than a direct assignment.  Therefore we clear OtherSymbol here:
      Result.Other_Symbol := Dictionary.NullSymbol;
      ExpStack.Push (Result, E_Stack);
   elsif Syntax_Node_Type (Node => Op_Node) /= SPSymbols.simple_expression_opt
     and then Syntax_Node_Type (Node => Op_Node) /= SPSymbols.annotation_simple_expression_opt then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Op_Node = simple_expression OR simple_expression_opt OR annotation_simple_expression OR annotation_simple_expression_opt in Wf_Simple_Expression");
   end if;
end Wf_Simple_Expression;
