-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit)
procedure Wf_Proof_Function_Declaration
  (Node     : in     STree.SyntaxNode;
   Scope    : in     Dictionary.Scopes;
   Func_Sym :    out Dictionary.Symbol) is
   Spec_Node, Constraint_Node, Formal_Part_Node, Null_Constraint_Node : STree.SyntaxNode;

   -----------------------------------------------------------

   procedure Check_Function_Specification
     (Node          : in     STree.SyntaxNode;
      Current_Scope : in     Dictionary.Scopes;
      Subprog_Sym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Subprog_Sym                from ContextManager.Ops.Unit_Stack,
   --#                                         Current_Scope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table;
   is
      Return_Type_Node : STree.SyntaxNode;
      Ident_Str        : LexTokenManager.Lex_String;
      Type_Sym         : Dictionary.Symbol;
   begin
      -- ASSUME Node = function_specification
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.function_specification,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = function_specification in Check_Function_Specification");
      Subprog_Sym := Dictionary.NullSymbol;
      Ident_Str   := Node_Lex_String (Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
      if Dictionary.IsDirectlyDefined (Ident_Str, Current_Scope, Dictionary.ProofContext) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Ident_Str);
      else
         Dictionary.AddSubprogram
           (Name          => Ident_Str,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                  End_Position   => Node_Position (Node => Node)),
            Scope         => Current_Scope,
            Context       => Dictionary.ProofContext,
            Subprogram    => Subprog_Sym);
      end if;
      Return_Type_Node := Last_Sibling_Of (Start_Node => Child_Node (Current_Node => Node));
      -- ASSUME Return_Type_Node = type_mark
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Return_Type_Node) = SPSymbols.type_mark,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Return_Type_Node = type_mark in Check_Function_Specification");
      Wf_Type_Mark
        (Node          => Return_Type_Node,
         Current_Scope => Current_Scope,
         Context       => Dictionary.ProofContext,
         Type_Sym      => Type_Sym);
      if Dictionary.IsUnconstrainedArrayType (Type_Sym) then
         Type_Sym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.Semantic_Error
           (Err_Num   => 39,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Return_Type_Node),
            Id_Str    => Node_Lex_String (Node => Return_Type_Node));
      elsif Dictionary.IsPredefinedSuspensionObjectType (Type_Sym) or else Dictionary.TypeIsProtected (Type_Sym) then
         Type_Sym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.Semantic_Error
           (Err_Num   => 905,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Return_Type_Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
      if Subprog_Sym /= Dictionary.NullSymbol then
         Dictionary.AddReturnType
           (TheFunction   => Subprog_Sym,
            TypeMark      => Type_Sym,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            TypeReference => Dictionary.Location'(Start_Position => Node_Position (Node => Return_Type_Node),
                                                  End_Position   => Node_Position (Node => Return_Type_Node)));
      end if;
   end Check_Function_Specification;

begin -- Wf_Proof_Function_Declaration

   -- ASSUME Node = proof_function_declaration
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.proof_function_declaration,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = proof_function_declaration in Wf_Proof_Function_Declaration");
   Spec_Node := Child_Node (Current_Node => Node);
   -- ASSUME Spec_Node = function_specification
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Spec_Node) = SPSymbols.function_specification,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Spec_Node = function_specification in Wf_Proof_Function_Declaration");
   Constraint_Node := Next_Sibling (Current_Node => Spec_Node);
   -- ASSUME Constraint_Node = function_constraint
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Constraint_Node) = SPSymbols.function_constraint,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Constraint_Node = function_constraint in Wf_Proof_Function_Declaration");
   Check_Function_Specification (Node          => Spec_Node,
                                 Current_Scope => Scope,
                                 Subprog_Sym   => Func_Sym);
   if Func_Sym /= Dictionary.NullSymbol then
      Formal_Part_Node := Next_Sibling (Current_Node => Child_Node (Current_Node => Spec_Node));
      -- ASSUME Formal_Part_Node = formal_part OR type_mark
      if Syntax_Node_Type (Node => Formal_Part_Node) = SPSymbols.formal_part then
         -- ASSUME Formal_Part_Node = formal_part
         Wf_Formal_Part
           (Node             => Formal_Part_Node,
            Current_Scope    => Scope,
            Subprog_Sym      => Func_Sym,
            First_Occurrence => True,
            Context          => Dictionary.ProofContext);
      elsif Syntax_Node_Type (Node => Formal_Part_Node) /= SPSymbols.type_mark then
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Formal_Part_Node = formal_part OR type_mark in Wf_Proof_Function_Declaration");
      end if;
   end if;

   Null_Constraint_Node := Child_Node (Current_Node => Constraint_Node);
   -- ASSUME Null_Constraint_Node = precondition OR return_expression OR NULL
   if Syntax_Node_Type (Node => Null_Constraint_Node) = SPSymbols.precondition
     or else Syntax_Node_Type (Node => Null_Constraint_Node) = SPSymbols.return_expression then
      -- ASSUME Null_Constraint_Node = precondition OR return_expression
      ErrorHandler.Semantic_Error
        (Err_Num   => 315,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Constraint_Node),
         Id_Str    => LexTokenManager.Null_String);
   elsif Null_Constraint_Node /= STree.NullNode then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Null_Constraint_Node = precondition OR return_expression OR NULL in Wf_Proof_Function_Declaration");
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Proof_Function
        (Comp_Unit   => ContextManager.Ops.Current_Unit,
         Parse_Tree  => Node,
         Scope       => Scope,
         Subprog_Sym => Func_Sym);
   end if;

end Wf_Proof_Function_Declaration;
