-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit)
procedure wf_global_definition
  (Node         : in     STree.SyntaxNode;
   CurrentScope : in     Dictionary.Scopes;
   SubprogSym   : in     Dictionary.Symbol;
   FirstSeen    : in     Boolean;
   SemErrFound  :    out Boolean) is
   Abstraction   : Dictionary.Abstractions;
   DoingFunction : Boolean;

   ----------------------------------------------------------------------

   procedure GlobalNotRefined
     (GlobalSym  : in out Dictionary.Symbol;
      ErrorPos   : in     LexTokenManager.Token_Position;
      SubProgSym : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorPos,
   --#                                         GlobalSym,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubProgSym &
   --#         GlobalSym,
   --#         SemErrFound                from *,
   --#                                         Dictionary.Dict,
   --#                                         GlobalSym,
   --#                                         SubProgSym;
   is
   begin
      if Dictionary.IsRefinedOwnVariable (GlobalSym)
        and then IsEnclosingPackage (Dictionary.GetOwner (GlobalSym), Dictionary.LocalScope (SubProgSym)) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 314,
            Reference => ErrorHandler.No_Reference,
            Position  => ErrorPos,
            Id_Str    => Dictionary.GetSimpleName (GlobalSym));
         GlobalSym   := Dictionary.NullSymbol;
         SemErrFound := True;
      end if;
   end GlobalNotRefined;

   ----------------------------------------------------------------------

   procedure CheckStreamModeConsistency
     (GlobalSym  : in out Dictionary.Symbol;
      Mode       : in     Dictionary.Modes;
      ErrorPos   : in     LexTokenManager.Token_Position;
      SubProgSym : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorPos,
   --#                                         GlobalSym,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubProgSym &
   --#         GlobalSym,
   --#         SemErrFound                from *,
   --#                                         Dictionary.Dict,
   --#                                         GlobalSym,
   --#                                         Mode,
   --#                                         SubProgSym;
   is
      PossibleStreamMode : Dictionary.Modes;
   begin
      PossibleStreamMode := Dictionary.GetOwnVariableOrConstituentMode (GlobalSym);
      -- PossibleStreamMode contains DefaultMode unless the global is a stream
      -- variable or a stream constituent in which case it will be either InMode
      -- or OutMode (it can never be InOutMode because of earlier wffs).

      -- checks are only required if the mode is something other than DefaultMode
      if PossibleStreamMode /= Dictionary.DefaultMode then
         if Dictionary.IsFunction (SubProgSym) then
            -- required check is that global has default mode or mode in
            if PossibleStreamMode = Dictionary.OutMode then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 709,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (GlobalSym));
               GlobalSym   := Dictionary.NullSymbol;
               SemErrFound := True;
            end if;

         else -- handle procedure

            --  required check is that a moded own variable must have
            --  the same mode as the global mode if one is given.
            --  Also, global mode in out cannot be used if the own
            --  variable is moded.
            if Mode = Dictionary.InMode and then PossibleStreamMode /= Mode then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 711,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (GlobalSym));
               GlobalSym   := Dictionary.NullSymbol;
               SemErrFound := True;
            elsif Mode = Dictionary.OutMode and then PossibleStreamMode /= Mode then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 710,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (GlobalSym));
               GlobalSym   := Dictionary.NullSymbol;
               SemErrFound := True;
            elsif Mode = Dictionary.InOutMode then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 712,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => Dictionary.GetSimpleName (GlobalSym));
               GlobalSym   := Dictionary.NullSymbol;
               SemErrFound := True;
            end if;
         end if;
      end if;
   end CheckStreamModeConsistency;

   ----------------------------------------------------------------------

   procedure UniqueGlobal
     (GlobalSym  : in out Dictionary.Symbol;
      Dotted     : in     Boolean;
      ErrorPos   : in     LexTokenManager.Token_Position;
      SubprogSym : in     Dictionary.Symbol)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Dotted,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrorPos,
   --#                                         GlobalSym,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym &
   --#         GlobalSym,
   --#         SemErrFound                from *,
   --#                                         Abstraction,
   --#                                         Dictionary.Dict,
   --#                                         Dotted,
   --#                                         GlobalSym,
   --#                                         LexTokenManager.State,
   --#                                         SubprogSym;
   is
      It        : Dictionary.Iterator;
      GlobalStr : LexTokenManager.Lex_String;

   begin
      GlobalStr := Dictionary.GetSimpleName (GlobalSym);
      if not Dotted and then Dictionary.IsSubprogram (SubprogSym) then -- task types have no parameters so skip search
         It := Dictionary.FirstSubprogramParameter (SubprogSym);
         while It /= Dictionary.NullIterator loop
            if LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => GlobalStr,
               Lex_Str2 => Dictionary.GetSimpleName (Dictionary.CurrentSymbol (It))) =
              LexTokenManager.Str_Eq then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 158,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => GlobalStr);
               GlobalSym   := Dictionary.NullSymbol;
               SemErrFound := True;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;

      if GlobalSym /= Dictionary.NullSymbol then
         It := Dictionary.FirstGlobalVariable (Abstraction, SubprogSym);
         while It /= Dictionary.NullIterator loop
            if GlobalSym = Dictionary.CurrentSymbol (It) then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 157,
                  Reference => ErrorHandler.No_Reference,
                  Position  => ErrorPos,
                  Id_Str    => GlobalStr);
               GlobalSym   := Dictionary.NullSymbol;
               SemErrFound := True;
               exit;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
   end UniqueGlobal;

   ----------------------------------------------------------------------

   procedure CheckImportInit (ImportNode            : in STree.SyntaxNode;
                              ImportSym, SubprogSym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ImportNode,
   --#                                         ImportSym,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         SemErrFound                from *,
   --#                                         Dictionary.Dict,
   --#                                         ImportSym,
   --#                                         SubprogSym;
   is

   begin
      if Dictionary.IsMainProgram (SubprogSym)
        and then Dictionary.IsFunction (SubprogSym)
        and then not Dictionary.OwnVariableIsInitialized (ImportSym)
        and then Dictionary.GetOwnVariableOrConstituentMode (ImportSym) = Dictionary.DefaultMode then
         SemErrFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 167,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ImportNode),
            Id_Str    => Dictionary.GetSimpleName (ImportSym));
      end if;
   end CheckImportInit;

   -----------------------------------------------------------------------

   procedure CheckOnGlobalRefinement_1
     (SubprogSym : in Dictionary.Symbol;
      VarSym     : in Dictionary.Symbol;
      ErrPos     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrPos,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym,
   --#                                         VarSym &
   --#         SemErrFound                from *,
   --#                                         Dictionary.Dict,
   --#                                         SubprogSym,
   --#                                         VarSym;
   is
      SubjectSym : Dictionary.Symbol;
      Found      : Boolean;
      It         : Dictionary.Iterator;

      function ValidRefinement (Constituent, Subprogram : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Owner           : Dictionary.Symbol;
         Region          : Dictionary.Symbol;
         EnclosingRegion : Dictionary.Symbol;
         Result          : Boolean;
      begin
         -- We regard Constituent as a ValidRefinement if:
         --   It is a refinement constituent of Subject AND
         --      (Subject is owned by the region in which the Subprogram is declared OR
         --       Subject is owned by the region in which the protected type in which the
         --          Subprogram is declared)
         Owner  := Dictionary.GetOwner (Dictionary.GetSubject (Constituent));
         Region := Dictionary.GetRegion (Dictionary.GetScope (Subprogram));

         Result := Owner = Region;
         if not Result and then Dictionary.IsProtectedType (Region) then
            EnclosingRegion := Dictionary.GetRegion (Dictionary.GetScope (Region));
            Result          := Owner = EnclosingRegion;
         end if;
         return Result;
      end ValidRefinement;

   begin
      if Dictionary.IsConstituent (VarSym) then
         -- look for refinement subject in first global anno
         SubjectSym := Dictionary.GetSubject (VarSym);
         -- if subject not from this package then use own var itself
         --if Dictionary.GetOwner (SubjectSym) /=
         --   Dictionary.GetRegion (Dictionary.GetScope (SubprogSym))
         if not ValidRefinement (VarSym, SubprogSym) then
            SubjectSym := VarSym;
         end if;
      else
         --look for global itself in first global anno
         SubjectSym := VarSym;
      end if;

      Found := False;
      It    := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, SubprogSym);
      loop
         exit when Dictionary.IsNullIterator (It);
         Found := SubjectSym = Dictionary.CurrentSymbol (It);
         exit when Found;
         It := Dictionary.NextSymbol (It);
      end loop;
      if not Found then
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 85,
            Reference => ErrorHandler.No_Reference,
            Position  => ErrPos,
            Sym       => VarSym,
            Scope     => Dictionary.GetScope (SubprogSym));
         SemErrFound := True;
      end if;
   end CheckOnGlobalRefinement_1;

   ----------------------------------------------------------------------

   -- Handle Special case for stream variables here:
   -- Error occurs if the subprogram references a global which is a mode IN refinement
   -- constituent whose abstract subject is unmoded.  This condition is illegal for
   -- a function subprogram (because it would conceal the stream side-efefct in the
   -- abstract view).  For a procedure, we are protected because derives refinement
   -- checks will always reveal the error; however, we can improve error reporting
   -- by putting something out here as well
   procedure CheckStreamVariableRefinement (SubprogSym : in Dictionary.Symbol;
                                            ErrPos     :    LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrPos,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym &
   --#         SemErrFound                from *,
   --#                                         Dictionary.Dict,
   --#                                         SubprogSym;
   is
      RefinedGlobal   : Dictionary.Symbol;
      AbstractGlobal  : Dictionary.Symbol;
      RefinedGlobList : Dictionary.Iterator;

      function IsLocalConstituent (Sym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      --#        in RefinedGlobal;
      --#        in SubprogSym;
      is
      begin
         return Dictionary.IsConstituent (Sym)
           and then (Dictionary.GetEnclosingPackage (Dictionary.GetScope (SubprogSym)) =
                       Dictionary.GetOwner (RefinedGlobal));
      end IsLocalConstituent;

   begin --CheckStreamVariableRefinement
      RefinedGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsRefined, SubprogSym);
      while not Dictionary.IsNullIterator (RefinedGlobList) loop
         RefinedGlobal := Dictionary.CurrentSymbol (RefinedGlobList);
         if IsLocalConstituent (RefinedGlobal) then
            AbstractGlobal := Dictionary.GetSubject (RefinedGlobal);
            if Dictionary.GetOwnVariableOrConstituentMode (RefinedGlobal) = Dictionary.InMode
              and then
              -- Abstract has no mode
              Dictionary.GetOwnVariableOrConstituentMode (AbstractGlobal) = Dictionary.DefaultMode
              and then
              -- Abstract is not protected
              not (Dictionary.IsOwnVariable (AbstractGlobal) and then Dictionary.GetOwnVariableProtected (AbstractGlobal)) then
               -- an error condition may exist
               --# accept Flow, 41, "Expected stable expression";
               if Dictionary.IsAdaFunction (SubprogSym) then
                  --# end accept;
                  -- an error condition does exist
                  ErrorHandler.Semantic_Error2
                    (Err_Num   => 721,
                     Reference => ErrorHandler.No_Reference,
                     Position  => ErrPos,
                     Id_Str1   => Dictionary.GetSimpleName (RefinedGlobal),
                     Id_Str2   => Dictionary.GetSimpleName (AbstractGlobal));
                  SemErrFound := True;
               else -- must be procedure

                  --  extra checks here - for there to be an error the
                  --  abstract own variable must be have global mode
                  --  IN and the refinement constituent as well
                  if Dictionary.GetGlobalMode (Dictionary.IsAbstract, SubprogSym, AbstractGlobal) = Dictionary.InMode
                    and then Dictionary.GetGlobalMode (Dictionary.IsRefined, SubprogSym, RefinedGlobal) = Dictionary.InMode then
                     ErrorHandler.Semantic_Error2
                       (Err_Num   => 722,
                        Reference => ErrorHandler.No_Reference,
                        Position  => ErrPos,
                        Id_Str1   => Dictionary.GetSimpleName (RefinedGlobal),
                        Id_Str2   => Dictionary.GetSimpleName (AbstractGlobal));
                     SemErrFound := True;
                  end if;
               end if;
            end if;
         end if;
         RefinedGlobList := Dictionary.NextSymbol (RefinedGlobList);
      end loop;
   end CheckStreamVariableRefinement;

   ----------------------------------------------------------------------

   function TypeTwoErrorPosition (Node : STree.SyntaxNode) return LexTokenManager.Token_Position
   --# global in STree.Table;
   is
      TempNode : STree.SyntaxNode;
   begin
      --assume Node = global_definition
      TempNode := Child_Node (Child_Node (Node));
      if Syntax_Node_Type (Node => TempNode) = SPSymbols.global_definition_rep then
         TempNode := Next_Sibling (TempNode);
      end if;
      -- TempNode now points at the right-most global_variable_clause
      TempNode := Child_Node (Next_Sibling (Child_Node (TempNode)));
      -- TempNode now either a global_variable or a global_variable_list
      if Syntax_Node_Type (Node => TempNode) = SPSymbols.global_variable_list then
         TempNode := Next_Sibling (TempNode);
      end if;
      -- TempNode is right-most global variable
      return Node_Position (Node => TempNode);
   end TypeTwoErrorPosition;

   ----------------------------------------------------------------------

   procedure CheckOnGlobalRefinement_2 (SubprogSym : in Dictionary.Symbol;
                                        ErrPos     :    LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         ErrPos,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         SubprogSym &
   --#         SemErrFound                from *,
   --#                                         Dictionary.Dict,
   --#                                         SubprogSym;
   is
      AGloVar : Dictionary.Symbol;
      --Found          : Boolean;
      FirstGlobList : Dictionary.Iterator;
      --SecondGlobList : Dictionary.Iterator;

      -- lifted out from loop in enclosing procedure below
      procedure LookForRefinementConstituent
      --# global in     AGloVar;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ErrPos;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SemErrFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from AGloVar,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrPos,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         SubprogSym &
      --#         SemErrFound                from *,
      --#                                         AGloVar,
      --#                                         Dictionary.Dict,
      --#                                         SubprogSym;
      is
         Found          : Boolean;
         SecondGlobList : Dictionary.Iterator;

      begin -- LookForRefinementConstituent
         Found          := False;
         SecondGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsRefined, SubprogSym);
         loop
            exit when Dictionary.IsNullIterator (SecondGlobList);
            Found := Dictionary.IsRefinement (AGloVar, Dictionary.CurrentSymbol (SecondGlobList));
            exit when Found;
            SecondGlobList := Dictionary.NextSymbol (SecondGlobList);
         end loop;
         if not Found then
            ErrorHandler.Semantic_Error_Sym
              (Err_Num   => 86,
               Reference => ErrorHandler.No_Reference,
               Position  => ErrPos,
               Sym       => AGloVar,
               Scope     => Dictionary.GetScope (SubprogSym));
            SemErrFound := True;
         end if;
      end LookForRefinementConstituent;

      -- new procedure similar to above but for unrefined global items
      procedure LookForSelf
      --# global in     AGloVar;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     ErrPos;
      --#        in     LexTokenManager.State;
      --#        in     SubprogSym;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SemErrFound;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from AGloVar,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         ErrPos,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         SubprogSym &
      --#         SemErrFound                from *,
      --#                                         AGloVar,
      --#                                         Dictionary.Dict,
      --#                                         SubprogSym;
      is
         Found          : Boolean;
         SecondGlobList : Dictionary.Iterator;

      begin -- LookForSelf
         Found          := False;
         SecondGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsRefined, SubprogSym);
         loop
            exit when Dictionary.IsNullIterator (SecondGlobList);
            Found := AGloVar = Dictionary.CurrentSymbol (SecondGlobList);
            exit when Found;
            SecondGlobList := Dictionary.NextSymbol (SecondGlobList);
         end loop;
         if not Found and then AGloVar /= Dictionary.GetNullVariable then -- no refinement of "null" needed
            ErrorHandler.Semantic_Error_Sym
              (Err_Num   => 723,
               Reference => ErrorHandler.No_Reference,
               Position  => ErrPos,
               Sym       => AGloVar,
               Scope     => Dictionary.GetScope (SubprogSym));
            SemErrFound := True;
         end if;
      end LookForSelf;

      function ConstituentRequired (TheGlobal, Subprogram : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Result : Boolean := False;
         Region : Dictionary.Symbol;
      begin
         -- We expect a refinement sonstituent of TheGlobal if:
         --   TheGlobal is an abstract own variable  and
         --      (its owner is the package in which Subprogram is declared or
         --       the Subprogram is declared in a protected type and the owner
         --       of TheGlobal is the package in which the protected type is declared)
         if Dictionary.IsRefinedOwnVariable (TheGlobal) then
            Result := IsEnclosingPackage (Dictionary.GetOwner (TheGlobal), Dictionary.GetScope (Subprogram));
            if not Result then
               Region := Dictionary.GetRegion (Dictionary.GetScope (Subprogram));
               if Dictionary.IsProtectedType (Region) then
                  Result := IsEnclosingPackage (Dictionary.GetOwner (TheGlobal), Dictionary.GetScope (Region));
               end if;
            end if;
         end if;
         return Result;
      end ConstituentRequired;

   begin -- CheckOnGlobalRefinement_2
      FirstGlobList := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, SubprogSym);
      while not Dictionary.IsNullIterator (FirstGlobList) loop
         AGloVar := Dictionary.CurrentSymbol (FirstGlobList);
         if ConstituentRequired (AGloVar, SubprogSym) then
            LookForRefinementConstituent; -- at least on constituent must be present
         else
            LookForSelf;                  -- own variable itself must be present
         end if;
         FirstGlobList := Dictionary.NextSymbol (FirstGlobList);
      end loop;
   end CheckOnGlobalRefinement_2;

   ----------------------------------------------------------------------

   procedure CheckMainProgramGlobal
     (Node         : in     STree.SyntaxNode;
      CurrentScope : in     Dictionary.Scopes;
      VarSym       :    out Dictionary.Symbol;
      Dotted       :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dotted,
   --#         STree.Table,
   --#         VarSym                     from CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
      IdNode, NextIdNode : STree.SyntaxNode;
      PIdStr, IdStr      : LexTokenManager.Lex_String;
      PSym, Sym          : Dictionary.Symbol;

   begin
      IdNode := Last_Child_Of (Start_Node => Node);
      IdStr  := Node_Lex_String (Node => IdNode);
      PIdStr := LexTokenManager.Null_String;
      Sym    := Dictionary.LookupItem (Name              => IdStr,
                                       Scope             => CurrentScope,
                                       Context           => Dictionary.ProofContext,
                                       Full_Package_Name => False);
      loop  -- introduced for multiple prefixes
         if Sym = Dictionary.NullSymbol then
            ErrorHandler.Semantic_Error2
              (Err_Num   => 144,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str1   => IdStr,
               Id_Str2   => PIdStr);
            exit;
         end if;

         NextIdNode := Next_Sibling (Parent_Node (Current_Node => IdNode));

         if Dictionary.IsOwnVariable (Sym) then
            -- entire variable check
            -- at this point Sym is a variable, final check that there is no dotted
            -- part to the right of it as there would be if a record field was there
            if NextIdNode /= STree.NullNode then
               ErrorHandler.Semantic_Error
                 (Err_Num   => 156,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Node),
                  Id_Str    => LexTokenManager.Null_String);
               Sym := Dictionary.NullSymbol;
            end if;
            exit;
         end if;

         if not Dictionary.IsPackage (Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 174,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => IdNode),
               Id_Str    => IdStr);
            Sym := Dictionary.NullSymbol;
            exit;
         end if;

         if NextIdNode = STree.NullNode then
            -- package without a selected component
            ErrorHandler.Semantic_Error
              (Err_Num   => 174,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => IdNode),
               Id_Str    => IdStr);
            Sym := Dictionary.NullSymbol;
            exit;
         end if;
         STree.Set_Node_Lex_String (Sym  => Sym,
                                    Node => IdNode);
         PIdStr := IdStr;
         IdNode := NextIdNode;
         IdStr  := Node_Lex_String (Node => IdNode);
         PSym   := Sym;
         Sym    :=
           Dictionary.LookupSelectedItem
           (Prefix   => PSym,
            Selector => IdStr,
            Scope    => Dictionary.VisibleScope (Sym),
            Context  => Dictionary.ProofContext);
         if Sym = Dictionary.NullSymbol then
            -- need also to search current scope for inherited child packages
            Sym :=
              Dictionary.LookupSelectedItem
              (Prefix   => PSym,
               Selector => IdStr,
               Scope    => CurrentScope,
               Context  => Dictionary.ProofContext);
         end if;
      end loop;

      Dotted :=
        LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => PIdStr,
                                                             Lex_Str2 => LexTokenManager.Null_String) /=
        LexTokenManager.Str_Eq
        and then Sym /= Dictionary.NullSymbol;
      VarSym := Sym;
   end CheckMainProgramGlobal;

   --------------------------------------------------

   procedure ProcessOneGlobal (Node : in STree.SyntaxNode;
                               Mode : in Dictionary.Modes)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     CurrentScope;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in     SubprogSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict            from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         ErrorHandler.Error_Context from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         SemErrFound                from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         STree.Table                from *,
   --#                                         CommandLineData.Content,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SubprogSym;
   --  pre     Syntax_Node_Type (Node => Node) = SPSymbols.global_variable;
   is
      Dotted    : Boolean;
      Sym       : Dictionary.Symbol;
      ParamMode : Dictionary.Modes;
   begin --ProcessOneGlobal
      if Dictionary.IsMainProgram (SubprogSym) then
         CheckMainProgramGlobal (Child_Node (Node), Dictionary.LocalScope (SubprogSym),
                                 --to get
                                 Sym, Dotted);
      else
         wf_entire_variable (Child_Node (Node), CurrentScope, In_Global_List,
                             --to get
                             Sym, Dotted);
      end if;

      --# assert True;
      if Sym = Dictionary.NullSymbol then
         SemErrFound := True;
      end if;

      if CommandLineData.Ravenscar_Selected
        and then (SubprogSym /= Dictionary.GetThePartition)
        and then Dictionary.IsOwnVariable (Sym)
        and then Dictionary.GetOwnVariableIsInterruptStream (Sym) then
         -- An interrupt stream variable is being used outside the partition
         -- wide flow annotation.
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 955,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Sym       => Sym,
            Scope     => CurrentScope);
         Sym := Dictionary.NullSymbol;
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol
        and then (Dictionary.IsMainProgram (SubprogSym) or Dictionary.IsTaskType (SubprogSym)) then
         -- check to ensure that global is initialized
         if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
           and then not CommandLineData.Content.Do_Information_Flow
           and then (Mode = Dictionary.InMode or Mode = Dictionary.InOutMode)
           and then not Dictionary.OwnVariableIsInitialized (Sym)
           and then Dictionary.GetOwnVariableOrConstituentMode (Sym) = Dictionary.DefaultMode then
            ErrorHandler.Semantic_Error
              (Err_Num   => 167,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Dictionary.GetSimpleName (Sym));
         end if;
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         GlobalNotRefined (Sym, Node_Position (Node => Child_Node (Node)), SubprogSym);
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         UniqueGlobal (Sym, Dotted, Node_Position (Node => Child_Node (Node)), SubprogSym);
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         CheckStreamModeConsistency (Sym, Mode, Node_Position (Node => Child_Node (Node)), SubprogSym);
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol and then Dictionary.IsOwnVariable (Sym) and then Dictionary.IsVirtualElement (Sym) then
         if Dictionary.IsOrIsInProtectedScope (CurrentScope)
           and then Dictionary.GetEnclosingProtectedRegion (CurrentScope) =
           Dictionary.GetRootType (Dictionary.GetType (Dictionary.GetVirtualElementOwner (Sym))) then

            -- Mark the virtual element as having been seen by its "owning"
            -- protected type.
            --
            Dictionary.SetVirtualElementSeenByOwner (Sym);
         else
            -- This is an access to a protected virtual element outside
            -- it's protected type.
            --
            ErrorHandler.Semantic_Error_Sym2
              (Err_Num   => 946,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Sym       => Sym,
               Sym2      => Dictionary.GetRootType (Dictionary.GetType (Dictionary.GetVirtualElementOwner (Sym))),
               Scope     => CurrentScope);
         end if;
      end if;

      --# assert True;
      if Sym /= Dictionary.NullSymbol then
         if not FirstSeen then
            CheckOnGlobalRefinement_1 (SubprogSym, Sym, Node_Position (Node => Child_Node (Node)));
         end if;

         CheckImportInit (Node, Sym, SubprogSym);

         -- check that mode on a global of an embedded procedure is compatible with
         -- parameters mode if the global is refering to a parameter of an enclosing
         -- procedure
         if Dictionary.IsSubprogramParameter (Sym) then
            ParamMode := Dictionary.GetSubprogramParameterMode (Sym);
            if ParamMode = Dictionary.InMode or ParamMode = Dictionary.DefaultMode then
               if Mode /= Dictionary.InMode and Mode /= Dictionary.DefaultMode then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 508,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Node),
                     Id_Str    => LexTokenManager.Null_String);

               end if;
            end if;
         end if;

         Dictionary.AddGlobalVariable
           (Abstraction       => Abstraction,
            Subprogram        => SubprogSym,
            Variable          => Sym,
            Mode              => Mode,
            PrefixNeeded      => Dotted,
            Comp_Unit         => ContextManager.Ops.Current_Unit,
            VariableReference => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                      End_Position   => Node_Position (Node => Node)));

         -- Mark the global variable as being referenced by a thread if the
         -- owner of this global variable is a thread (i.e. task, main program or
         -- interrupt handler) and the variable is not protected. This data is
         -- required by the main program shared variable checks.
         --
         if Dictionary.IsThread (SubprogSym)
           and then Dictionary.IsOwnVariable (Sym)
           and then not Dictionary.GetOwnVariableProtected (Sym) then
            Dictionary.SetUsesUnprotectedVariables (SubprogSym);
         end if;

      end if;
   end ProcessOneGlobal;

   ----------------------------------------------------------------------

   procedure ProcessGlobalList (Node : in STree.SyntaxNode;
                                Mode : in Dictionary.Modes)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     CurrentScope;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in     SubprogSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         SemErrFound                from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Mode,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym;
   is
      It : STree.Iterator;
   begin
      It := Find_First_Node (Node_Kind    => SPSymbols.global_variable,
                             From_Root    => Node,
                             In_Direction => STree.Down);

      while not STree.IsNull (It) loop
         ProcessOneGlobal (Get_Node (It => It), Mode);
         It := STree.NextNode (It);
      end loop;
   end ProcessGlobalList;

   ----------------------------------------------------------------------

   procedure wf_moded_global_definition (Node : in STree.SyntaxNode)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     CurrentScope;
   --#        in     DoingFunction;
   --#        in     FirstSeen;
   --#        in     LexTokenManager.State;
   --#        in     SubprogSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SemErrFound;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         DoingFunction,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         SubprogSym &
   --#         SemErrFound                from *,
   --#                                         Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         CurrentScope,
   --#                                         Dictionary.Dict,
   --#                                         DoingFunction,
   --#                                         FirstSeen,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         STree.Table,
   --#                                         SubprogSym;
   is
      It       : STree.Iterator;
      NextNode : STree.SyntaxNode;
      Mode     : Dictionary.Modes;

      function GetMode (Node : STree.SyntaxNode) return Dictionary.Modes
      --# global in STree.Table;
      --  pre    Syntax_Node_Type (Node => Node) = SPSymbols.global_variable_clause;
      is
         ModeNode : STree.SyntaxNode;
         Result   : Dictionary.Modes;
      begin
         ModeNode := Child_Node (Child_Node (Node));
         if ModeNode = STree.NullNode then
            Result := Dictionary.DefaultMode;
         else
            case Syntax_Node_Type (Node => ModeNode) is
               when SPSymbols.in_mode =>
                  Result := Dictionary.InMode;
               when SPSymbols.inout_mode =>
                  Result := Dictionary.InOutMode;
               when SPSymbols.out_mode =>
                  Result := Dictionary.OutMode;
               when others =>
                  Result := Dictionary.DefaultMode; --cannot occur
            end case;
         end if;
         return Result;
      end GetMode;

   begin -- wf_moded_global_definition
      It := Find_First_Node (Node_Kind    => SPSymbols.global_variable_clause,
                             From_Root    => Node,
                             In_Direction => STree.Down);

      while not STree.IsNull (It) loop
         NextNode := Get_Node (It => It);
         Mode     := GetMode (NextNode);
         --# accept Flow, 41, "Expected stable expression";
         if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83 then
            --# end accept;
            if not CommandLineData.Content.Do_Information_Flow
              and then not DoingFunction
              and then Mode = Dictionary.DefaultMode then
               SemErrFound := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 500,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => NextNode),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         end if;

         if DoingFunction and then (Mode = Dictionary.OutMode or else Mode = Dictionary.InOutMode) then
            SemErrFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 169,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => NextNode),
               Id_Str    => LexTokenManager.Null_String);
         end if;

         ProcessGlobalList (Next_Sibling (Child_Node (NextNode)), Mode);
         It := STree.NextNode (It);
      end loop;

   end wf_moded_global_definition;

begin -- wf_global_definition

   -- ASSUME Node = moded_global_definition
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.moded_global_definition,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = moded_global_definition in Wf_Global_Definition");
   SemErrFound   := False;
   DoingFunction := Dictionary.IsFunction (SubprogSym);

   if FirstSeen then
      Abstraction := Dictionary.IsAbstract;
   else
      Abstraction := Dictionary.IsRefined;
   end if;

   Dictionary.AddGlobalAnnotation
     (Abstraction => Abstraction,
      Subprogram  => SubprogSym,
      Comp_Unit   => ContextManager.Ops.Current_Unit,
      Annotation  => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                          End_Position   => Node_Position (Node => Node)));

   wf_moded_global_definition (Node);

   if not FirstSeen then
      CheckOnGlobalRefinement_2 (SubprogSym, TypeTwoErrorPosition (Node));
      CheckStreamVariableRefinement (SubprogSym, TypeTwoErrorPosition (Node));
   end if;

   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Global
        (Comp_Unit   => ContextManager.Ops.Current_Unit,
         Parse_Tree  => Node,
         Scope       => CurrentScope,
         Subprog_Sym => SubprogSym);
   end if;

end wf_global_definition;
