-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_function_constraint (Node      : in STree.SyntaxNode;
                                  Scope     : in Dictionary.Scopes;
                                  FirstSeen : in Boolean) is
   ConNode                : STree.SyntaxNode;
   ErrorsFound            : Boolean := False;
   ErrorsFoundInPredicate : Boolean;
   -- look up table: if FirstSeen then we are dealing with Abstract spec else Refined
   type WhichAbstractions is array (Boolean) of Dictionary.Abstractions;
   WhichAbstraction : constant WhichAbstractions :=
     WhichAbstractions'(False => Dictionary.IsRefined,
                        True  => Dictionary.IsAbstract);

   ---------------------------------------------

   procedure wf_return_expression (Node  : in STree.SyntaxNode;
                                   Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     FirstSeen;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out ErrorsFound;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         ErrorsFound,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         FirstSeen,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FirstSeen,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      RetExp                 : Exp_Record;
      ImplicitNode           : STree.SyntaxNode;
      ImplicitStr            : LexTokenManager.Lex_String;
      ImplicitVar            : Dictionary.Symbol;
      ReturnType             : Dictionary.Symbol;
      ErrorsFoundInPredicate : Boolean;
   begin
      --node is node below return_expression
      if Syntax_Node_Type (Node => Node) = SPSymbols.annotation_expression then
         ReturnType := Dictionary.GetType (Dictionary.GetRegion (Scope));
         WalkAnnotationExpression
           (Exp_Node       => Node,
            Scope          => Scope,
            Type_Context   => ReturnType,
            Context        => FunctionReturn,
            Result         => RetExp,
            Component_Data => GlobalComponentData);
         ErrorsFound := ErrorsFound or (RetExp.Errors_In_Expression);

         AssignmentCheck (Position   => Node_Position (Node => Node),
                          Scope      => Scope,
                          TargetType => ReturnType,
                          ExpResult  => RetExp);
         ErrorsFound := ErrorsFound or (RetExp.Errors_In_Expression);

      else --must be simple_name of implicit return expression
         ImplicitNode := Child_Node (Node);
         ImplicitStr  := Node_Lex_String (Node => ImplicitNode);
         if Dictionary.IsDefined (Name              => ImplicitStr,
                                  Scope             => Scope,
                                  Context           => Dictionary.ProofContext,
                                  Full_Package_Name => False) then
            ErrorsFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => ImplicitStr);
         else --valid implicit return variable so add it and then wf predicate
            Dictionary.AddImplicitReturnVariable
              (Abstraction => WhichAbstraction (FirstSeen),
               Comp_Unit   => ContextManager.Ops.Current_Unit,
               Declaration => Dictionary.Location'(Start_Position => Node_Position (Node => ImplicitNode),
                                                   End_Position   => Node_Position (Node => ImplicitNode)),
               Name        => ImplicitStr,
               TheFunction => Dictionary.GetRegion (Scope),
               Variable    => ImplicitVar);
            wf_predicate
              (Node        => Next_Sibling (Node),
               Scope       => Dictionary.LocalScope (ImplicitVar),
               Context     => FunctionReturn,
               ErrorsFound => ErrorsFoundInPredicate);
            ErrorsFound := ErrorsFound or ErrorsFoundInPredicate;
         end if;
      end if;
   end wf_return_expression;

   ---------------------------------------------

begin --wf_function_constraint
      --assume node is function_constraint
   ConNode := Child_Node (Node);
   if ConNode /= STree.NullNode then
      if Syntax_Node_Type (Node => ConNode) = SPSymbols.precondition then
         Dictionary.AddPrecondition
           (WhichAbstraction (FirstSeen),
            Dictionary.GetRegion (Scope),
            STree.NodeToRef (Child_Node (ConNode)),
            Dictionary.Location'(Start_Position => Node_Position (Node => ConNode),
                                 End_Position   => Node_Position (Node => ConNode)));
         wf_predicate
           (Node        => Child_Node (ConNode),
            Scope       => Scope,
            Context     => Precondition,
            ErrorsFound => ErrorsFoundInPredicate);
         ErrorsFound := ErrorsFound or ErrorsFoundInPredicate;
         ConNode     := Next_Sibling (ConNode);
      end if;

      if Syntax_Node_Type (Node => ConNode) = SPSymbols.return_expression then
         Dictionary.AddPostcondition
           (WhichAbstraction (FirstSeen),
            Dictionary.GetRegion (Scope),
            STree.NodeToRef (Child_Node (ConNode)),
            Dictionary.Location'(Start_Position => Node_Position (Node => ConNode),
                                 End_Position   => Node_Position (Node => ConNode)));
         wf_return_expression (Node  => Child_Node (ConNode),
                               Scope => Scope);
      end if;
      if ErrorsFound then
         Dictionary.SetSubprogramSignatureNotWellformed (WhichAbstraction (FirstSeen), Dictionary.GetRegion (Scope));
      end if;
   end if;
end wf_function_constraint;
