-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--Synopsis
--This procedure checks the validity of a pragma interface (Ada83) or pragma
--import (Ada95).  The checks made are:
-- 1.  Internal consistency of associations used, number of parameters etc.
-- 2.  The Entity/Subprogram name is that expected
--------------------------------------------------------------------------------

separate (Sem.CompUnit)
procedure wf_external_interface
  (Pragma_Node : in     STree.SyntaxNode;
   Entity_Sym  : in     Dictionary.Symbol;
   Error_Found :    out Boolean) is

   procedure CheckRepresentSameName (Exp_Node   : in STree.SyntaxNode;
                                     Entity_Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out Error_Found;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found,
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         Exp_Node,
   --#                                         LexTokenManager.State,
   --#                                         STree.Table;
   is
      IsChain          : Boolean;
      IdNode, NextNode : STree.SyntaxNode;
      Name             : LexTokenManager.Lex_String;
   begin
      Name   := Dictionary.GetSimpleName (Item => Entity_Sym);
      IdNode := Exp_Node;
      loop
         IsChain  := Next_Sibling (IdNode) = STree.NullNode;
         NextNode := Child_Node (IdNode);
         exit when not IsChain or NextNode = STree.NullNode;

         IdNode := NextNode;
      end loop;

      if IsChain
        and then Syntax_Node_Type (Node => IdNode) = SPSymbols.identifier
        and then LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => Node_Lex_String (Node => IdNode),
         Lex_Str2 => Name) =
        LexTokenManager.Str_Eq then
         STree.Set_Node_Lex_String (Sym  => Entity_Sym,
                                    Node => IdNode);
      else
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 71,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Exp_Node),
            Id_Str    => Name);
      end if;
   end CheckRepresentSameName;

   ------------------------------------------------------------------

   procedure wf_pragma_interface (Pragma_Node : in STree.SyntaxNode;
                                  Entity_Sym  : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out Error_Found;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found,
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         STree.Table;
   is
      ArgAssocRepNode : STree.SyntaxNode;
      SubprogNameNode : STree.SyntaxNode;
   begin
      ArgAssocRepNode := Child_Node (Next_Sibling (Child_Node (Pragma_Node)));
      if Syntax_Node_Type (Node => Child_Node (ArgAssocRepNode)) = SPSymbols.argument_association then --pragma has two arguments
         SubprogNameNode := Child_Node (Next_Sibling (ArgAssocRepNode));

         if Syntax_Node_Type (Node => SubprogNameNode) /= SPSymbols.ADA_expression then --form of expression wrong
            Error_Found := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 71,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => SubprogNameNode),
               Id_Str    => Dictionary.GetSimpleName (Item => Entity_Sym));
         else --form of expression ok so check name actually matches
            CheckRepresentSameName (Exp_Node   => SubprogNameNode,
                                    Entity_Sym => Entity_Sym);
         end if;

      else --pragma does nor have exatcly 2 arguments
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 69,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Pragma_Node),
            Id_Str    => LexTokenManager.Interface_Token);
      end if;
   end wf_pragma_interface;

   ----------------------

   procedure wf_pragma_import (Pragma_Node : in STree.SyntaxNode;
                               Entity_Sym  : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out Error_Found;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         Error_Found,
   --#         STree.Table                from *,
   --#                                         Dictionary.Dict,
   --#                                         Entity_Sym,
   --#                                         LexTokenManager.State,
   --#                                         Pragma_Node,
   --#                                         STree.Table;
   is
      type Args is (Illegal, Convention, Entity, ExternalName, LinkName);
      subtype LegalArgs is Args range Convention .. LinkName;
      type Founds is array (LegalArgs) of Boolean;

      Found                 : Founds           := Founds'(LegalArgs => False);
      UsingNamedAssociation : Boolean          := False;
      ArgAssNode            : STree.SyntaxNode;
      ArgCount              : Natural          := 0;
      MaxArgs               : constant Natural := 4;

      procedure CheckArgument (Node       : in STree.SyntaxNode;
                               Entity_Sym : in Dictionary.Symbol)
      --# global in     ArgCount;
      --#        in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in out ErrorHandler.Error_Context;
      --#        in out Error_Found;
      --#        in out Found;
      --#        in out SPARK_IO.File_Sys;
      --#        in out STree.Table;
      --#        in out UsingNamedAssociation;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from ArgCount,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         Entity_Sym,
      --#                                         ErrorHandler.Error_Context,
      --#                                         Found,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table,
      --#                                         UsingNamedAssociation &
      --#         Error_Found,
      --#         STree.Table                from *,
      --#                                         ArgCount,
      --#                                         Dictionary.Dict,
      --#                                         Entity_Sym,
      --#                                         Found,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         UsingNamedAssociation &
      --#         Found                      from *,
      --#                                         ArgCount,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         STree.Table,
      --#                                         UsingNamedAssociation &
      --#         UsingNamedAssociation      from *,
      --#                                         Node,
      --#                                         STree.Table;
      is
         ExpNode : STree.SyntaxNode;
         Arg     : Args;

         function GetArg (ArgString : LexTokenManager.Lex_String) return Args
         --# global in LexTokenManager.State;
         is
            Result : Args;
         begin
            if LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => ArgString,
               Lex_Str2 => LexTokenManager.Convention_Token) =
              LexTokenManager.Str_Eq then
               Result := Convention;
            elsif LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => ArgString,
               Lex_Str2 => LexTokenManager.Entity_Token) =
              LexTokenManager.Str_Eq then
               Result := Entity;
            elsif LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => ArgString,
               Lex_Str2 => LexTokenManager.External_Name_Token) =
              LexTokenManager.Str_Eq then
               Result := ExternalName;
            elsif LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => ArgString,
               Lex_Str2 => LexTokenManager.Link_Name_Token) =
              LexTokenManager.Str_Eq then
               Result := LinkName;
            else
               Result := Illegal;
            end if;
            return Result;
         end GetArg;

         ---------------

      begin --CheckArgument
         ExpNode := Child_Node (Node);
         --must be an identifier or an ADA_expression
         if Syntax_Node_Type (Node => ExpNode) = SPSymbols.identifier then
            --named association
            UsingNamedAssociation := True;
            Arg                   := GetArg (Node_Lex_String (Node => ExpNode));
            if Arg = Illegal then
               Error_Found := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 601,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ExpNode),
                  Id_Str    => LexTokenManager.Null_String);
            elsif Found (Arg) then
               Error_Found := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 602,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ExpNode),
                  Id_Str    => Node_Lex_String (Node => ExpNode));
            else
               Found (Arg) := True;
               if Arg = Entity then
                  ExpNode := Next_Sibling (ExpNode);
                  CheckRepresentSameName (Exp_Node   => ExpNode,
                                          Entity_Sym => Entity_Sym);
               end if;
            end if;

         else --positional association
            if UsingNamedAssociation then --illegal switch form named to positional assoc
               Error_Found := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 601,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ExpNode),
                  Id_Str    => LexTokenManager.Null_String);
            else
               Arg         := Args'Val (ArgCount);
               Found (Arg) := True;
               if Arg = Entity then
                  CheckRepresentSameName (Exp_Node   => ExpNode,
                                          Entity_Sym => Entity_Sym);
               end if;
            end if;
         end if;
      end CheckArgument;

      ----------------------

   begin --wf_pragma_import
      ArgAssNode := Next_Sibling (Child_Node (Pragma_Node));
      --should be top of argument_asociation_rep chain
      if Syntax_Node_Type (Node => ArgAssNode) = SPSymbols.argument_association_rep then
         while Syntax_Node_Type (Node => ArgAssNode) /= SPSymbols.argument_association loop
            ArgAssNode := Child_Node (ArgAssNode);
         end loop;
         --now pointing at leftmost argument association
         while ArgAssNode /= STree.NullNode loop
            if ArgCount = MaxArgs then
               Error_Found := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 600,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ArgAssNode),
                  Id_Str    => LexTokenManager.Null_String);
               exit;
            end if;
            ArgCount := ArgCount + 1;
            CheckArgument (Node       => ArgAssNode,
                           Entity_Sym => Entity_Sym);
            ArgAssNode := Next_Sibling (Parent_Node (Current_Node => ArgAssNode));
         end loop;
         if ArgCount < 2 then
            Error_Found := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 600,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Pragma_Node),
               Id_Str    => LexTokenManager.Convention_Token);

         else
            if not Found (Convention) then
               Error_Found := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 603,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Pragma_Node),
                  Id_Str    => LexTokenManager.Convention_Token);
            end if;
            if not Found (Entity) then
               Error_Found := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 603,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Pragma_Node),
                  Id_Str    => LexTokenManager.Entity_Token);
            end if;
         end if;
      else --there are no arguments
         Error_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 600,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Pragma_Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end wf_pragma_import;

   ----------------------

begin -- wf_external_interface
   Error_Found := False;
   case CommandLineData.Content.Language_Profile is
      when CommandLineData.SPARK83 =>

         wf_pragma_interface (Pragma_Node => Pragma_Node,
                              Entity_Sym  => Entity_Sym);

      when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

         wf_pragma_import (Pragma_Node => Pragma_Node,
                           Entity_Sym  => Entity_Sym);

   end case;
end wf_external_interface;
