-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure up_wf_subprogram_body (Node  : in     STree.SyntaxNode;
                                 Scope : in out Dictionary.Scopes) is
   IdentNode, EndDesigNode : STree.SyntaxNode;
   IdentStr                : LexTokenManager.Lex_String;
   SubProgSym              : Dictionary.Symbol;
   Abstraction             : Dictionary.Abstractions;

   ---------------------------------------------------------

   procedure CheckGlobalImportsAreInitialized (Sym  : in Dictionary.Symbol;
                                               Node : in STree.SyntaxNode)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym;
   is
      It     : Dictionary.Iterator;
      VarSym : Dictionary.Symbol;

   begin
      It := Dictionary.FirstGlobalVariable (Abstraction, Sym);
      while not Dictionary.IsNullIterator (It) loop
         VarSym := Dictionary.CurrentSymbol (It);
         if Dictionary.IsImport (Abstraction, Sym, VarSym)
           and then not Dictionary.OwnVariableIsInitialized (VarSym)
           and then Dictionary.GetOwnVariableOrConstituentMode (VarSym) = Dictionary.DefaultMode then
            ErrorHandler.Semantic_Error
              (Err_Num   => 167,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Dictionary.GetSimpleName (VarSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckGlobalImportsAreInitialized;

   ---------------------------------------------------------

   procedure MainProgramCeilingPriorityCheck (Sym  : in Dictionary.Symbol;
                                              Node : in STree.SyntaxNode)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from Abstraction,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Sym;
   is
      It                     : Dictionary.Iterator;
      MainProgramPriorityLex : LexTokenManager.Lex_String;
   begin
      if Dictionary.MainProgramPrioritySupplied then
         MainProgramPriorityLex := Dictionary.GetMainProgramPriority;
         if LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => MainProgramPriorityLex,
            Lex_Str2 => LexTokenManager.Null_String) /=
           LexTokenManager.Str_Eq then
            -- We have a valid value for the priority. This will have been range-checked if
            -- Priority has been supplied in the Config file. We can do the ceiling check
            -- irrespective of whether the range check was performed or not, as long as the
            -- priority values are known not to be out of any supplied range.
            -- The Lex value was created using StorageRep in CheckPriorityPragma, so we can
            -- convert it back to a Value using ValueRep.
            It := Dictionary.FirstGlobalVariable (Abstraction, Sym);
            CheckCeilingPriority
              (Sym              => Sym,
               Scope            => Scope,
               CheckList        => It,
               PriorityLexValue => MainProgramPriorityLex,
               ErrorNode        => Node);
         else
            -- An out of range Priority value was supplied for Main. This will have already
            -- been reported as a semantic error, so we don't need any further errors or
            -- warnings here, but of course we can't do the ceiling check.
            null;
         end if;
      elsif Dictionary.BodyIsHidden (Sym) then
         -- Pragma priority may be there but is unavailable.
         ErrorHandler.Semantic_Warning
           (Err_Num  => 311,
            Position => Node_Position (Node => Node),
            Id_Str   => Dictionary.GetSimpleName (Sym));
      else
         -- "A pragma Priority is required for the main program"
         ErrorHandler.Semantic_Error
           (Err_Num   => 933,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end MainProgramCeilingPriorityCheck;  -- Expect Unused is not used

   ---------------------------------------------------------

   procedure CheckDelayPropertyAccountedFor (ProcOrTask : in Dictionary.Symbol;
                                             Node       : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         ProcOrTask,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
   begin
      if Dictionary.HasDelayProperty (ProcOrTask)
        and then (not Dictionary.DelayPropertyIsAccountedFor (ProcOrTask))
        and then (not Dictionary.BodyIsHidden (ProcOrTask)) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 915,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Node),
            Id_Str    => Dictionary.GetSimpleName (ProcOrTask));
      end if;
   end CheckDelayPropertyAccountedFor;

   ---------------------------------------------------------

begin --up_wf_subprogram_body

   --  subprogram_body :
   --      overriding_indicator procedure_specification procedure_annotation RWis
   --        subprogram_implementation
   --    | overriding_indicator function_specification function_annotation RWis
   --        subprogram_implementation
   --    | procedure_specification procedure_annotation RWis
   --        subprogram_implementation
   --    | function_specification function_annotation RWis
   --        subprogram_implementation ;

   SubProgSym := Dictionary.GetRegion (Scope);
   --determine which annotation to use
   Abstraction := Dictionary.GetAbstraction (SubProgSym, Scope);

   -- If the overriding_indicator is present then the IdentNode is the
   -- Last_Child_Of the next sibling of the
   -- overriding_indicator node (Child_Node (Node)).

   if Syntax_Node_Type (Node => Child_Node (Node)) = SPSymbols.overriding_indicator then
      IdentNode := Last_Child_Of (Start_Node => Next_Sibling (Child_Node (Node)));
   else
      IdentNode := Last_Child_Of (Start_Node => Node);
   end if;

   IdentStr     := Node_Lex_String (Node => IdentNode);
   EndDesigNode := Last_Sibling_Of (Start_Node => Child_Node (Last_Sibling_Of (Start_Node => Child_Node (Node))));

   if Syntax_Node_Type (Node => EndDesigNode) = SPSymbols.designator then
      if LexTokenManager.Lex_String_Case_Insensitive_Compare
        (Lex_Str1 => IdentStr,
         Lex_Str2 => Node_Lex_String (Node => Child_Node (EndDesigNode))) /=
        LexTokenManager.Str_Eq then
         ErrorHandler.Semantic_Error
           (Err_Num   => 58,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => EndDesigNode),
            Id_Str    => IdentStr);
      end if;
   end if;

   if Dictionary.IsMainProgram (SubProgSym) then
      --check that global imports are initialized has been done in
      --wf_dependency_clause for procedure main programs but a check is
      --needed here for the (very unlikely) case of a function main prog
      if Dictionary.IsFunction (SubProgSym) then
         CheckGlobalImportsAreInitialized (SubProgSym, EndDesigNode);
      end if;

      if CommandLineData.Ravenscar_Selected then
         -- For Ravenscar, perform the ceiling priority check for the main program PO calls.
         MainProgramCeilingPriorityCheck (SubProgSym, Node);
      end if;
   end if;

   CheckEmbedBodies (SubProgSym, EndDesigNode);

   CheckDelayPropertyAccountedFor (SubProgSym, Node);
   CheckSuspendsListAccountedFor (SubProgSym, Node);

   Scope := Dictionary.GetEnclosingScope (Scope);
end up_wf_subprogram_body;
