///###////////////////////////////////////////////////////////////////////////
//
// Burton Computer Corporation
// http://www.burton-computer.com
// http://www.cooldevtools.com
// $Id: SimpleMultiLineStringCharReader.cc 56 2004-10-31 23:10:45Z brian $
//
// Copyright (C) 2000 Burton Computer Corporation
// ALL RIGHTS RESERVED
//
// This program is open source software; you can redistribute it
// and/or modify it under the terms of the Q Public License (QPL)
// version 1.0. Use of this software in whole or in part, including
// linking it (modified or unmodified) into other programs is
// subject to the terms of the QPL.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// Q Public License for more details.
//
// You should have received a copy of the Q Public License
// along with this program; see the file LICENSE.txt.  If not, visit
// the Burton Computer Corporation or CoolDevTools web site
// QPL pages at:
//
//    http://www.burton-computer.com/qpl.html
//    http://www.cooldevtools.com/qpl.html
//

#include "AbstractMultiLineString.h"
#include "SimpleMultiLineStringCharReader.h"

class SimpleMultiLineStringCharReaderPosition : public AbstractCharReaderPosition
{
public:
    SimpleMultiLineStringCharReaderPosition()
    {
        reset();
    }

    ~SimpleMultiLineStringCharReaderPosition()
    {
    }

    void copyPosition(const SimpleMultiLineStringCharReaderPosition *other)
    {
        assert(other);
        *this = *other;
    }

    void reset()
    {
        lineIndex = 0;
        charIndex = -1;
        inTerminator = false;
        atEnd = false;
        currentChar = ' ';
    }

    int lineIndex;
    int charIndex;
    bool inTerminator;
    bool atEnd;
    char currentChar;
};

SimpleMultiLineStringCharReader::SimpleMultiLineStringCharReader(const AbstractMultiLineString *target)
: m_target(target), m_position(new SimpleMultiLineStringCharReaderPosition())
{
    reset();
}

SimpleMultiLineStringCharReader::SimpleMultiLineStringCharReader(const AbstractMultiLineString *target,
                                                                 const string &terminator)
: m_target(target), m_terminator(terminator), m_position(new SimpleMultiLineStringCharReaderPosition())
{
    reset();
}

SimpleMultiLineStringCharReader::~SimpleMultiLineStringCharReader()
{
}

bool SimpleMultiLineStringCharReader::forward()
{
    return skip(1);
}

bool SimpleMultiLineStringCharReader::skip(int nchars)
{
    m_position->charIndex += nchars;
    return ensureCharReady();
}

bool SimpleMultiLineStringCharReader::hasChar()
{
    return m_position->lineIndex >= 0 && !m_position->atEnd;
}

bool SimpleMultiLineStringCharReader::atEnd()
{
    return m_position->atEnd;
}

void SimpleMultiLineStringCharReader::reset()
{
    m_position->reset();
}

AbstractCharReaderPosition *SimpleMultiLineStringCharReader::createMark()
{
    return new SimpleMultiLineStringCharReaderPosition(*m_position.get());
}

void SimpleMultiLineStringCharReader::returnToMark(AbstractCharReaderPosition *position)
{
    m_position->copyPosition(dynamic_cast<SimpleMultiLineStringCharReaderPosition *>(position));
    setCurrentChar(m_position->currentChar);
}

const string *SimpleMultiLineStringCharReader::getCurrentLine()
{
    return m_position->inTerminator ? &m_terminator : m_target->line(m_position->lineIndex);
}

bool SimpleMultiLineStringCharReader::ensureCharReady()
{
    if (m_position->atEnd) {
        return false;
    }

    while (m_position->inTerminator || m_position->lineIndex < m_target->lineCount()) {
        const string *line = getCurrentLine();
        if (m_position->charIndex < line->size()) {
            char ch = (*line)[m_position->charIndex];
            m_position->currentChar = ch;
            setCurrentChar(ch);
            return true;
        }

        m_position->charIndex -= line->size();
        if (m_position->inTerminator || m_terminator.size() == 0) {
            m_position->inTerminator = false;
            ++m_position->lineIndex;
        } else {
            m_position->inTerminator = true;
        }
    }

    m_position->atEnd = true;
    return false;
}
