#include <QTextStream>
#include <QFile>

#include "TextEditHighlighted.h"
#include "LanguageComboBox.h"
#include "StyleComboBox.h"

namespace srchiliteqt {

TextEditHighlighted::TextEditHighlighted() :
        highlighter(0),
        languageComboBox(0), styleComboBox(0),
        styleFile("default.style")
{
    // by default we use C++ as source language
        setHighlighter("cpp.lang");
}

void TextEditHighlighted::setHighlighter(const QString &langFile) {
    // otherwise there'll be more highlighters for the same doc!
    if (highlighter)
        delete highlighter;

    // set Qt4SyntaxHighlighter for highlighting context
      highlighter =
              new srchiliteqt::Qt4SyntaxHighlighter(document());
      highlighter->init(langFile, styleFile);

      changeColors(highlighter->getForegroundColor(), highlighter->getBackgroundColor());
}

void TextEditHighlighted::changeColors(const QString &fgColor, const QString &bgColor) {
    if (fgColor == "" && bgColor == "")
        return;

    QPalette p = palette();

    if (fgColor != "")
        p.setColor(QPalette::Text, fgColor);
    else
        p.setColor(QPalette::Text, "black");

    if (bgColor != "")
        p.setColor(QPalette::Base, bgColor);
    else
        p.setColor(QPalette::Base, "white");

    setPalette(p);
}

void TextEditHighlighted::connectLanguageComboBox(LanguageComboBox *lcb) {
    languageComboBox = lcb;
    connect(languageComboBox, SIGNAL(currentIndexChanged(QString)),
              this, SLOT(changeHighlightingLanguage(QString)));
    connect(this, SIGNAL(changedHighlightingLanguage(QString)),
            languageComboBox, SLOT(setCurrentLanguage(QString)));
}

void TextEditHighlighted::connectStyleComboBox(StyleComboBox *lcb) {
    styleComboBox = lcb;
    connect(styleComboBox, SIGNAL(currentIndexChanged(QString)),
              this, SLOT(changeHighlightingStyle(QString)));
    connect(this, SIGNAL(changedHighlightingStyle(QString)),
            styleComboBox, SLOT(setCurrentStyle(QString)));
}

void TextEditHighlighted::changeHighlightingLanguage(const QString &newLang) {
    // avoid to switch language if it's just the same
    if (newLang == highlighter->getLangFile())
        return;

    setHighlighter(newLang);

    emit changedHighlightingLanguage(newLang);
}

void TextEditHighlighted::changeHighlightingStyle(const QString &newStyle) {
    // avoid to switch language if it's just the same
    if (newStyle == highlighter->getFormattingStyle())
        return;

    // this will also rehighlight the contents
    highlighter->setFormattingStyle(newStyle);
    changeColors(highlighter->getForegroundColor(), highlighter->getBackgroundColor());

    styleFile = newStyle;

    emit changedHighlightingStyle(newStyle);
}

const QString TextEditHighlighted::loadFile(const QString &fileName)
{
      QFile file(fileName);
      if (!file.open(QFile::ReadOnly | QFile::Text)) {
            return file.errorString();
      }

      // use the file name to detect the source language for highlighting
      const QString langDefFile =
              highlighter->getLangDefFileFromFileName(fileName);

      // make sure possible previous content is now removed
      clear();

      // if the highlight could be detected then set it now
      if (langDefFile != "") {
          changeHighlightingLanguage(langDefFile);
      }

      QTextStream in(&file);
      setPlainText(in.readAll());

      return ""; // no error
}

void TextEditHighlighted::changeFileName(const QString &fileName)
{
      if (fileName.isEmpty())
          return;

      // use the file name to detect the source language for highlighting
      const QString langDefFile =
              highlighter->getLangDefFileFromFileName(fileName);

      // if the highlight could be detected then set it now
      if (langDefFile != "") {
          changeHighlightingLanguage(langDefFile);
      }
}

}
