

#ifndef CONFIG_H
#define CONFIG_H

#include "formatinfoloader.h"
#include "conversionoptions.h"

#include <qobject.h>
#include <qvaluelist.h>
#include <qstringlist.h>
#include <qmap.h>

class Logger;
class ConvertPlugin;
//class ConvertPluginLoader;
class ReplayGainPlugin;
//class ReplayGainPluginLoader;
class RipperPlugin;
struct ProfileData;


/**
 * @short Format items for the format list
 * @author Daniel Faust <hessijames@gmail.com>
 * @version 0.3
 */
class FormatItem
{
public:
    /**
     * Constructor
     */
    FormatItem();

    /**
     * Destructor
     */
    virtual ~FormatItem();

    QStringList mime_types;
    QStringList extensions; // for easy use
    QStringList correction_file_mime_types;
    QStringList correction_file_extensions; // for easy use
    QString description;
    FormatInfo::CompressionType compressionType;
    int compressionLevel; // the value from the config dialog
    bool internalReplayGain;
    int size;
    QValueList<ConvertPlugin*> encoders;
    QValueList<ConvertPlugin*> decoders;
    QValueList<ReplayGainPlugin*> replaygains;
    //QValueList<RepairerPlugin*> repairers;
    ConvertPlugin* encoder;
    ConvertPlugin* decoder;
    ReplayGainPlugin* replaygain;
    //RepairerPlugin* repairer;
};


/**
 * @short Configuration class
 * @author Daniel Faust <hessijames@gmail.com>
 * @version 0.3
 */
class Config : public QObject
{
    Q_OBJECT
public:
    /**
     * Constructor
     */
    Config( Logger* );

    /**
     * Destructor
     */
    virtual ~Config();

    /**
     * Read the preferences from the configuration file
     */
    void read();

    /**
     * Write the preferences to the configuration file
     */
    void write( bool sync = true );

    /**
     * Read the profiles from the profiles file
     */
    void readProfiles();

    /**
     * Write the profiles to the profiles file
     */
    void writeProfiles();

    /**
     * Get the encoder for a given file format (mime type or extension)
     */
    ConvertPlugin* encoderForFormat( const QString &format );

    /**
     * Get the decoder for a given file format (mime type or extension)
     */
    ConvertPlugin* decoderForFormat( const QString &format );

    /**
     * Get the decoder for a given file format (mime type or extension)
     */
    ReplayGainPlugin* replaygainForFormat( const QString &format );

    /**
     * Get the format information for a given file format (mime type or extension)
     */
    FormatItem* getFormatItem( const QString &format );

    /**
     * Get the current ripper
     */
    RipperPlugin* getCurrentRipper() { return currentRipper; }

    /**
     * Set the current ripper
     */
    void setCurrentRipper( RipperPlugin* ripper ) { currentRipper = ripper; }

    /**
     * Returns a list of all loaded rippers
     */
    QValueList<RipperPlugin*> allRippers() { return rippers; }

    /**
     * Returns a list of all loaded converters
     */
    QValueList<ConvertPlugin*> allConverters() { return converters; }

    /**
     * Returns a list of all loaded replaygains
     */
    QValueList<ReplayGainPlugin*> allReplayGains() { return replaygains; }

    /**
     * Returns a list of all known file formats
     */
    QStringList allFormats();

    /**
     * Returns a list of all known encodeable file formats
     */
    QStringList allEncodableFormats();

    /**
     * Returns a list of all known lossy encodeable file formats
     */
    QStringList allLossyEncodableFormats();

    /**
     * Returns a list of all known lossless encodeable file formats
     */
    QStringList allLosslessEncodableFormats();

    /**
     * Returns a list of all known hybrid encodeable file formats
     */
    QStringList allHybridEncodableFormats();

    /**
     * Returns the extension of the correction file format for the given format
     * If there is nor correction file format, the returned string is empty
     */
    QString getCorrectionExtension( const QString &format );

    /**
     * Returns a localized description for the given format
     */
    QString getFormatDescription( const QString &format );

    /**
     * Add a new profile
     */
    void addProfile( const QString &name, const ConversionOptions& profile );

    /**
     * Remove a new profile
     */
    void removeProfile( const QString &name );

    /**
     * Returns the conversion options for a profile
     */
    ConversionOptions getProfile( const QString &name );

    /**
     * Returns the name of the profile with conversion options @p options
     */
     QString getProfileName( const ConversionOptions& options );

    /**
     * Returns a list of all user defined profiles
     */
    QStringList getAllProfiles();

    /**
     * Returns true if the @p file can be added to the conversion list (can be decoded)
     */
    bool acceptFile( const QString& format );

    /**
     * Returns true if the @p file can be added to the replay gain tool
     */
    bool acceptReplayGainFile( const QString& format );

    /**
     * Returns a file filter suitable for the file open dialog
     */
    QString fileFilter( bool wav = true );

    /**
     * Returns a string list of supported file formats
     */
    QStringList fileTypes( bool wav = true );

    /**
     * Returns a file filter suitable for the file open dialog for the replay gain scanner
     */
    QString replayGainFilter();

    /**
     * Returns a string list of Replay Gain supported file formats
     */
    QStringList replayGainFileTypes();

    struct Data {
        struct General {
            int startTab;
            int lastTab;
            QString defaultProfile;
            QString defaultFormat;
//             QString defaultOutputDirectory;
            QString specifyOutputDirectory;
            QString metaDataOutputDirectory;
            QString copyStructureOutputDirectory;
            int priority;
            bool useVFATNames;
            int numFiles;
            int updateDelay;
            bool askForNewOptions;
            bool showToolBar;
        } general;
        struct Plugins {
            bool checkForUpdates;
        } plugins;
        struct Environment {
            QStringList directories;
            QStringList foundPrograms;
        } environment;
        struct App {
            int configVersion;
        } app;
    } data;

    bool onlinePluginsChanged;
    bool backendsChanged;

    QMap<QString, QString> binaries;

private:
    struct PluginMapData {
        int version;
        QString filename;
        QString type;
    };

    Logger* logger;

    /** holds all known formats */
    QValueList<FormatItem> formats;

    /** holds all known rippers */
    QValueList<RipperPlugin*> rippers;
    RipperPlugin* currentRipper;

    /** holds all known converters */
    QValueList<ConvertPlugin*> converters;
    /** holds all known replaygain apps */
    QValueList<ReplayGainPlugin*> replaygains;
    /* holds all known file repairing apps */
//     QValueList<RepairerItem> repairer;

    //ConvertPluginLoader* convertPluginLoader;
    //ReplayGainPluginLoader* replaygainPluginLoader;

    /** holds all user defined profiles */
    QValueList<ProfileData> profiles;

    /**
     * Load the files with the file infos (format descriptions, file size, etc.)
     */
    void loadFileInfos();

    /**
     * Load all plugins that can be found
     */
    void loadPlugins();

    /**
     * A plugin wants to register it's features for a file format
     * @p binary The binary name of the app that wants to register it's features
     * @p extension The file name extension for that is wants to register
     * @p features The features of the app for that extension
     */
    void registerFormatFeatures( const QString &mime_type,
                                 ConvertPlugin* encoder = 0, ConvertPlugin* decoder = 0,
                                 ReplayGainPlugin* replaygain = 0/*, RepairPlugin* repairer = 0*/,
                                 const QString &correction_file_mime_type = QString::null );

    /**
     * Generate the service menu
     */
    void writeServiceMenu();

    /**
     * Copy the amarok script
     */
    void writeAmarokScript();

    /*
     * We have new inforamtion for a file format
     * @p extension The file name extension for that is wants to register
     * @p newSynonym Register a new synonym for this file format?
     * @p newMimeType Register a new mime type for this file format?
     * @p newDescription Register a new description for this file format?
     * @p newSize Register a new size for this file format for calculating the progress?
     */
    /*void registerFileFormat( const QString &extension, const QString &newSynonym = QString::null,
                             const QString &newMimeType = QString::null,
                             const QString &newDescription = QString::null,
                             int newSize = 0, FormatInfo::CompressionType newCompressionType = FormatInfo::lossy );*/
    //void registerFileFormat( const QString &mime_type, const QString &option, const QString &value );
    // NOTE this function is obsolete.
    //       after loading all plugins and creating all neccessary file format items,
    //       sk will search for format info files and update the items.

signals:
    void configChanged();

};

#endif // CONFIG_H
