/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include <sds.h>
#include <string.h>

#ifdef ENABLE_GLIB
#include <glib.h>
#endif

static char *
__get_tmp_dir (void)
{
#ifdef ENABLE_GLIB
  return (char *) g_get_tmp_dir ();
#else
  return "/tmp/";
#endif
}

static int
__sds_open_unix (char *server)
{
  char *s = NULL;
  int fd;
  struct sockaddr_un saddr;


  if (server)
    s = strdup (server);
  else
    {
      int len = 0;
      DIR *d;
      struct dirent *dir;

      if (!(d = opendir (__get_tmp_dir ())))
	return -1;

      while ((dir = readdir (d)))
	{

	  if (!strncmp (dir->d_name, "sds-", 4)
	      && !strcmp (dir->d_name + strlen (dir->d_name) - 5, ".sock"))
	    {
	      len = strlen (__get_tmp_dir ()) + strlen (dir->d_name) + 2;

	      if (!(s = (char *) malloc (sizeof (char) * len)))
		return -1;

	      snprintf (s, len, "%s/%s", __get_tmp_dir (), dir->d_name);

	      break;
	    }
	}

      if (!s)
	return -1;
    }

  if ((fd = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
    return -1;

  saddr.sun_family = AF_UNIX;
  strncpy (saddr.sun_path, s, 108);

  if (connect (fd, (struct sockaddr *) &saddr, sizeof (saddr)) == -1)
    {
      close (fd);
      return -1;
    }

  return fd;
}

static int
__sds_open_tcp (char *server, int port)
{
  int fd;
  struct sockaddr_in sock;
  struct hostent *hp;

  if (!server)
    server = "localhost";

  if (!port)
    port = SDS_PORT;

  if ((fd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
    return -1;

  if (!(hp = gethostbyname (server)))
    return -1;

  memset ((void *) &sock, 0, sizeof (sock));
  sock.sin_family = AF_INET;
  sock.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;

  sock.sin_port = htons (port);

  if (connect (fd, (struct sockaddr *) &sock, sizeof (sock)) < 0)
    {
      close (fd);
      return -1;
    }

  return fd;
}

/* Open function */
int
sds_open (char *server)
{

  if (!strcmp (server, "unix"))
    return __sds_open_unix (NULL);

  if (!strcmp (server, "tcp"))
    return __sds_open_tcp (NULL, 0);

  if (!strncmp (server, "unix://", 7))
    {
      char *file;
      file = server + 7;

      return __sds_open_unix (file);
    }

  if (!strncmp (server, "tcp://", 6))
    {
      char *a;
      int i, port = 0;

      a = server + 6;

      for (i = 0; i < strlen (a); i++)
	{
	  if (a[i] == ':')
	    {

	      if (i + 1 < strlen (a))
		{
		  port = atoi (&a[i + 1]);
		}

	      a[i] = 0;
	    }
	}

      return __sds_open_tcp (a, port);
    }

  return -1;
}

/* Set your program like SdS Monitor */
int
sds_monitor (int fd)
{
  if (write (fd, SDS_MONITOR_CONNECT
#ifdef WORDS_BIGENDIAN
	     "1"
#else
	     "0"
#endif
	     "\n", 10) < 0)
    return -1;

  return 0;
}

/* Set your program like SdS Input */
int
sds_set (int fd, int rate, int channels, int bitrate, int volume, int balance,
	 int solo, int mute, char *controller)
{
  char buf[50];
  int len;

  len =
    snprintf (buf, 50, "%d %d %d %d %d %d %d %d %s", rate, channels, bitrate,
	      volume, balance, solo, mute,
#ifdef WORDS_BIGENDIAN
	      1
#else
	      0
#endif
	      , controller ? controller : "");

  if (write (fd, buf, len) < 0)
    return -1;

  return 0;
}

/* Read, write and close */
int
sds_read (int fd, void *data, size_t s)
{
  return read (fd, data, s);
}

int
sds_write (int fd, void *data, size_t s)
{
  return write (fd, data, s);
}

int
sds_close (int fd)
{
  return close (fd);
}

/* Set your program like SdS Monitor */
int
sds_test (int fd)
{
  if (write (fd, SDS_TEST_CONNECT, strlen (SDS_TEST_CONNECT)))
    return -1;

  return 0;
}

/* Open function */
sds_controller *
sds_controller_open (char *server)
{
  sds_controller *c;
  int id;

  if (!strcmp (server, "unix"))
    id = __sds_open_unix (NULL);

  if (!strcmp (server, "tcp"))
    id = __sds_open_tcp (NULL, 0);

  if (!strncmp (server, "unix://", 7))
    {
      char *file;
      file = server + 7;

      id = __sds_open_unix (file);
    }

  if (!strncmp (server, "tcp://", 6))
    {
      char *a;
      int i, port = 0;

      a = server + 6;

      for (i = 0; i < strlen (a); i++)
	{
	  if (a[i] == ':')
	    {

	      if (i + 1 < strlen (a))
		{
		  port = atoi (&a[i + 1]);
		}

	      a[i] = 0;
	    }
	}

      id = __sds_open_tcp (a, port);
    }

  else
    return NULL;

  if (sds_test (id))
    return NULL;

  if (!(c = (sds_controller *) malloc (sizeof (sds_controller))))
    return NULL;
  c->server = strdup (server);

  return c;
}

int
sds_controller_test (sds_controller * c)
{
  int fd;

  if ((fd = sds_open (c->server)) < 1)
    return -1;

  return sds_test (fd);
}

static sds_client *
__sds_controller_parse_line (char *str)
{
  sds_client *c;

  if (!(c = (sds_client *) malloc (sizeof (sds_client))))
    return NULL;
  memset (c, 0, sizeof (sds_client));
  sscanf (str, "%s %d %d %d %d %d %d %d %d %d %s\n",
	  c->addr, &c->mute, &c->solo, &c->volume_up, &c->volume_down,
	  &c->balance_up, &c->balance_down, &c->volume, &c->balance,
	  &c->autofader, c->socket);

  return c;
}

sds_client *
sds_controller_get_list (sds_controller * c)
{
  sds_client *client, *head, *tmp;
  int fd;
  int i = 0;
  char buf[1024];

  if ((fd = sds_open (c->server)) < 1)
    return NULL;

  if (write (fd, SDS_DATA_LIST_CONNECT, strlen (SDS_DATA_LIST_CONNECT)))
    return NULL;

  client = head = NULL;

  while (read (fd, buf + i, 1) == 1)
    {
      if (buf[i] == '\n')
	{
	  buf[i] = 0;
	  i = 0;

	  if ((tmp = __sds_controller_parse_line (buf)))
	    {

	      if (!head)
		head = tmp;
	      else
		client->next = tmp;

	      client = tmp;
	    }
	}
    }

  return head;
}

/* EOF */
