/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif
#ifdef ENABLE_MIC

#include "../../player.h"
#include "../../output.h"
#include "../../other.h"
#include "../../ftime.h"
#include "../../audio.h"
#include "mic.h"

#ifdef ENABLE_L_SOUNDCARD_H
#include <linux/soundcard.h>
#elif ENABLE_S_SOUNDCARD_H
#include <sys/soundcard.h>
#elif ENABLE_SOUNDCARD_H
#include <soundcard.h>
#endif

void
mic_run (void)
{
  format_data_mic *mic = (format_data_mic *) play->format_current->data;
  int fd, size, t_size;
  unsigned char *buf;
  int format;

  if (!mic)
    {
      play->microphone = 0;
      return;
    }

  play->play = 0;

  if (!mic->rate)
    mic->rate = 44100;

  if (!mic->channels)
    mic->channels = 2;

  if (!mic->bitrate)
    mic->bitrate = 16;

  switch(mic->bitrate) {
    case 8:
      format = AFMT_U8;
      break;

    case 16:
      format = AFMT_S16_NE;
      break;

    default:
      msg_error(_("DSP bitrate must be 8 or 16!"));
      play->microphone=0;
      return;
  }

  mic->start = timer_start ();

  if (!play->devdsp)
    {
#ifdef DEVDSP
      if (!(play->devdsp = strdup (DEVDSP)))
	fatal (_("Error: memory."));
#else
      fatal (_("Compile with devdsp support!"));
#endif
    }

  if ((fd = open (play->devdsp, O_RDONLY)) < 0)
    {
      msg_error (_("Device DSP open error."));
      play->microphone = 0;
      return;
    }


  if (ioctl (fd, SNDCTL_DSP_SETFMT, &format) < 0)
    {
      msg_error (_("Set bitrate in DSP error."));
      play->microphone = 0;
      return;
    }

  switch(format) {
    case AFMT_U8: mic->bitrate=8; break;
    case AFMT_S16_NE: mic->bitrate=16; break;
  }

  if (ioctl (fd, SNDCTL_DSP_CHANNELS, &mic->channels) < 0)
    {
      msg_error (_("Set channels in DSP error."));
      play->microphone = 0;
      return;
    }

  if (ioctl (fd, SNDCTL_DSP_SPEED, &mic->rate) < 0)
    {
      msg_error (_("Set rate in DSP error."));
      play->microphone = 0;
      return;
    }

  for (size = 20; size > 0; size--)
    {
      t_size = 0x7fff0000 | size;
      if (ioctl (fd, SNDCTL_DSP_SETFRAGMENT, &t_size) == 0)
	break;
    }

  if (size <= 0)
    msg_error (_("Fragment size on DSP dropped to zero... mmm..."));

  size = 256;
  if (ioctl (fd, SNDCTL_DSP_GETBLKSIZE, &size) != 0)
    {
      msg_error (_("Getblksize from DSP error."));
      play->microphone = 0;
      return;
    }

  if (!(buf = (unsigned char *) malloc (sizeof (unsigned char) * size)))
    fatal (_("Error: memory."));

  msg (_("Info Mic Audio:\n" "\tMic %d Hz - Channels %d - Bitrate %d"),
       mic->rate, mic->channels, mic->bitrate);

  msg (_("Play file audio: %s\n"), playing_name());

  output_open (mic->rate, mic->channels, mic->bitrate);
  output_info ();

  while (1)
    {

      play->play = 1;

      if (events.skip || events.quit)
	break;

      if (read (fd, buf, size) != size)
	{
	  msg_error (_("Read from device DSP error."));
	  play->microphone = 0;
	  break;
	}

      output_write (buf, size);
    }

  close (fd);

  output_close ();
  play->play = 0;

  timer_clear (mic->start);

}

char *
mic_get_time (void)
{
  static char output[13];
  double t;
  long min;
  double sec;
  format_data_mic *mic = (format_data_mic *) play->format_current->data;

  if (!mic)
    return NULL;

  if (!mic->start)
    mic->start = timer_start ();
  t = timer_time (mic->start);

  min = (long) t / (long) 60;
  sec = t - 60.0f * min;

  snprintf (output, 13, "T: %02li:%05.2f", min, sec);

  return output;
}

int
mic_check_option (char *opt)
{
  if (!strncmp (opt, "rate=", 5))
    return 1;
  if (!strncmp (opt, "bitrate=", 8))
    return 1;
  if (!strncmp (opt, "channels=", 9))
    return 1;

  return 0;
}

int
mic_option (char *opt, format_data_mic * fmt)
{
  int len;
  int ch = 0;
  struct stat st;

  fmt->rate = 44100;
  fmt->bitrate = 16;
  fmt->channels = 1;

  if (!lstat (opt, &st))
    {
      if (mic_read_config (opt, fmt))
	fatal (_("Mic file config error."));
    }
  else
    {


      len = strlen (opt);

      while (ch < len)
	{
	  switch (*(opt + ch))
	    {
	    case 'b':
	      if (!strncmp (opt + ch, "bitrate=", 8))
		{
		  ch += 8;

		  fmt->bitrate = audio_parse_int (opt, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in mic parameter: %s"), opt);
		  return 1;
		}
	      break;

	    case 'c':
	      if (!strncmp (opt + ch, "channels=", 9))
		{
		  ch += 9;

		  fmt->channels = audio_parse_int (opt, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in mic parameter: %s"), opt);
		  return 1;
		}
	      break;

	    case 'r':
	      if (!strncmp (opt + ch, "rate=", 5))
		{
		  ch += 5;

		  fmt->rate = audio_parse_int (opt, &ch, len);
		}
	      else
		{
		  msg_error (_("Error in mic parameter: %s"), opt);
		  return 1;
		}
	      break;

	    default:
	      msg_error (_("Error in mic parameter: %s"), opt);
	      return 1;
	    }

	  ch++;
	}
    }

  return 0;
}
#endif
