/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "format.h"
#include "file.h"
#include "other.h"
#include "sock.h"

#ifdef ENABLE_OGG
#include "input/ogg/ogg.h"
#endif
#ifdef ENABLE_MP3
#include "input/mp3/mp3.h"
#endif
#ifdef ENABLE_SNDFILE
#include "input/sndfile/fsndfile.h"
#endif
#ifdef ENABLE_DAEMON
#include "input/daemon/daemon.h"
#endif
#ifdef ENABLE_MIC
#include "input/mic/mic.h"
#endif
#ifdef ENABLE_CDAUDIO
#include "input/cdaudio/cdaudio.h"
#endif

/* Now the formats of file (and the functions for decode this  files)
 * are in a array of FORMATS_NUMBER. If I want change somaplayer in
 * a modular program, this array must be a list */


/* This functions returns the format of a type of file */
format *
format_search (int type)
{
  register int i;

  if (!type)
    return NULL;

  for (i = 0; i < FORMATS_NUMBER; i++)
    {
      if (play->formats[i])
	{
	  if (play->formats[i]->type == type)
	    return play->formats[i];
	}
    }

  return NULL;
}

/* Inizialize the formats array */
void
format_init (void)
{
  register int i;

  for (i = 0; i < FORMATS_NUMBER; i++)
    play->formats[i] = NULL;
}

/* Add a format.
 * When somaplayer check the file, with type_file, I have the
 * type of file. With this function I add the functions for this
 * type of file. */
void
format_add (int type)
{

  if (!type)
    return;

  if (type > FORMATS_NUMBER)
    fatal (_("Internal error."));

  if (play->formats[type - 1] != NULL)
    return;

  fprintf (stderr, _("\n* Init format audio "));

  switch (type)
    {
#ifdef ENABLE_MP3
    case TYPE_MP3:
      play->formats[type - 1] = format_mp3 ();
      break;
#endif

#ifdef ENABLE_OGG
    case TYPE_OGG:
      play->formats[type - 1] = format_ogg ();
      break;
#endif

#ifdef ENABLE_SNDFILE
    case TYPE_SNDFILE:
      play->formats[type - 1] = format_sndfile ();
      break;
#endif

#ifdef ENABLE_CDAUDIO
    case TYPE_CDAUDIO:
      play->formats[type - 1] = format_cdaudio ();
      break;
#endif

#ifdef ENABLE_MIC
    case TYPE_MIC:
      play->formats[type - 1] = format_mic ();
      break;
#endif

#ifdef ENABLE_DAEMON
    case TYPE_DAEMON_INPUT:
      play->formats[type - 1] = format_daemon_input ();
      break;
#endif

    }
}

/* Remove the formats */
void
format_free (void)
{
  register int i;

  for (i = 0; i < FORMATS_NUMBER; i++)
    {
      if (play->formats[i])
	{
	  if (play->formats[i]->data)
	    free (play->formats[i]->data);
	  free (play->formats[i]);
	}
    }
}

/* This functions are the single elements of formats */

#ifdef ENABLE_OGG
format *
format_ogg (void)
{
  format *fmt;
  format_data_ogg *data;

  fprintf (stderr, "Ogg/Vorbis\n");

  if (!(fmt = (format *) malloc (sizeof (format))))
    fatal (_("Error: memory."));
  if (!(data = (format_data_ogg *) malloc (sizeof (format_data_ogg))))
    fatal (_("Error: memory."));

  fmt->type = TYPE_OGG;
  fmt->run = ogg_run;
  fmt->get_time = ogg_get_time;
  fmt->get_name = NULL;
  fmt->get_buffer = NULL;
  fmt->data = (void *) data;

  return fmt;
}
#endif

#ifdef ENABLE_MP3
format *
format_mp3 (void)
{
  format *fmt;
  format_data_mp3 *data;

  fprintf (stderr, "Mp3\n");

  if (!(fmt = (format *) malloc (sizeof (format))))
    fatal (_("Error: memory."));
  if (!(data = (format_data_mp3 *) malloc (sizeof (format_data_mp3))))
    fatal (_("Error: memory."));

  fmt->type = TYPE_MP3;
  fmt->run = mp3_run;
  fmt->get_time = mp3_get_time;
  fmt->get_name = mp3_get_name;
  fmt->get_buffer = NULL;
  fmt->data = (void *) data;

  return fmt;
}
#endif

#ifdef ENABLE_SNDFILE
format *
format_sndfile (void)
{
  format *fmt;
  format_data_sndfile *data;

  fprintf (stderr, "SndFile\n");

  if (!(fmt = (format *) malloc (sizeof (format))))
    fatal (_("Error: memory."));
  if (!(data = (format_data_sndfile *) malloc (sizeof (format_data_sndfile))))
    fatal (_("Error: memory."));

  fmt->type = TYPE_SNDFILE;
  fmt->run = sndfile_run;
  fmt->get_time = sndfile_get_time;
  fmt->get_name = NULL;
  fmt->get_buffer = NULL;
  fmt->data = (void *) data;

  return fmt;
}
#endif

#ifdef ENABLE_DAEMON
format *
format_daemon_input (void)
{
  format *fmt;
  format_data_daemon_input *data;

  fprintf (stderr, "Daemon Client\n");

  if (!(fmt = (format *) malloc (sizeof (format))))
    fatal (_("Error: memory."));
  if (!
      (data =
       (format_data_daemon_input *)
       malloc (sizeof (format_data_daemon_input))))
    fatal (_("Error: memory."));

  if (!(data->sock = sock_parse (play->daemon_input_opt)))
    fatal (_("Error: parameter!"));

  fmt->type = TYPE_DAEMON_INPUT;
  fmt->run = daemon_input_run;
  fmt->get_time = daemon_input_get_time;
  fmt->get_name = NULL;
  fmt->get_buffer = NULL;
  fmt->data = (void *) data;

  return fmt;
}
#endif

#ifdef ENABLE_MIC
format *
format_mic (void)
{
  format *fmt;
  format_data_mic *data;

  fprintf (stderr, "Mic\n");

  if (!(fmt = (format *) malloc (sizeof (format))))
    fatal (_("Error: memory."));
  if (!(data = (format_data_mic *) malloc (sizeof (format_data_mic))))
    fatal (_("Error: memory."));

  if (play->mic_option)
    {
      if (mic_option (play->mic_option, data))
	fatal (_("Mic config error."));
    }

  else
    {
      data->rate = 44100;
      data->channels = 2;
      data->bitrate = 16;
    }

  fmt->type = TYPE_MIC;
  fmt->run = mic_run;
  fmt->get_time = mic_get_time;
  fmt->get_name = NULL;
  fmt->get_buffer = NULL;
  fmt->data = (void *) data;

  return fmt;
}
#endif

#ifdef ENABLE_CDAUDIO
format *
format_cdaudio (void)
{
  format *fmt;
  format_data_cdaudio *data;

  fprintf (stderr, "Cdaudio\n");

  if (!(fmt = (format *) malloc (sizeof (format))))
    fatal (_("Error: memory."));
  if (!(data = (format_data_cdaudio *) malloc (sizeof (format_data_cdaudio))))
    fatal (_("Error: memory."));

  fmt->type = TYPE_CDAUDIO;
  fmt->run = cdaudio_run;
  fmt->get_time = cdaudio_get_time;
  fmt->get_name = NULL;
  fmt->get_buffer = cdaudio_get_buffer;
  fmt->data = (void *) data;

  return fmt;
}
#endif

/* EOF */
