/*
 * -----------------------------------------------------------------------
 * Emulation of MFC5282 Chip select module 
 *
 * (C) 2008 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * state:  Not implemented
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * -----------------------------------------------------------------------
 */

#include <bus.h>
#include <sgstring.h>

#define CSM_CSAR(base,x)	((base) + 0x0 + (x) * 12)
#define CSM_CSMR(base,x)	((base) + 0x4 + (x) * 12)
#define		CSMR_V		(1<<0)
#define 	CSMR_UD		(1<<1)
#define		CSMR_UC		(1<<2)
#define		CSMR_SD		(1<<3)
#define		CSMR_SC		(1<<4)
#define		CSMR_CI		(1<<5)
#define		CSMR_AM		(1<<6)
#define		CSMR_WP		(1<<8)
#define		CSMR_BAM_MASK	(0xffff<<16)
#define		CSMR_BAM_SHIFT	(16)

#define CSM_CSCR(base,x)	((base) + 0x8 + (x) * 12)
#define		CSCR_BSTW	(1<<3)
#define		CSCR_BSTR	(1<<4)
#define		CSCR_BEM	(1<<5)
#define		CSCR_PS0	(1<<6)
#define		CSCR_PS1	(1<<7)
#define		CSCR_AA		(1<<8)
#define		CSCR_WS_MASK	(0xf << 10)
#define		CSCR_WS_SHIFT	(10)

typedef struct Csm Csm; 

typedef struct ChipSelect {
	Csm *csm;	
	BusDevice *dev;
	uint16_t csar;
	uint32_t csmr;
	uint32_t cscr;
} ChipSelect;

struct Csm {
	BusDevice bdev;
	ChipSelect cs[7];
};

static void
UpdateMappings(Csm *csm)
{
	if(!csm->cs[0].csmr & CSMR_V) {
		// Map it to everythere except ipsbar
	}
}
static uint32_t
csar_read(void *clientData,uint32_t address,int rqlen)
{
	ChipSelect *cs = (ChipSelect *) clientData;
        return cs->csar;
}

static void
csar_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	ChipSelect *cs = (ChipSelect *) clientData;
	cs->csar = value;
        fprintf(stderr,"CSM: csar not implemented\n");
	// update_mappings
}

static uint32_t
csmr_read(void *clientData,uint32_t address,int rqlen)
{
	ChipSelect *cs = (ChipSelect *) clientData;
        return cs->csmr;
}

static void
csmr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	ChipSelect *cs = (ChipSelect *) clientData;
	cs->csmr = value;
        fprintf(stderr,"CSM: csmr not implemented\n");
	// update_mappings
}

static uint32_t
cscr_read(void *clientData,uint32_t address,int rqlen)
{
	ChipSelect *cs = (ChipSelect *) clientData;
	if(rqlen == 4) {
		return cs->cscr << 16;
	} else if((rqlen == 2) && ((address & 2) == 2)) {
		return cs->cscr;
	}
        return 0;
}

static void
cscr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	ChipSelect *cs = (ChipSelect *) clientData;
	if(rqlen == 4) {
		cs->cscr  = value >> 16;
	} else if((rqlen == 2) && ((address & 2) == 2)) {
		cs->cscr = value;
	}
}

static void
Csm_Unmap(void *owner,uint32_t base,uint32_t mask)
{
	int i;
	for(i=0;i<7;i++) {
        	IOH_Delete32(CSM_CSAR(base,i));
        	IOH_Delete32(CSM_CSMR(base,i));
        	IOH_Delete32(CSM_CSCR(base,i));
	}
}

static void
Csm_Map(void *owner,uint32_t base,uint32_t mask,uint32_t mapflags)
{
        Csm *csm = (Csm *) owner;
	int i;
	for(i=0;i<7;i++) {
        	IOH_New32(CSM_CSAR(base,i),csar_read,csar_write,&csm->cs[i]);
        	IOH_New32(CSM_CSMR(base,i),csmr_read,csmr_write,&csm->cs[i]);
        	IOH_New32(CSM_CSCR(base,i),cscr_read,cscr_write,&csm->cs[i]);
	}
}

static void
Csm_RegisterDevice(Csm *csm,BusDevice *dev,unsigned int cs_nr) 
{
	ChipSelect *cs = &csm->cs[cs_nr];
	if(cs_nr >= 7) {
		fprintf(stderr,"Illegal Chip select %d\n",cs_nr);	
		exit(1);
	}
	cs = &csm->cs[cs_nr];
	cs->dev = dev;
	UpdateMappings(csm);
}

BusDevice *
MCF5282_CsmNew(const char *name)
{
        Csm *csm = sg_calloc(sizeof(Csm));
        csm->bdev.first_mapping=NULL;
        csm->bdev.Map=Csm_Map;
        csm->bdev.UnMap=Csm_Unmap;
        csm->bdev.owner=csm;
        csm->bdev.hw_flags=MEM_FLAG_WRITABLE|MEM_FLAG_READABLE;
        return &csm->bdev;
}
