/*
 * ------------------------------------------------------------------------------
 *
 * Emulation of MAX6651 Fan Controller 
 *
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 * State: basically working, but FAN-Speeds are constants. GPIO 
 *        emulation is not complete
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * ------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include "i2c.h"
#include "max6651.h"

#if 0
#define dprintf(x...) { fprintf(stderr,x); }
#else
#define dprintf(x...)
#endif

#define MAX_STATE_ADDR (1)
#define MAX_STATE_DATA (2)

struct MAX6651 {
	I2C_Slave i2c_slave;
	int state;
	int reg_addr;

	uint8_t speed;
	uint8_t config;
	uint8_t gpio_def;
	uint8_t dac;
	uint8_t alarm_enable;
	uint8_t alarm;
	uint32_t tach[4];
	uint8_t gpio_stat;
	uint8_t count;
};

/*
 * ------------------------------------
 * MAX6651 Write state machine 
 * ------------------------------------
 */
static int 
max6651_write(void *dev,uint8_t data) {
	MAX6651 *max = dev;
	if(max->state==MAX_STATE_ADDR) {
		dprintf("MAX6651 Addr 0x%02x\n",data);
		max->reg_addr= data;
		max->state = MAX_STATE_DATA;
	} else if(max->state==MAX_STATE_DATA) {
		dprintf("MAX6651 Write 0x%02x to %04x\n",data,max->reg_addr);
		switch(max->reg_addr) {
			case 0:
				max->speed=data;
				break;
			case 2:
				max->config = data;
				break;
			case 4:
				max->gpio_def = data;
				break;
			case 6:
				max->dac = data;
				break;
			case 8:
				max->alarm_enable=data;
				break;
			case 0x16:
				max->count = data;
				break;
			default:
				fprintf(stderr,"MAX6651 write to nonexisting register\n");
				return I2C_NACK; /* does the real device the same thing ? */
		} 
	}
	return I2C_ACK;
};

static void
update_fanspeed(MAX6651 *max,unsigned int fan) 
{
	struct timeval tv;
	if(fan>3) {
		return;
	}
	gettimeofday(&tv,NULL);
	max->tach[fan] = 100 + (tv.tv_usec & 0x7); 	
	
}
static int 
max6651_read(void *dev,uint8_t *data) 
{
	MAX6651 *max = dev;
	switch(max->reg_addr) {
			case 0x0:
				*data=max->speed;
				break;
			case 0x2:
				*data=max->config;
				break;
			case 0x4:
				*data=max->gpio_def;
				break;
			case 0x6:
				*data=max->dac;
				break;

			case 0x8:
				*data=max->alarm_enable;
				break;

			case 0xa:
				*data=max->alarm;
				break;

			case 0xc:
				update_fanspeed(max,0);
				*data=(max->tach[0]<<1)>>(3-max->count);
				break;

			case 0xe:
				update_fanspeed(max,1);
				*data=(max->tach[1]<<1)>>(3-max->count);
				break;

			case 0x10:
				update_fanspeed(max,2);
				*data=(max->tach[2]<<1)>>(3-max->count);
				break;

			case 0x12:
				update_fanspeed(max,3);
				*data=(max->tach[3]<<1)>>(3-max->count);
				break;

			case 0x14:
				*data=max->gpio_stat;
				break;
			case 0x16:
				*data=max->count;
				break;
			default:
				*data=0;
	}
	dprintf("MAX6651 read 0x%02x from %04x\n",*data,max->reg_addr);
	return I2C_DONE;
};

static int
max6651_start(void *dev,int i2c_addr,int operation) {
	MAX6651 *max = dev;
	dprintf("MAX6651 start\n");
	max->state = MAX_STATE_ADDR;
	return I2C_ACK;
}

static void 
max6651_stop(void *dev) {
	MAX6651 *max = dev;
	dprintf("MAX6651 stop\n");
	max->state =  MAX_STATE_ADDR; 
}


static I2C_SlaveOps max6651_ops = {
	.start = max6651_start,
	.stop =  max6651_stop,
	.read =  max6651_read,	
	.write = max6651_write	
};

I2C_Slave *
MAX6651_New(char *name) {
	MAX6651 *max = malloc(sizeof(MAX6651)); 
	I2C_Slave *i2c_slave;
	if(!max) {
		fprintf(stderr,"Out of memory\n");
		exit(8152);
	}
	memset(max,0,sizeof(MAX6651));
	max->config=0xa;
	max->gpio_def=0xff;
	max->gpio_stat=0x1f;
	max->tach[0]=0;
	max->tach[1]=0;
	max->tach[2]=0;
	max->tach[3]=0;
	max->count=0x02;
	i2c_slave = &max->i2c_slave;
	i2c_slave->devops = &max6651_ops; 
	i2c_slave->dev = max;
	i2c_slave->speed = I2C_SPEED_FAST;
	fprintf(stderr,"MAX6652 FAN-Controller \"%s\" created\n",name);
	return i2c_slave;
}
