/* s7, a Scheme interpreter
 *
 *   derived from TinyScheme 1.39, but not a single byte of that code remains
 *   SPDX-License-Identifier: 0BSD
 *
 * Bill Schottstaedt, bil@ccrma.stanford.edu
 *
 * Mike Scholz provided the FreeBSD support (complex trig funcs, etc)
 * Rick Taube, Andrew Burnson, Donny Ward, Greg Santucci, and Christos Vagias provided the MS Visual C++ support
 * Kjetil Matheussen provided the mingw support
 *
 * Documentation is in s7.h and s7.html.
 * s7test.scm is a regression test.
 * repl.scm is a vt100-based listener.
 * nrepl.scm is a notcurses-based listener.
 * cload.scm and lib*.scm tie in various C libraries.
 * lint.scm checks Scheme code for infelicities.
 * r7rs.scm implements some of r7rs (small).
 * write.scm currrently has pretty-print.
 * mockery.scm has the mock-data definitions.
 * reactive.scm has reactive-set and friends.
 * stuff.scm has some stuff.
 * profile.scm has code to display profile data.
 * debug.scm has debugging aids.
 * case.scm has case*, an extension of case to pattern matching.
 * timing tests are in the s7 tools directory
 *
 * s7.c is organized as follows:
 *    structs and type flags
 *    internal debugging stuff
 *    constants
 *    GC
 *    stacks
 *    symbols and keywords
 *    lets
 *    continuations
 *    numbers
 *    characters
 *    strings
 *    ports
 *    format
 *    lists
 *    vectors
 *    hash-tables
 *    c-objects
 *    functions
 *    equal?
 *    generic length, copy, reverse, fill!, append
 *    error handlers
 *    sundry leftovers
 *    the optimizers
 *    multiple-values, quasiquote
 *    eval
 *    *s7*
 *    initialization and free
 *    repl
 *
 * naming conventions: s7_* usually are C accessible (s7.h), g_* are scheme accessible,
 *   H_* are documentation strings, Q_* are procedure signatures,
 *   *_1 are ancillary functions, big_* refer to gmp,
 *   scheme "?" corresponds to C "is_", scheme "->" to C "_to_".
 *
 * ---------------- compile time switches ----------------
 */

#if defined __has_include
#  if __has_include ("mus-config.h")
#    include "mus-config.h"
#  endif
#else
  #include "mus-config.h"
#endif

/*
 * Your config file goes here, or just replace that #include line with the defines you need.
 * The compile-time switches involve booleans, complex numbers, and multiprecision arithmetic.
 * Currently we assume we have setjmp.h (used by the error handlers).
 *
 * Complex number support, which is problematic in C++, Solaris, and netBSD
 *   is on the HAVE_COMPLEX_NUMBERS switch. In OSX or Linux, if you're not using C++,
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 1
 *
 *   In C++ I use:
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 0
 *
 *   In Windows and tcc, both are 0.
 *
 *   Some systems (FreeBSD) have complex.h, but some random subset of the trig funcs, so
 *   HAVE_COMPLEX_NUMBERS means we can find
 *      cimag creal cabs csqrt carg conj
 *   and HAVE_COMPLEX_TRIG means we have
 *      cacos cacosh casin casinh catan catanh ccos ccosh cexp clog cpow csin csinh ctan ctanh
 *
 * When HAVE_COMPLEX_NUMBERS is 0, the complex functions are stubs that simply return their
 *   argument -- this will be very confusing for the s7 user because, for example, (sqrt -2)
 *   will return something bogus (it might not signal an error).
 *
 * so the incoming (non-s7-specific) compile-time switches are
 *     HAVE_COMPLEX_NUMBERS, HAVE_COMPLEX_TRIG, SIZEOF_VOID_P
 * if SIZEOF_VOID_P is not defined, we look for __SIZEOF_POINTER__ instead,
 *   the default is to assume that we're running on a 64-bit machine.
 *
 * To get multiprecision arithmetic, set WITH_GMP to 1.
 *   You'll also need libgmp, libmpfr, and libmpc (version 0.8.0 or later)
 *
 * and we use these predefined macros: __cplusplus, _MSC_VER, __GNUC__, __clang__, __ANDROID__
 *
 * if WITH_SYSTEM_EXTRAS is 1 (default is 1 unless _MSC_VER), various OS and file related functions are included.
 * if you want this file to compile into a stand-alone interpreter, define WITH_MAIN
 *   to use nrepl, also define WITH_NOTCURSES
 *
 * -O3 produces segfaults, and is often slower than -O2 (at least according to callgrind)
 * -march=native seems to improve tree-vectorization which is important in Snd
 * -ffast-math makes a mess of NaNs, and does not appear to be faster
 * this code doesn't compile anymore in gcc 4.3 -- c11 might be needed
 */

#if (defined(__GNUC__) || defined(__clang__)) /* s7 uses PRId64 so (for example) g++ 4.4 is too old */
  #define WITH_GCC 1
#else
  #define WITH_GCC 0
#endif


/* ---------------- initial sizes ---------------- */

#ifndef INITIAL_HEAP_SIZE
  /* #define INITIAL_HEAP_SIZE 128000 */
  #define INITIAL_HEAP_SIZE 64000         /* 29-Jul-21 -- seems faster */
#endif
/* the heap grows as needed, this is its initial size. If the initial heap is small, s7 can run in about 2.5 Mbytes of memory.
 * There are many cases where a bigger heap is faster (but harware cache size probably matters more).
 * The heap size must be a multiple of 32.  Each object takes 48 bytes.
 */

#ifndef SYMBOL_TABLE_SIZE
  #define SYMBOL_TABLE_SIZE 32749
#endif
/* names are hashed into the symbol table (a vector) and collisions are chained as lists */
/* 16381: thash +80 [string_to_symbol_p_p] +40 if 24001, tlet +80 [symbol_p_p], +32 24001 */

#ifndef INITIAL_STACK_SIZE
  #define INITIAL_STACK_SIZE 4096  /* was 2048 17-Mar-21 */
#endif
/* the stack grows as needed, each frame takes 4 entries, this is its initial size. (*s7* 'stack-top) divides size by 4 */

#define STACK_RESIZE_TRIGGER (INITIAL_STACK_SIZE / 2)

#ifndef INITIAL_PROTECTED_OBJECTS_SIZE
  #define INITIAL_PROTECTED_OBJECTS_SIZE 16
#endif
/* a vector of objects that are (semi-permanently) protected from the GC, grows as needed */

#ifndef GC_TEMPS_SIZE
  #define GC_TEMPS_SIZE 256
#endif
/* the number of recent objects that are temporarily gc-protected; 8 works for s7test and snd-test.
 *    For the FFI, this sets the lag between a call on s7_cons and the first moment when its result
 *    might be vulnerable to the GC.
 */


/* ---------------- scheme choices ---------------- */

#ifndef WITH_GMP
  #define WITH_GMP 0
  /* this includes multiprecision arithmetic for all numeric types and functions, using gmp, mpfr, and mpc
   * WITH_GMP adds the following functions: bignum and bignum?, and (*s7* 'bignum-precision)
   */
#endif

#ifndef DEFAULT_BIGNUM_PRECISION
  #define DEFAULT_BIGNUM_PRECISION 128 /* (*s7* 'bignum-precision) initial value, must be >= 2 */
#endif

#ifndef WITH_PURE_S7
  #define WITH_PURE_S7 0
#endif
#if WITH_PURE_S7
  #define WITH_EXTRA_EXPONENT_MARKERS 0
  #define WITH_IMMUTABLE_UNQUOTE 1
  /* also omitted: *-ci* functions, char-ready?, cond-expand, multiple-values-bind|set!, call-with-values
   *   and a lot more (inexact/exact, integer-length,  etc) -- see s7.html.
   */
#endif

#ifndef WITH_EXTRA_EXPONENT_MARKERS
  #define WITH_EXTRA_EXPONENT_MARKERS 0
#endif
/* if 1, s7 recognizes "d", "f", "l", and "s" as exponent markers, in addition to "e" (also "D", "F", "L", "S") */

#ifndef WITH_SYSTEM_EXTRAS
  #define WITH_SYSTEM_EXTRAS (!_MSC_VER)
  /* this adds several functions that access file info, directories, times, etc */
#endif

#ifndef WITH_IMMUTABLE_UNQUOTE
  #define WITH_IMMUTABLE_UNQUOTE 0
  /* this removes the name "unquote" */
#endif

#ifndef WITH_C_LOADER
  #if WITH_GCC && (!__MINGW32__) && (!__CYGWIN__)
    #define WITH_C_LOADER 1
  /* (load file.so [e]) looks for (e 'init_func) and if found, calls it as the shared object init function.
   * If WITH_SYSTEM_EXTRAS is 0, the caller needs to supply system and delete-file so that cload.scm works.
   */
  #else
    #define WITH_C_LOADER 0
    /* I think dlopen et al are available in MS C, but I have no way to test them; see load_shared_object below */
  #endif
#endif

#ifndef WITH_HISTORY
  #define WITH_HISTORY 0
  /* this includes a circular buffer of previous evaluations for debugging, ((owlet) 'error-history) and (*s7* 'history-size) */
#endif

#ifndef DEFAULT_HISTORY_SIZE
  #define DEFAULT_HISTORY_SIZE 8
  /* this is the default length of the eval history buffer */
#endif
#if WITH_HISTORY
  #define MAX_HISTORY_SIZE 1048576
#endif

#ifndef DEFAULT_PRINT_LENGTH
  #define DEFAULT_PRINT_LENGTH 12 /* (*s7* 'print-length) initial value, was 32 but Snd uses 12, 23-Jul-21 */
#endif

/* in case mus-config.h forgets these */
#ifdef _MSC_VER
  #ifndef HAVE_COMPLEX_NUMBERS
    #define HAVE_COMPLEX_NUMBERS 0
  #endif
  #ifndef HAVE_COMPLEX_TRIG
    #define HAVE_COMPLEX_TRIG 0
  #endif
#else
  #ifndef HAVE_COMPLEX_NUMBERS
    #if __TINYC__
      #define HAVE_COMPLEX_NUMBERS 0
    #else
      #define HAVE_COMPLEX_NUMBERS 1
    #endif
  #endif
  #if __cplusplus || __TINYC__
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 0
    #endif
  #else
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 1
    #endif
  #endif
#endif

#ifndef WITH_MULTITHREAD_CHECKS
  #define WITH_MULTITHREAD_CHECKS 0
  /* debugging aid if using s7 in a multithreaded program -- this code courtesy of Kjetil Matheussen */
#endif

#ifndef WITH_WARNINGS
  #define WITH_WARNINGS 0
  /* int+int overflows to real, etc: this adds warnings which are expensive even though they are never called (procedure overhead) */
#endif

#ifndef S7_DEBUGGING
  #define S7_DEBUGGING 0
#endif

#undef DEBUGGING
#define DEBUGGING typo!
#define HAVE_GMP typo!

#define SHOW_EVAL_OPS 0

#ifndef _GNU_SOURCE
  #define _GNU_SOURCE  /* for qsort_r, grumble... */
#endif

#ifndef _MSC_VER
  #include <unistd.h>
  #include <sys/param.h>
  #include <strings.h>
  #include <errno.h>
  #include <locale.h>
#else
  /* in Snd these are in mus-config.h */
  #ifndef MUS_CONFIG_H_LOADED
    #if _MSC_VER < 1900
      #define snprintf _snprintf
    #endif
    #if _MSC_VER > 1200
      #define _CRT_SECURE_NO_DEPRECATE 1
      #define _CRT_NONSTDC_NO_DEPRECATE 1
      #define _CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES 1
    #endif
  #endif
  #include <io.h>
  #pragma warning(disable: 4244) /* conversion might cause loss of data warning */
#endif

#if WITH_GCC && (!S7_DEBUGGING)
  #define Inline inline __attribute__((__always_inline__))
#else
  #ifdef _MSC_VER
    #define Inline __forceinline
  #else
    #define Inline inline
  #endif
#endif

#ifndef WITH_VECTORIZE
  #define WITH_VECTORIZE 1
#endif

#if (WITH_VECTORIZE) && (defined(__GNUC__) && __GNUC__ >= 5)
  #define Vectorized __attribute__((optimize("tree-vectorize")))
#else
  #define Vectorized
#endif

#if WITH_GCC
  #define Sentinel __attribute__((sentinel))
#else
  #define Sentinel
#endif

#ifdef _MSC_VER
  #define noreturn _Noreturn
#else
  #define noreturn __attribute__((noreturn))
  /* this is ok in gcc/g++/clang and tcc; pure attribute is rarely applicable here, and does not seem to be helpful (maybe safe_strlen) */
#endif

#ifndef S7_ALIGNED
  #define S7_ALIGNED 0
  /* memclr, local_strcmp and local_memset */
#endif

#include <stdio.h>
#include <limits.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <time.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdint.h>
#include <inttypes.h>
#include <setjmp.h>

#ifdef _MSC_VER
  #define MS_WINDOWS 1
#else
  #define MS_WINDOWS 0
#endif

#if defined(_MSC_VER) || defined(__MINGW32__)
  #define Jmp_Buf       jmp_buf
  #define SetJmp(A, B)  setjmp(A)
  #define LongJmp(A, B) longjmp(A, B)
#else
  #define Jmp_Buf       sigjmp_buf
  #define SetJmp(A, B)  sigsetjmp(A, B)
  #define LongJmp(A, B) siglongjmp(A, B)
  /* we need sigsetjmp, not setjmp for nrepl's interrupt (something to do with signal masks??)
   *   unfortunately sigsetjmp is noticeably slower than setjmp, especially when s7_optimize_1 is called a lot.
   *   In one case, the sigsetjmp version runs in 24 seconds, but the setjmp version takes 10 seconds, and
   *   yet callgrind says there is almost no difference? I removed setjmp from s7_optimize.
   */
#endif

#if (!MS_WINDOWS)
  #include <pthread.h>
#endif

#if __cplusplus
  #include <cmath>
#else
  #include <math.h>
#endif

/* there is also apparently __STDC_NO_COMPLEX__ */
#if HAVE_COMPLEX_NUMBERS
  #if __cplusplus
    #include <complex>
  #else
    #include <complex.h>
    #ifndef __SUNPRO_C
      #if defined(__sun) && defined(__SVR4)
        #undef _Complex_I
        #define _Complex_I 1.0fi
      #endif
    #endif
  #endif

  #ifndef CMPLX
    #if (!(defined(__cplusplus))) && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 7)) && !defined(__INTEL_COMPILER)
      #define CMPLX(x, y) __builtin_complex ((double) (x), (double) (y))
    #else
      #define CMPLX(r, i) ((r) + ((i) * _Complex_I))
    #endif
  #endif
#endif

#include "s7.h"

#ifndef M_PI
  #define M_PI 3.1415926535897932384626433832795029L
#endif

#ifndef INFINITY
  #ifndef HUGE_VAL
    #define INFINITY (1.0/0.0) /* -log(0.0) is triggering dumb complaints from cppcheck */
    /* there is sometimes a function, infinity(), MSC apparently uses HUGE_VALF, gcc has __builtin_huge_val() */
  #else
    #define INFINITY HUGE_VAL
  #endif
#endif

#ifndef NAN
  #define NAN (INFINITY / INFINITY)
#endif

#define BOLD_TEXT "\033[1m"
#define UNBOLD_TEXT "\033[22m"

#if ((!__NetBSD__) && ((_MSC_VER) || (!defined(__STC__)) || (defined(__STDC_VERSION__) && (__STDC_VERSION__ < 199901L))))
  #define __func__ __FUNCTION__
#endif

#define display(Obj)    string_value(s7_object_to_string(sc, Obj, false))
#define display_80(Obj) string_value(object_to_truncated_string(sc, Obj, 80))

#if (((defined(SIZEOF_VOID_P)) && (SIZEOF_VOID_P == 4)) || ((defined(__SIZEOF_POINTER__)) && (__SIZEOF_POINTER__ == 4)) || (!defined(__LP64__)))
  #define NUMBER_NAME_SIZE 2 /* pointless */
  #define POINTER_32 true
#else
  #define NUMBER_NAME_SIZE 22 /* leave 1 for uint8_t name len (byte 0), 1 for terminating nul */
  #define POINTER_32 false
#endif

#define WRITE_REAL_PRECISION 16
typedef long double long_double;

#define ld64 PRId64
#define p64 PRIdPTR

#define MAX_FLOAT_FORMAT_PRECISION 128

/* types */
enum {T_FREE = 0,
      T_PAIR, T_NIL, T_UNUSED, T_UNDEFINED, T_UNSPECIFIED, T_EOF, T_BOOLEAN, T_CHARACTER, T_SYNTAX, T_SYMBOL,
      T_INTEGER, T_RATIO, T_REAL, T_COMPLEX, T_BIG_INTEGER, T_BIG_RATIO, T_BIG_REAL, T_BIG_COMPLEX,
      T_STRING, T_C_OBJECT, T_VECTOR, T_INT_VECTOR, T_FLOAT_VECTOR, T_BYTE_VECTOR,
      T_CATCH, T_DYNAMIC_WIND, T_HASH_TABLE, T_LET, T_ITERATOR,
      T_STACK, T_COUNTER, T_SLOT, T_C_POINTER, T_OUTPUT_PORT, T_INPUT_PORT, T_RANDOM_STATE, T_CONTINUATION, T_GOTO,
      T_CLOSURE, T_CLOSURE_STAR, T_MACRO, T_MACRO_STAR, T_BACRO, T_BACRO_STAR,
      T_C_MACRO, T_C_FUNCTION_STAR, T_C_FUNCTION, T_C_RST_NO_REQ_FUNCTION,
      NUM_TYPES};
/* T_UNUSED, T_STACK, T_SLOT, T_DYNAMIC_WIND, T_CATCH, and T_COUNTER are internal */

static const char *s7_type_names[] =
  {"free", "pair", "nil", "unused", "undefined", "unspecified", "eof_object", "boolean", "character", "syntax", "symbol",
   "integer", "ratio", "real", "complex", "big_integer", "big_ratio", "big_real", "big_complex",
   "string", "c_object", "vector", "int_vector", "float_vector", "byte_vector",
   "catch", "dynamic_wind", "hash_table", "let", "iterator",
   "stack", "counter", "slot", "c_pointer", "output_port", "input_port", "random_state", "continuation", "goto",
   "closure", "closure*", "macro", "macro*", "bacro", "bacro*",
   "c_macro", "c_function*", "c_function", "c_rst_no_req_function",
   };

typedef struct block_t {
  union {
    void *data;
    s7_pointer d_ptr;
    s7_int *i_ptr;
  } dx;
  int32_t index;
  union {
    bool needs_free;
    uint32_t tag;
  } ln;
  s7_int size;
  union {
    struct block_t *next;
    char *documentation;
    s7_pointer ksym;
    s7_int nx_int;
    s7_int *ix_ptr;
    struct {
      uint32_t i1, i2;
    } ix;
  } nx;
  union {
    s7_pointer ex_ptr;
    void *ex_info;
    s7_int ckey;
  } ex;
} block_t;

#define NUM_BLOCK_LISTS 18
#define TOP_BLOCK_LIST 17
#define BLOCK_LIST 0

#define block_data(p)                    p->dx.data
#define block_index(p)                   p->index
#define block_set_index(p, Index)        p->index = Index
#define block_size(p)                    p->size
#define block_set_size(p, Size)          p->size = Size
#define block_next(p)                    p->nx.next
#define block_info(p)                    p->ex.ex_info

typedef block_t hash_entry_t;
#define hash_entry_key(p)                p->dx.d_ptr
#define hash_entry_value(p)              (p)->ex.ex_ptr
#define hash_entry_set_value(p, Val)     p->ex.ex_ptr = Val
#define hash_entry_next(p)               block_next(p)
#define hash_entry_raw_hash(p)           block_size(p)
#define hash_entry_set_raw_hash(p, Hash) block_set_size(p, Hash)

typedef block_t vdims_t;
#define vdims_rank(p)                    p->size
#define vector_elements_should_be_freed(p) p->ln.needs_free
#define vdims_dims(p)                    p->dx.i_ptr
#define vdims_offsets(p)                 p->nx.ix_ptr
#define vdims_original(p)                p->ex.ex_ptr


typedef enum {TOKEN_EOF, TOKEN_LEFT_PAREN, TOKEN_RIGHT_PAREN, TOKEN_DOT, TOKEN_ATOM, TOKEN_QUOTE, TOKEN_DOUBLE_QUOTE, TOKEN_BACK_QUOTE,
	      TOKEN_COMMA, TOKEN_AT_MARK, TOKEN_SHARP_CONST, TOKEN_VECTOR, TOKEN_BYTE_VECTOR, TOKEN_INT_VECTOR, TOKEN_FLOAT_VECTOR} token_t;

typedef enum {NO_ARTICLE, INDEFINITE_ARTICLE} article_t;
typedef enum {DWIND_INIT, DWIND_BODY, DWIND_FINISH} dwind_t;
enum {NO_SAFETY = 0, IMMUTABLE_VECTOR_SAFETY, MORE_SAFETY_WARNINGS};  /* (*s7* 'safety) settings */

typedef enum {FILE_PORT, STRING_PORT, FUNCTION_PORT} port_type_t;

typedef struct {
  int32_t (*read_character)(s7_scheme *sc, s7_pointer port);             /* function to read a character, int32_t for EOF */
  void (*write_character)(s7_scheme *sc, uint8_t c, s7_pointer port);    /* function to write a character */
  void (*write_string)(s7_scheme *sc, const char *str, s7_int len, s7_pointer port); /* function to write a string of known length */
  token_t (*read_semicolon)(s7_scheme *sc, s7_pointer port);             /* internal skip-to-semicolon reader */
  int32_t (*read_white_space)(s7_scheme *sc, s7_pointer port);           /* internal skip white space reader */
  s7_pointer (*read_name)(s7_scheme *sc, s7_pointer pt);                 /* internal get-next-name reader */
  s7_pointer (*read_sharp)(s7_scheme *sc, s7_pointer pt);                /* internal get-next-sharp-constant reader */
  s7_pointer (*read_line)(s7_scheme *sc, s7_pointer pt, bool eol_case);  /* function to read a string up to \n */
  void (*displayer)(s7_scheme *sc, const char *s, s7_pointer pt);
  void (*close_port)(s7_scheme *sc, s7_pointer p);                       /* close-in|output-port */
} port_functions_t;

typedef struct {
  bool needs_free, is_closed;
  port_type_t ptype;
  FILE *file;
  char *filename;
  block_t *filename_block;
  uint32_t line_number, file_number;
  s7_int filename_length;
  block_t *block;
  s7_pointer orig_str;    /* GC protection for string port string */
  const port_functions_t *pf;
  s7_pointer (*input_function)(s7_scheme *sc, s7_read_t read_choice, s7_pointer port);
  void (*output_function)(s7_scheme *sc, uint8_t c, s7_pointer port);
} port_t;

typedef enum {o_d_v, o_d_vd, o_d_vdd, o_d_vid, o_d_id, o_d_7pi, o_d_7pii, o_d_7piid, o_d_7piii, o_d_7piiid,
	      o_d_ip, o_d_pd, o_d_7pid, o_d, o_d_d, o_d_dd, o_d_7dd, o_d_ddd, o_d_dddd,
	      o_i_i, o_i_7i, o_i_ii, o_i_7ii, o_i_iii, o_i_7pi, o_i_7pii, o_i_7piii, o_d_p,
	      o_b_p, o_b_7p, o_b_pp, o_b_7pp, o_b_pp_unchecked, o_b_pi, o_b_ii, o_b_7ii, o_b_dd,
	      o_p, o_p_p, o_p_ii, o_p_d, o_p_dd, o_i_7d, o_i_7p, o_d_7d, o_p_pp, o_p_ppp, o_p_pi, o_p_pi_unchecked,
	      o_p_ppi, o_p_i, o_p_pii, o_p_pip, o_p_pip_unchecked, o_p_piip, o_b_i, o_b_d} opt_func_t;

typedef struct opt_funcs_t {
  opt_func_t typ;
  void *func;
  struct opt_funcs_t *next;
} opt_funcs_t;

typedef struct {
  const char *name;
  int32_t name_length;
  uint32_t id;
  const char *doc;
  opt_funcs_t *opt_data; /* vunion-functions (see below) */
  s7_pointer generic_ff, setter, signature, pars;
  s7_pointer (*chooser)(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops);
  /* arg_defaults|names call_args only T_C_FUNCTION_STAR -- call args for GC protection */
  union {
    s7_pointer *arg_defaults;
    s7_pointer bool_setter;
  } dam;
  union {
    s7_pointer *arg_names;
    s7_pointer c_sym;
  } sam;
  union {
    s7_pointer call_args;
    void (*marker)(s7_pointer p, s7_int len);
  } cam;
} c_proc_t;


typedef struct {
  s7_int type, outer_type;
  s7_pointer scheme_name, getter, setter;
  void (*mark)(void *val);
  void (*free)(void *value);           /* this will go away someday (use gc_free) */
  bool (*eql)(void *val1, void *val2); /* this will go away someday (use equal) */
#if (!DISABLE_DEPRECATED)
  char *(*print)(s7_scheme *sc, void *value);
#endif
  s7_pointer (*equal)      (s7_scheme *sc, s7_pointer args);
  s7_pointer (*equivalent) (s7_scheme *sc, s7_pointer args);
  s7_pointer (*ref)        (s7_scheme *sc, s7_pointer args);
  s7_pointer (*set)        (s7_scheme *sc, s7_pointer args);
  s7_pointer (*length)     (s7_scheme *sc, s7_pointer args);
  s7_pointer (*reverse)    (s7_scheme *sc, s7_pointer args);
  s7_pointer (*copy)       (s7_scheme *sc, s7_pointer args);
  s7_pointer (*fill)       (s7_scheme *sc, s7_pointer args);
  s7_pointer (*to_list)    (s7_scheme *sc, s7_pointer args);
  s7_pointer (*to_string)  (s7_scheme *sc, s7_pointer args);
  s7_pointer (*gc_mark)    (s7_scheme *sc, s7_pointer args);
  s7_pointer (*gc_free)    (s7_scheme *sc, s7_pointer args);
} c_object_t;


typedef s7_int (*hash_map_t)(s7_scheme *sc, s7_pointer table, s7_pointer key);          /* hash-table object->location mapper */
typedef hash_entry_t *(*hash_check_t)(s7_scheme *sc, s7_pointer table, s7_pointer key); /* hash-table object equality function */
static hash_map_t default_hash_map[NUM_TYPES];

typedef s7_int (*s7_i_7pi_t)(s7_scheme *sc, s7_pointer p, s7_int i1);
typedef s7_int (*s7_i_7pii_t)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2);
typedef s7_int (*s7_i_7piii_t)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2, s7_int i3);
typedef s7_int (*s7_i_iii_t)(s7_int i1, s7_int i2, s7_int i3);
typedef s7_int (*s7_i_7i_t)(s7_scheme *sc, s7_int i1);
typedef s7_int (*s7_i_7ii_t)(s7_scheme *sc, s7_int i1, s7_int i2);
typedef bool (*s7_b_pp_t)(s7_pointer p1, s7_pointer p2);
typedef bool (*s7_b_7pp_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
typedef bool (*s7_b_7p_t)(s7_scheme *sc, s7_pointer p1);
typedef bool (*s7_b_pi_t)(s7_scheme *sc, s7_pointer p1, s7_int i2);
typedef bool (*s7_b_d_t)(s7_double p1);
typedef bool (*s7_b_i_t)(s7_int p1);
typedef bool (*s7_b_ii_t)(s7_int p1, s7_int p2);
typedef bool (*s7_b_7ii_t)(s7_scheme *sc, s7_int p1, s7_int p2);
typedef bool (*s7_b_dd_t)(s7_double p1, s7_double p2);
typedef s7_pointer (*s7_p_t)(s7_scheme *sc);
typedef s7_pointer (*s7_p_ppi_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int i1);
typedef s7_pointer (*s7_p_pi_t)(s7_scheme *sc, s7_pointer p1, s7_int i1);
typedef s7_pointer (*s7_p_pii_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2);
typedef s7_pointer (*s7_p_pip_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2);
typedef s7_pointer (*s7_p_piip_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_pointer p3);
typedef s7_pointer (*s7_p_i_t)(s7_scheme *sc, s7_int i);
typedef s7_pointer (*s7_p_ii_t)(s7_scheme *sc, s7_int i1, s7_int i2);
typedef s7_pointer (*s7_p_dd_t)(s7_scheme *sc, s7_double x1, s7_double x2);
typedef s7_double (*s7_d_7d_t)(s7_scheme *sc, s7_double p1);
typedef s7_double (*s7_d_7dd_t)(s7_scheme *sc, s7_double p1, s7_double p2);
typedef s7_double (*s7_d_7pii_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2);
typedef s7_double (*s7_d_7piid_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_double x1);
typedef s7_double (*s7_d_7piii_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_int i3);
typedef s7_double (*s7_d_7piiid_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_int i3, s7_double x1);

typedef struct opt_info opt_info;

typedef union {
  s7_int i;
  s7_double x;
  s7_pointer p;
  void *obj;
  opt_info *o1;
  s7_function call;
  s7_double (*d_f)(void);
  s7_double (*d_d_f)(s7_double x);
  s7_double (*d_7d_f)(s7_scheme *sc, s7_double x);
  s7_double (*d_dd_f)(s7_double x1, s7_double x2);
  s7_double (*d_7dd_f)(s7_scheme *sc, s7_double x1, s7_double x2);
  s7_double (*d_ddd_f)(s7_double x1, s7_double x2, s7_double x3);
  s7_double (*d_dddd_f)(s7_double x1, s7_double x2, s7_double x3, s7_double x4);
  s7_double (*d_v_f)(void *obj);
  s7_double (*d_vd_f)(void *obj, s7_double fm);
  s7_double (*d_vdd_f)(void *obj, s7_double x1, s7_double x2);
  s7_double (*d_vid_f)(void *obj, s7_int i, s7_double fm);
  s7_double (*d_id_f)(s7_int i, s7_double fm);
  s7_double (*d_7pi_f)(s7_scheme *sc, s7_pointer obj, s7_int i1);
  s7_double (*d_7pid_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_double x);
  s7_double (*d_7pii_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_int i2);
  s7_double (*d_7piid_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_int i2, s7_double x);
  s7_double (*d_7piii_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_int i2, s7_int i3);
  s7_double (*d_7piiid_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_int i2, s7_int i3, s7_double x);
  s7_double (*d_ip_f)(s7_int i1, s7_pointer p);
  s7_double (*d_pd_f)(s7_pointer obj, s7_double x);
  s7_double (*d_p_f)(s7_pointer p);
  s7_int (*i_7d_f)(s7_scheme *sc, s7_double i1);
  s7_int (*i_7p_f)(s7_scheme *sc, s7_pointer i1);
  s7_int (*i_i_f)(s7_int i1);
  s7_int (*i_7i_f)(s7_scheme *sc, s7_int i1);
  s7_int (*i_ii_f)(s7_int i1, s7_int i2);
  s7_int (*i_7ii_f)(s7_scheme *sc, s7_int i1, s7_int i2);
  s7_int (*i_iii_f)(s7_int i1, s7_int i2, s7_int i3);
  s7_int (*i_7pi_f)(s7_scheme *sc, s7_pointer p, s7_int i1);
  s7_int (*i_7pii_f)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2);
  s7_int (*i_7piii_f)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2, s7_int i3);
  bool (*b_i_f)(s7_int p);
  bool (*b_d_f)(s7_double p);
  bool (*b_p_f)(s7_pointer p);
  bool (*b_pp_f)(s7_pointer p1, s7_pointer p2);
  bool (*b_7pp_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
  bool (*b_7p_f)(s7_scheme *sc, s7_pointer p1);
  bool (*b_pi_f)(s7_scheme *sc, s7_pointer p1, s7_int i2);
  bool (*b_ii_f)(s7_int i1, s7_int i2);
  bool (*b_7ii_f)(s7_scheme *sc, s7_int i1, s7_int i2);
  bool (*b_dd_f)(s7_double x1, s7_double x2);
  s7_pointer (*p_f)(s7_scheme *sc);
  s7_pointer (*p_p_f)(s7_scheme *sc, s7_pointer p);
  s7_pointer (*p_pp_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
  s7_pointer (*p_ppp_f)(s7_scheme *sc, s7_pointer p, s7_pointer p2, s7_pointer p3);
  s7_pointer (*p_pi_f)(s7_scheme *sc, s7_pointer p1, s7_int i1);
  s7_pointer (*p_pii_f)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2);
  s7_pointer (*p_ppi_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int i1);
  s7_pointer (*p_pip_f)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2);
  s7_pointer (*p_piip_f)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_pointer p3);
  s7_pointer (*p_i_f)(s7_scheme *sc, s7_int i);
  s7_pointer (*p_ii_f)(s7_scheme *sc, s7_int x1, s7_int x2);
  s7_pointer (*p_d_f)(s7_scheme *sc, s7_double x);
  s7_pointer (*p_dd_f)(s7_scheme *sc, s7_double x1, s7_double x2);
  s7_double (*fd)(opt_info *o);
  s7_int (*fi)(opt_info *o);
  bool (*fb)(opt_info *o);
  s7_pointer (*fp)(opt_info *o);
} vunion;

#define NUM_VUNIONS 15
struct opt_info {
  vunion v[NUM_VUNIONS];
  s7_scheme *sc;
};

#define O_WRAP (NUM_VUNIONS - 1)

#if WITH_GMP
typedef struct bigint {mpz_t n; struct bigint *nxt;} bigint;
typedef struct bigrat {mpq_t q; struct bigrat *nxt;} bigrat;
typedef struct bigflt {mpfr_t x; struct bigflt *nxt;} bigflt;
typedef struct bigcmp {mpc_t z; struct bigcmp *nxt;} bigcmp;

typedef struct {
  mpfr_t error, ux, x0, x1;
  mpz_t i, i0, i1, n;
  mpz_t p0, q0, r, r1, p1, q1, old_p1, old_q1;
  mpfr_t val, e0, e1, e0p, e1p, old_e0, old_e1, old_e0p;
  mpq_t q;
} rat_locals_t;
#endif

typedef intptr_t opcode_t;


/* -------------------------------- cell structure -------------------------------- */

typedef struct s7_cell {
  union {
    uint64_t flag;                /* type info */
    int64_t signed_flag;
    uint8_t type_field;
    uint16_t sflag;
    struct {
      uint32_t unused_low_flag;
      uint16_t opt_choice;
      uint16_t high_flag;
    } opts;
  } tf;
  union {

    union {                       /* integers, floats */
      s7_int integer_value;
      s7_double real_value;

      struct {                    /* ratios */
	s7_int numerator;
	s7_int denominator;
      } fraction_value;

      struct {                    /* complex numbers */
	s7_double rl;
	s7_double im;
      } complex_value;

#if WITH_GMP
      bigint *bgi;                /* bignums */
      bigrat *bgr;
      bigflt *bgf;
      bigcmp *bgc;
#endif
    } number;

    struct {
      s7_int unused1, unused2;    /* always int64_t so this is 16 bytes */
      uint8_t name[24];
    } number_name;

    struct {                      /* ports */
      port_t *port;
      uint8_t *data;
      s7_int size, point;
      block_t *block;
    } prt;

    struct{                       /* characters */
      uint8_t c, up_c;
      int32_t length;
      bool alpha_c, digit_c, space_c, upper_c, lower_c;
      char c_name[12];
    } chr;

    struct {                      /* c-pointers */
      void *c_pointer;
      s7_pointer c_type, info, weak1, weak2;
    } cptr;

    struct {                      /* vectors */
      s7_int length;
      union {
	s7_pointer *objects;
	s7_int *ints;
	s7_double *floats;
	uint8_t *bytes;
      } elements;
      block_t *block;
      s7_pointer (*vget)(s7_scheme *sc, s7_pointer vec, s7_int loc);
      union {
	s7_pointer (*vset)(s7_scheme *sc, s7_pointer vec, s7_int loc, s7_pointer val);
	s7_pointer fset;
      } setv;
    } vector;

    struct {                        /* stacks (internal) struct must match vector above for length/objects */
      s7_int length;
      s7_pointer *objects;
      block_t *block;
      int64_t top, flags;
    } stk;

    struct {                        /* hash-tables */
      s7_int mask;
      hash_entry_t **elements;
      hash_check_t hash_func;
      hash_map_t *loc;
      block_t *block;
    } hasher;

    struct {                        /* iterators */
      s7_pointer obj, cur;
      union {
	s7_int loc;
	s7_pointer lcur;
      } lc;
      union {
	s7_int len;
	s7_pointer slow;
	hash_entry_t *hcur;
      } lw;
      s7_pointer (*next)(s7_scheme *sc, s7_pointer iterator);
    } iter;

    struct {
      c_proc_t *c_proc;              /* C functions, macros */
      s7_function ff;
      s7_int required_args, optional_args, all_args; /* these could be uint32_t */
    } fnc;

    struct {                         /* pairs */
      s7_pointer car, cdr, opt1;
      union
      {
	s7_pointer opt2;
	s7_int n;
      } o2;
      union {
	s7_pointer opt3;
	s7_int n;
	uint8_t opt_type;
      } o3;
    } cons;

    struct {                         /* special purpose pairs (symbol-table etc) */
      s7_pointer unused_car, unused_cdr;
      uint64_t hash;
      const char *fstr;
      uint64_t location;            /* line/file/position, also used in symbol_table as raw_len */
    } sym_cons;

    struct {                        /* scheme functions */
      s7_pointer args, body, env, setter; /* args can be a symbol, as well as a list, setter can be #f as well as a procedure/closure */
      int32_t arity;
    } func;

    struct {                        /* strings */
      s7_int length;
      char *svalue;
      uint64_t hash;                /* string hash-index */
      block_t *block;
      block_t *gensym_block;
    } string;

    struct {                       /* symbols */
      s7_pointer name, global_slot, local_slot;
      int64_t id;                  /* which let last bound the symbol -- for faster symbol lookup */
      uint32_t ctr;                /* how many times has symbol been bound */
      uint32_t tag;                /* symbol as member of a set (tree-set-memq etc), high 32 bits are in symbol_info (the string block) */
    } sym;

    struct {                       /* syntax */
      s7_pointer symbol;
      opcode_t op;
      int32_t min_args, max_args;
      const char *documentation;
    } syn;

    struct {                       /* slots (bindings) */
      s7_pointer sym, val, nxt, pending_value, expr;
    } slt;

    struct {                       /* lets (environments) */
      s7_pointer slots, nxt;
      int64_t id;                  /* id of rootlet is -1 */
      union {
	struct {
	  s7_pointer function;     /* *function* (symbol) if this is a funclet */
	  uint32_t line, file;     /* *function* location if it is known */
	} efnc;
	struct {
	  s7_pointer dox1, dox2;   /* do loop variables */
	} dox;
	s7_int key;                /* sc->baffle_ctr type */
      } edat;
    } envr;

    struct {                        /* special stuff like #<unspecified> */
      s7_pointer car, cdr;          /* unique_car|cdr, for sc->nil these are sc->unspecified for faster assoc etc */
      int64_t unused_let_id;        /* let_id(sc->nil) is -1, so this needs to align with envr.id above, only used by sc->nil, so free elsewhere */
      const char *name;
      s7_int len;
    } unq;

    struct {                        /* #<...> */
      char *name;                   /* not const because the GC frees it */
      s7_int len;
    } undef;

    struct {                        /* #<eof> */
      const char *name;
      s7_int len;
    } eof;

    struct {                        /* counter (internal) */
      s7_pointer result, list, env, slots; /* env = counter_let (curlet after map/for-each let created) */
      uint64_t cap;                 /* sc->capture_let_counter for let reuse */
    } ctr;

    struct {                        /* random-state */
#if WITH_GMP
      gmp_randstate_t state;
#else
      uint64_t seed, carry;
#endif
    } rng;

    struct {                        /* additional object types (C) */
      s7_int type;
      void *value;                  /* the value the caller associates with the c_object */
      s7_pointer e;                 /* the method list, if any (openlet) */
      s7_scheme *sc;
    } c_obj;

    struct {                        /* continuations */
      block_t *block;
      s7_pointer stack, op_stack;
      s7_pointer *stack_start, *stack_end;
    } cwcc;

    struct {                        /* call-with-exit */
      uint64_t goto_loc, op_stack_loc;
      bool active;
      s7_pointer name;
    } rexit;

    struct {                        /* catch */
      uint64_t goto_loc, op_stack_loc;
      s7_pointer tag;
      s7_pointer handler;
      Jmp_Buf *cstack;
    } rcatch; /* C++ reserves "catch" I guess */

    struct {                        /* dynamic-wind */
      s7_pointer in, out, body;
      dwind_t state;
    } winder;
  } object;

#if S7_DEBUGGING
  int32_t current_alloc_line, previous_alloc_line, uses, explicit_free_line, gc_line;
  int64_t current_alloc_type, previous_alloc_type, debugger_bits;
  const char *current_alloc_func, *previous_alloc_func, *gc_func;
#endif
} s7_cell;


typedef struct s7_big_cell {
  s7_cell cell;
  int64_t big_hloc;
} s7_big_cell;
typedef struct s7_big_cell *s7_big_pointer;

typedef struct heap_block_t {
  intptr_t start, end;
  int64_t offset;
  struct heap_block_t *next;
} heap_block_t;

typedef struct {
  s7_pointer *objs;
  int32_t size, top, ref, size2;
  bool has_hits;
  int32_t *refs;
  s7_pointer cycle_port, init_port;
  s7_int cycle_loc, init_loc, ctr;
  bool *defined;
} shared_info_t;

typedef struct {
  s7_int loc, curly_len, ctr;
  char *curly_str;
  s7_pointer args, orig_str, curly_arg, port, strport;
} format_data_t;

typedef struct gc_obj_t {
  s7_pointer p;
  struct gc_obj_t *nxt;
} gc_obj_t;

typedef struct {
  s7_pointer *list;
  s7_int size, loc;
} gc_list_t;

typedef struct {
  s7_int size, top, excl_size, excl_top;
  s7_pointer *funcs, *let_names, *files;
  s7_int *timing_data, *excl, *lines;
} profile_data_t;


/* -------------------------------- s7_scheme struct -------------------------------- */
struct s7_scheme {
  s7_pointer code;    /* layout of first 4 entries should match stack frame layout */
  s7_pointer curlet;
  s7_pointer args;
  opcode_t cur_op;
  s7_pointer value, cur_code;
  token_t tok;

  s7_pointer stack;                   /* stack is a vector */
  uint32_t stack_size;
  s7_pointer *stack_start, *stack_end, *stack_resize_trigger;

  s7_pointer *op_stack, *op_stack_now, *op_stack_end;
  uint32_t op_stack_size, max_stack_size;

  s7_cell **heap, **free_heap, **free_heap_top, **free_heap_trigger, **previous_free_heap_top;
  int64_t heap_size, gc_freed, gc_total_freed, max_heap_size, gc_temps_size;
  s7_double gc_resize_heap_fraction, gc_resize_heap_by_4_fraction;
  s7_int gc_calls, gc_total_time, gc_start, gc_end;
  heap_block_t *heap_blocks;

#if WITH_HISTORY
  s7_pointer eval_history1, eval_history2, error_history, history_sink, history_pairs, old_cur_code;
  bool using_history1;
#endif

#if WITH_MULTITHREAD_CHECKS
  int32_t lock_count;
  pthread_mutex_t lock;
#endif

  gc_obj_t *permanent_objects, *permanent_lets;
  s7_pointer protected_objects, protected_setters, protected_setter_symbols;  /* vectors of gc-protected objects */
  s7_int *protected_objects_free_list;    /* to avoid a linear search for a place to store an object in sc->protected_objects */
  s7_int protected_objects_size, protected_setters_size, protected_setters_loc;
  s7_int protected_objects_free_list_loc;

  s7_pointer nil;                     /* empty list */
  s7_pointer T;                       /* #t */
  s7_pointer F;                       /* #f */
  s7_pointer undefined;               /* #<undefined> */
  s7_pointer unspecified;             /* #<unspecified> */
  s7_pointer no_value;                /* the (values) value */
  s7_pointer unused;                  /* a marker for an unoccupied slot in sc->protected_objects (and other similar stuff) */

  s7_pointer symbol_table;            /* symbol table */
  s7_pointer rootlet, shadow_rootlet; /* rootlet */
  s7_int rootlet_entries;
  s7_pointer unlet;                   /* original bindings of predefined functions */

  s7_pointer input_port;              /* current-input-port */
  s7_pointer *input_port_stack;       /*   input port stack (load and read internally) */
  uint32_t input_port_stack_size, input_port_stack_loc;

  s7_pointer output_port;             /* current-output-port */
  s7_pointer error_port;              /* current-error-port */
  s7_pointer owlet;                   /* owlet */
  s7_pointer error_type, error_data, error_code, error_line, error_file, error_position; /* owlet slots */
  s7_pointer standard_input, standard_output, standard_error;

  s7_pointer sharp_readers;           /* the binding pair for the global *#readers* list */
  s7_pointer load_hook;               /* *load-hook* hook object */
  s7_pointer autoload_hook;           /* *autoload-hook* hook object */
  s7_pointer unbound_variable_hook;   /* *unbound-variable-hook* hook object */
  s7_pointer missing_close_paren_hook, rootlet_redefinition_hook;
  s7_pointer error_hook, read_error_hook; /* *error-hook* hook object, and *read-error-hook* */
  bool gc_off;                        /* gc_off: if true, the GC won't run */
  uint32_t gc_stats, gensym_counter, f_class, add_class, multiply_class, subtract_class, num_eq_class;
  int32_t format_column, error_argnum;
  uint64_t capture_let_counter;
  bool short_print, is_autoloading, in_with_let, object_out_locked, has_openlets, is_expanding, accept_all_keyword_arguments, muffle_warnings;
  bool got_tc, got_rec, not_tc;
  s7_int rec_tc_args, continuation_counter;
  int64_t let_number;
  s7_double default_rationalize_error, equivalent_float_epsilon, hash_table_float_epsilon;
  s7_int default_hash_table_length, initial_string_port_length, print_length, objstr_max_len, history_size, true_history_size, output_port_data_size;
  s7_int max_vector_length, max_string_length, max_list_length, max_vector_dimensions, max_format_length, max_port_data_size, rec_loc, rec_len;
  s7_pointer stacktrace_defaults;

  s7_pointer rec_stack, rec_testp, rec_f1p, rec_f2p, rec_f3p, rec_f4p, rec_f5p, rec_f6p, rec_f7p, rec_f8p, rec_f9p;
  s7_pointer rec_resp, rec_slot1, rec_slot2, rec_slot3, rec_p1, rec_p2;
  s7_pointer *rec_els;
  s7_function rec_testf, rec_f1f, rec_f2f, rec_f3f, rec_f4f, rec_f5f, rec_f6f, rec_f7f, rec_f8f, rec_f9f, rec_resf, rec_fn;
  s7_int (*rec_fi1)(opt_info *o);
  s7_int (*rec_fi2)(opt_info *o);
  s7_int (*rec_fi3)(opt_info *o);
  s7_int (*rec_fi4)(opt_info *o);
  s7_int (*rec_fi5)(opt_info *o);
  s7_int (*rec_fi6)(opt_info *o);
  bool (*rec_fb1)(opt_info *o);
  bool (*rec_fb2)(opt_info *o);

  opt_info *rec_test_o, *rec_result_o, *rec_a1_o, *rec_a2_o, *rec_a3_o, *rec_a4_o, *rec_a5_o, *rec_a6_o;
  s7_i_ii_t rec_i_ii_f;
  s7_d_dd_t rec_d_dd_f;
  s7_pointer rec_val1, rec_val2;

  int32_t float_format_precision;
  vdims_t *wrap_only;

  char *typnam;
  int32_t typnam_len, print_width;
  s7_pointer *singletons;
  block_t *unentry;                   /* hash-table lookup failure indicator */

  #define INITIAL_FILE_NAMES_SIZE 8
  s7_pointer *file_names;
  int32_t file_names_size, file_names_top;

  #define INITIAL_STRBUF_SIZE 1024
  s7_int strbuf_size;
  char *strbuf;

  char *read_line_buf;
  s7_int read_line_buf_size;

  s7_pointer w, x, y, z;
  s7_pointer temp1, temp2, temp3, temp4, temp5, temp6, temp7, temp8, temp9;
  s7_pointer t1_1, t2_1, t2_2, t3_1, t3_2, t3_3, t4_1, u1_1, u2_1, u2_2;
  s7_pointer elist_1, elist_2, elist_3, elist_4, elist_5, elist_6, elist_7;
  s7_pointer plist_1, plist_2, plist_2_2, plist_3, qlist_2, qlist_3, clist_1, clist_2, dlist_1; /* dlist|clist and ulist can't overlap */

  Jmp_Buf *goto_start;
  bool longjmp_ok;
  int32_t setjmp_loc;

  void (*begin_hook)(s7_scheme *sc, bool *val);
  opcode_t begin_op;

  bool debug_or_profile, profiling_gensyms;
  s7_int current_line, s7_call_line, debug, profile, profile_position;
  s7_pointer profile_prefix;
  profile_data_t *profile_data;
  const char *current_file, *s7_call_file, *s7_call_name;

  shared_info_t *circle_info;
  format_data_t **fdats;
  int32_t num_fdats, last_error_line, safety;
  gc_list_t *strings, *vectors, *input_ports, *output_ports, *input_string_ports, *continuations, *c_objects, *hash_tables;
  gc_list_t *gensyms, *undefineds, *multivectors, *weak_refs, *weak_hash_iterators, *opt1_funcs;
#if (WITH_GMP)
  gc_list_t *big_integers, *big_ratios, *big_reals, *big_complexes, *big_random_states;
  mpz_t mpz_1, mpz_2, mpz_3, mpz_4;
  mpq_t mpq_1, mpq_2, mpq_3;
  mpfr_t mpfr_1, mpfr_2, mpfr_3;
  mpc_t mpc_1, mpc_2;
  rat_locals_t *ratloc;
  bigint *bigints;
  bigrat *bigrats;
  bigflt *bigflts;
  bigcmp *bigcmps;
#endif
  s7_pointer *setters;
  s7_int setters_size, setters_loc;
  s7_pointer *tree_pointers;
  int32_t tree_pointers_size, tree_pointers_top, permanent_cells, num_to_str_size;
  s7_pointer format_ports;
  uint32_t alloc_pointer_k, alloc_function_k, alloc_symbol_k;
  s7_cell *alloc_pointer_cells;
  c_proc_t *alloc_function_cells;
  uint32_t alloc_big_pointer_k;
  s7_big_cell *alloc_big_pointer_cells;
  s7_pointer string_wrappers, integer_wrappers, real_wrappers;
  uint8_t *alloc_symbol_cells;
  char *num_to_str;

  block_t *block_lists[NUM_BLOCK_LISTS];
  size_t alloc_string_k;
  char *alloc_string_cells;

  c_object_t **c_object_types;
  int32_t c_object_types_size, num_c_object_types;
  s7_pointer type_to_typers[NUM_TYPES];

  uint32_t syms_tag, syms_tag2;
  int32_t bignum_precision;
  s7_int baffle_ctr;
  s7_pointer default_rng;

  s7_pointer sort_body, sort_begin, sort_v1, sort_v2;
  opcode_t sort_op;
  s7_int sort_body_len;
  s7_b_7pp_t sort_f;
  opt_info *sort_o;
  bool (*sort_fb)(opt_info *o);

  #define INT_TO_STR_SIZE 32
  char int_to_str1[INT_TO_STR_SIZE], int_to_str2[INT_TO_STR_SIZE], int_to_str3[INT_TO_STR_SIZE], int_to_str4[INT_TO_STR_SIZE], int_to_str5[INT_TO_STR_SIZE];

  s7_pointer abs_symbol, acos_symbol, acosh_symbol, add_symbol, angle_symbol, append_symbol, apply_symbol, apply_values_symbol, arity_symbol,
             ash_symbol, asin_symbol, asinh_symbol, assoc_symbol, assq_symbol, assv_symbol, atan_symbol, atanh_symbol, autoload_symbol, autoloader_symbol,
             bacro_symbol, bacro_star_symbol, bignum_symbol, byte_vector_symbol, byte_vector_ref_symbol, byte_vector_set_symbol, byte_vector_to_string_symbol,
             c_pointer_symbol, c_pointer_info_symbol, c_pointer_to_list_symbol, c_pointer_type_symbol, c_pointer_weak1_symbol, c_pointer_weak2_symbol,
             caaaar_symbol, caaadr_symbol, caaar_symbol, caadar_symbol, caaddr_symbol, caadr_symbol,
             caar_symbol, cadaar_symbol, cadadr_symbol, cadar_symbol, caddar_symbol, cadddr_symbol, caddr_symbol, cadr_symbol,
             call_cc_symbol, call_with_current_continuation_symbol, call_with_exit_symbol, call_with_input_file_symbol,
             call_with_input_string_symbol, call_with_output_file_symbol, call_with_output_string_symbol, car_symbol,
             catch_symbol, cdaaar_symbol, cdaadr_symbol, cdaar_symbol, cdadar_symbol, cdaddr_symbol, cdadr_symbol, cdar_symbol,
             cddaar_symbol, cddadr_symbol, cddar_symbol, cdddar_symbol, cddddr_symbol, cdddr_symbol, cddr_symbol, cdr_symbol,
             ceiling_symbol, char_downcase_symbol, char_eq_symbol, char_geq_symbol, char_gt_symbol, char_leq_symbol, char_lt_symbol,
             char_position_symbol, char_to_integer_symbol, char_upcase_symbol, cload_directory_symbol, close_input_port_symbol,
             close_output_port_symbol, complex_symbol, cons_symbol, copy_symbol, cos_symbol, cosh_symbol, coverlet_symbol,
             curlet_symbol, current_error_port_symbol, current_input_port_symbol, current_output_port_symbol, cutlet_symbol, cyclic_sequences_symbol,
             denominator_symbol, dilambda_symbol, display_symbol, divide_symbol, documentation_symbol, dynamic_wind_symbol, dynamic_unwind_symbol,
             num_eq_symbol, error_symbol, eval_string_symbol, eval_symbol, exact_to_inexact_symbol, exit_symbol, exp_symbol, expt_symbol,
             features_symbol, file__symbol, fill_symbol, float_vector_ref_symbol, float_vector_set_symbol, float_vector_symbol, floor_symbol,
             flush_output_port_symbol, for_each_symbol, format_symbol, funclet_symbol, _function__symbol,
             gc_symbol, gcd_symbol, gensym_symbol, geq_symbol, get_output_string_symbol, gt_symbol,
             hash_table_entries_symbol, hash_table_ref_symbol, hash_table_set_symbol, hash_table_symbol, help_symbol,
             imag_part_symbol, immutable_symbol, inexact_to_exact_symbol, inlet_symbol, int_vector_ref_symbol, int_vector_set_symbol, int_vector_symbol,
             integer_decode_float_symbol, integer_to_char_symbol,
             is_aritable_symbol, is_bignum_symbol, is_boolean_symbol, is_byte_symbol, is_byte_vector_symbol,
             is_c_object_symbol, c_object_type_symbol, is_c_pointer_symbol, is_char_alphabetic_symbol, is_char_lower_case_symbol, is_char_numeric_symbol,
             is_char_symbol, is_char_upper_case_symbol, is_char_whitespace_symbol, is_complex_symbol, is_constant_symbol,
             is_continuation_symbol, is_defined_symbol, is_dilambda_symbol, is_eof_object_symbol, is_eq_symbol, is_equal_symbol,
             is_eqv_symbol, is_even_symbol, is_exact_symbol, is_float_vector_symbol, is_funclet_symbol,
             is_gensym_symbol, is_goto_symbol, is_hash_table_symbol, is_immutable_symbol,
             is_inexact_symbol, is_infinite_symbol, is_input_port_symbol, is_int_vector_symbol, is_integer_symbol, is_iterator_symbol,
             is_keyword_symbol, is_let_symbol, is_list_symbol, is_macro_symbol, is_equivalent_symbol, is_nan_symbol, is_negative_symbol,
             is_null_symbol, is_number_symbol, is_odd_symbol, is_openlet_symbol, is_output_port_symbol, is_pair_symbol,
             is_port_closed_symbol, is_positive_symbol, is_procedure_symbol, is_proper_list_symbol, is_provided_symbol,
             is_random_state_symbol, is_rational_symbol, is_real_symbol, is_sequence_symbol, is_string_symbol, is_subvector_symbol,
             is_symbol_symbol, is_syntax_symbol, is_vector_symbol, is_weak_hash_table_symbol, is_zero_symbol,
             is_float_symbol, is_integer_or_real_at_end_symbol, is_integer_or_any_at_end_symbol, is_unspecified_symbol, is_undefined_symbol,
             iterate_symbol, iterator_is_at_end_symbol, iterator_sequence_symbol,
             keyword_to_symbol_symbol,
             lcm_symbol, length_symbol, leq_symbol, let_ref_fallback_symbol, let_ref_symbol, let_set_fallback_symbol,
             let_set_symbol, let_temporarily_symbol, libraries_symbol, list_ref_symbol, list_set_symbol, list_symbol, list_tail_symbol, list_values_symbol,
             load_path_symbol, load_symbol, log_symbol, logand_symbol, logbit_symbol, logior_symbol, lognot_symbol, logxor_symbol, lt_symbol,
             macro_symbol, macro_star_symbol, magnitude_symbol, make_byte_vector_symbol, make_float_vector_symbol, make_hash_table_symbol,
             make_weak_hash_table_symbol, make_int_vector_symbol, make_iterator_symbol, string_to_keyword_symbol, make_list_symbol, make_string_symbol,
             make_vector_symbol, map_symbol, max_symbol, member_symbol, memq_symbol, memv_symbol, min_symbol, modulo_symbol, multiply_symbol,
             name_symbol, newline_symbol, not_symbol, number_to_string_symbol, numerator_symbol,
             object_to_string_symbol, object_to_let_symbol, open_input_file_symbol, open_input_function_symbol, open_input_string_symbol,
             open_output_file_symbol, open_output_function_symbol, open_output_string_symbol, openlet_symbol, outlet_symbol, owlet_symbol,
             pair_filename_symbol, pair_line_number_symbol, peek_char_symbol, pi_symbol, port_filename_symbol, port_line_number_symbol,
             port_file_symbol, port_position_symbol, procedure_source_symbol, provide_symbol,
             qq_append_symbol, quotient_symbol,
             random_state_symbol, random_state_to_list_symbol, random_symbol, rationalize_symbol, read_byte_symbol,
             read_char_symbol, read_line_symbol, read_string_symbol, read_symbol, real_part_symbol, remainder_symbol,
             require_symbol, reverse_symbol, reverseb_symbol, rootlet_symbol, round_symbol,
             setter_symbol, set_car_symbol, set_cdr_symbol,
             set_current_error_port_symbol, set_current_input_port_symbol, set_current_output_port_symbol,
             signature_symbol, sin_symbol, sinh_symbol, sort_symbol, sqrt_symbol,
             stacktrace_symbol, string_append_symbol, string_copy_symbol, string_downcase_symbol, string_eq_symbol, string_fill_symbol,
             string_geq_symbol, string_gt_symbol, string_leq_symbol, string_lt_symbol, string_position_symbol, string_ref_symbol,
             string_set_symbol, string_symbol, string_to_number_symbol, string_to_symbol_symbol, string_upcase_symbol,
             sublet_symbol, substring_symbol, subtract_symbol, subvector_symbol, subvector_position_symbol, subvector_vector_symbol,
             symbol_symbol, symbol_to_dynamic_value_symbol,
             symbol_to_keyword_symbol, symbol_to_string_symbol, symbol_to_value_symbol,
             tan_symbol, tanh_symbol, throw_symbol, string_to_byte_vector_symbol,
             tree_count_symbol, tree_leaves_symbol, tree_memq_symbol, tree_set_memq_symbol, tree_is_cyclic_symbol, truncate_symbol, type_of_symbol,
             unlet_symbol,
             values_symbol, varlet_symbol, vector_append_symbol, vector_dimension_symbol, vector_dimensions_symbol, vector_fill_symbol,
             vector_rank_symbol, vector_ref_symbol, vector_set_symbol, vector_symbol,
             weak_hash_table_symbol, with_input_from_file_symbol, with_input_from_string_symbol, with_output_to_file_symbol, with_output_to_string_symbol,
             write_byte_symbol, write_char_symbol, write_string_symbol, write_symbol,
             local_documentation_symbol, local_signature_symbol, local_setter_symbol, local_iterator_symbol;
  s7_pointer hash_code_symbol, dummy_equal_hash_table;
#if (!WITH_PURE_S7)
  s7_pointer is_char_ready_symbol, char_ci_leq_symbol, char_ci_lt_symbol, char_ci_eq_symbol, char_ci_geq_symbol, char_ci_gt_symbol,
             let_to_list_symbol, integer_length_symbol, string_ci_leq_symbol, string_ci_lt_symbol, string_ci_eq_symbol,
             string_ci_geq_symbol, string_ci_gt_symbol, string_to_list_symbol, vector_to_list_symbol, string_length_symbol,
             list_to_string_symbol, list_to_vector_symbol, vector_length_symbol;
#endif

  /* syntax symbols et al */
  s7_pointer else_symbol, lambda_symbol, lambda_star_symbol, let_symbol, quote_symbol, quasiquote_symbol, unquote_symbol, macroexpand_symbol,
             define_expansion_symbol, define_expansion_star_symbol, with_let_symbol, if_symbol, autoload_error_symbol,
             when_symbol, unless_symbol, begin_symbol, cond_symbol, case_symbol, and_symbol, or_symbol, do_symbol,
             define_symbol, define_star_symbol, define_constant_symbol, with_baffle_symbol, define_macro_symbol, no_setter_symbol,
             define_macro_star_symbol, define_bacro_symbol, define_bacro_star_symbol, letrec_symbol, letrec_star_symbol, let_star_symbol,
             rest_keyword, allow_other_keys_keyword, readable_keyword, display_keyword, write_keyword, value_symbol, type_symbol,
             baffled_symbol, set_symbol, body_symbol, class_name_symbol, feed_to_symbol, format_error_symbol, immutable_error_symbol,
             wrong_number_of_args_symbol, read_error_symbol, string_read_error_symbol, syntax_error_symbol, division_by_zero_symbol, bad_result_symbol,
             no_catch_symbol, io_error_symbol, invalid_escape_function_symbol, wrong_type_arg_symbol, out_of_range_symbol, out_of_memory_symbol,
             missing_method_symbol, unbound_variable_symbol, if_keyword, symbol_table_symbol, profile_in_symbol, trace_in_symbol;

  /* signatures of sequences used as applicable objects: ("hi" 1) */
  s7_pointer string_signature, vector_signature, float_vector_signature, int_vector_signature, byte_vector_signature,
             c_object_signature, let_signature, hash_table_signature, pair_signature;
  /* common signatures */
  s7_pointer pcl_bc, pcl_bs, pcl_bt, pcl_c, pcl_e, pcl_f, pcl_i, pcl_n, pcl_r, pcl_s, pcl_v, pl_bc, pl_bn, pl_bt, pl_p, pl_sf, pl_tl, pl_nn;

  /* optimizer s7_functions */
  s7_pointer add_2, add_3, add_1x, add_x1, subtract_1, subtract_2, subtract_3, subtract_x1, subtract_2f, subtract_f2, simple_char_eq,
             char_equal_2, char_greater_2, char_less_2, char_position_csi, string_equal_2, substring_uncopied, display_2, display_f,
             string_greater_2, string_less_2, symbol_to_string_uncopied, get_output_string_uncopied, string_equal_2c, string_c1, string_append_2,
             vector_ref_2, vector_ref_3, vector_set_3, vector_set_4, read_char_1, dynamic_wind_unchecked, dynamic_wind_body, dynamic_wind_init, append_2,
             fv_ref_2, fv_ref_3, fv_set_3, fv_set_unchecked, iv_ref_2, iv_ref_3, iv_set_3, bv_ref_2, bv_ref_3, bv_set_3, vector_2, vector_3,
             list_0, list_1, list_2, list_3, list_4, list_set_i, hash_table_ref_2, hash_table_2, list_ref_at_0, list_ref_at_1, list_ref_at_2,
             format_f, format_no_column, format_just_control_string, format_as_objstr, values_uncopied,
             memq_2, memq_3, memq_4, memq_any, tree_set_memq_syms, simple_inlet, profile_out, simple_list_values,
             lint_let_ref, lint_let_set, geq_2, add_i_random, is_defined_in_rootlet;

  s7_pointer multiply_2, invert_1, invert_x, divide_2, divide_by_2, max_2, min_2, max_3, min_3,
             num_eq_2, num_eq_xi, num_eq_ix, less_xi, less_xf, less_x0, less_2, greater_xi, greater_xf, greater_2,
             leq_xi, leq_2, leq_ixx, geq_xi, geq_xf, random_i, random_f, random_1,
             mul_2_ff, mul_2_ii, mul_2_if, mul_2_fi, mul_2_xi, mul_2_ix, mul_2_fx, mul_2_xf,
             add_2_ff, add_2_ii, add_2_if, add_2_fi, add_2_xi, add_2_ix, add_2_fx, add_2_xf;
  s7_pointer seed_symbol, carry_symbol;

  /* object->let symbols */
  s7_pointer active_symbol, goto_symbol, data_symbol, weak_symbol, dimensions_symbol, info_symbol, c_type_symbol, source_symbol, c_object_ref_symbol,
             at_end_symbol, sequence_symbol, position_symbol, entries_symbol, locked_symbol, function_symbol, open_symbol, alias_symbol, port_type_symbol,
             file_symbol, file_info_symbol, line_symbol, c_object_let_symbol, class_symbol, current_value_symbol, closed_symbol,
             mutable_symbol, size_symbol, original_vector_symbol, pointer_symbol;

#if WITH_SYSTEM_EXTRAS
  s7_pointer is_directory_symbol, file_exists_symbol, delete_file_symbol, getenv_symbol, system_symbol, directory_to_list_symbol, file_mtime_symbol;
#endif
  s7_pointer open_input_function_choices[S7_NUM_READ_CHOICES];
  s7_pointer closed_input_function, closed_output_function;
  s7_pointer vector_set_function, string_set_function, list_set_function, hash_table_set_function, let_set_function, c_object_set_function, last_function;
  s7_pointer wrong_type_arg_info, out_of_range_info, simple_wrong_type_arg_info, simple_out_of_range_info;

  #define NUM_SAFE_PRELISTS 8
  #define NUM_SAFE_LISTS 64               /* 36 is the biggest normally (lint.scm), 49 in s7test, 57 in snd-test */
  s7_pointer safe_lists[NUM_SAFE_LISTS];
  int32_t current_safe_list;

  s7_pointer autoload_table, s7_let, s7_let_symbol, let_temp_hook;
  const char ***autoload_names;
  s7_int *autoload_names_sizes;
  bool **autoloaded_already;
  s7_int autoload_names_loc, autoload_names_top;
  int32_t format_depth;
  bool undefined_identifier_warnings, undefined_constant_warnings, stop_at_error;

  opt_funcs_t *alloc_opt_func_cells;
  int32_t alloc_opt_func_k;

  int32_t pc;
  #define OPTS_SIZE 256          /* pqw-vox needs 178 */
  opt_info *opts[OPTS_SIZE + 1]; /* this form is a lot faster than opt_info**! */

  #define INITIAL_SAVED_POINTERS_SIZE 256
  void **saved_pointers;
  s7_int saved_pointers_loc, saved_pointers_size;

  s7_pointer prepackaged_type_names[NUM_TYPES];

#if S7_DEBUGGING
  int32_t *tc_rec_calls;
  int32_t last_gc_line;
  bool printing_gc_info;
#endif
};

#if S7_DEBUGGING
  static void gdb_break(void) {};
#endif
static s7_scheme *cur_sc = NULL; /* intended for gdb (see gdbinit), but also used if S7_DEBUGGING unfortunately */

static noreturn void s7_error_nr(s7_scheme *sc, s7_pointer type, s7_pointer info);

#if POINTER_32
static void *Malloc(size_t bytes)
{
  void *p = malloc(bytes);
  if (!p) s7_error_nr(cur_sc, cur_sc->out_of_memory_symbol, cur_sc->nil);
  return(p);
}

static void *Calloc(size_t nmemb, size_t size)
{
  void *p = calloc(nmemb, size);
  if (!p) s7_error_nr(cur_sc, cur_sc->out_of_memory_symbol, cur_sc->nil);
  return(p);
}

static void *Realloc(void *ptr, size_t size)
{
  void *p = realloc(ptr, size);
  if (!p) s7_error_nr(cur_sc, cur_sc->out_of_memory_symbol, cur_sc->nil);
  return(p);
}
#else
#define Malloc malloc
#define Calloc calloc
#define Realloc realloc
#endif


/* -------------------------------- mallocate -------------------------------- */

static void add_saved_pointer(s7_scheme *sc, void *p)
{
  if (sc->saved_pointers_loc == sc->saved_pointers_size)
    {
      sc->saved_pointers_size *= 2;
      sc->saved_pointers = (void **)Realloc(sc->saved_pointers, sc->saved_pointers_size * sizeof(void *));
    }
  sc->saved_pointers[sc->saved_pointers_loc++] = p;
}

static const int32_t intlen_bits[256] =
  {0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
   6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
   7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
   7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8};

static void memclr(void *s, size_t n)
{
  uint8_t *s2;
#if S7_ALIGNED
  s2 = (uint8_t *)s;
#else
#if (defined(__x86_64__) || defined(__i386__))
  if (n >= 8)
    {
      int64_t *s1 = (int64_t *)s;
      size_t n8 = n >> 3;
      do {*s1++ = 0;} while (--n8 > 0); /* LOOP_4 here is slower */
      n &= 7;
      s2 = (uint8_t *)s1;
    }
  else s2 = (uint8_t *)s;
#else
  s2 = (uint8_t *)s;
#endif
#endif
  while (n > 0)
    {
      *s2++ = 0;
      n--;
    }
}

#define LOOP_4(Code) do {Code; Code; Code; Code;} while (0)
#define LOOP_8(Code) do {Code; Code; Code; Code; Code; Code; Code; Code;} while (0)
#define STEP_8(Var) (((Var) & 0x7) == 0)
#define STEP_64(Var) (((Var) & 0x3f) == 0)

#if POINTER_32
#define memclr64 memclr
#else
static Vectorized void memclr64(void *p, size_t bytes)
{
  size_t n = bytes >> 3;
  int64_t *vals = (int64_t *)p;
  for (size_t i = 0; i < n; )
    LOOP_8(vals[i++] = 0);
}
#endif

static void init_block_lists(s7_scheme *sc)
{
  for (int32_t i = 0; i < NUM_BLOCK_LISTS; i++)
    sc->block_lists[i] = NULL;
}

static inline void liberate(s7_scheme *sc, block_t *p)
{
  if (block_index(p) != TOP_BLOCK_LIST)
    {
      block_next(p) = (struct block_t *)sc->block_lists[block_index(p)];
      sc->block_lists[block_index(p)] = p;
    }
  else
    {
      if (block_data(p))
	{
	  free(block_data(p));
	  block_data(p) = NULL;
	}
      block_next(p) = (struct block_t *)sc->block_lists[BLOCK_LIST];
      sc->block_lists[BLOCK_LIST] = p;
    }
}

static inline void liberate_block(s7_scheme *sc, block_t *p)
{
  block_next(p) = (struct block_t *)sc->block_lists[BLOCK_LIST]; /* BLOCK_LIST=0 */
  sc->block_lists[BLOCK_LIST] = p;
}

static void fill_block_list(s7_scheme *sc)
{
  #define BLOCK_MALLOC_SIZE 256
  block_t *b = (block_t *)Malloc(BLOCK_MALLOC_SIZE * sizeof(block_t)); /* batch alloc means blocks in this batch can't be freed, only returned to the list */
  add_saved_pointer(sc, b);
  sc->block_lists[BLOCK_LIST] = b;
  for (int32_t i = 0; i < BLOCK_MALLOC_SIZE - 1; b++, i++)
    block_next(b) = (block_t *)(b + 1);
  block_next(b) = NULL;
}

static inline block_t *mallocate_block(s7_scheme *sc)
{
  block_t *p;
  if (!sc->block_lists[BLOCK_LIST])
    fill_block_list(sc);                /* this is much faster than allocating blocks as needed */
  p = sc->block_lists[BLOCK_LIST];
  sc->block_lists[BLOCK_LIST] = (block_t *)(block_next(p));
  block_set_index(p, BLOCK_LIST);
  return(p);
}

static inline char *permalloc(s7_scheme *sc, size_t len)
{
  #define ALLOC_STRING_SIZE (65536 * 8) /* going up to 16 made no difference in timings */
  #define ALLOC_MAX_STRING (512 * 8)    /* was 256 -- sets max size of block space lost at the end, but smaller = more direct malloc calls */
  char *result;
  size_t next_k;

  len = (len + 7) & (~7);            /* 8-byte aligned -- more than half the time, len is already 8-byte aligned */
  next_k = sc->alloc_string_k + len;
  if (next_k > ALLOC_STRING_SIZE)
    {
      if (len >= ALLOC_MAX_STRING)
	{
	  result = (char *)Malloc(len);
	  add_saved_pointer(sc, result);
	  return(result);
	}
      sc->alloc_string_cells = (char *)Malloc(ALLOC_STRING_SIZE); /* get a new block */
      add_saved_pointer(sc, sc->alloc_string_cells);
      sc->alloc_string_k = 0;
      next_k = len;
    }
  result = &(sc->alloc_string_cells[sc->alloc_string_k]);
  sc->alloc_string_k = next_k;
  return(result);
}

static Inline block_t *mallocate(s7_scheme *sc, size_t bytes)
{
  block_t *p;
  if (bytes > 0)
    {
      int32_t index;
      if (bytes <= 8) /* presetting a version of intlen_bits with 3's gave only a small speed-up */
	index = 3;
      else
	{
	  if (bytes <= 256)
	    index = intlen_bits[bytes - 1];
	  else index = (bytes <= 65536) ? (8 + intlen_bits[(bytes - 1) >> 8]) : TOP_BLOCK_LIST;   /* expansion to (1 << 17) made no difference */
	}
      p = sc->block_lists[index];
      if (p)
	sc->block_lists[index] = (block_t *)block_next(p);
      else
	{
	  if (index < (TOP_BLOCK_LIST - 1))
	    {
	      p = sc->block_lists[index + 1];
	      if (p)
		{
		  /* we are "borrowing" a block from the next larger bin -- this saves space but costs a bit of time.
		   *   in a tauto run repeating every call 1000 times, the old form ends up at 129M and 31.1 secs,
		   *   whereas the borrowing form ends at 116M and 31.5 secs, but most of my tests show a slight
		   *   speed-up, probably because grabbing a block here is faster than making a new one.
		   *   Worst case is tlet: 8 slower in callgrind.
		   */
		  sc->block_lists[index + 1] = (block_t *)block_next(p);
		  block_set_size(p, bytes);
		  return(p);
		}}
	  p = mallocate_block(sc);
	  block_data(p) = (index < TOP_BLOCK_LIST) ? (void *)permalloc(sc, (size_t)(1 << index)) : Malloc(bytes);
	  block_set_index(p, index);
	}}
  else p = mallocate_block(sc);
  block_set_size(p, bytes);
  return(p);
}

static block_t *callocate(s7_scheme *sc, size_t bytes)
{
  block_t *p = mallocate(sc, bytes);
  if ((block_data(p)) && (block_index(p) != BLOCK_LIST))
    {
      if ((bytes & (~0x3f)) > 0)
	memclr64((void *)block_data(p), bytes & (~0x3f));
      if ((bytes & 0x3f) > 0)
	memclr((void *)((uint8_t *)block_data(p) + (bytes & (~0x3f))), bytes & 0x3f);
    }
  return(p);
}

static block_t *reallocate(s7_scheme *sc, block_t *op, size_t bytes)
{
  block_t *np = mallocate(sc, bytes);
  if (block_data(op))  /* presumably block_data(np) is not null */
    memcpy((uint8_t *)(block_data(np)), (uint8_t *)(block_data(op)), block_size(op));
  liberate(sc, op);
  return(np);
}

/* we can't export mallocate et al without also exporting block_t or accessors for it
 *   that is, the block_t* pointer returned can't be used as if it were the void* pointer returned by malloc
 */


/* -------------------------------------------------------------------------------- */

typedef enum {P_DISPLAY, P_WRITE, P_READABLE, P_KEY, P_CODE} use_write_t;

static s7_pointer too_many_arguments_string, not_enough_arguments_string, cant_bind_immutable_string,
  a_boolean_string, a_byte_vector_string, a_format_port_string, a_let_string, a_list_string, a_non_constant_symbol_string,
  a_non_negative_integer_string, a_normal_procedure_string, a_normal_real_string, a_number_string, a_procedure_string, a_procedure_or_a_macro_string,
  a_proper_list_string, a_random_state_object_string, a_rational_string, a_sequence_string, a_symbol_string, a_thunk_string,
  a_valid_radix_string, an_association_list_string, an_eq_func_string, an_input_file_port_string, an_input_port_string,
  an_input_string_port_string, an_open_port_string, an_output_file_port_string, an_output_port_string, an_output_string_port_string,
  an_unsigned_byte_string, caaar_a_list_string, caadr_a_list_string, caar_a_list_string, cadar_a_list_string, caddr_a_list_string,
  cadr_a_list_string, car_a_list_string, cdaar_a_list_string, cdadr_a_list_string, cdar_a_list_string, cddar_a_list_string,
  cdddr_a_list_string, cddr_a_list_string, cdr_a_list_string, immutable_error_string, its_infinite_string, its_nan_string,
  its_negative_string, its_too_large_string, its_too_small_string, parameter_set_twice_string, result_is_too_large_string,
  something_applicable_string, too_many_indices_string, intermediate_too_large_string,
  format_string_1, format_string_2, format_string_3, format_string_4, keyword_value_missing_string, a_named_function_string;

static bool t_number_p[NUM_TYPES], t_small_real_p[NUM_TYPES], t_rational_p[NUM_TYPES], t_real_p[NUM_TYPES], t_big_number_p[NUM_TYPES];
static bool t_simple_p[NUM_TYPES], t_structure_p[NUM_TYPES];
static bool t_any_macro_p[NUM_TYPES], t_any_closure_p[NUM_TYPES], t_has_closure_let[NUM_TYPES];
static bool t_mappable_p[NUM_TYPES], t_sequence_p[NUM_TYPES], t_vector_p[NUM_TYPES];
static bool t_procedure_p[NUM_TYPES], t_applicable_p[NUM_TYPES];
#if S7_DEBUGGING
static bool t_freeze_p[NUM_TYPES]; /* free_cell sanity check */
#endif

static void init_types(void)
{
  for (int32_t i = 0; i < NUM_TYPES; i++)
    {
      t_number_p[i] = false;
      t_small_real_p[i] = false;
      t_real_p[i] = false;
      t_rational_p[i] = false;
      t_simple_p[i] = false;
      t_structure_p[i] = false;
      t_any_macro_p[i] = false;
      t_any_closure_p[i] = false;
      t_has_closure_let[i] = false;
      t_sequence_p[i] = false;
      t_mappable_p[i] = false;
      t_vector_p[i] = false;
      t_applicable_p[i] = false;
      t_procedure_p[i] = false;
#if S7_DEBUGGING
      t_freeze_p[i] = false;
#endif
    }
  t_number_p[T_INTEGER] = true;
  t_number_p[T_RATIO] = true;
  t_number_p[T_REAL] = true;
  t_number_p[T_COMPLEX] = true;
  t_number_p[T_BIG_INTEGER] = true;
  t_number_p[T_BIG_RATIO] = true;
  t_number_p[T_BIG_REAL] = true;
  t_number_p[T_BIG_COMPLEX] = true;

  t_rational_p[T_INTEGER] = true;
  t_rational_p[T_RATIO] = true;
  t_rational_p[T_BIG_INTEGER] = true;
  t_rational_p[T_BIG_RATIO] = true;

  t_small_real_p[T_INTEGER] = true;
  t_small_real_p[T_RATIO] = true;
  t_small_real_p[T_REAL] = true;

  t_real_p[T_INTEGER] = true;
  t_real_p[T_RATIO] = true;
  t_real_p[T_REAL] = true;
  t_real_p[T_BIG_INTEGER] = true;
  t_real_p[T_BIG_RATIO] = true;
  t_real_p[T_BIG_REAL] = true;

  t_big_number_p[T_BIG_INTEGER] = true;
  t_big_number_p[T_BIG_RATIO] = true;
  t_big_number_p[T_BIG_REAL] = true;
  t_big_number_p[T_BIG_COMPLEX] = true;

  t_structure_p[T_PAIR] = true;
  t_structure_p[T_VECTOR] = true;
  t_structure_p[T_HASH_TABLE] = true;
  t_structure_p[T_SLOT] = true;
  t_structure_p[T_LET] = true;
  t_structure_p[T_ITERATOR] = true;
  t_structure_p[T_C_POINTER] = true;
  t_structure_p[T_C_OBJECT] = true;

  t_sequence_p[T_NIL] = true;
  t_sequence_p[T_PAIR] = true;
  t_sequence_p[T_STRING] = true;
  t_sequence_p[T_VECTOR] = true;
  t_sequence_p[T_INT_VECTOR] = true;
  t_sequence_p[T_FLOAT_VECTOR] = true;
  t_sequence_p[T_BYTE_VECTOR] = true;
  t_sequence_p[T_HASH_TABLE] = true;
  t_sequence_p[T_LET] = true;
  t_sequence_p[T_C_OBJECT] = true; /* this assumes the object has a length method? */

  t_mappable_p[T_PAIR] = true;
  t_mappable_p[T_STRING] = true;
  t_mappable_p[T_VECTOR] = true;
  t_mappable_p[T_INT_VECTOR] = true;
  t_mappable_p[T_FLOAT_VECTOR] = true;
  t_mappable_p[T_BYTE_VECTOR] = true;
  t_mappable_p[T_HASH_TABLE] = true;
  t_mappable_p[T_LET] = true;
  t_mappable_p[T_C_OBJECT] = true;
  t_mappable_p[T_ITERATOR] = true;
  t_mappable_p[T_C_MACRO] = true;
  t_mappable_p[T_MACRO] = true;
  t_mappable_p[T_BACRO] = true;
  t_mappable_p[T_MACRO_STAR] = true;
  t_mappable_p[T_BACRO_STAR] = true;
  t_mappable_p[T_CLOSURE] = true;
  t_mappable_p[T_CLOSURE_STAR] = true;

  t_vector_p[T_VECTOR] = true;
  t_vector_p[T_INT_VECTOR] = true;
  t_vector_p[T_FLOAT_VECTOR] = true;
  t_vector_p[T_BYTE_VECTOR] = true;

  t_applicable_p[T_PAIR] = true;
  t_applicable_p[T_STRING] = true;
  t_applicable_p[T_VECTOR] = true;
  t_applicable_p[T_INT_VECTOR] = true;
  t_applicable_p[T_FLOAT_VECTOR] = true;
  t_applicable_p[T_BYTE_VECTOR] = true;
  t_applicable_p[T_HASH_TABLE] = true;
  t_applicable_p[T_ITERATOR] = true;
  t_applicable_p[T_LET] = true;
  t_applicable_p[T_C_OBJECT] = true;
  t_applicable_p[T_C_MACRO] = true;
  t_applicable_p[T_MACRO] = true;
  t_applicable_p[T_BACRO] = true;
  t_applicable_p[T_MACRO_STAR] = true;
  t_applicable_p[T_BACRO_STAR] = true;
  t_applicable_p[T_SYNTAX] = true;
  t_applicable_p[T_C_FUNCTION] = true;
  t_applicable_p[T_C_FUNCTION_STAR] = true;
  t_applicable_p[T_C_RST_NO_REQ_FUNCTION] = true;
  t_applicable_p[T_CLOSURE] = true;
  t_applicable_p[T_CLOSURE_STAR] = true;
  t_applicable_p[T_GOTO] = true;
  t_applicable_p[T_CONTINUATION] = true;

  /* t_procedure_p[T_C_OBJECT] = true; */
  t_procedure_p[T_C_FUNCTION] = true;
  t_procedure_p[T_C_FUNCTION_STAR] = true;
  t_procedure_p[T_C_RST_NO_REQ_FUNCTION] = true;
  t_procedure_p[T_CLOSURE] = true;
  t_procedure_p[T_CLOSURE_STAR] = true;
  t_procedure_p[T_GOTO] = true;
  t_procedure_p[T_CONTINUATION] = true;

  t_any_macro_p[T_C_MACRO] = true;
  t_any_macro_p[T_MACRO] = true;
  t_any_macro_p[T_BACRO] = true;
  t_any_macro_p[T_MACRO_STAR] = true;
  t_any_macro_p[T_BACRO_STAR] = true;

  t_any_closure_p[T_CLOSURE] = true;
  t_any_closure_p[T_CLOSURE_STAR] = true;

  t_has_closure_let[T_MACRO] = true;
  t_has_closure_let[T_BACRO] = true;
  t_has_closure_let[T_MACRO_STAR] = true;
  t_has_closure_let[T_BACRO_STAR] = true;
  t_has_closure_let[T_CLOSURE] = true;
  t_has_closure_let[T_CLOSURE_STAR] = true;

  t_simple_p[T_NIL] = true;
  /* t_simple_p[T_UNDEFINED] = true; */ /* only #<undefined> itself will work with eq? */
  t_simple_p[T_EOF] = true;
  t_simple_p[T_BOOLEAN] = true;
  t_simple_p[T_CHARACTER] = true;
  t_simple_p[T_SYMBOL] = true;
  t_simple_p[T_SYNTAX] = true;
  t_simple_p[T_C_MACRO] = true;
  t_simple_p[T_C_FUNCTION] = true;
  t_simple_p[T_C_FUNCTION_STAR] = true;
  t_simple_p[T_C_RST_NO_REQ_FUNCTION] = true;
  /* not completely sure about the next ones */
  t_simple_p[T_LET] = true;
  t_simple_p[T_INPUT_PORT] = true;
  t_simple_p[T_OUTPUT_PORT] = true;

#if S7_DEBUGGING
  t_freeze_p[T_STRING] = true;
  t_freeze_p[T_BYTE_VECTOR] = true;
  t_freeze_p[T_VECTOR] = true;
  t_freeze_p[T_FLOAT_VECTOR] = true;
  t_freeze_p[T_INT_VECTOR] = true;
  t_freeze_p[T_UNDEFINED] = true;
  t_freeze_p[T_C_OBJECT] = true;
  t_freeze_p[T_HASH_TABLE] = true;
  t_freeze_p[T_C_FUNCTION] = true;
  t_freeze_p[T_CONTINUATION] = true;
  t_freeze_p[T_INPUT_PORT] = true;
  t_freeze_p[T_OUTPUT_PORT] = true;
#if WITH_GMP
  t_freeze_p[T_BIG_INTEGER] = true;
  t_freeze_p[T_BIG_RATIO] = true;
  t_freeze_p[T_BIG_REAL] = true;
  t_freeze_p[T_BIG_COMPLEX] = true;
  t_freeze_p[T_RANDOM_STATE] = true;
#endif
#endif
}

#if WITH_HISTORY
#define current_code(Sc)               car(Sc->cur_code)
#define set_current_code(Sc, Code)     do {Sc->cur_code = cdr(Sc->cur_code); set_car(Sc->cur_code, T_Pos(Code));} while (0)
#define replace_current_code(Sc, Code) set_car(Sc->cur_code, T_Pos(Code))
#define mark_current_code(Sc)          do {int32_t i; s7_pointer p; for (p = Sc->cur_code, i = 0; i < Sc->history_size; i++, p = cdr(p)) gc_mark(car(p));} while (0)
#else
#define current_code(Sc)               Sc->cur_code
#define set_current_code(Sc, Code)     Sc->cur_code = T_Pos(Code)
#define replace_current_code(Sc, Code) Sc->cur_code = T_Pos(Code)
#define mark_current_code(Sc)          gc_mark(Sc->cur_code)
#endif

#define full_type(p)  ((p)->tf.flag)
#define typesflag(p) ((p)->tf.sflag)
#define TYPE_MASK    0xff

#if S7_DEBUGGING
  static void print_gc_info(s7_scheme *sc, s7_pointer obj, int32_t line);
  static s7_pointer check_ref(s7_pointer p, uint8_t expected_type, const char *func, int32_t line, const char *func1, const char *func2);
  static s7_pointer check_ref7(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref11(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref16(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_nref(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_let_ref(s7_pointer p, uint64_t role, const char *func, int32_t line);

  #define unchecked_type(p) ((p)->tf.type_field)
#if WITH_GCC
  #define type(p) ({uint8_t _t_; _t_ = (p)->tf.type_field; if ((!cur_sc->printing_gc_info) && ((_t_ == T_FREE) || (_t_ >= NUM_TYPES))) print_gc_info(cur_sc, p, __LINE__); _t_;})
#else
  #define type(p) (p)->tf.type_field
#endif
  #define set_full_type(p, f) set_type_1(p, f, __func__, __LINE__)
  /* these check most s7_cell field references (and many type bits) for consistency */
  #define T_Any(P) check_cell(sc, P,                 __func__, __LINE__)                /* any cell */
  #define T_App(P) check_ref11(P,                    __func__, __LINE__)                /* applicable or #f */
  #define T_Arg(P) check_ref10(P,                    __func__, __LINE__)                /* closure arg (list, symbol) */
  #define T_BVc(P) check_ref(P, T_BYTE_VECTOR,       __func__, __LINE__, "sweep", NULL)
  #define T_Bgf(P) check_ref(P, T_BIG_RATIO,         __func__, __LINE__, "sweep", NULL)
  #define T_Bgi(P) check_ref(P, T_BIG_INTEGER,       __func__, __LINE__, "sweep", NULL)
  #define T_Bgr(P) check_ref(P, T_BIG_REAL,          __func__, __LINE__, "sweep", NULL)
  #define T_Bgz(P) check_ref(P, T_BIG_COMPLEX,       __func__, __LINE__, "sweep", NULL)
  #define T_CMac(P) check_ref(P, T_C_MACRO,          __func__, __LINE__, NULL, NULL)
  #define T_Cat(P) check_ref(P, T_CATCH,             __func__, __LINE__, NULL, NULL)
  #define T_Chr(P) check_ref(P, T_CHARACTER,         __func__, __LINE__, NULL, NULL)
  #define T_Clo(P) check_ref5(P,                     __func__, __LINE__)                /* has closure let */
  #define T_Cmp(P) check_ref(P, T_COMPLEX,           __func__, __LINE__, NULL, NULL)
  #define T_Con(P) check_ref(P, T_CONTINUATION,      __func__, __LINE__, "sweep", "process_continuation")
  #define T_Ctr(P) check_ref(P, T_COUNTER,           __func__, __LINE__, NULL, NULL)
  #define T_Dyn(P) check_ref(P, T_DYNAMIC_WIND,      __func__, __LINE__, NULL, NULL)
  #define T_Eof(P) check_ref(P, T_EOF,               __func__, __LINE__, "sweep", NULL)
  #define T_Fnc(P) check_ref6(P,                     __func__, __LINE__)                /* any c_function|c_macro */
  #define T_Frc(P) check_ref2(P, T_RATIO, T_INTEGER, __func__, __LINE__, NULL, NULL)
  #define T_Fst(P) check_ref(P, T_C_FUNCTION_STAR,   __func__, __LINE__, NULL, NULL)
  #define T_Fvc(P) check_ref(P, T_FLOAT_VECTOR,      __func__, __LINE__, "sweep", NULL)
  #define T_Got(P) check_ref(P, T_GOTO,              __func__, __LINE__, NULL, NULL)
  #define T_Hsh(P) check_ref(P, T_HASH_TABLE,        __func__, __LINE__, "sweep", "free_hash_table")
  #define T_Int(P) check_ref(P, T_INTEGER,           __func__, __LINE__, NULL, NULL)
  #define T_Itr(P) check_ref(P, T_ITERATOR,          __func__, __LINE__, "sweep", "process_iterator")
  #define T_Ivc(P) check_ref(P, T_INT_VECTOR,        __func__, __LINE__, "sweep", NULL)
  #define T_Let(P) check_ref(P, T_LET,               __func__, __LINE__, NULL, NULL)
  #define T_Lid(P) check_ref16(P,                    __func__, __LINE__)                /* let/nil */
  #define T_Lst(P) check_ref2(P, T_PAIR, T_NIL,      __func__, __LINE__, "gc", NULL)
  #define T_Mac(P) check_ref17(P,                    __func__, __LINE__)                /* a non-C macro */
  #define T_Met(P) check_ref9(P,                     __func__, __LINE__)                /* anything that might contain a method */
  #define T_Nmv(P) check_ref15(P,                    __func__, __LINE__)                /* not multiple-value, not free */
  #define T_Num(P) check_ref7(P,                     __func__, __LINE__)                /* any number (not bignums) */
  #define T_Nvc(P) check_ref(P, T_VECTOR,            __func__, __LINE__, "sweep", NULL)
  #define T_Obj(P) check_ref(P, T_C_OBJECT,          __func__, __LINE__, "sweep", "s7_c_object_value")
  #define T_Pair(P) check_ref(P, T_PAIR,             __func__, __LINE__, NULL, NULL)
  #define T_Pcs(P) check_ref2(P, T_PAIR, T_CLOSURE_STAR, __func__, __LINE__, NULL, NULL)
  #define T_Pos(P) check_nref(P,                     __func__, __LINE__)                /* not free */
  #define T_Prc(P) check_ref14(P,                    __func__, __LINE__)                /* any procedure (3-arg setters) or #f */
  #define T_Prt(P) check_ref3(P,                     __func__, __LINE__)                /* input|output_port */
  #define T_Ptr(P) check_ref(P, T_C_POINTER,         __func__, __LINE__, NULL, NULL)
  #define T_Ran(P) check_ref(P, T_RANDOM_STATE,      __func__, __LINE__, NULL, NULL)
  #define T_Rel(P) check_ref(P, T_REAL,              __func__, __LINE__, NULL, NULL)
  #define T_SVec(P) check_ref13(P,                   __func__, __LINE__)                /* subvector */
  #define T_Seq(P) check_ref8(P,                     __func__, __LINE__)                /* any sequence or structure */
  #define T_Sld(P) check_ref2(P, T_SLOT, T_UNDEFINED,__func__, __LINE__, NULL, NULL)
  #define T_Sln(P) check_ref12(P,                    __func__, __LINE__)                /* slot or nil */
  #define T_Slt(P) check_ref(P, T_SLOT,              __func__, __LINE__, NULL, NULL)
  #define T_Stk(P) check_ref(P, T_STACK,             __func__, __LINE__, NULL, NULL)
  #define T_Str(P) check_ref(P, T_STRING,            __func__, __LINE__, "sweep", NULL)
  #define T_Sym(P) check_ref(P, T_SYMBOL,            __func__, __LINE__, "sweep", "remove_gensym_from_symbol_table")
  #define T_Syn(P) check_ref(P, T_SYNTAX,            __func__, __LINE__, NULL, NULL)
  #define T_Undf(P) check_ref(P, T_UNDEFINED,        __func__, __LINE__, "sweep", NULL)
  #define T_Vec(P) check_ref4(P,                     __func__, __LINE__)                /* any vector */
#else
  /* if not debugging, all those checks go away */
  #define T_Any(P)  P
  #define T_App(P)  P
  #define T_Arg(P)  P
  #define T_BVc(P)  P
  #define T_Bgf(P)  P
  #define T_Bgi(P)  P
  #define T_Bgr(P)  P
  #define T_Bgz(P)  P
  #define T_CMac(P) P
  #define T_Cat(P)  P
  #define T_Chr(P)  P
  #define T_Clo(P)  P
  #define T_Cmp(P)  P
  #define T_Con(P)  P
  #define T_Ctr(P)  P
  #define T_Dyn(P)  P
  #define T_Eof(P)  P
  #define T_Fnc(P)  P
  #define T_Frc(P)  P
  #define T_Fst(P)  P
  #define T_Fvc(P)  P
  #define T_Got(P)  P
  #define T_Hsh(P)  P
  #define T_Int(P)  P
  #define T_Itr(P)  P
  #define T_Ivc(P)  P
  #define T_Let(P)  P
  #define T_Lid(P)  P
  #define T_Lst(P)  P
  #define T_Mac(P)  P
  #define T_Met(P)  P
  #define T_Nmv(P)  P
  #define T_Num(P)  P
  #define T_Nvc(P)  P
  #define T_Obj(P)  P
  #define T_Pair(P) P
  #define T_Pcs(P)  P
  #define T_Pos(P)  P
  #define T_Prc(P)  P
  #define T_Prt(P)  P
  #define T_Ptr(P)  P
  #define T_Ran(P)  P
  #define T_Rel(P)  P
  #define T_SVec(P) P
  #define T_Seq(P)  P
  #define T_Sld(P)  P
  #define T_Sln(P)  P
  #define T_Slt(P)  P
  #define T_Stk(P)  P
  #define T_Str(P)  P
  #define T_Sym(P)  P
  #define T_Syn(P)  P
  #define T_Undf(P) P
  #define T_Vec(P)  P

  #define unchecked_type(p)            ((p)->tf.type_field)
  #define type(p)                      ((p)->tf.type_field)
  #define set_full_type(p, f)          full_type(p) = f
#endif
#define signed_type(p)                 (p)->tf.signed_flag

#define is_number(P)                   t_number_p[type(P)]
#define is_small_real(P)               t_small_real_p[type(P)]
#define is_real(P)                     t_real_p[type(P)]
#define is_rational(P)                 t_rational_p[type(P)]
#define is_big_number(p)               t_big_number_p[type(p)]
#define is_t_integer(p)                (type(p) == T_INTEGER)
#define is_t_ratio(p)                  (type(p) == T_RATIO)
#define is_t_real(p)                   (type(p) == T_REAL)
#define is_t_complex(p)                (type(p) == T_COMPLEX)
#define is_t_big_integer(p)            (type(p) == T_BIG_INTEGER)
#define is_t_big_ratio(p)              (type(p) == T_BIG_RATIO)
#define is_t_big_real(p)               (type(p) == T_BIG_REAL)
#define is_t_big_complex(p)            (type(p) == T_BIG_COMPLEX)

#define is_free(p)                     (type(p) == T_FREE)
#define is_free_and_clear(p)           (full_type(p) == T_FREE)
#define is_simple(P)                   t_simple_p[type(P)]  /* eq? */
#define has_structure(P)               ((t_structure_p[type(P)]) && ((!is_normal_vector(P)) || (!has_simple_elements(P))))

#define is_any_macro(P)                t_any_macro_p[type(P)]
#define is_any_closure(P)              t_any_closure_p[type(P)]
#define is_any_procedure(P)            (type(P) >= T_CLOSURE)
#define has_closure_let(P)             t_has_closure_let[type(P)]

#define is_simple_sequence(P)          (t_sequence_p[type(P)])
#define is_sequence(P)                 ((t_sequence_p[type(P)]) || (has_methods(P)))
#define is_mutable_sequence(P)         (((t_sequence_p[type(P)]) || (has_methods(P))) && (!is_immutable(P)))
#define is_mappable(P)                 (t_mappable_p[type(P)])
#define is_applicable(P)               (t_applicable_p[type(P)])
/* this misses #() which is not applicable to anything, and "", and inapplicable c-objects like random-state */
#define is_procedure(p)                ((t_procedure_p[type(p)]) || ((is_c_object(p)) && (is_safe_procedure(p))))
#define is_t_procedure(p)              (t_procedure_p[type(p)])

/* the layout of these bits does matter in several cases -- don't shadow SYNTACTIC_PAIR and OPTIMIZED_PAIR */
#define TYPE_BITS                      8

#define set_type_bit(p, b)             full_type(p) |= (b)
#define clear_type_bit(p, b)           full_type(p) &= (~(b))
#define has_type_bit(p, b)             ((full_type(p) & (b)) != 0)

#define set_type0_bit(p, b)            typesflag(p) |= (b)      /* I don't think these type0's matter -- *_type_bit is the same speed */
#define clear_type0_bit(p, b)          typesflag(p) &= (~(b))
#define has_type0_bit(p, b)            ((typesflag(p) & (b)) != 0)

#define set_type1_bit(p, b)            (p)->tf.opts.high_flag |= (b)
#define clear_type1_bit(p, b)          (p)->tf.opts.high_flag &= (~(b))
#define has_type1_bit(p, b)            (((p)->tf.opts.high_flag & (b)) != 0)

#define T_SYNTACTIC                    (1 << (TYPE_BITS + 1))
#define is_symbol_and_syntactic(p)     (typesflag(T_Pos(p)) == (uint16_t)(T_SYMBOL | T_SYNTACTIC))
#define is_syntactic_symbol(p)         has_type0_bit(T_Sym(p), T_SYNTACTIC)
#define is_syntactic_pair(p)           has_type0_bit(T_Pair(p), T_SYNTACTIC)
#define clear_syntactic(p)             clear_type0_bit(T_Pair(p), T_SYNTACTIC)
#define set_syntactic_pair(p)          full_type(T_Pair(p)) = (T_PAIR | T_SYNTACTIC | (full_type(p) & (0xffffffffffff0000 & ~T_OPTIMIZED))) /* used only in pair_set_syntax_op */
/* this marks symbols that represent syntax objects, it should be in the second byte */

#define T_SIMPLE_ARG_DEFAULTS          (1 << (TYPE_BITS + 2))
#define lambda_has_simple_defaults(p)  has_type0_bit(T_Pair(closure_body(p)), T_SIMPLE_ARG_DEFAULTS)
#define lambda_set_simple_defaults(p)  set_type0_bit(T_Pair(p), T_SIMPLE_ARG_DEFAULTS)
/* are all lambda* default values simple? This is set on closure_body, so it doesn't mess up closure_is_ok_1 */

#define T_LIST_IN_USE                  T_SIMPLE_ARG_DEFAULTS
#define list_is_in_use(p)              has_type0_bit(T_Pair(p), T_LIST_IN_USE)
#define set_list_in_use(p)             set_type0_bit(T_Pair(p), T_LIST_IN_USE)
#define clear_list_in_use(p)           do {clear_type0_bit(T_Pair(p), T_LIST_IN_USE); sc->current_safe_list = 0;} while (0)
/* since the safe lists are not in the heap, if the list_in_use bit is off, the list won't be GC-protected even if
 *   it is gc_marked explicitly.  This happens, for example, in copy_proper_list where we try to protect the original list
 *   by sc->temp5 = lst; then in the GC, gc_mark(sc->temp5); but the safe_list probably is already marked, so its contents are not protected.
 */

#define T_ONE_FORM                     T_SIMPLE_ARG_DEFAULTS
#define set_closure_has_one_form(p)    set_type0_bit(T_Clo(p), T_ONE_FORM)
#define T_MULTIFORM                    (1 << (TYPE_BITS + 0))
#define set_closure_has_multiform(p)   set_type0_bit(T_Clo(p), T_MULTIFORM)
#define T_ONE_FORM_FX_ARG              (T_ONE_FORM | T_MULTIFORM)
#define set_closure_one_form_fx_arg(p) set_type0_bit(T_Clo(p), T_ONE_FORM_FX_ARG)
/* can't use T_HAS_FX here because closure_is_ok wants to examine typesflag */

#define T_OPTIMIZED                    (1 << (TYPE_BITS + 3))
#define set_optimized(p)               set_type0_bit(T_Pair(p), T_OPTIMIZED)
#define clear_optimized(p)             clear_type0_bit(T_Pair(p), T_OPTIMIZED | T_SYNTACTIC | T_HAS_FX | T_HAS_FN)
#define is_optimized(p)                (typesflag(T_Pos(p)) == (uint16_t)(T_PAIR | T_OPTIMIZED))
/* optimizer flag for an expression that has optimization info, it should be in the second byte */

#define T_SCOPE_SAFE                   T_OPTIMIZED
#define is_scope_safe(p)               has_type0_bit(T_Fnc(p), T_SCOPE_SAFE)
#define set_scope_safe(p)              set_type0_bit(T_Fnc(p), T_SCOPE_SAFE)

#define T_SAFE_CLOSURE                 (1 << (TYPE_BITS + 4))
#define is_safe_closure(p)             has_type0_bit(T_Clo(p), T_SAFE_CLOSURE)
#define set_safe_closure(p)            set_type0_bit(T_Clo(p), T_SAFE_CLOSURE)
#define is_safe_closure_body(p)        has_type0_bit(T_Pair(p), T_SAFE_CLOSURE)
#define set_safe_closure_body(p)       set_type0_bit(T_Pair(p), T_SAFE_CLOSURE)
#define clear_safe_closure_body(p)     clear_type0_bit(T_Pair(p), T_SAFE_CLOSURE)

/* optimizer flag for a closure body that is completely simple (every expression is safe)
 *   set_safe_closure happens in define_funchcecked letrec_setup_closures etc, clear only in procedure_source, bits only here
 *   this has to be separate from T_SAFE_PROCEDURE, and should be in the second byte (closure_is_ok_1 checks typesflag).
 * define -> optimize_lambda sets safe -> define_funchecked -> make_funclet for the let
 *   similarly, named let -> optimize_lambda, then let creates the let if safe
 *   thereafter, optimizer uses OP_SAFE_CLOSURE* which calls update_let*
 */

#define T_DONT_EVAL_ARGS               (1 << (TYPE_BITS + 5))
#define dont_eval_args(p)              has_type0_bit(T_Pos(p), T_DONT_EVAL_ARGS)
/* this marks things that don't evaluate their arguments */

#define T_EXPANSION                    (1 << (TYPE_BITS + 6))
#define is_expansion(p)                has_type0_bit(T_Any(p), T_EXPANSION)
#define clear_expansion(p)             clear_type0_bit(T_Sym(p), T_EXPANSION)
/* this marks the symbol and its run-time macro value, distinguishing it from an ordinary macro */

#define T_MULTIPLE_VALUE               (1 << (TYPE_BITS + 7))
#define is_multiple_value(p)           has_type0_bit(T_Pos(p), T_MULTIPLE_VALUE)
#if S7_DEBUGGING
#define set_multiple_value(p)          do {if (!in_heap(p)) {fprintf(stderr, "%s[%d]: mv\n", __func__, __LINE__); abort();} set_type0_bit(T_Pair(p), T_MULTIPLE_VALUE);} while (0)
#else
#define set_multiple_value(p)          set_type0_bit(T_Pair(p), T_MULTIPLE_VALUE)
#endif
#define clear_multiple_value(p)        clear_type0_bit(T_Pair(p), T_MULTIPLE_VALUE)
#define multiple_value(p)              p
/* this bit marks a list (from "values") that is waiting for a chance to be spliced into its caller's argument list */

#define T_MATCHED                      T_MULTIPLE_VALUE
#define is_matched_pair(p)             has_type0_bit(T_Pair(p), T_MATCHED)
#define clear_match_pair(p)            clear_type0_bit(T_Pair(p), T_MATCHED)
#define set_match_pair(p)              set_type0_bit(T_Pair(p), T_MATCHED)
#define set_match_symbol(p)            set_type0_bit(T_Sym(p), T_MATCHED)
#define is_matched_symbol(p)           has_type0_bit(T_Sym(p), T_MATCHED)
#define clear_match_symbol(p)          clear_type0_bit(T_Sym(p), T_MATCHED)

#define T_GLOBAL                       (1 << (TYPE_BITS + 8))
#define T_LOCAL                        (1 << (TYPE_BITS + 12))
#define is_global(p)                   has_type_bit(T_Sym(p), T_GLOBAL)
#define set_global(p)                  do {if ((full_type(T_Sym(p)) & T_LOCAL) == 0) full_type(p) |= T_GLOBAL;} while (0)
/* T_LOCAL marks a symbol that has been used locally */
/* T_GLOBAL marks something defined (bound) at the top-level, and never defined locally */

#define REPORT_ROOTLET_REDEF 0
#if REPORT_ROOTLET_REDEF
  /* to find who is stomping on our symbols: */
  static void set_local_1(s7_scheme *sc, s7_pointer symbol, const char *func, int32_t line);
  #define set_local(Symbol) set_local_1(sc, Symbol, __func__, __LINE__)
#else
  #define set_local(p)                 full_type(T_Sym(p)) = ((full_type(p) | T_LOCAL) & ~(T_DONT_EVAL_ARGS | T_GLOBAL | T_SYNTACTIC))
#endif

#define T_LOW_COUNT                    T_LOCAL
#define has_low_count(p)               has_type_bit(T_Pair(p), T_LOW_COUNT)
#define set_has_low_count(p)           set_type_bit(T_Pair(p), T_LOW_COUNT)

#define T_TC                           T_LOCAL
#define has_tc(p)                      has_type_bit(T_Pair(p), T_TC)
#define set_has_tc(p)                  set_type_bit(T_Pair(p), T_TC)

#define T_UNSAFE_DO                    T_GLOBAL
#define is_unsafe_do(p)                has_type_bit(T_Pair(p), T_UNSAFE_DO)
#define set_unsafe_do(p)               set_type_bit(T_Pair(p), T_UNSAFE_DO)
/* marks do-loops that resist optimization */

#define T_DOX_SLOT1                    T_GLOBAL
#define has_dox_slot1(p)               has_type_bit(T_Let(p), T_DOX_SLOT1)
#define set_has_dox_slot1(p)           set_type_bit(T_Let(p), T_DOX_SLOT1)
/* marks a let that includes the dox_slot1 */

#define T_COLLECTED                    (1 << (TYPE_BITS + 9))
#define is_collected(p)                has_type_bit(T_Seq(p), T_COLLECTED)
#define is_collected_unchecked(p)      has_type_bit(p, T_COLLECTED)
#define set_collected(p)               set_type_bit(T_Seq(p), T_COLLECTED)
/* #define clear_collected(p)          clear_type_bit(T_Seq(p), T_COLLECTED) */
/* this is a transient flag used by the printer to catch cycles.  It affects only objects that have structure.
 *   We can't use a low bit (bit 7 for example), because collect_shared_info inspects the object's type.
 */

#define T_LOCATION                     (1 << (TYPE_BITS + 10))
#define has_location(p)                has_type_bit(T_Pair(p), T_LOCATION)
#define set_has_location(p)            set_type_bit(T_Pair(p), T_LOCATION)
/* pair in question has line/file/position info added during read, or the environment has function placement info
 *   this bit should not be in the first byte -- SYNTACTIC_PAIR ignores it.
 */

#define T_LOADER_PORT                  T_LOCATION
#define is_loader_port(p)              has_type_bit(T_Prt(p), T_LOADER_PORT)
#define set_loader_port(p)             set_type_bit(T_Prt(p), T_LOADER_PORT)
#define clear_loader_port(p)           clear_type_bit(T_Prt(p), T_LOADER_PORT)
/* this bit marks a port used by the loader so that random load-time reads do not screw up the load process */

#define T_HAS_SETTER                   T_LOCATION
#define slot_has_setter(p)             has_type_bit(T_Slt(p), T_HAS_SETTER)
#define slot_set_has_setter(p)         set_type_bit(T_Slt(p), T_HAS_SETTER)
/* marks a slot that has a setter or symbol that might have a setter */

#define T_WITH_LET_LET                 T_LOCATION
#define is_with_let_let(p)             has_type_bit(T_Let(p), T_WITH_LET_LET)
#define set_with_let_let(p)            set_type_bit(T_Let(p), T_WITH_LET_LET)
/* marks a let that is the argument to with-let */

#define T_SIMPLE_DEFAULTS              T_LOCATION
#define c_func_has_simple_defaults(p)  has_type_bit(T_Fst(p), T_SIMPLE_DEFAULTS)
#define c_func_set_simple_defaults(p)  set_type_bit(T_Fst(p), T_SIMPLE_DEFAULTS)
#define c_func_clear_simple_defaults(p) clear_type_bit(T_Fst(p), T_SIMPLE_DEFAULTS)
/* flag c_func_star arg defaults that need GC protection */

#define T_NO_SETTER                    T_LOCATION
#define closure_no_setter(p)           has_type_bit(T_Clo(p), T_NO_SETTER)
#define closure_set_no_setter(p)       set_type_bit(T_Clo(p), T_NO_SETTER)

#define T_SHARED                       (1 << (TYPE_BITS + 11))
#define is_shared(p)                   has_type_bit(T_Seq(p), T_SHARED)
#define set_shared(p)                  set_type_bit(T_Seq(p), T_SHARED)
#define is_collected_or_shared(p)      has_type_bit(p, T_COLLECTED | T_SHARED)
#define clear_collected_and_shared(p)  clear_type_bit(p, T_COLLECTED | T_SHARED) /* this can clear free cells = calloc */
/* T_LOCAL is bit 12 */

#define T_SAFE_PROCEDURE               (1 << (TYPE_BITS + 13))
#define is_safe_procedure(p)           has_type_bit(T_App(p), T_SAFE_PROCEDURE) /* was T_Pos 19-Apr-21 */
#define is_safe_or_scope_safe_procedure(p) ((full_type(T_Fnc(p)) & (T_SCOPE_SAFE | T_SAFE_PROCEDURE)) != 0)
/* applicable objects that do not return or modify their arg list directly (no :rest arg in particular),
 *    and that can't call themselves either directly or via s7_call, and that don't mess with the stack.
 */

#define T_CHECKED                      (1 << (TYPE_BITS + 14))
#define set_checked(p)                 set_type_bit(T_Pair(p), T_CHECKED)
#define is_checked(p)                  has_type_bit(T_Pair(p), T_CHECKED)
#define clear_checked(p)               clear_type_bit(T_Pair(p), T_CHECKED)
#define set_checked_slot(p)            set_type_bit(T_Slt(p), T_CHECKED)
#define is_checked_slot(p)             has_type_bit(T_Slt(p), T_CHECKED)
#define clear_checked_slot(p)          clear_type_bit(T_Slt(p), T_CHECKED)

#define T_ALL_INTEGER                  T_CHECKED
#define is_all_integer(p)              has_type_bit(T_Sym(p), T_ALL_INTEGER)
#define set_all_integer(p)             set_type_bit(T_Sym(p), T_ALL_INTEGER)

#define T_UNSAFE                       (1 << (TYPE_BITS + 15))
#define set_unsafe(p)                  set_type_bit(T_Pair(p), T_UNSAFE)
#define set_unsafely_optimized(p)      full_type(T_Pair(p)) = (full_type(p) | T_UNSAFE | T_OPTIMIZED)
#define is_unsafe(p)                   has_type_bit(T_Pair(p), T_UNSAFE)
#define clear_unsafe(p)                clear_type_bit(T_Pair(p), T_UNSAFE)
#define is_safely_optimized(p)         ((full_type(T_Pair(p)) & (T_OPTIMIZED | T_UNSAFE)) == T_OPTIMIZED) /* was T_Pos 30-Jan-21 */
/* optimizer flag saying "this expression is not completely self-contained.  It might involve the stack, etc" */

#define T_CLEAN_SYMBOL                 T_UNSAFE
#define is_clean_symbol(p)             has_type_bit(T_Sym(p), T_CLEAN_SYMBOL)
#define set_clean_symbol(p)            set_type_bit(T_Sym(p), T_CLEAN_SYMBOL)
/* set if we know the symbol name can be printed without quotes (slashification) */

#define T_HAS_STEPPER                  T_UNSAFE
#define has_stepper(p)                 has_type_bit(T_Slt(p), T_HAS_STEPPER)
#define set_has_stepper(p)             set_type_bit(T_Slt(p), T_HAS_STEPPER)

#define T_DOX_SLOT2                    T_UNSAFE
#define has_dox_slot2(p)               has_type_bit(T_Let(p), T_DOX_SLOT2)
#define set_has_dox_slot2(p)           set_type_bit(T_Let(p), T_DOX_SLOT2)
/* marks a let that includes the dox_slot2 */

#define T_IMMUTABLE                    (1 << (TYPE_BITS + 16))
#define is_immutable(p)                has_type_bit(T_Pos(p), T_IMMUTABLE)
#define set_immutable(p)               set_type_bit(T_Pos(p), T_IMMUTABLE)
#define set_immutable_let(p)           set_type_bit(T_Lid(p), T_IMMUTABLE)
#define is_immutable_port(p)           has_type_bit(T_Prt(p), T_IMMUTABLE)
#define is_immutable_symbol(p)         has_type_bit(T_Sym(p), T_IMMUTABLE)
#define is_immutable_slot(p)           has_type_bit(T_Slt(p), T_IMMUTABLE)
#define is_immutable_pair(p)           has_type_bit(T_Pair(p), T_IMMUTABLE)
#define is_immutable_vector(p)         has_type_bit(T_Vec(p), T_IMMUTABLE)
#define is_immutable_string(p)         has_type_bit(T_Str(p), T_IMMUTABLE)
/* T_IMMUTABLE is compatible with T_MUTABLE -- the latter is an internal bit for locally mutable numbers */

#define T_SETTER                       (1 << (TYPE_BITS + 17))
#define set_is_setter(p)               set_type_bit(T_Sym(p), T_SETTER)
#define is_setter(p)                   has_type_bit(T_Sym(p), T_SETTER)
/* optimizer flag for a procedure that sets some variable (set-car! for example) */

#define T_ALLOW_OTHER_KEYS             T_SETTER
#define set_allow_other_keys(p)        set_type_bit(T_Pair(p), T_ALLOW_OTHER_KEYS)
#define allows_other_keys(p)           has_type_bit(T_Pair(p), T_ALLOW_OTHER_KEYS)
#define c_function_set_allow_other_keys(p) set_type_bit(T_Fst(p), T_ALLOW_OTHER_KEYS)
#define c_function_allows_other_keys(p)    has_type_bit(T_Fst(p), T_ALLOW_OTHER_KEYS)
/* marks arglist (or c_function*) that allows keyword args other than those in the parameter list;
 *   we can't allow (define* (f :allow-other-keys)...) because there's only one nil, and besides, it does say "other".
 */

#define T_LET_REMOVED                  T_SETTER
#define let_set_removed(p)             set_type_bit(T_Let(p), T_LET_REMOVED)
#define let_removed(p)                 has_type_bit(T_Let(p), T_LET_REMOVED)
/* mark lets that have been removed from the heap or checked for that possibility */

#define T_HAS_EXPRESSION               T_SETTER
#define slot_set_has_expression(p)     set_type_bit(T_Slt(p), T_HAS_EXPRESSION)
#define slot_has_expression(p)         has_type_bit(T_Slt(p), T_HAS_EXPRESSION)

#define T_MUTABLE                      (1 << (TYPE_BITS + 18))
#define is_mutable_number(p)           has_type_bit(T_Num(p), T_MUTABLE)
#define is_mutable_integer(p)          has_type_bit(T_Int(p), T_MUTABLE)
#define clear_mutable_number(p)        clear_type_bit(T_Num(p), T_MUTABLE)
#define clear_mutable_integer(p)       clear_type_bit(T_Int(p), T_MUTABLE)
/* used for mutable numbers, can occur with T_IMMUTABLE (outside view vs inside) */

#define T_HAS_KEYWORD                  T_MUTABLE
#define has_keyword(p)                 has_type_bit(T_Sym(p), T_HAS_KEYWORD)
#define set_has_keyword(p)             set_type_bit(T_Sym(p), T_HAS_KEYWORD)

#define T_MARK_SEQ                     T_MUTABLE
#define is_mark_seq(p)                 has_type_bit(T_Itr(p), T_MARK_SEQ)
#define set_mark_seq(p)                set_type_bit(T_Itr(p), T_MARK_SEQ)
/* used in iterators for GC mark of sequence */

#define T_STEP_END                     T_MUTABLE
#define is_step_end(p)                 has_type_bit(T_Slt(p), T_STEP_END)
#define step_end_fits(Slot, Len)       ((is_step_end(Slot)) && (denominator(slot_value(Slot)) <= Len))
#define set_step_end(p)                set_type_bit(T_Slt(p), T_STEP_END)
/* marks a slot that holds a do-loop's step-or-end variable, numerator=current, denominator=end */

#define T_NO_CELL_OPT                  T_MUTABLE
#define set_no_cell_opt(p)             set_type_bit(T_Pair(p), T_NO_CELL_OPT)
#define no_cell_opt(p)                 has_type_bit(T_Pair(p), T_NO_CELL_OPT)

#define T_NO_INT_OPT                   T_SETTER
#define set_no_int_opt(p)              set_type_bit(T_Pair(p), T_NO_INT_OPT)
#define no_int_opt(p)                  has_type_bit(T_Pair(p), T_NO_INT_OPT)

#define T_NO_FLOAT_OPT                 T_UNSAFE
#define set_no_float_opt(p)            set_type_bit(T_Pair(p), T_NO_FLOAT_OPT)
#define no_float_opt(p)                has_type_bit(T_Pair(p), T_NO_FLOAT_OPT)

#define T_NO_BOOL_OPT                  T_SAFE_STEPPER
#define set_no_bool_opt(p)             set_type_bit(T_Pair(p), T_NO_BOOL_OPT)
#define no_bool_opt(p)                 has_type_bit(T_Pair(p), T_NO_BOOL_OPT)

#define T_INTEGER_KEYS                 T_SETTER
#define set_has_integer_keys(p)        set_type_bit(T_Pair(p), T_INTEGER_KEYS)
#define has_integer_keys(p)            has_type_bit(T_Pair(p), T_INTEGER_KEYS)

#define T_SAFE_STEPPER                 (1 << (TYPE_BITS + 19))
#define is_safe_stepper(p)             has_type_bit(T_Slt(p), T_SAFE_STEPPER)
#define set_safe_stepper(p)            set_type_bit(T_Slt(p), T_SAFE_STEPPER)
#define clear_safe_stepper(p)          clear_type_bit(T_Slt(p), T_SAFE_STEPPER)
#define is_safe_stepper_expr(p)        has_type_bit(T_Pair(p), T_SAFE_STEPPER)
#define set_safe_stepper_expr(p)       set_type_bit(T_Pair(p), T_SAFE_STEPPER)

#define T_NUMBER_NAME                  T_SAFE_STEPPER
#define has_number_name(p)             has_type_bit(T_Num(p), T_NUMBER_NAME)
#define set_has_number_name(p)         set_type_bit(T_Num(p), T_NUMBER_NAME)
/* marks numbers that have a saved version of their string representation; this only matters in teq.scm, maybe tread.scm */

#define T_MAYBE_SAFE                   T_SAFE_STEPPER
#define is_maybe_safe(p)               has_type_bit(T_Fnc(p), T_MAYBE_SAFE)
#define set_maybe_safe(p)              set_type_bit(T_Fnc(p), T_MAYBE_SAFE)

#define T_PAIR_MACRO                   T_SAFE_STEPPER
#define has_pair_macro(p)              has_type_bit(T_Mac(p), T_PAIR_MACRO)
#define set_has_pair_macro(p)          set_type_bit(T_Mac(p), T_PAIR_MACRO)

#define T_HAS_LET_SET_FALLBACK         T_SAFE_STEPPER
#define T_HAS_LET_REF_FALLBACK         T_MUTABLE
#define has_let_ref_fallback(p)        ((full_type(T_Lid(p)) & (T_HAS_LET_REF_FALLBACK | T_HAS_METHODS)) == (T_HAS_LET_REF_FALLBACK | T_HAS_METHODS))
#define has_let_set_fallback(p)        ((full_type(T_Lid(p)) & (T_HAS_LET_SET_FALLBACK | T_HAS_METHODS)) == (T_HAS_LET_SET_FALLBACK | T_HAS_METHODS))
#define set_has_let_ref_fallback(p)    set_type_bit(T_Let(p), T_HAS_LET_REF_FALLBACK)
#define set_has_let_set_fallback(p)    set_type_bit(T_Let(p), T_HAS_LET_SET_FALLBACK)
#define has_let_fallback(p)            has_type_bit(T_Lid(p), (T_HAS_LET_REF_FALLBACK | T_HAS_LET_SET_FALLBACK))
#define set_all_methods(p, e)          full_type(T_Let(p)) |= (full_type(e) & (T_HAS_METHODS | T_HAS_LET_REF_FALLBACK | T_HAS_LET_SET_FALLBACK))

#define T_WEAK_HASH                    T_SAFE_STEPPER
#define set_weak_hash_table(p)         set_type_bit(T_Hsh(p), T_WEAK_HASH)
#define is_weak_hash_table(p)          has_type_bit(T_Hsh(p), T_WEAK_HASH)

#define T_ALL_FLOAT                    T_SAFE_STEPPER
#define is_all_float(p)                has_type_bit(T_Sym(p), T_ALL_FLOAT)
#define set_all_float(p)               set_type_bit(T_Sym(p), T_ALL_FLOAT)
#define set_all_integer_and_float(p)   set_type_bit(T_Sym(p), (T_ALL_INTEGER | T_ALL_FLOAT))

#define T_COPY_ARGS                    (1 << (TYPE_BITS + 20))
#define needs_copied_args(p)           has_type_bit(T_Pos(p), T_COPY_ARGS) /* set via explicit T_COPY_ARGS, on T_Pos see s7_apply_function */
#define set_needs_copied_args(p)       set_type_bit(T_Pair(p), T_COPY_ARGS)
#define clear_needs_copied_args(p)     clear_type_bit(T_Pair(p), T_COPY_ARGS)
/* this marks something that might mess with its argument list, it should not be in the second byte */

#define T_GENSYM                       (1 << (TYPE_BITS + 21))
#define is_gensym(p)                   has_type_bit(T_Sym(p), T_GENSYM)
/* symbol is from gensym (GC-able etc) */

#define T_FUNCLET                      T_GENSYM
#define is_funclet(p)                  has_type_bit(T_Let(p), T_FUNCLET)
#define set_funclet(p)                 set_type_bit(T_Let(p), T_FUNCLET)
/* this marks a funclet */

#define T_HASH_CHOSEN                  T_GENSYM
#define hash_chosen(p)                 has_type_bit(T_Hsh(p), T_HASH_CHOSEN)
#define hash_set_chosen(p)             set_type_bit(T_Hsh(p), T_HASH_CHOSEN)
#define hash_clear_chosen(p)           clear_type_bit(T_Hsh(p), T_HASH_CHOSEN)

#define T_DOCUMENTED                   T_GENSYM
#define is_documented(p)               has_type_bit(T_Str(p), T_DOCUMENTED)
#define set_documented(p)              set_type_bit(T_Str(p), T_DOCUMENTED)
/* this marks a symbol that has documentation (bit is set on name cell) */

#define T_FX_TREED                     T_GENSYM
#define is_fx_treed(p)                 has_type_bit(T_Pair(p), T_FX_TREED)
#define set_fx_treed(p)                set_type_bit(T_Pair(p), T_FX_TREED)

#define T_SUBVECTOR                    T_GENSYM
#define is_subvector(p)                has_type_bit(T_Vec(p), T_SUBVECTOR)

#define T_HAS_PENDING_VALUE            T_GENSYM
#define slot_set_has_pending_value(p)  set_type_bit(T_Slt(p), T_HAS_PENDING_VALUE)
#define slot_has_pending_value(p)      has_type_bit(T_Slt(p), T_HAS_PENDING_VALUE)
#define slot_clear_has_pending_value(p) clear_type_bit(T_Slt(p), T_HAS_PENDING_VALUE)

#define T_HAS_METHODS                  (1 << (TYPE_BITS + 22))
#define has_methods(p)                 has_type_bit(T_Pos(p), T_HAS_METHODS)
#define has_active_methods(sc, p)      ((has_type_bit(T_Pos(p), T_HAS_METHODS)) && (sc->has_openlets)) /* g_char #<eof> */
#define set_has_methods(p)             set_type_bit(T_Met(p), T_HAS_METHODS)
#define clear_has_methods(p)           clear_type_bit(T_Met(p), T_HAS_METHODS)
/* this marks an environment or closure that is "open" for generic functions etc, don't reuse this bit */

#define T_ITER_OK                      (1LL << (TYPE_BITS + 23))
#define iter_ok(p)                     has_type_bit(T_Itr(p), T_ITER_OK) /* was T_Pos 15-Apr-21 */
#define clear_iter_ok(p)               clear_type_bit(T_Itr(p), T_ITER_OK)

#define T_STEP_END_OK                  T_ITER_OK
#define step_end_ok(p)                 has_type_bit(T_Pair(p), T_STEP_END_OK)
#define set_step_end_ok(p)             set_type_bit(T_Pair(p), T_STEP_END_OK)

#define T_IMPLICIT_SET_OK              T_ITER_OK
#define implicit_set_ok(p)             has_type_bit(T_Pair(p), T_IMPLICIT_SET_OK)
#define set_implicit_set_ok(p)         set_type_bit(T_Pair(p), T_IMPLICIT_SET_OK)

#define T_IN_ROOTLET                   T_ITER_OK
#define in_rootlet(p)                  has_type_bit(T_Slt(p), T_IN_ROOTLET)
#define set_in_rootlet(p)              set_type_bit(T_Slt(p), T_IN_ROOTLET)

#define T_BOOL_FUNCTION                T_ITER_OK
#define is_bool_function(p)            has_type_bit(T_Prc(p), T_BOOL_FUNCTION)
#define set_is_bool_function(p)        set_type_bit(T_Fnc(p), T_BOOL_FUNCTION)

/* it's faster here to use the high_flag bits rather than typeflag bits */
#define BIT_ROOM                       16
#define T_FULL_SYMCONS                 (1LL << (TYPE_BITS + BIT_ROOM + 24))
#define T_SYMCONS                      (1 << 0)
#define is_possibly_constant(p)        has_type1_bit(T_Sym(p), T_SYMCONS)
#define set_possibly_constant(p)       set_type1_bit(T_Sym(p), T_SYMCONS)
#define is_probably_constant(p)        has_type_bit(T_Sym(p), (T_FULL_SYMCONS | T_IMMUTABLE))

#define T_HAS_LET_ARG                  T_SYMCONS
#define has_let_arg(p)                 has_type1_bit(T_Prc(p), T_HAS_LET_ARG)
#define set_has_let_arg(p)             set_type1_bit(T_Prc(p), T_HAS_LET_ARG)
/* p is a setter procedure, "let arg" refers to the setter's optional third (let) argument */

#define T_HASH_VALUE_TYPE              T_SYMCONS
#define has_hash_value_type(p)         has_type1_bit(T_Hsh(p), T_HASH_VALUE_TYPE)
#define set_has_hash_value_type(p)     set_type1_bit(T_Hsh(p), T_HASH_VALUE_TYPE)

#define T_INT_OPTABLE                  T_SYMCONS
#define is_int_optable(p)              has_type1_bit(T_Pair(p), T_INT_OPTABLE)
#define set_is_int_optable(p)          set_type1_bit(T_Pair(p), T_INT_OPTABLE)

/* symbol free here */
#define T_FULL_HAS_LET_FILE            (1LL << (TYPE_BITS + BIT_ROOM + 25))
#define T_HAS_LET_FILE                 (1 << 1)
#define has_let_file(p)                has_type1_bit(T_Let(p), T_HAS_LET_FILE)
#define set_has_let_file(p)            set_type1_bit(T_Let(p), T_HAS_LET_FILE)
#define clear_has_let_file(p)          clear_type1_bit(T_Let(p), T_HAS_LET_FILE)

#define T_TYPED_VECTOR                 T_HAS_LET_FILE
#define is_typed_vector(p)             has_type1_bit(T_Vec(p), T_TYPED_VECTOR)
#define set_typed_vector(p)            set_type1_bit(T_Vec(p), T_TYPED_VECTOR)

#define T_TYPED_HASH_TABLE             T_HAS_LET_FILE
#define is_typed_hash_table(p)         has_type1_bit(T_Hsh(p), T_TYPED_HASH_TABLE)
#define set_typed_hash_table(p)        set_type1_bit(T_Hsh(p), T_TYPED_HASH_TABLE)

#define T_BOOL_SETTER                  T_HAS_LET_FILE
#define c_function_has_bool_setter(p)  has_type1_bit(T_Fnc(p), T_BOOL_SETTER)
#define c_function_set_has_bool_setter(p) set_type1_bit(T_Fnc(p), T_BOOL_SETTER)

#define T_REST_SLOT                    T_HAS_LET_FILE
#define is_rest_slot(p)                has_type1_bit(T_Slt(p), T_REST_SLOT)
#define set_is_rest_slot(p)            set_type1_bit(T_Slt(p), T_REST_SLOT)

#define T_NO_DEFAULTS                  T_HAS_LET_FILE
#define T_FULL_NO_DEFAULTS             T_FULL_HAS_LET_FILE
#define has_no_defaults(p)             has_type1_bit(T_Pcs(p), T_NO_DEFAULTS)
#define set_has_no_defaults(p)         set_type1_bit(T_Pcs(p), T_NO_DEFAULTS)
/* pair=closure* body, transferred to closure* */

#define T_FULL_DEFINER                 (1LL << (TYPE_BITS + BIT_ROOM + 26))
#define T_DEFINER                      (1 << 2)
#define is_definer(p)                  has_type1_bit(T_Sym(p), T_DEFINER)
#define set_is_definer(p)              set_type1_bit(T_Sym(p), T_DEFINER)
#define is_func_definer(p)             has_type1_bit(T_Fnc(p), T_DEFINER)
#define set_func_is_definer(p)         do {set_type1_bit(T_Fnc(initial_value(p)), T_DEFINER); set_type1_bit(T_Sym(p), T_DEFINER);} while (0)
#define is_syntax_definer(p)           has_type1_bit(T_Syn(p), T_DEFINER)
#define set_syntax_is_definer(p)       do {set_type1_bit(T_Syn(initial_value(p)), T_DEFINER); set_type1_bit(T_Sym(p), T_DEFINER);} while (0)
/* this marks "definers" like define and define-macro */

#define T_MACLET                       T_DEFINER
#define is_maclet(p)                   has_type1_bit(T_Let(p), T_MACLET)
#define set_maclet(p)                  set_type1_bit(T_Let(p), T_MACLET)
/* this marks a maclet */

#define T_HAS_FX                       T_DEFINER
#define set_has_fx(p)                  set_type1_bit(T_Pair(p), T_HAS_FX)
#define has_fx(p)                      has_type1_bit(T_Pair(p), T_HAS_FX)
#define clear_has_fx(p)                clear_type1_bit(T_Pair(p), T_HAS_FX)

#define T_SLOT_DEFAULTS                T_DEFINER
#define slot_defaults(p)               has_type1_bit(T_Slt(p), T_SLOT_DEFAULTS)
#define set_slot_defaults(p)           set_type1_bit(T_Slt(p), T_SLOT_DEFAULTS)

#define T_WEAK_HASH_ITERATOR           T_DEFINER
#define is_weak_hash_iterator(p)       has_type1_bit(T_Itr(p), T_WEAK_HASH_ITERATOR)
#define set_weak_hash_iterator(p)      set_type1_bit(T_Itr(p), T_WEAK_HASH_ITERATOR)
#define clear_weak_hash_iterator(p)    clear_type1_bit(T_Itr(p), T_WEAK_HASH_ITERATOR)

#define T_HASH_KEY_TYPE                T_DEFINER
#define has_hash_key_type(p)           has_type1_bit(T_Hsh(p), T_HASH_KEY_TYPE)
#define set_has_hash_key_type(p)       set_type1_bit(T_Hsh(p), T_HASH_KEY_TYPE)

#define T_FULL_BINDER                  (1LL << (TYPE_BITS + BIT_ROOM + 27))
#define T_BINDER                       (1 << 3)
#define set_syntax_is_binder(p)        do {set_type1_bit(T_Syn(initial_value(p)), T_BINDER); set_type1_bit(T_Sym(p), T_BINDER);} while (0)
#define is_definer_or_binder(p)        has_type1_bit(T_Sym(p), T_DEFINER | T_BINDER)
/* this marks "binders" like let */

#define T_SEMISAFE                     T_BINDER
#define is_semisafe(p)                 has_type1_bit(T_Fnc(p), T_SEMISAFE)
#define set_is_semisafe(p)             set_type1_bit(T_Fnc(p), T_SEMISAFE)

/* #define T_TREE_COLLECTED            T_FULL_BINDER */
#define T_SHORT_TREE_COLLECTED         T_BINDER
#define tree_is_collected(p)           has_type1_bit(T_Pair(p), T_SHORT_TREE_COLLECTED)
#define tree_set_collected(p)          set_type1_bit(T_Pair(p), T_SHORT_TREE_COLLECTED)
#define tree_clear_collected(p)        clear_type1_bit(T_Pair(p), T_SHORT_TREE_COLLECTED)

#define T_SIMPLE_VALUES                T_BINDER
#define has_simple_values(p)           has_type1_bit(T_Hsh(p), T_SIMPLE_VALUES)
#define set_has_simple_values(p)       set_type1_bit(T_Hsh(p), T_SIMPLE_VALUES)

#define T_VERY_SAFE_CLOSURE            (1LL << (TYPE_BITS + BIT_ROOM + 28))
#define T_SHORT_VERY_SAFE_CLOSURE      (1 << 4)
#define is_very_safe_closure(p)        has_type1_bit(T_Clo(p), T_SHORT_VERY_SAFE_CLOSURE)
#define set_very_safe_closure(p)       set_type1_bit(T_Clo(p), T_SHORT_VERY_SAFE_CLOSURE)
#define closure_bits(p)                (full_type(T_Pair(p)) & (T_SAFE_CLOSURE | T_VERY_SAFE_CLOSURE | T_FULL_NO_DEFAULTS))
#define is_very_safe_closure_body(p)   has_type1_bit(T_Pair(p), T_SHORT_VERY_SAFE_CLOSURE)
#define set_very_safe_closure_body(p)  set_type1_bit(T_Pair(p), T_SHORT_VERY_SAFE_CLOSURE)

#define T_BAFFLE_LET                   T_SHORT_VERY_SAFE_CLOSURE
#define is_baffle_let(p)               has_type1_bit(T_Let(p), T_BAFFLE_LET)
#define set_baffle_let(p)              set_type1_bit(T_Let(p), T_BAFFLE_LET)

#define T_CYCLIC                       (1LL << (TYPE_BITS + BIT_ROOM + 29))
#define T_SHORT_CYCLIC                 (1 << 5)
#define is_cyclic(p)                   has_type1_bit(T_Seq(p), T_SHORT_CYCLIC)
#define set_cyclic(p)                  set_type1_bit(T_Seq(p), T_SHORT_CYCLIC)

#define T_CYCLIC_SET                   (1LL << (TYPE_BITS + BIT_ROOM + 30))
#define T_SHORT_CYCLIC_SET             (1 << 6)
#define is_cyclic_set(p)               has_type1_bit(T_Seq(p), T_SHORT_CYCLIC_SET) /* was T_Pos 30-Jan-21 */
#define set_cyclic_set(p)              set_type1_bit(T_Seq(p), T_SHORT_CYCLIC_SET)
#define clear_cyclic_bits(p)           clear_type_bit(p, T_COLLECTED | T_SHARED | T_CYCLIC | T_CYCLIC_SET)

#define T_KEYWORD                      (1LL << (TYPE_BITS + BIT_ROOM + 31))
#define T_SHORT_KEYWORD                (1 << 7)
#define is_keyword(p)                  has_type1_bit(T_Sym(p), T_SHORT_KEYWORD)
#define is_symbol_and_keyword(p)       ((is_symbol(p)) && (is_keyword(p)))
/* this bit distinguishes a symbol from a symbol that is also a keyword */

#define T_FX_TREEABLE                  T_SHORT_KEYWORD
#define is_fx_treeable(p)              has_type1_bit(T_Pair(p), T_FX_TREEABLE)
#define set_is_fx_treeable(p)          set_type1_bit(T_Pair(p), T_FX_TREEABLE)

#define T_FULL_SIMPLE_ELEMENTS         (1LL << (TYPE_BITS + BIT_ROOM + 32))
#define T_SIMPLE_ELEMENTS              (1 << 8)
#define has_simple_elements(p)         has_type1_bit(T_Nvc(p), T_SIMPLE_ELEMENTS)
#define set_has_simple_elements(p)     set_type1_bit(T_Nvc(p), T_SIMPLE_ELEMENTS)
#define c_function_has_simple_elements(p)     has_type1_bit(T_Fnc(p), T_SIMPLE_ELEMENTS)
#define c_function_set_has_simple_elements(p) set_type1_bit(T_Fnc(p), T_SIMPLE_ELEMENTS)
/* c_func case here refers to boolean? et al -- structure element type declaration that ensures a simple object */

#define T_SIMPLE_KEYS                  T_SIMPLE_ELEMENTS
#define has_simple_keys(p)             has_type1_bit(T_Hsh(p), T_SIMPLE_KEYS)
#define set_has_simple_keys(p)         set_type1_bit(T_Hsh(p), T_SIMPLE_KEYS)

#define T_SAFE_SETTER                  T_SIMPLE_ELEMENTS
#define is_safe_setter(p)              has_type1_bit(T_Sym(p), T_SAFE_SETTER)
#define set_is_safe_setter(p)          set_type1_bit(T_Sym(p), T_SAFE_SETTER)

#define T_FLOAT_OPTABLE                T_SIMPLE_ELEMENTS
#define is_float_optable(p)            has_type1_bit(T_Pair(p), T_FLOAT_OPTABLE)
#define set_is_float_optable(p)        set_type1_bit(T_Pair(p), T_FLOAT_OPTABLE)

#define T_FULL_CASE_KEY                (1LL << (TYPE_BITS + BIT_ROOM + 33))
#define T_CASE_KEY                     (1 << 9)
#define is_case_key(p)                 has_type1_bit(T_Pos(p), T_CASE_KEY)
#define set_case_key(p)                set_type1_bit(T_Sym(p), T_CASE_KEY)

#define T_OPT1_FUNC_LISTED             T_CASE_KEY
#define opt1_func_listed(p)            has_type1_bit(T_Pair(p), T_OPT1_FUNC_LISTED)
#define set_opt1_func_listed(p)        set_type1_bit(T_Pair(p), T_OPT1_FUNC_LISTED)

#define T_FULL_HAS_GX                  (1LL << (TYPE_BITS + BIT_ROOM + 34))
#define T_HAS_GX                       (1 << 10)
#define has_gx(p)                      has_type1_bit(T_Pair(p), T_HAS_GX)
#define set_has_gx(p)                  set_type1_bit(T_Pair(p), T_HAS_GX)

#define T_FULL_UNKNOPT                 (1LL << (TYPE_BITS + BIT_ROOM + 35))
#define T_UNKNOPT                      (1 << 11)
#define is_unknopt(p)                  has_type1_bit(T_Pair(p), T_UNKNOPT)
#define set_is_unknopt(p)              set_type1_bit(T_Pair(p), T_UNKNOPT)

#define T_MAC_OK                       T_UNKNOPT
#define mac_is_ok(p)                   has_type1_bit(T_Pair(p), T_MAC_OK)
#define set_mac_is_ok(p)               set_type1_bit(T_Pair(p), T_MAC_OK)
/* marks a macro (via (macro...)) that has been checked -- easier (and slower) than making 4 or 5 more ops, op_macro_unchecked and so on */

#define T_FULL_SAFETY_CHECKED          (1LL << (TYPE_BITS + BIT_ROOM + 36))
#define T_SAFETY_CHECKED               (1 << 12)
#define is_safety_checked(p)           has_type1_bit(T_Pair(p), T_SAFETY_CHECKED)
#define set_safety_checked(p)          set_type1_bit(T_Pair(p), T_SAFETY_CHECKED)

#define T_FULL_HAS_FN                  (1LL << (TYPE_BITS + BIT_ROOM + 37))
#define T_HAS_FN                       (1 << 13)
#define set_has_fn(p)                  set_type1_bit(T_Pair(p), T_HAS_FN)
#define has_fn(p)                      has_type1_bit(T_Pair(p), T_HAS_FN)

#define T_GC_MARK                      0x8000000000000000
#define is_marked(p)                   has_type_bit(p, T_GC_MARK)
#define set_mark(p)                    set_type_bit(T_Pos(p), T_GC_MARK)
#define clear_mark(p)                  clear_type_bit(p, T_GC_MARK)
/* using the sign bit, bit 23 (or 55) == 31 (or 63) for this makes a big difference in the GC */

#define T_UNHEAP                       0x4000000000000000
#define T_SHORT_UNHEAP                 (1 << 14)
#define in_heap(p)                     (((T_Pos(p))->tf.opts.high_flag & T_SHORT_UNHEAP) == 0)
#define unheap(sc, p)                  set_type1_bit(T_Pos(p), T_SHORT_UNHEAP)

#define is_eof(p)                      ((T_Pos(p)) == eof_object)
#define is_true(Sc, p)                 ((T_Pos(p)) != Sc->F)
#define is_false(Sc, p)                ((T_Pos(p)) == Sc->F)

#ifdef _MSC_VER
  static s7_pointer make_boolean(s7_scheme *sc, bool val) {if (val) return(sc->T); return(sc->F);}
#else
  #define make_boolean(sc, Val)        ((Val) ? sc->T : sc->F)
#endif

#define is_pair(p)                     (type(p) == T_PAIR)
#define is_mutable_pair(p)             ((is_pair(p)) && (!is_immutable(p))) /* same speed: ((full_type(T_Pos(p)) & (TYPE_MASK | T_IMMUTABLE)) == T_PAIR) */
#define is_null(p)                     ((T_Pos(p)) == sc->nil)
#define is_not_null(p)                 ((T_Pos(p)) != sc->nil)
#define is_list(p)                     ((is_pair(p)) || (type(p) == T_NIL))
#define is_quoted_pair(p)              ((is_pair(p)) && (car(p) == sc->quote_symbol))
#define is_unquoted_pair(p)            ((is_pair(p)) && (car(p) != sc->quote_symbol))
#define is_quoted_symbol(p)            ((is_pair(p)) && (car(p) == sc->quote_symbol) && (is_symbol(cadr(p))))


/* pair line/file/position */
#define PAIR_LINE_BITS                 24
#define PAIR_FILE_BITS                 12
#define PAIR_POSITION_BITS             28
#define PAIR_LINE_OFFSET               0
#define PAIR_FILE_OFFSET               PAIR_LINE_BITS
#define PAIR_POSITION_OFFSET           (PAIR_LINE_BITS + PAIR_FILE_BITS)
#define PAIR_LINE_MASK                 ((1 << PAIR_LINE_BITS) - 1)
#define PAIR_FILE_MASK                 ((1 << PAIR_FILE_BITS) - 1)
#define PAIR_POSITION_MASK             ((1 << PAIR_POSITION_BITS) - 1)

#define port_location(Pt)              (((port_line_number(Pt) & PAIR_LINE_MASK) << PAIR_LINE_OFFSET) | \
                                        ((port_file_number(Pt) & PAIR_FILE_MASK) << PAIR_FILE_OFFSET) | \
                                        ((port_position(Pt) & PAIR_POSITION_MASK) << PAIR_POSITION_OFFSET))

#define location_to_line(Loc)          ((Loc >> PAIR_LINE_OFFSET) & PAIR_LINE_MASK)
#define location_to_file(Loc)          ((Loc >> PAIR_FILE_OFFSET) & PAIR_FILE_MASK)
#define location_to_position(Loc)      ((Loc >> PAIR_POSITION_OFFSET) & PAIR_POSITION_MASK)

#define pair_line_number(p)            location_to_line(pair_location(p))
#define pair_file_number(p)            location_to_file(pair_location(p))
#define pair_position(p)               location_to_position(pair_location(p))

#if (!S7_DEBUGGING)
#define pair_location(p)               (p)->object.sym_cons.location
#define pair_set_location(p, X)        (p)->object.sym_cons.location = X
#define pair_raw_hash(p)               (p)->object.sym_cons.hash
#define pair_set_raw_hash(p, X)        (p)->object.sym_cons.hash = X
#define pair_raw_len(p)                (p)->object.sym_cons.location
#define pair_set_raw_len(p, X)         (p)->object.sym_cons.location = X
#define pair_raw_name(p)               (p)->object.sym_cons.fstr
#define pair_set_raw_name(p, X)        (p)->object.sym_cons.fstr = X
/* opt1 == raw_hash, opt2 == raw_name, opt3 == line|ctr + len, but hash/name/len only apply to the symbol table so there's no collision */

#define opt1(p, r)                     ((p)->object.cons.opt1)
#define set_opt1(p, x, r)              (p)->object.cons.opt1 = x
#define opt2(p, r)                     ((p)->object.cons.o2.opt2)
#define set_opt2(p, x, r)              (p)->object.cons.o2.opt2 = (s7_pointer)(x)
#define opt2_n(p, r)                   ((p)->object.cons.o2.n)
#define set_opt2_n(p, x, r)            (p)->object.cons.o2.n = x
#define opt3(p, r)                     ((p)->object.cons.o3.opt3)
#define set_opt3(p, x, r)              do {(p)->object.cons.o3.opt3 = x; clear_type_bit(p, T_LOCATION);} while (0)
#define opt3_n(p, r)                   ((p)->object.cons.o3.n)
#define set_opt3_n(p, x, r)            do {(p)->object.cons.o3.n = x; clear_type_bit(p, T_LOCATION);} while (0)

#else

/* the 3 opt fields hold most of the varigated optimizer info, so they are used in many conflicting ways.
 * the bits and funcs here try to track each such use, and report any cross-talk or collisions.
 * all of this machinery vanishes if debugging is turned off.
 */
#define OPT1_SET                       (1 << 0)
#define OPT2_SET                       (1 << 1)
#define OPT3_SET                       (1 << 2)

#define OPT1_FAST                      (1 << 3)   /* fast list in member/assoc circular list check */
#define OPT1_CFUNC                     (1 << 4)   /* c-function */
#define OPT1_CLAUSE                    (1 << 5)   /* case clause */
#define OPT1_LAMBDA                    (1 << 6)   /* lambda(*) */
#define OPT1_SYM                       (1 << 7)   /* symbol */
#define OPT1_PAIR                      (1 << 8)   /* pair */
#define OPT1_CON                       (1 << 9)   /* constant from eval's point of view */ /* 10 was opt1_goto, unused */
#define OPT1_ANY                       (1 << 11)  /* anything -- deliberate unchecked case */
#define OPT1_HASH                      (1 << 12)  /* hash code used in the symbol table (pair_raw_hash) */
#define OPT1_MASK                      (OPT1_FAST | OPT1_CFUNC | OPT1_CLAUSE | OPT1_LAMBDA | OPT1_SYM | OPT1_PAIR | OPT1_CON | OPT1_ANY | OPT1_HASH)

#define opt1_is_set(p)                 (((T_Pair(p))->debugger_bits & OPT1_SET) != 0)
#define set_opt1_is_set(p)             (T_Pair(p))->debugger_bits |= OPT1_SET
#define opt1_role_matches(p, Role)     (((T_Pair(p))->debugger_bits & OPT1_MASK) == Role)
#define set_opt1_role(p, Role)         (T_Pair(p))->debugger_bits = (Role | ((p)->debugger_bits & ~OPT1_MASK))
#define opt1(p, Role)                  opt1_1(sc, T_Pair(p), Role, __func__, __LINE__)
#define set_opt1(p, x, Role)           set_opt1_1(T_Pair(p), x, Role)

#define OPT2_KEY                       (1 << 13)  /* case key */
#define OPT2_SLOW                      (1 << 14)  /* slow list in member/assoc circular list check */
#define OPT2_SYM                       (1 << 15)  /* symbol */
#define OPT2_PAIR                      (1 << 16)  /* pair */
#define OPT2_CON                       (1 << 17)  /* constant as above */
#define OPT2_FX                        (1 << 18)  /* fx (fx_*) func (sc, form) */
#define OPT2_FN                        (1 << 19)  /* fn (s7_function) func (sc, arglist) */
#define OPT2_LAMBDA                    (1 << 20)  /* lambda form */
#define OPT2_NAME                      (1 << 21)  /* named used by symbol table (pair_raw_name) */
#define OPT2_DIRECT                    (1LL << 32)
#define OPT2_INT                       (1LL << 33)
#define OPT2_MASK                      (OPT2_KEY | OPT2_SLOW | OPT2_SYM | OPT2_PAIR | OPT2_CON | OPT2_FX | \
                                        OPT2_FN | OPT2_LAMBDA | OPT2_DIRECT | OPT2_NAME | OPT2_INT)

#define opt2_is_set(p)                 (((T_Pair(p))->debugger_bits & OPT2_SET) != 0)
#define set_opt2_is_set(p)             (T_Pair(p))->debugger_bits |= OPT2_SET
#define opt2_role_matches(p, Role)     (((T_Pair(p))->debugger_bits & OPT2_MASK) == Role)
#define set_opt2_role(p, Role)         (T_Pair(p))->debugger_bits = (Role | ((p)->debugger_bits & ~OPT2_MASK))
#define opt2(p, Role)                  opt2_1(sc, T_Pair(p), Role, __func__, __LINE__)
#define set_opt2(p, x, Role)           set_opt2_1(sc, T_Pair(p), (s7_pointer)(x), Role, __func__, __LINE__)
#define opt2_n(p, Role)                opt2_n_1(sc, T_Pair(p), Role, __func__, __LINE__)
#define set_opt2_n(p, x, Role)         set_opt2_n_1(sc, T_Pair(p), x, Role, __func__, __LINE__)

#define OPT3_ARGLEN                    (1 << 22)  /* arglist length */
#define OPT3_SYM                       (1 << 23)  /* expression symbol access */
#define OPT3_AND                       (1 << 24)  /* and second clause */
#define OPT3_DIRECT                    (1 << 25)  /* direct call info */
#define OPT3_ANY                       (1 << 26)
#define OPT3_LET                       (1 << 27)  /* let or #f */
#define OPT3_CON                       (1 << 28)
#define OPT3_LOCATION                  (1 << 29)
#define OPT3_LEN                       (1 << 30)
#define OPT3_BYTE                      (1LL << 31)
#define OPT3_INT                       (1LL << 34)
#define OPT3_MASK                      (OPT3_ARGLEN | OPT3_SYM | OPT3_AND | OPT3_ANY | OPT3_LET | OPT3_BYTE | \
                                        OPT3_LOCATION | OPT3_LEN | OPT3_DIRECT | OPT3_CON | OPT3_INT)

#define opt3_is_set(p)                 (((T_Pair(p))->debugger_bits & OPT3_SET) != 0)
#define set_opt3_is_set(p)             (T_Pair(p))->debugger_bits |= OPT3_SET
#define opt3_role_matches(p, Role)     (((T_Pair(p))->debugger_bits & OPT3_MASK) == Role)
#define set_opt3_role(p, Role)         (T_Pair(p))->debugger_bits = (Role | ((p)->debugger_bits & ~OPT3_MASK))
#define opt3(p, Role)                  opt3_1(sc, T_Pair(p), Role, __func__, __LINE__)
#define set_opt3(p, x, Role)           set_opt3_1(T_Pair(p), x, Role)
#define opt3_n(p, Role)                opt3_n_1(sc, T_Pair(p), Role, __func__, __LINE__)
#define set_opt3_n(p, x, Role)         set_opt3_n_1(T_Pair(p), x, Role)

#define pair_location(p)               opt3_location_1(sc, T_Pair(p), __func__, __LINE__)
#define pair_set_location(p, X)        set_opt3_location_1(T_Pair(p), X)
#define pair_raw_hash(p)               opt1_hash_1(sc, T_Pair(p), __func__, __LINE__)
#define pair_set_raw_hash(p, X)        set_opt1_hash_1(T_Pair(p), X)
#define pair_raw_len(p)                opt3_len_1(sc, T_Pair(p), __func__, __LINE__)
#define pair_set_raw_len(p, X)         set_opt3_len_1(T_Pair(p), X)
#define pair_raw_name(p)               opt2_name_1(sc, T_Pair(p), __func__, __LINE__)
#define pair_set_raw_name(p, X)        set_opt2_name_1(T_Pair(p), X)

#define L_HIT                          (1LL << 40) /* "L_SET" is taken */
#define L_FUNC                         (1LL << 41)
#define L_DOX                          (1LL << 42)
#define L_MASK                         (L_FUNC | L_DOX)
#endif

#define opt1_fast(P)                   T_Lst(opt1(P,                OPT1_FAST))
#define set_opt1_fast(P, X)            set_opt1(P, T_Pair(X),       OPT1_FAST)
#define opt1_cfunc(P)                  T_Pos(opt1(P,                OPT1_CFUNC))
#define set_opt1_cfunc(P, X)           set_opt1(P, T_Fnc(X),        OPT1_CFUNC)
#define opt1_lambda_unchecked(P)       opt1(P,                      OPT1_LAMBDA) /* can be free/null? from s7_call? */
#define opt1_lambda(P)                 T_Clo(opt1(P,                OPT1_LAMBDA))
#define set_opt1_lambda(P, X)          set_opt1(P, T_Clo(X),        OPT1_LAMBDA)
#define set_opt1_lambda_add(P, X)      do {set_opt1(P, T_Clo(X),    OPT1_LAMBDA); add_opt1_func(sc, P);} while (0)
#define opt1_clause(P)                 T_Pos(opt1(P,                OPT1_CLAUSE))
#define set_opt1_clause(P, X)          set_opt1(P, T_Pos(X),        OPT1_CLAUSE)
#define opt1_sym(P)                    T_Sym(opt1(P,                OPT1_SYM))
#define set_opt1_sym(P, X)             set_opt1(P, T_Sym(X),        OPT1_SYM)
#define opt1_pair(P)                   T_Lst(opt1(P,                OPT1_PAIR))
#define set_opt1_pair(P, X)            set_opt1(P, T_Lst(X),        OPT1_PAIR)
#define opt1_con(P)                    T_Pos(opt1(P,                OPT1_CON))
#define set_opt1_con(P, X)             set_opt1(P, T_Pos(X),        OPT1_CON)
#define opt1_any(P)                    opt1(P,                      OPT1_ANY)    /* can be free in closure_is_ok */
#define set_opt1_any(P, X)             set_opt1(P, X,               OPT1_ANY)

#define opt2_any(P)                    opt2(P,                      OPT2_KEY)
#define set_opt2_any(P, X)             set_opt2(P, X,               OPT2_KEY)
#define opt2_int(P)                    opt2_n(P,                    OPT2_INT)
#define set_opt2_int(P, X)             set_opt2_n(P, X,             OPT2_INT)
#define opt2_slow(P)                   T_Lst(opt2(P,                OPT2_SLOW))
#define set_opt2_slow(P, X)            set_opt2(P, T_Pair(X),       OPT2_SLOW)
#define opt2_sym(P)                    T_Sym(opt2(P,                OPT2_SYM))
#define set_opt2_sym(P, X)             set_opt2(P, T_Sym(X),        OPT2_SYM)
#define opt2_pair(P)                   T_Lst(opt2(P,                OPT2_PAIR))
#define set_opt2_pair(P, X)            set_opt2(P, T_Lst(X),        OPT2_PAIR)
#define opt2_con(P)                    T_Pos(opt2(P,                OPT2_CON))
#define set_opt2_con(P, X)             set_opt2(P, T_Pos(X),        OPT2_CON)
#define opt2_lambda(P)                 T_Pair(opt2(P,               OPT2_LAMBDA))
#define set_opt2_lambda(P, X)          set_opt2(P, T_Pair(X),       OPT2_LAMBDA)
#define opt2_direct(P)                 opt2(P,                      OPT2_DIRECT)
#define set_opt2_direct(P, X)          set_opt2(P, (s7_pointer)(X), OPT2_DIRECT)

#define opt3_arglen(P)                 opt3_n(P,                    OPT3_ARGLEN)
#define set_opt3_arglen(P, X)          set_opt3_n(P, X,             OPT3_ARGLEN)
#define opt3_int(P)                    opt3_n(P,                    OPT3_INT)
#define set_opt3_int(P, X)             set_opt3_n(P, X,             OPT3_INT)
#define opt3_sym(P)                    T_Sym(opt3(P,                OPT3_SYM))
#define set_opt3_sym(P, X)             set_opt3(P, T_Sym(X),        OPT3_SYM)
#define opt3_con(P)                    T_Pos(opt3(P,                OPT3_CON))
#define set_opt3_con(P, X)             set_opt3(P, T_Pos(X),        OPT3_CON)
#define opt3_pair(P)                   T_Pair(opt3(P,               OPT3_AND))
#define set_opt3_pair(P, X)            set_opt3(P, T_Pair(X),       OPT3_AND)
#define opt3_any(P)                    opt3(P,                      OPT3_ANY)
#define set_opt3_any(P, X)             set_opt3(P, X,               OPT3_ANY)
#define opt3_let(P)                    T_Lid(opt3(P,                OPT3_LET))
#define set_opt3_let(P, X)             set_opt3(P, T_Lid(X),        OPT3_LET)
#define opt3_direct(P)                 opt3(P,                      OPT3_DIRECT)
#define set_opt3_direct(P, X)          set_opt3(P, (s7_pointer)(X), OPT3_DIRECT)

#if S7_DEBUGGING
#define opt3_byte(p)                   opt3_byte_1(sc, T_Pair(p), OPT3_BYTE, __func__, __LINE__)
#define set_opt3_byte(p, x)            set_opt3_byte_1(T_Pair(p), x, OPT3_BYTE, __func__, __LINE__)
#else
#define opt3_byte(P)                   T_Pair(P)->object.cons.o3.opt_type /* op_if_is_type, opt_type == opt3 in cons */
#define set_opt3_byte(P, X)            do {T_Pair(P)->object.cons.o3.opt_type = X; clear_type_bit(P, T_LOCATION);} while (0)
#endif

#define pair_macro(P)                  opt2_sym(P)
#define set_pair_macro(P, Name)        set_opt2_sym(P, Name)

#define fn_proc(f)                     ((s7_function)(opt2(f, OPT2_FN)))
#define fx_proc(f)                     ((s7_function)(opt2(f, OPT2_FX)))
#define fn_proc_unchecked(f)           ((s7_function)(T_Pair(f)->object.cons.o2.opt2))
#define fx_proc_unchecked(f)           ((s7_function)(T_Pair(f)->object.cons.o2.opt2)) /* unused */

#define set_fx(f, _X_)                 do {s7_pointer X; X = (s7_pointer)(_X_); set_opt2(f, X, OPT2_FX); if (X) set_has_fx(f); else clear_has_fx(f);} while (0)
#define set_fx_direct(f, X)            do {set_opt2(f, (s7_pointer)(X), OPT2_FX); set_has_fx(f);} while (0)
#define set_fn(f, _X_)                 do {s7_pointer X; X = (s7_pointer)(_X_); set_opt2(f, X, OPT2_FN); if (X) set_has_fn(f); else clear_has_fx(f);} while (0)
#define set_fn_direct(f, X)            do {set_opt2(f, (s7_pointer)(X), OPT2_FN); set_has_fn(f);} while (0)

#if WITH_GCC
#define fx_call(Sc, F)                 ({s7_pointer _P_; _P_ = F; fx_proc(_P_)(Sc, car(_P_));})
#define fc_call(Sc, F)                 ({s7_pointer _P_; _P_ = F; fn_proc(_P_)(Sc, cdr(_P_));})
#define fn_call(Sc, F)                 ({s7_pointer _P_; _P_ = F; fn_proc(_P_)(Sc, cdr(_P_));})
#else
#define fx_call(Sc, F)                 fx_proc(F)(Sc, car(F))
#define fc_call(Sc, F)                 fn_proc(F)(Sc, cdr(F))
#define fn_call(Sc, F)                 fn_proc(F)(Sc, cdr(F))
#endif
/* fx_call can affect the stack and sc->value */

#define car(p)                         (T_Pair(p))->object.cons.car
#define set_car(p, Val)                car(p) = T_Pos(Val)
#define cdr(p)                         (T_Pair(p))->object.cons.cdr
#define set_cdr(p, Val)                cdr(p) = T_Pos(Val)
#define unchecked_car(p)               (T_Pos(p))->object.cons.car
#define unchecked_cdr(p)               (T_Pos(p))->object.cons.cdr

#define caar(p)                        car(car(p))
#define cadr(p)                        car(cdr(p))
#define set_cadr(p, Val)               car(cdr(p)) = T_Pos(Val)
#define cdar(p)                        cdr(car(p))
#define set_cdar(p, Val)               cdr(car(p)) = T_Pos(Val)
#define cddr(p)                        cdr(cdr(p))

#define caaar(p)                       car(car(car(p)))
#define cadar(p)                       car(cdr(car(p)))
#define cdadr(p)                       cdr(car(cdr(p)))
#define caddr(p)                       car(cdr(cdr(p)))
#define set_caddr(p, Val)              car(cdr(cdr(p))) = T_Pos(Val)
#define caadr(p)                       car(car(cdr(p)))
#define cdaar(p)                       cdr(car(car(p)))
#define cdddr(p)                       cdr(cdr(cdr(p)))
#define set_cdddr(p, Val)              cdr(cdr(cdr(p))) = T_Pos(Val)
#define cddar(p)                       cdr(cdr(car(p)))

#define caaadr(p)                      car(car(car(cdr(p))))
#define caadar(p)                      car(car(cdr(car(p))))
#define cadaar(p)                      car(cdr(car(car(p))))
#define cadddr(p)                      car(cdr(cdr(cdr(p))))
#define caaddr(p)                      car(car(cdr(cdr(p))))
#define cddddr(p)                      cdr(cdr(cdr(cdr(p))))
#define caddar(p)                      car(cdr(cdr(car(p))))
#define cdadar(p)                      cdr(car(cdr(car(p))))
#define cdaddr(p)                      cdr(car(cdr(cdr(p))))
#define caaaar(p)                      car(car(car(car(p))))
#define cadadr(p)                      car(cdr(car(cdr(p))))
#define cdaadr(p)                      cdr(car(car(cdr(p))))
#define cdaaar(p)                      cdr(car(car(car(p))))
#define cdddar(p)                      cdr(cdr(cdr(car(p))))
#define cddadr(p)                      cdr(cdr(car(cdr(p))))
#define cddaar(p)                      cdr(cdr(car(car(p))))

#define cadaddr(p)                     cadr(caddr(p))
#define caddadr(p)                     caddr(cadr(p))
#define caddaddr(p)                    caddr(caddr(p))

#if WITH_GCC
  /* slightly tricky because cons can be called recursively, macro here is faster than inline function */
  #define cons(Sc, A, B)   ({s7_pointer _X_, _A_, _B_; _A_ = A; _B_ = B; new_cell(Sc, _X_, T_PAIR | T_SAFE_PROCEDURE); set_car(_X_, _A_); set_cdr(_X_, _B_); _X_;})
#else
  #define cons(Sc, A, B)               s7_cons(Sc, A, B)
#endif

#define list_1(Sc, A)                  cons(Sc, A, Sc->nil)
#define list_1_unchecked(Sc, A)        cons_unchecked(Sc, A, Sc->nil)
#define list_2(Sc, A, B)               cons_unchecked(Sc, A, cons(Sc, B, Sc->nil))
#define list_2_unchecked(Sc, A, B)     cons_unchecked(Sc, A, cons_unchecked(Sc, B, Sc->nil))
#define list_3(Sc, A, B, C)            cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons(Sc, C, Sc->nil)))
#define list_4(Sc, A, B, C, D)         cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons_unchecked(Sc, C, cons(Sc, D, Sc->nil))))
#define with_list_t1(A)                (set_car(sc->t1_1, A), sc->t1_1) /* this is slower than explicit code, esp t3, procedures are same as this */
#define with_list_t2(A, B)             (set_car(sc->t2_1, A), set_car(sc->t2_2, B), sc->t2_1)
#define with_list_t3(A, B, C)          (set_car(sc->t3_1, A), set_car(sc->t3_2, B), set_car(sc->t3_3, C), sc->t3_1)
#define with_list_t4(A, B, C, D)       (set_car(sc->t4_1, A), set_car(sc->t3_1, B), set_car(sc->t3_2, C), set_car(sc->t3_3, D), sc->t4_1)

#define is_string(p)                   (type(p) == T_STRING)
#define is_mutable_string(p)           ((full_type(T_Pos(p)) & (TYPE_MASK | T_IMMUTABLE)) == T_STRING)
#define string_value(p)                (T_Str(p))->object.string.svalue
#define string_length(p)               (T_Str(p))->object.string.length
#define string_hash(p)                 (T_Str(p))->object.string.hash
#define string_block(p)                (T_Str(p))->object.string.block
#define unchecked_string_block(p)      p->object.string.block

#define character(p)                   (T_Chr(p))->object.chr.c
#define is_character(p)                (type(p) == T_CHARACTER)
#define upper_character(p)             (T_Chr(p))->object.chr.up_c
#define is_char_alphabetic(p)          (T_Chr(p))->object.chr.alpha_c
#define is_char_numeric(p)             (T_Chr(p))->object.chr.digit_c
#define is_char_whitespace(p)          (T_Chr(p))->object.chr.space_c
#define is_char_uppercase(p)           (T_Chr(p))->object.chr.upper_c
#define is_char_lowercase(p)           (T_Chr(p))->object.chr.lower_c
#define character_name(p)              (T_Chr(p))->object.chr.c_name
#define character_name_length(p)       (T_Chr(p))->object.chr.length

#define optimize_op(P)                 (T_Pos(P))->tf.opts.opt_choice
#define set_optimize_op(P, Op)         (T_Pos(P))->tf.opts.opt_choice = (Op)
#define OP_HOP_MASK                    0xfffe
#define optimize_op_match(P, Q)        ((is_optimized(P)) && ((optimize_op(P) & OP_HOP_MASK) == (Q)))
#define op_no_hop(P)                   (optimize_op(P) & OP_HOP_MASK)
#define op_has_hop(P)                  ((optimize_op(P) & 1) != 0)
#define clear_optimize_op(P)           set_optimize_op(P, OP_UNOPT)
#define set_safe_optimize_op(P, Q)     do {set_optimized(P); set_optimize_op(P, Q);} while (0)
#define set_unsafe_optimize_op(P, Q)   do {set_unsafely_optimized(P); set_optimize_op(P, Q);} while (0)

#define is_symbol(p)                   (type(p) == T_SYMBOL)
#define is_normal_symbol(p)            ((is_symbol(p)) && (!is_keyword(p)))
#define is_safe_symbol(p)              ((is_symbol(p)) && (is_slot(lookup_slot_from(p, sc->curlet))))
#define symbol_name_cell(p)            T_Str((T_Sym(p))->object.sym.name)
#define symbol_set_name_cell(p, S)     (T_Sym(p))->object.sym.name = T_Str(S)
#define symbol_name(p)                 string_value(symbol_name_cell(p))
#define symbol_name_length(p)          string_length(symbol_name_cell(p))
#define gensym_block(p)                symbol_name_cell(p)->object.string.gensym_block
#define pointer_map(p)                 (s7_int)((intptr_t)(p) >> 8)
#define symbol_id(p)                   (T_Sym(p))->object.sym.id
#define symbol_set_id_unchecked(p, X)  (T_Sym(p))->object.sym.id = X
#if S7_DEBUGGING
static void symbol_set_id(s7_pointer p, s7_int id)
{
  if (id < symbol_id(p))
    {
      fprintf(stderr, "%s[%d]: id mismatch: sym: %s %" ld64 ", let: %" ld64 "\n", __func__, __LINE__, symbol_name(p), symbol_id(p), id);
      abort();
    }
  (T_Sym(p))->object.sym.id = id;
}
#else
#define symbol_set_id(p, X)            (T_Sym(p))->object.sym.id = X
#endif
/* we need 64-bits here, since we don't want this thing to wrap around, and lets are created at a great rate
 *    callgrind says this is faster than an uint32_t!
 */
#define symbol_info(p)                 (symbol_name_cell(p))->object.string.block
#define symbol_type(p)                 (block_size(symbol_info(p)) & 0xff)   /* boolean function bool type */
#define symbol_set_type(p, Type)       block_size(symbol_info(p)) = ((block_size(symbol_info(p)) & ~0xff) | (Type & 0xff))
#define symbol_clear_type(p)           block_size(symbol_info(p)) = 0
#define symbol_s7_let(p)               ((uint8_t)((block_size(symbol_info(p)) >> 8) & 0xff))  /* *s7* field id */
#define symbol_set_s7_let(p, Field)    block_size(symbol_info(p)) = ((block_size(symbol_info(p)) & ~0xff00) | ((Field & 0xff) << 8))
#define initial_slot(p)                T_Sld(symbol_info(p)->ex.ex_ptr)
#define set_initial_slot(p, Val)       symbol_info(p)->ex.ex_ptr = T_Sld(Val)
#define global_slot(p)                 T_Sld((T_Sym(p))->object.sym.global_slot)
#define set_global_slot(p, Val)        (T_Sym(p))->object.sym.global_slot = T_Sld(Val)
#define local_slot(p)                  T_Sln((T_Sym(p))->object.sym.local_slot)
#define set_local_slot(p, Val)         (T_Sym(p))->object.sym.local_slot = T_Slt(Val)

#define initial_value(p)               slot_value(initial_slot(T_Sym(p)))
#define local_value(p)                 slot_value(local_slot(T_Sym(p)))
#define unchecked_local_value(p)       local_slot(p)->object.slt.val
#define global_value(p)                slot_value(global_slot(T_Sym(p)))

#define keyword_symbol(p)              symbol_info(p)->nx.ksym               /* keyword only, so does not collide with documentation */
#define keyword_set_symbol(p, Val)     symbol_info(p)->nx.ksym = T_Sym(Val)
#define symbol_help(p)                 symbol_info(p)->nx.documentation
#define symbol_set_help(p, Doc)        symbol_info(p)->nx.documentation = Doc
#define symbol_tag(p)                  (T_Sym(p))->object.sym.tag
#define symbol_set_tag(p, Val)         (T_Sym(p))->object.sym.tag = Val
#define symbol_ctr(p)                  (T_Sym(p))->object.sym.ctr            /* needs to be in the symbol object (not symbol_info) for speed */
#define symbol_clear_ctr(p)            (T_Sym(p))->object.sym.ctr = 0
#define symbol_increment_ctr(p)        (T_Sym(p))->object.sym.ctr++
#define symbol_tag2(p)                 symbol_info(p)->ln.tag
#define symbol_set_tag2(p, Val)        symbol_info(p)->ln.tag = Val
#define symbol_has_help(p)             (is_documented(symbol_name_cell(p)))
#define symbol_set_has_help(p)         set_documented(symbol_name_cell(p))
/* symbol_info->dx is free */

#define symbol_set_local_slot_unchecked(Symbol, Id, Slot) \
  do {(Symbol)->object.sym.local_slot = T_Sln(Slot); symbol_set_id_unchecked(Symbol, Id); symbol_increment_ctr(Symbol);} while (0)
#define symbol_set_local_slot_unchecked_and_unincremented(Symbol, Id, Slot) \
  do {(Symbol)->object.sym.local_slot = T_Sln(Slot); symbol_set_id_unchecked(Symbol, Id);} while (0)
#define symbol_set_local_slot(Symbol, Id, Slot) \
  do {set_local_slot(Symbol, Slot); symbol_set_id(Symbol, Id); symbol_increment_ctr(Symbol);} while (0)
#define symbol_set_local_slot_unincremented(Symbol, Id, Slot) \
  do {set_local_slot(Symbol, Slot); symbol_set_id(Symbol, Id);} while (0)
/* set slot before id in case Slot is an expression that tries to find the current Symbol slot (using its old Id obviously) */

#define is_slot(p)                     (type(p) == T_SLOT)
#define slot_symbol(p)                 T_Sym((T_Slt(p))->object.slt.sym)
#define slot_set_symbol(p, Sym)        (T_Slt(p))->object.slt.sym = T_Sym(Sym)
#define slot_value(p)                  T_Nmv((T_Slt(p))->object.slt.val)
#define slot_set_value(p, Val)         (T_Slt(p))->object.slt.val = T_Nmv(Val)
#define slot_set_symbol_and_value(Slot, Symbol, Value) do {slot_set_symbol(Slot, Symbol); slot_set_value(Slot, Value);} while (0)
#define slot_set_value_with_hook(Slot, Value) \
  do {if (hook_has_functions(sc->rootlet_redefinition_hook)) slot_set_value_with_hook_1(sc, Slot, T_Nmv(Value)); else slot_set_value(Slot, T_Nmv(Value));} while (0)
#define next_slot(p)                   T_Sln((T_Slt(p))->object.slt.nxt)
#define slot_set_next(p, Val)          (T_Slt(p))->object.slt.nxt = T_Sln(Val)
#define slot_set_pending_value(p, Val) do {(T_Slt(p))->object.slt.pending_value = T_Nmv(Val); slot_set_has_pending_value(p);} while (0)
#define slot_simply_set_pending_value(p, Val) (T_Slt(p))->object.slt.pending_value = T_Nmv(Val)
#if S7_DEBUGGING
static s7_pointer slot_pending_value(s7_pointer p) \
  {if (slot_has_pending_value(p)) return(p->object.slt.pending_value); fprintf(stderr, "%s[%d]: slot: no pending value\n", __func__, __LINE__); abort(); return(NULL);}
static s7_pointer slot_expression(s7_pointer p)    \
  {if (slot_has_expression(p)) return(p->object.slt.expr); fprintf(stderr, "%s[%d]: slot: no expression\n", __func__, __LINE__); abort(); return(NULL);}
#else
#define slot_pending_value(p)          (T_Slt(p))->object.slt.pending_value
#define slot_expression(p)             (T_Slt(p))->object.slt.expr
#endif
#define slot_set_expression(p, Val)    do {(T_Slt(p))->object.slt.expr = T_Pos(Val); slot_set_has_expression(p);} while (0)
#define slot_just_set_expression(p, Val) (T_Slt(p))->object.slt.expr = T_Pos(Val)
#define slot_setter(p)                 T_Prc(T_Slt(p)->object.slt.expr)
#define slot_set_setter_1(p, Val)      (T_Slt(p))->object.slt.expr = T_Prc(Val)
#if S7_DEBUGGING
#define tis_slot(p) ((p) && (T_Slt(p)))
#else
#define tis_slot(p) (p) /* used for loop through let slots which end in nil, not for general slot recognition */
#endif
#define slot_end(sc) NULL
#define is_slot_end(p) (!(p))

#define is_syntax(p)                   (type(p) == T_SYNTAX)
#define syntax_symbol(p)               T_Sym((T_Syn(p))->object.syn.symbol)
#define syntax_set_symbol(p, Sym)      (T_Syn(p))->object.syn.symbol = T_Sym(Sym)
#define syntax_opcode(p)               (T_Syn(p))->object.syn.op
#define syntax_min_args(p)             (T_Syn(p))->object.syn.min_args
#define syntax_max_args(p)             (T_Syn(p))->object.syn.max_args
#define syntax_documentation(p)        (T_Syn(p))->object.syn.documentation
#define pair_set_syntax_op(p, X)       do {set_optimize_op(p, X); set_syntactic_pair(p);} while (0)
#define symbol_syntax_op_checked(p)    ((is_syntactic_pair(p)) ? optimize_op(p) : symbol_syntax_op(car(p)))
#define symbol_syntax_op(p)            syntax_opcode(global_value(p))

#define INITIAL_ROOTLET_SIZE           512
#define let_id(p)                      (T_Lid(p))->object.envr.id
#define let_set_id(p, Id)              (T_Lid(p))->object.envr.id = Id
#define is_let(p)                      (type(p) == T_LET)
#define is_let_unchecked(p)            (unchecked_type(p) == T_LET)
#define let_slots(p)                   T_Sln((T_Let(p))->object.envr.slots)
#define let_outlet(p)                  T_Lid((T_Let(p))->object.envr.nxt)
#define let_set_outlet(p, ol)          (T_Let(p))->object.envr.nxt = T_Lid(ol)
#if S7_DEBUGGING
  #define let_set_slots(p, Slot)       do {if ((!in_heap(p)) && (Slot) && (in_heap(Slot))) fprintf(stderr, "%s[%d]: let+slot mismatch\n", __func__, __LINE__); \
                                           T_Let(p)->object.envr.slots = T_Sln(Slot);} while (0)
  #define C_Let(p, role)               check_let_ref(p, role, __func__, __LINE__)
  #define S_Let(p, role)               check_let_set(p, role, __func__, __LINE__)
#else
  #define let_set_slots(p, Slot)       (T_Let(p))->object.envr.slots = T_Sln(Slot)
  #define C_Let(p, role)               p
  #define S_Let(p, role)               p
#endif
#define funclet_function(p)            T_Sym((C_Let(p, L_FUNC))->object.envr.edat.efnc.function)
#define funclet_set_function(p, F)     (S_Let(p, L_FUNC))->object.envr.edat.efnc.function = T_Sym(F)
#define set_curlet(Sc, P)              Sc->curlet = T_Lid(P)

#define let_baffle_key(p)              (T_Let(p))->object.envr.edat.key
#define set_let_baffle_key(p, K)       (T_Let(p))->object.envr.edat.key = K

#define let_line(p)                    (C_Let(p, L_FUNC))->object.envr.edat.efnc.line
#define let_set_line(p, L)             (S_Let(p, L_FUNC))->object.envr.edat.efnc.line = L
#define let_file(p)                    (C_Let(p, L_FUNC))->object.envr.edat.efnc.file
#define let_set_file(p, F)             (S_Let(p, L_FUNC))->object.envr.edat.efnc.file = F

#define let_dox_slot1(p)               T_Slt((C_Let(p, L_DOX))->object.envr.edat.dox.dox1)
#define let_set_dox_slot1(p, S)        do {(S_Let(p, L_DOX))->object.envr.edat.dox.dox1 = T_Slt(S); set_has_dox_slot1(p);} while (0)
#define let_dox_slot2(p)               T_Sld((C_Let(p, L_DOX))->object.envr.edat.dox.dox2)
#define let_set_dox_slot2(p, S)        do {(S_Let(p, L_DOX))->object.envr.edat.dox.dox2 = T_Slt(S); set_has_dox_slot2(p);} while (0)
#define let_dox_slot2_unchecked(p)     T_Sld(C_Let(p, L_DOX)->object.envr.edat.dox.dox2)
#define let_set_dox_slot2_unchecked(p, S) do {S_Let(p, L_DOX)->object.envr.edat.dox.dox2 = T_Sld(S); set_has_dox_slot2(p);} while (0)
#define let_dox1_value(p)              slot_value(let_dox_slot1(p))
#define let_dox2_value(p)              slot_value(let_dox_slot2(p))

#define unique_name(p)                 (p)->object.unq.name /* not T_Uniq(p) here -- see make_unique */
#define unique_name_length(p)          (p)->object.unq.len
#define is_unspecified(p)              (type(p) == T_UNSPECIFIED)
#define unique_car(p)                  (p)->object.unq.car
#define unique_cdr(p)                  (p)->object.unq.cdr

#define is_undefined(p)                (type(p) == T_UNDEFINED)
#define undefined_name(p)              (T_Undf(p))->object.undef.name
#define undefined_name_length(p)       (T_Undf(p))->object.undef.len
#define undefined_set_name_length(p, L) (T_Undf(p))->object.undef.len = L
#define eof_name(p)                    (T_Eof(p))->object.eof.name
#define eof_name_length(p)             (T_Eof(p))->object.eof.len

#define is_any_vector(p)               t_vector_p[type(p)]
#define is_normal_vector(p)            (type(p) == T_VECTOR)
#define vector_length(p)               (p)->object.vector.length
#define unchecked_vector_elements(p)   (p)->object.vector.elements.objects
#define unchecked_vector_element(p, i) ((p)->object.vector.elements.objects[i])
#define vector_element(p, i)           ((T_Vec(p))->object.vector.elements.objects[i])
#define vector_elements(p)             (T_Vec(p))->object.vector.elements.objects
#define vector_getter(p)               (T_Vec(p))->object.vector.vget
#define vector_setter(p)               (T_Vec(p))->object.vector.setv.vset
#define vector_block(p)                (T_Vec(p))->object.vector.block
#define unchecked_vector_block(p)      p->object.vector.block

#define typed_vector_typer(p)          T_Prc((T_Vec(p))->object.vector.setv.fset)
#define typed_vector_set_typer(p, Fnc) (T_Vec(p))->object.vector.setv.fset = T_Prc(Fnc)
#define typed_vector_gc_mark(p)        ((is_c_function(typed_vector_typer(p))) ? c_function_marker(typed_vector_typer(p)) : mark_typed_vector_1)
#define typed_vector_typer_call(sc, p, Args) \
  ((is_c_function(typed_vector_typer(p))) ? c_function_call(typed_vector_typer(p))(sc, Args) : s7_apply_function(sc, typed_vector_typer(p), Args))

#define is_int_vector(p)               (type(p) == T_INT_VECTOR)
#define int_vector(p, i)               ((T_Ivc(p))->object.vector.elements.ints[i])
#define int_vector_ints(p)             (T_Ivc(p))->object.vector.elements.ints

#define is_float_vector(p)             (type(p) == T_FLOAT_VECTOR)
#define float_vector(p, i)             ((T_Fvc(p))->object.vector.elements.floats[i])
#define float_vector_floats(p)         (T_Fvc(p))->object.vector.elements.floats

#define is_byte_vector(p)              (type(p) == T_BYTE_VECTOR)
#define byte_vector_length(p)          (T_BVc(p))->object.vector.length
#define byte_vector_bytes(p)           (T_BVc(p))->object.vector.elements.bytes
#define byte_vector(p, i)              ((T_BVc(p))->object.vector.elements.bytes[i])
#define is_string_or_byte_vector(p)    ((type(p) == T_STRING) || (type(p) == T_BYTE_VECTOR))

#define vector_dimension_info(p)       ((vdims_t *)(T_Vec(p))->object.vector.block->ex.ex_info)
#define vector_set_dimension_info(p, d) (T_Vec(p))->object.vector.block->ex.ex_info = (void  *)d
#define vector_ndims(p)                vdims_rank(vector_dimension_info(p))
#define vector_dimension(p, i)         vdims_dims(vector_dimension_info(p))[i]
#define vector_dimensions(p)           vdims_dims(vector_dimension_info(p))
#define vector_offset(p, i)            vdims_offsets(vector_dimension_info(p))[i]
#define vector_offsets(p)              vdims_offsets(vector_dimension_info(p))
#define vector_rank(p)                 ((vector_dimension_info(p)) ? vector_ndims(p) : 1)
#define vector_has_dimension_info(p)   (vector_dimension_info(p))

#define subvector_vector(p)            T_Vec(((vector_dimension_info(T_SVec(p))) ? vdims_original(vector_dimension_info(p)) : (p)->object.vector.block->nx.ksym))
#define subvector_set_vector(p, vect)  (T_SVec(p))->object.vector.block->nx.ksym = T_Vec(vect)

#define rootlet_element(p, i)          unchecked_vector_element(p, i)
#define rootlet_elements(p)            unchecked_vector_elements(p)
#define rootlet_block(p)               unchecked_vector_block(p)

#define stack_element(p, i)            unchecked_vector_element(T_Stk(p), i)
#define stack_elements(p)              unchecked_vector_elements(T_Stk(p))
#define stack_block(p)                 unchecked_vector_block(T_Stk(p))
#define current_stack_top(Sc)          ((Sc)->stack_end - (Sc)->stack_start)
#define temp_stack_top(p)              (T_Stk(p))->object.stk.top
/* #define stack_flags(p)              (T_Stk(p))->object.stk.flags */
#define stack_clear_flags(p)           (T_Stk(p))->object.stk.flags = 0
#define stack_has_pairs(p)             (((T_Stk(p))->object.stk.flags & 1) != 0)
#define stack_set_has_pairs(p)         (T_Stk(p))->object.stk.flags |= 1
#define stack_has_counters(p)          (((T_Stk(p))->object.stk.flags & 2) != 0)
#define stack_set_has_counters(p)      (T_Stk(p))->object.stk.flags |= 2

#define is_hash_table(p)               (type(p) == T_HASH_TABLE)
#define is_mutable_hash_table(p)       ((full_type(T_Pos(p)) & (TYPE_MASK | T_IMMUTABLE)) == T_HASH_TABLE)
#define hash_table_mask(p)             (T_Hsh(p))->object.hasher.mask
#define hash_table_block(p)            (T_Hsh(p))->object.hasher.block
#define unchecked_hash_table_block(p)  p->object.hasher.block
#define hash_table_set_block(p, b)     (T_Hsh(p))->object.hasher.block = b
#define hash_table_element(p, i)       (T_Hsh(p))->object.hasher.elements[i]
#define hash_table_elements(p)         (T_Hsh(p))->object.hasher.elements /* block data (dx) */
#define hash_table_entries(p)          hash_table_block(p)->nx.nx_int
#define hash_table_checker(p)          (T_Hsh(p))->object.hasher.hash_func
#define hash_table_mapper(p)           (T_Hsh(p))->object.hasher.loc
#define hash_table_checker_locked(p)   (hash_table_mapper(p) != default_hash_map)
#define hash_table_procedures(p)       T_Lst(hash_table_block(p)->ex.ex_ptr)
#define hash_table_set_procedures(p, Lst)  hash_table_block(p)->ex.ex_ptr = T_Lst(Lst)
#define hash_table_procedures_checker(p)   car(hash_table_procedures(p))
#define hash_table_procedures_mapper(p)    cdr(hash_table_procedures(p))
#define hash_table_set_procedures_mapper(p, f) set_cdr(hash_table_procedures(p), f)
#define hash_table_key_typer(p)            T_Prc(opt1_any(hash_table_procedures(p)))
#define hash_table_set_key_typer(p, Fnc)   set_opt1_any(p, T_Prc(Fnc))
#define hash_table_value_typer(p)          T_Prc(opt2_any(hash_table_procedures(p)))
#define hash_table_set_value_typer(p, Fnc) set_opt2_any(p, T_Prc(Fnc))
#define weak_hash_iters(p)                 hash_table_block(p)->ln.tag

#if S7_DEBUGGING
#define T_Itr_Pos(p)                   titr_pos(sc, T_Itr(p), __func__, __LINE__)
#define T_Itr_Len(p)                   titr_len(sc, T_Itr(p), __func__, __LINE__)
#define T_Itr_Hash(p)                  titr_hash(sc, T_Itr(p), __func__, __LINE__)
#define T_Itr_Let(p)                   titr_let(sc, T_Itr(p), __func__, __LINE__)
#define T_Itr_Pair(p)                  titr_pair(sc, T_Itr(p), __func__, __LINE__)
#else
#define T_Itr_Pos(p)                   p
#define T_Itr_Len(p)                   p
#define T_Itr_Hash(p)                  p
#define T_Itr_Let(p)                   p
#define T_Itr_Pair(p)                  p
#endif

#define is_iterator(p)                 (type(p) == T_ITERATOR)
#define iterator_sequence(p)           (T_Itr(p))->object.iter.obj
#define iterator_position(p)           (T_Itr_Pos(p))->object.iter.lc.loc
#define iterator_length(p)             (T_Itr_Len(p))->object.iter.lw.len
#define iterator_next(p)               (T_Itr(p))->object.iter.next
#define iterator_is_at_end(p)          (!iter_ok(p))                                /* ((full_type(T_Itr(p)) & T_ITER_OK) == 0) */
#define iterator_slow(p)               T_Lst((T_Itr_Pair(p))->object.iter.lw.slow)
#define iterator_set_slow(p, Val)      (T_Itr_Pair(p))->object.iter.lw.slow = T_Lst(Val)
#define iterator_hash_current(p)       (T_Itr_Hash(p))->object.iter.lw.hcur
#define iterator_current(p)            (T_Itr(p))->object.iter.cur
#define iterator_current_slot(p)       T_Sln((T_Itr_Let(p))->object.iter.lc.lcur)
#define iterator_set_current_slot(p, Val) (T_Itr_Let(p))->object.iter.lc.lcur = T_Sln(Val)
#define iterator_let_cons(p)           (T_Itr_Let(p))->object.iter.cur

#define ITERATOR_END                   eof_object
#define ITERATOR_END_NAME              "#<eof>"

#define is_input_port(p)               (type(p) == T_INPUT_PORT)
#define is_output_port(p)              (type(p) == T_OUTPUT_PORT)
#define port_port(p)                   (T_Prt(p))->object.prt.port
#define is_string_port(p)              (port_type(p) == STRING_PORT)
#define is_file_port(p)                (port_type(p) == FILE_PORT)
#define is_function_port(p)            (port_type(p) == FUNCTION_PORT)
#define port_filename_block(p)         port_port(p)->filename_block
#define port_filename(p)               port_port(p)->filename
#define port_filename_length(p)        port_port(p)->filename_length
#define port_file(p)                   port_port(p)->file
#define port_data_block(p)             port_port(p)->block
#define unchecked_port_data_block(p)   p->object.prt.port->block
#define port_line_number(p)            port_port(p)->line_number
#define port_file_number(p)            port_port(p)->file_number
#define port_data(p)                   (T_Prt(p))->object.prt.data
#define port_data_size(p)              (T_Prt(p))->object.prt.size
#define port_position(p)               (T_Prt(p))->object.prt.point
#define port_block(p)                  (T_Prt(p))->object.prt.block
#define port_type(p)                   port_port(p)->ptype
#define port_is_closed(p)              port_port(p)->is_closed
#define port_set_closed(p, Val)        port_port(p)->is_closed = Val /* this can't be a type bit because sweep checks it after the type has been cleared */
#define port_needs_free(p)             port_port(p)->needs_free
#define port_next(p)                   port_block(p)->nx.next
#define port_output_function(p)        port_port(p)->output_function /* these two are for function ports */
#define port_input_function(p)         port_port(p)->input_function
#define port_string_or_function(p)     port_port(p)->orig_str
#define port_set_string_or_function(p, S) port_port(p)->orig_str = S

#define current_input_port(Sc)         Sc->input_port
#define set_current_input_port(Sc, P)  Sc->input_port = P
#define current_output_port(Sc)        Sc->output_port
#define set_current_output_port(Sc, P) Sc->output_port = P

#define port_read_character(p)         port_port(p)->pf->read_character
#define port_read_line(p)              port_port(p)->pf->read_line
#define port_display(p)                port_port(p)->pf->displayer
#define port_write_character(p)        port_port(p)->pf->write_character
#define port_write_string(p)           port_port(p)->pf->write_string
#define port_read_semicolon(p)         port_port(p)->pf->read_semicolon
#define port_read_white_space(p)       port_port(p)->pf->read_white_space
#define port_read_name(p)              port_port(p)->pf->read_name
#define port_read_sharp(p)             port_port(p)->pf->read_sharp
#define port_close(p)                  port_port(p)->pf->close_port

#define is_c_function(f)               (type(f) >= T_C_FUNCTION)
#define is_c_function_star(f)          (type(f) == T_C_FUNCTION_STAR)
#define is_any_c_function(f)           (type(f) >= T_C_FUNCTION_STAR)
#define c_function_data(f)             (T_Fnc(f))->object.fnc.c_proc
#define c_function_call(f)             (T_Fnc(f))->object.fnc.ff
#define c_function_min_args(f)         (T_Fnc(f))->object.fnc.required_args
#define c_function_optional_args(f)    (T_Fnc(f))->object.fnc.optional_args
#define c_function_max_args(f)         (T_Fnc(f))->object.fnc.all_args
#define c_function_is_aritable(f, N)   ((c_function_min_args(f) <= N) && (c_function_max_args(f) >= N))
#define c_function_name(f)             c_function_data(f)->name
#define c_function_name_length(f)      c_function_data(f)->name_length
#define c_function_documentation(f)    c_function_data(f)->doc
#define c_function_signature(f)        c_function_data(f)->signature
#define c_function_setter(f)           T_Prc(c_function_data(f)->setter)
#define c_function_set_setter(f, Val)  c_function_data(f)->setter = T_Prc(Val)
#define c_function_class(f)            c_function_data(f)->id
#define c_function_chooser(f)          c_function_data(f)->chooser
#define c_function_base(f)             T_Fnc(c_function_data(f)->generic_ff)
#define c_function_set_base(f, Val)    c_function_data(f)->generic_ff = T_Fnc(Val)
#define c_function_marker(f)           c_function_data(f)->cam.marker              /* the mark function for the vector (mark_vector_1 etc) */
#define c_function_set_marker(f, Val)  c_function_data(f)->cam.marker = Val
#define c_function_symbol(f)           c_function_data(f)->sam.c_sym

#define c_function_bool_setter(f)      c_function_data(f)->dam.bool_setter
#define c_function_set_bool_setter(f, Val) c_function_data(f)->dam.bool_setter = Val

#define c_function_arg_defaults(f)     c_function_data(T_Fst(f))->dam.arg_defaults
#define c_function_call_args(f)        c_function_data(T_Fst(f))->cam.call_args
#define c_function_arg_names(f)        c_function_data(T_Fst(f))->sam.arg_names

#define set_c_function(X, f)           do {set_opt1_cfunc(X, f); set_fn_direct(X, c_function_call(f));} while (0)
#define c_function_opt_data(f)         c_function_data(f)->opt_data

#define is_c_macro(p)                  (type(p) == T_C_MACRO)
#define c_macro_data(f)                (T_CMac(f))->object.fnc.c_proc
#define c_macro_call(f)                (T_CMac(f))->object.fnc.ff
#define c_macro_name(f)                c_macro_data(f)->name
#define c_macro_name_length(f)         c_macro_data(f)->name_length
#define c_macro_min_args(f)       (T_CMac(f))->object.fnc.required_args
#define c_macro_max_args(f)            (T_CMac(f))->object.fnc.all_args
#define c_macro_setter(f)              T_Prc(c_macro_data(f)->setter)
#define c_macro_set_setter(f, Val)     c_macro_data(f)->setter = T_Prc(Val)

#define is_random_state(p)             (type(p) == T_RANDOM_STATE)
#define random_gmp_state(p)            (p)->object.rng.state /* sweep sees free cell in big_random_state gc_list and needs to call gmprandclear on its value */
#define random_seed(p)                 (T_Ran(p))->object.rng.seed
#define random_carry(p)                (T_Ran(p))->object.rng.carry

#define continuation_block(p)          (T_Con(p))->object.cwcc.block
#define continuation_stack(p)          T_Stk(T_Con(p)->object.cwcc.stack)
#define continuation_set_stack(p, Val) (T_Con(p))->object.cwcc.stack = T_Stk(Val)
#define continuation_stack_end(p)      (T_Con(p))->object.cwcc.stack_end
#define continuation_stack_start(p)    (T_Con(p))->object.cwcc.stack_start
#define continuation_stack_top(p)      (continuation_stack_end(p) - continuation_stack_start(p))
#define continuation_op_stack(p)       (T_Con(p))->object.cwcc.op_stack
#define continuation_stack_size(p)     continuation_block(p)->nx.ix.i1
#define continuation_op_loc(p)         continuation_block(p)->nx.ix.i2
#define continuation_op_size(p)        continuation_block(p)->ln.tag
#define continuation_key(p)            continuation_block(p)->ex.ckey
/* this can overflow int32_t -- baffle_key is s7_int, so ckey should be also */
#define continuation_name(p)           continuation_block(p)->dx.d_ptr

#define call_exit_goto_loc(p)          (T_Got(p))->object.rexit.goto_loc
#define call_exit_op_loc(p)            (T_Got(p))->object.rexit.op_stack_loc
#define call_exit_active(p)            (T_Got(p))->object.rexit.active
#define call_exit_name(p)              (T_Got(p))->object.rexit.name

#define is_continuation(p)             (type(p) == T_CONTINUATION)
#define is_goto(p)                     (type(p) == T_GOTO)
#define is_macro(p)                    (type(p) == T_MACRO)
#define is_macro_star(p)               (type(p) == T_MACRO_STAR)
#define is_bacro(p)                    (type(p) == T_BACRO)
#define is_bacro_star(p)               (type(p) == T_BACRO_STAR)
#define is_either_macro(p)             ((is_macro(p)) || (is_macro_star(p)))
#define is_either_bacro(p)             ((type(p) == T_BACRO) || (type(p) == T_BACRO_STAR))

#define is_closure(p)                  (type(p) == T_CLOSURE)
#define is_closure_star(p)             (type(p) == T_CLOSURE_STAR)
#define closure_args(p)                T_Arg((T_Clo(p))->object.func.args)
#define closure_set_args(p, Val)       (T_Clo(p))->object.func.args = T_Arg(Val)
#define closure_body(p)                (T_Pair((T_Clo(p))->object.func.body))
#define closure_set_body(p, Val)       (T_Clo(p))->object.func.body = T_Pair(Val)
#define closure_let(p)                 T_Lid((T_Clo(p))->object.func.env)
#define closure_set_let(p, L)          (T_Clo(p))->object.func.env = T_Lid(L)
#define closure_arity(p)               (T_Clo(p))->object.func.arity
#define closure_set_arity(p, A)        (T_Clo(p))->object.func.arity = A

#define closure_setter(p)              (T_Prc((T_Clo(p))->object.func.setter))
#define closure_set_setter(p, Val)     (T_Clo(p))->object.func.setter = T_Prc(Val)
#define closure_map_list(p)            (T_Pair((T_Clo(p))->object.func.setter))
#define closure_set_map_list(p, Val)   (T_Clo(p))->object.func.setter = T_Pair(Val)
#define closure_setter_or_map_list(p)  (T_Clo(p)->object.func.setter)
/* closure_map_list refers to a cyclic list detector in map; since in this case map makes a new closure for its own use,
 *   closure_map_list doesn't collide with closure_setter.
 */

#define CLOSURE_ARITY_NOT_SET          0x40000000
#define MAX_ARITY                      0x20000000
#define closure_arity_unknown(p)       (closure_arity(p) == CLOSURE_ARITY_NOT_SET)
#define is_thunk(Sc, Fnc)              ((type(Fnc) >= T_GOTO) && (s7_is_aritable(Sc, Fnc, 0)))

#define hook_has_functions(p)          (is_pair(s7_hook_functions(sc, T_Clo(p))))

#define catch_tag(p)                   (T_Cat(p))->object.rcatch.tag
#define catch_goto_loc(p)              (T_Cat(p))->object.rcatch.goto_loc
#define catch_op_loc(p)                (T_Cat(p))->object.rcatch.op_stack_loc
#define catch_cstack(p)                (T_Cat(p))->object.rcatch.cstack
#define catch_handler(p)               T_Pos((T_Cat(p))->object.rcatch.handler)
#define catch_set_handler(p, val)      (T_Cat(p))->object.rcatch.handler = T_Pos(val)

#define dynamic_wind_state(p)          (T_Dyn(p))->object.winder.state
#define dynamic_wind_in(p)             (T_Dyn(p))->object.winder.in
#define dynamic_wind_out(p)            (T_Dyn(p))->object.winder.out
#define dynamic_wind_body(p)           (T_Dyn(p))->object.winder.body

#define is_c_object(p)                 (type(p) == T_C_OBJECT)
#define c_object_value(p)              (T_Obj(p))->object.c_obj.value
#define c_object_type(p)               (T_Obj(p))->object.c_obj.type
#define c_object_let(p)                T_Lid((T_Obj(p))->object.c_obj.e)
#define c_object_set_let(p, L)         (T_Obj(p))->object.c_obj.e = T_Lid(L)
#define c_object_s7(p)                 (T_Obj(p))->object.c_obj.sc

#define c_object_info(Sc, p)           Sc->c_object_types[c_object_type(T_Obj(p))]
#define c_object_free(Sc, p)           c_object_info(Sc, p)->free
#define c_object_mark(Sc, p)           c_object_info(Sc, p)->mark
#define c_object_gc_mark(Sc, p)        c_object_info(Sc, p)->gc_mark
#define c_object_gc_free(Sc, p)        c_object_info(Sc, p)->gc_free
#define c_object_ref(Sc, p)            c_object_info(Sc, p)->ref
#define c_object_getf(Sc, p)           c_object_info(Sc, p)->getter
#define c_object_set(Sc, p)            c_object_info(Sc, p)->set
#define c_object_setf(Sc, p)           c_object_info(Sc, p)->setter
#if (!DISABLE_DEPRECATED)
  #define c_object_print(Sc, p)        c_object_info(Sc, p)->print
#endif
#define c_object_len(Sc, p)            c_object_info(Sc, p)->length
#define c_object_eql(Sc, p)            c_object_info(Sc, p)->eql
#define c_object_equal(Sc, p)          c_object_info(Sc, p)->equal
#define c_object_equivalent(Sc, p)     c_object_info(Sc, p)->equivalent
#define c_object_fill(Sc, p)           c_object_info(Sc, p)->fill
#define c_object_copy(Sc, p)           c_object_info(Sc, p)->copy
#define c_object_reverse(Sc, p)        c_object_info(Sc, p)->reverse
#define c_object_to_list(Sc, p)        c_object_info(Sc, p)->to_list
#define c_object_to_string(Sc, p)      c_object_info(Sc, p)->to_string
#define c_object_scheme_name(Sc, p)    T_Str(c_object_info(Sc, p)->scheme_name)

#define c_pointer(p)                   (T_Ptr(p))->object.cptr.c_pointer
#define c_pointer_type(p)              (T_Ptr(p))->object.cptr.c_type
#define c_pointer_info(p)              (T_Ptr(p))->object.cptr.info
#define c_pointer_weak1(p)             (T_Ptr(p))->object.cptr.weak1
#define c_pointer_weak2(p)             (T_Ptr(p))->object.cptr.weak2
#define c_pointer_set_weak1(p, q)      (T_Ptr(p))->object.cptr.weak1 = T_Pos(q)
#define c_pointer_set_weak2(p, q)      (T_Ptr(p))->object.cptr.weak2 = T_Pos(q)
#define is_c_pointer(p)                (type(p) == T_C_POINTER)

#define is_counter(p)                  (type(p) == T_COUNTER)
#define counter_result(p)              (T_Ctr(p))->object.ctr.result
#define counter_set_result(p, Val)     (T_Ctr(p))->object.ctr.result = T_Pos(Val)
#define counter_list(p)                (T_Ctr(p))->object.ctr.list
#define counter_set_list(p, Val)       (T_Ctr(p))->object.ctr.list = T_Pos(Val)
#define counter_capture(p)             (T_Ctr(p))->object.ctr.cap
#define counter_set_capture(p, Val)    (T_Ctr(p))->object.ctr.cap = Val
#define counter_let(p)                 T_Lid((T_Ctr(p))->object.ctr.env)
#define counter_set_let(p, L)          (T_Ctr(p))->object.ctr.env = T_Lid(L)
#define counter_slots(p)               T_Sln(T_Ctr(p)->object.ctr.slots)
#define counter_set_slots(p, Val)      (T_Ctr(p))->object.ctr.slots = T_Sln(Val)

#if __cplusplus && HAVE_COMPLEX_NUMBERS
  using namespace std;                /* the code has to work in C as well as C++, so we can't scatter std:: all over the place */
  typedef complex<s7_double> s7_complex;
  static s7_double Real(complex<s7_double> x) {return(real(x));} /* protect the C++ name */
  static s7_double Imag(complex<s7_double> x) {return(imag(x));}
#endif

#define integer(p)                     (T_Int(p))->object.number.integer_value
#define set_integer(p, x)              integer(p) = x
#define real(p)                        (T_Rel(p))->object.number.real_value
#define set_real(p, x)                 real(p) = x
#define numerator(p)                   (T_Frc(p))->object.number.fraction_value.numerator
#define denominator(p)                 (T_Frc(p))->object.number.fraction_value.denominator
#define fraction(p)                    (((long_double)numerator(p)) / ((long_double)denominator(p)))
#define inverted_fraction(p)           (((long_double)denominator(p)) / ((long_double)numerator(p)))
#define real_part(p)                   (T_Cmp(p))->object.number.complex_value.rl
#define set_real_part(p, x)            real_part(p) = x
#define imag_part(p)                   (T_Cmp(p))->object.number.complex_value.im
#define set_imag_part(p, x)            imag_part(p) = x
#if HAVE_COMPLEX_NUMBERS
  #define to_c_complex(p)              CMPLX(real_part(p), imag_part(p))
#endif

#if WITH_GMP
#define big_integer(p)                 ((T_Bgi(p))->object.number.bgi->n)
#define big_integer_nxt(p)             (p)->object.number.bgi->nxt
#define big_integer_bgi(p)             (p)->object.number.bgi
#define big_ratio(p)                   ((T_Bgf(p))->object.number.bgr->q)
#define big_ratio_nxt(p)               (p)->object.number.bgr->nxt
#define big_ratio_bgr(p)               (p)->object.number.bgr
#define big_real(p)                    ((T_Bgr(p))->object.number.bgf->x)
#define big_real_nxt(p)                (p)->object.number.bgf->nxt
#define big_real_bgf(p)                (p)->object.number.bgf
#define big_complex(p)                 ((T_Bgz(p))->object.number.bgc->z)
#define big_complex_nxt(p)             (p)->object.number.bgc->nxt
#define big_complex_bgc(p)             (p)->object.number.bgc
#endif

#if S7_DEBUGGING
static void set_type_1(s7_pointer p, uint64_t f, const char *func, int32_t line)
{
  p->previous_alloc_line = p->current_alloc_line;
  p->previous_alloc_func = p->current_alloc_func;
  p->previous_alloc_type = p->current_alloc_type;
  p->current_alloc_line = line;
  p->current_alloc_func = func;
  p->current_alloc_type = f;
  p->explicit_free_line = 0;
  p->uses++;
  if (((f) & TYPE_MASK) == T_FREE)
    fprintf(stderr, "%d: set free, %p type to %" PRIx64 "\n", __LINE__, p, (int64_t)(f));
  else
    if (((f) & TYPE_MASK) >= NUM_TYPES)
      fprintf(stderr, "%d: set invalid type, %p type to %" PRIx64 "\n", __LINE__, p, (int64_t)(f));
    else
      {
	if (((full_type(p) & T_IMMUTABLE) != 0) && ((full_type(p) != (uint64_t)(f))))
	  {
	    fprintf(stderr, "%s[%d]: set immutable %p type %d to %" ld64 "\n", __func__, __LINE__, p, unchecked_type(p), (int64_t)(f));
	    abort();
	  }
	if (((full_type(p) & T_UNHEAP) != 0) && (((f) & T_UNHEAP) == 0))
	  fprintf(stderr, "%s[%d]: clearing unheap in set type!\n", __func__, __LINE__);
      }
  full_type(p) = f;
}
#endif

#define number_name(p)                 (char *)((T_Num(p))->object.number_name.name + 1)
#define number_name_length(p)          (T_Num(p))->object.number_name.name[0]

static void set_number_name(s7_pointer p, const char *name, int32_t len)
{
  /* if no number name: teq +110 tread +30 tform +90 */
  if ((len >= 0) && (len < NUMBER_NAME_SIZE) && (!is_mutable_number(p)))
    {
      set_has_number_name(p);
      number_name_length(p) = (uint8_t)len;
      memcpy((void *)number_name(p), (void *)name, len);
      (number_name(p))[len] = 0;
    }
}

static s7_int s7_int_min = 0;
static int32_t s7_int_digits_by_radix[17];

#define S7_INT_BITS 63

#define S7_INT64_MAX 9223372036854775807LL
#define S7_INT64_MIN (int64_t)(-S7_INT64_MAX - 1LL)

#define S7_INT32_MAX 2147483647LL
#define S7_INT32_MIN (-S7_INT32_MAX - 1LL)

static void init_int_limits(void)
{
#if WITH_GMP
  #define S7_LOG_INT64_MAX 36.736800
#else
  /* actually not safe = (log (- (expt 2 63) 1)) and (log (- (expt 2 31) 1)) (using 63 and 31 bits) */
  #define S7_LOG_INT64_MAX 43.668274
#endif
  s7_int_min = S7_INT64_MIN; /* see comment in s7_make_ratio -- we're trying to hack around a gcc bug (9.2.1 Ubuntu) */
  s7_int_digits_by_radix[0] = 0;
  s7_int_digits_by_radix[1] = 0;
  for (int32_t i = 2; i < 17; i++)
    s7_int_digits_by_radix[i] = (int32_t)(floor(S7_LOG_INT64_MAX / log((double)i)));
}

static s7_pointer make_permanent_integer(s7_int i)
{
  s7_pointer p = (s7_pointer)Calloc(1, sizeof(s7_cell)); /* Calloc to clear name */
  full_type(p) = T_IMMUTABLE | T_INTEGER | T_UNHEAP;
  integer(p) = i;
  return(p);
}

#define NUM_CHARS 256

#ifndef NUM_SMALL_INTS
  #define NUM_SMALL_INTS 8192
  /* 65536: tshoot -6, tvect -50, dup -26, trclo -27, tmap -48, tsort -14, tlet -16, trec -58, thash -40 */
#else
#if (NUM_SMALL_INTS < NUM_CHARS) /* g_char_to_integer assumes this is at least NUM_CHARS, as does the byte_vector stuff (256) */
  #error NUM_SMALL_INTS is less than NUM_CHARS which will not work
#endif
#endif

static s7_pointer *small_ints = NULL;
#define small_int(Val) small_ints[Val]
#define is_small_int(n) ((n & ~(NUM_SMALL_INTS - 1)) == 0)                 /* ((n >= 0) && (n < NUM_SMALL_INTS)) is slower */

static s7_pointer real_zero, real_NaN, complex_NaN, real_pi, real_one, arity_not_set, max_arity, real_infinity, real_minus_infinity;
static s7_pointer int_zero, int_one, int_two, int_three, minus_one, minus_two, mostfix, leastfix;

static void init_small_ints(void)
{
  const char *ones[10] = {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9"};
  s7_cell *cells = (s7_cell *)Malloc(NUM_SMALL_INTS * sizeof(s7_cell)); /* was calloc 14-Apr-22 */
  small_ints = (s7_pointer *)Malloc(NUM_SMALL_INTS * sizeof(s7_pointer));
  for (int32_t i = 0; i < NUM_SMALL_INTS; i++)
    {
      s7_pointer p;
      small_ints[i] = &cells[i];
      p = small_ints[i];
      full_type(p) = T_IMMUTABLE | T_INTEGER | T_UNHEAP;
      integer(p) = i;
    }
  for (int32_t i = 0; i < 10; i++)
    set_number_name(small_ints[i], ones[i], 1);

  /* setup a few other numbers while we're here */
  #define EXTRA_NUMBERS 11
  cells = (s7_cell *)Calloc(EXTRA_NUMBERS, sizeof(s7_cell));

  #define init_integer(Ptr, Num, Name, Name_Len) \
    do {set_full_type(Ptr, T_INTEGER | T_IMMUTABLE | T_UNHEAP); set_integer(Ptr, Num); set_number_name(Ptr, Name, Name_Len);} while (0)
  #define init_integer_no_name(Ptr, Num) \
    do {set_full_type(Ptr, T_INTEGER | T_IMMUTABLE | T_UNHEAP); set_integer(Ptr, Num);} while (0)

  #define init_real(Ptr, Num, Name, Name_Len) \
    do {set_full_type(Ptr, T_REAL | T_IMMUTABLE | T_UNHEAP); set_real(Ptr, Num); set_number_name(Ptr, Name, Name_Len);} while (0)
  #define init_real_no_name(Ptr, Num) \
    do {set_full_type(Ptr, T_REAL | T_IMMUTABLE | T_UNHEAP); set_real(Ptr, Num);} while (0)

  #define init_complex(Ptr, Real, Imag, Name, Name_Len) \
    do {set_full_type(Ptr, T_COMPLEX | T_IMMUTABLE | T_UNHEAP); set_real_part(Ptr, Real); set_imag_part(Ptr, Imag); set_number_name(Ptr, Name, Name_Len);} while (0)

  real_zero = &cells[0]; init_real(real_zero, 0.0, "0.0", 3);
  real_one = &cells[1]; init_real(real_one, 1.0, "1.0", 3);
  real_NaN = &cells[2]; init_real(real_NaN, NAN, "+nan.0", 6);
  complex_NaN = &cells[10]; init_complex(complex_NaN, NAN, NAN, "+nan.0+nan.0i", 13);
  real_infinity = &cells[3]; init_real(real_infinity, INFINITY, "+inf.0", 6);
  real_minus_infinity = &cells[4]; init_real(real_minus_infinity, -INFINITY, "-inf.0", 6);
  real_pi = &cells[5]; init_real_no_name(real_pi, 3.1415926535897932384626433832795029L);

  arity_not_set = &cells[6]; init_integer_no_name(arity_not_set, CLOSURE_ARITY_NOT_SET);
  max_arity = &cells[7]; init_integer_no_name(max_arity, MAX_ARITY);
  minus_one = &cells[8]; init_integer(minus_one, -1, "-1", 2);
  minus_two = &cells[9]; init_integer(minus_two, -2, "-2", 2);
  int_zero = small_ints[0];
  int_one = small_ints[1];
  int_two = small_ints[2];
  int_three = small_ints[3];

  mostfix = make_permanent_integer(S7_INT64_MAX);
  leastfix = make_permanent_integer(s7_int_min);
  set_number_name(mostfix, "9223372036854775807", 19);
  set_number_name(leastfix, "-9223372036854775808", 20);
}

/* -------------------------------------------------------------------------------- */
#if (defined(__FreeBSD__)) || ((defined(__linux__)) && (__GLIBC__ >= 2) && (__GLIBC_MINOR__ > 17)) || (defined(__OpenBSD__)) || (defined(__NetBSD__))
  static inline s7_int my_clock(void)
  {
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    /* coarse: 0.057u 0.007s, monotonic: 0.083u 0.007s, clock(): 0.624u 0.372s -- coarse since Linux 2.6.32, glibc > 2.17
     *   FreeBSD has CLOCK_MONOTONIC_FAST in place of COARSE, OpenBSD and netBSD have neither
     *   clock_getres places 1 in tv_nsec in linux, so I assume I divide billion/tv_nsec
     *   MacOSX has clock_get_time, and after Sierra 10.12 has clock_gettime
     *     apparently we include /usr/include/AvailabilityMacros.h, then #if MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
     *   Windows has QueryPerformanceCounter or something
     * maybe just check for POSIX compatibility?
     */
    return(ts.tv_sec * 1000000000 + ts.tv_nsec); /* accumulated into s7_int so this should be ok: s7.h gives it 64 bits */
  }

  static s7_int ticks_per_second(void)
  {
    struct timespec ts;
    clock_getres(CLOCK_MONOTONIC, &ts);
    return((ts.tv_nsec == 0) ? 1000000000 : (1000000000 / ts.tv_nsec));
  }
#else
  #define my_clock clock
  #define ticks_per_second() CLOCKS_PER_SEC
#endif

#ifndef GC_TRIGGER_SIZE
  #define GC_TRIGGER_SIZE 64
#endif

#if S7_DEBUGGING
static void try_to_call_gc_1(s7_scheme *sc, const char *func, int32_t line);
#define try_to_call_gc(Sc) try_to_call_gc_1(Sc, __func__, __LINE__)
#else
static void try_to_call_gc(s7_scheme *sc);
#endif

#define GC_STATS 1
#define HEAP_STATS 2
#define STACK_STATS 4
#define PROTECTED_OBJECTS_STATS 8

#define show_gc_stats(Sc)                ((Sc->gc_stats & GC_STATS) != 0)
#define show_stack_stats(Sc)             ((Sc->gc_stats & STACK_STATS) != 0)
#define show_heap_stats(Sc)              ((Sc->gc_stats & HEAP_STATS) != 0)
#define show_protected_objects_stats(Sc) ((Sc->gc_stats & PROTECTED_OBJECTS_STATS) != 0)


/* new_cell has to include the new cell's type.  In the free list, it is 0 (T_FREE).  If we remove it here,
 *   but then hit some error before setting the type, the GC sweep thinks it is a free cell already and
 *   does not return it to the free list: a memory leak.
 */
#if (!S7_DEBUGGING)
#define new_cell(Sc, Obj, Type)			\
  do {						\
    if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc); \
    Obj = (*(--(Sc->free_heap_top))); \
    set_full_type(Obj, Type);	      \
    } while (0)

#define new_cell_no_check(Sc, Obj, Type) do {Obj = (*(--(Sc->free_heap_top))); set_full_type(Obj, Type);} while (0)
  /* since sc->free_heap_trigger is GC_TRIGGER_SIZE above the free heap base, we don't need
   *   to check it repeatedly after the first such check.
   */
#else

#define new_cell(Sc, Obj, Type)						\
  do {									\
    if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc); \
    Obj = (*(--(Sc->free_heap_top)));					\
    Obj->debugger_bits = 0; Obj->gc_func = NULL; Obj->gc_line = 0;	\
    set_full_type(Obj, Type);						\
  } while (0)

#define new_cell_no_check(Sc, Obj, Type)				\
  do {									\
    Obj = (*(--(Sc->free_heap_top)));					\
    if (Sc->free_heap_top < Sc->free_heap) {fprintf(stderr, "%s[%d]: free heap exhausted\n", __func__, __LINE__); abort();}\
    Obj->debugger_bits = 0; Obj->gc_func = NULL; Obj->gc_line = 0;	\
    set_full_type(Obj, Type);						\
    } while (0)
#endif

/* #define gc_if_at_trigger(Sc) if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc) */

#if WITH_GCC
#define make_integer(Sc, N) \
  ({ s7_int _N_; _N_ = (N); (is_small_int(_N_) ? small_int(_N_) : ({ s7_pointer _I_; new_cell(Sc, _I_, T_INTEGER); integer(_I_) = _N_; _I_;}) ); })

#define make_real(Sc, X) ({ s7_pointer _R_; s7_double _N_ = (X); new_cell(Sc, _R_, T_REAL); set_real(_R_, _N_); _R_;})
#define make_real_unchecked(Sc, X) ({ s7_pointer _R_; s7_double _N_ = (X); new_cell_no_check(Sc, _R_, T_REAL); set_real(_R_, _N_); _R_;})

#define make_complex_not_0i(Sc, R, I) ({ s7_pointer _C_; new_cell(Sc, _C_, T_COMPLEX); set_real_part(_C_, R); set_imag_part(_C_, I); _C_;})
#define make_complex(Sc, R, I) \
  ({ s7_double _im_; _im_ = (I); ((_im_ == 0.0) ? make_real(Sc, R) : \
				  ({ s7_pointer _C_; new_cell(Sc, _C_, T_COMPLEX); set_real_part(_C_, R); set_imag_part(_C_, _im_); _C_;}) ); })

#define real_to_double(Sc, X, Caller) ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_REAL) ? real(_x_) : s7_number_to_real_with_caller(Sc, _x_, Caller)); })
#define rational_to_double(Sc, X)     ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_INTEGER) ? (s7_double)integer(_x_) : fraction(_x_)); })

#else

#define make_integer(Sc, N)           s7_make_integer(Sc, N)
#define make_real(Sc, X)              s7_make_real(Sc, X)
#define make_real_unchecked(Sc, X)    s7_make_real(Sc, X)
#define make_complex(Sc, R, I)        s7_make_complex(Sc, R, I)
#define make_complex_not_0i(Sc, R, I) s7_make_complex(Sc, R, I)
#define real_to_double(Sc, X, Caller) s7_number_to_real_with_caller(Sc, X, Caller)
#define rational_to_double(Sc, X)     s7_number_to_real(Sc, X)
#endif

static s7_pointer wrap_integer(s7_scheme *sc, s7_int x)
{
  s7_pointer p;
  if (is_small_int(x)) return(small_int(x));
  p = car(sc->integer_wrappers);
  integer(p) = x;
  sc->integer_wrappers = cdr(sc->integer_wrappers);
  return(p);
}

static s7_pointer wrap_real(s7_scheme *sc, s7_double x)
{
  s7_pointer p = car(sc->real_wrappers);
  real(p) = x;
  sc->real_wrappers = cdr(sc->real_wrappers);
  return(p);
}


/* --------------------------------------------------------------------------------
 * local versions of some standard C library functions
 * timing tests involving these are very hard to interpret, local_memset is faster using int64_t than int32_t
 */

static void local_memset(void *s, uint8_t val, size_t n)
{
  uint8_t *s2;
#if S7_ALIGNED
  s2 = (uint8_t *)s;
#else
#if (defined(__x86_64__) || defined(__i386__))
  if (n >= 8)
    {
      int64_t *s1 = (int64_t *)s;
      size_t n8 = n >> 3;
      int64_t ival = val | (val << 8) | (val << 16) | (((uint64_t)val) << 24); /* uint64_t casts make gcc/clang/fsanitize happy */
      ival = (((uint64_t)ival) << 32) | ival;
      if ((n8 & 0x3) == 0)
	while (n8 > 0) {LOOP_4(*s1++ = ival); n8 -= 4;}
      else do {*s1++ = ival;} while (--n8 > 0);
      n &= 7;
      s2 = (uint8_t *)s1;
    }
  else s2 = (uint8_t *)s;
#else
  s2 = (uint8_t *)s;
#endif
#endif
  while (n > 0)
    {
      *s2++ = val;
      n--;
    }
}

static inline s7_int safe_strlen(const char *str) /* this is safer than strlen, and slightly faster */
{
  const char *tmp = str;
  if ((!tmp) || (!(*tmp))) return(0);
  for (; *tmp; ++tmp);
  return(tmp - str);
}

static char *copy_string_with_length(const char *str, s7_int len)
{
  char *newstr;
  if ((S7_DEBUGGING) && ((len <= 0) || (!str))) fprintf(stderr, "%s[%d]: len: %" ld64 ", str: %s\n", __func__, __LINE__, len, str);
  if (len > (1LL << 48)) return(NULL); /* squelch an idiotic warning */
  newstr = (char *)Malloc(len + 1);
  memcpy((void *)newstr, (void *)str, len); /* we check len != 0 above -- 24-Jan-22 */
  newstr[len] = '\0';
  return(newstr);
}

static char *copy_string(const char *str) {return(copy_string_with_length(str, safe_strlen(str)));}

static bool local_strcmp(const char *s1, const char *s2)
{
  while (true)
    {
      if (*s1 != *s2++) return(false);
      if (*s1++ == 0) return(true);
    }
  return(true);
}

#define c_strings_are_equal(Str1, Str2) (local_strcmp(Str1, Str2))
/* scheme strings can have embedded nulls */

static bool safe_strcmp(const char *s1, const char *s2)
{
  if ((!s1) || (!s2)) return(s1 == s2);
  return(local_strcmp(s1, s2));
}

static bool local_strncmp(const char *s1, const char *s2, size_t n)
{
#if ((!S7_ALIGNED) && (defined(__x86_64__) || defined(__i386__))) /* unaligned accesses are safe on i386 hardware, sez everyone */
  if (n >= 8)
    {
      size_t n8 = n >> 3;
      int64_t *is1 = (int64_t *)s1, *is2 = (int64_t *)s2;
      do {if (*is1++ != *is2++) return(false);} while (--n8 > 0); /* in tbig LOOP_4 is slower? */
      s1 = (const char *)is1;
      s2 = (const char *)is2;
      n &= 7;
    }
#endif
  while (n > 0)
    {
      if (*s1++ != *s2++) return(false);
      n--;
    }
  return(true);
}

#define strings_are_equal_with_length(Str1, Str2, Len) (local_strncmp(Str1, Str2, Len))

static Sentinel size_t catstrs(char *dst, size_t len, ...) /* NULL-terminated arg list */
{
  const char *dend = (const char *)(dst + len - 1); /* -1 for null at end? */
  char *d = dst;
  va_list ap;
  while ((*d) && (d < dend)) d++; /* stop at NULL or end-of-buffer */
  va_start(ap, len);
  for (const char *s = va_arg(ap, const char *); s != NULL; s = va_arg(ap, const char *))
    while ((*s) && (d < dend)) {*d++ = *s++;}
  *d = '\0';
  va_end (ap);
  return(d - dst);
}

static Sentinel size_t catstrs_direct(char *dst, const char *s1, ...)
{ /* NULL-terminated arg list, dst is destination only (assumed empty), all args known to fit in dst */
  char *d = dst;
  va_list ap;
  va_start(ap, s1);
  for (const char *s = s1; s != NULL; s = va_arg(ap, const char *))
    while (*s) {*d++ = *s++;}
  *d = '\0';
  va_end (ap);
  return(d - dst);
}

static char *pos_int_to_str(s7_scheme *sc, s7_int num, s7_int *len, char endc)
{
  char *p = (char *)(sc->int_to_str3 + INT_TO_STR_SIZE - 1);
  char *op = p;
  *p-- = '\0';
  if (endc != '\0') *p-- = endc;
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  (*len) = op - p;           /* this includes the trailing #\null */
  return((char *)(p + 1));
}

static char *pos_int_to_str_direct(s7_scheme *sc, s7_int num)
{
  char *p = (char *)(sc->int_to_str4 + INT_TO_STR_SIZE - 1);
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  return((char *)(p + 1));
}

static char *pos_int_to_str_direct_1(s7_scheme *sc, s7_int num)
{
  char *p = (char *)(sc->int_to_str5 + INT_TO_STR_SIZE - 1);
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  return((char *)(p + 1));
}

#if S7_DEBUGGING && WITH_GCC
  static s7_pointer lookup_1(s7_scheme *sc, s7_pointer symbol);
  #define lookup(Sc, Sym) check_null_sym(Sc, lookup_1(Sc, Sym), Sym, __LINE__, __func__)
  static s7_pointer check_null_sym(s7_scheme *sc, s7_pointer p, s7_pointer sym, int32_t line, const char *func);
  #define lookup_unexamined(Sc, Sym) lookup_1(Sc, Sym)
#else
  static inline s7_pointer lookup(s7_scheme *sc, s7_pointer symbol);
  #define lookup_unexamined(Sc, Sym) lookup(Sc, Sym)
#endif

#if WITH_GCC
  #if S7_DEBUGGING
    #define lookup_checked(Sc, Sym) ({s7_pointer _x_; _x_ = lookup_1(Sc, Sym); ((_x_) ? _x_ : unbound_variable(Sc, Sym));})
  #else
    #define lookup_checked(Sc, Sym) ({s7_pointer _x_; _x_ = lookup(Sc, Sym); ((_x_) ? _x_ : unbound_variable(Sc, Sym));})
  #endif
#else
  #define lookup_checked(Sc, Sym) lookup(Sc, Sym)
#endif

static inline s7_pointer lookup_slot_from(s7_pointer symbol, s7_pointer e);
static s7_pointer object_to_truncated_string(s7_scheme *sc, s7_pointer p, s7_int len);
static s7_pointer wrap_string(s7_scheme *sc, const char *str, s7_int len);
static s7_pointer cons_unchecked(s7_scheme *sc, s7_pointer a, s7_pointer b);
static s7_pointer unbound_variable(s7_scheme *sc, s7_pointer sym);
static s7_pointer find_method_with_let(s7_scheme *sc, s7_pointer let, s7_pointer symbol);
static const char *type_name(s7_scheme *sc, s7_pointer arg, article_t article);

static noreturn void simple_wrong_type_arg_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer typnam, s7_pointer descr);
static noreturn void wrong_type_arg_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg_n, s7_pointer arg, s7_pointer typnam, s7_pointer descr);
static noreturn void out_of_range_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg_n, s7_pointer arg, s7_pointer descr);
static noreturn void simple_out_of_range_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer descr);

/* putting off the type description until s7_error via the sc->unused marker below makes it possible
 *    for gcc to speed up the functions that call these as tail-calls.  1-2% overall speedup!
 */
#define simple_wrong_type_argument(Sc, Caller, Arg, Desired_Type) \
  simple_wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, Sc->unused, Sc->prepackaged_type_names[Desired_Type])

#define wrong_type_argument(Sc, Caller, Num, Arg, Desired_Type)	\
  wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), make_integer(Sc, Num), Arg, Sc->unused, Sc->prepackaged_type_names[Desired_Type])

#define simple_wrong_type_argument_with_type(Sc, Caller, Arg, Type) \
  simple_wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, Sc->unused, Type)

#define wrong_type_argument_with_type(Sc, Caller, Num, Arg, Type)	\
  wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), make_integer(Sc, Num), Arg, Sc->unused, Type)

#define simple_out_of_range(Sc, Caller, Arg, Description)   simple_out_of_range_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, Description)
#define out_of_range(Sc, Caller, Arg_Num, Arg, Description) out_of_range_error_prepackaged(Sc, symbol_name_cell(Caller), Arg_Num, Arg, Description)


/* ---------------- evaluator ops ---------------- */

/* C=constant, S=symbol, A=fx-callable, Q=quote, N=any number of next >= 1, FX=list of A's, P=parlous?, O=one form, M=multiform */
enum {OP_UNOPT, OP_GC_PROTECT, /* must be an even number of ops here, op_gc_protect used below as lower boundary marker */

      OP_SAFE_C_NC, HOP_SAFE_C_NC, OP_SAFE_C_S, HOP_SAFE_C_S,
      OP_SAFE_C_SS, HOP_SAFE_C_SS, OP_SAFE_C_SC, HOP_SAFE_C_SC, OP_SAFE_C_CS, HOP_SAFE_C_CS, OP_SAFE_C_CQ, HOP_SAFE_C_CQ,
      OP_SAFE_C_SSS, HOP_SAFE_C_SSS, OP_SAFE_C_SCS, HOP_SAFE_C_SCS, OP_SAFE_C_SSC, HOP_SAFE_C_SSC, OP_SAFE_C_CSS, HOP_SAFE_C_CSS,
      OP_SAFE_C_SCC, HOP_SAFE_C_SCC, OP_SAFE_C_CSC, HOP_SAFE_C_CSC, OP_SAFE_C_CCS, HOP_SAFE_C_CCS,
      OP_SAFE_C_NS, HOP_SAFE_C_NS, OP_SAFE_C_opNCq, HOP_SAFE_C_opNCq, OP_SAFE_C_opSq, HOP_SAFE_C_opSq,
      OP_SAFE_C_opSSq, HOP_SAFE_C_opSSq, OP_SAFE_C_opSCq, HOP_SAFE_C_opSCq,
      OP_SAFE_C_opCSq, HOP_SAFE_C_opCSq, OP_SAFE_C_S_opSq, HOP_SAFE_C_S_opSq,
      OP_SAFE_C_C_opSCq, HOP_SAFE_C_C_opSCq, OP_SAFE_C_S_opSCq, HOP_SAFE_C_S_opSCq, OP_SAFE_C_S_opCSq, HOP_SAFE_C_S_opCSq,
      OP_SAFE_C_opSq_S, HOP_SAFE_C_opSq_S, OP_SAFE_C_opSq_C, HOP_SAFE_C_opSq_C,
      OP_SAFE_C_opSq_opSq, HOP_SAFE_C_opSq_opSq, OP_SAFE_C_S_opSSq, HOP_SAFE_C_S_opSSq, OP_SAFE_C_C_opSq, HOP_SAFE_C_C_opSq,
      OP_SAFE_C_opCSq_C, HOP_SAFE_C_opCSq_C, OP_SAFE_C_opSSq_C, HOP_SAFE_C_opSSq_C, OP_SAFE_C_C_opSSq, HOP_SAFE_C_C_opSSq,
      OP_SAFE_C_opSSq_opSSq, HOP_SAFE_C_opSSq_opSSq, OP_SAFE_C_opSSq_opSq, HOP_SAFE_C_opSSq_opSq, OP_SAFE_C_opSq_opSSq, HOP_SAFE_C_opSq_opSSq,
      OP_SAFE_C_opSSq_S, HOP_SAFE_C_opSSq_S, OP_SAFE_C_opCSq_S, HOP_SAFE_C_opCSq_S, OP_SAFE_C_opSCq_C, HOP_SAFE_C_opSCq_C,
      OP_SAFE_C_op_opSSqq_S, HOP_SAFE_C_op_opSSqq_S, OP_SAFE_C_op_opSqq, HOP_SAFE_C_op_opSqq,
      OP_SAFE_C_op_S_opSqq, HOP_SAFE_C_op_S_opSqq, OP_SAFE_C_op_opSq_Sq, HOP_SAFE_C_op_opSq_Sq, OP_SAFE_C_opSq_CS, HOP_SAFE_C_opSq_CS,

      OP_SAFE_C_A, HOP_SAFE_C_A, OP_SAFE_C_AA, HOP_SAFE_C_AA, OP_SAFE_C_SA, HOP_SAFE_C_SA, OP_SAFE_C_AS, HOP_SAFE_C_AS,
      OP_SAFE_C_CA, HOP_SAFE_C_CA, OP_SAFE_C_AC, HOP_SAFE_C_AC, OP_SAFE_C_AAA, HOP_SAFE_C_AAA, OP_SAFE_C_4A, HOP_SAFE_C_4A,
      OP_SAFE_C_NA, HOP_SAFE_C_NA, OP_SAFE_C_ALL_CA, HOP_SAFE_C_ALL_CA,
      OP_SAFE_C_SSA, HOP_SAFE_C_SSA, OP_SAFE_C_SAS, HOP_SAFE_C_SAS, OP_SAFE_C_SAA, HOP_SAFE_C_SAA,
      OP_SAFE_C_CSA, HOP_SAFE_C_CSA, OP_SAFE_C_SCA, HOP_SAFE_C_SCA, OP_SAFE_C_ASS, HOP_SAFE_C_ASS,
      OP_SAFE_C_CAC, HOP_SAFE_C_CAC, OP_SAFE_C_AGG, HOP_SAFE_C_AGG,
      OP_SAFE_C_opAq, HOP_SAFE_C_opAq, OP_SAFE_C_opAAq, HOP_SAFE_C_opAAq, OP_SAFE_C_opAAAq, HOP_SAFE_C_opAAAq,
      OP_SAFE_C_S_opAq, HOP_SAFE_C_S_opAq, OP_SAFE_C_opAq_S, HOP_SAFE_C_opAq_S, OP_SAFE_C_S_opAAq, HOP_SAFE_C_S_opAAq,
      OP_SAFE_C_STAR, HOP_SAFE_C_STAR, OP_SAFE_C_STAR_A, HOP_SAFE_C_STAR_A,
      OP_SAFE_C_STAR_AA, HOP_SAFE_C_STAR_AA, OP_SAFE_C_STAR_NA, HOP_SAFE_C_STAR_NA,

      OP_SAFE_C_P, HOP_SAFE_C_P, OP_SAFE_C_PP, HOP_SAFE_C_PP, OP_SAFE_C_FF, HOP_SAFE_C_FF, OP_SAFE_C_SP, HOP_SAFE_C_SP,
      OP_SAFE_C_CP, HOP_SAFE_C_CP, OP_SAFE_C_AP, HOP_SAFE_C_AP, OP_SAFE_C_PA, HOP_SAFE_C_PA, OP_SAFE_C_PS, HOP_SAFE_C_PS,
      OP_SAFE_C_PC, HOP_SAFE_C_PC, OP_SAFE_C_SSP, HOP_SAFE_C_SSP, OP_ANY_C_NP, HOP_ANY_C_NP, OP_SAFE_C_3P, HOP_SAFE_C_3P,

      OP_THUNK, HOP_THUNK, OP_THUNK_O, HOP_THUNK_O, OP_THUNK_ANY, HOP_THUNK_ANY,
      OP_SAFE_THUNK, HOP_SAFE_THUNK, OP_SAFE_THUNK_A, HOP_SAFE_THUNK_A, OP_SAFE_THUNK_ANY, HOP_SAFE_THUNK_ANY,

      OP_CLOSURE_S, HOP_CLOSURE_S, OP_CLOSURE_S_O, HOP_CLOSURE_S_O,
      OP_CLOSURE_A, HOP_CLOSURE_A, OP_CLOSURE_A_O, HOP_CLOSURE_A_O, OP_CLOSURE_P, HOP_CLOSURE_P,
      OP_CLOSURE_AP, HOP_CLOSURE_AP, OP_CLOSURE_PA, HOP_CLOSURE_PA, OP_CLOSURE_PP, HOP_CLOSURE_PP,
      OP_CLOSURE_FA, HOP_CLOSURE_FA, OP_CLOSURE_SS, HOP_CLOSURE_SS, OP_CLOSURE_SS_O, HOP_CLOSURE_SS_O,
      OP_CLOSURE_SC, HOP_CLOSURE_SC, OP_CLOSURE_SC_O, HOP_CLOSURE_SC_O,
      OP_CLOSURE_3S, HOP_CLOSURE_3S, OP_CLOSURE_3S_O, HOP_CLOSURE_3S_O, OP_CLOSURE_4S, HOP_CLOSURE_4S, OP_CLOSURE_4S_O, HOP_CLOSURE_4S_O,
      OP_CLOSURE_AA, HOP_CLOSURE_AA, OP_CLOSURE_AA_O, HOP_CLOSURE_AA_O, OP_CLOSURE_3A, HOP_CLOSURE_3A, OP_CLOSURE_4A, HOP_CLOSURE_4A,
      OP_CLOSURE_NA, HOP_CLOSURE_NA, OP_CLOSURE_ASS, HOP_CLOSURE_ASS, OP_CLOSURE_SAS, HOP_CLOSURE_SAS ,OP_CLOSURE_AAS, HOP_CLOSURE_AAS,
      OP_CLOSURE_SAA, HOP_CLOSURE_SAA, OP_CLOSURE_ASA, HOP_CLOSURE_ASA, OP_CLOSURE_NS, HOP_CLOSURE_NS,

      OP_SAFE_CLOSURE_S, HOP_SAFE_CLOSURE_S, OP_SAFE_CLOSURE_S_O, HOP_SAFE_CLOSURE_S_O,
      OP_SAFE_CLOSURE_S_A, HOP_SAFE_CLOSURE_S_A, OP_SAFE_CLOSURE_S_TO_S, HOP_SAFE_CLOSURE_S_TO_S, OP_SAFE_CLOSURE_S_TO_SC, HOP_SAFE_CLOSURE_S_TO_SC,
      OP_SAFE_CLOSURE_P, HOP_SAFE_CLOSURE_P, OP_SAFE_CLOSURE_P_A, HOP_SAFE_CLOSURE_P_A,
      OP_SAFE_CLOSURE_AP, HOP_SAFE_CLOSURE_AP, OP_SAFE_CLOSURE_PA, HOP_SAFE_CLOSURE_PA, OP_SAFE_CLOSURE_PP, HOP_SAFE_CLOSURE_PP,
      OP_SAFE_CLOSURE_A, HOP_SAFE_CLOSURE_A, OP_SAFE_CLOSURE_A_O, HOP_SAFE_CLOSURE_A_O, OP_SAFE_CLOSURE_A_A, HOP_SAFE_CLOSURE_A_A,
      OP_SAFE_CLOSURE_A_TO_SC, HOP_SAFE_CLOSURE_A_TO_SC,
      OP_SAFE_CLOSURE_SS, HOP_SAFE_CLOSURE_SS, OP_SAFE_CLOSURE_SS_O, HOP_SAFE_CLOSURE_SS_O, OP_SAFE_CLOSURE_SS_A, HOP_SAFE_CLOSURE_SS_A,
      OP_SAFE_CLOSURE_SC, HOP_SAFE_CLOSURE_SC, OP_SAFE_CLOSURE_SC_O, HOP_SAFE_CLOSURE_SC_O,
      OP_SAFE_CLOSURE_AA, HOP_SAFE_CLOSURE_AA, OP_SAFE_CLOSURE_AA_O, HOP_SAFE_CLOSURE_AA_O, OP_SAFE_CLOSURE_AA_A, HOP_SAFE_CLOSURE_AA_A,
      OP_SAFE_CLOSURE_SAA, HOP_SAFE_CLOSURE_SAA, OP_SAFE_CLOSURE_SSA, HOP_SAFE_CLOSURE_SSA,
      OP_SAFE_CLOSURE_AGG, HOP_SAFE_CLOSURE_AGG, OP_SAFE_CLOSURE_3A, HOP_SAFE_CLOSURE_3A, OP_SAFE_CLOSURE_NA, HOP_SAFE_CLOSURE_NA,
      OP_SAFE_CLOSURE_3S, HOP_SAFE_CLOSURE_3S, OP_SAFE_CLOSURE_NS, HOP_SAFE_CLOSURE_NS, /* safe_closure_4s gained very little */
      OP_SAFE_CLOSURE_3S_A, HOP_SAFE_CLOSURE_3S_A,
      /* ssa|saa|ns|na|3s|agg|3a|sc|ap|pa|pp_a ? thunk_o? op_closure_ns?  */

      OP_ANY_CLOSURE_3P, HOP_ANY_CLOSURE_3P, OP_ANY_CLOSURE_4P, HOP_ANY_CLOSURE_4P, OP_ANY_CLOSURE_NP, HOP_ANY_CLOSURE_NP,
      OP_ANY_CLOSURE_SYM, HOP_ANY_CLOSURE_SYM, OP_ANY_CLOSURE_A_SYM, HOP_ANY_CLOSURE_A_SYM,

      OP_CLOSURE_STAR_A, HOP_CLOSURE_STAR_A, OP_CLOSURE_STAR_NA, HOP_CLOSURE_STAR_NA,
      OP_SAFE_CLOSURE_STAR_A, HOP_SAFE_CLOSURE_STAR_A, OP_SAFE_CLOSURE_STAR_AA, HOP_SAFE_CLOSURE_STAR_AA,
      OP_SAFE_CLOSURE_STAR_AA_O, HOP_SAFE_CLOSURE_STAR_AA_O, OP_SAFE_CLOSURE_STAR_A1, HOP_SAFE_CLOSURE_STAR_A1,
      OP_SAFE_CLOSURE_STAR_KA, HOP_SAFE_CLOSURE_STAR_KA, OP_CLOSURE_STAR_KA, HOP_CLOSURE_STAR_KA, OP_SAFE_CLOSURE_STAR_3A, HOP_SAFE_CLOSURE_STAR_3A,
      OP_SAFE_CLOSURE_STAR_NA, HOP_SAFE_CLOSURE_STAR_NA, OP_SAFE_CLOSURE_STAR_NA_0, HOP_SAFE_CLOSURE_STAR_NA_0,
      OP_SAFE_CLOSURE_STAR_NA_1, HOP_SAFE_CLOSURE_STAR_NA_1, OP_SAFE_CLOSURE_STAR_NA_2, HOP_SAFE_CLOSURE_STAR_NA_2,

      OP_C_SS, HOP_C_SS, OP_C_S, HOP_C_S, OP_READ_S, HOP_READ_S, OP_C_P, HOP_C_P, OP_C_AP, HOP_C_AP,
      OP_C_A, HOP_C_A, OP_C_AA, HOP_C_AA, OP_C, HOP_C, OP_C_NA, HOP_C_NA,

      OP_CL_S, HOP_CL_S, OP_CL_SS, HOP_CL_SS, OP_CL_A, HOP_CL_A, OP_CL_AA, HOP_CL_AA,
      OP_CL_NA, HOP_CL_NA, OP_CL_FA, HOP_CL_FA, OP_CL_SAS, HOP_CL_SAS,
      /* end of h_opts */

      OP_APPLY_SS, OP_APPLY_SA, OP_APPLY_SL, OP_MACRO_D, OP_MACRO_STAR_D,
      OP_WITH_IO, OP_WITH_IO_1, OP_WITH_OUTPUT_TO_STRING, OP_WITH_IO_C, OP_CALL_WITH_OUTPUT_STRING,
      OP_S, OP_S_G, OP_S_A, OP_S_AA, OP_A_A, OP_A_AA, OP_P_S, OP_P_S_1, OP_MAP_FOR_EACH_FA, OP_MAP_FOR_EACH_FAA,
      OP_F, OP_F_A, OP_F_AA, OP_F_NP, OP_F_NP_1,

      OP_IMPLICIT_GOTO, OP_IMPLICIT_GOTO_A, OP_IMPLICIT_CONTINUATION_A, OP_IMPLICIT_ITERATE,
      OP_IMPLICIT_VECTOR_REF_A, OP_IMPLICIT_VECTOR_REF_AA, OP_IMPLICIT_VECTOR_SET_3, OP_IMPLICIT_VECTOR_SET_4,
      OP_IMPLICIT_STRING_REF_A, OP_IMPLICIT_C_OBJECT_REF_A, OP_IMPLICIT_PAIR_REF_A, OP_IMPLICIT_PAIR_REF_AA,
      OP_IMPLICIT_HASH_TABLE_REF_A, OP_IMPLICIT_HASH_TABLE_REF_AA,
      OP_IMPLICIT_LET_REF_C, OP_IMPLICIT_LET_REF_A, OP_IMPLICIT_S7_LET_REF_S,
      OP_UNKNOWN, OP_UNKNOWN_NS, OP_UNKNOWN_NA, OP_UNKNOWN_S, OP_UNKNOWN_GG, OP_UNKNOWN_A, OP_UNKNOWN_AA, OP_UNKNOWN_NP,

      OP_SYMBOL, OP_CONSTANT, OP_PAIR_SYM, OP_PAIR_PAIR, OP_PAIR_ANY, HOP_HASH_TABLE_INCREMENT, OP_CLEAR_OPTS,

      OP_READ_INTERNAL, OP_EVAL, OP_EVAL_ARGS, OP_EVAL_ARGS1, OP_EVAL_ARGS2, OP_EVAL_ARGS3, OP_EVAL_ARGS4, OP_EVAL_ARGS5,
      OP_APPLY, OP_EVAL_MACRO, OP_LAMBDA, OP_QUOTE, OP_QUOTE_UNCHECKED, OP_MACROEXPAND, OP_CALL_CC, OP_CALL_WITH_EXIT, OP_CALL_WITH_EXIT_O,
      OP_C_CATCH, OP_C_CATCH_ALL, OP_C_CATCH_ALL_O, OP_C_CATCH_ALL_A,

      OP_DEFINE, OP_DEFINE1, OP_BEGIN, OP_BEGIN_HOOK, OP_BEGIN_NO_HOOK, OP_BEGIN_UNCHECKED, OP_BEGIN_2_UNCHECKED, OP_BEGIN_NA, OP_BEGIN_AA,
      OP_IF, OP_IF1, OP_WHEN, OP_UNLESS, OP_SET, OP_SET1, OP_SET2,
      OP_LET, OP_LET1, OP_LET_STAR, OP_LET_STAR1, OP_LET_STAR2, OP_LET_STAR_SHADOWED,
      OP_LETREC, OP_LETREC1, OP_LETREC_STAR, OP_LETREC_STAR1,
      OP_LET_TEMPORARILY, OP_LET_TEMP_UNCHECKED, OP_LET_TEMP_INIT1, OP_LET_TEMP_INIT2, OP_LET_TEMP_DONE, OP_LET_TEMP_DONE1,
      OP_LET_TEMP_S7, OP_LET_TEMP_NA, OP_LET_TEMP_A, OP_LET_TEMP_SETTER, OP_LET_TEMP_UNWIND, OP_LET_TEMP_S7_UNWIND, OP_LET_TEMP_SETTER_UNWIND,
      OP_LET_TEMP_A_A,
      OP_COND, OP_COND1, OP_FEED_TO_1, OP_COND_SIMPLE, OP_COND1_SIMPLE, OP_COND_SIMPLE_O, OP_COND1_SIMPLE_O,
      OP_AND, OP_OR,
      OP_DEFINE_MACRO, OP_DEFINE_MACRO_STAR, OP_DEFINE_EXPANSION, OP_DEFINE_EXPANSION_STAR, OP_MACRO, OP_MACRO_STAR,
      OP_CASE,
      OP_READ_LIST, OP_READ_NEXT, OP_READ_DOT, OP_READ_QUOTE,
      OP_READ_QUASIQUOTE, OP_READ_UNQUOTE, OP_READ_APPLY_VALUES,
      OP_READ_VECTOR, OP_READ_BYTE_VECTOR, OP_READ_INT_VECTOR, OP_READ_FLOAT_VECTOR, OP_READ_DONE,
      OP_LOAD_RETURN_IF_EOF, OP_LOAD_CLOSE_AND_POP_IF_EOF, OP_EVAL_DONE, OP_SPLICE_VALUES, OP_NO_VALUES,
      OP_CATCH, OP_DYNAMIC_WIND, OP_DYNAMIC_UNWIND, OP_DYNAMIC_UNWIND_PROFILE, OP_PROFILE_IN,
      OP_DEFINE_CONSTANT, OP_DEFINE_CONSTANT1,
      OP_DO, OP_DO_END, OP_DO_END1, OP_DO_STEP, OP_DO_STEP2, OP_DO_INIT,
      OP_DEFINE_STAR, OP_LAMBDA_STAR, OP_LAMBDA_STAR_DEFAULT, OP_ERROR_QUIT, OP_UNWIND_INPUT, OP_UNWIND_OUTPUT, OP_ERROR_HOOK_QUIT,
      OP_WITH_LET, OP_WITH_LET1, OP_WITH_LET_UNCHECKED, OP_WITH_LET_S,
      OP_WITH_BAFFLE, OP_WITH_BAFFLE_UNCHECKED, OP_EXPANSION,
      OP_FOR_EACH, OP_FOR_EACH_1, OP_FOR_EACH_2, OP_FOR_EACH_3,
      OP_MAP, OP_MAP_1, OP_MAP_2, OP_MAP_GATHER, OP_MAP_GATHER_1, OP_MAP_GATHER_2, OP_MAP_GATHER_3,
      OP_BARRIER, OP_DEACTIVATE_GOTO,
      OP_DEFINE_BACRO, OP_DEFINE_BACRO_STAR, OP_BACRO, OP_BACRO_STAR,
      OP_GET_OUTPUT_STRING,
      OP_SORT, OP_SORT1, OP_SORT2, OP_SORT3, OP_SORT_PAIR_END, OP_SORT_VECTOR_END, OP_SORT_STRING_END,
      OP_EVAL_STRING,
      OP_MEMBER_IF, OP_ASSOC_IF, OP_MEMBER_IF1, OP_ASSOC_IF1,
      OP_LAMBDA_UNCHECKED, OP_LET_UNCHECKED, OP_CATCH_1, OP_CATCH_2, OP_CATCH_ALL,

      OP_SET_UNCHECKED, OP_SET_S_C, OP_SET_S_S, OP_SET_S_P, OP_SET_S_A,
      OP_SET_NORMAL, OP_SET_opSq_A, OP_SET_opSAq_A, OP_SET_opSAq_P, OP_SET_opSAq_P_1, OP_SET_opSAAq_A, OP_SET_opSAAq_P, OP_SET_opSAAq_P_1,
      OP_SET_FROM_SETTER, OP_SET_FROM_LET_TEMP, OP_SET_SAFE,
      OP_INCREMENT_BY_1, OP_DECREMENT_BY_1, OP_INCREMENT_SA, OP_INCREMENT_SAA, OP_SET_CONS,

      OP_LETREC_UNCHECKED, OP_LETREC_STAR_UNCHECKED, OP_COND_UNCHECKED,
      OP_LAMBDA_STAR_UNCHECKED, OP_DO_UNCHECKED, OP_DEFINE_UNCHECKED, OP_DEFINE_STAR_UNCHECKED, OP_DEFINE_FUNCHECKED, OP_DEFINE_CONSTANT_UNCHECKED,
      OP_DEFINE_WITH_SETTER,

      OP_LET_NO_VARS, OP_NAMED_LET, OP_NAMED_LET_NO_VARS, OP_NAMED_LET_A, OP_NAMED_LET_AA, OP_NAMED_LET_NA, OP_NAMED_LET_STAR,
      OP_LET_NA_OLD, OP_LET_NA_NEW, OP_LET_2A_OLD, OP_LET_2A_NEW, OP_LET_3A_OLD, OP_LET_3A_NEW,
      OP_LET_opaSSq_OLD, OP_LET_opaSSq_NEW, OP_LET_ONE_OLD, OP_LET_ONE_NEW, OP_LET_ONE_P_OLD, OP_LET_ONE_P_NEW,
      OP_LET_ONE_OLD_1, OP_LET_ONE_NEW_1, OP_LET_ONE_P_OLD_1, OP_LET_ONE_P_NEW_1,
      OP_LET_A_OLD, OP_LET_A_NEW, OP_LET_A_P_OLD, OP_LET_A_P_NEW,
      OP_LET_A_A_OLD, OP_LET_A_A_NEW, OP_LET_A_NA_OLD, OP_LET_A_NA_NEW, OP_LET_A_OLD_2, OP_LET_A_NEW_2,
      OP_LET_STAR_NA, OP_LET_STAR_NA_A,

      OP_CASE_A_E_S, OP_CASE_A_I_S, OP_CASE_A_G_S, OP_CASE_A_E_G, OP_CASE_A_G_G, OP_CASE_A_S_G,
      OP_CASE_P_E_S, OP_CASE_P_I_S, OP_CASE_P_G_S, OP_CASE_P_E_G, OP_CASE_P_G_G,
      OP_CASE_E_S, OP_CASE_I_S, OP_CASE_G_S, OP_CASE_E_G, OP_CASE_G_G,
      OP_CASE_A_I_S_A, OP_CASE_A_E_S_A, OP_CASE_A_G_S_A, OP_CASE_A_S_G_A,

      OP_IF_UNCHECKED, OP_AND_P, OP_AND_P1, OP_AND_AP, OP_AND_PAIR_P,
      OP_AND_SAFE_P1, OP_AND_SAFE_P2, OP_AND_SAFE_P3, OP_AND_SAFE_P_REST, OP_AND_2A, OP_AND_3A, OP_AND_N, OP_AND_S_2,
      OP_OR_P, OP_OR_P1, OP_OR_AP, OP_OR_2A, OP_OR_3A, OP_OR_N, OP_OR_S_2, OP_OR_S_TYPE_2,
      OP_WHEN_S, OP_WHEN_A, OP_WHEN_P, OP_WHEN_AND_AP, OP_WHEN_AND_2A, OP_WHEN_AND_3A, OP_UNLESS_S, OP_UNLESS_A, OP_UNLESS_P,

      OP_IF_A_C_C, OP_IF_A_A, OP_IF_A_A_A, OP_IF_S_A_A, OP_IF_AND2_S_A, OP_IF_NOT_A_A, OP_IF_NOT_A_A_A,
      OP_IF_B_A, OP_IF_B_P, OP_IF_B_R, OP_IF_B_A_P, OP_IF_B_P_A, OP_IF_B_P_P, OP_IF_B_N_N,
      OP_IF_A_A_P, OP_IF_A_P_A, OP_IF_S_P_A, OP_IF_S_A_P, OP_IF_S_P, OP_IF_S_P_P, OP_IF_S_R, OP_IF_S_N, OP_IF_S_N_N,
      OP_IF_opSq_P, OP_IF_opSq_P_P, OP_IF_opSq_R, OP_IF_opSq_N, OP_IF_opSq_N_N,
      OP_IF_IS_TYPE_S_P, OP_IF_IS_TYPE_S_P_P, OP_IF_IS_TYPE_S_R, OP_IF_IS_TYPE_S_N, OP_IF_IS_TYPE_S_N_N, OP_IF_IS_TYPE_S_P_A, OP_IF_IS_TYPE_S_A_A, OP_IF_IS_TYPE_S_A_P,
      OP_IF_A_P, OP_IF_A_P_P, OP_IF_A_R, OP_IF_A_N, OP_IF_A_N_N,
      OP_IF_AND2_P, OP_IF_AND2_P_P, OP_IF_AND2_R, OP_IF_AND2_N, OP_IF_AND2_N_N,
      OP_IF_AND3_P, OP_IF_AND3_P_P, OP_IF_AND3_R, OP_IF_AND3_N, OP_IF_AND3_N_N,  /* or3 got few hits */
      OP_IF_P_P, OP_IF_P_P_P, OP_IF_P_R, OP_IF_P_N, OP_IF_P_N_N,
      OP_IF_ANDP_P, OP_IF_ANDP_P_P, OP_IF_ANDP_R, OP_IF_ANDP_N, OP_IF_ANDP_N_N,
      OP_IF_ORP_P, OP_IF_ORP_P_P, OP_IF_ORP_R, OP_IF_ORP_N, OP_IF_ORP_N_N,
      OP_IF_OR2_P, OP_IF_OR2_P_P, OP_IF_OR2_R, OP_IF_OR2_N, OP_IF_OR2_N_N,
      OP_IF_PP, OP_IF_PPP, OP_IF_PN, OP_IF_PR, OP_IF_PRR, OP_WHEN_PP, OP_UNLESS_PP,

      OP_COND_NA_NA, OP_COND_NA_NP, OP_COND_NA_NP_1, OP_COND_NA_2E, OP_COND_NA_3E, OP_COND_NA_NP_O,
      OP_COND_FEED, OP_COND_FEED_1,

      OP_SIMPLE_DO, OP_SIMPLE_DO_STEP, OP_SAFE_DOTIMES, OP_SAFE_DOTIMES_STEP, OP_SAFE_DOTIMES_STEP_O,
      OP_SAFE_DO, OP_SAFE_DO_STEP, OP_DOX, OP_DOX_STEP, OP_DOX_STEP_O, OP_DOX_NO_BODY, OP_DOX_PENDING_NO_BODY, OP_DOX_INIT,
      OP_DOTIMES_P, OP_DOTIMES_STEP_O,
      OP_DO_NO_VARS, OP_DO_NO_VARS_NO_OPT, OP_DO_NO_VARS_NO_OPT_1,
      OP_DO_NO_BODY_NA_VARS, OP_DO_NO_BODY_NA_VARS_STEP, OP_DO_NO_BODY_NA_VARS_STEP_1,

      OP_SAFE_C_P_1, OP_SAFE_C_PP_1, OP_SAFE_C_PP_3_MV, OP_SAFE_C_PP_5, OP_SAFE_C_PP_6_MV,
      OP_SAFE_C_3P_1, OP_SAFE_C_3P_2, OP_SAFE_C_3P_3, OP_SAFE_C_3P_1_MV, OP_SAFE_C_3P_2_MV, OP_SAFE_C_3P_3_MV,
      OP_SAFE_C_SP_1, OP_SAFE_C_SP_MV, OP_SAFE_CONS_SP_1, OP_SAFE_ADD_SP_1, OP_SAFE_MULTIPLY_SP_1,
      OP_SAFE_C_PS_1, OP_SAFE_C_PC_1, OP_SAFE_C_PS_MV, OP_SAFE_C_PC_MV,
      OP_EVAL_MACRO_MV, OP_MACROEXPAND_1, OP_APPLY_LAMBDA,
      OP_ANY_C_NP_1, OP_ANY_C_NP_MV, OP_SAFE_C_SSP_1, OP_SAFE_C_SSP_MV,
      OP_C_P_1, OP_C_P_MV, OP_C_AP_1, OP_C_AP_MV, OP_ANY_C_NP_2, OP_SAFE_C_PA_1, OP_SAFE_C_PA_MV,
      OP_SET_WITH_LET_1, OP_SET_WITH_LET_2,

      OP_CLOSURE_AP_1, OP_CLOSURE_PA_1, OP_CLOSURE_PP_1, OP_CLOSURE_P_1,
      OP_SAFE_CLOSURE_P_1, OP_SAFE_CLOSURE_P_A_1, OP_SAFE_CLOSURE_AP_1, OP_SAFE_CLOSURE_PA_1, OP_SAFE_CLOSURE_PP_1,
      OP_ANY_CLOSURE_3P_1, OP_ANY_CLOSURE_3P_2, OP_ANY_CLOSURE_3P_3, OP_ANY_CLOSURE_NP_1, OP_ANY_CLOSURE_NP_MV,
      OP_ANY_CLOSURE_4P_1, OP_ANY_CLOSURE_4P_2, OP_ANY_CLOSURE_4P_3, OP_ANY_CLOSURE_4P_4, OP_ANY_CLOSURE_NP_2,

      OP_TC_AND_A_OR_A_LA, OP_TC_OR_A_AND_A_LA, OP_TC_AND_A_OR_A_LAA, OP_TC_OR_A_AND_A_LAA, OP_TC_AND_A_OR_A_L3A, OP_TC_OR_A_AND_A_L3A,
      OP_TC_OR_A_A_AND_A_A_LA, OP_TC_OR_A_AND_A_A_L3A, OP_TC_AND_A_OR_A_A_LA, OP_TC_OR_A_AND_A_A_LA,
      OP_TC_WHEN_LA, OP_TC_WHEN_LAA, OP_TC_WHEN_L3A, OP_TC_LET_WHEN_LAA, OP_TC_LET_UNLESS_LAA,
      OP_TC_COND_A_Z_A_Z_LAA, OP_TC_COND_A_Z_A_LAA_Z, OP_TC_COND_A_Z_A_LAA_LAA, OP_TC_LET_COND,
      OP_TC_IF_A_Z_LA, OP_TC_IF_A_Z_LAA, OP_TC_IF_A_Z_L3A, OP_TC_IF_A_L3A_Z, OP_TC_IF_A_LA_Z, OP_TC_IF_A_LAA_Z,
      OP_TC_IF_A_Z_IF_A_Z_LA, OP_TC_IF_A_Z_IF_A_LA_Z, OP_TC_IF_A_Z_IF_A_Z_LAA, OP_TC_IF_A_Z_IF_A_LAA_Z, OP_TC_IF_A_Z_IF_A_L3A_L3A,
      OP_TC_COND_A_Z_A_Z_LA, OP_TC_COND_A_Z_A_LA_Z, OP_TC_COND_A_Z_LA, OP_TC_COND_A_LA_Z, OP_TC_COND_A_Z_LAA, OP_TC_COND_A_LAA_Z,
      OP_TC_LET_IF_A_Z_LA, OP_TC_LET_IF_A_Z_LAA, OP_TC_IF_A_Z_LET_IF_A_Z_LAA,
      OP_TC_CASE_LA, OP_TC_AND_A_IF_A_Z_LA, OP_TC_AND_A_IF_A_LA_Z,

      OP_RECUR_IF_A_A_opA_LAq, OP_RECUR_IF_A_opA_LAq_A, OP_RECUR_IF_A_A_opLA_Aq, OP_RECUR_IF_A_opLA_Aq_A,
      OP_RECUR_IF_A_A_opLA_LAq, OP_RECUR_IF_A_opLA_LAq_A,
      OP_RECUR_IF_A_A_opA_LA_LAq, OP_RECUR_IF_A_opA_LA_LAq_A,
      OP_RECUR_IF_A_A_opLA_LA_LAq, OP_RECUR_IF_A_A_IF_A_A_opLA_LAq, OP_RECUR_IF_A_A_IF_A_A_opLAA_LAAq,
      OP_RECUR_IF_A_A_opA_LAAq, OP_RECUR_IF_A_opA_LAAq_A, OP_RECUR_IF_A_A_opA_L3Aq,
      OP_RECUR_IF_A_A_LopL3A_L3A_L3Aq, OP_RECUR_IF_A_A_AND_A_LAA_LAA,
      OP_RECUR_IF_A_A_IF_A_LAA_opA_LAAq, /* same as cond case below */
      OP_RECUR_COND_A_A_opA_LAq, OP_RECUR_COND_A_A_opA_LAAq,
      OP_RECUR_COND_A_A_A_A_opLA_LAq, OP_RECUR_COND_A_A_A_A_opLAA_LAAq, OP_RECUR_COND_A_A_A_A_opA_LAAq,
      OP_RECUR_COND_A_A_A_LAA_LopA_LAAq, OP_RECUR_COND_A_A_A_LAA_opA_LAAq,
      OP_RECUR_AND_A_OR_A_LAA_LAA,

      NUM_OPS};

#define is_tc_op(Op) ((Op >= OP_TC_AND_A_OR_A_LA) && (Op <= OP_TC_CASE_LA))

typedef enum{E_C_P, E_C_PP, E_C_CP, E_C_SP, E_C_PC, E_C_PS} combine_op_t;

static const char* op_names[NUM_OPS] =
     {
      "unopt", "gc_protect",

      "safe_c_nc", "h_safe_c_nc", "safe_c_s", "h_safe_c_s",
      "safe_c_ss", "h_safe_c_ss", "safe_c_sc", "h_safe_c_sc", "safe_c_cs", "h_safe_c_cs", "safe_c_cq", "h_safe_c_cq",
      "safe_c_sss", "h_safe_c_sss", "safe_c_scs", "h_safe_c_scs", "safe_c_ssc", "h_safe_c_ssc", "safe_c_css", "h_safe_c_css",
      "safe_c_scc", "h_safe_c_scc", "safe_c_csc", "h_safe_c_csc", "safe_c_ccs", "h_safe_c_ccs",
      "safe_c_ns", "h_safe_c_ns", "safe_c_opncq", "h_safe_c_opncq", "safe_c_opsq", "h_safe_c_opsq",
      "safe_c_opssq", "h_safe_c_opssq", "safe_c_opscq", "h_safe_c_opscq",
      "safe_c_opcsq", "h_safe_c_opcsq", "safe_c_s_opsq", "h_safe_c_s_opsq",
      "safe_c_c_opscq", "h_safe_c_c_opscq", "safe_c_s_opscq", "h_safe_c_s_opscq", "safe_c_s_opcsq", "h_safe_c_s_opcsq",
      "safe_c_opsq_s", "h_safe_c_opsq_s", "safe_c_opsq_c", "h_safe_c_opsq_c",
      "safe_c_opsq_opsq", "h_safe_c_opsq_opsq", "safe_c_s_opssq", "h_safe_c_s_opssq", "safe_c_c_opsq", "h_safe_c_c_opsq",
      "safe_c_opcsq_c", "h_safe_c_opcsq_c", "safe_c_opssq_c", "h_safe_c_opssq_c", "safe_c_c_opssq", "h_safe_c_c_opssq",
      "safe_c_opssq_opssq", "h_safe_c_opssq_opssq", "safe_c_opssq_opsq", "h_safe_c_opssq_opsq", "safe_c_opsq_opssq", "h_safe_c_opsq_opssq",
      "safe_c_opssq_s", "h_safe_c_opssq_s", "safe_c_opcsq_s", "h_safe_c_opcsq_s", "safe_c_opscq_c", "h_safe_c_opscq_c",
      "safe_c_op_opssqq_s", "h_safe_c_op_opssqq_s", "safe_c_op_opsqq", "h_safe_c_op_opsqq",
      "safe_c_op_s_opsqq", "h_safe_c_op_s_opsqq", "safe_c_op_opsq_sq", "h_safe_c_op_opsq_sq", "safe_c_opsq_cs", "h_safe_c_opsq_cs",

      "safe_c_a", "h_safe_c_a", "safe_c_aa", "h_safe_c_aa", "safe_c_sa", "h_safe_c_sa", "safe_c_as", "h_safe_c_as",
      "safe_c_ca", "h_safe_c_ca", "safe_c_ac", "h_safe_c_ac", "safe_c_aaa", "h_safe_c_aaa", "safe_c_4a", "h_safe_c_4a",
      "safe_c_na", "h_safe_c_na", "safe_c_all_ca", "h_safe_c_all_ca",
      "safe_c_ssa", "h_safe_c_ssa", "safe_c_sas", "h_safe_c_sas", "safe_c_saa", "h_safe_c_saa",
      "safe_c_csa", "h_safe_c_csa", "safe_c_sca", "h_safe_c_sca", "safe_c_ass", "h_safe_c_ass",
      "safe_c_cac", "h_safe_c_cac", "safe_c_agg", "h_safe_c_agg",
      "safe_c_opaq", "h_safe_c_opaq", "safe_c_opaaq", "h_safe_c_opaaq", "safe_c_opaaaq", "h_safe_c_opaaaq",
      "safe_c_s_opaq", "h_safe_c_s_opaq", "safe_c_opaq_s", "h_safe_c_opaq_s", "safe_c_s_opaaq", "h_safe_c_s_opaaq",
      "safe_c*", "h_safe_c*", "safe_c*_a", "h_safe_c*_a", "safe_c*_aa", "h_safe_c*_aa", "safe_c*_na", "h_safe_c*_na",

      "safe_c_p", "h_safe_c_p", "safe_c_pp", "h_safe_c_pp", "safe_c_ff", "h_safe_c_ff", "safe_c_sp", "h_safe_c_sp",
      "safe_c_cp", "h_safe_c_cp", "safe_c_ap", "h_safe_c_ap", "safe_c_pa", "h_safe_c_pa", "safe_c_ps", "h_safe_c_ps",
      "safe_c_pc", "h_safe_c_pc", "safe_c_ssp", "h_safe_c_ssp", "any_c_np", "h_any_c_np", "safe_c_3p", "h_safe_c_3p",

      "thunk", "h_thunk", "thunk_o", "h_thunk_o", "thunk_any", "h_thunk_any",
      "safe_thunk", "h_safe_thunk", "safe_thunk_a", "h_safe_thunk_a", "safe_thunk_any", "h_safe_thunk_any",

      "closure_s", "h_closure_s", "closure_s_o", "h_closure_s_o",
      "closure_a", "h_closure_a", "closure_a_o", "h_closure_a_o", "closure_p", "h_closure_p",
      "closure_ap", "h_closure_ap", "closure_pa", "h_closure_pa", "closure_pp", "h_closure_pp",
      "closure_fa", "h_closure_fa", "closure_ss", "h_closure_ss", "closure_ss_o", "h_closure_ss_o",
      "closure_sc", "h_closure_sc", "closure_sc_o", "h_closure_sc_o",
      "closure_3s", "h_closure_3s", "closure_3s_o", "h_closure_3s_o", "closure_4s", "h_closure_4s", "closure_4s_o", "h_closure_4s_o",
      "closure_aa", "h_closure_aa", "closure_aa_o", "h_closure_aa_o", "closure_3a", "h_closure_3a", "closure_4a", "h_closure_4a",
      "closure_na", "h_closure_na", "closure_ass", "h_closure_ass", "closure_sas", "h_closure_sas ","closure_aas", "h_closure_aas",
      "closure_saa", "h_closure_saa", "closure_asa", "h_closure_asa", "closure_ns", "h_closure_ns",

      "safe_closure_s", "h_safe_closure_s", "safe_closure_s_o", "h_safe_closure_s_o",
      "safe_closure_s_a", "h_safe_closure_s_a", "safe_closure_s_to_s", "h_safe_closure_s_to_s", "safe_closure_s_to_sc", "h_safe_closure_s_to_sc",
      "safe_closure_p", "h_safe_closure_p", "safe_closure_p_a", "h_safe_closure_p_a",
      "safe_closure_ap", "h_safe_closure_ap", "safe_closure_pa", "h_safe_closure_pa", "safe_closure_pp", "h_safe_closure_pp",
      "safe_closure_a", "h_safe_closure_a", "safe_closure_a_o", "h_safe_closure_a_o", "safe_closure_a_a", "h_safe_closure_a_a",
      "safe_closure_a_to_sc", "h_safe_closure_a_to_sc",
      "safe_closure_ss", "h_safe_closure_ss", "safe_closure_ss_o", "h_safe_closure_ss_o", "safe_closure_ss_a", "h_safe_closure_ss_a",
      "safe_closure_sc", "h_safe_closure_sc", "safe_closure_sc_o", "h_safe_closure_sc_o",
      "safe_closure_aa", "h_safe_closure_aa", "safe_closure_aa_o", "h_safe_closure_aa_o", "safe_closure_aa_a", "h_safe_closure_aa_a",
      "safe_closure_saa", "h_safe_closure_saa", "safe_closure_ssa", "h_safe_closure_ssa",
      "safe_closure_agg", "h_safe_closure_agg", "safe_closure_3a", "h_safe_closure_3a", "safe_closure_na", "h_safe_closure_na",
      "safe_closure_3s", "h_safe_closure_3s", "safe_closure_ns", "h_safe_closure_ns",
      "safe_closure_3s_a", "h_safe_closure_3s_a",

      "any_closure_3p", "h_any_closure_3p", "any_closure_4p", "h_any_closure_4p", "any_closure_np", "h_any_closure_np",
      "any_closure_sym", "h_any_closure_sym", "any_closure_a_sym", "h_any_closure_a_sym",

      "closure*_a", "h_closure*_a", "closure*_na", "h_closure*_na",
      "safe_closure*_a", "h_safe_closure*_a", "safe_closure*_aa", "h_safe_closure*_aa",
      "safe_closure*_aa_o", "h_safe_closure*_aa_o", "safe_closure*_a1", "h_safe_closure*_a1",
      "safe_closure*_ka", "h_safe_closure*_ka", "closure*_ka", "h_closure*_ka", "safe_closure*_3a", "h_safe_closure*_3a",
      "safe_closure*_na", "h_safe_closure*_na", "safe_closure*_na_0", "h_safe_closure*_na_0",
      "safe_closure*_na_1", "h_safe_closure*_na_1", "safe_closure*_na_2", "h_safe_closure*_na_2",

      "c_ss", "h_c_ss", "c_s", "h_c_s", "read_s", "h_read_s", "c_p", "h_c_p", "c_ap", "h_c_ap",
      "c_a", "h_c_a", "c_aa", "h_c_aa", "c", "h_c", "c_na", "h_c_na",

      "cl_s", "h_cl_s", "cl_ss", "h_cl_ss", "cl_a", "h_cl_a", "cl_aa", "h_cl_aa",
      "cl_na", "h_cl_na", "cl_fa", "h_cl_fa", "cl_sas", "h_cl_sas",

      "apply_ss", "apply_sa", "apply_sl", "macro_d", "macro*_d",
      "with_input_from_string", "with_input_from_string_1", "with_output_to_string", "with_input_from_string_c", "call_with_output_string",
      "s", "s_g", "s_a", "s_aa", "a_a", "a_aa", "p_s", "p_s_1", "map_for_each_fa", "map_for_each_faa",
      "f", "f_a", "f_aa", "f_np", "f_np_1",

      "implicit_goto", "implicit_goto_a", "implicit_continuation_a","implicit_iterate",
      "implicit_vector_ref_a", "implicit_vector_ref_aa", "implicit_vector_set_3", "implicit_vector_set_4",
      "implicit_string_ref_a", "implicit_c_object_ref_a", "implicit_pair_ref_a", "implicit_pair_ref_aa",
      "implicit_hash_table_ref_a", "implicit_hash_table_ref_aa",
      "implicit_let_ref_c", "implicit_let_ref_a", "implicit_*s7*_ref_s",
      "unknown_thunk", "unknown_ns", "unknown_na", "unknown_s", "unknown_gg", "unknown_a", "unknown_aa", "unknown_np",

      "symbol", "constant", "pair_sym", "pair_pair", "pair_any", "h_hash_table_increment", "clear_opts",

      "read_internal", "eval", "eval_args", "eval_args1", "eval_args2", "eval_args3", "eval_args4", "eval_args5",
      "apply", "eval_macro", "lambda", "quote", "quote_unchecked", "macroexpand", "call/cc", "call_with_exit", "call_with_exit_o",
      "c_catch", "c_catch_all", "c_catch_all_o", "c_catch_all_a",

      "define", "define1", "begin", "begin_hook", "begin_no_hook", "begin_unchecked", "begin_2_unchecked", "begin_na", "begin_aa",
      "if", "if1", "when", "unless", "set", "set1", "set2",
      "let", "let1", "let*", "let*1", "let*2", "let*-shadowed",
      "letrec", "letrec1", "letrec*", "letrec*1",
      "let_temporarily", "let_temp_unchecked", "let_temp_init1", "let_temp_init2", "let_temp_done", "let_temp_done1",
      "let_temp_s7", "let_temp_na", "let_temp_a", "let_temp_setter", "let_temp_unwind", "let_temp_s7_unwind", "let_temp_setter_unwind",
      "let_temp_a_a",
      "cond", "cond1", "feed_to_1", "cond_simple", "cond1_simple", "cond_simple_o", "cond1_simple_o",
      "and", "or",
      "define_macro", "define_macro*", "define_expansion", "define_expansion*", "macro", "macro*",
      "case", "read_list", "read_next", "read_dot", "read_quote",
      "read_quasiquote", "read_unquote", "read_apply_values",
      "read_vector", "read_byte_vector", "read_int_vector", "read_float_vector", "read_done",
      "load_return_if_eof", "load_close_and_pop_if_eof", "eval_done", "splice_values", "no_values",
      "catch", "dynamic_wind", "dynamic_unwind", "dynamic_unwind_profile", "profile_in",
      "define_constant", "define_constant1",
      "do", "do_end", "do_end1", "do_step", "do_step2", "do_init",
      "define*", "lambda*", "lambda*_default", "error_quit", "unwind_input", "unwind_output", "error_hook_quit",
      "with_let", "with_let1", "with_let_unchecked", "with_let_s",
      "with_baffle", "with_baffle_unchecked", "expansion",
      "for_each", "for_each_1", "for_each_2", "for_each_3",
      "map", "map_1", "map_2", "map_gather", "map_gather_1", "map_gather_2", "map_gather_3",
      "barrier", "deactivate_goto",
      "define_bacro", "define_bacro*", "bacro", "bacro*",
      "get_output_string",
      "sort", "sort1", "sort2", "sort3", "sort_pair_end", "sort_vector_end", "sort_string_end",
      "eval_string",
      "member_if", "assoc_if", "member_if1", "assoc_if1",
      "lambda_unchecked", "let_unchecked", "catch_1", "catch_2", "catch_all",
      "set_unchecked", "set_s_c", "set_s_s", "set_s_p", "set_a",
      "set_normal", "set_opsq_a", "set_opsaq_a", "set_opsaq_p", "set_opsaq_p_1", "set_opsaaq_a", "set_opsaaq_p", "set_opsaaq_p_1",
      "set_from_setter", "set_from_let_temp", "set_safe",
      "increment_1", "decrement_1", "increment_sa", "increment_saa", "set_cons",
      "letrec_unchecked", "letrec*_unchecked", "cond_unchecked",
      "lambda*_unchecked", "do_unchecked", "define_unchecked", "define*_unchecked", "define_funchecked", "define_constant_unchecked",
      "define_with_setter",

      "let_no_vars", "named_let", "named_let_no_vars", "named_let_a", "named_let_aa", "named_let_na", "named_let*",
      "let_na_old", "let_na_new", "let_2a_old", "let_2a_new", "let_3a_old", "let_3a_new",
      "let_opassq_old", "let_opassq_new", "let_one_old", "let_one_new", "let_one_p_old", "let_one_p_new",
      "let_one_old_1", "let_one_new_1", "let_one_p_old_1", "let_one_p_new_1",
      "let_a_old", "let_a_new", "let_a_p_old", "let_a_p_new",
      "let_a_a_old", "let_a_a_new", "let_a_na_old", "let_a_na_new", "let_a_old_2", "let_a_new_2",
      "let*_na", "let*_na_a",

      "case_a_e_s", "case_a_i_s", "case_a_g_s", "case_a_e_g", "case_a_g_g", "case_a_s_g",
      "case_p_e_s", "case_p_i_s", "case_p_g_s", "case_p_e_g", "case_p_g_g",
      "case_e_s", "case_i_s", "case_g_s", "case_e_g", "case_g_g",
      "case_a_i_s_a", "case_a_e_s_a", "case_a_g_s_a", "case_a_s_g_a",

      "if_unchecked", "and_p", "and_p1", "and_ap", "and_pair_p",
      "and_safe_p1", "op_and_safe_p2", "and_safe_p3", "and_safe_p_rest", "and_2a", "and_3a", "and_n", "and_s_2",
      "or_p", "or_p1", "or_ap", "or_2a", "or_3a", "or_n", "or_s_2", "or_s_type_2",
      "when_s", "when_a", "when_p", "when_and_ap", "when_and_2a", "when_and_3a", "unless_s", "unless_a", "unless_p",

      "if_a_c_c", "if_a_a", "if_a_a_a", "if_s_a_a", "if_and2_s_a", "if_not_a_a", "if_not_a_a_a",
      "if_b_a", "if_b_p", "if_b_r",  "if_b_a_p", "if_b_p_a", "if_b_p_p", "if_b_n_n",
      "if_a_a_p", "if_a_p_a", "if_s_p_a", "if_s_a_p", "if_s_p", "if_s_p_p", "if_s_r", "if_s_n", "if_s_n_n",
      "if_opsq_p", "if_opsq_p_p", "if_opsq_r", "if_opsq_n", "if_opsq_n_n",
      "if_is_type_s_p", "if_is_type_s_p_p", "if_is_type_s_r", "if_is_type_s_n", "if_is_type_s_n_n", "if_is_type_s_p_a", "if_is_type_s_a_a", "if_is_type_s_a_p",
      "if_a_p", "if_a_p_p", "if_a_r", "if_a_n", "if_a_n_n",
      "if_and2_p", "if_and2_p_p", "if_and2_r", "if_and2_n", "if_and2_n_n",
      "if_and3_p", "if_and3_p_p", "if_and3_r", "if_and3_n", "if_and3_n_n",
      "if_p_p", "if_p_p_p", "if_p_r", "if_p_n", "if_p_n_n",
      "if_andp_p", "if_andp_p_p", "if_andp_r", "if_andp_n", "if_andp_n_n",
      "if_orp_p", "if_orp_p_p", "if_orp_r", "if_orp_n", "if_orp_n_n",
      "if_or2_p", "if_or2_p_p", "if_or2_r", "if_or2_n", "if_or2_n_n",
      "if_pp", "if_ppp", "if_pn", "if_pr", "if_prr", "when_pp", "unless_pp",

      "cond_na_na", "cond_na_np", "cond_na_np_1", "cond_na_2e", "cond_na_3e", "cond_na_np_o",
      "cond_feed", "cond_feed_1",

      "simple_do", "simple_do_step", "safe_dotimes", "safe_dotimes_step", "safe_dotimes_step_o",
      "safe_do", "safe_do_step", "dox", "dox_step", "dox_step_o", "dox_no_body", "dox_pending_no_body", "dox_init",
      "dotimes_p", "dotimes_step_o",
      "do_no_vars", "do_no_vars_no_opt", "do_no_vars_no_opt_1",
      "do_no_body_na_vars", "do_no_body_na_vars_step", "do_no_body_na_vars_step_1",

      "safe_c_p_1", "safe_c_pp_1", "safe_c_pp_3_mv", "safe_c_pp_5", "safe_c_pp_6_mv",
      "safe_c_3p_1", "safe_c_3p_2", "safe_c_3p_3", "safe_c_3p_1_mv", "safe_c_3p_2_mv", "safe_c_3p_3_mv",
      "safe_c_sp_1", "safe_c_sp_mv", "safe_cons_sp_1", "safe_add_sp_1", "safe_multiply_sp_1",
      "safe_c_ps_1", "safe_c_pc_1", "safe_c_ps_mv", "safe_c_pc_mv",
      "eval_macro_mv", "macroexpand_1", "apply_lambda",
      "any_c_np_1", "any_c_np_mv", "safe_c_ssp_1", "safe_c_ssp_mv",
      "c_p_1", "c_p_mv", "c_ap_1", "c_ap_mv", "any_c_np_2", "safe_c_pa_1", "safe_c_pa_mv",
      "set_with_let_1", "set_with_let_2",

      "closure_ap_1", "closure_pa_1", "closure_pp_1", "closure_p_1",
      "safe_closure_p_1", "safe_closure_p_a_1", "safe_closure_ap_1", "safe_closure_pa_1", "safe_closure_pp_1",
      "any_closure_3p_1", "any_closure_3p_2", "any_closure_3p_3", "any_closure_np_1", "any_closure_np_mv",
      "any_closure_4p_1", "any_closure_4p_2", "any_closure_4p_3", "any_closure_4p_4", "any_closure_np_2",

      "tc_and_a_or_a_la", "tc_or_a_and_a_la", "tc_and_a_or_a_laa", "tc_or_a_and_a_laa", "tc_and_a_or_a_l3a", "tc_or_a_and_a_l3a",
      "tc_or_a_a_and_a_a_la", "tc_or_a_and_a_a_l3a", "tc_and_a_or_a_a_la", "tc_or_a_and_a_a_la",
      "tc_when_la", "tc_when_laa", "tc_when_l3a", "tc_let_when_laa", "tc_let_unless_laa",
      "tc_cond_a_z_a_z_laa", "tc_cond_a_z_a_laa_z", "tc_cond_a_z_a_laa_laa", "tc_let_cond",
      "tc_if_a_z_la", "tc_if_a_z_laa", "tc_if_a_z_l3a", "tc_if_a_l3a_z", "tc_if_a_la_z", "tc_if_a_laa_z",
      "tc_if_a_z_if_a_z_la", "tc_if_a_z_if_a_la_z", "tc_if_a_z_if_a_z_laa", "tc_if_a_z_if_a_laa_z", "tc_if_a_z_if_a_l3a_l3a",
      "tc_cond_a_z_a_z_la", "tc_cond_a_z_a_la_z", "tc_cond_a_z_la", "tc_cond_a_la_z", "tc_cond_a_z_laa", "tc_cond_a_laa_z",
      "tc_let_if_a_z_la", "tc_let_if_a_z_laa", "if_a_z_let_if_a_z_laa",
      "tc_case_la", "tc_and_a_if_a_z_la", "tc_and_a_if_a_la_z",

      "recur_if_a_a_opa_laq", "recur_if_a_opa_laq_a", "recur_if_a_a_opla_aq", "recur_if_a_opla_aq_a",
      "recur_if_a_a_opla_laq", "recur_if_a_opla_laq_a",
      "recur_if_a_a_opa_la_laq", "recur_if_a_opa_la_laq_a",
      "recur_if_a_a_opla_la_laq", "recur_if_a_a_if_a_a_opla_laq", "recur_if_a_a_if_a_a_oplaa_laaq",
      "recur_if_a_a_opa_laaq", "recur_if_a_opa_laaq_a", "recur_if_a_a_opa_l3aq",
      "recur_if_a_a_lopl3a_l3a_l3aq", "recur_if_a_a_and_a_laa_laa",
      "recur_if_a_a_if_a_laa_opa_laaq",
      "recur_cond_a_a_op_a_laq", "recur_cond_a_a_op_a_laaq",
      "recur_cond_a_a_a_a_opla_laq", "recur_cond_a_a_a_a_oplaa_laaq", "recur_cond_a_a_a_a_opa_laaq",
      "recur_cond_a_a_a_laa_lopa_laaq", "recur_cond_a_a_a_laa_opa_laaq",
      "recur_and_a_or_a_laa_laa",
};

#define is_safe_c_op(op)  ((op >= OP_SAFE_C_NC) && (op < OP_THUNK))
#define is_unknown_op(op) ((op >= OP_UNKNOWN) && (op <= OP_UNKNOWN_NP))
#define is_h_safe_c_nc(P) (optimize_op(P) == HOP_SAFE_C_NC)
#define is_safe_c_s(P)    ((optimize_op(P) == OP_SAFE_C_S) || (optimize_op(P) == HOP_SAFE_C_S))
#define is_h_safe_c_s(P)  (optimize_op(P) == HOP_SAFE_C_S)
#define FIRST_UNHOPPABLE_OP OP_APPLY_SS

static bool is_h_optimized(s7_pointer p)
{
  return((is_optimized(p)) &&
	 (op_has_hop(p)) &&
	 (optimize_op(p) < FIRST_UNHOPPABLE_OP) &&  /* was OP_S? */
	 (optimize_op(p) > OP_GC_PROTECT));
}


/* -------------------------------- internal debugging apparatus -------------------------------- */

static int64_t heap_location(s7_scheme *sc, s7_pointer p)
{
  for (heap_block_t *hp = sc->heap_blocks; hp; hp = hp->next)
    if (((intptr_t)p >= hp->start) && ((intptr_t)p < hp->end))
      return(hp->offset + (((intptr_t)p - hp->start) / sizeof(s7_cell)));
  return(((s7_big_pointer)p)->big_hloc);
}

#if TRAP_SEGFAULT
#include <signal.h>
static Jmp_Buf senv; /* global here is not a problem -- it is used only to protect s7_is_valid */
static volatile sig_atomic_t can_jump = 0;
static void segv(int32_t unused) {if (can_jump) LongJmp(senv, 1);}
#endif

bool s7_is_valid(s7_scheme *sc, s7_pointer arg)
{
  bool result = false;
  if (!arg) return(false);
#if TRAP_SEGFAULT
  if (SetJmp(senv, 1) == 0)
    {
      void (*old_segv)(int32_t sig);
      can_jump = 1;
      old_segv = signal(SIGSEGV, segv);
#endif
      if ((unchecked_type(arg) > T_FREE) &&
	  (unchecked_type(arg) < NUM_TYPES))
	{
	  if (!in_heap(arg))
	    result = true;
	  else
	    {
	      int64_t loc = heap_location(sc, arg);
	      if ((loc >= 0) && (loc < sc->heap_size))
		result = (sc->heap[loc] == arg);
	    }}
#if TRAP_SEGFAULT
      signal(SIGSEGV, old_segv);
    }
  else result = false;
  can_jump = 0;
#endif
  return(result);
}

void s7_show_let(s7_scheme *sc) /* debugging convenience */
{
  for (s7_pointer olet = sc->curlet; is_let(T_Lid(olet)); olet = let_outlet(olet))
    {
      if (olet == sc->owlet)
	fprintf(stderr, "(owlet): ");
      else
	if (is_funclet(olet))
	  fprintf(stderr, "(%s funclet): ", display(funclet_function(olet)));
	else
	  if (olet == sc->shadow_rootlet)
	    fprintf(stderr, "(shadow rootlet): ");
      fprintf(stderr, "%s\n", display(olet));
    }
}

#define safe_print(Code)	   \
  do {				   \
    bool old_open = sc->has_openlets, old_stop = sc->stop_at_error;  \
    sc->has_openlets = false;      \
    sc->stop_at_error = false;	   \
    Code;			   \
    sc->stop_at_error = old_stop;  \
    sc->has_openlets = old_open;   \
  } while (0)

void s7_show_history(s7_scheme *sc)
{
#if WITH_HISTORY
  if (sc->cur_code == sc->history_sink)
    fprintf(stderr, "history diabled\n");
  else
    {
      int32_t size = sc->history_size;
      s7_pointer p = cdr(sc->cur_code);
      fprintf(stderr, "history:\n");
      for (int32_t i = 0; i < size; i++, p = cdr(p)) /* stepper "i" is not redundant */
	safe_print(fprintf(stderr, "%d: %s\n", i, display_80(car(p))));
      fprintf(stderr, "\n");
    }
#else
  fprintf(stderr, "%s\n", display(sc->cur_code));
#endif
}

#define stack_code(Stack, Loc)  stack_element(Stack, Loc - 3)
#define stack_let(Stack, Loc)   stack_element(Stack, Loc - 2)
#define stack_args(Stack, Loc)  stack_element(Stack, Loc - 1)
#define stack_op(Stack, Loc)    ((opcode_t)(stack_element(Stack, Loc)))

void s7_show_stack(s7_scheme *sc)
{
  fprintf(stderr, "stack:\n");
  for (int64_t i = current_stack_top(sc) - 1; i >= 3; i -= 4)
    fprintf(stderr, "  %s\n", op_names[stack_op(sc->stack, i)]);
}

#define UNUSED_BITS 0xfc00000000c0 /* high 6 bits of optimizer code + high 2 bits of type */

static char *describe_type_bits(s7_scheme *sc, s7_pointer obj) /* used outside S7_DEBUGGING in display_fallback (for display_functions) */
{
  uint64_t full_typ = full_type(obj);
  uint8_t typ = unchecked_type(obj);
  char *buf;
  char str[900];

  str[0] = '\0';
  catstrs(str, 900,	  /* if debugging, all of these bits are being watched, so we need to access them directly */
	  /* bit 8 (the first 8 bits (after the 8 type bits) are easy...) */
	  ((full_typ & T_MULTIFORM) != 0) ?      ((is_any_closure(obj)) ?
						  (((full_typ & T_ONE_FORM) != 0) ? " closure-one-form-has-fx" : " closure-multiform") :
						  " ?0?") : "",
	  /* bit 9 */
	  ((full_typ & T_SYNTACTIC) != 0) ?      (((is_pair(obj)) || (is_syntax(obj)) || (is_normal_symbol(obj))) ?
						  " syntactic" :
						  " ?1?") : "",
	  /* bit 10 */
	  ((full_typ & T_SIMPLE_ARG_DEFAULTS) != 0) ? ((is_pair(obj)) ? " simple-args|in-use" :
						       ((is_any_closure(obj)) ? " closure-one-form" :
							" ?2?")) : "",
	  /* bit 11 */
	  ((full_typ & T_OPTIMIZED) != 0) ?      ((is_c_function(obj)) ? " scope-safe" :
						  ((is_pair(obj)) ? " optimized" :
						   " ?3?")) : "",
	  /* bit 12 */
	  ((full_typ & T_SAFE_CLOSURE) != 0) ?   (((has_closure_let(obj)) || (is_pair(obj))) ? " safe-closure" : " ?4?") : "",
	  /* bit 13 */
	  ((full_typ & T_DONT_EVAL_ARGS) != 0) ? (((is_any_macro(obj)) || (is_syntax(obj))) ? " dont-eval-args" : " ?5?") : "",
	  /* bit 14 */
	  ((full_typ & T_EXPANSION) != 0) ?      (((is_normal_symbol(obj)) || (is_either_macro(obj))) ? " expansion" : " ?6?") : "",
	  /* bit 15 */
	  ((full_typ & T_MULTIPLE_VALUE) != 0) ? ((is_symbol(obj)) ? " matched" :
						  ((is_pair(obj)) ? " values|matched" :
						   " ?7?")) : "",
	  /* bit 16 */
	  ((full_typ & T_GLOBAL) != 0) ?         ((is_pair(obj)) ? " unsafe-do" :
						  (((is_symbol(obj)) || (is_syntax(obj))) ? " global" :
						   ((is_let(obj)) ? " dox_slot1" :
						    " ?8?"))) : "",
	  /* bit 17 */
	  ((full_typ & T_COLLECTED) != 0) ?      " collected" : "",
	  /* bit 18 */
	  ((full_typ & T_LOCATION) != 0) ?       ((is_pair(obj)) ? " line-number" :
						  ((is_input_port(obj)) ? " loader-port" :
						   ((is_let(obj)) ? " with-let" :
						    ((is_any_procedure(obj)) ? " simple-defaults" :
						     ((is_slot(obj)) ? " has-setter" :
						      " ?10?"))))) : "",
	  /* bit 19 */
	  ((full_typ & T_SHARED) != 0) ?         ((is_sequence(obj)) ? " shared" : " ?11?") : "",
	  /* bit 20 */
	  ((full_typ & T_LOCAL) != 0) ?          ((is_normal_symbol(obj)) ? " local" :
						  ((is_pair(obj)) ? " high-c" :
						   " ?12?")) : "",
	  /* bit 21 */
	  ((full_typ & T_SAFE_PROCEDURE) != 0) ? ((is_applicable(obj)) ? " safe-procedure" : " ?13?") : "",
	  /* bit 22 */
	  ((full_typ & T_CHECKED) != 0) ?        (((is_pair(obj)) || (is_slot(obj))) ? " checked" :
						  ((is_symbol(obj)) ? " all-integer" :
						   " ?14?")) : "",
	  /* bit 23 */
	  ((full_typ & T_UNSAFE) != 0) ?         ((is_symbol(obj)) ? " clean-symbol" :
						  ((is_slot(obj)) ? " has-stepper" :
						   ((is_pair(obj)) ? " unsafely-opt|no-float-opt" :
						    ((is_let(obj)) ? " dox-slot2" :
						     " ?15?")))) : "",
	  /* bit 24 */
	  ((full_typ & T_IMMUTABLE) != 0) ?      " immutable" : "",
	  /* bit 25 */
	  ((full_typ & T_SETTER) != 0) ?         ((is_normal_symbol(obj)) ? " setter" :
						  ((is_pair(obj)) ? " allow-other-keys|no-int-opt" :
						   ((is_slot(obj)) ? " has-expression" :
						    ((is_c_function_star(obj)) ? " allow-other-keys" :
						     " ?17?")))) : "",
	  /* bit 26 */
	  ((full_typ & T_MUTABLE) != 0) ?        ((is_number(obj)) ? " mutable" :
						  ((is_symbol(obj)) ? " has-keyword" :
						   ((is_let(obj)) ? " ref-fallback" :
						    ((is_iterator(obj)) ? " mark-sequence" :
						     ((is_slot(obj)) ? " step-end" :
						      ((is_pair(obj)) ? " no-opt" :
						       " ?18?")))))) : "",
	  /* bit 27 */
	  ((full_typ & T_SAFE_STEPPER) != 0) ?   ((is_let(obj)) ? " set-fallback" :
						  ((is_slot(obj)) ? " safe-stepper" :
						   ((is_c_function(obj)) ? " maybe-safe" :
						    ((is_number(obj)) ? " print-name" :
						     ((is_pair(obj)) ? " direct-opt" :
						      ((is_hash_table(obj)) ? " weak-hash" :
						       ((is_any_macro(obj)) ? " pair-macro-set" :
							((is_symbol(obj)) ? " all-float" :
							 " ?19?")))))))) : "",
	  /* bit 28, for c_function case see sc->apply */
	  ((full_typ & T_COPY_ARGS) != 0) ?      (((is_pair(obj)) || (is_any_macro(obj)) || (is_syntax(obj)) ||
						   (is_any_closure(obj)) || (is_c_function(obj))) ? " copy-args" :
						  " ?20?") : "",
	  /* bit 29 */
	  ((full_typ & T_GENSYM) != 0) ?         ((is_let(obj)) ? " funclet" :
						  ((is_normal_symbol(obj)) ? " gensym" :
						   ((is_string(obj)) ? " documented-symbol" :
						    ((is_hash_table(obj)) ? " hash-chosen" :
						     ((is_pair(obj)) ? " fx-treed" :
						      ((is_any_vector(obj)) ? " subvector" :
						       ((is_slot(obj)) ? " has-pending-value" :
							((is_any_closure(obj)) ? " unknopt" :
							 " ?21?")))))))) : "",
	  /* bit 30 */
	  ((full_typ & T_HAS_METHODS) != 0) ?    (((is_let(obj)) || (is_c_object(obj)) || (is_any_closure(obj)) ||
						   (is_any_macro(obj)) || (is_c_pointer(obj))) ? " has-methods" : " ?22?") : "",
	  /* bit 31 */
	  ((full_typ & T_ITER_OK) != 0) ?        ((is_iterator(obj)) ? " iter-ok" :
						  ((is_pair(obj)) ? " step-end-ok/set-implicit-ok" :
						   ((is_slot(obj)) ? " in-rootlet" :
						    ((is_c_function(obj)) ? " bool-function" :
						     " ?23?")))) : "",
	  /* bit 24+24 */
	  ((full_typ & T_FULL_SYMCONS) != 0) ?   ((is_symbol(obj)) ? " possibly-constant" :
						  ((is_any_procedure(obj)) ? " has-let-arg" :
						   ((is_hash_table(obj)) ? " has-value-type" :
						    ((is_pair(obj)) ? " int-optable" :
						     " ?24?")))) : "",
	  /* bit 25+24 */
	  ((full_typ & T_FULL_HAS_LET_FILE) != 0) ? ((is_let(obj)) ? " has-let-file" :
						     ((is_any_vector(obj)) ? " typed-vector" :
						      ((is_hash_table(obj)) ? " typed-hash-table" :
						       ((is_c_function(obj)) ? " has-bool-setter" :
							((is_slot(obj)) ? " rest-slot" :
							 (((is_pair(obj)) || (is_closure_star(obj))) ? " no-defaults" :
							  " ?25?")))))) : "",
	  /* bit 26+24 */
	  ((full_typ & T_FULL_DEFINER) != 0) ?   ((is_normal_symbol(obj)) ? " definer" :
						  ((is_pair(obj)) ? " has-fx" :
						   ((is_slot(obj)) ? " slot-defaults" :
						    ((is_iterator(obj)) ? " weak-hash-iterator" :
						     ((is_hash_table(obj)) ? " has-key-type" :
						      ((is_let(obj)) ? " maclet" :
						       ((is_c_function(obj)) ? " func-definer" :
							((is_syntax(obj)) ? " syntax-definer" :
							 " ?26?")))))))) : "",
	  /* bit 27+24 */
	  ((full_typ & T_FULL_BINDER) != 0) ?    ((is_pair(obj)) ? " tree-collected" :
						  ((is_hash_table(obj)) ? " simple-values" :
						   ((is_normal_symbol(obj)) ? " binder" :
						    ((is_c_function(obj)) ? " safe-args" :
						     " ?27?")))) : "",
	  /* bit 28+24 */
	  ((full_typ & T_VERY_SAFE_CLOSURE) != 0) ? (((is_pair(obj)) || (is_any_closure(obj))) ? " very-safe-closure" :
						     ((is_let(obj)) ? " baffle-let" :
						      " ?28?")) : "",
	  /* bit 29+24 */
	  ((full_typ & T_CYCLIC) != 0) ?         (((is_simple_sequence(obj)) || (t_structure_p[type(obj)]) ||
						   (is_any_closure(obj))) ? " cyclic" : " ?29?") : "",
	  /* bit 30+24 */
	  ((full_typ & T_CYCLIC_SET) != 0) ?     (((is_simple_sequence(obj)) || (t_structure_p[type(obj)]) ||
						   (is_any_closure(obj))) ? " cyclic-set" : " ?30?") : "",
	  /* bit 31+24 */
	  ((full_typ & T_KEYWORD) != 0) ?        ((is_symbol(obj)) ? " keyword" :
						  ((is_pair(obj)) ? " fx-treeable" :
						   " ?31?")) : "",
	  /* bit 32+24 */
	  ((full_typ & T_FULL_SIMPLE_ELEMENTS) != 0) ? ((is_normal_vector(obj)) ? " simple-elements" :
							((is_hash_table(obj)) ? " simple-keys" :
							 ((is_normal_symbol(obj)) ? " safe-setter" :
							  ((is_pair(obj)) ? " float-optable" :
							   ((typ >= T_C_MACRO) ? " function-simple-elements" :
							    " 32?"))))) : "",
	  /* bit 33+24 */
	  ((full_typ & T_FULL_CASE_KEY) != 0) ?  ((is_symbol(obj)) ? " case-key" :
						  ((is_pair(obj)) ? " opt1-func-listed" :
						   " ?33?")) : "",
	  /* bit 34+24 */
	  ((full_typ & T_FULL_HAS_GX) != 0) ?    ((is_pair(obj)) ? " has-gx" : " ?34?") : "",
	  /* bit 35+24 */
	  ((full_typ & T_FULL_UNKNOPT) != 0) ?    ((is_pair(obj)) ? " unknopt" : " ?35?") : "",
	  /* bit 36+24 */
	  ((full_typ & T_FULL_SAFETY_CHECKED) != 0) ? ((is_pair(obj)) ? " safety-checked" : " ?36?") : "",
	  /* bit 37+24 */
	  ((full_typ & T_FULL_HAS_FN) != 0) ?    ((is_pair(obj)) ? " has-fn" : " ?37") : "",
	  /* bit 62 */
	  ((full_typ & T_UNHEAP) != 0) ?         " unheap" : "",
	  /* bit 63 */
	  ((full_typ & T_GC_MARK) != 0) ?        " gc-marked" : "",

	  ((full_typ & UNUSED_BITS) != 0) ?      " unused bits set?" : "",
	  ((is_symbol(obj)) && (((uint8_t)(symbol_type(obj) & 0xff) >= NUM_TYPES) || ((symbol_type(obj) & ~0xffff) != 0))) ? " bad-symbol-type" : "",
	  NULL);

  buf = (char *)Malloc(1024);
  snprintf(buf, 1024, "type: %s? (%d), opt_op: %d %s, flags: #x%" PRIx64 "%s",
	   type_name(sc, obj, NO_ARTICLE), typ, optimize_op(obj), (optimize_op(obj) < NUM_OPS) ? op_names[optimize_op(obj)] : "", full_typ, str);
  return(buf);
}

/* snprintf returns the number of bytes that would have been written: (display (c-pointer 123123123 (symbol (make-string 130 #\a)))) */
#define clamp_length(NLen, Len) (((NLen) < (Len)) ? (NLen) : (Len))

#if S7_DEBUGGING
static bool has_odd_bits(s7_pointer obj)
{
  uint64_t full_typ = full_type(obj);
  if ((full_typ & UNUSED_BITS) != 0) return(true);
  if (((full_typ & T_MULTIFORM) != 0) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_KEYWORD) != 0) && (!is_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_SYNTACTIC) != 0) && (!is_syntax(obj)) && (!is_pair(obj)) && (!is_normal_symbol(obj))) return(true);
  if (((full_typ & T_SIMPLE_ARG_DEFAULTS) != 0) && (!is_pair(obj)) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_OPTIMIZED) != 0) && (!is_c_function(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_SAFE_CLOSURE) != 0) && (!is_any_closure(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_SAFE_PROCEDURE) != 0) && (!is_applicable(obj))) return(true);
  if (((full_typ & T_EXPANSION) != 0) && (!is_normal_symbol(obj)) && (!is_either_macro(obj))) return(true);
  if (((full_typ & T_MULTIPLE_VALUE) != 0) && (!is_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_GLOBAL) != 0) && (!is_pair(obj)) && (!is_symbol(obj)) && (!is_let(obj)) && (!is_syntax(obj))) return(true);
  if (((full_typ & T_ITER_OK) != 0) && (!is_iterator(obj)) && (!is_pair(obj)) && (!is_slot(obj)) && (!is_c_function(obj))) return(true);
  if (((full_typ & T_LOCAL) != 0) && (!is_normal_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_UNSAFE) != 0) && (!is_symbol(obj)) && (!is_slot(obj)) && (!is_let(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_VERY_SAFE_CLOSURE) != 0) && (!is_pair(obj)) && (!is_any_closure(obj)) && (!is_let(obj))) return(true);
  if (((full_typ & T_FULL_CASE_KEY) != 0) && (!is_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_FULL_UNKNOPT) != 0) && (!is_pair(obj))) return(true);
  if (((full_typ & T_FULL_SAFETY_CHECKED) != 0) && (!is_pair(obj))) return(true);
  if (((full_typ & T_FULL_HAS_GX) != 0) && (!is_pair(obj))) return(true);
  if (((full_typ & T_DONT_EVAL_ARGS) != 0) && (!is_any_macro(obj)) && (!is_syntax(obj))) return(true);
  if (((full_typ & T_CHECKED) != 0) && (!is_slot(obj)) && (!is_pair(obj)) && (!is_symbol(obj))) return(true);
  if (((full_typ & T_SHARED) != 0) && (!t_sequence_p[type(obj)]) && (!t_structure_p[type(obj)]) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_COPY_ARGS) != 0) && (!is_pair(obj)) &&
      (!is_any_macro(obj)) && (!is_any_closure(obj)) && (!is_c_function(obj)) && (!is_syntax(obj)))
    return(true);
  if (((full_typ & T_FULL_SYMCONS) != 0) &&
      (!is_symbol(obj)) && (!is_any_procedure(obj)) && (!is_hash_table(obj)) && (!is_pair(obj)))
    return(true);
  if (((full_typ & T_FULL_BINDER) != 0) &&
      ((!is_pair(obj)) && (!is_hash_table(obj)) && (!is_normal_symbol(obj)) && (!is_c_function(obj)) && (!is_syntax(obj))))
    return(true);
  if (((full_typ & T_FULL_DEFINER) != 0) &&
      (!is_normal_symbol(obj)) && (!is_c_function(obj)) && (!is_pair(obj)) && (!is_slot(obj)) && (!is_iterator(obj)) &&
      (!is_hash_table(obj)) && (!is_let(obj)) && (!is_syntax(obj)))
    return(true);
  if (((full_typ & T_FULL_HAS_LET_FILE) != 0) &&
      (!is_let(obj)) && (!is_any_vector(obj)) && (!is_hash_table(obj)) && (!is_c_function(obj)) &&
      (!is_slot(obj)) && (!is_pair(obj)) && (!is_closure_star(obj)))
    return(true);
  if (((full_typ & T_SAFE_STEPPER) != 0) &&
      (!is_let(obj)) && (!is_slot(obj)) && (!is_c_function(obj)) && (!is_number(obj)) &&
      (!is_pair(obj)) && (!is_hash_table(obj)) && (!is_any_macro(obj)) && (!is_symbol(obj)))
    return(true);
  if (((full_typ & T_SETTER) != 0) &&
      (!is_slot(obj)) && (!is_normal_symbol(obj)) && (!is_pair(obj)) && (!is_let(obj)) && (!is_c_function_star(obj)))
    return(true);
  if (((full_typ & T_LOCATION) != 0) &&
      (!is_pair(obj)) && (!is_input_port(obj)) && (!is_let(obj)) && (!is_any_procedure(obj)) && (!is_slot(obj)))
    return(true);
  if (((full_typ & T_MUTABLE) != 0) &&
      (!is_number(obj)) && (!is_symbol(obj)) && (!is_let(obj)) && (!is_iterator(obj)) && (!is_slot(obj)) && (!is_let(obj)) && (!is_pair(obj)))
    return(true);
  if (((full_typ & T_GENSYM) != 0) && (!is_slot(obj)) && (!is_any_closure(obj)) &&
      (!is_let(obj)) && (!is_symbol(obj)) && (!is_string(obj)) && (!is_hash_table(obj)) && (!is_pair(obj)) && (!is_any_vector(obj)))
    return(true);
  if (((full_typ & T_FULL_SIMPLE_ELEMENTS) != 0) &&
      ((!is_normal_vector(obj)) && (!is_hash_table(obj)) && (!is_normal_symbol(obj)) && (!is_pair(obj)) && (unchecked_type(obj) < T_C_MACRO)))
    return(true);
  if (((full_typ & T_CYCLIC) != 0) && (!is_simple_sequence(obj)) && (!t_structure_p[type(obj)]) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_CYCLIC_SET) != 0) && (!is_simple_sequence(obj)) && (!t_structure_p[type(obj)]) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_FULL_HAS_FN) != 0) && (!is_pair(obj))) return(true);
  if (is_symbol(obj))
    {
      if ((uint8_t)(symbol_type(obj) & 0xff) >= NUM_TYPES)
	return(true);
      if ((symbol_type(obj) & ~0xffff) != 0)
	return(true);
    }
  if ((signed_type(obj) == 0) && ((full_typ & T_GC_MARK) != 0)) return(true);
  return(false);
}

static const char *check_name(s7_scheme *sc, int32_t typ)
{
  if ((typ >= 0) && (typ < NUM_TYPES))
    {
      s7_pointer p = sc->prepackaged_type_names[typ];
      if (is_string(p)) return(string_value(p));
    }
  return("unknown type!");
}

#if REPORT_ROOTLET_REDEF
static void set_local_1(s7_scheme *sc, s7_pointer symbol, const char *func, int32_t line)
{
  if (is_global(symbol))
    {
      fprintf(stderr, "%s[%d]: %s%s%s in %s\n",
	      func, line,
	      BOLD_TEXT, s7_object_to_c_string(sc, symbol), UNBOLD_TEXT,
	      display_80(sc->cur_code));
      /* gdb_break(); */
    }
  full_type(symbol) = (full_type(symbol) & ~(T_DONT_EVAL_ARGS | T_GLOBAL | T_SYNTACTIC));
}
#endif

static char *safe_object_to_string(s7_pointer p)
{
  char *buf = (char *)Malloc(128);
  snprintf(buf, 128, "type: %d", unchecked_type(p));
  return(buf);
}

static void complain(const char* complaint, s7_pointer p, const char *func, int32_t line, uint8_t typ)
{
  fprintf(stderr, complaint, BOLD_TEXT, func, line, check_name(cur_sc, typ), safe_object_to_string(p), UNBOLD_TEXT);
  if (cur_sc->stop_at_error) abort();
}

static char* show_debugger_bits(s7_pointer p)
{
  char *bits_str = (char *)Malloc(512);
  int64_t bits = p->debugger_bits;
  snprintf(bits_str, 512, " %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
	   ((bits & OPT1_SET) != 0) ? " opt1_set" : "",
	   ((bits & OPT1_FAST) != 0) ? " opt1_fast" : "",
	   ((bits & OPT1_CFUNC) != 0) ? " opt1_cfunc" : "",
	   ((bits & OPT1_CLAUSE) != 0) ? " opt1_clause" : "",
	   ((bits & OPT1_LAMBDA) != 0) ? " opt1_lambda" : "",
	   ((bits & OPT1_SYM) != 0) ? " opt1_sym" : "",
	   ((bits & OPT1_PAIR) != 0) ? " opt1_pair" : "",
	   ((bits & OPT1_CON) != 0) ? " opt1_con" : "",
	   ((bits & OPT1_ANY) != 0) ? " opt1_any" : "",
	   ((bits & OPT1_HASH) != 0) ? " opt1_raw_hash" : "",

	   ((bits & OPT2_SET) != 0) ? " opt2_set" : "",
	   ((bits & OPT2_KEY) != 0) ? " opt2_any" : "",
	   ((bits & OPT2_SLOW) != 0) ? " opt2_slow" : "",
	   ((bits & OPT2_SYM) != 0) ? " opt2_sym" : "",
	   ((bits & OPT2_PAIR) != 0) ? " opt2_pair" : "",
	   ((bits & OPT2_CON) != 0) ? " opt2_con" : "",
	   ((bits & OPT2_FX) != 0) ? " opt2_fx" : "",
	   ((bits & OPT2_FN) != 0) ? " opt2_fn" : "",
	   ((bits & OPT2_LAMBDA) != 0) ? " opt2_lambda" : "",
	   ((bits & OPT2_DIRECT) != 0) ? " opt2_direct" : "",
	   ((bits & OPT2_NAME) != 0) ? " opt2_raw_name" : "",
	   ((bits & OPT2_INT) != 0) ? " opt2_int" : "",

	   ((bits & OPT3_SET) != 0) ? " opt3_set" : "",
	   ((bits & OPT3_ARGLEN) != 0) ? " opt3_arglen" : "",
	   ((bits & OPT3_SYM) != 0) ? " opt3_sym" : "",
	   ((bits & OPT3_CON) != 0) ? " opt3_con" : "",
	   ((bits & OPT3_AND) != 0) ? " opt3_pair " : "",
	   ((bits & OPT3_ANY) != 0) ? " opt3_any " : "",
	   ((bits & OPT3_LET) != 0) ? " opt3_let " : "",
	   ((bits & OPT3_BYTE) != 0) ? " opt3_byte " : "",
	   ((bits & OPT3_DIRECT) != 0) ? " opt3_direct" : "",
	   ((bits & OPT3_LOCATION) != 0) ? " opt3_location" : "",
	   ((bits & OPT3_LEN) != 0) ? " opt3_len" : "",
	   ((bits & OPT3_INT) != 0) ? " opt3_int" : "",

	   ((bits & L_HIT) != 0) ? " let_set" : "",
	   ((bits & L_FUNC) != 0) ? " let_func" : "",
	   ((bits & L_DOX) != 0) ? " let_dox" : "");
  return(bits_str);
}

static s7_pointer check_ref(s7_pointer p, uint8_t expected_type, const char *func, int32_t line, const char *func1, const char *func2)
{
  if (!p)
    fprintf(stderr, "%s%s[%d]: null pointer passed to check_ref%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
  else
    {
      uint8_t typ = unchecked_type(p);
      if (typ != expected_type)
	{
	  if ((!func1) || (typ != T_FREE))
	    {
	      fprintf(stderr, "%s%s[%d]: not %s, but %s (%s)%s\n",
		      BOLD_TEXT,
		      func, line, check_name(cur_sc, expected_type), check_name(cur_sc, typ), safe_object_to_string(p),
		      UNBOLD_TEXT);
	      if (cur_sc->stop_at_error) abort();
	    }
	  else
	    if ((strcmp(func, func1) != 0) &&
		((!func2) || (strcmp(func, func2) != 0)))
	      {
		fprintf(stderr, "%s%s[%d]: free cell, not %s%s\n", BOLD_TEXT, func, line, check_name(cur_sc, expected_type), UNBOLD_TEXT);
		if (cur_sc->stop_at_error) abort();
	      }}}
  return(p);
}

static s7_pointer check_let_ref(s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_ref(p, T_LET, func, line, NULL, NULL);
  if ((p->debugger_bits & L_HIT) == 0) fprintf(stderr, "%s[%d]: let not set\n", func, line);
  if ((p->debugger_bits & L_MASK) != role) fprintf(stderr, "%s[%d]: let bad role\n", func, line);
  return(p);
}

static s7_pointer check_let_set(s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_ref(p, T_LET, func, line, NULL, NULL);
  p->debugger_bits &= (~L_MASK);
  p->debugger_bits |= (L_HIT | role);
  return(p);
}

static s7_pointer check_ref2(s7_pointer p, uint8_t expected_type, int32_t other_type, const char *func, int32_t line, const char *func1, const char *func2)
{
  if (!p)
    fprintf(stderr, "%s[%d]: null pointer passed to check_ref2\n", func, line);
  else
    {
      uint8_t typ = unchecked_type(p);
      if ((typ != expected_type) && (typ != other_type))
	return(check_ref(p, expected_type, func, line, func1, func2));
    }
  return(p);
}

static s7_pointer check_ref3(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if ((typ != T_INPUT_PORT) && (typ != T_OUTPUT_PORT) && (typ != T_FREE))
    complain("%s%s[%d]: not a port, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref4(s7_pointer p, const char *func, int32_t line)
{
  if ((strcmp(func, "sweep") != 0) &&
      (strcmp(func, "process_multivector") != 0))
    {
      uint8_t typ = unchecked_type(p);
      if (!t_vector_p[typ])
	complain("%s%s[%d]: not a vector, but %s (%s)%s\n", p, func, line, typ);
    }
  return(p);
}

static s7_pointer check_ref5(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if (!t_has_closure_let[typ])
    complain("%s%s[%d]: not a closure, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref6(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if (typ < T_C_MACRO)
    complain("%s%s[%d]: not a c function, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref7(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if ((typ < T_INTEGER) || (typ > T_COMPLEX))
    complain("%s%s[%d]: not a number, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref8(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if ((!t_sequence_p[typ]) && (!t_structure_p[typ]) && (!is_any_closure(p))) /* closure as iterator -- see s7test */
    complain("%s%s[%d]: not a sequence or structure, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref9(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if ((typ != T_LET) && (typ != T_C_OBJECT) && (!is_any_closure(p)) && (!is_any_macro(p)) && (typ != T_C_POINTER))
    complain("%s%s[%d]: not a possible method holder, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref10(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if ((typ != T_PAIR) && (typ != T_NIL) && (typ != T_SYMBOL))
    complain("%s%s[%d]: arglist is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref11(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  if ((!t_applicable_p[typ]) && (p != cur_sc->F))
    complain("%s%s[%d]: applicable object is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref12(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  if (is_slot_end(p)) return(p);
  typ = unchecked_type(p);
  if ((typ != T_SLOT) && (typ != T_NIL)) /* unset slots are nil */
    complain("%s%s[%d]: slot is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref13(s7_pointer p, const char *func, int32_t line)
{
  if (!is_any_vector(p))
    complain("%s%s[%d]: subvector is %s (%s)%s?\n", p, func, line, unchecked_type(p));
  if (!is_subvector(p))
    complain("%s%s[%d]: subvector is %s (%s), but not a subvector?%s\n", p, func, line, unchecked_type(p));
  return(p);
}

static s7_pointer check_ref14(s7_pointer p, const char *func, int32_t line)
{
  if ((!is_any_procedure(p)) && (!s7_is_boolean(p)))
    complain("%s%s[%d]: setter (with let arg) is %s (%s)%s?\n", p, func, line, unchecked_type(p));
  return(p);
}

static s7_pointer check_ref15(s7_pointer p, const char *func, int32_t line) /* called in mark_let so s7_scheme* for cur_sc is difficult */
{
  uint8_t typ = unchecked_type(p);
  check_nref(p, func, line);
  if ((is_multiple_value(p)) &&
      (!safe_strcmp(func, "mark_slot"))) /* match == multiple-values which causes false error messages */
    complain("%s%s[%d]: slot value is a multiple-value, %s (%s)%s?\n", p, func, line, typ);
  if (has_odd_bits(p))
    {char *s; fprintf(stderr, "odd bits: %s\n", s = describe_type_bits(cur_sc, p)); free(s);}
  return(p);
}

static s7_pointer check_ref16(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = unchecked_type(p);
  check_nref(p, func, line);
  if ((typ != T_LET) && (typ != T_NIL))
    complain("%s%s[%d]: not a let or nil, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref17(s7_pointer p, const char *func, int32_t line)
{
  if ((!is_any_macro(p)) || (is_c_macro(p)))
    complain("%s%s[%d]: macro is %s (%s)%s?\n", p, func, line, unchecked_type(p));
  return(p);
}

static s7_pointer check_cell(s7_scheme *sc, s7_pointer p, const char *func, int32_t line)
{
  if (!p)
    {
      fprintf(stderr, "%s%s[%d]: null pointer!%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  else
    if (unchecked_type(p) >= NUM_TYPES)
      {
	fprintf(stderr, "%s%s[%d]: attempt to use messed up cell (type: %d)%s\n", BOLD_TEXT, func, line, unchecked_type(p), UNBOLD_TEXT);
	if (sc->stop_at_error) abort();
      }
  return(p);
}

static void print_gc_info(s7_scheme *sc, s7_pointer obj, int32_t line)
{
  if (!obj)
    fprintf(stderr, "[%d]: obj is %p\n", line, obj);
  else
    if (unchecked_type(obj) != T_FREE)
      fprintf(stderr, "[%d]: %p type is %d?\n", line, obj, unchecked_type(obj));
    else
      {
	s7_int free_type = full_type(obj);
	char *bits;
	char fline[128];
	full_type(obj) = obj->current_alloc_type;
	sc->printing_gc_info = true;
	bits = describe_type_bits(sc, obj); /* this func called in type macro */
	sc->printing_gc_info = false;
	full_type(obj) = free_type;
	if (obj->explicit_free_line > 0)
	  snprintf(fline, 128, ", freed at %d, ", obj->explicit_free_line);
	fprintf(stderr, "%s%p is free (line %d, alloc type: %s %" ld64 " #x%" PRIx64 " (%s)), current: %s[%d], previous: %s[%d], %sgc: %s[%d]%s",
		BOLD_TEXT, obj, line, s7_type_names[obj->current_alloc_type & 0xff], obj->current_alloc_type, obj->current_alloc_type,
		bits, obj->current_alloc_func, obj->current_alloc_line, obj->previous_alloc_func, obj->previous_alloc_line,
		(obj->explicit_free_line > 0) ? fline : "", obj->gc_func, obj->gc_line,	UNBOLD_TEXT);
	if (S7_DEBUGGING) fprintf(stderr, ", last gc line: %d", sc->last_gc_line);
	fprintf(stderr, "\n");
	free(bits);
      }
  if (sc->stop_at_error) abort();
}

static s7_pointer check_nref(s7_pointer p, const char *func, int32_t line)
{
  check_cell(cur_sc, p, func, line);
  if (unchecked_type(p) == T_FREE)
    {
      fprintf(stderr, "%s%s[%d]: attempt to use free cell%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
      print_gc_info(cur_sc, p, line);
    }
  return(p);
}

static const char *opt1_role_name(uint64_t role)
{
  if (role == OPT1_FAST) return("opt1_fast");
  if (role == OPT1_CFUNC) return("opt1_cfunc");
  if (role == OPT1_LAMBDA) return("opt1_lambda");
  if (role == OPT1_CLAUSE) return("opt1_clause");
  if (role == OPT1_SYM) return("opt1_sym");
  if (role == OPT1_PAIR) return("opt1_pair");
  if (role == OPT1_CON) return("opt1_con");
  if (role == OPT1_ANY) return("opt1_any");
  return((role == OPT1_HASH) ? "opt1_hash" : "opt1_unknown");
}

static const char *opt2_role_name(uint64_t role)
{
  if (role == OPT2_FX) return("opt2_fx");
  if (role == OPT2_FN) return("opt2_fn");
  if (role == OPT2_KEY) return("opt2_any");
  if (role == OPT2_SLOW) return("opt2_slow");
  if (role == OPT2_SYM) return("opt2_sym");
  if (role == OPT2_PAIR) return("opt2_pair");
  if (role == OPT2_CON) return("opt2_con");
  if (role == OPT2_LAMBDA) return("opt2_lambda");
  if (role == OPT2_DIRECT) return("opt2_direct");
  if (role == OPT2_INT) return("opt2_int");
  return((role == OPT2_NAME) ? "opt2_raw_name" : "opt2_unknown");
}

static const char *opt3_role_name(uint64_t role)
{
  if (role == OPT3_ARGLEN) return("opt3_arglen");
  if (role == OPT3_SYM) return("opt3_sym");
  if (role == OPT3_CON) return("opt3_con");
  if (role == OPT3_AND) return("opt3_pair");
  if (role == OPT3_ANY) return("opt3_any");
  if (role == OPT3_LET) return("opt3_let");
  if (role == OPT3_BYTE) return("opt3_byte");
  if (role == OPT3_DIRECT) return("direct_opt3");
  if (role == OPT3_LEN) return("opt3_len");
  if (role == OPT3_INT) return("opt3_int");
  return((role == OPT3_LOCATION) ? "opt3_location" : "opt3_unknown");
}

static void show_opt1_bits(s7_pointer p, const char *func, int32_t line, uint64_t role)
{
  char *bits = show_debugger_bits(p);
  fprintf(stderr, "%s%s[%d]%s: opt1: %p->%p wants %s, debugger bits are %" PRIx64 "%s but expects %" ld64,
	  BOLD_TEXT, func, line, UNBOLD_TEXT,
	  p, p->object.cons.opt1,
	  opt1_role_name(role),
	  p->debugger_bits, bits, (s7_int)role);
  free(bits);
}

static s7_pointer opt1_1(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  if ((!opt1_is_set(p)) ||
      ((!opt1_role_matches(p, role)) &&
       (role != OPT1_ANY)))
    {
      show_opt1_bits(p, func, line, role);
      if (sc->stop_at_error) abort();
    }
  return(p->object.cons.opt1);
}

static void base_opt1(s7_pointer p, uint64_t role)
{
  set_opt1_role(p, role);
  set_opt1_is_set(p);
}

static s7_pointer set_opt1_1(s7_pointer p, s7_pointer x, uint64_t role)
{
  p->object.cons.opt1 = x;
  base_opt1(p, role);
  return(x);
}

static uint64_t opt1_hash_1(s7_scheme *sc, s7_pointer p, const char *func, int32_t line)
{
  if ((!opt1_is_set(p)) ||
      (!opt1_role_matches(p, OPT1_HASH)))
    {
      show_opt1_bits(p, func, line, (uint64_t)OPT1_HASH);
      if (sc->stop_at_error) abort();
    }
  return(p->object.sym_cons.hash);
}

static void set_opt1_hash_1(s7_pointer p, uint64_t x)
{
  p->object.sym_cons.hash = x;
  base_opt1(p, OPT1_HASH);
}

static void show_opt2_bits(s7_pointer p, const char *func, int32_t line, uint64_t role)
{
  char *bits = show_debugger_bits(p);
  fprintf(stderr, "%s%s[%d]%s: opt2: %p->%p wants %s, debugger bits are %" PRIx64 "%s but expects %" ld64 " %s",
	  BOLD_TEXT, func, line, UNBOLD_TEXT,
	  p, p->object.cons.o2.opt2,
	  opt2_role_name(role),
	  p->debugger_bits, bits, (s7_int)role,
	  opt2_role_name(role));
  free(bits);
}

static bool f_call_func_mismatch(const char *func)
{
  return((!safe_strcmp(func, "check_and")) &&  /* these reflect set_fx|unchecked where the destination checks for null fx_proc */
	 (!safe_strcmp(func, "check_or")) &&
	 (!safe_strcmp(func, "eval")) &&
	 (!safe_strcmp(func, "set_any_c_np")) &&
	 (!safe_strcmp(func, "set_any_closure_np")) &&
	 (!safe_strcmp(func, "optimize_func_two_args")) &&
	 (!safe_strcmp(func, "optimize_func_many_args")) &&
	 (!safe_strcmp(func, "optimize_func_three_args")) &&
	 (!safe_strcmp(func, "fx_c_ff")) &&
	 (!safe_strcmp(func, "op_map_for_each_fa")) &&
	 (!safe_strcmp(func, "op_map_for_each_faa")));
}

static void check_opt2_bits(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  if (!p)
    {
      fprintf(stderr, "%s%s[%d]: opt2 null!\n%s", BOLD_TEXT, func, line, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, role)))
    {
      show_opt2_bits(p, func, line, role);
      if (sc->stop_at_error) abort();
    }
}

static s7_pointer opt2_1(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_opt2_bits(sc, p, role, func, line);
  return(p->object.cons.o2.opt2);
}

static s7_int opt2_n_1(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_opt2_bits(sc, p, role, func, line);
  return(p->object.cons.o2.n);
}

static void base_opt2(s7_pointer p, uint64_t role)
{
  set_opt2_role(p, role);
  set_opt2_is_set(p);
}

static void set_opt2_1(s7_scheme *sc, s7_pointer p, s7_pointer x, uint64_t role, const char *func, int32_t line)
{
  if ((role == OPT2_FX) &&
      (x == NULL) &&
      (f_call_func_mismatch(func)))
    fprintf(stderr, "%s[%d]: set fx_proc for %s to null (%s%s%s)\n", func, line,
	    string_value(object_to_truncated_string(sc, p, 80)),
	    ((is_h_optimized(car(p))) && (is_safe_c_op(optimize_op(car(p))))) ? BOLD_TEXT : "",
	    op_names[optimize_op(car(p))],
	    ((is_h_optimized(car(p))) && (is_safe_c_op(optimize_op(car(p))))) ? UNBOLD_TEXT : "");
  if ((role != OPT2_FX) && (role != OPT2_DIRECT) && (has_fx(p))) /* sometimes opt2_direct just specializes fx */
    {
      fprintf(stderr, "%s[%d]: overwrite has_fx: %s %s\n", func, line, opt2_role_name(role), display_80(p));
      if (sc->stop_at_error) abort();
    }
  p->object.cons.o2.opt2 = x;
  base_opt2(p, role);
}

static void set_opt2_n_1(s7_scheme *unused_sc, s7_pointer p, s7_int x, uint64_t role, const char *unused_func, int32_t unused_line)
{
  p->object.cons.o2.n = x;
  base_opt2(p, role);
}

static const char *opt2_name_1(s7_scheme *sc, s7_pointer p, const char *func, int32_t line)
{
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, OPT2_NAME)))
    {
      show_opt2_bits(p, func, line, (uint64_t)OPT2_NAME);
      if (sc->stop_at_error) abort();
    }
  return(p->object.sym_cons.fstr);
}

static void set_opt2_name_1(s7_pointer p, const char *str)
{
  p->object.sym_cons.fstr = str;
  base_opt2(p, OPT2_NAME);
}

static void show_opt3_bits(s7_pointer p, const char *func, int32_t line, uint64_t role)
{
  char *bits = show_debugger_bits(p);
  fprintf(stderr, "%s%s[%d]%s: opt3: %s %" PRIx64 "%s", BOLD_TEXT, func, line, UNBOLD_TEXT, opt3_role_name(role), p->debugger_bits, bits);
  free(bits);
}

static void check_opt3_bits(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  if (!p)
    {
      fprintf(stderr, "%s%s[%d]: opt3 null!\n%s", BOLD_TEXT, func, line, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  if ((!opt3_is_set(p)) ||
      (!opt3_role_matches(p, role)))
    {
      show_opt3_bits(p, func, line, role);
      if (sc->stop_at_error) abort();
    }
}

static s7_pointer opt3_1(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_opt3_bits(sc, p, role, func, line);
  return(p->object.cons.o3.opt3);
}

static s7_int opt3_n_1(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_opt3_bits(sc, p, role, func, line);
  return(p->object.cons.o3.n);
}

static void base_opt3(s7_pointer p, uint64_t role)
{
  set_opt3_role(p, role);
  set_opt3_is_set(p);
}

static void set_opt3_1(s7_pointer p, s7_pointer x, uint64_t role)
{
  clear_type_bit(p, T_LOCATION);
  p->object.cons.o3.opt3 = x;
  base_opt3(p, role);
}

static void set_opt3_n_1(s7_pointer p, s7_int x, uint64_t role)
{
  clear_type_bit(p, T_LOCATION);
  p->object.cons.o3.n = x;
  base_opt3(p, role);
}

static uint8_t opt3_byte_1(s7_scheme *sc, s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_opt3_bits(sc, p, role, func, line);
  return(p->object.cons.o3.opt_type);
}

static void set_opt3_byte_1(s7_pointer p, uint8_t x, uint64_t role, const char *unused_func, int32_t unused_line)
{
  clear_type_bit(p, T_LOCATION);
  p->object.cons.o3.opt_type = x;
  base_opt3(p, role);
}

static uint64_t opt3_location_1(s7_scheme *sc, s7_pointer p, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      ((p->debugger_bits & OPT3_LOCATION) == 0) ||
      (!has_location(p)))
    {
      show_opt3_bits(p, func, line, (uint64_t)OPT3_LOCATION);
      if (sc->stop_at_error) abort();
    }
  return(p->object.sym_cons.location);
}

static void set_opt3_location_1(s7_pointer p, uint64_t x)
{
  p->object.sym_cons.location = x;
  (p)->debugger_bits = (OPT3_LOCATION | (p->debugger_bits & ~OPT3_LEN)); /* turn on line, cancel len */
  set_opt3_is_set(p);
}

static uint64_t opt3_len_1(s7_scheme *sc, s7_pointer p, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      ((p->debugger_bits & OPT3_LEN) == 0) ||
      (has_location(p)))
    {
      show_opt3_bits(p, func, line, (uint64_t)OPT3_LEN);
      if (sc->stop_at_error) abort();
    }
  return(p->object.sym_cons.location);
}

static void set_opt3_len_1(s7_pointer p, uint64_t x)
{
  clear_type_bit(p, T_LOCATION);
  p->object.sym_cons.location = x;
  (p)->debugger_bits = (OPT3_LEN | (p->debugger_bits & ~(OPT3_LOCATION)));
  set_opt3_is_set(p);
}

static void print_debugging_state(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  /* show current state, current allocated state, and previous allocated state */
  char *current_bits, *allocated_bits, *previous_bits, *str;
  int64_t save_full_type = full_type(obj);
  s7_int len, nlen;
  const char *excl_name = (is_free(obj)) ? "free cell!" : "unknown object!";
  block_t *b;

  current_bits = describe_type_bits(sc, obj);
  full_type(obj) = obj->current_alloc_type;
  allocated_bits = describe_type_bits(sc, obj);
  full_type(obj) = obj->previous_alloc_type;
  previous_bits = describe_type_bits(sc, obj);
  full_type(obj) = save_full_type;

  len = safe_strlen(excl_name) +
    safe_strlen(current_bits) + safe_strlen(allocated_bits) + safe_strlen(previous_bits) +
    safe_strlen(obj->previous_alloc_func) + safe_strlen(obj->current_alloc_func) + 512;

  b = mallocate(sc, len);
  str = (char *)block_data(b);
  nlen = snprintf(str, len,
		  "\n<%s %s,\n  current: %s[%d] %s,\n  previous: %s[%d] %s\n  %d uses>",
		  excl_name, current_bits,
		  obj->current_alloc_func, obj->current_alloc_line, allocated_bits,
		  obj->previous_alloc_func, obj->previous_alloc_line, previous_bits,
		  obj->uses);
  free(current_bits);
  free(allocated_bits);
  free(previous_bits);
  if (is_null(port))
    fprintf(stderr, "%p: %s\n", obj, str);
  else port_write_string(port)(sc, str, clamp_length(nlen, len), port);
  liberate(sc, b);
}

static s7_pointer symbol_to_local_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer e);
static s7_pointer check_null_sym(s7_scheme *sc, s7_pointer p, s7_pointer sym, int32_t line, const char *func)
{
  if (!p)
    {
      s7_pointer slot = symbol_to_local_slot(sc, sym, sc->curlet);
      char *s = describe_type_bits(sc, sym);
      fprintf(stderr, "%s%s[%d]: %s unbound%s\n", BOLD_TEXT, func, line, symbol_name(sym), UNBOLD_TEXT);
      fprintf(stderr, "  symbol_id: %" ld64 ", let_id: %" ld64 ", bits: %s", symbol_id(sym), let_id(sc->curlet), s);
      free(s);
      if (is_slot(slot)) fprintf(stderr, ", slot: %s", display(slot));
      fprintf(stderr, "\n");
      if (sc->stop_at_error) abort();
    }
  return(p);
}
#endif /* S7_DEBUGGING */
/* -------------------------------- end internal debugging apparatus -------------------------------- */


static s7_pointer set_elist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->elist_1, x1);
  return(sc->elist_1);
}

static s7_pointer set_elist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->elist_2, x1);
  set_cadr(sc->elist_2, x2);
  return(sc->elist_2);
}

static s7_pointer set_elist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p = sc->elist_3;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(sc->elist_3);
}

static s7_pointer set_elist_4(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p = sc->elist_4;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(sc->elist_4);
}

static s7_pointer set_elist_5(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5)
{
  set_car(sc->elist_5, x1);
  set_elist_4(sc, x2, x3, x4, x5);
  return(sc->elist_5);
}

static s7_pointer set_elist_6(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5, s7_pointer x6)
{
  set_car(sc->elist_6, x1);
  set_elist_5(sc, x2, x3, x4, x5, x6);
  return(sc->elist_6);
}

static s7_pointer set_elist_7(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5, s7_pointer x6, s7_pointer x7)
{
  set_car(sc->elist_7, x1);
  set_elist_6(sc, x2, x3, x4, x5, x6, x7);
  return(sc->elist_7);
}

static s7_pointer set_wlist_3(s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(lst);
}

static s7_pointer set_wlist_4(s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(lst);
}

static s7_pointer set_plist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->plist_1, x1);
  return(sc->plist_1);
}

static s7_pointer set_plist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->plist_2, x1);
  set_car(sc->plist_2_2, x2);
  return(sc->plist_2);
}

static s7_pointer set_plist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  return(set_wlist_3(sc->plist_3, x1, x2, x3));
}

static s7_pointer set_qlist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2) /* let_ref_fallback */
{
  set_car(sc->qlist_2, x1);
  set_cadr(sc->qlist_2, x2);
  return(sc->qlist_2);
}

static s7_pointer set_qlist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3) /* let_set_fallback */
{
  return(set_wlist_3(sc->qlist_3, x1, x2, x3));
}

static s7_pointer set_clist_1(s7_scheme *sc, s7_pointer x1) /* for c_object length method etc, a "weak" list */
{
  set_car(sc->clist_1, x1);
  return(sc->clist_1);
}

static s7_pointer set_clist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2) /* for c_object equal method etc, a "weak" list */
{
  set_car(sc->clist_2, x1);
  set_cadr(sc->clist_2, x2);
  return(sc->clist_2);
}

static s7_pointer set_dlist_1(s7_scheme *sc, s7_pointer x1) /* another like clist: temp usage, "weak" (not gc_marked), but permanent list */
{
  set_car(sc->dlist_1, x1);
  return(sc->dlist_1);
}

static s7_pointer set_ulist_1(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->u1_1, x1);
  set_cdr(sc->u1_1, x2);
  return(sc->u1_1);
}

static s7_pointer set_ulist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  set_car(sc->u2_1, x1);
  set_car(sc->u2_2, x2);
  set_cdr(sc->u2_2, x3);
  return(sc->u2_1);
}

static int32_t position_of(s7_pointer p, s7_pointer args)
{
  int32_t i;
  for (i = 1; p != args; i++, args = cdr(args));
  return(i);
}

s7_pointer s7_method(s7_scheme *sc, s7_pointer obj, s7_pointer method)
{
  if (has_active_methods(sc, obj))
    return(find_method_with_let(sc, obj, method));
  return(sc->undefined);
}

/* if a method is shadowing a built-in like abs, it should expect the same args as abs and behave the same -- no multiple values etc */
#define check_method(Sc, Obj, Method, Args)		\
  {							\
    s7_pointer func;					\
    if ((has_active_methods(Sc, Obj)) &&				\
	((func = find_method_with_let(Sc, Obj, Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, func, Args)); \
  }

static s7_pointer apply_boolean_method(s7_scheme *sc, s7_pointer obj, s7_pointer method)
{
  s7_pointer func = find_method_with_let(sc, obj, method);
  if (func == sc->undefined) return(sc->F);
  return(s7_apply_function(sc, func, set_plist_1(sc, obj)));
}

static noreturn void missing_method_error(s7_scheme *sc, s7_pointer method, s7_pointer obj)
{
  s7_error_nr(sc, sc->missing_method_symbol,
	      set_elist_3(sc, wrap_string(sc, "missing ~S method in ~A", 23), method,
			  (is_c_object(obj)) ? c_object_scheme_name(sc, obj) : obj));
}

/* this is a macro mainly to simplify the Checker handling */
#define check_boolean_method(Sc, Checker, Method, Args)	       \
  {							       \
    s7_pointer p = car(Args);				       \
    if (Checker(p)) return(Sc->T);			       \
    if (!has_active_methods(Sc, p)) return(Sc->F);	       \
    return(apply_boolean_method(Sc, p, Method));	       \
  }

static s7_pointer find_and_apply_method(s7_scheme *sc, s7_pointer obj, s7_pointer sym, s7_pointer args)
{
  s7_pointer func = find_method_with_let(sc, obj, sym);
  if (func == sc->undefined)
    missing_method_error(sc, sym, obj);
  return(s7_apply_function(sc, func, args));
}

static s7_pointer method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, uint8_t typ, int32_t num)
{
  if (!has_active_methods(sc, obj))
    wrong_type_argument(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, args));
}

static s7_pointer method_or_bust_p(s7_scheme *sc, s7_pointer obj, s7_pointer method, uint8_t typ)
{
  if (!has_active_methods(sc, obj))
    simple_wrong_type_argument(sc, method, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_1(sc, obj)));
}

static s7_pointer method_or_bust_pp(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer x1, s7_pointer x2, uint8_t typ, int32_t num)
{
  if (!has_active_methods(sc, obj))
    wrong_type_argument(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_2(sc, x1, x2)));
}

static s7_pointer method_or_bust_ppp(s7_scheme *sc, s7_pointer obj, s7_pointer method,
				     s7_pointer x1, s7_pointer x2, s7_pointer x3, uint8_t typ, int32_t num)
{
  if (!has_active_methods(sc, obj))
    wrong_type_argument(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_3(sc, x1, x2, x3)));
}

static noreturn void immutable_object_error(s7_scheme *sc, s7_pointer info) {s7_error_nr(sc, sc->immutable_error_symbol, info);}

static s7_pointer mutable_method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, uint8_t typ, int32_t num)
{
  if (has_active_methods(sc, obj))
    return(find_and_apply_method(sc, obj, method, args));
  if (type(obj) != typ)
    wrong_type_argument(sc, method, num, obj, typ);
  if (!is_immutable(obj))
    wrong_type_argument(sc, method, num, obj, typ);
  immutable_object_error(sc, set_elist_3(sc, immutable_error_string, method, obj));
  return(NULL);
}

static s7_pointer mutable_method_or_bust_ppp(s7_scheme *sc, s7_pointer obj, s7_pointer method,
					     s7_pointer x1, s7_pointer x2, s7_pointer x3, uint8_t typ, int32_t num)
{
  return(mutable_method_or_bust(sc, obj, method, set_plist_3(sc, x1, x2, x3), typ, num));
}

static s7_pointer method_or_bust_one_arg(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, uint8_t typ)
{
  if (!has_active_methods(sc, obj))
    simple_wrong_type_argument(sc, method, obj, typ);
  return(find_and_apply_method(sc, obj, method, args));
}

static s7_pointer method_or_bust_with_type(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, s7_pointer typ, int32_t num)
{
  if (!has_active_methods(sc, obj))
    wrong_type_argument_with_type(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, args));
}

static s7_pointer method_or_bust_with_type_pp(s7_scheme *sc, s7_pointer obj, s7_pointer method,
					      s7_pointer x1, s7_pointer x2, s7_pointer typ, int32_t num)
{
  if (!has_active_methods(sc, obj))
    wrong_type_argument_with_type(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_2(sc, x1, x2)));
}

static s7_pointer method_or_bust_with_type_and_loc_pp(s7_scheme *sc, s7_pointer obj, s7_pointer method,
						      s7_pointer x1, s7_pointer x2, s7_pointer typ, int32_t num)
{
  int32_t loc = sc->error_argnum + num;
  sc->error_argnum = 0;
  if (!has_active_methods(sc, obj))
    wrong_type_argument_with_type(sc, method, loc, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_2(sc, x1, x2)));
}

static s7_pointer method_or_bust_with_type_pi(s7_scheme *sc, s7_pointer obj, s7_pointer method,
					      s7_pointer x1, s7_int x2, s7_pointer typ, int32_t num)
{
  if (!has_active_methods(sc, obj))
    wrong_type_argument_with_type(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_2(sc, x1, make_integer(sc, x2))));
}

static s7_pointer method_or_bust_with_type_pf(s7_scheme *sc, s7_pointer obj, s7_pointer method,
					      s7_pointer x1, s7_double x2, s7_pointer typ, int32_t num)
{
  if (!has_active_methods(sc, obj))
    wrong_type_argument_with_type(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_2(sc, x1, make_real(sc, x2))));
}

static s7_pointer method_or_bust_with_type_one_arg_p(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer typ)
{
  if (!has_active_methods(sc, obj))
    simple_wrong_type_argument_with_type(sc, method, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_1(sc, obj)));
}

#define syntax_error_any(Sc, ErrType, ErrMsg, Len, Obj) \
  s7_error_nr(Sc, ErrType, set_elist_2(Sc, wrap_string(Sc, ErrMsg, Len), Obj))

#define syntax_error(Sc, ErrMsg, Len, Obj) \
  syntax_error_any(Sc, Sc->syntax_error_symbol, ErrMsg, Len, Obj)

#define syntax_error_with_caller(Sc, ErrMsg, Len, Caller, Obj) \
  s7_error_nr(Sc, Sc->syntax_error_symbol, set_elist_3(Sc, wrap_string(Sc, ErrMsg, Len), Caller, Obj))

#define syntax_error_with_caller2(Sc, ErrMsg, Len, Caller, Name, Obj) \
  s7_error_nr(Sc, Sc->syntax_error_symbol, set_elist_4(Sc, wrap_string(Sc, ErrMsg, Len), Caller, Name, Obj))


/* -------------------------------- constants -------------------------------- */

/* #f and #t */
s7_pointer s7_f(s7_scheme *sc) {return(sc->F);}
s7_pointer s7_t(s7_scheme *sc) {return(sc->T);}


/* () */
s7_pointer s7_nil(s7_scheme *sc) {return(sc->nil);}
bool s7_is_null(s7_scheme *sc, s7_pointer p) {return(is_null(p));}
static bool is_null_b_p(s7_pointer p) {return(type(p) == T_NIL);} /* faster than b_7p because opt_b_p is faster */

static s7_pointer g_is_null(s7_scheme *sc, s7_pointer args)
{
  #define H_is_null "(null? obj) returns #t if obj is the empty list"
  #define Q_is_null sc->pl_bt
  check_boolean_method(sc, is_null, sc->is_null_symbol, args);
}


/* #<undefined> and #<unspecified> */
s7_pointer s7_undefined(s7_scheme *sc)       {return(sc->undefined);}
s7_pointer s7_unspecified(s7_scheme *sc)     {return(sc->unspecified);}

bool s7_is_unspecified(s7_scheme *sc, s7_pointer val) {return(is_unspecified(val));}

static s7_pointer g_is_undefined(s7_scheme *sc, s7_pointer args)
{
  #define H_is_undefined "(undefined? val) returns #t if val is #<undefined> or its reader equivalent"
  #define Q_is_undefined sc->pl_bt
  check_boolean_method(sc, is_undefined, sc->is_undefined_symbol, args);
}

static s7_pointer g_is_unspecified(s7_scheme *sc, s7_pointer args)
{
  #define H_is_unspecified "(unspecified? val) returns #t if val is #<unspecified>"
  #define Q_is_unspecified sc->pl_bt
  check_boolean_method(sc, is_unspecified, sc->is_unspecified_symbol, args);
}


/* -------------------------------- eof-object? -------------------------------- */
s7_pointer eof_object = NULL;          /* #<eof> is an entry in the chars array, so it's not a part of sc */

s7_pointer s7_eof_object(s7_scheme *sc) {return(eof_object);}

static s7_pointer g_is_eof_object(s7_scheme *sc, s7_pointer args)
{
  #define H_is_eof_object "(eof-object? val) returns #t if val is the end-of-file object"
  #define Q_is_eof_object sc->pl_bt
  check_boolean_method(sc, is_eof, sc->is_eof_object_symbol, args);
}

static bool is_eof_object_b_p(s7_pointer p) {return(p == eof_object);}


/* -------------------------------- not -------------------------------- */
static bool not_b_7p(s7_scheme *sc, s7_pointer p) {return(p == sc->F);}
bool s7_boolean(s7_scheme *sc, s7_pointer x)      {return(x != sc->F);}
s7_pointer s7_make_boolean(s7_scheme *sc, bool x) {return(make_boolean(sc, x));}

static s7_pointer g_not(s7_scheme *sc, s7_pointer args)
{
  #define H_not "(not obj) returns #t if obj is #f, otherwise #t: (not ()) -> #f"
  #define Q_not sc->pl_bt
  return((car(args) == sc->F) ? sc->T : sc->F);
}


/* -------------------------------- boolean? -------------------------------- */
bool s7_is_boolean(s7_pointer x) {return(type(x) == T_BOOLEAN);}

static s7_pointer g_is_boolean(s7_scheme *sc, s7_pointer args)
{
  #define H_is_boolean "(boolean? obj) returns #t if obj is #f or #t: (boolean? ()) -> #f"
  #define Q_is_boolean sc->pl_bt
  check_boolean_method(sc, s7_is_boolean, sc->is_boolean_symbol, args);
}


/* -------------------------------- constant? -------------------------------- */
static inline bool is_constant_symbol(s7_scheme *sc, s7_pointer sym)
{
  if (is_immutable_symbol(sym))    /* for keywords */
    return(true);
  if (is_possibly_constant(sym))
    {
      s7_pointer slot = lookup_slot_from(sym, sc->curlet);
      return((is_slot(slot)) && (is_immutable_slot(slot)));
    }
  return(false);
}

#define is_constant(sc, p) ((type(p) != T_SYMBOL) || (is_constant_symbol(sc, p)))

static s7_pointer g_is_constant(s7_scheme *sc, s7_pointer args)
{
  #define H_is_constant "(constant? obj) returns #t if obj either evaluates to itself, or is a symbol whose binding is constant"
  #define Q_is_constant sc->pl_bt
  return(make_boolean(sc, is_constant(sc, car(args))));
}

static bool is_constant_b_7p(s7_scheme *sc, s7_pointer p) {return(is_constant(sc, p));}
static s7_pointer is_constant_p_p(s7_scheme *sc, s7_pointer p) {return(make_boolean(sc, is_constant(sc, p)));}


/* -------------------------------- immutable? -------------------------------- */
bool s7_is_immutable(s7_pointer p) {return(is_immutable(p));}

static s7_pointer g_is_immutable(s7_scheme *sc, s7_pointer args)
{
  #define H_is_immutable "(immutable? sequence) returns #t if the sequence is immutable"
  #define Q_is_immutable sc->pl_bt
  s7_pointer p = car(args);
  if (is_number(p)) return(sc->T); /* should these be marked immutable? should we use (type != SYMBOL) as above? */
  return(make_boolean(sc, is_immutable(p)));
}


/* -------------------------------- immutable! -------------------------------- */
s7_pointer s7_immutable(s7_pointer p)
{
  set_immutable(p);
  return(p);
}

static s7_pointer g_immutable(s7_scheme *sc, s7_pointer args)
{
  #define H_immutable "(immutable! x) declares that the x can't be changed. x is returned."
  #define Q_immutable s7_make_signature(sc, 2, sc->T, sc->T)
  s7_pointer p = car(args);
  if (is_symbol(p))
    {
      s7_pointer slot = lookup_slot_from(p, sc->curlet);
      if (is_slot(slot))
	{
	  set_immutable(slot);
	  return(p);   /* symbol is not set immutable ? */
	}}
  set_immutable(p);   /* could set_immutable save the current file/line? Then the immutable error checks for define-constant and this setting */
  return(p);
}

/* there's no way to make a slot setter (as setter) immutable (t_multiform as bit) */


/* -------------------------------- GC -------------------------------- */

/* in most code, pairs, lets, and slots dominate the heap -- each about 25% to 40% of the
 *   total cell allocations.  In snd-test, reals are 50%. slots need not be in the heap,
 *   but moving them out to their own free list was actually slower because we need (in that
 *   case) to manage them in the sweep process by tracking lets.
 */

#if S7_DEBUGGING
static s7_int gc_protect_2(s7_scheme *sc, s7_pointer x, int32_t line)
{
  s7_int loc = s7_gc_protect(sc, x);
  if (loc > 8192)
    {
      fprintf(stderr, "infinite loop or memory leak at line %d %s?\n", line, string_value(s7_object_to_string(sc, current_code(sc), false)));
      abort();
    }
  return(loc);
}
#define gc_protect_1(Sc, X) gc_protect_2(Sc, X, __LINE__)
#else
#define gc_protect_1(Sc, X) s7_gc_protect(Sc, X)
#endif

static void resize_gc_protect(s7_scheme *sc)
{
  s7_int size = sc->protected_objects_size;
  block_t *ob = vector_block(sc->protected_objects);
  s7_int new_size = 2 * size;
  block_t *nb = reallocate(sc, ob, new_size * sizeof(s7_pointer));
  block_info(nb) = NULL;
  vector_block(sc->protected_objects) = nb;
  vector_elements(sc->protected_objects) = (s7_pointer *)block_data(nb);
  vector_length(sc->protected_objects) = new_size;
  sc->protected_objects_size = new_size;
  sc->protected_objects_free_list = (s7_int *)Realloc(sc->protected_objects_free_list, new_size * sizeof(s7_int));
  for (s7_int i = size; i < new_size; i++)
    {
      vector_element(sc->protected_objects, i) = sc->unused;
      sc->protected_objects_free_list[++sc->protected_objects_free_list_loc] = i;
    }
}

s7_int s7_gc_protect(s7_scheme *sc, s7_pointer x)
{
  s7_int loc;
  if (sc->protected_objects_free_list_loc < 0)
    resize_gc_protect(sc);
  loc = sc->protected_objects_free_list[sc->protected_objects_free_list_loc--];
  vector_element(sc->protected_objects, loc) = x;
  return(loc);
}

void s7_gc_unprotect_at(s7_scheme *sc, s7_int loc)
{
  if (loc < sc->protected_objects_size)
    {
      if (vector_element(sc->protected_objects, loc) != sc->unused)
	sc->protected_objects_free_list[++sc->protected_objects_free_list_loc] = loc;
#if S7_DEBUGGING
      else fprintf(stderr, "redundant gc_unprotect_at location %" ld64 "\n", loc);
#endif
      vector_element(sc->protected_objects, loc) = sc->unused;
    }
}

s7_pointer s7_gc_protected_at(s7_scheme *sc, s7_int loc)
{
  s7_pointer obj = sc->unspecified;
  if (loc < sc->protected_objects_size)
    obj = vector_element(sc->protected_objects, loc);
  if (obj == sc->unused)
    return(sc->unspecified);
  return(obj);
}

#define gc_protected_at(Sc, Loc) vector_element(Sc->protected_objects, Loc)

s7_pointer s7_gc_protect_via_location(s7_scheme *sc, s7_pointer x, s7_int loc)
{
  vector_element(sc->protected_objects, loc) = x;
  return(x);
}

s7_pointer s7_gc_unprotect_via_location(s7_scheme *sc, s7_int loc)
{
  vector_element(sc->protected_objects, loc) = sc->F;
  return(sc->F);
}


static void (*mark_function[NUM_TYPES])(s7_pointer p);

void s7_mark(s7_pointer p) {if (!is_marked(p)) (*mark_function[unchecked_type(p)])(p);}

static inline void gc_mark(s7_pointer p) {if (!is_marked(p)) (*mark_function[unchecked_type(p)])(p);}

static inline void mark_slot(s7_pointer p)
{
  set_mark(T_Slt(p));
  gc_mark(slot_value(p));
  if (slot_has_setter(p))
    gc_mark(slot_setter(p));
  if (slot_has_pending_value(p))
    gc_mark(slot_pending_value(p));
  set_mark(slot_symbol(p));
}

static void mark_noop(s7_pointer unused_p) {}

static void close_output_port(s7_scheme *sc, s7_pointer p);
static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym);
static void cull_weak_hash_table(s7_scheme *sc, s7_pointer table);

static void process_iterator(s7_scheme *unused_sc, s7_pointer s1)
{
  if (is_weak_hash_iterator(s1))
    {
      s7_pointer h;
      clear_weak_hash_iterator(s1);
      h = iterator_sequence(s1);
      if (unchecked_type(h) == T_HASH_TABLE)
	weak_hash_iters(h)--;
    }
}

static void process_multivector(s7_scheme *sc, s7_pointer s1)
{
  vdims_t *info = vector_dimension_info(s1);  /* a multidimensional empty vector can have dimension info, wrapped vectors always have dimension info */
  if ((info) &&
      (info != sc->wrap_only))
    {
      if (vector_elements_should_be_freed(info)) /* a kludge for foreign code convenience */
	{
	  free(vector_elements(s1));
	  vector_elements_should_be_freed(info) = false;
	}
      liberate(sc, info);
      vector_set_dimension_info(s1, NULL);
    }
  liberate(sc, vector_block(s1));
}

static void process_input_string_port(s7_scheme *sc, s7_pointer s1)
{
#if S7_DEBUGGING
  /* this set of ports is a subset of the ports that respond true to is_string_port --
   *   the latter include file ports fully read into local memory; see read_file which uses add_input_port, not add_input_string_port
   */
  if (port_filename(s1))
    fprintf(stderr, "string input port has a filename: %s\n", port_filename(s1));
  if (port_needs_free(s1))
    fprintf(stderr, "string input port needs data release\n");
#endif
  liberate(sc, port_block(s1));
}

static void free_port_data(s7_scheme *sc, s7_pointer s1)
{
  if (port_data(s1))
    {
      liberate(sc, port_data_block(s1));
      port_data_block(s1) = NULL;
      port_data(s1) = NULL;
      port_data_size(s1) = 0;
    }
  port_needs_free(s1) = false;
}

static void close_input_function(s7_scheme *sc, s7_pointer p);

static void process_input_port(s7_scheme *sc, s7_pointer s1)
{
  if (!port_is_closed(s1))
    {
      if (is_file_port(s1))
	{
	  if (port_file(s1))
	    {
	      fclose(port_file(s1));
	      port_file(s1) = NULL;
	    }}
      else
	if (is_function_port(s1))
	  close_input_function(sc, s1);
    }
  if (port_needs_free(s1))
    free_port_data(sc, s1);

  if (port_filename(s1))
    {
      liberate(sc, port_filename_block(s1));
      port_filename(s1) = NULL;
    }
  liberate(sc, port_block(s1));
}

static void process_output_port(s7_scheme *sc, s7_pointer s1)
{
  close_output_port(sc, s1); /* needed for free filename, etc */
  liberate(sc, port_block(s1));
  if (port_needs_free(s1))
    {
      port_needs_free(s1) = false;
      if (port_data_block(s1))
	{
	  liberate(sc, port_data_block(s1));
	  port_data_block(s1) = NULL;
	}}
}

static void process_continuation(s7_scheme *sc, s7_pointer s1)
{
  continuation_op_stack(s1) = NULL;
  liberate_block(sc, continuation_block(s1));
}


#if WITH_GMP
#if ((__GNU_MP_VERSION < 6) || ((__GNU_MP_VERSION == 6) && (__GNU_MP_VERSION_MINOR == 0)))
static int32_t mpq_cmp_z(const mpq_t op1, const mpz_t op2)
{
  mpq_t z1;
  int32_t result;
  mpq_init(z1);
  mpq_set_z(z1, op2);
  result = mpq_cmp(op1, z1);
  mpq_clear(z1);
  return(result);
}
#endif

static s7_int big_integer_to_s7_int(s7_scheme *sc, mpz_t n);

static s7_int s7_integer_clamped_if_gmp(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p))
    return(integer(p));
  if (is_t_big_integer(p))
    return(big_integer_to_s7_int(sc, big_integer(p)));
  return(0);
}

static void free_big_integer(s7_scheme *sc, s7_pointer p)
{
  big_integer_nxt(p) = sc->bigints;
  sc->bigints = big_integer_bgi(p);
  big_integer_bgi(p) = NULL;
}

static void free_big_ratio(s7_scheme *sc, s7_pointer p)
{
  big_ratio_nxt(p) = sc->bigrats;
  sc->bigrats = big_ratio_bgr(p);
  big_ratio_bgr(p) = NULL;
}

static void free_big_real(s7_scheme *sc, s7_pointer p)
{
  big_real_nxt(p) = sc->bigflts;
  sc->bigflts = big_real_bgf(p);
  big_real_bgf(p) = NULL;
}

static void free_big_complex(s7_scheme *sc, s7_pointer p)
{
  big_complex_nxt(p) = sc->bigcmps;
  sc->bigcmps = big_complex_bgc(p);
  big_complex_bgc(p) = NULL;
}
#else
#define s7_integer_clamped_if_gmp(Sc, P) integer(P)
#endif


static void free_hash_table(s7_scheme *sc, s7_pointer table);

static void sweep(s7_scheme *sc)
{
  s7_int i, j;
  gc_list_t *gp;

  #define process_gc_list(Code)				\
    if (gp->loc > 0)					\
      {							\
        for (i = 0, j = 0; i < gp->loc; i++)	\
          {						\
            s7_pointer s1 = gp->list[i];		\
            if (is_free_and_clear(s1))			\
              {						\
                Code;					\
              }						\
            else gp->list[j++] = s1;			\
          }						\
        gp->loc = j;					\
      }							\

  gp = sc->strings;
  process_gc_list(liberate(sc, string_block(s1)))

  gp = sc->gensyms;
  process_gc_list(remove_gensym_from_symbol_table(sc, s1); liberate(sc, gensym_block(s1)))
  if (gp->loc == 0) mark_function[T_SYMBOL] = mark_noop;

  gp = sc->undefineds;
  process_gc_list(free(undefined_name(s1)))

  gp = sc->c_objects;
  process_gc_list((c_object_gc_free(sc, s1)) ? (void)(*(c_object_gc_free(sc, s1)))(sc, s1) : (void)(*(c_object_free(sc, s1)))(c_object_value(s1)))

  gp = sc->vectors;
  process_gc_list(liberate(sc, vector_block(s1)))

  gp = sc->multivectors;
  process_gc_list(process_multivector(sc, s1));

  gp = sc->hash_tables;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    free_hash_table(sc, s1);
	  else
	    {
	      if ((is_weak_hash_table(s1)) &&
		  (weak_hash_iters(s1) == 0))
		cull_weak_hash_table(sc, s1);
	      gp->list[j++] = s1;
	    }}
      gp->loc = j;
    }

  gp = sc->weak_hash_iterators;
  process_gc_list(process_iterator(sc, s1));

  gp = sc->opt1_funcs;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer s1 = gp->list[i];
	  if (!is_free_and_clear(s1))
	    gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->input_ports;
  process_gc_list(process_input_port(sc, s1));

  gp = sc->input_string_ports;
  process_gc_list(process_input_string_port(sc, s1));

  gp = sc->output_ports;
  process_gc_list(process_output_port(sc, s1));

  gp = sc->continuations;
  process_gc_list(process_continuation(sc, s1));

  gp = sc->weak_refs;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer s1 = gp->list[i];
	  if (!is_free_and_clear(s1))
	    {
	      if (is_free_and_clear(c_pointer_weak1(s1)))
		c_pointer_weak1(s1) = sc->F;
	      if (is_free_and_clear(c_pointer_weak2(s1)))
		c_pointer_weak2(s1) = sc->F;
	      if ((c_pointer_weak1(s1) != sc->F) ||
		  (c_pointer_weak2(s1) != sc->F))
		gp->list[j++] = s1;
	    }}
      gp->loc = j;
    }

#if WITH_GMP
  gp = sc->big_integers;
  process_gc_list(free_big_integer(sc, s1))

  gp = sc->big_ratios;
  process_gc_list(free_big_ratio(sc ,s1))

  gp = sc->big_reals;
  process_gc_list(free_big_real(sc, s1))

  gp = sc->big_complexes;
  process_gc_list(free_big_complex(sc, s1))

  gp = sc->big_random_states;
  process_gc_list(gmp_randclear(random_gmp_state(s1)))
#endif
}

static inline void add_to_gc_list(gc_list_t *gp, s7_pointer p)
{
  if (gp->loc == gp->size)
    {
      gp->size *= 2;
      gp->list = (s7_pointer *)Realloc(gp->list, gp->size * sizeof(s7_pointer));
    }
  gp->list[gp->loc++] = p;
}

static gc_list_t *make_gc_list(void)
{
  gc_list_t *gp = (gc_list_t *)Malloc(sizeof(gc_list_t));
  #define INIT_GC_CACHE_SIZE 4
  gp->size = INIT_GC_CACHE_SIZE;
  gp->loc = 0;
  gp->list = (s7_pointer *)Malloc(gp->size * sizeof(s7_pointer));
  return(gp);
}

static void just_mark(s7_pointer p) {set_mark(p);}

static void add_gensym(s7_scheme *sc, s7_pointer p)
{
  add_to_gc_list(sc->gensyms, p);
  mark_function[T_SYMBOL] = just_mark;
}

#define add_c_object(sc, p)          add_to_gc_list(sc->c_objects, p)
#define add_hash_table(sc, p)        add_to_gc_list(sc->hash_tables, p)
#define add_string(sc, p)            add_to_gc_list(sc->strings, p)
#define add_input_port(sc, p)        add_to_gc_list(sc->input_ports, p)
#define add_input_string_port(sc, p) add_to_gc_list(sc->input_string_ports, p)
#define add_output_port(sc, p)       add_to_gc_list(sc->output_ports, p)
#define add_continuation(sc, p)      add_to_gc_list(sc->continuations, p)
#define add_undefined(sc, p)         add_to_gc_list(sc->undefineds, p)
#define add_vector(sc, p)            add_to_gc_list(sc->vectors, p)
#define add_multivector(sc, p)       add_to_gc_list(sc->multivectors, p)
#define add_weak_ref(sc, p)          add_to_gc_list(sc->weak_refs, p)
#define add_weak_hash_iterator(sc, p) add_to_gc_list(sc->weak_hash_iterators, p)
#define add_opt1_func(sc, p) do {if (!opt1_func_listed(p)) add_to_gc_list(sc->opt1_funcs, p); set_opt1_func_listed(p);} while (0)

#if WITH_GMP
#define add_big_integer(sc, p)       add_to_gc_list(sc->big_integers, p)
#define add_big_ratio(sc, p)         add_to_gc_list(sc->big_ratios, p)
#define add_big_real(sc, p)          add_to_gc_list(sc->big_reals, p)
#define add_big_complex(sc, p)       add_to_gc_list(sc->big_complexes, p)
#define add_big_random_state(sc, p)  add_to_gc_list(sc->big_random_states, p)
#endif

static void init_gc_caches(s7_scheme *sc)
{
  sc->strings = make_gc_list();
  sc->gensyms = make_gc_list();
  sc->undefineds = make_gc_list();
  sc->vectors = make_gc_list();
  sc->multivectors = make_gc_list();
  sc->hash_tables = make_gc_list();
  sc->input_ports = make_gc_list();
  sc->input_string_ports = make_gc_list();
  sc->output_ports = make_gc_list();
  sc->continuations = make_gc_list();
  sc->c_objects = make_gc_list();
  sc->weak_refs = make_gc_list();
  sc->weak_hash_iterators = make_gc_list();
  sc->opt1_funcs = make_gc_list();
#if WITH_GMP
  sc->big_integers = make_gc_list();
  sc->big_ratios = make_gc_list();
  sc->big_reals = make_gc_list();
  sc->big_complexes = make_gc_list();
  sc->big_random_states = make_gc_list();
  sc->ratloc = NULL;
#endif
  /* slightly unrelated... */
  sc->setters_size = 4;
  sc->setters_loc = 0;
  sc->setters = (s7_pointer *)Malloc(sc->setters_size * sizeof(s7_pointer));
}

static s7_pointer permanent_cons(s7_scheme *sc, s7_pointer a, s7_pointer b, uint64_t type);

static void add_setter(s7_scheme *sc, s7_pointer p, s7_pointer setter)
{
  /* setters GC-protected. The c_function_setter field can't be used because the built-in functions
   *   are often removed from the heap and never thereafter marked.  Only closures and macros are protected here.
   */
  for (s7_int i = 0; i < sc->setters_loc; i++)
    {
      s7_pointer x = sc->setters[i];
      if (car(x) == p)
	{
	  set_cdr(x, setter);
	  return;
 	}}
  if (sc->setters_loc == sc->setters_size)
    {
      sc->setters_size *= 2;
      sc->setters = (s7_pointer *)Realloc(sc->setters, sc->setters_size * sizeof(s7_pointer));
    }
  sc->setters[sc->setters_loc++] = permanent_cons(sc, p, setter, T_PAIR | T_IMMUTABLE);
}

static void mark_symbol_vector(s7_pointer p, s7_int len)
{
  set_mark(p);
  if (mark_function[T_SYMBOL] != mark_noop) /* else no gensyms */
    {
      s7_pointer *e = vector_elements(p);
      for (s7_int i = 0; i < len; i++)
	if (is_gensym(e[i]))
	  set_mark(e[i]);
    }
}

static void mark_simple_vector(s7_pointer p, s7_int len)
{
  s7_pointer *e = vector_elements(p);
  set_mark(p);
  for (s7_int i = 0; i < len; i++)
    set_mark(e[i]);
}

static void just_mark_vector(s7_pointer p, s7_int unused_len) {set_mark(p);}

static void mark_vector_1(s7_pointer p, s7_int top)
{
  s7_pointer *tp = (s7_pointer *)(vector_elements(p));
  s7_pointer *tend, *tend4;
  set_mark(p);
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);
  tend4 = (s7_pointer *)(tend - 8);
  while (tp <= tend4)
    LOOP_8(gc_mark(*tp++));
  while (tp < tend)
    gc_mark(*tp++);
}

static void mark_typed_vector_1(s7_pointer p, s7_int top) /* for typed vectors with closure setters */
{
  gc_mark(typed_vector_typer(p));
  mark_vector_1(p, top);
}

static void mark_let(s7_pointer let)
{
  for (s7_pointer x = let; is_let(x) && (!is_marked(x)); x = let_outlet(x)) /* let can be sc->nil, e.g. closure_let */
    {
      set_mark(x);
      if (has_dox_slot1(x)) mark_slot(let_dox_slot1(x));
      if ((has_dox_slot2(x)) && (is_slot(let_dox_slot2(x)))) mark_slot(let_dox_slot2(x));
      /* it can happen (call/cc related) that let_dox_slot2 is a slot but invalid, but in that case has_dox_slot2 will not be set(?) */
      for (s7_pointer y = let_slots(x); tis_slot(y); y = next_slot(y))
	if (!is_marked(y)) /* slot value might be the enclosing let */
	  mark_slot(y);
    }
}

#if WITH_HISTORY
static void gc_owlet_mark(s7_pointer tp)
{
  /* gc_mark but if tp is a pair ignore the marked bit on unheaped entries */
  if (is_pair(tp))
    {
      s7_pointer p = tp;
      do {
	set_mark(p);
	gc_mark(car(p)); /* does this need to be gc_owlet_mark? I can't find a case */
	p = cdr(p);
      } while ((is_pair(p)) && (p != tp) && ((!in_heap(p)) || (!is_marked(p)))); /* ((full_type(p) & (TYPE_MASK | T_GC_MARK)) == T_PAIR) is much slower */
      gc_mark(p);
    }
  else
    if (!is_marked(tp))
      (*mark_function[unchecked_type(tp)])(tp);
}
#endif

static void mark_owlet(s7_scheme *sc)
{
#if WITH_HISTORY
  {
    s7_pointer p1 = sc->eval_history1;
    s7_pointer p2 = sc->eval_history2;
    s7_pointer p3 = sc->history_pairs;
    for (int32_t i = 1; ; i++, p2 = cdr(p2), p3 = cdr(p3))
      {
#if 0
	set_mark(p1); /* pointless? they're permanent */
	set_mark(p2);
	set_mark(p3);
#endif
	gc_owlet_mark(car(p1));
	gc_owlet_mark(car(p2));
	gc_owlet_mark(car(p3));
	p1 = cdr(p1);
	if (p1 == sc->eval_history1) break; /* these are circular lists */
      }}
#endif
  /* sc->error_type and friends are slots in owlet */
  mark_slot(sc->error_type);
  slot_set_value(sc->error_data, sc->F); /* or maybe mark_tree(slot_value(sc->error_data)) ? */
  mark_slot(sc->error_data);
  mark_slot(sc->error_code);
  mark_slot(sc->error_line);
  mark_slot(sc->error_file);
  mark_slot(sc->error_position);
#if WITH_HISTORY
  mark_slot(sc->error_history);
#endif
  set_mark(sc->owlet);
  mark_let(let_outlet(sc->owlet));
}

static void mark_c_pointer(s7_pointer p)
{
  set_mark(p);
  gc_mark(c_pointer_type(p));
  gc_mark(c_pointer_info(p));
}

static void mark_c_proc_star(s7_pointer p)
{
  set_mark(p);
  if ((!c_func_has_simple_defaults(p)) &&
      (c_function_call_args(p))) /* NULL if not a safe function */
    for (s7_pointer arg = c_function_call_args(p); is_pair(arg); arg = cdr(arg))
      gc_mark(car(arg));
}

static void mark_pair(s7_pointer p)
{
  do {
    set_mark(p);
    gc_mark(car(p)); /* expanding this to avoid recursion is slower */
    p = cdr(p);
  } while ((is_pair(p)) && (!is_marked(p))); /* ((full_type(p) & (TYPE_MASK | T_GC_MARK)) == T_PAIR) is much slower */
  gc_mark(p);
}

static void mark_counter(s7_pointer p)
{
  set_mark(p);
  gc_mark(counter_result(p));
  gc_mark(counter_list(p));
  gc_mark(counter_let(p));
}

static void mark_closure(s7_pointer p)
{
  set_mark(p);
  gc_mark(closure_args(p));
  gc_mark(closure_body(p));
  mark_let(closure_let(p));
  gc_mark(closure_setter_or_map_list(p));
}

static void mark_stack_1(s7_pointer p, s7_int top)
{
  s7_pointer *tp = (s7_pointer *)(stack_elements(p)), *tend;
  set_mark(p);
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);
  while (tp < tend)
    {
      gc_mark(*tp++);
      gc_mark(*tp++);
      gc_mark(*tp++);
      tp++;
    }
}

static void mark_stack(s7_pointer p)
{
  /* we can have a bare stack waiting for a continuation to hold it if the new_cell for the continuation triggers the GC!  But we need a top-of-stack?? */
  mark_stack_1(p, temp_stack_top(p));
}

static void mark_continuation(s7_pointer p)
{
  set_mark(p);
  if (!is_marked(continuation_stack(p))) /* can these be cyclic? */
    mark_stack_1(continuation_stack(p), continuation_stack_top(p));
  gc_mark(continuation_op_stack(p));
}

static void mark_vector(s7_pointer p)
{
  if (is_typed_vector(p))
    typed_vector_gc_mark(p)(p, vector_length(p));
  else mark_vector_1(p, vector_length(p));
}

static void mark_vector_possibly_shared(s7_pointer p)
{
  /* If a subvector (an inner dimension) of a vector is the only remaining reference
   *    to the main vector, we want to make sure the main vector is not GC'd until
   *    the subvector is also GC-able.  The subvector field either points to the
   *    parent vector, or it is sc->F, so we need to check for a vector parent if
   *    the current is multidimensional (this will include 1-dim slices).  We need
   *    to keep the parent case separate (i.e. sc->F means the current is the original)
   *    so that we only free once (or remove_from_heap once).
   *
   * If we have a subvector of a subvector, and the middle and original are not otherwise
   *   in use, we mark the middle one, but (since it itself is not in use anywhere else)
   *   we don't mark the original!  So we need to follow the share-vector chain marking every one.
   *
   * To remove a cell from the heap, we need its current heap location so that we can replace it.
   *   The heap is allocated as needed in monolithic blocks of (say) 1/2M s7_cells. When a cell
   *   is replaced, the new cell (at heap[x] say) is no longer from the original block. Since the
   *   GC clears all type bits when it frees a cell, we can't use a type bit to distinguish the
   *   replacements from the originals, but we need that info because in the base case, we use
   *   the distance of the cell from the base cell to get "x", its location.  In the replacement
   *   case, we add the location at the end of the s7_cell (s7_big_cell).  We track the current
   *   heap blocks via the sc->heap_blocks list.  To get the location of "p" above, we run through
   *   that list looking for a block it fits in.  If none is found, we assume it is an s7_big_cell
   *   and use the saved location.
   */
  if (is_subvector(p))
    mark_vector_possibly_shared(subvector_vector(p));

  /* mark_vector_1 does not check the marked bit, so if subvector below is in a cycle involving
   *   the calling vector, we get infinite recursion unless we check the mark bit here.
   */
  if (!is_marked(p))
    mark_vector_1(p, vector_length(p));
}

static void mark_int_or_float_vector(s7_pointer p) {set_mark(p);}

static void mark_int_or_float_vector_possibly_shared(s7_pointer p)
{
  if (is_subvector(p))
    mark_int_or_float_vector_possibly_shared(subvector_vector(p));
  set_mark(p);
}

static void mark_c_object(s7_pointer p)
{
  set_mark(p);
  if (c_object_gc_mark(c_object_s7(p), p))
    (*(c_object_gc_mark(c_object_s7(p), p)))(c_object_s7(p), p);
  else (*(c_object_mark(c_object_s7(p), p)))(c_object_value(p));
}

static void mark_catch(s7_pointer p)
{
  set_mark(p);
  gc_mark(catch_tag(p));
  gc_mark(catch_handler(p));
}

static void mark_dynamic_wind(s7_pointer p)
{
  set_mark(p);
  gc_mark(dynamic_wind_in(p));
  gc_mark(dynamic_wind_out(p));
  gc_mark(dynamic_wind_body(p));
}

/* if is_typed_hash_table then if c_function_marker(key|value_typer) is just_mark_vector, we can ignore that field,
 *    if it's mark_simple_vector, we just set_mark (key|value), else we gc_mark (none of this is implemented yet)
 */
static void mark_hash_table(s7_pointer p)
{
  set_mark(p);
  gc_mark(hash_table_procedures(p));
  if (hash_table_entries(p) > 0)
    {
      s7_int len = hash_table_mask(p) + 1;
      hash_entry_t **entries = hash_table_elements(p);
      hash_entry_t **last = (hash_entry_t **)(entries + len);

      if ((is_weak_hash_table(p)) &&
	  (weak_hash_iters(p) == 0))
	while (entries < last)
	  {
	    hash_entry_t *xp;
	    for (xp = *entries++; xp; xp = hash_entry_next(xp))
	      gc_mark(hash_entry_value(xp));
	    for (xp = *entries++; xp; xp = hash_entry_next(xp))
	      gc_mark(hash_entry_value(xp));
	  }
      else
	while (entries < last) /* counting entries here was slightly faster */
	  {
	    hash_entry_t *xp;
	    for (xp = *entries++; xp; xp = hash_entry_next(xp))
	      {
		gc_mark(hash_entry_key(xp));
		gc_mark(hash_entry_value(xp));
	      }
	    for (xp = *entries++; xp; xp = hash_entry_next(xp))
	      {
		gc_mark(hash_entry_key(xp));
		gc_mark(hash_entry_value(xp));
	      }}}
}

static void mark_iterator(s7_pointer p)
{
  set_mark(p);
  gc_mark(iterator_sequence(p));
  if (is_mark_seq(p))
    gc_mark(iterator_current(p));
}

static void mark_input_port(s7_pointer p)
{
  set_mark(p);
  gc_mark(port_string_or_function(p));
}

static void mark_output_port(s7_pointer p)
{
  set_mark(p);
  if (is_function_port(p))
    gc_mark(port_string_or_function(p));
}

#define clear_type(p) full_type(p) = T_FREE

static void init_mark_functions(void)
{
  mark_function[T_FREE]                = mark_noop;
  mark_function[T_UNDEFINED]           = just_mark;
  mark_function[T_EOF]                 = mark_noop;
  mark_function[T_UNSPECIFIED]         = mark_noop;
  mark_function[T_NIL]                 = mark_noop;
  mark_function[T_UNUSED]              = mark_noop;
  mark_function[T_BOOLEAN]             = mark_noop;
  mark_function[T_SYNTAX]              = mark_noop;
  mark_function[T_CHARACTER]           = mark_noop;
  mark_function[T_SYMBOL]              = mark_noop; /* this changes to just_mark when gensyms are in the heap */
  mark_function[T_STRING]              = just_mark;
  mark_function[T_INTEGER]             = just_mark;
  mark_function[T_RATIO]               = just_mark;
  mark_function[T_REAL]                = just_mark;
  mark_function[T_COMPLEX]             = just_mark;
  mark_function[T_BIG_INTEGER]         = just_mark;
  mark_function[T_BIG_RATIO]           = just_mark;
  mark_function[T_BIG_REAL]            = just_mark;
  mark_function[T_BIG_COMPLEX]         = just_mark;
  mark_function[T_RANDOM_STATE]        = just_mark;
  mark_function[T_GOTO]                = just_mark;
  mark_function[T_OUTPUT_PORT]         = just_mark; /* changed to mark_output_port if output function ports are active */
  mark_function[T_C_MACRO]             = just_mark;
  mark_function[T_C_POINTER]           = mark_c_pointer;
  mark_function[T_C_FUNCTION]          = just_mark;
  mark_function[T_C_FUNCTION_STAR]     = just_mark; /* changes to mark_c_proc_star if defaults involve an expression */
  mark_function[T_C_RST_NO_REQ_FUNCTION] = just_mark;
  mark_function[T_PAIR]                = mark_pair;
  mark_function[T_CLOSURE]             = mark_closure;
  mark_function[T_CLOSURE_STAR]        = mark_closure;
  mark_function[T_CONTINUATION]        = mark_continuation;
  mark_function[T_INPUT_PORT]          = mark_input_port;
  mark_function[T_VECTOR]              = mark_vector; /* this changes if subvector created (similarly below) */
  mark_function[T_INT_VECTOR]          = mark_int_or_float_vector;
  mark_function[T_FLOAT_VECTOR]        = mark_int_or_float_vector;
  mark_function[T_BYTE_VECTOR]         = just_mark;
  mark_function[T_MACRO]               = mark_closure;
  mark_function[T_BACRO]               = mark_closure;
  mark_function[T_MACRO_STAR]          = mark_closure;
  mark_function[T_BACRO_STAR]          = mark_closure;
  mark_function[T_C_OBJECT]            = mark_c_object;
  mark_function[T_CATCH]               = mark_catch;
  mark_function[T_DYNAMIC_WIND]        = mark_dynamic_wind;
  mark_function[T_HASH_TABLE]          = mark_hash_table;
  mark_function[T_ITERATOR]            = mark_iterator;
  mark_function[T_LET]                 = mark_let;
  mark_function[T_STACK]               = mark_stack;
  mark_function[T_COUNTER]             = mark_counter;
  mark_function[T_SLOT]                = mark_slot;
}

static void mark_op_stack(s7_scheme *sc)
{
  s7_pointer *p = sc->op_stack;
  s7_pointer *tp = sc->op_stack_now;
  while (p < tp)
    gc_mark(*p++);
}

static void mark_input_port_stack(s7_scheme *sc)
{
  s7_pointer *tp = (s7_pointer *)(sc->input_port_stack + sc->input_port_stack_loc);
  for (s7_pointer *p = sc->input_port_stack; p < tp; p++)
    gc_mark(*p);
}

static void mark_rootlet(s7_scheme *sc)
{
  s7_pointer ge = sc->rootlet;
  s7_pointer *tmp = rootlet_elements(ge);
  s7_pointer *top = (s7_pointer *)(tmp + sc->rootlet_entries);
  set_mark(ge);
  while (tmp < top)
    gc_mark(slot_value(*tmp++));
  /* slot_setter is handled below with an explicit list -- more code than its worth probably */
  /* we're not marking slot_symbol above which makes me worry that a top-level gensym won't be protected
   *   (apply define (gensym) '(32)), then try to get the GC to clobber {gensym}-0,
   *   but I can't get it to break, so they must be protected somehow; apparently they are
   *   removed from the heap!  At least: (define-macro (defit) (let ((n (gensym))) `(define (,n) (format #t "fun")))) (defit)
   *   removes the function from the heap (protecting the gensym).
   */
}

/* mark_closure calls mark_let on closure_let(func) which marks slot values.
 *   if we move rootlet to end, unmarked closures at that point could mark let/slot but not slot value?
 *   or save safe-closure lets to handle all at end?  or a gc_list of safe closure lets and only mark let if not safe?
 */

static void mark_permanent_objects(s7_scheme *sc)
{
  for (gc_obj_t *g = sc->permanent_objects; g; g = (gc_obj_t *)(g->nxt))
    gc_mark(g->p);
  /* permanent_objects also has lets (removed from heap) -- should they be handled like permanent_lets?
   *    if unmarked should either be removed from the list and perhaps placed on a free list?
   *    if outlet is free can the let potentially be in use?
   *    there are many more permanent_lets(slots) than permanent objects
   */
}
/* do we mark funclet slot values from the function as root?  Maybe treat them like permanent_lets here? */

static void unmark_permanent_objects(s7_scheme *sc)
{
  gc_obj_t *g;
  for (g = sc->permanent_objects; g; g = (gc_obj_t *)(g->nxt))
    clear_mark(g->p);
  for (g = sc->permanent_lets; g; g = (gc_obj_t *)(g->nxt)) /* there are lets and slots in this list */
    clear_mark(g->p);
}

#if (!MS_WINDOWS)
  #include <time.h>
  #include <sys/time.h>
#endif

#if S7_DEBUGGING
static bool has_odd_bits(s7_pointer obj);
#endif
static char *describe_type_bits(s7_scheme *sc, s7_pointer obj);
static s7_pointer make_symbol(s7_scheme *sc, const char *name);
#if WITH_GCC
static __attribute__ ((format (printf, 3, 4))) void s7_warn(s7_scheme *sc, s7_int len, const char *ctrl, ...);
#else
static void s7_warn(s7_scheme *sc, s7_int len, const char *ctrl, ...);
#endif

#if S7_DEBUGGING
#define call_gc(Sc) gc(Sc, __func__, __LINE__)
static int64_t gc(s7_scheme *sc, const char *func, int32_t line)
#else
#define call_gc(Sc) gc(Sc)
static int64_t gc(s7_scheme *sc)
#endif
{
  s7_cell **old_free_heap_top;
  s7_int i;
  s7_pointer p;

  sc->gc_start = my_clock();
  sc->gc_calls++;
#if S7_DEBUGGING
  sc->last_gc_line = line;
#endif
  sc->continuation_counter = 0;

  mark_rootlet(sc);
  mark_owlet(sc);

  gc_mark(sc->code);
  if (sc->args) gc_mark(sc->args);
  gc_mark(sc->curlet);   /* not mark_let because op_any_closure_3p uses sc->curlet as a temp!! */
  mark_current_code(sc); /* probably redundant if with_history */
  gc_mark(sc->value);

  mark_stack_1(sc->stack, current_stack_top(sc));
  set_mark(current_input_port(sc));
  mark_input_port_stack(sc);
  set_mark(current_output_port(sc));
  set_mark(sc->error_port);
  gc_mark(sc->stacktrace_defaults);
  gc_mark(sc->autoload_table);
  gc_mark(sc->default_rng);
  gc_mark(sc->let_temp_hook);

  gc_mark(sc->w);
  gc_mark(sc->x);
  gc_mark(sc->y);
  gc_mark(sc->z);
  gc_mark(sc->temp1);
  gc_mark(sc->temp2);
  gc_mark(sc->temp3);
  gc_mark(sc->temp4);
  gc_mark(sc->temp5);
  gc_mark(sc->temp6);
  gc_mark(sc->temp7);
  gc_mark(sc->temp8);
  gc_mark(sc->temp9);

  gc_mark(car(sc->t1_1));
  gc_mark(car(sc->t2_1)); gc_mark(car(sc->t2_2));
  gc_mark(car(sc->t3_1)); gc_mark(car(sc->t3_2)); gc_mark(car(sc->t3_3));
  gc_mark(car(sc->t4_1));
  gc_mark(car(sc->plist_1));
  gc_mark(car(sc->plist_2)); gc_mark(cadr(sc->plist_2));
  for (p = sc->plist_3; is_pair(p); p = cdr(p)) gc_mark(car(p));
  gc_mark(car(sc->qlist_2)); gc_mark(cadr(sc->qlist_2));
  gc_mark(car(sc->qlist_3));
  gc_mark(car(sc->u1_1));
  gc_mark(car(sc->u2_1)); gc_mark(car(sc->u2_2));

  gc_mark(sc->rec_p1);
  gc_mark(sc->rec_p2);

  /* these probably don't need to be marked */
  for (p = sc->wrong_type_arg_info; is_pair(p); p = cdr(p)) gc_mark(car(p));
  for (p = sc->simple_wrong_type_arg_info; is_pair(p); p = cdr(p)) gc_mark(car(p));
  for (p = sc->out_of_range_info; is_pair(p); p = cdr(p)) gc_mark(car(p));
  for (p = sc->simple_out_of_range_info; is_pair(p); p = cdr(p)) gc_mark(car(p));

  gc_mark(car(sc->elist_1));
  gc_mark(car(sc->elist_2));
  gc_mark(cadr(sc->elist_2));
  for (p = sc->elist_3; is_pair(p); p = cdr(p)) gc_mark(car(p));
  gc_mark(car(sc->elist_4));
  gc_mark(car(sc->elist_5));
  gc_mark(car(sc->elist_6));
  gc_mark(car(sc->elist_7));

  for (i = 1; i < NUM_SAFE_LISTS; i++)
    if ((is_pair(sc->safe_lists[i])) &&
	(list_is_in_use(sc->safe_lists[i])))
      for (p = sc->safe_lists[i]; is_pair(p); p = cdr(p))
	gc_mark(car(p));

  for (i = 0; i < sc->setters_loc; i++)
    gc_mark(cdr(sc->setters[i]));

  for (i = 0; i <= sc->format_depth; i++) /* sc->num_fdats is size of array */
    if (sc->fdats[i])
      gc_mark(sc->fdats[i]->curly_arg);

  if (sc->rec_stack)
    {
      just_mark(sc->rec_stack);
      for (i = 0; i < sc->rec_loc; i++)
	gc_mark(sc->rec_els[i]);
    }
  mark_vector(sc->protected_objects);
  mark_vector(sc->protected_setters);
  set_mark(sc->protected_setter_symbols);
  if ((is_symbol(sc->profile_prefix)) && (is_gensym(sc->profile_prefix))) set_mark(sc->profile_prefix);

  /* now protect recent allocations using the free_heap cells above the current free_heap_top (if any).
   * cells above sc->free_heap_top might be malloc'd garbage (after heap reallocation), so we keep track of
   *   where the last actually freed cells were after the previous GC call.  We're trying to
   *   GC protect the previous GC_TEMPS_SIZE allocated pointers so that the caller doesn't have
   *   to gc-protect every temporary cell.
   * There's one remaining possible problem.  remove_from_heap frees cells outside
   *   the GC and might push free_heap_top beyond its previous_free_heap_top, then
   *   an immediate explicit gc call might not see those temp cells.
   */
  {
    s7_pointer *tmps = sc->free_heap_top;
    s7_pointer *tmps_top = tmps + sc->gc_temps_size;
    if (tmps_top > sc->previous_free_heap_top)
      tmps_top = sc->previous_free_heap_top;
    while (tmps < tmps_top)
      gc_mark(*tmps++);
  }
  mark_op_stack(sc);
  mark_permanent_objects(sc);

  if (sc->profiling_gensyms)
    {
      profile_data_t *pd = sc->profile_data;
      for (i = 0; i < pd->top; i++)
	if ((pd->funcs[i]) && (is_gensym(pd->funcs[i])))
	  set_mark(pd->funcs[i]);
    }

  {
    gc_list_t *gp = sc->opt1_funcs;
    for (i = 0; i < gp->loc; i++)
      {
	s7_pointer s1 = T_Pair(gp->list[i]);
	if ((is_marked(s1)) && (!is_marked(opt1_any(s1)))) /* opt1_lambda, but op_unknown* can change to opt1_cfunc etc */
	  gc_mark(opt1_any(s1));                           /* not set_mark -- need to protect let/body/args as well */
      }}

  /* free up all unmarked objects */
  old_free_heap_top = sc->free_heap_top;
  {
    s7_pointer *fp = sc->free_heap_top;
    s7_pointer *tp = sc->heap, *heap_top;
    heap_top = (s7_pointer *)(sc->heap + sc->heap_size);

#if S7_DEBUGGING
  #define gc_object(Tp)							\
    p = (*Tp++);							\
    if (signed_type(p) > 0)						\
      {								        \
        p->debugger_bits = 0; p->gc_func = func; p->gc_line = line;	\
        /* if (unchecked_type(p) == T_PAIR) {p->object.cons.opt1 = NULL; p->object.cons.o2.opt2 = NULL; p->object.cons.o3.opt3 = NULL;} */\
        if (has_odd_bits(p)) {char *s; fprintf(stderr, "odd bits: %s\n", s = describe_type_bits(sc, p)); free(s);} \
        signed_type(p) = 0;						\
        (*fp++) = p;							\
      }									\
    else if (signed_type(p) < 0) clear_mark(p);
#else
  #define gc_object(Tp) p = (*Tp++); if (signed_type(p) > 0) {signed_type(p) = 0; (*fp++) = p;} else if (signed_type(p) < 0) clear_mark(p);
  /* this appears to be about 10% faster than the previous form
   *   if the sign bit is on, but no other bits, this version will take no action (it thinks the cell is on the free list), but
   *   it means we've marked a free cell as in-use: since types are set as soon as removed from the free list, this has to be a bug
   *   (this case is caught by has_odd_bits).  If ignored, the type will be set, and later the bit cleared, so no problem?
   *   An alternate form that simply calls clear_mark (no check for < 0) appears to be the same speed even in cases with lots
   *   of long-lived objects.
   */
#endif
    while (tp < heap_top)          /* != here or ^ makes no difference, and going to 64 (from 32) doesn't matter */
      {
	LOOP_8(gc_object(tp));
	LOOP_8(gc_object(tp));
	LOOP_8(gc_object(tp));
	LOOP_8(gc_object(tp));
      }
    /* I tried using pthreads here, since there is no need for a lock in this loop, but the *fp++ part needs to
     *   be local to each thread, then merged at the end.  In my timing tests, the current version was faster.
     *   If NUM_THREADS=2, and all thread variables are local, surely there's no "false sharing"?
     */
    sc->free_heap_top = fp;
    sweep(sc);
  }

  unmark_permanent_objects(sc);
  sc->gc_freed = (int64_t)(sc->free_heap_top - old_free_heap_top);
  sc->gc_total_freed += sc->gc_freed;
  sc->gc_end = my_clock();
  sc->gc_total_time += (sc->gc_end - sc->gc_start);

  if (show_gc_stats(sc))
    {
#if (!MS_WINDOWS)
#if S7_DEBUGGING
      s7_warn(sc, 512, "%s[%d]: gc freed %" ld64 "/%" ld64 " (free: %" p64 "), time: %f\n", func, line,
	      sc->gc_freed, sc->heap_size, (intptr_t)(sc->free_heap_top - sc->free_heap), (double)(sc->gc_end - sc->gc_start) / ticks_per_second());
#else
      s7_warn(sc, 256, "gc freed %" ld64 "/%" ld64 " (free: %" p64 "), time: %f\n",
	      sc->gc_freed, sc->heap_size, (intptr_t)(sc->free_heap_top - sc->free_heap), (double)(sc->gc_end - sc->gc_start) / ticks_per_second());
#endif
#else
      s7_warn(sc, 256, "gc freed %" ld64 "/%" ld64 "\n", sc->gc_freed, sc->heap_size);
#endif
    }
  if (show_protected_objects_stats(sc))
    {
      s7_int num, len = vector_length(sc->protected_objects); /* allocated at startup */
      for (i = 0, num = 0; i < len; i++)
	if (vector_element(sc->protected_objects, i) != sc->unused)
	  num++;
      s7_warn(sc, 256, "gc-protected-objects: %" ld64 " in use of %" ld64 "\n", num, len);
    }
  sc->previous_free_heap_top = sc->free_heap_top;
  return(sc->gc_freed);
}

#define GC_RESIZE_HEAP_BY_4_FRACTION 0.67
/*   .5+.1: test -3?, dup +86, tmap +45, tsort -3, thash +305.  .85+.7: dup -5 */

static void resize_heap_to(s7_scheme *sc, int64_t size)
{
  int64_t old_size = sc->heap_size;
  int64_t old_free = sc->free_heap_top - sc->free_heap;
  s7_cell *cells;
  s7_cell **cp;
  heap_block_t *hp;

  if (size == 0)
    {
      /* (sc->heap_size < 2048000) */  /* 8192000 here improves various gc benchmarks only slightly */
      /* maybe the choice of 4 should depend on how much space was freed rather than the current heap_size? */
      if (old_free < old_size * sc->gc_resize_heap_by_4_fraction)
	sc->heap_size *= 4;          /* *8 if < 1M (or whatever) doesn't make much difference */
      else sc->heap_size *= 2;
    }
  else
    if (size > sc->heap_size)
      while (sc->heap_size < size) sc->heap_size *= 2;
    else return;
  /* do not call new_cell here! */
#if POINTER_32
  if (((2 * sc->heap_size * sizeof(s7_cell *)) + ((sc->heap_size - old_size) * sizeof(s7_cell))) >= SIZE_MAX)
    { /* can this happen in 64-bit land?  SIZE_MAX is unsigned int in 32-bit, unsigned long in 64 bit = UINTPTR_MAX = 18446744073709551615UL */
      s7_warn(sc, 256, "heap size requested, %" ld64 " => %" ld64 " bytes, is greater than size_t: %u\n",
	      sc->heap_size,
	      (2 * sc->heap_size * sizeof(s7_cell *)) + ((sc->heap_size - old_size) * sizeof(s7_cell)),
	      SIZE_MAX);
      sc->heap_size = old_size + 64000;
    }
#endif
  cp = (s7_cell **)Realloc(sc->heap, sc->heap_size * sizeof(s7_cell *));
  if (cp)
    sc->heap = cp;
  else
    {
      s7_warn(sc, 256, "heap reallocation failed! tried to get %" ld64 " bytes (will retry with a smaller amount)\n",
	      (int64_t)(sc->heap_size * sizeof(s7_cell *)));
      sc->heap_size = old_size + 64000;
      sc->heap = (s7_cell **)Realloc(sc->heap, sc->heap_size * sizeof(s7_cell *));
    }
  sc->free_heap = (s7_cell **)Realloc(sc->free_heap, sc->heap_size * sizeof(s7_cell *));
  sc->free_heap_trigger = (s7_cell **)(sc->free_heap + GC_TRIGGER_SIZE);
  sc->free_heap_top = sc->free_heap + old_free; /* incremented below, added old_free 21-Aug-12?!? */

  cells = (s7_cell *)Calloc(sc->heap_size - old_size, sizeof(s7_cell)); /* Malloc + clear_type below is much slower?! */
  add_saved_pointer(sc, (void *)cells);
  {
    s7_pointer p = cells;
    for (int64_t k = old_size; k < sc->heap_size;)
      {
	LOOP_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
	LOOP_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
	LOOP_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
	LOOP_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
      }}
  hp = (heap_block_t *)Malloc(sizeof(heap_block_t));
  hp->start = (intptr_t)cells;
  hp->end = (intptr_t)cells + ((sc->heap_size - old_size) * sizeof(s7_cell));
  hp->offset = old_size;
  hp->next = sc->heap_blocks;
  sc->heap_blocks = hp;
  sc->previous_free_heap_top = sc->free_heap_top;

  if (show_heap_stats(sc))
    {
      const char *str = string_value(object_to_truncated_string(sc, current_code(sc), 80));
      if (size != 0)
	s7_warn(sc, 512, "heap grows to %" ld64 " (old free/size: %" ld64 "/%" ld64 ", requested %" ld64 ") from %s\n",
		sc->heap_size, old_free, old_size, size, str);
      else s7_warn(sc, 512, "heap grows to %" ld64 " (old free/size: %" ld64 "/%" ld64 ") from %s\n",
		   sc->heap_size, old_free, old_size, str);
    }
  if (sc->heap_size >= sc->max_heap_size)
    s7_error_nr(sc, make_symbol(sc, "heap-too-big"),
		set_elist_3(sc, wrap_string(sc, "heap has grown past (*s7* 'max-heap-size): ~S > ~S", 50),
			    wrap_integer(sc, sc->max_heap_size),
			    wrap_integer(sc, sc->heap_size)));
}

#define resize_heap(Sc) resize_heap_to(Sc, 0)

#ifndef GC_RESIZE_HEAP_FRACTION
  #define GC_RESIZE_HEAP_FRACTION 0.8
/* 1/2 is ok, 3/4 speeds up some GC benchmarks, 7/8 is a bit faster, 95/100 comes to a halt (giant heap)
 *    in my tests, only tvect.scm ends up larger if 3/4 used
 */
#endif

#if S7_DEBUGGING
static void try_to_call_gc_1(s7_scheme *sc, const char *func, int32_t line)
#else
static void try_to_call_gc(s7_scheme *sc)
#endif
{
  /* called only from new_cell */
  if (sc->gc_off)     /* we can't just return here!  Someone needs a new cell, and once the heap free list is exhausted, segfault */
    resize_heap(sc);
  else
    {
#if (!S7_DEBUGGING)
      int64_t freed_heap = gc(sc);
      if (freed_heap < (sc->heap_size * sc->gc_resize_heap_fraction))
	resize_heap(sc);
#else
      gc(sc, func, line);
      if ((int64_t)(sc->free_heap_top - sc->free_heap) < (sc->heap_size * sc->gc_resize_heap_fraction))
	resize_heap(sc);
#endif
    }
}
  /* originally I tried to mark each temporary value until I was done with it, but
   *   that way madness lies... By delaying GC of _every_ %$^#%@ pointer, I can dispense
   *   with hundreds of individual protections.  So the free_heap's last GC_TEMPS_SIZE
   *   allocated pointers are protected during the mark sweep.
   */

static s7_pointer g_gc(s7_scheme *sc, s7_pointer args)
{
  #define H_gc "(gc (on #t)) runs the garbage collector.  If 'on' is supplied, it turns the GC on or off. \
Evaluation produces a surprising amount of garbage, so don't leave the GC off for very long!"
  #define Q_gc s7_make_signature(sc, 2, sc->T, sc->is_boolean_symbol)

  /* g_gc can't be called in a situation where these lists matter (I think...) */
  set_plist_1(sc, sc->nil);
  set_plist_2(sc, sc->nil, sc->nil);
  set_plist_3(sc, sc->nil, sc->nil, sc->nil);
  set_qlist_2(sc, sc->nil, sc->nil);
  set_car(sc->qlist_3, sc->nil);
  set_elist_1(sc, sc->nil);
  set_elist_2(sc, sc->nil, sc->nil);
  set_elist_3(sc, sc->nil, sc->nil, sc->nil);
  set_car(sc->elist_4, sc->nil);
  set_car(sc->elist_5, sc->nil);
  set_car(sc->elist_6, sc->nil);
  set_car(sc->elist_7, sc->nil);
  set_car(sc->dlist_1, sc->nil);

  if (is_not_null(args))
    {
      if (!s7_is_boolean(car(args)))
	return(method_or_bust_one_arg(sc, car(args), sc->gc_symbol, args, T_BOOLEAN));
      sc->gc_off = (car(args) == sc->F);
      if (sc->gc_off)
	return(sc->F);
    }
  call_gc(sc);
  return(sc->unspecified);
}

s7_pointer s7_gc_on(s7_scheme *sc, bool on)
{
  sc->gc_off = !on;
  return(s7_make_boolean(sc, on));
}

#if S7_DEBUGGING
static void check_free_heap_size_1(s7_scheme *sc, s7_int size, const char *func, int32_t line)
#define check_free_heap_size(Sc, Size) check_free_heap_size_1(Sc, Size, __func__, __LINE__)
#else
static void check_free_heap_size(s7_scheme *sc, s7_int size)
#endif
{
  s7_int free_cells = sc->free_heap_top - sc->free_heap;
  if (free_cells < size)
    {
#if S7_DEBUGGING
      gc(sc, func, line);
#else
      gc(sc);
#endif
      while ((sc->free_heap_top - sc->free_heap) < size)
	resize_heap(sc);
    }
}

#define ALLOC_POINTER_SIZE 256
static s7_cell *alloc_pointer(s7_scheme *sc)
{
  if (sc->alloc_pointer_k == ALLOC_POINTER_SIZE)     /* if either no current block or the block is used up, make a new block */
    {
      sc->permanent_cells += ALLOC_POINTER_SIZE;
      sc->alloc_pointer_cells = (s7_cell *)Calloc(ALLOC_POINTER_SIZE, sizeof(s7_cell)); /* not Malloc here or below (maybe set full type to 0 if Malloc) */
      add_saved_pointer(sc, sc->alloc_pointer_cells);
      sc->alloc_pointer_k = 0;
    }
  return(&(sc->alloc_pointer_cells[sc->alloc_pointer_k++]));
}

#define ALLOC_BIG_POINTER_SIZE 256
static s7_big_cell *alloc_big_pointer(s7_scheme *sc, int64_t loc)
{
  s7_big_pointer p;
  if (sc->alloc_big_pointer_k == ALLOC_BIG_POINTER_SIZE)
    {
      sc->permanent_cells += ALLOC_BIG_POINTER_SIZE;
      sc->alloc_big_pointer_cells = (s7_big_cell *)Calloc(ALLOC_BIG_POINTER_SIZE, sizeof(s7_big_cell));
      add_saved_pointer(sc, sc->alloc_big_pointer_cells);
      sc->alloc_big_pointer_k = 0;
    }
  p = (&(sc->alloc_big_pointer_cells[sc->alloc_big_pointer_k++]));
  p->big_hloc = loc;
  /* needed if this new pointer is itself petrified later -- it's not from one of the heap blocks,
   *   but it's in the heap, and we'll need to know where it is in the heap to replace it
   */
  return(p);
}

static void add_permanent_object(s7_scheme *sc, s7_pointer obj) /* called by remove_from_heap */
{
  gc_obj_t *g = (gc_obj_t *)Malloc(sizeof(gc_obj_t));
  g->p = obj;
  g->nxt = sc->permanent_objects;
  sc->permanent_objects = g;
}

static void add_permanent_let_or_slot(s7_scheme *sc, s7_pointer obj)
{
  gc_obj_t *g = (gc_obj_t *)Malloc(sizeof(gc_obj_t));
  g->p = obj;
  g->nxt = sc->permanent_lets;
  sc->permanent_lets = g;
}

#if S7_DEBUGGING
static const char *type_name_from_type(int32_t typ, article_t article);

#define free_cell(Sc, P) free_cell_1(Sc, P, __LINE__)
static void free_cell_1(s7_scheme *sc, s7_pointer p, int32_t line)
#else
static void free_cell(s7_scheme *sc, s7_pointer p)
#endif
{
#if S7_DEBUGGING
  /* anything that needs gc_list attention should not be freed here */
  uint8_t typ = unchecked_type(p);
  gc_list_t *gp = sc->opt1_funcs;

  if ((t_freeze_p[typ]) || ((typ == T_SYMBOL) && (is_gensym(p))))
    fprintf(stderr, "free_cell of %s?\n", type_name_from_type(typ, NO_ARTICLE));
  if ((t_any_closure_p[typ]) && (gp->loc > 0))
    for (s7_int i = 0; i < gp->loc; i++)
      if (gp->list[i] == p)
	fprintf(stderr, "opt1_funcs free_cell of %s?\n", type_name_from_type(typ, NO_ARTICLE));
  gp = sc->weak_refs;
  if (gp->loc > 0)
    for (s7_int i = 0; i < gp->loc; i++)
      if (gp->list[i] == p)
	fprintf(stderr, "weak refs free_cell of %s?\n", type_name_from_type(typ, NO_ARTICLE));

  p->debugger_bits = 0;
  p->explicit_free_line = line;
#endif
  clear_type(p);
  (*(sc->free_heap_top++)) = p;
}

static inline s7_pointer petrify(s7_scheme *sc, s7_pointer x)
{
  int64_t loc = heap_location(sc, x);
  s7_pointer p = (s7_pointer)alloc_big_pointer(sc, loc);
  sc->heap[loc] = p;
  free_cell(sc, p);
  unheap(sc, x);        /* set_immutable(x); */ /* if there are GC troubles, this might catch them? */
  return(x);
}

#if S7_DEBUGGING
#define remove_gensym_from_heap(Sc, Gensym) remove_gensym_from_heap_1(Sc, Gensym, __func__, __LINE__)
static void remove_gensym_from_heap_1(s7_scheme *sc, s7_pointer x, const char *func, int line)
#else
static void remove_gensym_from_heap(s7_scheme *sc, s7_pointer x) /* x known to be a symbol and in the heap */
#endif
{
  int64_t loc = heap_location(sc, x);
  sc->heap[loc] = (s7_pointer)alloc_big_pointer(sc, loc);
  free_cell(sc, sc->heap[loc]);
#if S7_DEBUGGING
  x->gc_func = func;
  x->gc_line = line;
#endif
  unheap(sc, x); /* set UNHEAP bit in type(x) */
  {
    gc_list_t *gp = sc->gensyms;
    for (s7_int i = 0; i < gp->loc; i++) /* sc->gensyms reaches size 512 during s7test, but this search is called 3 times and costs nothing */
      if (gp->list[i] == x)
	{
	  for (s7_int j = i + 1; i < gp->loc - 1; i++, j++)
	    gp->list[i] = gp->list[j];
	  gp->list[i] = NULL;
	  gp->loc--;
	  if (gp->loc == 0) mark_function[T_SYMBOL] = mark_noop;
	  break;
	}}
}

static inline void remove_from_heap(s7_scheme *sc, s7_pointer x)
{
  /* global functions are very rarely redefined, so we can remove the function body from the heap when it is defined */
  if (!in_heap(x)) return;
  if (is_pair(x))
    {
      s7_pointer p = x;
      do {
	petrify(sc, p);
	remove_from_heap(sc, car(p));
	p = cdr(p);
      } while (is_pair(p) && (in_heap(p)));
      if (in_heap(p)) petrify(sc, p);
      return;
    }
  switch (type(x))
    {
    case T_LET: /* very rare */
      if (is_funclet(x)) set_immutable(x);
    case T_HASH_TABLE:
    case T_VECTOR:
      /* not int|float_vector or string because none of their elements are GC-able (so unheap below is ok)
       *   but hash-table and let seem like they need protection? And let does happen via define-class.
       */
      add_permanent_object(sc, x);
      return;

    case T_SYMBOL:
      if (is_gensym(x))
	remove_gensym_from_heap(sc, x);
      return;

    case T_CLOSURE: case T_CLOSURE_STAR:
    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
      /* these need to be GC-protected! */
      add_permanent_object(sc, x);
      return;

    default:
      break;
    }
  petrify(sc, x);
}


/* -------------------------------- stacks -------------------------------- */

#define OP_STACK_INITIAL_SIZE 64

#if S7_DEBUGGING
static void push_op_stack(s7_scheme *sc, s7_pointer op)
{
  (*sc->op_stack_now++) = T_Pos(op);
  if (sc->op_stack_now > (sc->op_stack + sc->op_stack_size))
    {
      fprintf(stderr, "%sop_stack overflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
}

static s7_pointer pop_op_stack(s7_scheme *sc)
{
  s7_pointer op = (*(--(sc->op_stack_now)));
  if (sc->op_stack_now < sc->op_stack)
    {
      fprintf(stderr, "%sop_stack underflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  return(T_Pos(op));
}
#else
#define push_op_stack(Sc, Op) (*Sc->op_stack_now++) = Op
#define pop_op_stack(Sc)      (*(--(Sc->op_stack_now)))
#endif

static void initialize_op_stack(s7_scheme *sc)
{
  sc->op_stack = (s7_pointer *)Malloc(OP_STACK_INITIAL_SIZE * sizeof(s7_pointer));
  sc->op_stack_size = OP_STACK_INITIAL_SIZE;
  sc->op_stack_now = sc->op_stack;
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
  for (int32_t i = 0; i < OP_STACK_INITIAL_SIZE; i++) sc->op_stack[i] = sc->nil;
}

static void resize_op_stack(s7_scheme *sc)
{
  int32_t new_size = sc->op_stack_size * 2;
  int32_t loc = (int32_t)(sc->op_stack_now - sc->op_stack);
  sc->op_stack = (s7_pointer *)Realloc((void *)(sc->op_stack), new_size * sizeof(s7_pointer));
  for (int32_t i = sc->op_stack_size; i < new_size; i++) sc->op_stack[i] = sc->nil;
  sc->op_stack_size = (uint32_t)new_size;
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + loc);
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
}

#if S7_DEBUGGING
#define pop_stack(Sc) pop_stack_1(Sc, __func__, __LINE__)
static void pop_stack_1(s7_scheme *sc, const char *func, int32_t line)
{
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start)
    {
      fprintf(stderr, "%s%s[%d]: stack underflow%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  /* here and in push_stack, both code and args might be non-free only because they've been retyped
   *   inline (as in named let) -- they actually don't make sense in these cases, but are ignored,
   *   and are carried around as GC protection in other cases.
   */
  sc->code = T_Pos(sc->stack_end[0]);
  sc->curlet = sc->stack_end[1];  /* not T_Lid|Pos, see op_any_closure_3p_end et al (stack used to pass args, not curlet) */
  sc->args = sc->stack_end[2];
  sc->cur_op = (opcode_t)(sc->stack_end[3]);
  if (sc->cur_op >= NUM_OPS)
    {
      fprintf(stderr, "%s%s[%d]: pop_stack invalid opcode: %" p64 " %s\n", BOLD_TEXT, func, line, sc->cur_op, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  if ((sc->cur_op != OP_GC_PROTECT) &&
      (!is_let(sc->stack_end[1])) && (!is_null(sc->stack_end[1])) &&
      (sc->cur_op != OP_ANY_CLOSURE_3P_3)) /* used as third GC protection field */
    fprintf(stderr, "%s[%d]: curlet not a let: %s\n", func, line, op_names[sc->cur_op]);
}

#define pop_stack_no_op(Sc) pop_stack_no_op_1(Sc, __func__, __LINE__)
static void pop_stack_no_op_1(s7_scheme *sc, const char *func, int32_t line)
{
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start)
    {
      fprintf(stderr, "%s%s[%d]: stack underflow%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  sc->code = T_Pos(sc->stack_end[0]);
  if ((sc->cur_op != OP_GC_PROTECT) && (!is_let(sc->stack_end[1])) && (!is_null(sc->stack_end[1])))
    fprintf(stderr, "%s[%d]: curlet not a let\n", func, line);
  sc->curlet = sc->stack_end[1]; /* not T_Lid|Pos: gc_protect can set this directly (not through push_stack) to anything */
  sc->args = sc->stack_end[2];
}

#define push_stack(Sc, Op, Args, Code)	\
  do {s7_pointer *_end_; _end_ = Sc->stack_end; push_stack_1(Sc, Op, Args, Code, _end_, __func__, __LINE__);} while (0)

static void push_stack_1(s7_scheme *sc, opcode_t op, s7_pointer args, s7_pointer code, s7_pointer *end, const char *func, int32_t line)
{
  if ((SHOW_EVAL_OPS) && (op == OP_EVAL_DONE)) fprintf(stderr, "%s[%d]: push eval_done\n", func, line);
  if (sc->stack_end >= sc->stack_start + sc->stack_size)
    {
      fprintf(stderr, "%s%s[%d]: stack overflow, %" ld64 " > %u, trigger: %" ld64 " %s\n",
	      BOLD_TEXT, func, line,
	      (s7_int)((intptr_t)(sc->stack_end - sc->stack_start)), sc->stack_size,
	      (s7_int)((intptr_t)(sc->stack_resize_trigger - sc->stack_start)),
	      UNBOLD_TEXT);
      if (S7_DEBUGGING) s7_show_stack(sc);
      if (sc->stop_at_error) abort();
    }
  if (sc->stack_end >= sc->stack_resize_trigger)
    fprintf(stderr, "%s%s[%d]: stack resize skipped%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
  if (sc->stack_end != end)
    fprintf(stderr, "%s[%d]: stack changed in push_stack\n", func, line);
  if (op >= NUM_OPS)
    {
      fprintf(stderr, "%s%s[%d]: push_stack invalid opcode: %" p64 " %s\n", BOLD_TEXT, func, line, sc->cur_op, UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
  if (code) sc->stack_end[0] = T_Pos(code);
  sc->stack_end[1] = T_Lid(sc->curlet);
  if ((args) && (unchecked_type(args) != T_FREE)) sc->stack_end[2] = T_Pos(args);
  sc->stack_end[3] = (s7_pointer)op;
  sc->stack_end += 4;
}

#define push_stack_no_code(Sc, Op, Args)        push_stack(Sc, Op, Args, Sc->unused)
#define push_stack_no_let_no_code(Sc, Op, Args) push_stack(Sc, Op, Args, Sc->unused)
#define push_stack_no_args(Sc, Op, Code)        push_stack(Sc, Op, Sc->unused, Code)
#define push_stack_no_let(Sc, Op, Args, Code)   push_stack(Sc, Op, Args, Code)
#define push_stack_op(Sc, Op)                   push_stack(Sc, Op, Sc->unused, Sc->unused)
#define push_stack_op_let(Sc, Op)               push_stack(Sc, Op, Sc->unused, Sc->unused)
#define push_stack_direct(Sc, Op)               push_stack(Sc, Op, Sc->args, Sc->code)
#define push_stack_no_args_direct(Sc, Op)       push_stack(Sc, Op, Sc->unused, Sc->code)
/* in the non-debugging case, the sc->unused's here are not set, so we can (later) pop free cells */

#else

#define pop_stack(Sc) do {Sc->stack_end -= 4; memcpy((void *)Sc, (void *)(Sc->stack_end), 4 * sizeof(s7_pointer));} while (0)
#define pop_stack_no_op(Sc) {Sc->stack_end -= 4; memcpy((void *)Sc, (void *)(Sc->stack_end), 3 * sizeof(s7_pointer));} while (0)

#define push_stack(Sc, Op, Args, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[1] = Sc->curlet; \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_direct(Sc, Op) \
  do { \
      Sc->cur_op = Op; \
      memcpy((void *)(Sc->stack_end), (void *)Sc, 4 * sizeof(s7_pointer)); \
      /* Sc->stack_end[3] = (s7_pointer)(Op); */ \
      Sc->stack_end += 4; \
  } while (0)
/* is this faster with cur_op because of the cast to s7_pointer, or is callgrind messing up memcpy stats?
 *   time's output is all over the map.  I think the cur_op form should be slower, but callgrind disagrees.
 */

#define push_stack_no_code(Sc, Op, Args) \
  do { \
      Sc->stack_end[1] = Sc->curlet; \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_let_no_code(Sc, Op, Args) \
  do { \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_args(Sc, Op, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[1] = Sc->curlet; \
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_args_direct(Sc, Op) \
  do { \
      memcpy((void *)(Sc->stack_end), (void *)Sc, 2 * sizeof(s7_pointer));	\
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_let(Sc, Op, Args, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_op(Sc, Op) \
  do { \
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_op_let(Sc, Op) \
  do { \
      Sc->stack_end[1] = Sc->curlet; \
      Sc->stack_end[3] = (s7_pointer)(Op); \
      Sc->stack_end += 4; \
  } while (0)
#endif
/* since we don't GC mark the stack past the stack_top, push_stack_no_args and friends can cause pop_stack to set
 *   sc->code and sc->args to currently free objects.
 */

#if S7_DEBUGGING
#define unstack(Sc) unstack_1(Sc, __func__, __LINE__)
static void unstack_1(s7_scheme *sc, const char *func, int32_t line)
{
  sc->stack_end -= 4;
  if (((opcode_t)sc->stack_end[3]) != OP_GC_PROTECT)
    {
      fprintf(stderr, "%s%s[%d]: popped %s?%s\n", BOLD_TEXT, func, line, op_names[(opcode_t)sc->stack_end[3]], UNBOLD_TEXT);
      fprintf(stderr, "    code: %s, args: %s\n", display(sc->code), display(sc->args));
      fprintf(stderr, "    cur_code: %s, estr: %s\n", display(current_code(sc)), display(s7_name_to_value(sc, "estr")));
      if (sc->stop_at_error) abort();
    }
}
#define unstack_with(Sc, Op) unstack_2(Sc, Op, __func__, __LINE__)
static void unstack_2(s7_scheme *sc, opcode_t op, const char *func, int32_t line)
{
  sc->stack_end -= 4;
  if (((opcode_t)sc->stack_end[3]) != op)
    {
      fprintf(stderr, "%s%s[%d]: popped %s?%s\n", BOLD_TEXT, func, line, op_names[(opcode_t)sc->stack_end[3]], UNBOLD_TEXT);
      fprintf(stderr, "    code: %s, args: %s\n", display(sc->code), display(sc->args));
      fprintf(stderr, "    cur_code: %s, estr: %s\n", display(current_code(sc)), display(s7_name_to_value(sc, "estr")));
      if (sc->stop_at_error) abort();
    }
}
#else
#define unstack(sc) sc->stack_end -= 4
#define unstack_with(sc, op) sc->stack_end -= 4
#endif

#define main_stack_op(Sc) ((opcode_t)(Sc->stack_end[-1]))
/* #define main_stack_args(Sc) (Sc->stack_end[-2]), #define main_stack_let(Sc) (Sc->stack_end[-3]), #define main_stack_code(Sc) (Sc->stack_end[-4]) */
/* beware of main_stack_code!  If a function has a tail-call, the main_stack_code that form sees
 *   if main_stack_op==op-begin1 can change from call to call -- the begin actually refers
 *   to the caller, which is dependent on where the current function was called, so we can't hard-wire
 *   any optimizations based on that sequence.
 */

static void stack_reset(s7_scheme *sc)
{
  sc->stack_end = sc->stack_start;
  push_stack_op(sc, OP_EVAL_DONE);
}

static void resize_stack(s7_scheme *sc)
{
  uint64_t loc = current_stack_top(sc);
  uint32_t new_size = sc->stack_size * 2;
  block_t *ob, *nb;

  /* how can we trap infinite recursion?  Is a warning in order here? I think I'll add 'max-stack-size */
  if (new_size > sc->max_stack_size)
    s7_error_nr(sc, make_symbol(sc, "stack-too-big"),
		set_elist_1(sc, wrap_string(sc, "stack has grown past (*s7* 'max-stack-size)", 43)));

  ob = stack_block(sc->stack);
  nb = reallocate(sc, ob, new_size * sizeof(s7_pointer));
  block_info(nb) = NULL;
  stack_block(sc->stack) = nb;
  stack_elements(sc->stack) = (s7_pointer *)block_data(nb);
  if (!stack_elements(sc->stack))
    s7_error_nr(sc, make_symbol(sc, "stack-too-big"), set_elist_1(sc, wrap_string(sc, "no room to expand stack?", 24)));
  {
    s7_pointer *orig = stack_elements(sc->stack);
    s7_int i = sc->stack_size;
    s7_int left = new_size - i - 8;
    while (i <= left)
      LOOP_8(orig[i++] = sc->nil);
    for (; i < new_size; i++)
      orig[i] = sc->nil;
  }
  vector_length(sc->stack) = new_size;
  sc->stack_size = new_size;
  sc->stack_start = stack_elements(sc->stack);
  sc->stack_end = (s7_pointer *)(sc->stack_start + loc);
  /* sc->stack_resize_trigger = (s7_pointer *)(sc->stack_start + sc->stack_size / 2); */
  sc->stack_resize_trigger = (s7_pointer *)(sc->stack_start + (new_size - STACK_RESIZE_TRIGGER));

  if (show_stack_stats(sc))
    s7_warn(sc, 128, "stack grows to %u\n", new_size);
}

#define check_stack_size(Sc) do {if (Sc->stack_end >= Sc->stack_resize_trigger) resize_stack(Sc);} while (0)

s7_pointer s7_gc_protect_via_stack(s7_scheme *sc, s7_pointer x)
{
  push_stack_no_let_no_code(sc, OP_GC_PROTECT, x);
  return(x);
}

s7_pointer s7_gc_unprotect_via_stack(s7_scheme *sc, s7_pointer x)
{
  unstack(sc);
  return(x);
}

#define stack_protected1(Sc) Sc->stack_end[-2]
#define stack_protected2(Sc) Sc->stack_end[-4]
#define stack_protected3(Sc) Sc->stack_end[-3]

static inline void gc_protect_via_stack(s7_scheme *sc, s7_pointer val)
{
  sc->stack_end[2] = val;
  sc->stack_end[3] = (s7_pointer)OP_GC_PROTECT;
  sc->stack_end += 4;
}

#define gc_protect_2_via_stack(Sc, X, Y) do {push_stack_no_let_no_code(Sc, OP_GC_PROTECT, X); stack_protected2(Sc) = Y;} while (0)
/* often X and Y are fx_calls, so push X, then set Y */


/* -------------------------------- symbols -------------------------------- */

static inline uint64_t raw_string_hash(const uint8_t *key, s7_int len)
{
  if (len <= 8)
    {
      uint64_t xs[1] = {0};
      memcpy((void *)xs, (void *)key, len);
      return(xs[0]);
    }
  else
    {
      uint64_t xs[2] = {0, 0};
      memcpy((void *)xs, (void *)key, (len > 16) ? 16 : len);  /* compiler complaint here is bogus */
      return(xs[0] + xs[1]);
    }
}

static uint8_t *alloc_symbol(s7_scheme *sc)
{
  #define SYMBOL_SIZE (3 * sizeof(s7_cell) + sizeof(block_t))
  #define ALLOC_SYMBOL_SIZE (64 * SYMBOL_SIZE)
  uint8_t *result;
  if (sc->alloc_symbol_k == ALLOC_SYMBOL_SIZE)
    {
      sc->alloc_symbol_cells = (uint8_t *)Malloc(ALLOC_SYMBOL_SIZE);
      add_saved_pointer(sc, sc->alloc_symbol_cells);
      sc->alloc_symbol_k = 0;
    }
  result = &(sc->alloc_symbol_cells[sc->alloc_symbol_k]);
  sc->alloc_symbol_k += SYMBOL_SIZE;
  return(result);
}

static s7_pointer make_permanent_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer value);
static inline s7_pointer make_symbol_with_length(s7_scheme *sc, const char *name, s7_int len); /* calls new_symbol */

static inline s7_pointer new_symbol(s7_scheme *sc, const char *name, s7_int len, uint64_t hash, uint32_t location)
{
  /* name might not be null-terminated, these are permanent symbols even in s7_gensym; g_gensym handles everything separately */
  uint8_t *base = alloc_symbol(sc);
  s7_pointer x = (s7_pointer)base;
  s7_pointer str = (s7_pointer)(base + sizeof(s7_cell));
  s7_pointer p = (s7_pointer)(base + 2 * sizeof(s7_cell));
  uint8_t *val = (uint8_t *)permalloc(sc, len + 1);
  memcpy((void *)val, (void *)name, len);
  val[len] = '\0';

  full_type(str) = T_STRING | T_IMMUTABLE | T_UNHEAP;       /* avoid debugging confusion involving set_type (also below) */
  string_length(str) = len;
  string_value(str) = (char *)val;
  string_hash(str) = hash;

  full_type(x) = T_SYMBOL | T_UNHEAP;
  symbol_set_name_cell(x, str);
  set_global_slot(x, sc->undefined);                       /* was sc->nil */
  symbol_info(x) = (block_t *)(base + 3 * sizeof(s7_cell));
  set_initial_slot(x, sc->undefined);
  symbol_set_local_slot_unchecked_and_unincremented(x, 0LL, sc->nil);
  symbol_set_tag(x, 0);
  symbol_set_tag2(x, 0);
  symbol_clear_ctr(x); /* alloc_symbol uses malloc */
  symbol_clear_type(x);

  if ((len > 1) &&                                    /* not 0, otherwise : is a keyword */
      ((name[0] == ':') || (name[len - 1] == ':')))   /* see s7test under keyword? for troubles if both colons are present */
    {
      s7_pointer slot, ksym;
      set_type_bit(x, T_IMMUTABLE | T_KEYWORD | T_GLOBAL);
      set_optimize_op(str, OP_CONSTANT);
      ksym = make_symbol_with_length(sc, (name[0] == ':') ? (char *)(name + 1) : name, len - 1);
      keyword_set_symbol(x, ksym);
      set_has_keyword(ksym);
      /* the keyword symbol needs to be permanent (not a gensym) else we have to laboriously gc-protect it */
      if ((is_gensym(ksym)) &&
	  (in_heap(ksym)))
	remove_gensym_from_heap(sc, ksym);
      slot = make_permanent_slot(sc, x, x);
      set_global_slot(x, slot);
      set_local_slot(x, slot);
    }

  full_type(p) = T_PAIR | T_IMMUTABLE | T_UNHEAP;  /* add x to the symbol table */
  set_car(p, x);
  set_cdr(p, vector_element(sc->symbol_table, location));
  vector_element(sc->symbol_table, location) = p;
  pair_set_raw_hash(p, hash);
  pair_set_raw_len(p, (uint64_t)len); /* symbol name length, so it ought to fit! */
  pair_set_raw_name(p, string_value(str));
  return(x);
}

static inline s7_pointer make_symbol_with_length(s7_scheme *sc, const char *name, s7_int len)
{
  uint64_t hash = raw_string_hash((const uint8_t *)name, len);
  uint32_t location = hash % SYMBOL_TABLE_SIZE;
  if (len <= 8)
    {
      for (s7_pointer x = vector_element(sc->symbol_table, location); is_pair(x); x = cdr(x))
	if ((hash == pair_raw_hash(x)) &&
	    ((uint64_t)len == pair_raw_len(x)))
	  return(car(x));
    }
  else
    for (s7_pointer x = vector_element(sc->symbol_table, location); is_pair(x); x = cdr(x))
      if ((hash == pair_raw_hash(x)) &&
	  ((uint64_t)len == pair_raw_len(x)) &&
	  (strings_are_equal_with_length(name, pair_raw_name(x), len))) /* length here because name might not be null-terminated */
	return(car(x));
  return(new_symbol(sc, name, len, hash, location));
}

static s7_pointer make_symbol(s7_scheme *sc, const char *name) {return(make_symbol_with_length(sc, name, safe_strlen(name)));}

s7_pointer s7_make_symbol(s7_scheme *sc, const char *name) {return((name) ? make_symbol_with_length(sc, name, safe_strlen(name)) : sc->F);}

static s7_pointer symbol_table_find_by_name(s7_scheme *sc, const char *name, uint64_t hash, uint32_t location, s7_int len)
{
  for (s7_pointer x = vector_element(sc->symbol_table, location); is_not_null(x); x = cdr(x))
    if ((hash == pair_raw_hash(x)) &&
	(strings_are_equal_with_length(name, pair_raw_name(x), len)))
      return(car(x));
  return(sc->nil);
}

s7_pointer s7_symbol_table_find_name(s7_scheme *sc, const char *name)
{
  s7_int len = safe_strlen(name);
  uint64_t hash = raw_string_hash((const uint8_t *)name, len);
  s7_pointer result = symbol_table_find_by_name(sc, name, hash, hash % SYMBOL_TABLE_SIZE, len);
  return((is_null(result)) ? NULL : result);
}


/* -------------------------------- symbol-table -------------------------------- */
static inline s7_pointer make_simple_vector(s7_scheme *sc, s7_int len);

static s7_pointer g_symbol_table(s7_scheme *sc, s7_pointer unused_args)
{
  #define H_symbol_table "(symbol-table) returns a vector containing the current symbol-table symbols"
  #define Q_symbol_table s7_make_signature(sc, 1, sc->is_vector_symbol)

  s7_pointer *entries = vector_elements(sc->symbol_table);
  int32_t syms = 0;

  /* this can't be optimized by returning the actual symbol-table (a vector of lists), because
   *    gensyms can cause the table's lists and symbols to change at any time.  This wreaks havoc
   *    on traversals like for-each.  So, symbol-table returns a snap-shot of the table contents
   *    at the time it is called.
   *    (define (for-each-symbol func num) (for-each (lambda (sym) (if (> num 0) (for-each-symbol func (- num 1)) (func sym))) (symbol-table)))
   *    (for-each-symbol (lambda (sym) (gensym) 1))
   */
  for (int32_t i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (s7_pointer x = entries[i]; is_not_null(x); x = cdr(x))
      syms++;
  sc->w = make_simple_vector(sc, syms);
  {
    s7_pointer *els = vector_elements(sc->w);
    for (int32_t i = 0, j = 0; i < SYMBOL_TABLE_SIZE; i++)
      for (s7_pointer x = entries[i]; is_not_null(x); x = cdr(x))
	els[j++] = car(x);
  }
  {
    s7_pointer lst = sc->w;
    sc->w = sc->nil;
    return(lst);
  }
}

bool s7_for_each_symbol_name(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, void *data), void *data)
{
  /* this includes the special constants #<unspecified> and so on for simplicity -- are there any others? */
  for (int32_t i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (s7_pointer x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      if (symbol_func(symbol_name(car(x)), data))
	return(true);

  return((symbol_func("#t", data))             ||
	 (symbol_func("#f", data))             ||
	 (symbol_func("#<unspecified>", data)) ||
	 (symbol_func("#<undefined>", data))   ||
	 (symbol_func("#<eof>", data))         ||
	 (symbol_func("#true", data))          ||
	 (symbol_func("#false", data)));
}

bool s7_for_each_symbol(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, void *data), void *data)
{
  for (int32_t i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (s7_pointer x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      if (symbol_func(symbol_name(car(x)), data))
	return(true);
  return(false);
}

/* -------------------------------- gensym -------------------------------- */
static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym)
{
  /* sym is a free cell at this point (we're called after the GC), but the name_cell is still intact */
  s7_pointer name = symbol_name_cell(sym);
  uint32_t location = string_hash(name) % SYMBOL_TABLE_SIZE;
  s7_pointer x = vector_element(sc->symbol_table, location);
  if (car(x) == sym)
    vector_element(sc->symbol_table, location) = cdr(x);
  else
    for (s7_pointer y = x, z = cdr(x); is_pair(z); y = z, z = cdr(z))
      if (car(z) == sym)
	{
	  set_cdr(y, cdr(z));
	  return;
	}
}

s7_pointer s7_gensym(s7_scheme *sc, const char *prefix)
{
  s7_int len = safe_strlen(prefix) + 32;
  block_t *b = mallocate(sc, len);
  char *name = (char *)block_data(b);
  /* there's no point in heroic efforts here to avoid name collisions -- the user can screw up no matter what we do */
  name[0] = '\0';
  {
    s7_int slen = catstrs(name, len, "{", (prefix) ? prefix : "", "}-", pos_int_to_str_direct(sc, sc->gensym_counter++), (char *)NULL);
    uint64_t hash = raw_string_hash((const uint8_t *)name, slen);
    int32_t location = hash % SYMBOL_TABLE_SIZE;
    s7_pointer x = new_symbol(sc, name, slen, hash, location);  /* not T_GENSYM -- might be called from outside */
    liberate(sc, b);
    return(x);
  }
}

static bool is_gensym_b_p(s7_pointer g) {return((is_symbol(g)) && (is_gensym(g)));}

static s7_pointer g_is_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_is_gensym "(gensym? sym) returns #t if sym is a gensym"
  #define Q_is_gensym sc->pl_bt
  check_boolean_method(sc, is_gensym_b_p, sc->is_gensym_symbol, args);
}

static s7_pointer g_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_gensym "(gensym (prefix \"gensym\")) returns a new, unused symbol"
  #define Q_gensym s7_make_signature(sc, 2, sc->is_gensym_symbol, sc->is_string_symbol)

  const char *prefix;
  char *name, *p, *base;
  s7_int len, plen, nlen;
  uint32_t location;
  uint64_t hash;
  s7_pointer x, str, stc;
  block_t *b, *ib;

  /* get symbol name */
  if (is_not_null(args))
    {
      s7_pointer gname = car(args);
      if (!is_string(gname))
	return(method_or_bust_one_arg(sc, gname, sc->gensym_symbol, args, T_STRING));
      prefix = string_value(gname);
      plen = string_length(gname); /* was safe_strlen(prefix): were we stopping at #\null deliberately? */
    }
  else
    {
      prefix = "gensym";
      plen = 6;
    }
  len = plen + 32; /* why 32 -- we need room for the gensym_counter integer, but (length "9223372036854775807") = 19 */

  b = mallocate(sc, len + sizeof(block_t) + 2 * sizeof(s7_cell));
  base = (char *)block_data(b);
  str = (s7_cell *)base;
  stc = (s7_cell *)(base + sizeof(s7_cell));
  ib = (block_t *)(base + 2 * sizeof(s7_cell));
  name = (char *)(base + sizeof(block_t) + 2 * sizeof(s7_cell));

  name[0] = '{';
  memcpy((void *)(name + 1), prefix, plen); /* memcpy is ok with plen==0, I think */
  name[plen + 1] = '}';
  name[plen + 2] = '-'; /* {gensym}-nnn */

  p = pos_int_to_str(sc, sc->gensym_counter++, &len, '\0');
  memcpy((void *)(name + plen + 3), (void *)p, len);
  nlen = len + plen + 2;
  hash = raw_string_hash((const uint8_t *)name, nlen);
  location = hash % SYMBOL_TABLE_SIZE;

  if ((WITH_WARNINGS) && (!is_null(symbol_table_find_by_name(sc, name, hash, location, nlen))))
    s7_warn(sc, nlen + 32, "%s is already in use!", name);

  /* make-string for symbol name */
  if (S7_DEBUGGING) full_type(str) = 0; /* here and below, this is needed to avoid set_type check errors (mallocate above) */
  set_full_type(str, T_STRING | T_IMMUTABLE | T_UNHEAP);
  string_length(str) = nlen;
  string_value(str) = name;
  string_hash(str) = hash;

  /* allocate the symbol in the heap so GC'd when inaccessible */
  new_cell(sc, x, T_SYMBOL | T_GENSYM);
  symbol_set_name_cell(x, str);
  symbol_info(x) = ib;
  set_global_slot(x, sc->undefined);  /* set_initial_slot(x, sc->undefined); */
  symbol_set_local_slot_unchecked(x, 0LL, sc->nil);
  symbol_clear_ctr(x);
  symbol_set_tag(x, 0);
  symbol_set_tag2(x, 0);
  symbol_clear_type(x);
  gensym_block(x) = b;

  /* place new symbol in symbol-table */
  if (S7_DEBUGGING) full_type(stc) = 0;
  set_full_type(stc, T_PAIR | T_IMMUTABLE | T_UNHEAP);
  set_car(stc, x);
  set_cdr(stc, vector_element(sc->symbol_table, location));
  vector_element(sc->symbol_table, location) = stc;
  pair_set_raw_hash(stc, hash);
  pair_set_raw_len(stc, (uint64_t)string_length(str));
  pair_set_raw_name(stc, string_value(str));

  add_gensym(sc, x);
  return(x);
}


/* -------------------------------- syntax? -------------------------------- */
bool s7_is_syntax(s7_pointer p) {return(is_syntax(p));}

static s7_pointer g_is_syntax(s7_scheme *sc, s7_pointer args)
{
  #define H_is_syntax "(syntax? obj) returns #t if obj is a syntactic value (e.g. lambda)"
  #define Q_is_syntax sc->pl_bt
  check_boolean_method(sc, is_syntax, sc->is_syntax_symbol, args);
}


/* -------------------------------- symbol? -------------------------------- */
bool s7_is_symbol(s7_pointer p) {return(is_symbol(p));}

static s7_pointer g_is_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_is_symbol "(symbol? obj) returns #t if obj is a symbol"
  #define Q_is_symbol sc->pl_bt
  check_boolean_method(sc, is_symbol, sc->is_symbol_symbol, args);
}

const char *s7_symbol_name(s7_pointer p) {return(symbol_name(p));}

s7_pointer s7_name_to_value(s7_scheme *sc, const char *name) {return(s7_symbol_value(sc, make_symbol(sc, name)));}
/* should this also handle non-symbols such as "+nan.0"? */


/* -------------------------------- symbol->string -------------------------------- */
static Inline s7_pointer inline_make_string_with_length(s7_scheme *sc, const char *str, s7_int len)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING | T_SAFE_PROCEDURE);
  string_block(x) = mallocate(sc, len + 1);
  string_value(x) = (char *)block_data(string_block(x));
  memcpy((void *)string_value(x), (void *)str, len);
  string_value(x)[len] = 0;
  string_length(x) = len;
  string_hash(x) = 0;
  add_string(sc, x);
  return(x);
}

static s7_pointer make_string_with_length(s7_scheme *sc, const char *str, s7_int len)
{
  return(inline_make_string_with_length(sc, str, len)); /* packaged to avoid inlining everywhere */
}

static s7_pointer g_symbol_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_string "(symbol->string sym) returns the symbol sym converted to a string"
  #define Q_symbol_to_string s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_symbol_symbol)

  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->symbol_to_string_symbol, args, T_SYMBOL));
  /* s7_make_string uses strlen which stops at an embedded null */
  return(inline_make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));    /* return a copy */
}

static s7_pointer g_symbol_to_string_uncopied(s7_scheme *sc, s7_pointer args)
{
  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->symbol_to_string_symbol, args, T_SYMBOL));
  if (is_gensym(sym))
    return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));    /* return a copy of gensym name (which will be freed) */
  return(symbol_name_cell(sym));
}

static s7_pointer symbol_to_string_p_p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->symbol_to_string_symbol, set_plist_1(sc, sym), T_SYMBOL));
  return(inline_make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));
}

static s7_pointer symbol_to_string_uncopied_p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->symbol_to_string_symbol, set_plist_1(sc, sym), T_SYMBOL));
  if (is_gensym(sym))
    return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));
  return(symbol_name_cell(sym));
}


/* -------------------------------- string->symbol -------------------------------- */
static inline s7_pointer g_string_to_symbol_1(s7_scheme *sc, s7_pointer str, s7_pointer caller)
{
  if (!is_string(str))
    return(method_or_bust_p(sc, str, caller, T_STRING));
  if (string_length(str) <= 0)
    simple_wrong_type_argument_with_type(sc, caller, str, wrap_string(sc, "a non-null string", 17));
  return(make_symbol_with_length(sc, string_value(str), string_length(str)));
}

static s7_pointer g_string_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_symbol "(string->symbol str) returns the string str converted to a symbol"
  #define Q_string_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_string_symbol)
  return(g_string_to_symbol_1(sc, car(args), sc->string_to_symbol_symbol));
}

static s7_pointer string_to_symbol_p_p(s7_scheme *sc, s7_pointer p) {return(g_string_to_symbol_1(sc, p, sc->string_to_symbol_symbol));}


/* -------------------------------- symbol -------------------------------- */
static s7_pointer g_string_append_1(s7_scheme *sc, s7_pointer args, s7_pointer caller);

static s7_pointer g_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol "(symbol str ...) returns its string arguments concatenated and converted to a symbol"
  #define Q_symbol s7_make_circular_signature(sc, 1, 2, sc->is_symbol_symbol, sc->is_string_symbol)

  s7_int len = 0, cur_len;
  s7_pointer p, sym;
  block_t *b;
  char *name;

  for (p = args; is_pair(p); p = cdr(p))
    if (is_string(car(p)))
      len += string_length(car(p));
    else break;

  if (is_pair(p))
    {
      if (is_null(cdr(args)))
	return(g_string_to_symbol_1(sc, car(args), sc->symbol_symbol));
      return(g_string_to_symbol_1(sc, g_string_append_1(sc, args, sc->symbol_symbol), sc->symbol_symbol));
    }
  if (len == 0)
    simple_wrong_type_argument_with_type(sc, sc->symbol_symbol, car(args), wrap_string(sc, "a non-null string", 17));

  b = mallocate(sc, len + 1);
  name = (char *)block_data(b);
  /* can't use catstrs_direct here because it stops at embedded null */
  for (cur_len = 0, p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer str = car(p);
      if (string_length(str) > 0)
	{
	  memcpy((void *)(name + cur_len), (void *)string_value(str), string_length(str));
	  cur_len += string_length(str);
	}}
  name[len] = '\0';
  sym = make_symbol_with_length(sc, name, len);
  liberate(sc, b);
  return(sym);
}

static s7_pointer symbol_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  char buf[256];
  s7_int len;
  if ((!is_string(p1)) || (!is_string(p2))) return(g_symbol(sc, set_plist_2(sc, p1, p2)));
  len = string_length(p1) + string_length(p2);
  if ((len == 0) || (len >= 256)) return(g_symbol(sc, set_plist_2(sc, p1, p2)));
  memcpy((void *)buf, (void *)string_value(p1), string_length(p1));
  memcpy((void *)(buf + string_length(p1)), (void *)string_value(p2), string_length(p2));
  return(make_symbol_with_length(sc, buf, len));
}


/* -------- symbol sets -------- */
static inline s7_pointer add_symbol_to_list(s7_scheme *sc, s7_pointer sym)
{
  symbol_set_tag(sym, sc->syms_tag);
  symbol_set_tag2(sym, sc->syms_tag2);
  return(sym);
}

static inline void clear_symbol_list(s7_scheme *sc)
{
  sc->syms_tag++;
  if (sc->syms_tag == 0)
    {
      sc->syms_tag = 1; /* we're assuming (in let_equal) that this tag is not 0 */
      sc->syms_tag2++;
    }
}

#define symbol_is_in_list(Sc, Sym) ((symbol_tag(Sym) == Sc->syms_tag) && (symbol_tag2(Sym) == Sc->syms_tag2))


/* -------------------------------- lets/slots -------------------------------- */

static Inline s7_pointer make_let(s7_scheme *sc, s7_pointer old_let)
{
  s7_pointer x;
  new_cell(sc, x, T_LET | T_SAFE_PROCEDURE);
  let_set_id(x, ++sc->let_number);
  let_set_slots(x, slot_end(sc));
  let_set_outlet(x, old_let);
  return(x);
}

static inline s7_pointer make_let_slowly(s7_scheme *sc, s7_pointer old_let)
{
  s7_pointer x;
  new_cell(sc, x, T_LET | T_SAFE_PROCEDURE);
  let_set_id(x, ++sc->let_number);
  let_set_slots(x, slot_end(sc));
  let_set_outlet(x, old_let);
  return(x);
}

static Inline s7_pointer make_let_with_slot(s7_scheme *sc, s7_pointer old_let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer new_let, slot;
  sc->value = value;
  new_cell(sc, new_let, T_LET | T_SAFE_PROCEDURE);
  let_set_id(new_let, ++sc->let_number);
  let_set_outlet(new_let, old_let);
  new_cell_no_check(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  symbol_set_local_slot(symbol, sc->let_number, slot);
  slot_set_next(slot, slot_end(sc));
  let_set_slots(new_let, slot);
  return(new_let);
}

static Inline s7_pointer make_let_with_two_slots(s7_scheme *sc, s7_pointer old_let,
						 s7_pointer symbol1, s7_pointer value1, s7_pointer symbol2, s7_pointer value2)
{
  /* we leave value1/value2 computation order to the C compiler here -- in the old macro, it was explicitly value1 then value2
   *   this means any let in old scheme code that actually depends on the order may break -- it should be let*.
   */
  s7_pointer new_let, slot1, slot2;
  new_cell(sc, new_let, T_LET | T_SAFE_PROCEDURE);
  let_set_id(new_let, ++sc->let_number);
  let_set_outlet(new_let, old_let);

  new_cell_no_check(sc, slot1, T_SLOT);
  slot_set_symbol_and_value(slot1, symbol1, value1);
  symbol_set_local_slot(symbol1, sc->let_number, slot1);
  let_set_slots(new_let, slot1);

  new_cell_no_check(sc, slot2, T_SLOT);
  slot_set_symbol_and_value(slot2, symbol2, value2);
  symbol_set_local_slot(symbol2, sc->let_number, slot2);
  slot_set_next(slot2, slot_end(sc));
  slot_set_next(slot1, slot2);
  return(new_let);
}

/* in all these functions, symbol_set_local_slot should follow slot_set_value so that we can evaluate the slot's value in its old state */
static inline void add_slot_unchecked(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value, uint64_t id)
{
  s7_pointer slot;
  new_cell_no_check(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  set_local(symbol);
  symbol_set_local_slot(symbol, id, slot);
}

static void add_slot_unchecked_no_local(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_no_check(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  set_local(symbol);
}

#define add_slot(Sc, Let, Symbol, Value) add_slot_unchecked(Sc, Let, Symbol, Value, let_id(Let))

static inline s7_pointer add_slot_checked(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  symbol_set_local_slot(symbol, let_id(let), slot);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  return(slot);
}

static inline s7_pointer add_slot_checked_with_id(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  set_local(symbol);
  if (let_id(let) >= symbol_id(symbol))
    symbol_set_local_slot(symbol, let_id(let), slot);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  return(slot);
}

static s7_pointer add_slot_unchecked_with_id(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_no_check(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  set_local(symbol);
  if (let_id(let) >= symbol_id(symbol))
    symbol_set_local_slot(symbol, let_id(let), slot);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  return(slot);
}

static Inline s7_pointer add_slot_at_end(s7_scheme *sc, uint64_t id, s7_pointer last_slot, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_no_check(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, slot_end(sc));
  symbol_set_local_slot(symbol, id, slot);
  slot_set_next(last_slot, slot);
  return(slot);
}

static s7_pointer add_slot_at_end_no_local(s7_scheme *sc, s7_pointer last_slot, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_no_check(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, slot_end(sc));
  slot_set_next(last_slot, slot);
  return(slot);
}

static inline void make_let_with_three_slots(s7_scheme *sc, s7_pointer func, s7_pointer val1, s7_pointer val2, s7_pointer val3)
{
  s7_pointer last_slot, cargs = closure_args(func);
  sc->curlet = make_let_with_two_slots(sc, closure_let(func), car(cargs), val1, cadr(cargs), val2);
  last_slot = next_slot(let_slots(sc->curlet));
  add_slot_at_end(sc, let_id(sc->curlet), last_slot, caddr(cargs), val3);
}

static inline void make_let_with_four_slots(s7_scheme *sc, s7_pointer func, s7_pointer val1, s7_pointer val2, s7_pointer val3, s7_pointer val4)
{
  s7_pointer last_slot, cargs = closure_args(func);
  sc->curlet = make_let_with_two_slots(sc, closure_let(func), car(cargs), val1, cadr(cargs), val2);
  cargs = cddr(cargs);
  last_slot = next_slot(let_slots(sc->curlet));
  last_slot = add_slot_at_end(sc, let_id(sc->curlet), last_slot, car(cargs), val3);
  add_slot_at_end(sc, let_id(sc->curlet), last_slot, cadr(cargs), val4);
}

static s7_pointer reuse_as_let(s7_scheme *sc, s7_pointer let, s7_pointer next_let)
{
  /* we're reusing let here as a let -- it was probably a pair */
#if S7_DEBUGGING
  let->debugger_bits = 0;
  if (!in_heap(let)) fprintf(stderr, "reusing an unheaped let?\n");
#endif
  set_full_type(let, T_LET | T_SAFE_PROCEDURE);
  let_set_slots(let, slot_end(sc));
  let_set_outlet(let, next_let);
  let_set_id(let, ++sc->let_number);
  return(let);
}

static s7_pointer reuse_as_slot(s7_scheme *sc, s7_pointer slot, s7_pointer symbol, s7_pointer value)
{
#if S7_DEBUGGING
  slot->debugger_bits = 0;
  if (!in_heap(slot)) fprintf(stderr, "reusing a permanent cell?\n");
  if (is_multiple_value(value))
    {
      fprintf(stderr, "%s%s[%d]: multiple-value %s %s%s\n", BOLD_TEXT, __func__, __LINE__, display(value), display(sc->code), UNBOLD_TEXT);
      if (sc->stop_at_error) abort();
    }
#endif
  set_full_type(slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  return(slot);
}

#define update_slot(Slot, Val, Id) do {s7_pointer sym; slot_set_value(Slot, Val); sym = slot_symbol(Slot); symbol_set_local_slot_unincremented(sym, Id, Slot);} while (0)

static s7_pointer update_let_with_slot(s7_scheme *sc, s7_pointer let, s7_pointer val)
{
  s7_pointer slot = let_slots(let);
  uint64_t id = ++sc->let_number;
  let_set_id(let, id);
  update_slot(slot, val, id);
  return(let);
}

static s7_pointer update_let_with_two_slots(s7_scheme *sc, s7_pointer let, s7_pointer val1, s7_pointer val2)
{
  s7_pointer slot = let_slots(let);
  uint64_t id = ++sc->let_number;
  let_set_id(let, id);
  update_slot(slot, val1, id);
  slot = next_slot(slot);
  update_slot(slot, val2, id);
  return(let);
}

static s7_pointer update_let_with_three_slots(s7_scheme *sc, s7_pointer let, s7_pointer val1, s7_pointer val2, s7_pointer val3)
{
  s7_pointer slot = let_slots(let);
  uint64_t id = ++sc->let_number;
  let_set_id(let, id);
  update_slot(slot, val1, id);
  slot = next_slot(slot);
  update_slot(slot, val2, id);
  slot = next_slot(slot);
  update_slot(slot, val3, id);
  return(let);
}

static s7_pointer update_let_with_four_slots(s7_scheme *sc, s7_pointer let, s7_pointer val1, s7_pointer val2, s7_pointer val3, s7_pointer val4)
{
  s7_pointer slot = let_slots(let);
  uint64_t id = ++sc->let_number;
  let_set_id(let, id);
  update_slot(slot, val1, id);
  slot = next_slot(slot);
  update_slot(slot, val2, id);
  slot = next_slot(slot);
  update_slot(slot, val3, id);
  slot = next_slot(slot);
  update_slot(slot, val4, id);
  return(let);
}

static s7_pointer make_permanent_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot = alloc_pointer(sc);
  set_full_type(slot, T_SLOT | T_UNHEAP);
  slot_set_symbol_and_value(slot, symbol, value);
  return(slot);
}

static s7_pointer make_permanent_let(s7_scheme *sc, s7_pointer vars)
{
  s7_pointer slot, let = alloc_pointer(sc);
  set_full_type(let, T_LET | T_SAFE_PROCEDURE | T_UNHEAP);
  let_set_id(let, ++sc->let_number);
  let_set_outlet(let, sc->curlet);
  slot = make_permanent_slot(sc, caar(vars), sc->F);
  add_permanent_let_or_slot(sc, slot);
  symbol_set_local_slot(caar(vars), sc->let_number, slot);
  let_set_slots(let, slot);
  for (s7_pointer var = cdr(vars); is_pair(var); var = cdr(var))
    {
      s7_pointer last_slot = slot;
      slot = make_permanent_slot(sc, caar(var), sc->F);
      add_permanent_let_or_slot(sc, slot);
      symbol_set_local_slot(caar(var), sc->let_number, slot);
      slot_set_next(last_slot, slot);
    }
  slot_set_next(slot, slot_end(sc));
  add_permanent_let_or_slot(sc, let); /* need to mark outlet and maybe slot values */
  return(let);
}

static s7_pointer find_let(s7_scheme *sc, s7_pointer obj)
{
  if (is_let(obj)) return(obj);
  switch (type(obj))
    {
    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
    case T_CLOSURE: case T_CLOSURE_STAR:
      return(closure_let(obj));

    case T_C_OBJECT:
      return(c_object_let(obj));

    case T_C_POINTER:
      if ((is_let(c_pointer_info(obj))) &&
	  (c_pointer_info(obj) != sc->rootlet))
	return(c_pointer_info(obj));
    }
  return(sc->nil);
}

static s7_pointer call_setter(s7_scheme *sc, s7_pointer slot, s7_pointer old_value);

static inline s7_pointer checked_slot_set_value(s7_scheme *sc, s7_pointer y, s7_pointer value)
{
  if (slot_has_setter(y))
    slot_set_value(y, call_setter(sc, y, value));
  else
    {
      if (is_immutable_slot(y))
	immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->let_set_symbol, slot_symbol(y)));
      slot_set_value(y, value);
    }
  return(slot_value(y));
}

static s7_pointer let_fill(s7_scheme *sc, s7_pointer args)
{
  s7_pointer e = car(args), val;
  if (e == sc->rootlet)
    out_of_range(sc, sc->fill_symbol, int_one, e, wrap_string(sc, "can't fill! rootlet", 19));
  if (e == sc->s7_let)
    out_of_range(sc, sc->fill_symbol, int_one, e, wrap_string(sc, "can't fill! *s7*", 16));
  if (e == sc->owlet)                 /* (owlet) copies sc->owlet, so this probably can't happen */
    out_of_range(sc, sc->fill_symbol, int_one, e, wrap_string(sc, "can't fill! owlet", 17));
  if (is_funclet(e))
    out_of_range(sc, sc->fill_symbol, int_one, e, wrap_string(sc, "can't fill! a funclet", 21));
  val = cadr(args);
  for (s7_pointer p = let_slots(e); tis_slot(p); p = next_slot(p))
    checked_slot_set_value(sc, p, val);
  return(val);
}

static s7_pointer find_method(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  s7_pointer slot;
  if (symbol_id(symbol) == 0) /* this means the symbol has never been used locally, so how can it be a method? */
    return(sc->undefined);
  slot = lookup_slot_from(symbol, let);
  if (slot != global_slot(symbol))
    return(slot_value(slot));
  return(sc->undefined);
}

static s7_pointer find_method_with_let(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  return(find_method(sc, find_let(sc, let), symbol));
}

static s7_int s7_let_length(void);

static s7_int let_length(s7_scheme *sc, s7_pointer e)
{
  /* used by length, applicable_length, copy, and some length optimizations */
  s7_int i;
  s7_pointer p;

  if (e == sc->rootlet)
    return(sc->rootlet_entries);
  if (e == sc->s7_let)
    return(s7_let_length());
  if (has_active_methods(sc, e))
    {
      s7_pointer length_func = find_method(sc, e, sc->length_symbol);
      if (length_func != sc->undefined)
	{
	  p = s7_apply_function(sc, length_func, set_plist_1(sc, e));
	  return((s7_is_integer(p)) ? s7_integer(p) : -1); /* ?? */
	}}
  for (i = 0, p = let_slots(e); tis_slot(p); i++, p = next_slot(p));
  return(i);
}


static void slot_set_setter(s7_pointer p, s7_pointer val)
{
  if ((type(val) == T_C_FUNCTION) &&
      (c_function_has_bool_setter(val)))
    slot_set_setter_1(p, c_function_bool_setter(val));
  else slot_set_setter_1(p, val);
}

static void slot_set_value_with_hook_1(s7_scheme *sc, s7_pointer slot, s7_pointer value)
{
  /* (set! (hook-functions *rootlet-redefinition-hook*) (list (lambda (hook) (format *stderr* "~A ~A~%" (hook 'name) (hook 'value))))) */
  s7_pointer symbol = slot_symbol(slot);
  if ((global_slot(symbol) == slot) &&
      (value != slot_value(slot)))
    s7_call(sc, sc->rootlet_redefinition_hook, set_plist_2(sc, symbol, value));
  slot_set_value(slot, value);
}

static void remove_function_from_heap(s7_scheme *sc, s7_pointer value); /* calls remove_let_from_heap */

static void remove_let_from_heap(s7_scheme *sc, s7_pointer lt)
{
  for (s7_pointer p = let_slots(lt); tis_slot(p); p = next_slot(p))
    {
      s7_pointer val = slot_value(p);
      if ((has_closure_let(val)) &&
	  (in_heap(closure_args(val))))
	remove_function_from_heap(sc, val);
    }
  let_set_removed(lt);
}

static void add_slot_to_rootlet(s7_scheme *sc, s7_pointer slot)
{
  s7_pointer ge = sc->rootlet;
  rootlet_element(ge, sc->rootlet_entries++) = slot;
  set_in_rootlet(slot);
  if (sc->rootlet_entries >= vector_length(ge))
    {
      s7_int len;
      block_t *ob, *nb;
      vector_length(ge) *= 2;
      len = vector_length(ge);
      ob = rootlet_block(ge);
      nb = reallocate(sc, ob, len * sizeof(s7_pointer));
      block_info(nb) = NULL;
      rootlet_block(ge) = nb;
      rootlet_elements(ge) = (s7_pointer *)block_data(nb);
      for (s7_int i = sc->rootlet_entries; i < len; i++) rootlet_element(ge, i) = sc->nil;
    }
}

static void remove_function_from_heap(s7_scheme *sc, s7_pointer value)
{
  s7_pointer lt;
  remove_from_heap(sc, closure_args(value));
  remove_from_heap(sc, closure_body(value));
  /* remove closure if it's local to current func (meaning (define f (let ...) (lambda ...)) removes the enclosing let) */
  lt = closure_let(value); /* closure_let and all its outlets can't be rootlet */
  if ((is_let(lt)) && (!let_removed(lt)) && (lt != sc->shadow_rootlet))
    {
      lt = let_outlet(lt);
      if ((is_let(lt)) && (!let_removed(lt)) && (lt != sc->shadow_rootlet))
	{
	  remove_let_from_heap(sc, lt);
	  lt = let_outlet(lt);
	  if ((is_let(lt)) && (!let_removed(lt)) && (lt != sc->shadow_rootlet))
	    remove_let_from_heap(sc, lt);
	}}
}

s7_pointer s7_make_slot(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if ((!is_let(let)) ||
      (let == sc->rootlet))
    {
      s7_pointer slot;
      if (is_immutable(sc->rootlet))
	immutable_object_error(sc, set_elist_2(sc, wrap_string(sc, "can't define '~S; rootlet is immutable", 38), symbol));
      if ((sc->safety <= NO_SAFETY) &&
	  (has_closure_let(value)))
	remove_function_from_heap(sc, value);

      /* first look for existing slot -- this is not always checked before calling s7_make_slot */
      if (is_slot(global_slot(symbol)))
	{
	  slot = global_slot(symbol);
	  symbol_increment_ctr(symbol);
	  slot_set_value_with_hook(slot, value);
	  return(slot);
	}

      slot = make_permanent_slot(sc, symbol, value);
      add_slot_to_rootlet(sc, slot);
      set_global_slot(symbol, slot);
      if (symbol_id(symbol) == 0)         /* never defined locally? */
	{
	  if ((!is_gensym(symbol)) &&
	      (initial_slot(symbol) == sc->undefined) &&
	      (!in_heap(value)) &&        /* else initial_slot value can be GC'd if symbol set! (initial != global, initial unprotected) */
	      ((!sc->unlet) ||            /* init_unlet creates sc->unlet (includes syntax), after that initial_slot is for c_functions?? */
	       (is_c_function(value))))   /* || (is_syntax(value)) -- we need 'else as a special case? */
	    set_initial_slot(symbol, make_permanent_slot(sc, symbol, value));
	  set_local_slot(symbol, slot);
	  set_global(symbol);
	}
      symbol_increment_ctr(symbol);
      if (is_gensym(symbol))
	remove_gensym_from_heap(sc, symbol);
      return(slot);
    }
  return(add_slot_checked_with_id(sc, let, symbol, value));
  /* there are about as many lets as local variables -- this strikes me as surprising, but it holds up across a lot of code */
}

static s7_pointer make_slot(s7_scheme *sc, s7_pointer variable, s7_pointer value)
{
  s7_pointer y;
  new_cell(sc, y, T_SLOT);
  slot_set_symbol_and_value(y, variable, value);
  return(y);
}


/* -------------------------------- let? -------------------------------- */
bool s7_is_let(s7_pointer e) {return(is_let(e));}

static s7_pointer g_is_let(s7_scheme *sc, s7_pointer args)
{
  #define H_is_let "(let? obj) returns #t if obj is a let."
  #define Q_is_let sc->pl_bt
  check_boolean_method(sc, is_let, sc->is_let_symbol, args);
}


/* -------------------------------- funclet? -------------------------------- */
static s7_pointer g_is_funclet(s7_scheme *sc, s7_pointer args)
{
  #define H_is_funclet "(funclet? obj) returns #t if obj is a funclet (a function's environment)."
  #define Q_is_funclet sc->pl_bt

  s7_pointer lt = car(args);
  if ((is_let(lt)) && ((is_funclet(lt)) || (is_maclet(lt))))
    return(sc->T);
  if (!has_active_methods(sc, lt))
    return(sc->F);
  return(apply_boolean_method(sc, lt, sc->is_funclet_symbol));
}


/* -------------------------------- unlet -------------------------------- */
static s7_pointer default_vector_setter(s7_scheme *sc, s7_pointer vec, s7_int loc, s7_pointer val);
static s7_pointer default_vector_getter(s7_scheme *sc, s7_pointer vec, s7_int loc);

#define UNLET_ENTRIES 512 /* 397 if not --disable-deprecated etc */

static void init_unlet(s7_scheme *sc)
{
  int32_t k = 0;
  s7_pointer *inits, *els;
  block_t *block = mallocate(sc, UNLET_ENTRIES * sizeof(s7_pointer));

  sc->unlet = (s7_pointer)Calloc(1, sizeof(s7_cell));
  set_full_type(sc->unlet, T_VECTOR | T_UNHEAP);
  vector_length(sc->unlet) = UNLET_ENTRIES;
  vector_block(sc->unlet) = block;
  vector_elements(sc->unlet) = (s7_pointer *)block_data(block);
  vector_set_dimension_info(sc->unlet, NULL);
  vector_getter(sc->unlet) = default_vector_getter;
  vector_setter(sc->unlet) = default_vector_setter;
  inits = vector_elements(sc->unlet);
  s7_vector_fill(sc, sc->unlet, sc->nil);
  els = vector_elements(sc->symbol_table);

  inits[k++] = initial_slot(sc->else_symbol);
  for (int32_t i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (s7_pointer x = els[i]; is_pair(x); x = cdr(x))
      {
	s7_pointer sym = car(x);
	if ((!is_gensym(sym)) && (is_slot(initial_slot(sym))))
	  {
	    s7_pointer val = initial_value(sym);
	    if ((is_c_function(val)) || (is_syntax(val)))  /* we assume the initial_slot value needs no GC protection */
	      inits[k++] = initial_slot(sym);
	    /* non-c_functions that are not set! (and therefore initial_slot GC) protected by default:
	     *    make-hook hook-functions
	     * if these initial_slot values are added to unlet, they need explicit GC protection.
	     */
	    if ((S7_DEBUGGING) && (k >= UNLET_ENTRIES)) fprintf(stderr, "unlet overflow\n");
	  }}
}

static s7_pointer g_unlet(s7_scheme *sc, s7_pointer unused_args)
{
  /* add sc->unlet bindings to the current environment */
  #define H_unlet "(unlet) returns a let that establishes the original bindings of all the predefined functions"
  #define Q_unlet s7_make_signature(sc, 1, sc->is_let_symbol)

  /* slightly confusing:
   *    ((unlet) 'abs) -> #<undefined>
   *    (defined? 'abs (unlet)) -> #t
   * this is because unlet sets up a local environment of unshadowed symbols, and s7_let_ref only looks at the local env chain
   *   (that is, if env is not the global env, then the global env is not searched).
   */
  s7_pointer *inits = vector_elements(sc->unlet);
  s7_pointer x;

  sc->w = make_let_slowly(sc, sc->curlet);
  for (int32_t i = 0; (i < UNLET_ENTRIES) && (is_slot(inits[i])); i++)
    {
      s7_pointer sym = slot_symbol(inits[i]);
      x = slot_value(inits[i]);
      if ((x != global_value(sym)) ||  /* it has been changed globally */
	  ((!is_global(sym)) &&        /* it might be shadowed locally */
	   (s7_symbol_local_value(sc, sym, sc->curlet) != global_value(sym))))
	add_slot_checked_with_id(sc, sc->w, sym, x);
    }
  /* if (set! + -) then + needs to be overridden, but the local bit isn't set, so we have to check the actual values in the non-local case.
   *   (define (f x) (with-let (unlet) (+ x 1)))
   */
  x = sc->w;
  sc->w = sc->nil;
  return(x);
}


/* -------------------------------- openlet? -------------------------------- */
bool s7_is_openlet(s7_pointer e) {return(has_methods(e));}

static s7_pointer g_is_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_is_openlet "(openlet? obj) returns #t is 'obj' has methods."
  #define Q_is_openlet sc->pl_bt

  s7_pointer e = car(args);  /* if e is not a let, should this raise an error? -- no, easier to use this way in cond */
  check_method(sc, e, sc->is_openlet_symbol, args);
  return(make_boolean(sc, has_methods(e)));
}


/* -------------------------------- openlet -------------------------------- */
s7_pointer s7_openlet(s7_scheme *sc, s7_pointer e)
{
  set_has_methods(e);
  return(e);
}

static s7_pointer g_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_openlet "(openlet e) tells the built-in generic functions that the let 'e might have an over-riding method."
  #define Q_openlet sc->pcl_e

  s7_pointer e = car(args), elet, func;
  if ((e == sc->rootlet) || (e == sc->nil))
    s7_error_nr(sc, sc->out_of_range_symbol, set_elist_1(sc, wrap_string(sc, "can't openlet rootlet", 21)));
  elet = find_let(sc, e); /* returns nil if no let found, so has to follow error check above */
  if (!is_let(elet))
    simple_wrong_type_argument_with_type(sc, sc->openlet_symbol, e, a_let_string);
  if ((has_active_methods(sc, e)) &&
      ((func = find_method(sc, elet, sc->openlet_symbol)) != sc->undefined))
    return(s7_apply_function(sc, func, args));
  set_has_methods(e);
  return(e);
}

/* -------------------------------- coverlet -------------------------------- */
static s7_pointer g_coverlet(s7_scheme *sc, s7_pointer args)
{
  #define H_coverlet "(coverlet e) undoes an earlier openlet."
  #define Q_coverlet sc->pcl_e

  s7_pointer e = car(args);
  check_method(sc, e, sc->coverlet_symbol, set_plist_1(sc, e));
  if ((e == sc->rootlet) || (e == sc->s7_let))
    s7_error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "can't coverlet ~S", 17), e));

  if ((is_let(e)) ||
      (has_closure_let(e)) ||
      ((is_c_object(e)) && (c_object_let(e) != sc->nil)) ||
      ((is_c_pointer(e)) && (is_let(c_pointer_info(e)))))
    {
      clear_has_methods(e);
      return(e);
    }
  simple_wrong_type_argument_with_type(sc, sc->coverlet_symbol, e, a_let_string);
  return(NULL);
}


/* -------------------------------- varlet -------------------------------- */
static void append_let(s7_scheme *sc, s7_pointer new_e, s7_pointer old_e)
{
  if ((old_e == sc->rootlet) || (new_e == sc->s7_let))
    return;
  if (new_e == sc->rootlet)
    for (s7_pointer x = let_slots(old_e); tis_slot(x); x = next_slot(x))
      {
	s7_pointer sym = slot_symbol(x), val = slot_value(x);
	if (is_slot(global_slot(sym)))
	  slot_set_value(global_slot(sym), val);
	else s7_make_slot(sc, new_e, sym, val);
      }
  else
    if (old_e == sc->s7_let)
      {
	s7_pointer iter = s7_make_iterator(sc, sc->s7_let);
	s7_int gc_loc = s7_gc_protect(sc, iter);
	iterator_current(iter) = cons_unchecked(sc, sc->F, sc->F);
	set_mark_seq(iter); /* so carrier is GC protected by mark_iterator */
	while (true)
	  {
	    s7_pointer y = s7_iterate(sc, iter);
	    if (iterator_is_at_end(iter)) break;
	    add_slot_checked_with_id(sc, new_e, car(y), cdr(y));
	  }
	s7_gc_unprotect_at(sc, gc_loc);
      }
    else
      for (s7_pointer x = let_slots(old_e); tis_slot(x); x = next_slot(x))
	add_slot_checked_with_id(sc, new_e, slot_symbol(x), slot_value(x)); /* not add_slot here because we might run off the free heap end */
}

static s7_pointer check_c_object_let(s7_scheme *sc, s7_pointer old_e, s7_pointer caller)
{
  if (is_c_object(old_e))
    old_e = c_object_let(old_e);
  if (!is_let(old_e))
    simple_wrong_type_argument_with_type(sc, caller, old_e, a_let_string);
  return(old_e);
}

s7_pointer s7_varlet(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(let))
    wrong_type_argument_with_type(sc, sc->varlet_symbol, 1, let, a_let_string);
  if (!is_symbol(symbol))
    wrong_type_argument_with_type(sc, sc->varlet_symbol, 2, symbol, a_symbol_string);

  if ((is_slot(global_slot(symbol))) &&
      (is_syntax(global_value(symbol))))
    wrong_type_argument_with_type(sc, sc->varlet_symbol, 2, symbol, wrap_string(sc, "a non-syntactic name", 20));

  if (let == sc->rootlet)
    {
      if (is_slot(global_slot(symbol)))
	slot_set_value(global_slot(symbol), value);
      else s7_make_slot(sc, let, symbol, value);
    }
  else add_slot_checked_with_id(sc, let, symbol, value);
  return(value);
}

static s7_pointer g_varlet(s7_scheme *sc, s7_pointer args)   /* varlet = with-let + define */
{
  #define H_varlet "(varlet let ...) adds its arguments (a let, a cons: symbol . value, or two arguments, the symbol and its value) \
to the let let, and returns let.  (varlet (curlet) 'a 1) adds 'a to the current environment with the value 1."
  #define Q_varlet s7_make_circular_signature(sc, 2, 4, sc->is_let_symbol, \
                     s7_make_signature(sc, 2, sc->is_let_symbol, sc->is_null_symbol), \
                       s7_make_signature(sc, 3, sc->is_pair_symbol, sc->is_symbol_symbol, sc->is_let_symbol), \
                         sc->T)

  s7_pointer e = car(args), val;
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->varlet_symbol, args);
      if (!is_let(e))
	wrong_type_argument_with_type(sc, sc->varlet_symbol, 1, e, a_let_string);
      if ((is_immutable(e)) || (e == sc->s7_let))
	s7_error_nr(sc, sc->immutable_error_symbol,
		    set_elist_3(sc, wrap_string(sc, "can't (varlet ~{~S~^ ~}), ~S is immutable", 41), args, e));
    }
  for (s7_pointer x = cdr(args); is_pair(x); x = cdr(x))
    {
      s7_pointer sym, p = car(x);
      switch (type(p))
	{
	case T_SYMBOL:
	  sym = (is_keyword(p)) ? keyword_symbol(p) : p;
	  if (!is_pair(cdr(x)))
	    s7_error_nr(sc, sc->syntax_error_symbol,
			set_elist_3(sc, wrap_string(sc, "varlet: keyword ~S, but no value: ~S", 36), p, args));
	  if (is_constant_symbol(sc, sym))
	    wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), sym, a_non_constant_symbol_string);
	  x = cdr(x);
	  val = car(x);
	  break;

	case T_PAIR:
	  sym = car(p);
	  if (!is_symbol(sym))
	    wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, a_symbol_string);
	  if (is_constant_symbol(sc, sym))
	    wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), sym, a_non_constant_symbol_string);
	  val = cdr(p);
	  break;

	case T_LET:
	  append_let(sc, e, check_c_object_let(sc, p, sc->varlet_symbol));
	  continue;

	default:
	  wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, a_symbol_string);
	}
      if (e == sc->rootlet)
	{
	  if (is_slot(global_slot(sym)))
	    {
	      if (is_syntax(global_value(sym)))
		wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, wrap_string(sc, "a non-syntactic keyword", 23));
	      /*  without this check we can end up turning our code into gibberish:
	       *    (set! quote 1) -> ;can't set! quote
	       *    (varlet (rootlet) '(quote . 1)), :quote -> 1
	       * or worse set quote to a function of one arg that tries to quote something -- infinite loop
	       */
	      slot_set_value_with_hook(global_slot(sym), val);
	    }
	  else s7_make_slot(sc, e, sym, val);
	}
      else
	{
	  if ((has_let_fallback(e)) &&
	      ((sym == sc->let_ref_fallback_symbol) || (sym == sc->let_set_fallback_symbol)))
	    s7_error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "varlet can't shadow ~S", 22), sym));

	  add_slot_checked_with_id(sc, e, sym, val);
	}}
      /* this used to check for sym already defined, and set its value, but that greatly slows down
       *   the most common use (adding a slot), and makes it hard to shadow explicitly.  Don't use
       *   varlet as a substitute for set!/let-set!.
       */
  return(e);
}


/* -------------------------------- cutlet -------------------------------- */
static s7_pointer g_cutlet(s7_scheme *sc, s7_pointer args)
{
  #define H_cutlet "(cutlet e symbol ...) removes symbols from the let e."
  #define Q_cutlet s7_make_circular_signature(sc, 2, 3, sc->is_let_symbol, sc->is_let_symbol, sc->is_symbol_symbol)

  s7_pointer e = car(args);
  s7_int the_un_id;
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->cutlet_symbol, args);
      if (!is_let(e))
	wrong_type_argument_with_type(sc, sc->cutlet_symbol, 1, e, a_let_string);
      if ((is_immutable(e)) || (e == sc->s7_let))
	immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->cutlet_symbol, e));
    }
  /* besides removing the slot we have to make sure the symbol_id does not match else
   *   let-ref and others will use the old slot!  What's the un-id?  Perhaps the next one?
   *   (let ((b 1)) (let ((b 2)) (cutlet (curlet) 'b)) b)
   */
  the_un_id = ++sc->let_number;

  for (s7_pointer syms = cdr(args); is_pair(syms); syms = cdr(syms))
    {
      s7_pointer sym = car(syms);

      if (!is_symbol(sym))
	wrong_type_argument_with_type(sc, sc->cutlet_symbol, position_of(syms, args), sym, a_symbol_string);
      if (is_keyword(sym))
	sym = keyword_symbol(sym);

      if (e == sc->rootlet)
	{
	  if (is_slot(global_slot(sym)))
	    {
	      symbol_set_id(sym, the_un_id);
	      slot_set_value(global_slot(sym), sc->undefined);
	    }}
      else
	{
	  s7_pointer slot;
	  if ((has_let_fallback(e)) &&
	      ((sym == sc->let_ref_fallback_symbol) || (sym == sc->let_set_fallback_symbol)))
	    s7_error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "cutlet can't remove ~S", 22), sym));

	  slot = let_slots(e);
	  if (tis_slot(slot))
	    {
	      if (slot_symbol(slot) == sym)
		{
		  let_set_slots(e, next_slot(let_slots(e)));
		  symbol_set_id(sym, the_un_id);
		}
	      else
		{
		  s7_pointer last_slot = slot;
		  for (slot = next_slot(let_slots(e)); tis_slot(slot); last_slot = slot, slot = next_slot(slot))
		    if (slot_symbol(slot) == sym)
		      {
			symbol_set_id(sym, the_un_id);
			slot_set_next(last_slot, next_slot(slot));
			break;
		      }}}}}
  return(e);
}


/* -------------------------------- sublet -------------------------------- */
static s7_pointer sublet_1(s7_scheme *sc, s7_pointer e, s7_pointer bindings, s7_pointer caller)
{
  s7_pointer new_e = make_let_slowly(sc, (e == sc->rootlet) ? sc->nil : e);
  set_all_methods(new_e, e);

  if (!is_null(bindings))
    {
      s7_pointer sp = NULL;
      sc->temp3 = new_e;
      for (s7_pointer x = bindings; is_pair(x); x = cdr(x))
	{
	  s7_pointer p = car(x), sym, val;

	  switch (type(p))
	    {
	      /* should this insist on one style of field arg?  i.e. (cons sym val) throughout, or :sym val etc? */
	    case T_SYMBOL:
	      sym = (is_keyword(p)) ? keyword_symbol(p) : p;
	      if (!is_pair(cdr(x)))
		s7_error_nr(sc, sc->syntax_error_symbol,
			    set_elist_4(sc, wrap_string(sc, "~A: entry ~S, but no value: ~S", 30), caller, p, bindings));
	      x = cdr(x);
	      val = car(x);
	      break;

	    case T_PAIR:
	      sym = car(p);
	      if (!is_symbol(sym))
		wrong_type_argument_with_type(sc, caller, 1 + position_of(x, bindings), p, a_symbol_string);
	      if (is_keyword(sym))
		sym = keyword_symbol(sym);
	      val = cdr(p);
	      break;

	    case T_LET:
	      append_let(sc, new_e, check_c_object_let(sc, p, caller));
	      continue;

	    default:
	      wrong_type_argument_with_type(sc, caller, 1 + position_of(x, bindings), p, a_symbol_string);
	    }

	  if (is_constant_symbol(sc, sym))
	    wrong_type_argument_with_type(sc, caller, 1 + position_of(x, bindings), sym, a_non_constant_symbol_string);
	  if ((is_slot(global_slot(sym))) &&
	      (is_syntax(global_value(sym))))
	    wrong_type_argument_with_type(sc, caller, 2, sym, wrap_string(sc, "a non-syntactic name", 20));

	  /* here we know new_e is a let and is not rootlet */
	  if (!sp)
	    sp = add_slot_checked_with_id(sc, new_e, sym, val);
	  else
	    {
	      if (sc->free_heap_top <= sc->free_heap_trigger) try_to_call_gc(sc); /* or maybe add add_slot_at_end_checked? */
	      sp = add_slot_at_end(sc, let_id(new_e), sp, sym, val);
	      set_local(sym); /* ? */
	    }
	  if (sym == sc->let_ref_fallback_symbol)
	    set_has_let_ref_fallback(new_e);
	  else
	    if (sym == sc->let_set_fallback_symbol)
	      set_has_let_set_fallback(new_e);
	}
      sc->temp3 = sc->nil;
    }
  return(new_e);
}

s7_pointer s7_sublet(s7_scheme *sc, s7_pointer e, s7_pointer bindings) {return(sublet_1(sc, e, bindings, sc->sublet_symbol));}

static s7_pointer g_sublet(s7_scheme *sc, s7_pointer args)
{
  #define H_sublet "(sublet let ...) makes a new let within the environment 'let', initializing it with the bindings"
  #define Q_sublet Q_varlet

  s7_pointer e = car(args);
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->sublet_symbol, args);
      if (!is_let(e))
	wrong_type_argument_with_type(sc, sc->sublet_symbol, 1, e, a_let_string);
    }
  return(sublet_1(sc, e, cdr(args), sc->sublet_symbol));
}


/* -------------------------------- inlet -------------------------------- */
s7_pointer s7_inlet(s7_scheme *sc, s7_pointer args)
{
  #define H_inlet "(inlet ...) adds its arguments, each a let, a cons: '(symbol . value), or a keyword/value pair, to a new let, and returns the \
new let. (inlet :a 1 :b 2) or (inlet '(a . 1) '(b . 2))"
  #define Q_inlet s7_make_circular_signature(sc, 1, 2, sc->is_let_symbol, sc->T)
  return(sublet_1(sc, sc->rootlet, args, sc->inlet_symbol));
}

#define g_inlet s7_inlet

static s7_pointer g_simple_inlet(s7_scheme *sc, s7_pointer args)
{
  /* here all args are paired with normal symbol/value, no fallbacks, no immutable symbols etc */
  s7_pointer new_e = make_let_slowly(sc, sc->nil);
  int64_t id = let_id(new_e);
  s7_pointer sp = NULL;

  sc->temp3 = new_e;
  for (s7_pointer x = args; is_pair(x); x = cddr(x))
    {
      s7_pointer symbol = car(x);
      if (is_keyword(symbol))                 /* (inlet ':allow-other-keys 3) */
	symbol = keyword_symbol(symbol);
      if (is_constant_symbol(sc, symbol))     /* (inlet 'pi 1) */
	wrong_type_argument_with_type(sc, sc->inlet_symbol, 1, symbol, a_non_constant_symbol_string);
      if (!sp)
	{
	  add_slot_unchecked(sc, new_e, symbol, cadr(x), id);
	  sp = let_slots(new_e);
	}
      else sp = add_slot_at_end(sc, id, sp, symbol, cadr(x));
    }
  sc->temp3 = sc->nil;
  return(new_e);
}

static s7_pointer inlet_p_pp(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x;

  if (!is_symbol(symbol))
    return(sublet_1(sc, sc->nil, set_plist_2(sc, symbol, value), sc->inlet_symbol));
  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);
  if (is_constant_symbol(sc, symbol))
    wrong_type_argument_with_type(sc, sc->inlet_symbol, 1, symbol, a_non_constant_symbol_string);
  if ((is_global(symbol)) &&
      (is_syntax(global_value(symbol))))
    wrong_type_argument_with_type(sc, sc->inlet_symbol, 1, symbol, wrap_string(sc, "a non-syntactic name", 20));

  new_cell(sc, x, T_LET | T_SAFE_PROCEDURE);
  sc->temp3 = x;
  let_set_id(x, ++sc->let_number);
  let_set_outlet(x, sc->nil);
  let_set_slots(x, slot_end(sc));
  add_slot_unchecked(sc, x, symbol, value, let_id(x));
  sc->temp3 = sc->nil;
  return(x);
}

static s7_pointer internal_inlet(s7_scheme *sc, s7_int num_args, ...)
{
  va_list ap;
  s7_pointer new_e = make_let_slowly(sc, sc->nil);
  int64_t id = let_id(new_e);
  s7_pointer sp = NULL;

  sc->temp3 = new_e;
  va_start(ap, num_args);
  for (s7_int i = 0; i < num_args; i += 2)
    {
      s7_pointer symbol = va_arg(ap, s7_pointer);
      s7_pointer value = va_arg(ap, s7_pointer);
      if ((S7_DEBUGGING) && (is_keyword(symbol))) fprintf(stderr, "internal_inlet key: %s??\n", display(symbol));
      if (!sp)
	{
	  add_slot_unchecked(sc, new_e, symbol, value, id);
	  sp = let_slots(new_e);
	}
      else sp = add_slot_at_end(sc, id, sp, symbol, value);
    }
  va_end(ap);
  sc->temp3 = sc->nil;
  return(new_e);
}

static bool is_proper_quote(s7_scheme *sc, s7_pointer p)
{
  return((is_quoted_pair(p)) &&
	 (is_pair(cdr(p))) &&
	 (is_null(cddr(p))) &&
	 (is_global(sc->quote_symbol)));
}

static s7_pointer inlet_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (!ops) return(f);
  if ((args > 0) &&
      ((args % 2) == 0))
    {
      for (s7_pointer p = cdr(expr); is_pair(p); p = cddr(p))
	if (!is_symbol_and_keyword(car(p)))
	  {
	    s7_pointer sym;
	    if (!is_proper_quote(sc, car(p)))             /* 'abs etc, but tricky: ':abs */
	      return(f);
	    sym = cadar(p);
	    if ((!is_symbol(sym)) ||
		(is_possibly_constant(sym)) ||            /* define-constant etc */
		(is_syntactic_symbol(sym))  ||            /* (inlet 'if 3) */
		((is_slot(global_slot(sym))) &&
		 (is_syntax(global_value(sym)))) ||
		(sym == sc->let_ref_fallback_symbol) ||
		(sym == sc->let_set_fallback_symbol))
	      return(f);
	  }
      return(sc->simple_inlet);
    }
  return(f);
}


/* -------------------------------- let->list -------------------------------- */
static s7_pointer proper_list_reverse_in_place(s7_scheme *sc, s7_pointer list);

s7_pointer s7_let_to_list(s7_scheme *sc, s7_pointer let)
{
  s7_pointer x;
  sc->temp3 = sc->w;
  sc->w = sc->nil;
  if (let == sc->rootlet)
    {
      s7_int i, lim2 = sc->rootlet_entries;
      s7_pointer *entries = rootlet_elements(let);

      if (lim2 & 1) lim2--;
      for (i = 0; i < lim2; )
	{
	  sc->w = cons_unchecked(sc, cons(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w); i++;
	  sc->w = cons_unchecked(sc, cons_unchecked(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w); i++;
	}
      if (lim2 < sc->rootlet_entries)
	sc->w = cons_unchecked(sc, cons(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w);
    }
  else
    {
      s7_pointer iter, func;
      s7_int gc_loc = -1;
      /* need to check make-iterator method before dropping into let->list */

      if ((has_active_methods(sc, let)) &&
	  ((func = find_method(sc, let, sc->make_iterator_symbol)) != sc->undefined))
	iter = s7_apply_function(sc, func, set_plist_1(sc, let));
      else
	if (let == sc->s7_let) /* (let->list *s7*) via s7_let_make_iterator */
	  {
	    iter = s7_make_iterator(sc, let);
	    gc_loc = s7_gc_protect(sc, iter);
	  }
	else iter = sc->nil;

      if (is_null(iter))
	for (x = let_slots(let); tis_slot(x); x = next_slot(x))
	  sc->w = cons_unchecked(sc, cons(sc, slot_symbol(x), slot_value(x)), sc->w);
      else
	/* (begin (load "mockery.scm") (let ((lt ((*mock-pair* 'mock-pair) 1 2 3))) (format *stderr* "~{~A ~}" lt))) */
	while (true)
	  {
	    x = s7_iterate(sc, iter);
	    if (iterator_is_at_end(iter)) break;
	    sc->w = cons(sc, x, sc->w);
	  }
      sc->w = proper_list_reverse_in_place(sc, sc->w);
      if (gc_loc != -1)
	s7_gc_unprotect_at(sc, gc_loc);
    }
  x = sc->w;
  sc->w = sc->temp3;
  sc->temp3 = sc->nil;
  return(x);
}

#if (!WITH_PURE_S7)
static s7_pointer g_let_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_let_to_list "(let->list let) returns let's bindings as a list of cons's: '(symbol . value)."
  #define Q_let_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_let_symbol)

  s7_pointer let = car(args);
  check_method(sc, let, sc->let_to_list_symbol, args);
  if (!is_let(let))
    {
      if (is_c_object(let))
	let = c_object_let(let);
      else
	if (is_c_pointer(let))
	  let = c_pointer_info(let);
      if (!is_let(let))
        simple_wrong_type_argument_with_type(sc, sc->let_to_list_symbol, let, a_let_string);
    }
  return(s7_let_to_list(sc, let));
}
#endif


/* -------------------------------- let-ref -------------------------------- */

static s7_pointer call_let_ref_fallback(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  s7_pointer p;
  push_stack_no_let(sc, OP_GC_PROTECT, sc->value, sc->code);
  p = s7_apply_function(sc, find_method(sc, let, sc->let_ref_fallback_symbol), set_qlist_2(sc, let, symbol));
  unstack(sc);
  sc->code = T_Pos(sc->stack_end[0]);
  sc->value = T_Pos(sc->stack_end[2]);
  return(p);
}

static s7_pointer call_let_set_fallback(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer p;
  push_stack_no_let(sc, OP_GC_PROTECT, sc->value, sc->code);
  p = s7_apply_function(sc, find_method(sc, let, sc->let_set_fallback_symbol), set_qlist_3(sc, let, symbol, value));
  unstack(sc);
  sc->code = T_Pos(sc->stack_end[0]);
  sc->value = T_Pos(sc->stack_end[2]);
  return(p);
}

inline s7_pointer s7_let_ref(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  /* (let ((a 1)) ((curlet) 'a)) or ((rootlet) 'abs) */
  if (!is_let(let))
    {
      let = find_let(sc, let);
      if (!is_let(let))
	wrong_type_argument_with_type(sc, sc->let_ref_symbol, 1, let, a_let_string);
    }
  if (!is_symbol(symbol))
    {
      if (has_let_ref_fallback(let))
	return(call_let_ref_fallback(sc, let, symbol));
      wrong_type_argument_with_type(sc, sc->let_ref_symbol, 2, symbol, a_symbol_string);
    }

  if (!is_global(sc->let_ref_symbol))
    check_method(sc, let, sc->let_ref_symbol, set_plist_2(sc, let, symbol));
  /* a let-ref method is almost impossible to write without creating an infinite loop:
   *   any reference to the let will probably call let-ref somewhere, calling us again, and looping.
   *   This is not a problem in c-objects and funclets because c-object-ref and funclet-ref don't exist.
   *   After much wasted debugging, I decided to make let-ref and let-set! immutable.
   */

  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);

  if (let == sc->rootlet)
    return((is_slot(global_slot(symbol))) ? global_value(symbol) : sc->undefined);

  if (let_id(let) == symbol_id(symbol))
    return(local_value(symbol)); /* this obviously has to follow the rootlet check */

  for (s7_pointer x = let; is_let(x); x = let_outlet(x))
    for (s7_pointer y = let_slots(x); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(slot_value(y));

  if (has_methods(let))
    {
      /* If a let is a mock-hash-table (for example), implicit
       *   indexing of the hash-table collides with the same thing for the let (field names
       *   versus keys), and we can't just try again here because that makes it too easy to
       *   get into infinite recursion.  So, 'let-ref-fallback...
       */
      if (has_let_ref_fallback(let))
	return(call_let_ref_fallback(sc, let, symbol));
    }
  return((is_slot(global_slot(symbol))) ? global_value(symbol) : sc->undefined); /* (let () ((curlet) 'pi)) */
}

static s7_pointer g_let_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_let_ref "(let-ref let sym) returns the value of the symbol sym in the let"
  #define Q_let_ref s7_make_signature(sc, 3, sc->T, sc->is_let_symbol, sc->is_symbol_symbol)
  return(s7_let_ref(sc, car(args), cadr(args)));
}

static s7_pointer slot_in_let(s7_scheme *sc, s7_pointer e, s7_pointer sym)
{
  for (s7_pointer y = let_slots(e); tis_slot(y); y = next_slot(y))
    if (slot_symbol(y) == sym)
      return(y);
  return(sc->undefined);
}

static s7_pointer lint_let_ref_p_pp(s7_scheme *sc, s7_pointer lt, s7_pointer sym)
{
  for (s7_pointer x = lt; is_let(x); x = let_outlet(x))
    for (s7_pointer y = let_slots(x); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == sym)
	return(slot_value(y));

  if ((has_methods(lt)) &&
      (has_let_ref_fallback(lt)))
    return(call_let_ref_fallback(sc, lt, sym));

  return((is_slot(global_slot(sym))) ? global_value(sym) : sc->undefined);
}

static inline s7_pointer g_lint_let_ref(s7_scheme *sc, s7_pointer args)
{
  s7_pointer lt = car(args), sym;
  if (!is_let(lt))
    wrong_type_argument_with_type(sc, sc->let_ref_symbol, 1, lt, a_let_string);
  sym = cadr(args);
  for (s7_pointer y = let_slots(lt); tis_slot(y); y = next_slot(y))
    if (slot_symbol(y) == sym)
      return(slot_value(y));
  return(lint_let_ref_p_pp(sc, let_outlet(lt), sym));
}

static s7_pointer let_ref_chooser(s7_scheme *sc, s7_pointer f, int32_t unused_args, s7_pointer expr, bool ops)
{
  if ((!ops) || (!is_global(sc->let_ref_symbol))) return(f);
  if (optimize_op(expr) == HOP_SAFE_C_opSq_C)
    {
      s7_pointer arg1 = cadr(expr), arg2 = caddr(expr);
      if ((car(arg1) == sc->cdr_symbol) &&
	  (is_quoted_symbol(arg2)) &&
	  (!is_possibly_constant(cadr(arg2))))
	{
	  set_opt3_sym(cdr(expr), cadr(arg2));
	  return(sc->lint_let_ref);
	}}
  return(f);
}

static bool op_implicit_let_ref_c(s7_scheme *sc)
{
  s7_pointer s = lookup_checked(sc, car(sc->code));
  if (!is_let(s)) {sc->last_function = s; return(false);}
  sc->value = s7_let_ref(sc, T_Pos(s), opt3_con(sc->code));
  return(true);
}

static bool op_implicit_let_ref_a(s7_scheme *sc)
{
  s7_pointer s = lookup_checked(sc, car(sc->code));
  if (!is_let(s)) {sc->last_function = s; return(false);}
  sc->value = s7_let_ref(sc, s, fx_call(sc, cdr(sc->code)));
  return(true);
}


/* -------------------------------- let-set! -------------------------------- */
static s7_pointer let_set_1(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);
  symbol_increment_ctr(symbol);

  if (let == sc->rootlet)
    {
      s7_pointer slot;
      if (is_constant_symbol(sc, symbol))  /* (let-set! (rootlet) 'pi #f) */
	wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, a_non_constant_symbol_string);

      slot = global_slot(symbol);
      if (!is_slot(slot))
	s7_error_nr(sc, sc->wrong_type_arg_symbol,
		    set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), symbol, let));
      if (is_syntax(slot_value(slot)))
	wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, wrap_string(sc, "a non-syntactic keyword", 23));

      slot_set_value(slot, (slot_has_setter(slot)) ? call_setter(sc, slot, value) : value);
      return(slot_value(slot));
    }

  if (let_id(let) == symbol_id(symbol))
   {
     s7_pointer slot = local_slot(symbol);
     if (is_slot(slot))
       return(checked_slot_set_value(sc, slot, value));
   }
  for (s7_pointer x = let; is_let(x); x = let_outlet(x))
    for (s7_pointer y = let_slots(x); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(checked_slot_set_value(sc, y, value));

  if ((!has_methods(let)) ||
      (!has_let_set_fallback(let)))
    s7_error_nr(sc, sc->wrong_type_arg_symbol,
		set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), symbol, let));
  /* not sure about this -- what's the most useful choice? */
  return(call_let_set_fallback(sc, let, symbol, value));
}

s7_pointer s7_let_set(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(let))
    wrong_type_argument_with_type(sc, sc->let_set_symbol, 1, let, a_let_string);
  if (!is_symbol(symbol))
    {
      if (has_let_set_fallback(let))
	return(call_let_set_fallback(sc, let, symbol, value));
      wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, a_symbol_string);
    }
  if (!is_global(sc->let_set_symbol))
    check_method(sc, let, sc->let_set_symbol, set_plist_3(sc, let, symbol, value));
  return(let_set_1(sc, let, symbol, value));
}

static s7_pointer g_let_set(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (set! ((curlet) 'a) 32) a) */
  #define H_let_set "(let-set! let sym val) sets the symbol sym's value in the let to val"
  #define Q_let_set s7_make_signature(sc, 4, sc->T, sc->is_let_symbol, sc->is_symbol_symbol, sc->T)
  return(s7_let_set(sc, car(args), cadr(args), caddr(args)));
}

static s7_pointer let_set_p_ppp_2(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_pointer p3)
{
  if (!is_symbol(p2))
    wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, p2, a_symbol_string);
  return(let_set_1(sc, p1, p2, p3));
}

static s7_pointer g_lint_let_set(s7_scheme *sc, s7_pointer args)
{
  s7_pointer y, lt = car(args), sym = cadr(args), val = caddr(args);

  if (!is_let(lt))
    wrong_type_argument_with_type(sc, sc->let_set_symbol, 1, lt, a_let_string);
  if (lt != sc->rootlet)
    {
      for (s7_pointer x = lt; is_let(x); x = let_outlet(x))
	for (y = let_slots(x); tis_slot(y); y = next_slot(y))
	  if (slot_symbol(y) == sym)
	    {
	      slot_set_value(y, (slot_has_setter(y)) ? call_setter(sc, y, val) : val);
	      return(slot_value(y));
	    }
      if ((has_methods(lt)) &&
	  (has_let_set_fallback(lt)))
	return(call_let_set_fallback(sc, lt, sym, val));
    }
  y = global_slot(sym);
  if (!is_slot(y))
    s7_error_nr(sc, sc->wrong_type_arg_symbol, set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), sym, lt));
  slot_set_value(y, (slot_has_setter(y)) ? call_setter(sc, y, val) : val);
  return(slot_value(y));
}

static s7_pointer let_set_chooser(s7_scheme *sc, s7_pointer f, int32_t unused_args, s7_pointer expr, bool ops)
{
  if ((!ops) || (!is_global(sc->let_set_symbol))) return(f);
  if (optimize_op(expr) == HOP_SAFE_C_opSq_CS)
    {
      s7_pointer arg1 = cadr(expr), arg2 = caddr(expr), arg3 = cadddr(expr);
      if ((car(arg1) == sc->cdr_symbol) &&
	  (car(arg2) == sc->quote_symbol) &&
	  (is_symbol(cadr(arg2))) &&
	  (!is_possibly_constant(cadr(arg2))) &&
	  (!is_possibly_constant(arg3)))
	return(sc->lint_let_set);
    }
  return(f);
}


static s7_pointer reverse_slots(s7_scheme *sc, s7_pointer list)
{
  s7_pointer p = list, result = slot_end(sc);
  while (tis_slot(p))
    {
      s7_pointer q = next_slot(p);
      slot_set_next(p, result);
      result = p;
      p = q;
    }
  return(result);
}

static s7_pointer let_copy(s7_scheme *sc, s7_pointer let)
{
  if (is_let(let))
    {
      s7_pointer new_e;

      if (let == sc->rootlet)   /* (copy (rootlet)) or (copy (funclet abs)) etc */
	return(sc->rootlet);

      /* we can't make copy handle lets-as-objects specially because the make-object function in define-class uses copy to make a new object!
       *   So if it is present, we get it here, and then there's almost surely trouble.
       */
      new_e = make_let_slowly(sc, let_outlet(let));
      set_all_methods(new_e, let);
      sc->temp3 = new_e;
      if (tis_slot(let_slots(let)))
	{
	  s7_int id = let_id(new_e);
	  s7_pointer y = NULL;
	  for (s7_pointer x = let_slots(let); tis_slot(x); x = next_slot(x))
	    {
	      s7_pointer z;
	      new_cell(sc, z, T_SLOT);
	      slot_set_symbol_and_value(z, slot_symbol(x), slot_value(x));
	      if (symbol_id(slot_symbol(z)) != id) /* keep shadowing intact */
		symbol_set_local_slot(slot_symbol(x), id, z);
	      if (slot_has_setter(x))
		{
		  slot_set_setter(z, slot_setter(x));
		  slot_set_has_setter(z);
		}
	      if (y)
		slot_set_next(y, z);
	      else let_set_slots(new_e, z);
	      slot_set_next(z, slot_end(sc));              /* in case GC runs during this loop */
	      y = z;
	    }}
      /* We can't do a (normal) loop here then reverse the slots later because the symbol's local_slot has to
       *    match the unshadowed slot, not the last in the list:
       *    (let ((e1 (inlet 'a 1 'a 2))) (let ((e2 (copy e1))) (list (equal? e1 e2) (equal? (e1 'a) (e2 'a)))))
       */
      sc->temp3 = sc->nil;
      return(new_e);
    }
  return(sc->nil);
}


/* -------------------------------- rootlet -------------------------------- */
static s7_pointer g_rootlet(s7_scheme *sc, s7_pointer unused)
{
  #define H_rootlet "(rootlet) returns the current top-level definitions (symbol bindings)."
  #define Q_rootlet s7_make_signature(sc, 1, sc->is_let_symbol)
  return(sc->rootlet);
}
/* as with the symbol-table, this function can lead to disaster -- user could
 *   clobber the let etc.  But we want it to be editable and augmentable,
 *   so I guess I'll leave it alone.  (See curlet|funclet as well).
 */

s7_pointer s7_rootlet(s7_scheme *sc) {return(sc->rootlet);}

/* shadow_rootlet is a convenience for foreign function writers -- the C code can act as if it were loading everything into rootlet,
 *   but when actually loaded, everything can be shunted into a separate namespace (*motif* for example).
 */
s7_pointer s7_shadow_rootlet(s7_scheme *sc) {return(sc->shadow_rootlet);}

s7_pointer s7_set_shadow_rootlet(s7_scheme *sc, s7_pointer let)
{
  s7_pointer old_let = sc->shadow_rootlet;
  sc->shadow_rootlet = let;
  return(old_let); /* like s7_set_curlet below */
}


/* -------------------------------- curlet -------------------------------- */
static s7_pointer g_curlet(s7_scheme *sc, s7_pointer unused_args)
{
  #define H_curlet "(curlet) returns the current definitions (symbol bindings)"
  #define Q_curlet s7_make_signature(sc, 1, sc->is_let_symbol)

  sc->capture_let_counter++;
  return((is_let(sc->curlet)) ? sc->curlet : sc->rootlet);
}

s7_pointer s7_curlet(s7_scheme *sc)
{
  sc->capture_let_counter++;
  return(sc->curlet);
}

static void update_symbol_ids(s7_scheme *sc, s7_pointer e)
{
  for (s7_pointer p = let_slots(e); tis_slot(p); p = next_slot(p))
    {
      s7_pointer sym = slot_symbol(p);
      if (symbol_id(sym) != sc->let_number)
	symbol_set_local_slot_unincremented(sym, sc->let_number, p);
    }
}

s7_pointer s7_set_curlet(s7_scheme *sc, s7_pointer e)
{
  s7_pointer old_e = sc->curlet;
  set_curlet(sc, e);
  if ((is_let(e)) && (let_id(e) > 0)) /* might be () [id=-1] or rootlet [id=0?] etc */
    {
      let_set_id(e, ++sc->let_number);
      update_symbol_ids(sc, e);
    }
  return(old_e);
}


/* -------------------------------- outlet -------------------------------- */
s7_pointer s7_outlet(s7_scheme *sc, s7_pointer let)
{
  if ((let == sc->rootlet) || (is_null(let_outlet(let))))
    return(sc->rootlet);
  return(let_outlet(let));
}

s7_pointer outlet_p_p(s7_scheme *sc, s7_pointer let)
{
  if (!is_let(let))
    return(s7_wrong_type_arg_error(sc, "outlet", 1, let, "a let")); /* not a method call here! */
  if ((let == sc->rootlet) || (is_null(let_outlet(let))))
    return(sc->rootlet);
  return(let_outlet(let));
}

static s7_pointer g_outlet(s7_scheme *sc, s7_pointer args)
{
  #define H_outlet "(outlet let) is the environment that contains let."
  #define Q_outlet s7_make_signature(sc, 2, sc->is_let_symbol, sc->is_let_symbol)
  return(outlet_p_p(sc, car(args)));
}


static s7_pointer g_set_outlet(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (let ((b 2)) (set! (outlet (curlet)) (rootlet)) ((curlet) 'a))) */
  s7_pointer let = car(args), new_outer;

  if (!is_let(let))
    return(s7_wrong_type_arg_error(sc, "set! outlet", 1, let, "a let"));
  if (let == sc->s7_let)
    s7_error_nr(sc, sc->out_of_range_symbol, set_elist_1(sc, wrap_string(sc, "can't set! (outlet *s7*)", 24)));
  if (is_immutable(let))
    immutable_object_error(sc, set_elist_4(sc, wrap_string(sc, "can't (set! (outlet ~S) ~S), ~S is immutable", 44), let, cadr(args), let));
  new_outer = cadr(args);
  if (!is_let(new_outer))
    return(s7_wrong_type_arg_error(sc, "set! outlet", 2, new_outer, "a let"));
  if (let != sc->rootlet)
    {
      /* here it's possible to get cyclic let chains; maybe do this check only if safety>0 */
      for (s7_pointer lt = new_outer; (is_let(lt)) && (lt != sc->rootlet); lt = let_outlet(lt))
	if (let == lt)
	  s7_error_nr(sc, make_symbol(sc, "cyclic-let"),
		      set_elist_2(sc, wrap_string(sc, "set! (outlet ~A) creates a cyclic let chain", 43), let));
      let_set_outlet(let, (new_outer == sc->rootlet) ? sc->nil : new_outer);  /* outlet rootlet->() so that slot search can use is_let(outlet) I think */
    }
  return(new_outer);
}

/* -------------------------------- symbol lookup -------------------------------- */

static inline s7_pointer lookup_from(s7_scheme *sc, const s7_pointer symbol, s7_pointer e)
{
#if S7_DEBUGGING
  if ((!is_let(e)) && (!is_null(e)))
    {
      fprintf(stderr, "%s[%d]: e is not a let\n", __func__, __LINE__);
      if (sc != cur_sc)	fprintf(stderr, "sc != cur_sc\n");
    }
#endif
  if (let_id(e) == symbol_id(symbol))
    return(local_value(symbol));
  if (symbol_id(symbol) < let_id(e))
    {
      do {e = let_outlet(e);} while (symbol_id(symbol) < let_id(e));
      if (let_id(e) == symbol_id(symbol))
	return(local_value(symbol));
    }
  for (; is_let(e); e = let_outlet(e))
    for (s7_pointer y = let_slots(e); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(slot_value(y));

  if (is_slot(global_slot(symbol)))
    return(global_value(symbol));
#if WITH_GCC
  return(NULL); /* much faster than various alternatives */
#else
  return(unbound_variable(sc, symbol));
#endif
}

static inline s7_pointer lookup_slot_from(s7_pointer symbol, s7_pointer e)
{
  if (let_id(e) == symbol_id(symbol))
    return(local_slot(symbol));
  if (symbol_id(symbol) < let_id(e))
    {
      do {e = let_outlet(e);} while (symbol_id(symbol) < let_id(e));
      if (let_id(e) == symbol_id(symbol))
	return(local_slot(symbol));
    }
  for (; is_let(e); e = let_outlet(e))
    for (s7_pointer y = let_slots(e); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(y);
  return(global_slot(symbol));
}

#if WITH_GCC && S7_DEBUGGING
static s7_pointer lookup_1(s7_scheme *sc, const s7_pointer symbol)
#else
static inline s7_pointer lookup(s7_scheme *sc, const s7_pointer symbol) /* lookup_checked includes the unbound_variable call */
#endif
{
  return(lookup_from(sc, symbol, sc->curlet));
}

s7_pointer s7_slot(s7_scheme *sc, s7_pointer symbol) {return(lookup_slot_from(symbol, sc->curlet));}

s7_pointer s7_slot_value(s7_pointer slot) {return(slot_value(slot));}

s7_pointer s7_slot_set_value(s7_scheme *sc, s7_pointer slot, s7_pointer value) {slot_set_value(slot, value); return(value);}

void s7_slot_set_real_value(s7_scheme *sc, s7_pointer slot, s7_double value) {set_real(slot_value(slot), value);}

static s7_pointer symbol_to_local_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer e)
{
  if (!is_let(e))
    return(global_slot(symbol));
  if (symbol_id(symbol) != 0)
    for (s7_pointer y = let_slots(e); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(y);
  return(sc->undefined);
}

s7_pointer s7_symbol_value(s7_scheme *sc, s7_pointer sym)
{
  s7_pointer x = lookup_slot_from(sym, sc->curlet);
  return((is_slot(x)) ? slot_value(x) : sc->undefined);
}

s7_pointer s7_symbol_local_value(s7_scheme *sc, s7_pointer sym, s7_pointer let)
{
  /* restrict the search to local let outward */
  if ((let == sc->rootlet) || (is_global(sym)))
    return((is_slot(global_slot(sym))) ? global_value(sym) : sc->undefined);

  if (!is_let(let))
    return(s7_symbol_value(sc, sym));

  if (let_id(let) == symbol_id(sym))
    return(local_value(sym));
  if (symbol_id(sym) < let_id(let))
    {
      do {let = let_outlet(let);} while (symbol_id(sym) < let_id(let));
      if (let_id(let) == symbol_id(sym))
	return(local_value(sym));
    }
  for (; is_let(let); let = let_outlet(let))
    for (s7_pointer y = let_slots(let); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == sym)
	return(slot_value(y));

  /* need to check rootlet before giving up */
  if (is_slot(global_slot(sym)))
    return(global_value(sym));

  /* (let ((e (curlet))) (let ((a 1)) (symbol->value 'a e))) -> #<undefined> not 1 */
  return(sc->undefined); /* 29-Nov-17 */
}


/* -------------------------------- symbol->value -------------------------------- */
#define lookup_global(Sc, Sym) ((is_global(Sym)) ? global_value(Sym) : lookup_checked(Sc, Sym))

static s7_pointer g_s7_let_ref_fallback(s7_scheme *sc, s7_pointer args);
static s7_pointer g_s7_let_set_fallback(s7_scheme *sc, s7_pointer args);

static s7_pointer g_symbol_to_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_value "(symbol->value sym (let (curlet))) returns the binding of (the value associated with) the \
symbol sym in the given let: (let ((x 32)) (symbol->value 'x)) -> 32"
  #define Q_symbol_to_value s7_make_signature(sc, 3, sc->T, sc->is_symbol_symbol, sc->is_let_symbol)
  /* (symbol->value 'x e) => (e 'x)? */

  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->symbol_to_value_symbol, args, T_SYMBOL, 1));

  if (is_not_null(cdr(args)))
    {
      s7_pointer local_let = cadr(args);
      if (local_let == sc->unlet_symbol)
	return((is_slot(initial_slot(sym))) ? initial_value(sym) : sc->undefined);

      if (!is_let(local_let))
	{
	  local_let = find_let(sc, local_let);
	  if (!is_let(local_let))
	    return(method_or_bust_with_type(sc, cadr(args), sc->symbol_to_value_symbol, args, a_let_string, 2));
	}
      if (local_let == sc->s7_let)
	return(g_s7_let_ref_fallback(sc, set_qlist_2(sc, local_let, sym)));

      return(s7_symbol_local_value(sc, sym, local_let));
    }
  if (is_global(sym))
    return(global_value(sym));
  return(s7_symbol_value(sc, sym));
}

s7_pointer s7_symbol_set_value(s7_scheme *sc, s7_pointer sym, s7_pointer val)
{
  s7_pointer x = lookup_slot_from(sym, sc->curlet);   /* if immutable should this return an error? */
  if (is_slot(x))
    slot_set_value(x, val); /* with_hook? */
  return(val);
}


/* -------------------------------- symbol->dynamic-value -------------------------------- */
static s7_pointer find_dynamic_value(s7_scheme *sc, s7_pointer x, s7_pointer sym, int64_t *id)
{
  for (; symbol_id(sym) < let_id(x); x = let_outlet(x));
  if (let_id(x) == symbol_id(sym))
    {
      (*id) = let_id(x);
      return(local_value(sym));
    }
  for (; (is_let(x)) && (let_id(x) > (*id)); x = let_outlet(x))
    for (s7_pointer y = let_slots(x); tis_slot(y); y = next_slot(y))
      if (slot_symbol(y) == sym)
	{
	  (*id) = let_id(x);
	  return(slot_value(y));
	}
  return(sc->unused);
}

static s7_pointer g_symbol_to_dynamic_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_dynamic_value "(symbol->dynamic-value sym) returns the dynamic binding of the symbol sym"
  #define Q_symbol_to_dynamic_value s7_make_signature(sc, 2, sc->T, sc->is_symbol_symbol)

  s7_pointer sym = car(args), val;
  int64_t top_id = -1;

  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->symbol_to_dynamic_value_symbol, args, T_SYMBOL, 1));

  if (is_global(sym))
    return(global_value(sym));

  if (let_id(sc->curlet) == symbol_id(sym))
    return(local_value(sym));

  val = find_dynamic_value(sc, sc->curlet, sym, &top_id);
  if (top_id == symbol_id(sym))
    return(val);

  for (int64_t i = current_stack_top(sc) - 1; i > 0; i -= 4)
    if (is_let_unchecked(stack_let(sc->stack, i))) /* OP_GC_PROTECT let slot can be anything (even free) */
      {
	s7_pointer cur_val = find_dynamic_value(sc, stack_let(sc->stack, i), sym, &top_id);
	if (cur_val != sc->unused)
	  val = cur_val;
	if (top_id == symbol_id(sym))
	  return(val);
      }
  return((val == sc->unused) ? s7_symbol_value(sc, sym) : val);
}


typedef bool (safe_sym_t)(s7_scheme *sc, s7_pointer sym, s7_pointer e);

static bool direct_memq(s7_pointer symbol, s7_pointer symbols)
{
  for (s7_pointer x = symbols; is_pair(x); x = cdr(x))
    if (car(x) == symbol)
	return(true);
  return(false);
}

static bool direct_assq(s7_pointer symbol, s7_pointer symbols)
{ /* used only below in do_symbol_is_safe */
  for (s7_pointer x = symbols; is_pair(x); x = cdr(x))
    if (caar(x) == symbol)
      return(true);
  return(false);
}

static bool do_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((is_slot(global_slot(sym))) ||
	 (direct_assq(sym, e)) ||
	 (is_slot(lookup_slot_from(sym, sc->curlet))));
}

static bool let_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  if (is_slot(global_slot(sym)))
    return(true);
  if (is_null(e))
    e = sc->rootlet;
  return((!is_with_let_let(e)) &&
	 (is_slot(lookup_slot_from(sym, sc->curlet))));
}

static bool let_symbol_is_safe_or_listed(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((symbol_is_in_list(sc, sym)) ||
	 (let_symbol_is_safe(sc, sym, e)));
}

static bool let_star_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((symbol_is_in_list(sc, sym)) ||
	 (is_slot(global_slot(sym))) ||
	 ((is_let(e)) && (!is_with_let_let(e)) && (is_slot(lookup_slot_from(sym, sc->curlet)))));
}

static bool pair_symbol_is_safe(s7_scheme *unused_sc, s7_pointer sym, s7_pointer e)
{
  return((is_slot(global_slot(sym))) ||
	 (direct_memq(sym, e)));
}

static s7_pointer collect_variables(s7_scheme *sc, s7_pointer lst, s7_pointer e)
{
  /* collect local variable names from let/do (pre-error-check), 20 overhead in tgen -> 14 if cons_unchecked below */
  sc->w = e;
  for (s7_pointer p = lst; is_pair(p); p = cdr(p))
    sc->w = cons(sc, add_symbol_to_list(sc, caar(p)), sc->w);
  return(sc->w);
}

static s7_pointer collect_parameters(s7_scheme *sc, s7_pointer lst, s7_pointer e)
{
  /* collect local variable names from lambda arglists (pre-error-check) */
  s7_pointer p;
  s7_int the_un_id = ++sc->let_number;
  if (is_symbol(lst))
    {
      symbol_set_id(lst, the_un_id);
      return(cons(sc, add_symbol_to_list(sc, lst), e));
    }
  sc->w = e;
  for (p = lst; is_pair(p); p = cdr(p))
    {
      s7_pointer car_p = car(p);
      if (is_pair(car_p))
	car_p = car(car_p);
      if (is_normal_symbol(car_p))
	{
	  symbol_set_id(car_p, the_un_id);
	  sc->w = cons(sc, add_symbol_to_list(sc, car_p), sc->w);
	}}
  if (is_symbol(p)) /* rest arg */
    {
      symbol_set_id(p, the_un_id);
      sc->w = cons(sc, add_symbol_to_list(sc, p), sc->w);
    }
  return(sc->w);
}

typedef enum {OPT_F, OPT_T, OPT_OOPS} opt_t;
static opt_t optimize(s7_scheme *sc, s7_pointer code, int32_t hop, s7_pointer e);

static void clear_all_optimizations(s7_scheme *sc, s7_pointer p)
{
  /* I believe that we would not have been optimized to begin with if the tree were circular,
   *   and this tree is supposed to be a function call + args -- a circular list here is a bug.
   */
  if (is_pair(p))
    {
      if ((is_optimized(p)) &&
 	  (((optimize_op(p) >= FIRST_UNHOPPABLE_OP) ||  /* avoid clearing hop ops, fx_function and op_unknown* need to be cleared */
	    (!op_has_hop(p)))))
	{
	  clear_optimized(p);     /* includes T_SYNTACTIC */
	  clear_optimize_op(p);
	}
      clear_all_optimizations(sc, cdr(p));
      clear_all_optimizations(sc, car(p));
    }
}

static s7_pointer add_trace(s7_scheme *sc, s7_pointer code)
{
  if ((is_pair(car(code))) && (caar(code) == sc->trace_in_symbol))
    return(code);
  return(cons_unchecked(sc, list_2(sc, sc->trace_in_symbol, list_1(sc, sc->curlet_symbol)), code));
}

static s7_pointer add_profile(s7_scheme *sc, s7_pointer code)
{
  s7_pointer p;
  if ((is_pair(car(code))) && (caar(code) == sc->profile_in_symbol))
    return(code);
  p = cons_unchecked(sc, list_3(sc, sc->profile_in_symbol, s7_make_integer(sc, sc->profile_position), list_1(sc, sc->curlet_symbol)), code);
  sc->profile_position++;
  set_unsafe_optimize_op(car(p), OP_PROFILE_IN);
  return(p);
}

static bool tree_has_definers(s7_scheme *sc, s7_pointer tree)
{
  for (s7_pointer p = tree; is_pair(p); p = cdr(p))
    if (tree_has_definers(sc, car(p)))
      return(true);
  return((is_symbol(tree)) && (is_definer(tree)));
}

static s7_pointer make_macro(s7_scheme *sc, opcode_t op, bool named)
{
  s7_pointer mac, body, mac_name = NULL;
  uint64_t typ;
  switch (op)
    {
    case OP_DEFINE_MACRO:      case OP_MACRO:      typ = T_MACRO;      break;
    case OP_DEFINE_MACRO_STAR: case OP_MACRO_STAR: typ = T_MACRO_STAR; break;
    case OP_DEFINE_BACRO:      case OP_BACRO:      typ = T_BACRO;      break;
    case OP_DEFINE_BACRO_STAR: case OP_BACRO_STAR: typ = T_BACRO_STAR; break;
    case OP_DEFINE_EXPANSION:      typ = T_MACRO | ((is_let(sc->curlet)) ? 0 : T_EXPANSION); break; /* local expansions are just normal macros */
    case OP_DEFINE_EXPANSION_STAR: typ = T_MACRO_STAR | ((is_let(sc->curlet)) ? 0 : T_EXPANSION); break;
    default:
      if (S7_DEBUGGING) fprintf(stderr, "%s[%d]: got %s\n", __func__, __LINE__, op_names[op]);
      typ = T_MACRO;
      break;
    }
  new_cell(sc, mac, typ | T_DONT_EVAL_ARGS);
  closure_set_args(mac, (named) ? cdar(sc->code) : car(sc->code));
  body = cdr(sc->code);
  closure_set_body(mac, body);
  closure_set_setter(mac, sc->F);
  closure_set_let(mac, sc->curlet);
  closure_set_arity(mac, CLOSURE_ARITY_NOT_SET);
  sc->capture_let_counter++;
  gc_protect_via_stack(sc, mac);

  if (named)
    {
      s7_pointer mac_slot;
      mac_name = caar(sc->code);

      if (((op == OP_DEFINE_EXPANSION) || (op == OP_DEFINE_EXPANSION_STAR)) &&
	  (!is_let(sc->curlet)))
	set_full_type(mac_name, T_EXPANSION | T_SYMBOL | (full_type(mac_name) & T_UNHEAP));

      /* symbol? macro name has already been checked, find name in let, and define it */
      mac_slot = symbol_to_local_slot(sc, mac_name, sc->curlet); /* returns global_slot(symbol) if sc->curlet == nil */
      if (is_slot(mac_slot))
	{
	  if ((sc->curlet == sc->nil) && (!in_rootlet(mac_slot)))
	    add_slot_to_rootlet(sc, mac_slot);
	  slot_set_value_with_hook(mac_slot, mac);
	}
      else s7_make_slot(sc, sc->curlet, mac_name, mac); /* was current but we've checked immutable already */
      if (tree_has_definers(sc, body))
	set_is_definer(mac_name);            /* (list-values 'define ...) aux-13 */
    }

  /* PERHAPS: if rest/dotted arg list, don't optimize */
  if ((!is_either_bacro(mac)) &&
      (optimize(sc, body, 1, collect_parameters(sc, closure_args(mac), sc->nil)) == OPT_OOPS))
    clear_all_optimizations(sc, body);

  if (sc->debug > 1) /* no profile here */
    closure_set_body(mac, add_trace(sc, body));

  unstack(sc);
  if (named)
    {
      set_pair_macro(closure_body(mac), mac_name);
      set_has_pair_macro(mac);
      if (has_location(car(sc->code)))
	{
	  pair_set_location(closure_body(mac), pair_location(car(sc->code)));
	  set_has_location(closure_body(mac));
	}}
  /* passed to maclet in apply_macro et al, copied in copy_closure */
  return(mac);
}

static s7_pointer make_closure_unchecked(s7_scheme *sc, s7_pointer args, s7_pointer code, uint64_t type, int32_t arity)
{
  s7_pointer x;
  new_cell_no_check(sc, x, (type | closure_bits(code)));
  closure_set_args(x, args);
  closure_set_let(x, sc->curlet);
  closure_set_setter(x, sc->F);
  closure_set_arity(x, arity);
  closure_set_body(x, code);
  if (is_pair(cdr(code))) set_closure_has_multiform(x); else set_closure_has_one_form(x);
  sc->capture_let_counter++;
  return(x);
}

static inline s7_pointer make_closure_gc_checked(s7_scheme *sc, s7_pointer args, s7_pointer code, uint64_t type, int32_t arity)
{
  s7_pointer x;
  new_cell(sc, x, (type | closure_bits(code)));
  closure_set_args(x, args);
  closure_set_let(x, sc->curlet);
  closure_set_setter(x, sc->F);
  closure_set_arity(x, arity);
  closure_set_body(x, code);
  if (is_pair(cdr(code))) set_closure_has_multiform(x); else set_closure_has_one_form(x);
  sc->capture_let_counter++;
  return(x);
}

static s7_pointer make_closure(s7_scheme *sc, s7_pointer args, s7_pointer code, uint64_t type, int32_t arity)
{
  /* this is called (almost?) every time a lambda form is evaluated, or during letrec, etc */
  s7_pointer x;
  new_cell(sc, x, (type | closure_bits(code)));
  closure_set_args(x, args);
  closure_set_let(x, sc->curlet);
  closure_set_setter(x, sc->F);
  closure_set_arity(x, arity);
  closure_set_body(x, code);           /* in case add_trace triggers GC, new func (x) needs some legit body for mark_closure */
  if (sc->debug_or_profile)
    {
      gc_protect_via_stack(sc, x);     /* GC protect func during add_trace */
      closure_set_body(x, (sc->debug > 1) ? add_trace(sc, code) : add_profile(sc, code));
      set_closure_has_multiform(x);
      unstack(sc);
    }
  else
    if (is_pair(cdr(code)))
      set_closure_has_multiform(x);
    else set_closure_has_one_form(x);
  sc->capture_let_counter++;
  return(x);
}

static int32_t closure_length(s7_scheme *sc, s7_pointer e)
{
  /* we can't use let_length(sc, closure_let(e)) because the closure_let(closure)
   *   changes.  So the open bit is not always on.  Besides, the fallbacks need to be for closures, not lets.
   */
  s7_pointer length_func = find_method(sc, closure_let(e), sc->length_symbol);
  if (length_func != sc->undefined)
    return((int32_t)s7_integer(s7_apply_function(sc, length_func, set_plist_1(sc, e))));
  /* there are cases where this should raise a wrong-type-arg error, but for now... */
  return(-1);
}

static s7_pointer cons_unchecked_with_type(s7_scheme *sc, s7_pointer p, s7_pointer a, s7_pointer b) /* (used only in copy_tree_with_type) */
{
  s7_pointer x;
  new_cell_no_check(sc, x, full_type(p) & (TYPE_MASK | T_IMMUTABLE | T_SAFE_PROCEDURE));
  set_car(x, a);
  set_cdr(x, b);
  return(x);
}

static s7_pointer copy_tree_with_type(s7_scheme *sc, s7_pointer tree)
{
  /* if sc->safety > NO_SAFETY, '(1 2) is set immutable by the reader, but eval (in that safety case) calls
   *   copy_body on the incoming tree, so we have to preserve T_IMMUTABLE in that case.
   * if tree is something like (+ 1 (car '#1=(2 . #1#))), we have to see the quoted list and not copy it.
   * Before getting here, we have checked that there is room for the entire tree (in copy_body), or 8192 cells (in list_values) in the free heap.
   */
#if WITH_GCC
  #define COPY_TREE_WITH_TYPE(P) ({s7_pointer _p; _p = P; \
                                   cons_unchecked_with_type(sc, _p, (is_unquoted_pair(car(_p))) ? copy_tree_with_type(sc, car(_p)) : car(_p), \
                                                                    (is_unquoted_pair(cdr(_p))) ? copy_tree_with_type(sc, cdr(_p)) : cdr(_p));})
#else
  #define COPY_TREE_WITH_TYPE(P) copy_tree_with_type(sc, P)
#endif
  return(cons_unchecked_with_type(sc, tree,
				  (is_unquoted_pair(car(tree))) ? COPY_TREE_WITH_TYPE(car(tree)) : car(tree),
				  (is_unquoted_pair(cdr(tree))) ? COPY_TREE_WITH_TYPE(cdr(tree)) : cdr(tree)));
}

static s7_pointer copy_tree(s7_scheme *sc, s7_pointer tree)
{
#if WITH_GCC
  #define COPY_TREE(P) ({s7_pointer _p; _p = P; \
                         cons_unchecked(sc, (is_unquoted_pair(car(_p))) ? copy_tree(sc, car(_p)) : car(_p), \
                                            (is_pair(cdr(_p))) ? copy_tree(sc, cdr(_p)) : cdr(_p));})
#else
  #define COPY_TREE(P) copy_tree(sc, P)
#endif
  return(cons_unchecked(sc,
			(is_unquoted_pair(car(tree))) ? COPY_TREE(car(tree)) : car(tree),
			(is_pair(cdr(tree))) ? COPY_TREE(cdr(tree)) : cdr(tree)));
}


/* -------------------------------- tree-cyclic? -------------------------------- */
#define TREE_NOT_CYCLIC 0
#define TREE_CYCLIC 1
#define TREE_HAS_PAIRS 2

static int32_t tree_is_cyclic_or_has_pairs(s7_scheme *sc, s7_pointer tree)
{
  s7_pointer fast = tree, slow = tree; /* we assume tree is a pair */
  bool has_pairs = false;
  while (true)
    {
      if (tree_is_collected(fast)) return(TREE_CYCLIC);
      if ((!has_pairs) && (is_unquoted_pair(car(fast))))
	has_pairs = true;
      fast = cdr(fast);
      if (!is_pair(fast))
	{
	  if (!has_pairs) return(TREE_NOT_CYCLIC);
	  break;
	}
      if (tree_is_collected(fast)) return(TREE_CYCLIC);
      if ((!has_pairs) && (is_unquoted_pair(car(fast))))
	has_pairs = true;
      fast = cdr(fast);
      if (!is_pair(fast))
	{
	  if (!has_pairs) return(TREE_NOT_CYCLIC);
	  break;
	}
      slow = cdr(slow);
      if (fast == slow) return(TREE_CYCLIC);
    }
  return(TREE_HAS_PAIRS);
}

/* we can't use shared_info here because tree_is_cyclic may be called in the midst of output that depends on sc->circle_info */

static bool tree_is_cyclic_1(s7_scheme *sc, s7_pointer tree)
{
  for (s7_pointer p = tree; is_pair(p); p = cdr(p))
    {
      tree_set_collected(p);
      if (sc->tree_pointers_top == sc->tree_pointers_size)
	{
	  if (sc->tree_pointers_size == 0)
	    {
	      sc->tree_pointers_size = 8;
	      sc->tree_pointers = (s7_pointer *)Malloc(sc->tree_pointers_size * sizeof(s7_pointer));
	    }
	  else
	    {
	      sc->tree_pointers_size *= 2;
	      sc->tree_pointers = (s7_pointer *)Realloc(sc->tree_pointers, sc->tree_pointers_size * sizeof(s7_pointer));
	    }}
      sc->tree_pointers[sc->tree_pointers_top++] = p;
      if (is_unquoted_pair(car(p)))
	{
	  int32_t old_top = sc->tree_pointers_top, result;
	  result = tree_is_cyclic_or_has_pairs(sc, car(p));
	  if ((result == TREE_CYCLIC) ||
	      (tree_is_cyclic_1(sc, car(p))))
	    return(true);
	  for (int32_t i = old_top; i < sc->tree_pointers_top; i++)
	    tree_clear_collected(sc->tree_pointers[i]);
	  sc->tree_pointers_top = old_top;
	}}
  return(false);
}

static bool tree_is_cyclic(s7_scheme *sc, s7_pointer tree)
{
  int32_t result;
  if (!is_pair(tree)) return(false);
  result = tree_is_cyclic_or_has_pairs(sc, tree);
  if (result == TREE_NOT_CYCLIC) return(false);
  if (result == TREE_CYCLIC) return(true);
  result = tree_is_cyclic_1(sc, tree);
  for (int32_t i = 0; i < sc->tree_pointers_top; i++)
    tree_clear_collected(sc->tree_pointers[i]);
  sc->tree_pointers_top = 0;
  return(result);
}

static s7_pointer g_tree_is_cyclic(s7_scheme *sc, s7_pointer args)
{
  #define H_tree_is_cyclic "(tree-cyclic? tree) returns #t if the tree has a cycle."
  #define Q_tree_is_cyclic sc->pl_bt
  return(make_boolean(sc, tree_is_cyclic(sc, car(args))));
}

static inline s7_int tree_len(s7_scheme *sc, s7_pointer p);

static s7_pointer copy_body(s7_scheme *sc, s7_pointer p)
{
  sc->w = p;
  if (!is_safety_checked(p))
    {
      if (tree_is_cyclic(sc, p))
	s7_error_nr(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "copy: tree is cyclic: ~S", 24), p));
      else set_safety_checked(p);
    }
  check_free_heap_size(sc, tree_len(sc, p) * 2);
  return((sc->safety > NO_SAFETY) ? copy_tree_with_type(sc, p) : copy_tree(sc, p));
}

static s7_pointer copy_closure(s7_scheme *sc, s7_pointer fnc)
{
  /* copy the source tree annotating (for eventual optimization), return a thing of the same type as fnc */
  s7_pointer x, body = copy_body(sc, closure_body(fnc));
  if ((is_any_macro(fnc)) && (has_pair_macro(fnc)))
    {
      set_pair_macro(body, pair_macro(closure_body(fnc)));
      set_has_pair_macro(fnc);
    }
  new_cell(sc, x, full_type(fnc) & (~T_COLLECTED)); /* I'm paranoid about that is_collected bit */
  closure_set_args(x, closure_args(fnc));
  closure_set_body(x, body);
  closure_set_setter(x, closure_setter(fnc));
  closure_set_arity(x, closure_arity(fnc));
  closure_set_let(x, closure_let(fnc));
  return(x);
}


/* -------------------------------- defined? -------------------------------- */
static s7_pointer g_is_defined(s7_scheme *sc, s7_pointer args)
{
  #define H_is_defined "(defined? symbol (let (curlet)) ignore-globals) returns #t if symbol has a binding (a value) in the let. \
Only the let is searched if ignore-globals is not #f."
#define Q_is_defined s7_make_signature(sc, 4, sc->is_boolean_symbol, sc->is_symbol_symbol, \
                       s7_make_signature(sc, 5, sc->is_let_symbol, sc->is_procedure_symbol, sc->is_macro_symbol, \
                                                sc->is_c_object_symbol, sc->is_c_pointer_symbol), sc->is_boolean_symbol)

  /* if the symbol has a global slot and e is unset or rootlet, this returns #t */
  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->is_defined_symbol, args, T_SYMBOL, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer e = cadr(args), b, x;
      if (!is_let(e))
	{
	  bool nil_is_rootlet = is_any_procedure(e); /* (defined? 'abs (lambda () 1)) -- unsure about this */
	  e = find_let(sc, e);
	  if ((is_null(e)) && (nil_is_rootlet))
	    e = sc->rootlet;
	  else
	    if (!is_let(e))
	      wrong_type_argument_with_type(sc, sc->is_defined_symbol, 2, cadr(args), a_let_string);
	}
      if (e == sc->s7_let)
	return(make_boolean(sc, symbol_s7_let(sym) != 0));
      if (is_pair(cddr(args)))
	{
	  b = caddr(args);
	  if (!s7_is_boolean(b))
	    return(method_or_bust_with_type(sc, b, sc->is_defined_symbol, args, a_boolean_string, 3));
	}
      else b = sc->F;
      if (e == sc->rootlet) /* we checked (let? e) above */
	{
	  if (b == sc->F)
	    return(make_boolean(sc, is_slot(global_slot(sym)))); /* new_symbol and gensym initialize global_slot to #<undefined> */
	  return(sc->F);
	}
      x = symbol_to_local_slot(sc, sym, e);
      if (is_slot(x))
	return(sc->T);
      return((b == sc->T) ? sc->F : make_boolean(sc, is_slot(global_slot(sym))));
    }
  return((is_global(sym)) ? sc->T : make_boolean(sc, is_slot(lookup_slot_from(sym, sc->curlet))));
}

static s7_pointer g_is_defined_in_rootlet(s7_scheme *sc, s7_pointer args)
{
  /* here we know arg2=(rootlet), and no arg3, arg1 is a symbol that needs to be looked-up */
  s7_pointer sym = lookup(sc, car(args));
  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->is_defined_symbol, args, T_SYMBOL, 1));
  return(make_boolean(sc, is_slot(global_slot(sym))));
}

static s7_pointer is_defined_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (!ops) return(f);
  if ((args == 2) && (is_symbol(cadr(expr))))
    {
      s7_pointer e = caddr(expr);
      if ((is_pair(e)) && (is_null(cdr(e))) && (car(e) == sc->rootlet_symbol))
	{
	  set_safe_optimize_op(expr, HOP_SAFE_C_NC);
	  return(sc->is_defined_in_rootlet);
	}}
  return(f);
}

bool s7_is_defined(s7_scheme *sc, const char *name)
{
  s7_pointer x = s7_symbol_table_find_name(sc, name);
  if (!x) return(false);
  x = lookup_slot_from(x, sc->curlet);
  return(is_slot(x));
}

static bool is_defined_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_symbol(p)) return(method_or_bust(sc, p, sc->is_defined_symbol, set_plist_1(sc, p), T_SYMBOL, 1) != sc->F);
  return(is_slot(lookup_slot_from(p, sc->curlet)));
}

static bool is_defined_b_7pp(s7_scheme *sc, s7_pointer p, s7_pointer e) {return(g_is_defined(sc, set_plist_2(sc, p, e)) != sc->F);}


void s7_define(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x;
  if ((let == sc->nil) || (let == sc->rootlet))
    let = sc->shadow_rootlet;
  x = symbol_to_local_slot(sc, symbol, let);
  if (is_slot(x))
    slot_set_value_with_hook(x, value);
  else
    {
      s7_make_slot(sc, let, symbol, value); /* I think this means C code can override "constant" defs */
      /* if let is sc->nil or rootlet, s7_make_slot makes a permanent_slot */
      if ((let == sc->shadow_rootlet) &&
	  (!is_slot(global_slot(symbol))))
	{
	  set_global(symbol); /* is_global => global_slot is usable -- is this a good idea? */
	  set_global_slot(symbol, local_slot(symbol));
	}}
}

s7_pointer s7_define_variable(s7_scheme *sc, const char *name, s7_pointer value)
{
  s7_pointer sym = make_symbol(sc, name);
  s7_define(sc, sc->nil, sym, value);
  return(sym);
}

s7_pointer s7_define_variable_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym = s7_define_variable(sc, name, value);
  symbol_set_has_help(sym);
  symbol_set_help(sym, copy_string(help));
  add_saved_pointer(sc, symbol_help(sym));
  return(sym);
}

s7_pointer s7_define_constant_with_environment(s7_scheme *sc, s7_pointer envir, const char *name, s7_pointer value)
{
  s7_pointer sym = make_symbol(sc, name);
  s7_define(sc, envir, sym, value);
  set_immutable(sym);
  set_possibly_constant(sym);
  set_immutable(global_slot(sym));
  set_immutable(local_slot(sym));
  return(sym);
}

s7_pointer s7_define_constant(s7_scheme *sc, const char *name, s7_pointer value)
{
  return(s7_define_constant_with_environment(sc, sc->nil, name, value));
}

/* (define (func a) (let ((cvar (+ a 1))) cvar)) (define-constant cvar 23) (func 1) -> ;can't bind an immutable object: cvar
 * (let ((aaa 1)) (define-constant aaa 32) (set! aaa 3)) -> set!: can't alter immutable object: aaa
 */

s7_pointer s7_define_constant_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym = s7_define_constant(sc, name, value);
  symbol_set_has_help(sym);
  symbol_set_help(sym, copy_string(help));
  add_saved_pointer(sc, symbol_help(sym));
  return(value); /* inconsistent with variable above, but consistent with define_function? */
}


/* -------------------------------- keyword? -------------------------------- */
bool s7_is_keyword(s7_pointer obj) {return(is_symbol_and_keyword(obj));}

static s7_pointer g_is_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_is_keyword "(keyword? obj) returns #t if obj is a keyword, (keyword? :rest) -> #t"
  #define Q_is_keyword sc->pl_bt
  check_boolean_method(sc, is_symbol_and_keyword, sc->is_keyword_symbol, args);
}


/* -------------------------------- string->keyword -------------------------------- */
s7_pointer s7_make_keyword(s7_scheme *sc, const char *key)
{
  s7_pointer sym;
  size_t slen = (size_t)safe_strlen(key);
  block_t *b = mallocate(sc, slen + 2);
  char *name = (char *)block_data(b);
  name[0] = ':';
  memcpy((void *)(name + 1), (void *)key, slen);
  name[slen + 1] = '\0';
  sym = make_symbol_with_length(sc, name, slen + 1); /* keyword slot etc taken care of here (in new_symbol actually) */
  liberate(sc, b);
  return(sym);
}

static s7_pointer g_string_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_keyword "(string->keyword str) prepends ':' to str and defines that as a keyword"
  #define Q_string_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_string_symbol)

  s7_pointer str = car(args);
  if (!is_string(str))
    return(method_or_bust_one_arg(sc, str, sc->string_to_keyword_symbol, args, T_STRING));
  if ((string_length(str) == 0) ||
      (string_value(str)[0] == '\0'))
    s7_error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "string->keyword wants a non-null string: ~S", 43), str));
  return(s7_make_keyword(sc, string_value(str)));
}


/* -------------------------------- keyword->symbol -------------------------------- */
static s7_pointer g_keyword_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_keyword_to_symbol "(keyword->symbol key) returns a symbol with the same name as key but no prepended colon"
  #define Q_keyword_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_keyword_symbol)

  s7_pointer sym = car(args);
  if (!is_symbol_and_keyword(sym))
    return(method_or_bust_with_type_one_arg_p(sc, sym, sc->keyword_to_symbol_symbol, wrap_string(sc, "a keyword", 9)));
  return(keyword_symbol(sym));
}

s7_pointer s7_keyword_to_symbol(s7_scheme *sc, s7_pointer key) {return(keyword_symbol(key));}


/* -------------------------------- symbol->keyword -------------------------------- */
#define symbol_to_keyword(Sc, Sym) s7_make_keyword(Sc, symbol_name(Sym))

static s7_pointer g_symbol_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_keyword "(symbol->keyword sym) returns a keyword with the same name as sym, but with a colon prepended"
  #define Q_symbol_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_symbol_symbol)

  if (!is_symbol(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->symbol_to_keyword_symbol, args, T_SYMBOL));
  return(symbol_to_keyword(sc, car(args)));
}


/* -------------------------------- c-pointer? -------------------------------- */
bool s7_is_c_pointer(s7_pointer arg) {return(is_c_pointer(arg));}

bool s7_is_c_pointer_of_type(s7_pointer arg, s7_pointer type) {return((is_c_pointer(arg)) && (c_pointer_type(arg) == type));}

static s7_pointer g_is_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_is_c_pointer "(c-pointer? obj type) returns #t if obj is a C pointer being held in s7.  If type is given, the c_pointer's type is also checked."
  #define Q_is_c_pointer s7_make_signature(sc, 3, sc->is_boolean_symbol, sc->T, sc->T)

  s7_pointer p = car(args);
  if (is_c_pointer(p))
    return((is_pair(cdr(args))) ? make_boolean(sc, c_pointer_type(p) == cadr(args)) : sc->T);
  if (!has_active_methods(sc, p)) return(sc->F);
  return(apply_boolean_method(sc, p, sc->is_c_pointer_symbol));
}


/* -------------------------------- c-pointer -------------------------------- */
void *s7_c_pointer(s7_pointer p) {return(c_pointer(p));}

void *s7_c_pointer_with_type(s7_scheme *sc, s7_pointer p, s7_pointer expected_type, const char *caller, s7_int argnum)
{
  if (!is_c_pointer(p))
    wrong_type_arg_error_prepackaged(sc, wrap_string(sc, caller, strlen(caller)),
					    make_integer(sc, argnum), p, sc->unused, sc->prepackaged_type_names[T_C_POINTER]);
  if ((c_pointer(p) != NULL) &&
      (c_pointer_type(p) != expected_type))
    s7_error_nr(sc, sc->wrong_type_arg_symbol,
		(argnum == 0) ?
		set_elist_4(sc, wrap_string(sc, "~S argument is a pointer of type ~S, but expected ~S", 52),
			    wrap_string(sc, caller, strlen(caller)), c_pointer_type(p), expected_type) :
		set_elist_5(sc, wrap_string(sc, "~S ~:D argument got a pointer of type ~S, but expected ~S", 56),
			    wrap_string(sc, caller, strlen(caller)),
			    make_integer(sc, argnum), c_pointer_type(p), expected_type));
  return(c_pointer(p));
}

s7_pointer s7_make_c_pointer_with_type(s7_scheme *sc, void *ptr, s7_pointer type, s7_pointer info)
{
  s7_pointer x;
  new_cell(sc, x, T_C_POINTER);
  c_pointer(x) = ptr;
  c_pointer_type(x) = type;
  c_pointer_info(x) = info;
  c_pointer_weak1(x) = sc->F;
  c_pointer_weak2(x) = sc->F;
  return(x);
}

s7_pointer s7_make_c_pointer(s7_scheme *sc, void *ptr) {return(s7_make_c_pointer_with_type(sc, ptr, sc->F, sc->F));}

static s7_pointer g_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer "(c-pointer int type info weak1 weak2) returns a c-pointer object. The type and info args are optional, defaulting to #f."
  #define Q_c_pointer s7_make_circular_signature(sc, 2, 3, sc->is_c_pointer_symbol, sc->is_integer_symbol, sc->T)

  s7_pointer arg = car(args), type = sc->F, info = sc->F, weak1 = sc->F, weak2 = sc->F, cp;
  intptr_t p;

  if (!s7_is_integer(arg))
    return(method_or_bust(sc, arg, sc->c_pointer_symbol, args, T_INTEGER, 1));
  p = (intptr_t)s7_integer_clamped_if_gmp(sc, arg);     /* (c-pointer (bignum "1234")) */
  args = cdr(args);
  if (is_pair(args))
    {
      type = car(args);
      args = cdr(args);
      if (is_pair(args))
	{
	  info = car(args);
	  args = cdr(args);
	  if (is_pair(args))
	    {
	      weak1 = car(args);
	      args = cdr(args);
	      if (is_pair(args))
		weak2 = car(args);
	    }}}
  cp = s7_make_c_pointer_with_type(sc, (void *)p, type, info);
  c_pointer_set_weak1(cp, weak1);
  c_pointer_set_weak2(cp, weak2);
  if ((weak1 != sc->F) || (weak2 != sc->F))
    add_weak_ref(sc, cp);
  return(cp);
}


/* -------------------------------- c-pointer-info -------------------------------- */
static s7_pointer c_pointer_info_p_p(s7_scheme *sc, s7_pointer p)
{
  if (!is_c_pointer(p))
    return(method_or_bust_p(sc, p, sc->c_pointer_info_symbol, T_C_POINTER));
  return(c_pointer_info(p));
}

static s7_pointer g_c_pointer_info(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_info "(c-pointer-info obj) returns the c-pointer info field"
  #define Q_c_pointer_info s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_info_p_p(sc, car(args)));
}


/* -------------------------------- c-pointer-type -------------------------------- */
static s7_pointer method_or_bust_lp(s7_scheme *sc, s7_pointer obj, s7_pointer method, uint8_t typ)
{ /* weird -- overhead goes berserk in callgrind if using the simpler method_or_bust_p! */
  if (!has_active_methods(sc, obj))
    wrong_type_argument(sc, method, 1, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_plist_1(sc, obj)));
}

s7_pointer s7_c_pointer_type(s7_pointer p) {return((is_c_pointer(p)) ? c_pointer_type(p) : NULL);}

static s7_pointer c_pointer_type_p_p(s7_scheme *sc, s7_pointer p)
{
  return((is_c_pointer(p)) ? c_pointer_type(p) : method_or_bust_lp(sc, p, sc->c_pointer_type_symbol, T_C_POINTER));
}

static s7_pointer g_c_pointer_type(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_type "(c-pointer-type obj) returns the c-pointer type field"
  #define Q_c_pointer_type s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_type_p_p(sc, car(args)));
}


/* -------------------------------- c-pointer-weak1/2 -------------------------------- */
static s7_pointer c_pointer_weak1_p_p(s7_scheme *sc, s7_pointer p)
{
  return((is_c_pointer(p)) ? c_pointer_weak1(p) : method_or_bust_lp(sc, p, sc->c_pointer_weak1_symbol, T_C_POINTER));
}

static s7_pointer g_c_pointer_weak1(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_weak1 "(c-pointer-weak1 obj) returns the c-pointer weak1 field"
  #define Q_c_pointer_weak1 s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_weak1_p_p(sc, car(args)));
}

static s7_pointer c_pointer_weak2_p_p(s7_scheme *sc, s7_pointer p)
{
  return((is_c_pointer(p)) ? c_pointer_weak2(p) : method_or_bust_lp(sc, p, sc->c_pointer_weak2_symbol, T_C_POINTER));
}

static s7_pointer g_c_pointer_weak2(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_weak2 "(c-pointer-weak2 obj) returns the c-pointer weak2 field"
  #define Q_c_pointer_weak2 s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_weak2_p_p(sc, car(args)));
}


/* -------------------------------- c-pointer->list -------------------------------- */
static s7_pointer g_c_pointer_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_to_list "(c-pointer->list obj) returns the c-pointer data as (list pointer-as-int type info)"
  #define Q_c_pointer_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_c_pointer_symbol)

  s7_pointer p = car(args);
  if (!is_c_pointer(p))
    return(method_or_bust(sc, p, sc->c_pointer_to_list_symbol, args, T_C_POINTER, 1));
  return(list_3(sc, make_integer(sc, (s7_int)((intptr_t)c_pointer(p))), c_pointer_type(p), c_pointer_info(p)));
}


/* -------------------------------- continuations and gotos -------------------------------- */

enum {NO_JUMP, CALL_WITH_EXIT_JUMP, THROW_JUMP, CATCH_JUMP, ERROR_JUMP, ERROR_QUIT_JUMP};
enum {NO_SET_JUMP, READ_SET_JUMP, LOAD_SET_JUMP, DYNAMIC_WIND_SET_JUMP, S7_CALL_SET_JUMP, EVAL_SET_JUMP};


/* ----------------------- continuation? -------------------------------- */
static s7_pointer g_is_continuation(s7_scheme *sc, s7_pointer args)
{
  #define H_is_continuation "(continuation? obj) returns #t if obj is a continuation"
  #define Q_is_continuation sc->pl_bt
  check_boolean_method(sc, is_continuation, sc->is_continuation_symbol, args);
  /* is this the right thing?  It returns #f for call-with-exit ("goto") because
   *   that form of continuation can't continue (via a jump back to its context).
   */
}

static bool is_continuation_b_p(s7_pointer p) {return(is_continuation(p));}

#if S7_DEBUGGING
static s7_pointer check_wrap_return(s7_pointer lst)
{
  for (s7_pointer fast = lst, slow = lst; is_pair(fast); slow = cdr(slow), fast = cdr(fast))
    {
      if (is_matched_pair(fast)) fprintf(stderr, "matched_pair not cleared\n");
      fast = cdr(fast);
      if (!is_pair(fast)) return(lst);
      if (fast == slow) return(lst);
      if (is_matched_pair(fast)) fprintf(stderr, "matched_pair not cleared\n");
    }
  return(lst);
}
#endif

static s7_pointer copy_any_list(s7_scheme *sc, s7_pointer a)
{
  s7_pointer slow = cdr(a);
  s7_pointer fast = slow;
  s7_pointer p;
#if S7_DEBUGGING
  #define wrap_return(W) do {fast = W; W = sc->nil; sc->y = sc->nil; return(check_wrap_return(fast));} while (0)
#else
  #define wrap_return(W) do {fast = W; W = sc->nil; sc->y = sc->nil; return(fast);} while (0)
#endif
  sc->y = a;
  sc->w = list_1(sc, car(a));
  p = sc->w;
  while (true)
    {
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    wrap_return(sc->w);
	  set_cdr(p, fast);
	  wrap_return(sc->w);
	}

      set_cdr(p, list_1(sc, car(fast)));
      p = cdr(p);

      fast = cdr(fast);
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    wrap_return(sc->w);
	  set_cdr(p, fast);
	  wrap_return(sc->w);
	}
      /* if unrolled further, it's a lot slower? */
      set_cdr(p, list_1_unchecked(sc, car(fast)));
      p = cdr(p);

      fast = cdr(fast);
      slow = cdr(slow);
      if (fast == slow)
	{
	  /* try to preserve the original cyclic structure */
	  s7_pointer p1, f1, p2, f2;
	  set_match_pair(a);
	  for (p1 = sc->w, f1 = a; !(is_matched_pair(cdr(f1))); f1 = cdr(f1), p1 = cdr(p1))
	    set_match_pair(f1);
	  for (p2 = sc->w, f2 = a; cdr(f1) != f2; f2 = cdr(f2), p2 = cdr(p2))
	    clear_match_pair(f2);
	  for (f1 = f2; is_pair(f1); f1 = cdr(f1), f2 = cdr(f2))
	    {
	      clear_match_pair(f1);
	      f1 = cdr(f1);
	      clear_match_pair(f1);
	      if (f1 == f2) break;
	    }
	  clear_match_pair(a);
	  if (is_null(p1))
	    set_cdr(p2, p2);
	  else set_cdr(p1, p2);
	  wrap_return(sc->w);
	}}
  wrap_return(sc->w);
}

static s7_pointer copy_counter(s7_scheme *sc, s7_pointer obj)
{
  s7_pointer nobj;
  new_cell(sc, nobj, T_COUNTER);
  counter_set_result(nobj, counter_result(obj));
  counter_set_list(nobj, counter_list(obj));
  counter_set_capture(nobj, counter_capture(obj));
  counter_set_let(nobj, counter_let(obj));
  counter_set_slots(nobj, counter_slots(obj));
  return(nobj);
}

static void copy_stack_list_set_immutable(s7_scheme *unused_sc, s7_pointer pold, s7_pointer pnew)
{
  for (s7_pointer p1 = pold, p2 = pnew, slow = pold; is_pair(p2); p1 = cdr(p1), p2 = cdr(p2))
    {
      if (is_immutable(p1)) set_immutable(p2);
      if (is_pair(cdr(p1)))
	{
	  p1 = cdr(p1);
	  p2 = cdr(p2);
	  if (is_immutable(p1)) set_immutable(p2);
	  if (p1 == slow) break;
	  slow = cdr(slow);
	}}
}

static s7_pointer copy_stack(s7_scheme *sc, s7_pointer new_v, s7_pointer old_v, int64_t top)
{
  bool has_pairs = false;
  s7_pointer *nv = stack_elements(new_v);
  s7_pointer *ov = stack_elements(old_v);
  memcpy((void *)nv, (void *)ov, top * sizeof(s7_pointer));
  stack_clear_flags(new_v);

  s7_gc_on(sc, false);
  if (stack_has_counters(old_v))
    {
      for (int64_t i = 2; i < top; i += 4)
	{
	  s7_pointer p = ov[i];                               /* args */
	  /* if op_gc_protect, any ov[i] (except op) can be a list, but it isn't the arglist, so it seems to be safe */
	  if (is_pair(p))                          /* args need not be a list (it can be a port or #f, etc) */
	    {
	      has_pairs = true;
	      if (is_null(cdr(p)))
		nv[i] = cons_unchecked(sc, car(p), sc->nil); /* GC is off -- could uncheck list_2 et al also */
	      else
		if ((is_pair(cdr(p))) && (is_null(cddr(p))))
		  nv[i] = list_2_unchecked(sc, car(p), cadr(p));
		else nv[i] = copy_any_list(sc, p);  /* args (copy is needed -- see s7test.scm) */
	      /* if op=eval_args4 for example, this has to be a proper list, and in many cases it doesn't need to be copied */
	      copy_stack_list_set_immutable(sc, p, nv[i]);
	    }
	  /* lst can be dotted or circular here.  The circular list only happens in a case like:
	   *    (dynamic-wind (lambda () (eq? (let ((lst (cons 1 2))) (set-cdr! lst lst) lst) (call/cc (lambda (k) k)))) (lambda () #f) (lambda () #f))
	   *    proper_list_reverse_in_place(sc->args) is one reason we need to copy, another reuse_as_let
	   */
	  else
	    if (is_counter(p))                  /* these can only occur in this context (not in a list etc) */
	      {
		stack_set_has_counters(new_v);
		nv[i] = copy_counter(sc, p);
	      }}}
  else
    for (int64_t i = 2; i < top; i += 4)
      if (is_pair(ov[i]))
	{
	  s7_pointer p = ov[i];
	  has_pairs = true;
	  if (is_null(cdr(p)))
	    nv[i] = cons_unchecked(sc, car(p), sc->nil);
	  else
	    if ((is_pair(cdr(p))) && (is_null(cddr(p))))
	      nv[i] = list_2_unchecked(sc, car(p), cadr(p));
	    else nv[i] = copy_any_list(sc, p);  /* args (copy is needed -- see s7test.scm) */
	  copy_stack_list_set_immutable(sc, p, nv[i]);
	}
  if (has_pairs) stack_set_has_pairs(new_v);
  s7_gc_on(sc, true);
  return(new_v);
}

static s7_pointer copy_op_stack(s7_scheme *sc)
{
  int32_t len = (int32_t)(sc->op_stack_now - sc->op_stack);
  s7_pointer nv = make_simple_vector(sc, len); /* not sc->op_stack_size */
  if (len > 0)
    {
      s7_pointer *src = sc->op_stack;
      s7_pointer *dst = (s7_pointer *)vector_elements(nv);
      for (int32_t i = len; i > 0; i--) *dst++ = *src++;
    }
  return(nv);
}

/* -------------------------------- with-baffle -------------------------------- */
/* (with-baffle . body) calls body guaranteeing that there can be no jumps into the
 *    middle of it from outside -- no outer evaluation of a continuation can jump across this
 *    barrier:  The flip-side of call-with-exit.
 */

static bool find_baffle(s7_scheme *sc, s7_int key)
{
  /* search backwards through sc->curlet for baffle_let with (continuation_)key as its baffle_key value */
  if (sc->baffle_ctr > 0)
    for (s7_pointer x = sc->curlet; is_let(x); x = let_outlet(x))
      if ((is_baffle_let(x)) &&
	  (let_baffle_key(x) == key))
	return(true);
  return(false);
}

#define NOT_BAFFLED -1

static s7_int find_any_baffle(s7_scheme *sc)
{
  /* search backwards through sc->curlet for any sc->baffle_symbol -- called by s7_make_continuation to set continuation_key */
  if (sc->baffle_ctr > 0)
    for (s7_pointer x = sc->curlet; is_let(x); x = let_outlet(x))
      if (is_baffle_let(x))
	return(let_baffle_key(x));
  return(NOT_BAFFLED);
}

static void check_with_baffle(s7_scheme *sc)
{
  if (!s7_is_proper_list(sc, sc->code))
    syntax_error(sc, "with-baffle: unexpected dot? ~A", 31, sc->code);
  pair_set_syntax_op(sc->code, OP_WITH_BAFFLE_UNCHECKED);
}

static bool op_with_baffle_unchecked(s7_scheme *sc)
{
  sc->code = cdr(sc->code);
  if (is_null(sc->code))
    {
      sc->value = sc->nil;
      return(true);
    }
  sc->curlet = make_let(sc, sc->curlet);
  set_baffle_let(sc->curlet);
  set_let_baffle_key(sc->curlet, sc->baffle_ctr++);
  return(false);
}


/* -------------------------------- call/cc -------------------------------- */
static void make_room_for_cc_stack(s7_scheme *sc)
{
  if ((int64_t)(sc->free_heap_top - sc->free_heap) < (int64_t)(sc->heap_size / 8)) /* we probably never need this much space -- very often we don't need any */
    {
      int64_t freed_heap = call_gc(sc);
      if (freed_heap < (int64_t)(sc->heap_size / 8))
	resize_heap(sc);
    }
}

s7_pointer s7_make_continuation(s7_scheme *sc)
{
  s7_pointer x, stack;
  int64_t loc;
  block_t *block;

  sc->continuation_counter++;
  make_room_for_cc_stack(sc);
  if (sc->continuation_counter > 2000) call_gc(sc); /* call_gc zeros cc counter, gc time up, but run time down -- try big cache */

  loc = current_stack_top(sc);
  stack = make_simple_vector(sc, loc);
  set_full_type(stack, T_STACK);
  temp_stack_top(stack) = loc;
  sc->temp7 = stack;
  copy_stack(sc, stack, sc->stack, loc);

  new_cell(sc, x, T_CONTINUATION);
  block = mallocate_block(sc);
  continuation_block(x) = block;
  continuation_set_stack(x, stack);
  continuation_stack_size(x) = vector_length(continuation_stack(x));
  continuation_stack_start(x) = stack_elements(continuation_stack(x));
  continuation_stack_end(x) = (s7_pointer *)(continuation_stack_start(x) + loc);
  continuation_op_stack(x) = copy_op_stack(sc);
  continuation_op_loc(x) = (int32_t)(sc->op_stack_now - sc->op_stack);
  continuation_op_size(x) = sc->op_stack_size;
  continuation_key(x) = find_any_baffle(sc);
  continuation_name(x) = sc->F;
  sc->temp7 = sc->nil;

  add_continuation(sc, x);
  return(x);
}

static void let_temp_done(s7_scheme *sc, s7_pointer args, s7_pointer let);
static void let_temp_unwind(s7_scheme *sc, s7_pointer slot, s7_pointer new_value);
static s7_pointer dynamic_unwind(s7_scheme *sc, s7_pointer func, s7_pointer e);
static s7_pointer eval(s7_scheme *sc, opcode_t first_op);

static bool check_for_dynamic_winds(s7_scheme *sc, s7_pointer c)
{
  /* called only from call_with_current_continuation.
   *   if call/cc jumps into a dynamic-wind, the init/finish funcs are wrapped in with-baffle
   *   so they'll complain.  Otherwise we're supposed to re-run the init func before diving
   *   into the body.  Similarly for let-temporarily.  If a call/cc jumps out of a dynamic-wind
   *   body-func, we're supposed to call the finish-func.  The continuation is called at
   *   current_stack_top(sc); the continuation form is at continuation_stack_top(c).
   *
   * check sc->stack for dynamic-winds we're jumping out of
   *    we need to check from the current stack top down to where the continuation stack matches the current stack??
   *    this was (i > 0), but that goes too far back; perhaps s7 should save the position of the call/cc invocation.
   *    also the two stacks can be different sizes (either can be larger)
   */
  int64_t top1 = current_stack_top(sc), top2 = continuation_stack_top(c);
  for (int64_t i = top1 - 1; (i > 0) && ((i >= top2) || (stack_code(sc->stack, i) != stack_code(continuation_stack(c), i))); i -= 4)
    {
      opcode_t op = stack_op(sc->stack, i);
      switch (op)
	{
	case OP_DYNAMIC_WIND:
	case OP_LET_TEMP_DONE:
	  {
	    s7_pointer x = stack_code(sc->stack, i);
	    int64_t s_base = 0;
	    for (int64_t j = 3; j < top2; j += 4)
	      if (((stack_op(continuation_stack(c), j) == OP_DYNAMIC_WIND) ||
		   (stack_op(continuation_stack(c), j) == OP_LET_TEMP_DONE)) &&
		  (x == stack_code(continuation_stack(c), j)))
		{
		  s_base = i;
		  break;
		}
	    if (s_base == 0)
	      {
		if (op == OP_DYNAMIC_WIND)
		  {
		    if (dynamic_wind_state(x) == DWIND_BODY)
		      {
			dynamic_wind_state(x) = DWIND_FINISH;
			if (dynamic_wind_out(x) != sc->F)
			  sc->value = s7_call(sc, dynamic_wind_out(x), sc->nil);
		      }}
		else let_temp_done(sc, stack_args(sc->stack, i), stack_let(sc->stack, i));
	      }}
	  break;

	case OP_DYNAMIC_UNWIND:
	case OP_DYNAMIC_UNWIND_PROFILE:
	  stack_element(sc->stack, i) = (s7_pointer)OP_GC_PROTECT;
	  break;

	case OP_LET_TEMP_UNWIND:
	  let_temp_unwind(sc, stack_code(sc->stack, i), stack_args(sc->stack, i));
	  break;

	case OP_LET_TEMP_S7_UNWIND:
	  g_s7_let_set_fallback(sc, set_plist_3(sc, sc->s7_let, stack_code(sc->stack, i), stack_args(sc->stack, i)));
	  break;

	case OP_BARRIER:
	  if (i > top2)                       /* otherwise it's some unproblematic outer eval-string? */
	    return(false);                    /*    but what if we've already evaluated a dynamic-wind closer? */
	  break;

	case OP_DEACTIVATE_GOTO:              /* here we're jumping out of an unrelated call-with-exit block */
	  if (i > top2)
	    call_exit_active(stack_args(sc->stack, i)) = false;
	  break;

	case OP_UNWIND_INPUT:
	  if (stack_args(sc->stack, i) != sc->unused)
	    set_current_input_port(sc, stack_args(sc->stack, i));         /* "args" = port that we shadowed */
	  break;

	case OP_UNWIND_OUTPUT:
	  if (stack_args(sc->stack, i) != sc->unused)
	    set_current_output_port(sc, stack_args(sc->stack, i));        /* "args" = port that we shadowed */
	  break;

	default:
	  break;
	}}

  /* check continuation-stack for dynamic-winds we're jumping into */
  for (int64_t i = current_stack_top(sc) - 1; i < top2; i += 4)
    {
      opcode_t op = stack_op(continuation_stack(c), i);
      if (op == OP_DYNAMIC_WIND)
	{
	  s7_pointer x = T_Dyn(stack_code(continuation_stack(c), i));
	  if (dynamic_wind_in(x) != sc->F)
	    sc->value = s7_call(sc, dynamic_wind_in(x), sc->nil);
	  dynamic_wind_state(x) = DWIND_BODY;
	}
      else
	if (op == OP_DEACTIVATE_GOTO)
	  call_exit_active(stack_args(continuation_stack(c), i)) = true;
      /* not let_temp_done here! */
      /* if op == OP_LET_TEMP_DONE, we're jumping back into a let-temporarily.  MIT and Chez scheme say they remember the
       *   let-temp vars (fluid-let or parameters in their terminology) at the point of the call/cc, and restore them
       *   on re-entry; that strikes me as incoherently complex -- they've wrapped a hidden dynamic-wind around the
       *   call/cc to restore all let-temp vars!  I think let-temp here should be the same as let -- if you jump back
       *   in, nothing hidden happens. So,
       *     (let ((x #f) (cc #f))
       *       (let-temporarily ((x 1))
       *         (set! x 2) (call/cc (lambda (r) (set! cc r))) (display x) (unless (= x 2) (newline) (exit)) (set! x 3) (cc)))
       *   behaves the same (in this regard) if let-temp is replaced with let.
       */
    }
  return(true);
}

static s7_pointer splice_in_values(s7_scheme *sc, s7_pointer args);

static void call_with_current_continuation(s7_scheme *sc)
{
  s7_pointer c = sc->code;  /* sc->args are the returned values */

  /* check for (baffle ...) blocking the current attempt to continue */
  if ((continuation_key(c) != NOT_BAFFLED) &&
      (!(find_baffle(sc, continuation_key(c)))))
    s7_error_nr(sc, sc->baffled_symbol,
		(is_symbol(continuation_name(sc->code))) ?
		set_elist_2(sc, wrap_string(sc, "continuation ~S can't jump into with-baffle", 43), continuation_name(sc->code)) :
		set_elist_1(sc, wrap_string(sc, "continuation can't jump into with-baffle", 40)));

  if (check_for_dynamic_winds(sc, c))
    {
      /* make_room_for_cc_stack(sc); */ /* 28-May-21 */
      /* we push_stack sc->code before calling an embedded eval above, so sc->code should still be c here, etc */
      if ((stack_has_pairs(continuation_stack(c))) ||
	  (stack_has_counters(continuation_stack(c))))
	{
	  make_room_for_cc_stack(sc);
	  copy_stack(sc, sc->stack, continuation_stack(c), continuation_stack_top(c));
	}
      else
	{
	  s7_pointer *nv = stack_elements(sc->stack);
	  s7_pointer *ov = stack_elements(continuation_stack(c));
	  memcpy((void *)nv, (void *)ov, continuation_stack_top(c) * sizeof(s7_pointer));
	}
      /* copy_stack(sc, sc->stack, continuation_stack(c), continuation_stack_top(c)); */
      sc->stack_end = (s7_pointer *)(sc->stack_start + continuation_stack_top(c));

      {
	int32_t top = continuation_op_loc(c);
	s7_pointer *src, *dst;
	sc->op_stack_now = (s7_pointer *)(sc->op_stack + top);
	sc->op_stack_size = continuation_op_size(c);
	sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
	src = (s7_pointer *)vector_elements(continuation_op_stack(c));
	dst = sc->op_stack;
	for (int32_t i = 0; i < top; i++) dst[i] = src[i];
      }
      sc->value = (is_null(sc->args)) ? sc->nil : ((is_null(cdr(sc->args))) ? car(sc->args) : splice_in_values(sc, sc->args));
    }
}

static s7_pointer g_call_cc(s7_scheme *sc, s7_pointer args)
{
  #define H_call_cc "(call-with-current-continuation (lambda (continuer)...)) is always a mistake!"
  #define Q_call_cc s7_make_signature(sc, 2, sc->T, sc->is_procedure_symbol)

  s7_pointer p = car(args);                  /* this is the procedure passed to call/cc */
  if (!is_t_procedure(p))                    /* this includes continuations */
    {
      check_method(sc, p, sc->call_cc_symbol, args);
      check_method(sc, p, sc->call_with_current_continuation_symbol, args);
      simple_wrong_type_argument_with_type(sc, sc->call_cc_symbol, p, a_procedure_string);
    }
  if (((!is_closure(p)) ||
       (closure_arity(p) != 1)) &&
      (!s7_is_aritable(sc, p, 1)))
    s7_error_nr(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "call/cc procedure, ~A, should take one argument", 47), p));

  sc->w = s7_make_continuation(sc);
  if ((is_any_closure(p)) && (is_pair(closure_args(p))) && (is_symbol(car(closure_args(p)))))
    continuation_name(sc->w) = car(closure_args(p));
  push_stack(sc, OP_APPLY, list_1_unchecked(sc, sc->w), p); /* apply function p to continuation sc->w */
  sc->w = sc->nil;
  return(sc->nil);
}

/* we can't naively optimize call/cc to call-with-exit if the continuation is only
 *   used as a function in the call/cc body because it might (for example) be wrapped
 *   in a lambda form that is being exported.  See b-func in s7test for an example.
 */

static void op_call_cc(s7_scheme *sc)
{
  sc->w = s7_make_continuation(sc);
  continuation_name(sc->w) = caar(opt2_pair(sc->code)); /* caadadr(sc->code) */
  sc->curlet = make_let_with_slot(sc, sc->curlet, continuation_name(sc->w), sc->w);
  sc->w = sc->nil;
  sc->code = cdr(opt2_pair(sc->code)); /* cddadr(sc->code) */
}

static bool op_implicit_continuation_a(s7_scheme *sc)
{
  s7_pointer code = sc->code; /* dumb-looking code, but it's faster than the pretty version, according to callgrind */
  s7_pointer s = lookup_checked(sc, car(code));
  if (!is_continuation(s)) {sc->last_function = s; return(false);}
  sc->code = s;
  sc->args = set_plist_1(sc, fx_call(sc, cdr(code)));
  call_with_current_continuation(sc);
  return(true);
}


/* -------------------------------- call-with-exit -------------------------------- */

static void pop_input_port(s7_scheme *sc);

static void call_with_exit(s7_scheme *sc)
{
  int64_t i, new_stack_top, quit = 0;

  if (!call_exit_active(sc->code))
    s7_error_nr(sc, sc->invalid_escape_function_symbol,
		set_elist_1(sc, wrap_string(sc, "call-with-exit escape procedure called outside its block", 56)));

  call_exit_active(sc->code) = false;
  new_stack_top = call_exit_goto_loc(sc->code);
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + call_exit_op_loc(sc->code));

  /* look for dynamic-wind in the stack section that we are jumping out of */
  i = current_stack_top(sc) - 1;
  do {
    switch (stack_op(sc->stack, i)) /* avoidable if we group these ops at the end and use op< */
      {
      case OP_DYNAMIC_WIND:
	{
	  s7_pointer lx = T_Dyn(stack_code(sc->stack, i));
	  if (dynamic_wind_state(lx) == DWIND_BODY)
	    {
	      dynamic_wind_state(lx) = DWIND_FINISH;
	      if (dynamic_wind_out(lx) != sc->F)
		{
		  s7_pointer arg = (sc->args == sc->plist_1) ? car(sc->plist_1) : sc->unused;  /* might also need GC protection here */
		  /* protect the sc->args value across this call if it is sc->plist_1 -- I can't find a broken case */
		  sc->value = s7_call(sc, dynamic_wind_out(lx), sc->nil);
		  if (arg != sc->unused) set_plist_1(sc, arg);
		}}}
	break;

      case OP_DYNAMIC_UNWIND:
      case OP_DYNAMIC_UNWIND_PROFILE:
	stack_element(sc->stack, i) = (s7_pointer)OP_GC_PROTECT;
	dynamic_unwind(sc, stack_code(sc->stack, i), stack_args(sc->stack, i));
	break;

      case OP_EVAL_STRING:
	s7_close_input_port(sc, current_input_port(sc));
	pop_input_port(sc);
	break;

      case OP_BARRIER:                /* oops -- we almost certainly went too far */
	goto SET_VALUE;

      case OP_DEACTIVATE_GOTO:        /* here we're jumping into an unrelated call-with-exit block */
	call_exit_active(stack_args(sc->stack, i)) = false;
	break;

      case OP_LET_TEMP_DONE:
	{
	  s7_pointer old_args = sc->args;
	  let_temp_done(sc, stack_args(sc->stack, i), stack_let(sc->stack, i));
	  sc->args = old_args;
	}
	break;

      case OP_LET_TEMP_UNWIND:
	let_temp_unwind(sc, stack_code(sc->stack, i), stack_args(sc->stack, i));
	break;

      case OP_LET_TEMP_S7_UNWIND:
	g_s7_let_set_fallback(sc, set_plist_3(sc, sc->s7_let, stack_code(sc->stack, i), stack_args(sc->stack, i)));
	break;

	/* call/cc does not close files, but I think call-with-exit should */
      case OP_GET_OUTPUT_STRING:
      case OP_UNWIND_OUTPUT:
	{
	  s7_pointer x = T_Prt(stack_code(sc->stack, i));  /* "code" = port that we opened */
	  s7_close_output_port(sc, x);
	  x = stack_args(sc->stack, i);                    /* "args" = port that we shadowed, if not #<unused> */
	  if (x != sc->unused)
	    set_current_output_port(sc, x);
	}
	break;

      case OP_UNWIND_INPUT:
	s7_close_input_port(sc, stack_code(sc->stack, i)); /* "code" = port that we opened */
	if (stack_args(sc->stack, i) != sc->unused)
	  set_current_input_port(sc, stack_args(sc->stack, i));       /* "args" = port that we shadowed */
	break;

      case OP_EVAL_DONE: /* goto called in a method -- put off the inner eval return(s) until we clean up the stack */
	quit++;
	break;

      default:
	break;
      }
    i -= 4;
  } while (i > new_stack_top);

 SET_VALUE:
  sc->stack_end = (s7_pointer *)(sc->stack_start + new_stack_top);

  /* the return value should have an implicit values call, just as in call/cc */
  sc->value = (is_null(sc->args)) ? sc->nil : ((is_null(cdr(sc->args))) ? car(sc->args) : splice_in_values(sc, sc->args));
  if (quit > 0)
    {
      if (sc->longjmp_ok)
	{
	  pop_stack(sc);
	  LongJmp(*(sc->goto_start), CALL_WITH_EXIT_JUMP);
	}
      for (i = 0; i < quit; i++)
	push_stack_op_let(sc, OP_EVAL_DONE);
    }
}

static s7_pointer g_is_goto(s7_scheme *sc, s7_pointer args)
{
  #define H_is_goto "(goto? obj) returns #t if obj is a call-with-exit exit function"
  #define Q_is_goto sc->pl_bt
  return(make_boolean(sc, is_goto(car(args))));
}

static inline s7_pointer make_goto(s7_scheme *sc, s7_pointer name)
{
  s7_pointer x;
  new_cell(sc, x, T_GOTO);
  call_exit_goto_loc(x) = current_stack_top(sc);
  call_exit_op_loc(x) = (int32_t)(sc->op_stack_now - sc->op_stack);
  call_exit_active(x) = true;
  call_exit_name(x) = name;
  return(x);
}

static s7_pointer g_call_with_exit(s7_scheme *sc, s7_pointer args)   /* (call-with-exit (lambda (return) ...)) */
{
  #define H_call_with_exit "(call-with-exit (lambda (exiter) ...)) is call/cc without the ability to jump back into a previous computation."
  #define Q_call_with_exit s7_make_signature(sc, 2, sc->values_symbol, sc->is_procedure_symbol)

  s7_pointer p = car(args), x;
  if (is_any_closure(p))
    {
      x = make_goto(sc, ((is_pair(closure_args(p))) && (is_symbol(car(closure_args(p))))) ? car(closure_args(p)) : sc->F);
      push_stack(sc, OP_DEACTIVATE_GOTO, x, p); /* this means call-with-exit is not tail-recursive */
      push_stack(sc, OP_APPLY, cons_unchecked(sc, x, sc->nil), p);
      return(sc->nil);
    }
  /* maybe just return an error here -- these gotos as args are stupid; also an error above if closure not aritable 1 */
  if (!is_t_procedure(p))
    return(method_or_bust_with_type_one_arg_p(sc, p, sc->call_with_exit_symbol, a_procedure_string));
  if (!s7_is_aritable(sc, p, 1))
    s7_error_nr(sc, sc->wrong_type_arg_symbol,
		set_elist_2(sc, wrap_string(sc, "call-with-exit argument should be a function of one argument: ~S", 64), p));
  x = make_goto(sc, sc->F);
  call_exit_active(x) = false;
  return((is_c_function(p)) ? c_function_call(p)(sc, set_plist_1(sc, x)) : s7_apply_function_star(sc, p, set_plist_1(sc, x)));
}

static inline void op_call_with_exit(s7_scheme *sc)
{
  s7_pointer go, args = opt2_pair(sc->code);
  go = make_goto(sc, caar(args));
  push_stack_no_let_no_code(sc, OP_DEACTIVATE_GOTO, go); /* was also pushing code */
  sc->curlet = make_let_with_slot(sc, sc->curlet, caar(args), go);
  sc->code = T_Pair(cdr(args));
}

static void op_call_with_exit_o(s7_scheme *sc)
{
  op_call_with_exit(sc);
  sc->code = car(sc->code);
}

static bool op_implicit_goto(s7_scheme *sc)
{
  s7_pointer g = lookup_checked(sc, car(sc->code));
  if (!is_goto(g)) {sc->last_function = g; return(false);}
  sc->args = sc->nil;
  sc->code = g;
  call_with_exit(sc);
  return(true);
}

static bool op_implicit_goto_a(s7_scheme *sc)
{
  s7_pointer g = lookup_checked(sc, car(sc->code));
  if (!is_goto(g)) {sc->last_function = g; return(false);}
  sc->args = set_plist_1(sc, fx_call(sc, cdr(sc->code)));
  sc->code = g;
  call_with_exit(sc);
  return(true);
}


/* -------------------------------- numbers -------------------------------- */

static block_t *string_to_block(s7_scheme *sc, const char *p, s7_int len)
{
  block_t *b = mallocate(sc, len + 1);
  char *bp = (char *)block_data(b);
  memcpy((void *)bp, (void *)p, len);
  bp[len] = '\0';
  return(b);
}

static s7_pointer block_to_string(s7_scheme *sc, block_t *block, s7_int len)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING | T_SAFE_PROCEDURE);
  string_block(x) = block;
  string_value(x) = (char *)block_data(block);
  string_length(x) = len;
  string_value(x)[len] = '\0';
  string_hash(x) = 0;
  add_string(sc, x);
  return(x);
}

static inline s7_pointer make_simple_ratio(s7_scheme *sc, s7_int num, s7_int den)
{
  s7_pointer x;
  if (den == 1)
    return(make_integer(sc, num));
  if (den == -1)
    return(make_integer(sc, -num));
  if ((den == S7_INT64_MIN) && ((num & 1) != 0))
    return(make_real(sc, (long_double)num / (long_double)den));
  new_cell(sc, x, T_RATIO);
  if (den < 0)
    {
      numerator(x) = -num;
      denominator(x) = -den;
    }
  else
    {
      numerator(x) = num;
      denominator(x) = den;
    }
  return(x);
}

static bool is_zero(s7_pointer x);
static bool is_positive(s7_scheme *sc, s7_pointer x);
static bool is_negative(s7_scheme *sc, s7_pointer x);
static s7_pointer make_ratio(s7_scheme *sc, s7_int a, s7_int b);

static bool is_NaN(s7_double x) {return(x != x);}
/* callgrind says this is faster than isnan, I think (very confusing data...) */

#if defined(__sun) && defined(__SVR4)
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));} /* there's no isinf in Solaris */
#else
#if (!MS_WINDOWS)
  #if __cplusplus
    #define is_inf(x) std::isinf(x)
  #else
    #define is_inf(x) isinf(x)
  #endif
#else
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));}  /* Another possibility: (x * 0) != 0 */

#if (_MSC_VER < 1700)
  /* in MS C, we need to provide inverse hyperbolic trig funcs and cbrt */
  static double asinh(double x) {return(log(x + sqrt(1.0 + x * x)));}
  static double acosh(double x) {return(log(x + sqrt(x * x - 1.0)));}
  /* perhaps less prone to numerical troubles (untested): 2.0 * log(sqrt(0.5 * (x + 1.0)) + sqrt(0.5 * (x - 1.0))) */
  static double atanh(double x) {return(log((1.0 + x) / (1.0 - x)) / 2.0);}
  static double cbrt(double x) {if (x >= 0.0) return(pow(x, 1.0 / 3.0)); return(-pow(-x, 1.0 / 3.0));}
#endif
#endif /* windows */
#endif /* not sun */


#if WITH_GMP
static mp_prec_t mpc_precision = DEFAULT_BIGNUM_PRECISION;
static mp_prec_t mpc_set_default_precision(mp_prec_t prec) {mpc_precision = prec; return(prec);}
#define mpc_init(Z) mpc_init2(Z, mpc_precision)

static bigint *alloc_bigint(s7_scheme *sc)
{
  bigint *p;
  if (sc->bigints)
    {
      p = sc->bigints;
      sc->bigints = p->nxt;
    }
  else
    {
      p = (bigint *)Malloc(sizeof(bigint));
      /* not permalloc here: gmp must be playing tricky games with realloc or something.  permalloc can lead
       *   to mpz_set_si overwriting adjacent memory (valgrind does not catch this), clobbering at least the
       *   bigint nxt field.  Someday I need to look at the source.
       */
      mpz_init(p->n);
    }
  return(p);
}

static bigrat *alloc_bigrat(s7_scheme *sc)
{
  bigrat *p;
  if (sc->bigrats)
    {
      p = sc->bigrats;
      sc->bigrats = p->nxt;
    }
  else
    {
      p = (bigrat *)Malloc(sizeof(bigrat));
      mpq_init(p->q);
    }
  return(p);
}

static bigflt *alloc_bigflt(s7_scheme *sc)
{
  bigflt *p;
  if (sc->bigflts)
    {
      p = sc->bigflts;
      sc->bigflts = p->nxt;
      mpfr_set_prec(p->x, sc->bignum_precision);
    }
  else
    {
      p = (bigflt *)Malloc(sizeof(bigflt));
      mpfr_init2(p->x, sc->bignum_precision);
    }
  return(p);
}

static bigcmp *alloc_bigcmp(s7_scheme *sc)
{
  bigcmp *p;
  if (sc->bigcmps)
    {
      p = sc->bigcmps;
      sc->bigcmps = p->nxt;
      mpc_set_prec(p->z, sc->bignum_precision);
    }
  else
    {
      p = (bigcmp *)Malloc(sizeof(bigcmp));
      mpc_init(p->z);
    }
  return(p);
}

static s7_pointer mpz_to_big_integer(s7_scheme *sc, mpz_t val)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_INTEGER);
  big_integer_bgi(x) = alloc_bigint(sc);
  mpz_set(big_integer(x), val);
  add_big_integer(sc, x);
  return(x);
}

static s7_pointer mpz_to_integer(s7_scheme *sc, mpz_t val)
{
  if (mpz_fits_slong_p(val))
    return(make_integer(sc, mpz_get_si(val)));
  return(mpz_to_big_integer(sc, val));
}

#if (!WITH_PURE_S7)
static s7_pointer mpz_to_big_real(s7_scheme *sc, mpz_t val)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_REAL);
  big_real_bgf(x) = alloc_bigflt(sc);
  add_big_real(sc, x);
  mpfr_set_z(big_real(x), val, MPFR_RNDN);
  return(x);
}
#endif

static s7_pointer mpq_to_big_ratio(s7_scheme *sc, mpq_t val)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_RATIO);
  big_ratio_bgr(x) = alloc_bigrat(sc);
  add_big_ratio(sc, x);
  mpq_set(big_ratio(x), val);
  return(x);
}

static s7_pointer mpq_to_rational(s7_scheme *sc, mpq_t val)
{
  if (mpz_cmp_ui(mpq_denref(val), 1) == 0)
    return(mpz_to_integer(sc, mpq_numref(val)));
#if S7_DEBUGGING
  mpq_canonicalize(val);
  if (mpz_cmp_ui(mpq_denref(val), 1) == 0)
    {
      fprintf(stderr, "mpq_to_rational: missing canonicalize\n");
      return(mpz_to_integer(sc, mpq_numref(val)));
    }
#endif
 if ((mpz_fits_slong_p(mpq_numref(val))) && (mpz_fits_slong_p(mpq_denref(val))))
    return(make_simple_ratio(sc, mpz_get_si(mpq_numref(val)), mpz_get_si(mpq_denref(val))));
  return(mpq_to_big_ratio(sc, val));
}

static s7_pointer mpq_to_canonicalized_rational(s7_scheme *sc, mpq_t mpq)
{
  mpq_canonicalize(mpq);
  return(mpq_to_rational(sc, mpq));
}

static s7_pointer mpz_to_rational(s7_scheme *sc, mpz_t n, mpz_t d) /* mpz_3 and mpz_4 */
{
  if (mpz_cmp_ui(d, 1) == 0)
    return(mpz_to_integer(sc, n));
  mpq_set_num(sc->mpq_1, n);
  mpq_set_den(sc->mpq_1, d);
  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
}

#if (!WITH_PURE_S7)
static s7_pointer mpq_to_big_real(s7_scheme *sc, mpq_t val)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_REAL);
  big_real_bgf(x) = alloc_bigflt(sc);
  add_big_real(sc, x);
  mpfr_set_q(big_real(x), val, MPFR_RNDN);
  return(x);
}
#endif

static s7_pointer any_rational_to_mpq(s7_scheme *sc, s7_pointer z, mpq_t bigq)
{
  switch (type(z))
    {
    case T_INTEGER:     mpq_set_si(bigq, integer(z), 1);                break;
    case T_BIG_INTEGER: mpq_set_z(bigq, big_integer(z));                break;
    case T_RATIO:       mpq_set_si(bigq, numerator(z), denominator(z)); break;
    case T_BIG_RATIO:   mpq_set(bigq, big_ratio(z));                    break;
    }
  return(z);
}

static s7_pointer mpfr_to_integer(s7_scheme *sc, mpfr_t val)
{
  mpfr_get_z(sc->mpz_4, val, MPFR_RNDN);
  return(mpz_to_integer(sc, sc->mpz_4));
}

static s7_pointer mpfr_to_big_real(s7_scheme *sc, mpfr_t val)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_REAL);
  add_big_real(sc, x);
  big_real_bgf(x) = alloc_bigflt(sc);
  mpfr_set(big_real(x), val, MPFR_RNDN);
  return(x);
}

static s7_pointer mpc_to_number(s7_scheme *sc, mpc_t val)
{
  s7_pointer x;
  if (mpfr_zero_p(mpc_imagref(val)))
    return(mpfr_to_big_real(sc, mpc_realref(val)));
  new_cell(sc, x, T_BIG_COMPLEX);
  big_complex_bgc(x) = alloc_bigcmp(sc);
  add_big_complex(sc, x);
  mpc_set(big_complex(x), val, MPC_RNDNN);
  return(x);
}

/* s7.h */
mpz_t  *s7_big_integer(s7_pointer x) {return(&big_integer(x));}
mpq_t  *s7_big_ratio(s7_pointer x)   {return(&big_ratio(x));}
mpfr_t *s7_big_real(s7_pointer x)    {return(&big_real(x));}
mpc_t  *s7_big_complex(s7_pointer x) {return(&big_complex(x));}

bool s7_is_big_integer(s7_pointer x) {return(is_t_big_integer(x));}
bool s7_is_big_ratio(s7_pointer x)   {return(is_t_big_ratio(x));}
bool s7_is_big_real(s7_pointer x)    {return(is_t_big_real(x));}
bool s7_is_big_complex(s7_pointer x) {return(is_t_big_complex(x));}

s7_pointer s7_make_big_integer(s7_scheme *sc, mpz_t *val) {return(mpz_to_integer(sc, *val));}
s7_pointer s7_make_big_ratio(s7_scheme *sc, mpq_t *val)   {return(mpq_to_rational(sc, *val));}
s7_pointer s7_make_big_real(s7_scheme *sc, mpfr_t *val)   {return(mpfr_to_big_real(sc, *val));}
s7_pointer s7_make_big_complex(s7_scheme *sc, mpc_t *val) {return(mpc_to_number(sc, *val));}

#if (!WITH_PURE_S7)
static s7_pointer big_integer_to_big_real(s7_scheme *sc, s7_pointer x) {return(mpz_to_big_real(sc, big_integer(x)));}
static s7_pointer big_ratio_to_big_real(s7_scheme *sc, s7_pointer x)   {return(mpq_to_big_real(sc, big_ratio(x)));}
#endif

static s7_pointer s7_int_to_big_integer(s7_scheme *sc, s7_int val)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_INTEGER);
  big_integer_bgi(x) = alloc_bigint(sc);
  mpz_set_si(big_integer(x), val);
  add_big_integer(sc, x);
  return(x);
}

static s7_pointer s7_int_to_big_ratio(s7_scheme *sc, s7_int num, s7_int den)
{
  /* (called only in g_bignum), den here always comes from denominator(x) or some positive constant so it is not negative */
  s7_pointer x;
  new_cell(sc, x, T_BIG_RATIO);
  big_ratio_bgr(x) = alloc_bigrat(sc);
  add_big_ratio(sc, x);
  mpq_set_si(big_ratio(x), num, den);
  return(x);
}

static s7_pointer s7_double_to_big_real(s7_scheme *sc, s7_double rl)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_REAL);
  big_real_bgf(x) = alloc_bigflt(sc);
  add_big_real(sc, x);
  mpfr_set_d(big_real(x), rl, MPFR_RNDN);
  return(x);
}

static s7_pointer s7_double_to_big_complex(s7_scheme *sc, s7_double rl, s7_double im)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_COMPLEX);
  add_big_complex(sc, x);
  big_complex_bgc(x) = alloc_bigcmp(sc);
  mpc_set_d_d(big_complex(x), rl, im, MPC_RNDNN);
  return(x);
}

static s7_pointer big_pi(s7_scheme *sc)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_REAL | T_IMMUTABLE);
  big_real_bgf(x) = alloc_bigflt(sc);
  add_big_real(sc, x);
  mpfr_const_pi(big_real(x), MPFR_RNDN);
  return(x);
}

static bool is_integer_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_integer(p))
    return(true);
  if (has_active_methods(sc, p))
    {
      s7_pointer f = find_method_with_let(sc, p, sc->is_integer_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, set_plist_1(sc, p))));
    }
  return(false);
}

#if (!WITH_PURE_S7)
static s7_pointer s7_number_to_big_real(s7_scheme *sc, s7_pointer p)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_REAL);
  big_real_bgf(x) = alloc_bigflt(sc);
  add_big_real(sc, x);

  switch (type(p))
    {
    case T_INTEGER:
      mpfr_set_si(big_real(x), integer(p), MPFR_RNDN);
      break;
    case T_RATIO:
      /* here we can't use fraction(number(p)) even though that uses long_double division because
       *   there are lots of int64_t ratios that will still look the same.
       *   We have to do the actual bignum divide by hand.
       */
      mpq_set_si(sc->mpq_1, numerator(p), denominator(p));
      mpfr_set_q(big_real(x), sc->mpq_1, MPFR_RNDN);
      break;
    default:
      mpfr_set_d(big_real(x), s7_real(p), MPFR_RNDN);
      break;
    }
  return(x);
}
#endif

static s7_pointer s7_number_to_big_complex(s7_scheme *sc, s7_pointer p)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_COMPLEX);
  big_complex_bgc(x) = alloc_bigcmp(sc);
  add_big_complex(sc, x);

  switch (type(p))
    {
    case T_INTEGER:
      mpc_set_si(big_complex(x), integer(p), MPC_RNDNN);
      break;
    case T_RATIO:
      /* can't use fraction here */
      mpq_set_si(sc->mpq_1, numerator(p), denominator(p));
      mpfr_set_q(sc->mpfr_1, sc->mpq_1, MPFR_RNDN);
      mpc_set_fr(big_complex(x), sc->mpfr_1, MPC_RNDNN);
      break;
    case T_REAL:
      mpc_set_d(big_complex(x), s7_real(p), MPC_RNDNN);
      break;
    default:
      mpc_set_d_d(big_complex(x), real_part(p), imag_part(p), MPC_RNDNN);
      break;
    }
  return(x);
}

static s7_pointer any_real_to_mpfr(s7_scheme *sc, s7_pointer p, mpfr_t bigx)
{
  switch (type(p))
    {
    case T_INTEGER:
      mpfr_set_si(bigx, integer(p), MPFR_RNDN);
      break;
    case T_RATIO:
      mpq_set_si(sc->mpq_1, numerator(p), denominator(p));
      mpfr_set_q(bigx, sc->mpq_1, MPFR_RNDN);
      break;
    case T_REAL:
      mpfr_set_d(bigx, real(p), MPFR_RNDN);
      if (is_NaN(real(p))) return(real_NaN);
      if (is_inf(real(p))) return(real_infinity);
      break;
    case T_BIG_INTEGER:
      mpfr_set_z(bigx, big_integer(p), MPFR_RNDN);
      break;
    case T_BIG_RATIO:
      mpfr_set_q(bigx, big_ratio(p), MPFR_RNDN);
      break;
    case T_BIG_REAL:
      mpfr_set(bigx, big_real(p), MPFR_RNDN);
      if (mpfr_nan_p(big_real(p))) return(real_NaN);
      if (mpfr_inf_p(big_real(p))) return(real_infinity);
      break;
    }
  return(NULL);
}

#define mpc_zero_p(z) ((mpfr_zero_p(mpc_realref(z))) && (mpfr_zero_p(mpc_imagref(z))))

static s7_pointer any_number_to_mpc(s7_scheme *sc, s7_pointer p, mpc_t bigz)
{
  switch (type(p))
    {
    case T_INTEGER:
      mpc_set_si(bigz, integer(p), MPC_RNDNN);
      break;
    case T_RATIO:
      mpq_set_si(sc->mpq_1, numerator(p), denominator(p));
      mpc_set_q(bigz, sc->mpq_1, MPC_RNDNN);
      break;
    case T_REAL:
      if (is_NaN(real(p))) return(real_NaN);
      if (is_inf(real(p))) return(real_infinity);
      mpc_set_d(bigz, real(p), MPC_RNDNN);
      break;
    case T_COMPLEX:
      if (is_NaN(imag_part(p))) return(complex_NaN);
      if (is_NaN(real_part(p))) return(real_NaN);
     mpc_set_d_d(bigz, real_part(p), imag_part(p), MPC_RNDNN);
      break;
    case T_BIG_INTEGER:
      mpc_set_z(bigz, big_integer(p), MPC_RNDNN);
      break;
    case T_BIG_RATIO:
      mpc_set_q(bigz, big_ratio(p), MPC_RNDNN);
      break;
    case T_BIG_REAL:
      mpc_set_fr(bigz, big_real(p), MPC_RNDNN);
      if (mpfr_nan_p(big_real(p))) return(real_NaN);
      if (mpfr_inf_p(big_real(p))) return(real_infinity);
      break;
    case T_BIG_COMPLEX:
      if (mpfr_nan_p(mpc_imagref(big_complex(p)))) return(complex_NaN);
      if (mpfr_nan_p(mpc_realref(big_complex(p)))) return(real_NaN);
      mpc_set(bigz, big_complex(p), MPC_RNDNN);
      break;
    }
  return(NULL);
}

static s7_pointer make_big_complex(s7_scheme *sc, mpfr_t rl, mpfr_t im)
{
  /* there is no mpc_get_str equivalent, so we need to split up str, use make_big_real to get the 2 halves, then mpc_init, then mpc_set_fr_fr */
  s7_pointer x;
  new_cell(sc, x, T_BIG_COMPLEX);
  big_complex_bgc(x) = alloc_bigcmp(sc);
  add_big_complex(sc, x);
  mpc_set_fr_fr(big_complex(x), rl ,im, MPC_RNDNN);
  return(x);
}

static block_t *mpfr_to_string(s7_scheme *sc, mpfr_t val, int32_t radix)
{
  char *str;
  mp_exp_t expptr;
  int32_t ep;
  s7_int i, len;
  block_t *b, *btmp;

  if (mpfr_zero_p(val))
    return(string_to_block(sc, "0.0", 3));
  if (mpfr_nan_p(val))
    return(string_to_block(sc, "+nan.0", 6));
  if (mpfr_inf_p(val))
    return((mpfr_signbit(val) == 0) ? string_to_block(sc, "+inf.0", 6) : string_to_block(sc, "-inf.0", 6));

  b = callocate(sc, sc->bignum_precision + 32);
  str = mpfr_get_str((char *)block_data(b), &expptr, radix, 0, val, MPFR_RNDN);
  ep = (int32_t)expptr;
  len = safe_strlen(str);

  /* remove trailing 0's */
  for (i = len - 1; i > 3; i--)
    if (str[i] != '0')
      break;
  if (i < len - 1)
    str[i + 1] = '\0';

  btmp = mallocate(sc, len + 64);
  if (str[0] == '-')
    snprintf((char *)block_data(btmp), len + 64, "-%c.%s%c%d", str[1], (char *)(str + 2), (radix <= 10) ? 'E' : '@', ep - 1);
  else snprintf((char *)block_data(btmp), len + 64, "%c.%s%c%d", str[0], (char *)(str + 1), (radix <= 10) ? 'E' : '@', ep - 1);

  liberate(sc, b);
  return(btmp);
}

static block_t *mpc_to_string(s7_scheme *sc, mpc_t val, int32_t radix, use_write_t use_write)
{
  block_t *rl, *im, *tmp;
  s7_int len;

  mpc_real(sc->mpfr_1, val, MPFR_RNDN);
  rl = mpfr_to_string(sc, sc->mpfr_1, radix);
  mpc_imag(sc->mpfr_2, val, MPFR_RNDN);
  im = mpfr_to_string(sc, sc->mpfr_2, radix);

  len = safe_strlen((char *)block_data(rl)) + safe_strlen((char *)block_data(im)) + 128;
  tmp = mallocate(sc, len);
  snprintf((char *)block_data(tmp), len, "%s%s%si",
	   (char *)block_data(rl),
	   ((((char *)block_data(im))[0] == '-') || (((char *)block_data(im))[0] == '+')) ? "" : "+", (char *)block_data(im));

  liberate(sc, rl);
  liberate(sc, im);
  return(tmp);
}

static block_t *big_number_to_string_with_radix(s7_scheme *sc, s7_pointer p, int32_t radix, s7_int width, s7_int *nlen, use_write_t use_write)
{
  block_t *str;
  switch (type(p))
    {
    case T_BIG_INTEGER:
      str = callocate(sc, mpz_sizeinbase(big_integer(p), radix) + 64);
      mpz_get_str((char *)block_data(str), radix, big_integer(p));
      break;
    case T_BIG_RATIO:
      mpz_set(sc->mpz_1, mpq_numref(big_ratio(p)));
      mpz_set(sc->mpz_2, mpq_denref(big_ratio(p)));
      str = callocate(sc, mpz_sizeinbase(sc->mpz_1, radix) + mpz_sizeinbase(sc->mpz_2, radix) + 64);
      mpq_get_str((char *)block_data(str), radix, big_ratio(p));
      break;
    case T_BIG_REAL:
      str = mpfr_to_string(sc, big_real(p), radix);
      break;
    default:
      str = mpc_to_string(sc, big_complex(p), radix, use_write);
      break;
    }
  if (width > 0)
    {
      s7_int len = safe_strlen((char *)block_data(str));
      if (width > len)
	{
	  int32_t spaces = width - len;
	  block_t *tmp = (block_t *)mallocate(sc, width + 1);
	  ((char *)block_data(tmp))[width] = '\0';
	  memmove((void *)((char *)block_data(tmp) + spaces), (void *)block_data(str), len);
	  local_memset((void *)block_data(tmp), (int)' ', spaces);
	  (*nlen) = width;
	  liberate(sc, str);
	  return(tmp);
	}
      (*nlen) = len;
    }
  else (*nlen) = safe_strlen((char *)block_data(str));
  return(str);
}

static s7_pointer string_to_big_integer(s7_scheme *sc, const char *str, int32_t radix)
{
  mpz_set_str(sc->mpz_4, (str[0] == '+') ? (const char *)(str + 1) : str, radix);
  return(mpz_to_integer(sc, sc->mpz_4));
}

static s7_pointer string_to_big_ratio(s7_scheme *sc, const char *str, int32_t radix)
{
  s7_pointer x;
  mpq_set_str(sc->mpq_1, str, radix);
  mpq_canonicalize(sc->mpq_1);
  if (mpz_cmp_ui(mpq_denref(sc->mpq_1), 1) == 0)
    return(mpz_to_integer(sc, mpq_numref(sc->mpq_1)));
  new_cell(sc, x, T_BIG_RATIO);
  big_ratio_bgr(x) = alloc_bigrat(sc);
  add_big_ratio(sc, x);
  mpq_set(big_ratio(x), sc->mpq_1);
  return(x);
}

static s7_pointer string_to_big_real(s7_scheme *sc, const char *str, int32_t radix)
{
  s7_pointer x;
  new_cell(sc, x, T_BIG_REAL);
  big_real_bgf(x) = alloc_bigflt(sc);
  add_big_real(sc, x);
  mpfr_set_str(big_real(x), str, radix, MPFR_RNDN);
  return(x);
}

static s7_int string_to_integer(const char *str, int32_t radix, bool *overflow);

static s7_pointer string_to_either_integer(s7_scheme *sc, const char *str, int32_t radix)
{
  bool overflow = false;
  s7_int val = string_to_integer(str, radix, &overflow);
  if (!overflow)
    return(make_integer(sc, val));
  return(string_to_big_integer(sc, str, radix));
}

static s7_pointer string_to_either_ratio(s7_scheme *sc, const char *nstr, const char *dstr, int32_t radix)
{
  bool overflow = false;
  /* gmp segfaults if passed a bignum/0 so this needs to check first that
   *   the denominator is not 0 before letting gmp screw up.  Also, if the
   *   first character is '+', gmp returns 0!
   */
  s7_int d = string_to_integer(dstr, radix, &overflow);
  if (!overflow)
    {
      s7_int n;
      if (d == 0)
	return(real_NaN);

      n = string_to_integer(nstr, radix, &overflow);
      if (!overflow)
	return(make_ratio(sc, n, d));
    }
  if (nstr[0] == '+')
    return(string_to_big_ratio(sc, (const char *)(nstr + 1), radix));
  return(string_to_big_ratio(sc, nstr, radix));
}

static s7_double string_to_double_with_radix(const char *ur_str, int32_t radix, bool *overflow);
static s7_pointer string_to_either_real(s7_scheme *sc, const char *str, int32_t radix)
{
  bool overflow = false;
  s7_double val = string_to_double_with_radix((char *)str, radix, &overflow);
  if (!overflow)
    return(make_real(sc, val));
  return(string_to_big_real(sc, str, radix));
}

static s7_pointer string_to_either_complex_1(s7_scheme *sc, char *q, char *slash1, char *ex1, bool has_dec_point1, int32_t radix, s7_double *d_rl)
{
  bool overflow = false;
  /* there's a real problem here -- we don't want to promote s7_double .1 to a bignum because
   *    its low order digits are garbage, causing (rationalize .1 0) to return 3602879701896397/36028797018963968
   *    no matter what the bignum-precision.  But we can't just fallback on gmp's reader because (for example)
   *    it reads 1/2+i or 1+0/0i as 1.0.  Also format gets screwed up.  And string->number signals an error
   *    where it should return #f.  I wonder what to do.
   */
  if ((has_dec_point1) ||
      (ex1))
    {
      (*d_rl) = string_to_double_with_radix(q, radix, &overflow);
      if (overflow)
	return(string_to_big_real(sc, q, radix));
    }
  else
    {
      if (slash1)
	{
	  s7_int n, d;
	  /* q can include the slash and denominator */
	  n = string_to_integer(q, radix, &overflow);
	  if (overflow)
	    return(string_to_big_ratio(sc, q, radix));
	  d = string_to_integer(slash1, radix, &overflow);
	  if (overflow) return(string_to_big_ratio(sc, q, radix));
	  (*d_rl) = (s7_double)n / (s7_double)d;
	}
      else
	{
	  s7_int val = string_to_integer(q, radix, &overflow);
	  if (overflow)
	    return(string_to_big_integer(sc, q, radix));
	  (*d_rl) = (s7_double)val;
	}}
  if ((*d_rl) == -0.0) (*d_rl) = 0.0;
  return(NULL);
}

static s7_pointer string_to_either_complex(s7_scheme *sc, char *q, char *slash1, char *ex1, bool has_dec_point1,
					   char *plus, char *slash2, char *ex2, bool has_dec_point2,
					   int32_t radix, int32_t has_plus_or_minus)
{
  /* this can be just about anything involving 2 real/ratio/int portions, +/- in between and 'i' at the end */
  double d_rl = 0.0, d_im = 0.0;
  s7_pointer p_rl = string_to_either_complex_1(sc, q, slash1, ex1, has_dec_point1, radix, &d_rl);
  s7_pointer p_im = string_to_either_complex_1(sc, plus, slash2, ex2, has_dec_point2, radix, &d_im);

  if ((d_im == 0.0) &&                     /* 1.0+0.0000000000000000000000000000i */
      ((!p_im) || (is_zero(p_im))))
    return((p_rl) ? p_rl : make_real(sc, d_rl));

  if ((!p_rl) && (!p_im))
    return(s7_make_complex(sc, d_rl, (has_plus_or_minus == -1) ? (-d_im) : d_im));

  if (p_rl)
    any_real_to_mpfr(sc, p_rl, sc->mpfr_1);
  else mpfr_set_d(sc->mpfr_1, d_rl, MPFR_RNDN);

  if (p_im)
    any_real_to_mpfr(sc, p_im, sc->mpfr_2);
  else mpfr_set_d(sc->mpfr_2, d_im, MPFR_RNDN);

  if (has_plus_or_minus == -1)
    mpfr_neg(sc->mpfr_2, sc->mpfr_2, MPFR_RNDN);
  return(make_big_complex(sc, sc->mpfr_1, sc->mpfr_2));
}

static bool big_numbers_are_eqv(s7_scheme *sc, s7_pointer a, s7_pointer b)
{
  /* either or both can be big here, but not neither, and types might not match at all */
  switch (type(a))
    {
    case T_INTEGER:
      return((is_t_big_integer(b)) && (mpz_cmp_si(big_integer(b), integer(a)) == 0));
    case T_BIG_INTEGER:
      if (is_t_big_integer(b)) return(mpz_cmp(big_integer(a), big_integer(b)) == 0);
      return((is_t_integer(b)) && (mpz_cmp_si(big_integer(a), integer(b)) == 0));
    case T_RATIO:
      if (!is_t_big_ratio(b)) return(false);
      mpq_set_si(sc->mpq_1, numerator(a), denominator(a));
      return(mpq_equal(sc->mpq_1, big_ratio(b)));
    case T_BIG_RATIO:
      if (is_t_big_ratio(b)) return(mpq_equal(big_ratio(a), big_ratio(b)));
      if (!is_t_ratio(b)) return(false);
      mpq_set_si(sc->mpq_1, numerator(b), denominator(b));
      return(mpq_equal(sc->mpq_1, big_ratio(a)));
    case T_REAL:
      if (is_NaN(real(a))) return(false);
      return((is_t_big_real(b)) && (!mpfr_nan_p(big_real(b))) && (mpfr_cmp_d(big_real(b), real(a)) == 0));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(a))) return(false);
      if (is_t_big_real(b)) return((!mpfr_nan_p(big_real(b))) && (mpfr_equal_p(big_real(a), big_real(b))));
      return((is_t_real(b)) && (!is_NaN(real(b))) && (mpfr_cmp_d(big_real(a), real(b)) == 0));
    case T_COMPLEX:
      if ((is_NaN(real_part(a))) || (is_NaN(imag_part(a)))) return(false);
      if (!is_t_big_complex(b)) return(false);
      if ((mpfr_nan_p(mpc_realref(big_complex(b)))) || (mpfr_nan_p(mpc_imagref(big_complex(b)))))
	return(false);
      mpc_set_d_d(sc->mpc_1, real_part(a), imag_part(a), MPC_RNDNN);
      return(mpc_cmp(sc->mpc_1, big_complex(b)) == 0);
    case T_BIG_COMPLEX:
      if ((mpfr_nan_p(mpc_realref(big_complex(a)))) || (mpfr_nan_p(mpc_imagref(big_complex(a)))))
	return(false);
      if (is_t_big_complex(b))
	{
	  if ((mpfr_nan_p(mpc_realref(big_complex(b)))) || (mpfr_nan_p(mpc_imagref(big_complex(b)))))
	    return(false);
	  return(mpc_cmp(big_complex(a), big_complex(b)) == 0);
	}
      if (is_t_complex(b))
	{
	  if ((is_NaN(real_part(b))) || (is_NaN(imag_part(b)))) return(false);
	  mpc_set_d_d(sc->mpc_2, real_part(b), imag_part(b), MPC_RNDNN);
	  return(mpc_cmp(big_complex(a), sc->mpc_1) == 0);
	}}
  return(false);
}

static s7_int big_integer_to_s7_int(s7_scheme *sc, mpz_t n)
{
  if (!mpz_fits_slong_p(n))
    s7_error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "bigint does not fit in s7_int: ~S", 34), mpz_to_big_integer(sc, n)));
  return(mpz_get_si(n));
}
#endif

#ifndef HAVE_OVERFLOW_CHECKS
  #if ((defined(__clang__) && (!POINTER_32) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4))) || (defined(__GNUC__) && __GNUC__ >= 5))
    #define HAVE_OVERFLOW_CHECKS 1
  #else
    #define HAVE_OVERFLOW_CHECKS 0
    #pragma message("no arithmetic overflow checks in this version of s7")
    /* these are untested */
    static bool add_overflow(s7_int A, s7_int B, s7_int *C) {*C = A + B; return(false);}        /* #define add_overflow(A, B, C) 0 */
    static bool subtract_overflow(s7_int A, s7_int B, s7_int *C) {*C = A - B; return(false);}   /* #define subtract_overflow(A, B, C) 0 */
    static bool multiply_overflow(s7_int A, s7_int B, s7_int *C) {*C = A * B; return(false);}   /* #define multiply_overflow(A, B, C) 0 */
  #endif
#endif

#if (defined(__clang__) && (!POINTER_32) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4)))
  #define subtract_overflow(A, B, C)       __builtin_ssubll_overflow((long long)A, (long long)B, (long long *)C)
  #define add_overflow(A, B, C)            __builtin_saddll_overflow((long long)A, (long long)B, (long long *)C)
  #define multiply_overflow(A, B, C)       __builtin_smulll_overflow((long long)A, (long long)B, (long long *)C)
  /* #define int32_subtract_overflow(A, B, C) __builtin_ssub_overflow(A, B, C) */
  #define int32_add_overflow(A, B, C)      __builtin_sadd_overflow(A, B, C)
  #define int32_multiply_overflow(A, B, C) __builtin_smul_overflow(A, B, C)
#else
#if (defined(__GNUC__) && __GNUC__ >= 5)
  #define subtract_overflow(A, B, C)       __builtin_sub_overflow(A, B, C)
  #define add_overflow(A, B, C)            __builtin_add_overflow(A, B, C)
  #define multiply_overflow(A, B, C)       __builtin_mul_overflow(A, B, C)
  /* #define int32_subtract_overflow(A, B, C) __builtin_sub_overflow(A, B, C) */
  #define int32_add_overflow(A, B, C)      __builtin_add_overflow(A, B, C)
  #define int32_multiply_overflow(A, B, C) __builtin_mul_overflow(A, B, C)
#endif
#endif

#if WITH_GCC
#define s7_int_abs(x) ({s7_int _X_; _X_ = x; _X_ >= 0 ? _X_ : -_X_;})
#else
#define s7_int_abs(x) ((x) >= 0 ? (x) : -(x))
#endif
/* can't use abs even in gcc -- it doesn't work with int64_ts! */

#if (!__NetBSD__)
  #define s7_fabsl(X) fabsl(X)
#else
  static double s7_fabsl(long_double x) {return((signbit(x)) ? -x : x);}
#endif

/* for g_log, we also need round. this version is from stackoverflow, see also r5rs_round below */
double s7_round(double number) {return((number < 0.0) ? ceil(number - 0.5) : floor(number + 0.5));}

#if HAVE_COMPLEX_NUMBERS
#if __cplusplus
  #define _Complex_I (complex<s7_double>(0.0, 1.0))
  #define creal(x) Real(x)
  #define cimag(x) Imag(x)
  #define carg(x) arg(x)
  #define cabs(x) abs(x)
  #define csqrt(x) sqrt(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define cexp(x) exp(x)
  #define csin(x) sin(x)
  #define ccos(x) cos(x)
  #define ctan(x) tan(x)
  #define csinh(x) sinh(x)
  #define ccosh(x) cosh(x)
  #define ctanh(x) tanh(x)
  #define casin(x) asin(x)
  #define cacos(x) acos(x)
  #define catan(x) atan(x)
  #define casinh(x) asinh(x)
  #define cacosh(x) acosh(x)
  #define catanh(x) atanh(x)
#else
  typedef double complex s7_complex;
#endif


#if (!HAVE_COMPLEX_TRIG)
#if (__cplusplus)

  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-_Complex_I * clog(_Complex_I * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-_Complex_I * clog(z + _Complex_I * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(_Complex_I * clog((_Complex_I + z) / (_Complex_I - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
#else

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 12)
static s7_complex clog(s7_complex z) {return(log(fabs(cabs(z))) + carg(z) * _Complex_I);}
static s7_complex cpow(s7_complex x, s7_complex y)
{
  s7_double r = cabs(x);
  s7_double theta = carg(x);
  s7_double yre = creal(y);
  s7_double yim = cimag(y);
  s7_double nr = exp(yre * log(r) - yim * theta);
  s7_double ntheta = yre * theta + yim * log(r);
  return(nr * cos(ntheta) + (nr * sin(ntheta)) * _Complex_I);
}
#endif
#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 9) /* untested -- this orignally looked at __FreeBSD_version which apparently no longer exists */
  static s7_complex cexp(s7_complex z) {return(exp(creal(z)) * cos(cimag(z)) + (exp(creal(z)) * sin(cimag(z))) * _Complex_I);}
#endif

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 10)
  static s7_complex csin(s7_complex z)   {return(sin(creal(z)) * cosh(cimag(z)) + (cos(creal(z)) * sinh(cimag(z))) * _Complex_I);}
  static s7_complex ccos(s7_complex z)   {return(cos(creal(z)) * cosh(cimag(z)) + (-sin(creal(z)) * sinh(cimag(z))) * _Complex_I);}
  static s7_complex csinh(s7_complex z)  {return(sinh(creal(z)) * cos(cimag(z)) + (cosh(creal(z)) * sin(cimag(z))) * _Complex_I);}
  static s7_complex ccosh(s7_complex z)  {return(cosh(creal(z)) * cos(cimag(z)) + (sinh(creal(z)) * sin(cimag(z))) * _Complex_I);}
  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-_Complex_I * clog(_Complex_I * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-_Complex_I * clog(z + _Complex_I * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(_Complex_I * clog((_Complex_I + z) / (_Complex_I - z)) / 2.0);}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
#endif /* not FreeBSD 10 */
#endif /* not c++ */
#endif /* not HAVE_COMPLEX_TRIG */

#else  /* not HAVE_COMPLEX_NUMBERS */
  typedef double s7_complex;
  #define _Complex_I 1
  #define creal(x) x
  #define cimag(x) x
  #define csin(x) sin(x)
  #define casin(x) x
  #define ccos(x) cos(x)
  #define cacos(x) x
  #define ctan(x) x
  #define catan(x) x
  #define csinh(x) x
  #define casinh(x) x
  #define ccosh(x) x
  #define cacosh(x) x
  #define ctanh(x) x
  #define catanh(x) x
  #define cexp(x) exp(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define csqrt(x) sqrt(x)
  #define conj(x) x
#endif

#ifdef __OpenBSD__
  /* openbsd's builtin versions of these functions are not usable */
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh_1(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
#endif
#ifdef __NetBSD__
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
#endif


bool s7_is_number(s7_pointer p)   {return(is_number(p));}
bool s7_is_complex(s7_pointer p)  {return(is_number(p));}
bool s7_is_real(s7_pointer p)     {return(is_real(p));}
bool s7_is_rational(s7_pointer p) {return(is_rational(p));}

bool s7_is_integer(s7_pointer p)
{
#if WITH_GMP
  return((is_t_integer(p)) || (is_t_big_integer(p)));
#else
  return(is_t_integer(p));
#endif
}

bool s7_is_ratio(s7_pointer p)
{
#if WITH_GMP
  return((is_t_ratio(p)) || (is_t_big_ratio(p)));
#else
  return(is_t_ratio(p));
#endif
}

static s7_int c_gcd(s7_int u, s7_int v)
{
  s7_int a, b;
  if ((u == s7_int_min) || (v == s7_int_min))
    {
      /* can't take abs of these (below) so do it by hand */
      s7_int divisor = 1;
      if (u == v) return(u);
      while (((u & 1) == 0) && ((v & 1) == 0))
	{
	  u /= 2;
	  v /= 2;
	  divisor *= 2;
	}
      return(divisor);
    }
  a = s7_int_abs(u);
  b = s7_int_abs(v);
  /* there are faster gcd algorithms but does it ever matter? */
  while (b != 0)
    {
      s7_int temp = a % b;
      a = b;
      b = temp;
    }
  /* if (a < 0) return(-a); */ /* why this? */
  return(a);
}

#define RATIONALIZE_LIMIT 1.0e12

static bool c_rationalize(s7_double ux, s7_double error, s7_int *numer, s7_int *denom)
{
  /* from CL code in Canny, Donald, Ressler, "A Rational Rotation Method for Robust Geometric Algorithms" */
  double x0, x1;
  s7_int i, p0, q0 = 1, p1, q1 = 1;
  double e0, e1, e0p, e1p;
  int32_t tries = 0;
  /* don't use long_double: the loop below will hang */

  /* #e1e19 is a killer -- it's bigger than most-positive-fixnum, but if we ceil(ux) below
   *   it turns into most-negative-fixnum.  1e19 is trouble in many places.
   */
  if (fabs(ux) > RATIONALIZE_LIMIT)
    {
      /* (rationalize most-positive-fixnum) should not return most-negative-fixnum
       *   but any number > 1e14 here is so inaccurate that rationalize is useless
       *   for example,
       *     default: (rationalize (/ (*s7* 'most-positive-fixnum) 31111.0)) -> 1185866354261165/4
       *     gmp:     (rationalize (/ (*s7* 'most-positive-fixnum) 31111.0)) -> 9223372036854775807/31111
       * can't return false here because that confuses some of the callers!
       */
      (*numer) = (s7_int)ux;
      (*denom) = 1;
      return(true);
    }

  if (error < 0.0) error = -error;
  x0 = ux - error;
  x1 = ux + error;
  i = (s7_int)ceil(x0);

  if (error >= 1.0) /* aw good grief! */
    {
      if (x0 < 0.0)
	(*numer) = (x1 < 0.0) ? (s7_int)floor(x1) : 0;
      else (*numer) = i;
      (*denom) = 1;
      return(true);
    }
  if (x1 >= i)
    {
      (*numer) = (i >= 0) ? i : (s7_int)floor(x1);
      (*denom) = 1;
      return(true);
    }

  p0 = (s7_int)floor(x0);
  p1 = (s7_int)ceil(x1);
  e0 = p1 - x0;
  e1 = x0 - p0;
  e0p = p1 - x1;
  e1p = x1 - p0;

  while (true)
    {
      s7_int old_p1, old_q1;
      double old_e0, old_e1, old_e0p, val, r, r1;
      val = (double)p0 / (double)q0;

      if (((x0 <= val) && (val <= x1)) || (e1 == 0.0) || (e1p == 0.0) || (tries > 100))
	{
	  if ((q0 == s7_int_min) && (p0 == 1)) /* (rationalize 1.000000004297917e-12) when error is 1e-12 */
	    {
	      (*numer) = 0;
	      (*denom) = 1;
	    }
	  else
	    {
	      (*numer) = p0;
	      (*denom) = q0;
	      if ((S7_DEBUGGING) && (q0 == 0)) fprintf(stderr, "%f %" ld64 "/0\n", ux, p0);
	    }
	  return(true);
	}
      tries++;

      r = (s7_int)floor(e0 / e1);
      r1 = (s7_int)ceil(e0p / e1p);
      if (r1 < r) r = r1;

      /* do handles all step vars in parallel */
      old_p1 = p1;
      p1 = p0;
      old_q1 = q1;
      q1 = q0;
      old_e0 = e0;
      e0 = e1p;
      old_e0p = e0p;
      e0p = e1;
      old_e1 = e1;

      p0 = old_p1 + r * p0;
      q0 = old_q1 + r * q0;
      e1 = old_e0p - r * e1p;
      /* if the error is set too low, we can get e1 = 0 here: (rationalize (/ pi) 1e-17) */
      e1p = old_e0 - r * old_e1;
    }
  return(false);
}

s7_pointer s7_rationalize(s7_scheme *sc, s7_double x, s7_double error)
{
  s7_int numer = 0, denom = 1;
  if (c_rationalize(x, error, &numer, &denom))
    return(make_ratio(sc, numer, denom));
  return(make_real(sc, x));
}

s7_pointer s7_make_integer(s7_scheme *sc, s7_int n)
{
  s7_pointer x;
  if (is_small_int(n))
    return(small_int(n));
  new_cell(sc, x, T_INTEGER);
  integer(x) = n;
  return(x);
}

static s7_pointer make_mutable_integer(s7_scheme *sc, s7_int n)
{
  s7_pointer x;
  new_cell(sc, x, T_INTEGER | T_MUTABLE | T_IMMUTABLE);
  integer(x) = n;
  return(x);
}

s7_pointer s7_make_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL);
  set_real(x, n);
  return(x);
}

s7_pointer s7_make_mutable_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL | T_MUTABLE | T_IMMUTABLE);
  set_real(x, n);
  return(x);
}

s7_pointer s7_make_complex(s7_scheme *sc, s7_double a, s7_double b)
{
  s7_pointer x;
  if (b == 0.0)
    {
      new_cell(sc, x, T_REAL);
      set_real(x, a);
    }
  else
    {
      new_cell(sc, x, T_COMPLEX);
      set_real_part(x, a);
      set_imag_part(x, b);
    }
  return(x);
}

static s7_complex s7_to_c_complex(s7_pointer p)
{
#if HAVE_COMPLEX_NUMBERS
  return(CMPLX(s7_real_part(p), s7_imag_part(p)));
#else
  return(0.0);
#endif
}

static inline s7_pointer c_complex_to_s7(s7_scheme *sc, s7_complex z) {return(make_complex(sc, creal(z), cimag(z)));}

static noreturn void division_by_zero_error_1(s7_scheme *sc, s7_pointer caller, s7_pointer x)
{
  s7_error_nr(sc, sc->division_by_zero_symbol,
	      set_elist_4(sc, wrap_string(sc, "~A: division by zero, (~A ~S)", 29), caller, caller, x));
}

static noreturn void division_by_zero_error_2(s7_scheme *sc, s7_pointer caller, s7_pointer x, s7_pointer y)
{
  s7_error_nr(sc, sc->division_by_zero_symbol,
	      set_elist_5(sc, wrap_string(sc, "~A: division by zero, (~A ~S ~S)", 32), caller, caller, x, y));
}

static s7_pointer make_ratio(s7_scheme *sc, s7_int a, s7_int b)
{
  s7_pointer x;
  if (b == s7_int_min)
    {
      /* This should not trigger an error during reading -- we might have the
       *   ratio on a switch with-bignums or whatever, so its mere occurrence is just an annoyance.
       */
      if (a & 1)
	return(make_real(sc, (long_double)a / (long_double)b));
      a /= 2;
      b /= 2;
    }
  if (b < 0)
    {
      a = -a;
      b = -b;
    }
  if (a == s7_int_min)
    {
      while (((a & 1) == 0) && ((b & 1) == 0))
	{
	  a /= 2;
	  b /= 2;
	}}
  else
    {
      s7_int b1 = b, divisor = s7_int_abs(a);
      do {
	s7_int temp = divisor % b1;
	divisor = b1;
	b1 = temp;
      } while (b1 != 0);
      if (divisor != 1)
	{
	  a /= divisor;
	  b /= divisor;
	}}
  if (b == 1)
    return(make_integer(sc, a));

  new_cell(sc, x, T_RATIO);
  numerator(x) = a;
  denominator(x) = b;
  return(x);
}

/* using "make-ratio" here is a desperate kludge trying to maintain backwards compatibility; internally we use make_ratio_with_div_check below */
s7_pointer s7_make_ratio(s7_scheme *sc, s7_int a, s7_int b)
{
  if (b == 0)
    division_by_zero_error_2(sc, wrap_string(sc, "make-ratio", 10), wrap_integer(sc, a), int_zero);
  return(make_ratio(sc, a, b));
}

static s7_pointer make_ratio_with_div_check(s7_scheme *sc, s7_pointer caller, s7_int a, s7_int b)
{
  if (b == 0)
    division_by_zero_error_2(sc, caller, wrap_integer(sc, a), int_zero);
  return(make_ratio(sc, a, b));
}


#define WITH_OVERFLOW_ERROR true
#define WITHOUT_OVERFLOW_ERROR false

#define INT64_TO_DOUBLE_LIMIT (1LL << 53)
#define DOUBLE_TO_INT64_LIMIT (1LL << 53)

#if (!WITH_PURE_S7)

/* 9007199254740991LL is where a truncated double starts to skip integers (expt 2 53) = ca 1e16
 *   (ceiling (+ 1e16 1)) -> 10000000000000000
 *   (> 9007199254740993.0 9007199254740992.0) -> #f ; in non-gmp 64-bit doubles
 * but we can't fix this except in the gmp case because:
 *   (integer-decode-float (+ (expt 2.0 62) 100)) -> (4503599627370496 10 1)
 *   (integer-decode-float (+ (expt 2.0 62) 500)) -> (4503599627370496 10 1)
 *   (> (+ (expt 2.0 62) 500) (+ (expt 2.0 62) 100)) -> #f ; non-gmp again
 * i.e. the bits are identical.  We can't even detect when it has happened (without tedious effort), so should
 *   we just give an error for any floor (or whatever) of an arg>1e16?  (sin has a similar problem)?
 *   I think in the non-gmp case I'll throw an error in these cases because the results are bogus:
 *   (floor (+ (expt 2.0 62) 512)) -> 4611686018427387904
 *   (floor (+ (expt 2.0 62) 513)) -> 4611686018427388928
 * another case at the edge: (round 9007199254740992.51) -> 9007199254740992
 * This spells trouble for normal arithmetic in this range.  If no gmp,
 *    (- (+ (expt 2.0 62) 512) (+ (expt 2.0 62) 513)) = -1024.0 (should be -1.0)
 *    but we don't currently give an error in this case -- not sure what the right thing is.
 */

static s7_pointer exact_to_inexact(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
#if WITH_GMP
      if ((integer(x) > INT64_TO_DOUBLE_LIMIT) || (integer(x) < -INT64_TO_DOUBLE_LIMIT))
	return(s7_number_to_big_real(sc, x));
#endif
      return(make_real(sc, (s7_double)(integer(x))));

    case T_RATIO:
#if WITH_GMP
      if ((numerator(x) > INT64_TO_DOUBLE_LIMIT) || (numerator(x) < -INT64_TO_DOUBLE_LIMIT) ||
 	  (denominator(x) > INT64_TO_DOUBLE_LIMIT))  /* just a guess */
	return(s7_number_to_big_real(sc, x));
#endif
      return(make_real(sc, (s7_double)(fraction(x))));

#if WITH_GMP
    case T_BIG_INTEGER:
      return(big_integer_to_big_real(sc, x));
    case T_BIG_RATIO:
      return(big_ratio_to_big_real(sc, x));
#endif
    case T_REAL:    case T_BIG_REAL:
    case T_COMPLEX: case T_BIG_COMPLEX:
      return(x); /* apparently (exact->inexact 1+i) is not an error */

    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->exact_to_inexact_symbol, a_number_string));
    }
}

#if WITH_GMP
static s7_pointer big_rationalize(s7_scheme *sc, s7_pointer args);
#endif

static s7_pointer inexact_to_exact(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER: case T_BIG_INTEGER:
    case T_RATIO:   case T_BIG_RATIO:
      return(x);

#if WITH_GMP
    case T_BIG_REAL:
      return(big_rationalize(sc, set_plist_1(sc, x)));
#endif

    case T_REAL:
      {
	s7_int numer = 0, denom = 1;
	s7_double val = real(x);
	if ((is_inf(val)) || (is_NaN(val)))
	  simple_wrong_type_argument_with_type(sc, sc->inexact_to_exact_symbol, x, a_normal_real_string);

	if ((val > DOUBLE_TO_INT64_LIMIT) || (val < -(DOUBLE_TO_INT64_LIMIT)))
	  {
#if WITH_GMP
	    return(big_rationalize(sc, set_plist_1(sc, x))); /* this can handle t_real as well as t_big_real */
#else
	    simple_out_of_range(sc, sc->inexact_to_exact_symbol, x, its_too_large_string);
#endif
	  }
	/* c_rationalize limit is RATIONALIZE_LIMIT=1e12 currently so this is a tighter limit than DOUBLE_TO_INT64_LIMIT */
	if (c_rationalize(val, sc->default_rationalize_error, &numer, &denom))
	  return(make_ratio(sc, numer, denom));
      }

    default:
      return(method_or_bust_p(sc, x, sc->inexact_to_exact_symbol, T_REAL));
    }
  return(x);
}
#endif

/* this is a mess -- it's too late to clean up s7.h (sigh) */
s7_double s7_number_to_real_with_caller(s7_scheme *sc, s7_pointer x, const char *caller)
{
  if (is_t_real(x)) return(real(x));
  switch (type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return(fraction(x));
#if WITH_GMP
    case T_BIG_INTEGER: return((s7_double)big_integer_to_s7_int(sc, big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long_double)big_integer_to_s7_int(sc, mpq_numref(big_ratio(x))) /
					   (long_double)big_integer_to_s7_int(sc, mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), MPFR_RNDN));
#endif
    }
  s7_wrong_type_arg_error(sc, caller, 0, x, "a real number");
  return(0.0);
}

s7_double s7_number_to_real(s7_scheme *sc, s7_pointer x) {return(s7_number_to_real_with_caller(sc, x, "s7_number_to_real"));}

s7_int s7_number_to_integer_with_caller(s7_scheme *sc, s7_pointer x, const char *caller)
{
  if (is_t_integer(x)) return(integer(x));
#if WITH_GMP
  if (is_t_big_integer(x)) return(big_integer_to_s7_int(sc, big_integer(x)));
#endif
  s7_wrong_type_arg_error(sc, caller, 0, x, "an integer");
  return(0);
}

s7_int s7_number_to_integer(s7_scheme *sc, s7_pointer x) {return(s7_number_to_integer_with_caller(sc, x, "s7_number_to_integer"));}

s7_int s7_numerator(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x));
    case T_RATIO:       return(numerator(x));
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_get_si(big_integer(x))); /* big_integer_to_s7_int but no sc -- no error if out of range */
    case T_BIG_RATIO:   return(mpz_get_si(mpq_numref(big_ratio(x))));
#endif
    }
  return(0);
}

s7_int s7_denominator(s7_pointer x)
{
  if (is_t_ratio(x)) return(denominator(x));
#if WITH_GMP
  if (is_t_big_ratio(x)) return(mpz_get_si(mpq_denref(big_ratio(x))));
#endif
  return(1);
}

s7_int s7_integer(s7_pointer p)
{
  if (is_t_integer(p)) return(integer(p));
#if WITH_GMP
  if (is_t_big_integer(p)) return(mpz_get_si(big_integer(p)));
#endif
  return(0);
}

s7_double s7_real(s7_pointer x)
{
  if (is_t_real(x)) return(real(x));
  switch (type(x))
    {
    case T_RATIO:       return(fraction(x));
    case T_INTEGER:     return((s7_double)integer(x));
#if WITH_GMP
    case T_BIG_INTEGER: return((s7_double)mpz_get_si(big_integer(x)));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), MPFR_RNDN));
    case T_BIG_RATIO:
      {
	s7_double result;
	mpfr_t bx;
	mpfr_init2(bx, DEFAULT_BIGNUM_PRECISION);
	mpfr_set_q(bx, big_ratio(x), MPFR_RNDN);
	result = mpfr_get_d(bx, MPFR_RNDN);
	mpfr_clear(bx);
	return(result);
      }
#endif
    }
  return(0.0);
}

static bool is_one(s7_pointer x)
{
  return(((is_t_integer(x)) && (integer(x) == 1)) ||
	 ((is_t_real(x)) && (real(x) == 1.0)));
}


/* -------- optimize exponents -------- */

#define MAX_POW 64
static double **pepow = NULL; /* [17][MAX_POW * 2]; */

static void init_pows(void)
{
  pepow = (double **)Malloc(17 * sizeof(double *));
  pepow[0] = NULL;
  pepow[1] = NULL;
  for (int32_t i = 2; i < 17; i++) pepow[i] = (double *)Malloc((MAX_POW * 2) * sizeof(double));
  for (int32_t i = 2; i < 17; i++)        /* radix between 2 and 16 */
    for (int32_t j = -MAX_POW; j < MAX_POW; j++) /* saved exponent between 0 and +/- MAX_POW */
      pepow[i][j + MAX_POW] = pow((double)i, (double)j);
}

static inline double dpow(int32_t x, int32_t y)
{
  if ((y >= MAX_POW) || (y < -MAX_POW)) /* this can happen (once in a blue moon) */
    return(pow((double)x, (double)y));
  return(pepow[x][y + MAX_POW]);
}


/* -------------------------------- number->string -------------------------------- */
#define WITH_DTOA 1
#if WITH_DTOA
/* fpconv, revised to fit the local coding style

   The MIT License

Copyright (c) 2013 Andreas Samoljuk

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#define dtoa_npowers     87
#define dtoa_steppowers  8
#define dtoa_firstpower -348 /* 10 ^ -348 */
#define dtoa_expmax     -32
#define dtoa_expmin     -60

typedef struct dtoa_np {uint64_t frac; int32_t exp;} dtoa_np;

static const dtoa_np dtoa_powers_ten[] = {
    { 18054884314459144840U, -1220 }, { 13451937075301367670U, -1193 }, { 10022474136428063862U, -1166 }, { 14934650266808366570U, -1140 },
    { 11127181549972568877U, -1113 }, { 16580792590934885855U, -1087 }, { 12353653155963782858U, -1060 }, { 18408377700990114895U, -1034 },
    { 13715310171984221708U, -1007 }, { 10218702384817765436U, -980 }, { 15227053142812498563U, -954 },  { 11345038669416679861U, -927 },
    { 16905424996341287883U, -901 },  { 12595523146049147757U, -874 }, { 9384396036005875287U,  -847 },  { 13983839803942852151U, -821 },
    { 10418772551374772303U, -794 },  { 15525180923007089351U, -768 }, { 11567161174868858868U, -741 },  { 17236413322193710309U, -715 },
    { 12842128665889583758U, -688 },  { 9568131466127621947U,  -661 }, { 14257626930069360058U, -635 },  { 10622759856335341974U, -608 },
    { 15829145694278690180U, -582 },  { 11793632577567316726U, -555 }, { 17573882009934360870U, -529 },  { 13093562431584567480U, -502 },
    { 9755464219737475723U,  -475 },  { 14536774485912137811U, -449 }, { 10830740992659433045U, -422 },  { 16139061738043178685U, -396 },
    { 12024538023802026127U, -369 },  { 17917957937422433684U, -343 }, { 13349918974505688015U, -316 },  { 9946464728195732843U,  -289 },
    { 14821387422376473014U, -263 },  { 11042794154864902060U, -236 }, { 16455045573212060422U, -210 },  { 12259964326927110867U, -183 },
    { 18268770466636286478U, -157 },  { 13611294676837538539U, -130 }, { 10141204801825835212U, -103 },  { 15111572745182864684U, -77 },
    { 11258999068426240000U, -50 },   { 16777216000000000000U, -24 }, { 12500000000000000000U,   3 },   { 9313225746154785156U,   30 },
    { 13877787807814456755U,  56 },   { 10339757656912845936U,  83 }, { 15407439555097886824U, 109 },   { 11479437019748901445U, 136 },
    { 17105694144590052135U, 162 },   { 12744735289059618216U, 189 }, { 9495567745759798747U,  216 },   { 14149498560666738074U, 242 },
    { 10542197943230523224U, 269 },   { 15709099088952724970U, 295 }, { 11704190886730495818U, 322 },   { 17440603504673385349U, 348 },
    { 12994262207056124023U, 375 },   { 9681479787123295682U,  402 }, { 14426529090290212157U, 428 },   { 10748601772107342003U, 455 },
    { 16016664761464807395U, 481 },   { 11933345169920330789U, 508 }, { 17782069995880619868U, 534 },   { 13248674568444952270U, 561 },
    { 9871031767461413346U,  588 },   { 14708983551653345445U, 614 }, { 10959046745042015199U, 641 },   { 16330252207878254650U, 667 },
    { 12166986024289022870U, 694 },   { 18130221999122236476U, 720 }, { 13508068024458167312U, 747 },   { 10064294952495520794U, 774 },
    { 14996968138956309548U, 800 },   { 11173611982879273257U, 827 }, { 16649979327439178909U, 853 },   { 12405201291620119593U, 880 },
    { 9242595204427927429U,  907 },   { 13772540099066387757U, 933 }, { 10261342003245940623U, 960 },   { 15290591125556738113U, 986 },
    { 11392378155556871081U, 1013 },  { 16975966327722178521U, 1039 },
    { 12648080533535911531U, 1066 }};

static dtoa_np dtoa_find_cachedpow10(int exp, int* k)
{
  const double one_log_ten = 0.30102999566398114;
  int32_t approx = -(exp + dtoa_npowers) * one_log_ten;
  int32_t idx = (approx - dtoa_firstpower) / dtoa_steppowers;
  while (true)
    {
      int32_t current = exp + dtoa_powers_ten[idx].exp + 64;
      if (current < dtoa_expmin)
	{
	  idx++;
	  continue;
        }
      if (current > dtoa_expmax)
	{
	  idx--;
	  continue;
        }
      *k = (dtoa_firstpower + idx * dtoa_steppowers);
      return(dtoa_powers_ten[idx]);
    }
}

#define dtoa_fracmask  0x000FFFFFFFFFFFFFU
#define dtoa_expmask   0x7FF0000000000000U
#define dtoa_hiddenbit 0x0010000000000000U
#define dtoa_signmask  0x8000000000000000U
#define dtoa_expbias   (1023 + 52)
#define dtoa_absv(n)   ((n) < 0 ? -(n) : (n))
#define dtoa_minv(a, b) ((a) < (b) ? (a) : (b))

static uint64_t dtoa_tens[] =
  { 10000000000000000000U, 1000000000000000000U, 100000000000000000U,
    10000000000000000U, 1000000000000000U, 100000000000000U,
    10000000000000U, 1000000000000U, 100000000000U,
    10000000000U, 1000000000U, 100000000U,
    10000000U, 1000000U, 100000U,
    10000U, 1000U, 100U,
    10U, 1U};

static uint64_t dtoa_get_dbits(double d)
{
  union {double dbl; uint64_t i;} dbl_bits = {d};
  return(dbl_bits.i);
}

static dtoa_np dtoa_build_np(double d)
{
  uint64_t bits = dtoa_get_dbits(d);
  dtoa_np fp;
  fp.frac = bits & dtoa_fracmask;
  fp.exp = (bits & dtoa_expmask) >> 52;
  if (fp.exp)
    {
      fp.frac += dtoa_hiddenbit;
      fp.exp -= dtoa_expbias;
    }
  else fp.exp = -dtoa_expbias + 1;
  return(fp);
}

static void dtoa_normalize(dtoa_np* fp)
{
  int32_t shift = 64 - 52 - 1;
  while ((fp->frac & dtoa_hiddenbit) == 0)
    {
      fp->frac <<= 1;
      fp->exp--;
    }
  fp->frac <<= shift;
  fp->exp -= shift;
}

static void dtoa_get_normalized_boundaries(dtoa_np* fp, dtoa_np* lower, dtoa_np* upper)
{
  int32_t u_shift, l_shift;
  upper->frac = (fp->frac << 1) + 1;
  upper->exp  = fp->exp - 1;
  while ((upper->frac & (dtoa_hiddenbit << 1)) == 0)
    {
      upper->frac <<= 1;
      upper->exp--;
    }
  u_shift = 64 - 52 - 2;
  upper->frac <<= u_shift;
  upper->exp = upper->exp - u_shift;
  l_shift = fp->frac == dtoa_hiddenbit ? 2 : 1;
  lower->frac = (fp->frac << l_shift) - 1;
  lower->exp = fp->exp - l_shift;
  lower->frac <<= lower->exp - upper->exp;
  lower->exp = upper->exp;
}

static dtoa_np dtoa_multiply(dtoa_np* a, dtoa_np* b)
{
  dtoa_np fp;
  uint64_t ah_bl, al_bh, al_bl, ah_bh, tmp;
  const uint64_t lomask = 0x00000000FFFFFFFF;

  ah_bl = (a->frac >> 32)    * (b->frac & lomask);
  al_bh = (a->frac & lomask) * (b->frac >> 32);
  al_bl = (a->frac & lomask) * (b->frac & lomask);
  ah_bh = (a->frac >> 32)    * (b->frac >> 32);
  tmp = (ah_bl & lomask) + (al_bh & lomask) + (al_bl >> 32);
  /* round up */
  tmp += 1U << 31;
  fp.frac = ah_bh + (ah_bl >> 32) + (al_bh >> 32) + (tmp >> 32);
  fp.exp = a->exp + b->exp + 64;
  return(fp);
}

static void dtoa_round_digit(char* digits, int32_t ndigits, uint64_t delta, uint64_t rem, uint64_t kappa, uint64_t frac)
{
  while ((rem < frac) && (delta - rem >= kappa) &&
	 ((rem + kappa < frac) || (frac - rem > rem + kappa - frac)))
    {
      digits[ndigits - 1]--;
      rem += kappa;
    }
}

static int32_t dtoa_generate_digits(dtoa_np* fp, dtoa_np* upper, dtoa_np* lower, char* digits, int* K)
{
  uint64_t part1, part2, wfrac = upper->frac - fp->frac, delta = upper->frac - lower->frac;
  uint64_t *unit;
  int32_t idx = 0, kappa = 10;
  dtoa_np one;

  one.frac = 1ULL << -upper->exp;
  one.exp  = upper->exp;
  part1 = upper->frac >> -one.exp;
  part2 = upper->frac & (one.frac - 1);

  /* 1000000000 */
  for (uint64_t *divp = dtoa_tens + 10; kappa > 0; divp++)
    {
      uint64_t tmp, div = *divp;
      unsigned digit = part1 / div;
      if (digit || idx)
	digits[idx++] = digit + '0';
      part1 -= digit * div;
      kappa--;
      tmp = (part1 << -one.exp) + part2;
      if (tmp <= delta)
	{
	  *K += kappa;
	  dtoa_round_digit(digits, idx, delta, tmp, div << -one.exp, wfrac);
	  return(idx);
        }}

  /* 10 */
  unit = dtoa_tens + 18;
  while(true)
    {
      unsigned digit;
      part2 *= 10;
      delta *= 10;
      kappa--;
      digit = part2 >> -one.exp;
      if (digit || idx)
	digits[idx++] = digit + '0';
      part2 &= one.frac - 1;
      if (part2 < delta)
	{
	  *K += kappa;
	  dtoa_round_digit(digits, idx, delta, part2, one.frac, wfrac * *unit);
	  return(idx);
	}
      unit--;
    }
}

static int32_t dtoa_grisu2(double d, char* digits, int* K)
{
  int32_t k;
  dtoa_np cp, w, lower, upper;
  w = dtoa_build_np(d);
  dtoa_get_normalized_boundaries(&w, &lower, &upper);
  dtoa_normalize(&w);
  cp = dtoa_find_cachedpow10(upper.exp, &k);
  w = dtoa_multiply(&w, &cp);
  upper = dtoa_multiply(&upper, &cp);
  lower = dtoa_multiply(&lower, &cp);
  lower.frac++;
  upper.frac--;
  *K = -k;
  return(dtoa_generate_digits(&w, &upper, &lower, digits, K));
}

static int32_t dtoa_emit_digits(char* digits, int32_t ndigits, char* dest, int32_t K, bool neg)
{
  int32_t exp, idx, cent;
  char sign;
  exp = dtoa_absv(K + ndigits - 1);

  /* write plain integer */
  if ((K >= 0) && (exp < (ndigits + 7)))
    {
      memcpy(dest, digits, ndigits);
      local_memset(dest + ndigits, '0', K);
      dest[ndigits + K] = '.';
      dest[ndigits + K + 1] = '0';
      return(ndigits + K + 2);
    }

  /* write decimal w/o scientific notation */
  if ((K < 0) && (K > -7 || exp < 4))
    {
      int32_t offset = ndigits - dtoa_absv(K);
      /* fp < 1.0 -> write leading zero */
      if (offset <= 0)
	{
	  offset = -offset;
	  dest[0] = '0';
	  dest[1] = '.';
	  local_memset(dest + 2, '0', offset);
	  memcpy(dest + offset + 2, digits, ndigits);
	  return(ndigits + 2 + offset);
	  /* fp > 1.0 */
	}
      else
	{
	  memcpy(dest, digits, offset);
	  dest[offset] = '.';
	  memcpy(dest + offset + 1, digits + offset, ndigits - offset);
	  return(ndigits + 1);
	}}

  /* write decimal w/ scientific notation */
  ndigits = dtoa_minv(ndigits, 18 - neg);
  idx = 0;
  dest[idx++] = digits[0];
  if (ndigits > 1)
    {
      dest[idx++] = '.';
      memcpy(dest + idx, digits + 1, ndigits - 1);
      idx += ndigits - 1;
    }
  dest[idx++] = 'e';
  sign = K + ndigits - 1 < 0 ? '-' : '+';
  dest[idx++] = sign;
  cent = 0;
  if (exp > 99)
    {
      cent = exp / 100;
      dest[idx++] = cent + '0';
      exp -= cent * 100;
    }
  if (exp > 9)
    {
      int32_t dec = exp / 10;
      dest[idx++] = dec + '0';
      exp -= dec * 10;
    }
  else
    if (cent)
      dest[idx++] = '0';

  dest[idx++] = exp % 10 + '0';
  return(idx);
}

static int32_t dtoa_filter_special(double fp, char* dest, bool neg)
{
  uint64_t bits;
  bool nan;
  if (fp == 0.0)
    {
      dest[0] = '0'; dest[1] = '.'; dest[2] = '0';
      return(3);
    }
  bits = dtoa_get_dbits(fp);
  nan = (bits & dtoa_expmask) == dtoa_expmask;
  if (!nan) return(0);

  if (!neg)
    {
      dest[0] = '+';
      dest++;
    }
  if (bits & dtoa_fracmask)
    {
      dest[0] = 'n'; dest[1] = 'a'; dest[2] = 'n'; dest[3] = '.'; dest[4] = '0';
    }
  else
    {
      dest[0] = 'i'; dest[1] = 'n'; dest[2] = 'f'; dest[3] = '.'; dest[4] = '0';
    }
  return((neg) ? 5 : 6);
}

static inline int32_t fpconv_dtoa(double d, char dest[24])
{
  char digit[18];
  int32_t str_len = 0, spec, K, ndigits;
  bool neg = false;

  if (dtoa_get_dbits(d) & dtoa_signmask)
    {
      dest[0] = '-';
      str_len++;
      neg = true;
    }
  spec = dtoa_filter_special(d, dest + str_len, neg);
  if (spec) return(str_len + spec);
  K = 0;
  ndigits = dtoa_grisu2(d, digit, &K);
  str_len += dtoa_emit_digits(digit, ndigits, dest + str_len, K, neg);
  return(str_len);
}
#endif


/* -------------------------------- number->string -------------------------------- */
static const char dignum[] = "0123456789abcdef";

static size_t integer_to_string_any_base(char *p, s7_int n, int32_t radix)  /* called by number_to_string_with_radix */
{
  s7_int i, len, end;
  bool sign;
  s7_int pown;

  if ((radix < 2) || (radix > 16))
    return(0);

  if (n == S7_INT64_MIN) /* can't negate this, so do it by hand */
    {
      static const char *mnfs[17] = {"","",
	"-1000000000000000000000000000000000000000000000000000000000000000", "-2021110011022210012102010021220101220222",
	"-20000000000000000000000000000000", "-1104332401304422434310311213", "-1540241003031030222122212",
	"-22341010611245052052301", "-1000000000000000000000", "-67404283172107811828",	"-9223372036854775808",
	"-1728002635214590698",	"-41a792678515120368", "-10b269549075433c38", "-4340724c6c71dc7a8", "-160e2ad3246366808", "-8000000000000000"};

      len = safe_strlen(mnfs[radix]);
      memcpy((void *)p, (void *)mnfs[radix], len);
      p[len] = '\0';
      return(len);
    }

  sign = (n < 0);
  if (sign) n = -n;

  /* the previous version that counted up to n, rather than dividing down below n, as here,
   *   could be confused by large ints on 64 bit machines
   */
  pown = n;
  for (i = 1; i < 100; i++)
    {
      if (pown < radix)
	break;
      pown /= (s7_int)radix;
    }
  len = i - 1;
  if (sign) len++;
  end = 0;
  if (sign)
    {
      p[0] = '-';
      end++;
    }
  for (i = len; i >= end; i--)
    {
      p[i] = dignum[n % radix];
      n /= radix;
    }
  p[len + 1] = '\0';
  return(len + 1);
}

static char *integer_to_string(s7_scheme *sc, s7_int num, s7_int *nlen) /* do not free the returned string */
{
  char *p, *op;
  bool sign;

  if (num == S7_INT64_MIN)
    {
      (*nlen) = 20;
      return((char *)"-9223372036854775808");
    }
  p = (char *)(sc->int_to_str1 + INT_TO_STR_SIZE - 1);
  op = p;
  *p-- = '\0';

  sign = (num < 0);
  if (sign) num = -num;  /* we need a positive index below */
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  if (sign)
    {
      *p = '-';
      (*nlen) = op - p;
      return(p);
    }
  (*nlen) = op - p - 1;
  return(++p);
}

static char *integer_to_string_no_length(s7_scheme *sc, s7_int num) /* do not free the returned string */
{
  char *p;
  bool sign;

  if (num == S7_INT64_MIN)
    return((char *)"-9223372036854775808");
  p = (char *)(sc->int_to_str2 + INT_TO_STR_SIZE - 1);
  *p-- = '\0';
  sign = (num < 0);
  if (sign) num = -num;
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  if (sign)
    {
      *p = '-';
      return(p);
    }
  return(++p);
}

static char *floatify(char *str, s7_int *nlen)
{
  if ((!strchr(str, '.')) && (!strchr(str, 'e'))) /* faster than (strcspn(str, ".e") >= (size_t)(*nlen)) */
    {
      s7_int len = *nlen;
      /* snprintf returns "nan" and "inf" but we (stupidly) want "+nan.0" and "+inf.0"; "-nan" and "-inf" will be handled by the normal case */
      if (len == 3)
	{
	  if (str[0] == 'n')
	    {
	      str[0] = '+'; str[1] = 'n'; str[2] = 'a'; str[3] = 'n';
	      len = 4;
	    }
	  if (str[0] == 'i')
	    {
	      str[0] = '+'; str[1] = 'i'; str[2] = 'n'; str[3] = 'f';
	      len = 4;
	    }}
      str[len]='.';
      str[len + 1]='0';
      str[len + 2]='\0';
      (*nlen) = len + 2;
    }
  return(str);
}

static void insert_spaces(s7_scheme *sc, const char *src, s7_int width, s7_int len)
{
  s7_int spaces = width - len;
  if (width >= sc->num_to_str_size)
    {
      sc->num_to_str_size = width + 1;
      sc->num_to_str = (char *)Realloc(sc->num_to_str, sc->num_to_str_size);
    }
  sc->num_to_str[width] = '\0';
  memmove((void *)(sc->num_to_str + spaces), (void *)src, len);
  local_memset((void *)(sc->num_to_str), (int)' ', spaces);
}

static char *number_to_string_base_10(s7_scheme *sc, s7_pointer obj, s7_int width, s7_int precision,
				      char float_choice, s7_int *nlen, use_write_t choice) /* don't free result */
{
  /* called by number_to_string_with_radix g_number_to_string, number_to_string_p_p number_to_port format_number */
  /* the rest of s7 assumes nlen is set to the correct length
   *   a tricky case: (format #f "~f" 1e308) -- tries to print 308 digits! so 256 as default len is too small.
   *   but then even worse: (format #f "~F" 1e308+1e308i)!
   */
  s7_int len = width + precision;
  len = (len > 512) ? (512 + 2 * len) : 1024;
  if (len > sc->num_to_str_size)
    {
      sc->num_to_str = (sc->num_to_str) ? (char *)Realloc(sc->num_to_str, len) : (char *)Malloc(len);
      sc->num_to_str_size = len;
    }

  /* bignums can't happen here */
  if (is_t_integer(obj))
    {
      char *p;
      if (width == 0)
	{
	  if (has_number_name(obj))
	    {
	      (*nlen) = number_name_length(obj);
	      return((char *)number_name(obj));
	    }
	  return(integer_to_string(sc, integer(obj), nlen));
	}
      p = integer_to_string(sc, integer(obj), &len);
      if (width > len)
	{
	  insert_spaces(sc, p, width, len);
	  (*nlen) = width;
	  return(sc->num_to_str);
	}
      (*nlen) = len;
      return(p);
    }

  if (is_t_real(obj))
    {
      if (width == 0)
	{
#if WITH_DTOA
	  if ((float_choice == 'g') &&
	      (precision == WRITE_REAL_PRECISION))
	    {
	      /* (number->string 0.0000001) is sensitive to (*s7* 'float-format-precision) and inconsistent: either 1e-7 or 0.0000001
	       *    because fpconv_dtoa has some complicated decision about 'g' vs 'f' -- not sure if this is a bug.
	       */
	      len = fpconv_dtoa(real(obj), sc->num_to_str);
	      sc->num_to_str[len] = '\0';
	      (*nlen) = len;
	      return(sc->num_to_str);
	    }
#endif
	  len = snprintf(sc->num_to_str, sc->num_to_str_size - 4,
			 (float_choice == 'g') ? "%.*g" : ((float_choice == 'f') ? "%.*f" : "%.*e"),
			 (int32_t)precision, real(obj)); /* -4 for floatify */
	}
      else len = snprintf(sc->num_to_str, sc->num_to_str_size - 4,
			  (float_choice == 'g') ? "%*.*g" : ((float_choice == 'f') ? "%*.*f" : "%*.*e"),
			  (int32_t)width, (int32_t)precision, real(obj)); /* -4 for floatify */
      (*nlen) = len;
      floatify(sc->num_to_str, nlen);
      return(sc->num_to_str);
    }

  if (is_t_complex(obj))
    {
      char *imag;
      sc->num_to_str[0] = '\0';
      imag = copy_string(number_to_string_base_10(sc, wrap_real(sc, imag_part(obj)), 0, precision, float_choice, &len, choice));

      sc->num_to_str[0] = '\0';
      number_to_string_base_10(sc, wrap_real(sc, real_part(obj)), 0, precision, float_choice, &len, choice);

      sc->num_to_str[len] = '\0';
      len = catstrs(sc->num_to_str, sc->num_to_str_size, ((imag[0] == '+') || (imag[0] == '-')) ? "" : "+", imag, "i", (char *)NULL);
      free(imag);

      if (width > len)  /* (format #f "~20g" 1+i) */
	{
	  insert_spaces(sc, sc->num_to_str, width, len); /* this checks sc->num_to_str_size */
	  (*nlen) = width;
	}
      else (*nlen) = len;
      return(sc->num_to_str);
    }

  /* ratio */
  len = catstrs_direct(sc->num_to_str, integer_to_string_no_length(sc, numerator(obj)), "/", pos_int_to_str_direct(sc, denominator(obj)), (const char *)NULL);
  if (width > len)
    {
      insert_spaces(sc, sc->num_to_str, width, len);
      (*nlen) = width;
    }
  else (*nlen) = len;
  return(sc->num_to_str);
}

static block_t *number_to_string_with_radix(s7_scheme *sc, s7_pointer obj, int32_t radix, s7_int width, s7_int precision, char float_choice, s7_int *nlen)
{
  /* called by s7_number_to_string (char*), g_number_to_string (strp), number_to_string_p_pp (strp), format_number (strp basically) */
  /* the rest of s7 assumes nlen is set to the correct length */
  block_t *b;
  char *p;
  s7_int len, str_len;

#if WITH_GMP
  if (s7_is_bignum(obj))
    return(big_number_to_string_with_radix(sc, obj, radix, width, nlen, P_WRITE));
  /* this ignores precision because it's way too hard to get the mpfr string to look like
   *   C's output -- we either have to call mpfr_get_str twice (the first time just to
   *   find out what the exponent is and how long the string actually is), or we have
   *   to do messy string manipulations.  So (format #f "",3F" pi) ignores the "3" and
   *   prints the full string.  And don't even think about mpfr_snprintf!
   */
#endif
  if (radix == 10)
    {
      p = number_to_string_base_10(sc, obj, width, precision, float_choice, nlen, P_WRITE);
      return(string_to_block(sc, p, *nlen));
    }

  switch (type(obj))
    {
    case T_INTEGER:
      {
	size_t len1;
	b = mallocate(sc, (128 + width));
	p = (char *)block_data(b);
	len1 = integer_to_string_any_base(p, integer(obj), radix);
	if ((size_t)width > len1)
	  {
	    size_t start = width - len1;
	    memmove((void *)(p + start), (void *)p, len1);
	    local_memset((void *)p, (int)' ', start);
	    p[width] = '\0';
	    *nlen = width;
	  }
	else *nlen = len1;
	return(b);
      }

    case T_RATIO:
      {
	size_t len1, len2;
	str_len = 256 + width;
	b = mallocate(sc, str_len);
	p = (char *)block_data(b);
	len1 = integer_to_string_any_base(p, numerator(obj), radix);
	p[len1] = '/';
	len2 = integer_to_string_any_base((char *)(p + len1 + 1), denominator(obj), radix);
        len = len1 + 1 + len2;
        p[len] = '\0';
      }
      break;

    case T_REAL:
      {
	int32_t i;
	s7_int int_part, nsize;
	s7_double x = real(obj), frac_part, min_frac, base;
	bool sign = false;
	char n[128], d[256];

	if (is_NaN(x))
	  return(string_to_block(sc, "+nan.0", *nlen = 6));
	if (is_inf(x))
	  {
	    if (x < 0.0)
	      return(string_to_block(sc, "-inf.0", *nlen = 6));
	    return(string_to_block(sc, "+inf.0", *nlen = 6));
	  }
	if (x < 0.0)
	  {
	    sign = true;
	    x = -x;
	  }
	if (x > 1.0e18) /* i.e. close to or greater than most-positive-fixnum (9.22e18), so the code below is unlikely to work, (format #f "~X" 1e19) */
	  {
	    int32_t ep = (int32_t)floor(log(x) / log((double)radix));
	    block_t *b1;
	    len = 0;
	    b = number_to_string_with_radix(sc, wrap_real(sc, x / pow((double)radix, (double)ep)), /* divide it down to one digit, then the fractional part */
					    radix, width, precision, float_choice, &len);
	    b1 = mallocate(sc, len + 8);
	    p = (char *)block_data(b1);
	    p[0] = '\0';
	    (*nlen) = catstrs(p, len + 8, (sign) ? "-" : "", (char *)block_data(b), (radix == 16) ? "@" : "e", integer_to_string_no_length(sc, ep), (char *)NULL);
	    liberate(sc, b);
	    return(b1);
	  }

	int_part = (s7_int)floor(x);
	frac_part = x - int_part;
	nsize = integer_to_string_any_base(n, int_part, radix);
	min_frac = dpow(radix, -precision);

	/* doesn't this assume precision < 128/256 and that we can fit in 256 digits (1e308)? */
	for (i = 0, base = radix; (i < precision) && (frac_part > min_frac); i++, base *= radix)
	  {
	    s7_int ipart = (s7_int)(frac_part * base);
	    if (ipart >= radix)         /* rounding confusion */
	      ipart = radix - 1;
	    frac_part -= (ipart / base);
	    /* d[i] = ((const char *)"0123456789abcdef")[ipart]; */
	    d[i] = dignum[ipart];
	  }
	if (i == 0)
	  d[i++] = '0';
	d[i] = '\0';
	b = mallocate(sc, 256);
        p = (char *)block_data(b);
	/* much faster than catstrs because we know the string lengths */
	{
	  char *pt = p;
	  if (sign) {pt[0] = '-'; pt++;}
	  memcpy(pt, n, nsize);
	  pt += nsize;
	  pt[0] = '.';
	  pt++;
	  memcpy(pt, d, i);
	  pt[i] = '\0';
	  /* len = ((sign) ? 1 : 0) + 1 + nsize + i; */
	  len = pt + i - p;
	}
	str_len = 256;
      }
      break;

    default:
      {
	char *pt;
	s7_int real_len = 0, imag_len = 0;
	block_t *n = number_to_string_with_radix(sc, wrap_real(sc, real_part(obj)), radix, 0, precision, float_choice, &real_len); /* include floatify */
	block_t *d = number_to_string_with_radix(sc, wrap_real(sc, imag_part(obj)), radix, 0, precision, float_choice, &imag_len);
	char *dp = (char *)block_data(d);
	b = mallocate(sc, 512);
	p = (char *)block_data(b);
	pt = p;
	memcpy(pt, (void *)block_data(n), real_len);
	pt += real_len;
	if ((dp[0] != '+') && (dp[0] != '-')) {pt[0] = '+'; pt++;}
	memcpy(pt, dp, imag_len);
	pt[imag_len] = 'i';
	pt[imag_len + 1] = '\0';
	len = pt + imag_len + 1 - p;
	str_len = 512;
	liberate(sc, n);
	liberate(sc, d);
      }
      break;
    }

  if (width > len)
    {
      s7_int spaces;
      if (width >= str_len)
	{
	  str_len = width + 1;
	  b = reallocate(sc, b, str_len);
	  p = (char *)block_data(b);
	}
      spaces = width - len;
      p[width] = '\0';
      memmove((void *)(p + spaces), (void *)p, len);
      local_memset((void *)p, (int)' ', spaces);
      (*nlen) = width;
    }
  else (*nlen) = len;
  return(b);
}

char *s7_number_to_string(s7_scheme *sc, s7_pointer obj, s7_int radix)
{
  s7_int nlen = 0;
  block_t *b = number_to_string_with_radix(sc, obj, radix, 0, 20, 'g', &nlen);  /* (log top 10) so we get all the digits in base 10 (??) */
  char *str = copy_string_with_length((char *)block_data(b), nlen);
  liberate(sc, b);
  return(str);
}

static s7_pointer g_number_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_number_to_string "(number->string num (radix 10)) converts the number num into a string."
  #define Q_number_to_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_number_symbol, sc->is_integer_symbol)

  s7_int nlen = 0, radix; /* ignore cppcheck complaint about radix! */
  char *res;
  s7_pointer x = car(args);

  if (!is_number(x))
    return(method_or_bust_with_type(sc, x, sc->number_to_string_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer y = cadr(args);
      if (s7_is_integer(y))
	radix = s7_integer_clamped_if_gmp(sc, y);
      else return(method_or_bust(sc, y, sc->number_to_string_symbol, args, T_INTEGER, 2));
      if ((radix < 2) || (radix > 16))
	out_of_range(sc, sc->number_to_string_symbol, int_two, y, a_valid_radix_string);
#if (WITH_GMP)
      if (!s7_is_bignum(x))
#endif
	{
	  block_t *b = number_to_string_with_radix(sc, x, radix, 0, sc->float_format_precision, 'g', &nlen);
	  return(block_to_string(sc, b, nlen));
	}}
#if WITH_GMP
  else radix = 10;
  if (s7_is_bignum(x))
    {
      block_t *b = big_number_to_string_with_radix(sc, x, radix, 0, &nlen, P_WRITE);
      return(block_to_string(sc, b, nlen));
    }
  res = number_to_string_base_10(sc, x, 0, sc->float_format_precision, 'g', &nlen, P_WRITE);
#else
  if (is_t_integer(x))
    {
      if (has_number_name(x))
	{
	  nlen = number_name_length(x);
	  res = (char *)number_name(x);
	}
      else res = integer_to_string(sc, integer(x), &nlen);
    }
  else res = number_to_string_base_10(sc, x, 0, sc->float_format_precision, 'g', &nlen, P_WRITE);
#endif
  return(inline_make_string_with_length(sc, res, nlen));
}

static s7_pointer number_to_string_p_p(s7_scheme *sc, s7_pointer p)
{
#if WITH_GMP
  return(g_number_to_string(sc, set_plist_1(sc, p)));
#else
  s7_int nlen = 0;
  char *res;
  if (!is_number(p))
    return(method_or_bust_with_type_one_arg_p(sc, p, sc->number_to_string_symbol, a_number_string));
  res = number_to_string_base_10(sc, p, 0, sc->float_format_precision, 'g', &nlen, P_WRITE);
  return(inline_make_string_with_length(sc, res, nlen));
#endif
}

static s7_pointer number_to_string_p_i(s7_scheme *sc, s7_int p)
{
  s7_int nlen = 0;
  char *res = integer_to_string(sc, p, &nlen);
  return(inline_make_string_with_length(sc, res, nlen));
}
/* not number_to_string_p_d! */

static s7_pointer number_to_string_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
#if WITH_GMP
  return(g_number_to_string(sc, set_plist_2(sc, p1, p2)));
#else
  s7_int nlen = 0, radix;
  block_t *b;

  if (!is_number(p1))
    wrong_type_argument_with_type(sc, sc->number_to_string_symbol, 1, p1, a_number_string);
  if (!is_t_integer(p2))
    wrong_type_argument(sc, sc->number_to_string_symbol, 2, p2, T_INTEGER);
  radix = integer(p2);
  if ((radix < 2) || (radix > 16))
    out_of_range(sc, sc->number_to_string_symbol, int_two, p2, a_valid_radix_string);

  b = number_to_string_with_radix(sc, p1, radix, 0, sc->float_format_precision, 'g', &nlen);
  return(block_to_string(sc, b, nlen));
#endif
}


/* -------------------------------------------------------------------------------- */
#define CTABLE_SIZE 256
static bool *exponent_table, *slashify_table, *char_ok_in_a_name, *white_space, *number_table, *symbol_slashify_table;
static int32_t *digits;

static void init_ctables(void)
{
  exponent_table = (bool *)Calloc(CTABLE_SIZE, sizeof(bool));
  slashify_table = (bool *)Calloc(CTABLE_SIZE, sizeof(bool));
  symbol_slashify_table = (bool *)Calloc(CTABLE_SIZE, sizeof(bool));
  char_ok_in_a_name = (bool *)Calloc(CTABLE_SIZE, sizeof(bool));
  white_space = (bool *)Calloc(CTABLE_SIZE + 1, sizeof(bool));
  white_space++;      /* leave white_space[-1] false for white_space[EOF] */
  number_table = (bool *)Calloc(CTABLE_SIZE, sizeof(bool));
  digits = (int32_t *)Calloc(CTABLE_SIZE, sizeof(int32_t));

  for (int32_t i = 0; i < CTABLE_SIZE; i++)
    {
      char_ok_in_a_name[i] = true;
      white_space[i] = false;
      digits[i] = 256;
      number_table[i] = false;
    }

  char_ok_in_a_name[0] = false;
  char_ok_in_a_name[(uint8_t)'('] = false;  /* cast for C++ */
  char_ok_in_a_name[(uint8_t)')'] = false;
  char_ok_in_a_name[(uint8_t)';'] = false;
  char_ok_in_a_name[(uint8_t)'\t'] = false;
  char_ok_in_a_name[(uint8_t)'\n'] = false;
  char_ok_in_a_name[(uint8_t)'\r'] = false;
  char_ok_in_a_name[(uint8_t)' '] = false;
  char_ok_in_a_name[(uint8_t)'"'] = false;

  white_space[(uint8_t)'\t'] = true;
  white_space[(uint8_t)'\n'] = true;
  white_space[(uint8_t)'\r'] = true;
  white_space[(uint8_t)'\f'] = true;
  white_space[(uint8_t)'\v'] = true;
  white_space[(uint8_t)' '] = true;
  white_space[(uint8_t)'\205'] = true; /* 133 */
  white_space[(uint8_t)'\240'] = true; /* 160 */

  /* surely only 'e' is needed... */
  exponent_table[(uint8_t)'e'] = true; exponent_table[(uint8_t)'E'] = true;
  exponent_table[(uint8_t)'@'] = true;
#if WITH_EXTRA_EXPONENT_MARKERS
  exponent_table[(uint8_t)'s'] = true; exponent_table[(uint8_t)'S'] = true;
  exponent_table[(uint8_t)'f'] = true; exponent_table[(uint8_t)'F'] = true;
  exponent_table[(uint8_t)'d'] = true; exponent_table[(uint8_t)'D'] = true;
  exponent_table[(uint8_t)'l'] = true; exponent_table[(uint8_t)'L'] = true;
#endif
  for (int32_t i = 0; i < 32; i++) slashify_table[i] = true;
  for (int32_t i = 127; i < 160; i++) slashify_table[i] = true;
  slashify_table[(uint8_t)'\\'] = true;
  slashify_table[(uint8_t)'"'] = true;
  slashify_table[(uint8_t)'\n'] = false;

  for (int32_t i = 0; i < CTABLE_SIZE; i++)
    symbol_slashify_table[i] = ((slashify_table[i]) || (!char_ok_in_a_name[i])); /* force use of (symbol ...) for cases like '(ab) as symbol */

  digits[(uint8_t)'0'] = 0; digits[(uint8_t)'1'] = 1; digits[(uint8_t)'2'] = 2; digits[(uint8_t)'3'] = 3; digits[(uint8_t)'4'] = 4;
  digits[(uint8_t)'5'] = 5; digits[(uint8_t)'6'] = 6; digits[(uint8_t)'7'] = 7; digits[(uint8_t)'8'] = 8; digits[(uint8_t)'9'] = 9;
  digits[(uint8_t)'a'] = 10; digits[(uint8_t)'A'] = 10;
  digits[(uint8_t)'b'] = 11; digits[(uint8_t)'B'] = 11;
  digits[(uint8_t)'c'] = 12; digits[(uint8_t)'C'] = 12;
  digits[(uint8_t)'d'] = 13; digits[(uint8_t)'D'] = 13;
  digits[(uint8_t)'e'] = 14; digits[(uint8_t)'E'] = 14;
  digits[(uint8_t)'f'] = 15; digits[(uint8_t)'F'] = 15;

  number_table[(uint8_t)'0'] = true; number_table[(uint8_t)'1'] = true; number_table[(uint8_t)'2'] = true; number_table[(uint8_t)'3'] = true;
  number_table[(uint8_t)'4'] = true; number_table[(uint8_t)'5'] = true; number_table[(uint8_t)'6'] = true; number_table[(uint8_t)'7'] = true;
  number_table[(uint8_t)'8'] = true; number_table[(uint8_t)'9'] = true; number_table[(uint8_t)'.'] = true;
  number_table[(uint8_t)'+'] = true;
  number_table[(uint8_t)'-'] = true;
  number_table[(uint8_t)'#'] = true;
}

#define is_white_space(C) white_space[C]
  /* this is much faster than C's isspace, and does not depend on the current locale.
   * if c == EOF (-1), it indexes into the empty (0) slot we preallocated below white_space
   */

/* -------------------------------- *#readers* -------------------------------- */
static s7_pointer check_sharp_readers(s7_scheme *sc, const char *name)
{
  s7_pointer value = sc->F, args = sc->F;
  bool need_loader_port = is_loader_port(current_input_port(sc));

  /* *#reader* is assumed to be an alist of (char . proc)
   *    where each proc takes one argument, the string from just beyond the "#" to the next delimiter.
   *    The procedure can call read-char to read ahead in the current-input-port.
   *    If it returns anything other than #f, that is the value of the sharp expression.
   *    Since #f means "nothing found", it is tricky to handle #F:
   *       (cons #\F (lambda (str) (and (string=? str "F") (list 'not #t)))) ; or ''#f used in lint.scm
   * This search happens after #|, #t, and #f (and #nD for multivectors?). #! has a fallback.  Added #_ later)
   */
  if (need_loader_port)
    clear_loader_port(current_input_port(sc));

  /* normally read* can't read from current_input_port(sc) if it is in use by the loader, but here we are deliberately making that possible */
  for (s7_pointer reader = slot_value(sc->sharp_readers); is_not_null(reader); reader = cdr(reader))
    if (name[0] == s7_character(caar(reader)))
      {
	if (args == sc->F)
	  args = set_plist_1(sc, s7_make_string_wrapper(sc, name));
	/* args is GC protected by s7_apply_function?? (placed on the stack) */
	value = s7_apply_function(sc, cdar(reader), args); /* this is much less error-safe than s7_call */
	if (value != sc->F)
	  break;
      }
  if (need_loader_port)
    set_loader_port(current_input_port(sc));
  return(value);
}

static s7_pointer g_sharp_readers_set(s7_scheme *sc, s7_pointer args)
{
  /* new value must be either () or a proper list of conses (char . func) */
  s7_pointer x;
  if (is_null(cadr(args))) return(cadr(args));
  if (!is_pair(cadr(args)))
    s7_error_nr(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args)));
  for (x = cadr(args); is_pair(x); x = cdr(x))
    if ((!is_pair(car(x))) ||
	(!is_character(caar(x))) ||
	(!is_procedure(cdar(x))))
      s7_error_nr(sc, sc->wrong_type_arg_symbol,
		  set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args)));
  if (!is_null(x))
    s7_error_nr(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args)));
  return(cadr(args));
}

static s7_pointer make_undefined(s7_scheme *sc, const char* name)
{
  s7_int len = safe_strlen(name);
  char *newstr = (char *)Malloc(len + 2);
  s7_pointer p;
  new_cell(sc, p, T_UNDEFINED | T_IMMUTABLE);
  newstr[0] = '#';
  memcpy((void *)(newstr + 1), (void *)name, len);
  newstr[len + 1] = '\0';
  if (sc->undefined_constant_warnings) s7_warn(sc, len + 32, "%s is undefined\n", newstr);
  undefined_set_name_length(p, len + 1);
  undefined_name(p) = newstr;
  add_undefined(sc, p);
  return(p);
}

static int32_t inchar(s7_pointer pt)
{
  int32_t c;
  if (is_file_port(pt))
    c = fgetc(port_file(pt)); /* not uint8_t! -- could be EOF */
  else
    {
      if (port_data_size(pt) <= port_position(pt))
	return(EOF);
      c = (uint8_t)port_data(pt)[port_position(pt)++];
    }
  if (c == '\n')
    port_line_number(pt)++;
  return(c);
}

static void backchar(char c, s7_pointer pt)
{
  if (c == '\n')
    port_line_number(pt)--;

  if (is_file_port(pt))
    ungetc(c, port_file(pt));
  else
    if (port_position(pt) > 0)
      port_position(pt)--;
}

static void resize_strbuf(s7_scheme *sc, s7_int needed_size)
{
  s7_int old_size = sc->strbuf_size;
  while (sc->strbuf_size <= needed_size) sc->strbuf_size *= 2;
  sc->strbuf = (char *)Realloc(sc->strbuf, sc->strbuf_size);
  for (s7_int i = old_size; i < sc->strbuf_size; i++) sc->strbuf[i] = '\0';
}

static s7_pointer *chars;

static s7_pointer unknown_sharp_constant(s7_scheme *sc, const char *name, s7_pointer pt)
{
  if (hook_has_functions(sc->read_error_hook))  /* check *read-error-hook* */
    {
      bool old_history_enabled = s7_set_history_enabled(sc, false); /* see sc->error_hook for a more robust way to handle this */
      s7_pointer result = s7_call(sc, sc->read_error_hook, set_plist_2(sc, sc->T, s7_make_string_wrapper(sc, name)));
      s7_set_history_enabled(sc, old_history_enabled);
      if (result != sc->unspecified)
	return(result);
    }
  if (pt) /* #<"..."> which gets here as name="#<" */
    {
      s7_int len = safe_strlen(name);
      if ((name[len - 1] != '>') &&
	  (is_input_port(pt)) &&
	  (pt != sc->standard_input))
	{
	  if (s7_peek_char(sc, pt) != chars[(uint8_t)'"']) /* if not #<"...">, just return it */
	    return(make_undefined(sc, name));

	  if (is_string_port(pt)) /* probably unnecessary (see below) */
	    {
	      s7_int c = inchar(pt);
	      const char *pstart = (const char *)(port_data(pt) + port_position(pt));
	      const char *p = strchr(pstart, (int)'"');
	      s7_int added_len;
	      char *buf;
	      s7_pointer res;

	      if (!p)
		{
		  backchar(c, pt);
		  return(make_undefined(sc, name));
		}
	      p++;
	      while (char_ok_in_a_name[(uint8_t)(*p)]) {p++;}
	      added_len = (s7_int)(p - pstart);                 /* p is one past '>' presumably */
	      /* we can't use strbuf here -- it might be the source of the "name" argument! */
	      buf = (char *)Malloc(len + added_len + 2);
	      memcpy((void *)buf, (void *)name, len);
	      buf[len] = '"';                                   /* from inchar */
	      memcpy((void *)(buf + len + 1), (void *)pstart, added_len);
	      buf[len + added_len + 1] = 0;
	      port_position(pt) += added_len;
	      res = make_undefined(sc, (const char *)buf);
	      free(buf);
	      return(res);
	    }}}
  return(make_undefined(sc, name));
}

static s7_pointer make_atom(s7_scheme *sc, char *q, int32_t radix, bool want_symbol, bool with_error);
#define SYMBOL_OK true
#define NO_SYMBOLS false

static s7_pointer make_sharp_constant(s7_scheme *sc, const char *name, bool with_error, s7_pointer pt, bool error_if_bad_number)
{
  /* name is the stuff after the '#', return sc->nil if not a recognized #... entity */
  if ((!name) || (!*name)) /* (string->number "#") for example */
    return(make_undefined(sc, name));

  /* stupid r7rs special cases */
  if ((name[0] == 't') &&
      ((name[1] == '\0') || (c_strings_are_equal(name, "true"))))
    return(sc->T);

  if ((name[0] == 'f') &&
      ((name[1] == '\0') || (c_strings_are_equal(name, "false"))))
    return(sc->F);

  if (name[0] == '_')
    {
      /* this needs to be unsettable via *#readers*:
       *    (set! *#readers* (list (cons #\_ (lambda (str) (string->symbol (substring str 1))))))
       *    (let ((+ -)) (#_+ 1 2)): -1
       */
      s7_pointer sym = make_symbol(sc, (char *)(name + 1));
      if ((!is_gensym(sym)) && (is_slot(initial_slot(sym))))
	return(initial_value(sym));
      /* here we should not necessarily raise an error that *_... is undefined.  reader-cond, for example, needs to
       *    read undefined #_ vals that it will eventually discard.
       */
      return(make_undefined(sc, name));    /* (define x (with-input-from-string "(#_asdf 1 2)" read)) (type-of (car x)) -> undefined? */
    }

  if (is_not_null(slot_value(sc->sharp_readers)))
    {
      s7_pointer x = check_sharp_readers(sc, name);
      if (x != sc->F)
	return(x);
    }

  if ((name[0] == '\0') || name[1] == '\0')
    return(unknown_sharp_constant(sc, name, pt)); /* pt here because #<"..."> comes here as "<" so name[1] is '\0'! */

  switch (name[0])
    {
      /* -------- #< ... > -------- */
    case '<':
      if (c_strings_are_equal(name, "<unspecified>")) return(sc->unspecified);
      if (c_strings_are_equal(name, "<undefined>"))   return(sc->undefined);
      if (c_strings_are_equal(name, "<eof>"))         return(eof_object);
      return(unknown_sharp_constant(sc, name, pt));

      /* -------- #o #x #b -------- */
    case 'o':   /* #o (octal) */
    case 'x':   /* #x (hex) */
    case 'b':   /* #b (binary) */
      {
	s7_pointer res = make_atom(sc, (char *)(name + 1), (name[0] == 'o') ? 8 : ((name[0] == 'x') ? 16 : 2), NO_SYMBOLS, with_error);
	if ((error_if_bad_number) && (res == sc->F)) /* #b32 etc but not if called from string->number */
	  {
	    char buf[256];
	    size_t len = snprintf(buf, 256, "#%s is not a number", name);
	    s7_error_nr(sc, sc->read_error_symbol,
			set_elist_1(sc, s7_make_string_with_length(sc, buf, len))); /* can't use wrap_string here (buf is local) */
	  }
	return(res);
      }

      /* -------- #\... -------- */
    case '\\':
      if (name[2] == 0)                             /* the most common case: #\a */
	return(chars[(uint8_t)(name[1])]);
      /* not uint32_t here!  (uint32_t)255 (as a char) returns -1!! */
      switch (name[1])
	{
	case 'n':
	  if ((c_strings_are_equal(name + 1, "null")) ||
	      (c_strings_are_equal(name + 1, "nul")))
	    return(chars[0]);

	  if (c_strings_are_equal(name + 1, "newline"))
	    return(chars[(uint8_t)'\n']);
	  break;

	case 's': if (c_strings_are_equal(name + 1, "space"))     return(chars[(uint8_t)' ']);  break;
	case 'r': if (c_strings_are_equal(name + 1, "return"))    return(chars[(uint8_t)'\r']); break;
	case 'l': if (c_strings_are_equal(name + 1, "linefeed"))  return(chars[(uint8_t)'\n']); break;
	case 't': if (c_strings_are_equal(name + 1, "tab"))       return(chars[(uint8_t)'\t']); break;
	case 'a': if (c_strings_are_equal(name + 1, "alarm"))     return(chars[7]);             break;
	case 'b': if (c_strings_are_equal(name + 1, "backspace")) return(chars[8]);             break;
	case 'e': if (c_strings_are_equal(name + 1, "escape"))    return(chars[0x1b]);          break;
	case 'd': if (c_strings_are_equal(name + 1, "delete"))    return(chars[0x7f]);          break;

	case 'x':
	  /* #\x is just x, but apparently #\x<num> is int->char? #\x65 -> #\e, and #\xcebb is lambda? */
	  {
	    /* sscanf here misses errors like #\x1.4, but make_atom misses #\x6/3,
	     *   #\x#b0, #\x#e0.0, #\x-0, #\x#e0e100 etc, so we have to do it at an even lower level.
	     * another problem: #\xbdca2cbec overflows so lval is -593310740 -> segfault unless caught
	     */
	    bool happy = true;
	    char *tmp = (char *)(name + 2);
	    int32_t lval = 0;

	    while ((*tmp) && (happy) && (lval >= 0) && (lval < 256))
	      {
		int32_t dig = digits[(int32_t)(*tmp++)];
		if (dig < 16)
		  lval = dig + (lval * 16);
		else happy = false;
	      }
	    if ((happy) &&
		(lval < 256) &&
		(lval >= 0))
	      return(chars[lval]);
	  }
	  break;
	}}
  return(unknown_sharp_constant(sc, name, NULL));
}

static s7_int string_to_integer(const char *str, int32_t radix, bool *overflow)
{
  bool negative = false;
  s7_int lval = 0;
  int32_t dig;
  char *tmp = (char *)str;
#if WITH_GMP
  char *tmp1;
#endif
  if (str[0] == '+')
    tmp++;
  else
    if (str[0] == '-')
      {
	negative = true;
	tmp++;
      }
  while (*tmp == '0') {tmp++;};
#if WITH_GMP
  tmp1 = tmp;
#endif
 if (radix == 10)
    {
      while (true)
	{
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig > 9) break;
#if HAVE_OVERFLOW_CHECKS
	  if ((multiply_overflow(lval, (s7_int)10, &lval)) ||
	      (add_overflow(lval, (s7_int)dig, &lval)))
	    {
	      if ((radix == 10) &&
		  (strncmp(str, "-9223372036854775808", 20) == 0) &&
		  (digits[(uint8_t)(*tmp++)] > 9)) /* i.e. next thing is not a digit -- "/" for example */
		return(S7_INT64_MIN);
	      *overflow = true;
	      return((negative) ? S7_INT64_MIN : S7_INT64_MAX);
	      break;
	    }
#else
	  lval = dig + (lval * 10);
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig > 9) break;
	  lval = dig + (lval * 10);
#endif
	}}
  else
    while (true)
      {
	dig = digits[(uint8_t)(*tmp++)];
	if (dig >= radix) break;
#if HAVE_OVERFLOW_CHECKS && (!WITH_GMP)
	{
	  s7_int oval = 0;
	  if (multiply_overflow(lval, (s7_int)radix, &oval))
	    {
	      /* maybe a bad idea!  #xffffffffffffffff -> -1??? this is needed for 64-bit number hacks (see s7test.scm bit-reverse) */
	      if ((radix == 16) &&
		  (digits[(uint8_t)(*tmp)] >= radix))
		{
		  lval -= 576460752303423488LL; /* turn off sign bit */
		  lval *= radix;
		  lval += dig;
		  lval -= 9223372036854775807LL;
		  return(lval - 1);
		}
	      lval = oval; /* old case */
	      if ((lval == S7_INT64_MIN)  && (digits[(uint8_t)(*tmp++)] > 9))
		return(lval);
	      *overflow = true;
	      break;
	    }
	  else lval = oval;
	  if (add_overflow(lval, (s7_int)dig, &lval))
	    {
	      if (lval == S7_INT64_MIN) return(lval);
	      *overflow = true;
	      break;
	    }}
#else
	lval = dig + (lval * radix);
	dig = digits[(uint8_t)(*tmp++)];
	if (dig >= radix) break;
	lval = dig + (lval * radix);
#endif
      }

#if WITH_GMP
 if (!(*overflow))
   (*overflow) = ((lval > S7_INT32_MAX) ||
		  ((tmp - tmp1) > s7_int_digits_by_radix[radix]));
  /* this tells the string->number readers to create a bignum.  We need to be very conservative here to catch contexts such as (/ 1/524288 19073486328125) */
#endif
  return((negative) ? -lval : lval);
}

/*  9223372036854775807                9223372036854775807
 * -9223372036854775808               -9223372036854775808
 * 0000000000000000000000000001.0     1.0
 * 1.0000000000000000000000000000     1.0
 * 1000000000000000000000000000.0e-40 1.0e-12
 * 0.0000000000000000000000000001e40  1.0e12
 * 1.0e00000000000000000001           10.0
 */

#if WITH_GMP
static s7_double string_to_double_with_radix(const char *ur_str, int32_t radix, bool *overflow)
#else
#define string_to_double_with_radix(Str, Rad, Over) string_to_double_with_radix_1(Str, Rad)
static s7_double string_to_double_with_radix_1(const char *ur_str, int32_t radix)
#endif
{
  /* strtod follows LANG which is not what we want (only "." is decimal point in Scheme).
   *   To overcome LANG in strtod would require screwing around with setlocale which never works.
   *   So we use our own code -- according to valgrind, this function is much faster than strtod.
   * comma as decimal point causes ambiguities: `(+ ,1 2) etc
   */

  int32_t i, sign = 1, frac_len, int_len, dig, exponent = 0;
  int32_t max_len = s7_int_digits_by_radix[radix];
  int64_t int_part = 0, frac_part = 0;
  char *str = (char *)ur_str;
  char *ipart, *fpart;
  s7_double dval = 0.0;

  /* there's an ambiguity in number notation here if we allow "1e1" or "1.e1" in base 16 (or 15) -- is e a digit or an exponent marker?
   *   but 1e+1, for example disambiguates it -- kind of messy! -- the scheme spec says "e" can only occur in base 10.
   * '@' can now be used as the exponent marker (26-Mar-12).
   * Another slight ambiguity: 1+1/2i is parsed as 1 + 0.5i, not 1+1/(2i), or (1+1)/(2i) or (1+1/2)i etc
   */
  if (*str == '-')
    {
      str++;
      sign = -1;
    }
  else
    if (*str == '+')
      str++;
  while (*str == '0') {str++;};

  ipart = str;
  while (digits[(int32_t)(*str)] < radix) str++;
  int_len = str - ipart;

  if (*str == '.') str++;
  fpart = str;
  while (digits[(int32_t)(*str)] < radix) str++;
  frac_len = str - fpart;

  if ((*str) && (exponent_table[(uint8_t)(*str)]))
    {
      bool exp_negative = false;
      str++;
      if (*str == '+')
	str++;
      else
	if (*str == '-')
	  {
	    str++;
	    exp_negative = true;
	  }
      while ((dig = digits[(int32_t)(*str++)]) < 10) /* exponent itself is always base 10 */
	{
#if HAVE_OVERFLOW_CHECKS
	  if ((int32_multiply_overflow(exponent, 10, &exponent)) ||
	      (int32_add_overflow(exponent, dig, &exponent)))
	    {
	      exponent = 1000000; /* see below */
	      break;
	    }
#else
	  exponent = dig + (exponent * 10);
#endif
	}
#if (!defined(__GNUC__)) || ((__GNUC__ < 5) && (!defined(__clang__)))
      if (exponent < 0)         /* we overflowed, so make sure we notice it below (need to check for 0.0e... first) (Brian Damgaard) */
	exponent = 1000000;     /*   see below for examples -- this number needs to be very big but not too big for add */
#endif
      if (exp_negative)
	exponent = -exponent;

      /*           2e12341234123123123123213123123123 -> 0.0
       * but exp len is not the decider: 2e00000000000000000000000000000000000000001 -> 20.0
       * first zero: 2e123412341231231231231
       * then:     2e12341234123123123123123123 -> inf
       * then:     2e123412341231231231231231231231231231 -> 0.0
       *           2e-123412341231231231231 -> inf
       * but:      0e123412341231231231231231231231231231
       */
    }

#if WITH_GMP
  /* 9007199254740995.0 */
  if (int_len + frac_len >= max_len)
    {
      (*overflow) = true;
      return(0.0);
    }
#endif
  str = ipart;
  if ((int_len + exponent) > max_len)
    {
      /*  12341234.56789e12                   12341234567889999872.0              1.234123456789e+19
       * -1234567890123456789.0              -1234567890123456768.0              -1.2345678901235e+18
       *  12345678901234567890.0              12345678901234567168.0              1.2345678901235e+19
       *  123.456e30                          123456000000000012741097792995328.0 1.23456e+32
       *  12345678901234567890.0e12           12345678901234569054409354903552.0  1.2345678901235e+31
       *  1.234567890123456789012e30          1234567890123456849145940148224.0   1.2345678901235e+30
       *  1e20                                100000000000000000000.0             1e+20
       *  1234567890123456789.0               1234567890123456768.0               1.2345678901235e+18
       *  123.456e16                          1234560000000000000.0               1.23456e+18
       *  98765432101234567890987654321.0e-5  987654321012345728401408.0          9.8765432101235e+23
       *  98765432101234567890987654321.0e-10 9876543210123456512.0               9.8765432101235e+18
       *  0.00000000000000001234e20           1234.0
       *  0.000000000000000000000000001234e30 1234.0
       *  0.0000000000000000000000000000000000001234e40 1234.0
       *  0.000000000012345678909876543210e15 12345.678909877
       *  0e1000                              0.0
       */

      for (i = 0; i < max_len; i++)
	{
	  dig = digits[(int32_t)(*str++)];
	  if (dig < radix)
	    int_part = dig + (int_part * radix);
	  else break;
	}

      /* if the exponent is huge, check for 0 int_part and frac_part before complaining (0e1000 or 0.0e1000)
       */
      if ((int_part == 0) &&
	  (exponent > max_len))
	{
	  /* if frac_part is also 0, return 0.0 */
	  if (frac_len == 0) return(0.0);
	  str = fpart;
	  while ((dig = digits[(int32_t)(*str++)]) < radix)
	    frac_part = dig + (frac_part * radix);
	  if (frac_part == 0) return(0.0);
#if WITH_GMP
	  (*overflow) = true;
#endif
	}
#if WITH_GMP
      (*overflow) = ((int_part > 0) || (exponent > 20));    /* .1e310 is a tricky case */
#endif
      if (int_part != 0) /* 0.<310 zeros here>1e310 for example -- pow (via dpow) thinks it has to be too big, returns Nan,
			  *   then Nan * 0 -> Nan and the NaN propagates
			  */
	{
	  if (int_len <= max_len)
	    dval = int_part * dpow(radix, exponent);
	  else dval = int_part * dpow(radix, exponent + int_len - max_len);
	}
      else dval = 0.0;

      /* shift by exponent, but if int_len > max_len then we assumed (see below) int_len - max_len 0's on the left */
      /*   using int_to_int or table lookups here instead of pow did not make any difference in speed */

      if (int_len < max_len)
	{
	  str = fpart;
	  for (int32_t k = 0; (frac_len > 0) && (k < exponent); k += max_len)
	    {
	      int32_t flen = (frac_len > max_len) ? max_len : frac_len; /* ? */
	      frac_len -= max_len;
	      frac_part = 0;
	      for (i = 0; i < flen; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
	      if (frac_part != 0)                                /* same pow->NaN problem as above can occur here */
		dval += frac_part * dpow(radix, exponent - flen - k);
	    }}
      else
	/* some of the fraction is in the integer part before the negative exponent shifts it over */
	if (int_len > max_len)
	  {
	    int32_t ilen = int_len - max_len;                          /* we read these above */
	    /* str should be at the last digit we read */
	    if (ilen > max_len)
	      ilen = max_len;
	    for (i = 0; i < ilen; i++)
	      frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
	    dval += frac_part * dpow(radix, exponent - ilen);
	  }
      return(sign * dval);
    }

  /* int_len + exponent <= max_len */
  if (int_len <= max_len)
    {
      int32_t int_exponent = exponent;
      /* a better algorithm (since the inaccuracies are in the radix^exponent portion):
       *   strip off leading zeros and possible sign,
       *   strip off digits beyond max_len, then remove any trailing zeros.
       *     (maybe fiddle with the lowest order digit here for rounding, but I doubt it matters)
       *   read digits until end of number or max_len reached, ignoring the decimal point
       *   get exponent and use it and decimal point location to position the current result integer
       * this always combines the same integer and the same exponent no matter how the number is expressed.
       */
      if (int_len > 0)
	{
	  char *iend = (char *)(str + int_len - 1);
	  while ((*iend == '0') && (iend != str)) {iend--; int_exponent++;}
	  while (str <= iend)
	    int_part = digits[(int32_t)(*str++)] + (int_part * radix);
	}
      dval = (int_exponent == 0) ? (s7_double)int_part : int_part * dpow(radix, int_exponent);
    }
  else
    {
      int32_t flen, len = int_len + exponent;
      int64_t frpart = 0;

      /* 98765432101234567890987654321.0e-20    987654321.012346
       * 98765432101234567890987654321.0e-29    0.98765432101235
       * 98765432101234567890987654321.0e-30    0.098765432101235
       * 98765432101234567890987654321.0e-28    9.8765432101235
       */
      for (i = 0; i < len; i++)
	int_part = digits[(int32_t)(*str++)] + (int_part * radix);
      flen = -exponent;
      if (flen > max_len)
	flen = max_len;
      for (i = 0; i < flen; i++)
	frpart = digits[(int32_t)(*str++)] + (frpart * radix);
      if (len <= 0)
	dval = int_part + frpart * dpow(radix, len - flen);
      else dval = int_part + frpart * dpow(radix, -flen);
    }

  if (frac_len > 0)
    {
      str = fpart;
      if (frac_len <= max_len)
	{
	  /* splitting out base 10 case saves very little here */
	  /* this ignores trailing zeros, so that 0.3 equals 0.300 */
	  char *fend = (char *)(str + frac_len - 1);

	  while ((*fend == '0') && (fend != str)) {fend--; frac_len--;} /* (= .6 0.6000) */
	  if ((frac_len & 1) == 0)
	    {
	      while (str <= fend)
		{
		  frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
		  frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
		}}
	  else
	    while (str <= fend)
	      frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	  dval += frac_part * dpow(radix, exponent - frac_len);

	  /* 0.6:    frac:    6, exp: 0.10000000000000000555, val: 0.60000000000000008882
	   * 0.60:   frac:   60, exp: 0.01000000000000000021, val: 0.59999999999999997780
	   * 0.6000: frac: 6000, exp: 0.00010000000000000000, val: 0.59999999999999997780
	   * (= 0.6 0.60): #f
	   * (= #i3/5 0.6): #f
	   * so (string->number (number->string num)) == num only if both num's are the same text (or you get lucky)
	   * (= 0.6 6e-1): #t ; but not 60e-2
	   * to fix the 0.60 case, we need to ignore trailing post-dot zeros.
	   */
	}
      else
	{
	  if (exponent <= 0)
	    {
	      for (i = 0; i < max_len; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	      dval += frac_part * dpow(radix, exponent - max_len);
	    }
	  else
	    {
	      /* 1.0123456789876543210e1         10.12345678987654373771
	       * 1.0123456789876543210e10        10123456789.87654304504394531250
	       * 0.000000010000000000000000e10   100.0
	       * 0.000000010000000000000000000000000000000000000e10 100.0
	       * 0.000000012222222222222222222222222222222222222e10 122.22222222222222
	       * 0.000000012222222222222222222222222222222222222e17 1222222222.222222
	       */
	      int_part = 0;
	      for (i = 0; i < exponent; i++)
		int_part = digits[(int32_t)(*str++)] + (int_part * radix);
	      frac_len -= exponent;
	      if (frac_len > max_len)
		frac_len = max_len;
	      for (i = 0; i < frac_len; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
	      dval += int_part + frac_part * dpow(radix, -frac_len);
	    }}}
#if WITH_GMP
  if ((int_part == 0) &&
      (frac_part == 0))
    return(0.0);
  (*overflow) = ((frac_len - exponent) > max_len);
#endif
  return(sign * dval);
}

#if (!WITH_GMP)
static s7_pointer make_undefined_bignum(s7_scheme *sc, const char *name)
{
  s7_int len = safe_strlen(name) + 16;
  block_t *b = mallocate(sc, len);
  char *buf = (char *)block_data(b);
  s7_pointer res;
  snprintf(buf, len, "<bignum: %s>", name);
  res = make_undefined(sc, (const char *)buf); /* 123123123123123123123123123123 -> +inf.0 originally, but now #<bignum: 123123...> */
  liberate(sc, b);
  return(res);
}
#endif

static s7_pointer nan1_or_bust(s7_scheme *sc, s7_double x, char *p, char *q, int32_t radix, bool want_symbol)
{
  s7_int len = safe_strlen(p);
  if (p[len - 1] == 'i')        /* +nan.0[+/-]...i */
    {
      if (len == 6)            /* +nan.0+i */
	return(make_complex_not_0i(sc, x, (p[4] == '+') ? 1.0 : -1.0));
      if ((len > 5) && (len < 1024)) /* make compiler happy */
	{
	  char *ip = copy_string_with_length((const char *)(p + 4), len - 5);
	  s7_pointer imag = make_atom(sc, ip, radix, NO_SYMBOLS, WITHOUT_OVERFLOW_ERROR);
	  free(ip);
	  if (is_real(imag))
	    return(make_complex(sc, x, real_to_double(sc, imag, __func__))); /* +nan.0+2/3i etc */
	}}
  return((want_symbol) ? make_symbol(sc, q) : sc->F);
}

static s7_pointer nan2_or_bust(s7_scheme *sc, s7_double x, char *q, int32_t radix, bool want_symbol)
{
  s7_int len = safe_strlen(q);
  if ((len > 7) && (len < 1024)) /* make compiler happy */
    {
      char *ip = copy_string_with_length((const char *)q, len - 7);
      s7_pointer rl = make_atom(sc, ip, radix, NO_SYMBOLS, WITHOUT_OVERFLOW_ERROR);
      free(ip);
      if (is_real(rl))
	return(make_complex(sc, real_to_double(sc, rl, __func__), x));
    }
  return((want_symbol) ? make_symbol(sc, q) : sc->F);
}

static s7_pointer make_atom(s7_scheme *sc, char *q, int32_t radix, bool want_symbol, bool with_error)
{
  /* make symbol or number from string, a number starts with + - . or digit, but so does 1+ for example */
  #define IS_DIGIT(Chr, Rad) (digits[(uint8_t)Chr] < Rad)

  char c, *p = q;
  bool has_dec_point1 = false;

  c = *p++;
  switch (c)
    {
    case '#':
      /* from string->number, (string->number #xc) */
      return(make_sharp_constant(sc, p, with_error, NULL, false)); /* make_sharp_constant expects the '#' to be removed */

    case '+':
    case '-':
      c = *p++;
      if (c == '.')
	{
	  has_dec_point1 = true;
	  c = *p++;
	}
      if (!c)
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      if (!IS_DIGIT(c, radix))
	{
	  if (has_dec_point1)
	    return((want_symbol) ? make_symbol(sc, q) : sc->F);
	  if (c == 'n')
	    {
	      if (local_strcmp(p, "an.0"))      /* +nan.0 */
		return(real_NaN);
	      if ((local_strncmp(p, "an.0", 4)) &&
		  ((p[4] == '+') || (p[4] == '-')))
		return(nan1_or_bust(sc, NAN, p, q, radix, want_symbol));
	    }
	  if (c == 'i')
	    {
	      if (local_strcmp(p, "nf.0"))  /* +inf.0 */
		return((q[0] == '+') ? real_infinity : real_minus_infinity);
	      if ((local_strncmp(p, "nf.0", 4)) &&
		  ((p[4] == '+') || (p[4] == '-')))
		return(nan1_or_bust(sc, (q[0] == '-') ? -INFINITY : INFINITY, p, q, radix, want_symbol));
	    }
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);
	}
      break;

    case '.':
      has_dec_point1 = true;
      c = *p++;

      if ((!c) || (!IS_DIGIT(c, radix)))
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      break;

    case 'n':
      return((want_symbol) ? make_symbol(sc, q) : sc->F);

    case 'i':
      return((want_symbol) ? make_symbol(sc, q) : sc->F);

    case '0':        /* these two are always digits */
    case '1':
      break;

    default:
      if (!IS_DIGIT(c, radix))
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      break;
    }

  /* now it's possibly a number -- the first character(s) could be part of a number in the current radix */
  {
    char *slash1 = NULL, *slash2 = NULL, *plus = NULL, *ex1 = NULL, *ex2 = NULL;
    bool has_i = false, has_dec_point2 = false;
    int32_t has_plus_or_minus = 0, current_radix;
#if (!WITH_GMP)
    bool overflow = false; /* for string_to_integer */
#endif
    current_radix = radix;  /* current_radix is 10 for the exponent portions, but radix for all the rest */

    for ( ; (c = *p) != 0; ++p)
      {
	/* what about embedded null? (string->number (string #\1 (integer->char 0) #\0))
	 *   currently we stop and return 1, but Guile returns #f.
	 *   this also means we can't use substring_uncopied if (string->number (substring...))
	 */
	if (!IS_DIGIT(c, current_radix))         /* moving this inside the switch statement was much slower */
	  {
	    current_radix = radix;

	    switch (c)
	      {
		/* -------- decimal point -------- */
	      case '.':
		if ((!IS_DIGIT(p[1], current_radix)) &&
		    (!IS_DIGIT(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    if ((has_dec_point1) || (slash1))
		      return((want_symbol) ? make_symbol(sc, q) : sc->F);
		    has_dec_point1 = true;
		  }
		else
		  {
		    if ((has_dec_point2) || (slash2))
		      return((want_symbol) ? make_symbol(sc, q) : sc->F);
		    has_dec_point2 = true;
		  }
		continue;

		/* -------- exponent marker -------- */
#if WITH_EXTRA_EXPONENT_MARKERS
		/* 1st 3d-perspective 0.0f 128.0f 3d 1s -- in 2 million lines of public scheme code, not one actual use! */
	      case 's': case 'S':
	      case 'd': case 'D':
	      case 'f': case 'F':
	      case 'l': case 'L':
#endif
	      case 'e': case 'E':
		if (current_radix > 10) /* see above */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);
		/* fall through -- if '@' used, radices>10 are ok */

	      case '@':
		current_radix = 10;

		if (((ex1) ||
		     (slash1)) &&
		    (has_plus_or_minus == 0)) /* ee */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (((ex2) ||
		     (slash2)) &&
		    (has_plus_or_minus != 0)) /* 1+1.0ee */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if ((!IS_DIGIT(p[-1], radix)) && /* was current_radix but that's always 10! */
		    (p[-1] != '.'))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    ex1 = p;
		    has_dec_point1 = true; /* decimal point illegal from now on */
		  }
		else
		  {
		    ex2 = p;
		    has_dec_point2 = true;
		  }
		p++;
		if ((*p == '-') || (*p == '+')) p++;
		if (IS_DIGIT(*p, current_radix))
		  continue;
		break;

		/* -------- internal + or - -------- */
	      case '+':
	      case '-':
		if (has_plus_or_minus != 0) /* already have the separator */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		has_plus_or_minus = (c == '+') ? 1 : -1;
		plus = (char *)(p + 1);
		/* now check for nan/inf as imaginary part */

		if ((plus[0] == 'n') &&
		    (local_strcmp(plus, "nan.0i")))
		  return(nan2_or_bust(sc, (c == '+') ? NAN : -NAN, q, radix, want_symbol));
		if ((plus[0] == 'i') &&
		    (local_strcmp(plus, "inf.0i")))
		  return(nan2_or_bust(sc, (c == '+') ? INFINITY : -INFINITY, q, radix, want_symbol));
		continue;

		/* ratio marker */
	      case '/':
		if ((has_plus_or_minus == 0) &&
		    ((ex1) ||
		     (slash1) ||
		     (has_dec_point1)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if ((has_plus_or_minus != 0) &&
		    ((ex2) ||
		     (slash2) ||
		     (has_dec_point2)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  slash1 = (char *)(p + 1);
		else slash2 = (char *)(p + 1);

		if ((!IS_DIGIT(p[1], current_radix)) ||
		    (!IS_DIGIT(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		continue;

		/* -------- i for the imaginary part -------- */
	      case 'i':
		if ((has_plus_or_minus != 0) &&
		    (!has_i))
		  {
		    has_i = true;
		    continue;
		  }
		break;

	      default:
		break;
	      }
	    return((want_symbol) ? make_symbol(sc, q) : sc->F);
	  }}

    if ((has_plus_or_minus != 0) &&        /* that is, we have an internal + or - */
	(!has_i))                          /*   but no i for the imaginary part */
      return((want_symbol) ? make_symbol(sc, q) : sc->F);

    if (has_i)
      {
#if (!WITH_GMP)
	s7_double rl = 0.0, im = 0.0;
#else
	char e1 = 0, e2 = 0;
#endif
	s7_pointer result;
	s7_int len = safe_strlen(q);
	char ql1, pl1;

	if (q[len - 1] != 'i')
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);

	/* save original string */
	ql1 = q[len - 1];
	pl1 = (*(plus - 1));
#if WITH_GMP
	if (ex1) {e1 = *ex1; (*ex1) = '@';} /* for mpfr */
	if (ex2) {e2 = *ex2; (*ex2) = '@';}
#endif
	/* look for cases like 1+i */
	q[len - 1] = ((q[len - 2] == '+') || (q[len - 2] == '-')) ? '1' : '\0'; /* remove 'i' */

	(*((char *)(plus - 1))) = '\0';

#if (!WITH_GMP)
	if ((has_dec_point1) ||
	    (ex1))  /* (string->number "1100.1+0.11i" 2) -- need to split into 2 honest reals before passing to non-base-10 str->dbl */
	  rl = string_to_double_with_radix(q, radix, ignored);
	else /* no decimal point, no exponent, a ratio (1/2+i for example, but 1+2/3i is handled below) */
	  {
	    if (slash1)
	      {
		/* here the overflow could be innocuous if it's in the denominator and the numerator is 0: 0/100000000000000000000000000000000000000 */
		s7_int num, den;
		num = string_to_integer(q, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
		den = string_to_integer(slash1, radix, &overflow);
		if (den == 0)
		  rl = NAN;        /* real_part if complex */
		else
		  {
		    if (num == 0)
		      {
			rl = 0.0;
			overflow = false;
		      }
		    else
		      {
			if (overflow) return(make_undefined_bignum(sc, q)); /* denominator overflow */
			rl = (long_double)num / (long_double)den; /* no gmp, so we do what we can */
		      }}}
	    else
	      {
		rl = (s7_double)string_to_integer(q, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
	      }}
	if (rl == -0.0) rl = 0.0;

	if ((has_dec_point2) ||
	    (ex2))
	  im = string_to_double_with_radix(plus, radix, ignored);
	else
	  {
	    if (slash2) /* complex part I think */
	      {
		/* same as above: 0-0/100000000000000000000000000000000000000i */
		s7_int num, den;
		num = string_to_integer(plus, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
		den = string_to_integer(slash2, radix, &overflow);
		if (den == 0)
		  im = NAN;
		else
		  {
		    if (num == 0)
		      {
			im = 0.0;
			overflow = false;
		      }
		    else
		      {
			if (overflow) return(make_undefined_bignum(sc, q)); /* denominator overflow */
			im = (long_double)num / (long_double)den;
		      }}}
	    else
	      {
		im = (s7_double)string_to_integer(plus, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
	      }}
	if ((has_plus_or_minus == -1) &&
	    (im != 0.0))
	  im = -im;
	result = s7_make_complex(sc, rl, im);
#else
	result = string_to_either_complex(sc, q, slash1, ex1, has_dec_point1, plus, slash2, ex2, has_dec_point2, radix, has_plus_or_minus);
#endif
	/* restore original string */
	q[len - 1] = ql1;
	(*((char *)(plus - 1))) = pl1;
#if WITH_GMP
	if (ex1) (*ex1) = e1;
	if (ex2) (*ex2) = e2;
#endif
	return(result);
      }

    /* not complex */
    if ((has_dec_point1) ||
	(ex1))
      {
	s7_pointer result;

	if (slash1)  /* not complex, so slash and "." is not a number */
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);

#if (!WITH_GMP)
	result = make_real(sc, string_to_double_with_radix(q, radix, ignored));
#else
	{
	  char old_e = 0;
	  if (ex1)
	    {
	      old_e = (*ex1);
	      (*ex1) = '@';
	    }
	  result = string_to_either_real(sc, q, radix);
	  if (ex1)
	    (*ex1) = old_e;
	}
#endif
	return(result);
      }

    /* rational */
    if (slash1)
#if (!WITH_GMP)
      {
	s7_int n, d;

	n = string_to_integer(q, radix, &overflow);
	if (overflow) return(make_undefined_bignum(sc, q));
	d = string_to_integer(slash1, radix, &overflow);

	if ((n == 0) && (d != 0))                        /* 0/100000000000000000000000000000000000000 */
	  return(int_zero);
	if (d == 0)
	  return(real_NaN);
	if (overflow) return(make_undefined_bignum(sc, q));
	/* it would be neat to return 1 from 10000000000000000000000000000/10000000000000000000000000000
	 *   but q is the entire number ('/' included) and slash1 is the stuff after the '/', and every
	 *   big number comes through here, so there's no clean and safe way to check that q == slash1.
	 */
	return(make_ratio(sc, n, d));
      }
#else
    return(string_to_either_ratio(sc, q, slash1, radix));
#endif
    /* integer */
#if (!WITH_GMP)
    {
      s7_int x = string_to_integer(q, radix, &overflow);
      if (overflow) return(make_undefined_bignum(sc, q));
      return(make_integer(sc, x));
    }
#else
    return(string_to_either_integer(sc, q, radix));
#endif
  }
}


/* -------------------------------- string->number -------------------------------- */

static s7_pointer string_to_number(s7_scheme *sc, char *str, int32_t radix)
{
  s7_pointer x = make_atom(sc, str, radix, NO_SYMBOLS, WITHOUT_OVERFLOW_ERROR);
  return((is_number(x)) ? x : sc->F);  /* only needed because str might start with '#' and not be a number (#t for example) */
}

static s7_pointer string_to_number_p_p(s7_scheme *sc, s7_pointer str1)
{
  char *str;
  if (!is_string(str1))
    wrong_type_argument(sc, sc->string_to_number_symbol, 1, str1, T_STRING);
  str = (char *)string_value(str1);
  return(((!str) || (!(*str))) ? sc->F : string_to_number(sc, str, 10));
}

static s7_pointer string_to_number_p_pp(s7_scheme *sc, s7_pointer str1, s7_pointer radix1)
{
  s7_int radix;
  char *str;
  if (!is_string(str1))
    wrong_type_argument(sc, sc->string_to_number_symbol, 1, str1, T_STRING);

  if (!is_t_integer(radix1))
    wrong_type_argument(sc, sc->string_to_number_symbol, 2, radix1, T_INTEGER);
  radix = integer(radix1);
  if ((radix < 2) || (radix > 16))
    out_of_range(sc, sc->string_to_number_symbol, int_two, radix1, a_valid_radix_string);

  str = (char *)string_value(str1);
  if ((!str) || (!(*str)))
    return(sc->F);
  return(string_to_number(sc, str, radix));
}

static s7_pointer g_string_to_number_1(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  s7_int radix;
  char *str;
  if (!is_string(car(args)))
    return(method_or_bust(sc, car(args), caller, args, T_STRING, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer rad = cadr(args);
      if (!s7_is_integer(rad))
	return(method_or_bust(sc, rad, caller, args, T_INTEGER, 2));
      radix = s7_integer_clamped_if_gmp(sc, rad);
      if ((radix < 2) || (radix > 16))
	out_of_range(sc, caller, int_two, rad, a_valid_radix_string);
    }
  else radix = 10;
  str = (char *)string_value(car(args));
  if ((!str) || (!(*str)))
    return(sc->F);
  return(string_to_number(sc, str, radix));
}

static s7_pointer g_string_to_number(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_number "(string->number str (radix 10)) converts str into a number. \
If str does not represent a number, string->number returns #f.  If 'str' has an embedded radix, \
the optional 'radix' argument is ignored: (string->number \"#x11\" 2) -> 17 not 3."
  #define Q_string_to_number s7_make_signature(sc, 3, \
                               s7_make_signature(sc, 2, sc->is_number_symbol, sc->not_symbol), \
                               sc->is_string_symbol, sc->is_integer_symbol)
  return(g_string_to_number_1(sc, args, sc->string_to_number_symbol));
}


/* -------------------------------- abs -------------------------------- */
static inline s7_pointer abs_p_p(s7_scheme *sc, s7_pointer x)
{
#if (!WITH_GMP)
  if (is_t_integer(x))
    {
      if (integer(x) >= 0) return(x);
      if (integer(x) > S7_INT64_MIN) return(make_integer(sc, -integer(x)));
    }
  if (is_t_real(x))
    {
      if (is_NaN(real(x)))                  /* (abs -nan.0) -> +nan.0, not -nan.0 */
	return(real_NaN);
      return((signbit(real(x))) ? make_real(sc, -real(x)) : x);
    }
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) >= 0) return(x);
#if WITH_GMP
      if (integer(x) == S7_INT64_MIN)
	{
	  x = s7_int_to_big_integer(sc, integer(x));
	  mpz_neg(big_integer(x), big_integer(x));
	  return(x);
	}
#else
      if (integer(x) == S7_INT64_MIN)
	simple_out_of_range(sc, sc->abs_symbol, set_elist_1(sc, x), result_is_too_large_string);
#endif
      return(make_integer(sc, -integer(x)));

    case T_RATIO:
      if (numerator(x) >= 0) return(x);
#if WITH_GMP && (!POINTER_32)
      if (numerator(x) == S7_INT64_MIN)
	{
	  s7_pointer p;
	  mpz_set_si(sc->mpz_1, S7_INT64_MIN);
	  mpz_neg(sc->mpz_1, sc->mpz_1);
	  mpz_set_si(sc->mpz_2, denominator(x));
	  new_cell(sc, p, T_BIG_RATIO);
	  big_ratio_bgr(p) = alloc_bigrat(sc);
	  add_big_ratio(sc, p);
	  mpq_set_num(big_ratio(p), sc->mpz_1);
	  mpq_set_den(big_ratio(p), sc->mpz_2);
	  return(p);
	}
#else
      if (numerator(x) == S7_INT64_MIN)
	return(make_ratio(sc, S7_INT64_MAX, denominator(x)));
#endif
      return(make_simple_ratio(sc, -numerator(x), denominator(x)));

    case T_REAL:
      if (is_NaN(real(x)))                  /* (abs -nan.0) -> +nan.0, not -nan.0 */
	return(real_NaN);
      return((signbit(real(x))) ? make_real(sc, -real(x)) : x); /* (abs -0.0) returns -0.0 -- Shiro Kawai */
#if WITH_GMP
    case T_BIG_INTEGER:
      mpz_abs(sc->mpz_1, big_integer(x));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
      mpq_abs(sc->mpq_1, big_ratio(x));
      return(mpq_to_rational(sc, sc->mpq_1));
    case T_BIG_REAL:
      mpfr_abs(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->abs_symbol, T_REAL));
    }
}

static s7_pointer g_abs(s7_scheme *sc, s7_pointer args)
{
  #define H_abs "(abs x) returns the absolute value of the real number x"
  #define Q_abs s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_real_symbol)
  return(abs_p_p(sc, car(args)));
}

static s7_double abs_d_d(s7_double x) {return((signbit(x)) ? (-x) : x);}
static s7_int abs_i_i(s7_int x) {return((x < 0) ? (-x) : x);}


/* -------------------------------- magnitude -------------------------------- */
static double my_hypot(double x, double y)
{
  /* according to callgrind, this is much faster than libc's hypot */
  if (x == 0.0) return(fabs(y));
  if (y == 0.0) return(fabs(x));
  if (x == y) return(1.414213562373095 * fabs(x));
  if ((is_NaN(x)) || (is_NaN(y))) return(NAN);
  return(sqrt(x * x + y * y));
}

static s7_pointer magnitude_p_p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_complex(x))
    return(make_real(sc, my_hypot(imag_part(x), real_part(x))));

  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == S7_INT64_MIN) return(mostfix);
      /* (magnitude -9223372036854775808) -> -9223372036854775808
       *   same thing happens in abs, lcm and gcd: (gcd -9223372036854775808) -> -9223372036854775808
       */
      return((integer(x) < 0) ? make_integer(sc, -integer(x)) : x);

    case T_RATIO:
      return((numerator(x) < 0) ? make_simple_ratio(sc, -numerator(x), denominator(x)) : x);

    case T_REAL:
      if (is_NaN(real(x)))                 /* (magnitude -nan.0) -> +nan.0, not -nan.0 */
	return(real_NaN);
      return((signbit(real(x))) ? make_real(sc, -real(x)) : x);

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
    case T_BIG_REAL:
      return(abs_p_p(sc, x));

    case T_BIG_COMPLEX:
      mpc_abs(sc->mpfr_1, big_complex(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif

    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->magnitude_symbol, a_number_string));
    }
}

static s7_pointer g_magnitude(s7_scheme *sc, s7_pointer args)
{
  #define H_magnitude "(magnitude z) returns the magnitude of z"
  #define Q_magnitude s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  return(magnitude_p_p(sc, car(args)));
}


/* -------------------------------- rationalize -------------------------------- */
#if WITH_GMP

static rat_locals_t *init_rat_locals_t(s7_scheme *sc)
{
  rat_locals_t *r = (rat_locals_t *)Malloc(sizeof(rat_locals_t));
  sc->ratloc = r;
  mpz_inits(r->i, r->i0, r->i1, r->n, r->p0, r->q0, r->r, r->r1, r->p1, r->q1, r->old_p1, r->old_q1, NULL);
  mpq_init(r->q);
  mpfr_inits2(sc->bignum_precision, r->error, r->ux, r->x0, r->x1, r->val, r->e0, r->e1, r->e0p, r->e1p, r->old_e0, r->old_e1, r->old_e0p, NULL);
  return(r);
}

static void free_rat_locals(s7_scheme *sc)
{
  rat_locals_t *r = sc->ratloc;
  mpz_clears(r->i, r->i0, r->i1, r->n, r->p0, r->q0, r->r, r->r1, r->p1, r->q1, r->old_p1, r->old_q1, NULL);
  mpq_clear(r->q);
  mpfr_clears(r->error, r->ux, r->x0, r->x1, r->val, r->e0, r->e1, r->e0p, r->e1p, r->old_e0, r->old_e1, r->old_e0p, NULL);
  free(r);
}

static s7_pointer big_rationalize(s7_scheme *sc, s7_pointer args)
{
  /* can return be non-rational? */
  /* currently (rationalize 1/0 1e18) -> 0
   * remember to pad with many trailing zeros:
   *    (rationalize 0.1 0)                -> 3602879701896397/36028797018963968
   *    (rationalize 0.1000000000000000 0) -> 1/10
   * perhaps gmp number reader used if gmp -- could this be the trailing zeros problem?  (why is the non-gmp case ok?)
   *         also the bignum function is faking it.
   *         (rationalize (real-part (bignum "0.1+i")) 0) -> 3602879701896397/36028797018963968
   * a confusing case:
   *   (rationalize 5925563891587147521650777143.74135805596e05) should be 148139097289678688041269428593533951399/250000
   * but that requires more than 128 bits of bignum-precision.
   */

  s7_pointer pp0 = car(args);
  rat_locals_t *r = (sc->ratloc) ? sc->ratloc : init_rat_locals_t(sc);

  switch (type(pp0))
    {
    case T_INTEGER:
      mpfr_set_si(r->ux, integer(pp0), MPFR_RNDN);
      break;
    case T_RATIO:
      mpq_set_si(sc->mpq_1, numerator(pp0), denominator(pp0));
      mpfr_set_q(r->ux, sc->mpq_1, MPFR_RNDN);
      break;
    case T_REAL:
      if (is_NaN(real(pp0)))
	out_of_range(sc, sc->rationalize_symbol, int_one, pp0, its_nan_string);
      if (is_inf(real(pp0)))
	out_of_range(sc, sc->rationalize_symbol, int_one, pp0, its_infinite_string);
      mpfr_set_d(r->ux, real(pp0), MPFR_RNDN);
      break;
    case T_BIG_INTEGER:
      mpfr_set_z(r->ux, big_integer(pp0), MPFR_RNDN);
      break;
    case T_BIG_RATIO:
      mpfr_set_q(r->ux, big_ratio(pp0), MPFR_RNDN);
      break;
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(pp0)))
	out_of_range(sc, sc->rationalize_symbol, int_one, pp0, its_nan_string);
      if (mpfr_inf_p(big_real(pp0)))
	out_of_range(sc, sc->rationalize_symbol, int_one, pp0, its_infinite_string);
      mpfr_set(r->ux, big_real(pp0), MPFR_RNDN);
      break;
    case T_COMPLEX:
    case T_BIG_COMPLEX:
      wrong_type_argument(sc, sc->rationalize_symbol, 1, pp0, T_REAL);
    default:
      return(method_or_bust(sc, pp0, sc->rationalize_symbol, args, T_REAL, 1));
    }

  if (is_null(cdr(args)))
    mpfr_set_d(r->error, sc->default_rationalize_error, MPFR_RNDN);
  else
    {
      s7_pointer pp1 = cadr(args);
      switch (type(pp1))
	{
	case T_INTEGER:
	  mpfr_set_si(r->error, integer(pp1), MPFR_RNDN);
	  break;
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(pp1), denominator(pp1));
	  mpfr_set_q(r->error, sc->mpq_1, MPFR_RNDN);
	  break;
	case T_REAL:
	  if (is_NaN(real(pp1)))
	    out_of_range(sc, sc->rationalize_symbol, int_two, pp1, its_nan_string);
	  if (is_inf(real(pp1)))
	    return(int_zero);
	  mpfr_set_d(r->error, real(pp1), MPFR_RNDN);
	  break;
	case T_BIG_INTEGER:
	  mpfr_set_z(r->error, big_integer(pp1), MPFR_RNDN);
	  break;
	case T_BIG_RATIO:
	  mpfr_set_q(r->error, big_ratio(pp1), MPFR_RNDN);
	  break;
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(pp1)))
	    out_of_range(sc, sc->rationalize_symbol, int_two, pp1, its_nan_string);
	  if (mpfr_inf_p(big_real(pp1)))
	    return(int_zero);
	  mpfr_set(r->error, big_real(pp1), MPFR_RNDN);
	  break;
	case T_COMPLEX:
	case T_BIG_COMPLEX:
	  wrong_type_argument(sc, sc->rationalize_symbol, 2, pp1, T_REAL);
	default:
	  return(method_or_bust(sc, pp1, sc->rationalize_symbol, args, T_REAL, 2));
	}
      mpfr_abs(r->error, r->error, MPFR_RNDN);
    }

  mpfr_set(r->x0, r->ux, MPFR_RNDN);            /* x0 = ux - error */
  mpfr_sub(r->x0, r->x0, r->error, MPFR_RNDN);
  mpfr_set(r->x1, r->ux, MPFR_RNDN);            /* x1 = ux + error */
  mpfr_add(r->x1, r->x1, r->error, MPFR_RNDN);
  mpfr_get_z(r->i, r->x0, MPFR_RNDU);           /* i = ceil(x0) */

  if (mpfr_cmp_ui(r->error, 1) >= 0)            /* if (error >= 1.0) */
    {
      if (mpfr_cmp_ui(r->x0, 0) < 0)            /* if (x0 < 0) */
	{
	  if (mpfr_cmp_ui(r->x1, 0) < 0)        /*   if (x1 < 0) */
	    mpfr_get_z(r->n, r->x1, MPFR_RNDD); /*     num = floor(x1) */
	  else mpz_set_ui(r->n, 0);             /*   else num = 0 */
	}
      else mpz_set(r->n, r->i);                 /* else num = i */
      return(mpz_to_integer(sc, r->n));
    }

  if (mpfr_cmp_z(r->x1, r->i) >= 0)             /* if (x1 >= i) */
    {
      if (mpz_cmp_ui(r->i, 0) >= 0)             /* if (i >= 0) */
	mpz_set(r->n, r->i);                    /*   num = i */
      else mpfr_get_z(r->n, r->x1, MPFR_RNDD);  /* else num = floor(x1) */
      return(mpz_to_integer(sc, r->n));
    }

  mpfr_get_z(r->i0, r->x0, MPFR_RNDD);          /* i0 = floor(x0) */
  mpfr_get_z(r->i1, r->x1, MPFR_RNDU);          /* i1 = ceil(x1) */

  mpz_set(r->p0, r->i0);                        /* p0 = i0 */
  mpz_set_ui(r->q0, 1);                         /* q0 = 1 */
  mpz_set(r->p1, r->i1);                        /* p1 = i1 */
  mpz_set_ui(r->q1, 1);                         /* q1 = 1 */
  mpfr_sub_z(r->e0, r->x0, r->i1, MPFR_RNDN);   /* e0 = i1 - x0 */
  mpfr_neg(r->e0, r->e0, MPFR_RNDN);
  mpfr_sub_z(r->e1, r->x0, r->i0, MPFR_RNDN);   /* e1 = x0 - i0 */
  mpfr_sub_z(r->e0p, r->x1, r->i1, MPFR_RNDN);  /* e0p = i1 - x1 */
  mpfr_neg(r->e0p, r->e0p, MPFR_RNDN);
  mpfr_sub_z(r->e1p, r->x1, r->i0, MPFR_RNDN);  /* e1p = x1 - i0 */

  while (true)
    {
      mpfr_set_z(r->val, r->p0, MPFR_RNDN);
      mpfr_div_z(r->val, r->val, r->q0, MPFR_RNDN);  /* val = p0/q0 */

      if (((mpfr_lessequal_p(r->x0, r->val)) &&        /* if ((x0 <= val) && (val <= x1)) */
	   (mpfr_lessequal_p(r->val, r->x1))) ||
	  (mpfr_cmp_ui(r->e1, 0) == 0) ||
	  (mpfr_cmp_ui(r->e1p, 0) == 0))
	/* these last 2 are probably not needed -- they protect against running out of bits in the non-gmp case above */
	{
	  mpq_set_num(r->q, r->p0);                /* return(p0/q0) */
	  mpq_set_den(r->q, r->q0);
	  return(mpq_to_rational(sc, r->q));
	}

      mpfr_div(r->val, r->e0, r->e1, MPFR_RNDN);
      mpfr_get_z(r->r, r->val, MPFR_RNDD);           /* r = floor(e0/e1) */
      mpfr_div(r->val, r->e0p, r->e1p, MPFR_RNDN);
      mpfr_get_z(r->r1, r->val, MPFR_RNDU);          /* r1 = ceil(e0p/e1p) */
      if (mpz_cmp(r->r1, r->r) < 0)                 /* if (r1 < r) */
	mpz_set(r->r, r->r1);                       /*   r = r1 */

      mpz_set(r->old_p1, r->p1);                    /* old_p1 = p1 */
      mpz_set(r->p1, r->p0);                        /* p1 = p0 */
      mpz_set(r->old_q1, r->q1);                    /* old_q1 = q1 */
      mpz_set(r->q1, r->q0);                        /* q1 = q0 */

      mpfr_set(r->old_e0, r->e0, MPFR_RNDN);         /* old_e0 = e0 */
      mpfr_set(r->e0, r->e1p, MPFR_RNDN);            /* e0 = e1p */
      mpfr_set(r->old_e0p, r->e0p, MPFR_RNDN);       /* old_e0p = e0p */
      mpfr_set(r->e0p, r->e1, MPFR_RNDN);            /* e0p = e1 */
      mpfr_set(r->old_e1, r->e1, MPFR_RNDN);         /* old_e1 = e1 */

      mpz_mul(r->p0, r->p0, r->r);                  /* p0 = old_p1 + r * p0 */
      mpz_add(r->p0, r->p0, r->old_p1);

      mpz_mul(r->q0, r->q0, r->r);                  /* q0 = old_q1 + r * q0 */
      mpz_add(r->q0, r->q0, r->old_q1);

      mpfr_mul_z(r->e1, r->e1p, r->r, MPFR_RNDN);    /* e1 = old_e0p - r * e1p */
      mpfr_sub(r->e1, r->old_e0p, r->e1, MPFR_RNDN);

      mpfr_mul_z(r->e1p, r->old_e1, r->r, MPFR_RNDN);/* e1p = old_e0 - r * old_e1 */
      mpfr_sub(r->e1p, r->old_e0, r->e1p, MPFR_RNDN);
    }
}
#endif

static s7_pointer g_rationalize(s7_scheme *sc, s7_pointer args)
{
  #define H_rationalize "(rationalize x err) returns the ratio with smallest denominator within err of x"
  #define Q_rationalize s7_make_signature(sc, 3, sc->is_rational_symbol, sc->is_real_symbol, sc->is_real_symbol)
  /* I can't find a case where this returns a non-rational result */

  s7_double err;
  s7_pointer x = car(args);

#if WITH_GMP
  if (is_big_number(x))
    return(big_rationalize(sc, args));
#endif
  if (!is_real(x))
    return(method_or_bust(sc, x, sc->rationalize_symbol, args, T_REAL, 1));

  if (is_null(cdr(args)))
    err = sc->default_rationalize_error;
  else
    {
      s7_pointer ex = cadr(args);
#if WITH_GMP
      if (is_big_number(ex))
	return(big_rationalize(sc, args));
#endif
      if (!is_real(ex))
	return(method_or_bust(sc, ex, sc->rationalize_symbol, args, T_REAL, 2));
      err = real_to_double(sc, ex, "rationalize");
      if (is_NaN(err))
	out_of_range(sc, sc->rationalize_symbol, int_two, cadr(args), its_nan_string);
      if (err < 0.0) err = -err;
    }

  switch (type(x))
    {
    case T_INTEGER:
      {
	s7_int a, b, pa;
	if (err < 1.0) return(x);
	a = integer(x);
	pa = (a < 0) ? -a : a;
	if (err >= pa) return(int_zero);
	b = (s7_int)err;
	pa -= b;
	return((a < 0) ? make_integer(sc, -pa) : make_integer(sc, pa));
      }

    case T_RATIO:
      if (err == 0.0)
	return(x);

    case T_REAL:
      {
	s7_double rat = s7_real(x); /* possible fall through from above */
	s7_int numer = 0, denom = 1;

	if ((is_NaN(rat)) || (is_inf(rat)))
	  out_of_range(sc, sc->rationalize_symbol, int_one, x, a_normal_real_string);

	if (err >= fabs(rat))
	  return(int_zero);

#if WITH_GMP
	if (fabs(rat) > RATIONALIZE_LIMIT)
	  return(big_rationalize(sc, set_plist_2(sc, x, wrap_real(sc, err))));
#else
	if (fabs(rat) > RATIONALIZE_LIMIT)
	  out_of_range(sc, sc->rationalize_symbol, int_one, x, its_too_large_string);
#endif
	if ((fabs(rat) + fabs(err)) < 1.0e-18)
	  err = 1.0e-18;
	/* (/ 1.0 most-positive-fixnum) is 1.0842021e-19, so if we let err be less than that,
	 * (rationalize 1e-19 1e-20) hangs, but this only affects the initial ceiling, I believe.
	 */

	if (fabs(rat) < fabs(err))
	  return(int_zero);

	return((c_rationalize(rat, err, &numer, &denom)) ? make_ratio(sc, numer, denom) : sc->F);
      }}
  return(sc->F); /* make compiler happy */
}

static s7_int rationalize_i_i(s7_int x) {return(x);}
static s7_pointer rationalize_p_i(s7_scheme *sc, s7_int x) {return(make_integer(sc, x));}
static s7_pointer rationalize_p_d(s7_scheme *sc, s7_double x)
{
  if ((is_NaN(x)) || (is_inf(x)))
    out_of_range(sc, sc->rationalize_symbol, int_one, wrap_real(sc, x), a_normal_real_string); /* was make_real, also below */
  if (fabs(x) > RATIONALIZE_LIMIT)
#if WITH_GMP
    return(big_rationalize(sc, set_plist_1(sc, wrap_real(sc, x))));
#else
    out_of_range(sc, sc->rationalize_symbol, int_one, wrap_real(sc, x), its_too_large_string);
#endif
  return(s7_rationalize(sc, x, sc->default_rationalize_error));
}


/* -------------------------------- angle -------------------------------- */
static s7_pointer g_angle(s7_scheme *sc, s7_pointer args)
{
  #define H_angle "(angle z) returns the angle of z"
  #define Q_angle s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)

  s7_pointer x = car(args);  /* (angle inf+infi) -> 0.78539816339745 ? I think this should be -pi < ang <= pi */
  switch (type(x))
    {
    case T_INTEGER: return((integer(x) < 0) ? real_pi : int_zero);
    case T_RATIO:   return((numerator(x) < 0) ? real_pi : int_zero);
    case T_COMPLEX: return(make_real(sc, atan2(imag_part(x), real_part(x))));

    case T_REAL:
      if (is_NaN(real(x))) return(x);
      return((real(x) < 0.0) ? real_pi : real_zero);
#if WITH_GMP
    case T_BIG_INTEGER: return((mpz_cmp_ui(big_integer(x), 0) >= 0) ? int_zero : big_pi(sc));
    case T_BIG_RATIO:   return((mpq_cmp_ui(big_ratio(x), 0, 1) >= 0) ? int_zero : big_pi(sc));

    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x))) return(x);
      return((mpfr_cmp_d(big_real(x), 0.0) >= 0) ? real_zero : big_pi(sc));

    case T_BIG_COMPLEX:
      {
	s7_pointer z;
	new_cell(sc, z, T_BIG_REAL);
	big_real_bgf(z) = alloc_bigflt(sc);
	add_big_real(sc, z);
	mpc_arg(big_real(z), big_complex(x), MPFR_RNDN);
	return(z);
      }
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->angle_symbol, a_number_string));
    }
}


/* -------------------------------- complex -------------------------------- */
static s7_pointer g_complex(s7_scheme *sc, s7_pointer args)
{
  #define H_complex "(complex x1 x2) returns a complex number with real-part x1 and imaginary-part x2"
  #define Q_complex s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_real_symbol, sc->is_real_symbol)

  s7_pointer x = car(args), y = cadr(args);

#if WITH_GMP
  if ((is_big_number(x)) || (is_big_number(y)))
    {
      s7_pointer p0 = x, p1 = y, p = NULL;

      if (!is_real(p0))
	return(method_or_bust(sc, p0, sc->complex_symbol, args, T_REAL, 1));
      if (!is_real(p1))
	return(method_or_bust(sc, p1, sc->complex_symbol, args, T_REAL, 2));

      switch (type(p1))
	{
	case T_INTEGER: case T_RATIO: case T_REAL:
	  {
	    s7_double iz = s7_real(p1);
	    if (iz == 0.0)                      /* imag-part is 0.0 */
	      return(p0);
	    new_cell(sc, p, T_BIG_COMPLEX);
	    big_complex_bgc(p) = alloc_bigcmp(sc);
	    mpfr_set_d(mpc_imagref(big_complex(p)), iz, MPFR_RNDN);
	  }
	  break;

	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(p1))) return(p0);
	  new_cell(sc, p, T_BIG_COMPLEX);
	  big_complex_bgc(p) = alloc_bigcmp(sc);
	  mpfr_set(mpc_imagref(big_complex(p)), big_real(p1), MPFR_RNDN);
	  break;

	case T_BIG_RATIO:
	  new_cell(sc, p, T_BIG_COMPLEX);
	  big_complex_bgc(p) = alloc_bigcmp(sc);
	  mpfr_set_q(mpc_imagref(big_complex(p)), big_ratio(p1), MPFR_RNDN);
	  break;

	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(p1), 0) == 0) return(p0);
	  new_cell(sc, p, T_BIG_COMPLEX);
	  big_complex_bgc(p) = alloc_bigcmp(sc);
	  mpfr_set_z(mpc_imagref(big_complex(p)), big_integer(p1), MPFR_RNDN);
	  break;
	}

      switch (type(p0))
	{
	case T_INTEGER: case T_RATIO: case T_REAL:
	  mpfr_set_d(mpc_realref(big_complex(p)), s7_real(p0), MPFR_RNDN);
	  break;
	case T_BIG_REAL:
	  mpfr_set(mpc_realref(big_complex(p)), big_real(p0), MPFR_RNDN);
	  break;
	case T_BIG_RATIO:
	  mpfr_set_q(mpc_realref(big_complex(p)), big_ratio(p0), MPFR_RNDN);
	  break;
	case T_BIG_INTEGER:
	  mpfr_set_z(mpc_realref(big_complex(p)), big_integer(p0), MPFR_RNDN);
	  break;
	}
      add_big_complex(sc, p);
      return(p);
    }
#endif
  switch (type(y))
    {
    case T_INTEGER:
      switch (type(x))
	{
	case T_INTEGER: return((integer(y) == 0) ? x : s7_make_complex(sc, (s7_double)integer(x), (s7_double)integer(y)));
	  /* these int->dbl's are problematic:
	   *   (complex 9223372036854775807 9007199254740995): 9223372036854776000.0+9007199254740996.0i
	   * should we raise an error?
	   */
	case T_RATIO:  return((integer(y) == 0) ? x : s7_make_complex(sc, (s7_double)fraction(x), (s7_double)integer(y)));
	case T_REAL:   return((integer(y) == 0) ? x : s7_make_complex(sc, real(x), (s7_double)integer(y)));
	default:       return(method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1));
	}
    case T_RATIO:
      switch (type(x))
	{
	case T_INTEGER: return(s7_make_complex(sc, (s7_double)integer(x), (s7_double)fraction(y))); /* can fraction be 0.0? */
	case T_RATIO:   return(s7_make_complex(sc, (s7_double)fraction(x), (s7_double)fraction(y)));
	case T_REAL:    return(s7_make_complex(sc, real(x), (s7_double)fraction(y)));
	default:	return(method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1));
	}
    case T_REAL:
      switch (type(x))
	{
	case T_INTEGER: return((real(y) == 0.0) ? x : s7_make_complex(sc, (s7_double)integer(x), real(y)));
	case T_RATIO:	return((real(y) == 0.0) ? x : s7_make_complex(sc, (s7_double)fraction(x), real(y)));
	case T_REAL:    return((real(y) == 0.0) ? x : s7_make_complex(sc, real(x), real(y)));
	default:	return(method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1));
	}
    default:
      return(method_or_bust(sc, (is_let(x)) ? x : y, sc->complex_symbol, args, T_REAL, 2));
    }
}

static s7_pointer complex_p_ii(s7_scheme *sc, s7_int x, s7_int y)
{
  return((y == 0.0) ? make_integer(sc, x) : make_complex_not_0i(sc, (s7_double)x, (s7_double)y));
}

static s7_pointer complex_p_dd(s7_scheme *sc, s7_double x, s7_double y)
{
  return((y == 0.0) ? make_real(sc, x) : make_complex_not_0i(sc, x, y));
}


/* -------------------------------- bignum -------------------------------- */
static s7_pointer g_bignum(s7_scheme *sc, s7_pointer args)
{
  #define H_bignum "(bignum val (radix 10)) returns a multiprecision version of the string 'val'. If the argument is a number \
bignum returns that number as a bignum"
#if WITH_GMP
  #define Q_bignum s7_make_signature(sc, 3, sc->is_bignum_symbol, s7_make_signature(sc, 2, sc->is_number_symbol, sc->is_string_symbol), sc->is_integer_symbol)
#else
  #define Q_bignum s7_make_signature(sc, 3, \
                     s7_make_signature(sc, 2, sc->is_number_symbol, sc->not_symbol), \
                     s7_make_signature(sc, 2, sc->is_number_symbol, sc->is_string_symbol), \
                     sc->is_integer_symbol)
#endif

  s7_pointer p = car(args);
  if (is_number(p))
    {
      if (!is_null(cdr(args)))
	s7_error_nr(sc, make_symbol(sc, "bignum-error"),
		    set_elist_2(sc, wrap_string(sc, "bignum of a number takes only one argument: ~S", 46), args));
#if WITH_GMP
      switch (type(p))
	{
	case T_INTEGER: return(s7_int_to_big_integer(sc, integer(p)));
	case T_RATIO:   return(s7_int_to_big_ratio(sc, numerator(p), denominator(p)));
	case T_REAL:    return(s7_double_to_big_real(sc, real(p)));
	case T_COMPLEX: return(s7_double_to_big_complex(sc, real_part(p), imag_part(p)));
	default:        return(p);
	}
#else
      return(p);
#endif
    }
  p = g_string_to_number_1(sc, args, sc->bignum_symbol);
  if (is_false(sc, p))                                       /* (bignum "1/3.0") */
    s7_error_nr(sc, make_symbol(sc, "bignum-error"),
		set_elist_2(sc, wrap_string(sc, "bignum string argument does not represent a number: ~S", 54), car(args)));
#if WITH_GMP
  switch (type(p))
    {
    case T_INTEGER:   return(s7_int_to_big_integer(sc, integer(p)));
    case T_RATIO:     return(s7_int_to_big_ratio(sc, numerator(p), denominator(p)));
    case T_COMPLEX:   return(s7_number_to_big_complex(sc, p));
    case T_REAL:
      if (is_NaN(real(p))) return(p);
      return(s7_double_to_big_real(sc, real(p)));
      /* 9Sep21: this was return(string_to_big_real(sc, string_value(car(args)), (is_pair(cdr(args))) ? s7_integer_clamped_if_gmp(sc, cadr(args)) : 10)); */
    default:
      return(p);
    }
#else
  return(p);
#endif
}


/* -------------------------------- exp -------------------------------- */
#if (!HAVE_COMPLEX_NUMBERS)
  static s7_pointer no_complex_numbers_string;
#endif

#define EXP_LIMIT 100.0

#if WITH_GMP
static s7_pointer exp_1(s7_scheme *sc, s7_double x)
{
  mpfr_set_d(sc->mpfr_1, x, MPFR_RNDN);
  mpfr_exp(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
}

static s7_pointer exp_2(s7_scheme *sc, s7_double x, s7_double y)
{
  mpc_set_d_d(sc->mpc_1, x, y, MPC_RNDNN);
  mpc_exp(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
  return(mpc_to_number(sc, sc->mpc_1));
}
#endif

static s7_pointer exp_p_p(s7_scheme *sc, s7_pointer x)
{
  s7_double z;
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_one);                       /* (exp 0) -> 1 */
      z = (s7_double)integer(x);
#if WITH_GMP
      if (fabs(z) > EXP_LIMIT)
	return(exp_1(sc, z));
#endif
      return(make_real(sc, exp(z)));

    case T_RATIO:
      z = (s7_double)fraction(x);
#if WITH_GMP
      if (fabs(z) > EXP_LIMIT)
	return(exp_1(sc, z));
#endif
      return(make_real(sc, exp(z)));

    case T_REAL:
#if WITH_GMP
      if (fabs(real(x)) > EXP_LIMIT)
	return(exp_1(sc, real(x)));
#endif
      return(make_real(sc, exp(real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
#if WITH_GMP
      if ((fabs(real_part(x)) > EXP_LIMIT) ||
	  (fabs(imag_part(x)) > EXP_LIMIT))
	return(exp_2(sc, real_part(x), imag_part(x)));
#endif
      return(c_complex_to_s7(sc, cexp(to_c_complex(x))));
      /* this is inaccurate for large arguments:
       *   (exp 0+1e20i) -> -0.66491178990701-0.74692189125949i, not 7.639704044417283004001468027378811228331E-1-6.45251285265780844205811711312523007406E-1i
       */
#else
      out_of_range(sc, sc->exp_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      mpfr_exp(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      mpfr_exp(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_exp(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_exp(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->exp_symbol, a_number_string));
    }
}

static s7_pointer g_exp(s7_scheme *sc, s7_pointer args)
{
  #define H_exp "(exp z) returns e^z, (exp 1) is 2.718281828459"
  #define Q_exp sc->pl_nn
  return(exp_p_p(sc, car(args)));
}

#if (!WITH_GMP)
static s7_double exp_d_d(s7_double x) {return(exp(x));}
#endif


/* -------------------------------- log -------------------------------- */
#if __cplusplus
#define LOG_2 1.4426950408889634074
#else
#define LOG_2 1.4426950408889634073599246810018921L /* (/ (log 2.0)) */
#endif

#if WITH_GMP
static s7_pointer big_log(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p0 = car(args), p1 = NULL, res;

  if (!is_number(p0))
    return(method_or_bust_with_type(sc, p0, sc->log_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      p1 = cadr(args);
      if (!is_number(p1))
	return(method_or_bust_with_type(sc, p1, sc->log_symbol, args, a_number_string, 2));
    }

  if (is_real(p0))
    {
      res = any_real_to_mpfr(sc, p0, sc->mpfr_1);
      if (res == real_NaN) return(res);
      if ((is_positive(sc, p0)) &&
	  ((!p1) ||
	   ((is_real(p1)) && (is_positive(sc, p1)))))
	{
	  if (res) return(res);
	  mpfr_log(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  if (p1)
	    {
	      res = any_real_to_mpfr(sc, p1, sc->mpfr_2);
	      if (res)
		return((res == real_infinity) ? real_zero : res);
	      if (mpfr_zero_p(sc->mpfr_2))
		out_of_range(sc, sc->log_symbol, int_two, p1, wrap_string(sc, "can't be zero", 13));
	      mpfr_log(sc->mpfr_2, sc->mpfr_2, MPFR_RNDN);
	      mpfr_div(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
	    }
	  if ((mpfr_integer_p(sc->mpfr_1)) && ((is_rational(p0)) && ((!p1) || (is_rational(p1)))))
	    return(mpfr_to_integer(sc, sc->mpfr_1));
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}}

  if (p1)
    {
      res = any_number_to_mpc(sc, p1, sc->mpc_2);
      if (res)
	return((res == real_infinity) ? real_zero : complex_NaN);
      if (mpc_zero_p(sc->mpc_2))
	out_of_range(sc, sc->log_symbol, int_two, p1, wrap_string(sc, "can't be zero", 13));
    }
  res = any_number_to_mpc(sc, p0, sc->mpc_1);
  if (res)
    {
      if ((res == real_infinity) && (p1) && ((is_negative(sc, p0))))
	return(make_complex_not_0i(sc, INFINITY, -NAN));
      return((res == real_NaN) ? complex_NaN : res);
    }
  mpc_log(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
  if (p1)
    {
      mpc_log(sc->mpc_2, sc->mpc_2, MPC_RNDNN);
      mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
    }
  if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
    return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
  return(mpc_to_number(sc, sc->mpc_1));
}
#endif

static s7_pointer g_log(s7_scheme *sc, s7_pointer args)
{
  #define H_log "(log z1 (z2 e)) returns log(z1) / log(z2) where z2 (the base) defaults to e: (log 8 2) = 3"
  #define Q_log sc->pcl_n

  s7_pointer x = car(args);

#if WITH_GMP
  if (is_big_number(x)) return(big_log(sc, args));
#endif

  if (!is_number(x))
    return(method_or_bust_with_type(sc, x, sc->log_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer y = cadr(args);
      if (!(is_number(y)))
	return(method_or_bust_with_type(sc, y, sc->log_symbol, args, a_number_string, 2));

#if WITH_GMP
      if (is_big_number(y)) return(big_log(sc, args));
#endif
      if ((is_t_integer(y)) && (integer(y) == 2))
	{
	  /* (define (2^n? x) (and (not (zero? x)) (zero? (logand x (- x 1))))) */
	  if (is_t_integer(x))
	    {
	      s7_int ix = integer(x);
	      if (ix > 0)
		{
		  s7_double fx;
#if (__ANDROID__) || (MS_WINDOWS) || (((__GNUC__) && ((__GNUC__ < 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ <= 4)))) && (!defined(__clang__)))
		  /* just a guess -- log2 gets a warning in gcc 4.3.2, but not in 4.4.4 */
		  fx = log((double)ix) / log(2.0);
#else
		  fx = log2((double)ix);
#endif
		  /* (s7_int)fx rounds (log 8 2) to 2 in FreeBSD! */
#if (((__GNUC__) && ((__GNUC__ < 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ < 4)))) && (!defined(__clang__)))
		  return(make_real(sc, fx));
#else
		  return(((ix & (ix - 1)) == 0) ? make_integer(sc, (s7_int)s7_round(fx)) : make_real(sc, fx));
#endif
		}}
	  if ((is_real(x)) &&
	      (is_positive(sc, x)))
	    return(make_real(sc, log(s7_real(x)) * LOG_2));
	  return(c_complex_to_s7(sc, clog(s7_to_c_complex(x)) * LOG_2));
	}

      if ((is_t_integer(x)) && (integer(x) == 1) && (is_t_integer(y)) && (integer(y) == 1))  /* (log 1 1) -> 0 (this is NaN in the bignum case) */
	return(int_zero);

      /* (log 1 0) must be 0 since everyone says (expt 0 0) is 1 */
      if (is_zero(y))
	{
	  if ((is_t_integer(y)) && (is_t_integer(x)) && (integer(x) == 1))
	    return(y);
	  out_of_range(sc, sc->log_symbol, int_two, y, wrap_string(sc, "can't be zero", 13));
	}

      if ((is_t_real(x)) && (is_NaN(real(x))))
	return(real_NaN);
      if (is_one(y))                                     /* this used to raise an error, but the bignum case is simpler if we return inf */
	return((is_one(x)) ? real_zero : real_infinity); /* but (log 1.0 1.0) -> 0.0, currently (log 1/0 1) is inf? */

      if ((is_real(x)) &&
	  (is_real(y)) &&
	  (is_positive(sc, x)) &&
	  (is_positive(sc, y)))
	{
	  if ((is_rational(x)) &&
	      (is_rational(y)))
	    {
	      s7_double res = log(rational_to_double(sc, x)) / log(rational_to_double(sc, y));
	      s7_int ires = (s7_int)res;
	      if (res - ires == 0.0)
		return(make_integer(sc, ires));   /* (log 8 2) -> 3 or (log 1/8 2) -> -3 */
	      /* since x and y are rational here, it seems reasonable to try to rationalize the result, but not go overboard?
	       *   what about (expt 16 3/2) -> 64?  also 2 as base is handled above and always returns a float.
	       */
	      if (fabs(res) < RATIONALIZE_LIMIT)
		{
		  s7_int num, den;
		  if ((c_rationalize(res, sc->default_rationalize_error, &num, &den)) &&
		      (s7_int_abs(num) < 100) && (s7_int_abs(den) < 100))
		    return(make_simple_ratio(sc, num, den));
		}
	      return(make_real(sc, res));
	    }
	  return(make_real(sc, log(s7_real(x)) / log(s7_real(y))));
	}
      if ((is_t_real(x)) && (is_NaN(real(x))))
	return(real_NaN);
      if ((is_t_complex(y)) && ((is_NaN(real_part(y))) || (is_NaN(imag_part(y)))))
	return(real_NaN);
      return(c_complex_to_s7(sc, clog(s7_to_c_complex(x)) / clog(s7_to_c_complex(y))));
    }

  if (!is_real(x))
    return(c_complex_to_s7(sc, clog(s7_to_c_complex(x))));
  if (is_positive(sc, x))
    return(make_real(sc, log(s7_real(x))));
  return(make_complex_not_0i(sc, log(-s7_real(x)), M_PI));
}


/* -------------------------------- sin -------------------------------- */
#define SIN_LIMIT 1.0e16
#define SINH_LIMIT 20.0
/* (- (sinh (bignum 30.0)) (sinh 30.0)): -3.718172657214174140191915872003397016115E-4
 * (- (sinh (bignum 20.0)) (sinh 20.0)): -7.865629467297586346406367346575835463792E-10, slightly worse (e-8) if imag-part
 */

static s7_pointer sin_p_p(s7_scheme *sc, s7_pointer x)
{
#if (!WITH_GMP)
  if (is_t_real(x)) return(make_real(sc, sin(real(x)))); /* range check in gmp case */
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);           /* (sin 0) -> 0 */
#if WITH_GMP
      if (integer(x) > SIN_LIMIT)
	{
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpfr_set_z(sc->mpfr_1, sc->mpz_1, MPFR_RNDN);
	  mpfr_sin(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, sin((s7_double)(integer(x))))); /* bogus for very large integers, but so is the equivalent real (see SIN_LIMIT) */

    case T_RATIO:
      return(make_real(sc, sin((s7_double)(fraction(x)))));

    case T_REAL:
      {
	s7_double y = real(x);
#if WITH_GMP
	if (fabs(y) > SIN_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
	    mpfr_sin(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, sin(y)));
      }

    case T_COMPLEX:
#if WITH_GMP
      if ((fabs(real_part(x)) > SIN_LIMIT) || (fabs(imag_part(x)) > SINH_LIMIT))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sin(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if HAVE_COMPLEX_NUMBERS
      return(c_complex_to_s7(sc, csin(to_c_complex(x))));
#else
      out_of_range(sc, sc->sin_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      mpfr_sin(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      mpfr_sin(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_sin(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_sin(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->sin_symbol, a_number_string));
    }
  /* sin is inaccurate over about 1e30.  There's a way to get true results, but it involves fancy "range reduction" techniques.
   * (sin 1e32): 0.5852334864823946
   *   but it should be 3.901970254333630491697613212893425767786E-1
   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22 !! (it's now a range error)
   *   it should be 5.263007914620499494429139986095833592117E0
   * before comparing imag-part to 0, we need to look for NaN and inf, else:
   *    (sinh 0+0/0i) -> 0.0 and (sinh (log 0.0)) -> inf.0
   */
}

static s7_pointer g_sin(s7_scheme *sc, s7_pointer args)
{
  #define H_sin "(sin z) returns sin(z)"
  #define Q_sin sc->pl_nn
  return(sin_p_p(sc, car(args)));
}

#if WITH_GMP
static s7_pointer sin_p_d(s7_scheme *sc, s7_double x)
{
  if (fabs(x) <= SIN_LIMIT)
    return(make_real(sc, sin(x)));
  mpfr_set_d(sc->mpfr_1, x, MPFR_RNDN);
  mpfr_sin(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
}
#else
static s7_pointer sin_p_d(s7_scheme *sc, s7_double x) {return(make_real(sc, sin(x)));}
#endif

static s7_double sin_d_d(s7_double x) {return(sin(x));}


/* -------------------------------- cos -------------------------------- */
static s7_pointer cos_p_p(s7_scheme *sc, s7_pointer x)
{
#if (!WITH_GMP)
  if (is_t_real(x)) return(make_real(sc, cos(real(x)))); /* range check in gmp case */
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_one);             /* (cos 0) -> 1 */
#if WITH_GMP
      if (integer(x) > SIN_LIMIT)
	{
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpfr_set_z(sc->mpfr_1, sc->mpz_1, MPFR_RNDN);
	  mpfr_cos(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, cos((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, cos((s7_double)(fraction(x)))));

    case T_REAL: /* if with_gmp */
      {
	s7_double y = real(x);
#if WITH_GMP
	if (fabs(y) > SIN_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
	    mpfr_cos(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, cos(y)));
      }

    case T_COMPLEX:
#if WITH_GMP
      if ((fabs(real_part(x)) > SIN_LIMIT) || (fabs(imag_part(x)) > SINH_LIMIT))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_cos(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if HAVE_COMPLEX_NUMBERS
      return(c_complex_to_s7(sc, ccos(to_c_complex(x))));
#else
      out_of_range(sc, sc->cos_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      mpfr_cos(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      mpfr_cos(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_cos(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_cos(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->cos_symbol, a_number_string));
    }
}

static s7_pointer g_cos(s7_scheme *sc, s7_pointer args)
{
  #define H_cos "(cos z) returns cos(z)"
  #define Q_cos sc->pl_nn
  return(cos_p_p(sc, car(args)));
}

#if WITH_GMP
static s7_pointer cos_p_d(s7_scheme *sc, s7_double x)
{
  if (fabs(x) <= SIN_LIMIT)
    return(make_real(sc, cos(x)));
  mpfr_set_d(sc->mpfr_1, x, MPFR_RNDN);
  mpfr_cos(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
}
#else
static s7_pointer cos_p_d(s7_scheme *sc, s7_double x) {return(make_real(sc, cos(x)));}
#endif

static s7_double cos_d_d(s7_double x) {return(cos(x));}


/* -------------------------------- tan -------------------------------- */
#define TAN_LIMIT 1.0e18

static s7_pointer tan_p_p(s7_scheme *sc, s7_pointer x)
{
#if (!WITH_GMP)
  if (is_t_real(x)) return(make_real(sc, tan(real(x))));
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);                      /* (tan 0) -> 0 */
#if WITH_GMP
      if (integer(x) > TAN_LIMIT)
	{
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpfr_set_z(sc->mpfr_1, sc->mpz_1, MPFR_RNDN);
	  mpfr_tan(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, tan((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, tan((s7_double)(fraction(x)))));

#if WITH_GMP
    case T_REAL:
      if (fabs(real(x)) > TAN_LIMIT)
	{
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_tan(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, tan(real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      if (imag_part(x) > 350.0)
	return(make_complex_not_0i(sc, 0.0, 1.0));
      return((imag_part(x) < -350.0) ? s7_make_complex(sc, 0.0, -1.0) : c_complex_to_s7(sc, ctan(to_c_complex(x))));
#else
      out_of_range(sc, sc->tan_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      mpfr_tan(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      mpfr_tan(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_tan(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      if ((MPC_INEX_IM(mpc_cmp_si_si(big_complex(x), 1, 350))) > 0)
	return(make_complex_not_0i(sc, 0.0, 1.0));
      if ((MPC_INEX_IM(mpc_cmp_si_si(big_complex(x), 1, -350))) < 0)
	return(make_complex_not_0i(sc, 0.0, -1.0));
      mpc_tan(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->tan_symbol, a_number_string));
    }
}

static s7_pointer g_tan(s7_scheme *sc, s7_pointer args)
{
  #define H_tan "(tan z) returns tan(z)"
  #define Q_tan sc->pl_nn
  return(tan_p_p(sc, car(args)));
}

static s7_double tan_d_d(s7_double x) {return(tan(x));}


/* -------------------------------- asin -------------------------------- */
static s7_pointer c_asin(s7_scheme *sc, s7_double x)
{
  s7_double absx = fabs(x), recip;
  s7_complex result;

  if (absx <= 1.0)
    return(make_real(sc, asin(x)));

  /* otherwise use maxima code: */
  recip = 1.0 / absx;
  result = (M_PI / 2.0) - (_Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip)))));
  return((x < 0.0) ? c_complex_to_s7(sc, -result) : c_complex_to_s7(sc, result));
}

static s7_pointer asin_p_p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_real(p)) return(c_asin(sc, real(p)));
  switch (type(p))
    {
    case T_INTEGER:
      if (integer(p) == 0) return(int_zero);                    /* (asin 0) -> 0 */
      /* in netBSD, (asin 2) returns 0.25383842987008+0.25383842987008i according to Peter Bex */
      return(c_asin(sc, (s7_double)integer(p)));

    case T_RATIO:
      return(c_asin(sc, fraction(p)));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      /* if either real or imag part is very large, use explicit formula, not casin */
      /*   this code taken from sbcl's src/code/irrat.lisp; break is around x+70000000i */
      if ((fabs(real_part(p)) > 1.0e7) ||
	  (fabs(imag_part(p)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z = to_c_complex(p);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);
	  return(s7_make_complex(sc, atan(real_part(p) / creal(sq1mz * sq1pz)), asinh(cimag(sq1pz * conj(sq1mz)))));
	}
      return(c_complex_to_s7(sc, casin(to_c_complex(p))));
#else
      out_of_range(sc, sc->asin_symbol, int_one, p, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(p), MPFR_RNDN);
      goto ASIN_BIG_REAL;
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(p), MPFR_RNDN);
      goto ASIN_BIG_REAL;
    case T_BIG_REAL:
      if (mpfr_inf_p(big_real(p)))
	{
	  if (mpfr_cmp_ui(big_real(p), 0) < 0)
	    return(make_complex_not_0i(sc, NAN, INFINITY)); /* match non-bignum choice */
	  return(make_complex_not_0i(sc, NAN, -INFINITY));
	}
      mpfr_set(sc->mpfr_1, big_real(p), MPFR_RNDN);
    ASIN_BIG_REAL:
      mpfr_set_ui(sc->mpfr_2, 1, MPFR_RNDN);
      if (mpfr_cmpabs(sc->mpfr_1, sc->mpfr_2) <= 0)
	{
	  mpfr_asin(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      mpc_set_fr(sc->mpc_1, sc->mpfr_1, MPC_RNDNN);
      mpc_asin(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_COMPLEX:
      mpc_asin(sc->mpc_1, big_complex(p), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, p, sc->asin_symbol, a_number_string));
    }
}

static s7_pointer g_asin(s7_scheme *sc, s7_pointer args)
{
  #define H_asin "(asin z) returns asin(z); (sin (asin x)) = x"
  #define Q_asin sc->pl_nn
  return(asin_p_p(sc, car(args)));
}


/* -------------------------------- acos -------------------------------- */
static s7_pointer c_acos(s7_scheme *sc, s7_double x)
{
  s7_double absx = fabs(x), recip;
  s7_complex result;

  if (absx <= 1.0)
    return(make_real(sc, acos(x)));

  /* else follow maxima again: */
  recip = 1.0 / absx;
  if (x > 0.0)
    result = _Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  else result = M_PI - _Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  return(c_complex_to_s7(sc, result));
}

static s7_pointer acos_p_p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_real(p)) return(c_acos(sc, real(p)));
  switch (type(p))
    {
    case T_INTEGER:
      return((integer(p) == 1) ? int_zero : c_acos(sc, (s7_double)integer(p)));

    case T_RATIO:
      return(c_acos(sc, fraction(p)));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      /* if either real or imag part is very large, use explicit formula, not cacos */
      /*   this code taken from sbcl's src/code/irrat.lisp */

      if ((fabs(real_part(p)) > 1.0e7) ||
	  (fabs(imag_part(p)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z = to_c_complex(p);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);	  /* creal(sq1pz) can be 0.0 */
	  if (creal(sq1pz) == 0.0)        /* so the atan arg will be inf, so the real part will be pi/2(?) */
	    return(s7_make_complex(sc, M_PI / 2.0, asinh(cimag(sq1mz * conj(sq1pz)))));
	  return(s7_make_complex(sc, 2.0 * atan(creal(sq1mz) / creal(sq1pz)), asinh(cimag(sq1mz * conj(sq1pz)))));
	}
      return(c_complex_to_s7(sc, cacos(s7_to_c_complex(p))));
#else
      out_of_range(sc, sc->acos_symbol, int_one, p, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(p), MPFR_RNDN);
      goto ACOS_BIG_REAL;

    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(p), MPFR_RNDN);
      goto ACOS_BIG_REAL;

    case T_BIG_REAL:
      if (mpfr_inf_p(big_real(p)))
	{
	  if (mpfr_cmp_ui(big_real(p), 0) < 0)
	    return(make_complex_not_0i(sc, -NAN, -INFINITY)); /* match non-bignum choice */
	  return(make_complex_not_0i(sc, -NAN, INFINITY));
	}
      mpfr_set(sc->mpfr_1, big_real(p), MPFR_RNDN);
    ACOS_BIG_REAL:
      mpfr_set_ui(sc->mpfr_2, 1, MPFR_RNDN);
      if (mpfr_cmpabs(sc->mpfr_1, sc->mpfr_2) <= 0)
	{
	  mpfr_acos(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      mpc_set_fr(sc->mpc_1, sc->mpfr_1, MPC_RNDNN);
      mpc_acos(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));

    case T_BIG_COMPLEX:
      mpc_acos(sc->mpc_1, big_complex(p), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, p, sc->acos_symbol, a_number_string));
    }
}

static s7_pointer g_acos(s7_scheme *sc, s7_pointer args)
{
  #define H_acos "(acos z) returns acos(z); (cos (acos 1)) = 1"
  #define Q_acos sc->pl_nn
  return(acos_p_p(sc, car(args)));
}


/* -------------------------------- atan -------------------------------- */
static s7_pointer g_atan(s7_scheme *sc, s7_pointer args)
{
  #define H_atan "(atan z) returns atan(z), (atan y x) returns atan(y/x)"
  #define Q_atan s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_number_symbol, sc->is_real_symbol)
  /* actually if there are two args, both should be real, but how to express that in the signature? */

  s7_pointer x = car(args), y;
  /* currently (atan inf.0 inf.0) -> 0.78539816339745, and (atan inf.0 -inf.0) -> 2.3561944901923 (etc) */

  if (!is_pair(cdr(args)))
    {
      switch (type(x))
	{
	case T_INTEGER:  return((integer(x) == 0) ? int_zero : make_real(sc, atan((double)integer(x))));
	case T_RATIO:    return(make_real(sc, atan(fraction(x))));
	case T_REAL:     return(make_real(sc, atan(real(x))));

	case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
	  return(c_complex_to_s7(sc, catan(to_c_complex(x))));
#else
	  out_of_range(sc, sc->atan_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
	case T_BIG_INTEGER:
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  mpfr_atan(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  mpfr_atan(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_atan(sc->mpfr_1, big_real(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_atan(sc->mpc_1, big_complex(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_one_arg_p(sc, x, sc->atan_symbol, a_number_string));
	}}

  y = cadr(args);
  switch (type(x))
    {
    case T_INTEGER: case T_RATIO: case T_REAL:
      if (is_small_real(y))
	return(make_real(sc, atan2(s7_real(x), s7_real(y))));
#if WITH_GMP
      if (!is_real(y))
	return(method_or_bust(sc, y, sc->atan_symbol, args, T_REAL, 2));
      mpfr_set_d(sc->mpfr_1, s7_real(x), MPFR_RNDN);
      goto ATAN2_BIG_REAL;
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      goto ATAN2_BIG_REAL;
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      goto ATAN2_BIG_REAL;
    case T_BIG_REAL:
      mpfr_set(sc->mpfr_1, big_real(x), MPFR_RNDN);
      goto ATAN2_BIG_REAL;
#endif
    default:
      return(method_or_bust(sc, x, sc->atan_symbol, args, T_REAL, 1));
    }
#if WITH_GMP
 ATAN2_BIG_REAL:
  if (is_small_real(y))
    mpfr_set_d(sc->mpfr_2, s7_real(y), MPFR_RNDN);
  else
    if (is_t_big_real(y))
      mpfr_set(sc->mpfr_2, big_real(y), MPFR_RNDN);
    else
      if (is_t_big_integer(y))
	mpfr_set_z(sc->mpfr_2, big_integer(y), MPFR_RNDN);
      else
	if (is_t_big_ratio(y))
	  mpfr_set_q(sc->mpfr_2, big_ratio(y), MPFR_RNDN);
	else return(method_or_bust(sc, y, sc->atan_symbol, args, T_REAL, 2));
  mpfr_atan2(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif
}

static s7_double atan_d_dd(s7_double x, s7_double y) {return(atan2(x, y));}


/* -------------------------------- sinh -------------------------------- */
static s7_pointer sinh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);                    /* (sinh 0) -> 0 */

    case T_REAL:
    case T_RATIO:
      {
	s7_double y = s7_real(x);
#if WITH_GMP
	if (fabs(y) > SINH_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
	    mpfr_sinh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, sinh(y)));
      }

    case T_COMPLEX:
#if WITH_GMP
      if ((fabs(real_part(x)) > SINH_LIMIT) || (fabs(imag_part(x)) > SINH_LIMIT))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sinh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if HAVE_COMPLEX_NUMBERS
      return(c_complex_to_s7(sc, csinh(to_c_complex(x))));
#else
      out_of_range(sc, sc->sinh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      mpfr_sinh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      mpfr_sinh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_sinh(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_sinh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->sinh_symbol, a_number_string));
    }
}

static s7_pointer g_sinh(s7_scheme *sc, s7_pointer args)
{
  #define H_sinh "(sinh z) returns sinh(z)"
  #define Q_sinh sc->pl_nn
  return(sinh_p_p(sc, car(args)));
}

#if (!WITH_GMP)
static s7_double sinh_d_d(s7_double x) {return(sinh(x));}
#endif


/* -------------------------------- cosh -------------------------------- */
static s7_pointer cosh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_one);                   /* (cosh 0) -> 1 */

    case T_REAL:
    case T_RATIO:
      {
	s7_double y = s7_real(x);
#if WITH_GMP
	if (fabs(y) > SINH_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
	    mpfr_cosh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, cosh(y)));
      }

    case T_COMPLEX:
#if WITH_GMP
      if ((fabs(real_part(x)) > SINH_LIMIT) || (fabs(imag_part(x)) > SINH_LIMIT))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_cosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if HAVE_COMPLEX_NUMBERS
      return(c_complex_to_s7(sc, ccosh(to_c_complex(x))));
#else
      out_of_range(sc, sc->cosh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      mpfr_cosh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      mpfr_cosh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_cosh(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_cosh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->cosh_symbol, a_number_string));
    }
}

static s7_pointer g_cosh(s7_scheme *sc, s7_pointer args)
{
  #define H_cosh "(cosh z) returns cosh(z)"
  #define Q_cosh sc->pl_nn
  return(cosh_p_p(sc, car(args)));
}

#if (!WITH_GMP)
static s7_double cosh_d_d(s7_double x) {return(cosh(x));}
#endif


/* -------------------------------- tanh -------------------------------- */
#define TANH_LIMIT 350.0
static s7_pointer g_tanh(s7_scheme *sc, s7_pointer args)
{
  #define H_tanh "(tanh z) returns tanh(z)"
  #define Q_tanh sc->pl_nn

  s7_pointer x = car(args);
  switch (type(x))
    {
    case T_INTEGER: return((integer(x) == 0) ? int_zero : make_real(sc, tanh((s7_double)integer(x))));
    case T_RATIO:   return(make_real(sc, tanh(fraction(x))));
    case T_REAL:    return(make_real(sc, tanh(real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      if (real_part(x) > TANH_LIMIT)
	return(real_one);                         /* closer than 0.0 which is what ctanh is about to return! */
      if (real_part(x) < -TANH_LIMIT)
	return(make_real(sc, -1.0));              /* closer than ctanh's -0.0 */
      return(c_complex_to_s7(sc, ctanh(to_c_complex(x))));
#else
      out_of_range(sc, sc->tanh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      goto BIG_REAL_TANH;

    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      goto BIG_REAL_TANH;

    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x))) return(real_NaN);
      mpfr_set(sc->mpfr_1, big_real(x), MPFR_RNDN);

    BIG_REAL_TANH:
      if (mpfr_cmp_d(sc->mpfr_1, TANH_LIMIT) > 0) return(real_one);
      if (mpfr_cmp_d(sc->mpfr_1, -TANH_LIMIT) < 0) return(make_real(sc, -1.0));
      mpfr_tanh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));

    case T_BIG_COMPLEX:
      if ((MPC_INEX_RE(mpc_cmp_si_si(big_complex(x), TANH_LIMIT, 1))) > 0)
	return(real_one);
      if ((MPC_INEX_RE(mpc_cmp_si_si(big_complex(x), -TANH_LIMIT, 1))) < 0)
	return(make_real(sc, -1.0));

      if ((mpfr_nan_p(mpc_imagref(big_complex(x)))) ||
	  (mpfr_inf_p(mpc_imagref(big_complex(x)))))
	{
	  if (mpfr_cmp_ui(mpc_realref(big_complex(x)), 0) == 0)
	    return(make_complex_not_0i(sc, 0.0, NAN)); /* match non-bignum choice */
	  return(complex_NaN);
	}

      mpc_tanh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->tanh_symbol, a_number_string));
    }
}

static s7_double tanh_d_d(s7_double x) {return(tanh(x));}


/* -------------------------------- asinh -------------------------------- */
static s7_pointer asinh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER: return((integer(x) == 0) ? int_zero : make_real(sc, asinh((s7_double)integer(x))));
    case T_RATIO:   return(make_real(sc, asinh(fraction(x))));
    case T_REAL:    return(make_real(sc, asinh(real(x))));
    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(c_complex_to_s7(sc, casinh_1(to_c_complex(x))));
  #else
      return(c_complex_to_s7(sc, casinh(to_c_complex(x))));
  #endif
#else
      out_of_range(sc, sc->asinh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
      mpfr_asinh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      mpfr_asinh(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_asinh(sc->mpfr_1, big_real(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_asinh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->asinh_symbol, a_number_string));
    }
}

static s7_pointer g_asinh(s7_scheme *sc, s7_pointer args)
{
  #define H_asinh "(asinh z) returns asinh(z)"
  #define Q_asinh sc->pl_nn
  return(asinh_p_p(sc, car(args)));
}


/* -------------------------------- acosh -------------------------------- */
static s7_pointer acosh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 1) return(int_zero);
    case T_REAL:
    case T_RATIO:
      {
	s7_double x1 = s7_real(x);
	if (x1 >= 1.0)
	  return(make_real(sc, acosh(x1)));
      }
    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #ifdef __OpenBSD__
      return(c_complex_to_s7(sc, cacosh_1(s7_to_c_complex(x))));
  #else
      return(c_complex_to_s7(sc, cacosh(s7_to_c_complex(x)))); /* not to_c_complex because x might not be complex */
  #endif
#else
      /* since we can fall through to this branch, we need a better error message than "must be a number, not 0.0" */
      out_of_range(sc, sc->acosh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
      mpc_acosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_RATIO:
      mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
      mpc_acosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_REAL:
      mpc_set_fr(sc->mpc_1, big_real(x), MPC_RNDNN);
      mpc_acosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_COMPLEX:
      mpc_acosh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->acosh_symbol, a_number_string));
    }
}

static s7_pointer g_acosh(s7_scheme *sc, s7_pointer args)
{
  #define H_acosh "(acosh z) returns acosh(z)"
  #define Q_acosh sc->pl_nn
  return(acosh_p_p(sc, car(args)));
}


/* -------------------------------- atanh -------------------------------- */
static s7_pointer g_atanh(s7_scheme *sc, s7_pointer args)
{
  #define H_atanh "(atanh z) returns atanh(z)"
  #define Q_atanh sc->pl_nn

  s7_pointer x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);                    /* (atanh 0) -> 0 */
    case T_REAL:
    case T_RATIO:
      {
	s7_double x1 = s7_real(x);
	if (fabs(x1) < 1.0)
	  return(make_real(sc, atanh(x1)));
      }
      /* if we can't distinguish x from 1.0 even with long_doubles, we'll get inf.0:
       *    (atanh 9223372036854775/9223372036854776) -> 18.714973875119
       *    (atanh 92233720368547758/92233720368547757) -> inf.0
       *    (atanh (bignum 92233720368547758/92233720368547757)) -> 1.987812468492420421418925013176932317086E1+1.570796326794896619231321691639751442098E0i
       *    but the imaginary part is unnecessary
       */
    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(c_complex_to_s7(sc, catanh_1(s7_to_c_complex(x))));
  #else
      return(c_complex_to_s7(sc, catanh(s7_to_c_complex(x))));
  #endif
#else
      out_of_range(sc, sc->atanh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_2, big_integer(x), MPFR_RNDN);
      goto ATANH_BIG_REAL;

    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_2, big_ratio(x), MPFR_RNDN);
      goto ATANH_BIG_REAL;

    case T_BIG_REAL:
      mpfr_set(sc->mpfr_2, big_real(x), MPFR_RNDN);
    ATANH_BIG_REAL:
      mpfr_set_ui(sc->mpfr_1, 1, MPFR_RNDN);
      if (mpfr_cmpabs(sc->mpfr_2, sc->mpfr_1) < 0)
	{
	  mpfr_atanh(sc->mpfr_2, sc->mpfr_2, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_2));
	}
      mpc_set_fr(sc->mpc_1, sc->mpfr_2, MPC_RNDNN);
      mpc_atanh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));

    case T_BIG_COMPLEX:
      mpc_atanh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->atanh_symbol, a_number_string));
    }
}


/* -------------------------------- sqrt -------------------------------- */
static s7_pointer sqrt_p_p(s7_scheme *sc, s7_pointer p)
{
  switch (type(p))
    {
    case T_INTEGER:
      {
	s7_double sqx;
	if (integer(p) >= 0)
	  {
	    s7_int ix;
#if WITH_GMP
	    mpz_set_si(sc->mpz_1, integer(p));
	    mpz_sqrtrem(sc->mpz_1, sc->mpz_2, sc->mpz_1);
	    if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	      return(make_integer(sc, mpz_get_si(sc->mpz_1)));
	    mpfr_set_si(sc->mpfr_1, integer(p), MPFR_RNDN);
	    mpfr_sqrt(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif
	    sqx = sqrt((s7_double)integer(p));
	    ix = (s7_int)sqx;
	    return(((ix * ix) == integer(p)) ? make_integer(sc, ix) : make_real(sc, sqx));
	    /* Mark Weaver notes that (zero? (- (sqrt 9007199136250226) 94906265.0)) -> #t
	     * but (* 94906265 94906265) -> 9007199136250225 -- oops
	     * if we use bigfloats, we're ok:
	     *    (* (sqrt 9007199136250226.0) (sqrt 9007199136250226.0)) -> 9.007199136250226000000000000000000000026E15
	     * at least we return a real here, not an incorrect integer and (sqrt 9007199136250225) -> 94906265
	     */
	  }
#if HAVE_COMPLEX_NUMBERS
#if WITH_GMP
	mpc_set_si(sc->mpc_1, integer(p), MPC_RNDNN);
	mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	return(mpc_to_number(sc, sc->mpc_1));
#endif
	sqx = (s7_double)integer(p); /* we're trying to protect against (sqrt -9223372036854775808) where we can't negate the integer argument */
	return(s7_make_complex(sc, 0.0, sqrt((s7_double)(-sqx))));
#else
	out_of_range(sc, sc->sqrt_symbol, int_one, p, no_complex_numbers_string);
#endif
      }

    case T_RATIO:
      if (numerator(p) > 0) /* else it's complex, so it can't be a ratio */
	{
	  s7_int nm = (s7_int)sqrt(numerator(p));
	  if (nm * nm == numerator(p))
	    {
	      s7_int dn = (s7_int)sqrt(denominator(p));
	      if (dn * dn == denominator(p))
		return(make_ratio(sc, nm, dn));
	    }
	  return(make_real(sc, sqrt((s7_double)fraction(p))));
	}
#if HAVE_COMPLEX_NUMBERS
      return(s7_make_complex(sc, 0.0, sqrt((s7_double)(-fraction(p)))));
#else
      out_of_range(sc, sc->sqrt_symbol, int_one, p, no_complex_numbers_string);
#endif

    case T_REAL:
      if (is_NaN(real(p)))
	return(real_NaN);
      if (real(p) >= 0.0)
	return(make_real(sc, sqrt(real(p))));
      return(make_complex_not_0i(sc, 0.0, sqrt(-real(p))));

    case T_COMPLEX:    /* (* inf.0 (sqrt -1)) -> -nan+infi, but (sqrt -inf.0) -> 0+infi */
#if HAVE_COMPLEX_NUMBERS
      return(c_complex_to_s7(sc, csqrt(to_c_complex(p)))); /* sqrt(+inf.0+1.0i) -> +inf.0 */
#else
      out_of_range(sc, sc->sqrt_symbol, int_one, p, no_complex_numbers_string);
#endif

#if WITH_GMP
    case T_BIG_INTEGER:
      if (mpz_cmp_ui(big_integer(p), 0) >= 0)
	{
	  mpz_sqrtrem(sc->mpz_1, sc->mpz_2, big_integer(p));
	  if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	    return(mpz_to_integer(sc, sc->mpz_1));
	  mpfr_set_z(sc->mpfr_1, big_integer(p), MPFR_RNDN);
	  mpfr_sqrt(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      mpc_set_z(sc->mpc_1, big_integer(p), MPC_RNDNN);
      mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));

    case T_BIG_RATIO: /* if big ratio, check both num and den for squares */
      if (mpq_cmp_ui(big_ratio(p), 0, 1) < 0)
	{
	  mpc_set_q(sc->mpc_1, big_ratio(p), MPC_RNDNN);
	  mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
      mpz_sqrtrem(sc->mpz_1, sc->mpz_2, mpq_numref(big_ratio(p)));
      if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	{
	  mpz_sqrtrem(sc->mpz_3, sc->mpz_2, mpq_denref(big_ratio(p)));
	  if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	    {
	      mpq_set_num(sc->mpq_1, sc->mpz_1);
	      mpq_set_den(sc->mpq_1, sc->mpz_3);
	      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	    }}
      mpfr_set_q(sc->mpfr_1, big_ratio(p), MPFR_RNDN);
      mpfr_sqrt(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));

    case T_BIG_REAL:
      if (mpfr_cmp_ui(big_real(p), 0) < 0)
	{
	  mpc_set_fr(sc->mpc_1, big_real(p), MPC_RNDNN);
	  mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
      mpfr_sqrt(sc->mpfr_1, big_real(p), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));

    case T_BIG_COMPLEX:
      mpc_sqrt(sc->mpc_1, big_complex(p), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, p, sc->sqrt_symbol, a_number_string));
    }
}

static s7_pointer g_sqrt(s7_scheme *sc, s7_pointer args)
{
  #define H_sqrt "(sqrt z) returns the square root of z"
  #define Q_sqrt sc->pl_nn
  return(sqrt_p_p(sc, car(args)));
}


/* -------------------------------- expt -------------------------------- */
static s7_int int_to_int(s7_int x, s7_int n)
{
  /* from GSL */
  s7_int value = 1;
  do {
    if (n & 1) value *= x;
    n >>= 1;
#if HAVE_OVERFLOW_CHECKS
    if (multiply_overflow(x, x, &x))
      break;
#else
    x *= x;
#endif
  } while (n);
  return(value);
}

static const int64_t nth_roots[63] = {
  S7_INT64_MAX, S7_INT64_MAX, 3037000499LL, 2097151, 55108, 6208, 1448, 511, 234, 127, 78, 52, 38, 28, 22,
  18, 15, 13, 11, 9, 8, 7, 7, 6, 6, 5, 5, 5, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2};

static bool int_pow_ok(s7_int x, s7_int y)
{
  return((y < S7_INT_BITS) && (nth_roots[y] >= s7_int_abs(x)));
}

#if WITH_GMP
static s7_pointer real_part_p_p(s7_scheme *sc, s7_pointer p);
static bool lt_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2);

static s7_pointer big_expt(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = car(args), y = cadr(args), res;
  if (!is_number(x))
    return(method_or_bust_with_type(sc, x, sc->expt_symbol, args, a_number_string, 1));
  if (!is_number(y))
    return(method_or_bust_with_type(sc, y, sc->expt_symbol, args, a_number_string, 2));

  if (is_zero(x))
    {
      if ((s7_is_integer(x)) &&
	  (s7_is_integer(y)) &&
	  (is_zero(y)))
	return(int_one);

      if (is_real(y))
	{
	  if (is_negative(sc, y))
	    division_by_zero_error_2(sc, sc->expt_symbol, x, y);
	}
      else
	if (is_negative(sc, real_part_p_p(sc, y))) /* handle big_complex as well as complex */
	  division_by_zero_error_2(sc, sc->expt_symbol, x, y);

      if ((is_rational(x)) &&
	  (is_rational(y)))
	return(int_zero);
      return(real_zero);
    }

  if (s7_is_integer(y))
    {
      s7_int yval = s7_integer_clamped_if_gmp(sc, y);
      if (yval == 0)
	return((is_rational(x)) ? int_one : real_one);

      if (yval == 1)
	return(x);

      if ((!is_big_number(x)) &&
	  ((is_one(x)) || (is_zero(x))))
	return(x);

      if ((yval < S7_INT32_MAX) &&
	  (yval > S7_INT32_MIN))
	{
	  /* (protect against gmp exception if for example (expt 1/9223372036854775807 -9223372036854775807) */
	  if (s7_is_integer(x))
	    {
	      if (is_t_big_integer(x))
		mpz_set(sc->mpz_2, big_integer(x));
	      else mpz_set_si(sc->mpz_2, integer(x));
	      if (yval >= 0)
		{
		  mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)yval);
		  return(mpz_to_integer(sc, sc->mpz_2));
		}
	      mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)(-yval));
	      mpq_set_z(sc->mpq_1, sc->mpz_2);
	      mpq_inv(sc->mpq_1, sc->mpq_1);
	      if (mpz_cmp_ui(mpq_denref(sc->mpq_1), 1) == 0)
		return(mpz_to_integer(sc, mpq_numref(sc->mpq_1)));
	      return(mpq_to_big_ratio(sc, sc->mpq_1));
	    }

	  if (s7_is_ratio(x)) /* here y is an integer */
	    {
	      if (is_t_big_ratio(x))
		{
		  mpz_set(sc->mpz_1, mpq_numref(big_ratio(x)));
		  mpz_set(sc->mpz_2, mpq_denref(big_ratio(x)));
		}
	      else
		{
		  mpz_set_si(sc->mpz_1, numerator(x));
		  mpz_set_si(sc->mpz_2, denominator(x));
		}
	      if (yval >= 0)
		{
		  mpz_pow_ui(sc->mpz_1, sc->mpz_1, (uint32_t)yval);
		  mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)yval);
		  mpq_set_num(sc->mpq_1, sc->mpz_1);
		  mpq_set_den(sc->mpq_1, sc->mpz_2);
		}
	      else
		{
		  yval = -yval;
		  mpz_pow_ui(sc->mpz_1, sc->mpz_1, (uint32_t)yval);
		  mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)yval);
		  mpq_set_num(sc->mpq_1, sc->mpz_2);
		  mpq_set_den(sc->mpq_1, sc->mpz_1);
		  mpq_canonicalize(sc->mpq_1);
		}
	      if (mpz_cmp_ui(mpq_denref(sc->mpq_1), 1) == 0)
		return(mpz_to_integer(sc, mpq_numref(sc->mpq_1)));
	      return(mpq_to_big_ratio(sc, sc->mpq_1));
	    }

	  if (is_real(x))
	    {
	      if (is_t_big_real(x))
		mpfr_set(sc->mpfr_1, big_real(x), MPFR_RNDN);
	      else mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	      mpfr_pow_si(sc->mpfr_1, sc->mpfr_1, yval, MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }}}

  if ((is_t_ratio(y)) &&              /* not s7_is_ratio which accepts bignums */
      (numerator(y) == 1))
    {
      if (denominator(y) == 2)
	return(sqrt_p_p(sc, x));

      if ((is_real(x)) &&
	  (denominator(y) == 3))
	{
	  any_real_to_mpfr(sc, x, sc->mpfr_1);
	  mpfr_cbrt(sc->mpfr_1, sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}}

  res = any_number_to_mpc(sc, y, sc->mpc_2);
  if (res == real_infinity)
    {
      if (is_one(x)) return(int_one);
      if (!is_real(x)) return((is_negative(sc, y)) ? real_zero : complex_NaN);
      if (is_zero(x))
	{
	  if (is_negative(sc, y)) division_by_zero_error_2(sc, sc->expt_symbol, x, y);
	  return(real_zero);
	}
      if (lt_b_pi(sc, x, 0))
	{
	  if (lt_b_pi(sc, x, -1))
	    return((is_positive(sc, y)) ? real_infinity : real_zero);
	  return((is_positive(sc, y)) ? real_zero : real_infinity);
	}
      if (lt_b_pi(sc, x, 1))
	return((is_positive(sc, y)) ? real_zero : real_infinity);
      return((is_positive(sc, y)) ? real_infinity : real_zero);
    }
  if (res) return(complex_NaN);

  if ((is_real(x)) &&
      (is_real(y)) &&
      (is_positive(sc, x)))
    {
      res = any_real_to_mpfr(sc, x, sc->mpfr_1);
      if (res)
	{
	  if (res == real_infinity)
	    {
	      if (is_negative(sc, y)) return(real_zero);
	      return((is_zero(y)) ? real_one : real_infinity);
	    }
	  return(complex_NaN);
	}
      mpfr_pow(sc->mpfr_1, sc->mpfr_1, mpc_realref(sc->mpc_2), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    }

  res = any_number_to_mpc(sc, x, sc->mpc_1);
  if (res)
    {
      if ((res == real_infinity) && (is_real(y)))
	{
	  if (is_negative(sc, y)) return(real_zero);
	  return((is_zero(y)) ? real_one : real_infinity);
	}
      return(complex_NaN);
    }
  if (mpc_cmp_si_si(sc->mpc_1, 0, 0) == 0)
    return(int_zero);
  if (mpc_cmp_si_si(sc->mpc_1, 1, 0) == 0)
    return(int_one);

  mpc_pow(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);

  if ((!mpfr_nan_p(mpc_imagref(sc->mpc_1))) && (mpfr_cmp_ui(mpc_imagref(sc->mpc_1), 0) == 0)) /* (expt -inf.0 1/3) -> +inf.0+nan.0i in mpc */
    {
      if ((is_rational(car(args))) &&
	  (is_rational(cadr(args))) &&
	  (mpfr_integer_p(mpc_realref(sc->mpc_1)) != 0))
	{
	  /* mpfr_integer_p can be confused: (expt 2718/1000 (bignum 617/5)) returns an int32_t if precision=128, float if 512 */
	  /*   so first make sure we're within (say) 31 bits */
	  mpfr_set_ui(sc->mpfr_1, S7_INT32_MAX, MPFR_RNDN);
	  if (mpfr_cmpabs(mpc_realref(sc->mpc_1), sc->mpfr_1) < 0)
	    {
	      mpfr_get_z(sc->mpz_1, mpc_realref(sc->mpc_1), MPFR_RNDN);
	      return(mpz_to_integer(sc, sc->mpz_1));
	    }}
      mpfr_set(sc->mpfr_1, mpc_realref(sc->mpc_1), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    }
  return(mpc_to_number(sc, sc->mpc_1));
}
#endif

static s7_pointer expt_p_pp(s7_scheme *sc, s7_pointer n, s7_pointer pw)
{
  if (!is_number(n))
    return(method_or_bust_with_type_pp(sc, n, sc->expt_symbol, n, pw, a_number_string, 1));
  if (!is_number(pw))
    return(method_or_bust_with_type_pp(sc, pw, sc->expt_symbol, n, pw, a_number_string, 2));

  if (is_zero(n))
    {
      if (is_zero(pw))
	{
	  if ((s7_is_integer(n)) && (s7_is_integer(pw)))       /* (expt 0 0) -> 1 */
	    return(int_one);
	  return(real_zero);                                   /* (expt 0.0 0) -> 0.0 */
	}
      if (is_real(pw))
	{
	  if (is_negative(sc, pw))                              /* (expt 0 -1) */
	    division_by_zero_error_2(sc, sc->expt_symbol, n, pw);
	  /* (Clisp gives divide-by-zero error here, Guile returns inf.0) */

	  if (is_NaN(s7_real(pw)))                             /* (expt 0 +nan.0) */
	    return(pw);
	}
      else
	{                                                      /* (expt 0 a+bi) */
	  if (real_part(pw) < 0.0)                             /* (expt 0 -1+i) */
	    division_by_zero_error_2(sc, sc->expt_symbol, n, pw);
	  if ((is_NaN(real_part(pw))) ||                       /* (expt 0 0+1/0i) */
	      (is_NaN(imag_part(pw))))
	    return(real_NaN);
	}
      if ((s7_is_integer(n)) && (s7_is_integer(pw)))           /* pw != 0, (expt 0 2312) */
	return(int_zero);
      return(real_zero);                                       /* (expt 0.0 123123) */
    }

  if (is_one(pw))
    {
      if (s7_is_integer(pw))                                   /* (expt x 1) */
	return(n);
      if (is_rational(n))                                      /* (expt ratio 1.0) */
	return(make_real(sc, rational_to_double(sc, n)));
      return(n);
    }
  if (is_t_integer(pw))
    {
      s7_int y = integer(pw);
      if (y == 0)
	{
	  if (is_rational(n))                                 /* (expt 3 0) */
	    return(int_one);
	  if ((is_NaN(s7_real_part(n))) ||                    /* (expt 1/0 0) -> NaN */
	      (is_NaN(s7_imag_part(n))))                      /* (expt (complex 0 1/0) 0) -> NaN */
	    return(n);
	  return(real_one);                                   /* (expt 3.0 0) */
	}
      switch (type(n))
	{
	case T_INTEGER:
	  {
	    s7_int x = integer(n);
	    if (x == 1)                                       /* (expt 1 y) */
	      return(n);

	    if (x == -1)
	      {
		if (y == S7_INT64_MIN)                        /* (expt -1 most-negative-fixnum) */
		  return(int_one);
		if (s7_int_abs(y) & 1)                        /* (expt -1 odd-int) */
		  return(n);
		return(int_one);                              /* (expt -1 even-int) */
	      }

	    if (y == S7_INT64_MIN)                            /* (expt x most-negative-fixnum) */
	      return(int_zero);
	    if (x == S7_INT64_MIN)                            /* (expt most-negative-fixnum y) */
	      return(make_real(sc, pow((double)x, (double)y)));

	    if (int_pow_ok(x, s7_int_abs(y)))
	      {
		if (y > 0)
		  return(make_integer(sc, int_to_int(x, y)));
		return(make_ratio(sc, 1, int_to_int(x, -y)));
	      }}
	  break;

	case T_RATIO:
	  {
	    s7_int nm = numerator(n), dn = denominator(n);
	    if (y == S7_INT64_MIN)
	      {
		if (s7_int_abs(nm) > dn)
		  return(int_zero);                  /* (expt 4/3 most-negative-fixnum) -> 0? */
		return(real_infinity);               /* (expt 3/4 most-negative-fixnum) -> inf? */
	      }
	    if ((int_pow_ok(nm, s7_int_abs(y))) &&
		(int_pow_ok(dn, s7_int_abs(y))))
	      {
		if (y > 0)
		  return(make_ratio(sc, int_to_int(nm, y), int_to_int(dn, y)));
		return(make_ratio_with_div_check(sc, sc->expt_symbol, int_to_int(dn, -y), int_to_int(nm, -y)));
	      }}
	  break;
	  /* occasionally int^rat can be int32_t but it happens so infrequently it's probably not worth checking
	   *  one possibly easy case: (expt 1 1/2) -> 1 (-1?) etc
	   */

	case T_REAL:
	  /* (expt -1.0 most-positive-fixnum) should be -1.0
	   * (expt -1.0 (+ (expt 2 53) 1)) -> -1.0
	   * (expt -1.0 (- 1 (expt 2 54))) -> -1.0
	   */
	  if (real(n) == -1.0)
	    {
	      if (y == S7_INT64_MIN)
		return(real_one);
	      return((s7_int_abs(y) & 1) ? n : real_one);
	    }
	  break;

	case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
	  if ((s7_real_part(n) == 0.0) &&
	      ((s7_imag_part(n) == 1.0) ||
	       (s7_imag_part(n) == -1.0)))
	    {
	      bool yp = (y > 0), np = (s7_imag_part(n) > 0.0);
	      switch (s7_int_abs(y) % 4)
		{
		case 0: return(real_one);
		case 1: return(make_complex_not_0i(sc, 0.0, (yp == np) ? 1.0 : -1.0));
		case 2: return(make_real(sc, -1.0));
		case 3: return(make_complex_not_0i(sc, 0.0, (yp == np) ? -1.0 : 1.0));
		}}
#else
	  out_of_range(sc, sc->expt_symbol, int_two, n, no_complex_numbers_string);
#endif
	  break;
	}}

  if ((is_real(n)) &&
      (is_real(pw)))
    {
      s7_double x, y;

      if ((is_t_ratio(pw)) &&
	  (numerator(pw) == 1))
	{
	  if (denominator(pw) == 2)
	    return(sqrt_p_p(sc, n));
	  if (denominator(pw) == 3)
	    return(make_real(sc, cbrt(s7_real(n)))); /* (expt 27 1/3) should be 3, not 3.0... */
	  /* but: (expt 512/729 1/3) -> 0.88888888888889, and 4 -> sqrt(sqrt...) etc? */
	}

      x = s7_real(n);
      y = s7_real(pw);
      if (is_NaN(x)) return(n);
      if (is_NaN(y)) return(pw);
      if (y == 0.0) return(real_one);
      /* I think pow(rl, inf) is ok */
      if (x > 0.0)
	return(make_real(sc, pow(x, y)));      /* tricky cases abound here: (expt -1 1/9223372036854775807) */
    }

  /* (expt 0+i 1e+16) = 0.98156860153485-0.19111012657867i ?
   * (expt 0+i 1+1/0i) = 0.0 ??
   */
  return(c_complex_to_s7(sc, cpow(s7_to_c_complex(n), s7_to_c_complex(pw))));
}

static s7_pointer g_expt(s7_scheme *sc, s7_pointer args)
{
  #define H_expt "(expt z1 z2) returns z1^z2"
  #define Q_expt sc->pcl_n
#if WITH_GMP
  return(big_expt(sc, args));
  /* big_expt sometimes chooses a different value: g_expt (expt -1 1/3) is -1, but big_expt (expt -1 (bignum 1/3)) is (complex 1/2 (/ (sqrt 3) 2)) */
#endif
  return(expt_p_pp(sc, car(args), cadr(args)));
}


/* -------------------------------- lcm -------------------------------- */
#if WITH_GMP
static s7_pointer big_lcm(s7_scheme *sc, s7_int num, s7_int den, s7_pointer args)
{
  mpz_set_si(sc->mpz_3, num);
  mpz_set_si(sc->mpz_4, den);

  for (s7_pointer x = args; is_pair(x); x = cdr(x))
    {
      s7_pointer rat = car(x);
      switch (type(rat))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(rat));
	  mpz_lcm(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  mpz_set_si(sc->mpz_4, 1);
	  break;
	case T_RATIO:
	  mpz_set_si(sc->mpz_1, numerator(rat));
	  mpz_set_si(sc->mpz_2, denominator(rat));
	  mpz_lcm(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  mpz_gcd(sc->mpz_4, sc->mpz_4, sc->mpz_2);
	  break;
	case T_BIG_INTEGER:
	  mpz_lcm(sc->mpz_3, sc->mpz_3, big_integer(rat));
	  mpz_set_si(sc->mpz_4, 1);
	  break;
	case T_BIG_RATIO:
	  mpz_lcm(sc->mpz_3, sc->mpz_3, mpq_numref(big_ratio(rat)));
	  mpz_gcd(sc->mpz_4, sc->mpz_4, mpq_denref(big_ratio(rat)));
	  break;
	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_argument_with_type(sc, sc->lcm_symbol, position_of(x, args), rat, a_rational_string);
	default:
	  return(method_or_bust_with_type(sc, rat, sc->lcm_symbol,
					  set_ulist_1(sc, mpz_to_rational(sc, sc->mpz_3, sc->mpz_4), x),
					  a_rational_string, position_of(x, args)));
	}}
  return(mpz_to_rational(sc, sc->mpz_3, sc->mpz_4));
}
#endif

static s7_pointer g_lcm(s7_scheme *sc, s7_pointer args)
{
  /* (/ (* m n) (gcd m n)), (lcm a b c) -> (lcm a (lcm b c)) */
  #define H_lcm "(lcm ...) returns the least common multiple of its rational arguments"
  #define Q_lcm sc->pcl_f

  s7_int n = 1, d = 0;

  if (!is_pair(args))
    return(int_one);

  if (!is_pair(cdr(args)))
    {
      if (!is_rational(car(args)))
	return(method_or_bust_with_type(sc, car(args), sc->lcm_symbol, args, a_rational_string, 1));
      return(g_abs(sc, args));
    }

  for (s7_pointer p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer x = car(p);
      s7_int b;
#if HAVE_OVERFLOW_CHECKS
      s7_int n1;
#endif
      switch (type(x))
	{
	case T_INTEGER:
	  d = 1;
	  if (integer(x) == 0) /* return 0 unless there's a wrong-type-arg (geez what a mess) */
	    {
	      for (p = cdr(p); is_pair(p); p = cdr(p))
		{
		  s7_pointer x1 = car(p);
		  if (is_number(x1))
		    {
		      if (!is_rational(x1))
			wrong_type_argument_with_type(sc, sc->lcm_symbol, position_of(p, args), x1, a_rational_string);
		    }
		  else
		    if (has_active_methods(sc, x1))
		      {
			s7_pointer f = find_method_with_let(sc, x1, sc->is_rational_symbol);
			if ((f == sc->undefined) ||
			    (is_false(sc, s7_apply_function(sc, f, set_plist_1(sc, x1)))))
			  wrong_type_argument_with_type(sc, sc->lcm_symbol, position_of(p, args), x1, a_rational_string);
		      }
		    else wrong_type_argument_with_type(sc, sc->lcm_symbol, position_of(p, args), x1, a_rational_string);
		}
	      return(int_zero);
	    }
	  b = integer(x);
	  if (b < 0)
	    {
	      if (b == S7_INT64_MIN)
#if WITH_GMP
		return(big_lcm(sc, n, d, p));
#else
		simple_out_of_range(sc, sc->lcm_symbol, args, its_too_large_string);
#endif
	      b = -b;
	    }
#if HAVE_OVERFLOW_CHECKS
	  if (multiply_overflow(n / c_gcd(n, b), b, &n1))
#if WITH_GMP
	    return(big_lcm(sc, n, d, p));
#else
	    simple_out_of_range(sc, sc->lcm_symbol, args, result_is_too_large_string);
#endif
	  n = n1;
#else
	  n = (n / c_gcd(n, b)) * b;
#endif
	  break;

	case T_RATIO:
	  b = numerator(x);
	  if (b < 0)
	    {
	      if (b == S7_INT64_MIN)
#if WITH_GMP
		return(big_lcm(sc, n, d, p));
#else
		simple_out_of_range(sc, sc->lcm_symbol, args, its_too_large_string);
#endif
	      b = -b;
	    }
#if HAVE_OVERFLOW_CHECKS
	  if (multiply_overflow(n / c_gcd(n, b), b, &n1))  /* (lcm 92233720368547758/3 3005/2) */
#if WITH_GMP
	    return(big_lcm(sc, n, d, p));
#else
	    simple_out_of_range(sc, sc->lcm_symbol, args, intermediate_too_large_string);
#endif
          n = n1;
#else
	  n = (n / c_gcd(n, b)) * b;
#endif
	  if (d == 0)
	    d = (p == args) ? denominator(x) : 1;
	  else d = c_gcd(d, denominator(x));
	  break;

#if WITH_GMP
	case T_BIG_INTEGER:
	  d = 1;
	case T_BIG_RATIO:
	  return(big_lcm(sc, n, d, p));
#endif
	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_argument_with_type(sc, sc->lcm_symbol, position_of(p, args), x, a_rational_string);

	default:
	  return(method_or_bust_with_type(sc, x, sc->lcm_symbol,
					  set_ulist_1(sc, (d <= 1) ? make_integer(sc, n) : make_ratio_with_div_check(sc, sc->lcm_symbol, n, d), p),
					  a_rational_string, position_of(p, args)));
	}}
  return((d <= 1) ? make_integer(sc, n) : make_simple_ratio(sc, n, d));
}


/* -------------------------------- gcd -------------------------------- */
#if WITH_GMP
static s7_pointer big_gcd(s7_scheme *sc, s7_int num, s7_int den, s7_pointer args)
{
  mpz_set_si(sc->mpz_3, num);
  mpz_set_si(sc->mpz_4, den);

  for (s7_pointer x = args; is_pair(x); x = cdr(x))
    {
      s7_pointer rat = car(x);
      switch (type(rat))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(rat));
	  mpz_gcd(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  break;
	case T_RATIO:
	  mpz_set_si(sc->mpz_1, numerator(rat));
	  mpz_set_si(sc->mpz_2, denominator(rat));
	  mpz_gcd(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  mpz_lcm(sc->mpz_4, sc->mpz_4, sc->mpz_2);
	  break;
	case T_BIG_INTEGER:
	  mpz_gcd(sc->mpz_3, sc->mpz_3, big_integer(rat));
	  break;
	case T_BIG_RATIO:
	  mpz_gcd(sc->mpz_3, sc->mpz_3, mpq_numref(big_ratio(rat)));
	  mpz_lcm(sc->mpz_4, sc->mpz_4, mpq_denref(big_ratio(rat)));
	  break;
	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_argument_with_type(sc, sc->gcd_symbol, position_of(x, args), rat, a_rational_string);
	default:
	  return(method_or_bust_with_type(sc, rat, sc->gcd_symbol,
					  set_ulist_1(sc, mpz_to_rational(sc, sc->mpz_3, sc->mpz_4), x),
					  a_rational_string, position_of(x, args)));
	}}
  return(mpz_to_rational(sc, sc->mpz_3, sc->mpz_4));
}
#endif

static s7_pointer g_gcd(s7_scheme *sc, s7_pointer args)
{
  #define H_gcd "(gcd ...) returns the greatest common divisor of its rational arguments"
  #define Q_gcd sc->pcl_f

  s7_int n = 0, d = 1;

  if (!is_pair(args))       /* (gcd) */
    return(int_zero);

  if (!is_pair(cdr(args)))  /* (gcd 3/4) */
    {
      if (!is_rational(car(args)))
	return(method_or_bust_with_type(sc, car(args), sc->gcd_symbol, args, a_rational_string, 1));
      return(abs_p_p(sc, car(args)));
    }

  for (s7_pointer p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer x = car(p);
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == S7_INT64_MIN)
#if WITH_GMP
	    return(big_gcd(sc, n, d, p));
#else
	    simple_out_of_range(sc, sc->lcm_symbol, args, its_too_large_string);
#endif
	  n = c_gcd(n, integer(x));
	  break;

	case T_RATIO:
	  {
#if HAVE_OVERFLOW_CHECKS
	    s7_int dn;
#endif
	    n = c_gcd(n, numerator(x));
	    if (d == 1)
	      d = denominator(x);
	    else
	      {
		s7_int b = denominator(x);
#if HAVE_OVERFLOW_CHECKS
		if (multiply_overflow(d / c_gcd(d, b), b, &dn)) /* (gcd 1/92233720368547758 1/3005) */
#if WITH_GMP
		  return(big_gcd(sc, n, d, x));
#else
		  simple_out_of_range(sc, sc->gcd_symbol, args, intermediate_too_large_string);
#endif
		d = dn;
#else
		d = (d / c_gcd(d, b)) * b;
#endif
	      }}
	  break;

#if WITH_GMP
	case T_BIG_INTEGER:
	case T_BIG_RATIO:
	  return(big_gcd(sc, n, d, p));
#endif

	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_argument_with_type(sc, sc->gcd_symbol, position_of(p, args), x, a_rational_string);

	default:
	  return(method_or_bust_with_type(sc, x, sc->gcd_symbol,
					  set_ulist_1(sc, (d <= 1) ? make_integer(sc, n) : make_ratio_with_div_check(sc, sc->gcd_symbol, n, d), p),
					  a_rational_string, position_of(p, args)));
	}}
  return((d <= 1) ? make_integer(sc, n) : make_simple_ratio(sc, n, d));
}


/* -------------------------------- floor -------------------------------- */
static s7_pointer floor_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      {
	s7_int val = numerator(x) / denominator(x);
	/* C "/" truncates? -- C spec says "truncation toward 0" */
	/* we're avoiding "floor" here because the int->double conversion introduces inaccuracies for big numbers
	 *   but it's used by opt_i_d_c (via s7_number_to_real) so floor_i_7d below can return different results:
	 *   (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (display (floor 3441313796169221281/1720656898084610641)) (newline))) (func)): 1
	 *   (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (display (/ (floor 3441313796169221281/1720656898084610641))) (newline))) (func)): 1/2
	 */
	return((numerator(x) < 0) ? make_integer(sc, val - 1) : make_integer(sc, val)); /* not "val" because it might be truncated to 0 */
      }
    case T_REAL:
      {
	s7_double z = real(x);
	if (is_NaN(z))
	  simple_out_of_range(sc, sc->floor_symbol, x, its_nan_string);
	if (is_inf(z))
	  simple_out_of_range(sc, sc->floor_symbol, x, its_infinite_string);
#if WITH_GMP
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, z, MPFR_RNDN);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_1, MPFR_RNDD);
	    return(mpz_to_integer(sc, sc->mpz_1));
	  }
#else
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  simple_out_of_range(sc, sc->floor_symbol, x, its_too_large_string);
#endif
	return(make_integer(sc, (s7_int)floor(z)));
	/* floor here rounds down, whereas a straight int<=real coercion apparently rounds towards 0 */
      }
#if WITH_GMP
    case T_BIG_INTEGER:
      return(x);
    case T_BIG_RATIO:
      mpz_fdiv_q(sc->mpz_1, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	simple_out_of_range(sc, sc->floor_symbol, x, its_nan_string);
      if (mpfr_inf_p(big_real(x)))
	simple_out_of_range(sc, sc->floor_symbol, x, its_infinite_string);
      mpfr_get_z(sc->mpz_1, big_real(x), MPFR_RNDD);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      return(s7_wrong_type_arg_error(sc, "floor", 0, x, "a real number"));
    default:
      return(method_or_bust_p(sc, x, sc->floor_symbol, T_REAL));
    }
}

static s7_pointer g_floor(s7_scheme *sc, s7_pointer args)
{
  #define H_floor "(floor x) returns the integer closest to x toward -inf"
  #define Q_floor s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(floor_p_p(sc, car(args)));
}

static s7_int floor_i_i(s7_int i) {return(i);}

#if (!WITH_GMP)
static s7_int floor_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    simple_out_of_range(sc, sc->floor_symbol, real_NaN, its_nan_string);
  if (fabs(x) > DOUBLE_TO_INT64_LIMIT)
    simple_out_of_range(sc, sc->floor_symbol, wrap_real(sc, x), its_too_large_string);
  return((s7_int)floor(x));
}

static s7_int floor_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p)) return(integer(p));
  if (is_t_real(p)) return(floor_i_7d(sc, real(p)));
  if (is_t_ratio(p)) /* for consistency with floor_p_p, don't use floor(fraction(p)) */
    {
      s7_int val = numerator(p) / denominator(p);
      return((numerator(p) < 0) ? val - 1 : val);
    }
  return(s7_integer(method_or_bust_p(sc, p, sc->floor_symbol, T_REAL)));
}
#endif


/* -------------------------------- ceiling -------------------------------- */
static s7_pointer ceiling_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      {
	s7_int val = numerator(x) / denominator(x);
	return((numerator(x) < 0) ? make_integer(sc, val) : make_integer(sc, val + 1));
      }
    case T_REAL:
      {
	s7_double z = real(x);
	if (is_NaN(z))
	  simple_out_of_range(sc, sc->ceiling_symbol, x, its_nan_string);
	if (is_inf(z))
	  simple_out_of_range(sc, sc->ceiling_symbol, x, its_infinite_string);
#if WITH_GMP
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, z, MPFR_RNDN);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_1, MPFR_RNDU);
	    return(mpz_to_integer(sc, sc->mpz_1));
	  }
#else
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  simple_out_of_range(sc, sc->ceiling_symbol, x, its_too_large_string);
#endif
	return(make_integer(sc, (s7_int)ceil(real(x))));
      }
#if WITH_GMP
    case T_BIG_INTEGER:
      return(x);
    case T_BIG_RATIO:
      mpz_cdiv_q(sc->mpz_1, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	simple_out_of_range(sc, sc->ceiling_symbol, x, its_nan_string);
      if (mpfr_inf_p(big_real(x)))
	simple_out_of_range(sc, sc->ceiling_symbol, x, its_infinite_string);
      mpfr_get_z(sc->mpz_1, big_real(x), MPFR_RNDU);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      return(s7_wrong_type_arg_error(sc, "ceiling", 0, x, "a real number"));
    default:
      return(method_or_bust_p(sc, x, sc->ceiling_symbol, T_REAL));
    }
}

static s7_pointer g_ceiling(s7_scheme *sc, s7_pointer args)
{
  #define H_ceiling "(ceiling x) returns the integer closest to x toward inf"
  #define Q_ceiling s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(ceiling_p_p(sc, car(args)));
}

static s7_int ceiling_i_i(s7_int i) {return(i);}

#if (!WITH_GMP)
static s7_int ceiling_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    simple_out_of_range(sc, sc->ceiling_symbol, real_NaN, its_nan_string);
  if ((is_inf(x)) ||
      (x > DOUBLE_TO_INT64_LIMIT) || (x < -DOUBLE_TO_INT64_LIMIT))
    simple_out_of_range(sc, sc->ceiling_symbol, wrap_real(sc, x), its_too_large_string);
  return((s7_int)ceil(x));
}

static s7_int ceiling_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p)) return(integer(p));
  if (is_t_real(p)) return(ceiling_i_7d(sc, real(p)));
  if (is_t_ratio(p)) return((s7_int)(ceil(fraction(p))));
  return(s7_integer(method_or_bust_p(sc, p, sc->ceiling_symbol, T_REAL)));
}
#endif


/* -------------------------------- truncate -------------------------------- */
static s7_pointer truncate_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      return(make_integer(sc, (s7_int)(numerator(x) / denominator(x)))); /* C "/" already truncates (but this divide is not accurate over e13) */
    case T_REAL:
      {
	s7_double z = real(x);
	if (is_NaN(z))
	  simple_out_of_range(sc, sc->truncate_symbol, x, its_nan_string);
	if (is_inf(z))
	  simple_out_of_range(sc, sc->truncate_symbol, x, its_infinite_string);
#if WITH_GMP
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_1, MPFR_RNDZ);
	    return(mpz_to_integer(sc, sc->mpz_1));
	  }
#else
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  simple_out_of_range(sc, sc->truncate_symbol, x, its_too_large_string);
#endif
	return((z > 0.0) ? make_integer(sc, (s7_int)floor(z)) : make_integer(sc, (s7_int)ceil(z)));
      }
#if WITH_GMP
    case T_BIG_INTEGER:
      return(x);
    case T_BIG_RATIO:
      mpz_tdiv_q(sc->mpz_1, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	simple_out_of_range(sc, sc->truncate_symbol, x, its_nan_string);
      if (mpfr_inf_p(big_real(x)))
	simple_out_of_range(sc, sc->truncate_symbol, x, its_infinite_string);
      mpfr_get_z(sc->mpz_1, big_real(x), MPFR_RNDZ);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      return(s7_wrong_type_arg_error(sc, "truncate", 0, x, "a real number"));
    default:
      return(method_or_bust_p(sc, x, sc->truncate_symbol, T_REAL));
    }
}

static s7_pointer g_truncate(s7_scheme *sc, s7_pointer args)
{
  #define H_truncate "(truncate x) returns the integer closest to x toward 0"
  #define Q_truncate s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(truncate_p_p(sc, car(args)));
}

static s7_int truncate_i_i(s7_int i) {return(i);}

#if (!WITH_GMP)
static s7_int truncate_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    simple_out_of_range(sc, sc->truncate_symbol, real_NaN, its_nan_string);
  if (is_inf(x))
    simple_out_of_range(sc, sc->truncate_symbol, wrap_real(sc, x), its_infinite_string);
  if (fabs(x) > DOUBLE_TO_INT64_LIMIT)
    simple_out_of_range(sc, sc->truncate_symbol, wrap_real(sc, x), its_too_large_string);
  return((x > 0.0) ? (s7_int)floor(x) : (s7_int)ceil(x));
}
#endif


/* -------------------------------- round -------------------------------- */
static s7_double r5rs_round(s7_double x)
{
  s7_double fl = floor(x), ce = ceil(x);
  s7_double dfl = x - fl;
  s7_double dce = ce - x;
  if (dfl > dce) return(ce);
  if (dfl < dce) return(fl);
  return((fmod(fl, 2.0) == 0.0) ? fl : ce);
}

static s7_pointer round_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      {
	s7_int truncated = numerator(x) / denominator(x), remains = numerator(x) % denominator(x);
	long_double frac = s7_fabsl((long_double)remains / (long_double)denominator(x));
	if ((frac > 0.5) ||
	    ((frac == 0.5) &&
	     (truncated % 2 != 0)))
	  return((numerator(x) < 0) ? make_integer(sc, truncated - 1) : make_integer(sc, truncated + 1));
	return(make_integer(sc, truncated));
      }
    case T_REAL:
      {
	s7_double z = real(x);
	if (is_NaN(z))
	  simple_out_of_range(sc, sc->round_symbol, x, its_nan_string);
	if (is_inf(z))
	  simple_out_of_range(sc, sc->round_symbol, x, its_infinite_string);
#if WITH_GMP
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  {
	    mpfr_set_d(sc->mpfr_1, z, MPFR_RNDN);
	    mpfr_rint(sc->mpfr_2, sc->mpfr_1, MPFR_RNDN); /* mpfr_roundeven in mpfr 4.0.0 */
	    mpfr_get_z(sc->mpz_3, sc->mpfr_2, MPFR_RNDN);
	    return(mpz_to_integer(sc, sc->mpz_3));
	  }
#else
	if (fabs(z) > DOUBLE_TO_INT64_LIMIT)
	  simple_out_of_range(sc, sc->round_symbol, x, its_too_large_string);
#endif
	return(make_integer(sc, (s7_int)r5rs_round(z)));
      }
#if WITH_GMP
      case T_BIG_INTEGER:
	return(x);
    case T_BIG_RATIO:
      {
	int32_t rnd;
	mpz_fdiv_qr(sc->mpz_1, sc->mpz_2, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
	mpz_mul_ui(sc->mpz_2, sc->mpz_2, 2);
	rnd = mpz_cmpabs(sc->mpz_2, mpq_denref(big_ratio(x)));
	mpz_fdiv_q(sc->mpz_2, sc->mpz_2, mpq_denref(big_ratio(x)));
	if (rnd > 0)
	  mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	else
	  if ((rnd == 0) &&
	      (mpz_odd_p(sc->mpz_1)))
	    mpz_add_ui(sc->mpz_1, sc->mpz_1, 1);
	return(mpz_to_integer(sc, sc->mpz_1));
      }
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	simple_out_of_range(sc, sc->round_symbol, x, its_nan_string);
      if (mpfr_inf_p(big_real(x)))
	simple_out_of_range(sc, sc->round_symbol, x, its_infinite_string);
      mpfr_set(sc->mpfr_1, big_real(x), MPFR_RNDN);
      mpfr_rint(sc->mpfr_2, sc->mpfr_1, MPFR_RNDN);
      mpfr_get_z(sc->mpz_3, sc->mpfr_2, MPFR_RNDN);
      return(mpz_to_integer(sc, sc->mpz_3));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      return(s7_wrong_type_arg_error(sc, "round", 0, x, "a real number"));
    default:
      return(method_or_bust_p(sc, x, sc->round_symbol, T_REAL));
    }
}

static s7_pointer g_round(s7_scheme *sc, s7_pointer args)
{
  #define H_round "(round x) returns the integer closest to x"
  #define Q_round s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(round_p_p(sc, car(args)));
}

static s7_int round_i_i(s7_int i) {return(i);}

#if (!WITH_GMP)
static s7_int round_i_7d(s7_scheme *sc, s7_double z)
{
  if (is_NaN(z))
    simple_out_of_range(sc, sc->round_symbol, real_NaN, its_nan_string);
  if ((is_inf(z)) ||
      (z > DOUBLE_TO_INT64_LIMIT) || (z < -DOUBLE_TO_INT64_LIMIT))
    simple_out_of_range(sc, sc->round_symbol, wrap_real(sc, z), its_too_large_string);
  return((s7_int)r5rs_round(z));
}
#endif


/* ---------------------------------------- add ---------------------------------------- */

static inline s7_pointer add_if_overflow_to_real_or_big_integer(s7_scheme *sc, s7_int x, s7_int y)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val;
  if (add_overflow(x, y, &val))
#if WITH_GMP
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_set_si(sc->mpz_2, y);
      mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      return(mpz_to_big_integer(sc, sc->mpz_1));
    }
#else
    {
      if (WITH_WARNINGS) s7_warn(sc, 128, "integer add overflow: (+ %" ld64 " %" ld64 ")\n", x, y);
      return(make_real(sc, (long_double)x + (long_double)y));
    }
#endif
  return(make_integer(sc, val));
#else
  return(make_integer(sc, x + y));
#endif
}

static s7_pointer integer_ratio_add_if_overflow_to_real_or_rational(s7_scheme *sc, s7_pointer x, s7_pointer y) /* x: int, y:ratio */
{
#if HAVE_OVERFLOW_CHECKS
  s7_int z;
  if ((multiply_overflow(integer(x), denominator(y), &z)) ||
      (add_overflow(z, numerator(y), &z)))
#if WITH_GMP
    {
      mpz_set_si(sc->mpz_1, integer(x));
      mpz_mul_si(sc->mpz_1, sc->mpz_1, denominator(y));
      mpz_set_si(sc->mpz_2, numerator(y));
      mpz_add(mpq_numref(sc->mpq_1), sc->mpz_2, sc->mpz_1);
      mpz_set_si(mpq_denref(sc->mpq_1), denominator(y));
      return(mpq_to_rational(sc, sc->mpq_1));
    }
#else
    {
      if (WITH_WARNINGS) s7_warn(sc, 128, "integer + ratio overflow: (+ %" ld64 " %" ld64 "/%" ld64 ")\n", integer(x), numerator(y), denominator(y));
      return(make_real(sc, (long_double)integer(x) + fraction(y)));
    }
#endif
    return(make_ratio(sc, z, denominator(y)));
#else
  return(make_ratio(sc, integer(x) * denominator(y) + numerator(y), denominator(y)));
#endif
}

#define parcel_out_fractions(X, Y) do {d1 = denominator(x); n1 = numerator(x); d2 = denominator(y); n2 = numerator(y);} while (0)
/* add_out_x|y here (as in lt_out_x|y) gives a small speed-up, say 3-7 callgrind units, about 2% */

static s7_pointer add_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* an experiment: try to avoid the switch statement */
  /* this wins in most s7 cases, not so much elsewhere? parallel subtract/multiply code is slower */
  if (is_t_integer(x))
    {
      if (is_t_integer(y))
	return(add_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));
    }
  else
    if (is_t_real(x))
      {
	if (is_t_real(y))
	  return(make_real(sc, real(x) + real(y)));
      }
    else
      if ((is_t_complex(x)) && (is_t_complex(y)))
	return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(add_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));
	case T_RATIO:
	  return(integer_ratio_add_if_overflow_to_real_or_rational(sc, x, y));
	case T_REAL:
#if WITH_GMP
	  if (s7_int_abs(integer(x)) >= INT64_TO_DOUBLE_LIMIT) /* (+ 9223372036854775807 .1), >= needed for (+ 9007199254740992 1.0) */
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), MPFR_RNDN);
	      mpfr_add_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (long_double)integer(x) + real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, (long_double)integer(x) + (long_double)real_part(y), imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpz_add(sc->mpz_1, sc->mpz_1, big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_add(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_add_si(sc->mpfr_1, big_real(y), integer(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_si(sc->mpc_1, integer(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer_ratio_add_if_overflow_to_real_or_rational(sc, y, x));
	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
	    if (d1 == d2)
	      {
#if HAVE_OVERFLOW_CHECKS
		s7_int q;
		if (add_overflow(n1, n2, &q))
#if WITH_GMP
		  {
		    mpq_set_si(sc->mpq_1, n1, d1);
		    mpq_set_si(sc->mpq_2, n2, d2);
		    mpq_add(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		    return(mpq_to_rational(sc, sc->mpq_1));
		  }
#else
		  {
 		    if (WITH_WARNINGS) s7_warn(sc, 128, "ratio + ratio overflow: (/ (+ %" ld64 " %" ld64 ") %" ld64 ")\n", n1, n2, d1);
		    return(make_real(sc, ((long_double)n1 + (long_double)n2) / (long_double)d1));
		  }
#endif
	        return(make_ratio_with_div_check(sc, sc->add_symbol, q, d1));
#else
		return(make_ratio_with_div_check(sc, sc->add_symbol, n1 + n2, d1));
#endif
	      }

#if HAVE_OVERFLOW_CHECKS
	    {
	      s7_int n1d2, n2d1, d1d2, q;
	      if ((multiply_overflow(d1, d2, &d1d2)) ||
		  (multiply_overflow(n1, d2, &n1d2)) ||
		  (multiply_overflow(n2, d1, &n2d1)) ||
		  (add_overflow(n1d2, n2d1, &q)))
#if WITH_GMP
		{
		  mpq_set_si(sc->mpq_1, n1, d1);
		  mpq_set_si(sc->mpq_2, n2, d2);
		  mpq_add(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		  return(mpq_to_rational(sc, sc->mpq_1));
		}
#else
	        {
 		  if (WITH_WARNINGS) s7_warn(sc, 128, "ratio + ratio overflow: (+ %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
	          return(make_real(sc, ((long_double)n1 / (long_double)d1) + ((long_double)n2 / (long_double)d2)));
		}
#endif
	      return(make_ratio_with_div_check(sc, sc->add_symbol, q, d1d2));
	    }
#else
	    return(make_ratio_with_div_check(sc, sc->add_symbol, n1 * d2 + n2 * d1, d1 * d2));
#endif
	  }
	case T_REAL:
	  return(make_real(sc, fraction(x) + real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, fraction(x) + real_part(y), imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_set_z(sc->mpq_2, big_integer(y));
	  mpq_add(sc->mpq_1, sc->mpq_1, sc->mpq_2);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_add(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_add_q(sc->mpfr_1, big_real(y), sc->mpq_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
#if WITH_GMP
	  if (s7_int_abs(integer(y)) >= INT64_TO_DOUBLE_LIMIT) /* (+ .1 9223372036854775807) */
	    {
	      mpfr_set_si(sc->mpfr_1, integer(y), MPFR_RNDN);
	      mpfr_add_d(sc->mpfr_1, sc->mpfr_1, real(x), MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, real(x) + (long_double)integer(y)));
	case T_RATIO:
	  return(make_real(sc, real(x) + fraction(y)));
	case T_REAL:
	  return(make_real(sc, real(x) + real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, real(x) + real_part(y), imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_add_z(sc->mpfr_1, sc->mpfr_1, big_integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_add_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_add_d(sc->mpfr_1, big_real(y), real(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_complex_not_0i(sc, real_part(x) + integer(y), imag_part(x)));
	case T_RATIO:
	  return(make_complex_not_0i(sc, real_part(x) + fraction(y), imag_part(x)));
	case T_REAL:
	  return(make_complex_not_0i(sc, real_part(x) + real(y), imag_part(x)));
	case T_COMPLEX:
	  return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_add_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(y));
	  mpz_add(sc->mpz_1, big_integer(x), sc->mpz_1);
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_add(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_set_d(sc->mpfr_1, real(y), MPFR_RNDN);
	  mpfr_add_z(sc->mpfr_1, sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpz_add(sc->mpz_1, big_integer(x), big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  mpq_add(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_add_z(sc->mpfr_1, big_real(y), big_integer(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_add(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_add(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_set_d(sc->mpfr_1, real(y), MPFR_RNDN);
	  mpfr_add_q(sc->mpfr_1, sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_add(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_add(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_add_q(sc->mpfr_1, big_real(y), big_ratio(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  mpfr_add_si(sc->mpfr_1, big_real(x), integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_add_q(sc->mpfr_1, big_real(x), sc->mpq_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_add_d(sc->mpfr_1, big_real(x), real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_add_fr(sc->mpc_1, sc->mpc_1, big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpfr_add_z(sc->mpfr_1, big_real(x), big_integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_add_q(sc->mpfr_1, big_real(x), big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_add(sc->mpfr_1, big_real(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_add_fr(sc->mpc_1, big_complex(y), big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  mpc_set_si(sc->mpc_1, integer(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(real_NaN); */
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_add_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_add(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}
#endif
      default:
	return(method_or_bust_with_type_pp(sc, x, sc->add_symbol, x, y, a_number_string, 1));
    }
}

static s7_pointer add_p_ppp(s7_scheme *sc, s7_pointer p0, s7_pointer p1, s7_pointer p2)
{
  if ((is_t_integer(p0)) && (is_t_integer(p1)) && (is_t_integer(p2)))
    {
#if HAVE_OVERFLOW_CHECKS
      s7_int val;
      if ((!add_overflow(integer(p0), integer(p1), &val)) &&
	  (!add_overflow(val, integer(p2), &val)))
	return(make_integer(sc, val));
#if WITH_GMP
      mpz_set_si(sc->mpz_1, integer(p0));
      mpz_set_si(sc->mpz_2, integer(p1));
      mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      mpz_set_si(sc->mpz_2, integer(p2));
      mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      return(mpz_to_integer(sc, sc->mpz_1));
#else
      if (WITH_WARNINGS) s7_warn(sc, 128, "integer add overflow: (+ %" ld64 " %" ld64 " %" ld64 ")\n", integer(p0), integer(p1), integer(p2));
      return(make_real(sc, (long_double)integer(p0) + (long_double)integer(p1) + (long_double)integer(p2)));
#endif
#else
      return(make_integer(sc, integer(p0) + integer(p1) + integer(p2)));
#endif
    }
  if ((is_t_real(p0)) && (is_t_real(p1)) && (is_t_real(p2)))
    return(make_real(sc, real(p0) + real(p1) + real(p2)));
  {
    s7_pointer p = add_p_pp(sc, p0, p1);
    sc->error_argnum = 1;
    p = add_p_pp(sc, p, p2);
    sc->error_argnum = 0;
    return(p);
  }
}

static s7_pointer g_add(s7_scheme *sc, s7_pointer args)
{
  #define H_add "(+ ...) adds its arguments"
  #define Q_add sc->pcl_n

  s7_pointer x, p;
  if (is_null(args))
    return(int_zero);
  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	return(method_or_bust_with_type_one_arg_p(sc, x, sc->add_symbol, a_number_string));
      return(x);
    }
  if (is_null(cdr(p)))
    return(add_p_pp(sc, x, car(p)));
  for (sc->error_argnum = 0; is_pair(p); p = cdr(p), sc->error_argnum++)
    x = add_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_add_2(s7_scheme *sc, s7_pointer args) {return(add_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_add_3(s7_scheme *sc, s7_pointer args) {return(add_p_ppp(sc, car(args), cadr(args), caddr(args)));}

static s7_pointer g_add_x1_1(s7_scheme *sc, s7_pointer x, int32_t pos)
{
  if (is_t_integer(x))
    return(add_if_overflow_to_real_or_big_integer(sc, integer(x), 1));

  switch (type(x))
    {
    case T_RATIO:   return(integer_ratio_add_if_overflow_to_real_or_rational(sc, int_one, x)); /* return(add_p_pp(sc, x, int_one)) */
    case T_REAL:    return(make_real(sc, real(x) + 1.0));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) + 1.0, imag_part(x)));
#if WITH_GMP
    case T_BIG_INTEGER:
      mpz_set_si(sc->mpz_1, 1);
      mpz_add(sc->mpz_1, big_integer(x), sc->mpz_1);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
    case T_BIG_REAL:
    case T_BIG_COMPLEX:
      return(add_p_pp(sc, x, int_one));
#endif
    default:
      return(method_or_bust_with_type(sc, x, sc->add_symbol,
				      (pos == 1) ? set_plist_2(sc, x, int_one) : set_plist_2(sc, int_one, x),
				      a_number_string, pos));
    }
  return(x);
}

#if WITH_GMP
static s7_pointer g_add_x1(s7_scheme *sc, s7_pointer args) {return(g_add_x1_1(sc, car(args), 1));}
#else
static s7_pointer g_add_x1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = car(args);
  if (is_t_integer(x)) return(add_if_overflow_to_real_or_big_integer(sc, integer(x), 1)); /* return(make_integer(sc, integer(x) + 1)); */
  if (is_t_real(x)) return(make_real(sc, real(x) + 1.0));
  if (is_t_complex(x)) return(make_complex_not_0i(sc, real_part(x) + 1.0, imag_part(x)));
  return(add_p_pp(sc, x, int_one));
}
#endif
static s7_pointer g_add_1x(s7_scheme *sc, s7_pointer args) {return(g_add_x1_1(sc, cadr(args), 2));}

static s7_pointer g_add_xi(s7_scheme *sc, s7_pointer x, s7_int y, int32_t loc)
{
  if (is_t_integer(x))
    return(add_if_overflow_to_real_or_big_integer(sc, integer(x), y));

  switch (type(x))
    {
    case T_RATIO:   return(add_p_pp(sc, x, wrap_integer(sc, y)));
    case T_REAL:    return(make_real(sc, real(x) + y));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) + y, imag_part(x)));
#if WITH_GMP
    case T_BIG_INTEGER:
      mpz_set_si(sc->mpz_1, y);
      mpz_add(sc->mpz_1, big_integer(x), sc->mpz_1);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
    case T_BIG_REAL:
    case T_BIG_COMPLEX:
      return(add_p_pp(sc, x, wrap_integer(sc, y)));
#endif
    default: return(method_or_bust_with_type_pi(sc, x, sc->add_symbol, x, y, a_number_string, loc));
    }
  return(x);
}

static s7_pointer add_p_pi(s7_scheme *sc, s7_pointer p1, s7_int i1) {return(g_add_xi(sc, p1, i1, 1));}

static s7_pointer g_add_xf(s7_scheme *sc, s7_pointer x, s7_double y, int32_t loc)
{
  if (is_t_real(x)) return(make_real(sc, real(x) + y));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) + y));
    case T_RATIO:   return(make_real(sc, fraction(x) + y));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) + y, imag_part(x)));
#if WITH_GMP
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(add_p_pp(sc, x, wrap_real(sc, y)));
#endif
    default: return(method_or_bust_with_type_pf(sc, x, sc->add_symbol, x, y, a_number_string, loc));
    }
  return(x);
}

static s7_pointer g_add_2_ff(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  if ((is_t_real(car(args))) && (is_t_real(cadr(args))))
    return(make_real(sc, real(car(args)) + real(cadr(args))));
  return(add_p_pp(sc, car(args), cadr(args)));
#else
  return(make_real(sc, real(car(args)) + real(cadr(args))));
#endif
}

static s7_pointer g_add_2_ii(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  if ((is_t_integer(car(args))) && (is_t_integer(cadr(args))))
#endif
    return(add_if_overflow_to_real_or_big_integer(sc, integer(car(args)), integer(cadr(args))));
#if WITH_GMP
  return(g_add(sc, args)); /* possibly bigint? */
#endif
}

#if WITH_GMP
static s7_pointer add_2_if(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if ((is_t_integer(x)) && (is_t_real(y)))
    {
      if (s7_int_abs(integer(x)) >= INT64_TO_DOUBLE_LIMIT)
	{
	  mpfr_set_si(sc->mpfr_1, integer(x), MPFR_RNDN);
	  mpfr_add_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      return(make_real(sc, integer(x) + real(y)));
    }
  return(add_p_pp(sc, x, y));
}

static s7_pointer g_add_2_if(s7_scheme *sc, s7_pointer args) {return(add_2_if(sc, car(args), cadr(args)));}
static s7_pointer g_add_2_fi(s7_scheme *sc, s7_pointer args) {return(add_2_if(sc, cadr(args), car(args)));}

static s7_pointer g_add_2_xi(s7_scheme *sc, s7_pointer args) {if (is_t_integer(cadr(args))) return(g_add_xi(sc, car(args), integer(cadr(args)), 1)); return(g_add(sc, args));}
static s7_pointer g_add_2_ix(s7_scheme *sc, s7_pointer args) {if (is_t_integer(car(args))) return(g_add_xi(sc, cadr(args), integer(car(args)), 2)); return(g_add(sc, args));}
static s7_pointer g_add_2_xf(s7_scheme *sc, s7_pointer args) {if (is_t_real(cadr(args))) return(g_add_xf(sc, car(args), real(cadr(args)), 1)); return(g_add(sc, args));}
static s7_pointer g_add_2_fx(s7_scheme *sc, s7_pointer args) {if (is_t_real(car(args))) return(g_add_xf(sc, cadr(args), real(car(args)), 2)); return(g_add(sc, args));}

#else

static s7_pointer g_add_2_if(s7_scheme *sc, s7_pointer args) {return(make_real(sc, integer(car(args)) + real(cadr(args))));}
static s7_pointer g_add_2_fi(s7_scheme *sc, s7_pointer args) {return(make_real(sc, real(car(args)) + integer(cadr(args))));}
static s7_pointer g_add_2_xi(s7_scheme *sc, s7_pointer args) {return(g_add_xi(sc, car(args), integer(cadr(args)), 1));}
static s7_pointer g_add_2_ix(s7_scheme *sc, s7_pointer args) {return(g_add_xi(sc, cadr(args), integer(car(args)), 2));}
static s7_pointer g_add_2_xf(s7_scheme *sc, s7_pointer args) {return(g_add_xf(sc, car(args), real(cadr(args)), 1));}
static s7_pointer g_add_2_fx(s7_scheme *sc, s7_pointer args) {return(g_add_xf(sc, cadr(args), real(car(args)), 2));}
#endif

static s7_pointer add_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 + x2));}
static s7_pointer add_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_integer(sc, x1 + x2));}

/* add_p_ii and add_d_id unhittable apparently -- this (d_id) is due to the order of d_dd_ok and d_id_ok in float_optimize,
 *   but d_dd is much more often hit, and the int arg (if constant) is turned into a float in d_dd
 */
static s7_double add_d_d(s7_double x) {return(x);}
static s7_double add_d_dd(s7_double x1, s7_double x2) {return(x1 + x2);}
static s7_double add_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 + x2 + x3);}
static s7_double add_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 + x2 + x3 + x4);}

static s7_int add_i_ii(s7_int i1, s7_int i2) {return(i1 + i2);}
static s7_int add_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 + i2 + i3);}

static s7_pointer argument_type(s7_scheme *sc, s7_pointer arg1)
{
  if (is_pair(arg1))
    {
      if (car(arg1) == sc->quote_symbol)
	return((is_pair(cdr(arg1))) ? s7_type_of(sc, cadr(arg1)) : NULL);    /* arg1 = (quote) */

      if ((is_h_optimized(arg1)) &&
	  (is_safe_c_op(optimize_op(arg1))) &&
	  (is_c_function(opt1_cfunc(arg1))))
	{
	  s7_pointer sig = c_function_signature(opt1_cfunc(arg1));
	  if ((sig) &&
	      (is_pair(sig)) &&
	      (is_symbol(car(sig))))
	    return(car(sig));
	}
      /* perhaps add closure sig if we can depend on it (immutable func etc) */
    }
  else
    if (!is_symbol(arg1))
      return(s7_type_of(sc, arg1));
  return(NULL);
}

static s7_pointer chooser_check_arg_types(s7_scheme *sc, s7_pointer arg1, s7_pointer arg2, s7_pointer fallback,
					  s7_pointer f_2_ff, s7_pointer f_2_ii, s7_pointer f_2_if, s7_pointer f_2_fi,
					  s7_pointer f_2_xi, s7_pointer f_2_ix, s7_pointer f_2_fx, s7_pointer f_2_xf)
{
  s7_pointer arg1_type = argument_type(sc, arg1);
  s7_pointer arg2_type = argument_type(sc, arg2);
  if ((arg1_type) || (arg2_type))
    {
      if (arg1_type == sc->is_float_symbol)
	{
	  if (arg2_type == sc->is_float_symbol)
	    return(f_2_ff);
	  return((arg2_type == sc->is_integer_symbol) ? f_2_fi : f_2_fx);
	}
      if (arg1_type == sc->is_integer_symbol)
	{
	  if (arg2_type == sc->is_float_symbol)
	    return(f_2_if);
	  return((arg2_type == sc->is_integer_symbol) ? f_2_ii : f_2_ix);
	}
      if (arg2_type == sc->is_float_symbol)
	return(f_2_xf);
      if (arg2_type == sc->is_integer_symbol)
	return(f_2_xi);
    }
  return(fallback);
}

static s7_pointer g_random_i(s7_scheme *sc, s7_pointer args);

static s7_pointer add_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  /* (+ s f) (+ (* s s) s) (+ s s) (+ s (* s s)) */
  if (args == 2)
    {
      if (ops)
	{
	  s7_pointer arg1 = cadr(expr), arg2 = caddr(expr);
	  if (arg2 == int_one)                          /* (+ ... 1) */
	    return(sc->add_x1);
	  if ((is_t_integer(arg1)) && ((is_pair(arg2)) && (is_optimized(arg2)) && (is_h_safe_c_nc(arg2)) && (fn_proc(arg2) == g_random_i)))
	    {
	      set_opt3_int(cdr(expr), integer(cadr(arg2)));
	      set_safe_optimize_op(expr, HOP_SAFE_C_NC); /* op if r op? */
	      return(sc->add_i_random);
	    }
	  if (arg1 == int_one)
	    return(sc->add_1x);
	  return(chooser_check_arg_types(sc, arg1, arg2, sc->add_2,
					 sc->add_2_ff, sc->add_2_ii, sc->add_2_if, sc->add_2_fi,
					 sc->add_2_xi, sc->add_2_ix, sc->add_2_fx, sc->add_2_xf));
	}
      return(sc->add_2);
    }
  return((args == 3) ? sc->add_3 : f);
}

/* ---------------------------------------- subtract ---------------------------------------- */

static s7_pointer negate_p_p(s7_scheme *sc, s7_pointer p)     /* can't use "negate" because it confuses C++! */
{
  switch (type(p))
    {
    case T_INTEGER:
      if (integer(p) == S7_INT64_MIN)
#if WITH_GMP
	{
	  mpz_set_si(sc->mpz_1, S7_INT64_MIN);
	  mpz_neg(sc->mpz_1, sc->mpz_1);
	  return(mpz_to_big_integer(sc, sc->mpz_1));
	}
#else
	simple_out_of_range(sc, sc->subtract_symbol, p, wrap_string(sc, "most-negative-fixnum can't be negated", 37));
#endif
      return(make_integer(sc, -integer(p)));

    case T_RATIO:   return(make_simple_ratio(sc, -numerator(p), denominator(p)));
    case T_REAL:    return(make_real(sc, -real(p)));
    case T_COMPLEX: return(make_complex_not_0i(sc, -real_part(p), -imag_part(p)));

#if WITH_GMP
    case T_BIG_INTEGER:
      mpz_neg(sc->mpz_1, big_integer(p));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
      mpq_neg(sc->mpq_1, big_ratio(p));
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    case T_BIG_REAL:
      mpfr_neg(sc->mpfr_1, big_real(p), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_neg(sc->mpc_1, big_complex(p), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, p, sc->subtract_symbol, a_number_string));
    }
}

static inline s7_pointer subtract_if_overflow_to_real_or_big_integer(s7_scheme *sc, s7_int x, s7_int y)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val;
  if (subtract_overflow(x, y, &val))
#if WITH_GMP
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_set_si(sc->mpz_2, y);
      mpz_sub(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      return(mpz_to_big_integer(sc, sc->mpz_1));
    }
#else
    {
      if (WITH_WARNINGS) s7_warn(sc, 128, "integer subtract overflow: (- %" ld64 " %" ld64 ")\n", x, y);
      return(make_real(sc, (long_double)x - (long_double)y));
    }
#endif
  return(make_integer(sc, val));
#else
  return(make_integer(sc, x - y));
#endif
}

static s7_pointer subtract_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0)
	return(negate_p_p(sc, y));
      switch (type(y))
	{
	case T_INTEGER:
	  return(subtract_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));

	case T_RATIO:
	  {
#if HAVE_OVERFLOW_CHECKS
	    s7_int z;
	    if ((multiply_overflow(integer(x), denominator(y), &z)) ||
		(subtract_overflow(z, numerator(y), &z)))
#if WITH_GMP
	      {
		mpz_set_si(sc->mpz_1, integer(x));
		mpz_mul_si(sc->mpz_1, sc->mpz_1, denominator(y));
		mpz_set_si(sc->mpz_2, numerator(y));
		mpz_sub(mpq_numref(sc->mpq_1), sc->mpz_1, sc->mpz_2);
		mpz_set_si(mpq_denref(sc->mpq_1), denominator(y));
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
	      {
		if (WITH_WARNINGS) s7_warn(sc, 128, "integer - ratio overflow: (- %" ld64 " %" ld64 "/%" ld64 ")\n", integer(x), numerator(y), denominator(y));
		return(make_real(sc, (long_double)integer(x) - fraction(y)));
	      }
#endif
	      return(make_ratio(sc, z, denominator(y)));
#else
	    return(make_ratio(sc, integer(x) * denominator(y) - numerator(y), denominator(y)));
#endif
	  }
	case T_REAL:
#if WITH_GMP
	  if (s7_int_abs(integer(x)) >= INT64_TO_DOUBLE_LIMIT) /* (- 9223372036854775807 .1) */
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), MPFR_RNDN);
	      mpfr_sub_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (long_double)integer(x) - real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, (long_double)integer(x) - real_part(y), -imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpz_sub(sc->mpz_1, sc->mpz_1, big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_sub(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_si_sub(sc->mpfr_1, integer(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_si(sc->mpc_1, integer(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  {
#if HAVE_OVERFLOW_CHECKS
	    s7_int z;
	    if ((multiply_overflow(integer(y), denominator(x), &z)) ||
		(subtract_overflow(numerator(x), z, &z)))
#if WITH_GMP
	      {
		mpz_set_si(sc->mpz_1, integer(y));
		mpz_mul_si(sc->mpz_1, sc->mpz_1, denominator(x));
		mpz_set_si(sc->mpz_2, numerator(x));
		mpz_sub(mpq_numref(sc->mpq_1), sc->mpz_2, sc->mpz_1);
		mpz_set_si(mpq_denref(sc->mpq_1), denominator(x));
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
	      {
		if (WITH_WARNINGS) s7_warn(sc, 128, "ratio - integer overflow: (- %" ld64 "/%" ld64 " %" ld64 ")\n", numerator(x), denominator(x), integer(y));
		return(make_real(sc, fraction(x) - (long_double)integer(y)));
	      }
#endif
	    return(make_ratio(sc, z, denominator(x)));
#else
	    return(make_ratio(sc, numerator(x) - (integer(y) * denominator(x)), denominator(x)));
#endif
	  }
	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
	    if (d1 == d2)
	      {
#if HAVE_OVERFLOW_CHECKS
		s7_int q;
		if (subtract_overflow(n1, n2, &q))
#if WITH_GMP
		  {
		    mpq_set_si(sc->mpq_1, n1, d1);
		    mpq_set_si(sc->mpq_2, n2, d2);
		    mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		    return(mpq_to_rational(sc, sc->mpq_1));
		  }
#else
		  {
		    if (WITH_WARNINGS) s7_warn(sc, 128, "ratio - ratio overflow: (- %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
		    return(make_real(sc, ((long_double)n1 - (long_double)n2) / (long_double)d1));
		  }
#endif
	        return(make_ratio_with_div_check(sc, sc->subtract_symbol, q, d1));
#else
		return(make_ratio(sc, numerator(x) - numerator(y), denominator(x)));
#endif
	      }

#if HAVE_OVERFLOW_CHECKS
	    {
	      s7_int n1d2, n2d1, d1d2, q;
	      if ((multiply_overflow(d1, d2, &d1d2)) ||
		  (multiply_overflow(n1, d2, &n1d2)) ||
		  (multiply_overflow(n2, d1, &n2d1)) ||
		  (subtract_overflow(n1d2, n2d1, &q)))
#if WITH_GMP
		{
		  mpq_set_si(sc->mpq_1, n1, d1);
		  mpq_set_si(sc->mpq_2, n2, d2);
		  mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		  return(mpq_to_rational(sc, sc->mpq_1));
		}
#else
	        {
		  if (WITH_WARNINGS) s7_warn(sc, 128, "ratio - ratio overflow: (- %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
		  return(make_real(sc, ((long_double)n1 / (long_double)d1) - ((long_double)n2 / (long_double)d2)));
		}
#endif
	      return(make_ratio_with_div_check(sc, sc->subtract_symbol, q, d1d2));
	    }
#else
	    return(make_ratio_with_div_check(sc, sc->subtract_symbol, n1 * d2 - n2 * d1, d1 * d2));
#endif
	  }
	case T_REAL:
	  return(make_real(sc, fraction(x) - real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, fraction(x) - real_part(y), -imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_set_z(sc->mpq_2, big_integer(y));
	  mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_sub(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_set_q(sc->mpfr_1, sc->mpq_1, MPFR_RNDN);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
#if WITH_GMP
	  if (s7_int_abs(integer(y)) >= INT64_TO_DOUBLE_LIMIT) /* (- .1 92233720368547758071) */
	    {
	      mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	      mpfr_sub_si(sc->mpfr_1, sc->mpfr_1, integer(y), MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, real(x) - (long_double)integer(y))); /* long_double saves (- 9007199254740996.0 9007199254740995): 1.0 */
	case T_RATIO:
	  return(make_real(sc, real(x) - fraction(y)));
	case T_REAL:
	  return(make_real(sc, real(x) - real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, real(x) - real_part(y), -imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_sub_z(sc->mpfr_1, sc->mpfr_1, big_integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_sub_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_d_sub(sc->mpfr_1, real(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_complex_not_0i(sc, real_part(x) - integer(y), imag_part(x)));
	case T_RATIO:
	  return(make_complex_not_0i(sc, real_part(x) - fraction(y), imag_part(x)));
	case T_REAL:
	  return(make_complex_not_0i(sc, real_part(x) - real(y), imag_part(x)));
	case T_COMPLEX:
	  return(make_complex(sc, real_part(x) - real_part(y), imag_part(x) - imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sub_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(y));
	  mpz_sub(sc->mpz_1, big_integer(x), sc->mpz_1);
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_sub(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  mpfr_sub_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpz_sub(sc->mpz_1, big_integer(x), big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  mpq_sub(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_sub(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_sub(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  mpfr_sub_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_sub(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_sub(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  mpfr_sub_si(sc->mpfr_1, big_real(x), integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_sub_q(sc->mpfr_1, big_real(x), sc->mpq_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_sub_d(sc->mpfr_1, big_real(x), real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_fr_sub(sc->mpc_1, big_real(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpfr_sub_z(sc->mpfr_1, big_real(x), big_integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_sub_q(sc->mpfr_1, big_real(x), big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_sub(sc->mpfr_1, big_real(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_fr_sub(sc->mpc_1, big_real(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  mpc_set_si(sc->mpc_2, integer(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(real_NaN); */
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_sub_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_sub(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}
#endif
      default:
	return(method_or_bust_with_type_pp(sc, x, sc->subtract_symbol, x, y, a_number_string, 1));
    }
}

static s7_pointer g_subtract(s7_scheme *sc, s7_pointer args)
{
  #define H_subtract "(- x1 ...) subtracts its trailing arguments from the first, or negates the first if only one it is given"
  #define Q_subtract sc->pcl_n

  s7_pointer x = car(args), p = cdr(args);
  if (is_null(p))
    return(negate_p_p(sc, x));
  for (sc->error_argnum = 0; is_pair(p); p = cdr(p), sc->error_argnum++)
    x = subtract_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_subtract_1(s7_scheme *sc, s7_pointer args) {return(negate_p_p(sc, car(args)));}
static s7_pointer g_subtract_2(s7_scheme *sc, s7_pointer args) {return(subtract_p_pp(sc, car(args), cadr(args)));}

static s7_pointer g_subtract_3(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = car(args);
  x = subtract_p_pp(sc, x, cadr(args));
  sc->error_argnum = 1;
  x = subtract_p_pp(sc, x, caddr(args));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer minus_c1(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER: return(subtract_if_overflow_to_real_or_big_integer(sc, integer(x), 1));
    case T_RATIO:   return(subtract_p_pp(sc, x, int_one));
    case T_REAL:    return(make_real(sc, real(x) - 1.0));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) - 1.0, imag_part(x)));
#if WITH_GMP
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, x, int_one));
#endif
    default:
      return(method_or_bust_with_type_pp(sc, x, sc->subtract_symbol, x, int_one, a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_x1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p = car(args);
#if WITH_GMP
  return(subtract_p_pp(sc, p, int_one));
#endif
  /* return((is_t_integer(p)) ? make_integer(sc, integer(p) - 1) : minus_c1(sc, p)); */
  return((is_t_integer(p)) ? subtract_if_overflow_to_real_or_big_integer(sc, integer(p), 1) : minus_c1(sc, p));
}

static s7_pointer g_subtract_2f(s7_scheme *sc, s7_pointer args) /* (- x f) */
{
  s7_pointer x = car(args);
  s7_double n = real(cadr(args)); /* checked below is_t_real */
  if (is_t_real(x)) return(make_real(sc, real(x) - n));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) - n));
    case T_RATIO:   return(make_real(sc, fraction(x) - n));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) - n, imag_part(x)));
#if WITH_GMP
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, x, cadr(args)));
#endif
    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_f2(s7_scheme *sc, s7_pointer args) /* (- f x) */
{
  s7_pointer x = cadr(args);
  s7_double n = real(car(args)); /* checked below is_t_real */

  if (is_t_real(x)) return(make_real(sc, n - real(x)));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, n - integer(x)));
    case T_RATIO:   return(make_real(sc, n - fraction(x)));
    case T_COMPLEX: return(make_complex_not_0i(sc, n - real_part(x), -imag_part(x)));
#if WITH_GMP
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, car(args), x));
#endif
    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1));
    }
  return(x);
}

static s7_int subtract_i_ii(s7_int i1, s7_int i2) {return(i1 - i2);}
static s7_int subtract_i_i(s7_int x) {return(-x);}
static s7_int subtract_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 - i2 - i3);}

static s7_double subtract_d_d(s7_double x) {return(-x);}
static s7_double subtract_d_dd(s7_double x1, s7_double x2) {return(x1 - x2);}
static s7_double subtract_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 - x2 - x3);}
static s7_double subtract_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 - x2 - x3 - x4);}

static s7_pointer subtract_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 - x2));}
static s7_pointer subtract_p_ii(s7_scheme *sc, s7_int i1, s7_int i2) {return(make_integer(sc, i1 - i2));}

static s7_pointer g_sub_xi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x))
    return(subtract_if_overflow_to_real_or_big_integer(sc, integer(x), y));

  switch (type(x))
    {
    case T_RATIO:   return(make_ratio(sc, numerator(x) - (y * denominator(x)), denominator(x)));
    case T_REAL:    return(make_real(sc, real(x) - y));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) - y, imag_part(x)));
#if WITH_GMP
    case T_BIG_INTEGER:
      mpz_set_si(sc->mpz_1, y);
      mpz_sub(sc->mpz_1, big_integer(x), sc->mpz_1);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
    case T_BIG_REAL:
    case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, x, wrap_integer(sc, y)));
#endif
    default: return(method_or_bust_with_type_pi(sc, x, sc->subtract_symbol, x, y, a_number_string, 1));
    }
  return(x);
}

static s7_pointer subtract_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 1)
    return(sc->subtract_1);
  if (args == 2)
    {
      if (ops)
	{
	  s7_pointer arg1 = cadr(expr), arg2 = caddr(expr);
	  if (arg2 == int_one) return(sc->subtract_x1);
	  if (is_t_real(arg1)) return(sc->subtract_f2);
	  if (is_t_real(arg2)) return(sc->subtract_2f);
	}
      return(sc->subtract_2);
    }
  return((args == 3) ? sc->subtract_3 : f);
}


/* ---------------------------------------- multiply ---------------------------------------- */

#define QUOTIENT_FLOAT_LIMIT 1e13
#define QUOTIENT_INT_LIMIT 10000000000000
/* fraction(x) is not accurate enough if it involves numbers over e18 even when done with long_doubles */

static inline s7_pointer multiply_if_overflow_to_real_or_big_integer(s7_scheme *sc, s7_int x, s7_int y)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val;
  if (multiply_overflow(x, y, &val))
#if WITH_GMP
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_mul_si(sc->mpz_1, sc->mpz_1, y);
      return(mpz_to_big_integer(sc, sc->mpz_1));
    }
#else
    {
      if (WITH_WARNINGS) s7_warn(sc, 128, "integer multiply overflow: (* %" ld64 " %" ld64 ")\n", x, y);
      return(make_real(sc, (s7_double)x * (s7_double)y));
    }
#endif
    return(make_integer(sc, val));
#else
  return(make_integer(sc, x * y));
#endif
}

static s7_pointer integer_ratio_multiply_if_overflow_to_real_or_ratio(s7_scheme *sc, s7_int x, s7_pointer y)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int z;
  if (multiply_overflow(x, numerator(y), &z))
#if WITH_GMP
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_mul_si(sc->mpz_1, sc->mpz_1, numerator(y));
      mpq_set_si(sc->mpq_1, 1, denominator(y));
      mpq_set_num(sc->mpq_1, sc->mpz_1);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    }
#else
    {
      if (WITH_WARNINGS) s7_warn(sc, 128, "integer * ratio overflow: (* %" ld64 " %" ld64 "/%" ld64 ")\n", x, numerator(y), denominator(y));
      return(make_real(sc, (s7_double)x * fraction(y)));
    }
#endif
    return(make_ratio(sc, z, denominator(y)));
#else
  return(make_ratio(sc, x * numerator(y), denominator(y)));
#endif
}

static s7_pointer multiply_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(multiply_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));
	case T_RATIO:
	  return(integer_ratio_multiply_if_overflow_to_real_or_ratio(sc, integer(x), y));
	case T_REAL:
#if WITH_GMP
	  if (s7_int_abs(integer(x)) > QUOTIENT_INT_LIMIT)
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), MPFR_RNDN);
	      mpfr_set_d(sc->mpfr_2, real(y), MPFR_RNDN);
	      mpfr_mul(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (long_double)integer(x) * real(y)));
	case T_COMPLEX:
	  return(s7_make_complex(sc, (long_double)integer(x) * real_part(y), (long_double)integer(x) * imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpz_mul_si(sc->mpz_1, big_integer(y), integer(x));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_mul(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_mul_si(sc->mpfr_1, big_real(y), integer(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_mul_si(sc->mpc_1, big_complex(y), integer(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* x might be 0 */
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer_ratio_multiply_if_overflow_to_real_or_ratio(sc, integer(y), x));
	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
#if HAVE_OVERFLOW_CHECKS
	    {
	      s7_int n1n2, d1d2;
	      if ((multiply_overflow(d1, d2, &d1d2)) ||
		  (multiply_overflow(n1, n2, &n1n2)))
#if WITH_GMP
		{
		  mpq_set_si(sc->mpq_1, n1, d1);
		  mpq_set_si(sc->mpq_2, n2, d2);
		  mpq_mul(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
		}
#else
	        {
		  if (WITH_WARNINGS) s7_warn(sc, 128, "ratio * ratio overflow: (* %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
		  return(make_real(sc, fraction(x) * fraction(y)));
		}
#endif
	      return(make_ratio_with_div_check(sc, sc->multiply_symbol, n1n2, d1d2));
	    }
#else
	    return(make_ratio_with_div_check(sc, sc->multiply_symbol, n1 * n2, d1 * d2));
#endif
	  }
	case T_REAL:
#if WITH_GMP
	  if (numerator(x) > QUOTIENT_INT_LIMIT)
	    {
	      mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	      mpfr_set_q(sc->mpfr_1, sc->mpq_1, MPFR_RNDN);
	      mpfr_mul_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, fraction(x) * real(y)));
	case T_COMPLEX:
	  return(s7_make_complex(sc, fraction(x) * real_part(y), fraction(x) * imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_set_z(sc->mpq_2, big_integer(y));
	  mpq_mul(sc->mpq_1, sc->mpq_1, sc->mpq_2);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_mul(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_mul_q(sc->mpfr_1, big_real(y), sc->mpq_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
#if WITH_GMP
	  if (s7_int_abs(integer(y)) > QUOTIENT_INT_LIMIT)
	    {
	      mpfr_set_si(sc->mpfr_1, integer(y), MPFR_RNDN);
	      mpfr_set_d(sc->mpfr_2, real(x), MPFR_RNDN);
	      mpfr_mul(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, real(x) * (long_double)integer(y)));
	case T_RATIO:
#if WITH_GMP
	  if (numerator(y) > QUOTIENT_INT_LIMIT)
	    {
	      mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	      mpfr_set_q(sc->mpfr_1, sc->mpq_1, MPFR_RNDN);
	      mpfr_mul_d(sc->mpfr_1, sc->mpfr_1, real(x), MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, fraction(y) * real(x)));
	case T_REAL:
	  return(make_real(sc, real(x) * real(y)));
	case T_COMPLEX:
	  return(make_complex(sc, real(x) * real_part(y), real(x) * imag_part(y)));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_mul_z(sc->mpfr_1, sc->mpfr_1, big_integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_mul_d(sc->mpfr_1, big_real(y), real(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* x might = 0.0 */
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_complex(sc, real_part(x) * integer(y), imag_part(x) * integer(y)));
	case T_RATIO:
	  return(s7_make_complex(sc, real_part(x) * fraction(y), imag_part(x) * fraction(y)));
	case T_REAL:
	  return(make_complex(sc, real_part(x) * real(y), imag_part(x) * real(y)));
	case T_COMPLEX:
	  {
	    s7_double r1 = real_part(x), r2 = real_part(y), i1 = imag_part(x), i2 = imag_part(y);
	    return(make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	  }
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_mul_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  mpz_mul_si(sc->mpz_1, big_integer(x), integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_mul(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_set_d(sc->mpfr_1, real(y), MPFR_RNDN);
	  mpfr_mul_z(sc->mpfr_1, sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpz_mul(sc->mpz_1, big_integer(x), big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  mpq_mul(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_mul_z(sc->mpfr_1, big_real(y), big_integer(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_mul(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_mul(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_set_d(sc->mpfr_1, real(y), MPFR_RNDN);
	  mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_mul(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_mul(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_mul_q(sc->mpfr_1, big_real(y), big_ratio(x), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  mpfr_mul_si(sc->mpfr_1, big_real(x), integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_mul_q(sc->mpfr_1, big_real(x), sc->mpq_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  mpfr_mul_d(sc->mpfr_1, big_real(x), real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_mul_fr(sc->mpc_1, sc->mpc_1, big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpfr_mul_z(sc->mpfr_1, big_real(x), big_integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_mul_q(sc->mpfr_1, big_real(x), big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_mul(sc->mpfr_1, big_real(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_mul_fr(sc->mpc_1, big_complex(y), big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* 0.0? */
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  mpc_mul_si(sc->mpc_1, big_complex(x), integer(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(real_NaN); */
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_mul_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_mul(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}
#endif
      default:
	return(method_or_bust_with_type_pp(sc, x, sc->multiply_symbol, x, y, a_number_string, 1));
    }
}

static s7_pointer multiply_p_ppp(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_pointer z)
{
  x = multiply_p_pp(sc, x, y);
  sc->error_argnum = 1;
  x = multiply_p_pp(sc, x, z);
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer multiply_method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer args, s7_pointer typ, int32_t num)
{
  if (has_active_methods(sc, obj))
    return(find_and_apply_method(sc, obj, sc->multiply_symbol, args));
  if (num == 0)
    simple_wrong_type_argument_with_type(sc, sc->multiply_symbol, obj, typ);
  wrong_type_argument_with_type(sc, sc->multiply_symbol, num, obj, typ);
  return(NULL);
}

static s7_pointer g_multiply(s7_scheme *sc, s7_pointer args)
{
  #define H_multiply "(* ...) multiplies its arguments"
  #define Q_multiply sc->pcl_n

  s7_pointer x, p;
  if (is_null(args))
    return(int_one);
  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	return(multiply_method_or_bust(sc, x, args, a_number_string, 0));
      return(x);
    }
  for (sc->error_argnum = 0; is_pair(p); p = cdr(p), sc->error_argnum++)
    x = multiply_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_multiply_2(s7_scheme *sc, s7_pointer args) {return(multiply_p_pp(sc, car(args), cadr(args)));}

static s7_pointer g_mul_xi(s7_scheme *sc, s7_pointer x, s7_int n, int32_t loc)
{
  switch (type(x))
    {
    case T_INTEGER: return(multiply_if_overflow_to_real_or_big_integer(sc, integer(x), n));
    case T_RATIO:   return(integer_ratio_multiply_if_overflow_to_real_or_ratio(sc, n, x));
    case T_REAL:    return(make_real(sc, real(x) * n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * n, imag_part(x) * n));
#if WITH_GMP
    case T_BIG_INTEGER:
      mpz_mul_si(sc->mpz_1, big_integer(x), n);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
    case T_BIG_REAL:
    case T_BIG_COMPLEX:
      return(multiply_p_pp(sc, x, wrap_integer(sc, n)));
#endif
    default:
      /* we can get here from mul_2_xi for example so the non-integer argument might not be a symbol */
      return(method_or_bust_with_type_pi(sc, x, sc->multiply_symbol, x, n, a_number_string, loc));
    }
  return(x);
}

static s7_pointer multiply_p_pi(s7_scheme *sc, s7_pointer p1, s7_int i1) {return(g_mul_xi(sc, p1, i1, 1));}

static s7_pointer g_mul_xf(s7_scheme *sc, s7_pointer x, s7_double y, int32_t num)
{
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) * y));
    case T_RATIO:   return(make_real(sc, numerator(x) * y / denominator(x)));
    case T_REAL:    return(make_real(sc, real(x) * y));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * y, imag_part(x) * y));
#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
      mpfr_mul_z(sc->mpfr_1, sc->mpfr_1, big_integer(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
      mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, big_ratio(x), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_mul_d(sc->mpfr_1, big_real(x), y, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
      mpc_mul_fr(sc->mpc_1, big_complex(x), sc->mpfr_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default: return(method_or_bust_with_type_pf(sc, x, sc->multiply_symbol, x, y, a_number_string, num));
    }
  return(x);
}

#if WITH_GMP
static s7_pointer g_mul_2_if(s7_scheme *sc, s7_pointer args) {
  if ((is_t_integer(car(args))) && (is_t_real(cadr(args))))
    return(make_real(sc, integer(car(args)) * real(cadr(args))));
  return(multiply_p_pp(sc, car(args), cadr(args)));
}

static s7_pointer g_mul_2_fi(s7_scheme *sc, s7_pointer args)
{
  if ((is_t_integer(cadr(args))) && (is_t_real(car(args))))
    return(make_real(sc, real(car(args)) * integer(cadr(args))));
  return(multiply_p_pp(sc, car(args), cadr(args)));
}

static s7_pointer g_mul_2_xi(s7_scheme *sc, s7_pointer args) {if (is_t_integer(cadr(args))) return(g_mul_xi(sc, car(args), integer(cadr(args)), 1)); return(g_multiply(sc, args));}
static s7_pointer g_mul_2_ix(s7_scheme *sc, s7_pointer args) {if (is_t_integer(car(args))) return(g_mul_xi(sc, cadr(args), integer(car(args)), 2)); return(g_multiply(sc, args));}
static s7_pointer g_mul_2_xf(s7_scheme *sc, s7_pointer args) {if (is_t_real(cadr(args))) return(g_mul_xf(sc, car(args), real(cadr(args)), 1)); return(g_multiply(sc, args));}
static s7_pointer g_mul_2_fx(s7_scheme *sc, s7_pointer args) {if (is_t_real(car(args))) return(g_mul_xf(sc, cadr(args), real(car(args)), 2)); return(g_multiply(sc, args));}
static s7_pointer g_mul_2_ff(s7_scheme *sc, s7_pointer args) {return(multiply_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_mul_2_ii(s7_scheme *sc, s7_pointer args) {return(multiply_p_pp(sc, car(args), cadr(args)));}
#else
static s7_pointer g_mul_2_if(s7_scheme *sc, s7_pointer args) {return(make_real(sc, integer(car(args)) * real(cadr(args))));}
static s7_pointer g_mul_2_fi(s7_scheme *sc, s7_pointer args) {return(make_real(sc, real(car(args)) * integer(cadr(args))));}
static s7_pointer g_mul_2_xi(s7_scheme *sc, s7_pointer args) {return(g_mul_xi(sc, car(args), integer(cadr(args)), 1));}
static s7_pointer g_mul_2_ix(s7_scheme *sc, s7_pointer args) {return(g_mul_xi(sc, cadr(args), integer(car(args)), 2));}
static s7_pointer g_mul_2_xf(s7_scheme *sc, s7_pointer args) {return(g_mul_xf(sc, car(args), real(cadr(args)), 1));} /* split out t_real is slower */
static s7_pointer g_mul_2_fx(s7_scheme *sc, s7_pointer args) {return(g_mul_xf(sc, cadr(args), real(car(args)), 2));}
static s7_pointer g_mul_2_ff(s7_scheme *sc, s7_pointer args) {return(make_real(sc, real(car(args)) * real(cadr(args))));}

static s7_pointer g_mul_2_ii(s7_scheme *sc, s7_pointer args)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val, x = integer(car(args)), y = integer(cadr(args));
  if (multiply_overflow(x, y, &val))
    {
      if (WITH_WARNINGS) s7_warn(sc, 128, "integer multiply overflow: (* %" ld64 " %" ld64 ")\n", x, y);
      return(make_real(sc, (s7_double)x * (s7_double)y));
    }
  return(make_integer(sc, val));
#else
  return(make_integer(sc, integer(car(args)) * integer(cadr(args))));
#endif
}
#endif

static s7_int multiply_i_ii(s7_int i1, s7_int i2)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val;
  if (multiply_overflow(i1, i2, &val))
    {
      if (WITH_WARNINGS) s7_warn(cur_sc, 64, "integer multiply overflow: (* %" ld64 " %" ld64 ")\n", i1, i2);
      return(S7_INT64_MAX); /* this is inconsistent with other unopt cases where an overflow -> double result */
    }
  /* (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (do ((j 0 (+ j 1))) ((= j 1)) (even? (* (ash 1 43) (ash 1 43)))))) (define (hi) (func)) (hi)) */
  return(val);
#else
  return(i1 * i2);
#endif
}

static s7_int multiply_i_iii(s7_int i1, s7_int i2, s7_int i3)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val1, val2;
  if ((multiply_overflow(i1, i2, &val1)) ||
      (multiply_overflow(val1, i3, &val2)))
    {
      if (WITH_WARNINGS) s7_warn(cur_sc, 64, "integer multiply overflow: (* %" ld64 " %" ld64 " %" ld64 ")\n", i1, i2, i3);
      return(S7_INT64_MAX);
    }
  return(val2);
#else
  return(i1 * i2 * i3);
#endif
}

static s7_double multiply_d_d(s7_double x) {return(x);}
static s7_double multiply_d_dd(s7_double x1, s7_double x2) {return(x1 * x2);}
static s7_double multiply_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 * x2 * x3);}
static s7_double multiply_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 * x2 * x3 * x4);}
static s7_pointer mul_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 * x2));}

static s7_pointer multiply_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 2)
    {
      if (ops)
	return(chooser_check_arg_types(sc, cadr(expr), caddr(expr), sc->multiply_2,
				       sc->mul_2_ff, sc->mul_2_ii, sc->mul_2_if, sc->mul_2_fi,
				       sc->mul_2_xi, sc->mul_2_ix, sc->mul_2_fx, sc->mul_2_xf));
      return(sc->multiply_2);
    }
  return(f);
}


/* ---------------------------------------- divide ---------------------------------------- */

static s7_pointer complex_invert(s7_scheme *sc, s7_pointer p)
{
  s7_double r2 = real_part(p), i2 = imag_part(p);
  s7_double den = (r2 * r2 + i2 * i2);
  /* here if p is, for example, -inf.0+i, den is +inf.0 so -i2/den is -0.0 (in gcc anyway), so the imag part is 0.0 */
  return(s7_make_complex(sc, r2 / den, -i2 / den));
}

static s7_pointer invert_p_p(s7_scheme *sc, s7_pointer p)
{
#if WITH_GMP
  s7_pointer x;
#endif
  switch (type(p))
    {
    case T_INTEGER:
#if WITH_GMP && (!POINTER_32)
      if (integer(p) == S7_INT64_MIN) /* (/ 1 (*s7* 'most-negative-fixnum)) -> -1/9223372036854775808 */
	{
	  new_cell(sc, x, T_BIG_RATIO);
	  big_ratio_bgr(x) = alloc_bigrat(sc);
	  add_big_ratio(sc, x);
	  mpz_set_si(sc->mpz_1, S7_INT64_MAX);
	  mpz_set_si(sc->mpz_2, 1);
	  mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	  mpq_set_si(big_ratio(x), -1, 1);
	  mpq_set_den(big_ratio(x), sc->mpz_1); /* geez... */
	  return(x);
	}
#endif
      if (integer(p) == 0)
	division_by_zero_error_1(sc, sc->divide_symbol, p);
      return(make_simple_ratio(sc, 1, integer(p)));  /* this checks for int */
    case T_RATIO:
      return(make_simple_ratio(sc, denominator(p), numerator(p)));
    case T_REAL:
      if (real(p) == 0.0)
	division_by_zero_error_1(sc, sc->divide_symbol, p);
      return(make_real(sc, 1.0 / real(p)));
    case T_COMPLEX:
      return(complex_invert(sc, p));

#if WITH_GMP
    case T_BIG_INTEGER:
      if (mpz_cmp_ui(big_integer(p), 0) == 0)
	division_by_zero_error_1(sc, sc->divide_symbol, p);
      if ((mpz_cmp_ui(big_integer(p), 1) == 0) || (mpz_cmp_si(big_integer(p), -1) == 0))
	return(p);
      new_cell(sc, x, T_BIG_RATIO);
      big_ratio_bgr(x) = alloc_bigrat(sc);
      add_big_ratio(sc, x);
      mpq_set_si(big_ratio(x), 1, 1);
      mpq_set_den(big_ratio(x), big_integer(p));
      mpq_canonicalize(big_ratio(x));
      return(x);

    case T_BIG_RATIO:
      if (mpz_cmp_ui(mpq_numref(big_ratio(p)), 1) == 0)
	return(mpz_to_integer(sc, mpq_denref(big_ratio(p))));
      if (mpz_cmp_si(mpq_numref(big_ratio(p)), -1) == 0)
	{
	  mpz_neg(sc->mpz_1, mpq_denref(big_ratio(p)));
	  return(mpz_to_integer(sc, sc->mpz_1));
	}
      new_cell(sc, x, T_BIG_RATIO);
      big_ratio_bgr(x) = alloc_bigrat(sc);
      add_big_ratio(sc, x);
      mpq_inv(big_ratio(x), big_ratio(p));
      mpq_canonicalize(big_ratio(x));
      return(x);

    case T_BIG_REAL:
      if (mpfr_zero_p(big_real(p)))
	division_by_zero_error_1(sc, sc->divide_symbol, p);
      x = mpfr_to_big_real(sc, big_real(p));
      mpfr_ui_div(big_real(x), 1, big_real(x), MPFR_RNDN);
      return(x);

    case T_BIG_COMPLEX:
      if ((!mpfr_number_p(mpc_realref(big_complex(p)))) || (!mpfr_number_p(mpc_imagref(big_complex(p)))))
	return(complex_NaN);
      mpc_ui_div(sc->mpc_1, 1, big_complex(p), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1)); /* x might be 0+0i if real-part is inf? */
#endif
    default:
      check_method(sc, p, sc->divide_symbol, set_plist_1(sc, p));
      wrong_type_argument_with_type(sc, sc->divide_symbol, 1, p, a_number_string);
    }
  return(NULL);
}

static s7_pointer divide_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* splitting out real/real here saves very little */
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	  /* -------- integer x -------- */
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  if (integer(x) == 1)  /* mainly to handle (/ 1 -9223372036854775808) correctly! */
	    return(invert_p_p(sc, y));
	  return(make_ratio(sc, integer(x), integer(y)));

	case T_RATIO:
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(integer(x), denominator(y), &dn))
#if WITH_GMP
	      {
		mpq_set_si(sc->mpq_1, integer(x), 1);
		mpq_set_si(sc->mpq_2, numerator(y), denominator(y));
		mpq_div(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	      }
#else
              {
		if (WITH_WARNINGS) s7_warn(sc, 128, "integer / ratio overflow: (/ %" ld64 " %" ld64 "/%" ld64 ")\n", integer(x), numerator(y), denominator(y));
  	        return(make_real(sc, integer(x) * inverted_fraction(y)));
	      }
#endif
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, dn, numerator(y)));
	  }
#else
	  return(make_ratio_with_div_check(sc, sc->divide_symbol, integer(x) * denominator(y), numerator(y)));
#endif

	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  if (is_inf(real(y))) return(real_zero);
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
#if WITH_GMP
	  if ((s7_int_abs(integer(x))) > QUOTIENT_INT_LIMIT)
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), MPFR_RNDN);
	      mpfr_set_d(sc->mpfr_2, real(y), MPFR_RNDN);
	      mpfr_div(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (s7_double)(integer(x)) / real(y)));

	case T_COMPLEX:
	  {
	    s7_double den, r1 = (s7_double)integer(x), r2 = real_part(y), i2 = imag_part(y);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    /* we could avoid the squaring (see Knuth II p613 16), not a big deal: (/ 1.0e308+1.0e308i 2.0e308+2.0e308i) => nan, (gmp case is ok here) */
	    return(s7_make_complex(sc, r1 * r2 * den, -(r1 * i2 * den)));
	  }

#if WITH_GMP
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_set_den(sc->mpq_1, big_integer(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_div(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_si_div(sc->mpfr_1, integer(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_si(sc->mpc_1, integer(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* x might be 0? */
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
      break;

      /* -------- ratio x -------- */
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(denominator(x), integer(y), &dn))
#if WITH_GMP
	      {
		mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
		mpq_set_si(sc->mpq_2, integer(y), 1);
		mpq_div(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
              {
		if (WITH_WARNINGS)
		  s7_warn(sc, 128, "ratio / integer overflow: (/ %" ld64 "/%" ld64 " %" ld64 ")\n", numerator(x), denominator(x), integer(y));
	        return(make_real(sc, (long_double)numerator(x) / ((long_double)denominator(x) * (long_double)integer(y))));
	      }
#endif
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, numerator(x), dn));
	  }
#else
	  return(make_ratio_with_div_check(sc, sc->divide_symbol, numerator(x), denominator(x) * integer(y)));
#endif

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
	    if (d1 == d2)
	      return(make_ratio_with_div_check(sc, sc->divide_symbol, n1, n2));
#if HAVE_OVERFLOW_CHECKS
	    if ((multiply_overflow(n1, d2, &n1)) ||
		(multiply_overflow(n2, d1, &d1)))
	      {
#if WITH_GMP
		mpq_set_si(sc->mpq_1, numerator(x), denominator(x)); /* not n1 and d1! they are garbage here */
		mpq_set_si(sc->mpq_2, n2, d2);
		mpq_div(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_rational(sc, sc->mpq_1));
#else
		s7_double r1, r2;
		if (WITH_WARNINGS)
		  s7_warn(sc, 128, "ratio / ratio overflow: (/ %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", numerator(x), denominator(x), numerator(y), denominator(y));
		r1 = fraction(x);
		r2 = inverted_fraction(y);
		return(make_real(sc, r1 * r2));
#endif
	      }
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, n1, d1));
#else
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, n1 * d2, n2 * d1));
#endif
	  }

	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  return(make_real(sc, fraction(x) / real(y)));

	case T_COMPLEX:
	  {
	    s7_double rx = fraction(x), r2 = real_part(y), i2 = imag_part(y);
	    s7_double den = 1.0 / (r2 * r2 + i2 * i2);
	    return(s7_make_complex(sc, rx * r2 * den, -rx * i2 * den)); /* not unchecked: (/ 3/4 -inf.0+i) */
	  }

#if WITH_GMP
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_set_si(sc->mpq_2, numerator(x), denominator(x));
	  mpq_div(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_div(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_set_q(sc->mpfr_1, sc->mpq_1, MPFR_RNDN);
	  mpfr_div(sc->mpfr_1, sc->mpfr_1, big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}

      /* -------- real x -------- */
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  if (is_NaN(real(x))) return(real_NaN); /* what is (/ +nan.0 0)? */
	  if (is_inf(real(x)))
	    return((real(x) > 0.0) ? ((integer(y) > 0) ? real_infinity : real_minus_infinity) : ((integer(y) > 0) ? real_minus_infinity : real_infinity));
	  return(make_real(sc, (long_double)real(x) / (long_double)integer(y)));

	case T_RATIO:
	  if (is_NaN(real(x))) return(real_NaN);
	  if (is_inf(real(x)))
	    return((real(x) > 0) ? ((numerator(y) > 0) ? real_infinity : real_minus_infinity) : ((numerator(y) > 0) ? real_minus_infinity : real_infinity));
	  return(make_real(sc, real(x) * inverted_fraction(y)));

	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  if (is_NaN(real(x))) return(real_NaN);
	  if (is_inf(real(y)))
	    return((is_inf(real(x))) ? real_NaN : real_zero);
	  return(make_real(sc, real(x) / real(y)));

	case T_COMPLEX:
	  {
	    s7_double den, r2, i2;
	    if (is_NaN(real(x))) return(complex_NaN);
	    r2 = real_part(y);
	    i2 = imag_part(y);
	    if ((is_NaN(r2)) || (is_inf(r2))) return(complex_NaN);
	    if ((is_NaN(i2)) || (is_inf(i2))) return(complex_NaN);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    return(s7_make_complex(sc, real(x) * r2 * den, -real(x) * i2 * den));
	  }

#if WITH_GMP
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_set_z(sc->mpfr_1, big_integer(y), MPFR_RNDN);
	  mpfr_d_div(sc->mpfr_1, real(x), sc->mpfr_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  mpfr_div_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_d_div(sc->mpfr_1, real(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((is_NaN(real(x))) || (!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}

      /* -------- complex x -------- */
    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  {
	    s7_double r1;
	    if (integer(y) == 0)
	      division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	    r1 = (long_double)1.0 / (long_double)integer(y);
	    return(s7_make_complex(sc, real_part(x) * r1, imag_part(x) * r1));
	  }

	case T_RATIO:
	  {
	    s7_double frac = inverted_fraction(y);
	    return(make_complex(sc, real_part(x) * frac, imag_part(x) * frac));
	  }

	case T_REAL:
	  {
	    s7_double r1;
	    if (real(y) == 0.0)
	      division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	    r1 = 1.0 / real(y);
	    return(make_complex(sc, real_part(x) * r1, imag_part(x) * r1)); /* (/ 0.0+1.0i +inf.0) */
	  }

	case T_COMPLEX:
	  {
	    s7_double r1 = real_part(x), r2, i1, i2, den;
	    if (is_NaN(r1)) return(real_NaN);
	    i1 = imag_part(x);
	    if (is_NaN(i1)) return(real_NaN);
	    r2 = real_part(y);
	    if (is_NaN(r2)) return(real_NaN);
	    if (is_inf(r2)) return(complex_NaN);
	    i2 = imag_part(y);
	    if (is_NaN(i2)) return(real_NaN);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    return(s7_make_complex(sc, (r1 * r2 + i1 * i2) * den, (r2 * i1 - r1 * i2) * den));
	  }

#if WITH_GMP
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_div_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpz_set_si(sc->mpz_1, integer(y));
	  mpq_set_num(sc->mpq_1, big_integer(x));
	  mpq_set_den(sc->mpq_1, sc->mpz_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y)); /* can't invert here, mpq den=unsigned */
	  mpq_div(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  mpfr_div_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y))) ||
	      (is_inf(real_part(y))) || (is_inf(imag_part(y))))
	    return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpq_set_num(sc->mpq_1, big_integer(x));
	  mpq_set_den(sc->mpq_1, big_integer(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, 0, 1);
	  mpq_set_num(sc->mpq_1, big_integer(x));
	  mpq_div(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  mpfr_div(sc->mpfr_1, sc->mpfr_1, big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_div(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_div(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  mpfr_div_d(sc->mpfr_1, sc->mpfr_1, real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y))) ||
	      (is_inf(real_part(y))) || (is_inf(imag_part(y))))
	    return(complex_NaN);
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_div(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_div(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  mpfr_div(sc->mpfr_1, sc->mpfr_1, big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_div_si(sc->mpfr_1, big_real(x), integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_div_q(sc->mpfr_1, big_real(x), sc->mpq_1, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_div_d(sc->mpfr_1, big_real(x), real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y))) ||
	      (is_inf(real_part(y))) || (is_inf(imag_part(y))))
	    return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_fr_div(sc->mpc_1, big_real(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_div_z(sc->mpfr_1, big_real(x), big_integer(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_div_q(sc->mpfr_1, big_real(x), big_ratio(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpfr_div(sc->mpfr_1, big_real(x), big_real(y), MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_fr_div(sc->mpc_1, big_real(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpc_set_si(sc->mpc_1, integer(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(real_NaN); */
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y))) ||
	      (is_inf(real_part(y))) || (is_inf(imag_part(y))))
	    return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2(sc, sc->divide_symbol, x, y);
	  mpc_div_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_div(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
#endif

    default: /* x is not a built-in number */
      return(method_or_bust_with_type_pp(sc, x, sc->divide_symbol, x, y, a_number_string, 1)); /* not args here! y = apply * to cdr(args) */
    }
  return(NULL); /* make the compiler happy */
}

static s7_pointer g_divide(s7_scheme *sc, s7_pointer args)
{
  #define H_divide "(/ x1 ...) divides its first argument by the rest, or inverts the first if there is only one argument"
  #define Q_divide sc->pcl_n

  s7_pointer x = car(args), p = cdr(args);
  if (is_null(p))            /* (/ x) */
    {
      if (!is_number(x))
	return(method_or_bust_with_type_one_arg_p(sc, x, sc->divide_symbol, a_number_string));
      return(invert_p_p(sc, x));
    }
  for (sc->error_argnum = 0; is_pair(p); p = cdr(p), sc->error_argnum++)
    x = divide_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_invert_1(s7_scheme *sc, s7_pointer args) {return(invert_p_p(sc, car(args)));}
static s7_pointer g_divide_2(s7_scheme *sc, s7_pointer args) {return(divide_p_pp(sc, car(args), cadr(args)));}

static s7_pointer g_divide_by_2(s7_scheme *sc, s7_pointer args)
{
  /* (/ x 2) */
  s7_pointer num = car(args);
  if (is_t_integer(num))
    {
      s7_int i = integer(num);
      if (i & 1)
	{
	  s7_pointer x;
	  new_cell(sc, x, T_RATIO);
	  numerator(x) = i;
	  denominator(x) = 2;
	  return(x);
	}
      return(make_integer(sc, i >> 1));
    }
  switch (type(num))
    {
    case T_RATIO:
#if HAVE_OVERFLOW_CHECKS
      {
	s7_int dn;
	if (multiply_overflow(denominator(num), 2, &dn))
	  {
	    if ((numerator(num) & 1) == 1)
#if WITH_GMP
	      {
		mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
		mpq_set_si(sc->mpq_2, 1, 2);
		mpq_mul(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
	      {
		if (WITH_WARNINGS) s7_warn(sc, 128, "ratio / 2 overflow: (/ %" ld64 "/%" ld64 " 2)\n", numerator(num), denominator(num));
	        return(make_real(sc, ((long_double)numerator(num) * 0.5) / (long_double)denominator(num)));
	      }
#endif
	    return(make_ratio(sc, numerator(num) / 2, denominator(num)));
	  }
	return(make_ratio_with_div_check(sc, sc->divide_symbol, numerator(num), dn));
      }
#else
      return(make_ratio(sc, numerator(num), denominator(num) * 2));
#endif

    case T_REAL:    return(make_real(sc, real(num) * 0.5));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(num) * 0.5, imag_part(num) * 0.5));

#if WITH_GMP
    case T_BIG_INTEGER:
      mpq_set_z(sc->mpq_1, big_integer(num));
      mpz_mul_ui(mpq_denref(sc->mpq_1), mpq_denref(sc->mpq_1), 2);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    case T_BIG_RATIO:
      mpq_set_si(sc->mpq_1, 2, 1);
      mpq_div(sc->mpq_1, big_ratio(num), sc->mpq_1);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    case T_BIG_REAL:
      mpfr_div_si(sc->mpfr_1, big_real(num), 2, MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_set_si(sc->mpc_1, 2, MPC_RNDNN);
      mpc_div(sc->mpc_1, big_complex(num), sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type_pp(sc, num, sc->divide_symbol, num, int_two, a_number_string, 1));
    }
}

static s7_pointer g_invert_x(s7_scheme *sc, s7_pointer args)
{
  /* (/ 1.0 x) */
  if (is_t_real(cadr(args)))
    {
      s7_double rl = real(cadr(args));
      if (rl == 0.0)
	division_by_zero_error_2(sc, sc->divide_symbol, car(args), cadr(args));
      return((is_NaN(rl)) ? real_NaN : make_real(sc, 1.0 / rl));
    }
  return(g_divide(sc, args));
}

static s7_double divide_d_7d(s7_scheme *sc, s7_double x)
{
  if (x == 0.0) division_by_zero_error_1(sc, sc->divide_symbol, real_zero);
  return(1.0 / x);
}

static s7_double divide_d_7dd(s7_scheme *sc, s7_double x1, s7_double x2)
{
  if (x2 == 0.0) division_by_zero_error_1(sc, sc->divide_symbol, real_zero);
  return(x1 / x2);
}

static s7_pointer divide_p_ii(s7_scheme *sc, s7_int x, s7_int y) {return(make_ratio_with_div_check(sc, sc->divide_symbol, x, y));}
static s7_pointer divide_p_i(s7_scheme *sc, s7_int x) {return(make_ratio_with_div_check(sc, sc->divide_symbol, 1, x));}

static s7_pointer divide_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 1)
    return(sc->invert_1);
  if ((ops) && (args == 2))
    {
      s7_pointer arg1 = cadr(expr);
      if ((is_t_real(arg1)) && (real(arg1) == 1.0))
	return(sc->invert_x);
      return(((is_t_integer(caddr(expr))) && (integer(caddr(expr)) == 2)) ? sc->divide_by_2 : sc->divide_2);
    }
  return(f);
}


/* -------------------------------- quotient -------------------------------- */
static inline s7_int quotient_i_7ii(s7_scheme *sc, s7_int x, s7_int y)
{
  if ((y > 0) || (y < -1)) return(x / y);
  if (y == 0)
    division_by_zero_error_2(sc, sc->quotient_symbol, wrap_integer(sc, x), int_zero);
  if ((y == -1) && (x == S7_INT64_MIN))   /* (quotient most-negative-fixnum -1) */
    simple_out_of_range(sc, sc->quotient_symbol, set_elist_2(sc, leastfix, minus_one), its_too_large_string);
  return(x / y);
}

#if (!WITH_GMP)
static s7_pointer s7_truncate(s7_scheme *sc, s7_pointer caller, s7_double xf)   /* can't use "truncate" -- it's in unistd.h */
{
  if (fabs(xf) > QUOTIENT_FLOAT_LIMIT)
    simple_out_of_range(sc, caller, wrap_real(sc, xf), its_too_large_string);
  return((xf > 0.0) ? make_integer(sc, (s7_int)floor(xf)) : make_integer(sc, (s7_int)ceil(xf)));
}

static s7_int c_quo_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_double xf;
  if (y == 0.0)
    division_by_zero_error_2(sc, sc->quotient_symbol, wrap_real(sc, x), real_zero);
  if ((is_inf(y)) || (is_NaN(y))) /* here we can't return NAN so I guess we should signal an error */
    wrong_type_argument_with_type(sc, sc->quotient_symbol, 2, wrap_real(sc, y), a_normal_real_string);
  xf = x / y;
  if (fabs(xf) > QUOTIENT_FLOAT_LIMIT)
    simple_out_of_range(sc, sc->quotient_symbol, wrap_real(sc, xf), its_too_large_string);
  return((xf > 0.0) ? (s7_int)floor(xf) : (s7_int)ceil(xf));
}
#endif

static s7_int quotient_i_ii_unchecked(s7_int i1, s7_int i2) {return(i1 / i2);} /* i2 > 0 */

static s7_pointer quotient_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if WITH_GMP
  if ((is_real(x)) && (is_real(y)))
    {
      if (is_zero(y))
	division_by_zero_error_2(sc, sc->quotient_symbol, x, y);
      if ((s7_is_integer(x)) && (s7_is_integer(y)))
	{
	  if (is_t_integer(x)) mpz_set_si(sc->mpz_1, integer(x)); else mpz_set(sc->mpz_1, big_integer(x));
	  if (is_t_integer(y)) mpz_set_si(sc->mpz_2, integer(y)); else mpz_set(sc->mpz_2, big_integer(y));
	  mpz_tdiv_q(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	}
      else
	if ((!is_rational(x)) || (!is_rational(y)))
	  {
	    if (any_real_to_mpfr(sc, x, sc->mpfr_1)) return(real_NaN);
	    if (any_real_to_mpfr(sc, y, sc->mpfr_2)) return(real_NaN);
	    mpfr_div(sc->mpfr_3, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_3, MPFR_RNDZ);
	  }
	else
	  {
	    any_rational_to_mpq(sc, x, sc->mpq_1);
	    any_rational_to_mpq(sc, y, sc->mpq_2);
	    mpq_div(sc->mpq_3, sc->mpq_1, sc->mpq_2);
	    mpz_tdiv_q(sc->mpz_1, mpq_numref(sc->mpq_3), mpq_denref(sc->mpq_3));
	  }
      return(mpz_to_integer(sc, sc->mpz_1));
    }
  return(method_or_bust_pp(sc, (is_real(x)) ? y : x, sc->quotient_symbol, x, y, T_REAL, (is_real(x)) ? 2 : 1));
#else

  s7_int d1, d2, n1, n2;
  if ((is_t_integer(x)) && (is_t_integer(y)))
    return(make_integer(sc, quotient_i_7ii(sc, integer(x), integer(y))));

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, quotient_i_7ii(sc, integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  /* (quotient -9223372036854775808 -1/9223372036854775807): arithmetic exception in the no-overflow-checks case */
	  goto RATIO_QUO_RATIO;

	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->quotient_symbol, x, y);
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(real_NaN);
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)integer(x) / real(y))); /* s7_truncate returns an integer */

	default:
	  return(method_or_bust_pp(sc, y, sc->quotient_symbol, x, y, T_REAL, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->quotient_symbol, x, y);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);
	  d2 = 1;
	  goto RATIO_QUO_RATIO;
	  /* this can lose:
	   *   (quotient 1 2305843009213693952/4611686018427387903) -> 2, not 1
	   *   (quotient 21053343141/6701487259 3587785776203/1142027682075) -> 1, not 0
	   */

	case T_RATIO:
	  parcel_out_fractions(x, y);
	RATIO_QUO_RATIO:
	  if (d1 == d2)
	    return(make_integer(sc, n1 / n2));              /* (quotient 3/9223372036854775807 1/9223372036854775807) */
	  if (n1 == n2)
	    return(make_integer(sc, d2 / d1));              /* (quotient 9223372036854775807/2 9223372036854775807/8) */
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int n1d2, n2d1;
	    if ((multiply_overflow(n1, d2, &n1d2)) ||
		(multiply_overflow(n2, d1, &n2d1)))
	      return(s7_truncate(sc, sc->quotient_symbol, ((long_double)n1 / (long_double)n2) * ((long_double)d2 / (long_double)d1)));
	    return(make_integer(sc, n1d2 / n2d1));
	  }
#else
	  return(make_integer(sc, (n1 * d2) / (n2 * d1)));
#endif

	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->quotient_symbol, x, y);
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(real_NaN);
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)fraction(x) / real(y)));

	default:
	  return(method_or_bust_pp(sc, y, sc->quotient_symbol, x, y, T_REAL, 2));
	}

    case T_REAL:
      if (((is_inf(real(x))) || (is_NaN(real(x)))) && (is_real(y)))
	return(real_NaN);
      /* if infs allowed we need to return infs/nans, else:
       *    (quotient inf.0 1e-309) -> -9223372036854775808
       *    (quotient inf.0 inf.0) -> -9223372036854775808
       */

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->quotient_symbol, x, y);
	  return(s7_truncate(sc, sc->quotient_symbol, (long_double)real(x) / (long_double)integer(y)));

	case T_RATIO: return(s7_truncate(sc, sc->quotient_symbol, real(x) / (s7_double)fraction(y)));
	case T_REAL:  return(make_integer(sc, c_quo_dbl(sc, real(x), real(y)))); /* c_quo_dbl returns an integer */
	default:      return(method_or_bust_pp(sc, y, sc->quotient_symbol, x, y, T_REAL, 2));
	}

    default:
      return(method_or_bust_pp(sc, x, sc->quotient_symbol, x, y, T_REAL, 2));
    }
#endif
}

static s7_pointer quotient_p_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if ((is_t_integer(x)) && ((y > 0) || (y < -1))) return(make_integer(sc, integer(x) / y));
  return(quotient_p_pp(sc, x, wrap_integer(sc, y)));
}

static s7_pointer g_quotient(s7_scheme *sc, s7_pointer args)
{
  #define H_quotient "(quotient x1 x2) returns the integer quotient of x1 and x2; (quotient 4 3) = 1"
  #define Q_quotient sc->pcl_r
  /* sig was '(integer? ...) but quotient can return NaN */
  /* (define (quo x1 x2) (truncate (/ x1 x2))) ; slib */
  return(quotient_p_pp(sc, car(args), cadr(args)));
}


/* -------------------------------- remainder -------------------------------- */
#if WITH_GMP
static s7_pointer big_mod_or_rem(s7_scheme *sc, s7_pointer x, s7_pointer y, bool use_floor)
{
  if ((is_real(x)) && (is_real(y)))
    {
      if ((s7_is_integer(x)) && (s7_is_integer(y)))
	{
	  if (is_t_integer(x)) mpz_set_si(sc->mpz_1, integer(x)); else mpz_set(sc->mpz_1, big_integer(x));
	  if (is_t_integer(y)) mpz_set_si(sc->mpz_2, integer(y)); else mpz_set(sc->mpz_2, big_integer(y));
	  if (use_floor)
	    mpz_fdiv_q(sc->mpz_3, sc->mpz_1, sc->mpz_2);
	  else mpz_tdiv_q(sc->mpz_3, sc->mpz_1, sc->mpz_2);
	  mpz_mul(sc->mpz_3, sc->mpz_3, sc->mpz_2);
	  mpz_sub(sc->mpz_1, sc->mpz_1, sc->mpz_3);
	  return(mpz_to_integer(sc, sc->mpz_1));
	}
      if ((!is_rational(x)) || (!is_rational(y)))
	{
	  any_real_to_mpfr(sc, x, sc->mpfr_1);
	  any_real_to_mpfr(sc, y, sc->mpfr_2);
	  mpfr_div(sc->mpfr_3, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
	  if (use_floor)
	    mpfr_get_z(sc->mpz_1, sc->mpfr_3, MPFR_RNDD);
	  else mpfr_get_z(sc->mpz_1, sc->mpfr_3, MPFR_RNDZ);
	  mpfr_mul_z(sc->mpfr_2, sc->mpfr_2, sc->mpz_1, MPFR_RNDN);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, MPFR_RNDN);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      any_rational_to_mpq(sc, x, sc->mpq_1);
      any_rational_to_mpq(sc, y, sc->mpq_2);
      mpq_div(sc->mpq_3, sc->mpq_1, sc->mpq_2);
      if (use_floor)
	mpz_fdiv_q(sc->mpz_1, mpq_numref(sc->mpq_3), mpq_denref(sc->mpq_3));
      else mpz_tdiv_q(sc->mpz_1, mpq_numref(sc->mpq_3), mpq_denref(sc->mpq_3));
      mpz_mul(mpq_numref(sc->mpq_2), sc->mpz_1, mpq_numref(sc->mpq_2));
      mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    }
  return(method_or_bust_pp(sc, (is_real(x)) ? y : x, (use_floor) ? sc->modulo_symbol : sc->remainder_symbol, x, y, T_REAL, (is_real(x)) ? 2 : 1));
}
#endif

#define REMAINDER_FLOAT_LIMIT 1e13

static inline s7_int remainder_i_7ii(s7_scheme *sc, s7_int x, s7_int y)
{
  if ((y > 1) || (y < -1)) return(x % y); /* avoid floating exception if (remainder -9223372036854775808 -1)! */
  if (y == 0)
    division_by_zero_error_2(sc, sc->remainder_symbol, wrap_integer(sc, x), int_zero);
  return(0);
}

static s7_double c_rem_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_int quo;
  s7_double pre_quo;
  if ((is_inf(y)) || (is_NaN(y)))
    return(NAN);
  pre_quo = x / y;
  if (fabs(pre_quo) > REMAINDER_FLOAT_LIMIT)
    simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, wrap_real(sc, x), wrap_real(sc, y)), its_too_large_string);
  quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
  return(x - (y * quo));
}

static s7_int remainder_i_ii_unchecked(s7_int i1, s7_int i2) {return(i1 % i2);} /* i2 > 1 */
static s7_double remainder_d_7dd(s7_scheme *sc, s7_double x1, s7_double x2)
{
  if (x2 == 0.0)
    division_by_zero_error_2(sc, sc->remainder_symbol, wrap_real(sc, x1), real_zero);
  if ((is_inf(x1)) || (is_NaN(x1))) /* match remainder_p_pp */
    return(NAN);
  return(c_rem_dbl(sc, x1, x2));
}

static s7_pointer remainder_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if WITH_GMP
  if (is_zero(y))
    division_by_zero_error_2(sc, sc->remainder_symbol, x, y);
  return(big_mod_or_rem(sc, x, y, false));
#else
  s7_int quo, d1, d2, n1, n2;
  s7_double pre_quo;

  if ((is_t_integer(x)) && (is_t_integer(y)))
    return(make_integer(sc, remainder_i_7ii(sc, integer(x), integer(y))));

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, remainder_i_7ii(sc, integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  goto RATIO_REM_RATIO;

	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->remainder_symbol, x, y);
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(real_NaN);
	  pre_quo = (long_double)integer(x) / (long_double)real(y);
	  if (fabs(pre_quo) > REMAINDER_FLOAT_LIMIT)
	    simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, x, y), its_too_large_string);
	  quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	  return(make_real(sc, integer(x) - real(y) * quo));

	default:
	  return(method_or_bust_pp(sc, y, sc->remainder_symbol, x, y, T_REAL, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  n2 = integer(y);
 	  if (n2 == 0)
 	    division_by_zero_error_2(sc, sc->remainder_symbol, x, y);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  d2 = 1;
	  goto RATIO_REM_RATIO;

	case T_RATIO:
	  parcel_out_fractions(x, y);
	RATIO_REM_RATIO:
	  if (d1 == d2)
	    quo = (s7_int)(n1 / n2);
	  else
	    {
	      if (n1 == n2)
		quo = (s7_int)(d2 / d1);
	      else
		{
#if HAVE_OVERFLOW_CHECKS
		  s7_int n1d2, n2d1;
		  if ((multiply_overflow(n1, d2, &n1d2)) ||
		      (multiply_overflow(n2, d1, &n2d1)))
		    {
		      pre_quo = ((long_double)n1 / (long_double)n2) * ((long_double)d2 / (long_double)d1);
		      if (fabs(pre_quo) > REMAINDER_FLOAT_LIMIT)
			simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, x, y), its_too_large_string);
		      quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
		    }
		  else quo = n1d2 / n2d1;
#else
		  quo = (n1 * d2) / (n2 * d1);
#endif
		}}
	  if (quo == 0)
	    return(x);

#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn, nq;
	    if (!multiply_overflow(n2, quo, &nq))
	      {
		if ((d1 == d2) &&
		    (!subtract_overflow(n1, nq, &dn)))
		  return(make_ratio_with_div_check(sc, sc->remainder_symbol, dn, d1));

		if ((!multiply_overflow(n1, d2, &dn)) &&
		    (!multiply_overflow(nq, d1, &nq)) &&
		    (!subtract_overflow(dn, nq, &nq)) &&
		    (!multiply_overflow(d1, d2, &d1)))
		  return(make_ratio_with_div_check(sc, sc->remainder_symbol, nq, d1));
	      }}
#else
	  if (d1 == d2)
	    return(make_ratio_with_div_check(sc, sc->remainder_symbol, n1 - n2 * quo, d1));

	  return(make_ratio_with_div_check(sc, sc->remainder_symbol, n1 * d2 - n2 * d1 * quo, d1 * d2));
#endif
	  simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, x, y), intermediate_too_large_string);

	case T_REAL:
	  {
	    s7_double frac;
	    if (real(y) == 0.0)
	      division_by_zero_error_2(sc, sc->remainder_symbol, x, y);
	    if ((is_inf(real(y))) || (is_NaN(real(y))))
	      return(real_NaN);
	    if (s7_int_abs(numerator(x)) > QUOTIENT_INT_LIMIT)
	      return(subtract_p_pp(sc, x, multiply_p_pp(sc, y, quotient_p_pp(sc, x, y))));
	    frac = (s7_double)fraction(x);
	    pre_quo = frac / real(y);
	    if (fabs(pre_quo) > REMAINDER_FLOAT_LIMIT)
	      simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, x, y), its_too_large_string);
	    quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	    return(make_real(sc, frac - real(y) * quo));
	  }

	default:
	  return(method_or_bust_pp(sc, y, sc->remainder_symbol, x, y, T_REAL, 2));
	}

    case T_REAL:
      if (((is_inf(real(x))) || (is_NaN(real(x)))) && (is_real(y)))
	{
	  if (is_zero(y))
	    division_by_zero_error_2(sc, sc->remainder_symbol, x, y);
	  return(real_NaN);
	}
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2(sc, sc->remainder_symbol, x, y);
	  /* actually here (and elsewhere) if y > INT64_TO_DOUBLE_LIMIT, the result is probably wrong */
	  pre_quo = (long_double)real(x) / (long_double)integer(y);
	  if (fabs(pre_quo) > REMAINDER_FLOAT_LIMIT)
	    simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, x, y), its_too_large_string);
	  quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	  return(make_real(sc, real(x) - integer(y) * quo));
	  /* but... (remainder 1e+18 9223372036854775807) -> 1e+18 */

	case T_RATIO:
	  if (s7_int_abs(numerator(y)) > QUOTIENT_INT_LIMIT)
	    return(subtract_p_pp(sc, x, multiply_p_pp(sc, y, quotient_p_pp(sc, x, y))));
	  {
	    s7_double frac = (s7_double)fraction(y);
	    pre_quo = real(x) / frac;
	    if (fabs(pre_quo) > REMAINDER_FLOAT_LIMIT)
	      simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, x, y), its_too_large_string);
	    quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	    return(make_real(sc, real(x) - frac * quo));
	  }

	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2(sc, sc->remainder_symbol, x, y);
	  return(make_real(sc, c_rem_dbl(sc, real(x), real(y))));
	  /* see under sin -- this calculation is completely bogus if "a" is large
	   * (quotient 1e22 (* 2 pi)) -> -9223372036854775808 but it should be 1591549430918953357688,
	   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22 -- the "remainder" is greater than the original argument!
	   * Clisp gives 0.0 here, as does sbcl, currently s7 throws an error (out-of-range).
	   */

	default:
	  return(method_or_bust_pp(sc, y, sc->remainder_symbol, x, y, T_REAL, 2));
	}

    default:
      return(method_or_bust_pp(sc, x, sc->remainder_symbol, x, y, T_REAL, 1));
    }
#endif
}

static s7_pointer remainder_p_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if ((is_t_integer(x)) && ((y > 1) || (y < -1))) return(make_integer(sc, integer(x) % y));
  return(remainder_p_pp(sc, x, wrap_integer(sc, y)));
}

static s7_pointer g_remainder(s7_scheme *sc, s7_pointer args)
{
  #define H_remainder "(remainder x1 x2) returns the remainder of x1/x2; (remainder 10 3) = 1"
  #define Q_remainder sc->pcl_r
  /* (define (rem x1 x2) (- x1 (* x2 (quo x1 x2)))) ; slib, if x2 is an integer (- x1 (truncate x1 x2)), fractional part: (remainder x 1) */

  s7_pointer x = car(args), y = cadr(args);
  if ((is_t_integer(x)) && (is_t_integer(y)))
    return(make_integer(sc, remainder_i_7ii(sc, integer(x), integer(y))));
  return(remainder_p_pp(sc, x, y));
}


/* -------------------------------- modulo -------------------------------- */
static s7_int modulo_i_ii(s7_int x, s7_int y)
{
  s7_int z;
  if (y > 1)
    {
      z = x % y;
      return((z >= 0) ? z : z + y);
    }
  if (y < -1)
    {
      z = x % y;
      return((z > 0) ? z + y : z);
    }
  if (y == 0) return(x);     /* else arithmetic exception */
  return(0);
}

static s7_int modulo_i_ii_unchecked(s7_int i1, s7_int i2) /* here we know i2 > 1 */
{
  s7_int z = i1 % i2;
  return((z < 0) ? (z + i2) : z);
}

static s7_double modulo_d_7dd(s7_scheme *sc, s7_double x1, s7_double x2)
{
  s7_double c;
  if ((is_NaN(x1)) || (is_NaN(x2)) || (is_inf(x1)) || (is_inf(x2))) return(NAN);
  if (x2 == 0.0) return(x1);
  if (fabs(x1) > 1e17)
    out_of_range(sc, sc->modulo_symbol, int_one, wrap_real(sc, x1), its_too_large_string);
  c = x1 / x2;
  if ((c > 1e19) || (c < -1e19))
    simple_out_of_range(sc, sc->modulo_symbol,
			set_elist_3(sc, sc->divide_symbol, wrap_real(sc, x1), wrap_real(sc, x2)),
			intermediate_too_large_string);
  return(x1 - x2 * (s7_int)floor(c));
}

static s7_pointer modulo_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if WITH_GMP
  /* as tricky as expt, so just use bignums; mpz_mod|_ui = mpz_fdiv_r_ui, but sign ignored -- probably not worth the code
   *   originally   subtract_p_pp(sc, x, multiply_p_pp(sc, y, floor_p_p(sc, divide_p_pp(sc, x, y))))
   *   quotient is                                            truncate_p_p(sc, divide_p_pp(sc, x, y))
   *   remainder is subtract_p_pp(sc, x, multiply_p_pp(sc, y, quotient_p_pp(sc, x, y)))
   */
  if (!is_zero(y)) return(big_mod_or_rem(sc, x, y, true));
  if (is_real(x)) return(x);
  return(method_or_bust_pp(sc, x, sc->modulo_symbol, x, y, T_REAL, 1));
#else
  s7_double a, b;
  s7_int n1, n2, d1, d2;

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, modulo_i_ii(integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  if ((n1 == n2) && (d1 > d2)) return(x); /* signs match so this should be ok */
	  goto RATIO_MOD_RATIO;

	case T_REAL:
	  if ((integer(x) == S7_INT64_MIN) || (s7_int_abs(integer(x)) > QUOTIENT_INT_LIMIT))
	    out_of_range(sc, sc->modulo_symbol, int_one, x, its_too_large_string);
	  b = real(y);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  if (is_inf(b)) return(real_NaN);
	  a = (s7_double)integer(x);
	  goto REAL_MOD;

	default:
	  return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, T_REAL, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);

	  if ((n2 > 0) && (n1 > 0) && (n2 > n1)) return(x);
	  if ((n2 < 0) && (n1 < 0) && (n2 < n1)) return(x);
	  if (n2 == S7_INT64_MIN)
	    simple_out_of_range(sc, sc->modulo_symbol,
				       set_elist_3(sc, sc->divide_symbol, x, y),
				       intermediate_too_large_string);
	  /* the problem here is that (modulo 3/2 most-negative-fixnum)
	   * will segfault with signal SIGFPE, Arithmetic exception, so try to trap it.
	   */
	  if ((n1 == n2) && (d1 > 1)) return(x);
	  d2 = 1;
	  goto RATIO_MOD_RATIO;

	case T_RATIO:
	  parcel_out_fractions(x, y);
	  if (d1 == d2)
	    return(make_ratio_with_div_check(sc, sc->modulo_symbol, modulo_i_ii(n1, n2), d1));
	  if ((n1 == n2) && (d1 > d2)) return(x);

	RATIO_MOD_RATIO:
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int n2d1, n1d2, d1d2, fl;
	    if (!multiply_overflow(n2, d1, &n2d1))
	      {
		if ((n2d1 == 1) || (n2d1 == -1)) /* (modulo 100 -1/2) */
		  return(int_zero);

		if (!multiply_overflow(n1, d2, &n1d2))
		  {
		    fl = (s7_int)(n1d2 / n2d1);
		    if (((n1 < 0) && (n2 > 0)) ||
			((n1 > 0) && (n2 < 0)))
		      fl -= 1;
		    if (fl == 0)
		      return(x);

		    if ((!multiply_overflow(d1, d2, &d1d2)) &&
			(!multiply_overflow(fl, n2d1, &fl)) &&
			(!subtract_overflow(n1d2, fl, &fl)))
		      return(make_ratio_with_div_check(sc, sc->modulo_symbol, fl, d1d2));
		  }}}
#else
	  {
	    s7_int n1d2, n2d1, fl;
	    n1d2 = n1 * d2;
	    n2d1 = n2 * d1;

	    if (n2d1 == 1)
	      return(int_zero);

	    /* can't use "floor" here (float->int ruins everything) */
	    fl = (s7_int)(n1d2 / n2d1);
	    if (((n1 < 0) && (n2 > 0)) ||
		((n1 > 0) && (n2 < 0)))
	      fl -= 1;

	    if (fl == 0)
	      return(x);

	    return(make_ratio_with_div_check(sc, sc->modulo_symbol, n1d2 - (n2d1 * fl), d1 * d2));
	  }
#endif
	  simple_out_of_range(sc, sc->modulo_symbol,
				     set_elist_3(sc, sc->divide_symbol, x, y),
				     intermediate_too_large_string);
	case T_REAL:
	  b = real(y);
	  if (is_inf(b)) return(real_NaN);
	  if (fabs(b) > 1e17)
	    out_of_range(sc, sc->modulo_symbol, int_two, y, its_too_large_string);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  a = fraction(x);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	default:
	  return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, T_REAL, 2));
	}

    case T_REAL:
      {
	s7_double c;
	a = real(x);
	if (!is_real(y))
	  return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, T_REAL, 2));
	if (is_NaN(a)) return(x);
	if (is_inf(a)) return(real_NaN); /* not b */
	if (fabs(a) > 1e17)
	  out_of_range(sc, sc->modulo_symbol, int_one, x, its_too_large_string);

	switch (type(y))
	  {
	  case T_INTEGER:
	    if (integer(y) == 0) return(x);
	    if ((integer(y) == S7_INT64_MIN) || (s7_int_abs(integer(y)) > QUOTIENT_INT_LIMIT))
	      out_of_range(sc, sc->modulo_symbol, int_two, y, its_too_large_string);
	    b = (s7_double)integer(y);
	    goto REAL_MOD;

	  case T_RATIO:
	    b = fraction(y);
	    goto REAL_MOD;

	  case T_REAL:
	    b = real(y);
	    if (b == 0.0) return(x);
	    if (is_NaN(b)) return(y);
	    if (is_inf(b)) return(real_NaN);
	  REAL_MOD:
	    c = a / b;
	    if (fabs(c) > 1e19)
	      simple_out_of_range(sc, sc->modulo_symbol,
					 set_elist_3(sc, sc->divide_symbol, x, y),
					 intermediate_too_large_string);
	    return(make_real(sc, a - b * (s7_int)floor(c)));

	  default:
	    return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, T_REAL, 2));
	  }}

    default:
      return(method_or_bust_pp(sc, x, sc->modulo_symbol, x, y, T_REAL, 1));
    }
#endif
}

static s7_pointer modulo_p_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x)) return(make_integer(sc, modulo_i_ii(integer(x), y)));
  return(modulo_p_pp(sc, x, wrap_integer(sc, y)));
}

static s7_pointer g_modulo(s7_scheme *sc, s7_pointer args)
{
  #define H_modulo "(modulo x1 x2) returns x1 mod x2; (modulo 4 3) = 1.  The arguments can be real numbers."
  #define Q_modulo sc->pcl_r
  /* (define (mod x1 x2) (- x1 (* x2 (floor (/ x1 x2))))) from slib
   * (mod x 0) = x according to "Concrete Mathematics"
   */
  return(modulo_p_pp(sc, car(args), cadr(args)));
}


/* ---------------------------------------- max ---------------------------------------- */
static bool is_real_via_method_1(s7_scheme *sc, s7_pointer p)
{
  s7_pointer f = find_method_with_let(sc, p, sc->is_real_symbol);
  if (f != sc->undefined)
    return(is_true(sc, s7_apply_function(sc, f, set_plist_1(sc, p))));
  return(false);
}

#define is_real_via_method(sc, p) ((is_real(p)) || ((has_active_methods(sc, p)) && (is_real_via_method_1(sc, p))))

#define max_out_x(Sc, X, Y) method_or_bust_pp(Sc, X, Sc->max_symbol, X, Y, T_REAL, 1)
#define max_out_y(Sc, X, Y) method_or_bust_pp(Sc, Y, Sc->max_symbol, X, Y, T_REAL, 2)

static s7_pointer max_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* same basic code as lt_b_7_pp (or any relop) but max returns NaN if NaN encountered, and methods for < and max return
   *    different results, so it seems simpler to repeat the other code.
   */
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return((integer(x) < integer(y)) ? y : x);
      if (is_t_real(x))
	return(((is_NaN(real(x))) || (real(x) >= real(y))) ? x : y);
      if (is_t_ratio(x))
	return((fraction(x) < fraction(y)) ? y : x);
#if WITH_GMP
      if (is_t_big_integer(x))
	return((mpz_cmp(big_integer(x), big_integer(y)) < 0) ? y : x);
      if (is_t_big_ratio(x))
	return((mpq_cmp(big_ratio(x), big_ratio(y)) < 0) ? y : x);
      if (is_t_big_real(x))
	return(((mpfr_nan_p(big_real(x)) != 0) || (mpfr_greaterequal_p(big_real(x), big_real(y)))) ? x : y); /* ?? */
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:
	  return((integer(x) < fraction(y)) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  return((integer(x) < real(y)) ? y : x);
#if WITH_GMP
	case T_BIG_INTEGER:
	  return((mpz_cmp_si(big_integer(y), integer(x)) < 0) ? x : y);
	case T_BIG_RATIO:
	  return((mpq_cmp_si(big_ratio(y), integer(x), 1) < 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_si(big_real(y), integer(x)) < 0) ? x : y);
#endif
	default:
	  return(max_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((fraction(x) < integer(y)) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  return((fraction(x) < real(y)) ? y : x);
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpq_cmp_z(sc->mpq_1, big_integer(y)) < 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) < 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpfr_cmp_q(big_real(y), sc->mpq_1) < 0) ? x : y);
#endif
	default:
	  return(max_out_y(sc, x, y));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(x);
	  return((real(x) < integer(y)) ? y : x);
	case T_RATIO:
	  return((real(x) < fraction(y)) ? y : x);
#if WITH_GMP
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return((mpfr_cmp_z(sc->mpfr_1, big_integer(y)) < 0) ? y : x);

	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return((mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) < 0) ? y : x);

	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(x);
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_d(big_real(y), real(x)) < 0) ? x : y);
#endif
	default:
	  return(max_out_y(sc, x, y));
	}
      break;

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpz_cmp_si(big_integer(x), integer(y)) < 0) ? y : x);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return((mpq_cmp_z(sc->mpq_1, big_integer(x)) < 0) ? x : y);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) < 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_z(big_ratio(y), big_integer(x)) < 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_z(big_real(y), big_integer(x)) < 0) ? x : y);
	default:
	  return(max_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpq_cmp_si(big_ratio(x), integer(y), 1) < 0) ? y : x);
	case T_RATIO:
	  return((mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) < 0) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) < 0) ? y : x);
	case T_BIG_INTEGER:
	  return((mpq_cmp_z(big_ratio(x), big_integer(y)) < 0) ? y : x);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_q(big_real(y), big_ratio(x)) < 0) ? x : y);
	default:
	  return(max_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_si(big_real(x), integer(y)) < 0) ? y : x);
	case T_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return((mpfr_cmp_q(big_real(x), sc->mpq_1) < 0) ? y : x);
	case T_REAL:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  if (is_NaN(real(y))) return(y);
	  return((mpfr_cmp_d(big_real(x), real(y)) < 0) ? y : x);
	case T_BIG_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_z(big_real(x), big_integer(y)) < 0) ? y : x);
	case T_BIG_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_q(big_real(x), big_ratio(y)) < 0) ? y : x);
	default:
	  return(max_out_y(sc, x, y));
	}
#endif
    default:
      return(max_out_x(sc, x, y));
    }
  return(x);
}

static s7_pointer g_max(s7_scheme *sc, s7_pointer args)
{
  #define H_max "(max ...) returns the maximum of its arguments"
  #define Q_max sc->pcl_r

  s7_pointer x = car(args);
  if (is_null(cdr(args)))
    {
      if (is_real(x)) return(x);
      return(method_or_bust_p(sc, x, sc->max_symbol, T_REAL));
    }
  for (s7_pointer p = cdr(args); is_pair(p); p = cdr(p))
    x = max_p_pp(sc, x, car(p));
  return(x);
}

static s7_pointer g_max_2(s7_scheme *sc, s7_pointer args) {return(max_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_max_3(s7_scheme *sc, s7_pointer args) {return(max_p_pp(sc, max_p_pp(sc, car(args), cadr(args)), caddr(args)));}

static s7_pointer max_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer unused_expr, bool unused_ops)
{
  return((args == 2) ? sc->max_2 : ((args == 3) ? sc->max_3 : f));
}

static s7_int max_i_ii(s7_int i1, s7_int i2) {return((i1 > i2) ? i1 : i2);}
static s7_int max_i_iii(s7_int i1, s7_int i2, s7_int i3) {return((i1 > i2) ? ((i1 > i3) ? i1 : i3) : ((i2 > i3) ? i2 : i3));}
static s7_double max_d_dd(s7_double x1, s7_double x2) {if (is_NaN(x1)) return(x1); return((x1 > x2) ? x1 : x2);}
static s7_double max_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(max_d_dd(x1, max_d_dd(x2, x3)));}
static s7_double max_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(max_d_dd(x1, max_d_ddd(x2, x3, x4)));}


/* ---------------------------------------- min ---------------------------------------- */
#define min_out_x(Sc, X, Y) method_or_bust_pp(Sc, X, Sc->min_symbol, X, Y, T_REAL, 1)
#define min_out_y(Sc, X, Y) method_or_bust_pp(Sc, Y, Sc->min_symbol, X, Y, T_REAL, 2)

static s7_pointer min_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return((integer(x) > integer(y)) ? y : x);
      if (is_t_real(x))
	return(((is_NaN(real(x))) || (real(x) <= real(y))) ? x : y);
      if (is_t_ratio(x))
	return((fraction(x) > fraction(y)) ? y : x);
#if WITH_GMP
      if (is_t_big_integer(x))
	return((mpz_cmp(big_integer(x), big_integer(y)) > 0) ? y : x);
      if (is_t_big_ratio(x))
	return((mpq_cmp(big_ratio(x), big_ratio(y)) > 0) ? y : x);
      if (is_t_big_real(x))
	return(((mpfr_nan_p(big_real(x)) != 0) || (mpfr_lessequal_p(big_real(x), big_real(y)))) ? x : y); /* ?? */
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:       return((integer(x) > fraction(y)) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  return((integer(x) > real(y)) ? y : x);
#if WITH_GMP
	case T_BIG_INTEGER: return((mpz_cmp_si(big_integer(y), integer(x)) > 0) ? x : y);
	case T_BIG_RATIO:   return((mpq_cmp_si(big_ratio(y), integer(x), 1) > 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_si(big_real(y), integer(x)) > 0) ? x : y);
#endif
	default:
	  return(min_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((fraction(x) > integer(y)) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  return((fraction(x) > real(y)) ? y : x);
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpq_cmp_z(sc->mpq_1, big_integer(y)) > 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) > 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpfr_cmp_q(big_real(y), sc->mpq_1) > 0) ? x : y);
#endif
	default:
	  return(min_out_y(sc, x, y));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(x);
	  return((real(x) > integer(y)) ? y : x);
	case T_RATIO:
	  return((real(x) > fraction(y)) ? y : x);
#if WITH_GMP
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return((mpfr_cmp_z(sc->mpfr_1, big_integer(y)) > 0) ? y : x);

	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return((mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) > 0) ? y : x);

	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(x);
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_d(big_real(y), real(x)) > 0) ? x : y);
#endif
	default:
	  return(min_out_y(sc, x, y));
	}
      break;

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpz_cmp_si(big_integer(x), integer(y)) > 0) ? y : x);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return((mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) > 0) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) > 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_z(big_ratio(y), big_integer(x)) > 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_z(big_real(y), big_integer(x)) > 0) ? x : y);
	default:
	  return(min_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpq_cmp_si(big_ratio(x), integer(y), 1) > 0) ? y : x);
	case T_RATIO:
	  return((mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) > 0) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) > 0) ? y : x);
	case T_BIG_INTEGER:
	  return((mpq_cmp_z(big_ratio(x), big_integer(y)) > 0) ? y : x);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_q(big_real(y), big_ratio(x)) > 0) ? x : y);
	default:
	  return(min_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_si(big_real(x), integer(y)) > 0) ? y : x);
	case T_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return((mpfr_cmp_q(big_real(x), sc->mpq_1) > 0) ? y : x);
	case T_REAL:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  if (is_NaN(real(y))) return(y);
	  return((mpfr_cmp_d(big_real(x), real(y)) > 0) ? y : x);
	case T_BIG_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_z(big_real(x), big_integer(y)) > 0) ? y : x);
	case T_BIG_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_q(big_real(x), big_ratio(y)) > 0) ? y : x);
	default:
	  return(min_out_y(sc, x, y));
	}
#endif
    default:
      return(min_out_x(sc, x, y));
    }
  return(x);
}

static s7_pointer g_min(s7_scheme *sc, s7_pointer args)
{
  #define H_min "(min ...) returns the minimum of its arguments"
  #define Q_min sc->pcl_r

  s7_pointer x = car(args);
  if (is_null(cdr(args)))
    {
      if (is_real(x)) return(x);
      return(method_or_bust_p(sc, x, sc->min_symbol, T_REAL));
    }
  for (s7_pointer p = cdr(args); is_pair(p); p = cdr(p))
    x = min_p_pp(sc, x, car(p));
  return(x);
}

static s7_pointer g_min_2(s7_scheme *sc, s7_pointer args) {return(min_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_min_3(s7_scheme *sc, s7_pointer args) {return(min_p_pp(sc, min_p_pp(sc, car(args), cadr(args)), caddr(args)));}

static s7_pointer min_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer unused_expr, bool unused_ops)
{
  return((args == 2) ? sc->min_2 : ((args == 3) ? sc->min_3 : f));
}

static s7_int min_i_ii(s7_int i1, s7_int i2) {return((i1 < i2) ? i1 : i2);}
static s7_int min_i_iii(s7_int i1, s7_int i2, s7_int i3) {return((i1 < i2) ? ((i1 < i3) ? i1 : i3) : ((i2 < i3) ? i2 : i3));}
static s7_double min_d_dd(s7_double x1, s7_double x2) {if (is_NaN(x1)) return(x1); return((x1 < x2) ? x1 : x2);}
static s7_double min_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(min_d_dd(x1, min_d_dd(x2, x3)));}
static s7_double min_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(min_d_dd(x1, min_d_ddd(x2, x3, x4)));}


/* ---------------------------------------- = ---------------------------------------- */
static bool eq_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->num_eq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument_with_type(sc, sc->num_eq_symbol, 1, x, a_number_string);
  return(false);
}

static bool eq_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->num_eq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument_with_type(sc, sc->num_eq_symbol, 2, y, a_number_string);
  return(false);
}

static bool num_eq_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) == integer(y));
      if (is_t_real(x))
	return(real(x) == real(y));
      if (is_t_complex(x))
	return((real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y)));
      if (is_t_ratio(x))
	return((numerator(x) == numerator(y)) && (denominator(x) == denominator(y)));
#if WITH_GMP
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) == 0);
      if (is_t_big_ratio(x))
	return(mpq_equal(big_ratio(x), big_ratio(y)));
      if (is_t_big_real(x))
	return(mpfr_equal_p(big_real(x), big_real(y)));
      if (is_t_big_complex(x)) /* mpc_cmp can't handle NaN */
	{
	  if ((mpfr_nan_p(mpc_realref(big_complex(x)))) || (mpfr_nan_p(mpc_imagref(big_complex(x)))) ||
	      (mpfr_nan_p(mpc_realref(big_complex(y)))) || (mpfr_nan_p(mpc_imagref(big_complex(y)))))
	    return(false);
	  return(mpc_cmp(big_complex(x), big_complex(y)) == 0);
	}
#endif
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:
	  return(false);
	case T_REAL:
#if WITH_GMP
	  if (s7_int_abs(integer(x)) >= INT64_TO_DOUBLE_LIMIT)
	    {
	      if (is_NaN(real(y))) return(false);
	      mpfr_set_d(sc->mpfr_1, real(y), MPFR_RNDN);
	      return(mpfr_cmp_si(sc->mpfr_1, integer(x)) == 0);
	    }
#endif
	  return(integer(x) == real(y));
	case T_COMPLEX:
	  return(false);
#if WITH_GMP
	case T_BIG_INTEGER:
	  return((mpz_fits_slong_p(big_integer(y))) && (integer(x) == mpz_get_si(big_integer(y))));
	case T_BIG_RATIO:
	  return(false);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_si(big_real(y), integer(x)) == 0));
	case T_BIG_COMPLEX:
	  return(false);
#endif
	default: return(eq_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(false);
	case T_REAL:    return(fraction(x) == real(y));
	case T_COMPLEX: return(false);
#if WITH_GMP
	case T_BIG_INTEGER:
	  return(false);
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_equal(sc->mpq_1, big_ratio(y)));
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(false);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) == 0);
	case T_BIG_COMPLEX:
	  return(false);
#endif
	default: return(eq_out_y(sc, x, y));
	}
      break;

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(real(x) == integer(y));
	case T_RATIO:
	  return(real(x) == fraction(y));
	case T_COMPLEX:
	  return(false);
#if WITH_GMP
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) == 0);
	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) == 0);
	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(false);
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_d(big_real(y), real(x)) == 0));
	case T_BIG_COMPLEX:
	  return(false);
#endif
	default: return(eq_out_y(sc, x, y));
	}
      break;

    case T_COMPLEX:
      if (is_real(y)) return(false);
#if WITH_GMP
      if (is_t_big_complex(y))
	{
	  if ((is_NaN(real_part(x))) || (is_NaN(imag_part(x))) ||
	      (mpfr_nan_p(mpc_realref(big_complex(y)))) || (mpfr_nan_p(mpc_imagref(big_complex(y)))))
	    return(false);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  return(mpc_cmp(big_complex(y), sc->mpc_1) == 0);
	}
#endif
      return(eq_out_y(sc, x, y));

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpz_fits_slong_p(big_integer(x))) && (integer(y) == mpz_get_si(big_integer(x))));
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(y), MPFR_RNDN);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(x)) == 0);
	case T_RATIO: case T_COMPLEX: case T_BIG_RATIO: case T_BIG_COMPLEX:
	  return(false);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_z(big_real(y), big_integer(x)) == 0));
	default: return(eq_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpq_equal(sc->mpq_1, big_ratio(x)));
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(y), MPFR_RNDN);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(x)) == 0);
	case T_INTEGER: case T_BIG_INTEGER: case T_COMPLEX: case T_BIG_COMPLEX:
	  return(false);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_q(big_real(y), big_ratio(x)) == 0));
	default: return(eq_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      if ((is_number(y)) && (mpfr_nan_p(big_real(x)))) return(false);
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) == 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) == 0);
	case T_REAL:
	  return((!is_NaN(real(y))) && (mpfr_cmp_d(big_real(x), real(y)) == 0));
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) == 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) == 0);
	case T_COMPLEX: case T_BIG_COMPLEX:
	  return(false);
	default: return(eq_out_y(sc, x, y));
	}

    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_RATIO: case T_REAL: case T_INTEGER: case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL:
	  return(false);
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y))) ||
	      (mpfr_nan_p(mpc_realref(big_complex(x)))) || (mpfr_nan_p(mpc_imagref(big_complex(x)))))
	    return(false);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  return(mpc_cmp(big_complex(x), sc->mpc_1) == 0); /* NaN's not allowed! */
	default: return(eq_out_y(sc, x, y));
	}
#endif
    default: return(eq_out_x(sc, x, y));
    }
  return(false);
}

static bool is_number_via_method(s7_scheme *sc, s7_pointer p)
{
  if (is_number(p))
    return(true);
  if (has_active_methods(sc, p))
    {
      s7_pointer f = find_method_with_let(sc, p, sc->is_number_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, set_plist_1(sc, p))));
    }
  return(false);
}

static s7_pointer g_num_eq(s7_scheme *sc, s7_pointer args)
{
  #define H_num_eq "(= z1 ...) returns #t if all its arguments are equal"
  #define Q_num_eq s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_number_symbol)

  s7_pointer x = car(args), p = cdr(args);
  if (is_null(cdr(p)))
    return(make_boolean(sc, num_eq_b_7pp(sc, x, car(p))));

  for (; is_pair(p); p = cdr(p))
    if (!num_eq_b_7pp(sc, x, car(p)))
      {
	for (p = cdr(p); is_pair(p); p = cdr(p))
	  if (!is_number_via_method(sc, car(p)))
	    wrong_type_argument_with_type(sc, sc->num_eq_symbol, position_of(p, args), car(p), a_number_string);
	return(sc->F);
      }
  return(sc->T);
}

static bool num_eq_b_ii(s7_int i1, s7_int i2) {return(i1 == i2);}
static bool num_eq_b_dd(s7_double i1, s7_double i2) {return(i1 == i2);}

static s7_pointer num_eq_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 == x2));}
static s7_pointer num_eq_p_ii(s7_scheme *sc, s7_int x1, s7_int x2)       {return(make_boolean(sc, x1 == x2));}
static s7_pointer num_eq_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, num_eq_b_7pp(sc, x, y)));}

static s7_pointer num_eq_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_t_integer(p1))
    return((integer(p1) == p2) ? sc->T : sc->F);
  if (is_t_real(p1))
    return((real(p1) == p2) ? sc->T : sc->F);
#if WITH_GMP
  if (is_t_big_integer(p1))
    return(((mpz_fits_slong_p(big_integer(p1))) && (p2 == mpz_get_si(big_integer(p1)))) ? sc->T : sc->F);
  if (is_t_big_real(p1))
    return((mpfr_cmp_si(big_real(p1), p2) == 0) ? sc->T : sc->F);
#endif
  return((is_number(p1)) ? sc->F : make_boolean(sc, eq_out_x(sc, p1, make_integer(sc, p2))));
}

static bool num_eq_b_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x))
    return(integer(x) == y);
  if (is_t_real(x))
    return(real(x) == y);
#if WITH_GMP
  if (is_t_big_integer(x))
    return((mpz_fits_slong_p(big_integer(x))) && (y == mpz_get_si(big_integer(x))));
  if (is_t_big_real(x))
    return(mpfr_cmp_si(big_real(x), y) == 0);
#endif
  if (!is_number(x)) /* complex/ratio */
    simple_wrong_type_argument_with_type(sc, sc->num_eq_symbol, x, a_number_string);
    /* return(eq_out_x(sc, x, make_integer(sc, y))); */ /* much slower? see thash */
  return(false);
}

static s7_pointer g_num_eq_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = car(args), y = cadr(args);
  if ((is_t_integer(x)) && (is_t_integer(y))) /* this is by far the most common case (ratios aren't used much, and = with floats is frowned upon) */
    return(make_boolean(sc, integer(x) == integer(y)));
  return(make_boolean(sc, num_eq_b_7pp(sc, x, y)));
}

static inline s7_pointer num_eq_xx(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) == integer(y)));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) == integer(y)));
  if (!is_number(x))
    return(make_boolean(sc, eq_out_x(sc, x, y)));
#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), integer(y)) == 0));
  if (is_t_big_real(x))
    {
      if (mpfr_nan_p(big_real(x))) return(sc->F);
      return(make_boolean(sc, mpfr_cmp_si(big_real(x), integer(y)) == 0));
    }
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), integer(y), 1) == 0));
#endif
  return(sc->F);
}

static s7_pointer g_num_eq_xi(s7_scheme *sc, s7_pointer args) {return(num_eq_xx(sc, car(args), cadr(args)));}
static s7_pointer g_num_eq_ix(s7_scheme *sc, s7_pointer args) {return(num_eq_xx(sc, cadr(args), car(args)));}

static s7_pointer num_eq_chooser(s7_scheme *sc, s7_pointer ur_f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 2)
    {
      if ((ops) && (is_t_integer(caddr(expr))))
	return(sc->num_eq_xi);
      return(((ops) && (is_t_integer(cadr(expr)))) ? sc->num_eq_ix : sc->num_eq_2);
    }
  return(ur_f);
}


/* ---------------------------------------- < ---------------------------------------- */
static bool lt_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->lt_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->lt_symbol, 1, x, T_REAL);
  return(false);
}

static bool lt_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->lt_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->lt_symbol, 2, y, T_REAL);
  return(false);
}

static bool lt_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) < integer(y));
      if (is_t_real(x))
	return(real(x) < real(y));
      if (is_t_ratio(x))
	return(fraction(x) < fraction(y));
#if WITH_GMP
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) < 0);
      if (is_t_big_ratio(x))
	return(mpq_cmp(big_ratio(x), big_ratio(y)) < 0);
      if (is_t_big_real(x))
	return(mpfr_less_p(big_real(x), big_real(y)));
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:	return(integer(x) < fraction(y)); /* ?? */
	case T_REAL:	return(integer(x) < real(y));
#if WITH_GMP
	case T_BIG_INTEGER: return(mpz_cmp_si(big_integer(y), integer(x)) > 0);
	case T_BIG_RATIO:   return(mpq_cmp_si(big_ratio(y), integer(x), 1) > 0);
	case T_BIG_REAL:    return(mpfr_cmp_si(big_real(y), integer(x)) > 0);
#endif
	default: return(lt_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(fraction(x) < integer(y));
	case T_REAL:    return(fraction(x) < real(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_cmp_z(sc->mpq_1, big_integer(y)) < 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) > 0);
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) > 0);
#endif
	default: return(lt_out_y(sc, x, y));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(real(x) < integer(y));
	case T_RATIO:	return(real(x) < fraction(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) < 0);

	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) < 0);

	case T_BIG_REAL:
	  return(mpfr_cmp_d(big_real(y), real(x)) > 0);
#endif
	default: return(lt_out_y(sc, x, y));
	}
      break;

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpz_cmp_si(big_integer(x), integer(y)) < 0);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return(mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) < 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) < 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_z(big_ratio(y), big_integer(x)) > 0);
	case T_BIG_REAL:
	  return(mpfr_cmp_z(big_real(y), big_integer(x)) > 0);
	default: return(lt_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpq_cmp_si(big_ratio(x), integer(y), 1) < 0);
	case T_RATIO:
	  return(mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) < 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) < 0);
	case T_BIG_INTEGER:
	  return(mpq_cmp_z(big_ratio(x), big_integer(y)) < 0);
	case T_BIG_REAL:
	  return(mpfr_cmp_q(big_real(y), big_ratio(x)) > 0);
	default: return(lt_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) < 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) < 0);
	case T_REAL:
	  return(mpfr_cmp_d(big_real(x), real(y)) < 0);
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) < 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) < 0);
	default: return(lt_out_y(sc, x, y));
	}
#endif
    default: return(lt_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer g_less(s7_scheme *sc, s7_pointer args)
{
  #define H_less "(< x1 ...) returns #t if its arguments are in increasing order"
  #define Q_less s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x = car(args), p = cdr(args);
  if (is_null(cdr(p)))
    return(make_boolean(sc, lt_b_7pp(sc, x, car(p))));

  for (; is_pair(p); p = cdr(p))
    {
      if (!lt_b_7pp(sc, x, car(p)))
	{
	  for (p = cdr(p); is_pair(p); p = cdr(p))
	    if (!is_real_via_method(sc, car(p)))
	      wrong_type_argument(sc, sc->lt_symbol, position_of(p, args), car(p), T_REAL);
	  return(sc->F);
	}
      x = car(p);
    }
  return(sc->T);
}

static bool ratio_lt_pi(s7_pointer x, s7_int y)
{
  if ((y >= 0) && (numerator(x) < 0))
    return(true);
  if ((y <= 0) && (numerator(x) > 0))
    return(false);
  if (denominator(x) < S7_INT32_MAX)
    return(numerator(x) < (y * denominator(x)));
  return(fraction(x) < y);
}

static s7_pointer g_less_x0(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = car(args);
  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) < 0));
  if (is_small_real(x))
    return(make_boolean(sc, is_negative(sc, x)));
#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), 0) < 0));
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_si(big_real(x), 0) < 0));
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), 0, 1) < 0));
#endif
  return(method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1));
}

static s7_pointer g_less_xi(s7_scheme *sc, s7_pointer args)
{
  s7_int y = integer(cadr(args));
  s7_pointer x = car(args);

  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) < y));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) < y));
  if (is_t_ratio(x))
    return(make_boolean(sc, ratio_lt_pi(x, y)));
#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), y) < 0));
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_si(big_real(x), y) < 0));
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), y, 1) < 0));
#endif
  return(method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1));
}

static s7_pointer g_less_xf(s7_scheme *sc, s7_pointer args)
{
  s7_double y = real(cadr(args)); /* chooser below checks is_t_real(y) */
  s7_pointer x = car(args);

  if (is_t_real(x))
    return(make_boolean(sc, real(x) < y));
  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) < y));
  if (is_t_ratio(x))
    return(make_boolean(sc, fraction(x) < y));
#if WITH_GMP
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_d(big_real(x), y) < 0));
  if (is_t_big_integer(x))
    {
      mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
      return(make_boolean(sc, mpfr_cmp_z(sc->mpfr_1, big_integer(x)) > 0));
    }
  if (is_t_big_ratio(x))
    {
      mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
      return(make_boolean(sc, mpfr_cmp_q(sc->mpfr_1, big_ratio(x)) > 0));
    }
#endif
  return(method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1));
}

static inline s7_pointer lt_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, lt_b_7pp(sc, p1, p2)));}
static bool lt_b_ii(s7_int i1, s7_int i2) {return(i1 < i2);}
static bool lt_b_dd(s7_double i1, s7_double i2) {return(i1 < i2);}
static s7_pointer lt_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 < x2));}
static s7_pointer lt_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 < x2));}

static bool lt_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_t_integer(p1)) return(integer(p1) < p2);
  if (is_t_real(p1))  return(real(p1) < p2);
  if (is_t_ratio(p1)) return(ratio_lt_pi(p1, p2));
#if WITH_GMP
  if (is_t_big_integer(p1))
    return(mpz_cmp_si(big_integer(p1), p2) < 0);
  if (is_t_big_real(p1))
    return(mpfr_cmp_si(big_real(p1), p2) < 0);
  if (is_t_big_ratio(p1))
    return(mpq_cmp_si(big_ratio(p1), p2, 1) < 0);
#endif
  return(lt_out_x(sc, p1, make_integer(sc, p2)));
}

static s7_pointer g_less_2(s7_scheme *sc, s7_pointer args) {return(lt_p_pp(sc, car(args), cadr(args)));}
static s7_pointer lt_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, lt_b_pi(sc, p1, p2)));}

static s7_pointer less_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 2)
    {
      if (ops)
	{
	  s7_pointer arg2 = caddr(expr);
	  if (is_t_integer(arg2))
	    {
	      if (integer(arg2) == 0)
		return(sc->less_x0);

	      if ((integer(arg2) < S7_INT32_MAX) &&
		  (integer(arg2) > S7_INT32_MIN))
		return(sc->less_xi);
	    }
	  if (is_t_real(arg2))
	    return(sc->less_xf);
	}
      return(sc->less_2);
    }
  return(f);
}


/* ---------------------------------------- <= ---------------------------------------- */
static bool leq_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->leq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->leq_symbol, 1, x, T_REAL);
  return(false);
}

static bool leq_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->leq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->leq_symbol, 2, y, T_REAL);
  return(false);
}

static bool leq_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) <= integer(y));
      if (is_t_real(x))
	return(real(x) <= real(y));
      if (is_t_ratio(x))
	return(fraction(x) <= fraction(y));
#if WITH_GMP
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) <= 0);
      if (is_t_big_ratio(x))
	return(mpq_cmp(big_ratio(x), big_ratio(y)) <= 0);
      if (is_t_big_real(x))
	return(mpfr_lessequal_p(big_real(x), big_real(y)));
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:	return(integer(x) <= fraction(y)); /* ?? */
	case T_REAL:	return(integer(x) <= real(y));
#if WITH_GMP
	case T_BIG_INTEGER: return(mpz_cmp_si(big_integer(y), integer(x)) >= 0);
	case T_BIG_RATIO:   return(mpq_cmp_si(big_ratio(y), integer(x), 1) >= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_si(big_real(y), integer(x)) >= 0));
#endif
	default: return(leq_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(fraction(x) <= integer(y));
	case T_REAL:    return(fraction(x) <= real(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_cmp_z(sc->mpq_1, big_integer(y)) <= 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) >= 0);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(false);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) >= 0);
#endif
	default: return(leq_out_y(sc, x, y));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(real(x) <= integer(y));
	case T_RATIO:	return(real(x) <= fraction(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) <= 0);

	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) <= 0);

	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(false);
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_d(big_real(y), real(x)) >= 0));
#endif
	default: return(leq_out_y(sc, x, y));
	}
      break;

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpz_cmp_si(big_integer(x), integer(y)) <= 0);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return(mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) <= 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) <= 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_z(big_ratio(y), big_integer(x)) >= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_z(big_real(y), big_integer(x)) >= 0));
	default: return(leq_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpq_cmp_si(big_ratio(x), integer(y), 1) <= 0);
	case T_RATIO:
	  return(mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) <= 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) <= 0);
	case T_BIG_INTEGER:
	  return(mpq_cmp_z(big_ratio(x), big_integer(y)) <= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_q(big_real(y), big_ratio(x)) >= 0));
	default: return(leq_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      if ((is_real(y)) && (mpfr_nan_p(big_real(x)))) return(false);
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) <= 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) <= 0);
	case T_REAL:
	  return((!is_NaN(real(y))) && (mpfr_cmp_d(big_real(x), real(y)) <= 0));
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) <= 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) <= 0);
	default: return(leq_out_y(sc, x, y));
	}
#endif
    default: return(leq_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer g_less_or_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_less_or_equal "(<= x1 ...) returns #t if its arguments are in non-decreasing order"
  #define Q_less_or_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x = car(args), p = cdr(args);

  if (is_null(cdr(p)))
    return(make_boolean(sc, leq_b_7pp(sc, x, car(p))));

  for (; is_pair(p); x = car(p), p = cdr(p))
    if (!leq_b_7pp(sc, x, car(p)))
      {
	for (p = cdr(p); is_pair(p); p = cdr(p))
	  if (!is_real_via_method(sc, car(p)))
	    wrong_type_argument(sc, sc->leq_symbol, position_of(p, args), car(p), T_REAL);
	return(sc->F);
      }
  return(sc->T);
}

static inline s7_pointer leq_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, leq_b_7pp(sc, p1, p2)));}
static bool leq_b_ii(s7_int i1, s7_int i2) {return(i1 <= i2);}
static bool leq_b_dd(s7_double i1, s7_double i2) {return(i1 <= i2);}
static s7_pointer leq_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 <= x2));}
static s7_pointer leq_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 <= x2));}

static bool ratio_leq_pi(s7_pointer x, s7_int y)
{
  if ((y >= 0) && (numerator(x) <= 0))
    return(true);
  if ((y <= 0) && (numerator(x) > 0))
    return(false);
  if (denominator(x) < S7_INT32_MAX)
    return(numerator(x) <= (y * denominator(x)));
  return(fraction(x) <= y);
}

static s7_pointer g_leq_xi(s7_scheme *sc, s7_pointer args)
{
  s7_int y = integer(cadr(args));
  s7_pointer x = car(args);

  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) <= y));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) <= y));
  if (is_t_ratio(x))
    return(make_boolean(sc, ratio_leq_pi(x, y)));
#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), y) <= 0));
  if (is_t_big_real(x))
    {
      if (mpfr_nan_p(big_real(x))) return(sc->F);
      return(make_boolean(sc, mpfr_cmp_si(big_real(x), y) <= 0));
    }
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), y, 1) <= 0));
#endif
  return(method_or_bust(sc, x, sc->leq_symbol, args, T_REAL, 1));
}

static bool leq_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_t_integer(p1)) return(integer(p1) <= p2);
  if (is_t_real(p1))  return(real(p1) <= p2);
  if (is_t_ratio(p1)) return(ratio_leq_pi(p1, p2));
#if WITH_GMP
  if (is_t_big_integer(p1))
    return(mpz_cmp_si(big_integer(p1), p2) <= 0);
  if (is_t_big_real(p1))
    return(mpfr_cmp_si(big_real(p1), p2) <= 0);
  if (is_t_big_ratio(p1))
    return(mpq_cmp_si(big_ratio(p1), p2, 1) <= 0);
#endif
  return(leq_out_x(sc, p1, make_integer(sc, p2)));
}

static s7_pointer leq_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, leq_b_pi(sc, p1, p2)));}
static s7_pointer g_leq_2(s7_scheme *sc, s7_pointer args) {return(make_boolean(sc, leq_b_7pp(sc, car(args), cadr(args))));}
static s7_pointer g_leq_ixx(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p = cdr(args);
  if (is_t_integer(car(p)))
    {
      if (integer(car(args)) > integer(car(p)))
	{
	  if (!is_real_via_method(sc, cadr(p)))
	    wrong_type_argument(sc, sc->leq_symbol, 3, cadr(p), T_REAL);
	  return(sc->F);
	}
      if (is_t_integer(cadr(p)))
	return((integer(car(p)) > integer(cadr(p))) ? sc->F : sc->T);
    }
  return(g_less_or_equal(sc, args));
}

static s7_pointer leq_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 2)
    {
      if (ops)
	{
	  s7_pointer arg2 = caddr(expr);
	  if ((is_t_integer(arg2)) &&
	      (integer(arg2) < S7_INT32_MAX) &&
	      (integer(arg2) > S7_INT32_MIN))
	    return(sc->leq_xi);
	}
      return(sc->leq_2);
    }
  if ((args == 3) && (is_t_integer(cadr(expr))))
    return(sc->leq_ixx);
  return(f);
}


/* ---------------------------------------- > ---------------------------------------- */
static bool gt_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->gt_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->gt_symbol, 1, x, T_REAL);
  return(false);
}

static bool gt_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->gt_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->gt_symbol, 2, y, T_REAL);
  return(false);
}

static bool gt_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) > integer(y));
      if (is_t_real(x))
	return(real(x) > real(y));
      if (is_t_ratio(x))
	return(fraction(x) > fraction(y));
#if WITH_GMP
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) > 0);
      if (is_t_big_ratio(x))
	return(mpq_cmp(big_ratio(x), big_ratio(y)) > 0);
      if (is_t_big_real(x))
	return(mpfr_greater_p(big_real(x), big_real(y)));
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:	return(integer(x) > fraction(y)); /* ?? */
	case T_REAL:	return(integer(x) > real(y));
#if WITH_GMP
	case T_BIG_INTEGER: return(mpz_cmp_si(big_integer(y), integer(x)) < 0);
	case T_BIG_RATIO:   return(mpq_cmp_si(big_ratio(y), integer(x), 1) < 0);
	case T_BIG_REAL:    return(mpfr_cmp_si(big_real(y), integer(x)) < 0);
#endif
	default: return(gt_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(fraction(x) > integer(y));
	case T_REAL:    return(fraction(x) > real(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_cmp_z(sc->mpq_1, big_integer(y)) > 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) < 0);
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) < 0);
#endif
	default: return(gt_out_y(sc, x, y));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(real(x) > integer(y));
	case T_RATIO:	return(real(x) > fraction(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) > 0);

	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) > 0);

	case T_BIG_REAL:
	  return(mpfr_cmp_d(big_real(y), real(x)) < 0);
#endif
	default: return(gt_out_y(sc, x, y));
	}
      break;

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpz_cmp_si(big_integer(x), integer(y)) > 0);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return(mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) > 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) > 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_z(big_ratio(y), big_integer(x)) < 0);
	case T_BIG_REAL:
	  return(mpfr_cmp_z(big_real(y), big_integer(x)) < 0);
	default: return(gt_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpq_cmp_si(big_ratio(x), integer(y), 1) > 0);
	case T_RATIO:
	  return(mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) > 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) > 0);
	case T_BIG_INTEGER:
	  return(mpq_cmp_z(big_ratio(x), big_integer(y)) > 0);
	case T_BIG_REAL:
	  return(mpfr_cmp_q(big_real(y), big_ratio(x)) < 0);
	default: return(gt_out_y(sc, x, y));
	}
    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) > 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) > 0);
	case T_REAL:
	  return(mpfr_cmp_d(big_real(x), real(y)) > 0);
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) > 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) > 0);
	default: return(gt_out_y(sc, x, y));
	}
#endif
    default: return(gt_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer g_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_greater "(> x1 ...) returns #t if its arguments are in decreasing order"
  #define Q_greater s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x = car(args), p = cdr(args);

  if (is_null(cdr(p)))
    return(make_boolean(sc, gt_b_7pp(sc, x, car(p))));

  for (; is_pair(p); x = car(p), p = cdr(p))
    if (!gt_b_7pp(sc, x, car(p)))
      {
	for (p = cdr(p); is_pair(p); p = cdr(p))
	  if (!is_real_via_method(sc, car(p)))
	    wrong_type_argument(sc, sc->gt_symbol, position_of(p, args), car(p), T_REAL);
	return(sc->F);
      }
  return(sc->T);
}

static s7_pointer g_greater_xi(s7_scheme *sc, s7_pointer args)
{
  s7_int y = integer(cadr(args));
  s7_pointer x = car(args);

  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) > y));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) > y));
  if (is_t_ratio(x))
    return(make_boolean(sc, !ratio_leq_pi(x, y)));
#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), y) > 0));
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_si(big_real(x), y) > 0));
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), y, 1) > 0));
#endif
  return(method_or_bust_with_type(sc, x, sc->gt_symbol, args, a_number_string, 1));
}

static s7_pointer g_greater_xf(s7_scheme *sc, s7_pointer args)
{
  s7_double y = real(cadr(args));
  s7_pointer x = car(args);

  if (is_t_real(x))
    return(make_boolean(sc, real(x) > y));

  switch (type(x))
    {
    case T_INTEGER: return(make_boolean(sc, integer(x) > y));

    case T_RATIO:
      /* (> 9223372036854775807/9223372036854775806 1.0) */
      if (denominator(x) < S7_INT32_MAX) /* y range check was handled in greater_chooser */
	return(make_boolean(sc, (numerator(x) > (y * denominator(x)))));
      return(make_boolean(sc, fraction(x) > y));

#if WITH_GMP
    case T_BIG_INTEGER:
      mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
      return(make_boolean(sc, mpfr_cmp_z(sc->mpfr_1, big_integer(x)) < 0));

    case T_BIG_RATIO:
      mpfr_set_d(sc->mpfr_1, y, MPFR_RNDN);
      return(make_boolean(sc, mpfr_cmp_q(sc->mpfr_1, big_ratio(x)) < 0));

    case T_BIG_REAL:
      return(make_boolean(sc, mpfr_cmp_d(big_real(x), y) > 0));
#endif
    default:
      return(method_or_bust_with_type(sc, x, sc->gt_symbol, args, a_number_string, 1));
    }
  return(sc->T);
}

static inline s7_pointer gt_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, gt_b_7pp(sc, p1, p2)));}
static bool gt_b_ii(s7_int i1, s7_int i2) {return(i1 > i2);}
static bool gt_b_dd(s7_double i1, s7_double i2) {return(i1 > i2);}
static s7_pointer gt_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 > x2));}
static s7_pointer gt_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 > x2));}

static bool gt_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_t_integer(p1)) return(integer(p1) > p2);
  if (is_t_real(p1))  return(real(p1) > p2);
  if (is_t_ratio(p1)) return(!ratio_leq_pi(p1, p2));
#if WITH_GMP
  if (is_t_big_integer(p1))
    return(mpz_cmp_si(big_integer(p1), p2) > 0);
  if (is_t_big_real(p1))
    return(mpfr_cmp_si(big_real(p1), p2) > 0);
  if (is_t_big_ratio(p1))
    return(mpq_cmp_si(big_ratio(p1), p2, 1) > 0);
#endif
  return(gt_out_x(sc, p1, make_integer(sc, p2)));
}

static s7_pointer gt_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, gt_b_pi(sc, p1, p2)));}

static s7_pointer g_greater_2(s7_scheme *sc, s7_pointer args)
{
  /* ridiculous repetition, but overheads are killing this poor thing */
  s7_pointer x = car(args), y = cadr(args);
  if (type(x) == type(y))
    {
      if (is_t_integer(x)) return(make_boolean(sc, integer(x) > integer(y)));
      if (is_t_real(x))    return(make_boolean(sc, real(x) > real(y)));
      if (is_t_ratio(x))   return(make_boolean(sc, fraction(x) > fraction(y)));
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:   return(gt_p_pp(sc, x, y));
	case T_REAL:    return(make_boolean(sc, integer(x) > real(y)));
#if WITH_GMP
	case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL:
	  return(gt_p_pp(sc, x, y));
#endif
	default:        return(make_boolean(sc, gt_out_y(sc, x, y)));
	}
      break;

    case T_RATIO:       return(gt_p_pp(sc, x, y));

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_boolean(sc, real(x) > integer(y)));
	case T_RATIO:   return(make_boolean(sc, real(x) > fraction(y)));
#if WITH_GMP
	case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL:
	  return(gt_p_pp(sc, x, y));
#endif
	default:        return(make_boolean(sc, gt_out_y(sc, x, y)));
	}
      break;
#if WITH_GMP
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL:
      return(gt_p_pp(sc, x, y));
#endif

    default:            return(make_boolean(sc, gt_out_x(sc, x, y)));
    }
  return(sc->T);
}

static s7_pointer greater_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 2)
    {
      if (ops)
	{
	  s7_pointer arg2 = caddr(expr);
	  if ((is_t_integer(arg2)) &&
	      (integer(arg2) < S7_INT32_MAX) &&
	      (integer(arg2) > S7_INT32_MIN))
	    return(sc->greater_xi);
	  if ((is_t_real(arg2)) &&
	      (real(arg2) < S7_INT32_MAX) &&
	      (real(arg2) > S7_INT32_MIN))
	    return(sc->greater_xf);
	}
      return(sc->greater_2);
    }
  return(f);
}


/* ---------------------------------------- >= ---------------------------------------- */
static bool geq_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->geq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->geq_symbol, 1, x, T_REAL);
  return(false);
}

static bool geq_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->geq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->geq_symbol, 2, y, T_REAL);
  return(false);
}

static bool geq_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) >= integer(y));
      if (is_t_real(x))
	return(real(x) >= real(y));
      if (is_t_ratio(x))
	return(fraction(x) >= fraction(y));
#if WITH_GMP
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) >= 0);
      if (is_t_big_ratio(x))
	return(mpq_cmp(big_ratio(x), big_ratio(y)) >= 0);
      if (is_t_big_real(x))
	return(mpfr_greaterequal_p(big_real(x), big_real(y)));
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:	return(integer(x) >= fraction(y)); /* ?? */
	case T_REAL:	return(integer(x) >= real(y));
#if WITH_GMP
	case T_BIG_INTEGER: return(mpz_cmp_si(big_integer(y), integer(x)) <= 0);
	case T_BIG_RATIO:   return(mpq_cmp_si(big_ratio(y), integer(x), 1) <= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_si(big_real(y), integer(x)) <= 0));
#endif
	default: return(geq_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(fraction(x) >= integer(y));
	case T_REAL:    return(fraction(x) >= real(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_cmp_z(sc->mpq_1, big_integer(y)) >= 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) <= 0);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(false);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) <= 0);
#endif
	default: return(geq_out_y(sc, x, y));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(real(x) >= integer(y));
	case T_RATIO:	return(real(x) >= fraction(y));
#if WITH_GMP
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) >= 0);
	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), MPFR_RNDN);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) >= 0);
	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(false);
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_d(big_real(y), real(x)) <= 0));
#endif
	default: return(geq_out_y(sc, x, y));
	}
      break;

#if WITH_GMP
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpz_cmp_si(big_integer(x), integer(y)) >= 0);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return(mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) >= 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) >= 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_z(big_ratio(y), big_integer(x)) <= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_z(big_real(y), big_integer(x)) <= 0));
	default: return(geq_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpq_cmp_si(big_ratio(x), integer(y), 1) >= 0);
	case T_RATIO:
	  return(mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) >= 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), MPFR_RNDN);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) >= 0);
	case T_BIG_INTEGER:
	  return(mpq_cmp_z(big_ratio(x), big_integer(y)) >= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_q(big_real(y), big_ratio(x)) <= 0));
	default: return(geq_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      if ((is_real(y)) && (mpfr_nan_p(big_real(x)))) return(false);
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) >= 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) >= 0);
	case T_REAL:
	  return((!is_NaN(real(y))) && (mpfr_cmp_d(big_real(x), real(y)) >= 0));
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) >= 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) >= 0);
	default: return(geq_out_y(sc, x, y));
	}
#endif
    default: return(geq_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer g_greater_or_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_greater_or_equal "(>= x1 ...) returns #t if its arguments are in non-increasing order"
  #define Q_greater_or_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x = car(args), p = cdr(args);

  if (is_null(cdr(p)))
    return(make_boolean(sc, geq_b_7pp(sc, x, car(p))));

  for (; is_pair(p); x = car(p), p = cdr(p))
    if (!geq_b_7pp(sc, x, car(p)))
      {
	for (p = cdr(p); is_pair(p); p = cdr(p))
	  if (!is_real_via_method(sc, car(p)))
	    wrong_type_argument(sc, sc->geq_symbol, position_of(p, args), car(p), T_REAL);
	return(sc->F);
      }
  return(sc->T);
}

static inline s7_pointer geq_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, geq_b_7pp(sc, p1, p2)));}
static bool geq_b_ii(s7_int i1, s7_int i2) {return(i1 >= i2);}
static bool geq_b_dd(s7_double i1, s7_double i2) {return(i1 >= i2);}
static s7_pointer geq_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 >= x2));}
static s7_pointer geq_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 >= x2));}

static s7_pointer g_geq_2(s7_scheme *sc, s7_pointer args) {return(make_boolean(sc, geq_b_7pp(sc, car(args), cadr(args))));}

static s7_pointer g_geq_xf(s7_scheme *sc, s7_pointer args)
{
  s7_double y = real(cadr(args));
  s7_pointer x = car(args);
  return(make_boolean(sc, ((is_t_real(x)) ? (real(x) >= y) : geq_b_7pp(sc, car(args), cadr(args)))));
}

static s7_pointer g_geq_xi(s7_scheme *sc, s7_pointer args)
{
  s7_int y = integer(cadr(args));
  s7_pointer x = car(args);

  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) >= y));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) >= y));
  if (is_t_ratio(x))
    return(make_boolean(sc, !ratio_lt_pi(x, y)));
#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), y) >= 0));
  if (is_t_big_real(x))
    {
      if (mpfr_nan_p(big_real(x))) return(sc->F);
      return(make_boolean(sc, mpfr_cmp_si(big_real(x), y) >= 0));
    }
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), y, 1) >= 0));
#endif
  return(method_or_bust(sc, x, sc->geq_symbol, args, T_REAL, 1));
}

static bool geq_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_t_integer(p1)) return(integer(p1) >= p2);
  if (is_t_real(p1))  return(real(p1) >= p2);
  if (is_t_ratio(p1)) return(!ratio_lt_pi(p1, p2));
#if WITH_GMP
  if (is_t_big_integer(p1))
    return(mpz_cmp_si(big_integer(p1), p2) >= 0);
  if (is_t_big_real(p1))
    return((!mpfr_nan_p(big_real(p1))) && (mpfr_cmp_si(big_real(p1), p2) >= 0));
  if (is_t_big_ratio(p1))
    return(mpq_cmp_si(big_ratio(p1), p2, 1) >= 0);
#endif
  return(geq_out_x(sc, p1, make_integer(sc, p2)));
}

static s7_pointer geq_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, geq_b_pi(sc, p1, p2)));}

static s7_pointer geq_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 2)
    {
      if (ops)
	{
	  s7_pointer arg2 = caddr(expr);
	  if ((is_t_integer(arg2)) &&
	      (integer(arg2) < S7_INT32_MAX) &&
	      (integer(arg2) > S7_INT32_MIN))
	    return(sc->geq_xi);
	  if ((is_t_real(arg2)) &&
	      (real(arg2) < S7_INT32_MAX) &&
	      (real(arg2) > S7_INT32_MIN))
	    return(sc->geq_xf);
	}
      return(sc->geq_2);
    }
  return(f);
}


/* ---------------------------------------- real-part ---------------------------------------- */
s7_double s7_real_part(s7_pointer x)
{
  switch(type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return(fraction(x));
    case T_REAL:        return(real(x));
    case T_COMPLEX:     return(real_part(x));
#if WITH_GMP
    case T_BIG_INTEGER: return((s7_double)mpz_get_si(big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long_double)mpz_get_si(mpq_numref(big_ratio(x))) /
					   (long_double)mpz_get_si(mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), MPFR_RNDN));
    case T_BIG_COMPLEX: return((s7_double)mpfr_get_d(mpc_realref(big_complex(x)), MPFR_RNDN));
#endif
    }
  return(0.0);
}

static s7_pointer real_part_p_p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_complex(p)) return(make_real(sc, real_part(p)));
  switch (type(p))
    {
    case T_INTEGER: case T_RATIO: case T_REAL:
      return(p);
#if WITH_GMP
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL:
      return(p);
    case T_BIG_COMPLEX:
      {
	s7_pointer x;
	new_cell(sc, x, T_BIG_REAL);
	big_real_bgf(x) = alloc_bigflt(sc);
	add_big_real(sc, x);
	mpc_real(big_real(x), big_complex(p), MPFR_RNDN);
	return(x);
      }
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, p, sc->real_part_symbol, a_number_string));
    }
}

static s7_pointer g_real_part(s7_scheme *sc, s7_pointer args)
{
  #define H_real_part "(real-part num) returns the real part of num"
  #define Q_real_part s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  return(real_part_p_p(sc, car(args)));
}


/* ---------------------------------------- imag-part ---------------------------------------- */
s7_double s7_imag_part(s7_pointer x)
{
  if (is_t_complex(x))
    return(imag_part(x));
#if WITH_GMP
  if (is_t_big_complex(x))
    return((s7_double)mpfr_get_d(mpc_imagref(big_complex(x)), MPFR_RNDN));
#endif
  return(0.0);
}

static s7_pointer imag_part_p_p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_complex(p)) return(make_real(sc, imag_part(p)));
  switch (type(p))
    {
    case T_INTEGER: case T_RATIO:
      return(int_zero);
    case T_REAL:
      return(real_zero);
#if WITH_GMP
    case T_BIG_INTEGER: case T_BIG_RATIO:
      return(int_zero);
    case T_BIG_REAL:
      return(real_zero);
    case T_BIG_COMPLEX:
      {
	s7_pointer x;
	new_cell(sc, x, T_BIG_REAL);
	big_real_bgf(x) = alloc_bigflt(sc);
	add_big_real(sc, x);
	mpc_imag(big_real(x), big_complex(p), MPFR_RNDN);
	return(x);
      }
#endif
    default:
      return(method_or_bust_with_type_one_arg_p(sc, p, sc->imag_part_symbol, a_number_string));
    }
}

static s7_pointer g_imag_part(s7_scheme *sc, s7_pointer args)
{
  #define H_imag_part "(imag-part num) returns the imaginary part of num"
  #define Q_imag_part s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  /* currently (imag-part +nan.0) -> 0.0 ? it's true but maybe confusing */
  return(imag_part_p_p(sc, car(args)));
}


/* ---------------------------------------- numerator denominator ---------------------------------------- */
static s7_int numerator_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_ratio(p)) return(numerator(p));
  if (is_t_integer(p)) return(integer(p));
#if WITH_GMP
  if (is_t_big_ratio(p)) return(mpz_get_si(mpq_numref(big_ratio(p))));
  if (is_t_big_integer(p)) return(mpz_get_si(big_integer(p)));
#endif
  return(integer(method_or_bust_with_type_one_arg_p(sc, p, sc->numerator_symbol, a_rational_string)));
}

static s7_pointer g_numerator(s7_scheme *sc, s7_pointer args)
{
  #define H_numerator "(numerator rat) returns the numerator of the rational number rat"
  #define Q_numerator s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_rational_symbol)

  s7_pointer x = car(args);
  switch (type(x))
    {
    case T_RATIO:       return(make_integer(sc, numerator(x)));
    case T_INTEGER:     return(x);
#if WITH_GMP
    case T_BIG_INTEGER: return(x);
    case T_BIG_RATIO:   return(mpz_to_integer(sc, mpq_numref(big_ratio(x))));
#endif
    default:            return(method_or_bust_with_type_one_arg_p(sc, x, sc->numerator_symbol, a_rational_string));
    }
}


static s7_pointer g_denominator(s7_scheme *sc, s7_pointer args)
{
  #define H_denominator "(denominator rat) returns the denominator of the rational number rat"
  #define Q_denominator s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_rational_symbol)

  s7_pointer x = car(args);
  switch (type(x))
    {
    case T_RATIO:       return(make_integer(sc, denominator(x)));
    case T_INTEGER:     return(int_one);
#if WITH_GMP
    case T_BIG_INTEGER: return(int_one);
    case T_BIG_RATIO:   return(mpz_to_integer(sc, mpq_denref(big_ratio(x))));
#endif
    default:            return(method_or_bust_with_type_one_arg_p(sc, x, sc->denominator_symbol, a_rational_string));
    }
}

static s7_int denominator_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_ratio(p)) return(denominator(p));
  if (is_t_integer(p)) return(1);
#if WITH_GMP
  if (is_t_big_ratio(p)) return(mpz_get_si(mpq_denref(big_ratio(p))));
  if (is_t_big_integer(p)) return(1);
#endif
  return(integer(method_or_bust_with_type_one_arg_p(sc, p, sc->denominator_symbol, a_rational_string)));
}


/* ---------------------------------------- number? bignum? complex? integer? byte? rational? real?  ---------------------------------------- */
static s7_pointer g_is_number(s7_scheme *sc, s7_pointer args)
{
  #define H_is_number "(number? obj) returns #t if obj is a number"
  #define Q_is_number sc->pl_bt
  check_boolean_method(sc, is_number, sc->is_number_symbol, args);
}

bool s7_is_bignum(s7_pointer obj) {return(is_big_number(obj));}

static s7_pointer g_is_bignum(s7_scheme *sc, s7_pointer args)
{
  #define H_is_bignum "(bignum? obj) returns #t if obj is a multiprecision number."
  #define Q_is_bignum sc->pl_bt
  return(s7_make_boolean(sc, is_big_number(car(args))));
}

static s7_pointer g_is_integer(s7_scheme *sc, s7_pointer args)
{
  #define H_is_integer "(integer? obj) returns #t if obj is an integer"
  #define Q_is_integer sc->pl_bt
  check_boolean_method(sc, s7_is_integer, sc->is_integer_symbol, args);
}

static bool is_byte(s7_pointer p) {return((s7_is_integer(p)) && (s7_integer(p) >= 0) && (s7_integer(p) < 256));}
static s7_pointer g_is_byte(s7_scheme *sc, s7_pointer args)
{
  #define H_is_byte "(byte? obj) returns #t if obj is a byte (an integer between 0 and 255)"
  #define Q_is_byte sc->pl_bt
  check_boolean_method(sc, is_byte, sc->is_byte_symbol, args);
}

static s7_pointer g_is_real(s7_scheme *sc, s7_pointer args)
{
  #define H_is_real "(real? obj) returns #t if obj is a real number"
  #define Q_is_real sc->pl_bt
  check_boolean_method(sc, is_real, sc->is_real_symbol, args);
}

static s7_pointer g_is_complex(s7_scheme *sc, s7_pointer args)
{
  #define H_is_complex "(complex? obj) returns #t if obj is a number"
  #define Q_is_complex sc->pl_bt
  check_boolean_method(sc, is_number, sc->is_complex_symbol, args);
}

static s7_pointer g_is_rational(s7_scheme *sc, s7_pointer args)
{
  #define H_is_rational "(rational? obj) returns #t if obj is a rational number (either an integer or a ratio)"
  #define Q_is_rational sc->pl_bt
  check_boolean_method(sc, is_rational, sc->is_rational_symbol, args);
  /* in the non-gmp case, (rational? 455702434782048082459/86885567283849955830) -> #f, not #t, and similarly for exact? etc */
}

static s7_pointer g_is_float(s7_scheme *sc, s7_pointer args)
{
  #define H_is_float "(float? x) returns #t is x is real and not rational."
  #define Q_is_float sc->pl_bt
  s7_pointer p = car(args);
#if WITH_GMP
  return(make_boolean(sc, (is_t_real(p)) || (is_t_big_real(p)))); /* (float? pi) */
#else
  return(make_boolean(sc, is_t_real(p)));
#endif
}

#if WITH_GMP
static bool is_float_b(s7_pointer p) {return((is_t_real(p)) || (is_t_big_real(p)));}
#else
static bool is_float_b(s7_pointer p) {return(is_t_real(p));}
#endif


/* ---------------------------------------- nan? ---------------------------------------- */
static bool is_nan_b_7p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_real(x)) return(is_NaN(real(x)));
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:   return(false);
    /* case T_REAL:    return(is_NaN(real(x))); */
    case T_COMPLEX: return((is_NaN(real_part(x))) || (is_NaN(imag_part(x))));
#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:   return(false);
    case T_BIG_REAL:    return(mpfr_nan_p(big_real(x)) != 0);
    case T_BIG_COMPLEX: return((mpfr_nan_p(mpc_realref(big_complex(x))) != 0) || (mpfr_nan_p(mpc_imagref(big_complex(x))) != 0));
#endif
    default:
      if (is_number(x))
	return(method_or_bust_with_type_one_arg_p(sc, x, sc->is_nan_symbol, a_number_string) != sc->F);
    }
  return(false);
}

static s7_pointer g_is_nan(s7_scheme *sc, s7_pointer args)
{
  #define H_is_nan "(nan? obj) returns #t if obj is a NaN"
  #define Q_is_nan sc->pl_bt
  return(make_boolean(sc, is_nan_b_7p(sc, car(args))));
}


/* ---------------------------------------- infinite? ---------------------------------------- */
static bool is_infinite_b_7p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:    return(false);
    case T_REAL:     return(is_inf(real(x)));
    case T_COMPLEX:  return((is_inf(real_part(x))) || (is_inf(imag_part(x))));
#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO: return(false);
    case T_BIG_REAL:  return(mpfr_inf_p(big_real(x)) != 0);
    case T_BIG_COMPLEX:
      return((mpfr_inf_p(mpc_realref(big_complex(x))) != 0) ||
	     (mpfr_inf_p(mpc_imagref(big_complex(x))) != 0));
#endif
    default:
      if (is_number(x))
	return(method_or_bust_with_type_one_arg_p(sc, x, sc->is_infinite_symbol, a_number_string) != sc->F);
    }
  return(false);
}

static s7_pointer g_is_infinite(s7_scheme *sc, s7_pointer args)
{
  #define H_is_infinite "(infinite? obj) returns #t if obj is an infinite real"
  #define Q_is_infinite sc->pl_bt
  return(make_boolean(sc, is_infinite_b_7p(sc, car(args))));
}


/* ---------------------------------------- even? odd?---------------------------------------- */
static bool is_even_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p))
    return((integer(p) & 1) == 0);
#if WITH_GMP
  if (is_t_big_integer(p))
    return(mpz_even_p(big_integer(p)));
#endif
  return(method_or_bust_p(sc, p, sc->is_even_symbol, T_INTEGER) != sc->F);
}

static s7_pointer is_even_p_p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_integer(x))
    return(make_boolean(sc, (integer(x) & 1) == 0));
  return(make_boolean(sc, is_even_b_7p(sc, x)));
}

static bool is_even_i(s7_int i1) {return((i1 & 1) == 0);}

static s7_pointer g_is_even(s7_scheme *sc, s7_pointer args)
{
  #define H_is_even "(even? int) returns #t if the integer int32_t is even"
  #define Q_is_even s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_integer_symbol)
  return(make_boolean(sc, is_even_b_7p(sc, car(args))));
}


static bool is_odd_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p))
    return((integer(p) & 1) == 1);
#if WITH_GMP
  if (is_t_big_integer(p))
    return(mpz_odd_p(big_integer(p)));
#endif
  return(method_or_bust_p(sc, p, sc->is_odd_symbol, T_INTEGER) != sc->F);
}

static s7_pointer is_odd_p_p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_integer(x))
    return(make_boolean(sc, (integer(x) & 1) == 1));
  return(make_boolean(sc, is_odd_b_7p(sc, x)));
}

static bool is_odd_i(s7_int i1) {return((i1 & 1) == 1);}

static s7_pointer g_is_odd(s7_scheme *sc, s7_pointer args)
{
  #define H_is_odd "(odd? int) returns #t if the integer int32_t is odd"
  #define Q_is_odd s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_integer_symbol)
  return(make_boolean(sc, is_odd_b_7p(sc, car(args))));
}


/* ---------------------------------------- zero? ---------------------------------------- */
static bool is_zero(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x) == 0);
    case T_REAL:        return(real(x) == 0.0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(x), 0) == 0);
    case T_BIG_REAL:    return(mpfr_zero_p(big_real(x)));
#endif
    default:
      return(false); /* ratios and complex numbers here are already collapsed into integers and reals */
    }
}

static bool is_zero_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p)) return(integer(p) == 0);
  if (is_t_real(p))    return(real(p) == 0.0);
  if (is_number(p))    return(is_zero(p));
  return(method_or_bust_with_type_one_arg_p(sc, p, sc->is_zero_symbol, a_number_string) != sc->F);
}

static s7_pointer g_is_zero(s7_scheme *sc, s7_pointer args)
{
  #define H_is_zero "(zero? num) returns #t if the number num is zero"
  #define Q_is_zero sc->pl_bn
  return(make_boolean(sc, is_zero_b_7p(sc, car(args))));
}

static s7_pointer is_zero_p_p(s7_scheme *sc, s7_pointer p) {return(make_boolean(sc, is_zero_b_7p(sc, p)));}
static bool is_zero_i(s7_int p) {return(p == 0);}
static bool is_zero_d(s7_double p) {return(p == 0.0);}


/* -------------------------------- positive? -------------------------------- */
static bool is_positive(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x) > 0);
    case T_RATIO:       return(numerator(x) > 0);
    case T_REAL:        return(real(x) > 0.0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(x), 0) > 0);
    case T_BIG_RATIO:   return(mpq_cmp_ui(big_ratio(x), 0, 1) > 0);
    case T_BIG_REAL:    return(mpfr_cmp_ui(big_real(x), 0) > 0);
#endif
    default:
      simple_wrong_type_argument(sc, sc->is_positive_symbol, x, T_REAL);
    }
  return(false);
}

static bool is_positive_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p)) return(integer(p) > 0);
  if (is_t_real(p))    return(real(p) > 0.0);
  if (is_number(p))    return(is_positive(sc, p));
  return(method_or_bust_p(sc, p, sc->is_positive_symbol, T_REAL) != sc->F);
}

static s7_pointer g_is_positive(s7_scheme *sc, s7_pointer args)
{
  #define H_is_positive "(positive? num) returns #t if the real number num is positive (greater than 0)"
  #define Q_is_positive s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_real_symbol)
  return(make_boolean(sc, is_positive_b_7p(sc, car(args))));
}

static s7_pointer is_positive_p_p(s7_scheme *sc, s7_pointer p) {return(make_boolean(sc, is_positive_b_7p(sc, p)));}
static bool is_positive_i(s7_int p) {return(p > 0);}
static bool is_positive_d(s7_double p) {return(p > 0.0);}


/* -------------------------------- negative? -------------------------------- */
static bool is_negative(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x) < 0);
    case T_RATIO:       return(numerator(x) < 0);
    case T_REAL:        return(real(x) < 0.0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(x), 0) < 0);
    case T_BIG_RATIO:   return(mpq_cmp_ui(big_ratio(x), 0, 1) < 0);
    case T_BIG_REAL:    return(mpfr_cmp_ui(big_real(x), 0) < 0);
#endif
    default:
      simple_wrong_type_argument(sc, sc->is_negative_symbol, x, T_REAL);
    }
  return(false);
}

static bool is_negative_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p)) return(integer(p) < 0);
  if (is_t_real(p))    return(real(p) < 0.0);
  if (is_number(p))    return(is_negative(sc, p));
  return(method_or_bust_p(sc, p, sc->is_negative_symbol, T_REAL) != sc->F);
}

static s7_pointer g_is_negative(s7_scheme *sc, s7_pointer args)
{
  #define H_is_negative "(negative? num) returns #t if the real number num is negative (less than 0)"
  #define Q_is_negative s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_real_symbol)
  return(make_boolean(sc, is_negative_b_7p(sc, car(args))));
}

static s7_pointer is_negative_p_p(s7_scheme *sc, s7_pointer p) {return(make_boolean(sc, is_negative_b_7p(sc, p)));}
static bool is_negative_i(s7_int p) {return(p < 0);}
static bool is_negative_d(s7_double p) {return(p < 0.0);}


#if (!WITH_PURE_S7)
/* ---------------------------------------- exact<->inexact exact? inexact? ---------------------------------------- */
static s7_pointer g_exact_to_inexact(s7_scheme *sc, s7_pointer args)
{
  #define H_exact_to_inexact "(exact->inexact num) converts num to an inexact number; (exact->inexact 3/2) = 1.5"
  #define Q_exact_to_inexact s7_make_signature(sc, 2, sc->is_number_symbol, sc->is_number_symbol)
  /* arg can be complex -> itself! */
  return(exact_to_inexact(sc, car(args)));
}

static s7_pointer g_inexact_to_exact(s7_scheme *sc, s7_pointer args)
{
  #define H_inexact_to_exact "(inexact->exact num) converts num to an exact number; (inexact->exact 1.5) = 3/2"
  #define Q_inexact_to_exact s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_real_symbol)
  return(inexact_to_exact(sc, car(args)));
}

static s7_pointer g_is_exact(s7_scheme *sc, s7_pointer args)
{
  #define H_is_exact "(exact? num) returns #t if num is exact (an integer or a ratio)"
  #define Q_is_exact sc->pl_bn

  s7_pointer x = car(args);
  switch (type(x))
    {
    case T_INTEGER: case T_BIG_INTEGER:
    case T_RATIO:   case T_BIG_RATIO:
      return(sc->T);
    case T_REAL:    case T_BIG_REAL:
    case T_COMPLEX: case T_BIG_COMPLEX:
      return(sc->F);
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->is_exact_symbol, a_number_string));
    }
}

static bool is_exact_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_number(p))
    return(method_or_bust_with_type_one_arg_p(sc, p, sc->is_exact_symbol, a_number_string) != sc->F);
  return(is_rational(p));
}


static s7_pointer g_is_inexact(s7_scheme *sc, s7_pointer args)
{
  #define H_is_inexact "(inexact? num) returns #t if num is inexact (neither an integer nor a ratio)"
  #define Q_is_inexact sc->pl_bn

  s7_pointer x = car(args);
  switch (type(x))
    {
    case T_INTEGER:  case T_BIG_INTEGER:
    case T_RATIO:    case T_BIG_RATIO:
      return(sc->F);
    case T_REAL:     case T_BIG_REAL:
    case T_COMPLEX:  case T_BIG_COMPLEX:
      return(sc->T);
    default:
      return(method_or_bust_with_type_one_arg_p(sc, x, sc->is_inexact_symbol, a_number_string));
    }
}

static bool is_inexact_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_number(p))
    return(method_or_bust_with_type_one_arg_p(sc, p, sc->is_inexact_symbol, a_number_string) != sc->F);
  return(!is_rational(p));
}


/* ---------------------------------------- integer-length ---------------------------------------- */
static int32_t integer_length(s7_int a)
{
  if (a < 0)
    {
      if (a == S7_INT64_MIN) return(63);
      a = -a;
    }
  if (a < 256LL) return(intlen_bits[a]); /* in gmp, sbcl and clisp (integer-length 0) is 0 */
  if (a < 65536LL) return(8 + intlen_bits[a >> 8]);
  if (a < 16777216LL) return(16 + intlen_bits[a >> 16]);
  if (a < 4294967296LL) return(24 + intlen_bits[a >> 24]);
  if (a < 1099511627776LL) return(32 + intlen_bits[a >> 32]);
  if (a < 281474976710656LL) return(40 + intlen_bits[a >> 40]);
  if (a < 72057594037927936LL) return(48 + intlen_bits[a >> 48]);
  return(56 + intlen_bits[a >> 56]);
}

static s7_pointer g_integer_length(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_length "(integer-length arg) returns the number of bits required to represent the integer 'arg': \
(ceiling (log (if (< arg 0) (- arg) (+ arg 1)) 2))"
  #define Q_integer_length sc->pcl_i

  s7_pointer p = car(args);
  if (is_t_integer(p))
    {
      s7_int x = integer(p);
      return((x < 0) ? small_int(integer_length(-(x + 1))) : small_int(integer_length(x)));
    }
#if WITH_GMP
  if (is_t_big_integer(p))
    return(make_integer(sc, mpz_sizeinbase(big_integer(p), 2)));
#endif
  return(method_or_bust_one_arg(sc, p, sc->integer_length_symbol, args, T_INTEGER));
}

static s7_int integer_length_i_i(s7_int x) {return((x < 0) ? integer_length(-(x + 1)) : integer_length(x));}
#endif /* !pure s7 */


/* ---------------------------------------- integer-decode-float ---------------------------------------- */
static s7_pointer g_integer_decode_float(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_decode_float "(integer-decode-float x) returns a list containing the significand, exponent, and \
sign of 'x' (1 = positive, -1 = negative).  (integer-decode-float 0.0): (0 0 1)"
  #define Q_integer_decode_float s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_float_symbol)

  typedef union {
    int64_t ix;
    double fx;
  } decode_float_t;

  decode_float_t num;
  s7_pointer x = car(args);
  if (is_t_real(x))
    {
      if (real(x) == 0.0)
	return(list_3(sc, int_zero, int_zero, int_one));
      num.fx = (double)real(x);
      return(list_3(sc,
		    make_integer(sc, (s7_int)((num.ix & 0xfffffffffffffLL) | 0x10000000000000LL)),
		    make_integer(sc, (s7_int)(((num.ix & 0x7fffffffffffffffLL) >> 52) - 1023 - 52)),
		    ((num.ix & 0x8000000000000000LL) != 0) ? minus_one : int_one));
    }
#if WITH_GMP
  if (is_t_big_real(x))
    {
      mp_exp_t exp_n;
      bool neg;
      exp_n = mpfr_get_z_exp(sc->mpz_1, big_real(x));
      neg = (mpz_cmp_ui(sc->mpz_1, 0) < 0);
      if (neg) mpz_abs(sc->mpz_1, sc->mpz_1);
      return(list_3(sc, mpz_to_integer(sc, sc->mpz_1), make_integer(sc, exp_n), (neg) ? minus_one : int_one));
      /* not gmp: (integer-decode-float +nan.0): (6755399441055744 972 1), gmp: (integer-decode-float (bignum +nan.0)): (0 -1073741823 1) */
    }
#endif
  return(method_or_bust_with_type_one_arg_p(sc, x, sc->integer_decode_float_symbol, wrap_string(sc, "a non-rational real", 19)));
}


/* -------------------------------- logior -------------------------------- */
#if WITH_GMP
static s7_pointer big_logior(s7_scheme *sc, s7_int start, s7_pointer args)
{
  mpz_set_si(sc->mpz_1, start);
  for (s7_pointer x = args; is_not_null(x); x = cdr(x))
    {
      s7_pointer i = car(x);
      switch (type(i))
	{
	case T_BIG_INTEGER:
	  mpz_ior(sc->mpz_1, sc->mpz_1, big_integer(i));
	  break;
	case T_INTEGER:
	  mpz_set_si(sc->mpz_2, integer(i));
	  mpz_ior(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	  break;
	default:
	  if (!is_integer_via_method(sc, i))
	    wrong_type_argument(sc, sc->logior_symbol, position_of(x, args), i, T_INTEGER);
	  return(method_or_bust(sc, i, sc->logior_symbol,
				set_ulist_1(sc, mpz_to_integer(sc, sc->mpz_1), x),
				T_INTEGER, position_of(x, args)));
	}}
  return(mpz_to_integer(sc, sc->mpz_1));
}
#endif

static s7_pointer g_logior(s7_scheme *sc, s7_pointer args)
{
  #define H_logior "(logior int32_t ...) returns the OR of its integer arguments (the bits that are on in any of the arguments)"
  #define Q_logior sc->pcl_i

  s7_int result = 0;
  for (s7_pointer x = args; is_not_null(x); x = cdr(x))
    {
#if WITH_GMP
      if (is_t_big_integer(car(x)))
	return(big_logior(sc, result, x));
#endif
      if (!is_t_integer(car(x)))
	return(method_or_bust(sc, car(x), sc->logior_symbol,
			      (result == 0) ? x : set_ulist_1(sc, make_integer(sc, result), x),
			      T_INTEGER, position_of(x, args)));
      result |= integer(car(x));
    }
  return(make_integer(sc, result));
}

static s7_int logior_i_ii(s7_int i1, s7_int i2) {return(i1 | i2);}
static s7_int logior_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 | i2 | i3);}


/* -------------------------------- logxor -------------------------------- */
#if WITH_GMP
static s7_pointer big_logxor(s7_scheme *sc, s7_int start, s7_pointer args)
{
  mpz_set_si(sc->mpz_1, start);
  for (s7_pointer x = args; is_not_null(x); x = cdr(x))
    {
      s7_pointer i = car(x);
      switch (type(i))
	{
	case T_BIG_INTEGER:
	  mpz_xor(sc->mpz_1, sc->mpz_1, big_integer(i));
	  break;
	case T_INTEGER:
	  mpz_set_si(sc->mpz_2, integer(i));
	  mpz_xor(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	  break;
	default:
	  if (!is_integer_via_method(sc, i))
	    wrong_type_argument(sc, sc->logxor_symbol, position_of(x, args), i, T_INTEGER);
	  return(method_or_bust(sc, i, sc->logxor_symbol,
				set_ulist_1(sc, mpz_to_integer(sc, sc->mpz_1), x),
				T_INTEGER, position_of(x, args)));
	}}
  return(mpz_to_integer(sc, sc->mpz_1));
}
#endif

static s7_pointer g_logxor(s7_scheme *sc, s7_pointer args)
{
  #define H_logxor "(logxor int32_t ...) returns the XOR of its integer arguments (the bits that are on in an odd number of the arguments)"
  #define Q_logxor sc->pcl_i

  s7_int result = 0;
  for (s7_pointer x = args; is_not_null(x); x = cdr(x))
    {
#if WITH_GMP
      if (is_t_big_integer(car(x)))
	return(big_logxor(sc, result, x));
#endif
      if (!is_t_integer(car(x)))
	return(method_or_bust(sc, car(x), sc->logxor_symbol,
			      (result == 0) ? x : set_ulist_1(sc, make_integer(sc, result), x),
			      T_INTEGER, position_of(x, args)));
      result ^= integer(car(x));
    }
  return(make_integer(sc, result));
}

static s7_int logxor_i_ii(s7_int i1, s7_int i2) {return(i1 ^ i2);}
static s7_int logxor_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 ^ i2 ^ i3);}


/* -------------------------------- logand -------------------------------- */
#if WITH_GMP
static s7_pointer big_logand(s7_scheme *sc, s7_int start, s7_pointer args)
{
  mpz_set_si(sc->mpz_1, start);
  for (s7_pointer x = args; is_not_null(x); x = cdr(x))
    {
      s7_pointer i = car(x);
      switch (type(i))
	{
	case T_BIG_INTEGER:
	  mpz_and(sc->mpz_1, sc->mpz_1, big_integer(i));
	  break;
	case T_INTEGER:
	  mpz_set_si(sc->mpz_2, integer(i));
	  mpz_and(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	  break;
	default:
	  if (!is_integer_via_method(sc, i))
	    wrong_type_argument(sc, sc->logand_symbol, position_of(x, args), i, T_INTEGER);
	  return(method_or_bust(sc, i, sc->logand_symbol,
				set_ulist_1(sc, mpz_to_integer(sc, sc->mpz_1), x),
				T_INTEGER, position_of(x, args)));
	}}
  return(mpz_to_integer(sc, sc->mpz_1));
}
#endif

static s7_pointer g_logand(s7_scheme *sc, s7_pointer args)
{
  #define H_logand "(logand int32_t ...) returns the AND of its integer arguments (the bits that are on in every argument)"
  #define Q_logand sc->pcl_i

  s7_int result = -1;
  for (s7_pointer x = args; is_not_null(x); x = cdr(x))
    {
#if WITH_GMP
      if (is_t_big_integer(car(x)))
	return(big_logand(sc, result, x));
#endif
      if (!is_t_integer(car(x)))
	return(method_or_bust(sc, car(x), sc->logand_symbol,
			      (result == -1) ? x : set_ulist_1(sc, make_integer(sc, result), x),
			      T_INTEGER, position_of(x, args)));
      result &= integer(car(x));
    }
  return(make_integer(sc, result));
}

static s7_int logand_i_ii(s7_int i1, s7_int i2) {return(i1 & i2);}
static s7_int logand_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 & i2 & i3);}


/* -------------------------------- lognot -------------------------------- */
static s7_pointer g_lognot(s7_scheme *sc, s7_pointer args)
{
  #define H_lognot "(lognot num) returns the negation of num (its complement, the bits that are not on): (lognot 0) -> -1"
  #define Q_lognot sc->pcl_i

  s7_pointer x = car(args);
  if (is_t_integer(x))
    return(make_integer(sc, ~integer(x)));

#if WITH_GMP
  if (is_t_big_integer(x))
    {
      mpz_com(sc->mpz_1, big_integer(x));
      return(mpz_to_integer(sc, sc->mpz_1));
    }
#endif
  return(method_or_bust_one_arg(sc, x, sc->lognot_symbol, args, T_INTEGER));
}

static s7_int lognot_i_i(s7_int i1) {return(~i1);}


/* -------------------------------- logbit? -------------------------------- */
/* logbit?  CL is (logbitp index int) using 2^index, but that order strikes me as backwards
 *   at least gmp got the arg order right!
 */

static s7_pointer g_logbit(s7_scheme *sc, s7_pointer args)
{
  #define H_logbit "(logbit? int index) returns #t if the index-th bit is on in int, otherwise #f. The argument \
order here follows gmp, and is the opposite of the CL convention.  (logbit? int bit) is the same as (not (zero? (logand int (ash 1 bit))))."
  #define Q_logbit s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_integer_symbol)

  s7_pointer x = car(args), y = cadr(args);
  s7_int index;      /* index in gmp is mp_bitcnt which is an unsigned long int */

  if (!s7_is_integer(x))
    return(method_or_bust(sc, x, sc->logbit_symbol, args, T_INTEGER, 1));
  if (!s7_is_integer(y))
    return(method_or_bust(sc, y, sc->logbit_symbol, args, T_INTEGER, 2));

  index = s7_integer_clamped_if_gmp(sc, y);
  if (index < 0)
    out_of_range(sc, sc->logbit_symbol, int_two, y, its_negative_string);

#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, (mpz_tstbit(big_integer(x), index) != 0)));
#endif

  if (index >= S7_INT_BITS)           /* not sure about the >: (logbit? -1 64) ?? */
    return(make_boolean(sc, integer(x) < 0));
  /* (zero? (logand most-positive-fixnum (ash 1 63))) -> ash argument 2, 63, is out of range (shift is too large)
   *   so logbit? has a wider range than the logand/ash shuffle above.
   */

  /* all these int64_ts are necessary, else C turns it into an int, gets confused about signs etc */
  return(make_boolean(sc, ((((int64_t)(1LL << (int64_t)index)) & (int64_t)integer(x)) != 0)));
}

static bool logbit_b_7ii(s7_scheme *sc, s7_int i1, s7_int i2)
{
  if (i2 < 0)
    {
      out_of_range(sc, sc->logbit_symbol, int_two, wrap_integer(sc, i1), its_negative_string);
      return(false);
    }
  if (i2 >= S7_INT_BITS) return(i1 < 0);
  return((((int64_t)(1LL << (int64_t)i2)) & (int64_t)i1) != 0);
}

static bool logbit_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  if (is_t_integer(p1))
    {
      if (is_t_integer(p2))
	return(logbit_b_7ii(sc, integer(p1), integer(p2)));
      return(method_or_bust(sc, p2, sc->logbit_symbol, set_plist_2(sc, p1, p2), T_INTEGER, 2) != sc->F);
    }
#if WITH_GMP
  return(g_logbit(sc, set_plist_2(sc, p1, p2)));
#else
  return(method_or_bust(sc, p1, sc->logbit_symbol, set_plist_2(sc, p1, p2), T_INTEGER, 1) != sc->F);
#endif
}


/* -------------------------------- ash -------------------------------- */
static s7_int c_ash(s7_scheme *sc, s7_int arg1, s7_int arg2)
{
  if (arg1 == 0) return(0);
  if (arg2 >= S7_INT_BITS)
    {
      if ((arg1 == -1) && (arg2 == 63))   /* (ash -1 63): most-negative-fixnum */
	return(S7_INT64_MIN);
      out_of_range(sc, sc->ash_symbol, int_two, wrap_integer(sc, arg2), its_too_large_string);
    }
  if (arg2 < -S7_INT_BITS)
    return((arg1 < 0) ? -1 : 0);        /* (ash -31 -100) */

  /* I can't see any point in protecting this: (ash 9223372036854775807 1) -> -2, but anyone using ash must know something about bits */
  if (arg2 < 0)
    return(arg1 >> -arg2);
  if (arg1 < 0)
    {
      uint64_t z = (uint64_t)arg1;
      return((s7_int)(z << arg2));
    }
  return(arg1 << arg2);
}

static s7_pointer g_ash(s7_scheme *sc, s7_pointer args)
{
  #define H_ash "(ash i1 i2) returns i1 shifted right or left i2 times, i1 << i2, (ash 1 3) -> 8, (ash 8 -3) -> 1"
  #define Q_ash sc->pcl_i

#if WITH_GMP
  /* here, as in expt, there are cases like (ash 1 63) which need to be handled as bignums */
  s7_pointer p0 = car(args), p1 = cadr(args);

  /* here, as in expt, there are cases like (ash 1 63) which need to be bignums so there's no easy way to tell when it's safe to drop into g_ash instead */
  if ((s7_is_integer(p0)) && /* this includes bignum ints... */
      (s7_is_integer(p1)))
    {
      s7_int shift;
      bool p0_is_big = is_big_number(p0);
      int32_t p0_compared_to_zero = 0;

      if (p0_is_big)
	p0_compared_to_zero = mpz_cmp_ui(big_integer(p0), 0);
      else
	if (integer(p0) > 0)
	  p0_compared_to_zero = 1;
	else p0_compared_to_zero = (integer(p0) < 0) ? -1 : 0;

      if (p0_compared_to_zero == 0)
	return(int_zero);

      if (is_big_number(p1))
	{
	  if (!mpz_fits_sint_p(big_integer(p1)))
	    {
	      if (mpz_cmp_ui(big_integer(p1), 0) > 0)
		out_of_range(sc, sc->ash_symbol, int_two, p1, its_too_large_string);

	      /* here if p0 is negative, we need to return -1 */
	      return((p0_compared_to_zero == 1) ? int_zero : minus_one);
	    }
	  shift = mpz_get_si(big_integer(p1));
	}
      else
	{
	  shift = integer(p1);
	  if (shift < S7_INT32_MIN)
	    return((p0_compared_to_zero == 1) ? int_zero : minus_one);
	}
      if (shift > S7_INT32_MAX)
	out_of_range(sc, sc->ash_symbol, int_two, p1, its_too_large_string); /* gmp calls abort if overflow here */

      if (is_t_big_integer(p0))
	mpz_set(sc->mpz_1, big_integer(p0));
      else mpz_set_si(sc->mpz_1, integer(p0));

      if (shift > 0)     /* left */
	mpz_mul_2exp(sc->mpz_1, sc->mpz_1, shift);
      else
	if (shift < 0) /* right */
	  mpz_fdiv_q_2exp(sc->mpz_1, sc->mpz_1, (uint32_t)(-shift));

      return(mpz_to_integer(sc, sc->mpz_1));
    }
  /* else fall through */
#endif
  s7_pointer x = car(args), y = cadr(args);

  if (!s7_is_integer(x))
    return(method_or_bust(sc, x, sc->ash_symbol, args, T_INTEGER, 1));
  if (!s7_is_integer(y))
    return(method_or_bust(sc, y, sc->ash_symbol, args, T_INTEGER, 2));
  return(make_integer(sc, c_ash(sc, s7_integer_clamped_if_gmp(sc, x), s7_integer_clamped_if_gmp(sc, y))));
}

#if (!WITH_GMP)
  static s7_int ash_i_7ii(s7_scheme *sc, s7_int i1, s7_int i2) {return(c_ash(sc, i1, i2));}
#endif
static s7_int lsh_i_ii_unchecked(s7_int i1, s7_int i2) {return(i1 << i2);} /* this may need gmp special handling, and out-of-range as in c_ash */
static s7_int rsh_i_ii_unchecked(s7_int i1, s7_int i2) {return(i1 >> (-i2));}
static s7_int rsh_i_i2_direct(s7_int i1, s7_int unused_i2) {return(i1 >> 1);}


/* -------------------------------- random-state -------------------------------- */
/* random numbers.  The simple version used in clm.c is probably adequate, but here I'll use Marsaglia's MWC algorithm.
 *     (random num) -> a number (0..num), if num == 0 return 0, use global default state
 *     (random num state) -> same but use this state
 *     (random-state seed) -> make a new state
 *   to save the current seed, use copy, to save it across load, random-state->list and list->random-state.
 *   random-state? returns #t if its arg is one of these guys
 */

s7_pointer s7_random_state(s7_scheme *sc, s7_pointer args)
{
  #define H_random_state "(random-state seed (carry plausible-default)) returns a new random number state initialized with 'seed'. \
Pass this as the second argument to 'random' to get a repeatable random number sequence:\n\
    (let ((seed (random-state 1234))) (random 1.0 seed))"
  #define Q_random_state s7_make_circular_signature(sc, 1, 2, sc->is_random_state_symbol, sc->is_integer_symbol)

#if WITH_GMP
  s7_pointer r, seed = car(args);
  if (!s7_is_integer(seed))
    return(method_or_bust_one_arg(sc, seed, sc->random_state_symbol, args, T_INTEGER));

  if (is_t_integer(seed))
    seed = s7_int_to_big_integer(sc, integer(seed));

  new_cell(sc, r, T_RANDOM_STATE);
  gmp_randinit_default(random_gmp_state(r));            /* Mersenne twister */
  gmp_randseed(random_gmp_state(r), big_integer(seed)); /* this is ridiculously slow! */
  add_big_random_state(sc, r);
  return(r);
#else
  s7_pointer r1 = car(args), r2, p;
  s7_int i1, i2;

  if (!s7_is_integer(r1))
    return(method_or_bust(sc, r1, sc->random_state_symbol, args, T_INTEGER, 1));
  i1 = integer(r1);
  if (i1 < 0)
    out_of_range(sc, sc->random_state_symbol, int_one, r1, its_negative_string);

  if (is_null(cdr(args)))
    {
      new_cell(sc, p, T_RANDOM_STATE);
      random_seed(p) = (uint64_t)i1;
      random_carry(p) = 1675393560;                          /* should this be dependent on the seed? */
      return(p);
    }

  r2 = cadr(args);
  if (!s7_is_integer(r2))
    return(method_or_bust(sc, r2, sc->random_state_symbol, args, T_INTEGER, 2));
  i2 = integer(r2);
  if (i2 < 0)
    out_of_range(sc, sc->random_state_symbol, int_two, r2, its_negative_string);

  new_cell(sc, p, T_RANDOM_STATE);
  random_seed(p) = (uint64_t)i1;
  random_carry(p) = (uint64_t)i2;
  return(p);
#endif
}

#define g_random_state s7_random_state

static s7_pointer rng_copy(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  return(sc->F); /* I can't find a way to copy a gmp random generator */
#else
  s7_pointer new_r, obj = car(args);
  if (!is_random_state(obj)) return(sc->F);
  new_cell(sc, new_r, T_RANDOM_STATE);
  random_seed(new_r) = random_seed(obj);
  random_carry(new_r) = random_carry(obj);
  return(new_r);
#endif
}


/* -------------------------------- random-state? -------------------------------- */
static s7_pointer g_is_random_state(s7_scheme *sc, s7_pointer args)
{
  #define H_is_random_state "(random-state? obj) returns #t if obj is a random-state object (from random-state)."
  #define Q_is_random_state sc->pl_bt
  check_boolean_method(sc, is_random_state, sc->is_random_state_symbol, args);
}

bool s7_is_random_state(s7_pointer p) {return(type(p) == T_RANDOM_STATE);}


/* -------------------------------- random-state->list -------------------------------- */
s7_pointer s7_random_state_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_random_state_to_list "(random-state->list r) returns the random state object as a list.\
You can later apply random-state to this list to continue a random number sequence from any point."
  #define Q_random_state_to_list s7_make_signature(sc, 2, (WITH_GMP) ? sc->is_list_symbol : sc->is_pair_symbol, sc->is_random_state_symbol)

#if WITH_GMP
  if ((is_pair(args)) &&
      (!is_random_state(car(args))))
    return(method_or_bust_with_type(sc, car(args), sc->random_state_to_list_symbol, args, a_random_state_object_string, 1));
  return(sc->nil);
#else
  s7_pointer r = (is_null(args)) ? sc->default_rng : car(args);
  if (!is_random_state(r))
    return(method_or_bust_with_type(sc, r, sc->random_state_to_list_symbol, args, a_random_state_object_string, 1));
  return(list_2(sc, make_integer(sc, random_seed(r)), make_integer(sc, random_carry(r))));
#endif
}

#define g_random_state_to_list s7_random_state_to_list

void s7_set_default_random_state(s7_scheme *sc, s7_int seed, s7_int carry)
{
#if (!WITH_GMP)
  s7_pointer p;
  new_cell(sc, p, T_RANDOM_STATE);
  random_seed(p) = (uint64_t)seed;
  random_carry(p) = (uint64_t)carry;
  sc->default_rng = p;
#endif
}


/* -------------------------------- random -------------------------------- */
#if WITH_GMP
static double next_random(s7_scheme *sc)
#else
static double next_random(s7_pointer r)
#endif
{
#if (!WITH_GMP)
  /* The multiply-with-carry generator for 32-bit integers:
   *        x(n)=a*x(n-1) + carry mod 2^32
   * Choose multiplier a from this list:
   *   1791398085 1929682203 1683268614 1965537969 1675393560 1967773755 1517746329 1447497129 1655692410 1606218150
   *   2051013963 1075433238 1557985959 1781943330 1893513180 1631296680 2131995753 2083801278 1873196400 1554115554
   * ( or any 'a' for which both a*2^32-1 and a*2^31-1 are prime)
   */
  double result;
  uint64_t temp;
  #define RAN_MULT 2131995753UL

  temp = random_seed(r) * RAN_MULT + random_carry(r);
  random_seed(r) = (temp & 0xffffffffUL);
  random_carry(r) = (temp >> 32);
  result = (double)((uint32_t)(random_seed(r))) / 4294967295.5;
  /* divisor was 2^32-1 = 4294967295.0, but somehow this can round up once in a billion tries?
   *   do we want the double just less than 2^32?
   * can the multiply-add+logand above return 0? I'm getting 0's from (random (expt 2 62))
   */

  /* (let ((mx 0) (mn 1000)) (do ((i 0 (+ i 1))) ((= i 10000)) (let ((val (random 123))) (set! mx (max mx val)) (set! mn (min mn val)))) (list mn mx)) */
  return(result);
#else
  mpfr_urandomb(sc->mpfr_1, random_gmp_state(sc->default_rng));
  return(mpfr_get_d(sc->mpfr_1, MPFR_RNDN));
#endif
}

static s7_pointer g_random(s7_scheme *sc, s7_pointer args)
{
  #define H_random "(random num (state #f)) returns a random number of the same type as num between zero and num, equalling num only if num is zero"
  #define Q_random s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_number_symbol, sc->is_random_state_symbol)
  s7_pointer r, num;

  /* if we disallow (random 0) the programmer has to protect every call on random with (if (eqv? x 0) 0 (random x)).  If
   *   we claim we're using a half-open interval, then we should also disallow (random 0.0); otherwise the following
   *   must be true: (let* ((x 0.0) (y (random x))) (and (>= y 0.0) (< y x))).  The definition above is consistent
   *   with (random 0) -> 0, simpler to use in practice, and certainly no worse than (/ 0 0) -> 1.
   */
  if (is_null(cdr(args)))
    r = sc->default_rng;
  else
    {
      r = cadr(args);
      if (!is_random_state(r))
	return(method_or_bust_with_type(sc, r, sc->random_symbol, args, a_random_state_object_string, 2));
    }
  num = car(args);
  switch (type(num))
    {
#if (!WITH_GMP)
    case T_INTEGER:
      return(make_integer(sc, (s7_int)(integer(num) * next_random(r))));

    case T_RATIO:
      {
	s7_double x = fraction(num), error;
	s7_int numer = 0, denom = 1;
	/* the error here needs to take the size of the fraction into account.  Otherwise, if
	 *    error is (say) 1e-6 and the fraction is (say) 9000000/9223372036854775807,
	 *    c_rationalize will always return 0.  But even that isn't foolproof:
	 *    (random 1/562949953421312) -> 1/376367230475000
	 */
	if ((x < 1.0e-10) && (x > -1.0e-10))
	  {
	    /* 1e-12 is not tight enough:
	     *    (random 1/2251799813685248) -> 1/2250240579436280
	     *    (random -1/4503599627370496) -> -1/4492889778435526
	     *    (random 1/140737488355328) -> 1/140730223985746
	     *    (random -1/35184372088832) -> -1/35183145492420
	     *    (random -1/70368744177664) -> -1/70366866392738
	     *    (random 1/4398046511104) -> 1/4398033095756
	     *    (random 1/137438953472) -> 1/137438941127
	     */
	    if (numerator(num) < -10)
	      numer = -(s7_int)(floor(-numerator(num) * next_random(r)));
	    else
	      if (numerator(num) > 10)
		numer = (s7_int)floor(numerator(num) * next_random(r));
	      else
		{
		  int64_t diff = S7_INT64_MAX - denominator(num);
		  numer = numerator(num);
		  if (diff < 100)
		    return(make_ratio(sc, numer, denominator(num)));
		  denom = denominator(num) + (s7_int)floor(diff * next_random(r));
		  return(make_ratio_with_div_check(sc, sc->random_symbol, numer, denom));
		}
	    return(make_ratio(sc, numer, denominator(num)));
	  }
	error = ((x < 1e-6) && (x > -1e-6)) ? 1e-18 : 1e-12;
	c_rationalize(x * next_random(r), error, &numer, &denom);
	return(make_ratio(sc, numer, denom));
      }

    case T_REAL:
      return(make_real(sc, real(num) * next_random(r)));

    case T_COMPLEX:
      return(s7_make_complex(sc, real_part(num) * next_random(r), imag_part(num) * next_random(r)));

#else

    case T_INTEGER:
      if (integer(num) == 0) return(int_zero);
      mpz_set_si(sc->mpz_1, integer(num));
      mpz_urandomm(sc->mpz_1, random_gmp_state(r), sc->mpz_1);
      if (integer(num) < 0) mpz_neg(sc->mpz_1, sc->mpz_1);
      return(make_integer(sc, mpz_get_si(sc->mpz_1)));

    case T_BIG_INTEGER:
      if (mpz_cmp_si(big_integer(num), 0) == 0) return(int_zero);
      mpz_urandomm(sc->mpz_1, random_gmp_state(r), big_integer(num));
      /* this does not work if num is a negative number -- you get positive results. so check num for sign, and negate result if necessary */
      if (mpz_cmp_ui(big_integer(num), 0) < 0)
	mpz_neg(sc->mpz_1, sc->mpz_1);
      return(mpz_to_integer(sc, sc->mpz_1));

    case T_RATIO:
      mpfr_urandomb(sc->mpfr_1, random_gmp_state(r));
      mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
      mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, sc->mpq_1, MPFR_RNDN);
      mpfr_mul_d(sc->mpfr_2, sc->mpfr_1, sc->default_rationalize_error, MPFR_RNDN);
      return(big_rationalize(sc, set_plist_2(sc, mpfr_to_big_real(sc, sc->mpfr_1), mpfr_to_big_real(sc, sc->mpfr_2))));

    case T_BIG_RATIO:
      mpfr_urandomb(sc->mpfr_1, random_gmp_state(r));
      mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, big_ratio(num), MPFR_RNDN);
      mpfr_mul_d(sc->mpfr_2, sc->mpfr_1, sc->default_rationalize_error, MPFR_RNDN);
      return(big_rationalize(sc, set_plist_2(sc, mpfr_to_big_real(sc, sc->mpfr_1), mpfr_to_big_real(sc, sc->mpfr_2))));

    case T_REAL:
      mpfr_urandomb(sc->mpfr_1, random_gmp_state(r));
      mpfr_mul_d(sc->mpfr_1, sc->mpfr_1, real(num), MPFR_RNDN);
      return(make_real(sc, mpfr_get_d(sc->mpfr_1, MPFR_RNDN)));

    case T_BIG_REAL:
      mpfr_urandomb(sc->mpfr_1, random_gmp_state(r));
      mpfr_mul(sc->mpfr_1, sc->mpfr_1, big_real(num), MPFR_RNDN);
      return(mpfr_to_big_real(sc, sc->mpfr_1));

    case T_COMPLEX:
      mpc_urandom(sc->mpc_1, random_gmp_state(r));
      mpfr_mul_d(mpc_realref(sc->mpc_1), mpc_realref(sc->mpc_1), real_part(num), MPFR_RNDN);
      mpfr_mul_d(mpc_imagref(sc->mpc_1), mpc_imagref(sc->mpc_1), imag_part(num), MPFR_RNDN);
      return(s7_make_complex(sc, mpfr_get_d(mpc_realref(sc->mpc_1), MPFR_RNDN), mpfr_get_d(mpc_imagref(sc->mpc_1), MPFR_RNDN)));

    case T_BIG_COMPLEX:
      mpc_urandom(sc->mpc_1, random_gmp_state(r));
      mpfr_mul(mpc_realref(sc->mpc_1), mpc_realref(sc->mpc_1), mpc_realref(big_complex(num)), MPFR_RNDN);
      mpfr_mul(mpc_imagref(sc->mpc_1), mpc_imagref(sc->mpc_1), mpc_imagref(big_complex(num)), MPFR_RNDN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_with_type(sc, num, sc->random_symbol, args, a_number_string, 1));
    }
  return(sc->F);
}

s7_double s7_random(s7_scheme *sc, s7_pointer state)
{
#if WITH_GMP
  mpfr_set_ui(sc->mpfr_1, 1, MPFR_RNDN);
  mpfr_urandomb(sc->mpfr_1, random_gmp_state((state) ? state : sc->default_rng));
  return((s7_double)mpfr_get_d(sc->mpfr_1, MPFR_RNDN));
#else
  return(next_random((state) ? state : sc->default_rng));
#endif
}

static s7_double random_d_7d(s7_scheme *sc, s7_double x)
{
#if WITH_GMP
  return(real(g_random(sc, set_plist_1(sc, wrap_real(sc, x)))));
#else
  return(x * next_random(sc->default_rng));
#endif
}

static s7_int random_i_7i(s7_scheme *sc, s7_int i)
{
#if WITH_GMP
  return(integer(g_random(sc, set_plist_1(sc, wrap_integer(sc, i)))));
#else
  return((s7_int)(i * next_random(sc->default_rng)));
#endif
}

static s7_pointer g_random_i(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  return(g_random(sc, args));
#else
  return(make_integer(sc, (s7_int)(integer(car(args)) * next_random(sc->default_rng))));
#endif
}

static s7_pointer g_random_f(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  return(g_random(sc, args));
#else
  return(make_real(sc, real(car(args)) * next_random(sc->default_rng)));
#endif
}

static s7_pointer g_random_1(s7_scheme *sc, s7_pointer args)
{
#if (!WITH_GMP)
  s7_pointer num = car(args), r = sc->default_rng;
  if (is_t_integer(num))
    return(make_integer(sc, (s7_int)(integer(num) * next_random(r))));
  if (is_t_real(num))
    return(make_real(sc, real(num) * next_random(r)));
#endif
  return(g_random(sc, args));
}

static s7_pointer random_p_p(s7_scheme *sc, s7_pointer num)
{
#if (!WITH_GMP)
  if (is_t_integer(num))
    return(make_integer(sc, (s7_int)(integer(num) * next_random(sc->default_rng))));
  if (is_t_real(num))
    return(make_real(sc, real(num) * next_random(sc->default_rng)));
#endif
  return(g_random(sc, set_plist_1(sc, num)));
}

static s7_pointer random_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if ((ops) && (args == 1))
    {
      s7_pointer arg1 = cadr(expr);
      if (is_t_integer(arg1))
	return(sc->random_i);
      return((is_t_real(arg1)) ? sc->random_f : sc->random_1);
    }
  return(f);
}

static s7_pointer g_add_i_random(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  return(add_p_pp(sc, car(args), random_p_p(sc, cadadr(args))));
#else
  s7_int x = integer(car(args)), y = opt3_int(args); /* cadadr */
  return(make_integer(sc, x + (s7_int)(y * next_random(sc->default_rng)))); /* (+ -1 (random 1)) -- placement of the (s7_int) cast matters! */
#endif
}


/* -------------------------------- characters -------------------------------- */
/* -------------------------------- char<->integer -------------------------------- */
static s7_pointer g_char_to_integer(s7_scheme *sc, s7_pointer args)
{
  #define H_char_to_integer "(char->integer c) converts the character c to an integer"
  #define Q_char_to_integer s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_char_symbol)

  if (!is_character(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->char_to_integer_symbol, args, T_CHARACTER));
  return(small_int(character(car(args))));
}

static s7_int char_to_integer_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_character(p))
    return(integer(method_or_bust_p(sc, p, sc->char_to_integer_symbol, T_CHARACTER)));
  return(character(p));
}

static s7_pointer char_to_integer_p_p(s7_scheme *sc, s7_pointer p)
{
  if (!is_character(p))
    return(method_or_bust_p(sc, p, sc->char_to_integer_symbol, T_CHARACTER));
  return(make_integer(sc, character(p)));
}

static s7_pointer integer_to_char_p_p(s7_scheme *sc, s7_pointer x)
{
  s7_int ind;
  if (!s7_is_integer(x))
    return(method_or_bust_p(sc, x, sc->integer_to_char_symbol, T_INTEGER));
  ind = s7_integer_clamped_if_gmp(sc, x);
  if ((ind >= 0) && (ind < NUM_CHARS))
    return(chars[(uint8_t)ind]);
  return(s7_out_of_range_error(sc, "integer->char", 1, x, "it doen't fit in an unsigned byte"));
}

static s7_pointer g_integer_to_char(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_to_char "(integer->char i) converts the non-negative integer i to a character"
  #define Q_integer_to_char s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_integer_symbol)
  return(integer_to_char_p_p(sc, car(args)));
}

static s7_pointer integer_to_char_p_i(s7_scheme *sc, s7_int ind)
{
  if ((ind >= 0) && (ind < NUM_CHARS))
    return(chars[(uint8_t)ind]);
  return(s7_out_of_range_error(sc, "integer->char", 1, wrap_integer(sc, ind), "it doen't fit in an unsigned byte")); /* int2 s7_out... uses 1 */
}


static uint8_t uppers[256], lowers[256];
static void init_uppers(void)
{
  for (int32_t i = 0; i < 256; i++)
    {
      uppers[i] = (uint8_t)toupper(i);
      lowers[i] = (uint8_t)tolower(i);
    }
}

static void init_chars(void)
{
  s7_cell *cells = (s7_cell *)Calloc(NUM_CHARS + 1, sizeof(s7_cell));

  chars = (s7_pointer *)Malloc((NUM_CHARS + 1) * sizeof(s7_pointer)); /* chars is declared far above */
  chars[0] = &cells[0];
  eof_object = chars[0];
  set_full_type(eof_object, T_EOF | T_IMMUTABLE | T_UNHEAP);
  eof_name_length(eof_object) = 6;
  eof_name(eof_object) = "#<eof>";
  chars++;                    /* now chars[EOF] == chars[-1] == #<eof> */
  cells++;

  for (int32_t i = 0; i < NUM_CHARS; i++)
    {
      s7_pointer cp = &cells[i];
      uint8_t c = (uint8_t)i;

      set_type_bit(cp, T_IMMUTABLE | T_CHARACTER | T_UNHEAP);
      set_optimize_op(cp, OP_CONSTANT);
      character(cp) = c;
      upper_character(cp) = (uint8_t)toupper(i);
      is_char_alphabetic(cp) = (bool)isalpha(i);
      is_char_numeric(cp) = (bool)isdigit(i);
      is_char_whitespace(cp) = white_space[i];
      is_char_uppercase(cp) = (((bool)isupper(i)) || ((i >= 192) && (i < 208)));
      is_char_lowercase(cp) = (bool)islower(i);
      chars[i] = cp;

      #define make_character_name(S) memcpy((void *)(&(character_name(cp))), (const void *)(S), character_name_length(cp) = strlen(S))
      switch (c)
	{
	case ' ':	 make_character_name("#\\space");     break;
	case '\n':       make_character_name("#\\newline");   break;
	case '\r':       make_character_name("#\\return");    break;
	case '\t':       make_character_name("#\\tab");       break;
	case '\0':       make_character_name("#\\null");      break;
	case (char)0x1b: make_character_name("#\\escape");    break;
	case (char)0x7f: make_character_name("#\\delete");    break;
	case (char)7:    make_character_name("#\\alarm");     break;
	case (char)8:    make_character_name("#\\backspace"); break;
	default:
          #define P_SIZE 12
	  character_name_length(cp) = snprintf((char *)(&(character_name(cp))), P_SIZE, ((c < 32) || (c >= 127)) ? "#\\x%x" : "#\\%c", c);
	  break;
	}}
}


/* -------------------------------- char-upcase, char-downcase ----------------------- */
static s7_pointer char_upcase_p_p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    return(method_or_bust_p(sc, c, sc->char_upcase_symbol, T_CHARACTER));
  return(chars[upper_character(c)]);
}

static s7_pointer char_upcase_p_p_unchecked(s7_scheme *unused_sc, s7_pointer c) {return(chars[upper_character(c)]);}

static s7_pointer g_char_upcase(s7_scheme *sc, s7_pointer args)
{
  #define H_char_upcase "(char-upcase c) converts the character c to upper case"
  #define Q_char_upcase sc->pcl_c
  return(char_upcase_p_p(sc, car(args)));
}

static s7_pointer g_char_downcase(s7_scheme *sc, s7_pointer args)
{
  #define H_char_downcase "(char-downcase c) converts the character c to lower case"
  #define Q_char_downcase sc->pcl_c
  if (!is_character(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->char_downcase_symbol, args, T_CHARACTER));
  return(chars[lowers[character(car(args))]]);
}


/* -------------------------------- char-alphabetic? char-numeric? char-whitespace? -------------------------------- */
static s7_pointer g_is_char_alphabetic(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_alphabetic "(char-alphabetic? c) returns #t if the character c is alphabetic"
  #define Q_is_char_alphabetic sc->pl_bc
  if (!is_character(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->is_char_alphabetic_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_alphabetic(car(args))));
  /* isalpha returns #t for (integer->char 226) and others in that range */
}

static bool is_char_alphabetic_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_alphabetic_symbol, c, T_CHARACTER);
    /* return(method_or_bust_one_arg(sc, c, sc->is_char_alphabetic_symbol, set_plist_1(sc, c), T_CHARACTER) != sc->F); */ /* slower? see tmisc */
  return(is_char_alphabetic(c));
}

static s7_pointer is_char_alphabetic_p_p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    return(method_or_bust_one_arg(sc, c, sc->is_char_alphabetic_symbol, set_plist_1(sc, c), T_CHARACTER));
  return(make_boolean(sc, is_char_alphabetic(c)));
}

static s7_pointer g_is_char_numeric(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_numeric "(char-numeric? c) returns #t if the character c is a digit"
  #define Q_is_char_numeric sc->pl_bc

  s7_pointer arg = car(args);
  if (!is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_numeric_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_numeric(arg)));
}

static bool is_char_numeric_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_numeric_symbol, c, T_CHARACTER);
  /* return(method_or_bust_one_arg(sc, c, sc->is_char_numeric_symbol, set_plist_1(sc, c), T_CHARACTER) != sc->F); */ /* as above */
  return(is_char_numeric(c));
}

static s7_pointer is_char_numeric_p_p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    return(method_or_bust_one_arg(sc, c, sc->is_char_numeric_symbol, set_plist_1(sc, c), T_CHARACTER));
  return(make_boolean(sc, is_char_numeric(c)));
}


static s7_pointer g_is_char_whitespace(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_whitespace "(char-whitespace? c) returns #t if the character c is non-printing character"
  #define Q_is_char_whitespace sc->pl_bc

  s7_pointer arg = car(args);
  if (!is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_whitespace_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_whitespace(arg)));
}

static bool is_char_whitespace_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_whitespace_symbol, c, T_CHARACTER);
  return(is_char_whitespace(c));
}

static s7_pointer is_char_whitespace_p_p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    return(method_or_bust_one_arg(sc, c, sc->is_char_whitespace_symbol, set_plist_1(sc, c), T_CHARACTER));
  return(make_boolean(sc, is_char_whitespace(c)));
}

static s7_pointer is_char_whitespace_p_p_unchecked(s7_scheme *sc, s7_pointer c) {return(make_boolean(sc, is_char_whitespace(c)));}


/* -------------------------------- char-upper-case? char-lower-case? -------------------------------- */
static s7_pointer g_is_char_upper_case(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_upper_case "(char-upper-case? c) returns #t if the character c is in upper case"
  #define Q_is_char_upper_case sc->pl_bc

  s7_pointer arg = car(args);
  if (!is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_upper_case_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_uppercase(arg)));
}

static bool is_char_upper_case_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    return(method_or_bust_one_arg(sc, c, sc->is_char_upper_case_symbol, set_plist_1(sc, c), T_CHARACTER) != sc->F);
  return(is_char_uppercase(c));
}

static s7_pointer g_is_char_lower_case(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_lower_case "(char-lower-case? c) returns #t if the character c is in lower case"
  #define Q_is_char_lower_case sc->pl_bc

  s7_pointer arg = car(args);
  if (!is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_lower_case_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_lowercase(arg)));
}

static bool is_char_lower_case_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!is_character(c))
    return(method_or_bust_one_arg(sc, c, sc->is_char_lower_case_symbol, set_plist_1(sc, c), T_CHARACTER) != sc->F);
  return(is_char_lowercase(c));
}


/* -------------------------------- char? -------------------------------- */
static s7_pointer g_is_char(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char "(char? obj) returns #t if obj is a character"
  #define Q_is_char sc->pl_bt
  check_boolean_method(sc, is_character, sc->is_char_symbol, args);
}

static s7_pointer is_char_p_p(s7_scheme *sc, s7_pointer p) {return((is_character(p)) ? sc->T : sc->F);}

s7_pointer s7_make_character(s7_scheme *sc, uint8_t c) {return(chars[c]);}

bool s7_is_character(s7_pointer p) {return(is_character(p));}

uint8_t s7_character(s7_pointer p) {return(character(p));}


/* -------------------------------- char<? char<=? char>? char>=? char=? -------------------------------- */
static int32_t charcmp(uint8_t c1, uint8_t c2)
{
  return((c1 == c2) ? 0 : (c1 < c2) ? -1 : 1);
  /* not tolower here -- the single case is apparently supposed to be upper case
   *   this matters in a case like (char-ci<? #\_ #\e) which Guile and Gauche say is #f
   *   although (char<? #\_ #\e) is #t -- the spec does not say how to interpret this!
   */
}

static bool is_character_via_method(s7_scheme *sc, s7_pointer p)
{
  if (is_character(p))
    return(true);
  if (has_active_methods(sc, p))
    {
      s7_pointer f = find_method_with_let(sc, p, sc->is_char_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, set_plist_1(sc, p))));
    }
  return(false);
}

static s7_pointer char_with_error_check(s7_scheme *sc, s7_pointer x, s7_pointer args, s7_pointer caller)
{
  for (s7_pointer y = cdr(x); is_pair(y); y = cdr(y)) /* before returning #f, check for bad trailing arguments */
    if (!is_character_via_method(sc, car(y)))
      wrong_type_argument(sc, caller, position_of(y, args), car(y), T_CHARACTER);
  return(sc->F);
}

static s7_pointer g_char_cmp(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer y = car(args);
  if (!is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));
  for (s7_pointer x = cdr(args); is_pair(x); y = car(x), x = cdr(x))
    {
      if (!is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, set_ulist_1(sc, y, x), T_CHARACTER, position_of(x, args)));
      if (charcmp(character(y), character(car(x))) != val)
	return(char_with_error_check(sc, x, args, sym));
    }
  return(sc->T);
}

static s7_pointer g_char_cmp_not(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer y = car(args);
  if (!is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));
  for (s7_pointer x = cdr(args); is_pair(x); y = car(x), x = cdr(x))
    {
      if (!is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, set_ulist_1(sc, y, x), T_CHARACTER, position_of(x, args)));
      if (charcmp(character(y), character(car(x))) == val)
	return(char_with_error_check(sc, x, args, sym));
    }
  return(sc->T);
}

static s7_pointer g_chars_are_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_equal "(char=? char ...) returns #t if all the character arguments are equal"
  #define Q_chars_are_equal sc->pcl_bc

  s7_pointer y = car(args);
  if (!is_character(y))
    return(method_or_bust(sc, y, sc->char_eq_symbol, args, T_CHARACTER, 1));
  for (s7_pointer x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!is_character(car(x)))
	return(method_or_bust(sc, car(x), sc->char_eq_symbol, set_ulist_1(sc, y, x), T_CHARACTER, position_of(x, args)));
      if (car(x) != y)
	return(char_with_error_check(sc, x, args, sc->char_eq_symbol));
    }
  return(sc->T);
}


static s7_pointer g_chars_are_less(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_less "(char<? char ...) returns #t if all the character arguments are increasing"
  #define Q_chars_are_less sc->pcl_bc
  return(g_char_cmp(sc, args, -1, sc->char_lt_symbol));
}

static s7_pointer g_chars_are_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_greater "(char>? char ...) returns #t if all the character arguments are decreasing"
  #define Q_chars_are_greater sc->pcl_bc
  return(g_char_cmp(sc, args, 1, sc->char_gt_symbol));
}

static s7_pointer g_chars_are_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_geq "(char>=? char ...) returns #t if all the character arguments are equal or decreasing"
  #define Q_chars_are_geq sc->pcl_bc
  return(g_char_cmp_not(sc, args, -1, sc->char_geq_symbol));
}

static s7_pointer g_chars_are_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_leq "(char<=? char ...) returns #t if all the character arguments are equal or increasing"
  #define Q_chars_are_leq sc->pcl_bc
  return(g_char_cmp_not(sc, args, 1, sc->char_leq_symbol));
}

static s7_pointer g_simple_char_eq(s7_scheme *sc, s7_pointer args) {return(make_boolean(sc, car(args) == cadr(args)));} /* chooser checks types */

#define check_char2_args(Sc, Caller, P1, P2) \
  do { \
      if (!is_character(P1)) return(method_or_bust(Sc, P1, Caller, set_plist_2(Sc, P1, P2), T_CHARACTER, 1) != sc->F); \
      if (!is_character(P2)) return(method_or_bust(Sc, P2, Caller, set_plist_2(Sc, P1, P2), T_CHARACTER, 2) != sc->F); \
     } while (0)

static bool char_lt_b_unchecked(s7_pointer p1, s7_pointer p2) {return(p1 < p2);}
static bool char_lt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_lt_symbol, p1, p2);
  return(p1 < p2);
}

static bool char_leq_b_unchecked(s7_pointer p1, s7_pointer p2) {return(p1 <= p2);}
static bool char_leq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_leq_symbol, p1, p2);
  return(p1 <= p2);
}

static bool char_gt_b_unchecked(s7_pointer p1, s7_pointer p2) {return(p1 > p2);}
static bool char_gt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_gt_symbol, p1, p2);
  return(p1 > p2);
}

static bool char_geq_b_unchecked(s7_pointer p1, s7_pointer p2) {return(p1 >= p2);}
static bool char_geq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_geq_symbol, p1, p2);
  return(p1 >= p2);
}

static bool char_eq_b_unchecked(s7_pointer p1, s7_pointer p2) {return(p1 == p2);}

static bool char_eq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  if (!is_character(p1)) return(method_or_bust(sc, p1, sc->char_eq_symbol, set_plist_2(sc, p1, p2), T_CHARACTER, 1) != sc->F);
  if (p1 == p2) return(true);
  if (!is_character(p2)) return(method_or_bust(sc, p2, sc->char_eq_symbol, set_plist_2(sc, p1, p2), T_CHARACTER, 2) != sc->F);
  return(false);
}

static s7_pointer char_eq_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  if (!is_character(p1)) return(method_or_bust(sc, p1, sc->char_eq_symbol, set_plist_2(sc, p1, p2), T_CHARACTER, 1));
  if (p1 == p2) return(sc->T);
  if (!is_character(p2)) return(method_or_bust(sc, p2, sc->char_eq_symbol, set_plist_2(sc, p1, p2), T_CHARACTER, 2));
  return(sc->F);
}

static s7_pointer g_char_equal_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_character(car(args))) return(method_or_bust(sc, car(args), sc->char_eq_symbol, args, T_CHARACTER, 1));
  if (car(args) == cadr(args))
    return(sc->T);
  if (!is_character(cadr(args))) return(method_or_bust(sc, cadr(args), sc->char_eq_symbol, args, T_CHARACTER, 2));
  return(sc->F);
}

static s7_pointer g_char_less_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_character(car(args))) return(method_or_bust(sc, car(args), sc->char_lt_symbol, args, T_CHARACTER, 1));
  if (!is_character(cadr(args))) return(method_or_bust(sc, cadr(args), sc->char_lt_symbol, args, T_CHARACTER, 2));
  return(make_boolean(sc, character(car(args)) < character(cadr(args))));
}

static s7_pointer g_char_greater_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_character(car(args))) return(method_or_bust(sc, car(args), sc->char_gt_symbol, args, T_CHARACTER, 1));
  if (!is_character(cadr(args))) return(method_or_bust(sc, cadr(args), sc->char_gt_symbol, args, T_CHARACTER, 2));
  return(make_boolean(sc, character(car(args)) > character(cadr(args))));
}

static bool returns_char(s7_scheme *sc, s7_pointer arg) {return(argument_type(sc, arg) == sc->is_char_symbol);}

static s7_pointer char_equal_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (args == 2)
    {
      if (ops)
	{
	  s7_pointer arg1 = cadr(expr), arg2 = caddr(expr);
	  if ((returns_char(sc, arg1)) && (returns_char(sc, arg2)))
	    return(sc->simple_char_eq);
	}
      return(sc->char_equal_2);
    }
  return(f);
}

static s7_pointer char_less_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer unused_expr, bool unused_ops)
{
  return((args == 2) ? sc->char_less_2 : f);
}

static s7_pointer char_greater_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer unused_expr, bool unused_ops)
{
  return((args == 2) ? sc->char_greater_2 : f);
}


/* -------------------------------- char-ci<? char-ci<=? char-ci>? char-ci>=? char-ci=? -------------------------------- */
#if (!WITH_PURE_S7)
static s7_pointer g_char_cmp_ci(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer y = car(args);
  if (!is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));

  for (s7_pointer x = cdr(args); is_pair(x); y = car(x), x = cdr(x))
    {
      if (!is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, set_ulist_1(sc, y, x), T_CHARACTER, position_of(x, args)));
      if (charcmp(upper_character(y), upper_character(car(x))) != val)
	return(char_with_error_check(sc, x, args, sym));
    }
  return(sc->T);
}

static s7_pointer g_char_cmp_ci_not(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer y = car(args);
  if (!is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));
  for (s7_pointer x = cdr(args); is_pair(x); y = car(x), x = cdr(x))
    {
      if (!is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, set_ulist_1(sc, y, x), T_CHARACTER, position_of(x, args)));
      if (charcmp(upper_character(y), upper_character(car(x))) == val)
	return(char_with_error_check(sc, x, args, sym));
    }
  return(sc->T);
}

static s7_pointer g_chars_are_ci_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_equal "(char-ci=? char ...) returns #t if all the character arguments are equal, ignoring case"
  #define Q_chars_are_ci_equal sc->pcl_bc
  return(g_char_cmp_ci(sc, args, 0, sc->char_ci_eq_symbol));
}

static s7_pointer g_chars_are_ci_less(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_less "(char-ci<? char ...) returns #t if all the character arguments are increasing, ignoring case"
  #define Q_chars_are_ci_less sc->pcl_bc
  return(g_char_cmp_ci(sc, args, -1, sc->char_ci_lt_symbol));
}

static s7_pointer g_chars_are_ci_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_greater "(char-ci>? char ...) returns #t if all the character arguments are decreasing, ignoring case"
  #define Q_chars_are_ci_greater sc->pcl_bc
  return(g_char_cmp_ci(sc, args, 1, sc->char_ci_gt_symbol));
}

static s7_pointer g_chars_are_ci_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_geq "(char-ci>=? char ...) returns #t if all the character arguments are equal or decreasing, ignoring case"
  #define Q_chars_are_ci_geq sc->pcl_bc
  return(g_char_cmp_ci_not(sc, args, -1, sc->char_ci_geq_symbol));
}

static s7_pointer g_chars_are_ci_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_leq "(char-ci<=? char ...) returns #t if all the character arguments are equal or increasing, ignoring case"
  #define Q_chars_are_ci_leq sc->pcl_bc
  return(g_char_cmp_ci_not(sc, args, 1, sc->char_ci_leq_symbol));
}


static bool char_ci_lt_b_unchecked(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) < upper_character(p2));}
static bool char_ci_lt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_lt_symbol, p1, p2);
  return(upper_character(p1) < upper_character(p2));
}

static bool char_ci_leq_b_unchecked(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) <= upper_character(p2));}
static bool char_ci_leq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_leq_symbol, p1, p2);
  return(upper_character(p1) <= upper_character(p2));
}

static bool char_ci_gt_b_unchecked(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) > upper_character(p2));}
static bool char_ci_gt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_gt_symbol, p1, p2);
  return(upper_character(p1) > upper_character(p2));
}

static bool char_ci_geq_b_unchecked(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) >= upper_character(p2));}
static bool char_ci_geq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_geq_symbol, p1, p2);
  return(upper_character(p1) >= upper_character(p2));
}

static bool char_ci_eq_b_unchecked(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) == upper_character(p2));}
static bool char_ci_eq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_eq_symbol, p1, p2);
  return(upper_character(p1) == upper_character(p2));
}

#endif /* not pure s7 */


/* -------------------------------- char-position -------------------------------- */
static s7_pointer g_char_position(s7_scheme *sc, s7_pointer args)
{
  #define H_char_position "(char-position char-or-str str (start 0)) returns the position of the first occurrence of char in str, or #f"
  #define Q_char_position s7_make_signature(sc, 4, \
                            s7_make_signature(sc, 2, sc->is_integer_symbol, sc->not_symbol), \
                            s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_string_symbol), \
                            sc->is_string_symbol, sc->is_integer_symbol)
  const char *porig, *pset;
  s7_int start, pos, len; /* not "int" because start arg might be most-negative-fixnum */
  s7_pointer arg1 = car(args), arg2;

  if ((!is_character(arg1)) &&
      (!is_string(arg1)))
    return(method_or_bust(sc, arg1, sc->char_position_symbol, args, T_CHARACTER, 1));

  arg2 = cadr(args);
  if (!is_string(arg2))
    return(method_or_bust(sc, arg2, sc->char_position_symbol, args, T_STRING, 2));

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	return(method_or_bust(sc, arg3, sc->char_position_symbol, args, T_INTEGER, 3));
      start = s7_integer_clamped_if_gmp(sc, arg3);
      if (start < 0)
	wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, arg3, a_non_negative_integer_string);
    }
  else start = 0;

  porig = string_value(arg2);
  len = string_length(arg2);
  if (start >= len) return(sc->F);

  if (is_character(arg1))
    {
      char c = character(arg1);
      const char *p = strchr((const char *)(porig + start), (int)c); /* use strchrnul in Gnu C to catch embedded null case */
      return((p) ? make_integer(sc, p - porig) : sc->F);
    }
  if (string_length(arg1) == 0)
    return(sc->F);
  pset = string_value(arg1);

  pos = strcspn((const char *)(porig + start), (const char *)pset);
  if ((pos + start) < len)
    return(make_integer(sc, pos + start));

  /* if the string has an embedded null, we can get erroneous results here --
   *   perhaps check for null at pos+start?  What about a searched-for string that also has embedded nulls?
   */
  return(sc->F);
}

static s7_pointer char_position_p_ppi(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int start)
{
  /* p1 is char, p2 is string */
  const char *porig, *p;
  s7_int len;
  char c;

  if (!is_string(p2))
    wrong_type_argument(sc, sc->char_position_symbol, 2, p2, T_STRING);
  if (start < 0)
    wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, make_integer(sc, start), a_non_negative_integer_string);

  c = character(p1);
  len = string_length(p2);
  porig = string_value(p2);
  if (start >= len) return(sc->F);
  p = strchr((const char *)(porig + start), (int)c);
  if (p) return(make_integer(sc, p - porig));
  return(sc->F);
}

static s7_pointer g_char_position_csi(s7_scheme *sc, s7_pointer args)
{
  /* assume char arg1, no end */
  const char *porig, *p;
  char c = character(car(args));
  s7_pointer arg2 = cadr(args);
  s7_int start, len;

  if (!is_string(arg2))
    return(g_char_position(sc, args));

  len = string_length(arg2); /* can't return #f here if len==0 -- need start error check first */
  porig = string_value(arg2);

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	return(g_char_position(sc, args));
      start = s7_integer_clamped_if_gmp(sc, arg3);
      if (start < 0)
	wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, arg3, a_non_negative_integer_string);
      if (start >= len) return(sc->F);
    }
  else start = 0;

  if (len == 0) return(sc->F);
  p = strchr((const char *)(porig + start), (int)c);
  return((p) ? make_integer(sc, p - porig) : sc->F);
}

static s7_pointer char_position_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (!ops) return(f);
  if ((is_character(cadr(expr))) && ((args == 2) || (args == 3)))
    return(sc->char_position_csi);
  return(f);
}


/* -------------------------------- string-position -------------------------------- */
static s7_pointer g_string_position(s7_scheme *sc, s7_pointer args)
{
  #define H_string_position "(string-position str1 str2 (start 0)) returns the starting position of str1 in str2 or #f"
  #define Q_string_position s7_make_signature(sc, 4, \
                              s7_make_signature(sc, 2, sc->is_integer_symbol, sc->not_symbol), \
                              sc->is_string_symbol, sc->is_string_symbol, sc->is_integer_symbol)
  const char *s1, *s2, *p2;
  s7_int start = 0;
  s7_pointer s1p = car(args), s2p;

  if (!is_string(s1p))
    return(method_or_bust(sc, s1p, sc->string_position_symbol, args, T_STRING, 1));

  s2p = cadr(args);
  if (!is_string(s2p))
    return(method_or_bust(sc, s2p, sc->string_position_symbol, args, T_STRING, 2));

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	return(method_or_bust(sc, arg3, sc->string_position_symbol, args, T_INTEGER, 3));
      start = s7_integer_clamped_if_gmp(sc, arg3);
      if (start < 0)
	wrong_type_argument_with_type(sc, sc->string_position_symbol, 3, caddr(args), a_non_negative_integer_string);
    }

  if (string_length(s1p) == 0)
    return(sc->F);
  s1 = string_value(s1p);
  s2 = string_value(s2p);
  if (start >= string_length(s2p))
    return(sc->F);

  p2 = strstr((const char *)(s2 + start), s1);
  return((p2) ? make_integer(sc, p2 - s2) : sc->F);
}


/* -------------------------------- strings -------------------------------- */

static s7_pointer nil_string; /* permanent "" */

bool s7_is_string(s7_pointer p) {return(is_string(p));}

const char *s7_string(s7_pointer p) {return(string_value(p));}

s7_int s7_string_length(s7_pointer str) {return(string_length(str));}

s7_pointer s7_make_string_with_length(s7_scheme *sc, const char *str, s7_int len) {return(make_string_with_length(sc, str, len));}

#define NUM_STRING_WRAPPERS 8

static s7_pointer wrap_string(s7_scheme *sc, const char *str, s7_int len)
{
  s7_pointer x = car(sc->string_wrappers);
  sc->string_wrappers = cdr(sc->string_wrappers);
  string_value(x) = (char *)str;
  string_length(x) = len;
  return(x);
}

s7_pointer s7_make_string_wrapper(s7_scheme *sc, const char *str) {return(wrap_string(sc, str, safe_strlen(str)));}

static Inline s7_pointer inline_make_empty_string(s7_scheme *sc, s7_int len, char fill)
{
  s7_pointer x;
  block_t *b;
  if (len == 0) return(nil_string);
  new_cell(sc, x, T_STRING);
  b = mallocate(sc, len + 1);
  string_block(x) = b;
  string_value(x) = (char *)block_data(b);
  if (fill != '\0')
    local_memset((void *)(string_value(x)), fill, len);
  string_value(x)[len] = 0;
  string_hash(x) = 0;
  string_length(x) = len;
  add_string(sc, x);
  return(x);
}

static s7_pointer make_empty_string(s7_scheme *sc, s7_int len, char fill) {return(inline_make_empty_string(sc, len, fill));}

s7_pointer s7_make_string(s7_scheme *sc, const char *str) {return((str) ? make_string_with_length(sc, str, safe_strlen(str)) : nil_string);}

static char *make_permanent_c_string(s7_scheme *sc, const char *str)
{
  s7_int len = safe_strlen(str);
  char *x = (char *)permalloc(sc, len + 1);
  memcpy((void *)x, (void *)str, len);
  x[len] = 0;
  return(x);
}

s7_pointer s7_make_permanent_string(s7_scheme *sc, const char *str)
{
  /* for the symbol table which is never GC'd */
  s7_pointer x;
  s7_int len;
  if (!str) return(nil_string);
  x = alloc_pointer(sc);
  set_full_type(x, T_STRING | T_IMMUTABLE | T_UNHEAP);
  set_optimize_op(x, OP_CONSTANT);
  len = safe_strlen(str);
  string_length(x) = len;
  string_block(x) = NULL;
  string_value(x) = (char *)permalloc(sc, len + 1);
  memcpy((void *)string_value(x), (void *)str, len);
  string_value(x)[len] = 0;
  string_hash(x) = 0;
  return(x);
}

static s7_pointer g_is_string(s7_scheme *sc, s7_pointer args)
{
  #define H_is_string "(string? obj) returns #t if obj is a string"
  #define Q_is_string sc->pl_bt
  check_boolean_method(sc, is_string, sc->is_string_symbol, args);
}

static s7_pointer make_permanent_string(const char *str)
{
  s7_pointer x = (s7_pointer)Calloc(1, sizeof(s7_cell));
  s7_int len = safe_strlen(str);
  set_full_type(x, T_STRING | T_IMMUTABLE | T_UNHEAP);
  set_optimize_op(x, OP_CONSTANT);
  string_length(x) = len;
  string_block(x) = NULL;
  string_value(x) = (char *)str;
  string_hash(x) = 0;
  return(x);
}

static void init_strings(void)
{
  nil_string = make_permanent_string("");
  nil_string->tf.flag = T_STRING | T_UNHEAP;
  set_optimize_op(nil_string, OP_CONSTANT);

  car_a_list_string = make_permanent_string("a pair whose car is also a pair");
  cdr_a_list_string = make_permanent_string("a pair whose cdr is also a pair");

  caar_a_list_string = make_permanent_string("a pair whose caar is also a pair");
  cadr_a_list_string = make_permanent_string("a pair whose cadr is also a pair");
  cdar_a_list_string = make_permanent_string("a pair whose cdar is also a pair");
  cddr_a_list_string = make_permanent_string("a pair whose cddr is also a pair");

  caaar_a_list_string = make_permanent_string("a pair whose caaar is also a pair");
  caadr_a_list_string = make_permanent_string("a pair whose caadr is also a pair");
  cadar_a_list_string = make_permanent_string("a pair whose cadar is also a pair");
  caddr_a_list_string = make_permanent_string("a pair whose caddr is also a pair");
  cdaar_a_list_string = make_permanent_string("a pair whose cdaar is also a pair");
  cdadr_a_list_string = make_permanent_string("a pair whose cdadr is also a pair");
  cddar_a_list_string = make_permanent_string("a pair whose cddar is also a pair");
  cdddr_a_list_string = make_permanent_string("a pair whose cdddr is also a pair");

  a_list_string =                 make_permanent_string("a list");
  an_eq_func_string =             make_permanent_string("a procedure that can take two arguments");
  an_association_list_string =    make_permanent_string("an association list");
  a_normal_real_string =          make_permanent_string("a normal real");
  a_rational_string =             make_permanent_string("an integer or a ratio");
  a_number_string =               make_permanent_string("a number");
  a_procedure_string =            make_permanent_string("a procedure");
  a_procedure_or_a_macro_string = make_permanent_string("a procedure or a macro");
  a_normal_procedure_string =     make_permanent_string("a normal procedure");
  a_let_string =                  make_permanent_string("a let (environment)");
  a_proper_list_string =          make_permanent_string("a proper list");
  a_boolean_string =              make_permanent_string("a boolean");
  a_byte_vector_string =          make_permanent_string("a byte-vector");
  an_input_port_string =          make_permanent_string("an input port");
  an_open_port_string =           make_permanent_string("an open port");
  an_output_port_string =         make_permanent_string("an output port");
  an_input_string_port_string =   make_permanent_string("an input string port");
  an_input_file_port_string =     make_permanent_string("an input file port");
  an_output_string_port_string =  make_permanent_string("an output string port");
  an_output_file_port_string =    make_permanent_string("an output file port");
  a_thunk_string =                make_permanent_string("a thunk");
  a_symbol_string =               make_permanent_string("a symbol");
  a_non_negative_integer_string = make_permanent_string("a non-negative integer");
  an_unsigned_byte_string =       make_permanent_string("an unsigned byte");
  something_applicable_string =   make_permanent_string("a procedure or something applicable");
  a_random_state_object_string =  make_permanent_string("a random-state object");
  a_format_port_string =          make_permanent_string("#f, #t, (), or an open output port");
  a_non_constant_symbol_string =  make_permanent_string("a non-constant symbol");
  a_sequence_string =             make_permanent_string("a sequence");
  a_valid_radix_string =          make_permanent_string("it should be between 2 and 16");
  result_is_too_large_string =    make_permanent_string("result is too large");
  its_too_large_string =          make_permanent_string("it is too large");
  its_too_small_string =          make_permanent_string("it is less than the start position");
  its_negative_string =           make_permanent_string("it is negative");
  its_nan_string =                make_permanent_string("NaN usually indicates a numerical error");
  its_infinite_string =           make_permanent_string("it is infinite");
  too_many_indices_string =       make_permanent_string("too many indices");
  parameter_set_twice_string =    make_permanent_string("parameter set twice, ~S in ~S");
  immutable_error_string =        make_permanent_string("can't ~S ~S (it is immutable)");
  cant_bind_immutable_string =    make_permanent_string("~A: can't bind an immutable object: ~S");
  intermediate_too_large_string = make_permanent_string("intermediate result is too large");
#if (!HAVE_COMPLEX_NUMBERS)
  no_complex_numbers_string =     make_permanent_string("this version of s7 does not support complex numbers");
#endif
  keyword_value_missing_string =  make_permanent_string("~A: keyword argument's value is missing: ~S in ~S");
  a_named_function_string =       make_permanent_string("a named function");

  format_string_1 = make_permanent_string("format: ~S ~{~S~^ ~}: ~A");
  format_string_2 = make_permanent_string("format: ~S: ~A");
  format_string_3 = make_permanent_string("format: ~S ~{~S~^ ~}~&~NT^: ~A");
  format_string_4 = make_permanent_string("format: ~S~&~NT^: ~A");

  too_many_arguments_string = make_permanent_string("~S: too many arguments: ~A");
  not_enough_arguments_string = make_permanent_string("~S: not enough arguments: ~A");
}


/* -------------------------------- make-string -------------------------------- */
static s7_pointer g_make_string(s7_scheme *sc, s7_pointer args)
{
  #define H_make_string "(make-string len (val #\\space)) makes a string of length len filled with the character val (default: space)"
  #define Q_make_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_integer_symbol, sc->is_char_symbol)

  s7_pointer n = car(args);
  s7_int len;
  char fill;
  if (!s7_is_integer(n))
    {
      check_method(sc, n, sc->make_string_symbol, args);
      wrong_type_argument(sc, sc->make_string_symbol, 1, n, T_INTEGER);
    }
  if ((is_pair(cdr(args))) &&
      (!is_character(cadr(args))))
    return(method_or_bust(sc, cadr(args), sc->make_string_symbol, args, T_CHARACTER, 2));

  len = s7_integer_clamped_if_gmp(sc, n);
  if (len == 0) return(nil_string);
  if ((len < 0) || (len > sc->max_string_length))
    out_of_range(sc, sc->make_string_symbol, int_one, n, (len < 0) ? its_negative_string : its_too_large_string);
  if (is_null(cdr(args)))
    return(make_empty_string(sc, len, '\0')); /* #\null here means "don't fill/clear" */
  fill = s7_character(cadr(args));
  n = make_empty_string(sc, len, fill);
  if (fill == '\0')
    memclr((void *)string_value(n), (size_t)len);
  return(n);
}

static s7_pointer make_string_p_i(s7_scheme *sc, s7_int len)
{
  if (len == 0) return(nil_string);
  if ((len < 0) || (len > sc->max_string_length))
    out_of_range(sc, sc->make_string_symbol, int_one, wrap_integer(sc, len), (len < 0) ? its_negative_string : its_too_large_string);
  return(make_empty_string(sc, len, '\0'));
}


#if (!WITH_PURE_S7)
/* -------------------------------- string-length -------------------------------- */
static s7_pointer g_string_length(s7_scheme *sc, s7_pointer args)
{
  #define H_string_length "(string-length str) returns the length of the string str"
  #define Q_string_length s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_string_symbol)
  s7_pointer p = car(args);
  if (!is_string(p))
    return(method_or_bust_one_arg(sc, p, sc->string_length_symbol, args, T_STRING));
  return(make_integer(sc, string_length(p)));
}

static s7_int string_length_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_string(p))
    return(integer(method_or_bust_p(sc, p, sc->string_length_symbol, T_STRING)));
  return(string_length(p));
}
#endif


/* -------------------------------- string-up|downcase -------------------------------- */
static s7_pointer g_string_downcase(s7_scheme *sc, s7_pointer args)
{
  #define H_string_downcase "(string-downcase str) returns the lower case version of str."
  #define Q_string_downcase sc->pcl_s

  s7_pointer p = car(args), newstr;
  s7_int i, len;
  uint8_t *nstr;
  const uint8_t *ostr;

  if (!is_string(p))
    return(method_or_bust_p(sc, p, sc->string_downcase_symbol, T_STRING));
  len = string_length(p);
  newstr = make_empty_string(sc, len, 0);

  ostr = (const uint8_t *)string_value(p);
  nstr = (uint8_t *)string_value(newstr);
  if (len >= 128)
    {
      i = len - 1;
      while (i >= 8)
	LOOP_8(nstr[i] = lowers[(uint8_t)ostr[i]]; i--);
      while (i >= 0) {nstr[i] = lowers[(uint8_t)ostr[i]]; i--;}
    }
  else
    for (i = 0; i < len; i++) nstr[i] = lowers[(uint8_t)ostr[i]];
  return(newstr);
}

static s7_pointer g_string_upcase(s7_scheme *sc, s7_pointer args)
{
  #define H_string_upcase "(string-upcase str) returns the upper case version of str."
  #define Q_string_upcase sc->pcl_s

  s7_pointer p = car(args), newstr;
  s7_int i, len;
  uint8_t *nstr;
  const uint8_t *ostr;

  if (!is_string(p))
    return(method_or_bust_p(sc, p, sc->string_upcase_symbol, T_STRING));
  len = string_length(p);
  newstr = make_empty_string(sc, len, 0);

  ostr = (const uint8_t *)string_value(p);
  nstr = (uint8_t *)string_value(newstr);
  if (len >= 128)
    {
      i = len - 1;
      while (i >= 8)
	LOOP_8(nstr[i] = uppers[(uint8_t)ostr[i]]; i--);
      while (i >= 0) {nstr[i] = uppers[(uint8_t)ostr[i]]; i--;}
    }
  else
    for (i = 0; i < len; i++) nstr[i] = uppers[(uint8_t)ostr[i]];
  return(newstr);
}


/* -------------------------------- string-ref -------------------------------- */
static s7_pointer string_ref_1(s7_scheme *sc, s7_pointer strng, s7_pointer index)
{
  char *str;
  s7_int ind;

  if (!s7_is_integer(index))
    return(method_or_bust_pp(sc, index, sc->string_ref_symbol, strng, index, T_INTEGER, 2));
  ind = s7_integer_clamped_if_gmp(sc, index);
  if (ind < 0)
    out_of_range(sc, sc->string_ref_symbol, int_two, index, its_negative_string);
  if (ind >= string_length(strng))
    out_of_range(sc, sc->string_ref_symbol, int_two, index, its_too_large_string);

  str = string_value(strng);
  return(chars[((uint8_t *)str)[ind]]);
}

static s7_pointer g_string_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_string_ref "(string-ref str index) returns the character at the index-th element of the string str"
  #define Q_string_ref s7_make_signature(sc, 3, sc->is_char_symbol, sc->is_string_symbol, sc->is_integer_symbol)

  s7_pointer strng = car(args);
  if (!is_string(strng))
    return(method_or_bust(sc, strng, sc->string_ref_symbol, args, T_STRING, 1));
  return(string_ref_1(sc, strng, cadr(args)));
}

static s7_pointer string_ref_p_pi(s7_scheme *sc, s7_pointer p1, s7_int i1)
{
  if (!is_string(p1))
    return(method_or_bust(sc, p1, sc->string_ref_symbol, set_plist_2(sc, p1, make_integer(sc, i1)), T_STRING, 1));
  if ((i1 >= 0) && (i1 < string_length(p1)))
    return(chars[((uint8_t *)string_value(p1))[i1]]);
  out_of_range(sc, sc->string_ref_symbol, int_two, wrap_integer(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p1);
}

static s7_pointer string_ref_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer i1)
{
  if (!is_string(p1))
    return(method_or_bust_pp(sc, p1, sc->string_ref_symbol, p1, i1, T_STRING, 1));
  return(string_ref_1(sc, p1, i1));
}

static s7_pointer string_ref_p_p0(s7_scheme *sc, s7_pointer p1, s7_pointer unused_i1)
{
  if (!is_string(p1))
    return(method_or_bust_pp(sc, p1, sc->string_ref_symbol, p1, int_zero, T_STRING, 1));
  if (string_length(p1) > 0)
    return(chars[((uint8_t *)string_value(p1))[0]]);
  out_of_range(sc, sc->string_ref_symbol, int_two, int_zero, its_too_large_string);
  return(p1);
}

static s7_pointer string_plast_via_method(s7_scheme *sc, s7_pointer p1)
{
  s7_pointer len = method_or_bust_p(sc, p1, sc->length_symbol, T_STRING);
  return(method_or_bust_with_type_pi(sc, p1, sc->string_ref_symbol, p1, integer(len) - 1, sc->prepackaged_type_names[T_STRING], 1));
}

static s7_pointer string_ref_p_plast(s7_scheme *sc, s7_pointer p1, s7_pointer unused_i1)
{
  if (!is_string(p1))
    return(string_plast_via_method(sc, p1));
  if (string_length(p1) > 0)
    return(chars[((uint8_t *)string_value(p1))[string_length(p1) - 1]]);
  out_of_range(sc, sc->string_ref_symbol, int_two, wrap_integer(sc, string_length(p1) - 1), its_too_large_string);
  return(p1);
}

static inline s7_pointer string_ref_p_pi_unchecked(s7_scheme *sc, s7_pointer p1, s7_int i1)
{
  if ((i1 >= 0) && (i1 < string_length(p1)))
    return(chars[((uint8_t *)string_value(p1))[i1]]);
  out_of_range(sc, sc->string_ref_symbol, int_two, wrap_integer(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p1);
}

static s7_pointer string_ref_p_pi_direct(s7_scheme *unused_sc, s7_pointer p1, s7_int i1) {return(chars[((uint8_t *)string_value(p1))[i1]]);}


/* -------------------------------- string-set! -------------------------------- */
static s7_pointer g_string_set(s7_scheme *sc, s7_pointer args)
{
  #define H_string_set "(string-set! str index chr) sets the index-th element of the string str to the character chr"
  #define Q_string_set s7_make_signature(sc, 4, sc->is_char_symbol, sc->is_string_symbol, sc->is_integer_symbol, sc->is_char_symbol)

  s7_pointer strng = car(args), c, index;
  char *str;
  s7_int ind;

  if (!is_mutable_string(strng))
    return(mutable_method_or_bust(sc, strng, sc->string_set_symbol, args, T_STRING, 1));

  index = cadr(args);
  if (!s7_is_integer(index))
    return(method_or_bust(sc, index, sc->string_set_symbol, args, T_INTEGER, 2));
  ind = s7_integer_clamped_if_gmp(sc, index);
  if (ind < 0)
    out_of_range(sc, sc->string_set_symbol, int_two, index, a_non_negative_integer_string);
  if (ind >= string_length(strng))
    out_of_range(sc, sc->string_set_symbol, int_two, index, its_too_large_string);

  str = string_value(strng);
  c = caddr(args);
  if (!is_character(c))
    return(method_or_bust(sc, c, sc->string_set_symbol, args, T_CHARACTER, 3));

  str[ind] = (char)s7_character(c);
  return(c);
}

static s7_pointer string_set_p_pip(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2)
{
  if (!is_string(p1))
    wrong_type_argument(sc, sc->string_set_symbol, 1, p1, T_STRING);
  if (!is_character(p2))
    wrong_type_argument(sc, sc->string_set_symbol, 2, p2, T_CHARACTER);
  if ((i1 >= 0) && (i1 < string_length(p1)))
    string_value(p1)[i1] = s7_character(p2);
  else out_of_range(sc, sc->string_set_symbol, int_two, wrap_integer(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p2);
}

static s7_pointer string_set_p_pip_unchecked(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2)
{
  if ((i1 >= 0) && (i1 < string_length(p1)))
    string_value(p1)[i1] = s7_character(p2);
  else out_of_range(sc, sc->string_set_symbol, int_two, wrap_integer(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p2);
}

static s7_pointer string_set_p_pip_direct(s7_scheme *unused_sc, s7_pointer p1, s7_int i1, s7_pointer p2) {string_value(p1)[i1] = s7_character(p2); return(p2);}


/* -------------------------------- string-append -------------------------------- */
static s7_pointer c_object_length(s7_scheme *sc, s7_pointer obj);

static bool sequence_is_empty(s7_scheme *sc, s7_pointer obj) /* "is_empty" is some C++ struct?? */
{
  switch (type(obj))
    {
    case T_BYTE_VECTOR: case T_INT_VECTOR: case T_FLOAT_VECTOR:
    case T_VECTOR:     return(vector_length(obj) == 0);
    case T_NIL:        return(true);
    case T_PAIR:       return(false);
    case T_STRING:     return(string_length(obj) == 0);
    case T_HASH_TABLE: return(hash_table_entries(obj) == 0);
    case T_LET:        return(!tis_slot(let_slots(obj)));
    case T_C_OBJECT:   return(s7_is_eqv(sc, c_object_length(sc, obj), int_zero));
    default:           return(false);
    }
}

static s7_int sequence_length(s7_scheme *sc, s7_pointer lst)
{
  switch (type(lst))
    {
    case T_PAIR:
      {
	s7_int len = s7_list_length(sc, lst);
	return((len == 0) ? -1 : len);
      }
    case T_NIL:         return(0);
    case T_BYTE_VECTOR: case T_INT_VECTOR: case T_FLOAT_VECTOR:
    case T_VECTOR:      return(vector_length(lst));
    case T_STRING:      return(string_length(lst));
    case T_HASH_TABLE:  return(hash_table_entries(lst));
    case T_LET:         return(let_length(sc, lst));
    case T_C_OBJECT:
      {
	s7_pointer x = c_object_length(sc, lst);
	if (s7_is_integer(x))
	  return(s7_integer_clamped_if_gmp(sc, x));
      }}
  return(-1);
}

static s7_pointer s7_copy_1(s7_scheme *sc, s7_pointer caller, s7_pointer args);

static void string_append_2(s7_scheme *sc, s7_pointer newstr, s7_pointer args, s7_pointer stop_arg, s7_pointer caller)
{
  s7_int len;
  char *pos;
  s7_pointer x;
  for (pos = string_value(newstr), x = args; x != stop_arg; x = cdr(x))
    if (is_string(car(x)))
      {
	len = string_length(car(x));
	if (len > 0)
	  {
	    memcpy(pos, string_value(car(x)), len);
	    pos += len;
	  }}
    else
      if (!sequence_is_empty(sc, car(x)))
	{
	  char *old_str = string_value(newstr);
	  string_value(newstr) = pos;
	  len = sequence_length(sc, car(x));
	  s7_copy_1(sc, caller, set_plist_2(sc, car(x), newstr));
	  string_value(newstr) = old_str;
	  pos += len;
	}
}

static s7_pointer g_string_append_1(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  #define H_string_append "(string-append str1 ...) appends all its string arguments into one string"
  #define Q_string_append sc->pcl_s

  s7_int len = 0;
  s7_pointer x, newstr;
  bool just_strings = true;

  if (is_null(args))
    return(nil_string);

  s7_gc_protect_via_stack(sc, args);
  /* get length for new string */
  for (x = args; is_not_null(x); x = cdr(x))
    {
      s7_pointer p = car(x);
      if (is_string(p))
	len += string_length(p);
      else
	{
	  s7_int newlen;
	  if (!is_sequence(p))
	    {
	      unstack(sc);
	      wrong_type_argument(sc, caller, position_of(x, args), p, T_STRING);
	    }
	  if (has_active_methods(sc, p)) /* look for string-append and if found, cobble up a plausible intermediate call */
	    {
	      s7_pointer func = find_method_with_let(sc, p, caller);
	      if (func != sc->undefined)
		{
		  if (len == 0)
		    {
		      unstack(sc);
		      return(s7_apply_function(sc, func, x)); /* not args (string-append "" "" ...) */
		    }
		  /* TODO: s7test this with non-string earlier arg */
		  newstr = make_empty_string(sc, len, 0);
		  string_append_2(sc, newstr, args, x, caller);
		  unstack(sc);
		  return(s7_apply_function(sc, func, set_ulist_1(sc, newstr, x)));
		}}
	  if ((caller == sc->string_append_symbol) || (caller == sc->symbol_symbol))
	    {
	      unstack(sc);
	      wrong_type_argument(sc, caller, position_of(x, args), p, T_STRING);
	    }
	  newlen = sequence_length(sc, p);
	  if (newlen < 0)
	    {
	      unstack(sc);
	      wrong_type_argument(sc, caller, position_of(x, args), p, T_STRING); /* TODO: something that can be turned into a seq of char */
	    }
	  just_strings = false;
	  len += newlen;
	}}
  if (len == 0)
    {
      unstack(sc);
      return(nil_string);
    }
  if (len > sc->max_string_length)
    {
      unstack(sc);
      s7_error_nr(sc, sc->out_of_range_symbol,
		  set_elist_4(sc, wrap_string(sc, "~S new string length, ~D, is larger than (*s7* 'max-string-length): ~D", 70),
			      caller, wrap_integer(sc, len), wrap_integer(sc, sc->max_string_length)));
    }
  newstr = inline_make_empty_string(sc, len, 0);
  if (just_strings)
    {
      char *pos;
      for (pos = string_value(newstr), x = args; is_not_null(x); x = cdr(x))
	{
	  len = string_length(car(x));
	  if (len > 0)
	    {
	      memcpy(pos, string_value(car(x)), len);
	      pos += len;
	    }}}
  else string_append_2(sc, newstr, args, sc->nil, caller);
  unstack(sc);
  return(newstr);
}

static s7_pointer g_string_append(s7_scheme *sc, s7_pointer args) {return(g_string_append_1(sc, args, sc->string_append_symbol));}

static inline s7_pointer string_append_1(s7_scheme *sc, s7_pointer s1, s7_pointer s2)
{
  if ((is_string(s1)) && (is_string(s2)))
    {
      s7_int len, pos = string_length(s1);
      s7_pointer newstr;
      if (pos == 0) return(make_string_with_length(sc, string_value(s2), string_length(s2)));
      len = pos + string_length(s2);
      if (len == pos) return(make_string_with_length(sc, string_value(s1), string_length(s1)));
      if (len > sc->max_string_length)
	s7_error_nr(sc, sc->out_of_range_symbol,
		    set_elist_4(sc, wrap_string(sc, "~S new string length, ~D, is larger than (*s7* 'max-string-length): ~D", 70),
				sc->string_append_symbol, wrap_integer(sc, len), wrap_integer(sc, sc->max_string_length)));
      newstr = make_empty_string(sc, len, 0); /* len+1 0-terminated */
      memcpy(string_value(newstr), string_value(s1), pos);
      memcpy((char *)(string_value(newstr) + pos), string_value(s2), string_length(s2));
      return(newstr);
    }
  return(g_string_append_1(sc, list_2(sc, s1, s2), sc->string_append_symbol));
}

static s7_pointer string_append_p_pp(s7_scheme *sc, s7_pointer s1, s7_pointer s2) {return(string_append_1(sc, s1, s2));}

static s7_pointer g_string_append_2(s7_scheme *sc, s7_pointer args) {return(string_append_1(sc, car(args), cadr(args)));}

static void check_for_substring_temp(s7_scheme *sc, s7_pointer expr);

static s7_pointer string_append_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool unused_ops)
{
  check_for_substring_temp(sc, expr);
  return((args == 2) ? sc->string_append_2 : f);
}


/* -------------------------------- substring -------------------------------- */
static s7_pointer start_and_end(s7_scheme *sc, s7_pointer caller, s7_pointer args, int32_t position, s7_pointer index_args, s7_int *start, s7_int *end)
{
  /* we assume that *start=0 and *end=length, that end is "exclusive", return true if the start/end points are not changed */
  s7_pointer pstart = car(index_args);
  s7_int index;

  if (!s7_is_integer(pstart))
    return(method_or_bust(sc, pstart, caller, args, T_INTEGER, position));
  index = s7_integer_clamped_if_gmp(sc, pstart);
  if ((index < 0) ||
      (index > *end)) /* *end == length here */
    out_of_range(sc, caller, small_int(position), pstart, (index < 0) ? its_negative_string : its_too_large_string);
  *start = index;

  if (is_pair(cdr(index_args)))
    {
      s7_pointer pend = cadr(index_args);
      if (!s7_is_integer(pend))
	return(method_or_bust(sc, pend, caller, args, T_INTEGER, position + 1));
      index = s7_integer_clamped_if_gmp(sc, pend);
      if ((index < *start) ||
	  (index > *end))
	out_of_range(sc, caller, small_int(position + 1), pend, (index < *start) ? its_too_small_string : its_too_large_string);
      *end = index;
    }
  return(sc->unused);
}

static s7_pointer g_substring(s7_scheme *sc, s7_pointer args)
{
  #define H_substring "(substring str start (end (length str))) returns the portion of the string str between start and \
end: (substring \"01234\" 1 2) -> \"1\""
  #define Q_substring s7_make_circular_signature(sc, 2, 3, sc->is_string_symbol, sc->is_string_symbol, sc->is_integer_symbol)

  s7_pointer x, str = car(args);
  s7_int start = 0, end, len;
  char *s;

  if (!is_string(str))
    return(method_or_bust(sc, str, sc->substring_symbol, args, T_STRING, 1));
  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      x = start_and_end(sc, sc->substring_symbol, args, 2, cdr(args), &start, &end);
      if (x != sc->unused) return(x);
    }
  s = string_value(str);
  len = end - start;
  if (len == 0) return(nil_string);
  x = inline_make_string_with_length(sc, (char *)(s + start), len);
  string_value(x)[len] = 0;
  return(x);
}

static s7_pointer g_substring_uncopied(s7_scheme 