/* s7, a Scheme interpreter
 *
 *    derived from:
 *
 * --------------------------------------------------------------------------------
 * T I N Y S C H E M E    1 . 3 9
 *   Dimitrios Souflis (dsouflis@acm.org)
 *   Based on MiniScheme (original credits follow)
 * (MINISCM)               coded by Atsushi Moriwaki (11/5/1989)
 * (MINISCM)           E-MAIL :  moriwaki@kurims.kurims.kyoto-u.ac.jp
 * (MINISCM) This version has been modified by R.C. Secrist.
 * (MINISCM)
 * (MINISCM) Mini-Scheme is now maintained by Akira KIDA.
 * (MINISCM)
 * (MINISCM) This is a revised and modified version by Akira KIDA.
 * (MINISCM)	current version is 0.85k4 (15 May 1994)
 * --------------------------------------------------------------------------------
 *
 * apparently tinyScheme is under the BSD license, so I guess s7 is too.
 * Here is Snd's verbiage which can apply here:
 *
 *     The authors hereby grant permission to use, copy, modify, distribute,
 *     and license this software and its documentation for any purpose.  No
 *     written agreement, license, or royalty fee is required.  Modifications
 *     to this software may be copyrighted by their authors and need not
 *     follow the licensing terms described here.
 *
 * followed by the usual all-caps shouting about liability.
 *
 * --------------------------------------------------------------------------------
 *
 * s7, Bill Schottstaedt, Aug-08, bil@ccrma.stanford.edu
 *
 * Mike Scholz provided the FreeBSD support (complex trig funcs, etc)
 * Rick Taube, Andrew Burnson, Donny Ward, and Greg Santucci provided the MS Visual C++ support
 *
 * Documentation is in s7.h and s7.html.
 * s7test.scm is a regression test.
 * glistener.c is a gtk-based listener.
 * repl.scm is a vt100-based listener.
 * cload.scm and lib*.scm tie in various C libraries.
 * lint.scm checks Scheme code for infelicities.
 * r7rs.scm implements some of r7rs (small).
 * write.scm currrently has pretty-print.
 * mockery.scm has the mock-data definitions.
 * stuff.scm has some stuff.
 *
 * s7.c is organized as follows:
 *
 *    structs and type flags
 *    constants
 *    GC
 *    stacks
 *    symbols and keywords
 *    environments
 *    continuations
 *    numbers
 *    characters
 *    strings
 *    ports
 *    format
 *    lists
 *    vectors
 *    hash-tables
 *    c-objects
 *    functions
 *    equal?
 *    generic length, copy, reverse, fill!, append
 *    error handlers
 *    sundry leftovers
 *    multiple-values, quasiquote
 *    eval
 *    multiprecision arithmetic
 *    *s7* environment
 *    initialization
 *    repl
 *
 * naming conventions: s7_* usually are C accessible (s7.h), g_* are scheme accessible (FFI), 
 *   H_* are documentation strings, Q_* are procedure signatures,
 *   *_1 are auxilliary functions, big_* refer to gmp, 
 *   scheme "?" corresponds to C "is_", scheme "->" to C "_to_".
 *
 * ---------------- compile time switches ----------------
 */

#include "mus-config.h"

/*
 * Your config file goes here, or just replace that #include line with the defines you need.
 * The compile-time switches involve booleans, complex numbers, and multiprecision arithmetic.
 * Currently we assume we have setjmp.h (used by the error handlers).
 *
 * Complex number support which is problematic in C++, Solaris, and netBSD
 *   is on the HAVE_COMPLEX_NUMBERS switch. In OSX or Linux, if you're not using C++,
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 1
 *
 *   In C++ I use:
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 0
 *
 *   In windows, both are 0.
 *
 *   Some systems (FreeBSD) have complex.h, but some random subset of the trig funcs, so
 *   HAVE_COMPLEX_NUMBERS means we can find
 *      cimag creal cabs csqrt carg conj
 *   and HAVE_COMPLEX_TRIG means we have
 *      cacos cacosh casin casinh catan catanh ccos ccosh cexp clog cpow csin csinh ctan ctanh
 *
 * When HAVE_COMPLEX_NUMBERS is 0, the complex functions are stubs that simply return their
 *   argument -- this will be very confusing for the s7 user because, for example, (sqrt -2)
 *   will return something bogus (it will not signal an error).
 *
 * so the incoming (non-s7-specific) compile-time switches are
 *     HAVE_COMPLEX_NUMBERS, HAVE_COMPLEX_TRIG, SIZEOF_VOID_P
 * if SIZEOF_VOID_P is not defined, we look for __SIZEOF_POINTER__ instead
 *   the default is to assume that we're running on a 64-bit machine.
 *
 * To get multiprecision arithmetic, set WITH_GMP to 1.
 *   You'll also need libgmp, libmpfr, and libmpc (version 0.8.0 or later)
 *   In highly numerical contexts, the gmp version of s7 is about 50(!) times slower than the non-gmp version.
 *
 * and we use these predefined macros: __cplusplus, _MSC_VER, __GNUC__, __clang__, __ANDROID__
 *
 * if WITH_SYSTEM_EXTRAS is 1 (default is 1 unless _MSC_VER), various OS and file related functions are included.
 * in openBSD I think you need to include -ftrampolines in CFLAGS.
 * if you want this file to compile into a stand-alone interpreter, define WITH_MAIN
 *
 * -O3 is sometimes slower, sometimes faster
 * -march=native -fomit-frame-pointer -m64 -funroll-loops gains about .1%
 * -ffast-math makes a mess of NaNs, and does not appear to be faster
 * for timing tests, I use: -O2 -DINITIAL_HEAP_SIZE=1024000 -march=native -fomit-frame-pointer -funroll-loops
 */


/* ---------------- initial sizes ---------------- */

#ifndef INITIAL_HEAP_SIZE
#define INITIAL_HEAP_SIZE 128000
/* the heap grows as needed, this is its initial size.
 * If the initial heap is small, s7 can run in about 2.5 Mbytes of memory. There are (many) cases where a bigger heap is faster.
 * The heap size must be a multiple of 32.  Each object takes about 50 bytes.
 *
 * repl runs in    4Mb (18v) (64bit) if heap is 8192
 *                11Mb (25v)         if 128k heap
 *  snd (no gui)  15Mb (151v)
 *  snd (motif)   12Mb (285v)
 *  snd (gtk)     32Mb (515v!)
 */
#endif

#ifndef SYMBOL_TABLE_SIZE
#define SYMBOL_TABLE_SIZE 13567
/* names are hashed into the symbol table (a vector) and collisions are chained as lists.
 */
#endif

#define INITIAL_STACK_SIZE 512
/* the stack grows as needed, each frame takes 4 entries, this is its initial size.
 *   this needs to be big enough to handle the eval_c_string's at startup (ca 100)
 *   In s7test.scm, the maximum stack size is ca 440.  In snd-test.scm, it's ca 200.
 *   This number matters only because call/cc copies the stack, which requires filling
 *   the unused portion of the new stack, which requires memcpy of #<unspecified>'s.
 */

#define INITIAL_PROTECTED_OBJECTS_SIZE 16
/* a vector of objects that are (semi-permanently) protected from the GC, grows as needed */

#define GC_TEMPS_SIZE 256
/* the number of recent objects that are temporarily gc-protected; 8 works for s7test and snd-test.
 *    For the FFI, this sets the lag between a call on s7_cons and the first moment when its result
 *    might be vulnerable to the GC.
 */


/* ---------------- scheme choices ---------------- */

#ifndef WITH_GMP
  #define WITH_GMP 0
  /* this includes multiprecision arithmetic for all numeric types and functions, using gmp, mpfr, and mpc
   * WITH_GMP adds the following functions: bignum, bignum?, bignum-precision
   * using gmp with precision=128 is about 50 times slower than using C doubles and long long ints.
   */
#endif

#if WITH_GMP
  #define DEFAULT_BIGNUM_PRECISION 128
#endif

#ifndef WITH_PURE_S7
  #define WITH_PURE_S7 0
#endif
#if WITH_PURE_S7
  #define WITH_EXTRA_EXPONENT_MARKERS 0
  #define WITH_IMMUTABLE_UNQUOTE 1
  /* also omitted: *-ci* functions, char-ready?, cond-expand, multiple-values-bind|set!, call-with-values, defmacro(*)
   *   and a lot more (inexact/exact, integer-length,  etc) -- see s7.html.
   */
#endif

#ifndef WITH_EXTRA_EXPONENT_MARKERS
  #define WITH_EXTRA_EXPONENT_MARKERS 0
  /* if 1, s7 recognizes "d", "f", "l", and "s" as exponent markers, in addition to "e" (also "D", "F", "L", "S") */
#endif

#ifndef WITH_SYSTEM_EXTRAS
  #define WITH_SYSTEM_EXTRAS (!_MSC_VER)
  /* this adds several functions that access file info, directories, times, etc
   *    this may be replaced by the cload business below
   */
#endif

#ifndef WITH_IMMUTABLE_UNQUOTE
  #define WITH_IMMUTABLE_UNQUOTE 0
  /* this removes the name "unquote" */
#endif

#ifndef WITH_C_LOADER
  #define WITH_C_LOADER WITH_GCC
  /* (load file.so [e]) looks for (e 'init_func) and if found, calls it
   *   as the shared object init function.  If WITH_SYSTEM_EXTRAS is 0, the caller
   *   needs to supply system and delete-file so that cload.scm works.
   */
#endif

#ifndef WITH_HISTORY
  #define WITH_HISTORY 0
  /* this includes a circular buffer of previous evaluations for debugging, ((owlet) 'error-history) and (*s7* 'history-size) */
#endif

#ifndef DEFAULT_HISTORY_SIZE
  #define DEFAULT_HISTORY_SIZE 8
  /* this is the default length of the eval history buffer */
#endif

#ifndef WITH_PROFILE
  #define WITH_PROFILE 0
  /* this includes profiling data collection accessible from scheme via the hash-table (*s7* 'profile-info) */
#endif


#define WITH_GCC (defined(__GNUC__) || defined(__clang__))

/* in case mus-config.h forgets these */
#ifdef _MSC_VER
  #ifndef HAVE_COMPLEX_NUMBERS
    #define HAVE_COMPLEX_NUMBERS 0
  #endif
  #ifndef HAVE_COMPLEX_TRIG
    #define HAVE_COMPLEX_TRIG 0
  #endif
#else
  #ifndef HAVE_COMPLEX_NUMBERS
    #define HAVE_COMPLEX_NUMBERS 1
  #endif
  #if __cplusplus
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 0
    #endif
  #else
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 1
    #endif
  #endif
#endif

/* -------------------------------------------------------------------------------- */

#ifndef DEBUGGING
  #define DEBUGGING 0
#endif
#ifndef OP_NAMES
  #define OP_NAMES 0
#endif

#define WITH_ADD_PF 0

#ifndef _MSC_VER
  #include <unistd.h>
  #include <sys/param.h>
  #include <strings.h>
  #include <errno.h>
  #include <locale.h>
#else
  /* in Snd these are in mus-config.h */
  #ifndef MUS_CONFIG_H_LOADED
    #define snprintf _snprintf 
    #if _MSC_VER > 1200
      #define _CRT_SECURE_NO_DEPRECATE 1
      #define _CRT_NONSTDC_NO_DEPRECATE 1
      #define _CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES 1
    #endif
  #endif
  #include <io.h>
  #pragma warning(disable: 4244)
#endif

#include <limits.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <time.h>
#include <stdarg.h>
#include <stddef.h>

#if __cplusplus
  #include <cmath>
#else
  #include <math.h>
#endif

#if HAVE_COMPLEX_NUMBERS
  #if __cplusplus
    #include <complex>
  #else
    #include <complex.h>
    #ifndef __SUNPRO_C
      #if defined(__sun) && defined(__SVR4)
        #undef _Complex_I
        #define _Complex_I 1.0fi
      #endif
    #endif
  #endif
#ifndef CMPLX
  /* c11 addition? */
  #define CMPLX(r, i) ((r) + ((i) * _Complex_I))
#endif
#endif

#include <setjmp.h>

#include "s7.h"

enum {NO_JUMP, CALL_WITH_EXIT_JUMP, THROW_JUMP, CATCH_JUMP, ERROR_JUMP, ERROR_QUIT_JUMP};
enum {NO_SET_JUMP, READ_SET_JUMP, LOAD_SET_JUMP, DYNAMIC_WIND_SET_JUMP, S7_CALL_SET_JUMP, EVAL_SET_JUMP};


#ifndef M_PI
  #define M_PI 3.1415926535897932384626433832795029L
#endif

#ifndef INFINITY
  #define INFINITY (-log(0.0))
  /* 1.0 / 0.0 is also used, there is sometimes a function, infinity(), MSC apparently uses HUGE_VALF */
#endif

#ifndef NAN
  #define NAN (INFINITY / INFINITY)
#endif

#define BOLD_TEXT "\033[1m"
#define UNBOLD_TEXT "\033[22m"

#define WRITE_REAL_PRECISION 16
static int float_format_precision = WRITE_REAL_PRECISION;

#if ((!__NetBSD__) && ((_MSC_VER) || (!defined(__STC__)) || (defined(__STDC_VERSION__) && (__STDC_VERSION__ < 199901L))))
  #define __func__ __FUNCTION__
#endif

#define DISPLAY(Obj) s7_object_to_c_string(sc, Obj)
#define DISPLAY_80(Obj) object_to_truncated_string(sc, Obj, 80)

#if (((defined(SIZEOF_VOID_P)) && (SIZEOF_VOID_P == 4)) || ((defined(__SIZEOF_POINTER__)) && (__SIZEOF_POINTER__ == 4)))
  #define opcode_t unsigned int
  #define PRINT_NAME_PADDING 8
  #define PRINT_NAME_SIZE (20 - PRINT_NAME_PADDING - 2)
  #define ptr_int unsigned int
  #define INT_FORMAT "%u"
  #ifndef WITH_OPTIMIZATION
    #define WITH_OPTIMIZATION 0
    /* 32-bit optimized case gets inexplicable NaNs in float-vector ops.
     *   only the rf cases are faulty, so it is possible to set this flag to 1, then make s7_rf_set_function a no-op,
     *   and comment out the 2 syntax_rp cases. 
     * In standard scheme code, this flag does not matter much, but it makes CLM run about 3 times as fast.
     */
  #endif
#else
  #define opcode_t unsigned long long int
  #define ptr_int unsigned long long int
  #define INT_FORMAT "%llu"
  #define PRINT_NAME_PADDING 16
  #define PRINT_NAME_SIZE (40 - PRINT_NAME_PADDING - 2)
  #ifndef WITH_OPTIMIZATION
    #define WITH_OPTIMIZATION 1
  #endif
#endif


/* types */
#define T_FREE                 0
#define T_PAIR                 1
#define T_NIL                  2
#define T_UNIQUE               3
#define T_UNSPECIFIED          4
#define T_BOOLEAN              5
#define T_CHARACTER            6
#define T_SYMBOL               7
#define T_SYNTAX               8

#define T_INTEGER              9
#define T_RATIO               10
#define T_REAL                11
#define T_COMPLEX             12

#define T_BIG_INTEGER         13 /* these four used only if WITH_GMP -- order matters */
#define T_BIG_RATIO           14
#define T_BIG_REAL            15
#define T_BIG_COMPLEX         16  

#define T_STRING              17
#define T_C_OBJECT            18
#define T_VECTOR              19
#define T_INT_VECTOR          20
#define T_FLOAT_VECTOR        21

#define T_CATCH               22
#define T_DYNAMIC_WIND        23
#define T_HASH_TABLE          24
#define T_LET                 25
#define T_ITERATOR            26
#define T_STACK               27
#define T_COUNTER             28
#define T_SLOT                29
#define T_C_POINTER           30
#define T_OUTPUT_PORT         31
#define T_INPUT_PORT          32
#define T_BAFFLE              33
#define T_RANDOM_STATE        34

#define T_GOTO                35
#define T_CONTINUATION        36
#define T_CLOSURE             37
#define T_CLOSURE_STAR        38
#define T_C_MACRO             39
#define T_MACRO               40
#define T_MACRO_STAR          41
#define T_BACRO               42
#define T_BACRO_STAR          43
#define T_C_FUNCTION_STAR     44
#define T_C_FUNCTION          45
#define T_C_ANY_ARGS_FUNCTION 46
#define T_C_OPT_ARGS_FUNCTION 47
#define T_C_RST_ARGS_FUNCTION 48

#define NUM_TYPES        49

/* T_STACK, T_SLOT, T_BAFFLE, T_DYNAMIC_WIND, and T_COUNTER are internal
 * I tried T_CASE_SELECTOR that turned a case statement into an array, but it was slower!
 */

typedef enum {TOKEN_EOF, TOKEN_LEFT_PAREN, TOKEN_RIGHT_PAREN, TOKEN_DOT, TOKEN_ATOM, TOKEN_QUOTE, TOKEN_DOUBLE_QUOTE,
	      TOKEN_BACK_QUOTE, TOKEN_COMMA, TOKEN_AT_MARK, TOKEN_SHARP_CONST,
	      TOKEN_VECTOR, TOKEN_BYTE_VECTOR} token_t;

typedef enum {FILE_PORT, STRING_PORT, FUNCTION_PORT} port_type_t;

typedef struct {
  bool needs_free;
  FILE *file;
  char *filename;
  int filename_length, gc_loc; /* gc_loc uses -1 as unset flag -- kinda ugly */
  void *next;
  s7_pointer (*input_function)(s7_scheme *sc, s7_read_t read_choice, s7_pointer port);
  void (*output_function)(s7_scheme *sc, unsigned char c, s7_pointer port);
  /* a version of string ports using a pointer to the current location and a pointer to the end
   *   (rather than an integer for both, indexing from the base string) was not faster.
   */
  s7_pointer orig_str;                                                   /* GC protection for string port string */
  int (*read_character)(s7_scheme *sc, s7_pointer port);                 /* function to read a character */
  void (*write_character)(s7_scheme *sc, int c, s7_pointer port);        /* function to write a character */
  void (*write_string)(s7_scheme *sc, const char *str, int len, s7_pointer port); /* function to write a string of known length */
  token_t (*read_semicolon)(s7_scheme *sc, s7_pointer port);             /* internal skip-to-semicolon reader */
  int (*read_white_space)(s7_scheme *sc, s7_pointer port);               /* internal skip white space reader */
  s7_pointer (*read_name)(s7_scheme *sc, s7_pointer pt);                 /* internal get-next-name reader */
  s7_pointer (*read_sharp)(s7_scheme *sc, s7_pointer pt);                /* internal get-next-sharp-constant reader */
  s7_pointer (*read_line)(s7_scheme *sc, s7_pointer pt, bool eol_case, bool copied);  /* function to read a string up to \n */
  void (*display)(s7_scheme *sc, const char *s, s7_pointer pt);
} port_t;


typedef struct {
  const char *name;
  int name_length;
  unsigned int id;
  char *doc;
  s7_pointer generic_ff;
  s7_pointer signature;
  s7_pointer (*chooser)(s7_scheme *sc, s7_pointer f, int args, s7_pointer expr);
  s7_pointer *arg_defaults, *arg_names;
  s7_pointer call_args;
  s7_rp_t rp;
  s7_ip_t ip;
  s7_pp_t pp, gp;
} c_proc_t;


typedef struct {               /* call/cc */
  unsigned int stack_size, op_stack_loc, op_stack_size;
  int local_key;   /* for with-baffle */
} continuation_t;


typedef struct vdims_t {
  unsigned int ndims;
  bool elements_allocated, dimensions_allocated; /* these are allocated as bytes, not ints, so the struct size is 32 */
  s7_int *dims, *offsets;
  s7_pointer original;
} vdims_t;


typedef struct {
  int type;
  unsigned int outer_type;
  const char *name;
  s7_pointer scheme_name;
  char *(*print)(s7_scheme *sc, void *value);
  void (*free)(void *value);
  bool (*equal)(void *val1, void *val2);
  void (*gc_mark)(void *val);
  s7_pointer (*ref)(s7_scheme *sc, s7_pointer obj, s7_pointer args);
  s7_pointer (*set)(s7_scheme *sc, s7_pointer obj, s7_pointer args);
  s7_pointer (*length)(s7_scheme *sc, s7_pointer obj);
  s7_pointer (*copy)(s7_scheme *sc, s7_pointer args);
  s7_pointer (*reverse)(s7_scheme *sc, s7_pointer obj);
  s7_pointer (*fill)(s7_scheme *sc, s7_pointer args);
  char *(*print_readably)(s7_scheme *sc, void *value);
  s7_pointer (*direct_ref)(s7_scheme *sc, s7_pointer obj, s7_int index);
  s7_pointer (*direct_set)(s7_scheme *sc, s7_pointer obj, s7_int index, s7_pointer val);
  s7_ip_t ip, set_ip;
  s7_rp_t rp, set_rp;
} c_object_t;


typedef struct hash_entry_t {
  s7_pointer key, value;
  struct hash_entry_t *next;
  unsigned int raw_hash;
} hash_entry_t;

typedef unsigned int (*hash_map_t)(s7_scheme *sc, s7_pointer table, s7_pointer key);    /* hash-table object->location mapper */
typedef hash_entry_t *(*hash_check_t)(s7_scheme *sc, s7_pointer table, s7_pointer key); /* hash-table object equality function */
static hash_map_t *default_hash_map;


/* cell structure */
typedef struct s7_cell {
  union {
    unsigned int flag;
    unsigned char type_field;
    unsigned short sflag;
  } tf;
  int hloc;
  union {

    union {
      s7_int integer_value;
      s7_double real_value;

      struct {
	char padding[PRINT_NAME_PADDING];
	char name[PRINT_NAME_SIZE + 2];
      } pval;

      struct {
	s7_int numerator;
	s7_int denominator;
      } fraction_value;

      struct {
	s7_double rl;
	s7_double im;
      } complex_value;

      unsigned long ul_value;           /* these two are not used by s7 in any way */
      unsigned long long ull_value;

#if WITH_GMP
      mpz_t big_integer;
      mpq_t big_ratio;
      mpfr_t big_real;
      mpc_t big_complex;
      /* using free_lists here was not faster, and avoiding the extra init/clear too tricky.  These make up
       *   no more than ca. 5% of the gmp computation -- it is totally dominated by stuff like __gmpz_mul,
       *   so I can't see much point in optimizing the background noise.  In a very numerical context,
       *   gmp slows us down by a factor of 50.
       */
#endif
    } number;

    struct {
      port_t *port;
      unsigned char *data;
      unsigned int size, point;        /* these limit the in-core portion of a string-port to 2^31 bytes */
      unsigned int line_number, file_number;
      bool is_closed;
      port_type_t ptype;
    } prt;

    struct{
      unsigned char c, up_c;
      int length;
      bool alpha_c, digit_c, space_c, upper_c, lower_c;
      char c_name[12];
    } chr;

    void *c_pointer;

    int baffle_key;

    struct {
      s7_int length;
      union {
	s7_pointer *objects;
	s7_int *ints;
	s7_double *floats;
      } elements;
      vdims_t *dim_info;
      s7_pointer (*vget)(s7_scheme *sc, s7_pointer vec, s7_int loc);
      s7_pointer (*vset)(s7_scheme *sc, s7_pointer vec, s7_int loc, s7_pointer val);
    } vector;

    struct {
      s7_int length;
      s7_pointer *objects;
      vdims_t *dim_info;
      int top;
    } stk;

    struct {
      unsigned int mask, entries;
      hash_entry_t **elements;
      hash_check_t hash_func;
      hash_map_t *loc;
      s7_pointer dproc;
    } hasher;

    struct {
      s7_pointer obj, cur;
      union {
	s7_int loc;
	s7_pointer lcur;
      } lc;
      union {
	s7_int len;
	s7_pointer slow;
	hash_entry_t *hcur;
      } lw;
      s7_pointer (*next)(s7_scheme *sc, s7_pointer iterator);
    } iter;

    struct {
      c_proc_t *c_proc;                /* C functions, macros */
      s7_function ff;
      s7_pointer setter;
      unsigned int required_args, optional_args, all_args;
      bool rest_arg;
    } fnc;

    struct {                           /* pairs */
      s7_pointer car, cdr, opt1, opt2, opt3;
    } cons;

    struct {
      s7_pointer sym_car, sym_cdr;
      unsigned long long int hash;
      const char *fstr;
      unsigned int op, line;
    } sym_cons;

    struct {
      s7_pointer args, body, env, setter;
      int arity;
    } func;

    struct {
      unsigned int length;
      union {
	bool needs_free;
	unsigned int accessor;
	int temp_len;
      } str_ext;
      char *svalue;
      unsigned long long int hash;          /* string hash-index */
      s7_pointer initial_slot;
      union {
	char *documentation;
	s7_pointer ksym;
      } doc;
    } string;

    struct {                       /* symbols */
      s7_pointer name, global_slot, local_slot;
      long long int id;
      unsigned int op, tag;
    } sym;

    struct {                       /* syntax */
      s7_pointer symbol;
      int op;
      short min_args, max_args;
      s7_rp_t rp;
      s7_ip_t ip;
      s7_pp_t pp;
    } syn;

    struct {                       /* slots (bindings) */
      s7_pointer sym, val, nxt, pending_value, expr;
    } slt;

    struct {                       /* environments (frames) */
      s7_pointer slots, nxt;
      long long int id;            /* id of rootlet is -1 */
      union {
	struct {
	  s7_pointer function;     /* __func__ (code) if this is a funclet */
	  unsigned int line, file; /* __func__ location if it is known */
	} efnc;
	struct {
	  s7_pointer dox1, dox2;   /* do loop variables */
	} dox;
	struct {                   /* (catch #t ...) opts */
	  s7_pointer result;
	  unsigned int op_stack_loc, goto_loc;
	} ctall;
      } edat;
    } envr;

    struct {
      /* these 3 are just place-holders */
      s7_pointer unused_slots, unused_nxt;
      long long int unused_id;
      /* these two fields are for some special case objects like #<unspecified> */
      const char *name;
      int len;
    } unq;

    struct {                        /* counter (internal) */
      s7_pointer result, list, env, slots; /* env = counter_let (curlet after map/for-each frame created) */
      unsigned long long int cap;   /* sc->capture_let_counter for frame reuse */
    } ctr;

    struct {
#if WITH_GMP
      gmp_randstate_t state;
#else
      unsigned long long int seed, carry;
#endif
    } rng;

    struct {               /* additional object types (C) */
      int type;
      void *value;         /*  the value the caller associates with the object */
      s7_pointer e;        /*   the method list, if any (openlet) */
      s7_pointer (*ref)(s7_scheme *sc, s7_pointer obj, s7_int pos);
    } c_obj;

    struct {
      continuation_t *continuation;
      s7_pointer stack;
      s7_pointer *stack_start, *stack_end, *op_stack;
    } cwcc;

    struct {               /* call-with-exit */
      unsigned int goto_loc, op_stack_loc;
      bool active;
    } rexit;

    struct {               /* catch */
      unsigned int goto_loc, op_stack_loc;
      s7_pointer tag;
      s7_pointer handler;
    } rcatch; /* C++ reserves "catch" I guess */

    struct {               /* dynamic-wind */
      s7_pointer in, out, body;
      unsigned int state;
    } winder;
  } object;

#if DEBUGGING
  int current_alloc_line, previous_alloc_line, current_alloc_type, previous_alloc_type, debugger_bits, gc_line, clear_line, alloc_line, uses;
  const char *current_alloc_func, *previous_alloc_func, *gc_func, *alloc_func;
#endif

} s7_cell;


typedef struct {
  s7_pointer *objs;
  int size, top, ref;
  bool has_hits;
  int *refs;
} shared_info;


typedef struct {
  int loc, curly_len, ctr;
  char *curly_str;
  s7_pointer args, orig_str, curly_arg;
  s7_pointer port, strport;
} format_data;


typedef struct gc_obj {
  s7_pointer p;
  struct gc_obj *nxt;
} gc_obj;


typedef struct xf_t {
  s7_pointer *data, *cur, *end;
  s7_pointer e;
  int size;
  gc_obj *gc_list;
  struct xf_t *next;
} xf_t;


static s7_pointer *small_ints, *chars;
static s7_pointer real_zero, real_NaN, real_pi, real_one, arity_not_set, max_arity, real_infinity, real_minus_infinity, minus_one, minus_two;


struct s7_scheme {
  opcode_t op;                        /* making this global is much slower! */
  s7_pointer value;
  s7_pointer args;                    /* arguments of current function */
  s7_pointer code, cur_code;          /* current code */
  s7_pointer envir;                   /* curlet */
  token_t tok;

  s7_pointer stack;                   /* stack is a vector */
  unsigned int stack_size;
  s7_pointer *stack_start, *stack_end, *stack_resize_trigger;

  s7_pointer *op_stack, *op_stack_now, *op_stack_end;
  unsigned int op_stack_size, max_stack_size;

  s7_cell **heap, **free_heap, **free_heap_top, **free_heap_trigger, **previous_free_heap_top;
  unsigned int heap_size;
  int gc_freed;

#if WITH_HISTORY
  s7_pointer eval_history1, eval_history2, error_history;
  bool using_history1;
#endif
  /* "int" or "unsigned int" seems safe here:
   *      sizeof(s7_cell) = 48 bytes
   *      so to get more than 2^32 actual objects would require ca 206 GBytes RAM
   *      vectors might be full of the same object (sc->nil for example), so there
   *      we need ca 38 GBytes RAM (8 bytes per pointer).
   */

  gc_obj *permanent_objects;

  s7_pointer protected_objects, protected_accessors;       /* a vector of gc-protected objects */
  unsigned int *gpofl;
  unsigned int protected_objects_size, protected_accessors_size, protected_accessors_loc;
  int gpofl_loc;

  s7_pointer nil;                     /* empty list */
  s7_pointer T;                       /* #t */
  s7_pointer F;                       /* #f */
  s7_pointer eof_object;              /* #<eof> */
  s7_pointer undefined;               /* #<undefined> */
  s7_pointer unspecified;             /* #<unspecified> */
  s7_pointer no_value;                /* the (values) value */
  s7_pointer else_object;             /* else */
  s7_pointer gc_nil;                  /* a marker for an unoccupied slot in sc->protected_objects (and other similar stuff) */

  s7_pointer symbol_table;            /* symbol table */
  s7_pointer rootlet, shadow_rootlet; /* rootlet */
  s7_int rootlet_entries;
  s7_pointer unlet;                   /* original bindings of predefined functions */

  s7_pointer input_port;              /* current-input-port */
  s7_pointer input_port_stack;        /*   input port stack (load and read internally) */
  s7_pointer output_port;             /* current-output-port */
  s7_pointer error_port;              /* current-error-port */
  s7_pointer owlet;                   /* owlet */
  s7_pointer error_type, error_data, error_code, error_line, error_file; /* owlet slots */
  s7_pointer standard_input, standard_output, standard_error;

  s7_pointer sharp_readers;           /* the binding pair for the global *#readers* list */
  s7_pointer load_hook;               /* *load-hook* hook object */
  s7_pointer unbound_variable_hook;   /* *unbound-variable-hook* hook object */
  s7_pointer missing_close_paren_hook;
  s7_pointer error_hook, read_error_hook; /* *error-hook* hook object, and *read-error-hook* */
  s7_pointer direct_str;

  bool gc_off;                        /* gc_off: if true, the GC won't run */
  unsigned int gc_stats;
  unsigned int gensym_counter, cycle_counter, f_class, add_class, multiply_class, subtract_class, equal_class;
  int format_column;
  unsigned long long int capture_let_counter;
  bool symbol_table_is_locked, short_print;
  long long int let_number;
  double default_rationalize_error, morally_equal_float_epsilon, hash_table_float_epsilon;
  s7_int default_hash_table_length, initial_string_port_length, print_length, history_size, true_history_size;
  s7_int max_vector_length, max_string_length, max_list_length, max_vector_dimensions;
  s7_pointer stacktrace_defaults;
  vdims_t *wrap_only;

  char *typnam;
  int typnam_len;
  char *help_arglist;
  int print_width;
  s7_pointer *singletons;

  #define INITIAL_TMP_STR_SIZE 16
  s7_pointer *tmp_strs;

  #define INITIAL_FILE_NAMES_SIZE 8
  s7_pointer *file_names;
  int file_names_size, file_names_top;

  #define INITIAL_STRBUF_SIZE 1024
  unsigned int strbuf_size;
  #define TMPBUF_SIZE 1024
  char *strbuf, *tmpbuf;

  char *read_line_buf;
  unsigned int read_line_buf_size;

  s7_pointer v, w, x, y, z;         /* evaluator local vars */
  s7_pointer temp1, temp2, temp3, temp4, temp5, temp6, temp7, temp8, temp9, temp10, temp11;
  s7_pointer temp_cell, temp_cell_1, temp_cell_2;
  s7_pointer d1, d2, d3, d4;
  s7_pointer t1_1, t2_1, t2_2, t3_1, t3_2, t3_3, z2_1, z2_2;
  s7_pointer a1_1, a2_1, a2_2, a3_1, a3_2, a3_3, a4_1, a4_2, a4_3, a4_4;

  jmp_buf goto_start;
  bool longjmp_ok;
  int setjmp_loc;

  void (*begin_hook)(s7_scheme *sc, bool *val);

  int no_values, current_line, s7_call_line, safety;
  const char *current_file, *s7_call_file, *s7_call_name;

  shared_info *circle_info;
  format_data **fdats;
  int num_fdats;
  s7_pointer elist_1, elist_2, elist_3, elist_4, elist_5, plist_1, plist_2, plist_3;

  s7_pointer *strings, *vectors, *input_ports, *output_ports, *continuations, *c_objects, *hash_tables, *gensyms, *setters;
  unsigned int strings_size, vectors_size, input_ports_size, output_ports_size, continuations_size, c_objects_size, hash_tables_size, gensyms_size, setters_size;
  unsigned int strings_loc, vectors_loc, input_ports_loc, output_ports_loc, continuations_loc, c_objects_loc, hash_tables_loc, gensyms_loc, setters_loc;

  unsigned int syms_tag;
  int ht_iter_tag, baffle_ctr, bignum_precision;
  s7_pointer default_rng;

  /* these symbols are primarily for the generic function search */
  s7_pointer abs_symbol, acos_symbol, acosh_symbol, add_symbol, angle_symbol, append_symbol, apply_symbol, arity_symbol,
             ash_symbol, asin_symbol, asinh_symbol, assoc_symbol, assq_symbol, assv_symbol, atan_symbol, atanh_symbol,
             autoload_symbol, autoloader_symbol,
             byte_vector_symbol, byte_vector_ref_symbol, byte_vector_set_symbol, 
             c_pointer_symbol, caaaar_symbol, caaadr_symbol, caaar_symbol, caadar_symbol, caaddr_symbol, caadr_symbol,
             caar_symbol, cadaar_symbol, cadadr_symbol, cadar_symbol, caddar_symbol, cadddr_symbol, caddr_symbol, cadr_symbol,
             call_cc_symbol, call_with_current_continuation_symbol, call_with_exit_symbol, call_with_input_file_symbol, 
             call_with_input_string_symbol, call_with_output_file_symbol, call_with_output_string_symbol, car_symbol,
             catch_symbol, cdaaar_symbol, cdaadr_symbol, cdaar_symbol, cdadar_symbol, cdaddr_symbol, cdadr_symbol, cdar_symbol,
             cddaar_symbol, cddadr_symbol, cddar_symbol, cdddar_symbol, cddddr_symbol, cdddr_symbol, cddr_symbol, cdr_symbol,
             ceiling_symbol, char_downcase_symbol, char_eq_symbol, char_geq_symbol, char_gt_symbol, char_leq_symbol, char_lt_symbol,
             char_position_symbol, char_to_integer_symbol, char_upcase_symbol, cload_directory_symbol, close_input_port_symbol, 
             close_output_port_symbol, complex_symbol, cons_symbol, copy_symbol, cos_symbol, cosh_symbol, coverlet_symbol, 
             curlet_symbol, current_error_port_symbol, current_input_port_symbol, current_output_port_symbol, cutlet_symbol, 
             denominator_symbol, dilambda_symbol, display_symbol, divide_symbol, dynamic_wind_symbol,
             eq_symbol, error_symbol, eval_string_symbol, eval_symbol, exact_to_inexact_symbol, exp_symbol, expt_symbol,
             features_symbol, fill_symbol, float_vector_ref_symbol, float_vector_set_symbol, float_vector_symbol, floor_symbol, 
             flush_output_port_symbol, for_each_symbol, format_symbol, funclet_symbol,
             gc_symbol, gcd_symbol, gensym_symbol, geq_symbol, get_output_string_symbol, gt_symbol,
             hash_table_entries_symbol, hash_table_ref_symbol, hash_table_set_symbol, hash_table_star_symbol, hash_table_symbol,
             help_symbol, 
             imag_part_symbol, inexact_to_exact_symbol, inlet_symbol, int_vector_ref_symbol, int_vector_set_symbol, int_vector_symbol,
             integer_decode_float_symbol, integer_to_char_symbol, is_aritable_symbol, is_boolean_symbol, is_byte_vector_symbol,
             is_c_object_symbol, is_c_pointer_symbol, is_char_alphabetic_symbol, is_char_lower_case_symbol, is_char_numeric_symbol,
             is_char_symbol, is_char_upper_case_symbol, is_char_whitespace_symbol, is_complex_symbol, is_constant_symbol,
             is_continuation_symbol, is_defined_symbol, is_dilambda_symbol, is_eof_object_symbol, is_eq_symbol, is_equal_symbol,
             is_eqv_symbol, is_even_symbol, is_exact_symbol, is_float_vector_symbol, is_gensym_symbol, is_hash_table_symbol,
             is_inexact_symbol, is_infinite_symbol, is_input_port_symbol, is_int_vector_symbol, is_integer_symbol, is_iterator_symbol,
             is_keyword_symbol, is_let_symbol, is_list_symbol, is_macro_symbol, is_morally_equal_symbol, is_nan_symbol, is_negative_symbol,
             is_null_symbol, is_number_symbol, is_odd_symbol, is_openlet_symbol, is_output_port_symbol, is_pair_symbol, 
             is_port_closed_symbol, is_positive_symbol, is_procedure_symbol, is_proper_list_symbol, is_provided_symbol,
             is_random_state_symbol, is_rational_symbol, is_real_symbol, is_sequence_symbol, is_string_symbol, is_symbol_symbol,
             is_vector_symbol, is_zero_symbol, iterate_symbol, iterator_is_at_end_symbol, iterator_sequence_symbol,
             is_float_symbol, is_integer_or_real_at_end_symbol, is_integer_or_any_at_end_symbol, is_unspecified_symbol,
             keyword_to_symbol_symbol, 
             lcm_symbol, length_symbol, leq_symbol, let_ref_fallback_symbol, let_ref_symbol, let_set_fallback_symbol,
             let_set_symbol, list_ref_symbol, list_set_symbol, list_symbol, list_tail_symbol, load_path_symbol,
             load_symbol, log_symbol, logand_symbol, logbit_symbol, logior_symbol, lognot_symbol, logxor_symbol, lt_symbol,
             magnitude_symbol, make_byte_vector_symbol, make_float_vector_symbol, make_hash_table_symbol, make_int_vector_symbol,
             make_iterator_symbol, string_to_keyword_symbol, make_list_symbol, make_shared_vector_symbol, make_string_symbol,
             make_vector_symbol, map_symbol, max_symbol, member_symbol, memq_symbol, memv_symbol, min_symbol, modulo_symbol,
             multiply_symbol, 
             newline_symbol, not_symbol, number_to_string_symbol, numerator_symbol, 
             object_to_string_symbol, object_to_let_symbol, open_input_file_symbol, open_input_string_symbol, open_output_file_symbol, 
             openlet_symbol, outlet_symbol, owlet_symbol, 
             pair_filename_symbol, pair_line_number_symbol, peek_char_symbol, pi_symbol, port_filename_symbol, port_line_number_symbol,
             procedure_documentation_symbol, procedure_signature_symbol, procedure_source_symbol, provide_symbol,
             quotient_symbol, 
             random_state_symbol, random_state_to_list_symbol, random_symbol, rationalize_symbol, read_byte_symbol,
             read_char_symbol, read_line_symbol, read_string_symbol, read_symbol, real_part_symbol, remainder_symbol,
             require_symbol, reverse_symbol, reverseb_symbol, rootlet_symbol, round_symbol,
             set_car_symbol, set_cdr_symbol, sin_symbol, sinh_symbol, sort_symbol, sqrt_symbol,
             stacktrace_symbol, string_append_symbol, string_downcase_symbol, string_eq_symbol, string_fill_symbol,
             string_geq_symbol, string_gt_symbol, string_leq_symbol, string_lt_symbol, string_position_symbol, string_ref_symbol,
             string_set_symbol, string_symbol, string_to_number_symbol, string_to_symbol_symbol, string_upcase_symbol,
             sublet_symbol, substring_symbol, subtract_symbol, symbol_access_symbol, symbol_symbol, symbol_to_dynamic_value_symbol,
             symbol_to_keyword_symbol, symbol_to_string_symbol, symbol_to_value_symbol, 
             tan_symbol, tanh_symbol, throw_symbol, string_to_byte_vector_symbol, truncate_symbol,
             unlet_symbol, 
             values_symbol, varlet_symbol, vector_append_symbol, vector_dimensions_symbol, vector_fill_symbol, vector_ref_symbol,
             vector_set_symbol, vector_symbol, 
             with_input_from_file_symbol, with_input_from_string_symbol, with_output_to_file_symbol, with_output_to_string_symbol,
             write_byte_symbol, write_char_symbol, write_string_symbol, write_symbol;
         
#if (!WITH_PURE_S7)
  s7_pointer is_char_ready_symbol, char_ci_leq_symbol, char_ci_lt_symbol, char_ci_eq_symbol, char_ci_geq_symbol, char_ci_gt_symbol, 
             let_to_list_symbol, integer_length_symbol, string_ci_leq_symbol, string_ci_lt_symbol, string_ci_eq_symbol,
             string_ci_geq_symbol, string_ci_gt_symbol, string_to_list_symbol, vector_to_list_symbol, string_length_symbol, 
             string_copy_symbol, list_to_string_symbol, list_to_vector_symbol, vector_length_symbol, make_polar_symbol, 
             make_rectangular_symbol;
#endif

  /* s7 env symbols */
  s7_pointer stack_top_symbol, symbol_table_is_locked_symbol, heap_size_symbol, gc_freed_symbol, gc_protected_objects_symbol,
             free_heap_size_symbol, file_names_symbol, symbol_table_symbol, cpu_time_symbol, c_objects_symbol, float_format_precision_symbol,
             stack_size_symbol, rootlet_size_symbol, c_types_symbol, safety_symbol, max_stack_size_symbol, gc_stats_symbol,
             strings_symbol, vectors_symbol, input_ports_symbol, output_ports_symbol, continuations_symbol, hash_tables_symbol, gensyms_symbol,
             catches_symbol, exits_symbol, stack_symbol, default_rationalize_error_symbol, max_string_length_symbol, default_random_state_symbol,
             max_list_length_symbol, max_vector_length_symbol, max_vector_dimensions_symbol, default_hash_table_length_symbol, profile_info_symbol,
             hash_table_float_epsilon_symbol, morally_equal_float_epsilon_symbol, initial_string_port_length_symbol, memory_usage_symbol,
             undefined_identifier_warnings_symbol, print_length_symbol, bignum_precision_symbol, stacktrace_defaults_symbol, history_size_symbol;

  /* syntax symbols et al */
  s7_pointer else_symbol, lambda_symbol, lambda_star_symbol, let_symbol, quote_symbol, unquote_symbol, macroexpand_symbol, 
             define_expansion_symbol, baffle_symbol, with_let_symbol, documentation_symbol, signature_symbol, if_symbol, 
             when_symbol, unless_symbol, begin_symbol, cond_symbol, case_symbol, and_symbol, or_symbol, do_symbol, 
             define_symbol, define_star_symbol, define_constant_symbol, with_baffle_symbol, define_macro_symbol, 
             define_macro_star_symbol, define_bacro_symbol, define_bacro_star_symbol, letrec_symbol, letrec_star_symbol, 
             let_star_symbol, key_rest_symbol, key_allow_other_keys_symbol, key_readable_symbol, value_symbol, type_symbol, 
             baffled_symbol, __func___symbol, set_symbol, body_symbol, class_name_symbol, feed_to_symbol, format_error_symbol, 
             wrong_number_of_args_symbol, read_error_symbol, string_read_error_symbol, syntax_error_symbol, division_by_zero_symbol, 
             no_catch_symbol, io_error_symbol, invalid_escape_function_symbol, wrong_type_arg_symbol, out_of_range_symbol;

  /* optimizer symbols */
  s7_pointer and_p2_symbol, and_p_symbol, and_unchecked_symbol, begin_unchecked_symbol, case_simple_symbol, case_simpler_1_symbol,
             case_simpler_ss_symbol, case_simpler_symbol, case_simplest_ss_symbol, case_simplest_symbol, case_unchecked_symbol,
             cond_all_x_2_symbol, cond_all_x_symbol, cond_s_symbol, cond_simple_symbol, cond_unchecked_symbol, decrement_1_symbol,
             define_constant_unchecked_symbol, define_funchecked_symbol, define_star_unchecked_symbol, define_unchecked_symbol,
             do_unchecked_symbol, dotimes_p_symbol, dox_symbol, if_a_p_p_symbol, if_a_p_symbol, if_and2_p_p_symbol, if_and2_p_symbol,
             if_andp_p_p_symbol, if_andp_p_symbol, if_cc_p_p_symbol, if_cc_p_symbol, if_cs_p_p_symbol, if_cs_p_symbol, if_csc_p_p_symbol,
             if_csc_p_symbol, if_csq_p_p_symbol, if_csq_p_symbol, if_css_p_p_symbol, if_css_p_symbol, if_is_pair_p_p_symbol,
             if_is_pair_p_symbol, if_is_symbol_p_p_symbol, if_is_symbol_p_symbol, if_not_s_p_p_symbol, if_not_s_p_symbol,
             if_opssq_p_p_symbol, if_opssq_p_symbol, if_orp_p_p_symbol, if_orp_p_symbol, if_p_feed_symbol, if_p_p_p_symbol,
             if_p_p_symbol, if_s_opcq_p_p_symbol, if_s_opcq_p_symbol, if_s_p_p_symbol, if_s_p_symbol, if_unchecked_symbol,
             if_z_p_p_symbol, if_z_p_symbol, increment_1_symbol, increment_sa_symbol, increment_saa_symbol, increment_ss_symbol,
             increment_sss_symbol, increment_sz_symbol, lambda_star_unchecked_symbol, lambda_unchecked_symbol, let_all_c_symbol,
             let_all_opsq_symbol, let_all_s_symbol, let_all_x_symbol, let_c_symbol, let_no_vars_symbol, let_one_symbol,
             let_opcq_symbol, let_opsq_p_symbol, let_opsq_symbol, let_opssq_symbol, let_s_symbol, let_star2_symbol,
             let_star_all_x_symbol, let_star_unchecked_symbol, let_unchecked_symbol, let_z_symbol, letrec_star_unchecked_symbol,
             letrec_unchecked_symbol, named_let_no_vars_symbol, named_let_star_symbol, named_let_symbol, or_p2_symbol, or_p_symbol,
             or_unchecked_symbol, quote_unchecked_symbol, safe_do_symbol, safe_dotimes_symbol, set_cons_symbol, set_let_all_x_symbol,
             set_let_s_symbol, set_normal_symbol, set_pair_a_symbol, set_pair_c_p_symbol, set_pair_c_symbol, set_pair_p_symbol,
             set_pair_symbol, set_pair_z_symbol, set_pair_za_symbol, set_pws_symbol, set_symbol_a_symbol, set_symbol_c_symbol,
             set_symbol_opcq_symbol, set_symbol_opsq_symbol, set_symbol_opssq_symbol, set_symbol_opsssq_symbol, set_symbol_p_symbol,
             set_symbol_q_symbol, set_symbol_s_symbol, set_symbol_z_symbol, set_unchecked_symbol, simple_do_a_symbol,
             simple_do_e_symbol, simple_do_p_symbol, simple_do_symbol, unless_s_symbol, unless_unchecked_symbol, when_s_symbol,
             when_unchecked_symbol, with_baffle_unchecked_symbol, with_let_s_symbol, with_let_unchecked_symbol, 
             dox_slot_symbol;

#if WITH_GMP
  s7_pointer bignum_symbol, is_bignum_symbol;
  s7_pointer *bigints, *bigratios, *bigreals, *bignumbers;
  int bigints_size, bigratios_size, bigreals_size, bignumbers_size;
  int bigints_loc, bigratios_loc, bigreals_loc, bignumbers_loc;
#endif

#if WITH_SYSTEM_EXTRAS
  s7_pointer is_directory_symbol, file_exists_symbol, delete_file_symbol, getenv_symbol, system_symbol, directory_to_list_symbol, file_mtime_symbol;
#endif

  /* setter and quasiquote functions */
  s7_pointer vector_set_function, string_set_function, list_set_function, hash_table_set_function, let_set_function, object_set_function,
             qq_list_function, qq_apply_values_function, qq_append_function, multivector_function, 
             apply_function, vector_function;

  s7_pointer wrong_type_arg_info, out_of_range_info, simple_wrong_type_arg_info, simple_out_of_range_info;
  s7_pointer too_many_arguments_string, not_enough_arguments_string, division_by_zero_error_string;
  s7_pointer *safe_lists, *syn_docs; /* prebuilt evaluator arg lists, syntax doc strings */

  s7_pointer autoload_table, libraries, profile_info;
  const char ***autoload_names;
  int *autoload_names_sizes;
  bool **autoloaded_already;
  int autoload_names_loc, autoload_names_top;
  port_t *port_heap;

  int format_depth;
  int slash_str_size;
  char *slash_str;

  xf_t *cur_rf;
  xf_t *rf_free_list, *rf_stack;
  bool undefined_identifier_warnings;
};

typedef enum {USE_DISPLAY, USE_WRITE, USE_READABLE_WRITE, USE_WRITE_WRONG} use_write_t;

#define NUM_SAFE_LISTS 16
#define INITIAL_AUTOLOAD_NAMES_SIZE 4


static s7_pointer prepackaged_type_names[NUM_TYPES];

static bool t_number_p[NUM_TYPES], t_real_p[NUM_TYPES], t_rational_p[NUM_TYPES];
static bool t_simple_p[NUM_TYPES];
static bool t_big_number_p[NUM_TYPES];
static bool t_structure_p[NUM_TYPES];
static bool t_any_macro_p[NUM_TYPES];
static bool t_any_closure_p[NUM_TYPES];
static bool t_has_closure_let[NUM_TYPES];
static bool t_sequence_p[NUM_TYPES];
static bool t_vector_p[NUM_TYPES];
static bool t_applicable_p[NUM_TYPES];

static void init_types(void)
{
  int i;
  for (i = 0; i < NUM_TYPES; i++)
    {
      t_number_p[i] = false;
      t_real_p[i] = false;
      t_rational_p[i] = false;
      t_simple_p[i] = false;
      t_structure_p[i] = false;
      t_any_macro_p[i] = false;
      t_any_closure_p[i] = false;
      t_has_closure_let[i] = false;
      t_sequence_p[i] = false;
      t_vector_p[i] = false;
      t_applicable_p[i] = false;
    }
  t_number_p[T_INTEGER] = true;
  t_number_p[T_RATIO] = true;
  t_number_p[T_REAL] = true;
  t_number_p[T_COMPLEX] = true;

  t_rational_p[T_INTEGER] = true;
  t_rational_p[T_RATIO] = true;

  t_real_p[T_INTEGER] = true;
  t_real_p[T_RATIO] = true;
  t_real_p[T_REAL] = true;

  t_big_number_p[T_BIG_INTEGER] = true;
  t_big_number_p[T_BIG_RATIO] = true;
  t_big_number_p[T_BIG_REAL] = true;
  t_big_number_p[T_BIG_COMPLEX] = true;

  t_structure_p[T_PAIR] = true;
  t_structure_p[T_VECTOR] = true;
  t_structure_p[T_HASH_TABLE] = true;
  t_structure_p[T_SLOT] = true;
  t_structure_p[T_LET] = true;
  t_structure_p[T_ITERATOR] = true;

  t_sequence_p[T_NIL] = true;
  t_sequence_p[T_PAIR] = true;
  t_sequence_p[T_STRING] = true;
  t_sequence_p[T_VECTOR] = true;
  t_sequence_p[T_INT_VECTOR] = true;
  t_sequence_p[T_FLOAT_VECTOR] = true;
  t_sequence_p[T_HASH_TABLE] = true;
  t_sequence_p[T_LET] = true;
  t_sequence_p[T_C_OBJECT] = true;

  t_vector_p[T_VECTOR] = true;
  t_vector_p[T_INT_VECTOR] = true;
  t_vector_p[T_FLOAT_VECTOR] = true;

  t_applicable_p[T_PAIR] = true;
  t_applicable_p[T_STRING] = true;
  t_applicable_p[T_VECTOR] = true;
  t_applicable_p[T_INT_VECTOR] = true;
  t_applicable_p[T_FLOAT_VECTOR] = true;
  t_applicable_p[T_HASH_TABLE] = true;
  t_applicable_p[T_ITERATOR] = true;
  t_applicable_p[T_LET] = true;
  t_applicable_p[T_C_OBJECT] = true;
  t_applicable_p[T_C_MACRO] = true;
  t_applicable_p[T_MACRO] = true;
  t_applicable_p[T_BACRO] = true;
  t_applicable_p[T_MACRO_STAR] = true;
  t_applicable_p[T_BACRO_STAR] = true;
  t_applicable_p[T_SYNTAX] = true;
  t_applicable_p[T_C_FUNCTION] = true;
  t_applicable_p[T_C_FUNCTION_STAR] = true;
  t_applicable_p[T_C_ANY_ARGS_FUNCTION] = true;
  t_applicable_p[T_C_OPT_ARGS_FUNCTION] = true;
  t_applicable_p[T_C_RST_ARGS_FUNCTION] = true;
  t_applicable_p[T_CLOSURE] = true;
  t_applicable_p[T_CLOSURE_STAR] = true;
  t_applicable_p[T_GOTO] = true;
  t_applicable_p[T_CONTINUATION] = true;

  t_any_macro_p[T_C_MACRO] = true;
  t_any_macro_p[T_MACRO] = true;
  t_any_macro_p[T_BACRO] = true;
  t_any_macro_p[T_MACRO_STAR] = true;
  t_any_macro_p[T_BACRO_STAR] = true;

  t_any_closure_p[T_CLOSURE] = true;
  t_any_closure_p[T_CLOSURE_STAR] = true;

  t_has_closure_let[T_MACRO] = true;
  t_has_closure_let[T_BACRO] = true;
  t_has_closure_let[T_MACRO_STAR] = true;
  t_has_closure_let[T_BACRO_STAR] = true;
  t_has_closure_let[T_CLOSURE] = true;
  t_has_closure_let[T_CLOSURE_STAR] = true;

  t_simple_p[T_NIL] = true;
  t_simple_p[T_UNIQUE] = true;
  t_simple_p[T_BOOLEAN] = true;
  t_simple_p[T_CHARACTER] = true;
  t_simple_p[T_SYMBOL] = true;
  t_simple_p[T_SYNTAX] = true;
  t_simple_p[T_C_MACRO] = true;
  t_simple_p[T_C_FUNCTION] = true;
  t_simple_p[T_C_FUNCTION_STAR] = true;
  t_simple_p[T_C_ANY_ARGS_FUNCTION] = true;
  t_simple_p[T_C_OPT_ARGS_FUNCTION] = true;
  t_simple_p[T_C_RST_ARGS_FUNCTION] = true;
  /* not completely sure about the next ones */
  t_simple_p[T_LET] = true;
  t_simple_p[T_INPUT_PORT] = true;
  t_simple_p[T_OUTPUT_PORT] = true;
}

#if WITH_HISTORY
#define current_code(Sc) car(Sc->cur_code)
#define set_current_code(Sc, Code) do {Sc->cur_code = cdr(Sc->cur_code); set_car(Sc->cur_code, Code);} while (0)
#define mark_current_code(Sc) do {int i; s7_pointer p; for (p = Sc->cur_code, i = 0; i < sc->history_size; i++, p = cdr(p)) S7_MARK(car(p));} while (0)
#else
#define current_code(Sc) Sc->cur_code
#define set_current_code(Sc, Code) Sc->cur_code = Code
#define mark_current_code(Sc) S7_MARK(Sc->cur_code)
#endif

#define typeflag(p)  ((p)->tf.flag)
#define typesflag(p) ((p)->tf.sflag)

static s7_scheme *hidden_sc = NULL;

#if DEBUGGING
  static const char *check_name(int typ);
  static s7_pointer check_seti(s7_scheme *sc, s7_pointer x, const char *func, int line);
  static s7_pointer check_ref(s7_pointer p, int expected_type, const char *func, int line, const char *func1, const char *func2);
  static s7_pointer check_ref2(s7_pointer p, int expected_type, int other_type, const char *func, int line, const char *func1, const char *func2);
  static s7_pointer check_ref3(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref4(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref5(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref6(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref7(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref8(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref9(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref10(s7_pointer p, const char *func, int line);
  static s7_pointer check_ref11(s7_pointer p, const char *func, int line);
  static s7_pointer check_nref(s7_pointer p, const char *func, int line);
  static void print_gc_info(s7_pointer obj, int line);

  static s7_pointer opt1_1(s7_scheme *sc, s7_pointer p, unsigned int role, const char *func, int line);
  static s7_pointer set_opt1_1(s7_scheme *sc, s7_pointer p, s7_pointer x, unsigned int role, const char *func, int line);
  static s7_pointer opt2_1(s7_scheme *sc, s7_pointer p, unsigned int role, const char *func, int line);
  static void set_opt2_1(s7_scheme *sc, s7_pointer p, s7_pointer x, unsigned int role, const char *func, int line);
  static s7_pointer opt3_1(s7_scheme *sc, s7_pointer p, unsigned int role, const char *func, int line);
  static void set_opt3_1(s7_scheme *sc, s7_pointer p, s7_pointer x, unsigned int role, const char *func, int line);

  static unsigned long long int s_hash_1(s7_scheme *sc, s7_pointer p, const char *func, int line);
  static void set_s_hash_1(s7_scheme *sc, s7_pointer p, unsigned long long int x, const char *func, int line);
  static const char *s_name_1(s7_scheme *sc, s7_pointer p, const char *func, int line);
  static void set_s_name_1(s7_scheme *sc, s7_pointer p, const char *str, const char *func, int line);
  static unsigned int s_line_1(s7_scheme *sc, s7_pointer p, const char *func, int line);
  static void set_s_line_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line);
  static unsigned int s_len_1(s7_scheme *sc, s7_pointer p, const char *func, int line);
  static void set_s_len_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line);
  static unsigned int s_op_1(s7_scheme *sc, s7_pointer p, const char *func, int line);
  static void set_s_op_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line);
  static unsigned int s_syn_op_1(s7_scheme *sc, s7_pointer p, const char *func, int line);
  static void set_s_syn_op_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line);

  #define unchecked_type(p)           ((p)->tf.type_field)
  #define type(p) ({unsigned char _t_; _t_ = (p)->tf.type_field; if (((_t_ == T_FREE)) || (_t_ >= NUM_TYPES)) print_gc_info(p, __LINE__); _t_;})

  #define set_type(p, f)						\
    do {								\
      p->previous_alloc_line = p->current_alloc_line;			\
      p->previous_alloc_func = p->current_alloc_func;			\
      p->previous_alloc_type = p->current_alloc_type;			\
      p->current_alloc_line = __LINE__;					\
      p->current_alloc_func = __func__;					\
      p->current_alloc_type = f;					\
      p->uses++; p->clear_line = 0;					\
      if ((((f) & 0xff) == T_FREE) || (((f) & 0xff) >= NUM_TYPES))	\
        fprintf(stderr, "%d: set free %p type to %x\n", __LINE__, p, f); \
      else								\
	{								\
	  if (((typeflag(p) & T_IMMUTABLE) != 0) && ((typeflag(p) != (f))))						\
	    fprintf(stderr, "%d: set immutable %p type %x to %x\n", __LINE__, p, unchecked_type(p), f); \
	  if (((typeflag(p) & T_LINE_NUMBER) != 0) && (((typeflag(p)) & 0xff) == T_PAIR) && (((f) & T_LINE_NUMBER) == 0)) \
            fprintf(stderr, "%d unsets line_number\n", __LINE__); \
	}								\
      typeflag(p) = f;							\
    } while (0)

  #define clear_type(p) do {p->clear_line = __LINE__; typeflag(p) = T_FREE;} while (0)

  /* these check most s7cell field references (and many type bits) for consistency */
  #define _TI(P)   check_ref(P, T_INTEGER,           __func__, __LINE__, NULL, NULL)
  #define _TR(P)   check_ref(P, T_REAL,              __func__, __LINE__, NULL, NULL)
  #define _TF(P)   check_ref2(P, T_RATIO, T_INTEGER, __func__, __LINE__, NULL, NULL)
  #define _TZ(P)   check_ref(P, T_COMPLEX,           __func__, __LINE__, NULL, NULL)
  #define _TBgi(P) check_ref(P, T_BIG_INTEGER,       __func__, __LINE__, "sweep", NULL)
  #define _TBgr(P) check_ref(P, T_BIG_REAL,          __func__, __LINE__, "sweep", NULL)
  #define _TBgf(P) check_ref(P, T_BIG_RATIO,         __func__, __LINE__, "sweep", NULL)
  #define _TBgz(P) check_ref(P, T_BIG_COMPLEX,       __func__, __LINE__, "sweep", NULL)

  #define _TChr(P) check_ref(P, T_CHARACTER,         __func__, __LINE__, NULL, NULL)
  #define _TCtr(P) check_ref(P, T_COUNTER,           __func__, __LINE__, NULL, NULL)
  #define _TPtr(P) check_ref(P, T_C_POINTER,         __func__, __LINE__, NULL, NULL)
  #define _TBfl(P) check_ref(P, T_BAFFLE,            __func__, __LINE__, NULL, NULL)
  #define _TGot(P) check_ref(P, T_GOTO,              __func__, __LINE__, NULL, NULL)
  #define _TStk(P) check_ref(P, T_STACK,             __func__, __LINE__, NULL, NULL)
  #define _TPair(P) check_ref(P, T_PAIR,             __func__, __LINE__, NULL, NULL)
  #define _TCat(P) check_ref(P, T_CATCH,             __func__, __LINE__, NULL, NULL)
  #define _TDyn(P) check_ref(P, T_DYNAMIC_WIND,      __func__, __LINE__, NULL, NULL)
  #define _TSlt(P) check_ref(P, T_SLOT,              __func__, __LINE__, NULL, NULL)
  #define _TSlp(P) check_ref2(P, T_SLOT, T_PAIR,     __func__, __LINE__, NULL, NULL)
  #define _TSln(P) check_ref2(P, T_SLOT, T_NIL,      __func__, __LINE__, NULL, NULL)
  #define _TSld(P) check_ref2(P, T_SLOT, T_UNIQUE,   __func__, __LINE__, NULL, NULL)
  #define _TSyn(P) check_ref(P, T_SYNTAX,            __func__, __LINE__, NULL, NULL)
  #define _TMac(P) check_ref(P, T_C_MACRO,           __func__, __LINE__, NULL, NULL)
  #define _TLet(P) check_ref(P, T_LET,               __func__, __LINE__, NULL, NULL)
  #define _TLid(P) check_ref2(P, T_LET, T_NIL,       __func__, __LINE__, NULL, NULL)
  #define _TRan(P) check_ref(P, T_RANDOM_STATE,      __func__, __LINE__, NULL, NULL)
  #define _TLst(P) check_ref2(P, T_PAIR, T_NIL,      __func__, __LINE__, "gc", NULL)
  #define _TStr(P) check_ref(P, T_STRING,            __func__, __LINE__, "sweep", NULL)
  #define _TObj(P) check_ref(P, T_C_OBJECT,          __func__, __LINE__, "free_object", NULL)
  #define _THsh(P) check_ref(P, T_HASH_TABLE,        __func__, __LINE__, "sweep", "free_hash_table")
  #define _TItr(P) check_ref(P, T_ITERATOR,          __func__, __LINE__, "sweep", NULL)
  #define _TCon(P) check_ref(P, T_CONTINUATION,      __func__, __LINE__, "sweep", NULL)
  #define _TFvc(P) check_ref(P, T_FLOAT_VECTOR,      __func__, __LINE__, "sweep", NULL)
  #define _TIvc(P) check_ref(P, T_INT_VECTOR,        __func__, __LINE__, "sweep", NULL)
  #define _TSym(P) check_ref(P, T_SYMBOL,            __func__, __LINE__, "sweep", "remove_gensym_from_symbol_table")

  #define _TPrt(P) check_ref3(P,                     __func__, __LINE__) /* input|output_port, or free */
  #define _TVec(P) check_ref4(P,                     __func__, __LINE__) /* any vector or free */
  #define _TClo(P) check_ref5(P,                     __func__, __LINE__) /* has closure let */
  #define _TFnc(P) check_ref6(P,                     __func__, __LINE__) /* any c_function|c_macro */
  #define _TNum(P) check_ref7(P,                     __func__, __LINE__) /* any number (not bignums I think) */
  #define _TSeq(P) check_ref8(P,                     __func__, __LINE__) /* any sequence or structure */
  #define _TMet(P) check_ref9(P,                     __func__, __LINE__) /* anything that might contain a method */
  #define _TArg(P) check_ref10(P,                    __func__, __LINE__) /* closure arg (list, symbol) */
  #define _TApp(P) check_ref11(P,                    __func__, __LINE__) /* setter (any_procedure or #f) */
  #define _NFre(P) check_nref(P,                     __func__, __LINE__) /* not free */
  #define _TSet(P) check_seti(sc, P,                 __func__, __LINE__) /* set of immutable value */

#else
  #define unchecked_type(p)           ((p)->tf.type_field)
  #define type(p)                     ((p)->tf.type_field)
  #define set_type(p, f)              typeflag(p) = f
  #define clear_type(p)               typeflag(p) = T_FREE
  #define _TSet(P)                    P
  #define _TI(P)                      P
  #define _TR(P)                      P
  #define _TF(P)                      P
  #define _TZ(P)                      P
  #define _TBgi(P)                    P
  #define _TBgr(P)                    P
  #define _TBgf(P)                    P
  #define _TBgz(P)                    P
  #define _TStr(P)                    P
  #define _TSyn(P)                    P
  #define _TChr(P)                    P
  #define _TObj(P)                    P
  #define _TCtr(P)                    P
  #define _THsh(P)                    P
  #define _TItr(P)                    P
  #define _TPtr(P)                    P
  #define _TBfl(P)                    P
  #define _TGot(P)                    P
  #define _TCon(P)                    P
  #define _TStk(P)                    P
  #define _TPrt(P)                    P
  #define _TIvc(P)                    P
  #define _TFvc(P)                    P
  #define _TVec(P)                    P
  #define _TPair(P)                   P
  #define _TRan(P)                    P
  #define _TDyn(P)                    P
  #define _TCat(P)                    P
  #define _TClo(P)                    P
  #define _TFnc(P)                    P
  #define _TSlt(P)                    P
  #define _TSln(P)                    P
  #define _TSld(P)                    P
  #define _TSlp(P)                    P
  #define _TSym(P)                    P
  #define _TLet(P)                    P
  #define _TLid(P)                    P
  #define _TLst(P)                    P
  #define _TNum(P)                    P
  #define _TSeq(P)                    P
  #define _TMet(P)                    P
  #define _TMac(P)                    P
  #define _TArg(P)                    P
  #define _TApp(P)                    P
  #define _NFre(P)                    P
#endif

#define is_number(P)                  t_number_p[type(P)]
#define is_integer(P)                 (type(P) == T_INTEGER)
#define is_real(P)                    t_real_p[type(P)]
#define is_rational(P)                t_rational_p[type(P)]
#define is_big_number(p)              t_big_number_p[type(p)]
#define is_t_integer(p)               (type(p) == T_INTEGER)
#define is_t_ratio(p)                 (type(p) == T_RATIO)
#define is_t_real(p)                  (type(p) == T_REAL)
#define is_t_complex(p)               (type(p) == T_COMPLEX)
#define is_t_big_integer(p)           (type(p) == T_BIG_INTEGER)
#define is_t_big_ratio(p)             (type(p) == T_BIG_RATIO)
#define is_t_big_real(p)              (type(p) == T_BIG_REAL)
#define is_t_big_complex(p)           (type(p) == T_BIG_COMPLEX)

#define is_free(p)                    (type(p) == T_FREE)
#define is_free_and_clear(p)          (typeflag(p) == T_FREE)
#define is_simple(P)                  t_simple_p[type(P)]
#define has_structure(P)              t_structure_p[type(P)]

#define is_any_macro(P)               t_any_macro_p[type(P)]
#define is_any_closure(P)             t_any_closure_p[type(P)]
#define is_procedure_or_macro(P)      ((t_any_macro_p[type(P)]) || ((typeflag(P) & T_PROCEDURE) != 0))
#define is_any_procedure(P)           (type(P) >= T_CLOSURE)
#define has_closure_let(P)            t_has_closure_let[type(P)]

#define is_simple_sequence(P)         (t_sequence_p[type(P)])
#define is_sequence(P)                ((t_sequence_p[type(P)]) || (has_methods(P)))
#define is_applicable(P)              (t_applicable_p[type(P)])
/* this misses #() which actually is not applicable to anything, probably "" also, and inapplicable c-objects like random-state */


/* the layout of these bits does matter in several cases -- in particular, don't use the second byte for anything
 *   that might shadow SYNTACTIC_PAIR and OPTIMIZED_PAIR.
 */
#define TYPE_BITS                     8

#define T_KEYWORD                     (1 << (TYPE_BITS + 0))
#define is_keyword(p)                 ((typesflag(_NFre(p)) & T_KEYWORD) != 0)
/* this bit distinguishes a symbol from a symbol that is also a keyword
 * this should be ok in the second byte because keywords are constants in s7 (never syntax)
 */

#define T_SYNTACTIC                   (1 << (TYPE_BITS + 1))
#define is_syntactic(p)               ((typesflag(_NFre(p)) & T_SYNTACTIC) != 0)
#define is_syntactic_symbol(p)        ((typesflag(_NFre(p)) & (T_SYNTACTIC | 0xff)) == (T_SYMBOL | T_SYNTACTIC))
#define SYNTACTIC_TYPE                (unsigned short)(T_SYMBOL | T_DONT_EVAL_ARGS | T_SYNTACTIC)
#define SYNTACTIC_PAIR                (unsigned short)(T_PAIR | T_SYNTACTIC)
/* this marks symbols that represent syntax objects, it should be in the second byte */
#define set_syntactic_pair(p)         typeflag(p) = (SYNTACTIC_PAIR | (typeflag(p) & 0xffff0000))

#define T_PROCEDURE                   (1 << (TYPE_BITS + 2))
#define is_procedure(p)               ((typesflag(_NFre(p)) & T_PROCEDURE) != 0)
/* closure, c_function, applicable object, goto or continuation, should be in second byte */

#define T_OPTIMIZED                   (1 << (TYPE_BITS + 3))
#define set_optimized(p)              typesflag(_TPair(p)) |= T_OPTIMIZED
#define clear_optimized(p)            typesflag(_TPair(p)) &= (~T_OPTIMIZED)
#define OPTIMIZED_PAIR                (unsigned short)(T_PAIR | T_OPTIMIZED)
#define is_optimized(p)               (typesflag(p) == OPTIMIZED_PAIR)
/*   this is faster than the bit extraction above and the same speed as xor */
/* optimizer flag for an expression that has optimization info, it should be in the second byte
 */

#define T_SAFE_CLOSURE                (1 << (TYPE_BITS + 4))
#define is_safe_closure(p)            ((typesflag(_NFre(p)) & T_SAFE_CLOSURE) != 0)
#define set_safe_closure(p)           typesflag(p) |= T_SAFE_CLOSURE
#define clear_safe_closure(p)         typesflag(p) &= (~T_SAFE_CLOSURE)
/* optimizer flag for a closure body that is completely simple (every expression is safe)
 *   set_safe_closure happens only in optimize_lambda, clear only in procedure_source, bits only here
 *   this has to be separate from T_SAFE_PROCEDURE, and should be in the second byte.
 *   It can be set on either the body (a pair) or the closure itself.
 */

#define T_DONT_EVAL_ARGS              (1 << (TYPE_BITS + 5))
#define dont_eval_args(p)             ((typesflag(_NFre(p)) & T_DONT_EVAL_ARGS) != 0)
/* this marks things that don't evaluate their arguments */

#define T_EXPANSION                   (1 << (TYPE_BITS + 6))
#define is_expansion(p)               ((typesflag(_NFre(p)) & T_EXPANSION) != 0)
#define clear_expansion(p)            typesflag(_TSym(p)) &= (~T_EXPANSION)
/* this marks the symbol associated with a run-time macro and distinguishes the value from an ordinary macro */

#define T_MULTIPLE_VALUE              (1 << (TYPE_BITS + 7))
#define is_multiple_value(p)          ((typesflag(_NFre(p)) & T_MULTIPLE_VALUE) != 0)
#define set_multiple_value(p)         typesflag(_TPair(p)) |= T_MULTIPLE_VALUE
#define clear_multiple_value(p)       typesflag(_TPair(p)) &= (~T_MULTIPLE_VALUE)
#define multiple_value(p)             p
/* this bit marks a list (from "values") that is waiting for a
 *    chance to be spliced into its caller's argument list.  It is normally
 *    on only for a very short time.
 */

#define T_MATCHED                     T_MULTIPLE_VALUE
#define is_matched_pair(p)            ((typesflag(_TPair(p)) & T_MATCHED) != 0)
#define set_match_pair(p)             typesflag(_TPair(p)) |= T_MATCHED
#define clear_match_pair(p)           typesflag(_TPair(p)) &= (~T_MATCHED)
#define is_matched_symbol(p)          ((typesflag(_TSym(p)) & T_MATCHED) != 0)
#define set_match_symbol(p)           typesflag(_TSym(p)) |= T_MATCHED
#define clear_match_symbol(p)         typesflag(_TSym(p)) &= (~T_MATCHED)

#define T_GLOBAL                      (1 << (TYPE_BITS + 8))
#define is_global(p)                  ((typeflag(_TSym(p)) & T_GLOBAL) != 0)
#define set_global(p)                 typeflag(_TSym(p)) |= T_GLOBAL
#if 0
  /* to find who is stomping on our symbols: */
  static char *object_to_truncated_string(s7_scheme *sc, s7_pointer p, int len);

  static void set_local_1(s7_scheme *sc, s7_pointer symbol, const char *func, int line)
  {
    if ((is_global(symbol)) || (is_syntactic(symbol)))
      fprintf(stderr, "%s[%d]: %s%s%s in %s\n", func, line, BOLD_TEXT, DISPLAY(symbol), UNBOLD_TEXT, DISPLAY_80(current_code(sc)));
    typeflag(symbol) = (typeflag(symbol) & ~(T_DONT_EVAL_ARGS | T_GLOBAL | T_SYNTACTIC));
  }
  #define set_local(Symbol) set_local_1(sc, Symbol, __func__, __LINE__)
#else
#define set_local(p)                  typeflag(_TSym(p)) &= ~(T_DONT_EVAL_ARGS | T_GLOBAL | T_SYNTACTIC)
#endif
/* this marks something defined (bound) at the top-level, and never defined locally */

#define T_UNSAFE_DO                   T_GLOBAL
#define is_unsafe_do(p)               ((typeflag(_TPair(p)) & T_UNSAFE_DO) != 0)
#define set_unsafe_do(p)              typeflag(_TPair(p)) |= T_UNSAFE_DO
#define is_unsafe_sort(p)             is_unsafe_do(p)
#define set_unsafe_sort(p)            set_unsafe_do(p)
/* marks do-loops (and sort functions) that resist optimization */

#define T_COLLECTED                   (1 << (TYPE_BITS + 9))
#define is_collected(p)               ((typeflag(_TSeq(p)) & T_COLLECTED) != 0)
#define set_collected(p)              typeflag(_TSeq(p)) |= T_COLLECTED
/* #define clear_collected(p)         typeflag(_TSeq(p)) &= (~T_COLLECTED) */
/* this is a transient flag used by the printer to catch cycles.  It affects only objects that have structure.  
 *   We can't use a low bit (bit 7 for example), because collect_shared_info inspects the object's type.
 */

#define T_LINE_NUMBER                 (1 << (TYPE_BITS + 10))
#define has_line_number(p)            ((typeflag(_TPair(p)) & T_LINE_NUMBER) != 0)
#define set_has_line_number(p)        typeflag(_TPair(p)) |= T_LINE_NUMBER
/* pair in question has line/file info added during read, or the environment has function placement info 
 *   this bit should not be in the first byte -- SYNTACTIC_PAIR ignores it.
 */

#define T_LOADER_PORT                 T_LINE_NUMBER
#define is_loader_port(p)             ((typeflag(_TPrt(p)) & T_LOADER_PORT) != 0)
#define set_loader_port(p)            typeflag(_TPrt(p)) |= T_LOADER_PORT
#define clear_loader_port(p)          typeflag(_TPrt(p)) &= (~T_LOADER_PORT)
/* to block random load-time reads from screwing up the load process, this bit marks a port used by the loader */

#define T_HAS_ACCESSOR                T_LINE_NUMBER
#define symbol_has_accessor(p)        ((typeflag(_TSym(p)) & T_HAS_ACCESSOR) != 0)
#define symbol_set_has_accessor(p)    typeflag(_TSym(p)) |= T_HAS_ACCESSOR
#define slot_has_accessor(p)          ((typeflag(_TSlt(p)) & T_HAS_ACCESSOR) != 0)
#define slot_set_has_accessor(p)      typeflag(_TSlt(p)) |= T_HAS_ACCESSOR
/* marks a slot or symbol that has a setter */

#define T_WITH_LET_LET                T_LINE_NUMBER
#define is_with_let_let(p)            ((typeflag(_TLet(p)) & T_WITH_LET_LET) != 0)
#define set_with_let_let(p)           typeflag(_TLet(p)) |= T_WITH_LET_LET
/* marks a let that is the argument to with-let */

#define T_SIMPLE_DEFAULTS             T_LINE_NUMBER
#define has_simple_defaults(p)        ((typeflag(_TFnc(p)) & T_SIMPLE_DEFAULTS) != 0)
#define set_simple_defaults(p)        typeflag(_TFnc(p)) |= T_SIMPLE_DEFAULTS
#define clear_simple_defaults(p)      typeflag(_TFnc(p)) &= (~T_SIMPLE_DEFAULTS)
/* flag c_func_star arg defaults that need GC protection */

#define T_SHARED                      (1 << (TYPE_BITS + 11))
#define is_shared(p)                  ((typeflag(_TSeq(p)) & T_SHARED) != 0)
#define set_shared(p)                 typeflag(_TSeq(p)) |= T_SHARED
/* #define clear_shared(p)            typeflag(_TSeq(p)) &= (~T_SHARED) */
#define clear_collected_and_shared(p) typeflag(p) &= (~(T_COLLECTED | T_SHARED)) /* this can clear free cells = calloc */

#define T_OVERLAY                     (1 << (TYPE_BITS + 12))
#define set_overlay(p)                typeflag(_TPair(p)) |= T_OVERLAY
#define is_overlaid(p)                ((typeflag(_TPair(p)) & T_OVERLAY) != 0)
/* optimizer flag that marks a cell whose opt_back [ie opt1] points to the previous cell in a list */

#define T_SAFE_PROCEDURE              (1 << (TYPE_BITS + 13))
#define is_safe_procedure(p)          ((typeflag(_NFre(p)) & T_SAFE_PROCEDURE) != 0)
/* applicable objects that do not return or modify their arg list directly (no :rest arg in particular),
 *    and that can't call apply themselves either directly or via s7_call, and that don't mess with the stack.
 */

#define T_CHECKED                     (1 << (TYPE_BITS + 14))
#define set_checked(p)                typeflag(_TPair(p)) |= T_CHECKED
#define is_checked(p)                 ((typeflag(_TPair(p)) & T_CHECKED) != 0)
#define clear_checked(p)              typeflag(_TPair(p)) &= (~T_CHECKED)

#define set_checked_slot(p)           typeflag(_TSlt(p)) |= T_CHECKED
#define is_checked_slot(p)            ((typeflag(_TSlt(p)) & T_CHECKED) != 0)
#define is_not_checked_slot(p)        ((typeflag(_TSlt(p)) & T_CHECKED) == 0)


#define T_UNSAFE                      (1 << (TYPE_BITS + 15))
#define set_unsafe(p)                 typeflag(_TPair(p)) |= T_UNSAFE
#define set_unsafely_optimized(p)     typeflag(_TPair(p)) |= (T_UNSAFE | T_OPTIMIZED)
#define is_unsafe(p)                  ((typeflag(_TPair(p)) & T_UNSAFE) != 0)
#define clear_unsafe(p)               typeflag(_TPair(p)) &= (~T_UNSAFE)
#define is_safely_optimized(p)        ((typeflag(p) & (T_OPTIMIZED | T_UNSAFE)) == T_OPTIMIZED)
/* optimizer flag saying "this expression is not completely self-contained.  It might involve the stack, etc" */

#define T_CLEAN_SYMBOL                T_UNSAFE
#define is_clean_symbol(p)            ((typeflag(_TSym(p)) & T_CLEAN_SYMBOL) != 0)
#define set_clean_symbol(p)           typeflag(_TSym(p)) |= T_CLEAN_SYMBOL
/* set if we know the symbol name can be printed without quotes (slashification) */

#define T_IMMUTABLE                   (1 << (TYPE_BITS + 16))
#define is_immutable(p)               ((typeflag(_NFre(p)) & T_IMMUTABLE) != 0)
#define is_immutable_port(p)          ((typeflag(_TPrt(p)) & T_IMMUTABLE) != 0)
#define is_immutable_symbol(p)        ((typeflag(_TSym(p)) & T_IMMUTABLE) != 0)
#define is_immutable_integer(p)       ((typeflag(_TI(p)) & T_IMMUTABLE) != 0)
#define is_immutable_real(p)          ((typeflag(_TR(p)) & T_IMMUTABLE) != 0)
#define set_immutable(p)              typeflag(_TSym(p)) |= T_IMMUTABLE
/* immutable means the value can't be changed via set! or bind -- this is separate from the symbol access stuff
 * this bit can't be in the second byte -- with-let, for example, is immutable, but we use SYNTACTIC_TYPE to 
 * recognize syntax in do loop optimizations.
 */

#define T_SETTER                      (1 << (TYPE_BITS + 17))
#define set_setter(p)                 typeflag(_TSym(p)) |= T_SETTER
#define is_setter(p)                  ((typeflag(_TSym(p)) & T_SETTER) != 0)
/* optimizer flag for a procedure that sets some variable (set-car! for example). */

#define T_ALLOW_OTHER_KEYS            T_SETTER
#define set_allow_other_keys(p)       typeflag(_TPair(p)) |= T_ALLOW_OTHER_KEYS
#define allows_other_keys(p)          ((typeflag(_TPair(p)) & T_ALLOW_OTHER_KEYS) != 0)
/* marks arglist that allows keyword args other than those in the parameter list; can't allow
 *   (define* (f :allow-other-keys)...) because there's only one nil, and besides, it does say "other".
 */

#define T_MUTABLE                     (1 << (TYPE_BITS + 18))
#define is_mutable(p)                 ((typeflag(_TNum(p)) & T_MUTABLE) != 0)
/* #define set_mutable(p)             typeflag(_TNum(p)) |= T_MUTABLE */
/* used for mutable numbers */

#define T_MARK_SEQ                    T_MUTABLE
#define is_mark_seq(p)                ((typeflag(_TItr(p)) & T_MARK_SEQ) != 0)
#define set_mark_seq(p)               typeflag(_TItr(p)) |= T_MARK_SEQ
/* used in iterators for GC mark of sequence */

#define T_BYTE_VECTOR                 T_MUTABLE
#define is_byte_vector(p)             ((typeflag(_TStr(p)) & T_BYTE_VECTOR) != 0)
#define set_byte_vector(p)            typeflag(_TStr(p)) |= T_BYTE_VECTOR
/* marks a string that the caller considers a byte_vector */

#define T_STEPPER                     T_MUTABLE
#define is_stepper(p)                 ((typeflag(_TSlt(p)) & T_STEPPER) != 0)
#define set_stepper(p)                typeflag(_TSlt(p)) |= T_STEPPER
bool s7_is_stepper(s7_pointer p)      {return(is_stepper(p));}
/* marks a slot that holds a do-loop's step variable (if int, can be numerator=current, denominator=end) */

#define T_SAFE_STEPPER                (1 << (TYPE_BITS + 19))
#define is_safe_stepper(p)            ((typeflag(_TSlp(p)) & T_SAFE_STEPPER) != 0)
#define set_safe_stepper(p)           typeflag(_TSlp(p)) |= T_SAFE_STEPPER
#define is_unsafe_stepper(p)          ((typeflag(_TSlp(p)) & (T_STEPPER | T_SAFE_STEPPER)) == T_STEPPER)
/* an experiment */

#define T_PRINT_NAME                  T_SAFE_STEPPER
#define has_print_name(p)             ((typeflag(_TNum(p)) & T_PRINT_NAME) != 0)
#define set_has_print_name(p)         typeflag(_TNum(p)) |= T_PRINT_NAME
/* marks numbers that have a saved version of their string representation */

#define T_POSSIBLY_SAFE               T_SAFE_STEPPER
#define is_possibly_safe(p)           ((typeflag(_TFnc(p)) & T_POSSIBLY_SAFE) != 0)
#define set_is_possibly_safe(p)       typeflag(_TFnc(p)) |= T_POSSIBLY_SAFE
/* marks c_functions that are not always unsafe -- this bit didn't work out as intended */

#define T_HAS_SET_FALLBACK            T_SAFE_STEPPER
#define T_HAS_REF_FALLBACK            T_MUTABLE
#define has_ref_fallback(p)           ((typeflag(_TLid(p)) & T_HAS_REF_FALLBACK) != 0)
#define has_set_fallback(p)           ((typeflag(_TLid(p)) & T_HAS_SET_FALLBACK) != 0)
#define set_has_ref_fallback(p)       typeflag(_TLet(p)) |= T_HAS_REF_FALLBACK
#define set_has_set_fallback(p)       typeflag(_TLet(p)) |= T_HAS_SET_FALLBACK
#define set_all_methods(p, e)         typeflag(_TLet(p)) |= (typeflag(e) & (T_HAS_METHODS | T_HAS_REF_FALLBACK | T_HAS_SET_FALLBACK))

#define T_COPY_ARGS                   (1 << (TYPE_BITS + 20))
#define needs_copied_args(p)          ((typeflag(_NFre(p)) & T_COPY_ARGS) != 0)
/* this marks something that might mess with its argument list, it should not be in the second byte */

#define T_GENSYM                      (1 << (TYPE_BITS + 21))
#define is_gensym(p)                  ((typeflag(_TSym(p)) & T_GENSYM) != 0)
/* symbol is from gensym (GC-able etc) */

#define T_SIMPLE_ARGS                 T_GENSYM
#define has_simple_args(p)            ((typeflag(_TPair(p)) & T_SIMPLE_ARGS) != 0)
#define set_simple_args(p)            typeflag(_TPair(p)) |= T_SIMPLE_ARGS
/* are all lambda* default values simple? */

#define T_LIST_IN_USE                 T_GENSYM
#define list_is_in_use(p)             ((typeflag(_TPair(p)) & T_LIST_IN_USE) != 0)
#define set_list_in_use(p)            typeflag(_TPair(p)) |= T_LIST_IN_USE
#define clear_list_in_use(p)          typeflag(_TPair(p)) &= (~T_LIST_IN_USE)
/* these could all be one permanent list, indexed from inside, and this bit is never actually protecting anything across a call */

#define T_FUNCTION_ENV                T_GENSYM
#define is_function_env(p)            ((typeflag(_TLet(p)) & T_FUNCTION_ENV) != 0)
#define set_function_env(p)           typeflag(_TLet(p)) |= T_FUNCTION_ENV
/* this marks a funclet */

#define T_DOCUMENTED                  T_GENSYM
#define is_documented(p)              ((typeflag(_TStr(p)) & T_DOCUMENTED) != 0)
#define set_documented(p)             typeflag(_TStr(p)) |= T_DOCUMENTED
/* this marks a symbol that has documentation (bit is set on name cell) */

#define T_HAS_METHODS                 (1 << (TYPE_BITS + 22))
#define has_methods(p)                ((typeflag(_NFre(p)) & T_HAS_METHODS) != 0)
#define set_has_methods(p)            typeflag(_TMet(p)) |= T_HAS_METHODS
#define clear_has_methods(p)          typeflag(_TMet(p)) &= (~T_HAS_METHODS)
/* this marks an environment or closure that is "opened" up to generic functions etc
 * don't reuse this bit if possible
 */

#define T_GC_MARK                     0x80000000            /* (1 << (TYPE_BITS + 23)) but that makes gcc unhappy */
#define is_marked(p)                  ((typeflag(p) &  T_GC_MARK) != 0)
#define set_mark(p)                   typeflag(_NFre(p)) |= T_GC_MARK
#define clear_mark(p)                 typeflag(p) &= (~T_GC_MARK)
/* using bit 23 for this makes a big difference in the GC */


static int not_heap = -1;
#define heap_location(p)              (p)->hloc
#define not_in_heap(p)                ((_NFre(p))->hloc < 0)
#define unheap(p)                     (p)->hloc = not_heap--

#define is_eof(p)                     (_NFre(p) == sc->eof_object)
#define is_true(Sc, p)                ((_NFre(p)) != Sc->F)
#define is_false(Sc, p)               ((_NFre(p)) == Sc->F)

#ifdef _MSC_VER
  #define MS_WINDOWS 1
  static s7_pointer make_boolean(s7_scheme *sc, bool val) {if (val) return(sc->T); return(sc->F);}
#else
  #define MS_WINDOWS 0
  #define make_boolean(sc, Val)       ((Val) ? sc->T : sc->F)
#endif

#define is_pair(p)                    (type(p) == T_PAIR)
#define is_null(p)                    ((_NFre(p)) == sc->nil)
#define is_not_null(p)                ((_NFre(p)) != sc->nil)


#if (!DEBUGGING)

#define opt1(p, r)                    ((p)->object.cons.opt1)
#define set_opt1(p, x, r)             (p)->object.cons.opt1 = x
#define opt2(p, r)                    ((p)->object.cons.opt2)
#define set_opt2(p, x, r)             (p)->object.cons.opt2 = (s7_pointer)(x)
#define opt3(p, r)                    ((p)->object.cons.opt3)
#define set_opt3(p, x, r)             do {(p)->object.cons.opt3 = x; typeflag(p) &= ~(T_OPTIMIZED | T_LINE_NUMBER);} while (0)

#define pair_line(p)                  (p)->object.sym_cons.line
#define pair_set_line(p, X)           (p)->object.sym_cons.line = X
#define pair_raw_hash(p)              (p)->object.sym_cons.hash
#define pair_set_raw_hash(p, X)       (p)->object.sym_cons.hash = X
#define pair_raw_len(p)               (p)->object.sym_cons.op
#define pair_set_raw_len(p, X)        (p)->object.sym_cons.op = X
#define pair_raw_name(p)              (p)->object.sym_cons.fstr
#define pair_set_raw_name(p, X)       (p)->object.sym_cons.fstr = X

/* opt1 == raw_hash, opt2 == raw_name, opt3 == line+op|len, but hash/name/len only apply to the symbol table so there's no collision */

#else

/* these 3 fields (or 8 counting sym_cons) hold most of the varigated optimizer info, so they are used in many conflicting ways.
 * the bits and funcs here try to track each such use, and report any cross-talk or collisions.
 * all of this machinery vanishes if debugging is turned off.
 */
#define S_NAME                        (1 << 26)
#define S_HASH                        (1 << 27)
#define S_OP                          (1 << 28)
#define S_LINE                        (1 << 29)
#define S_LEN                         (1 << 30)
#define S_SYNOP                       0x80000000 /* (1 << 31) */

#define E_SET                         (1 << 0)
#define E_FAST                        (1 << 6)   /* fast list in member/assoc circular list check */
#define E_CFUNC                       (1 << 7)   /* c-function */
#define E_CLAUSE                      (1 << 8)   /* case clause */
#define E_BACK                        (1 << 9)   /* back pointer for doubly-linked list */
#define E_LAMBDA                      (1 << 10)  /* lambda(*) */
#define E_SYM                         (1 << 11)  /* symbol */
#define E_PAIR                        (1 << 12)  /* pair */
#define E_CON                         (1 << 13)  /* constant from eval's point of view */
#define E_GOTO                        (1 << 14)  /* call-with-exit exit func */
#define E_VECTOR                      (1 << 15)  /* vector (any kind) */
#define E_ANY                         (1 << 16)  /* anything -- deliberate unchecked case */
#define E_SLOT                        (1 << 17)  /* slot */
#define E_MASK                        (E_FAST | E_CFUNC | E_CLAUSE | E_BACK | E_LAMBDA | E_SYM | E_PAIR | E_CON | E_GOTO | E_VECTOR | E_ANY | E_SLOT | S_HASH)

#define opt1_is_set(p)                (((p)->debugger_bits & E_SET) != 0)
#define set_opt1_is_set(p)            (p)->debugger_bits |= E_SET
#define opt1_role_matches(p, Role)    (((p)->debugger_bits & E_MASK) == Role)
#define set_opt1_role(p, Role)        (p)->debugger_bits = (Role | ((p)->debugger_bits & ~E_MASK))
#define opt1(p, Role)                 opt1_1(hidden_sc, _TPair(p), Role, __func__, __LINE__)
#define set_opt1(p, x, Role)          set_opt1_1(hidden_sc, _TPair(p), x, Role, __func__, __LINE__)

#define F_SET                         (1 << 1)   /* bit 18 is free */
#define F_KEY                         (1 << 19)  /* case key */
#define F_SLOW                        (1 << 20)  /* slow list in member/assoc circular list check */
#define F_SYM                         (1 << 21)  /* symbol */
#define F_PAIR                        (1 << 22)  /* pair */
#define F_CON                         (1 << 23)  /* constant as above */
#define F_CALL                        (1 << 24)  /* c-func */
#define F_LAMBDA                      (1 << 25)  /* lambda form */
#define F_MASK                        (F_KEY | F_SLOW | F_SYM | F_PAIR | F_CON | F_CALL | F_LAMBDA | S_NAME)

#define opt2_is_set(p)                (((p)->debugger_bits & F_SET) != 0)
#define set_opt2_is_set(p)            (p)->debugger_bits |= F_SET
#define opt2_role_matches(p, Role)    (((p)->debugger_bits & F_MASK) == Role)
#define set_opt2_role(p, Role)        (p)->debugger_bits = (Role | ((p)->debugger_bits & ~F_MASK))
#define opt2(p, Role)                 opt2_1(hidden_sc, _TPair(p), Role, __func__, __LINE__)
#define set_opt2(p, x, Role)          set_opt2_1(hidden_sc, _TPair(p), (s7_pointer)x, Role, __func__, __LINE__)

/* opt3 collides with optimization and line number stuff (T_LINE_NUMBER, T_OPTIMIZED) */
#define G_SET                         (1 << 2)
#define G_ARGLEN                      (1 << 3)  /* arglist length */
#define G_SYM                         (1 << 4)  /* expression symbol access */
#define G_AND                         (1 << 5)  /* and second clause */
#define G_MASK                        (G_ARGLEN | G_SYM | G_AND | S_OP | S_LINE | S_LEN | S_SYNOP)

#define opt3_is_set(p)                (((p)->debugger_bits & G_SET) != 0)
#define set_opt3_is_set(p)            (p)->debugger_bits |= G_SET
#define opt3_role_matches(p, Role)    (((p)->debugger_bits & G_MASK) == Role)
#define set_opt3_role(p, Role)        (p)->debugger_bits = (Role | ((p)->debugger_bits & ~G_MASK))
#define opt3(p, Role)                 opt3_1(hidden_sc, _TPair(p), Role, __func__, __LINE__)
#define set_opt3(p, x, Role)          set_opt3_1(hidden_sc, _TPair(p), x, Role, __func__, __LINE__)

/* opt1 == s_hash, opt2 == s_fstr, opt3 == s_op|len|line and op==len so they are contradictory (but only op/line|opt3 actually collide)
 * line|len|op: unsigned int set G_SET and S_* if S_LEN -> not op and vice versa
 * another collider: pair_syntax_op|optimize_op below.  Both need bits: S_SYNOP?
 */

#define pair_line(p)                  s_line_1(sc, _TPair(p), __func__, __LINE__)
#define pair_set_line(p, X)           set_s_line_1(sc, _TPair(p), X, __func__, __LINE__)
#define pair_raw_hash(p)              s_hash_1(sc, _TPair(p), __func__, __LINE__)
#define pair_set_raw_hash(p, X)       set_s_hash_1(sc, _TPair(p), X, __func__, __LINE__)
#define pair_raw_len(p)               s_len_1(sc, _TPair(p), __func__, __LINE__)
#define pair_set_raw_len(p, X)        set_s_len_1(sc, _TPair(p), X, __func__, __LINE__)
#define pair_raw_name(p)              s_name_1(sc, _TPair(p), __func__, __LINE__)
#define pair_set_raw_name(p, X)       set_s_name_1(sc, _TPair(p), X, __func__, __LINE__)
#endif

#define opt_fast(P)                   _TLst(opt1(P,              E_FAST))
#define set_opt_fast(P, X)            set_opt1(P, _TPair(X),      E_FAST)
#define opt_back(P)                   _TPair(opt1(P,              E_BACK))
#define set_opt_back(P)               set_opt1(cdr(P), _TPair(P), E_BACK)
#define has_opt_back(P)               (cdr(opt_back(P)) == P )
#define opt_cfunc(P)                  opt1(P,                    E_CFUNC)
#define set_opt_cfunc(P, X)           set_opt1(P, X,             E_CFUNC)
#define opt_lambda_unchecked(P)       opt1(P,                    E_LAMBDA)
#define opt_lambda(P)                 _TClo(opt1(P,              E_LAMBDA))
#define set_opt_lambda(P, X)          set_opt1(P, X,             E_LAMBDA)
#define opt_goto(P)                   _TGot(opt1(P,              E_GOTO))
#define set_opt_goto(P, X)            set_opt1(P, _TGot(X),      E_GOTO)
#define opt_vector(P)                 _TVec(opt1(P,              E_VECTOR))
#define set_opt_vector(P, X)          set_opt1(P, _TVec(X),      E_VECTOR)
#define opt_clause(P)                 opt1(P,                    E_CLAUSE)
#define set_opt_clause(P, X)          set_opt1(P, X,             E_CLAUSE)
#define opt_sym1(P)                   _TSym(opt1(P,              E_SYM))
#define set_opt_sym1(P, X)            set_opt1(P, _TSym(X),      E_SYM)
#define opt_pair1(P)                  _TLst(opt1(P,              E_PAIR))
#define set_opt_pair1(P, X)           set_opt1(P, _TLst(X),      E_PAIR)
#define opt_con1(P)                   opt1(P,                    E_CON)
#define set_opt_con1(P, X)            set_opt1(P, X,             E_CON)
#define opt_any1(P)                   opt1(P,                    E_ANY)
#define opt_slot1(P)                  _TSlt(opt1(P,              E_SLOT))
#define set_opt_slot1(P, X)           set_opt1(P, _TSlt(X),      E_SLOT)

#define c_callee(f)                   ((s7_function)opt2(f,      F_CALL))
#define c_call(f)                     ((s7_function)opt2(f,      F_CALL))
#define set_c_call(f, X)              set_opt2(f, (s7_pointer)X, F_CALL)
#define opt_key(P)                    opt2(P,                    F_KEY)
#define set_opt_key(P, X)             set_opt2(P, X,             F_KEY)
#define opt_slow(P)                   _TLst(opt2(P,              F_SLOW))
#define set_opt_slow(P, X)            set_opt2(P, _TPair(X),      F_SLOW)
#define opt_sym2(P)                   _TSym(opt2(P,              F_SYM))
#define set_opt_sym2(P, X)            set_opt2(P, _TSym(X),      F_SYM)
#define opt_pair2(P)                  _TLst(opt2(P,              F_PAIR))
#define set_opt_pair2(P, X)           set_opt2(P, _TLst(X),      F_PAIR)
#define opt_con2(P)                   opt2(P,                    F_CON)
#define set_opt_con2(P, X)            set_opt2(P, X,             F_CON)
#define opt_lambda2(P)                _TPair(opt2(P,              F_LAMBDA))
#define set_opt_lambda2(P, X)         set_opt2(P, _TPair(X),      F_LAMBDA)

#define arglist_length(P)             _TI(opt3(cdr(P),           G_ARGLEN))
#define set_arglist_length(P, X)      set_opt3(cdr(P), _TI(X),   G_ARGLEN)
#define opt_sym3(P)                   _TSym(opt3(P,              G_SYM))
#define set_opt_sym3(P, X)            set_opt3(P, _TSym(X),      G_SYM)
#define opt_and_2_test(P)             _TPair(opt3(P,              G_AND))
#define set_opt_and_2_test(P, X)      set_opt3(P, _TPair(X),      G_AND)


#define car(p)                        (_TLst(p))->object.cons.car
#define set_car(p, Val)               (_TLst(p))->object.cons.car = _NFre(Val)
#define cdr(p)                        (_TLst(p))->object.cons.cdr
#define set_cdr(p, Val)               (_TLst(p))->object.cons.cdr = _NFre(Val)
#define unchecked_car(p)              (_NFre(p))->object.cons.car
#define unchecked_cdr(p)              (_NFre(p))->object.cons.cdr

#define caar(p)                       car(car(p))
#define cadr(p)                       car(cdr(p))
#define set_cadr(p, Val)              (_TLst(p))->object.cons.cdr->object.cons.car = _NFre(Val)
#define cdar(p)                       cdr(car(p))
#define set_cdar(p, Val)              (_TLst(p))->object.cons.car->object.cons.cdr = _NFre(Val)
#define cddr(p)                       cdr(cdr(p))

#define caaar(p)                      car(car(car(p)))
#define cadar(p)                      car(cdr(car(p)))
#define cdadr(p)                      cdr(car(cdr(p)))
#define caddr(p)                      car(cdr(cdr(p)))
#define set_caddr(p, Val)             (_TLst(p))->object.cons.cdr->object.cons.cdr->object.cons.car = _NFre(Val)
#define caadr(p)                      car(car(cdr(p)))
#define cdaar(p)                      cdr(car(car(p)))
#define cdddr(p)                      cdr(cdr(cdr(p)))
#define cddar(p)                      cdr(cdr(car(p)))

#define caaadr(p)                     car(car(car(cdr(p))))
#define caadar(p)                     car(car(cdr(car(p))))
#define cadaar(p)                     car(cdr(car(car(p))))
#define cadddr(p)                     car(cdr(cdr(cdr(p))))
#define caaddr(p)                     car(car(cdr(cdr(p))))
#define cddddr(p)                     cdr(cdr(cdr(cdr(p))))
#define caddar(p)                     car(cdr(cdr(car(p))))
#define cdadar(p)                     cdr(car(cdr(car(p))))
#define cdaddr(p)                     cdr(car(cdr(cdr(p))))
#define caaaar(p)                     car(car(car(car(p))))
#define cadadr(p)                     car(cdr(car(cdr(p))))
#define cdaadr(p)                     cdr(car(car(cdr(p))))
#define cdaaar(p)                     cdr(car(car(car(p))))
#define cdddar(p)                     cdr(cdr(cdr(car(p))))
#define cddadr(p)                     cdr(cdr(car(cdr(p))))
#define cddaar(p)                     cdr(cdr(car(car(p))))

#if WITH_GCC
  /* slightly tricky because cons can be called recursively */
  #define cons(Sc, A, B)   ({s7_pointer _X_, _A_, _B_; _A_ = A; _B_ = B; new_cell(sc, _X_, T_PAIR | T_SAFE_PROCEDURE); set_car(_X_, _A_); set_cdr(_X_, _B_); _X_;})
#else
  #define cons(Sc, A, B)              s7_cons(Sc, A, B)
#endif

#define list_1(Sc, A)                 cons(Sc, A, Sc->nil)
#define list_2(Sc, A, B)              cons_unchecked(Sc, A, cons(Sc, B, Sc->nil))
#define list_3(Sc, A, B, C)           cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons(Sc, C, Sc->nil)))
#define list_4(Sc, A, B, C, D)        cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons_unchecked(Sc, C, cons(Sc, D, Sc->nil))))

#define is_string(p)                  (type(p) == T_STRING)
#define string_value(p)               (_TStr(p))->object.string.svalue
#define string_length(p)              (_TStr(p))->object.string.length
#define string_hash(p)                (_TStr(p))->object.string.hash
#define string_needs_free(p)          (_TStr(p))->object.string.str_ext.needs_free
#define string_temp_true_length(p)    (_TStr(p))->object.string.str_ext.temp_len

#define tmpbuf_malloc(P, Len)         do {if ((Len) < TMPBUF_SIZE) P = sc->tmpbuf; else P = (char *)malloc((Len) * sizeof(char));} while (0)
#define tmpbuf_calloc(P, Len)         do {if ((Len) < TMPBUF_SIZE) {P = sc->tmpbuf; memset((void *)P, 0, Len);} else P = (char *)calloc(Len, sizeof(char));} while (0)
#define tmpbuf_free(P, Len)           do {if ((Len) >= TMPBUF_SIZE) free(P);} while (0)

#define character(p)                  (_TChr(p))->object.chr.c
#define upper_character(p)            (_TChr(p))->object.chr.up_c
#define is_char_alphabetic(p)         (_TChr(p))->object.chr.alpha_c
#define is_char_numeric(p)            (_TChr(p))->object.chr.digit_c
#define is_char_whitespace(p)         (_TChr(p))->object.chr.space_c
#define is_char_uppercase(p)          (_TChr(p))->object.chr.upper_c
#define is_char_lowercase(p)          (_TChr(p))->object.chr.lower_c
#define character_name(p)             (_TChr(p))->object.chr.c_name
#define character_name_length(p)      (_TChr(p))->object.chr.length

#if (!DEBUGGING)
  #define optimize_op(p)              (_TPair(p))->object.sym_cons.op
  #define set_optimize_op(P, Op)      optimize_op(P) = Op
#else
  #define optimize_op(p)              s_op_1(hidden_sc, _TPair(p), __func__, __LINE__)
  #define set_optimize_op(p, Op)      set_s_op_1(hidden_sc, _TPair(p), Op, __func__, __LINE__)
#endif

#define optimize_op_match(P, Q)       ((is_optimized(P)) && ((optimize_op(P) & 0xfffe) == Q))
#define op_no_hop(P)                  (optimize_op(P) & 0xfffe)
#define clear_hop(P)                  set_optimize_op(P, op_no_hop(P))
#define clear_optimize_op(P)          set_optimize_op(P, 0)
#define set_safe_optimize_op(P, Q)    do {set_optimized(P); set_optimize_op(P, Q);} while (0)
#define set_unsafe_optimize_op(P, Q)  do {set_unsafely_optimized(P); set_optimize_op(P, Q);} while (0)

#define is_symbol(p)                  (type(p) == T_SYMBOL)
#define symbol_name_cell(p)           _TStr((_TSym(p))->object.sym.name)
#define symbol_set_name_cell(p, S)    (_TSym(p))->object.sym.name = _TStr(S)
#define symbol_name(p)                string_value(symbol_name_cell(p))
#define symbol_name_length(p)         string_length(symbol_name_cell(p))
#define symbol_hmap(p)                s7_int_abs(heap_location(p))
#define symbol_global_accessor_index(p) (symbol_name_cell(p))->object.string.str_ext.accessor
#define symbol_id(p)                  (_TSym(p))->object.sym.id
#define symbol_set_id(p, X)           (_TSym(p))->object.sym.id = X
/* we need 64-bits here, since we don't want this thing to wrap around, and frames are created at a great rate
 *    callgrind says this is faster than an unsigned int!
 */
#define symbol_syntax_op(p)           (_TSym(p))->object.sym.op

#define global_slot(p)                (_TSym(p))->object.sym.global_slot
#define set_global_slot(p, Val)       (_TSym(p))->object.sym.global_slot = _TSld(Val)
#define initial_slot(p)               (symbol_name_cell(p))->object.string.initial_slot
#define set_initial_slot(p, Val)      (symbol_name_cell(p))->object.string.initial_slot = _TSld(Val)
#define local_slot(p)                 (_TSym(p))->object.sym.local_slot
#define set_local_slot(p, Val)        (_TSym(p))->object.sym.local_slot = _TSln(Val)
#define keyword_symbol(p)             (symbol_name_cell(p))->object.string.doc.ksym
#define keyword_set_symbol(p, Val)    (symbol_name_cell(p))->object.string.doc.ksym = _TSym(Val)
#define symbol_help(p)                (symbol_name_cell(p))->object.string.doc.documentation
#define symbol_tag(p)                 (_TSym(p))->object.sym.tag
#define symbol_set_tag(p, Val)        (_TSym(p))->object.sym.tag = Val
#define symbol_has_help(p)            (is_documented(symbol_name_cell(p)))
#define symbol_set_has_help(p)        set_documented(symbol_name_cell(p))

#define symbol_set_local(Symbol, Id, Slot) do {set_local_slot(Symbol, Slot); symbol_set_id(Symbol, Id);} while (0)
/* set slot before id in case Slot is an expression that tries to find the current Symbol slot (using its old Id obviously) */

#define is_slot(p)                    (type(p) == T_SLOT)
#define slot_value(p)                 _NFre((_TSlt(p))->object.slt.val)
#define slot_set_value(p, Val)        (_TSlt(p))->object.slt.val = _NFre(Val)
#define slot_symbol(p)                _TSym((_TSlt(p))->object.slt.sym)
#define slot_set_symbol(p, Sym)       (_TSlt(p))->object.slt.sym = _TSym(Sym)
#define next_slot(p)                  (_TSlt(p))->object.slt.nxt
#define set_next_slot(p, Val)         (_TSlt(p))->object.slt.nxt = _TSln(Val)
#define slot_pending_value(p)         (_TSlt(p))->object.slt.pending_value
#define slot_set_pending_value(p, Val) (_TSlt(p))->object.slt.pending_value = _NFre(Val)
#define slot_expression(p)            (_TSlt(p))->object.slt.expr
#define slot_set_expression(p, Val)   (_TSlt(p))->object.slt.expr = _NFre(Val)
#define slot_accessor(p)              slot_expression(p)
#define slot_set_accessor(p, Val)     slot_expression(p) = _TApp(Val)

#define is_syntax(p)                  (type(p) == T_SYNTAX)
#define syntax_symbol(p)              _TSym((_TSyn(p))->object.syn.symbol)
#define syntax_set_symbol(p, Sym)     (_TSyn(p))->object.syn.symbol = _TSym(Sym)
#define syntax_opcode(p)              (_TSyn(p))->object.syn.op
#define syntax_min_args(p)            (_TSyn(p))->object.syn.min_args
#define syntax_max_args(p)            (_TSyn(p))->object.syn.max_args
#define syntax_documentation(p)       sc->syn_docs[syntax_opcode(p)]
#define syntax_rp(p)                  (_TSyn(p))->object.syn.rp
#define syntax_ip(p)                  (_TSyn(p))->object.syn.ip
#define syntax_pp(p)                  (_TSyn(p))->object.syn.pp

#if (!DEBUGGING)
  #define pair_syntax_op(p)           (p)->object.sym_cons.op
  #define pair_set_syntax_op(p, X)    (p)->object.sym_cons.op = X
#else
  #define pair_syntax_op(p)           s_syn_op_1(hidden_sc, _TPair(p), __func__, __LINE__)
  #define pair_set_syntax_op(p, Op)   set_s_syn_op_1(hidden_sc, _TPair(p), Op, __func__, __LINE__)
#endif
#define pair_syntax_symbol(P)         car(opt_back(P))
static void pair_set_syntax_symbol(s7_pointer p, s7_pointer op) {set_car(opt_back(p), op); pair_set_syntax_op(opt_back(p), symbol_syntax_op(op));}

#define ROOTLET_SIZE 512
#define let_id(p)                     (_TLid(p))->object.envr.id
#define is_let(p)                     (type(p) == T_LET)
#define let_slots(p)                  (_TLet(p))->object.envr.slots
#define let_set_slots(p, Slot)        (_TLet(p))->object.envr.slots = _TSln(Slot)
#define outlet(p)                     (_TLet(p))->object.envr.nxt
#define set_outlet(p, ol)             (_TLet(p))->object.envr.nxt = _TLid(ol)
#define funclet_function(p)           _TSym((_TLet(p))->object.envr.edat.efnc.function)
#define funclet_set_function(p, F)    (_TLet(p))->object.envr.edat.efnc.function = _TSym(F)
#define let_line(p)                   (_TLet(p))->object.envr.edat.efnc.line
#define let_set_line(p, L)            (_TLet(p))->object.envr.edat.efnc.line = L
#define let_file(p)                   (_TLet(p))->object.envr.edat.efnc.file
#define let_set_file(p, F)            (_TLet(p))->object.envr.edat.efnc.file = F
#define dox_slot1(p)                  _TSlt((_TLet(p))->object.envr.edat.dox.dox1)
#define dox_set_slot1(p, S)           (_TLet(p))->object.envr.edat.dox.dox1 = _TSlt(S)
#define dox_slot2(p)                  _TSlt((_TLet(p))->object.envr.edat.dox.dox2)
#define dox_set_slot2(p, S)           (_TLet(p))->object.envr.edat.dox.dox2 = _TSlt(S)

#define unique_name(p)                (p)->object.unq.name
#define unique_name_length(p)         (p)->object.unq.len
#define is_unspecified(p)             (type(p) == T_UNSPECIFIED)
#define unique_cdr(p)                 (p)->object.unq.unused_nxt

#define vector_length(p)              ((p)->object.vector.length)
#define vector_element(p, i)          ((p)->object.vector.elements.objects[i])
#define vector_elements(p)            (p)->object.vector.elements.objects
#define vector_getter(p)              (_TVec(p))->object.vector.vget
#define vector_setter(p)              (_TVec(p))->object.vector.vset
#define int_vector_element(p, i)      ((_TIvc(p))->object.vector.elements.ints[i])
#define int_vector_elements(p)        (_TIvc(p))->object.vector.elements.ints
#define float_vector_element(p, i)    ((_TFvc(p))->object.vector.elements.floats[i])
#define float_vector_elements(p)      (_TFvc(p))->object.vector.elements.floats
#define is_normal_vector(p)           (type(p) == T_VECTOR)
#define is_int_vector(p)              (type(p) == T_INT_VECTOR)
#define is_float_vector(p)            (type(p) == T_FLOAT_VECTOR)

#define vector_ndims(p)               ((_TVec(p))->object.vector.dim_info->ndims)
#define vector_dimension(p, i)        ((_TVec(p))->object.vector.dim_info->dims[i])
#define vector_dimensions(p)          ((_TVec(p))->object.vector.dim_info->dims)
#define vector_offset(p, i)           ((_TVec(p))->object.vector.dim_info->offsets[i])
#define vector_offsets(p)             ((_TVec(p))->object.vector.dim_info->offsets)
#define vector_dimension_info(p)      ((_TVec(p))->object.vector.dim_info)
#define shared_vector(p)              ((_TVec(p))->object.vector.dim_info->original)
#define vector_rank(p)                ((vector_dimension_info(p)) ? vector_ndims(p) : 1)
#define vector_has_dimensional_info(p) (vector_dimension_info(p))
#define vector_elements_allocated(p)  ((_TVec(p))->object.vector.dim_info->elements_allocated)
#define vector_dimensions_allocated(p) ((_TVec(p))->object.vector.dim_info->dimensions_allocated)

#define is_hash_table(p)              (type(p) == T_HASH_TABLE)
#define hash_table_mask(p)            (_THsh(p))->object.hasher.mask
#define hash_table_element(p, i)      ((_THsh(p))->object.hasher.elements[i])
#define hash_table_elements(p)        (_THsh(p))->object.hasher.elements
#define hash_table_entries(p)         (_THsh(p))->object.hasher.entries
#define hash_table_checker(p)         (_THsh(p))->object.hasher.hash_func
#define hash_table_mapper(p)          (_THsh(p))->object.hasher.loc
#define hash_table_checker_locked(p)  (hash_table_mapper(p) != default_hash_map)
#define hash_table_procedures(p)      _TLst((_THsh(p))->object.hasher.dproc)
#define hash_table_set_procedures(p, Lst) (_THsh(p))->object.hasher.dproc = _TLst(Lst)
#define hash_table_procedures_checker(p) car(hash_table_procedures(p))
#define hash_table_procedures_mapper(p) cdr(hash_table_procedures(p))

#define is_iterator(p)                (type(p) == T_ITERATOR)
#define iterator_sequence(p)          (_TItr(p))->object.iter.obj
#define iterator_position(p)          (_TItr(p))->object.iter.lc.loc
#define iterator_length(p)            (_TItr(p))->object.iter.lw.len
#define iterator_slow(p)              _TLst((_TItr(p))->object.iter.lw.slow)
#define iterator_set_slow(p, Val)     (_TItr(p))->object.iter.lw.slow = _TLst(Val)
#define iterator_hash_current(p)      (_TItr(p))->object.iter.lw.hcur
#define iterator_current(p)           (_TItr(p))->object.iter.cur
#define iterator_current_slot(p)      _TSln((_TItr(p))->object.iter.lc.lcur)
#define iterator_set_current_slot(p, Val) (_TItr(p))->object.iter.lc.lcur = _TSln(Val)
#define iterator_let_cons(p)          (_TItr(p))->object.iter.cur
#define iterator_next(p)              (_TItr(p))->object.iter.next
#define iterator_is_at_end(p)         (iterator_next(p) == iterator_finished)

#define ITERATOR_END eof_object
#define ITERATOR_END_NAME "#<eof>"

#define is_input_port(p)              (type(p) == T_INPUT_PORT)
#define is_output_port(p)             (type(p) == T_OUTPUT_PORT)
#define port_port(p)                  (_TPrt(p))->object.prt.port
#define port_type(p)                  (_TPrt(p))->object.prt.ptype
#define is_string_port(p)             (port_type(p) == STRING_PORT)
#define is_file_port(p)               (port_type(p) == FILE_PORT)
#define is_function_port(p)           (port_type(p) == FUNCTION_PORT)
#define port_line_number(p)           (_TPrt(p))->object.prt.line_number
#define port_file_number(p)           (_TPrt(p))->object.prt.file_number
#define port_filename(p)              port_port(p)->filename
#define port_filename_length(p)       port_port(p)->filename_length
#define port_file(p)                  port_port(p)->file
#define port_is_closed(p)             (_TPrt(p))->object.prt.is_closed
#define port_data(p)                  (_TPrt(p))->object.prt.data
#define port_data_size(p)             (_TPrt(p))->object.prt.size
#define port_position(p)              (_TPrt(p))->object.prt.point
#define port_needs_free(p)            port_port(p)->needs_free
#define port_output_function(p)       port_port(p)->output_function
#define port_input_function(p)        port_port(p)->input_function
#define port_original_input_string(p) port_port(p)->orig_str
#define port_read_character(p)        port_port(p)->read_character
#define port_read_line(p)             port_port(p)->read_line
#define port_display(p)               port_port(p)->display
#define port_write_character(p)       port_port(p)->write_character
#define port_write_string(p)          port_port(p)->write_string
#define port_read_semicolon(p)        port_port(p)->read_semicolon
#define port_read_white_space(p)      port_port(p)->read_white_space
#define port_read_name(p)             port_port(p)->read_name
#define port_read_sharp(p)            port_port(p)->read_sharp
#define port_gc_loc(p)                port_port(p)->gc_loc

#define is_c_function(f)              (type(f) >= T_C_FUNCTION)
#define is_c_function_star(f)         (type(f) == T_C_FUNCTION_STAR)
#define is_any_c_function(f)          (type(f) >= T_C_FUNCTION_STAR)
#define c_function_data(f)            (_TFnc(f))->object.fnc.c_proc
#define c_function_call(f)            (_TFnc(f))->object.fnc.ff
#define c_function_required_args(f)   (_TFnc(f))->object.fnc.required_args
#define c_function_optional_args(f)   (_TFnc(f))->object.fnc.optional_args
#define c_function_has_rest_arg(f)    (_TFnc(f))->object.fnc.rest_arg
#define c_function_all_args(f)        (_TFnc(f))->object.fnc.all_args
#define c_function_setter(f)          _TApp((_TFnc(f))->object.fnc.setter)
#define c_function_set_setter(f, Val) (_TFnc(f))->object.fnc.setter = _TApp(Val)
#define c_function_name(f)            c_function_data(f)->name
#define c_function_name_length(f)     c_function_data(f)->name_length
#define c_function_documentation(f)   c_function_data(f)->doc
#define c_function_signature(f)       c_function_data(f)->signature
#define c_function_class(f)           c_function_data(f)->id
#define c_function_chooser(f)         c_function_data(f)->chooser
#define c_function_base(f)            _TApp(c_function_data(f)->generic_ff)
#define c_function_set_base(f, Val)   c_function_data(f)->generic_ff = _TApp(Val)
#define c_function_arg_defaults(f)    c_function_data(f)->arg_defaults
#define c_function_call_args(f)       c_function_data(f)->call_args
#define c_function_arg_names(f)       c_function_data(f)->arg_names
#define c_function_rp(f)              c_function_data(f)->rp
#define c_function_ip(f)              c_function_data(f)->ip
#define c_function_pp(f)              c_function_data(f)->pp
#define c_function_gp(f)              c_function_data(f)->gp
#define set_c_function(f, X)          do {set_opt_cfunc(f, X); set_c_call(f, c_function_call(opt_cfunc(f)));} while (0)

#define is_c_macro(p)                 (type(p) == T_C_MACRO)
#define c_macro_data(f)               (_TMac(f))->object.fnc.c_proc
#define c_macro_call(f)               (_TMac(f))->object.fnc.ff
#define c_macro_name(f)               c_macro_data(f)->name
#define c_macro_name_length(f)        c_macro_data(f)->name_length
#define c_macro_required_args(f)      (_TMac(f))->object.fnc.required_args
#define c_macro_all_args(f)           (_TMac(f))->object.fnc.all_args
#define c_macro_setter(f)             _TApp((_TMac(f))->object.fnc.setter)
#define c_macro_set_setter(f, Val)    (_TMac(f))->object.fnc.setter = _TApp(Val)

#define is_random_state(p)            (type(p) == T_RANDOM_STATE)
#if WITH_GMP
#define random_gmp_state(p)           (_TRan(p))->object.rng.state
#else
#define random_seed(p)                (_TRan(p))->object.rng.seed
#define random_carry(p)               (_TRan(p))->object.rng.carry
#endif

#define continuation_data(p)          (_TCon(p))->object.cwcc.continuation
#define continuation_stack(p)         (_TCon(p))->object.cwcc.stack
#define continuation_set_stack(p, Val) (_TCon(p))->object.cwcc.stack = _TStk(Val)
#define continuation_stack_end(p)     (_TCon(p))->object.cwcc.stack_end
#define continuation_stack_start(p)   (_TCon(p))->object.cwcc.stack_start
#define continuation_stack_size(p)    (_TCon(p))->object.cwcc.continuation->stack_size
#define continuation_stack_top(p)     (continuation_stack_end(p) - continuation_stack_start(p))
#define continuation_op_stack(p)      (_TCon(p))->object.cwcc.op_stack
#define continuation_op_loc(p)        (_TCon(p))->object.cwcc.continuation->op_stack_loc
#define continuation_op_size(p)       (_TCon(p))->object.cwcc.continuation->op_stack_size
#define continuation_key(p)           (_TCon(p))->object.cwcc.continuation->local_key

#define call_exit_goto_loc(p)         (_TGot(p))->object.rexit.goto_loc
#define call_exit_op_loc(p)           (_TGot(p))->object.rexit.op_stack_loc
#define call_exit_active(p)           (_TGot(p))->object.rexit.active

#define temp_stack_top(p)             (_TStk(p))->object.stk.top
#define s7_stack_top(Sc)              ((Sc)->stack_end - (Sc)->stack_start)

#define is_continuation(p)            (type(p) == T_CONTINUATION)
#define is_goto(p)                    (type(p) == T_GOTO)
#define is_macro(p)                   (type(p) == T_MACRO)
/* #define is_bacro(p)                (type(p) == T_BACRO) */
#define is_macro_star(p)              (type(p) == T_MACRO_STAR)
#define is_bacro_star(p)              (type(p) == T_BACRO_STAR)

#define is_closure(p)                 (type(p) == T_CLOSURE)
#define is_closure_star(p)            (type(p) == T_CLOSURE_STAR)
#define closure_args(p)               (_TClo(p))->object.func.args
#define closure_set_args(p, Val)      (_TClo(p))->object.func.args = _TArg(Val)
#define closure_body(p)               (_TPair((_TClo(p))->object.func.body))
#define closure_set_body(p, Val)      (_TClo(p))->object.func.body = _TPair(Val)
#define closure_let(p)                _TLid((_TClo(p))->object.func.env)
#define closure_set_let(p, L)         (_TClo(p))->object.func.env = _TLid(L)
#define closure_setter(p)             _TApp((_TClo(p))->object.func.setter)
#define closure_set_setter(p, Val)    (_TClo(p))->object.func.setter = _TApp(Val)
#define closure_arity(p)              (_TClo(p))->object.func.arity
#define CLOSURE_ARITY_NOT_SET         0x40000000
#define MAX_ARITY                     0x20000000
#define closure_arity_unknown(p)      (closure_arity(p) == CLOSURE_ARITY_NOT_SET)
#define is_thunk(Sc, Fnc)             ((type(Fnc) >= T_GOTO) && (s7_is_aritable(Sc, Fnc, 0)))

#define hook_has_functions(p)         (is_pair(s7_hook_functions(sc, _TClo(p))))

#define catch_tag(p)                  (_TCat(p))->object.rcatch.tag
#define catch_goto_loc(p)             (_TCat(p))->object.rcatch.goto_loc
#define catch_op_loc(p)               (_TCat(p))->object.rcatch.op_stack_loc
#define catch_handler(p)              (_TCat(p))->object.rcatch.handler

#define catch_all_goto_loc(p)         (_TLet(p))->object.envr.edat.ctall.goto_loc
#define catch_all_set_goto_loc(p, L)  (_TLet(p))->object.envr.edat.ctall.goto_loc = L
#define catch_all_op_loc(p)           (_TLet(p))->object.envr.edat.ctall.op_stack_loc
#define catch_all_set_op_loc(p, L)    (_TLet(p))->object.envr.edat.ctall.op_stack_loc = L
#define catch_all_result(p)           _NFre((_TLet(p))->object.envr.edat.ctall.result)
#define catch_all_set_result(p, R)    (_TLet(p))->object.envr.edat.ctall.result = R

enum {DWIND_INIT, DWIND_BODY, DWIND_FINISH};
#define dynamic_wind_state(p)         (_TDyn(p))->object.winder.state
#define dynamic_wind_in(p)            (_TDyn(p))->object.winder.in
#define dynamic_wind_out(p)           (_TDyn(p))->object.winder.out
#define dynamic_wind_body(p)          (_TDyn(p))->object.winder.body

#define is_c_object(p)                (type(p) == T_C_OBJECT)
#define c_object_value(p)             (_TObj(p))->object.c_obj.value
#define c_object_type(p)              (_TObj(p))->object.c_obj.type
#define c_object_let(p)               _TLid((_TObj(p))->object.c_obj.e)
#define c_object_set_let(p, L)        (_TObj(p))->object.c_obj.e = _TLid(L)
#define c_object_cref(p)              (_TObj(p))->object.c_obj.ref

static c_object_t **object_types = NULL;
static int object_types_size = 0;
static int num_object_types = 0;

#define c_object_info(p)              object_types[c_object_type(_TObj(p))]
#define c_object_ref(p)               c_object_info(p)->ref
#define c_object_set(p)               c_object_info(p)->set
#define c_object_print(p)             c_object_info(p)->print
#define c_object_print_readably(p)    c_object_info(p)->print_readably
#define c_object_length(p)            c_object_info(p)->length
#define c_object_eql(p)               c_object_info(p)->equal
#define c_object_fill(p)              c_object_info(p)->fill
#define c_object_copy(p)              c_object_info(p)->copy
#define c_object_free(p)              c_object_info(p)->free
#define c_object_mark(p)              c_object_info(p)->gc_mark
#define c_object_reverse(p)           c_object_info(p)->reverse
#define c_object_direct_ref(p)        c_object_info(p)->direct_ref
#define c_object_direct_set(p)        c_object_info(p)->direct_set
#define c_object_ip(p)                c_object_info(p)->ip
#define c_object_rp(p)                c_object_info(p)->rp
#define c_object_set_ip(p)            c_object_info(p)->set_ip
#define c_object_set_rp(p)            c_object_info(p)->set_rp
#define c_object_scheme_name(p)       _TStr(c_object_info(p)->scheme_name)
/* #define c_object_outer_type(p)     c_object_info(p)->outer_type */

#define raw_pointer(p)                (_TPtr(p))->object.c_pointer

#define is_counter(p)                 (type(p) == T_COUNTER)
#define counter_result(p)             (_TCtr(p))->object.ctr.result
#define counter_set_result(p, Val)    (_TCtr(p))->object.ctr.result = _NFre(Val)
#define counter_list(p)               (_TCtr(p))->object.ctr.list
#define counter_set_list(p, Val)      (_TCtr(p))->object.ctr.list = _NFre(Val)
#define counter_capture(p)            (_TCtr(p))->object.ctr.cap
#define counter_set_capture(p, Val)   (_TCtr(p))->object.ctr.cap = Val
#define counter_let(p)                _TLid((_TCtr(p))->object.ctr.env)
#define counter_set_let(p, L)         (_TCtr(p))->object.ctr.env = _TLid(L)
#define counter_slots(p)              (_TCtr(p))->object.ctr.slots
#define counter_set_slots(p, Val)     (_TCtr(p))->object.ctr.slots = _TSln(Val)

#define is_baffle(p)                  (type(p) == T_BAFFLE)
#define baffle_key(p)                 (_TBfl(p))->object.baffle_key

#if __cplusplus && HAVE_COMPLEX_NUMBERS
  using namespace std;                /* the code has to work in C as well as C++, so we can't scatter std:: all over the place */
  typedef complex<s7_double> s7_complex;
  static s7_double Real(complex<s7_double> x) {return(real(x));} /* protect the C++ name */
  static s7_double Imag(complex<s7_double> x) {return(imag(x));}
#endif

#define integer(p)                    (_TI(p))->object.number.integer_value
#define real(p)                       (_TR(p))->object.number.real_value
#define set_real(p, x)                real(p) = x
#define numerator(p)                  (_TF(p))->object.number.fraction_value.numerator
#define denominator(p)                (_TF(p))->object.number.fraction_value.denominator
#define fraction(p)                   (((long double)numerator(p)) / ((long double)denominator(p)))
#define inverted_fraction(p)          (((long double)denominator(p)) / ((long double)numerator(p)))
#define real_part(p)                  (_TZ(p))->object.number.complex_value.rl
#define set_real_part(p, x)           real_part(p) = x
#define imag_part(p)                  (_TZ(p))->object.number.complex_value.im
#define set_imag_part(p, x)           imag_part(p) = x
#if HAVE_COMPLEX_NUMBERS
  #define as_c_complex(p)             CMPLX(real_part(p), imag_part(p))
#endif

#if WITH_GMP
#define big_integer(p)                ((_TBgi(p))->object.number.big_integer)
#define big_ratio(p)                  ((_TBgf(p))->object.number.big_ratio)
#define big_real(p)                   ((_TBgr(p))->object.number.big_real)
#define big_complex(p)                ((_TBgz(p))->object.number.big_complex)
#endif

#define NUM_SMALL_INTS 2048
#define small_int(Val)                small_ints[Val]
#define is_small(n)                   ((n & ~(NUM_SMALL_INTS - 1)) == 0)

#define print_name(p)                 (char *)((_TNum(p))->object.number.pval.name + 1)
#define print_name_length(p)          (_TNum(p))->object.number.pval.name[0]

static void set_print_name(s7_pointer p, const char *name, int len)
{
  if ((len < PRINT_NAME_SIZE) &&
      (!is_mutable(p)))
    {
      set_has_print_name(p);
      print_name_length(p) = (unsigned char)(len & 0xff);
      memcpy((void *)print_name(p), (void *)name, len);
    }
}

#if WITH_GCC
#define make_integer(Sc, N) \
  ({ s7_int _N_; _N_ = (N); (is_small(_N_) ? small_int(_N_) : ({ s7_pointer _X_; new_cell(Sc, _X_, T_INTEGER); integer(_X_) = _N_; _X_;}) ); })

#define make_real(Sc, X) \
  ({ s7_double _N_ = (X); ((_N_ == 0.0) ? real_zero : ({ s7_pointer _X_; new_cell(Sc, _X_, T_REAL); set_real(_X_, _N_); _X_;}) ); })
                     /* the x == 0.0 check saves more than it costs */

#define make_complex(Sc, R, I)						\
  ({ s7_double im; im = (I); ((im == 0.0) ? make_real(Sc, R) : ({ s7_pointer _X_; new_cell(Sc, _X_, T_COMPLEX); set_real_part(_X_, R); set_imag_part(_X_, im); _X_;}) ); })

#define real_to_double(Sc, X, Caller)   ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_REAL) ? real(_x_) : s7_number_to_real_with_caller(sc, _x_, Caller)); })
#define rational_to_double(Sc, X)       ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_INTEGER) ? (s7_double)integer(_x_) : fraction(_x_)); })

#else

#define make_integer(Sc, N)           s7_make_integer(Sc, N)
#define make_real(Sc, X)              s7_make_real(Sc, X)
#define make_complex(Sc, R, I)        s7_make_complex(Sc, R, I)
#define real_to_double(Sc, X, Caller) s7_number_to_real_with_caller(Sc, X, Caller)
#define rational_to_double(Sc, X)     s7_number_to_real(Sc, X)
#endif

#define S7_LLONG_MAX 9223372036854775807LL
#define S7_LLONG_MIN (-S7_LLONG_MAX - 1LL)

#define S7_LONG_MAX 2147483647LL
#define S7_LONG_MIN (-S7_LONG_MAX - 1LL)

#define S7_SHORT_MAX 32767
#define S7_SHORT_MIN -32768

static s7_int s7_int_max = 0, s7_int_min = 0;

/* 9007199254740991LL is where a truncated double starts to skip integers (expt 2 53) = ca 1e16
 *   :(ceiling (+ 1e16 1))
 *   10000000000000000
 *   :(> 9007199254740993.0 9007199254740992.0)
 *   #f ; in non-gmp 64-bit doubles
 *
 * but we can't fix this except in the gmp case because:
 *   :(integer-decode-float (+ (expt 2.0 62) 100))
 *   (4503599627370496 10 1)
 *   :(integer-decode-float (+ (expt 2.0 62) 500))
 *   (4503599627370496 10 1)
 *   :(> (+ (expt 2.0 62) 500) (+ (expt 2.0 62) 100))
 *   #f ; non-gmp again
 *
 * i.e. the bits are identical.  We can't even detect when it has happened, so should
 *   we just give an error for any floor (or whatever) of an arg>1e16?  (sin has a similar problem)?
 *   I think in the non-gmp case I'll throw an error in these cases because the results are
 *   bogus:
 *   :(floor (+ (expt 2.0 62) 512))
 *   4611686018427387904
 *   :(floor (+ (expt 2.0 62) 513))
 *   4611686018427388928
 *
 * another case at the edge: (round 9007199254740992.51) -> 9007199254740992
 *
 * This spells trouble for normal arithmetic in this range.  If no gmp,
 *    (- (+ (expt 2.0 62) 512) (+ (expt 2.0 62) 513)) = -1024.0 (should be -1.0)
 *    but we don't currently give an error in this case -- not sure what the right thing is.
 */


/* --------------------------------------------------------------------------------
 * local versions of some standard C library functions
 * timing tests involving these are very hard to interpret -- pervasive inconsistency!
 */

static int safe_strlen(const char *str)
{
  /* this is safer than strlen, and slightly faster */
  char *tmp = (char *)str;
  if ((!tmp) || (!(*tmp))) return(0);
  while (*tmp++) {};
  return(tmp - str - 1);
}


static int safe_strlen5(const char *str)
{
  /* safe_strlen but we quit counting if len>5 */
  char *tmp = (char *)str;
  char *end;
  if ((!tmp) || (!(*tmp))) return(0);
  end = (char *)(tmp + 6);
  while ((*tmp++) && (tmp < end)) {};
  return(tmp - str - 1);
}


static char *copy_string_with_length(const char *str, int len)
{
  char *newstr;
  newstr = (char *)malloc((len + 1) * sizeof(char));
  if (len != 0)
    memcpy((void *)newstr, (void *)str, len + 1);
  else newstr[0] = 0;
  return(newstr);
}


static char *copy_string(const char *str)
{
  return(copy_string_with_length(str, safe_strlen(str)));
}


static bool local_strcmp(const char *s1, const char *s2)
{
  while (true)
    {
      if (*s1 != *s2++) return(false);
      if (*s1++ == 0) return(true);
    }
  return(true);
}

#define strings_are_equal(Str1, Str2) (local_strcmp(Str1, Str2))
/* this should only be used for internal strings -- scheme strings can have embedded nulls. */

static bool safe_strcmp(const char *s1, const char *s2)
{
  if ((!s1) || (!s2)) return(s1 == s2);
  return(local_strcmp(s1, s2));
}


static bool local_strncmp(const char *s1, const char *s2, unsigned int n)
{
#if defined(__x86_64__) || defined(__i386__) /* unaligned accesses are safe on i386 hardware, sez everyone */
  if (n >= 4)
    {
      int *is1, *is2;
      int n4 = n >> 2;
      is1 = (int *)s1;
      is2 = (int *)s2;
      do {if (*is1++ != *is2++) return(false);} while (--n4 > 0);
      s1 = (const char *)is1;
      s2 = (const char *)is2;
      n &= 3;
    }
#endif
  while (n > 0)
    {
      if (*s1++ != *s2++) return(false);
      n--;
    }
  return(true);
}

#define strings_are_equal_with_length(Str1, Str2, Len) (local_strncmp(Str1, Str2, Len))


 static void memclr(void *s, size_t n)
{
  unsigned char *s2;
#if defined(__x86_64__) || defined(__i386__)
  if (n >= 4)
    {
      int *s1 = (int *)s;
      size_t n4 = n >> 2;
      do {*s1++ = 0;} while (--n4 > 0);
      n &= 3;
      s2 = (unsigned char *)s1;
    }
  else s2 = (unsigned char *)s;
#else
  s2 = (unsigned char *)s;
#endif
  while (n > 0)
    {
      *s2++ = 0;
      n--;
    }
}


/* ---------------- forward decls ---------------- */

static char *number_to_string_base_10(s7_pointer obj, int width, int precision, char float_choice, int *nlen, use_write_t choice);
static bool is_proper_list(s7_scheme *sc, s7_pointer lst);
static s7_pointer iterator_finished(s7_scheme *sc, s7_pointer iterator);
static bool is_all_x_safe(s7_scheme *sc, s7_pointer p);
static void annotate_args(s7_scheme *sc, s7_pointer args, s7_pointer e);
static void annotate_arg(s7_scheme *sc, s7_pointer arg, s7_pointer e);
static s7_pointer eval(s7_scheme *sc, opcode_t first_op);
static s7_pointer division_by_zero_error(s7_scheme *sc, s7_pointer caller, s7_pointer arg);
static s7_pointer file_error(s7_scheme *sc, const char *caller, const char *descr, const char *name);
static s7_pointer prepackaged_type_name(s7_scheme *sc, s7_pointer x);
static void s7_warn(s7_scheme *sc, int len, const char *ctrl, ...);
static s7_pointer safe_reverse_in_place(s7_scheme *sc, s7_pointer list);
static s7_pointer cons_unchecked(s7_scheme *sc, s7_pointer a, s7_pointer b);
static s7_pointer permanent_cons(s7_pointer a, s7_pointer b, unsigned int type);
static s7_pointer permanent_list(s7_scheme *sc, int len);
static void free_object(s7_pointer a);
static s7_pointer make_atom(s7_scheme *sc, char *q, int radix, bool want_symbol, bool with_error);
static s7_pointer apply_error(s7_scheme *sc, s7_pointer obj, s7_pointer args);
static int remember_file_name(s7_scheme *sc, const char *file);
static const char *type_name(s7_scheme *sc, s7_pointer arg, int article);
static s7_pointer make_vector_1(s7_scheme *sc, s7_int len, bool filled, unsigned int typ);
static s7_pointer make_string_uncopied_with_length(s7_scheme *sc, char *str, int len);
static s7_pointer make_string_wrapper_with_length(s7_scheme *sc, const char *str, int len);
static s7_pointer make_string_wrapper(s7_scheme *sc, const char *str);
static void check_for_substring_temp(s7_scheme *sc, s7_pointer expr);
static s7_pointer splice_in_values(s7_scheme *sc, s7_pointer args);
static void pop_input_port(s7_scheme *sc);
static char *object_to_truncated_string(s7_scheme *sc, s7_pointer p, int len);
static token_t token(s7_scheme *sc);
static s7_pointer implicit_index(s7_scheme *sc, s7_pointer obj, s7_pointer indices);
static bool s7_is_morally_equal(s7_scheme *sc, s7_pointer x, s7_pointer y);
static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym);
static s7_pointer unbound_variable(s7_scheme *sc, s7_pointer sym);
static s7_pointer optimize_lambda(s7_scheme *sc, bool unstarred_lambda, s7_pointer func, s7_pointer args, s7_pointer body);
static bool optimize_expression(s7_scheme *sc, s7_pointer expr, int hop, s7_pointer e);
static s7_pointer optimize(s7_scheme *sc, s7_pointer code, int hop, s7_pointer e);
static void free_hash_table(s7_pointer table);

#if WITH_GMP
static s7_int big_integer_to_s7_int(mpz_t n);
#else
static double next_random(s7_pointer r);
#endif

#if DEBUGGING && WITH_GCC
  static s7_pointer find_symbol_unchecked_1(s7_scheme *sc, s7_pointer symbol);
  #define find_symbol_unchecked(Sc, Sym) check_null_sym(Sc, find_symbol_unchecked_1(Sc, Sym), Sym, __LINE__, __func__)
  static s7_pointer check_null_sym(s7_scheme *sc, s7_pointer p, s7_pointer sym, int line, const char *func);
  #define find_symbol_unexamined(Sc, Sym) find_symbol_unchecked_1(Sc, Sym)
#else
  static s7_pointer find_symbol_unchecked(s7_scheme *sc, s7_pointer symbol);
  #define find_symbol_unexamined(Sc, Sym) find_symbol_unchecked(Sc, Sym)
#endif

#if WITH_GCC
  #if DEBUGGING
    #define find_symbol_checked(Sc, Sym) ({s7_pointer _x_; _x_ = find_symbol_unchecked_1(Sc, Sym); ((_x_) ? _x_ : unbound_variable(Sc, Sym));})
  #else
    #define find_symbol_checked(Sc, Sym) ({s7_pointer _x_; _x_ = find_symbol_unchecked(Sc, Sym); ((_x_) ? _x_ : unbound_variable(Sc, Sym));})
  #endif
#else
  #define find_symbol_checked(Sc, Sym) find_symbol_unchecked(Sc, Sym)
#endif

static s7_pointer find_method(s7_scheme *sc, s7_pointer env, s7_pointer symbol);
static s7_pointer find_let(s7_scheme *sc, s7_pointer obj);
static bool call_begin_hook(s7_scheme *sc);
static s7_pointer default_vector_setter(s7_scheme *sc, s7_pointer vec, s7_int loc, s7_pointer val);
static s7_pointer default_vector_getter(s7_scheme *sc, s7_pointer vec, s7_int loc);

static s7_pointer simple_wrong_type_arg_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer typnam, s7_pointer descr);
static s7_pointer wrong_type_arg_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg_n, s7_pointer arg, s7_pointer typnam, s7_pointer descr);
static s7_pointer out_of_range_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg_n, s7_pointer arg, s7_pointer descr);
static s7_pointer simple_out_of_range_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer descr);

/* putting off the type description until s7_error via the sc->gc_nil marker below makes it possible
 *    for gcc to speed up the functions that call these as tail-calls.  1-2% overall speedup!
 */
#define simple_wrong_type_argument(Sc, Caller, Arg, Desired_Type) \
  simple_wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, sc->gc_nil, prepackaged_type_names[Desired_Type])

#define wrong_type_argument(Sc, Caller, Num, Arg, Desired_Type)	\
  wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), make_integer(Sc, Num), Arg, sc->gc_nil, prepackaged_type_names[Desired_Type])

#define simple_wrong_type_argument_with_type(Sc, Caller, Arg, Type) \
  simple_wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, sc->gc_nil, Type)

#define wrong_type_argument_with_type(Sc, Caller, Num, Arg, Type)	\
  wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), make_integer(Sc, Num), Arg, sc->gc_nil, Type)


#define simple_out_of_range(Sc, Caller, Arg, Description) \
  simple_out_of_range_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, Description)

#define out_of_range(Sc, Caller, Arg_Num, Arg, Description)		\
  out_of_range_error_prepackaged(Sc, symbol_name_cell(Caller), Arg_Num, Arg, Description)


static s7_pointer car_a_list_string, cdr_a_list_string, caar_a_list_string, cadr_a_list_string, cdar_a_list_string,
                  cddr_a_list_string, caaar_a_list_string, caadr_a_list_string, cadar_a_list_string, caddr_a_list_string,
                  cdaar_a_list_string, cdadr_a_list_string, cddar_a_list_string, cdddr_a_list_string, a_list_string, 
                  an_association_list_string, an_output_port_string, an_input_port_string, an_open_port_string, 
                  a_normal_real_string, a_rational_string, a_boolean_string, a_number_string, a_let_string, 
                  a_procedure_string, a_proper_list_string, a_thunk_string, something_applicable_string, a_symbol_string, 
                  a_non_negative_integer_string, a_format_port_string, an_unsigned_byte_string, a_binding_string, 
                  a_non_constant_symbol_string, an_eq_func_string, a_sequence_string, its_too_small_string, 
                  a_normal_procedure_string, its_too_large_string, its_negative_string, result_is_too_large_string, 
                  its_nan_string, its_infinite_string, too_many_indices_string, a_valid_radix_string, an_input_string_port_string, 
                  an_input_file_port_string, an_output_string_port_string, an_output_file_port_string, a_random_state_object_string;

#if (!HAVE_COMPLEX_NUMBERS)
static s7_pointer no_complex_numbers_string;
#endif


/* ---------------- evaluator ops ---------------- */

enum {OP_NO_OP,
      OP_READ_INTERNAL, OP_EVAL,
      OP_EVAL_ARGS, OP_EVAL_ARGS1, OP_EVAL_ARGS2, OP_EVAL_ARGS3, OP_EVAL_ARGS4, OP_EVAL_ARGS5,
      OP_APPLY, OP_EVAL_MACRO, OP_LAMBDA, OP_QUOTE, OP_MACROEXPAND,
      OP_DEFINE, OP_DEFINE1, OP_BEGIN, OP_BEGIN_UNCHECKED, OP_BEGIN1,
      OP_IF, OP_IF1, OP_WHEN, OP_WHEN1, OP_UNLESS, OP_UNLESS1, OP_SET, OP_SET1, OP_SET2,
      OP_LET, OP_LET1, OP_LET_STAR, OP_LET_STAR1, OP_LET_STAR2,
      OP_LETREC, OP_LETREC1, OP_LETREC_STAR, OP_LETREC_STAR1, OP_COND, OP_COND1, OP_COND1_1, OP_COND_SIMPLE, OP_COND1_SIMPLE,
      OP_AND, OP_AND1, OP_OR, OP_OR1,
      OP_DEFINE_MACRO, OP_DEFINE_MACRO_STAR, OP_DEFINE_EXPANSION,
      OP_CASE, OP_CASE1, OP_READ_LIST, OP_READ_NEXT, OP_READ_DOT, OP_READ_QUOTE,
      OP_READ_QUASIQUOTE, OP_READ_UNQUOTE, OP_READ_APPLY_VALUES,
      OP_READ_VECTOR, OP_READ_BYTE_VECTOR, OP_READ_DONE,
      OP_LOAD_RETURN_IF_EOF, OP_LOAD_CLOSE_AND_POP_IF_EOF, OP_EVAL_DONE,
      OP_CATCH, OP_DYNAMIC_WIND, OP_DEFINE_CONSTANT, OP_DEFINE_CONSTANT1,
      OP_DO, OP_DO_END, OP_DO_END1, OP_DO_STEP, OP_DO_STEP2, OP_DO_INIT,
      OP_DEFINE_STAR, OP_LAMBDA_STAR, OP_LAMBDA_STAR_DEFAULT, OP_ERROR_QUIT, OP_UNWIND_INPUT, OP_UNWIND_OUTPUT,
      OP_ERROR_HOOK_QUIT,
      OP_WITH_LET, OP_WITH_LET1, OP_WITH_LET_UNCHECKED, OP_WITH_LET_S,
      OP_WITH_BAFFLE, OP_WITH_BAFFLE_UNCHECKED, OP_EXPANSION,
      OP_FOR_EACH, OP_FOR_EACH_1, OP_FOR_EACH_2, OP_FOR_EACH_3,
      OP_MAP, OP_MAP_1, OP_MAP_GATHER, OP_MAP_GATHER_1, OP_BARRIER, OP_DEACTIVATE_GOTO,

      OP_DEFINE_BACRO, OP_DEFINE_BACRO_STAR,
      OP_GET_OUTPUT_STRING, OP_GET_OUTPUT_STRING_1,
      OP_SORT, OP_SORT1, OP_SORT2, OP_SORT3, OP_SORT_PAIR_END, OP_SORT_VECTOR_END, OP_SORT_STRING_END,
      OP_EVAL_STRING_1, OP_EVAL_STRING_2,
      OP_MEMBER_IF, OP_ASSOC_IF, OP_MEMBER_IF1, OP_ASSOC_IF1,

      OP_QUOTE_UNCHECKED, OP_LAMBDA_UNCHECKED, OP_LET_UNCHECKED, OP_CASE_UNCHECKED, OP_WHEN_UNCHECKED, OP_UNLESS_UNCHECKED,

      OP_SET_UNCHECKED, OP_SET_SYMBOL_C, OP_SET_SYMBOL_S, OP_SET_SYMBOL_Q, OP_SET_SYMBOL_P, OP_SET_SYMBOL_Z, OP_SET_SYMBOL_A,
      OP_SET_SYMBOL_opSq, OP_SET_SYMBOL_opCq, OP_SET_SYMBOL_opSSq, OP_SET_SYMBOL_opSSSq, 
      OP_SET_NORMAL, OP_SET_PAIR, OP_SET_PAIR_Z, OP_SET_PAIR_A, OP_SET_PAIR_P, OP_SET_PAIR_ZA,
      OP_SET_PAIR_P_1, OP_SET_WITH_ACCESSOR, OP_SET_PWS, OP_SET_LET_S, OP_SET_LET_ALL_X,
      OP_SET_PAIR_C, OP_SET_PAIR_C_P, OP_SET_PAIR_C_P_1, OP_SET_SAFE,
      OP_INCREMENT_1, OP_DECREMENT_1, OP_SET_CONS,
      OP_INCREMENT_SS, OP_INCREMENT_SSS, OP_INCREMENT_SZ, OP_INCREMENT_SA, OP_INCREMENT_SAA,

      OP_LET_STAR_UNCHECKED, OP_LETREC_UNCHECKED, OP_LETREC_STAR_UNCHECKED, OP_COND_UNCHECKED,
      OP_LAMBDA_STAR_UNCHECKED, OP_DO_UNCHECKED, OP_DEFINE_UNCHECKED, OP_DEFINE_STAR_UNCHECKED, OP_DEFINE_FUNCHECKED, OP_DEFINE_CONSTANT_UNCHECKED,
      OP_DEFINE_WITH_ACCESSOR, OP_DEFINE_MACRO_WITH_ACCESSOR,

      OP_LET_NO_VARS, OP_NAMED_LET, OP_NAMED_LET_NO_VARS, OP_NAMED_LET_STAR,
      OP_LET_C, OP_LET_S, OP_LET_ALL_C, OP_LET_ALL_S, OP_LET_ALL_X,
      OP_LET_STAR_ALL_X, OP_LET_opCq, OP_LET_opSSq,
      OP_LET_opSq, OP_LET_ALL_opSq, OP_LET_opSq_P, OP_LET_ONE, OP_LET_ONE_1, OP_LET_Z, OP_LET_Z_1,

      OP_CASE_SIMPLE, OP_CASE_SIMPLER, OP_CASE_SIMPLER_1, OP_CASE_SIMPLER_SS, OP_CASE_SIMPLEST, OP_CASE_SIMPLEST_SS, 
      OP_IF_UNCHECKED, OP_AND_UNCHECKED, OP_AND_P, OP_AND_P1, OP_AND_P2, OP_OR_UNCHECKED, OP_OR_P, OP_OR_P1, OP_OR_P2,
      OP_IF_P_FEED, OP_IF_P_FEED_1, OP_WHEN_S, OP_UNLESS_S,

      OP_IF_S_P, OP_IF_S_P_P, OP_IF_NOT_S_P, OP_IF_NOT_S_P_P, OP_IF_CC_P, OP_IF_CC_P_P,
      OP_IF_CS_P, OP_IF_CS_P_P, OP_IF_CSQ_P, OP_IF_CSQ_P_P, OP_IF_CSS_P, OP_IF_CSS_P_P,
      OP_IF_CSC_P, OP_IF_CSC_P_P, OP_IF_IS_PAIR_P, OP_IF_IS_PAIR_P_P, OP_IF_opSSq_P, OP_IF_opSSq_P_P, OP_IF_S_opCq_P, OP_IF_S_opCq_P_P,
      OP_IF_IS_SYMBOL_P, OP_IF_IS_SYMBOL_P_P, OP_IF_A_P, OP_IF_A_P_P, OP_IF_AND2_P, OP_IF_AND2_P_P,
      OP_IF_Z_P, OP_IF_Z_P_P, OP_IF_P_P_P, OP_IF_P_P, OP_IF_ANDP_P, OP_IF_ANDP_P_P, OP_IF_ORP_P, OP_IF_ORP_P_P,
      OP_IF_PPP, OP_IF_PP,

      OP_CATCH_1, OP_CATCH_2, OP_CATCH_ALL, OP_COND_ALL_X, OP_COND_ALL_X_2, OP_COND_S,
      OP_SIMPLE_DO, OP_SIMPLE_DO_STEP, OP_SAFE_DOTIMES, OP_SAFE_DOTIMES_STEP, OP_SAFE_DOTIMES_STEP_P, OP_SAFE_DOTIMES_STEP_O, OP_SAFE_DOTIMES_STEP_A,
      OP_SAFE_DO, OP_SAFE_DO_STEP, OP_SIMPLE_DO_P, OP_SIMPLE_DO_STEP_P, OP_DOX, OP_DOX_STEP, OP_DOX_STEP_P, 
      OP_DOTIMES_P, OP_DOTIMES_STEP_P, OP_SIMPLE_DO_A, OP_SIMPLE_DO_STEP_A, OP_SIMPLE_DO_E, OP_SIMPLE_DO_STEP_E,

      OP_SAFE_C_P_1, OP_SAFE_C_PP_1, OP_SAFE_C_PP_2, OP_SAFE_C_PP_3, OP_SAFE_C_PP_4, OP_SAFE_C_PP_5, OP_SAFE_C_PP_6,
      OP_EVAL_ARGS_P_2, OP_EVAL_ARGS_P_2_MV, OP_EVAL_ARGS_P_3, OP_EVAL_ARGS_P_4, OP_EVAL_ARGS_P_3_MV, OP_EVAL_ARGS_P_4_MV,
      OP_EVAL_ARGS_SSP_1, OP_EVAL_ARGS_SSP_MV, OP_EVAL_MACRO_MV, OP_MACROEXPAND_1,

      OP_SAFE_C_ZZ_1, OP_SAFE_C_ZZ_2, OP_SAFE_C_ZC_1, OP_SAFE_C_SZ_1, OP_SAFE_C_ZA_1, OP_INCREMENT_SZ_1, OP_SAFE_C_SZ_SZ,
      OP_SAFE_C_ZAA_1, OP_SAFE_C_AZA_1, OP_SAFE_C_AAZ_1, OP_SAFE_C_SSZ_1,
      OP_SAFE_C_ZZA_1, OP_SAFE_C_ZZA_2, OP_SAFE_C_ZAZ_1, OP_SAFE_C_ZAZ_2, OP_SAFE_C_AZZ_1, OP_SAFE_C_AZZ_2,
      OP_SAFE_C_ZZZ_1, OP_SAFE_C_ZZZ_2, OP_SAFE_C_ZZZ_3,
      OP_SAFE_C_opSq_P_1, OP_SAFE_C_opSq_P_MV, OP_C_P_1, OP_C_P_2, OP_C_SP_1, OP_C_SP_2,
      OP_CLOSURE_P_1, OP_CLOSURE_P_2, OP_SAFE_CLOSURE_P_1,

      OP_SET_WITH_LET_1, OP_SET_WITH_LET_2,
      OP_MAX_DEFINED_1};

#define OP_MAX_DEFINED (OP_MAX_DEFINED_1 + 1)

typedef enum{E_C_P, E_C_PP, E_C_CP, E_C_SP, E_C_PC, E_C_PS} combine_op_t;

enum {OP_SAFE_C_C, HOP_SAFE_C_C, OP_SAFE_C_S, HOP_SAFE_C_S,
      OP_SAFE_C_SS, HOP_SAFE_C_SS, OP_SAFE_C_SC, HOP_SAFE_C_SC, OP_SAFE_C_CS, HOP_SAFE_C_CS,
      OP_SAFE_C_Q, HOP_SAFE_C_Q, OP_SAFE_C_SQ, HOP_SAFE_C_SQ, OP_SAFE_C_QS, HOP_SAFE_C_QS, OP_SAFE_C_QQ, HOP_SAFE_C_QQ,
      OP_SAFE_C_CQ, HOP_SAFE_C_CQ, OP_SAFE_C_QC, HOP_SAFE_C_QC,
      OP_SAFE_C_SSS, HOP_SAFE_C_SSS, OP_SAFE_C_SCS, HOP_SAFE_C_SCS, OP_SAFE_C_SSC, HOP_SAFE_C_SSC, OP_SAFE_C_CSS, HOP_SAFE_C_CSS,
      OP_SAFE_C_SCC, HOP_SAFE_C_SCC, OP_SAFE_C_CSC, HOP_SAFE_C_CSC,
      OP_SAFE_C_ALL_S, HOP_SAFE_C_ALL_S, OP_SAFE_C_ALL_X, HOP_SAFE_C_ALL_X, OP_SAFE_C_SSA, HOP_SAFE_C_SSA, OP_SAFE_C_SAS, HOP_SAFE_C_SAS,
      OP_SAFE_C_CSA, HOP_SAFE_C_CSA, OP_SAFE_C_SCA, HOP_SAFE_C_SCA, OP_SAFE_C_CAS, HOP_SAFE_C_CAS,
      OP_SAFE_C_A, HOP_SAFE_C_A, OP_SAFE_C_AA, HOP_SAFE_C_AA, OP_SAFE_C_AAA, HOP_SAFE_C_AAA, OP_SAFE_C_AAAA, HOP_SAFE_C_AAAA,
      OP_SAFE_C_SQS, HOP_SAFE_C_SQS, OP_SAFE_C_opAq, HOP_SAFE_C_opAq, OP_SAFE_C_opAAq, HOP_SAFE_C_opAAq, OP_SAFE_C_opAAAq, HOP_SAFE_C_opAAAq,
      OP_SAFE_C_S_opAq, HOP_SAFE_C_S_opAq, OP_SAFE_C_S_opAAq, HOP_SAFE_C_S_opAAq, OP_SAFE_C_S_opAAAq, HOP_SAFE_C_S_opAAAq,

      OP_SAFE_C_opCq, HOP_SAFE_C_opCq, OP_SAFE_C_opSq, HOP_SAFE_C_opSq,
      OP_SAFE_C_opSSq, HOP_SAFE_C_opSSq, OP_SAFE_C_opSCq, HOP_SAFE_C_opSCq, OP_SAFE_C_opSQq, HOP_SAFE_C_opSQq,
      OP_SAFE_C_opCSq, HOP_SAFE_C_opCSq, OP_SAFE_C_S_opSq, HOP_SAFE_C_S_opSq,
      OP_SAFE_C_C_opSCq, HOP_SAFE_C_C_opSCq,
      OP_SAFE_C_S_opSCq, HOP_SAFE_C_S_opSCq, OP_SAFE_C_S_opCSq, HOP_SAFE_C_S_opCSq,
      OP_SAFE_C_opSq_S, HOP_SAFE_C_opSq_S, OP_SAFE_C_opSq_C, HOP_SAFE_C_opSq_C,
      OP_SAFE_C_opSq_opSq, HOP_SAFE_C_opSq_opSq, OP_SAFE_C_S_opSSq, HOP_SAFE_C_S_opSSq, OP_SAFE_C_C_opSq, HOP_SAFE_C_C_opSq,
      OP_SAFE_C_C_opCSq, HOP_SAFE_C_C_opCSq, OP_SAFE_C_opCSq_C, HOP_SAFE_C_opCSq_C,
      OP_SAFE_C_S_opCq, HOP_SAFE_C_S_opCq, OP_SAFE_C_opSSq_C, HOP_SAFE_C_opSSq_C, OP_SAFE_C_C_opSSq, HOP_SAFE_C_C_opSSq,
      OP_SAFE_C_C_opCq, HOP_SAFE_C_C_opCq, OP_SAFE_C_opCq_S, HOP_SAFE_C_opCq_S,
      OP_SAFE_C_opCq_opCq, HOP_SAFE_C_opCq_opCq, OP_SAFE_C_opCq_C, HOP_SAFE_C_opCq_C,
      OP_SAFE_C_opSCq_opSCq, HOP_SAFE_C_opSCq_opSCq, OP_SAFE_C_opSSq_opSSq, HOP_SAFE_C_opSSq_opSSq,
      OP_SAFE_C_opSSq_opCq, HOP_SAFE_C_opSSq_opCq, OP_SAFE_C_opSSq_opSq, HOP_SAFE_C_opSSq_opSq, OP_SAFE_C_opSq_opSSq, HOP_SAFE_C_opSq_opSSq,
      OP_SAFE_C_opSSq_S, HOP_SAFE_C_opSSq_S, OP_SAFE_C_opSCq_S, HOP_SAFE_C_opSCq_S, OP_SAFE_C_opCSq_S, HOP_SAFE_C_opCSq_S,
      OP_SAFE_C_opSCq_C, HOP_SAFE_C_opSCq_C, OP_SAFE_C_opCq_opSSq, HOP_SAFE_C_opCq_opSSq,
      OP_SAFE_C_S_op_opSSq_Sq, HOP_SAFE_C_S_op_opSSq_Sq, OP_SAFE_C_S_op_S_opSSqq, HOP_SAFE_C_S_op_S_opSSqq,
      OP_SAFE_C_op_opSSq_q_C, HOP_SAFE_C_op_opSSq_q_C, OP_SAFE_C_op_opSq_q_C, HOP_SAFE_C_op_opSq_q_C, 
      OP_SAFE_C_op_opSSq_q_S, HOP_SAFE_C_op_opSSq_q_S, OP_SAFE_C_op_opSq_q_S, HOP_SAFE_C_op_opSq_q_S, 
      OP_SAFE_C_S_op_opSSq_opSSqq, HOP_SAFE_C_S_op_opSSq_opSSqq,
      OP_SAFE_C_op_opSq_q, HOP_SAFE_C_op_opSq_q, OP_SAFE_C_C_op_S_opCqq, HOP_SAFE_C_C_op_S_opCqq,
      OP_SAFE_C_op_S_opSq_q, HOP_SAFE_C_op_S_opSq_q, 
      OP_SAFE_C_opSq_Q, HOP_SAFE_C_opSq_Q, OP_SAFE_C_opSq_Q_S, HOP_SAFE_C_opSq_Q_S,

      OP_SAFE_C_Z, HOP_SAFE_C_Z, OP_SAFE_C_ZZ, HOP_SAFE_C_ZZ, OP_SAFE_C_SZ, HOP_SAFE_C_SZ, OP_SAFE_C_ZS, HOP_SAFE_C_ZS,
      OP_SAFE_C_CZ, HOP_SAFE_C_CZ, OP_SAFE_C_ZC, HOP_SAFE_C_ZC,
      OP_SAFE_C_opCq_Z, HOP_SAFE_C_opCq_Z, OP_SAFE_C_S_opSZq, HOP_SAFE_C_S_opSZq,
      OP_SAFE_C_AZ, HOP_SAFE_C_AZ, OP_SAFE_C_ZA, HOP_SAFE_C_ZA,
      OP_SAFE_C_ZAA, HOP_SAFE_C_ZAA, OP_SAFE_C_AZA, HOP_SAFE_C_AZA, OP_SAFE_C_AAZ, HOP_SAFE_C_AAZ, OP_SAFE_C_SSZ, HOP_SAFE_C_SSZ,
      OP_SAFE_C_ZZA, HOP_SAFE_C_ZZA, OP_SAFE_C_ZAZ, HOP_SAFE_C_ZAZ, OP_SAFE_C_AZZ, HOP_SAFE_C_AZZ,
      OP_SAFE_C_ZZZ, HOP_SAFE_C_ZZZ, 

      OP_THUNK, HOP_THUNK,
      OP_CLOSURE_S, HOP_CLOSURE_S, OP_CLOSURE_C, HOP_CLOSURE_C, OP_CLOSURE_Q, HOP_CLOSURE_Q,
      OP_CLOSURE_SS, HOP_CLOSURE_SS, OP_CLOSURE_SC, HOP_CLOSURE_SC, OP_CLOSURE_CS, HOP_CLOSURE_CS,
      OP_CLOSURE_A, HOP_CLOSURE_A, OP_CLOSURE_AA, HOP_CLOSURE_AA,
      OP_CLOSURE_ALL_X, HOP_CLOSURE_ALL_X, OP_CLOSURE_ALL_S, HOP_CLOSURE_ALL_S,

      OP_GLOSURE_A, HOP_GLOSURE_A, OP_GLOSURE_S, HOP_GLOSURE_S, OP_GLOSURE_P, HOP_GLOSURE_P,

      OP_CLOSURE_STAR_S, HOP_CLOSURE_STAR_S, OP_CLOSURE_STAR_SX, HOP_CLOSURE_STAR_SX,
      OP_CLOSURE_STAR, HOP_CLOSURE_STAR, OP_CLOSURE_STAR_ALL_X, HOP_CLOSURE_STAR_ALL_X,

      OP_SAFE_THUNK, HOP_SAFE_THUNK, OP_SAFE_THUNK_E, HOP_SAFE_THUNK_E, OP_SAFE_THUNK_P, HOP_SAFE_THUNK_P,
      OP_SAFE_CLOSURE_S, HOP_SAFE_CLOSURE_S, OP_SAFE_CLOSURE_C, HOP_SAFE_CLOSURE_C, OP_SAFE_CLOSURE_Q, HOP_SAFE_CLOSURE_Q,
      OP_SAFE_CLOSURE_SS, HOP_SAFE_CLOSURE_SS, OP_SAFE_CLOSURE_SC, HOP_SAFE_CLOSURE_SC, OP_SAFE_CLOSURE_CS, HOP_SAFE_CLOSURE_CS,
      OP_SAFE_CLOSURE_A, HOP_SAFE_CLOSURE_A, OP_SAFE_CLOSURE_SA, HOP_SAFE_CLOSURE_SA, OP_SAFE_CLOSURE_S_P, HOP_SAFE_CLOSURE_S_P,
      OP_SAFE_CLOSURE_SAA, HOP_SAFE_CLOSURE_SAA,
      OP_SAFE_CLOSURE_ALL_X, HOP_SAFE_CLOSURE_ALL_X, OP_SAFE_CLOSURE_AA, HOP_SAFE_CLOSURE_AA,

      OP_SAFE_GLOSURE_A, HOP_SAFE_GLOSURE_A, OP_SAFE_GLOSURE_S, HOP_SAFE_GLOSURE_S, OP_SAFE_GLOSURE_S_E, HOP_SAFE_GLOSURE_S_E, 
      OP_SAFE_GLOSURE_P, HOP_SAFE_GLOSURE_P,

      OP_SAFE_CLOSURE_STAR_S, HOP_SAFE_CLOSURE_STAR_S, OP_SAFE_CLOSURE_STAR_SS, HOP_SAFE_CLOSURE_STAR_SS,
      OP_SAFE_CLOSURE_STAR_SC, HOP_SAFE_CLOSURE_STAR_SC, OP_SAFE_CLOSURE_STAR_SA, HOP_SAFE_CLOSURE_STAR_SA, OP_SAFE_CLOSURE_STAR_S0, HOP_SAFE_CLOSURE_STAR_S0,
      OP_SAFE_CLOSURE_STAR, HOP_SAFE_CLOSURE_STAR, OP_SAFE_CLOSURE_STAR_ALL_X, HOP_SAFE_CLOSURE_STAR_ALL_X,

      /* these can't be embedded, and have to be the last thing called */
      OP_APPLY_SS, HOP_APPLY_SS,
      OP_C_ALL_X, HOP_C_ALL_X, OP_CALL_WITH_EXIT, HOP_CALL_WITH_EXIT, OP_C_CATCH, HOP_C_CATCH, OP_C_CATCH_ALL, HOP_C_CATCH_ALL,
      OP_C_S_opSq, HOP_C_S_opSq, OP_C_S_opCq, HOP_C_S_opCq, OP_C_SS, HOP_C_SS,
      OP_C_S, HOP_C_S, OP_READ_S, HOP_READ_S, OP_C_P, HOP_C_P, OP_C_Z, HOP_C_Z, OP_C_SP, HOP_C_SP,
      OP_C_SZ, HOP_C_SZ, OP_C_A, HOP_C_A, OP_C_SCS, HOP_C_SCS,

      OP_GOTO, HOP_GOTO, OP_GOTO_C, HOP_GOTO_C, OP_GOTO_S, HOP_GOTO_S, OP_GOTO_A, HOP_GOTO_A,

      OP_VECTOR_C, HOP_VECTOR_C, OP_VECTOR_S, HOP_VECTOR_S, OP_VECTOR_A, HOP_VECTOR_A, OP_VECTOR_CC, HOP_VECTOR_CC,
      OP_STRING_C, HOP_STRING_C, OP_STRING_S, HOP_STRING_S, OP_STRING_A, HOP_STRING_A,
      OP_C_OBJECT, HOP_C_OBJECT, OP_C_OBJECT_C, HOP_C_OBJECT_C, OP_C_OBJECT_S, HOP_C_OBJECT_S, OP_C_OBJECT_A, HOP_C_OBJECT_A, 
      OP_PAIR_C, HOP_PAIR_C, OP_PAIR_S, HOP_PAIR_S, OP_PAIR_A, HOP_PAIR_A,
      OP_HASH_TABLE_C, HOP_HASH_TABLE_C, OP_HASH_TABLE_S, HOP_HASH_TABLE_S, OP_HASH_TABLE_A, HOP_HASH_TABLE_A,
      OP_ENVIRONMENT_S, HOP_ENVIRONMENT_S, OP_ENVIRONMENT_Q, HOP_ENVIRONMENT_Q, OP_ENVIRONMENT_A, HOP_ENVIRONMENT_A, OP_ENVIRONMENT_C, HOP_ENVIRONMENT_C,

      OP_UNKNOWN, HOP_UNKNOWN, OP_UNKNOWN_ALL_S, HOP_UNKNOWN_ALL_S, OP_UNKNOWN_ALL_X, HOP_UNKNOWN_ALL_X,
      OP_UNKNOWN_G, HOP_UNKNOWN_G, OP_UNKNOWN_GG, HOP_UNKNOWN_GG, OP_UNKNOWN_A, HOP_UNKNOWN_A, OP_UNKNOWN_AA, HOP_UNKNOWN_AA,

      OP_SAFE_C_PP, HOP_SAFE_C_PP,
      OP_SAFE_C_opSq_P, HOP_SAFE_C_opSq_P, 
      OP_SAFE_C_SP, HOP_SAFE_C_SP, OP_SAFE_C_CP, HOP_SAFE_C_CP, OP_SAFE_C_QP, HOP_SAFE_C_QP, OP_SAFE_C_AP, HOP_SAFE_C_AP,
      OP_SAFE_C_PS, HOP_SAFE_C_PS, OP_SAFE_C_PC, HOP_SAFE_C_PC, OP_SAFE_C_PQ, HOP_SAFE_C_PQ,
      OP_SAFE_C_SSP, HOP_SAFE_C_SSP,
      OPT_MAX_DEFINED
};

#if DEBUGGING || OP_NAMES

static const char *op_names[OP_MAX_DEFINED_1] = {
      "no_op",
      "read_internal", "eval",
      "eval_args", "eval_args1", "eval_args2", "eval_args3", "eval_args4", "eval_args5",
      "apply", "eval_macro", "lambda", "quote", "macroexpand",
      "define", "define1", "begin", "begin_unchecked", "begin1",
      "if", "if1", "when", "when1", "unless", "unless1", "set", "set1", "set2",
      "let", "let1", "let_star", "let_star1", "let_star2",
      "letrec", "letrec1", "letrec_star", "letrec_star1", "cond", "cond1", "cond1_1", "cond_simple", "cond1_simple",
      "and", "and1", "or", "or1",
      "define_macro", "define_macro_star", "define_expansion",
      "case", "case1", "read_list", "read_next", "read_dot", "read_quote",
      "read_quasiquote", "read_unquote", "read_apply_values",
      "read_vector", "read_byte_vector", "read_done",
      "load_return_if_eof", "load_close_and_pop_if_eof", "eval_done",
      "catch", "dynamic_wind", "define_constant", "define_constant1",
      "do", "do_end", "do_end1", "do_step", "do_step2", "do_init",
      "define_star", "lambda_star", "lambda_star_default", "error_quit", "unwind_input", "unwind_output",
      "error_hook_quit",
      "with_let", "with_let1", "with_let_unchecked", "with_let_s",
      "with_baffle", "with_baffle_unchecked", "expansion",
      "for_each", "for_each_1", "for_each_2", "for_each_3", 
      "map", "map_1", "map_gather", "map_gather_1", "barrier", "deactivate_goto",

      "define_bacro", "define_bacro_star",
      "get_output_string", "get_output_string_1",
      "sort", "sort1", "sort2", "sort3", "sort_pair_end", "sort_vector_end", "sort_string_end",
      "eval_string_1", "eval_string_2",
      "member_if", "assoc_if", "member_if1", "assoc_if1",

      "quote_unchecked", "lambda_unchecked", "let_unchecked", "case_unchecked", "when_unchecked", "unless_unchecked",

      "set_unchecked", "set_symbol_c", "set_symbol_s", "set_symbol_q", "set_symbol_p", "set_symbol_z", "set_symbol_a",
      "set_symbol_opsq", "set_symbol_opcq", "set_symbol_opssq", "set_symbol_opsssq", 
      "set_normal", "set_pair", "set_pair_z", "set_pair_a", "set_pair_p", "set_pair_za",
      "set_pair_p_1", "set_with_accessor", "set_pws", "set_let_s", "set_let_all_x",
      "set_pair_c", "set_pair_c_p", "set_pair_c_p_1", "set_safe",
      "increment_1", "decrement_1", "set_cons",
      "increment_ss", "increment_sss", "increment_sz", "increment_sa", "increment_saa",

      "let_star_unchecked", "letrec_unchecked", "letrec_star_unchecked", "cond_unchecked",
      "lambda_star_unchecked", "do_unchecked", "define_unchecked", "define_star_unchecked", "define_funchecked", "define_constant_unchecked",
      "define_with_accessor", "define_macro_with_accessor",

      "let_no_vars", "named_let", "named_let_no_vars", "named_let_star",
      "let_c", "let_s", "let_all_c", "let_all_s", "let_all_x",
      "let_star_all_x", "let_opcq", "let_opssq",
      "let_opsq", "let_all_opsq", "let_opsq_p", "let_one", "let_one_1", "let_z", "let_z_1",

      "case_simple", "case_simpler", "case_simpler_1", "case_simpler_ss", "case_simplest", "case_simplest_ss", 
      "if_unchecked", "and_unchecked", "and_p", "and_p1", "and_p2", "or_unchecked", "or_p", "or_p1", "or_p2",
      "if_p_feed", "if_p_feed_1", "when_s", "unless_s",

      "if_s_p", "if_s_p_p", "if_not_s_p", "if_not_s_p_p", "if_cc_p", "if_cc_p_p",
      "if_cs_p", "if_cs_p_p", "if_csq_p", "if_csq_p_p", "if_css_p", "if_css_p_p",
      "if_csc_p", "if_csc_p_p", "if_is_pair_p", "if_is_pair_p_p", "if_opssq_p", "if_opssq_p_p", "if_s_opcq_p", "if_s_opcq_p_p",
      "if_is_symbol_p", "if_is_symbol_p_p", "if_a_p", "if_a_p_p", "if_and2_p", "if_and2_p_p",
      "if_z_p", "if_z_p_p", "if_p_p_p", "if_p_p", "if_andp_p", "if_andp_p_p", "if_orp_p", "if_orp_p_p",
      "if_ppp", "if_pp",

      "catch_1", "catch_2", "catch_all", "cond_all_x", "cond_all_x_2", "cond_s",
      "simple_do", "simple_do_step", "safe_dotimes", "safe_dotimes_step", "safe_dotimes_step_p", "safe_dotimes_step_o", "safe_dotimes_step_a",
      "safe_do", "safe_do_step", "simple_do_p", "simple_do_step_p", "dox", "dox_step", "dox_step_p", 
      "dotimes_p", "dotimes_step_p", "simple_do_a", "simple_do_step_a", "simple_do_e", "simple_do_step_e",

      "safe_c_p_1", "safe_c_pp_1", "safe_c_pp_2", "safe_c_pp_3", "safe_c_pp_4", "safe_c_pp_5", "safe_c_pp_6",
      "eval_args_p_2", "eval_args_p_2_mv", "eval_args_p_3", "eval_args_p_4", "eval_args_p_3_mv", "eval_args_p_4_mv",
      "eval_args_ssp_1", "eval_args_ssp_mv", "eval_macro_mv", "macroexpand_1",

      "safe_c_zz_1", "safe_c_zz_2", "safe_c_zc_1", "safe_c_sz_1", "safe_c_za_1", "increment_sz_1", "safe_c_sz_sz",
      "safe_c_zaa_1", "safe_c_aza_1", "safe_c_aaz_1", "safe_c_ssz_1",
      "safe_c_zza_1", "safe_c_zza_2", "safe_c_zaz_1", "safe_c_zaz_2", "safe_c_azz_1", "safe_c_azz_2",
      "safe_c_zzz_1", "safe_c_zzz_2", "safe_c_zzz_3",

      "safe_c_opsq_p_1", "safe_c_opsq_p_mv", "c_p_1", "c_p_2", "c_sp_1", "c_sp_2",
      "closure_p_1", "closure_p_2", "safe_closure_p_1",

      "set-with-let-1", "set-with-let-2",
};

static const char* opt_names[OPT_MAX_DEFINED] =
     {"safe_c_c", "h_safe_c_c", "safe_c_s", "h_safe_c_s",
      "safe_c_ss", "h_safe_c_ss", "safe_c_sc", "h_safe_c_sc", "safe_c_cs", "h_safe_c_cs",
      "safe_c_q", "h_safe_c_q", "safe_c_sq", "h_safe_c_sq", "safe_c_qs", "h_safe_c_qs", "safe_c_qq", "h_safe_c_qq",
      "safe_c_cq", "h_safe_c_cq", "safe_c_qc", "h_safe_c_qc",
      "safe_c_sss", "h_safe_c_sss", "safe_c_scs", "h_safe_c_scs", "safe_c_ssc", "h_safe_c_ssc", "safe_c_css", "h_safe_c_css",
      "safe_c_scc", "h_safe_c_scc", "safe_c_csc", "h_safe_c_csc",
      "safe_c_all_s", "h_safe_c_all_s", "safe_c_all_x", "h_safe_c_all_x", "safe_c_ssa", "h_safe_c_ssa", "safe_c_sas", "h_safe_c_sas",
      "safe_c_csa", "h_safe_c_csa", "safe_c_sca", "h_safe_c_sca", "safe_c_cas", "h_safe_c_cas",
      "safe_c_a", "h_safe_c_a", "safe_c_aa", "h_safe_c_aa", "safe_c_aaa", "h_safe_c_aaa", "safe_c_aaaa", "h_safe_c_aaaa",
      "safe_c_sqs", "h_safe_c_sqs", "safe_c_opaq", "h_safe_c_opaq", "safe_c_opaaq", "h_safe_c_opaaq", "safe_c_opaaaq", "h_safe_c_opaaaq",
      "safe_c_s_opaq", "h_safe_c_s_opaq", "safe_c_s_opaaq", "h_safe_c_s_opaaq", "safe_c_s_opaaaq", "h_safe_c_s_opaaaq",

      "safe_c_opcq", "h_safe_c_opcq", "safe_c_opsq", "h_safe_c_opsq",
      "safe_c_opssq", "h_safe_c_opssq", "safe_c_opscq", "h_safe_c_opscq", "safe_c_opsqq", "h_safe_c_opsqq",
      "safe_c_opcsq", "h_safe_c_opcsq", "safe_c_s_opsq", "h_safe_c_s_opsq",
      "safe_c_c_opscq", "h_safe_c_c_opscq",
      "safe_c_s_opscq", "h_safe_c_s_opscq", "safe_c_s_opcsq", "h_safe_c_s_opcsq",
      "safe_c_opsq_s", "h_safe_c_opsq_s", "safe_c_opsq_c", "h_safe_c_opsq_c",
      "safe_c_opsq_opsq", "h_safe_c_opsq_opsq", "safe_c_s_opssq", "h_safe_c_s_opssq", "safe_c_c_opsq", "h_safe_c_c_opsq",
      "safe_c_c_opcsq", "h_safe_c_c_opcsq", "safe_c_opcsq_c", "h_safe_c_opcsq_c",
      "safe_c_s_opcq", "h_safe_c_s_opcq", "safe_c_opssq_c", "h_safe_c_opssq_c", "safe_c_c_opssq", "h_safe_c_c_opssq",
      "safe_c_c_opcq", "h_safe_c_c_opcq", "safe_c_opcq_s", "h_safe_c_opcq_s",
      "safe_c_opcq_opcq", "h_safe_c_opcq_opcq", "safe_c_opcq_c", "h_safe_c_opcq_c",
      "safe_c_opscq_opscq", "h_safe_c_opscq_opscq", "safe_c_opssq_opssq", "h_safe_c_opssq_opssq",
      "safe_c_opssq_opcq", "h_safe_c_opssq_opcq", "safe_c_opssq_opsq", "h_safe_c_opssq_opsq", "safe_c_opsq_opssq", "h_safe_c_opsq_opssq",
      "safe_c_opssq_s", "h_safe_c_opssq_s", "safe_c_opscq_s", "h_safe_c_opscq_s", "safe_c_opcsq_s", "h_safe_c_opcsq_s",
      "safe_c_opscq_c", "h_safe_c_opscq_c", "safe_c_opcq_opssq", "h_safe_c_opcq_opssq",
      "safe_c_s_op_opssq_sq", "h_safe_c_s_op_opssq_sq", "safe_c_s_op_s_opssqq", "h_safe_c_s_op_s_opssqq",
      "safe_c_op_opssq_q_c", "h_safe_c_op_opssq_q_c", "safe_c_op_opsq_q_c", "h_safe_c_op_opsq_q_c", 
      "safe_c_op_opssq_q_s", "h_safe_c_op_opssq_q_s", "safe_c_op_opsq_q_s", "h_safe_c_op_opsq_q_s", 
      "safe_c_s_op_opssq_opssqq", "h_safe_c_s_op_opssq_opssqq",
      "safe_c_op_opsq_q", "h_safe_c_op_opsq_q", "safe_c_c_op_s_opcqq", "h_safe_c_c_op_s_opcqq",
      "safe_c_op_s_opsq_q", "h_safe_c_op_s_opsq_q",
      "safe_c_opsq_q", "h_safe_c_opsq_q", "safe_c_opsq_q_s", "h_safe_c_opsq_q_s",

      "safe_c_z", "h_safe_c_z", "safe_c_zz", "h_safe_c_zz", "safe_c_sz", "h_safe_c_sz", "safe_c_zs", "h_safe_c_zs",
      "safe_c_cz", "h_safe_c_cz", "safe_c_zc", "h_safe_c_zc",
      "safe_c_opcq_z", "h_safe_c_opcq_z", "safe_c_s_opszq", "h_safe_c_s_opszq",
      "safe_c_az", "h_safe_c_az", "safe_c_za", "h_safe_c_za",
      "safe_c_zaa", "h_safe_c_zaa", "safe_c_aza", "h_safe_c_aza", "safe_c_aaz", "h_safe_c_aaz", "safe_c_ssz", "h_safe_c_ssz",
      "safe_c_zza", "h_safe_c_zza", "safe_c_zaz", "h_safe_c_zaz", "safe_c_azz", "h_safe_c_azz",
      "safe_c_zzz", "h_safe_c_zzz", 

      "thunk", "h_thunk",
      "closure_s", "h_closure_s", "closure_c", "h_closure_c", "closure_q", "h_closure_q",
      "closure_ss", "h_closure_ss", "closure_sc", "h_closure_sc", "closure_cs", "h_closure_cs",
      "closure_a", "h_closure_a", "closure_aa", "h_closure_aa",
      "closure_all_x", "h_closure_all_x", "closure_all_s", "h_closure_all_s",

      "glosure_a", "h_glosure_a", "glosure_s", "h_glosure_s", "glosure_p", "h_glosure_p",

      "closure_star_s", "h_closure_star_s", "closure_star_sx", "h_closure_star_sx",
      "closure_star", "h_closure_star", "closure_star_all_x", "h_closure_star_all_x",

      "safe_thunk", "h_safe_thunk", "safe_thunk_e", "h_safe_thunk_e", "safe_thunk_p", "h_safe_thunk_p",
      "safe_closure_s", "h_safe_closure_s", "safe_closure_c", "h_safe_closure_c", "safe_closure_q", "h_safe_closure_q",
      "safe_closure_ss", "h_safe_closure_ss", "safe_closure_sc", "h_safe_closure_sc", "safe_closure_cs", "h_safe_closure_cs",
      "safe_closure_a", "h_safe_closure_a", "safe_closure_sa", "h_safe_closure_sa", "safe_closure_s_p", "h_safe_closure_s_p",
      "safe_closure_saa", "h_safe_closure_saa",
      "safe_closure_all_x", "h_safe_closure_all_x", "safe_closure_aa", "h_safe_closure_aa",

      "safe_glosure_a", "h_safe_glosure_a", "safe_glosure_s", "h_safe_glosure_s", "safe_glosure_s_e", "h_safe_glosure_s_e", 
      "safe_glosure_p", "h_safe_glosure_p",

      "safe_closure_star_s", "h_safe_closure_star_s", "safe_closure_star_ss", "h_safe_closure_star_ss",
      "safe_closure_star_sc", "h_safe_closure_star_sc", "safe_closure_star_sa", "h_safe_closure_star_sa", "safe_closure_star_s0", "h_safe_closure_star_s0",
      "safe_closure_star", "h_safe_closure_star", "safe_closure_star_all_x", "h_safe_closure_star_all_x",

      "apply_ss", "h_apply_ss",
      "c_all_x", "h_c_all_x", "call_with_exit", "h_call_with_exit", "c_catch", "h_c_catch", "c_catch_all", "h_c_catch_all",
      "c_s_opsq", "h_c_s_opsq", "c_s_opcq", "h_c_s_opcq", "c_ss", "h_c_ss",
      "c_s", "h_c_s", "read_s", "h_read_s", "c_p", "h_c_p", "c_z", "h_c_z", "c_sp", "h_c_sp",
      "c_sz", "h_c_sz", "c_a", "h_c_a", "c_scs", "h_c_scs",

      "goto", "h_goto", "goto_c", "h_goto_c", "goto_s", "h_goto_s", "goto_a", "h_goto_a",
      "vector_c", "h_vector_c", "vector_s", "h_vector_s", "vector_a", "h_vector_a", "vector_cc", "h_vector_cc",
      "string_c", "h_string_c", "string_s", "h_string_s", "string_a", "h_string_a",
      "c_object", "h_c_object", "c_object_c", "h_c_object_c", "c_object_s", "h_c_object_s", "c_object_a", "h_c_object_a", 
      "pair_c", "h_pair_c", "pair_s", "h_pair_s", "pair_a", "h_pair_a",
      "hash_table_c", "h_hash_table_c", "hash_table_s", "h_hash_table_s", "hash_table_a", "h_hash_table_a",
      "environment_s", "h_environment_s", "environment_q", "h_environment_q", "environment_a", "h_environment_a", "environment_c", "h_environment_c",

      "unknown", "h_unknown", "unknown_all_s", "h_unknown_all_s", "unknown_all_x", "h_unknown_all_x",
      "unknown_g", "h_unknown_g", "unknown_gg", "h_unknown_gg", "unknown_a", "h_unknown_a", "unknown_aa", "h_unknown_aa",

      "safe_c_pp", "h_safe_c_pp",
      "safe_c_opsq_p", "h_safe_c_opsq_p", 
      "safe_c_sp", "h_safe_c_sp", "safe_c_cp", "h_safe_c_cp", "safe_c_qp", "h_safe_c_qp", "safe_c_ap", "h_safe_c_ap",
      "safe_c_ps", "h_safe_c_ps", "safe_c_pc", "h_safe_c_pc", "safe_c_pq", "h_safe_c_pq",
      "safe_c_ssp", "h_safe_c_ssp",
};
#endif

#define is_safe_c_op(op) (op < OP_THUNK)                               /* used only in safe_stepper */
#define is_unknown_op(op) ((op >= OP_UNKNOWN) && (op < OP_SAFE_C_PP))
#define is_callable_c_op(op) ((op < OP_THUNK) || (op > OP_UNKNOWN_AA)) /* used only in check_set */

static bool is_h_optimized(s7_pointer p)
{
  return((is_optimized(p)) &&
	 ((optimize_op(p) & 1) != 0) &&
	 (!is_unknown_op(optimize_op(p))));
}

#define is_h_safe_c_c(P) ((is_optimized(P)) && (optimize_op(P) == HOP_SAFE_C_C))
#define is_h_safe_c_s(P) ((is_optimized(P)) && (optimize_op(P) == HOP_SAFE_C_S))
#define is_safe_c_s(P)   ((is_optimized(P)) && (op_no_hop(P) == OP_SAFE_C_S))

static int position_of(s7_pointer p, s7_pointer args)
{
  int i;
  for (i = 1; p != args; i++, args = cdr(args));
  return(i);
}

s7_pointer s7_method(s7_scheme *sc, s7_pointer obj, s7_pointer method)
{
  if (has_methods(obj))
    return(find_method(sc, find_let(sc, obj), method));
  return(sc->undefined);
}


/* if a method is shadowing a built-in like abs, it should expect the same args as abs and
 *   behave the same -- no multiple values etc.
 */
#define check_method(Sc, Obj, Method, Args)     \
  {                                             \
    s7_pointer func; 	                        \
    if ((has_methods(Obj)) && ((func = find_method(Sc, find_let(Sc, Obj), Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, func, Args)); \
  }

#define check_two_methods(Sc, Obj, Method1, Method2, Args)	                            \
  if (has_methods(Obj))                                                                     \
    {                                                                                       \
      s7_pointer func;							                    \
      func = find_method(Sc, find_let(Sc, Obj), Method1);                           \
      if ((func == Sc->undefined) && (Method1 != Method2) && (Method2)) func = find_method(Sc, find_let(Sc, Obj), Method2); \
      if (func != Sc->undefined) return(s7_apply_function(Sc, func, Args)); \
    }

static s7_pointer check_values(s7_scheme *sc, s7_pointer obj, s7_pointer args)
{
  check_method(sc, obj, sc->values_symbol, args);
  return(sc->gc_nil);
}

/* unfortunately, in the simplest cases, where a function (like number?) accepts any argument,
 *   this costs about a factor of 1.5 in speed (we're doing the normal check like s7_is_number,
 *   but then have to check has_methods before returning #f).  We can't use the old form until
 *   openlet is seen because the prior code might use #_number? which gets the value
 *   before the switch.  These simple functions normally do not dominate timing info, so I'll
 *   go ahead. It's mostly boilerplate:
 */

#define check_boolean_method(Sc, Checker, Method, Args)      \
  {                                                          \
    s7_pointer p;                                            \
    p = car(Args);                                           \
    if (Checker(p)) return(Sc->T);                           \
    check_method(Sc, p, Method, Args);                       \
    return(Sc->F);                                           \
  }

#define check_boolean_not_method(Sc, Checker, Method, Args)  \
  {                                                          \
    s7_pointer p, func;					     \
    p = find_symbol_checked(Sc, cadar(Args));                \
    if (Checker(p)) return(Sc->F);                           \
    if ((has_methods(p)) && ((func = find_method(Sc, find_let(Sc, p), Method)) != Sc->undefined) && \
	(s7_apply_function(Sc, func, list_1(Sc, p)) != Sc->F))		\
      return(Sc->F);                                         \
    return(Sc->T);                                           \
  }

 #define method_or_bust(Sc, Obj, Method, Args, Type, Num)	\
  {                                             \
    s7_pointer func; 	                        \
    if ((has_methods(Obj)) && ((func = find_method(Sc, find_let(Sc, Obj), Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, func, Args)); \
    if (Num == 0) return(simple_wrong_type_argument(Sc, Method, Obj, Type)); \
    return(wrong_type_argument(Sc, Method, Num, Obj, Type));		\
  }

#define method_or_bust_with_type(Sc, Obj, Method, Args, Type, Num)	\
  {                                             \
    s7_pointer func; 	                        \
    if ((has_methods(Obj)) && ((func = find_method(Sc, find_let(Sc, Obj), Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, func, Args)); \
    if (Num == 0) return(simple_wrong_type_argument_with_type(Sc, Method, Obj, Type)); \
    return(wrong_type_argument_with_type(Sc, Method, Num, Obj, Type));		\
  }


#define eval_error_any(Sc, ErrType, ErrMsg, Obj) \
  do {static s7_pointer _Err_ = NULL; if (!_Err_) _Err_ = s7_make_permanent_string(ErrMsg); \
      return(s7_error(Sc, ErrType, set_elist_2(Sc, _Err_, Obj)));} while (0)

#define eval_error(Sc, ErrMsg, Obj)       eval_error_any(Sc, Sc->syntax_error_symbol, ErrMsg, Obj)
#define eval_type_error(Sc, ErrMsg, Obj)  eval_error_any(Sc, Sc->wrong_type_arg_symbol, ErrMsg, Obj)
#define eval_range_error(Sc, ErrMsg, Obj) eval_error_any(Sc, Sc->out_of_range_symbol, ErrMsg, Obj)

#define eval_error_no_return(Sc, ErrType, ErrMsg, Obj)	\
  do {static s7_pointer _Err_ = NULL; \
      if (!_Err_) _Err_ = s7_make_permanent_string(ErrMsg); \
      s7_error(Sc, ErrType, set_elist_2(Sc, _Err_, Obj));} while (0)

#define eval_error_with_caller(Sc, ErrMsg, Caller, Obj)	\
  do {static s7_pointer _Err_ = NULL; \
      if (!_Err_) _Err_ = s7_make_permanent_string(ErrMsg); \
      return(s7_error(Sc, Sc->syntax_error_symbol, set_elist_3(Sc, _Err_, Caller, Obj)));} while (0)

static s7_pointer print_truncate(s7_scheme *sc, s7_pointer code);

#define eval_error_with_caller_and_print_limit(Sc, ErrMsg, Caller, Obj)	\
  do {static s7_pointer _Err_ = NULL; \
      if (!_Err_) _Err_ = s7_make_permanent_string(ErrMsg); \
      return(s7_error(Sc, Sc->syntax_error_symbol, set_elist_3(Sc, _Err_, Caller, print_truncate(Sc, Obj))));} while (0)

static s7_pointer set_elist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->elist_1, x1);
  return(sc->elist_1);
} 

static s7_pointer set_elist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->elist_2, x1);
  set_cadr(sc->elist_2, x2);
  return(sc->elist_2);
} 

static s7_pointer set_elist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p;
  p = sc->elist_3;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(sc->elist_3);
} 

static s7_pointer set_elist_4(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p;
  p = sc->elist_4;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(sc->elist_4);
} 

static s7_pointer set_elist_5(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5)
{
  s7_pointer p;
  p = sc->elist_5;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4); p = cdr(p);
  set_car(p, x5);
  return(sc->elist_5);
} 

static s7_pointer set_wlist_3(s7_scheme *sc, s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p;
  p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(lst);
} 

static s7_pointer set_wlist_4(s7_scheme *sc, s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p;
  p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(lst);
} 

static s7_pointer set_plist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->plist_1, x1);
  return(sc->plist_1);
} 

static s7_pointer set_plist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->plist_2, x1);
  set_cadr(sc->plist_2, x2);
  return(sc->plist_2);
} 

static s7_pointer set_plist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  return(set_wlist_3(sc, sc->plist_3, x1, x2, x3));
} 


/* -------------------------------- constants -------------------------------- */

s7_pointer s7_f(s7_scheme *sc)
{
  return(sc->F);
}


s7_pointer s7_t(s7_scheme *sc)
{
  return(sc->T);
}


s7_pointer s7_nil(s7_scheme *sc)
{
  return(sc->nil);
}


bool s7_is_null(s7_scheme *sc, s7_pointer p)
{
  return(is_null(p));
}


s7_pointer s7_undefined(s7_scheme *sc)
{
  return(sc->undefined);
}


s7_pointer s7_unspecified(s7_scheme *sc)
{
  return(sc->unspecified);
}


bool s7_is_unspecified(s7_scheme *sc, s7_pointer val)
{
  return(is_unspecified(val));
}


s7_pointer s7_eof_object(s7_scheme *sc)          /* returns #<eof> -- not equivalent to "eof-object?" */
{
  return(sc->eof_object);
}


static s7_pointer g_not(s7_scheme *sc, s7_pointer args)
{
  #define H_not "(not obj) returns #t if obj is #f, otherwise #t: (not ()) -> #f"
  #define Q_not pl_bt
  return(make_boolean(sc, is_false(sc, car(args))));
}


bool s7_boolean(s7_scheme *sc, s7_pointer x)
{
  return(x != sc->F);
}


bool s7_is_boolean(s7_pointer x)
{
  return(type(x) == T_BOOLEAN);
}


s7_pointer s7_make_boolean(s7_scheme *sc, bool x)
{
  return(make_boolean(sc, x));
}


static s7_pointer g_is_boolean(s7_scheme *sc, s7_pointer args)
{
  #define H_is_boolean "(boolean? obj) returns #t if obj is #f or #t: (boolean? ()) -> #f"
  #define Q_is_boolean pl_bt
  check_boolean_method(sc, s7_is_boolean, sc->is_boolean_symbol, args);
}


bool s7_is_constant(s7_pointer p)
{
  /* this means "always evaluates to the same thing", sort of, not "evaluates to itself":
   *   (let ((x 'x)) (and (not (constant? x)) (equal? x (eval x))))
   *   (and (constant? (list + 1)) (not (equal? (list + 1) (eval (list + 1)))))
   */
  return((type(p) != T_SYMBOL) || (is_immutable_symbol(p)));
}


static s7_pointer g_is_constant(s7_scheme *sc, s7_pointer args)
{
  #define H_is_constant "(constant? obj) returns #t if obj is a constant (unsettable): (constant? pi) -> #t"
  #define Q_is_constant pl_bt
  check_boolean_method(sc, s7_is_constant, sc->is_constant_symbol, args);
}


/* -------------------------------- GC -------------------------------- */

unsigned int s7_gc_protect(s7_scheme *sc, s7_pointer x)
{
  unsigned int loc;

  if (sc->gpofl_loc < 0)
    {
      unsigned int i, size, new_size;
      size = sc->protected_objects_size;
      new_size = 2 * size;
      vector_elements(sc->protected_objects) = (s7_pointer *)realloc(vector_elements(sc->protected_objects), new_size * sizeof(s7_pointer));
      vector_length(sc->protected_objects) = new_size;
      sc->protected_objects_size = new_size;
      sc->gpofl = (unsigned int *)realloc(sc->gpofl, new_size * sizeof(unsigned int));
      for (i = size; i < new_size; i++)
	{
	  vector_element(sc->protected_objects, i) = sc->gc_nil;
	  sc->gpofl[++sc->gpofl_loc] = i;
	}
    }

  loc = sc->gpofl[sc->gpofl_loc--];
#if DEBUGGING
  if (loc >= sc->protected_objects_size)
    fprintf(stderr, "sc->gpofl loc: %u (%d)\n", loc, sc->protected_objects_size);
  if (vector_element(sc->protected_objects, loc) != sc->gc_nil)
    fprintf(stderr, "protected object at %u about to be clobbered? %s\n", loc, DISPLAY(vector_element(sc->protected_objects, loc)));
#endif
  vector_element(sc->protected_objects, loc) = x;
  return(loc);
}

void s7_gc_unprotect(s7_scheme *sc, s7_pointer x)
{
  unsigned int i;

  for (i = 0; i < sc->protected_objects_size; i++)
    if (vector_element(sc->protected_objects, i) == x)
      {
	vector_element(sc->protected_objects, i) = sc->gc_nil;
	sc->gpofl[++sc->gpofl_loc] = i;
	return;
      }
}


void s7_gc_unprotect_at(s7_scheme *sc, unsigned int loc)
{
  if (loc < sc->protected_objects_size)
    {
      if (vector_element(sc->protected_objects, loc) != sc->gc_nil)
	sc->gpofl[++sc->gpofl_loc] = loc;
      vector_element(sc->protected_objects, loc) = sc->gc_nil;
    }
}


s7_pointer s7_gc_protected_at(s7_scheme *sc, unsigned int loc)
{
  s7_pointer obj;

  obj = sc->unspecified;
  if (loc < sc->protected_objects_size)
    obj = vector_element(sc->protected_objects, loc);

  if (obj == sc->gc_nil)
    return(sc->unspecified);

  return(obj);
}

#define gc_protected_at(Sc, Loc) vector_element(Sc->protected_objects, Loc)


static void (*mark_function[NUM_TYPES])(s7_pointer p);

#define S7_MARK(Obj) do {s7_pointer _p_; _p_ = Obj; if (!is_marked(_p_)) (*mark_function[unchecked_type(_p_)])(_p_);} while (0)

static void mark_symbol(s7_pointer p)
{
  if (is_gensym(p))
    set_mark(p);
  /* don't set the mark bit of a normal symbol!  It wrecks the check against SYNTACTIC_TYPE,
   *   slowing everything down by a large amount.
   */
}

static void mark_noop(s7_pointer p) {}

/* ports can be alloc'd and freed at a frightening pace, so I think I'll make a special free_list for them. */

static port_t *alloc_port(s7_scheme *sc)
{
  if (sc->port_heap)
    {
      port_t *p;
      p = sc->port_heap;
      sc->port_heap = (port_t *)(p->next);
      return(p);
    }
  return((port_t *)calloc(1, sizeof(port_t)));
}


static void free_port(s7_scheme *sc, port_t *p)
{
  p->next = (void *)(sc->port_heap);
  sc->port_heap = p;
}

static void close_output_port(s7_scheme *sc, s7_pointer p);

static void sweep(s7_scheme *sc)
{
  unsigned int i, j;
  if (sc->strings_loc > 0)
    {
      /* unrolling this loop is not an improvement */
      for (i = 0, j = 0; i < sc->strings_loc; i++)
	{
	  s7_pointer s1;
	  s1 = sc->strings[i];
	  if (is_free_and_clear(s1))
	    {
	      if (string_needs_free(s1))
		free(string_value(s1));
	    }
	  else sc->strings[j++] = s1;
	}
      sc->strings_loc = j;
    }

  if (sc->gensyms_loc > 0)
    {
      for (i = 0, j = 0; i < sc->gensyms_loc; i++)
	{
	  s7_pointer s1;
	  s1 = sc->gensyms[i];
	  if (is_free_and_clear(s1))
	    {
	      remove_gensym_from_symbol_table(sc, s1); /* this uses symbol_name_cell data */
	      free(symbol_name(s1));
	      if ((is_documented(s1)) &&
		  (symbol_help(s1)))
		{
		  free(symbol_help(s1));
		  symbol_help(s1) = NULL;
		}
	      free(symbol_name_cell(s1));
	    }
	  else sc->gensyms[j++] = s1;
	}
      sc->gensyms_loc = j;
      if (j == 0) mark_function[T_SYMBOL] = mark_noop;
    }

  if (sc->c_objects_loc > 0)
    {
      for (i = 0, j = 0; i < sc->c_objects_loc; i++)
	{
	  if (is_free_and_clear(sc->c_objects[i]))
	    free_object(sc->c_objects[i]);
	  else sc->c_objects[j++] = sc->c_objects[i];
	}
      sc->c_objects_loc = j;
    }

  if (sc->vectors_loc > 0)
    {
      for (i = 0, j = 0; i < sc->vectors_loc; i++)
	{
	  if (is_free_and_clear(sc->vectors[i]))
	    {
	      s7_pointer a;
	      a = sc->vectors[i];

	      /* a multidimensional empty vector can have dimension info, wrapped vectors always have dimension info */
	      if (vector_dimension_info(a))
		{
		  if (vector_dimensions_allocated(a))
		    {
		      free(vector_dimensions(a));
		      free(vector_offsets(a));
		    }
		  if (vector_elements_allocated(a))
		    free(vector_elements(a));      /* I think this will work for any vector (int/float too) */
		  if (vector_dimension_info(a) != sc->wrap_only)
		    free(vector_dimension_info(a));
		}
	      else
		{
		  if (vector_length(a) != 0)
		    free(vector_elements(a));
		}
	    }
	  else sc->vectors[j++] = sc->vectors[i];
	  /* here (in the else branch) if a vector constant in a global function has been removed from the heap,
	   *   not_in_heap(heap_location(v)), and we'll never see it freed, so if there were a lot of these, they might
	   *   glom up this loop.  Surely not a big deal!?
	   */
	}
      sc->vectors_loc = j;
    }

  if (sc->hash_tables_loc > 0)
    {
      for (i = 0, j = 0; i < sc->hash_tables_loc; i++)
	{
	  if (is_free_and_clear(sc->hash_tables[i]))
	    {
	      if (hash_table_mask(sc->hash_tables[i]) > 0)
		free_hash_table(sc->hash_tables[i]);
	    }
	  else sc->hash_tables[j++] = sc->hash_tables[i];
	}
      sc->hash_tables_loc = j;
    }

  if (sc->input_ports_loc > 0)
    {
      for (i = 0, j = 0; i < sc->input_ports_loc; i++)
	{
	  if (is_free_and_clear(sc->input_ports[i]))
	    {
	      s7_pointer a;
	      a = sc->input_ports[i];
	      if (port_needs_free(a))
		{
		  if (port_data(a))
		    {
		      free(port_data(a));
		      port_data(a) = NULL;
		      port_data_size(a) = 0;
		    }
		  port_needs_free(a) = false;
		}

	      if (port_filename(a))
		{
		  free(port_filename(a));
		  port_filename(a) = NULL;
		}
	      free_port(sc, port_port(a));
	    }
	  else sc->input_ports[j++] = sc->input_ports[i];
	}
      sc->input_ports_loc = j;
    }

  if (sc->output_ports_loc > 0)
    {
      for (i = 0, j = 0; i < sc->output_ports_loc; i++)
	{
	  if (is_free_and_clear(sc->output_ports[i]))
	    {
	      close_output_port(sc, sc->output_ports[i]); /* needed for free filename, etc */
	      free_port(sc, port_port(sc->output_ports[i]));
	    }
	  else sc->output_ports[j++] = sc->output_ports[i];
	}
      sc->output_ports_loc = j;
    }

  if (sc->continuations_loc > 0)
    {
      for (i = 0, j = 0; i < sc->continuations_loc; i++)
	{
	  if (is_free_and_clear(sc->continuations[i]))
	    {
	      s7_pointer c;
	      c = sc->continuations[i];
	      if (continuation_op_stack(c))
		{
		  free(continuation_op_stack(c));
		  continuation_op_stack(c) = NULL;
		}
	      free(continuation_data(c));
	    }
	  else sc->continuations[j++] = sc->continuations[i];
	}
      sc->continuations_loc = j;
    }

#if WITH_GMP
  if (sc->bigints_loc > 0)
    {
      for (i = 0, j = 0; i < sc->bigints_loc; i++)
	{
	  s7_pointer s1;
	  s1 = sc->bigints[i];
	  if (is_free_and_clear(s1))
	    mpz_clear(big_integer(s1));
	  else sc->bigints[j++] = s1;
	}
      sc->bigints_loc = j;
    }

  if (sc->bigratios_loc > 0)
    {
      for (i = 0, j = 0; i < sc->bigratios_loc; i++)
	{
	  s7_pointer s1;
	  s1 = sc->bigratios[i];
	  if (is_free_and_clear(s1))
	    mpq_clear(big_ratio(s1));
	  else sc->bigratios[j++] = s1;
	}
      sc->bigratios_loc = j;
    }

  if (sc->bigreals_loc > 0)
    {
      for (i = 0, j = 0; i < sc->bigreals_loc; i++)
	{
	  s7_pointer s1;
	  s1 = sc->bigreals[i];
	  if (is_free_and_clear(s1))
	    mpfr_clear(big_real(s1));
	  else sc->bigreals[j++] = s1;
	}
      sc->bigreals_loc = j;
    }

  if (sc->bignumbers_loc > 0)
    {
      for (i = 0, j = 0; i < sc->bignumbers_loc; i++)
	{
	  s7_pointer s1;
	  s1 = sc->bignumbers[i];
	  if (is_free_and_clear(s1))
	    mpc_clear(big_complex(s1));
	  else sc->bignumbers[j++] = s1;
	}
      sc->bignumbers_loc = j;
    }
#endif
}


static void add_string(s7_scheme *sc, s7_pointer p)
{
  if (sc->strings_loc == sc->strings_size)
    {
      sc->strings_size *= 2;
      sc->strings = (s7_pointer *)realloc(sc->strings, sc->strings_size * sizeof(s7_pointer));
    }
  sc->strings[sc->strings_loc++] = p;
}

#define Add_String(Str) if (sc->strings_loc == sc->strings_size) add_string(sc, Str); else sc->strings[sc->strings_loc++] = Str


static void add_gensym(s7_scheme *sc, s7_pointer p)
{
  if (sc->gensyms_loc == sc->gensyms_size)
    {
      sc->gensyms_size *= 2;
      sc->gensyms = (s7_pointer *)realloc(sc->gensyms, sc->gensyms_size * sizeof(s7_pointer));
    }
  sc->gensyms[sc->gensyms_loc++] = p;
  mark_function[T_SYMBOL] = mark_symbol;
}


static void add_c_object(s7_scheme *sc, s7_pointer p)
{
  if (sc->c_objects_loc == sc->c_objects_size)
    {
      sc->c_objects_size *= 2;
      sc->c_objects = (s7_pointer *)realloc(sc->c_objects, sc->c_objects_size * sizeof(s7_pointer));
    }
  sc->c_objects[sc->c_objects_loc++] = p;
}


static void add_hash_table(s7_scheme *sc, s7_pointer p)
{
  if (sc->hash_tables_loc == sc->hash_tables_size)
    {
      sc->hash_tables_size *= 2;
      sc->hash_tables = (s7_pointer *)realloc(sc->hash_tables, sc->hash_tables_size * sizeof(s7_pointer));
    }
  sc->hash_tables[sc->hash_tables_loc++] = p;
}


static void add_vector(s7_scheme *sc, s7_pointer p)
{
  if (sc->vectors_loc == sc->vectors_size)
    {
      sc->vectors_size *= 2;
      sc->vectors = (s7_pointer *)realloc(sc->vectors, sc->vectors_size * sizeof(s7_pointer));
    }
  sc->vectors[sc->vectors_loc++] = p;
}

#define Add_Vector(Vec) if (sc->vectors_loc == sc->vectors_size) add_vector(sc, Vec); else sc->vectors[sc->vectors_loc++] = Vec

static void add_input_port(s7_scheme *sc, s7_pointer p)
{
  if (sc->input_ports_loc == sc->input_ports_size)
    {
      sc->input_ports_size *= 2;
      sc->input_ports = (s7_pointer *)realloc(sc->input_ports, sc->input_ports_size * sizeof(s7_pointer));
    }
  sc->input_ports[sc->input_ports_loc++] = p;
}


static void add_output_port(s7_scheme *sc, s7_pointer p)
{
  if (sc->output_ports_loc == sc->output_ports_size)
    {
      sc->output_ports_size *= 2;
      sc->output_ports = (s7_pointer *)realloc(sc->output_ports, sc->output_ports_size * sizeof(s7_pointer));
    }
  sc->output_ports[sc->output_ports_loc++] = p;
}


static void add_continuation(s7_scheme *sc, s7_pointer p)
{
  if (sc->continuations_loc == sc->continuations_size)
    {
      sc->continuations_size *= 2;
      sc->continuations = (s7_pointer *)realloc(sc->continuations, sc->continuations_size * sizeof(s7_pointer));
    }
  sc->continuations[sc->continuations_loc++] = p;
}

#if WITH_GMP
static void add_bigint(s7_scheme *sc, s7_pointer p)
{
  if (sc->bigints_loc == sc->bigints_size)
    {
      sc->bigints_size *= 2;
      sc->bigints = (s7_pointer *)realloc(sc->bigints, sc->bigints_size * sizeof(s7_pointer));
    }
  sc->bigints[sc->bigints_loc++] = p;
}


static void add_bigratio(s7_scheme *sc, s7_pointer p)
{
  if (sc->bigratios_loc == sc->bigratios_size)
    {
      sc->bigratios_size *= 2;
      sc->bigratios = (s7_pointer *)realloc(sc->bigratios, sc->bigratios_size * sizeof(s7_pointer));
    }
  sc->bigratios[sc->bigratios_loc++] = p;
}


static void add_bigreal(s7_scheme *sc, s7_pointer p)
{
  if (sc->bigreals_loc == sc->bigreals_size)
    {
      sc->bigreals_size *= 2;
      sc->bigreals = (s7_pointer *)realloc(sc->bigreals, sc->bigreals_size * sizeof(s7_pointer));
    }
  sc->bigreals[sc->bigreals_loc++] = p;
}


static void add_bignumber(s7_scheme *sc, s7_pointer p)
{
  if (sc->bignumbers_loc == sc->bignumbers_size)
    {
      sc->bignumbers_size *= 2;
      sc->bignumbers = (s7_pointer *)realloc(sc->bignumbers, sc->bignumbers_size * sizeof(s7_pointer));
    }
  sc->bignumbers[sc->bignumbers_loc++] = p;
}
#endif


#define INIT_GC_CACHE_SIZE 64
static void init_gc_caches(s7_scheme *sc)
{
  sc->strings_size = INIT_GC_CACHE_SIZE * 16;
  sc->strings_loc = 0;
  sc->strings = (s7_pointer *)malloc(sc->strings_size * sizeof(s7_pointer));
  sc->gensyms_size = INIT_GC_CACHE_SIZE;
  sc->gensyms_loc = 0;
  sc->gensyms = (s7_pointer *)malloc(sc->gensyms_size * sizeof(s7_pointer));
  sc->vectors_size = INIT_GC_CACHE_SIZE * 8;
  sc->vectors_loc = 0;
  sc->vectors = (s7_pointer *)malloc(sc->vectors_size * sizeof(s7_pointer));
  sc->hash_tables_size = INIT_GC_CACHE_SIZE;
  sc->hash_tables_loc = 0;
  sc->hash_tables = (s7_pointer *)malloc(sc->hash_tables_size * sizeof(s7_pointer));
  sc->input_ports_size = INIT_GC_CACHE_SIZE;
  sc->input_ports_loc = 0;
  sc->input_ports = (s7_pointer *)malloc(sc->input_ports_size * sizeof(s7_pointer));
  sc->output_ports_size = INIT_GC_CACHE_SIZE;
  sc->output_ports_loc = 0;
  sc->output_ports = (s7_pointer *)malloc(sc->output_ports_size * sizeof(s7_pointer));
  sc->continuations_size = INIT_GC_CACHE_SIZE;
  sc->continuations_loc = 0;
  sc->continuations = (s7_pointer *)malloc(sc->continuations_size * sizeof(s7_pointer));
  sc->c_objects_size = INIT_GC_CACHE_SIZE;
  sc->c_objects_loc = 0;
  sc->c_objects = (s7_pointer *)malloc(sc->c_objects_size * sizeof(s7_pointer));
#if WITH_GMP
  sc->bigints_size = INIT_GC_CACHE_SIZE;
  sc->bigints_loc = 0;
  sc->bigints = (s7_pointer *)malloc(sc->bigints_size * sizeof(s7_pointer));
  sc->bigratios_size = INIT_GC_CACHE_SIZE;
  sc->bigratios_loc = 0;
  sc->bigratios = (s7_pointer *)malloc(sc->bigratios_size * sizeof(s7_pointer));
  sc->bigreals_size = INIT_GC_CACHE_SIZE;
  sc->bigreals_loc = 0;
  sc->bigreals = (s7_pointer *)malloc(sc->bigreals_size * sizeof(s7_pointer));
  sc->bignumbers_size = INIT_GC_CACHE_SIZE;
  sc->bignumbers_loc = 0;
  sc->bignumbers = (s7_pointer *)malloc(sc->bignumbers_size * sizeof(s7_pointer));
#endif

  /* slightly unrelated... */
  sc->setters_size = 4;
  sc->setters_loc = 0;
  sc->setters = (s7_pointer *)malloc(sc->c_objects_size * sizeof(s7_pointer));
}


static void add_setter(s7_scheme *sc, s7_pointer p, s7_pointer setter)
{
  /* procedure-setters GC-protected. The c_function_setter field can't be used because the built-in functions
   *   are often removed from the heap and never thereafter marked.
   */
  unsigned int i;
  for (i = 0; i < sc->setters_loc; i++)
    {
      s7_pointer x;
      x = sc->setters[i];
      if (car(x) == p)
	{
	  set_cdr(x, setter);
	  return;
 	}
    }
  if (sc->setters_loc == sc->setters_size)
    {
      sc->setters_size *= 2;
      sc->setters = (s7_pointer *)realloc(sc->setters, sc->setters_size * sizeof(s7_pointer));
    }
  sc->setters[sc->setters_loc++] = permanent_cons(p, setter, T_PAIR | T_IMMUTABLE);
}


static void mark_vector_1(s7_pointer p, s7_int top)
{
  s7_pointer *tp, *tend, *tend4;

  set_mark(p);

  tp = (s7_pointer *)(vector_elements(p));
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);

  tend4 = (s7_pointer *)(tend - 4);
  while (tp <= tend4)
    {
      S7_MARK(*tp++);
      S7_MARK(*tp++);
      S7_MARK(*tp++);
      S7_MARK(*tp++);
    }

  while (tp < tend)
    S7_MARK(*tp++);
}

static void mark_slot(s7_pointer p)
{
  set_mark(p);
  S7_MARK(slot_value(p));
  if (slot_has_accessor(p))
    S7_MARK(slot_accessor(p));

  if (is_gensym(slot_symbol(p))) /* (let () (apply define (gensym) (list 32)) (gc) (gc) (curlet)) */
    set_mark(slot_symbol(p));
}

static void mark_let(s7_pointer env)
{
  s7_pointer x;
  for (x = env; is_let(x) && (!is_marked(x)); x = outlet(x))
    {
      s7_pointer y;
      set_mark(x);
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (!is_marked(y)) /* slot value might be the enclosing let */
	  mark_slot(y);
    }
}

static void just_mark(s7_pointer p)
{
  set_mark(p);
}

static void mark_c_proc_star(s7_pointer p)
{
  set_mark(p);
  if (!has_simple_defaults(p))
    {
      s7_pointer arg;
      for (arg = c_function_call_args(p); is_pair(arg); arg = cdr(arg))
	S7_MARK(car(arg));
    }
}

static void mark_pair(s7_pointer p)
{
  s7_pointer x;
  set_mark(p);
  S7_MARK(car(p));
  /* if the list is huge, recursion to cdr(p) is problematic when there are strict limits on the stack size
   *  so I'll try something else... (This form is faster according to callgrind).
   *
   * in snd-14 or so through 15.3, sc->temp_cell_2|3 were used for trailing args in eval, but that meant
   *   the !is_marked check below (which is intended to catch cyclic lists) caused cells to be missed;
   *   since sc->args could contain permanently marked cells, if these were passed to g_vector, for example, and
   *   make_vector_1 triggered a GC call, we needed to mark both the permanent (always marked) cell and its contents,
   *   and continue through the rest of the list.  But adding temp_cell_2|3 to sc->permanent_objects was not enough.
   *   Now I've already forgotten the rest of the story, and it was just an hour ago! -- the upshot is that temp_cell_2|3
   *   are not now used as arg list members.
   */
  for (x = cdr(p); is_pair(x) && (!is_marked(x)); x = cdr(x))
    {
      set_mark(x);
      S7_MARK(car(x));
    }
  S7_MARK(x);
}

static void mark_counter(s7_pointer p)
{
  set_mark(p);
  S7_MARK(counter_result(p));
  S7_MARK(counter_list(p));
  S7_MARK(counter_let(p));
}

static void mark_closure(s7_pointer p)
{
  set_mark(p);
  S7_MARK(closure_args(p));
  S7_MARK(closure_body(p));
  mark_let(closure_let(p));
  S7_MARK(closure_setter(p));
}

static void mark_stack_1(s7_pointer p, s7_int top)
{
  s7_pointer *tp, *tend;
  set_mark(p);

  tp = (s7_pointer *)(vector_elements(p));
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);

  while (tp < tend)
    {
      S7_MARK(*tp++);
      S7_MARK(*tp++);
      S7_MARK(*tp++);
      tp++;
    }
}

static void mark_stack(s7_pointer p)
{
  /* we can have a bare stack awaiting a continuation to hold it if the new_cell for the continuation
   *    triggers the GC!  But we need a top-of-stack??
   */
  mark_stack_1(p, temp_stack_top(p));
}

static void mark_continuation(s7_pointer p)
{
  unsigned int i;
  set_mark(p);
  mark_stack_1(continuation_stack(p), continuation_stack_top(p));
  for (i = 0; i < continuation_op_loc(p); i++)
    S7_MARK(continuation_op_stack(p)[i]);
}

static void mark_vector(s7_pointer p)
{
  mark_vector_1(p, vector_length(p));
}

static void mark_vector_possibly_shared(s7_pointer p)
{
  /* If a subvector (an inner dimension) of a vector is the only remaining reference
   *    to the main vector, we want to make sure the main vector is not GC'd until
   *    the subvector is also GC-able.  The shared_vector field either points to the
   *    parent vector, or it is sc->F, so we need to check for a vector parent if
   *    the current is multidimensional (this will include 1-dim slices).  We need
   *    to keep the parent case separate (i.e. sc->F means the current is the original)
   *    so that we only free once (or remove_from_heap once).  
   *
   * If we have a shared-vector of a shared-vector, and the middle and original are not otherwise
   *   in use, we mark the middle one, but (since it itself is not in use anywhere else)
   *   we don't mark the original!  So we need to follow the share-vector chain marking every one.
   */
  if ((vector_has_dimensional_info(p)) &&
      (s7_is_vector(shared_vector(p))))
    mark_vector_possibly_shared(shared_vector(p));

  mark_vector_1(p, vector_length(p));
}

static void mark_int_or_float_vector(s7_pointer p)
{
  set_mark(p);
}

static void mark_int_or_float_vector_possibly_shared(s7_pointer p)
{
  if ((vector_has_dimensional_info(p)) &&
      (s7_is_vector(shared_vector(p))))
    mark_int_or_float_vector_possibly_shared(shared_vector(p));

  set_mark(p);
}

static void mark_c_object(s7_pointer p)
{
  set_mark(p);
  (*(c_object_mark(p)))(c_object_value(p));
}

static void mark_catch(s7_pointer p)
{
  set_mark(p);
  S7_MARK(catch_tag(p));
  S7_MARK(catch_handler(p));
}

static void mark_dynamic_wind(s7_pointer p)
{
  set_mark(p);
  S7_MARK(dynamic_wind_in(p));
  S7_MARK(dynamic_wind_out(p));
  S7_MARK(dynamic_wind_body(p));
}

static void mark_hash_table(s7_pointer p)
{
  set_mark(p);
  S7_MARK(hash_table_procedures(p));
  if (hash_table_entries(p) > 0)
    {
      unsigned int i, len;
      hash_entry_t **entries;
      entries = hash_table_elements(p);
      len = hash_table_mask(p) + 1;
      for (i = 0; i < len; i++)
	{
	  hash_entry_t *xp;
	  for (xp = entries[i++]; xp; xp = xp->next)
	    {
	      S7_MARK(xp->key);
	      S7_MARK(xp->value);
	    }
	  for (xp = entries[i]; xp; xp = xp->next)
	    {
	      S7_MARK(xp->key);
	      S7_MARK(xp->value);
	    }
	}
    }
}

static void mark_iterator(s7_pointer p)
{
  set_mark(p);
  S7_MARK(iterator_sequence(p));
  if (is_mark_seq(p))
    S7_MARK(iterator_current(p));
}

static void mark_input_port(s7_pointer p)
{
  set_mark(p);
  set_mark(port_original_input_string(p));
}

static void gf_mark(s7_scheme *sc)
{
  gc_obj *p;
  if (sc->cur_rf)
    for (p = sc->cur_rf->gc_list; p; p = p->nxt)
      S7_MARK(p->p);
}


static void init_mark_functions(void)
{
  mark_function[T_FREE]                = mark_noop;
  mark_function[T_UNIQUE]              = mark_noop;
  mark_function[T_UNSPECIFIED]         = mark_noop;
  mark_function[T_NIL]                 = mark_noop;
  mark_function[T_BOOLEAN]             = mark_noop;
  mark_function[T_STRING]              = just_mark;
  mark_function[T_INTEGER]             = just_mark;
  mark_function[T_RATIO]               = just_mark;
  mark_function[T_REAL]                = just_mark;
  mark_function[T_COMPLEX]             = just_mark;
  mark_function[T_BIG_INTEGER]         = just_mark;
  mark_function[T_BIG_RATIO]           = just_mark;
  mark_function[T_BIG_REAL]            = just_mark;
  mark_function[T_BIG_COMPLEX]         = just_mark;
  mark_function[T_SYMBOL]              = mark_noop; /* this changes to mark_symbol when gensyms are in the heap */
  mark_function[T_PAIR]                = mark_pair;
  mark_function[T_CLOSURE]             = mark_closure;
  mark_function[T_CLOSURE_STAR]        = mark_closure;
  mark_function[T_CONTINUATION]        = mark_continuation;
  mark_function[T_CHARACTER]           = mark_noop;
  mark_function[T_INPUT_PORT]          = mark_input_port;
  mark_function[T_VECTOR]              = mark_vector; /* this changes if shared vector created (similarly below) */
  mark_function[T_INT_VECTOR]          = mark_int_or_float_vector;
  mark_function[T_FLOAT_VECTOR]        = mark_int_or_float_vector; 
  mark_function[T_MACRO]               = mark_closure;
  mark_function[T_BACRO]               = mark_closure;
  mark_function[T_MACRO_STAR]          = mark_closure;
  mark_function[T_BACRO_STAR]          = mark_closure;
  mark_function[T_C_OBJECT]            = mark_c_object;
  mark_function[T_RANDOM_STATE]        = just_mark;
  mark_function[T_GOTO]                = just_mark;
  mark_function[T_OUTPUT_PORT]         = just_mark;
  mark_function[T_CATCH]               = mark_catch;
  mark_function[T_DYNAMIC_WIND]        = mark_dynamic_wind;
  mark_function[T_HASH_TABLE]          = mark_hash_table;
  mark_function[T_ITERATOR]            = mark_iterator;
  mark_function[T_SYNTAX]              = mark_noop;
  mark_function[T_LET]                 = mark_let;
  mark_function[T_STACK]               = mark_stack;
  mark_function[T_COUNTER]             = mark_counter;
  mark_function[T_SLOT]                = mark_slot;
  mark_function[T_BAFFLE]              = just_mark;
  mark_function[T_C_MACRO]             = just_mark;
  mark_function[T_C_POINTER]           = just_mark;
  mark_function[T_C_FUNCTION]          = just_mark;
  mark_function[T_C_FUNCTION_STAR]     = just_mark;  /* changes to mark_c_proc_star if defaults involve an expression */
  mark_function[T_C_ANY_ARGS_FUNCTION] = just_mark;
  mark_function[T_C_OPT_ARGS_FUNCTION] = just_mark;
  mark_function[T_C_RST_ARGS_FUNCTION] = just_mark;
}


static void mark_op_stack(s7_scheme *sc)
{
  s7_pointer *p, *tp;
  tp = sc->op_stack_now;
  p = sc->op_stack;
  while (p < tp)
    S7_MARK(*p++);
}

static void mark_rootlet(s7_scheme *sc)
{
  s7_pointer ge;
  s7_pointer *tmp, *top;

  ge = sc->rootlet;
  tmp = vector_elements(ge);
  top = (s7_pointer *)(tmp + sc->rootlet_entries);

  set_mark(ge);
  while (tmp < top)
    S7_MARK(slot_value(*tmp++));
}

void s7_mark_object(s7_pointer p)
{
  S7_MARK(p);
}

static void mark_permanent_objects(s7_scheme *sc)
{
  gc_obj *g;
  for (g = sc->permanent_objects; g; g = (gc_obj *)(g->nxt))
    S7_MARK(g->p);
}

static void unmark_permanent_objects(s7_scheme *sc)
{
  gc_obj *g;
  for (g = sc->permanent_objects; g; g = (gc_obj *)(g->nxt))
    clear_mark(g->p);
}


#ifndef _MSC_VER
  #include <time.h>
  #include <sys/time.h>
  static struct timeval start_time;
  static struct timezone z0;
#endif


#if DEBUGGING
static int last_gc_line = 0;
static const char *last_gc_func = NULL;
#endif

#define GC_STATS 1
#define HEAP_STATS 2
#define STACK_STATS 4

#define show_gc_stats(Sc) ((Sc->gc_stats & GC_STATS) != 0)
#define show_stack_stats(Sc) ((Sc->gc_stats & STACK_STATS) != 0)
#define show_heap_stats(Sc) ((Sc->gc_stats & HEAP_STATS) != 0)


static int gc(s7_scheme *sc)
{
  s7_cell **old_free_heap_top;
  /* mark all live objects (the symbol table is in permanent memory, not the heap) */
#if DEBUGGING
  #define gc_call(P, Tp) \
    p = (*tp++); \
    if (is_marked(p)) \
       clear_mark(p); \
    else \
      { \
        if (!is_free_and_clear(p))		\
          {								\
	    p->debugger_bits = 0; p->gc_line = last_gc_line; p->gc_func = last_gc_func;	\
            clear_type(p);	\
            (*fp++) = p;\
          }}
#else
  #define gc_call(P, Tp) p = (*tp++); if (is_marked(p)) clear_mark(p); else {if (!is_free_and_clear(p)) {clear_type(p); (*fp++) = p;}}
#endif

  if (show_gc_stats(sc))
    {
      fprintf(stdout, "gc ");
#if DEBUGGING
      fprintf(stdout, "%s[%d] ", last_gc_func, last_gc_line);
#endif
#ifndef _MSC_VER
      /* this is apparently deprecated in favor of clock_gettime -- what compile-time switch to use here?
       *   _POSIX_TIMERS, or perhaps use CLOCK_REALTIME, but clock_gettime requires -lrt -- no thanks.
       */
      gettimeofday(&start_time, &z0);
#endif
    }

  mark_rootlet(sc);
  S7_MARK(sc->args);
  mark_let(sc->envir);

  slot_set_value(sc->error_data, sc->F); 
  /* the other choice here is to explicitly mark slot_value(sc->error_data) as we do eval_history1/2 below.
   *    in both cases, the values are permanent lists that do not mark impermanent contents.
   *    this will need circular list checks, and can't depend on marked to exit early
   */
  mark_let(sc->owlet);
#if WITH_HISTORY
  {
    s7_pointer p1, p2;
    for (p1 = sc->eval_history1, p2 = sc->eval_history2; ; p2 = cdr(p2))
      {
	S7_MARK(car(p1));
	S7_MARK(car(p2));
	p1 = cdr(p1);
	if (p1 == sc->eval_history1) break; /* these are circular lists */
      }
  }
#endif

  S7_MARK(sc->code);
  mark_current_code(sc);
  mark_stack_1(sc->stack, s7_stack_top(sc));
  S7_MARK(sc->v);
  S7_MARK(sc->w);
  S7_MARK(sc->x);
  S7_MARK(sc->y);
  S7_MARK(sc->z);
  S7_MARK(sc->value);

  S7_MARK(sc->temp1);
  S7_MARK(sc->temp2);
  S7_MARK(sc->temp3);
  S7_MARK(sc->temp4);
  S7_MARK(sc->temp5);
  S7_MARK(sc->temp6);
  S7_MARK(sc->temp7);
  S7_MARK(sc->temp8);
  S7_MARK(sc->temp9);
  S7_MARK(sc->temp10);
  S7_MARK(sc->temp11);
  gf_mark(sc);

  set_mark(sc->input_port);
  S7_MARK(sc->input_port_stack);
  set_mark(sc->output_port);
  set_mark(sc->error_port);
  S7_MARK(sc->stacktrace_defaults);
  S7_MARK(sc->autoload_table);
  S7_MARK(sc->default_rng);

  mark_pair(sc->temp_cell_1);
  mark_pair(sc->temp_cell_2);
  S7_MARK(car(sc->t1_1));
  S7_MARK(car(sc->t2_1));
  S7_MARK(car(sc->t2_2));
  S7_MARK(car(sc->t3_1));
  S7_MARK(car(sc->t3_2));
  S7_MARK(car(sc->t3_3));

  S7_MARK(car(sc->a4_1));
  S7_MARK(car(sc->a4_2));
  S7_MARK(car(sc->a4_3));
  S7_MARK(car(sc->a4_4));

  S7_MARK(car(sc->plist_1));
  S7_MARK(car(sc->plist_2));
  S7_MARK(cadr(sc->plist_2));
  S7_MARK(car(sc->plist_3));
  S7_MARK(cadr(sc->plist_3));
  S7_MARK(caddr(sc->plist_3));

  {
    unsigned int i;
    s7_pointer p;
    for (i = 1; i < NUM_SAFE_LISTS; i++)
      if (list_is_in_use(sc->safe_lists[i]))
	for (p = sc->safe_lists[i]; is_pair(p); p = cdr(p))
	  S7_MARK(car(p));
    for (i = 0; i < sc->setters_loc; i++)
      S7_MARK(cdr(sc->setters[i]));
  }
  {
    int i;
    for (i = 0; i < sc->num_fdats; i++)
      if (sc->fdats[i])
	S7_MARK(sc->fdats[i]->curly_arg);
  }
  S7_MARK(sc->protected_objects);
  S7_MARK(sc->protected_accessors);

  /* now protect recent allocations using the free_heap cells above the current free_heap_top (if any).
   *
   * cells above sc->free_heap_top might be malloc'd garbage (after heap reallocation), so we keep track of
   *   where the last actually freed cells were after the previous GC call.  We're trying to
   *   GC protect the previous GC_TEMPS_SIZE allocated pointers so that the caller doesn't have
   *   to gc-protect every temporary cell.
   *
   * There's one remaining possible problem.  s7_remove_from_heap frees cells outside
   *   the GC and might push free_heap_top beyond its previous_free_heap_top, then
   *   an immediate explicit gc call might not see those temp cells.
   */
  {
    s7_pointer *tmps, *tmps_top;

    tmps = sc->free_heap_top;
    tmps_top = tmps + GC_TEMPS_SIZE;
    if (tmps_top > sc->previous_free_heap_top)
      tmps_top = sc->previous_free_heap_top;

    while (tmps < tmps_top)
      S7_MARK(*tmps++);
  }
  mark_op_stack(sc);
  mark_permanent_objects(sc);

  /* free up all unmarked objects */
  old_free_heap_top = sc->free_heap_top;

  {
    s7_pointer *fp, *tp, *heap_top;
    fp = sc->free_heap_top;

    tp = sc->heap;
    heap_top = (s7_pointer *)(sc->heap + sc->heap_size);

    while (tp < heap_top)          /* != here or ^ makes no difference */
      {
	s7_pointer p;
	/* from here down is gc_call, but I wanted one case explicit for readability */
	p = (*tp++);

	if (is_marked(p))          /* this order is faster than checking typeflag(p) != T_FREE first */
	  clear_mark(p);
	else
	  {
	    if (!is_free_and_clear(p)) /* if T_FREE, it's an already-free object -- the free_heap is usually not empty when we call the GC */
	      {
#if DEBUGGING
		p->debugger_bits = 0;
#endif
		clear_type(p); /* (this is needed -- otherwise we try to free some objects twice) */
		(*fp++) = p;
	      }
	  }

	/* this looks crazy, but it speeds up the entire GC process by 25%!
	 *   going from 16 to 32 saves .2% so it may not matter.
	 */
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);

	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);

	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);

	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);

	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);

	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);

	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);

	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
	gc_call(p, tp);
      }

    sc->free_heap_top = fp;
    sweep(sc);
  }

  unmark_permanent_objects(sc);
  sc->gc_freed = (int)(sc->free_heap_top - old_free_heap_top);

  if (show_gc_stats(sc))
    {
#ifndef _MSC_VER
      struct timeval t0;
      double secs;
      gettimeofday(&t0, &z0);
      secs = (t0.tv_sec - start_time.tv_sec) +  0.000001 * (t0.tv_usec - start_time.tv_usec);
#if (PRINT_NAME_PADDING == 8)
      fprintf(stdout, "freed %d/%u (free: %d), time: %f\n", sc->gc_freed, sc->heap_size, sc->free_heap_top - sc->free_heap, secs);
#else
      fprintf(stdout, "freed %d/%u (free: %ld), time: %f\n", sc->gc_freed, sc->heap_size, sc->free_heap_top - sc->free_heap, secs);
#endif
#else
      fprintf(stdout, "freed %d/%u\n", sc->gc_freed, sc->heap_size);
#endif
    }
  
  /* if (sc->begin_hook) call_begin_hook(sc); */
  sc->previous_free_heap_top = sc->free_heap_top;
  return(sc->gc_freed); /* needed by cell allocator to decide when to increase heap size */
}

void s7_gc_stats(s7_scheme *sc, bool on) {sc->gc_stats = (on) ? GC_STATS : 0;}
unsigned int s7_heap_size(s7_scheme *sc) {return(sc->heap_size);}
int s7_gc_freed(s7_scheme *sc) {return(sc->gc_freed);}


#define GC_TRIGGER_SIZE 64

/* new_cell has to include the new cell's type.  In the free list, it is 0 (T_FREE).  If we remove it here,
 *   but then hit some error before setting the type, the GC sweep thinks it is a free cell already and
 *   does not return it to the free list: a memory leak.
 */

#if (!DEBUGGING)
#define new_cell(Sc, Obj, Type)			\
  do {						\
    if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc); \
    Obj = (*(--(Sc->free_heap_top))); \
    set_type(Obj, Type);	      \
    } while (0)

#define new_cell_no_check(Sc, Obj, Type) do {Obj = (*(--(Sc->free_heap_top))); set_type(Obj, Type);} while (0)
  /* since sc->free_heap_trigger is GC_TRIGGER_SIZE above the free heap base, we don't need
   *   to check it repeatedly after the first such check.
   */
#else
static bool for_any_other_reason(s7_scheme *sc, int line)
{
#if 0
  static int ctr = 0;
  if ((sc->default_rng) &&
      (!sc->gc_off) &&
      (ctr > GC_TRIGGER_SIZE))
    {
      s7_double x;
      x = next_random(sc->default_rng);
      if (x > .995)
	{
	  ctr = 0;
	  return(true);
	}
    }
  ctr++;
#endif
  return(false);
}

#define new_cell(Sc, Obj, Type)			\
  do {						\
    if ((Sc->free_heap_top <= Sc->free_heap_trigger) || (for_any_other_reason(sc, __LINE__))) {last_gc_line = __LINE__; last_gc_func = __func__; try_to_call_gc(Sc);} \
    Obj = (*(--(Sc->free_heap_top))); \
    Obj->alloc_line = __LINE__;	 Obj->alloc_func = __func__;	\
    set_type(Obj, Type);	      \
    } while (0)

#define new_cell_no_check(Sc, Obj, Type)		\
  do {						\
    Obj = (*(--(Sc->free_heap_top)));					\
    Obj->alloc_line = __LINE__;	 Obj->alloc_func = __func__;		\
    set_type(Obj, Type);						\
    } while (0)
#endif


static void resize_heap(s7_scheme *sc)
{
  /* alloc more heap */
  unsigned int old_size, old_free, k;
  s7_cell *cells;
  s7_pointer p;

  old_size = sc->heap_size;
  old_free = sc->free_heap_top - sc->free_heap;

  if (sc->heap_size < 512000)
    sc->heap_size *= 2;
  else sc->heap_size += 512000;

  sc->heap = (s7_cell **)realloc(sc->heap, sc->heap_size * sizeof(s7_cell *));
  if (!(sc->heap))
    s7_warn(sc, 256, "heap reallocation failed! tried to get %lu bytes\n", (unsigned long)(sc->heap_size * sizeof(s7_cell *)));

  sc->free_heap = (s7_cell **)realloc(sc->free_heap, sc->heap_size * sizeof(s7_cell *));
  if (!(sc->free_heap))
    s7_warn(sc, 256, "free heap reallocation failed! tried to get %lu bytes\n", (unsigned long)(sc->heap_size * sizeof(s7_cell *)));

  sc->free_heap_trigger = (s7_cell **)(sc->free_heap + GC_TRIGGER_SIZE);
  sc->free_heap_top = sc->free_heap + old_free; /* incremented below, added old_free 21-Aug-12?!? */

  /* optimization suggested by K Matheussen */
  cells = (s7_cell *)calloc(sc->heap_size - old_size, sizeof(s7_cell));
  for (p = cells, k = old_size; k < sc->heap_size;)
    {
      sc->heap[k] = p;
      heap_location(p) = k++;
      (*sc->free_heap_top++) = p++;
      sc->heap[k] = p;
      heap_location(p) = k++;
      (*sc->free_heap_top++) = p++;
      sc->heap[k] = p;
      heap_location(p) = k++;
      (*sc->free_heap_top++) = p++;
      sc->heap[k] = p;
      heap_location(p) = k++;
      (*sc->free_heap_top++) = p++;
    }
  sc->previous_free_heap_top = sc->free_heap_top;

  if (show_heap_stats(sc))
    fprintf(stderr, "heap grows to %u\n", sc->heap_size);
}

static void try_to_call_gc(s7_scheme *sc)
{
  /* called only from new_cell and cons */
  if (sc->gc_off)
    {
      /* we can't just return here!  Someone needs a new cell, and once the heap free list is exhausted, segfault */
      resize_heap(sc);
    }
  else
    {
#if (!DEBUGGING)
      unsigned int freed_heap;
      freed_heap = gc(sc);
      if ((freed_heap < sc->heap_size / 2) &&
	  (freed_heap < 1000000)) /* if huge heap */
	resize_heap(sc);
#else
      gc(sc);
      if ((unsigned int)(sc->free_heap_top - sc->free_heap) < sc->heap_size / 2)
	resize_heap(sc);
#endif
    }
}

  /* originally I tried to mark each temporary value until I was done with it, but
   *   that way madness lies... By delaying GC of _every_ %$^#%@ pointer, I can dispense
   *   with hundreds of individual protections.  So the free_heap's last GC_TEMPS_SIZE
   *   allocated pointers are protected during the mark sweep.
   */


static s7_pointer g_gc(s7_scheme *sc, s7_pointer args)
{
  #define H_gc "(gc (on #t)) runs the garbage collector.  If 'on' is supplied, it turns the GC on or off. \
Evaluation produces a surprising amount of garbage, so don't leave the GC off for very long!"
  #define Q_gc s7_make_signature(sc, 2, sc->T, sc->is_boolean_symbol)

  if (is_not_null(args))
    {
      if (!s7_is_boolean(car(args)))
	method_or_bust(sc, car(args), sc->gc_symbol, args, T_BOOLEAN, 0);
      sc->gc_off = (car(args) == sc->F);
      if (sc->gc_off)
	return(sc->F);
    }
#if DEBUGGING
  last_gc_line = __LINE__; 
  last_gc_func = __func__;
#endif
  gc(sc);
  return(sc->unspecified);
}


s7_pointer s7_gc_on(s7_scheme *sc, bool on)
{
  sc->gc_off = !on;
  return(s7_make_boolean(sc, on));
}


static int permanent_cells = 0;
#if (!WITH_THREADS)
static s7_cell *alloc_pointer(void)
{
  #define ALLOC_SIZE 256
  static unsigned int alloc_k = ALLOC_SIZE;
  static s7_cell *alloc_cells = NULL;

  if (alloc_k == ALLOC_SIZE)     /* if either no current block or the block is used up */
    {                            /*   make a new block */
      permanent_cells += ALLOC_SIZE;
      alloc_cells = (s7_cell *)calloc(ALLOC_SIZE, sizeof(s7_cell));
      alloc_k = 0;
    }
  return(&alloc_cells[alloc_k++]);
}
#else
#define alloc_pointer() (s7_cell *)calloc(1, sizeof(s7_cell))
#endif


static void add_permanent_object(s7_scheme *sc, s7_pointer obj)
{
  gc_obj *g;
  g = (gc_obj *)malloc(sizeof(gc_obj));
  g->p = obj;
  g->nxt = sc->permanent_objects;
  sc->permanent_objects = g;
}


static void free_cell(s7_scheme *sc, s7_pointer p)
{
#if DEBUGGING
  p->debugger_bits = 0;
#endif
  clear_type(p);
  (*(sc->free_heap_top++)) = p;
}


static void s7_remove_from_heap(s7_scheme *sc, s7_pointer x)
{
  int loc;
  s7_pointer p;

  /* global functions are very rarely redefined, so we can remove the function body from
   *   the heap when it is defined.  If redefined, we currently lose the memory held by the
   *   old definition.  (It is not trivial to recover this memory because it is allocated
   *   in blocks, not by the pointer, I think, but s7_define is the point to try).
   *
   * There is at least one problem with this: if, for example, a function has
   *    a quoted (constant) list, then uses list-set! to change an element of it,
   *    then a GC happens, and the new element is GC'd because no one in the heap
   *    points to it, then we call the function again, and it tries to access
   *    that element.
   *
   *    (define (bad-idea)
   *      (let ((lst '(1 2 3)))
   *        (let ((result (list-ref lst 1)))
   *          (list-set! lst 1 (* 2.0 16.6))
   *          (gc)
   *          result)))
   *
   *     put that in a file, load it (to force removal), than call bad-idea a few times.
   * so... if (*s7* 'safety) is not 0, remove-from-heap is disabled.
   */
  loc = heap_location(x);
  if (not_in_heap(x)) return;

  switch (type(x))
    {
    case T_PAIR:
      unheap(x);
      p = alloc_pointer();
      sc->heap[loc] = p;
      (*sc->free_heap_top++) = p;
      heap_location(p) = loc;
#if 0
      /* this code fixes the problem above, but at some cost (gc + mark_pair up by about 2% in the worst case (snd-test.scm)) */
      if ((car(x) == sc->quote_symbol) &&
	  (is_pair(cadr(x))))
	{
	  add_permanent_object(sc, cdr(x));
	}
      else
	{
	  s7_remove_from_heap(sc, car(x));
	  s7_remove_from_heap(sc, cdr(x));
	}
#else
      s7_remove_from_heap(sc, car(x));
      s7_remove_from_heap(sc, cdr(x));
#endif
      return;

    case T_HASH_TABLE:
    case T_LET:
    case T_VECTOR: 
      /* not int|float_vector or string because none of their elements are GC-able (so unheap below is ok)
       *   but hash-table and let seem like they need protection? And let does happen via define-class.
       */
      add_permanent_object(sc, x);
      return;

    case T_SYNTAX:
      return;

    case T_SYMBOL:
      if (is_gensym(x))
	{
	  unsigned int i;
	  sc->heap[loc] = alloc_pointer();
	  free_cell(sc, sc->heap[loc]);
	  heap_location(sc->heap[loc]) = loc;

	  /* unheap(x); */
	  heap_location(x) = -heap_location(x);
	  /* if gensym is a hash-table key, then is removed from the heap, we need to be sure the hash-table map to it
	   *   continues to be valid.  symbol_hmap is abs(heap_location), and the possible overlap with other not-in-heap
	   *   ints is not problematic (they'll just hash to the same location).
	   */
	  for (i = 0; i < sc->gensyms_loc; i++) /* sc->gensyms reaches size 512 during s7test, but this search is called 3 times and costs nothing */
	    if (sc->gensyms[i] == x)
	      {
		unsigned int j;
		for (j = i + 1; i < sc->gensyms_loc - 1; i++, j++)
		  sc->gensyms[i] = sc->gensyms[j];
		sc->gensyms[i] = NULL;
		sc->gensyms_loc--;
		if (sc->gensyms_loc == 0) mark_function[T_SYMBOL] = mark_noop;
		break;
	      }
	}
      return;

    case T_CLOSURE: case T_CLOSURE_STAR:
    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
      unheap(x);
      p = alloc_pointer();
      free_cell(sc, p);
      sc->heap[loc] = p;
      heap_location(p) = loc;

      s7_remove_from_heap(sc, closure_args(x));
      s7_remove_from_heap(sc, closure_body(x));
      return;

    default:
      break;
    }

  unheap(x);
  p = alloc_pointer();
  free_cell(sc, p);
  sc->heap[loc] = p;
  heap_location(p) = loc;
}



/* -------------------------------- stacks -------------------------------- */

#define OP_STACK_INITIAL_SIZE 32

#if DEBUGGING
#define stop_at_error true

static void push_op_stack(s7_scheme *sc, s7_pointer op)
{
  (*sc->op_stack_now++) = _NFre(op);
  if (sc->op_stack_now > (sc->op_stack + sc->op_stack_size)) 
    {
      fprintf(stderr, "%sop_stack overflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
}

static s7_pointer pop_op_stack(s7_scheme *sc)
{
  s7_pointer op;
  op = (*(--(sc->op_stack_now)));
  if (sc->op_stack_now < sc->op_stack) 
    {
      fprintf(stderr, "%sop_stack underflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(_NFre(op));
}
#else
#define push_op_stack(Sc, Op) (*Sc->op_stack_now++) = Op
#define pop_op_stack(Sc)      (*(--(Sc->op_stack_now)))
#endif

static void initialize_op_stack(s7_scheme *sc)
{
  int i;
  sc->op_stack = (s7_pointer *)malloc(OP_STACK_INITIAL_SIZE * sizeof(s7_pointer));
  sc->op_stack_size = OP_STACK_INITIAL_SIZE;
  sc->op_stack_now = sc->op_stack;
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
  for (i = 0; i < OP_STACK_INITIAL_SIZE; i++)
    sc->op_stack[i] = sc->nil;
}


static void resize_op_stack(s7_scheme *sc)
{
  int i, loc, new_size;
  loc = (int)(sc->op_stack_now - sc->op_stack);
  new_size = sc->op_stack_size * 2;
  sc->op_stack = (s7_pointer *)realloc((void *)(sc->op_stack), new_size * sizeof(s7_pointer));
  for (i = sc->op_stack_size; i < new_size; i++)
    sc->op_stack[i] = sc->nil;
  sc->op_stack_size = (unsigned int)new_size;
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + loc);
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
}


#define stack_code(Stack, Loc)  vector_element(_TStk(Stack), Loc - 3)
#define stack_let(Stack, Loc)   vector_element(_TStk(Stack), Loc - 2)
#define stack_args(Stack, Loc)  vector_element(_TStk(Stack), Loc - 1)
#define stack_op(Stack, Loc)    ((opcode_t)(vector_element(_TStk(Stack), Loc)))

#if DEBUGGING
static void pop_stack(s7_scheme *sc)
{
  opcode_t cur_op;
  cur_op = sc->op;
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start) 
    {
      fprintf(stderr, "%sstack underflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  sc->code =  sc->stack_end[0];
  sc->envir = _TLid(sc->stack_end[1]);
  sc->args =  sc->stack_end[2];
  sc->op =    (opcode_t)(sc->stack_end[3]);
  if (sc->op > OP_MAX_DEFINED) 
    {
      fprintf(stderr, "%spop_stack[%d] invalid opcode: " INT_FORMAT "%s\n", BOLD_TEXT, __LINE__, sc->op, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if (unchecked_type(sc->code) == T_FREE)
    {
      fprintf(stderr, "%s%s[%d]: stack code is free, op: %s -> %s%s\n", BOLD_TEXT, __func__, __LINE__, op_names[cur_op], op_names[sc->op], UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if (unchecked_type(sc->args) == T_FREE)
    {
      fprintf(stderr, "%s%s[%d]: stack args is free, op: %s -> %s%s\n", BOLD_TEXT, __func__, __LINE__, op_names[cur_op], op_names[sc->op], UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
}

static void pop_stack_no_op(s7_scheme *sc)
{
  opcode_t cur_op;
  cur_op = sc->op;
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start) 
    {
      fprintf(stderr, "%sstack underflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  sc->code =  sc->stack_end[0];
  sc->envir = _TLid(sc->stack_end[1]);
  sc->args =  sc->stack_end[2];
  if (unchecked_type(sc->code) == T_FREE)
    {
      fprintf(stderr, "%s%s[%d]: stack code is free, op: %s -> %s%s\n", BOLD_TEXT, __func__, __LINE__, op_names[cur_op], op_names[sc->op], UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if (unchecked_type(sc->args) == T_FREE)
    {
      fprintf(stderr, "%s%s[%d]: stack args is free, op: %s -> %s%s\n", BOLD_TEXT, __func__, __LINE__, op_names[cur_op], op_names[sc->op], UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
}

static void push_stack(s7_scheme *sc, opcode_t op, s7_pointer args, s7_pointer code)
{
  if (sc->stack_end >= sc->stack_start + sc->stack_size) 
    {
      fprintf(stderr, "%sstack overflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if (op > OP_MAX_DEFINED) 
    {
      fprintf(stderr, "%spush_stack[%d] invalid opcode: " INT_FORMAT "%s\n", BOLD_TEXT, __LINE__, sc->op, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if (code) sc->stack_end[0] = _NFre(code);
  sc->stack_end[1] = _TLid(sc->envir);
  if (args) sc->stack_end[2] = _NFre(args);
  sc->stack_end[3] = (s7_pointer)op;
  sc->stack_end += 4;
}

#define push_stack_no_code(Sc, Op, Args) push_stack(Sc, Op, Args, Sc->gc_nil)
#define push_stack_no_args(Sc, Op, Code) push_stack(Sc, Op, Sc->gc_nil, Code)
/* in the non-debugging case, the sc->F's here are not set, so we can (later) pop free cells */

#else
/* these macros are faster than the equivalent simple function calls.  If the s7_scheme struct is set up to reflect the
 *    stack order [code envir args op], we can use memcpy here:
 *      #define pop_stack(Sc) do {Sc->stack_end -= 4; memcpy((void *)Sc, (void *)(Sc->stack_end), 4 * sizeof(s7_pointer));} while (0)
 *    but it is only slightly faster (.2% at best)!
 */

#define pop_stack(Sc) \
  do { \
      Sc->stack_end -= 4; \
      Sc->code =  Sc->stack_end[0]; \
      Sc->envir = Sc->stack_end[1]; \
      Sc->args =  Sc->stack_end[2]; \
      Sc->op =    (opcode_t)(Sc->stack_end[3]); \
  } while (0)

#define pop_stack_no_op(Sc) \
  do { \
      Sc->stack_end -= 4; \
      Sc->code =  Sc->stack_end[0]; \
      Sc->envir = Sc->stack_end[1]; \
      Sc->args =  Sc->stack_end[2]; \
  } while (0)

#define push_stack(Sc, Op, Args, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[1] = Sc->envir; \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_code(Sc, Op, Args) \
  do { \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_args(Sc, Op, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[1] = Sc->envir; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)
#endif
/* since we don't GC mark the stack past the stack_top, push_stack_no_args and friends can cause pop_stack to set
 *   sc->code and sc->args to currently free objects.
 */

#define main_stack_op(Sc)   ((opcode_t)(Sc->stack_end[-1]))
/* #define main_stack_args(Sc) (Sc->stack_end[-2]) */
/* #define main_stack_let(Sc)  (Sc->stack_end[-3]) */
/* #define main_stack_code(Sc) (Sc->stack_end[-4]) */
/* #define pop_main_stack(Sc)  Sc->stack_end -= 4 */

/* beware of main_stack_code!  If a function has a tail-call, the main_stack_code that form sees
 *   if main_stack_op==op-begin1 can change from call to call -- the begin actually refers
 *   to the caller, which is dependent on where the current function was called, so we can't hard-wire
 *   any optimizations based on that sequence.
 */

static void stack_reset(s7_scheme *sc)
{
  sc->stack_end = sc->stack_start;
  push_stack(sc, OP_EVAL_DONE, sc->nil, sc->nil);
  push_stack(sc, OP_BARRIER, sc->nil, sc->nil);
}


static void resize_stack(s7_scheme *sc)
{
  unsigned int i, new_size, loc;  /* long long ints?? sc->stack_size also is an unsigned int */

  loc = s7_stack_top(sc);
  new_size = sc->stack_size * 2;

  /* how can we trap infinite recursions?  Is a warning in order here?
   *   I think I'll add 'max-stack-size
   *   size currently reaches 8192 in s7test
   */
  if (new_size > sc->max_stack_size)
    s7_error(sc, s7_make_symbol(sc, "stack-too-big"), set_elist_1(sc, make_string_wrapper(sc, "stack has grown past (*s7* 'max-stack-size)")));

  vector_elements(sc->stack) = (s7_pointer *)realloc(vector_elements(sc->stack), new_size * sizeof(s7_pointer));
  if (!vector_elements(sc->stack))
    s7_error(sc, s7_make_symbol(sc, "stack-too-big"), set_elist_1(sc, make_string_wrapper(sc, "no room to expand stack?")));

  for (i = sc->stack_size; i < new_size; i++)
    vector_element(sc->stack, i) = sc->nil;
  vector_length(sc->stack) = new_size;
  sc->stack_size = new_size;

  sc->stack_start = vector_elements(sc->stack);
  sc->stack_end = (s7_pointer *)(sc->stack_start + loc);
  sc->stack_resize_trigger = (s7_pointer *)(sc->stack_start + sc->stack_size / 2);

  if (show_stack_stats(sc))
    fprintf(stderr, "stack grows to %u\n", new_size);
}

#define check_stack_size(Sc) \
  if (Sc->stack_end >= Sc->stack_resize_trigger) \
    { \
      if ((Sc->begin_hook) && (call_begin_hook(Sc))) return(Sc->F); \
      resize_stack(Sc); \
    }



/* -------------------------------- symbols -------------------------------- */

static unsigned long long int raw_string_hash(const unsigned char *key, unsigned int len)
{
  unsigned long long int x;
  unsigned char *cx = (unsigned char *)&x;

  x = 0;
  if (len <= 8)
    memcpy((void *)cx, (void *)key, len);
  else
    {
      unsigned long long int y;
      unsigned char *cy = (unsigned char *)&y;

      memcpy((void *)cx, (void *)key, 8);
      y = 0;
      len -= 8;
      memcpy((void *)cy, (void *)(key + 8), (len > 8) ? 8 : len);
      x |= y;
    }
  return(x);
}


static s7_pointer make_symbol_with_length(s7_scheme *sc, const char *name, unsigned int len);

static s7_pointer new_symbol(s7_scheme *sc, const char *name, unsigned int len, unsigned long long int hash, unsigned int location)
{
  s7_pointer x, str, p;
  unsigned char *base, *val;

  if (sc->symbol_table_is_locked)
    return(s7_error(sc, sc->error_symbol, set_elist_1(sc, make_string_wrapper(sc, "can't make symbol: symbol table is locked!"))));

  base = (unsigned char *)malloc(sizeof(s7_cell) * 3 + len + 1);
  x = (s7_pointer)base;
  str = (s7_pointer)(base + sizeof(s7_cell));
  p = (s7_pointer)(base + 2 * sizeof(s7_cell));
  val = (unsigned char *)(base + 3 * sizeof(s7_cell));
  memcpy((void *)val, (void *)name, len);
  val[len] = '\0';

  unheap(str);
  typeflag(str) = T_STRING | T_IMMUTABLE;                  /* avoid debugging confusion involving set_type (also below) */
  string_length(str) = len;
  string_value(str) = (char *)val;
  string_hash(str) = hash;
  string_needs_free(str) = false;

  unheap(x);
  typeflag(x) = T_SYMBOL;
  symbol_set_name_cell(x, str);
  set_global_slot(x, sc->undefined);                       /* was sc->nil */
  set_initial_slot(x, sc->undefined);
  symbol_set_local(x, 0LL, sc->nil);
  symbol_set_tag(x, 0);

  if (symbol_name_length(x) > 1)                           /* not 0, otherwise : is a keyword */
    {
      if (name[0] == ':')
	{
	  typeflag(x) |= (T_IMMUTABLE | T_KEYWORD);
	  keyword_set_symbol(x, make_symbol_with_length(sc, (char *)(name + 1), len - 1));
	  set_global_slot(x, s7_make_slot(sc, sc->nil, x, x));
	}
      else
	{
	  char c;
	  c = name[symbol_name_length(x) - 1];
	  if (c == ':')
	    {
	      char *kstr;
	      unsigned int klen;
	      klen = symbol_name_length(x) - 1;
	      /* can't used tmpbuf_* here (or not safely I think) because name is already using tmpbuf */
	      kstr = (char *)malloc((klen + 1) * sizeof(char));
	      memcpy((void *)kstr, (void *)name, klen);
	      kstr[klen] = 0;
	      typeflag(x) |= (T_IMMUTABLE | T_KEYWORD);
	      keyword_set_symbol(x, make_symbol_with_length(sc, kstr, klen));
	      set_global_slot(x, s7_make_slot(sc, sc->nil, x, x));
	      free(kstr);
	    }
	}
    }

  unheap(p);
  typeflag(p) = T_PAIR | T_IMMUTABLE;
  set_car(p, x);
  set_cdr(p, vector_element(sc->symbol_table, location));
  vector_element(sc->symbol_table, location) = p;
  pair_set_raw_hash(p, hash);
  pair_set_raw_len(p, len);
  pair_set_raw_name(p, string_value(str));
  return(x);
}

static s7_pointer make_symbol_with_length(s7_scheme *sc, const char *name, unsigned int len)
{
  s7_pointer x;
  unsigned long long int hash;
  unsigned int location;

  hash = raw_string_hash((const unsigned char *)name, len);
  location = hash % SYMBOL_TABLE_SIZE;

  if (len <= 8)
    {
      for (x = vector_element(sc->symbol_table, location); is_pair(x); x = cdr(x))
	if ((hash == pair_raw_hash(x)) &&
	    (len == pair_raw_len(x)))
	  return(car(x));
    }
  else
    {
      for (x = vector_element(sc->symbol_table, location); is_pair(x); x = cdr(x))
	if ((hash == pair_raw_hash(x)) &&
	    (len == pair_raw_len(x)) &&
	    (strings_are_equal_with_length(name, pair_raw_name(x), len))) /* length here because name might not be null-terminated */
	  return(car(x));
    }
  return(new_symbol(sc, name, len, hash, location));
}


static s7_pointer make_symbol(s7_scheme *sc, const char *name)
{
  return(make_symbol_with_length(sc, name, safe_strlen(name)));
}


s7_pointer s7_make_symbol(s7_scheme *sc, const char *name)
{
  if (!name) return(sc->F);
  return(make_symbol_with_length(sc, name, safe_strlen(name)));
}


static s7_pointer symbol_table_find_by_name(s7_scheme *sc, const char *name, unsigned long long int hash, unsigned int location)
{
  s7_pointer x;
  for (x = vector_element(sc->symbol_table, location); is_not_null(x); x = cdr(x))
    if ((hash == pair_raw_hash(x)) &&
	(strings_are_equal(name, pair_raw_name(x))))
      return(car(x));
  return(sc->nil);
}


s7_pointer s7_symbol_table_find_name(s7_scheme *sc, const char *name)
{
  unsigned long long int hash;
  unsigned int location;
  s7_pointer result;

  hash = raw_string_hash((const unsigned char *)name, safe_strlen(name));
  location = hash % SYMBOL_TABLE_SIZE;
  result = symbol_table_find_by_name(sc, name, hash, location);
  if (is_null(result))
    return(NULL);

  return(result);
}


#define FILLED true
#define NOT_FILLED false

static s7_pointer g_symbol_table(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_table "(symbol-table) returns a vector containing the current symbol-table symbols"
  #define Q_symbol_table s7_make_signature(sc, 1, sc->is_vector_symbol)

  s7_pointer lst, x;
  s7_pointer *els;
  int i, j, syms = 0;

  /* this can't be optimized by returning the actual symbol-table (a vector of lists), because
   *    gensyms can cause the table's lists and symbols to change at any time.  This wreaks havoc
   *    on traversals like for-each.  So, symbol-table returns a snap-shot of the table contents
   *    at the time it is called, and we call gc before making the list.  I suppose the next step
   *    is to check that we have room, and increase the heap here if necessary!
   *
   *    (define (for-each-symbol func num) (for-each (lambda (sym) (if (> num 0) (for-each-symbol func (- num 1)) (func sym))) (symbol-table)))
   *    (for-each-symbol (lambda (sym) (gensym) 1))
   */

  for (i = 0; i < vector_length(sc->symbol_table); i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      syms++;
  sc->w = make_vector_1(sc, syms, NOT_FILLED, T_VECTOR);
  els = vector_elements(sc->w);

  for (i = 0, j = 0; i < vector_length(sc->symbol_table); i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      els[j++] = car(x);

  lst = sc->w;
  sc->w = sc->nil;
  return(lst);
}


bool s7_for_each_symbol_name(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, void *data), void *data)
{
  /* this includes the special constants #<unspecified> and so on for simplicity -- are there any others? */
  int i;
  s7_pointer x;

  for (i = 0; i < vector_length(sc->symbol_table); i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      if (symbol_func(symbol_name(car(x)), data))
	return(true);

  return((symbol_func("#t", data))             ||
	 (symbol_func("#f", data))             ||
	 (symbol_func("#<unspecified>", data)) ||
	 (symbol_func("#<undefined>", data))   ||
	 (symbol_func("#<eof>", data))         ||
	 (symbol_func("#true", data))          ||
	 (symbol_func("#false", data)));
}


bool s7_for_each_symbol(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, s7_pointer value, void *data), void *data)
{
  int i;
  s7_pointer x;

  for (i = 0; i < vector_length(sc->symbol_table); i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      if (symbol_func(symbol_name(car(x)), cdr(x), data))
	return(true);

  return(false);
}


static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym)
{
  /* sym is a free cell at this point (we're called after the GC), but the name_cell is still intact */
  s7_pointer x, name;
  unsigned int location;

  name = symbol_name_cell(sym);
  location = string_hash(name) % SYMBOL_TABLE_SIZE;
  x = vector_element(sc->symbol_table, location);

  if (car(x) == sym)
    {
      vector_element(sc->symbol_table, location) = cdr(x);
      free(x);
    }
  else
    {
      s7_pointer y;
      for (y = x, x = cdr(x); is_pair(x); y = x, x = cdr(x))
	{
	  if (car(x) == sym)
	    {
	      set_cdr(y, cdr(x));
	      free(x);
	      return;
	    }
	}
#if DEBUGGING
      fprintf(stderr, "could not remove %s?\n", string_value(name));
#endif
    }
}


s7_pointer s7_gensym(s7_scheme *sc, const char *prefix)
{
  char *name;
  unsigned int len, location;
  unsigned long long int hash;
  s7_pointer x;

  len = safe_strlen(prefix) + 32;
  tmpbuf_malloc(name, len);
  /* there's no point in heroic efforts here to avoid name collisions -- the user can screw up no matter what we do */
  len = snprintf(name, len, "{%s}-%u", prefix, sc->gensym_counter++);
  hash = raw_string_hash((const unsigned char *)name, len);
  location = hash % SYMBOL_TABLE_SIZE;
  x = new_symbol(sc, name, len, hash, location);  /* not T_GENSYM -- might be called from outside */
  tmpbuf_free(name, len);
  return(x);
}


static bool s7_is_gensym(s7_pointer g) {return((is_symbol(g)) && (is_gensym(g)));}

static s7_pointer g_is_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_is_gensym "(gensym? sym) returns #t if sym is a gensym"
  #define Q_is_gensym pl_bt

  check_boolean_method(sc, s7_is_gensym, sc->is_gensym_symbol, args);
}


static char *pos_int_to_str(s7_int num, unsigned int *len, char endc)
{
  #define INT_TO_STR_SIZE 32
  static char itos[INT_TO_STR_SIZE];
  char *p, *op;

  p = (char *)(itos + INT_TO_STR_SIZE - 1);
  op = p;
  *p-- = '\0';
  if (endc != '\0') *p-- = endc;
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  (*len) = op - p;           /* this includes the trailing #\null */
  return((char *)(p + 1));
}

static s7_pointer g_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_gensym "(gensym (prefix \"gensym\")) returns a new, unused symbol"
  #define Q_gensym s7_make_signature(sc, 2, sc->is_gensym_symbol, sc->is_string_symbol)

  const char *prefix;
  char *name, *p;
  unsigned int len, plen, nlen, location;
  unsigned long long int hash;
  s7_pointer x, str, stc;

  /* get symbol name */
  if (is_not_null(args))
    {
      s7_pointer name;
      name = car(args);
      if (!is_string(name))
	method_or_bust(sc, name, sc->gensym_symbol, args, T_STRING, 0);
      prefix = string_value(name);
    }
  else prefix = "gensym";
  plen = safe_strlen(prefix);
  len = plen + 32;
  name = (char *)malloc(len * sizeof(char));
  name[0] = '{';
  if (plen > 0) memcpy((void *)(name + 1), prefix, plen);
  name[plen + 1] = '}';
  name[plen + 2] = '-';

  p = pos_int_to_str(sc->gensym_counter++, &len, '\0');
  memcpy((void *)(name + plen + 3), (void *)p, len);
  nlen = len + plen + 2;

  hash = raw_string_hash((const unsigned char *)name, nlen);
  location = hash % SYMBOL_TABLE_SIZE;

  /* make-string for symbol name */
  str = (s7_cell *)malloc(sizeof(s7_cell));  /* was calloc? */
  unheap(str);
#if DEBUGGING
  typeflag(str) = 0;
#endif
  set_type(str, T_STRING | T_IMMUTABLE);
  string_length(str) = nlen;
  string_value(str) = name;
  string_needs_free(str) = false;
  string_hash(str) = hash;

  /* allocate the symbol in the heap so GC'd when inaccessible */
  new_cell(sc, x, T_SYMBOL | T_GENSYM);
  symbol_set_name_cell(x, str);
  set_global_slot(x, sc->undefined);
  set_initial_slot(x, sc->undefined);
  symbol_set_local(x, 0LL, sc->nil);

  /* place new symbol in symbol-table, but using calloc so we can easily free it (remove it from the table) in GC sweep */
  stc = (s7_cell *)malloc(sizeof(s7_cell)); /* was calloc? */
#if DEBUGGING
  typeflag(stc) = 0;
#endif
  unheap(stc);
  set_type(stc, T_PAIR | T_IMMUTABLE);
  set_car(stc, x);
  set_cdr(stc, vector_element(sc->symbol_table, location));
  vector_element(sc->symbol_table, location) = stc;
  pair_set_raw_hash(stc, hash);
  pair_set_raw_len(stc, string_length(str));
  pair_set_raw_name(stc, string_value(str));

  add_gensym(sc, x);
  return(x);
}


s7_pointer s7_name_to_value(s7_scheme *sc, const char *name)
{
  return(s7_symbol_value(sc, make_symbol(sc, name)));
}


bool s7_is_symbol(s7_pointer p)
{
  return(is_symbol(p));
}


bool s7_is_syntax(s7_pointer p)
{
  return(is_syntax(p));
}


static s7_pointer g_is_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_is_symbol "(symbol? obj) returns #t if obj is a symbol"
  #define Q_is_symbol pl_bt

  check_boolean_method(sc, is_symbol, sc->is_symbol_symbol, args);
}


const char *s7_symbol_name(s7_pointer p)
{
  return(symbol_name(p));
}


static s7_pointer g_symbol_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_string "(symbol->string sym) returns the symbol sym converted to a string"
  #define Q_symbol_to_string s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_symbol_symbol)
  s7_pointer sym;

  sym = car(args);
  if (!is_symbol(sym))
    method_or_bust(sc, sym, sc->symbol_to_string_symbol, args, T_SYMBOL, 0);
  /* s7_make_string uses strlen which stops at an embedded null */
  return(s7_make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));    /* return a copy */
}

static s7_pointer symbol_to_string_uncopied;
static s7_pointer g_symbol_to_string_uncopied(s7_scheme *sc, s7_pointer args)
{
  s7_pointer sym;

  sym = car(args);
  if (!is_symbol(sym))
    method_or_bust(sc, sym, sc->symbol_to_string_symbol, args, T_SYMBOL, 0);
  return(symbol_name_cell(sym));
}


static s7_pointer g_string_to_symbol_1(s7_scheme *sc, s7_pointer str, s7_pointer caller)
{
  if (!is_string(str))
    method_or_bust(sc, str, caller, list_1(sc, str), T_STRING, 0);
  if (string_length(str) == 0)
    return(simple_wrong_type_argument_with_type(sc, caller, str, make_string_wrapper(sc, "a non-null string")));

  /* currently if the string has an embedded null, it marks the end of the new symbol name. */
  return(make_symbol_with_length(sc, string_value(str), string_length(str)));
}


static s7_pointer g_string_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_symbol "(string->symbol str) returns the string str converted to a symbol"
  #define Q_string_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_string_symbol)
  return(g_string_to_symbol_1(sc, car(args), sc->string_to_symbol_symbol));
}


static s7_pointer g_string_append(s7_scheme *sc, s7_pointer args);
static s7_pointer g_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol "(symbol str ...) returns its string arguments concatenated and converted to a symbol"
  #define Q_symbol s7_make_circular_signature(sc, 1, 2, sc->is_symbol_symbol, sc->is_string_symbol)
  if (is_null(cdr(args)))
    return(g_string_to_symbol_1(sc, car(args), sc->symbol_symbol));
  return(g_string_to_symbol_1(sc, g_string_append(sc, args), sc->symbol_symbol));
}


static s7_pointer add_sym_to_list(s7_scheme *sc, s7_pointer sym)
{
  symbol_set_tag(sym, sc->syms_tag);
  return(sym);
}

#define clear_syms_in_list(Sc) Sc->syms_tag++



/* -------------------------------- environments -------------------------------- */

#define new_frame(Sc, Old_Env, New_Env)		      \
  do {						      \
    s7_pointer _x_;				      \
      new_cell(Sc, _x_, T_LET); \
      let_id(_x_) = ++sc->let_number;		      \
      let_set_slots(_x_, Sc->nil);	              \
      set_outlet(_x_, Old_Env);	      \
      New_Env = _x_;				      \
  } while (0)


static s7_pointer new_frame_in_env(s7_scheme *sc, s7_pointer old_env)
{
  /* return(cons(sc, sc->nil, old_env)); */
  s7_pointer x;
  new_cell(sc, x, T_LET);
  let_id(x) = ++sc->let_number;
  let_set_slots(x, sc->nil);
  set_outlet(x, old_env);
  return(x);
}


static s7_pointer make_simple_let(s7_scheme *sc)
{
  s7_pointer frame;
  new_cell(sc, frame, T_LET);
  let_id(frame) = sc->let_number + 1;
  let_set_slots(frame, sc->nil);
  set_outlet(frame, sc->envir);
  return(frame);
}


/* in all these macros, symbol_set_local should follow slot_set_value so that we can evaluate the
 *    slot's value in its old state.
 */
#define add_slot(Frame, Symbol, Value)			\
  do {							\
    s7_pointer _slot_, _sym_, _val_;			\
    _sym_ = Symbol; _val_ = Value;			\
    new_cell_no_check(sc, _slot_, T_SLOT);\
    slot_set_symbol(_slot_, _sym_);			\
    slot_set_value(_slot_, _val_);			\
    symbol_set_local(_sym_, let_id(Frame), _slot_);	\
    set_next_slot(_slot_, let_slots(Frame));		\
    let_set_slots(Frame, _slot_);	                \
  } while (0)

#define add_slot_checked(Frame, Symbol, Value)		\
  do {							\
    s7_pointer _slot_, _sym_, _val_;			\
    _sym_ = Symbol; _val_ = Value;			\
    new_cell(sc, _slot_, T_SLOT);		        \
    slot_set_symbol(_slot_, _sym_);			\
    slot_set_value(_slot_, _val_);			\
    symbol_set_local(_sym_, let_id(Frame), _slot_);	\
    set_next_slot(_slot_, let_slots(Frame));		\
    let_set_slots(Frame, _slot_); 	                \
  } while (0)

/* no set_local here -- presumably done earlier in check_* */

#define new_frame_with_slot(Sc, Old_Env, New_Env, Symbol, Value) \
  do {								 \
    s7_pointer _x_, _slot_, _sym_, _val_;			 \
    _sym_ = Symbol; _val_ = Value;				\
    new_cell(Sc, _x_, T_LET);			                \
    let_id(_x_) = ++sc->let_number;				\
    set_outlet(_x_, Old_Env);			                \
    New_Env = _x_;						\
    new_cell_no_check(Sc, _slot_, T_SLOT);	                \
    slot_set_symbol(_slot_, _sym_);				\
    slot_set_value(_slot_, _val_);	                        \
    symbol_set_local(_sym_, sc->let_number, _slot_);            \
    set_next_slot(_slot_, sc->nil);			        \
    let_set_slots(_x_, _slot_);					\
  } while (0)


#define new_frame_with_two_slots(Sc, Old_Env, New_Env, Symbol1, Value1, Symbol2, Value2) \
  do {                                   \
    s7_pointer _x_, _slot_, _sym1_, _val1_, _sym2_, _val2_;		\
    _sym1_ = Symbol1; _val1_ = Value1;					\
    _sym2_ = Symbol2; _val2_ = Value2;					\
    new_cell(Sc, _x_, T_LET);				                \
    let_id(_x_) = ++sc->let_number;					\
    set_outlet(_x_, Old_Env);				                \
    New_Env = _x_;							\
    new_cell_no_check(Sc, _slot_, T_SLOT);		                \
    slot_set_symbol(_slot_, _sym1_);					\
    slot_set_value(_slot_, _val1_);					\
    symbol_set_local(_sym1_, sc->let_number, _slot_);			\
    let_set_slots(_x_, _slot_);			                        \
    new_cell_no_check(Sc, _x_, T_SLOT);			                \
    slot_set_symbol(_x_, _sym2_);					\
    slot_set_value(_x_, _val2_);					\
    symbol_set_local(_sym2_, sc->let_number, _x_);			\
    set_next_slot(_x_, sc->nil);				        \
    set_next_slot(_slot_, _x_);			                        \
  } while (0)


static s7_pointer old_frame_in_env(s7_scheme *sc, s7_pointer frame, s7_pointer next_frame)
{
  set_type(frame, T_LET);
  let_set_slots(frame, sc->nil);
  set_outlet(frame, next_frame);
  let_id(frame) = ++sc->let_number;
  return(frame);
}


static s7_pointer old_frame_with_slot(s7_scheme *sc, s7_pointer env, s7_pointer val)
{
  s7_pointer x, sym;
  unsigned long long int id;

  id = ++sc->let_number;
  let_id(env) = id;
  x = let_slots(env);
  slot_set_value(x, val);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);

  return(env);
}


static s7_pointer old_frame_with_two_slots(s7_scheme *sc, s7_pointer env, s7_pointer val1, s7_pointer val2)
{
  s7_pointer x, sym;
  unsigned long long int id;

  id = ++sc->let_number;
  let_id(env) = id;
  x = let_slots(env);
  slot_set_value(x, val1);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);
  x = next_slot(x);
  slot_set_value(x, val2);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);

  return(env);
}


static s7_pointer old_frame_with_three_slots(s7_scheme *sc, s7_pointer env, s7_pointer val1, s7_pointer val2, s7_pointer val3)
{
  s7_pointer x, sym;
  unsigned long long int id;

  id = ++sc->let_number;
  let_id(env) = id;
  x = let_slots(env);

  slot_set_value(x, val1);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);
  x = next_slot(x);

  slot_set_value(x, val2);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);
  x = next_slot(x);

  slot_set_value(x, val3);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);

  return(env);
}


static s7_pointer permanent_slot(s7_pointer symbol, s7_pointer value)
{
  s7_pointer x;
  x = alloc_pointer();
  unheap(x);
  set_type(x, T_SLOT);
  slot_set_symbol(x, symbol);
  slot_set_value(x, value);
  return(x);
}


static s7_pointer find_let(s7_scheme *sc, s7_pointer obj)
{
  if (is_let(obj)) return(obj);
  switch (type(obj))
    {
    case T_LET:
      return(obj);

    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
    case T_CLOSURE: case T_CLOSURE_STAR:
      return(closure_let(obj));

    case T_C_OBJECT:
      return(c_object_let(obj));
    }
  return(sc->nil);
}


static s7_pointer free_let(s7_scheme *sc, s7_pointer e)
{
  s7_pointer p;
#if DEBUGGING
  for (p = let_slots(e); is_slot(p);)
    {
      s7_pointer n;
      n = next_slot(p); /* grab it before we free p, or the type check stuff will complain */
      free_cell(sc, p);
      p = n;
    }
#else
  for (p = let_slots(e); is_slot(p); p = next_slot(p))
    free_cell(sc, p);
#endif
  free_cell(sc, e);
  return(sc->nil);
}


static s7_pointer find_method(s7_scheme *sc, s7_pointer env, s7_pointer symbol)
{
  s7_pointer x;
  if (symbol_id(symbol) == 0) /* this means the symbol has never been used locally, so how can it be a method? */
    return(sc->undefined);

  /* I think the symbol_id is in sync with let_id, so the standard search should work */
  if (let_id(env) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));

  for (x = env; symbol_id(symbol) < let_id(x); x = outlet(x));

  if (let_id(x) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));

  for (; is_let(x); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(slot_value(y));
    }
  return(sc->undefined);
}


static int let_length(s7_scheme *sc, s7_pointer e)
{
  /* used by length, applicable_length, and some length optimizations */
  int i;
  s7_pointer p;

  if (e == sc->rootlet)
    return(sc->rootlet_entries);

  if (has_methods(e))
    {
      s7_pointer length_func;
      length_func = find_method(sc, e, sc->length_symbol);
      if (length_func != sc->undefined)
	{
	  p = s7_apply_function(sc, length_func, list_1(sc, e));
	  if (s7_is_integer(p))
	    return((int)s7_integer(p));
	  return(-1); /* ?? */
	}
    }

  for (i = 0, p = let_slots(e); is_slot(p); i++, p = next_slot(p));
  return(i);
}


static s7_pointer make_slot_1(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  /* env is not rootlet and is a let */
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol(slot, symbol);
  slot_set_value(slot, value);
  set_next_slot(slot, let_slots(env));
  let_set_slots(env, slot);
  set_local(symbol);
  /* this is called by varlet so we have to be careful about the resultant let_id
   *   check for greater to ensure shadowing stays in effect, and equal to do updates (set! in effect)
   */
  if (let_id(env) >= symbol_id(symbol))
    symbol_set_local(symbol, let_id(env), slot);
  return(slot);
}


s7_pointer s7_make_slot(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  if ((!is_let(env)) ||
      (env == sc->rootlet))
    {
      s7_pointer ge, slot;

      if ((sc->safety == 0) && (has_closure_let(value)))
	{
	  s7_remove_from_heap(sc, closure_args(value));
	  s7_remove_from_heap(sc, closure_body(value));
	}

      /* first look for existing slot -- this is not always checked before calling s7_make_slot */
      if (is_slot(global_slot(symbol)))
	{
	  slot = global_slot(symbol);
	  slot_set_value(slot, value);
	  return(slot);
	}

      ge = sc->rootlet;
      slot = permanent_slot(symbol, value);
      vector_element(ge, sc->rootlet_entries++) = slot;
      if (sc->rootlet_entries >= vector_length(ge))
	{
	  int i;
	  vector_length(ge) *= 2;
	  vector_elements(ge) = (s7_pointer *)realloc(vector_elements(ge), vector_length(ge) * sizeof(s7_pointer));
	  for (i = sc->rootlet_entries; i < vector_length(ge); i++)
	    vector_element(ge, i) = sc->nil;
	}
      set_global_slot(symbol, slot);
      
      if (symbol_id(symbol) == 0) /* never defined locally? */
	{
	  if (initial_slot(symbol) == sc->undefined)
	    set_initial_slot(symbol, permanent_slot(symbol, value));
	  set_local_slot(symbol, slot);
	  set_global(symbol);
	}
      if (is_gensym(symbol))
	s7_remove_from_heap(sc, symbol);
      return(slot);
    }

  return(make_slot_1(sc, env, symbol, value));
  /* there are about the same number of frames as local variables -- this
   *   strikes me as surprising, but it holds up across a lot of code.
   */
}


static s7_pointer make_slot(s7_scheme *sc, s7_pointer variable, s7_pointer value)
{
  /* this is for a do-loop optimization -- an unattached slot */
  s7_pointer y;
  new_cell(sc, y, T_SLOT);
  slot_set_symbol(y, variable);
  if (!is_symbol(variable)) abort();
  slot_set_value(y, value);
  return(y);
}


/* -------------------------------- let? -------------------------------- */
bool s7_is_let(s7_pointer e)
{
  return(is_let(e));
}

static s7_pointer g_is_let(s7_scheme *sc, s7_pointer args)
{
  #define H_is_let "(let? obj) returns #t if obj is a let (an environment)."
  #define Q_is_let pl_bt

  check_boolean_method(sc, is_let, sc->is_let_symbol, args);
}


/* -------------------------------- unlet -------------------------------- */
#define UNLET_ENTRIES 410 /* 401 if not --disable-deprecated etc */

static void save_unlet(s7_scheme *sc)
{
  int i, k = 0;
  s7_pointer x;
  s7_pointer *inits;

  sc->unlet = (s7_pointer)calloc(1, sizeof(s7_cell));
  set_type(sc->unlet, T_VECTOR);
  vector_length(sc->unlet) = UNLET_ENTRIES;
  vector_elements(sc->unlet) = (s7_pointer *)malloc(UNLET_ENTRIES * sizeof(s7_pointer));
  vector_getter(sc->unlet) = default_vector_getter;
  vector_setter(sc->unlet) = default_vector_setter;
  inits = vector_elements(sc->unlet);
  s7_vector_fill(sc, sc->unlet, sc->nil);
  unheap(sc->unlet);

  for (i = 0; i < vector_length(sc->symbol_table); i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      {
	s7_pointer sym;
	sym = car(x);
	if (is_slot(initial_slot(sym)))
	  {
	    s7_pointer val;
	    val = slot_value(initial_slot(sym));
	    if ((is_procedure(val)) || (is_syntax(val)))
	      inits[k++] = initial_slot(sym);

	    /* (let ((begin +)) (with-let (unlet) (begin 1 2))) */
#if DEBUGGING
	    if (k >= UNLET_ENTRIES)
	      fprintf(stderr, "unlet overflow\n");
#endif
	  }
      }
}

static s7_pointer g_unlet(s7_scheme *sc, s7_pointer args)
{
  /* add sc->unlet bindings to the current environment */
  #define H_unlet "(unlet) establishes the original bindings of all the predefined functions"
  #define Q_unlet s7_make_signature(sc, 1, sc->is_let_symbol)

  /* slightly confusing:
   *    :((unlet) 'abs)
   *    #<undefined>
   *    :(defined? 'abs (unlet))
   *    #t
   * this is because unlet sets up a local environment of unshadowed symbols,
   *   and s7_let_ref below only looks at the local env chain (that is, if env is not
   *   the global env, then the global env is not searched).
   *
   * Also (define hi 3) #_hi => 3, (set! hi 4), #_hi -> 3 but (with-let (unlet) hi) -> 4!
   */
  int i;
  s7_pointer *inits;
  s7_pointer x;

  sc->w = new_frame_in_env(sc, sc->envir);
  inits = vector_elements(sc->unlet);

  for (i = 0; (i < UNLET_ENTRIES) && (is_slot(inits[i])); i++)
    {
      s7_pointer sym;
      x = slot_value(inits[i]);
      sym = slot_symbol(inits[i]);
      if (is_procedure(x))
	{
	  if (((!is_global(sym)) &&                  /* it might be shadowed locally */
	       (s7_symbol_local_value(sc, sym, sc->envir) != slot_value(global_slot(sym)))) ||
	      (x != slot_value(global_slot(sym))))   /* it's not shadowed, but has been changed globally */
	    make_slot_1(sc, sc->w, sym, x);
	}
      else
	{
	  if ((is_syntax(x)) &&
	      (local_slot(sym) != sc->nil))          /* this can be a freed cell, will be nil if unchanged */
	    make_slot_1(sc, sc->w, sym, x);
	}
    }
  /* if (set! + -) then + needs to be overridden, but the local bit isn't set,
   *   so we have to check the actual values in the non-local case.
   *   (define (f x) (with-let (unlet) (+ x 1)))
   */

  x = sc->w;
  sc->w = sc->nil;
  return(x);
}


/* -------------------------------- openlet? -------------------------------- */
bool s7_is_openlet(s7_pointer e)
{
  return(has_methods(e));
}

static s7_pointer g_is_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_is_openlet "(openlet? obj) returns #t is 'obj' has methods."
  #define Q_is_openlet pl_bt

  /* if car(args) is not a let (or possibly have one), should this raise an error? */
  check_method(sc, car(args), sc->is_openlet_symbol, args);
  return(make_boolean(sc, has_methods(car(args))));
}


/* -------------------------------- openlet -------------------------------- */
s7_pointer s7_openlet(s7_scheme *sc, s7_pointer e)
{
  set_has_methods(e);
  return(e);
}

static s7_pointer g_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_openlet "(openlet e) tells the built-in generic functions that the environment 'e might have an over-riding method."
  #define Q_openlet pcl_e
  s7_pointer e;

  e = car(args);
  check_method(sc, e, sc->openlet_symbol, args);
  if (((is_let(e)) && (e != sc->rootlet)) ||
      (has_closure_let(e)) ||
      ((is_c_object(e)) && (c_object_let(e) != sc->nil)))
    {
      set_has_methods(e);
      return(e);
    }
  return(simple_wrong_type_argument_with_type(sc, sc->openlet_symbol, e, a_let_string));
}


/* -------------------------------- coverlet -------------------------------- */
static s7_pointer c_coverlet(s7_scheme *sc, s7_pointer e)
{
  sc->temp3 = e;
  check_method(sc, e, sc->coverlet_symbol, list_1(sc, e));
  if (((is_let(e)) && (e != sc->rootlet)) ||
      (has_closure_let(e)) ||
      ((is_c_object(e)) && (c_object_let(e) != sc->nil)))
    {
      clear_has_methods(e);
      return(e);
    }
  return(simple_wrong_type_argument_with_type(sc, sc->coverlet_symbol, e, a_let_string));
}

static s7_pointer g_coverlet(s7_scheme *sc, s7_pointer args)
{
  #define H_coverlet "(coverlet e) undoes an earlier openlet."
  #define Q_coverlet pcl_e
  return(c_coverlet(sc, car(args)));
}


/* -------------------------------- varlet -------------------------------- */
static void append_let(s7_scheme *sc, s7_pointer new_e, s7_pointer old_e)
{
  s7_pointer x;

  if (old_e == sc->rootlet)
    return;

  if (new_e != sc->rootlet)
    {
      for (x = let_slots(old_e); is_slot(x); x = next_slot(x))
	make_slot_1(sc, new_e, slot_symbol(x), slot_value(x)); /* not add_slot here because we might run off the free heap end */
    }
  else
    {
      for (x = let_slots(old_e); is_slot(x); x = next_slot(x))
	{
	  s7_pointer sym, val;
	  sym = slot_symbol(x);
	  val = slot_value(x);
	  if (is_slot(global_slot(sym)))
	    slot_set_value(global_slot(sym), val);
	  else s7_make_slot(sc, new_e, sym, val);
	}
    }
}

static s7_pointer check_c_obj_env(s7_scheme *sc, s7_pointer old_e, s7_pointer caller)
{
  if (is_c_object(old_e))
    old_e = c_object_let(old_e);
  if (!is_let(old_e))
    return(simple_wrong_type_argument_with_type(sc, caller, old_e, a_let_string));
  return(old_e);
}


s7_pointer s7_varlet(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(env))
    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 1, env, a_let_string));

  if (!is_symbol(symbol))
    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 2, symbol, a_symbol_string));

  if (env == sc->rootlet)
    {
      if (is_slot(global_slot(symbol)))
	{
	  if (is_syntax(slot_value(global_slot(symbol))))
		return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 2, symbol, make_string_wrapper(sc, "a non-syntactic keyword")));
	  slot_set_value(global_slot(symbol), value);
	}
      else s7_make_slot(sc, env, symbol, value);
    }
  else make_slot_1(sc, env, symbol, value);
  return(value);
}


static s7_pointer g_varlet(s7_scheme *sc, s7_pointer args)
{
  #define H_varlet "(varlet env ...) adds its arguments (an environment, a cons: symbol . value, or a pair of arguments, the symbol and its value) \
to the environment env, and returns the environment."
  #define Q_varlet s7_make_circular_signature(sc, 2, 3, sc->is_let_symbol, sc->is_let_symbol, sc->T)
  /* varlet = with-let + define */

  s7_pointer x, e, sym, val, p;

  e = car(args);
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->varlet_symbol, args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 1, e, a_let_string));
    }

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      p = car(x);
      switch (type(p))
	{
	case T_SYMBOL:
	  if (is_keyword(p))
	    sym = keyword_symbol(p);
	  else sym = p;
	  if (!is_pair(cdr(x)))
	    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, a_binding_string));
	  x = cdr(x);
	  val = car(x);
	  break;

	case T_PAIR:
	  sym = car(p);
	  if (!is_symbol(sym))
	    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, a_symbol_string));
	  val = cdr(p);
	  break;

	case T_LET:
	  append_let(sc, e, check_c_obj_env(sc, p, sc->varlet_symbol));
	  continue;

	default:
	  return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, a_symbol_string));
	}

      if (is_immutable_symbol(sym))
	return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), sym, a_non_constant_symbol_string));

      if (e == sc->rootlet)
	{
	  if (is_slot(global_slot(sym)))
	    {
	      if (is_syntax(slot_value(global_slot(sym))))
		return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, make_string_wrapper(sc, "a non-syntactic keyword")));
	      /*  without this check we can end up turning our code into gibberish:
	       *   :(set! quote 1)
	       *   ;can't set! quote
	       *   :(varlet (rootlet) '(quote . 1))
	       *   :quote
	       *   1
	       * or worse set quote to a function of one arg that tries to quote something -- infinite loop
	       */
	      slot_set_value(global_slot(sym), val);
	    }
	  else s7_make_slot(sc, e, sym, val);
	}
      else make_slot_1(sc, e, sym, val);
      /* this used to check for sym already defined, and set its value, but that greatly slows down
       *   the most common use (adding a slot), and makes it hard to shadow explicitly.  Don't use
       *   varlet as a substitute for set!/let-set!.
       */
    }
  return(e);
}


/* -------------------------------- cutlet -------------------------------- */
static s7_pointer g_cutlet(s7_scheme *sc, s7_pointer args)
{
  #define H_cutlet "(cutlet e symbol ...) removes symbols from the environment e."
  #define Q_cutlet s7_make_circular_signature(sc, 2, 3, sc->is_let_symbol, sc->is_let_symbol, sc->is_symbol_symbol)

  s7_pointer e, syms;
  #define THE_UN_ID ++sc->let_number

  e = car(args);
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->cutlet_symbol, args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->cutlet_symbol, 1, e, a_let_string));
    }
  /* besides removing the slot we have to make sure the symbol_id does not match else
   *   let-ref and others will use the old slot!  What's the un-id?  Perhaps the next one?
   *   (let ((b 1)) (let ((b 2)) (cutlet (curlet) 'b)) b)
   */
  for (syms = cdr(args); is_pair(syms); syms = cdr(syms))
    {
      s7_pointer sym, slot;
      sym = car(syms);

      if (!is_symbol(sym))
	return(wrong_type_argument_with_type(sc, sc->cutlet_symbol, position_of(syms, args), sym, a_symbol_string));

      if (is_keyword(sym))
	sym = keyword_symbol(sym);

      if (e == sc->rootlet)
	{
	  if (is_slot(global_slot(sym)))
	    {
	      symbol_set_id(sym, THE_UN_ID);
	      slot_set_value(global_slot(sym), sc->undefined);
	    }
	}
      else
	{
	  slot = let_slots(e);
	  if (is_slot(slot))
	    {
	      if (slot_symbol(slot) == sym)
		{
		  let_set_slots(e, next_slot(let_slots(e)));
		  symbol_set_id(sym, THE_UN_ID);
		}
	      else
		{
		  s7_pointer last_slot;
		  last_slot = slot;
		  for (slot = next_slot(let_slots(e)); is_slot(slot); last_slot = slot, slot = next_slot(slot))
		    {
		      if (slot_symbol(slot) == sym)
			{
			  symbol_set_id(sym, THE_UN_ID);
			  set_next_slot(last_slot, next_slot(slot));
			  break; 
			}
		    }
		}
	    }
	}
    }
  return(e);
}


/* -------------------------------- sublet -------------------------------- */
static s7_pointer sublet_1(s7_scheme *sc, s7_pointer e, s7_pointer bindings, s7_pointer caller)
{
  s7_pointer new_e;

  if (e == sc->rootlet)
    new_e = new_frame_in_env(sc, sc->nil);
  else new_e = new_frame_in_env(sc, e);
  set_all_methods(new_e, e);

  if (!is_null(bindings))
    {
      s7_pointer x;
      sc->temp3 = new_e;

      for (x = bindings; is_not_null(x); x = cdr(x))
	{
	  s7_pointer p, sym, val;

	  p = car(x);
	  switch (type(p))
	    {
	    case T_SYMBOL:
	      if (is_keyword(p))
		sym = keyword_symbol(p);
	      else sym = p;
	      if (!is_pair(cdr(x)))
		return(wrong_type_argument_with_type(sc, caller, position_of(x, bindings), p, a_binding_string));
	      x = cdr(x);
	      val = car(x);
	      break;

	    case T_PAIR:
	      sym = car(p);
	      if (!is_symbol(sym))
		return(wrong_type_argument_with_type(sc, caller, position_of(x, bindings), p, a_symbol_string));
	      val = cdr(p);
	      break;

	    case T_LET:
	      append_let(sc, new_e, check_c_obj_env(sc, p, caller));
	      continue;

	    default:
	      return(wrong_type_argument_with_type(sc, caller, position_of(x, bindings), p, a_symbol_string));
	    }

	  if (is_immutable_symbol(sym))
	    return(wrong_type_argument_with_type(sc, caller, position_of(x, bindings), sym, a_non_constant_symbol_string));

	  /* here we know new_e is a let and is not rootlet */
	  make_slot_1(sc, new_e, sym, val);
	  if (sym == sc->let_ref_fallback_symbol)
	    set_has_ref_fallback(new_e);
	  else
	    {
	      if (sym == sc->let_set_fallback_symbol)
		set_has_set_fallback(new_e);
	    }
	}
      sc->temp3 = sc->nil;
    }
  return(new_e);
}

s7_pointer s7_sublet(s7_scheme *sc, s7_pointer e, s7_pointer bindings)
{
  return(sublet_1(sc, e, bindings, sc->sublet_symbol));
}

static s7_pointer g_sublet(s7_scheme *sc, s7_pointer args)
{
  #define H_sublet "(sublet env ...) adds its \
arguments (each an environment or a cons: symbol . value) to the environment env, and returns the \
new environment."
  #define Q_sublet s7_make_circular_signature(sc, 2, 3, sc->is_let_symbol, s7_make_signature(sc, 2, sc->is_let_symbol, sc->is_null_symbol), sc->T)

  s7_pointer e;

  e = car(args);
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->sublet_symbol, args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->sublet_symbol, 1, e, a_let_string));
    }
  return(sublet_1(sc, e, cdr(args), sc->sublet_symbol));
}


/* -------------------------------- inlet -------------------------------- */
s7_pointer s7_inlet(s7_scheme *sc, s7_pointer args)
{
  #define H_inlet "(inlet ...) adds its \
arguments, each an environment, a cons: '(symbol . value), or a keyword/value pair, to a new environment, and returns the \
new environment. (inlet :a 1 :b 2) or (inlet '(a . 1) '(b . 2))"
  #define Q_inlet s7_make_circular_signature(sc, 1, 2, sc->is_let_symbol, sc->T)

  return(sublet_1(sc, sc->rootlet, args, sc->inlet_symbol));
}

#define g_inlet s7_inlet


/* -------------------------------- let->list -------------------------------- */
s7_pointer s7_let_to_list(s7_scheme *sc, s7_pointer env)
{
  s7_pointer x;

  sc->temp3 = sc->w;
  sc->w = sc->nil;

  if (env == sc->rootlet)
    {
      unsigned int i, lim2;
      s7_pointer *entries;

      entries = vector_elements(env);
      lim2 = sc->rootlet_entries;
      if (lim2 & 1) lim2--;

      for (i = 0; i < lim2; )
	{
	  sc->w = cons_unchecked(sc, cons(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w); i++;
	  sc->w = cons_unchecked(sc, cons_unchecked(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w); i++;
	}
      if (lim2 < sc->rootlet_entries)
	sc->w = cons_unchecked(sc, cons(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w);
    }
  else
    {
      s7_pointer iter, func;
      /* need to check make-iterator method before dropping into let->list */

      if ((has_methods(env)) && ((func = find_method(sc, env, sc->make_iterator_symbol)) != sc->undefined))
	iter = s7_apply_function(sc, func, list_1(sc, env));
      else iter = sc->nil;

      if (is_null(iter))
	{
	  for (x = let_slots(env); is_slot(x); x = next_slot(x))
	    sc->w = cons_unchecked(sc, cons(sc, slot_symbol(x), slot_value(x)), sc->w);
	}
      else
	{
	  /* (begin (load "mockery.scm") (let ((lt ((*mock-pair* 'mock-pair) 1 2 3))) (format *stderr* "~{~A ~}" lt))) */
	  while (true)
	    {
	      x = s7_iterate(sc, iter);
	      if (iterator_is_at_end(iter)) break;
	      sc->w = cons(sc, x, sc->w);
	    }
	  sc->w = safe_reverse_in_place(sc, sc->w);
	}
    }
  x = sc->w;
  sc->w = sc->temp3;
  sc->temp3 = sc->nil;
  return(x);
}

#if (!WITH_PURE_S7)
static s7_pointer g_let_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_let_to_list "(let->list env) returns env's bindings as a list of cons's: '(symbol . value)."
  #define Q_let_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_let_symbol)

  s7_pointer env;
  env = car(args);
  check_method(sc, env, sc->let_to_list_symbol, args);
  if (!is_let(env))
    {
      if (is_c_object(env))
	env = c_object_let(env);
      if (!is_let(env))
        return(simple_wrong_type_argument_with_type(sc, sc->let_to_list_symbol, env, a_let_string));
    }
  return(s7_let_to_list(sc, env));
}
#endif


/* -------------------------------- let-ref -------------------------------- */
static s7_pointer let_ref_1(s7_scheme *sc, s7_pointer env, s7_pointer symbol)
{
  s7_pointer x, y;
  /* (let ((a 1)) ((curlet) 'a))
   * ((rootlet) 'abs)
   */
  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);

  if (env == sc->rootlet)
    {
      y = global_slot(symbol);
      if (is_slot(y))
	return(slot_value(y));
      return(sc->undefined);
    }

  if (let_id(env) == symbol_id(symbol))
    return(slot_value(local_slot(symbol))); /* this obviously has to follow the global-env check */

  for (x = env; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(slot_value(y));

  /* now for a horrible kludge.  If a let is a mock-hash-table (for example), implicit
   *   indexing of the hash-table collides with the same thing for the let (field names
   *   versus keys), and we can't just try again here because that makes it too easy to
   *   get into infinite recursion.  So, 'let-ref-fallback...
   */
  if (has_ref_fallback(env))
    check_method(sc, env, sc->let_ref_fallback_symbol, sc->w = list_2(sc, env, symbol));

  /* why did this ignore a global value? Changed 24-May-16 to check rootlet if no methods --
   *   apparently I was using #<undefined> here (pre-rootlet-check) to indicate that an
   *   open let did not have a particular method (locally).  This seems inconsistent now,
   *   but it was far worse before.  At least (let () ((curlet) 'pi)) is pi!
   */
  if (!has_methods(env))
    {
      y = global_slot(symbol);
      if (is_slot(y))
	return(slot_value(y));
    }

  return(sc->undefined);
}

s7_pointer s7_let_ref(s7_scheme *sc, s7_pointer env, s7_pointer symbol)
{
  if (!is_let(env))
    return(wrong_type_argument_with_type(sc, sc->let_ref_symbol, 1, env, a_let_string));
    
  if (!is_symbol(symbol))
    {
      check_method(sc, env, sc->let_ref_symbol, sc->w = list_2(sc, env, symbol));
      if (has_ref_fallback(env))
	check_method(sc, env, sc->let_ref_fallback_symbol, sc->w = list_2(sc, env, symbol));
      return(wrong_type_argument_with_type(sc, sc->let_ref_symbol, 2, symbol, a_symbol_string));
    }
  return(let_ref_1(sc, env, symbol));
}

static s7_pointer g_let_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_let_ref "(let-ref env sym) returns the value of the symbol sym in the environment env"
  #define Q_let_ref s7_make_signature(sc, 3, sc->T, sc->is_let_symbol, sc->is_symbol_symbol)
  s7_pointer e, s;

  e = car(args);
  if (!is_let(e))
    return(wrong_type_argument_with_type(sc, sc->let_ref_symbol, 1, e, a_let_string));

  s = cadr(args);
  if (!is_symbol(s))
    {
      check_method(sc, e, sc->let_ref_symbol, args);
      if (has_ref_fallback(e))
	check_method(sc, e, sc->let_ref_fallback_symbol, args);
      return(wrong_type_argument_with_type(sc, sc->let_ref_symbol, 2, s, a_symbol_string));
    }
  return(let_ref_1(sc, e, s));
}


/* -------------------------------- let-set! -------------------------------- */
static s7_pointer call_accessor(s7_scheme *sc, s7_pointer slot, s7_pointer old_value)
{
  s7_pointer func, new_value;

  /* new_value = sc->error_symbol; */
  func = slot_accessor(slot);

  if (is_procedure_or_macro(func))
    {
      if (is_c_function(func))
	{
	  set_car(sc->t2_1, slot_symbol(slot));
	  set_car(sc->t2_2, old_value);
      	  new_value = c_function_call(func)(sc, sc->t2_1);
	}
      else
	{
	  bool old_off;
	  old_off = sc->gc_off;
	  sc->gc_off = true;
	  new_value = s7_apply_function(sc, func, list_2(sc, slot_symbol(slot), old_value));
	  sc->gc_off = old_off;
	}
    }
  else return(old_value);

  if (new_value == sc->error_symbol)
    return(s7_error(sc, sc->error_symbol, set_elist_3(sc, make_string_wrapper(sc, "can't set! ~S to ~S"), slot_symbol(slot), old_value)));
  return(new_value);
}

static s7_pointer let_set_1(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x, y;

  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);

  if (env == sc->rootlet)
    {
      if (is_immutable_symbol(symbol))  /* (let-set! (rootlet) :rest #f) */
	return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, a_non_constant_symbol_string));
      y = global_slot(symbol);
      if (is_slot(y))
	{
	  if (slot_has_accessor(y))
	    slot_set_value(y, call_accessor(sc, y, value));
	  else slot_set_value(y, value);
	  return(slot_value(y));
	}
      return(sc->undefined);
    }

  for (x = env; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	{
	  if (slot_has_accessor(y))
	    slot_set_value(y, call_accessor(sc, y, value));
	  else slot_set_value(y, value);
	  return(slot_value(y));
	}

  if (has_set_fallback(env))
    check_method(sc, env, sc->let_set_fallback_symbol, sc->w = list_3(sc, env, symbol, value));

  if (!has_methods(env))
    {
      y = global_slot(symbol);
      if (is_slot(y))
	{
	  if (slot_has_accessor(y))
	    slot_set_value(y, call_accessor(sc, y, value));
	  else slot_set_value(y, value);
	  return(slot_value(y));
	}
    }
  return(sc->undefined);
}

s7_pointer s7_let_set(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(env))
    return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 1, env, a_let_string));

  if (!is_symbol(symbol))
    {
      check_method(sc, env, sc->let_set_symbol, sc->w = list_3(sc, env, symbol, value));
      if (has_set_fallback(env))
	check_method(sc, env, sc->let_set_fallback_symbol, sc->w = list_3(sc, env, symbol, value));
      return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, a_symbol_string));
    }

  return(let_set_1(sc, env, symbol, value));
}

static s7_pointer g_let_set(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (set! ((curlet) 'a) 32) a) */
  #define H_let_set "(let-set! env sym val) sets the symbol sym's value in the environment env to val"
  #define Q_let_set s7_make_signature(sc, 4, sc->T, sc->is_let_symbol, sc->is_symbol_symbol, sc->T)

  return(s7_let_set(sc, car(args), cadr(args), caddr(args)));
}


static s7_pointer reverse_slots(s7_scheme *sc, s7_pointer list)
{
  s7_pointer p = list, result, q;
  result = sc->nil;

  while (is_slot(p))
    {
      q = next_slot(p);
      set_next_slot(p, result);
      result = p;
      p = q;
    }
  return(result);
}


static s7_pointer let_copy(s7_scheme *sc, s7_pointer env)
{
  if (is_let(env))
    {
      s7_pointer new_e;

      if (env == sc->rootlet)   /* (copy (rootlet)) or (copy (funclet abs)) etc */
	return(sc->rootlet);

      /* we can't make copy handle environments-as-objects specially because the
       *   make-object function in define-class uses copy to make a new object!
       *   So if it is present, we get it here, and then there's almost surely trouble.
       */
      new_e = new_frame_in_env(sc, outlet(env));
      set_all_methods(new_e, env);
      sc->temp3 = new_e;
      if (is_slot(let_slots(env)))
	{
	  s7_int id;
	  s7_pointer x, y = NULL;

	  id = let_id(new_e);
	  for (x = let_slots(env); is_slot(x); x = next_slot(x))
	    {
	      s7_pointer z;
	      new_cell(sc, z, T_SLOT);
	      slot_set_symbol(z, slot_symbol(x));
	      slot_set_value(z, slot_value(x));
	      if (symbol_id(slot_symbol(z)) != id) /* keep shadowing intact */
		symbol_set_local(slot_symbol(x), id, z);
	      if (is_slot(let_slots(new_e)))
		set_next_slot(y, z);
	      else let_set_slots(new_e, z);
	      set_next_slot(z, sc->nil);              /* in case GC runs during this loop */
	      y = z;
	    }
	}
      /* We can't do a (normal) loop here then reverse the slots later because the symbol's local_slot has to
       *    match the unshadowed slot, not the last in the list:
       *    (let ((e1 (inlet 'a 1 'a 2))) (let ((e2 (copy e1))) (list (equal? e1 e2) (equal? (e1 'a) (e2 'a)))))
       */
      sc->temp3 = sc->nil;
      return(new_e);
    }
  return(sc->nil);
}


/* -------------------------------- rootlet -------------------------------- */
static s7_pointer g_rootlet(s7_scheme *sc, s7_pointer ignore)
{
  #define H_rootlet "(rootlet) returns the current top-level definitions (symbol bindings)."
  #define Q_rootlet s7_make_signature(sc, 1, sc->is_let_symbol)
  return(sc->rootlet);
}
/* as with the symbol-table, this function can lead to disaster -- user could
 *   clobber the environment etc.  But we want it to be editable and augmentable,
 *   so I guess I'll leave it alone.  (See curlet|funclet as well).
 */

s7_pointer s7_rootlet(s7_scheme *sc)
{
  return(sc->rootlet);
}

s7_pointer s7_shadow_rootlet(s7_scheme *sc)
{
  return(sc->shadow_rootlet);
}

s7_pointer s7_set_shadow_rootlet(s7_scheme *sc, s7_pointer let)
{
  sc->shadow_rootlet = let;
  return(let);
}


/* -------------------------------- curlet -------------------------------- */
static s7_pointer g_curlet(s7_scheme *sc, s7_pointer args)
{
  #define H_curlet "(curlet) returns the current definitions (symbol bindings)"
  #define Q_curlet s7_make_signature(sc, 1, sc->is_let_symbol)

  sc->capture_let_counter++;
  if (is_let(sc->envir))
    return(sc->envir);
  return(sc->rootlet);
}

s7_pointer s7_curlet(s7_scheme *sc)
{
  sc->capture_let_counter++;
  return(sc->envir);
}

s7_pointer s7_set_curlet(s7_scheme *sc, s7_pointer e)
{
  s7_pointer p, old_e;
  old_e = sc->envir;
  sc->envir = e;

  if ((is_let(e)) && (let_id(e) > 0)) /* might be () [id=-1] or rootlet [id=0] etc */
    {
      let_id(e) = ++sc->let_number;
      for (p = let_slots(e); is_slot(p); p = next_slot(p))
	{
	  s7_pointer sym;
	  sym = slot_symbol(p);
	  if (symbol_id(sym) != sc->let_number)
	    symbol_set_local(sym, sc->let_number, p);
	}
    }

  return(old_e);
}


/* -------------------------------- outlet -------------------------------- */
s7_pointer s7_outlet(s7_scheme *sc, s7_pointer e)
{
  return(outlet(e));
}

static s7_pointer g_outlet(s7_scheme *sc, s7_pointer args)
{
  #define H_outlet "(outlet env) is the environment that contains env."
  #define Q_outlet s7_make_signature(sc, 2, sc->is_let_symbol, sc->is_let_symbol)

  s7_pointer env;
  env = car(args);
  if (!is_let(env))
    method_or_bust_with_type(sc, env, sc->outlet_symbol, args, a_let_string, 0);

  if ((env == sc->rootlet) ||
      (is_null(outlet(env))))
    return(sc->rootlet);
  return(outlet(env));
}

static s7_pointer g_set_outlet(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (let ((b 2)) (set! (outlet (curlet)) (rootlet)) ((curlet) 'a))) */
  s7_pointer env, new_outer;

  env = car(args);
  if (!is_let(env))
    return(s7_wrong_type_arg_error(sc, "set! outlet", 1, env, "a let"));

  new_outer = cadr(args);
  if (!is_let(new_outer))
    return(s7_wrong_type_arg_error(sc, "set! outlet", 2, new_outer, "a let"));
  if (new_outer == sc->rootlet)
    new_outer = sc->nil;

  if (env != sc->rootlet)
    set_outlet(env, new_outer);
  return(new_outer);
}



static s7_pointer find_symbol(s7_scheme *sc, s7_pointer symbol)
{
  s7_pointer x;

  if (let_id(sc->envir) == symbol_id(symbol))
    return(local_slot(symbol));

  for (x = sc->envir; symbol_id(symbol) < let_id(x); x = outlet(x));

  if (let_id(x) == symbol_id(symbol))
    return(local_slot(symbol));

  for (; is_let(x); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(y);
    }

  return(global_slot(symbol));
}

#if WITH_GCC && DEBUGGING
static s7_pointer find_symbol_unchecked_1(s7_scheme *sc, s7_pointer symbol)
#else
static s7_pointer find_symbol_unchecked(s7_scheme *sc, s7_pointer symbol) /* find_symbol_checked includes the unbound_variable call */
#endif
{
  s7_pointer x;
  /* fprintf(stderr, "let_id: %lld, %s id: %lld\n", let_id(sc->envir), DISPLAY(symbol), symbol_id(symbol)); */

  if (let_id(sc->envir) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));

  for (x = sc->envir; symbol_id(symbol) < let_id(x); x = outlet(x));

  /* this looks redundant, but every attempt to improve it is much slower! */
  if (let_id(x) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));

  for (; is_let(x); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(slot_value(y));
    }

  x = global_slot(symbol);
  if (is_slot(x))
    return(slot_value(x));

#if WITH_GCC
  return(NULL);
#else
  return(unbound_variable(sc, symbol));
#endif
}


s7_pointer s7_slot(s7_scheme *sc, s7_pointer symbol)
{
  return(find_symbol(sc, symbol));
}


s7_pointer s7_slot_value(s7_pointer slot)
{
  return(slot_value(slot));
}


s7_pointer s7_slot_set_value(s7_scheme *sc, s7_pointer slot, s7_pointer value)
{
  slot_set_value(slot, value);
  return(value);
}


void s7_slot_set_real_value(s7_scheme *sc, s7_pointer slot, s7_double value)
{
  set_real(slot_value(slot), value);
}


s7_double s7_slot_real_value(s7_scheme *sc, s7_pointer slot, const char *caller)
{
  return(real_to_double(sc, slot_value(slot), caller));
}

s7_int s7_slot_integer_value(s7_pointer slot)
{
  return(integer(slot_value(slot)));
}


static s7_pointer find_local_symbol(s7_scheme *sc, s7_pointer symbol, s7_pointer e)
{
  if (!is_let(e))
    return(global_slot(symbol));

  if (symbol_id(symbol) != 0)
    {
      s7_pointer y;
      for (y = let_slots(e); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(y);
    }
  return(sc->undefined);
}


static s7_pointer s7_local_slot(s7_scheme *sc, s7_pointer symbol)
{
  s7_pointer y;
  for (y = let_slots(sc->envir); is_slot(y); y = next_slot(y))
    if (slot_symbol(y) == symbol)
      return(y);
  return(NULL);
}


s7_pointer s7_symbol_value(s7_scheme *sc, s7_pointer sym)
{
  s7_pointer x;

  x = find_symbol(sc, sym);
  if (is_slot(x))
    return(slot_value(x));

  return(sc->undefined);
}


s7_pointer s7_symbol_local_value(s7_scheme *sc, s7_pointer sym, s7_pointer local_env)
{
  if (is_let(local_env))
    {
      s7_pointer x;
      for (x = local_env; is_let(x); x = outlet(x))
	{
	  s7_pointer y;
	  for (y = let_slots(x); is_slot(y); y = next_slot(y))
	    if (slot_symbol(y) == sym)
	      return(slot_value(y));
	}
    }
  return(s7_symbol_value(sc, sym));
}


/* -------------------------------- symbol->value -------------------------------- */

#define find_global_symbol_checked(Sc, Sym) ((is_global(Sym)) ? slot_value(global_slot(Sym)) : find_symbol_checked(Sc, Sym))

static s7_pointer g_symbol_to_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_value "(symbol->value sym (env (curlet))) returns the binding of (the value associated with) the \
symbol sym in the given environment: (let ((x 32)) (symbol->value 'x)) -> 32"
  #define Q_symbol_to_value s7_make_signature(sc, 3, sc->T, sc->is_symbol_symbol, sc->is_let_symbol)
  /* (symbol->value 'x e) => (e 'x)? */

  s7_pointer sym;
  sym = car(args);

  if (!is_symbol(sym))
    method_or_bust(sc, sym, sc->symbol_to_value_symbol, args, T_SYMBOL, 1);

  if (is_not_null(cdr(args)))
    {
      s7_pointer local_env;

      local_env = cadr(args);
      if (local_env == sc->unlet_symbol)
	return((is_slot(initial_slot(sym))) ? slot_value(initial_slot(sym)) : sc->undefined);

      if (!is_let(local_env))
	method_or_bust_with_type(sc, local_env, sc->symbol_to_value_symbol, args, a_let_string, 2);

      if (local_env == sc->rootlet)
	{
	  s7_pointer x;
	  x = global_slot(sym);
	  if (is_slot(x))
	    return(slot_value(x));
	  return(sc->undefined);
	}
      return(s7_symbol_local_value(sc, sym, local_env));
    }

  if (is_global(sym))
    return(slot_value(global_slot(sym)));

  return(s7_symbol_value(sc, sym));
}


s7_pointer s7_symbol_set_value(s7_scheme *sc, s7_pointer sym, s7_pointer val)
{
  s7_pointer x;
  /* if immutable should this return an error? */
  x = find_symbol(sc, sym);
  if (is_slot(x))
    slot_set_value(x, val);
  return(val);
}


/* -------------------------------- symbol->dynamic-value -------------------------------- */

static s7_pointer find_dynamic_value(s7_scheme *sc, s7_pointer x, s7_pointer sym, long long int *id)
{
  for (; symbol_id(sym) < let_id(x); x = outlet(x));

  if (let_id(x) == symbol_id(sym))
    {
      (*id) = let_id(x);
      return(slot_value(local_slot(sym)));
    }
  for (; (is_let(x)) && (let_id(x) > (*id)); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == sym)
	  {
	    (*id) = let_id(x);
	    return(slot_value(y));
	  }
    }
  return(sc->gc_nil);
}


static s7_pointer g_symbol_to_dynamic_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_dynamic_value "(symbol->dynamic-value sym) returns the dynamic binding of the symbol sym"
  #define Q_symbol_to_dynamic_value s7_make_signature(sc, 2, sc->T, sc->is_symbol_symbol)

  s7_pointer sym, val;
  long long int top_id;
  int i;

  sym = car(args);
  if (!is_symbol(sym))
    method_or_bust(sc, sym, sc->symbol_to_dynamic_value_symbol, args, T_SYMBOL, 1);

  if (is_global(sym))
    return(slot_value(global_slot(sym)));

  if (let_id(sc->envir) == symbol_id(sym))
    return(slot_value(local_slot(sym)));

  top_id = -1;
  val = find_dynamic_value(sc, sc->envir, sym, &top_id);
  if (top_id == symbol_id(sym))
    return(val);

  for (i = s7_stack_top(sc) - 1; i > 0; i -= 4)
    {
      s7_pointer cur_val;
      cur_val = find_dynamic_value(sc, stack_let(sc->stack, i), sym, &top_id);
      if (cur_val != sc->gc_nil)
	val = cur_val;
      if (top_id == symbol_id(sym))
	return(val);
    }

  if (val == sc->gc_nil)
    return(s7_symbol_value(sc, sym));
  return(val);
}


typedef bool (safe_sym_t)(s7_scheme *sc, s7_pointer sym, s7_pointer e);
static s7_function all_x_eval(s7_scheme *sc, s7_pointer arg, s7_pointer e, safe_sym_t *checker);

static bool direct_memq(s7_pointer symbol, s7_pointer symbols)
{
  s7_pointer x;
  for (x = symbols; is_pair(x); x = unchecked_cdr(x))
    {
      if (car(x) == symbol)
	return(true);
      x = cdr(x);
      if (unchecked_car(x) == symbol)
	return(true);
    }
  return(false);
}

static bool indirect_memq(s7_pointer symbol, s7_pointer symbols)
{ /* used only below in do_symbol_is_safe */
  s7_pointer x;
  for (x = symbols; is_pair(x); x = cdr(x))
    if (caar(x) == symbol)
      return(true);
  return(false);
}

static bool do_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((is_slot(global_slot(sym))) || 
	 (indirect_memq(sym, e)) ||
	 (is_slot(find_symbol(sc, sym))));
}

static bool let_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((is_slot(global_slot(sym))) || ((!is_with_let_let(e)) && (is_slot(find_symbol(sc, sym)))));
}

static bool pair_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((is_slot(global_slot(sym))) || (direct_memq(sym, e)));
}


/* make macros and closures */

static s7_pointer make_macro(s7_scheme *sc)
{
  s7_pointer cx, mac;
  unsigned int typ;

  if (sc->op == OP_DEFINE_MACRO)
    typ = T_MACRO | T_DONT_EVAL_ARGS | T_COPY_ARGS;
  else
    {
      if (sc->op == OP_DEFINE_MACRO_STAR)
	typ = T_MACRO_STAR | T_DONT_EVAL_ARGS | T_COPY_ARGS;
      else
	{
	  if (sc->op == OP_DEFINE_BACRO)
	    typ = T_BACRO | T_DONT_EVAL_ARGS | T_COPY_ARGS;
	  else
	    {
	      if (sc->op == OP_DEFINE_BACRO_STAR)
		typ = T_BACRO_STAR | T_DONT_EVAL_ARGS | T_COPY_ARGS;
	      else
		{
		  if ((sc->op == OP_DEFINE_EXPANSION) &&
		      (!is_let(sc->envir)))        /* local expansions are just normal macros */
		    typ = T_MACRO | T_EXPANSION | T_DONT_EVAL_ARGS | T_COPY_ARGS;
		  else typ = T_MACRO | T_DONT_EVAL_ARGS | T_COPY_ARGS;
		}
	    }
	}
    }

  new_cell_no_check(sc, mac, typ);
  sc->temp6 = mac;
  closure_set_args(mac, cdar(sc->code));
  closure_set_body(mac, cdr(sc->code));
  closure_set_setter(mac, sc->F);
  closure_set_let(mac, sc->envir);
  closure_arity(mac) = CLOSURE_ARITY_NOT_SET;

  sc->capture_let_counter++;
  sc->code = caar(sc->code);
  if ((sc->op == OP_DEFINE_EXPANSION) &&
      (!is_let(sc->envir)))
    set_type(sc->code, T_EXPANSION | T_SYMBOL); /* see comment under READ_TOK */
  /* symbol? macro name has already been checked, find name in environment, and define it */
  cx = find_local_symbol(sc, sc->code, sc->envir);
  if (is_slot(cx))
    slot_set_value(cx, mac);
  else s7_make_slot(sc, sc->envir, sc->code, mac); /* was current but we've checked immutable already */

  optimize(sc, closure_body(mac), 0, sc->nil);
  sc->temp6 = sc->nil;
  return(mac);
}


static s7_pointer make_closure(s7_scheme *sc, s7_pointer args, s7_pointer code, int type)
{
  /* this is called every time a lambda form is evaluated, or during letrec, etc */

  s7_pointer x;
  unsigned int typ;

  if (is_safe_closure(code))
    {
      if (type == T_CLOSURE)
	typ = T_CLOSURE | T_PROCEDURE | T_SAFE_CLOSURE | T_COPY_ARGS;
      else typ = T_CLOSURE_STAR | T_PROCEDURE | T_SAFE_CLOSURE;
    }
  else
    {
      if (type == T_CLOSURE)
	typ = T_CLOSURE | T_PROCEDURE | T_COPY_ARGS;
      else typ = T_CLOSURE_STAR | T_PROCEDURE;
    }

  new_cell(sc, x, typ);
  closure_set_args(x, args);
  closure_set_body(x, code);
  closure_set_setter(x, sc->F);
  if (is_null(args))
    closure_arity(x) = 0;
  else closure_arity(x) = CLOSURE_ARITY_NOT_SET;
  closure_set_let(x, sc->envir);
  sc->capture_let_counter++;
  return(x);
}


#define make_closure_with_let(Sc, X, Args, Code, Env)	\
  do {							\
    unsigned int _T_;							\
    if (is_safe_closure(Code))						\
      _T_ = T_CLOSURE | T_PROCEDURE | T_SAFE_CLOSURE | T_COPY_ARGS;	\
    else _T_ = T_CLOSURE | T_PROCEDURE | T_COPY_ARGS;			\
    new_cell(Sc, X, _T_);						\
    closure_set_args(X, Args);						\
    closure_set_body(X, Code);				                \
    closure_set_setter(X, sc->F);					\
    if (is_null(Args)) closure_arity(X) = 0; else closure_arity(X) = CLOSURE_ARITY_NOT_SET; \
    closure_set_let(X, Env); \
    sc->capture_let_counter++;						\
  } while (0)


#define make_closure_without_capture(Sc, X, Args, Code, Env)	\
  do {								\
    unsigned int _T_;							\
    if (is_safe_closure(Code))						\
      _T_ = T_CLOSURE | T_PROCEDURE | T_SAFE_CLOSURE | T_COPY_ARGS;	\
    else _T_ = T_CLOSURE | T_PROCEDURE | T_COPY_ARGS;			\
    new_cell(Sc, X, _T_);						\
    closure_set_args(X, Args);						\
    closure_set_body(X, Code);				                \
    closure_set_setter(X, sc->F);					\
    if (is_null(Args)) closure_arity(X) = 0; else closure_arity(X) = CLOSURE_ARITY_NOT_SET; \
    closure_set_let(X, Env); \
  } while (0)


static int closure_length(s7_scheme *sc, s7_pointer e)
{
  /* we can't use let_length(sc, closure_let(e)) because the closure_let(closure)
   *   changes.  So the open bit is not always on.  Besides, the fallbacks need to be for closures, not environments.
   */
  s7_pointer length_func;
  length_func = find_method(sc, closure_let(e), sc->length_symbol);
  if (length_func != sc->undefined)
    return((int)s7_integer(s7_apply_function(sc, length_func, list_1(sc, e))));

  /* there are cases where this should raise a wrong-type-arg error, but for now... */
  return(-1);
}

#define check_closure_for(Sc, Fnc, Sym)				    \
  if ((has_closure_let(Fnc)) && (is_let(closure_let(Fnc))))	    \
    {								    \
      s7_pointer val;						    \
      val = find_local_symbol(Sc, Sym, closure_let(Fnc));	    \
      if ((!is_slot(val)) && (is_let(outlet(closure_let(Fnc)))))    \
	val = find_local_symbol(Sc, Sym, outlet(closure_let(Fnc))); \
      if (is_slot(val))						    \
	return(slot_value(val));				    \
    }

static s7_pointer copy_tree(s7_scheme *sc, s7_pointer tree)
{
#if WITH_GCC
  #define COPY_TREE(P) ({s7_pointer _p; _p = P; cons_unchecked(sc, (is_pair(car(_p))) ? copy_tree(sc, car(_p)) : car(_p), (is_pair(cdr(_p))) ? copy_tree(sc, cdr(_p)) : cdr(_p));})
#else
  #define COPY_TREE(P) copy_tree(sc, P)
#endif

  return(cons_unchecked(sc, 
			(is_pair(car(tree))) ? COPY_TREE(car(tree)) : car(tree),
			(is_pair(cdr(tree))) ? COPY_TREE(cdr(tree)) : cdr(tree)));
}

static void annotate_expansion(s7_pointer p)
{
  if ((is_symbol(car(p))) &&
      (is_pair(cdr(p))))
    {
      set_opt_back(p);
      set_overlay(cdr(p));
    }
  else
    {
      if (is_pair(car(p)))
	annotate_expansion(car(p));
    }
  for (p = cdr(p); is_pair(p); p = cdr(p))
    if (is_pair(car(p)))
      annotate_expansion(car(p));
}

static s7_pointer copy_body(s7_scheme *sc, s7_pointer p)
{
  if (8192 >= (sc->free_heap_top - sc->free_heap))
    {
      gc(sc);
      while (8192 >= (sc->free_heap_top - sc->free_heap))
	resize_heap(sc);
    }
  sc->w = copy_tree(sc, p);
  annotate_expansion(sc->w);
  p = sc->w;
  sc->w = sc->nil;
  return(p);
}

static s7_pointer copy_closure(s7_scheme *sc, s7_pointer fnc)
{
  /* copy the source tree annotating (for eventual optimization), return a thing of the same type as fnc */
  s7_pointer x, body;

  body = copy_body(sc, closure_body(fnc));
  new_cell(sc, x, typeflag(fnc));
  closure_set_args(x, closure_args(fnc));
  closure_set_body(x, body);
  closure_set_setter(x, closure_setter(fnc));
  closure_arity(x) = closure_arity(fnc);
  closure_set_let(x, closure_let(fnc));
  return(x);
}

/* -------------------------------- defined? -------------------------------- */
static s7_pointer g_is_defined(s7_scheme *sc, s7_pointer args)
{
  #define H_is_defined "(defined? obj (env (curlet)) ignore-globals) returns #t if obj has a binding (a value) in the environment env"
  #define Q_is_defined s7_make_signature(sc, 4, sc->is_boolean_symbol, sc->is_symbol_symbol, sc->is_let_symbol, sc->is_boolean_symbol)

  s7_pointer sym;

  /* is this correct?
   *    (defined? '_x) #f (symbol->value '_x) #<undefined>
   *    (define x #<undefined>) (defined? 'x) #t
   */

  sym = car(args);
  if (!is_symbol(sym))
    method_or_bust(sc, sym, sc->is_defined_symbol, args, T_SYMBOL, 1);

  if (is_pair(cdr(args)))
    {
      s7_pointer e, b, x;
      e = cadr(args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->is_defined_symbol, 2, e, a_let_string));

      if (is_pair(cddr(args)))
	{
	  b = caddr(args);
	  if (!s7_is_boolean(b))
	    method_or_bust_with_type(sc, b, sc->is_defined_symbol, args, a_boolean_string, 3);
	}
      else b = sc->F;

      if (e == sc->rootlet)
	return(make_boolean(sc, is_slot(global_slot(sym)))); /* new_symbol and gensym initialize global_slot to #<undefined> */

      x = find_local_symbol(sc, sym, e);
      if (is_slot(x))
	return(sc->T);

      if (b == sc->T)
	return(sc->F);

      /* here we can't fall back on find_symbol:
       *    (let ((b 2))
       *      (let ((e (curlet)))
       *        (let ((a 1))
       *          (if (defined? 'a e)
       *              (format #t "a: ~A in ~{~A ~}" (symbol->value 'a e) e))))
       *    "a: 1 in (b . 2)"
       *
       * but we also can't just return #f:
       *    (let ((b 2))
       *      (let ((e (curlet)))
       *        (let ((a 1))
       *          (format #t "~A: ~A" (defined? 'abs e) (eval '(abs -1) e)))))
       *    "#f: 1"
       */
      return(make_boolean(sc, is_slot(global_slot(sym))));
    }
  else
    {
      if (is_global(sym))
	return(sc->T);
    }
  return(make_boolean(sc, is_slot(find_symbol(sc, sym))));
}


bool s7_is_defined(s7_scheme *sc, const char *name)
{
  s7_pointer x;
  x = s7_symbol_table_find_name(sc, name);
  if (x)
    {
      x = find_symbol(sc, x);
      return(is_slot(x));
    }
  return(false);
}


void s7_define(s7_scheme *sc, s7_pointer envir, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x;
  if ((envir == sc->nil) ||
      (envir == sc->rootlet))
    envir = sc->shadow_rootlet;
  x = find_local_symbol(sc, symbol, envir);
  if (is_slot(x))
    slot_set_value(x, value);
  else
    {
      s7_make_slot(sc, envir, symbol, value); /* I think this means C code can override "constant" defs */
      if ((envir == sc->shadow_rootlet) &&
	  (!is_slot(global_slot(symbol))))
	{
	  set_global(symbol); /* is_global => global_slot is usable */
	  set_global_slot(symbol, local_slot(symbol));
	}
    }
}


s7_pointer s7_define_variable(s7_scheme *sc, const char *name, s7_pointer value)
{
  s7_pointer sym;
  sym = make_symbol(sc, name);
  s7_define(sc, sc->nil, sym, value);
  return(sym);
}


s7_pointer s7_define_variable_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym;
  sym = s7_define_variable(sc, name, value);
  symbol_set_has_help(sym);
  symbol_help(sym) = copy_string(help);
  return(sym);
}


s7_pointer s7_define_constant(s7_scheme *sc, const char *name, s7_pointer value)
{
  s7_pointer sym;
  sym = make_symbol(sc, name);
  s7_define(sc, sc->nil, sym, value);
  set_immutable(sym);
  return(sym);
}

/* (define (func a) (let ((cvar (+ a 1))) cvar)) (define-constant cvar 23) (func 1) -> ;can't bind an immutable object: cvar
 * (let ((aaa 1)) (define-constant aaa 32) (set! aaa 3)) -> set!: can't alter immutable object: aaa
 */

s7_pointer s7_define_constant_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym;
  sym = s7_define_constant(sc, name, value);
  symbol_set_has_help(sym);
  symbol_help(sym) = copy_string(help);
  return(value); /* inconsistent with variable above, but consistent with define_function? */
}


char *s7_symbol_documentation(s7_scheme *sc, s7_pointer sym)
{
  if (is_keyword(sym)) return(NULL);
  if ((is_symbol(sym)) &&
      (symbol_has_help(sym)))
    return(symbol_help(sym));
  return(NULL);
}


char *s7_symbol_set_documentation(s7_scheme *sc, s7_pointer sym, const char *new_doc)
{
  if (is_keyword(sym)) return(NULL);
  if ((is_symbol(sym)) &&
      (symbol_has_help(sym)) &&
      (symbol_help(sym)))
    free(symbol_help(sym));
  symbol_set_has_help(sym);
  symbol_help(sym) = copy_string(new_doc);
  return(symbol_help(sym));
}


/* -------------------------------- keyword? -------------------------------- */

bool s7_is_keyword(s7_pointer obj)
{
  return(is_keyword(obj));
}


static s7_pointer g_is_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_is_keyword "(keyword? obj) returns #t if obj is a keyword, (keyword? :rest) -> #t"
  #define Q_is_keyword pl_bt
  check_boolean_method(sc, is_keyword, sc->is_keyword_symbol, args);
}


/* -------------------------------- string->keyword -------------------------------- */
s7_pointer s7_make_keyword(s7_scheme *sc, const char *key)
{
  s7_pointer sym;
  char *name;
  unsigned int slen;
  slen = safe_strlen(key);
  tmpbuf_malloc(name, slen + 2);
  name[0] = ':';                                     /* prepend ":" */
  name[1] = '\0';
  memcpy((void *)(name + 1), (void *)key, slen);
  sym = make_symbol_with_length(sc, name, slen + 1); /* keyword slot etc taken care of here (in new_symbol actually) */
  tmpbuf_free(name, slen + 2);
  return(sym);
}


static s7_pointer g_string_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_keyword "(string->keyword str) prepends ':' to str and defines that as a keyword"
  #define Q_string_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_string_symbol)

  if (!is_string(car(args)))
    method_or_bust(sc, car(args), sc->string_to_keyword_symbol, args, T_STRING, 0);
  return(s7_make_keyword(sc, string_value(car(args))));
}

static s7_pointer c_string_to_keyword(s7_scheme *sc, s7_pointer x)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_to_keyword_symbol, list_1(sc, x), T_STRING, 0);
  return(s7_make_keyword(sc, string_value(x)));
}


/* -------------------------------- keyword->symbol -------------------------------- */
static s7_pointer g_keyword_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_keyword_to_symbol "(keyword->symbol key) returns a symbol with the same name as key but no prepended colon"
  #define Q_keyword_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_keyword_symbol)

  s7_pointer sym;
  sym = car(args);
  if (!is_keyword(sym))
    method_or_bust_with_type(sc, sym, sc->keyword_to_symbol_symbol, args, make_string_wrapper(sc, "a keyword"), 0);
  return(keyword_symbol(sym));
}

static s7_pointer c_keyword_to_symbol(s7_scheme *sc, s7_pointer sym)
{
  if (!is_keyword(sym))
    method_or_bust_with_type(sc, sym, sc->keyword_to_symbol_symbol, list_1(sc, sym), make_string_wrapper(sc, "a keyword"), 0);
  return(keyword_symbol(sym));
}


/* -------------------------------- symbol->keyword -------------------------------- */
static s7_pointer g_symbol_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_keyword "(symbol->keyword sym) returns a keyword with the same name as sym, but with a colon prepended"
  #define Q_symbol_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_symbol_symbol)

  if (!is_symbol(car(args)))
    method_or_bust(sc, car(args), sc->symbol_to_keyword_symbol, args, T_SYMBOL, 0);
  return(s7_make_keyword(sc, symbol_name(car(args))));
}

static s7_pointer c_symbol_to_keyword(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    method_or_bust(sc, sym, sc->symbol_to_keyword_symbol, list_1(sc, sym), T_SYMBOL, 0);
  return(s7_make_keyword(sc, symbol_name(sym)));
}



/* ---------------- uninterpreted pointers ---------------- */

bool s7_is_c_pointer(s7_pointer arg)
{
  return(type(arg) == T_C_POINTER);
}


void *s7_c_pointer(s7_pointer p)
{
  if ((is_number(p)) &&
      (s7_integer(p) == 0))
    return(NULL); /* special case where the null pointer has been cons'd up by hand */

  if (type(p) != T_C_POINTER)
    return(NULL);

  return(raw_pointer(p));
}


s7_pointer s7_make_c_pointer(s7_scheme *sc, void *ptr)
{
  s7_pointer x;
  new_cell(sc, x, T_C_POINTER);
  raw_pointer(x) = ptr;
  return(x);
}


static s7_pointer g_is_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_is_c_pointer "(c-pointer? obj) returns #t if obj is a C pointer being held in s7."
  #define Q_is_c_pointer pl_bt

  check_boolean_method(sc, s7_is_c_pointer, sc->is_c_pointer_symbol, args);
}


static s7_pointer c_c_pointer(s7_scheme *sc, s7_pointer arg)
{
  ptr_int p;
  if (!s7_is_integer(arg))
    method_or_bust(sc, arg, sc->c_pointer_symbol, list_1(sc, arg), T_INTEGER, 1);
  p = (ptr_int)s7_integer(arg);             /* (c-pointer (bignum "1234")) */
  return(s7_make_c_pointer(sc, (void *)p));
}

static s7_pointer g_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer "(c-pointer int) returns a c-pointer object."
  #define Q_c_pointer s7_make_signature(sc, 2, sc->is_c_pointer_symbol, sc->is_integer_symbol)
  return(c_c_pointer(sc, car(args)));
}



/* --------------------------------- rf (CLM optimizer) ----------------------------------------------- */

s7_pointer *s7_xf_start(s7_scheme *sc)
{
  sc->cur_rf->cur = sc->cur_rf->data;
  return(sc->cur_rf->cur);
}

static void resize_xf(s7_scheme *sc, xf_t *rc)
{
  /* if we're saving pointers into this array (for later fill-in), this realloc
   *   means earlier (backfill) pointers are not valid, so we have to save the position to be
   *   filled, not the pointer to it.
   */
  s7_int loc;
  loc = rc->cur - rc->data;

#if DEBUGGING
  int i;
  s7_pointer *old;
  old = rc->data;
  rc->data = (s7_pointer *)calloc(rc->size * 2, sizeof(s7_pointer));
  for (i = 0; i < rc->size; i++)
    {
      rc->data[i] = old[i];
      old[i] = NULL;
    }
#else
  rc->data = (s7_pointer *)realloc(rc->data, rc->size * 2 * sizeof(s7_pointer));
#endif
  rc->cur = (s7_pointer *)(rc->data + loc);
  rc->size *= 2;
  rc->end = (s7_pointer *)(rc->data + rc->size);
}

#define rc_loc(sc)     (ptr_int)(sc->cur_rf->cur - sc->cur_rf->data)
#define rc_go(sc, loc) (s7_pointer *)(sc->cur_rf->data + loc)

#define xf_init(N) do {rc = sc->cur_rf; if ((rc->cur + N) >= rc->end) resize_xf(sc, rc);} while (0)
#define xf_store(Val) do {(*(rc->cur)) = Val; rc->cur++;} while (0)
#define xf_save_loc(Loc) do {Loc = rc->cur - rc->data; rc->cur++;} while (0)
#define xf_save_loc2(Loc1, Loc2) do {Loc1 = rc->cur - rc->data; Loc2 = Loc1 + 1; rc->cur += 2;} while (0)
#define xf_save_loc3(Loc1, Loc2, Loc3) do {Loc1 = rc->cur - rc->data; Loc2 = Loc1 + 1; Loc3 = Loc2 + 1; rc->cur += 3;} while (0)
#define xf_store_at(Loc, Val) rc->data[Loc] = Val
#define xf_go(loc) rc->cur = (s7_pointer *)(rc->data + loc)
/* #define xf_loc() (ptr_int)(rc->cur - rc->data) */

s7_int s7_xf_store(s7_scheme *sc, s7_pointer val)
{
  s7_pointer *cur;
  xf_t *rc;
  rc = sc->cur_rf;
  if (rc->cur == rc->end)
    resize_xf(sc, rc);
  cur = rc->cur++;
  (*cur) = val;
  return(cur - rc->data);
}

void s7_xf_store_at(s7_scheme *sc, s7_int index, s7_pointer val)
{
  sc->cur_rf->data[index] = val;
}

void *s7_xf_new(s7_scheme *sc, s7_pointer e)
{
  xf_t *result;
  if (sc->rf_free_list)
    {
      result = sc->rf_free_list;
      sc->rf_free_list = sc->rf_free_list->next;
    }
  else
    {
      result = (xf_t *)malloc(sizeof(xf_t));
      result->size = 8;
      result->data = (s7_pointer *)calloc(result->size, sizeof(s7_pointer));
      result->end = (s7_pointer *)(result->data + result->size);
    }
  if (sc->cur_rf)
    {
      sc->cur_rf->next = sc->rf_stack;
      sc->rf_stack = sc->cur_rf;
    }
  sc->cur_rf = result;
  result->cur = result->data;
  result->e = e; /* set only here? */
  result->gc_list = NULL;
  return((void *)result);
}

static void s7_xf_clear(s7_scheme *sc)
{
  while (sc->cur_rf) {s7_xf_free(sc);}
}

bool s7_xf_is_stepper(s7_scheme *sc, s7_pointer sym)
{
  s7_pointer e, p;
  e = sc->cur_rf->e;
  if (!e) return(false);
  for (p = let_slots(e); is_slot(p); p = next_slot(p))
    if (slot_symbol(p) == sym)
      return(true);
  return(false);
}


static void xf_clear_list(s7_scheme *sc, xf_t *r)
{
  gc_obj *p, *op;
  for (p = r->gc_list; p; p = op)
    {
      op = p->nxt;
      free(p);
    }
  r->gc_list = NULL;
}

void *s7_xf_detach(s7_scheme *sc)
{
  xf_t *r;
  r = sc->cur_rf;
  sc->cur_rf = sc->rf_stack;
  if (sc->rf_stack)
    sc->rf_stack = sc->rf_stack->next;
  return((void *)r);
}

void s7_xf_attach(s7_scheme *sc, void *ur)
{
  xf_t *r = (xf_t *)ur;
  r->next = sc->rf_free_list;
  sc->rf_free_list = r;
  xf_clear_list(sc, r);
}

s7_pointer *s7_xf_top(s7_scheme *sc, void *ur)
{
  xf_t *r = (xf_t *)ur;
  return(r->data);
}


static s7_pointer xf_push(s7_scheme *sc, s7_pointer obj)
{
  gc_obj *p;
  p = (gc_obj *)malloc(sizeof(gc_obj));
  p->nxt = sc->cur_rf->gc_list;
  sc->cur_rf->gc_list = p;
  p->p = obj;
  return(obj);
}

#if WITH_ADD_PF
static s7_pointer xf_pop(s7_scheme *sc)
{
  if ((sc->cur_rf) && 
      (sc->cur_rf->gc_list))
    {
      s7_pointer p;
      gc_obj *g;
      g = sc->cur_rf->gc_list;
      p = g->p;
      sc->cur_rf->gc_list = g->nxt;
      free(g);
      return(p);
    }
  return(NULL);
}
#endif

void s7_xf_free(s7_scheme *sc)
{
  sc->cur_rf->next = sc->rf_free_list;
  sc->rf_free_list = sc->cur_rf;
  xf_clear_list(sc, sc->cur_rf);
  sc->cur_rf = sc->rf_stack;
  if (sc->rf_stack)
    sc->rf_stack = sc->rf_stack->next;
}

static s7_if_t implicit_int_vector_ref(s7_scheme *sc, s7_pointer expr);
static s7_rf_t implicit_float_vector_ref(s7_scheme *sc, s7_pointer expr);
static s7_pf_t implicit_pf_sequence_ref(s7_scheme *sc, s7_pointer expr);
static s7_pf_t implicit_gf_sequence_ref(s7_scheme *sc, s7_pointer expr);

#if WITH_OPTIMIZATION
static s7_pf_t implicit_pf_sequence_set(s7_scheme *sc, s7_pointer v, s7_pointer ind, s7_pointer val);
static s7_pf_t implicit_gf_sequence_set(s7_scheme *sc, s7_pointer v, s7_pointer ind, s7_pointer val);
#endif

/* set cases are via set_if/set_rf -- but set_gp|pf would need to be restricted to non-symbol settees */

/* need to make sure sequence is not a step var, also set cases */

static s7_rp_t rf_function(s7_pointer f)
{
  switch (type(f))
    {
    case T_C_FUNCTION_STAR: case T_C_FUNCTION: case T_C_ANY_ARGS_FUNCTION: case T_C_OPT_ARGS_FUNCTION: case T_C_RST_ARGS_FUNCTION:
      return(c_function_rp(f));

    case T_FLOAT_VECTOR: 
      return(implicit_float_vector_ref);

    case T_C_OBJECT:
      return(c_object_rp(f));

    case T_SYNTAX:       
      return(syntax_rp(f));
    }
  return(NULL);
}

static s7_ip_t if_function(s7_pointer f)
{
  switch (type(f))
    {
    case T_C_FUNCTION_STAR: case T_C_FUNCTION: case T_C_ANY_ARGS_FUNCTION: case T_C_OPT_ARGS_FUNCTION: case T_C_RST_ARGS_FUNCTION:
      return(c_function_ip(f));

    case T_INT_VECTOR: 
      return(implicit_int_vector_ref);

    case T_C_OBJECT:
      return(c_object_ip(f));

    case T_SYNTAX:       
      return(syntax_ip(f));
    }
  return(NULL);
}

static s7_pp_t pf_function(s7_pointer f)
{
  switch (type(f))
    {
    case T_C_FUNCTION_STAR: case T_C_FUNCTION: case T_C_ANY_ARGS_FUNCTION: case T_C_OPT_ARGS_FUNCTION: case T_C_RST_ARGS_FUNCTION:
      return(c_function_pp(f));

    case T_PAIR: case T_STRING: case T_VECTOR: case T_HASH_TABLE: case T_LET:
      return(implicit_pf_sequence_ref);

    case T_SYNTAX:       
      return(syntax_pp(f));
    }
  return(NULL);
}

static s7_pp_t gf_function(s7_pointer f)
{
  switch (type(f))
    {
    case T_C_FUNCTION_STAR: case T_C_FUNCTION: case T_C_ANY_ARGS_FUNCTION: case T_C_OPT_ARGS_FUNCTION: case T_C_RST_ARGS_FUNCTION:
      return(c_function_gp(f));

    case T_PAIR: case T_STRING: case T_VECTOR: case T_HASH_TABLE: case T_LET: case T_C_OBJECT: case T_INT_VECTOR: case T_FLOAT_VECTOR:
      return(implicit_gf_sequence_ref);
    }
  return(NULL);
}

s7_rp_t s7_rf_function(s7_scheme *sc, s7_pointer func) {return(rf_function(func));}
s7_ip_t s7_if_function(s7_scheme *sc, s7_pointer func) {return(if_function(func));}
s7_pp_t s7_pf_function(s7_scheme *sc, s7_pointer func) {return(pf_function(func));}
s7_pp_t s7_gf_function(s7_scheme *sc, s7_pointer func) {return(gf_function(func));}

void s7_rf_set_function(s7_pointer f, s7_rp_t rp)
{
#if WITH_OPTIMIZATION
  if (!is_c_function(f)) return;
  c_function_rp(f) = rp;
#else
  return;
#endif
}

void s7_if_set_function(s7_pointer f, s7_ip_t ip)
{
#if WITH_OPTIMIZATION
  if (!is_c_function(f)) return;
  c_function_ip(f) = ip;
#else
  return;
#endif
}

void s7_pf_set_function(s7_pointer f, s7_pp_t pp)
{
#if WITH_OPTIMIZATION
  if (!is_c_function(f)) return;
  c_function_pp(f) = pp;
#else
  return;
#endif
}

void s7_gf_set_function(s7_pointer f, s7_pp_t gp)
{
#if WITH_OPTIMIZATION
  if (!is_c_function(f)) return;
  c_function_gp(f) = gp;
#else
  return;
#endif
}

static s7_rp_t pair_to_rp(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer val_sym, val;
  val_sym = car(expr);
  if (!s7_is_symbol(val_sym)) return(NULL);
  if (s7_local_slot(sc, val_sym)) return(NULL);
  val = s7_symbol_value(sc, val_sym);
  return(s7_rf_function(sc, val)); 
}

static s7_ip_t pair_to_ip(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer val_sym, val;
  val_sym = car(expr);
  if (!s7_is_symbol(val_sym)) return(NULL);
  if (s7_local_slot(sc, val_sym)) return(NULL);
  val = s7_symbol_value(sc, val_sym);
  return(s7_if_function(sc, val)); 
}

static s7_pp_t pair_to_pp(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer val_sym, val;
  val_sym = car(expr);
  if (!s7_is_symbol(val_sym)) return(NULL);
  if (s7_local_slot(sc, val_sym)) return(NULL);
  val = s7_symbol_value(sc, val_sym);
  return(s7_pf_function(sc, val)); 
}

static s7_pp_t pair_to_gp(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer val_sym, val;
  val_sym = car(expr);
  if (!s7_is_symbol(val_sym)) return(NULL);
  if (s7_local_slot(sc, val_sym)) return(NULL);
  val = s7_symbol_value(sc, val_sym);
  return(s7_gf_function(sc, val)); 
}

static s7_pf_t xf_opt(s7_scheme *sc, s7_pointer lp)
{
  s7_int loc;
  s7_pointer f;
  s7_rp_t rp;
  s7_ip_t xp;
  s7_pp_t pp;
  xf_t *rc;

  f = find_symbol(sc, car(lp));
  if (!is_slot(f)) return(NULL);
  f = slot_value(f);

  xf_init(3);
  xf_save_loc(loc);

  xp = if_function(f);
  if (xp)
    {
      s7_if_t xf;
      xf = xp(sc, lp);
      if (xf)
	{
	  xf_store_at(loc, (s7_pointer)xf);
	  return((s7_pf_t)xf);
	}
      xf_go(loc + 1);
    }

  rp = rf_function(f);
  if (rp)
    {
      s7_rf_t rf;
      rf = rp(sc, lp);
      if (rf)
	{
	  xf_store_at(loc, (s7_pointer)rf);
	  return((s7_pf_t)rf);
	}
      xf_go(loc + 1);
    }

  pp = pf_function(f);
  if (pp)
    {
      s7_pf_t pf;
      pf = pp(sc, lp);
      if (pf)
	{
	  xf_store_at(loc, (s7_pointer)pf);
	  return(pf);
	}
      xf_go(loc + 1);
    }

  pp = gf_function(f);
  if (pp)
    {
      s7_pf_t pf;
      pf = pp(sc, lp);
      if (pf)
	{
	  xf_store_at(loc, (s7_pointer)pf);
	  return(pf);
	}
    }
  return(NULL);
}

#if 0
static s7_pointer if_to_pf(s7_scheme *sc, s7_pointer **p)
{
  s7_if_t xf;
  s7_int x;
  xf = (s7_if_t)(**p); (*p)++;
  x = xf(sc, p);
  return(make_integer(sc, x));
}

static s7_pointer rf_to_pf(s7_scheme *sc, s7_pointer **p)
{
  s7_rf_t rf;
  s7_double x;
  rf = (s7_rf_t)(**p); (*p)++;
  x = rf(sc, p);
  return(make_real(sc, x));
}

static s7_pf_t pf_opt(s7_scheme *sc, s7_pointer lp)
{
  s7_int loc, loc1;
  s7_pointer f;
  s7_rp_t rp;
  s7_ip_t xp;
  s7_pp_t pp;
  xf_t *rc;

  f = find_symbol(sc, car(lp));
  if (!is_slot(f)) return(NULL);
  f = slot_value(f);

  xf_init(3);
  xf_save_loc(loc);

  xp = if_function(f);
  if (xp)
    {
      s7_if_t xf;
      xf_save_loc(loc1);
      xf = xp(sc, lp);
      if (xf)
	{
	  xf_store_at(loc, (s7_pointer)if_to_pf);
	  xf_store_at(loc1, (s7_pointer)xf);
	  return((s7_pf_t)if_to_pf);
	}
      xf_go(loc + 1);
    }

  rp = rf_function(f);
  if (rp)
    {
      s7_rf_t rf;
      xf_save_loc(loc1);
      rf = rp(sc, lp);
      if (rf)
	{
	  xf_store_at(loc, (s7_pointer)rf_to_pf);
	  xf_store_at(loc1, (s7_pointer)rf);
	  return((s7_pf_t)rf_to_pf);
	}
      xf_go(loc + 1);
    }

  pp = pf_function(f);
  if (pp)
    {
      s7_pf_t pf;
      pf = pp(sc, lp);
      if (pf)
	{
	  xf_store_at(loc, (s7_pointer)pf);
	  return(pf);
	}
    }
  return(NULL);
}
#endif

static s7_double rf_c(s7_scheme *sc, s7_pointer **p)
{
  s7_double x;
  x = s7_number_to_real(sc, **p); (*p)++;
  return(x);
}

static s7_double rf_s(s7_scheme *sc, s7_pointer **p)
{
  s7_double x;
  x = s7_number_to_real(sc, slot_value(**p)); (*p)++;
  return(x);
}

static bool arg_to_rf(s7_scheme *sc, s7_pointer a1, s7_int in_loc)
{
  s7_int loc;
  xf_t *rc;

  xf_init(2);
  if (in_loc == -1)
    xf_save_loc(loc);
  else loc = in_loc;

  if (is_pair(a1))
    {
      s7_rp_t rp;
      s7_rf_t rf;
      rp = pair_to_rp(sc, a1);
      if (!rp) return(false);
      rf = rp(sc, a1);
      if (!rf) return(false);
      xf_store_at(loc, (s7_pointer)rf);
      return(true);
    }

  if (is_symbol(a1))
    {
      s7_pointer slot;
      slot = s7_slot(sc, a1);
      if ((is_slot(slot)) && 
	  (is_real(slot_value(slot))))
	{
	  xf_store(slot);
	  xf_store_at(loc, (s7_pointer)rf_s);
	  return(true);
	}
      return(false);
    }

  if (is_real(a1))
    {
      xf_store(a1);
      xf_store_at(loc, (s7_pointer)rf_c);
      return(true);
    }

  return(false);
}

bool s7_arg_to_rf(s7_scheme *sc, s7_pointer a1)
{
  return(arg_to_rf(sc, a1, -1));
}

static s7_int if_c(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer i;
  i = **p; (*p)++;
  return(integer(i));
}

static s7_int if_s(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer x;
  x = slot_value(**p); (*p)++;
  if (!is_integer(x)) s7_wrong_type_arg_error(sc, "", 0, x, "an integer");
  return(integer(x));
}

static bool arg_to_if(s7_scheme *sc, s7_pointer a1, s7_int in_loc)
{
  s7_int loc;
  xf_t *rc;

  xf_init(2);
  if (in_loc == -1)
    xf_save_loc(loc);
  else loc = in_loc;

  if (is_pair(a1))
    {
      s7_ip_t ip;
      s7_if_t xf;
      ip = pair_to_ip(sc, a1);
      if (!ip) return(false);
      xf = ip(sc, a1);
      if (!xf) return(false);
      xf_store_at(loc, (s7_pointer)xf);
      return(true);
    }

  if (is_symbol(a1))
    {
      s7_pointer slot;
      slot = s7_slot(sc, a1);
      if ((is_slot(slot)) && 
	  (is_integer(slot_value(slot))))
	{
	  xf_store(slot);
	  xf_store_at(loc, (s7_pointer)if_s);
	  return(true);
	}
      return(false);
    }

  if (is_integer(a1))
    {
      xf_store(a1);
      xf_store_at(loc, (s7_pointer)if_c);
      return(true);
    }

  return(false);
}

bool s7_arg_to_if(s7_scheme *sc, s7_pointer a1)
{
  return(arg_to_if(sc, a1, -1));
}

static s7_pointer pf_c(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer x;
  x = **p; (*p)++;
  return(x);
}

static s7_pointer pf_s(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer x;
  x = slot_value(**p); (*p)++;
  return(x);
}

static bool arg_to_pf(s7_scheme *sc, s7_pointer a1, s7_int in_loc)
{
  s7_int loc;
  xf_t *rc;

  xf_init(2);
  if (in_loc == -1)
    xf_save_loc(loc);
  else loc = in_loc;

  if (is_pair(a1))
    {
      s7_pp_t pp;
      s7_pf_t pf;
      pp = pair_to_pp(sc, a1);
      if (!pp) return(false);
      pf = pp(sc, a1);
      if (!pf) return(false);
      xf_store_at(loc, (s7_pointer)pf);
      return(true);
    }

  if (is_symbol(a1))
    {
      s7_pointer slot;
      slot = s7_slot(sc, a1);
      if (is_slot(slot))
	{
	  xf_store(slot);
	  xf_store_at(loc, (s7_pointer)pf_s);
	  return(true);
	}
      return(false);
    }

  xf_store(a1);
  xf_store_at(loc, (s7_pointer)pf_c);
  return(true);
}

bool s7_arg_to_pf(s7_scheme *sc, s7_pointer a1)
{
  return(arg_to_pf(sc, a1, -1));
}

static bool arg_to_gf(s7_scheme *sc, s7_pointer a1, s7_int in_loc)
{
  if (is_pair(a1))
    {
      s7_pp_t gp;
      gp = pair_to_gp(sc, a1);
      if (gp)
	{
	  xf_t *rc;
	  s7_pf_t gf;
	  s7_int loc;

	  xf_init(1);
	  if (in_loc == -1)
	    xf_save_loc(loc);
	  else loc = in_loc;
	  gf = gp(sc, a1);
	  if (gf)
	    {
	      xf_store_at(loc, (s7_pointer)gf);
	      return(true);
	    }
	}
    }
  return(false);
}

bool s7_arg_to_gf(s7_scheme *sc, s7_pointer a1)
{
  return(arg_to_gf(sc, a1, -1));
}

static s7_rf_t pair_to_rf(s7_scheme *sc, s7_pointer a1, s7_rf_t x)
{
  if (s7_arg_to_rf(sc, a1))
    return(x);
  return(NULL);
}

static s7_rf_t pair_to_rf_via_if(s7_scheme *sc, s7_pointer a1, s7_rf_t x)
{
  if (s7_arg_to_if(sc, a1))
    return(x);
  return(NULL);
}


s7_rf_t s7_rf_1(s7_scheme *sc, s7_pointer expr, s7_rf_t r, s7_rf_t s, s7_rf_t x)
{
  s7_pointer a1;
  xf_t *rc;

  if ((is_null(cdr(expr))) || (!is_null(cddr(expr)))) return(NULL);
  a1 = cadr(expr);

  xf_init(1);
  if (is_real(a1))
    {
      xf_store(a1);
      return(r);
    }

  if (is_symbol(a1))
    {
      a1 = s7_slot(sc, a1);
      if ((!is_slot(a1)) || (is_t_complex(slot_value(a1)))) return(NULL);
      xf_store(a1);
      return(s);
    }

  if (is_pair(a1))
    return(pair_to_rf(sc, a1, x));

  return(NULL);
}

s7_rf_t s7_rf_2(s7_scheme *sc, s7_pointer expr, s7_rf_t rr, s7_rf_t sr, s7_rf_t xr, s7_rf_t rs, s7_rf_t ss, s7_rf_t xs, s7_rf_t rx, s7_rf_t sx, s7_rf_t xx)
{
  s7_pointer a1, a2;
  xf_t *rc;

  if ((is_null(cdr(expr))) || (!is_null(cdddr(expr)))) return(NULL);
  a1 = cadr(expr);
  a2 = caddr(expr);

  xf_init(2);
  if (is_real(a1))
    {
      xf_store(a1);
      if (is_real(a2))
	{
	  xf_store(a2);
	  return(rr);
	}
      if (is_symbol(a2))
	{
	  a2 = s7_slot(sc, a2);
	  if ((!is_slot(a2)) || (is_t_complex(slot_value(a2)))) return(NULL);
	  xf_store(a2);
	  return(rs);
	}
      if (is_pair(a2))
	return(pair_to_rf(sc, a2, rx));
      return(NULL);
    }

  if (is_symbol(a1))
    {
      a1 = s7_slot(sc, a1);
      if ((!is_slot(a1)) || (is_t_complex(slot_value(a1)))) return(NULL);
      xf_store(a1);
      if (is_real(a2))
	{
	  xf_store(a2);
	  return(sr);
	}
      if (is_symbol(a2))
	{
	  a2 = s7_slot(sc, a2);
	  if ((!is_slot(a2)) || (is_t_complex(slot_value(a2)))) return(NULL);
	  xf_store(a2);
	  return(ss);
	}
      if (is_pair(a2))
	return(pair_to_rf(sc, a2, sx));
      return(NULL);
    }

  if (is_pair(a1))
    {
      s7_int loc;
      s7_rp_t rp;
      s7_rf_t rf;

      xf_save_loc(loc);
      rp = pair_to_rp(sc, a1);
      if (!rp) return(NULL);
      rf = rp(sc, a1);
      if (!rf) return(NULL);
      xf_store_at(loc, (s7_pointer)rf);

      if (is_real(a2))
	{
	  xf_store(a2);
	  return(xr);
	}
      if (is_symbol(a2))
	{
	  a2 = s7_slot(sc, a2);
	  if ((!is_slot(a2)) || (is_t_complex(slot_value(a2)))) return(NULL);
	  xf_store(a2);
	  return(xs);
	}
      if (is_pair(a2))
	return(pair_to_rf(sc, a2, xx));
      return(NULL);
    }
  return(NULL);
}

#if (!WITH_GMP)
typedef struct {s7_rf_t none, r, s, p, rs, rp, ss, sp, pp, rss, rsp, rpp, sss, ssp, spp, ppp;} rf_ops;
static rf_ops *add_r_ops, *multiply_r_ops;

static s7_rf_t com_rf_2(s7_scheme *sc, s7_pointer expr, rf_ops *a)
{
  /* expr len is assumed to be 3 (2 args) */
  s7_pointer a1, a2, p1 = NULL, p2 = NULL, s1 = NULL, s2 = NULL, c1 = NULL, c2 = NULL;
  xf_t *rc;

  a1 = cadr(expr);
  if (is_pair(a1)) p1 = a1; else {if (is_symbol(a1)) s1 = a1; else {if (is_real(a1)) c1 = a1; else return(NULL);}}
  a2 = caddr(expr);
  if (is_pair(a2)) p2 = a2; else {if (is_symbol(a2)) s2 = a2; else {if (is_real(a2)) c2 = a2; else return(NULL);}}
  
  xf_init(2);
  if (!c1) {c1 = c2; c2 = NULL;}
  if (c2)
    {
      if ((is_t_real(c1)) || (is_t_real(c2)))
	{
	  s7_pointer x;
	  s7_double x1, x2;
	  x1 = real_to_double(sc, c1, (a == add_r_ops) ? "+" : "*");
	  x2 = real_to_double(sc, c2, (a == add_r_ops) ? "+" : "*");
	  if (a == add_r_ops)
	    x = make_real(sc, x1 + x2);
	  else x = make_real(sc, x1 * x2);
	  if (!is_immutable_real(x))
	    xf_push(sc, x);
	  xf_store(x);
	  return(a->r);
	}
      return(NULL);
    }
  if (!s1) {s1 = s2; s2 = NULL;} 
  if (!p1) {p1 = p2; p2 = NULL;}
  
  if (s1)
    {
      bool s1_real;
      s1 = s7_slot(sc, s1);
      if ((!is_slot(s1)) || (is_unsafe_stepper(s1)) || (is_t_complex(slot_value(s1)))) return(NULL);
      s1_real = (is_t_real(slot_value(s1)));
      xf_store(s1);
      if (s2)
	{
	  s2 = s7_slot(sc, s2);
	  if ((!is_slot(s2)) || (is_unsafe_stepper(s2)) || (is_t_complex(slot_value(s2)))) return(NULL);
	  
	  if ((s1_real) ||                         /* TODO: look at step etc */
	      (is_t_real(slot_value(s2))))
	    {
	      xf_store(s2);
	      return(a->ss);
	    }
	  return(NULL);
	}
      if (c1)
	{
	  if ((s1_real) || (is_t_real(c1)))
	    {
	      xf_store(c1);
	      return(a->rs);
	    }
	  return(NULL);
	}
      if (s7_arg_to_rf(sc, p1))
	return(a->sp);
      return(NULL);
    }

  /* must be p1 here, c1 or p2 */
  if (c1)
    {
      xf_store(c1);
      if (s7_arg_to_rf(sc, p1))
	return(a->rp);
      return(NULL);
    }
  
  if ((s7_arg_to_rf(sc, p1)) &&
      (s7_arg_to_rf(sc, p2)))
    return(a->pp);

  return(NULL);
}

static s7_rf_t com_rf_3(s7_scheme *sc, s7_pointer expr, rf_ops *a)
{
  /* expr len is assumed to be 4 (3 args) */
  s7_pointer a1, a2, a3, p1 = NULL, p2 = NULL, p3 = NULL, s1 = NULL, s2 = NULL, s3 = NULL, c1 = NULL, c2 = NULL, c3 = NULL;
  bool s1_real = false;
  xf_t *rc;
  
  a1 = cadr(expr);
  if (is_pair(a1)) p1 = a1; else {if (is_symbol(a1)) s1 = a1; else {if (is_real(a1)) c1 = a1; else return(NULL);}}
  a2 = caddr(expr);
  if (is_pair(a2)) p2 = a2; else {if (is_symbol(a2)) s2 = a2; else {if (is_real(a2)) c2 = a2; else return(NULL);}}
  a3 = cadddr(expr);
  if (is_pair(a3)) p3 = a3; else {if (is_symbol(a3)) s3 = a3; else {if (is_real(a3)) c3 = a3; else return(NULL);}}
  
  if (!s2) {s2 = s3; s3 = NULL;}
  if (!s1) {s1 = s2; s2 = s3; s3 = NULL;} 

  xf_init(3);
  if (s1)
    {
      s1 = s7_slot(sc, s1);
      if ((!is_slot(s1)) || (is_unsafe_stepper(s1)) || (is_t_complex(slot_value(s1)))) return(NULL);
      s1_real = (is_t_real(slot_value(s1)));
      xf_store(s1);
    }

  if (!p2) {p2 = p3; p3 = NULL;}
  if (!p1) {p1 = p2; p2 = p3; p3 = NULL;}

  if (!c2) {c2 = c3; c3 = NULL;}
  if (!c1) {c1 = c2; c2 = c3; c3 = NULL;}
  if (c2)
    {
      if ((is_t_real(c1)) || (is_t_real(c2)) || ((c3) && (is_t_real(c3))))
	{
	  s7_pointer x;
	  s7_double x1, x2, x3;
	  x1 = real_to_double(sc, c1, (a == add_r_ops) ? "+" : "*");
	  x2 = real_to_double(sc, c2, (a == add_r_ops) ? "+" : "*");
	  if (c3) x3 = real_to_double(sc, c3, (a == add_r_ops) ? "+" : "*"); else x3 = ((a == add_r_ops) ? 0.0 : 1.0);
	  if (a == add_r_ops)
	    x = make_real(sc, x1 + x2 + x3);
	  else x = make_real(sc, x1 * x2 * x3);
	  if (!is_immutable_real(x))
	    xf_push(sc, x);
	  xf_store(x);
	  if (c3) return(a->r);
	  if (s1) return(a->rs);
	  if (s7_arg_to_rf(sc, p1))
	    return(a->rp);
	}
      return(NULL);
    }

  if (s1)
    {
      if (s2)
	{
	  bool s2_real;
	  s2 = s7_slot(sc, s2);
	  if ((!is_slot(s2)) || (is_unsafe_stepper(s2)) || (is_t_complex(slot_value(s2)))) return(NULL);
	  s2_real = (is_t_real(slot_value(s2)));
	  xf_store(s2);
	  if (s3)
	    {
	      s3 = s7_slot(sc, s3);
	      if ((!is_slot(s3)) || (is_unsafe_stepper(s3)) || (is_t_complex(slot_value(s3)))) return(NULL);
	      if ((s1_real) || (s2_real) || (is_t_real(slot_value(s3))))
		{
		  xf_store(s3);
		  return(a->sss);
		}
	      return(NULL);
	    }
	  if (c1)
	    {
	      if ((s1_real) || (s2_real) || (is_t_real(c1)))
		{
		  xf_store(c1);
		  return(a->rss);
		}
	      return(NULL);
	    }
	  if (s7_arg_to_rf(sc, p1))
	    return(a->ssp);
	  return(NULL);
	}
      if (c1)
	{
	  xf_store(c1);
	  if (s7_arg_to_rf(sc, p1))
	    return(a->rsp);
	  return(NULL);
	}
      if ((s7_arg_to_rf(sc, p1)) &&
	  (s7_arg_to_rf(sc, p2)))
	return(a->spp);
      return(NULL);
    }
  
  if (c1)
    {
      xf_store(c1);
      if ((s7_arg_to_rf(sc, p1)) &&
	  (s7_arg_to_rf(sc, p2)))
	return(a->rpp);
      return(NULL);
    }

  if ((s7_arg_to_rf(sc, p1)) &&
      (s7_arg_to_rf(sc, p2)) &&
      (s7_arg_to_rf(sc, p3)))
    return(a->ppp);
  return(NULL);
}

typedef struct {s7_if_t none, r, s, p, rs, rp, ss, sp, pp, rss, rsp, rpp, sss, ssp, spp, ppp;} if_ops;
static if_ops *add_i_ops, *multiply_i_ops;

static s7_if_t com_if_2(s7_scheme *sc, s7_pointer expr, if_ops *a)
{
  /* expr len is assumed to be 3 (2 args) */
  s7_pointer a1, a2, p1 = NULL, p2 = NULL, s1 = NULL, s2 = NULL, c1 = NULL, c2 = NULL;
  xf_t *rc;

  a1 = cadr(expr);
  if (is_pair(a1)) p1 = a1; else {if (is_symbol(a1)) s1 = a1; else {if (is_real(a1)) c1 = a1; else return(NULL);}}
  a2 = caddr(expr);
  if (is_pair(a2)) p2 = a2; else {if (is_symbol(a2)) s2 = a2; else {if (is_real(a2)) c2 = a2; else return(NULL);}}
  
  xf_init(2);
  if (!c1) {c1 = c2; c2 = NULL;}
  if ((c1) && (!is_t_integer(c1))) return(NULL);
  if (c2)
    {
      s7_pointer x;
      if (!(is_t_integer(c2))) return(NULL);
      if (a == add_i_ops)
	x = make_integer(sc, integer(c1) + integer(c2));
      else x = make_integer(sc, integer(c1) * integer(c2));
      if (!is_immutable_integer(x))
	xf_push(sc, x);
      xf_store(x);
      return(a->r);
    }
  if (!s1) {s1 = s2; s2 = NULL;} 
  if (!p1) {p1 = p2; p2 = NULL;}
  
  if (s1)
    {
      s1 = s7_slot(sc, s1);
      if ((!is_slot(s1)) || (is_unsafe_stepper(s1)) || (!is_t_integer(slot_value(s1)))) return(NULL);
      xf_store(s1);
      if (s2)
	{
	  s2 = s7_slot(sc, s2);
	  if ((!is_slot(s2)) || (is_unsafe_stepper(s2)) || (!is_t_integer(slot_value(s2)))) return(NULL);
	  xf_store(s2);
	  return(a->ss);
	}
      if (c1)
	{
	  xf_store(c1);
	  return(a->rs);
	}
      if (s7_arg_to_if(sc, p1))
	return(a->sp);
      return(NULL);
    }

  /* must be p1 here, c1 or p2 */
  if (c1)
    {
      xf_store(c1);
      if (s7_arg_to_if(sc, p1))
	return(a->rp);
      return(NULL);
    }
  
  if ((s7_arg_to_if(sc, p1)) &&
      (s7_arg_to_if(sc, p2)))
    return(a->pp);

  return(NULL);
}

static s7_if_t com_if_3(s7_scheme *sc, s7_pointer expr, if_ops *a)
{
  /* expr len is assumed to be 4 (3 args) */
  s7_pointer a1, a2, a3, p1 = NULL, p2 = NULL, p3 = NULL, s1 = NULL, s2 = NULL, s3 = NULL, c1 = NULL, c2 = NULL, c3 = NULL;
  xf_t *rc;

  a1 = cadr(expr);
  if (is_pair(a1)) p1 = a1; else {if (is_symbol(a1)) s1 = a1; else {if (is_real(a1)) c1 = a1; else return(NULL);}}
  a2 = caddr(expr);
  if (is_pair(a2)) p2 = a2; else {if (is_symbol(a2)) s2 = a2; else {if (is_real(a2)) c2 = a2; else return(NULL);}}
  a3 = cadddr(expr);
  if (is_pair(a3)) p3 = a3; else {if (is_symbol(a3)) s3 = a3; else {if (is_real(a3)) c3 = a3; else return(NULL);}}
  
  xf_init(3);
  if (!s2) {s2 = s3; s3 = NULL;}
  if (!s1) {s1 = s2; s2 = s3; s3 = NULL;}
  if (s1)
    {
      s1 = s7_slot(sc, s1);
      if ((!is_slot(s1)) || (is_unsafe_stepper(s1)) || (!is_t_integer(slot_value(s1)))) return(NULL);
      xf_store(s1);
    }

  if (!p2) {p2 = p3; p3 = NULL;}
  if (!p1) {p1 = p2; p2 = p3; p3 = NULL;}

  if (!c2) {c2 = c3; c3 = NULL;}
  if (!c1) {c1 = c2; c2 = c3; c3 = NULL;}
  if (c1)
    {
      if (!is_t_integer(c1)) return(NULL);
      if (c2)
	{
	  s7_pointer x;
	  if (!is_t_integer(c2)) return(NULL);
	  if ((c3) && (!is_t_integer(c3))) return(NULL);
	  if (a == add_i_ops)
	    x = make_integer(sc, integer(c1) + integer(c2) + ((c3) ? integer(c3) : 0));
	  else x = make_integer(sc, integer(c1) * integer(c2) * ((c3) ? integer(c3) : 1));
	  if (!is_immutable_integer(x))
	    xf_push(sc, x);
	  xf_store(x);
	  if (c3) return(a->r);
	  if (s1) return(a->rs);
	  if (s7_arg_to_if(sc, p1))
	    return(a->rp);
	}
      return(NULL);
    }

  if (s1)
    {
      if (s2)
	{
	  s2 = s7_slot(sc, s2);
	  if ((!is_slot(s2)) || (is_unsafe_stepper(s2)) || (!is_t_integer(slot_value(s2)))) return(NULL);
	  xf_store(s2);
	  if (s3)
	    {
	      s3 = s7_slot(sc, s3);
	      if ((!is_slot(s3)) || (is_unsafe_stepper(s3)) || (!is_t_integer(slot_value(s3)))) return(NULL);
	      xf_store(s3);
	      return(a->sss);
	    }
	  if (c1)
	    {
	      xf_store(c1);
	      return(a->rss);
	    }
	  if (s7_arg_to_if(sc, p1))
	    return(a->ssp);
	  return(NULL);
	}
      if (c1)
	{
	  xf_store(c1);
	  if (s7_arg_to_if(sc, p1))
	    return(a->rsp);
	  return(NULL);
	}
      if ((s7_arg_to_if(sc, p1)) &&
	  (s7_arg_to_if(sc, p2)))
	return(a->spp);
      return(NULL);
    }
  
  if (c1)
    {
      xf_store(c1);
      if ((s7_arg_to_if(sc, p1)) &&
	  (s7_arg_to_if(sc, p2)))
	return(a->rpp);
      return(NULL);
    }

  if ((s7_arg_to_if(sc, p1)) &&
      (s7_arg_to_if(sc, p2)) &&
      (s7_arg_to_if(sc, p3)))
    return(a->ppp);
  return(NULL);
}
#endif

#if WITH_OPTIMIZATION
static s7_double set_rf_sr(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, c1;
  s7_double x;
  s1 = (**p); (*p)++;
  c1 = (**p); (*p)++;
  x = real(c1);
  slot_set_value(s1, make_real(sc, x));
  return(x);
}

#if 0
static s7_double set_rf_ss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s7_double x;
  s1 = (**p); (*p)++;
  s2 = (**p); (*p)++;
  x = real_to_double(sc, slot_value(s2), "set!");
  slot_set_value(s1, make_real(sc, x));
  return(x);
}
#endif

static s7_double set_rf_sx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_double x;
  s7_rf_t r1;
  s1 = (**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  slot_set_value(s1, make_real(sc, x));
  return(x);
}

static s7_int set_if_sx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_int x;
  s7_if_t i1;
  s1 = (**p); (*p)++;
  i1 = (s7_if_t)(**p); (*p)++;
  x = i1(sc, p);
  slot_set_value(s1, make_integer(sc, x));
  return(x);
}

static s7_rf_t float_vector_set_rf_expanded(s7_scheme *sc, s7_pointer fv, s7_pointer ind_sym, s7_pointer val_expr);
static s7_if_t int_vector_set_if_expanded(s7_scheme *sc, s7_pointer iv, s7_pointer ind_sym, s7_pointer val_expr);

static s7_rf_t set_rf(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer slot, a1;
  xf_t *rc;

  if (is_pair(cdddr(expr))) return(NULL);
  a1 = cadr(expr);
  if (!is_symbol(a1)) /* look for implicit index case */
    {
      s7_pointer fv;
      if ((!is_pair(a1)) || (!is_symbol(car(a1))) || (!is_null(cddr(a1)))) return(NULL);
      fv = s7_symbol_value(sc, car(a1));
      if (is_float_vector(fv))
	return(float_vector_set_rf_expanded(sc, fv, cadr(a1), caddr(expr)));
      if ((is_c_object(fv)) &&
	  (c_object_set_rp(fv)))
	return(c_object_set_rp(fv)(sc, expr));
      return(NULL);
    }

  /* if sym has real value and new val is real, we're ok */
  slot = s7_slot(sc, a1);
  if (!is_slot(slot)) return(NULL);

  xf_init(2);
  if (is_t_real(slot_value(slot)))
    {
      s7_pointer a2;
      xf_store(slot);
      a2 = caddr(expr);
      if (is_t_real(a2))
	{
	  xf_store(a2);
	  return(set_rf_sr);
	}
#if 0
      if (is_symbol(a2))
	{
	  s7_pointer a2_slot;
	  a2_slot = s7_slot(sc, a2);
	  if (!is_slot(a2_slot)) return(NULL);
	  if (type(slot_value(a2_slot)) != T_REAL) return(NULL);
	  xf_store(a2_slot);
	  return(set_rf_ss);
	}
#endif
      if (is_pair(a2))
	{
	  s7_rp_t rp;
	  s7_rf_t rf;
	  s7_int loc;
	  xf_save_loc(loc);
	  rp = pair_to_rp(sc, a2);
	  if (!rp) return(NULL);
	  rf = rp(sc, a2);
	  if (!rf) return(NULL);
	  xf_store_at(loc, (s7_pointer)rf);
	  return(set_rf_sx);
	}
    }
  return(NULL);
}

static s7_if_t set_if(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer slot, a1;

  if (is_pair(cdddr(expr))) return(NULL);
  a1 = cadr(expr);

  if (!is_symbol(a1)) /* look for implicit index case */
    {
      s7_pointer fv;
      if ((!is_pair(a1)) || (!is_symbol(car(a1))) || (!is_null(cddr(a1)))) return(NULL);
      fv = s7_symbol_value(sc, car(a1));
      if (is_int_vector(fv))
	return(int_vector_set_if_expanded(sc, fv, cadr(a1), caddr(expr)));
      if ((is_c_object(fv)) &&
	  (c_object_set_ip(fv)))
	return(c_object_set_ip(fv)(sc, expr));
      return(NULL);
    }

  if (!is_symbol(a1)) return(NULL);
  slot = s7_slot(sc, a1);
  if (!is_slot(slot)) return(NULL);

  if (is_t_integer(slot_value(slot)))
    {
      s7_pointer a2;
      xf_t *rc;
      xf_init(1);
      xf_store(slot);
      a2 = caddr(expr);
      if ((is_pair(a2)) &&
	  (s7_arg_to_if(sc, a2)))
	return(set_if_sx);
    }
  return(NULL);
}

static s7_pf_t set_pf(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer a1;
  if (is_pair(cdddr(expr))) return(NULL);
  a1 = cadr(expr);
  if (is_pair(a1)) /* look for implicit index case */
    {
      s7_pointer v;
      if ((!is_symbol(car(a1))) || (!is_pair(cdr(a1))) || (!is_null(cddr(a1)))) return(NULL);
      v = s7_slot(sc, car(a1));
      if (!is_slot(v)) return(NULL);
      switch (type(slot_value(v)))
	{
	case T_PAIR: case T_STRING: case T_VECTOR: case T_HASH_TABLE: case T_LET:
	  return(implicit_pf_sequence_set(sc, v, cadr(a1), caddr(expr)));

	case T_INT_VECTOR: case T_FLOAT_VECTOR:
	  return(implicit_gf_sequence_set(sc, v, cadr(a1), caddr(expr)));
	}
    }
  return(NULL);
}
#endif

typedef s7_pointer (*p0_pf_t)(s7_scheme *sc);
static s7_pointer p0_pf_1(s7_scheme *sc, s7_pointer **p, p0_pf_t fnc)
{
  return(fnc(sc));
}

static s7_pf_t pf_0(s7_scheme *sc, s7_pointer expr, s7_pf_t fnc)
{
  if (!is_null(cdr(expr))) return(NULL);
  return(fnc);
}

#define PF_0(CName, Pfnc) \
  static s7_pointer CName ## _pf_0(s7_scheme *sc, s7_pointer **rp) {return(p0_pf_1(sc, rp, Pfnc));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(pf_0(sc, expr, CName ## _pf_0));}

PF_0(curlet, s7_curlet)
PF_0(rootlet, s7_rootlet)
PF_0(current_input_port, s7_current_input_port)
PF_0(current_output_port, s7_current_output_port)
PF_0(current_error_port, s7_current_error_port)

static s7_pointer c_unlet(s7_scheme *sc) {return(g_unlet(sc, sc->nil));}
PF_0(unlet, c_unlet)
static s7_pointer c_gc(s7_scheme *sc) {return(g_gc(sc, sc->nil));}
PF_0(gc, c_gc)


/* -------- PF_TO_PF -------- */
typedef s7_pointer (*pf_pf_t)(s7_scheme *sc, s7_pointer x);
static s7_pointer pf_pf_1(s7_scheme *sc, s7_pointer **p, pf_pf_t fnc)
{
  s7_pf_t f;
  s7_pointer x;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

static s7_pointer pf_pf_s(s7_scheme *sc, s7_pointer **p, pf_pf_t fnc)
{
  s7_pointer x;
  (*p)++; x = slot_value(**p); (*p)++;
  return(fnc(sc, x));
}

static s7_pf_t pf_1(s7_scheme *sc, s7_pointer expr, s7_pf_t f1, s7_pf_t f2)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))))
    {
      ptr_int loc;
      s7_pointer a1;
      a1 = cadr(expr);
      loc = rc_loc(sc);
      if (s7_arg_to_pf(sc, a1)) return((is_symbol(a1)) ? f2 : f1);
      sc->cur_rf->cur = rc_go(sc, loc);
      if (s7_arg_to_gf(sc, a1)) return((is_symbol(a1)) ? f2 : f1);
    }
  return(NULL);
}

#define PF_TO_PF(CName, Pfnc)						\
  static s7_pointer CName ## _pf_p(s7_scheme *sc, s7_pointer **rp) {return(pf_pf_1(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_s(s7_scheme *sc, s7_pointer **rp) {return(pf_pf_s(sc, rp, Pfnc));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(pf_1(sc, expr, CName ## _pf_p, CName ## _pf_s));}

static s7_pointer c_symbol_to_value(s7_scheme *sc, s7_pointer x) {return(g_symbol_to_value(sc, set_plist_1(sc, x)));}
PF_TO_PF(symbol_to_value, c_symbol_to_value)
static s7_pointer c_symbol_to_string(s7_scheme *sc, s7_pointer p) {return(g_symbol_to_string(sc, set_plist_1(sc, p)));}
PF_TO_PF(symbol_to_string, c_symbol_to_string)
static s7_pointer c_gensym(s7_scheme *sc, s7_pointer p) {return(g_gensym(sc, set_plist_1(sc, p)));}
PF_TO_PF(gensym, c_gensym)

static s7_pointer c_not(s7_scheme *sc, s7_pointer x) {return((x == sc->F) ? sc->T : sc->F);}
PF_TO_PF(not, c_not)
PF_TO_PF(outlet, s7_outlet)
PF_TO_PF(openlet, s7_openlet)
PF_TO_PF(funclet, s7_funclet)
PF_TO_PF(coverlet, c_coverlet)

#define bool_with_method(Name, Checker, Method)				\
  static s7_pointer c_ ## Name (s7_scheme *sc, s7_pointer p)		\
  {									\
    s7_pointer func;							\
    if (Checker(p)) return(sc->T);					\
    if ((has_methods(p)) &&						\
        ((func = find_method(sc, find_let(sc, p), Method)) != sc->undefined)) \
      return(s7_apply_function(sc, func, list_1(sc, p)));		\
    return(sc->F);							\
  }									\
  PF_TO_PF(Name, c_ ## Name)

bool_with_method(is_char, s7_is_character, sc->is_char_symbol)
bool_with_method(is_boolean, s7_is_boolean, sc->is_boolean_symbol)
bool_with_method(is_byte_vector, is_byte_vector, sc->is_byte_vector_symbol)
bool_with_method(is_complex, is_number, sc->is_complex_symbol)
bool_with_method(is_constant, s7_is_constant, sc->is_constant_symbol)
bool_with_method(is_continuation, is_continuation, sc->is_continuation_symbol)
bool_with_method(is_c_pointer, s7_is_c_pointer, sc->is_c_pointer_symbol)
bool_with_method(is_dilambda, s7_is_dilambda, sc->is_dilambda_symbol)
bool_with_method(is_eof_object, is_eof, sc->is_eof_object_symbol)
bool_with_method(is_float_vector, is_float_vector, sc->is_float_vector_symbol)
bool_with_method(is_gensym, is_gensym, sc->is_gensym_symbol)
bool_with_method(is_hash_table, is_hash_table, sc->is_hash_table_symbol)
bool_with_method(is_input_port, is_input_port, sc->is_input_port_symbol)
bool_with_method(is_integer, is_integer, sc->is_integer_symbol)
bool_with_method(is_int_vector, is_int_vector, sc->is_int_vector_symbol)
bool_with_method(is_iterator, is_iterator, sc->is_iterator_symbol)
bool_with_method(is_keyword, is_keyword, sc->is_keyword_symbol)
bool_with_method(is_let, is_let, sc->is_let_symbol)
bool_with_method(is_macro, is_macro, sc->is_macro_symbol)
bool_with_method(is_null, is_null, sc->is_null_symbol)
bool_with_method(is_number, is_number, sc->is_number_symbol)
bool_with_method(is_openlet, s7_is_openlet, sc->is_openlet_symbol)
bool_with_method(is_output_port, is_output_port, sc->is_output_port_symbol)
bool_with_method(is_pair, is_pair, sc->is_pair_symbol)
bool_with_method(is_procedure, is_procedure, sc->is_procedure_symbol)
bool_with_method(is_rational, is_rational, sc->is_rational_symbol)
bool_with_method(is_real, is_real, sc->is_real_symbol)
bool_with_method(is_string, is_string, sc->is_string_symbol)
bool_with_method(is_symbol, is_symbol, sc->is_symbol_symbol)
bool_with_method(is_vector, s7_is_vector, sc->is_vector_symbol)
#define opt_is_list(p) s7_is_list(sc, p)
bool_with_method(is_list, opt_is_list, sc->is_list_symbol)
bool_with_method(iterator_is_at_end, iterator_is_at_end, sc->iterator_is_at_end_symbol)
bool_with_method(is_random_state, is_random_state, sc->is_random_state_symbol)

PF_TO_PF(string_to_keyword, c_string_to_keyword)
PF_TO_PF(keyword_to_symbol, c_keyword_to_symbol)
PF_TO_PF(symbol_to_keyword, c_symbol_to_keyword)

static s7_pointer c_symbol(s7_scheme *sc, s7_pointer x) {return(g_string_to_symbol_1(sc, x, sc->symbol_symbol));} 
PF_TO_PF(symbol, c_symbol)

#if 0
static s7_pointer symbol_pf_p(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t f;
  s7_pointer x;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(g_string_to_symbol_1(sc, x, sc->symbol_symbol));
}
#endif

/* an experiment -- we need a temp pointer per func? */
static s7_pointer string_to_symbol_pf_p(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t f;
  s7_pointer x;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(g_string_to_symbol_1(sc, x, sc->string_to_symbol_symbol));
}

static s7_pointer number_to_string_pf_p(s7_scheme *sc, s7_pointer **p);
static s7_pointer number_to_string_pf_s(s7_scheme *sc, s7_pointer **p);
static s7_pointer number_to_string_pf_temp(s7_scheme *sc, s7_pointer **p);
static s7_pointer number_to_string_pf_s_temp(s7_scheme *sc, s7_pointer **p);

static s7_pf_t string_to_symbol_pf(s7_scheme *sc, s7_pointer expr)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))))
    {
      ptr_int loc;
      loc = rc_loc(sc);
      if (s7_arg_to_pf(sc, cadr(expr))) 
	return(string_to_symbol_pf_p);
      sc->cur_rf->cur = rc_go(sc, loc);
      if (s7_arg_to_gf(sc, cadr(expr)))
	{
	  if (sc->cur_rf->data[loc] == (s7_pointer)number_to_string_pf_p)
	    sc->cur_rf->data[loc] = (s7_pointer)number_to_string_pf_temp;
	  if (sc->cur_rf->data[loc] == (s7_pointer)number_to_string_pf_s)
	    sc->cur_rf->data[loc] = (s7_pointer)number_to_string_pf_s_temp;
	  return(string_to_symbol_pf_p);
	}
    }
  return(NULL);
}

#if (!WITH_PURE_S7)
PF_TO_PF(let_to_list, s7_let_to_list)
#endif


/* -------- PF2_TO_PF -------- */
typedef s7_pointer (*pf2_pf_t)(s7_scheme *sc, s7_pointer x, s7_pointer y);
static s7_pointer pf2_pf_1(s7_scheme *sc, s7_pointer **p, pf2_pf_t fnc)
{
  s7_pf_t f;
  s7_pointer x, y;
  f = (s7_pf_t)(**p); (*p)++;
  x = f(sc, p);
  f = (s7_pf_t)(**p); (*p)++;	
  y = f(sc, p);
  return(fnc(sc, x, y));
}

static s7_pointer pf2_pf_sp(s7_scheme *sc, s7_pointer **p, pf2_pf_t fnc)
{
  s7_pf_t f;
  s7_pointer x, y;
  x = slot_value(**p); (*p)++;
  f = (s7_pf_t)(**p); (*p)++;	
  y = f(sc, p);
  return(fnc(sc, x, y));
}

static s7_pointer pf2_pf_ss(s7_scheme *sc, s7_pointer **p, pf2_pf_t fnc)
{
  s7_pointer x, y;
  x = slot_value(**p); (*p)++;
  y = slot_value(**p); (*p)++;
  return(fnc(sc, x, y));
}

static s7_pointer pf2_pf_sc(s7_scheme *sc, s7_pointer **p, pf2_pf_t fnc)
{
  s7_pointer x, y;
  x = slot_value(**p); (*p)++;
  y = (**p); (*p)++;
  return(fnc(sc, x, y));
}

static s7_pointer pf2_pf_pc(s7_scheme *sc, s7_pointer **p, pf2_pf_t fnc)
{
  s7_pf_t f;
  s7_pointer x, y;
  f = (s7_pf_t)(**p); (*p)++;
  x = f(sc, p);
  y = (**p); (*p)++;
  return(fnc(sc, x, y));
}

static s7_pf_t pf_2(s7_scheme *sc, s7_pointer expr, s7_pf_t fpp, s7_pf_t fsp, s7_pf_t fss, s7_pf_t fsc, s7_pf_t fpc)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_null(cdddr(expr))))
    {
      s7_pointer a1, a2;
      xf_t *rc;

      xf_init(2);
      a1 = cadr(expr);
      a2 = caddr(expr);
      if (is_symbol(a1))
	{
	  a1 = s7_slot(sc, a1);
	  if (!is_slot(a1)) return(NULL);
	  xf_store(a1);
	  if (is_symbol(a2))
	    {
	      a2 = s7_slot(sc, a2);
	      if (!is_slot(a2)) return(NULL);
	      xf_store(a2);
	      return(fss);
	    }
	  if (is_pair(a2))
	    {
	      if (!s7_arg_to_pf(sc, a2)) return(NULL);
	      return(fsp);
	    }
	  xf_store(a2);
	  return(fsc);
	}
      if (s7_arg_to_pf(sc, a1))
	{
	  if ((!is_pair(a2)) && (!is_symbol(a2)))
	    {
	      xf_store(a2);
	      return(fpc);
	    }
	  if (s7_arg_to_pf(sc, a2))
	    return(fpp);
	}
    }
  return(NULL);
}

#define PF2_TO_PF(CName, Pfnc)						\
  static s7_pointer CName ## _pf_p2(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_1(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_p2_sp(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_sp(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_p2_ss(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_ss(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_p2_sc(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_sc(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_p2_pc(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_pc(sc, rp, Pfnc));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) \
  { \
    return(pf_2(sc, expr, CName ## _pf_p2, CName ## _pf_p2_sp, CName ## _pf_p2_ss, CName ## _pf_p2_sc, CName ## _pf_p2_pc));\
  }


static s7_pf_t pf_2_x(s7_scheme *sc, s7_pointer expr, bool (*checker)(s7_scheme *sc, s7_pointer obj), 
		      s7_pf_t fpp, s7_pf_t fpp_x, s7_pf_t fsp, s7_pf_t fss, s7_pf_t fsc, s7_pf_t fpc, s7_pf_t fpc_x)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_null(cdddr(expr))))
    {
      s7_pointer a1, a2;
      xf_t *rc;

      xf_init(2);
      a1 = cadr(expr);
      a2 = caddr(expr);
      if (is_symbol(a1))
	{
	  a1 = s7_slot(sc, a1);
	  if (!is_slot(a1)) return(NULL);
	  xf_store(a1);
	  if (is_symbol(a2))
	    {
	      a2 = s7_slot(sc, a2);
	      if (!is_slot(a2)) return(NULL);
	      xf_store(a2);
	      return(fss);
	    }
	  if (is_pair(a2))
	    {
	      if (!s7_arg_to_pf(sc, a2)) return(NULL);
	      return(fsp);
	    }
	  xf_store(a2);
	  return(fsc);
	}
      if (s7_arg_to_pf(sc, a1))
	{
	  if ((!is_pair(a2)) && (!is_symbol(a2)))
	    {
	      xf_store(a2);
	      if ((checker(sc, a1)) && (checker(sc, a2)))
		return(fpc_x);
	      return(fpc);
	    }
	  if (s7_arg_to_pf(sc, a2))
	    {
	      if ((checker(sc, a1)) && (checker(sc, a2)))
		return(fpp_x);
	      return(fpp);
	    }
	}
    }
  return(NULL);
}

#define PF2_TO_PF_X(CName, Checker, Pfnc1, Pfnc2)				\
  static s7_pointer CName ## _pf_p2_pp(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_1(sc, rp, Pfnc1));} \
  static s7_pointer CName ## _pf_p2_ppx(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_1(sc, rp, Pfnc2));} \
  static s7_pointer CName ## _pf_p2_pc(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_pc(sc, rp, Pfnc1));} \
  static s7_pointer CName ## _pf_p2_pcx(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_pc(sc, rp, Pfnc2));} \
  static s7_pointer CName ## _pf_p2_sp(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_sp(sc, rp, Pfnc1));} \
  static s7_pointer CName ## _pf_p2_ss(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_ss(sc, rp, Pfnc1));} \
  static s7_pointer CName ## _pf_p2_sc(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_sc(sc, rp, Pfnc1));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) \
   {\
     return(pf_2_x(sc, expr, Checker, \
                   CName ## _pf_p2_pp, CName ## _pf_p2_ppx, \
                   CName ## _pf_p2_sp, CName ## _pf_p2_ss, CName ## _pf_p2_sc,	\
                   CName ## _pf_p2_pc, CName ## _pf_p2_pcx));		\
   }


static s7_pointer c_is_eq(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, x == y));}
PF2_TO_PF(is_eq, c_is_eq)
static s7_pointer c_is_eqv(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, s7_is_eqv(x, y)));}
PF2_TO_PF(is_eqv, c_is_eqv)
static s7_pointer c_is_equal(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, s7_is_equal(sc, x, y)));}
PF2_TO_PF(is_equal, c_is_equal)
static s7_pointer c_is_morally_equal(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, s7_is_morally_equal(sc, x, y)));}
PF2_TO_PF(is_morally_equal, c_is_morally_equal)
PF2_TO_PF(let_ref, s7_let_ref)

static s7_pointer c_cutlet(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_cutlet(sc, set_plist_2(sc, x, y)));}
PF2_TO_PF(cutlet, c_cutlet)
static s7_pointer c_inlet(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(s7_inlet(sc, set_plist_2(sc, x, y)));}
PF2_TO_PF(inlet, c_inlet)


/* -------- PF3_TO_PF -------- */
typedef s7_pointer (*pf3_pf_t)(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_pointer z);
static s7_pointer pf3_pf_1(s7_scheme *sc, s7_pointer **p, pf3_pf_t fnc)
{
  s7_pf_t f;
  s7_pointer x, y, z;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  f = (s7_pf_t)(**p); (*p)++;	
  y = f(sc, p);
  f = (s7_pf_t)(**p); (*p)++;	
  z = f(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_pointer pf3_pf_s(s7_scheme *sc, s7_pointer **p, pf3_pf_t fnc)
{
  s7_pf_t f;
  s7_pointer x, y, z;
  x = slot_value(**p); (*p)++;
  f = (s7_pf_t)(**p); (*p)++;	
  y = f(sc, p);
  f = (s7_pf_t)(**p); (*p)++;	
  z = f(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_pf_t pf_3(s7_scheme *sc, s7_pointer expr, s7_pf_t fp, s7_pf_t fs)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_pair(cdddr(expr))) && (is_null(cddddr(expr))))
    {
      s7_pointer a1;

      a1 = cadr(expr);
      if (is_symbol(a1))
	{
	  s7_pointer slot;
	  slot = s7_slot(sc, a1);
	  if (!is_slot(slot)) return(NULL);
	  s7_xf_store(sc, slot);
	}
      else
	{
	  if (!s7_arg_to_pf(sc, a1)) return(NULL);
	}
      if ((s7_arg_to_pf(sc, caddr(expr))) &&
	  (s7_arg_to_pf(sc, cadddr(expr))))
	return((is_symbol(a1)) ? fs : fp);
    }
  return(NULL);
}

#define PF3_TO_PF(CName, Pfnc)						\
  static s7_pointer CName ## _pf_p3(s7_scheme *sc, s7_pointer **rp) {return(pf3_pf_1(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_p3_s(s7_scheme *sc, s7_pointer **rp) {return(pf3_pf_s(sc, rp, Pfnc));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(pf_3(sc, expr, CName ## _pf_p3, CName ## _pf_p3_s));}

PF3_TO_PF(let_set, s7_let_set)
PF3_TO_PF(varlet, s7_varlet)
PF_TO_PF(c_pointer, c_c_pointer)


/* -------- PIF_TO_PF -------- */
typedef s7_pointer (*pif_pf_t)(s7_scheme *sc, s7_pointer x, s7_int y);
static s7_pointer pif_pf_1(s7_scheme *sc, s7_pointer **p, pif_pf_t fnc)
{
  s7_pf_t pf;
  s7_if_t xf;
  s7_pointer x;
  s7_int y;
  pf = (s7_pf_t)(**p); (*p)++;	
  x = pf(sc, p);
  xf = (s7_if_t)(**p); (*p)++;	
  y = xf(sc, p);
  return(fnc(sc, x, y));
}

static s7_pointer pif_pf_s(s7_scheme *sc, s7_pointer **p, pif_pf_t fnc)
{
  s7_if_t xf;
  s7_pointer x;
  s7_int y;
  x = slot_value(**p); (*p)++;
  xf = (s7_if_t)(**p); (*p)++;	
  y = xf(sc, p);
  return(fnc(sc, x, y));
}

static s7_pointer pif_pf_pp(s7_scheme *sc, s7_pointer **p, pif_pf_t fnc)
{
  s7_pf_t pf;
  s7_pointer x, y;
  pf = (s7_pf_t)(**p); (*p)++;	
  x = pf(sc, p);
  pf = (s7_pf_t)(**p); (*p)++;	
  y = pf(sc, p);
  if (!is_integer(y)) 
    return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, make_string_wrapper(sc, "~A should be an integer"), y)));
  return(fnc(sc, x, integer(y)));
}

static s7_pointer pif_pf_sp(s7_scheme *sc, s7_pointer **p, pif_pf_t fnc)
{
  s7_pf_t pf;
  s7_pointer x, y;
  x = slot_value(**p); (*p)++;
  pf = (s7_pf_t)(**p); (*p)++;	
  y = pf(sc, p);
  if (!is_integer(y)) 
    return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, make_string_wrapper(sc, "~A should be an integer"), y)));
  return(fnc(sc, x, integer(y)));
}

static s7_pf_t pif_1(s7_scheme *sc, s7_pointer expr, s7_pf_t fpi, s7_pf_t fsi, s7_pf_t fpp, s7_pf_t fsp) 
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_null(cdddr(expr))))
    {
      s7_pointer a1, a2;
      ptr_int loc;
      a1 = cadr(expr);
      a2 = caddr(expr);
      if (is_symbol(a1))
	{
	  s7_pointer slot;
	  slot = s7_slot(sc, a1);
	  if (!is_slot(slot)) return(NULL);
	  s7_xf_store(sc, slot);
	}
      else 
	{
	  if (!s7_arg_to_pf(sc, a1))
	    return(NULL);
	}
      loc = rc_loc(sc);
      if (s7_arg_to_if(sc, a2))
	return((is_symbol(a1)) ? fsi : fpi);

      sc->cur_rf->cur = rc_go(sc, loc);
      if (s7_arg_to_pf(sc, a2))
	return((is_symbol(a1)) ? fsp : fpp);
    }
  return(NULL);
}

#define PIF_TO_PF(CName, Pfnc)						\
  static s7_pointer CName ## _pf_pi(s7_scheme *sc, s7_pointer **rp) {return(pif_pf_1(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_si(s7_scheme *sc, s7_pointer **rp) {return(pif_pf_s(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_pp(s7_scheme *sc, s7_pointer **rp) {return(pif_pf_pp(sc, rp, Pfnc));} \
  static s7_pointer CName ## _pf_sp(s7_scheme *sc, s7_pointer **rp) {return(pif_pf_sp(sc, rp, Pfnc));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(pif_1(sc, expr, CName ## _pf_pi, CName ## _pf_si, CName ## _pf_pp, CName ## _pf_sp));}


/* -------- PPIF_TO_PF -------- */
typedef s7_pointer (*ppif_pf_t)(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_int z);
static s7_pointer ppif_pf_1(s7_scheme *sc, s7_pointer **p, ppif_pf_t fnc) /* other case is pf2_pf_1, type pf2_pf_t */
{
  s7_pf_t pf;
  s7_if_t xf;
  s7_pointer x, y;
  s7_int z;
  pf = (s7_pf_t)(**p); (*p)++;	
  x = pf(sc, p);
  pf = (s7_pf_t)(**p); (*p)++;	
  y = pf(sc, p);
  xf = (s7_if_t)(**p); (*p)++;	
  z = xf(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_pf_t ppif_1(s7_scheme *sc, s7_pointer expr, s7_pf_t f1, s7_pf_t f2)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))))
    {
      ptr_int loc;
      if (!s7_arg_to_pf(sc, cadr(expr))) return(NULL);
      loc = rc_loc(sc);
      if (!s7_arg_to_pf(sc, caddr(expr)))
	{
	  sc->cur_rf->cur = rc_go(sc, loc);
	  if (!s7_arg_to_gf(sc, caddr(expr))) return(NULL);
	}
      if (is_null(cdddr(expr)))	return(f1);
      if (!is_null(cddddr(expr))) return(NULL);
      if (s7_arg_to_if(sc, cadddr(expr))) return(f2);
    }
  return(NULL);
}

#define PPIF_TO_PF(CName, Pfnc1, Pfnc2)					\
  static s7_pointer CName ## _pf_pp(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_1(sc, rp, Pfnc1));} \
  static s7_pointer CName ## _pf_ppi(s7_scheme *sc, s7_pointer **rp) {return(ppif_pf_1(sc, rp, Pfnc2));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(ppif_1(sc, expr, CName ## _pf_pp, CName ## _pf_ppi));}


/* -------- PIPF_TO_PF -------- */
typedef s7_pointer (*pipf_pf_t)(s7_scheme *sc, s7_pointer x, s7_int y, s7_pointer z);
static s7_pointer pipf_pf_slot(s7_scheme *sc, s7_pointer **p, pipf_pf_t fnc)
{
  s7_pf_t pf;
  s7_pointer x, z;
  s7_int y;
  x = (s7_pointer)(**p); (*p)++;
  y = s7_integer(slot_value(**p)); (*p)++;
  pf = (s7_pf_t)(**p); (*p)++;	
  z = pf(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_pointer pipf_pf_s(s7_scheme *sc, s7_pointer **p, pipf_pf_t fnc)
{
  s7_pf_t pf;
  s7_if_t xf;
  s7_pointer x, z;
  s7_int y;
  x = (s7_pointer)(**p); (*p)++;
  xf = (s7_if_t)(**p); (*p)++;
  y = xf(sc, p);
  pf = (s7_pf_t)(**p); (*p)++;	
  z = pf(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_pointer pipf_pf_seq(s7_scheme *sc, s7_pointer **p, pipf_pf_t fnc) /* used in implicit_sequence_set */
{
  s7_pf_t pf;
  s7_if_t xf;
  s7_pointer x, z;
  s7_int y;
  x = slot_value(**p); (*p)++;
  xf = (s7_if_t)(**p); (*p)++;
  y = xf(sc, p);
  pf = (s7_pf_t)(**p); (*p)++;	
  z = pf(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_pointer pipf_pf_a(s7_scheme *sc, s7_pointer **p, pipf_pf_t fnc)
{
  s7_pf_t pf;
  s7_if_t xf;
  s7_pointer x, z;
  s7_int y;
  pf = (s7_pf_t)(**p); (*p)++;	
  x = pf(sc, p);
  xf = (s7_if_t)(**p); (*p)++;
  y = xf(sc, p);
  pf = (s7_pf_t)(**p); (*p)++;	
  z = pf(sc, p);
  return(fnc(sc, x, y, z));
}

enum {TEST_NO_S, TEST_SS, TEST_SI, TEST_SQ}; /* si = sym ind, ss = sym sym for first two */
typedef int (*pf_i_t)(s7_scheme *sc, s7_pointer x);
static s7_pf_t pipf_1(s7_scheme *sc, s7_pointer expr, s7_pf_t f1, s7_pf_t f2, s7_pf_t f3, pf_i_t tester)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_pair(cdddr(expr))) && (is_null(cddddr(expr))))
    {
      int choice;
      choice = tester(sc, expr);
      if ((choice == TEST_SS) || (choice == TEST_SI) ||
	  ((choice == TEST_NO_S) &&
	   (s7_arg_to_pf(sc, cadr(expr))) &&
	   (s7_arg_to_if(sc, caddr(expr)))))
	{
	  ptr_int loc;
	  loc = rc_loc(sc);
	  if (s7_arg_to_pf(sc, cadddr(expr)))
	    return((choice == TEST_SS) ?  f1 : ((choice == TEST_SI) ? f2 : f3));
	  sc->cur_rf->cur = rc_go(sc, loc);
	  if (s7_arg_to_gf(sc, cadddr(expr)))
	    return((choice == TEST_SS) ? f1 : ((choice == TEST_SI) ? f2 : f3));
	}
    }
  return(NULL);
}

#define PIPF_TO_PF(CName, F1, F2, Tester) \
  static s7_pointer CName ## _pf_slot(s7_scheme *sc, s7_pointer **rp) {return(pipf_pf_slot(sc, rp, F1));} \
  static s7_pointer CName ## _pf_s(s7_scheme *sc, s7_pointer **rp) {return(pipf_pf_s(sc, rp, F1));} \
  static s7_pointer CName ## _pf_seq(s7_scheme *sc, s7_pointer **rp) {return(pipf_pf_seq(sc, rp, F1));} \
  static s7_pointer CName ## _pf_a(s7_scheme *sc, s7_pointer **rp) {return(pipf_pf_a(sc, rp, F2));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(pipf_1(sc, expr, CName ## _pf_slot, CName ## _pf_s, CName ## _pf_a, Tester));}


/* -------- IF_TO_IF -------- */
typedef s7_int (*if_if_t)(s7_scheme *sc, s7_int x);
static s7_int if_if_1(s7_scheme *sc, s7_pointer **p, if_if_t fnc)
{
  s7_if_t f;
  s7_int x;
  f = (s7_if_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

static s7_if_t if_1(s7_scheme *sc, s7_pointer expr, s7_if_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && (s7_arg_to_if(sc, cadr(expr))))
    return(f);
  return(NULL);
}

#define IF_TO_IF(CName, Ifnc)			\
  static s7_int CName ## _if_i(s7_scheme *sc, s7_pointer **rp) {return(if_if_1(sc, rp, Ifnc));} \
  static s7_if_t CName ## _if(s7_scheme *sc, s7_pointer expr) {return(if_1(sc, expr, CName ## _if_i));}

#if (!WITH_GMP)

/* -------- IF2_TO_IF -------- */
typedef s7_int (*if2_if_t)(s7_scheme *sc, s7_int x, s7_int y);
static s7_int if2_if_1(s7_scheme *sc, s7_pointer **p, if2_if_t fnc)
{
  s7_if_t f;
  s7_int x, y;
  f = (s7_if_t)(**p); (*p)++;	
  x = f(sc, p);
  f = (s7_if_t)(**p); (*p)++;	
  y = f(sc, p);
  return(fnc(sc, x, y));
}

static s7_if_t if_2(s7_scheme *sc, s7_pointer expr, s7_if_t f)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_null(cdddr(expr))) &&
      (s7_arg_to_if(sc, cadr(expr))) &&
      (s7_arg_to_if(sc, caddr(expr))))
    return(f);
  return(NULL);
}

#define IF2_TO_IF(CName, Ifnc)						\
  static s7_int CName ## _if_i2(s7_scheme *sc, s7_pointer **rp) {return(if2_if_1(sc, rp, Ifnc));} \
  static s7_if_t CName ## _if(s7_scheme *sc, s7_pointer expr) {return(if_2(sc, expr, CName ## _if_i2));}


/* -------- IF_3_TO_IF -------- */

typedef s7_int (*if3_if_t)(s7_scheme *sc, s7_int x, s7_int y, s7_int z);
static s7_int if3_if_1(s7_scheme *sc, s7_pointer **p, if3_if_t fnc)
{
  s7_if_t f;
  s7_int x, y, z;
  f = (s7_if_t)(**p); (*p)++;	
  x = f(sc, p);
  f = (s7_if_t)(**p); (*p)++;	
  y = f(sc, p);
  f = (s7_if_t)(**p); (*p)++;	
  z = f(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_if_t if_3(s7_scheme *sc, s7_pointer expr, s7_if_t f1, s7_if_t f2, s7_if_t f3)
{
  if (!is_pair(cdr(expr))) return(NULL);
  if (!s7_arg_to_if(sc, cadr(expr))) return(NULL);
  if (is_null(cddr(expr))) return(f1);
  if (!s7_arg_to_if(sc, caddr(expr))) return(NULL);
  if (is_null(cdddr(expr))) return(f2);
  if (!s7_arg_to_if(sc, cadddr(expr))) return(NULL);
  if (is_null(cddddr(expr))) return(f3);
  return(NULL);
}

#define IF_3_TO_IF(CName, Ifnc1, Ifnc2, Ifnc3)				\
  static s7_int CName ## _if_i1(s7_scheme *sc, s7_pointer **rp) {return(if_if_1(sc, rp, Ifnc1));} \
  static s7_int CName ## _if_i2(s7_scheme *sc, s7_pointer **rp) {return(if2_if_1(sc, rp, Ifnc2));} \
  static s7_int CName ## _if_i3(s7_scheme *sc, s7_pointer **rp) {return(if3_if_1(sc, rp, Ifnc3));} \
  static s7_if_t CName ## _if(s7_scheme *sc, s7_pointer expr) {return(if_3(sc, expr, CName ## _if_i1, CName ## _if_i2, CName ## _if_i3));}
#endif /* gmp */


/* -------- IF_TO_PF -------- */
typedef s7_pointer (*if_pf_t)(s7_scheme *sc, s7_int x);
static s7_pointer if_p_1(s7_scheme *sc, s7_pointer **p, if_pf_t fnc)
{
  s7_if_t f;
  s7_int x;
  f = (s7_if_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

static s7_pf_t if_pf_1(s7_scheme *sc, s7_pointer expr, s7_pf_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && (s7_arg_to_if(sc, cadr(expr))))
    return(f);
  return(NULL);
}

#define IF_TO_PF(CName, Ifnc)			\
  static s7_pointer CName ## _pf_i(s7_scheme *sc, s7_pointer **rp) {return(if_p_1(sc, rp, Ifnc));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(if_pf_1(sc, expr, CName ## _pf_i));}


/* -------- PF_TO_IF -------- */
typedef s7_int (*pf_if_t)(s7_scheme *sc, s7_pointer x);
static s7_int pf_i_1(s7_scheme *sc, s7_pointer **p, pf_if_t fnc)
{
  s7_pf_t f;
  s7_pointer x;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

static s7_if_t pf_if_1(s7_scheme *sc, s7_pointer expr, s7_if_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && (s7_arg_to_pf(sc, cadr(expr))))
    return(f);
  return(NULL);
}

#define PF_TO_IF(CName, Pfnc)			\
  static s7_int CName ## _if_p(s7_scheme *sc, s7_pointer **rp) {return(pf_i_1(sc, rp, Pfnc));} \
  static s7_if_t CName ## _if(s7_scheme *sc, s7_pointer expr) {return(pf_if_1(sc, expr, CName ## _if_p));}


/* -------- PF_TO_RF -------- */
typedef s7_double (*pf_rf_t)(s7_scheme *sc, s7_pointer x);
static s7_double pf_r_1(s7_scheme *sc, s7_pointer **p, pf_rf_t fnc)
{
  s7_pf_t f;
  s7_pointer x;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

static s7_rf_t pf_rf_1(s7_scheme *sc, s7_pointer expr, s7_rf_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && (s7_arg_to_rf(sc, cadr(expr))))
    return(f);
  return(NULL);
}

#define PF_TO_RF(CName, Pfnc)			\
  static s7_double CName ## _rf_p(s7_scheme *sc, s7_pointer **rp) {return(pf_r_1(sc, rp, Pfnc));} \
  static s7_rf_t CName ## _rf(s7_scheme *sc, s7_pointer expr) {return(pf_rf_1(sc, expr, CName ## _rf_p));}


#if (!WITH_GMP)

/* -------- RF_TO_IF -------- */
typedef s7_int (*rf_if_t)(s7_scheme *sc, s7_double x);
static s7_int rf_i_1(s7_scheme *sc, s7_pointer **p, rf_if_t fnc)
{
  s7_rf_t f;
  s7_double x;
  f = (s7_rf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

static s7_if_t rf_if_1(s7_scheme *sc, s7_pointer expr, s7_if_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && (s7_arg_to_rf(sc, cadr(expr))))
    return(f);
  return(NULL);
}

#define RF_TO_IF(CName, Rfnc)			\
  static s7_int CName ## _if_r(s7_scheme *sc, s7_pointer **rp) {return(rf_i_1(sc, rp, Rfnc));} \
  static s7_if_t CName ## _if(s7_scheme *sc, s7_pointer expr) {return(rf_if_1(sc, expr, CName ## _if_r));}

#endif /* gmp */

/* -------- RF_TO_PF -------- */
typedef s7_pointer (*rf_pf_t)(s7_scheme *sc, s7_double x);
static s7_pointer rf_p_1(s7_scheme *sc, s7_pointer **p, rf_pf_t fnc)
{
  s7_rf_t f;
  s7_double x;
  f = (s7_rf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

#if (!WITH_GMP)

static s7_pf_t rf_pf_1(s7_scheme *sc, s7_pointer expr, s7_pf_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && (s7_arg_to_rf(sc, cadr(expr))))
    return(f);
  return(NULL);
}

#define RF_TO_PF(CName, Pfnc)			\
  static s7_pointer CName ## _pf_r(s7_scheme *sc, s7_pointer **rp) {return(rf_p_1(sc, rp, Pfnc));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(rf_pf_1(sc, expr, CName ## _pf_r));}


/* -------- RF_TO_RF -------- */
typedef s7_double (*rf_rf_t)(s7_scheme *sc, s7_double x);
static s7_double rf_rf_1(s7_scheme *sc, s7_pointer **p, rf_rf_t fnc)
{
  s7_rf_t f;
  s7_double x;
  f = (s7_rf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(fnc(sc, x));
}

static s7_rf_t rf_1(s7_scheme *sc, s7_pointer expr, s7_rf_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && (s7_arg_to_rf(sc, cadr(expr))))
    return(f);
  return(NULL);
}

#define RF_TO_RF(CName, Rfnc)						\
  static s7_double CName ## _rf_r(s7_scheme *sc, s7_pointer **rp) {return(rf_rf_1(sc, rp, Rfnc));} \
  static s7_rf_t CName ## _rf(s7_scheme *sc, s7_pointer expr) {return(rf_1(sc, expr, CName ## _rf_r));}

#define DIRECT_RF_TO_RF(CName) \
  static s7_double CName ## _rf_r(s7_scheme *sc, s7_pointer **p) {s7_rf_t f; s7_double x; f = (s7_rf_t)(**p); (*p)++; x = f(sc, p); return(CName(x));} \
  static s7_rf_t CName ## _rf(s7_scheme *sc, s7_pointer expr) {if (s7_arg_to_rf(sc, s7_cadr(expr))) return(CName ## _rf_r); return(NULL);}



/* -------- RF2_TO_RF -------- */
typedef s7_double (*rf2_rf_t)(s7_scheme *sc, s7_double x, s7_double y);
static s7_double rf2_rf_1(s7_scheme *sc, s7_pointer **p, rf2_rf_t fnc)
{
  s7_rf_t f;
  s7_double x, y;
  f = (s7_rf_t)(**p); (*p)++;	
  x = f(sc, p);
  f = (s7_rf_t)(**p); (*p)++;	
  y = f(sc, p);
  return(fnc(sc, x, y));
}

static s7_rf_t rf_2(s7_scheme *sc, s7_pointer expr, s7_rf_t f)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))) && 
      (s7_arg_to_rf(sc, cadr(expr))) &&
      (s7_arg_to_rf(sc, caddr(expr))))
    return(f);
  return(NULL);
}

#define RF2_TO_RF(CName, Rfnc)						\
  static s7_double CName ## _rf_r2(s7_scheme *sc, s7_pointer **rp) {return(rf2_rf_1(sc, rp, Rfnc));} \
  static s7_rf_t CName ## _rf(s7_scheme *sc, s7_pointer expr) {return(rf_2(sc, expr, CName ## _rf_r2));}


/* -------- RF_3_TO_RF -------- */

typedef s7_double (*rf3_rf_t)(s7_scheme *sc, s7_double x, s7_double y, s7_double z);
static s7_double rf3_rf_1(s7_scheme *sc, s7_pointer **p, rf3_rf_t fnc)
{
  s7_rf_t f;
  s7_double x, y, z;
  f = (s7_rf_t)(**p); (*p)++;	
  x = f(sc, p);
  f = (s7_rf_t)(**p); (*p)++;	
  y = f(sc, p);
  f = (s7_rf_t)(**p); (*p)++;	
  z = f(sc, p);
  return(fnc(sc, x, y, z));
}

static s7_rf_t rf_3(s7_scheme *sc, s7_pointer expr, s7_rf_t f1, s7_rf_t f2, s7_rf_t f3)
{
  if (!is_pair(cdr(expr))) return(NULL);
  if (!s7_arg_to_rf(sc, cadr(expr))) return(NULL);
  if (is_null(cddr(expr))) return(f1);
  if (!s7_arg_to_rf(sc, caddr(expr))) return(NULL);
  if (is_null(cdddr(expr))) return(f2);
  if (!s7_arg_to_rf(sc, cadddr(expr))) return(NULL);
  if (is_null(cddddr(expr))) return(f3);
  return(NULL);
}

#define RF_3_TO_RF(CName, Rfnc1, Rfnc2, Rfnc3)				\
  static s7_double CName ## _rf_r1(s7_scheme *sc, s7_pointer **rp) {return(rf_rf_1(sc, rp, Rfnc1));} \
  static s7_double CName ## _rf_r2(s7_scheme *sc, s7_pointer **rp) {return(rf2_rf_1(sc, rp, Rfnc2));} \
  static s7_double CName ## _rf_r3(s7_scheme *sc, s7_pointer **rp) {return(rf3_rf_1(sc, rp, Rfnc3));} \
  static s7_rf_t CName ## _rf(s7_scheme *sc, s7_pointer expr) {return(rf_3(sc, expr, CName ## _rf_r1, CName ## _rf_r2, CName ## _rf_r3));}


/* -------- R_P_F_TO_PF -------- */
static s7_pf_t rpf_pf_1(s7_scheme *sc, s7_pointer expr, s7_pf_t fnc1, s7_pf_t fnc2, s7_pf_t fnc3)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))))
    {
      ptr_int loc;
      loc = rc_loc(sc);
      if (s7_arg_to_rf(sc, cadr(expr))) return(fnc1);
      sc->cur_rf->cur = rc_go(sc, loc);
      if (s7_arg_to_pf(sc, cadr(expr)))	return(fnc2);
      sc->cur_rf->cur = rc_go(sc, loc);
      if (s7_arg_to_gf(sc, cadr(expr)))	return(fnc3);
    }
  return(NULL);
}

#define R_P_F_TO_PF(CName, PFnc1, PFnc2, PFnc3)				\
  static s7_pointer CName ## _pf_r(s7_scheme *sc, s7_pointer **rp) {return(rf_p_1(sc, rp, PFnc1));} \
  static s7_pointer CName ## _pf_p(s7_scheme *sc, s7_pointer **rp) {return(pf_pf_1(sc, rp, PFnc2));} \
  static s7_pointer CName ## _pf_g(s7_scheme *sc, s7_pointer **rp) {return(pf_pf_1(sc, rp, PFnc3));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(rpf_pf_1(sc, expr, CName ## _pf_r, CName ## _pf_p, CName ## _pf_g));}

#endif /* gmp */

/* -------- XF_TO_PF -------- */
static s7_pf_t xf_pf_1(s7_scheme *sc, s7_pointer expr, s7_pf_t f1, s7_pf_t f2, s7_pf_t f3)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))))
    {
      ptr_int loc;
      loc = rc_loc(sc);
      if (s7_arg_to_if(sc, cadr(expr))) return(f1);
      sc->cur_rf->cur = rc_go(sc, loc);
      if (s7_arg_to_rf(sc, cadr(expr))) return(f2);
      sc->cur_rf->cur = rc_go(sc, loc);
      if (s7_arg_to_pf(sc, cadr(expr))) return(f3);
    }
  return(NULL);
}

#define XF_TO_PF(CName, PFnc1, PFnc2, PFnc3)					\
  static s7_pointer CName ## _pf_i(s7_scheme *sc, s7_pointer **rp) {return(if_p_1(sc, rp, PFnc1));} \
  static s7_pointer CName ## _pf_r(s7_scheme *sc, s7_pointer **rp) {return(rf_p_1(sc, rp, PFnc2));} \
  static s7_pointer CName ## _pf_p(s7_scheme *sc, s7_pointer **rp) {return(pf_pf_1(sc, rp, PFnc3));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) {return(xf_pf_1(sc, expr, CName ## _pf_i, CName ## _pf_r, CName ## _pf_p));}


/* -------- XF2_TO_PF -------- */
typedef s7_pointer (*if2_pf_t)(s7_scheme *sc, s7_int x, s7_int y);
typedef s7_pointer (*rf2_pf_t)(s7_scheme *sc, s7_double x, s7_double y);
static s7_pointer if2_pf_1(s7_scheme *sc, s7_pointer **p, if2_pf_t fnc)
{
  s7_if_t f;
  s7_int x, y;
  f = (s7_if_t)(**p); (*p)++;	x = f(sc, p);
  f = (s7_if_t)(**p); (*p)++;	y = f(sc, p);
  return(fnc(sc, x, y));
}

static s7_pointer rf2_pf_1(s7_scheme *sc, s7_pointer **p, rf2_pf_t fnc)
{
  s7_rf_t f;
  s7_double x, y;
  f = (s7_rf_t)(**p); (*p)++;	x = f(sc, p);
  f = (s7_rf_t)(**p); (*p)++;	y = f(sc, p);
  return(fnc(sc, x, y));
}

static s7_pointer rf2_pf_sc(s7_scheme *sc, s7_pointer **p, rf2_pf_t fnc)
{
  s7_pointer xp, yp;
  (*p)++; 
  xp = slot_value(**p); (*p) += 2;
  yp = (**p); (*p)++;
  if ((is_t_real(xp)) && (is_t_real(yp)))
    return(fnc(sc, real(xp), real(yp)));
  return(fnc(sc, s7_number_to_real(sc, xp), s7_number_to_real(sc, yp)));
}

static s7_pf_t xf2_pf_1(s7_scheme *sc, s7_pointer expr, s7_pf_t f1, s7_pf_t f2, s7_pf_t f3, s7_pf_t f4, s7_pf_t f5)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_null(cdddr(expr))))
    {
      ptr_int loc;
      s7_pointer a1, a2;
      a1 = cadr(expr);
      a2 = caddr(expr);
      if ((is_symbol(a1)) && (is_symbol(a2)))
	{
	  a1 = s7_slot(sc, a1);
	  if (!is_slot(a1)) return(NULL);
	  s7_xf_store(sc, a1);
	  a2 = s7_slot(sc, a2);
	  if (!is_slot(a2)) return(NULL);
	  s7_xf_store(sc, a2);
	  return(f5);
	}
      loc = rc_loc(sc);
      if ((s7_arg_to_if(sc, a1)) && (s7_arg_to_if(sc, a2))) return(f1);
      sc->cur_rf->cur = rc_go(sc, loc);
      if ((s7_arg_to_rf(sc, a1)) && (s7_arg_to_rf(sc, a2))) return(((is_symbol(a1)) && (is_real(a2))) ? f3 : f2);
      sc->cur_rf->cur = rc_go(sc, loc);	
      if ((s7_arg_to_pf(sc, a1)) && (s7_arg_to_pf(sc, a2))) return(f4);
    }
  return(NULL);
}

#define XF2_TO_PF(CName, PFnc1, PFnc2, PFnc3)					\
  static s7_pointer CName ## _pf_i2(s7_scheme *sc, s7_pointer **rp) {return(if2_pf_1(sc, rp, PFnc1));} \
  static s7_pointer CName ## _pf_r2(s7_scheme *sc, s7_pointer **rp) {return(rf2_pf_1(sc, rp, PFnc2));} \
  static s7_pointer CName ## _pf_r2_sc(s7_scheme *sc, s7_pointer **rp) {return(rf2_pf_sc(sc, rp, PFnc2));} \
  static s7_pointer CName ## _pf_p2(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_1(sc, rp, PFnc3));} \
  static s7_pointer CName ## _pf_ss(s7_scheme *sc, s7_pointer **rp) {return(pf2_pf_ss(sc, rp, PFnc3));} \
  static s7_pf_t CName ## _pf(s7_scheme *sc, s7_pointer expr) \
  {\
    return(xf2_pf_1(sc, expr, CName ## _pf_i2, CName ## _pf_r2, CName ## _pf_r2_sc, CName ## _pf_p2, CName ## _pf_ss));	\
  }

#if WITH_OPTIMIZATION
static s7_pointer if_pf_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t test, t;
  s7_pointer val;
  ptr_int e1;
  
  test = (s7_pf_t)(**p); (*p)++;
  t = (s7_pf_t)(**p); (*p)++;
  e1 = (ptr_int)(**p); (*p)++;

  val = test(sc, p);
  if (val != sc->F)
    val = t(sc, p);
  else val = sc->unspecified;
  (*p) = rc_go(sc, e1);

  return(val);
}

static s7_pointer if_pf_not_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t test, t;
  s7_pointer val;
  ptr_int e1; 
  
  test = (s7_pf_t)(**p); (*p)++;
  t = (s7_pf_t)(**p); (*p)++;
  e1 = (ptr_int)(**p); (*p)++;

  val = test(sc, p);
  if (val == sc->F)
    val = t(sc, p);
  else val = sc->unspecified;
  (*p) = rc_go(sc, e1);

  return(val);
}

#if (!WITH_GMP)
static s7_pointer equal_p2(s7_scheme *sc, s7_pointer **p);
#endif
static s7_pointer c_equal_2(s7_scheme *sc, s7_pointer x, s7_pointer y);

static s7_pointer if_pf_not_equal_2(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t t, eq2;
  s7_pointer val, x, y;
  ptr_int e1; 
  
  (*p)++;
  t = (s7_pf_t)(**p); (*p)++;
  e1 = (ptr_int)(**p); (*p)++;

  eq2 = (s7_pf_t)(**p); (*p)++;	
  x = eq2(sc, p);
  eq2 = (s7_pf_t)(**p); (*p)++;	
  y = eq2(sc, p);

  if (c_equal_2(sc, x, y) == sc->F)
    val = t(sc, p);
  else val = sc->unspecified;
  (*p) = rc_go(sc, e1);

  return(val);
}

static s7_pointer if_pf_xxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer x;
  s7_pf_t r1, r2;
  s7_pf_t pf;
  s7_pointer val;
  ptr_int e1, e2;

  pf = (s7_pf_t)(**p); (*p)++;
  r1 = (s7_pf_t)(**p); (*p)++;
  e1 = (ptr_int)(**p); (*p)++;
  r2 = (s7_pf_t)(**p); (*p)++;
  e2 = (ptr_int)(**p); (*p)++;

  val = pf(sc, p);
  if (val != sc->F)
    {
      x = r1(sc, p);
      (*p) = rc_go(sc, e2);
    }
  else 
    {
      (*p) = rc_go(sc, e1);
      x = r2(sc, p);
    }
  return(x);
}

static s7_pf_t if_pf(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer test, t, f = NULL;
  s7_int test_loc, t_loc, f_loc = 0, e1_loc, e2_loc = 0;
  bool not_case = false;
  ptr_int loc;
  xf_t *rc;

  if ((is_null(cdr(expr))) || (is_null(cddr(expr)))) return(NULL);
  test = cadr(expr);
  if ((is_pair(test)) && (car(test) == sc->not_symbol))
    {
      not_case = true;
      test = cadr(test);
    }
  t = caddr(expr);

  xf_init(5);
  xf_save_loc3(test_loc, t_loc, e1_loc);

  if (is_pair(cdddr(expr)))
    {
      f = cadddr(expr);
      xf_save_loc2(f_loc, e2_loc);
    }

  if (!arg_to_pf(sc, test, test_loc)) return(NULL);
  loc = rc_loc(sc);
  if (!arg_to_pf(sc, t, t_loc))
    {
      sc->cur_rf->cur = rc_go(sc, loc);
      if (!arg_to_if(sc, t, t_loc)) return(NULL);
    }
  xf_store_at(e1_loc, (s7_pointer)rc_loc(sc));

  if (f)
    {
      if (!arg_to_pf(sc, f, f_loc)) return(NULL);
      xf_store_at(e2_loc, (s7_pointer)rc_loc(sc));
    }

  if (!f)
    {
      if (not_case) 
	{
#if (!WITH_GMP)
	  if ((s7_pointer)equal_p2 == sc->cur_rf->data[test_loc])
	    return(if_pf_not_equal_2);
#endif
	  return(if_pf_not_xx);
	}
      return(if_pf_xx);
    }
  return(if_pf_xxx);
}


static s7_double if_rf_xxx(s7_scheme *sc, s7_pointer **p)
{
  s7_double x;
  s7_rf_t r1, r2;
  s7_pf_t pf;
  s7_pointer val;
  ptr_int e1, e2;
  
  pf = (s7_pf_t)(**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  r2 = (s7_rf_t)(**p); (*p)++;
  e1 = (ptr_int)(**p); (*p)++;
  e2 = (ptr_int)(**p); (*p)++;

  val = pf(sc, p);
  if (val != sc->F)
    {
      x = r1(sc, p);
      (*p) = rc_go(sc, e2);
    }
  else 
    {
      (*p) = rc_go(sc, e1);
      x = r2(sc, p);
    }
  return(x);
}

static s7_rf_t if_rf(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer test, t, f;
  s7_int test_loc, t_loc, f_loc = 0, e1_loc = 0, e2_loc;
  xf_t *rc;

  if ((is_null(cdr(expr))) || (is_null(cddr(expr))) || (is_null(cdddr(expr)))) return(NULL);
  test = cadr(expr);
  t = caddr(expr);
  f = cadddr(expr);
  xf_init(5);

  xf_save_loc3(test_loc, t_loc, f_loc);
  xf_save_loc2(e1_loc, e2_loc);

  if (!arg_to_pf(sc, test, test_loc)) return(NULL);
  if (!arg_to_rf(sc, t, t_loc)) return(NULL);
  xf_store_at(e1_loc, (s7_pointer)rc_loc(sc));
  if (!arg_to_rf(sc, f, f_loc)) return(NULL);
  xf_store_at(e2_loc, (s7_pointer)rc_loc(sc));

  return(if_rf_xxx);
}

static s7_pointer quote_pf_s(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s;
  s = **p; (*p)++;
  return(s);
}

static s7_pf_t quote_pf(s7_scheme *sc, s7_pointer expr)
{
  if (is_symbol(cadr(expr)))
    {
      xf_t *rc;
      xf_init(1);
      xf_store(cadr(expr));
      return(quote_pf_s);
    }
  return(NULL);
}

static s7_pointer or_pf_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t pf1, pf2;
  ptr_int e1;
  s7_pointer val;

  pf1 = (s7_pf_t)(**p); (*p)++;
  pf2 = (s7_pf_t)(**p); (*p)++;
  e1 = (ptr_int)(**p); (*p)++;

  val = pf1(sc, p);
  if (val != sc->F)
    {
      (*p) = rc_go(sc, e1);
      return(val);
    }
  return(pf2(sc, p));
}

static s7_pf_t or_pf(s7_scheme *sc, s7_pointer expr)
{
  int len;
  len = s7_list_length(sc, expr);
  if (len == 3)
    {
      int loc1, loc2, eloc;
      xf_t *rc;
      xf_init(3);
      xf_save_loc3(loc1, loc2, eloc);

      if (!arg_to_pf(sc, cadr(expr), loc1)) return(NULL);
      if (!arg_to_pf(sc, caddr(expr), loc2)) return(NULL);
      xf_store_at(eloc, (s7_pointer)rc_loc(sc));

      return(or_pf_xx);
    }
  return(NULL);
}

static s7_pointer and_pf_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t pf1, pf2;
  ptr_int e1;

  pf1 = (s7_pf_t)(**p); (*p)++;
  pf2 = (s7_pf_t)(**p); (*p)++;
  e1 = (ptr_int)(**p); (*p)++;

  if (pf1(sc, p) == sc->F) 
    {
      (*p) = rc_go(sc, e1);
      return(sc->F);
    }
  return(pf2(sc, p));
}

static s7_pf_t and_pf(s7_scheme *sc, s7_pointer expr)
{
  int len;
  len = s7_list_length(sc, expr);
  if (len == 3)
    {
      s7_int loc1, loc2, eloc;
      xf_t *rc;
      xf_init(3);
      xf_save_loc3(loc1, loc2, eloc);

      if (!arg_to_pf(sc, cadr(expr), loc1)) return(NULL);
      if (!arg_to_pf(sc, caddr(expr), loc2)) return(NULL);
      xf_store_at(eloc, (s7_pointer)rc_loc(sc));

      return(and_pf_xx);
    }
  return(NULL);
}
#endif


/* -------------------------------- continuations and gotos -------------------------------- */

static s7_pointer g_is_continuation(s7_scheme *sc, s7_pointer args)
{
  #define H_is_continuation "(continuation? obj) returns #t if obj is a continuation"
  #define Q_is_continuation pl_bt

  check_boolean_method(sc, is_continuation, sc->is_continuation_symbol, args);
  /* is this the right thing?  It returns #f for call-with-exit ("goto") because
   *   that form of continuation can't continue (via a jump back to its context).
   * how to recognize the call-with-exit function?  "goto" is an internal name.
   */
}


static s7_pointer protected_list_copy(s7_scheme *sc, s7_pointer a)
{
  s7_pointer slow, fast, p;

  sc->w = cons(sc, car(a), sc->nil);
  p = sc->w;

  slow = fast = cdr(a);
  while (true)
    {
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    return(sc->w);
	  set_cdr(p, fast);
	  return(sc->w);
	}

      set_cdr(p, cons(sc, car(fast), sc->nil));
      p = cdr(p);

      fast = cdr(fast);
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    return(sc->w);
	  set_cdr(p, fast);
	  return(sc->w);
	}
      /* if unrolled further, it's a lot slower? */
      set_cdr(p, cons(sc, car(fast), sc->nil));
      p = cdr(p);

      fast = cdr(fast);
      slow = cdr(slow);
      if (fast == slow)
	{
	  /* try to preserve the original cyclic structure */
	  s7_pointer p1, f1, p2, f2;
	  set_match_pair(a);
	  for (p1 = sc->w, f1 = a; !(is_matched_pair(cdr(f1))); f1 = cdr(f1), p1 = cdr(p1))
	    set_match_pair(f1);
	  for (p2 = sc->w, f2 = a; cdr(f1) != f2; f2 = cdr(f2), p2 = cdr(p2))
	    clear_match_pair(f2);
	  for (f1 = f2; is_pair(f1); f1 = cdr(f1), f2 = cdr(f2))
	    {
	      clear_match_pair(f1);
	      f1 = cdr(f1);
	      clear_match_pair(f1);
	      if (f1 == f2) break;
	    }
	  if (is_null(p1))
	    set_cdr(p2, p2);
	  else set_cdr(p1, p2);
	  return(sc->w);
	}
    }
  return(sc->w);
}


static s7_pointer copy_counter(s7_scheme *sc, s7_pointer obj)
{
  s7_pointer nobj;
  new_cell(sc, nobj, T_COUNTER);
  counter_set_result(nobj, counter_result(obj));
  counter_set_list(nobj, counter_list(obj));
  counter_set_capture(nobj, counter_capture(obj));
  counter_set_let(nobj, counter_let(obj));
  counter_set_slots(nobj, counter_slots(obj));
  return(nobj);
}


static s7_pointer copy_stack(s7_scheme *sc, s7_pointer old_v, int top)
{
  #define CC_INITIAL_STACK_SIZE 256 /* 128 is too small here */
  int i, len;
  s7_pointer new_v;
  s7_pointer *nv, *ov;

  /* stacks can grow temporarily, so sc->stack_size grows, but we don't normally need all that
   *   leftover space here, so choose the original stack size if it's smaller.
   */
  len = vector_length(old_v);
  if (len > CC_INITIAL_STACK_SIZE)
    {
      if (top < CC_INITIAL_STACK_SIZE / 4)
	len = CC_INITIAL_STACK_SIZE;
    }
  else
    {
      if (len < CC_INITIAL_STACK_SIZE)
	len = CC_INITIAL_STACK_SIZE;
    }
  if ((int)(sc->free_heap_top - sc->free_heap) < (int)(sc->heap_size / 4)) gc(sc);
  /* this gc call is needed if there are lots of call/cc's -- by pure bad luck
   *   we can end up hitting the end of the gc free list time after time while
   *   in successive copy_stack's below, causing s7 to core up until it runs out of memory.
   */

  new_v = make_vector_1(sc, len, NOT_FILLED, T_VECTOR);
  set_type(new_v, T_STACK);
  temp_stack_top(new_v) = top;
  nv = vector_elements(new_v);
  ov = vector_elements(old_v);
  if (len > 0)
    memcpy((void *)nv, (void *)ov, len * sizeof(s7_pointer));

  s7_gc_on(sc, false);
  for (i = 2; i < top; i += 4)
    {
      s7_pointer p;
      p = ov[i];                            /* args */
      if (is_pair(p))                       /* args need not be a list (it can be a port or #f, etc) */
	nv[i] = protected_list_copy(sc, p); /* args (copy is needed -- see s7test.scm) */
      /* lst can be dotted or circular here.  The circular list only happens in a case like:
       *    (dynamic-wind (lambda () (eq? (let ((lst (cons 1 2))) (set-cdr! lst lst) lst) (call/cc (lambda (k) k)))) (lambda () #f) (lambda () #f))
       */
      else
	{
	  if (is_counter(p))              /* these can only occur in this context */
	    nv[i] = copy_counter(sc, p);
	}
    }
  s7_gc_on(sc, true);
  return(new_v);
}


static s7_pointer make_goto(s7_scheme *sc)
{
  s7_pointer x;
  new_cell(sc, x, T_GOTO | T_PROCEDURE);
  call_exit_goto_loc(x) = s7_stack_top(sc);
  call_exit_op_loc(x) = (int)(sc->op_stack_now - sc->op_stack);
  call_exit_active(x) = true;
  return(x);
}


static s7_pointer *copy_op_stack(s7_scheme *sc)
{
  int len;
  s7_pointer *ops;
  ops = (s7_pointer *)malloc(sc->op_stack_size * sizeof(s7_pointer));
  len = (int)(sc->op_stack_now - sc->op_stack);
  if (len > 0)
    memcpy((void *)ops, (void *)(sc->op_stack), len * sizeof(s7_pointer));
  return(ops);
}


/* (with-baffle . body) calls body guaranteeing that there can be no jumps into the
 *    middle of it from outside -- no outer evaluation of a continuation can jump across this
 *    barrier:  The flip-side of call-with-exit.
 *    It sets a T_BAFFLE var in a new env, that has a unique key.  Call/cc then always
 *    checks the env chain for any such variable, saving the localmost.  Apply of a continuation
 *    looks for such a saved variable, if none, go ahead, else check the current env (before the
 *    jump) for that variable.  If none, error, else go ahead.  This is different from a delimited
 *    continuation which simply delimits the extent of the continuation (why not use lambda?) -- we want to block it
 *    from coming at us from some unknown place.
 */

static s7_pointer make_baffle(s7_scheme *sc)
{
  s7_pointer x;
  new_cell(sc, x, T_BAFFLE);
  baffle_key(x) = sc->baffle_ctr++;
  return(x);
}


static bool find_baffle(s7_scheme *sc, int key)
{
  /* search backwards through sc->envir for sc->baffle_symbol with key as value
   */
  s7_pointer x, y;
  for (x = sc->envir; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if ((slot_symbol(y) == sc->baffle_symbol) &&
	  (baffle_key(slot_value(y)) == key))
	return(true);

  if ((is_slot(global_slot(sc->baffle_symbol))) &&
      (is_baffle(slot_value(global_slot(sc->baffle_symbol)))))
    return(baffle_key(slot_value(global_slot(sc->baffle_symbol))) == key);

  return(false);
}


static int find_any_baffle(s7_scheme *sc)
{
  /* search backwards through sc->envir for any sc->baffle_symbol
   */
  if (sc->baffle_ctr > 0)
    {
      s7_pointer x, y;
      for (x = sc->envir; is_let(x); x = outlet(x))
	for (y = let_slots(x); is_slot(y); y = next_slot(y))
	  if (slot_symbol(y) == sc->baffle_symbol)
	    return(baffle_key(slot_value(y)));

      if ((is_slot(global_slot(sc->baffle_symbol))) &&
	  (is_baffle(slot_value(global_slot(sc->baffle_symbol)))))
	return(baffle_key(slot_value(global_slot(sc->baffle_symbol))));
    }
  return(-1);
}


s7_pointer s7_make_continuation(s7_scheme *sc)
{
  s7_pointer x, stack;
  int loc;

  loc = s7_stack_top(sc);
  stack = copy_stack(sc, sc->stack, loc);
  sc->temp8 = stack;

  new_cell(sc, x, T_CONTINUATION | T_PROCEDURE);
  continuation_data(x) = (continuation_t *)malloc(sizeof(continuation_t));
  continuation_set_stack(x, stack);
  continuation_stack_size(x) = vector_length(continuation_stack(x));   /* copy_stack can return a smaller stack than the current one */
  continuation_stack_start(x) = vector_elements(continuation_stack(x));
  continuation_stack_end(x) = (s7_pointer *)(continuation_stack_start(x) + loc);
  continuation_op_stack(x) = copy_op_stack(sc);                        /* no heap allocation here */
  continuation_op_loc(x) = (int)(sc->op_stack_now - sc->op_stack);
  continuation_op_size(x) = sc->op_stack_size;
  continuation_key(x) = find_any_baffle(sc);

  add_continuation(sc, x);
  return(x);
}


static bool check_for_dynamic_winds(s7_scheme *sc, s7_pointer c)
{
  int i, s_base = 0, c_base = -1;
  opcode_t op;

  for (i = s7_stack_top(sc) - 1; i > 0; i -= 4)
    {
      op = stack_op(sc->stack, i);
      switch (op)
	{
	case OP_DYNAMIC_WIND:
	  {
	    s7_pointer x;
	    int j;
	    x = stack_code(sc->stack, i);
	    for (j = 3; j < continuation_stack_top(c); j += 4)
	      if ((stack_op(continuation_stack(c), j) == OP_DYNAMIC_WIND) &&
		  (x == stack_code(continuation_stack(c), j)))
		{
		  s_base = i;
		  c_base = j;
		  break;
		}

	    if (s_base != 0)
	      break;

	    if (dynamic_wind_state(x) == DWIND_BODY)
	      {
		dynamic_wind_state(x) = DWIND_FINISH;
		if (dynamic_wind_out(x) != sc->F)
		  {
		    push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);
		    sc->args = sc->nil;
		    sc->code = dynamic_wind_out(x);
		    eval(sc, OP_APPLY);
		  }
	      }
	  }
	  break;

	case OP_BARRIER:
	  if (i > continuation_stack_top(c))  /* otherwise it's some unproblematic outer eval-string? */
	    return(false);                    /*    but what if we've already evaluated a dynamic-wind closer? */
	  break;

	case OP_DEACTIVATE_GOTO:              /* here we're jumping out of an unrelated call-with-exit block */
	  if (i > continuation_stack_top(c))
	    call_exit_active(stack_args(sc->stack, i)) = false;
	  break;

	default:
	  break;
	}
    }

  for (i = c_base + 4; i < continuation_stack_top(c); i += 4)
    {
      op = stack_op(continuation_stack(c), i);

      if (op == OP_DYNAMIC_WIND)
	{
	  s7_pointer x;
	  x = stack_code(continuation_stack(c), i);
	  if (dynamic_wind_in(x) != sc->F)
	    {
	      /* this can cause an infinite loop if the call/cc is trying to jump back into
	       *   a dynamic-wind init function -- it's even possible to trick with-baffle!
	       *   I can't find any fool-proof way to catch this problem.
	       */
	      push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);
	      sc->args = sc->nil;
	      sc->code = dynamic_wind_in(x);
	      eval(sc, OP_APPLY);
	    }
	  dynamic_wind_state(x) = DWIND_BODY;
	}
      else
	{
	  if (op == OP_DEACTIVATE_GOTO)
	    call_exit_active(stack_args(continuation_stack(c), i)) = true;
	}
    }
  return(true);
}


static bool call_with_current_continuation(s7_scheme *sc)
{
  s7_pointer c;
  c = sc->code;

  /* check for (baffle ...) blocking the current attempt to continue */
  if ((continuation_key(c) >= 0) &&
      (!(find_baffle(sc, continuation_key(c))))) /* should this raise an error? */
    return(false);

  if (!check_for_dynamic_winds(sc, c)) /* if OP_BARRIER on stack deeper than continuation top(?), but can this happen? (it doesn't in s7test) */
    return(true);

  /* we push_stack sc->code before calling an embedded eval above, so sc->code should still be c here, etc
   */
  sc->stack = copy_stack(sc, continuation_stack(c), continuation_stack_top(c));
  sc->stack_size = continuation_stack_size(c);
  sc->stack_start = vector_elements(sc->stack);
  sc->stack_end = (s7_pointer *)(sc->stack_start + continuation_stack_top(c));
  sc->stack_resize_trigger = (s7_pointer *)(sc->stack_start + sc->stack_size / 2);

  {
    int i, top;
    top = continuation_op_loc(c);
    sc->op_stack_now = (s7_pointer *)(sc->op_stack + top);
    sc->op_stack_size = continuation_op_size(c);
    sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
    for (i = 0; i < top; i++)
      sc->op_stack[i] = continuation_op_stack(c)[i];
  }

  if (is_null(sc->args))
    sc->value = sc->nil;
  else
    {
      if (is_null(cdr(sc->args)))
	sc->value = car(sc->args);
      else sc->value = splice_in_values(sc, sc->args);
    }
  return(true);
}


static void call_with_exit(s7_scheme *sc)
{
  int i, new_stack_top, quit = 0;

  if (!call_exit_active(sc->code))
    {
      static s7_pointer call_with_exit_error = NULL;
      if (!call_with_exit_error)
	call_with_exit_error = s7_make_permanent_string("call-with-exit escape procedure called outside its block");
      s7_error(sc, sc->invalid_escape_function_symbol, set_elist_1(sc, call_with_exit_error));
    }

  call_exit_active(sc->code) = false;
  new_stack_top = call_exit_goto_loc(sc->code);
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + call_exit_op_loc(sc->code));

  /* look for dynamic-wind in the stack section that we are jumping out of */
  for (i = s7_stack_top(sc) - 1; i > new_stack_top; i -= 4)
    {
      opcode_t op;

      op = stack_op(sc->stack, i);
      switch (op)
	{
	case OP_DYNAMIC_WIND:
	  {
	    s7_pointer lx;
	    lx = stack_code(sc->stack, i);
	    if (dynamic_wind_state(lx) == DWIND_BODY)
	      {
		dynamic_wind_state(lx) = DWIND_FINISH;
		if (dynamic_wind_out(lx) != sc->F)
		  {
		    push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);
		    sc->args = sc->nil;
		    sc->code = dynamic_wind_out(lx);
		    eval(sc, OP_APPLY);
		  }
	      }
	  }
	  break;

	case OP_EVAL_STRING_2:
	  s7_close_input_port(sc, sc->input_port);
	  pop_input_port(sc);
	  break;

	case OP_BARRIER:                /* oops -- we almost certainly went too far */
	  return;

	case OP_DEACTIVATE_GOTO:        /* here we're jumping into an unrelated call-with-exit block */
	  call_exit_active(stack_args(sc->stack, i)) = false;
	  break;

	  /* call/cc does not close files, but I think call-with-exit should */
	case OP_GET_OUTPUT_STRING_1:
	case OP_UNWIND_OUTPUT:
	  {
	    s7_pointer x;
	    x = stack_code(sc->stack, i);                /* "code" = port that we opened */
	    s7_close_output_port(sc, x);
	    x = stack_args(sc->stack, i);                /* "args" = port that we shadowed, if not #f */
	    if (x != sc->F)
	      sc->output_port = x;
	  }
	  break;

	case OP_UNWIND_INPUT:
	  s7_close_input_port(sc, stack_code(sc->stack, i)); /* "code" = port that we opened */
	  sc->input_port = stack_args(sc->stack, i);         /* "args" = port that we shadowed */
	  break;

	case OP_EVAL_DONE: /* goto called in a method -- put off the inner eval return(s) until we clean up the stack */
	  quit++;
	  break;

	default:
	  break;
	}
    }

  sc->stack_end = (s7_pointer *)(sc->stack_start + new_stack_top);

  /* the return value should have an implicit values call, just as in call/cc */
  if (is_null(sc->args))
    sc->value = sc->nil;
  else
    {
      if (is_null(cdr(sc->args)))
	sc->value = car(sc->args);
      else sc->value = splice_in_values(sc, sc->args);
    }

  if (quit > 0)
    {
      if (sc->longjmp_ok)
	{
	  pop_stack(sc);
	  longjmp(sc->goto_start, CALL_WITH_EXIT_JUMP);
	}
      for (i = 0; i < quit; i++)
	push_stack(sc, OP_EVAL_DONE, sc->nil, sc->nil);
    }
}


static s7_pointer g_call_cc(s7_scheme *sc, s7_pointer args)
{
  #define H_call_cc "(call-with-current-continuation func) is always a mistake!"
  #define Q_call_cc s7_make_signature(sc, 2, sc->values_symbol, sc->is_procedure_symbol)
  /* I think the intent is that sc->values_symbol as the proc-sig return type indicates multiple values are possible (otherwise use #t). */

  s7_pointer p;
  p = car(args);                             /* this is the procedure passed to call/cc */
  if (!is_procedure(p))                      /* this includes continuations */
    {
      check_two_methods(sc, p, sc->call_cc_symbol, sc->call_with_current_continuation_symbol, args);
      return(simple_wrong_type_argument_with_type(sc, sc->call_cc_symbol, p, a_procedure_string));
    }
  if (!s7_is_aritable(sc, p, 1))
    return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, make_string_wrapper(sc, "call/cc procedure, ~A, should take one argument"), p)));

  sc->w = s7_make_continuation(sc);
  push_stack(sc, OP_APPLY, list_1(sc, sc->w), p);
  sc->w = sc->nil;

  return(sc->nil);
}

/* we can't naively optimize call/cc to call-with-exit if the continuation is only
 *   used as a function in the call/cc body because it might (for example) be wrapped
 *   in a lambda form that is being exported.  See b-func in s7test for an example.
 */


static s7_pointer g_call_with_exit(s7_scheme *sc, s7_pointer args)
{
  #define H_call_with_exit "(call-with-exit func) is call/cc without the ability to jump back into a previous computation."
  #define Q_call_with_exit s7_make_signature(sc, 2, sc->values_symbol, sc->is_procedure_symbol)

  s7_pointer p, x;
  /* (call-with-exit (lambda (return) ...)) */
  p = car(args);
  if (!is_procedure(p))                           /* this includes continuations */
    method_or_bust_with_type(sc, p, sc->call_with_exit_symbol, args, a_procedure_string, 0);

  x = make_goto(sc);
  push_stack(sc, OP_DEACTIVATE_GOTO, x, p); /* this means call-with-exit is not tail-recursive */
  push_stack(sc, OP_APPLY, cons_unchecked(sc, x, sc->nil), p);

  /* if the lambda body calls the argument as a function,
   *   it is applied to its arguments, apply notices that it is a goto, and...
   *
   *      (conceptually...) sc->stack_top = call_exit_goto_loc(sc->code);
   *      s_pop(sc, (is_not_null(sc->args)) ? car(sc->args) : sc->nil);
   *
   *   which jumps to the point of the goto returning car(args).
   *
   * There is one gotcha: we can't jump back in from outside, so if the caller saves the goto
   *   and tries to invoke it outside the call-with-exit block, we have to
   *   make sure it triggers an error.  So, if the escape is called, it then
   *   deactivates itself.  Otherwise the block returns, we pop to OP_DEACTIVATE_GOTO,
   *   and it finds the goto in sc->args.
   * Even worse:
   *
       (let ((cc #f))
         (call-with-exit
           (lambda (c3)
             (call/cc (lambda (ret) (set! cc ret)))
             (c3)))
         (cc))
   *
   * where we jump back into a call-with-exit body via call/cc, the goto has to be
   * re-established.
   *
   * I think call-with-exit could be based on catch, but it's a simpler notion,
   *   and certainly at the source level it is easier to read.
   */
  return(sc->nil);
}



/* -------------------------------- numbers -------------------------------- */

#if WITH_GMP
  static char *big_number_to_string_with_radix(s7_pointer p, int radix, int width, int *nlen, use_write_t use_write);
  static bool big_numbers_are_eqv(s7_pointer a, s7_pointer b);
  static s7_pointer string_to_either_integer(s7_scheme *sc, const char *str, int radix);
  static s7_pointer string_to_either_ratio(s7_scheme *sc, const char *nstr, const char *dstr, int radix);
  static s7_pointer string_to_either_real(s7_scheme *sc, const char *str, int radix);
  static s7_pointer string_to_either_complex(s7_scheme *sc, char *q, char *slash1, char *ex1, bool has_dec_point1,
					     char *plus, char *slash2, char *ex2, bool has_dec_point2, int radix, int has_plus_or_minus);
  static s7_pointer big_add(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_subtract(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_multiply(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_divide(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_random(s7_scheme *sc, s7_pointer args);
  static s7_pointer s7_int_to_big_integer(s7_scheme *sc, s7_int val);
  static s7_pointer s7_ratio_to_big_ratio(s7_scheme *sc, s7_int num, s7_int den);
  static s7_pointer s7_number_to_big_real(s7_scheme *sc, s7_pointer p);
  static s7_pointer promote_number(s7_scheme *sc, int type, s7_pointer x);
  static s7_pointer big_equal(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_negate(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_invert(s7_scheme *sc, s7_pointer args);
#if (!WITH_PURE_S7)
  static s7_pointer big_inexact_to_exact(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_exact_to_inexact(s7_scheme *sc, s7_pointer args);
#endif
  static s7_pointer mpz_to_big_integer(s7_scheme *sc, mpz_t val);
  static s7_pointer mpq_to_big_ratio(s7_scheme *sc, mpq_t val);
  static s7_pointer mpfr_to_big_real(s7_scheme *sc, mpfr_t val);
  static s7_pointer mpc_to_big_complex(s7_scheme *sc, mpc_t val);
#endif

#define HAVE_OVERFLOW_CHECKS ((defined(__clang__) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4))) || \
                              (defined(__GNUC__) && __GNUC__ >= 5))

#if (defined(__clang__) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4))) 
  #define subtract_overflow(A, B, C)     __builtin_ssubll_overflow(A, B, C)
  #define add_overflow(A, B, C)          __builtin_saddll_overflow(A, B, C)
  #define multiply_overflow(A, B, C)     __builtin_smulll_overflow(A, B, C)
  #define int_subtract_overflow(A, B, C) __builtin_ssub_overflow(A, B, C)
  #define int_add_overflow(A, B, C)      __builtin_sadd_overflow(A, B, C)
  #define int_multiply_overflow(A, B, C) __builtin_smul_overflow(A, B, C)
#else
#if (defined(__GNUC__) && __GNUC__ >= 5)
  #define subtract_overflow(A, B, C)     __builtin_sub_overflow(A, B, C)
  #define add_overflow(A, B, C)          __builtin_add_overflow(A, B, C)
  #define multiply_overflow(A, B, C)     __builtin_mul_overflow(A, B, C)
  #define int_subtract_overflow(A, B, C) __builtin_sub_overflow(A, B, C)
  #define int_add_overflow(A, B, C)      __builtin_add_overflow(A, B, C)
  #define int_multiply_overflow(A, B, C) __builtin_mul_overflow(A, B, C)
#endif
#endif


#define s7_int_abs(x) ((x) >= 0 ? (x) : -(x))
/* can't use abs even in gcc -- it doesn't work with long long ints! */

#if (!__NetBSD__)
  #define s7_fabsl(X) fabsl(X)
#else
  static double s7_fabsl(long double x) {if (x < 0.0) return(-x);  return(x);}
#endif


static bool is_NaN(s7_double x) {return(x != x);}
/* callgrind says this is faster than isnan, I think (very confusing data...) */


#if defined(__sun) && defined(__SVR4)
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));} /* there's no isinf in Solaris */
#else
#if (!MS_WINDOWS)

  #if __cplusplus
    #define is_inf(x) std::isinf(x)
  #else
    #define is_inf(x) isinf(x)
  #endif

#else
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));}  /* Another possibility: (x * 0) != 0 */

  /* in MS C, we need to provide inverse hyperbolic trig funcs and cbrt */
  static double asinh(double x) {return(log(x + sqrt(1.0 + x * x)));}
  static double acosh(double x) {return(log(x + sqrt(x * x - 1.0)));}
  /* perhaps less prone to numerical troubles (untested): 2.0 * log(sqrt(0.5 * (x + 1.0)) + sqrt(0.5 * (x - 1.0))) */
  static double atanh(double x) {return(log((1.0 + x) / (1.0 - x)) / 2.0);}
  static double cbrt(double x) {if (x >= 0.0) return(pow(x, 1.0 / 3.0)); return(-pow(-x, 1.0 / 3.0));}
#endif /* windows */
#endif /* sun */


/* for g_log, we also need round. this version is from stackoverflow, see also round_per_R5RS below */
double s7_round(double number) {return((number < 0.0) ? ceil(number - 0.5) : floor(number + 0.5));}

#if HAVE_COMPLEX_NUMBERS
#if __cplusplus
  #define _Complex_I (complex<s7_double>(0.0, 1.0))
  #define creal(x) Real(x)
  #define cimag(x) Imag(x)
  #define carg(x) arg(x)
  #define cabs(x) abs(x)
  #define csqrt(x) sqrt(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define cexp(x) exp(x)
  #define csin(x) sin(x)
  #define ccos(x) cos(x)
  #define csinh(x) sinh(x)
  #define ccosh(x) cosh(x)
#else
  typedef double complex s7_complex;
#endif


#if (!HAVE_COMPLEX_TRIG)
#if (__cplusplus)

  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-_Complex_I * clog(_Complex_I * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-_Complex_I * clog(z + _Complex_I * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(_Complex_I * clog((_Complex_I + z) / (_Complex_I - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
#else

/* still not in FreeBSD! */
static s7_complex clog(s7_complex z) {return(log(fabs(cabs(z))) + carg(z) * _Complex_I);}
static s7_complex cpow(s7_complex x, s7_complex y)
{
  s7_double r = cabs(x);
  s7_double theta = carg(x);
  s7_double yre = creal(y);
  s7_double yim = cimag(y);
  s7_double nr = exp(yre * log(r) - yim * theta);
  s7_double ntheta = yre * theta + yim * log(r);
  return(nr * cos(ntheta) + (nr * sin(ntheta)) * _Complex_I); /* make-polar */
}

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 9) /* untested -- this orignally looked at __FreeBSD_version which apparently no longer exists */
  static s7_complex cexp(s7_complex z) {return(exp(creal(z)) * cos(cimag(z)) + (exp(creal(z)) * sin(cimag(z))) * _Complex_I);}
#endif

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 10)
  static s7_complex csin(s7_complex z)   {return(sin(creal(z)) * cosh(cimag(z)) + (cos(creal(z)) * sinh(cimag(z))) * _Complex_I);}
  static s7_complex ccos(s7_complex z)   {return(cos(creal(z)) * cosh(cimag(z)) + (-sin(creal(z)) * sinh(cimag(z))) * _Complex_I);}
  static s7_complex csinh(s7_complex z)  {return(sinh(creal(z)) * cos(cimag(z)) + (cosh(creal(z)) * sin(cimag(z))) * _Complex_I);}
  static s7_complex ccosh(s7_complex z)  {return(cosh(creal(z)) * cos(cimag(z)) + (sinh(creal(z)) * sin(cimag(z))) * _Complex_I);}
  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-_Complex_I * clog(_Complex_I * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-_Complex_I * clog(z + _Complex_I * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(_Complex_I * clog((_Complex_I + z) / (_Complex_I - z)) / 2.0);}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
  /* perhaps less prone to numerical troubles (untested): 2.0 * clog(csqrt(0.5 * (z + 1.0)) + csqrt(0.5 * (z - 1.0))) */
#endif /* not FreeBSD 10 */
#endif /* not c++ */
#endif /* not HAVE_COMPLEX_TRIG */

#else  /* not HAVE_COMPLEX_NUMBERS */
  typedef double s7_complex;
  #define _Complex_I 1
  #define creal(x) x
  #define cimag(x) x
  #define csin(x) sin(x)
  #define casin(x) x
  #define ccos(x) cos(x)
  #define cacos(x) x
  #define ctan(x) x
  #define catan(x) x
  #define csinh(x) x
  #define casinh(x) x
  #define ccosh(x) x
  #define cacosh(x) x
  #define ctanh(x) x
  #define catanh(x) x
  #define cexp(x) exp(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define csqrt(x) sqrt(x)
  #define conj(x) x
#endif

#ifdef __OpenBSD__
  /* openbsd's builtin versions of these functions are not usable */
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh_1(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
#endif
#ifdef __NetBSD__
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
#endif


bool s7_is_number(s7_pointer p)
{
#if WITH_GMP
  return((is_number(p)) || (is_big_number(p)));
#else
  return(is_number(p));
#endif
}


bool s7_is_integer(s7_pointer p)
{
#if WITH_GMP
  return((is_t_integer(p)) ||
	 (is_t_big_integer(p)));
#else
  return(is_integer(p));
#endif
}

bool s7_is_real(s7_pointer p)
{
#if WITH_GMP
  return((is_real(p)) ||
	 (is_t_big_integer(p)) ||
	 (is_t_big_ratio(p)) ||
	 (is_t_big_real(p)));
#else
  return(is_real(p)); /* in GSL, a NaN or inf is not a real, or perhaps better, finite = not (nan or inf) */
#endif
}


bool s7_is_rational(s7_pointer p)
{
#if WITH_GMP
  return((is_rational(p)) ||
	 (is_t_big_integer(p)) ||
	 (is_t_big_ratio(p)));
#else
  return(is_rational(p));
#endif
}


bool s7_is_ratio(s7_pointer p)
{
#if WITH_GMP
  return((is_t_ratio(p)) ||
	 (is_t_big_ratio(p)));
#else
  return(is_t_ratio(p));
#endif
}


bool s7_is_complex(s7_pointer p)
{
#if WITH_GMP
  return((is_number(p)) || (is_big_number(p)));
#else
  return(is_number(p));
#endif
}


static s7_int c_gcd(s7_int u, s7_int v)
{
  s7_int a, b;

  if ((u == s7_int_min) || (v == s7_int_min))
    {
      /* can't take abs of these (below) so do it by hand */
      s7_int divisor = 1;
      if (u == v) return(u);
      while (((u & 1) == 0) && ((v & 1) == 0))
	{
	  u /= 2;
	  v /= 2;
	  divisor *= 2;
	}
      return(divisor);
    }

  a = s7_int_abs(u);
  b = s7_int_abs(v);
  while (b != 0)
    {
      s7_int temp;
      temp = a % b;
      a = b;
      b = temp;
    }
  if (a < 0)
    return(-a);
  return(a);
}


static bool c_rationalize(s7_double ux, s7_double error, s7_int *numer, s7_int *denom)
{
  /*
    (define* (rat ux (err 0.0000001))
      ;; translated from CL code in Canny, Donald, Ressler, "A Rational Rotation Method for Robust Geometric Algorithms"
      (let ((x0 (- ux error))
	    (x1 (+ ux error)))
        (let ((i (ceiling x0))
	      (i0 (floor x0))
	      (i1 (ceiling x1))
	      (r 0))
          (if (>= x1 i)
	      i
	      (do ((p0 i0 (+ p1 (* r p0)))
	           (q0 1 (+ q1 (* r q0)))
	           (p1 i1 p0)
	           (q1 1 q0)
	           (e0 (- i1 x0) e1p)
	           (e1 (- x0 i0) (- e0p (* r e1p)))
	           (e0p (- i1 x1) e1)
	           (e1p (- x1 i0) (- e0 (* r e1))))
	          ((<= x0 (/ p0 q0) x1)
	           (/ p0 q0))
	        (set! r (min (floor (/ e0 e1))
			     (ceiling (/ e0p e1p)))))))))
  */

  double x0, x1;
  s7_int i, i0, i1, p0, q0, p1, q1;
  double e0, e1, e0p, e1p;
  int tries = 0;
  /* don't use s7_double here;  if it is "long double", the loop below will hang */

  /* #e1e19 is a killer -- it's bigger than most-positive-fixnum, but if we ceil(ux) below
   *   it turns into most-negative-fixnum.  1e19 is trouble in many places.
   */
  if ((ux > s7_int_max) || (ux < s7_int_min))
    {
      /* can't return false here because that confuses some of the callers!
       */
      if (ux > s7_int_min) (*numer) = s7_int_max; else (*numer) = s7_int_min;
      (*denom) = 1;
      return(true);
    }

  if (error < 0.0) error = -error;
  x0 = ux - error;
  x1 = ux + error;
  i = (s7_int)ceil(x0);

  if (error >= 1.0) /* aw good grief! */
    {
      if (x0 < 0)
	{
	  if (x1 < 0)
	    (*numer) = (s7_int)floor(x1);
	  else (*numer) = 0;
	}
      else (*numer) = i;
      (*denom) = 1;
      return(true);
    }

  if (x1 >= i)
    {
      if (i >= 0)
	(*numer) = i;
      else (*numer) = (s7_int)floor(x1);
      (*denom) = 1;
      return(true);
    }

  i0 = (s7_int)floor(x0);
  i1 = (s7_int)ceil(x1);

  p0 = i0;
  q0 = 1;
  p1 = i1;
  q1 = 1;
  e0 = i1 - x0;
  e1 = x0 - i0;
  e0p = i1 - x1;
  e1p = x1 - i0;

  while (true)
    {
      s7_int old_p1, old_q1;
      double old_e0, old_e1, old_e0p, val, r, r1;
      val = (double)p0 / (double)q0;

      if (((x0 <= val) && (val <= x1)) ||
	  (e1 == 0)                    ||
	  (e1p == 0)                   ||
	  (tries > 100))
	{
	  (*numer) = p0;
	  (*denom) = q0;
	  return(true);
	}
      tries++;

      r = (s7_int)floor(e0 / e1);
      r1 = (s7_int)ceil(e0p / e1p);
      if (r1 < r) r = r1;

      /* do handles all step vars in parallel */
      old_p1 = p1;
      p1 = p0;
      old_q1 = q1;
      q1 = q0;
      old_e0 = e0;
      e0 = e1p;
      old_e0p = e0p;
      e0p = e1;
      old_e1 = e1;

      p0 = old_p1 + r * p0;
      q0 = old_q1 + r * q0;
      e1 = old_e0p - r * e1p;
      /* if the error is set too low, we can get e1 = 0 here: (rationalize (/ pi) 1e-17) */
      e1p = old_e0 - r * old_e1;
    }
  return(false);
}


s7_pointer s7_rationalize(s7_scheme *sc, s7_double x, s7_double error)
{
  s7_int numer = 0, denom = 1;
  if (c_rationalize(x, error, &numer, &denom))
    return(s7_make_ratio(sc, numer, denom));
  return(make_real(sc, x));
}


static s7_int number_to_numerator(s7_pointer n)
{
  if (is_t_ratio(n))
    return(numerator(n));
  return(integer(n));
}


static s7_int number_to_denominator(s7_pointer n)
{
  if (is_t_ratio(n))
    return(denominator(n));
  return(1);
}


s7_pointer s7_make_integer(s7_scheme *sc, s7_int n)
{
  s7_pointer x;
  if (is_small(n))              /* ((n >= 0) && (n < NUM_SMALL_INTS)) is slower */
    return(small_int(n));

  new_cell(sc, x, T_INTEGER);
  integer(x) = n;
  return(x);
}


static s7_pointer make_mutable_integer(s7_scheme *sc, s7_int n)
{
  s7_pointer x;
  new_cell(sc, x, T_INTEGER | T_MUTABLE);
  integer(x) = n;
  return(x);
}


static s7_pointer make_permanent_integer_unchecked(s7_int i)
{
  s7_pointer p;
  p = (s7_pointer)calloc(1, sizeof(s7_cell));
  typeflag(p) = T_IMMUTABLE | T_INTEGER;
  unheap(p);
  integer(p) = i;
  return(p);
}

static s7_pointer make_permanent_integer(s7_int i)
{
  if (is_small(i)) return(small_int(i));

  if (i == MAX_ARITY) return(max_arity);
  if (i == CLOSURE_ARITY_NOT_SET) return(arity_not_set);
  if (i == -1) return(minus_one);
  if (i == -2) return(minus_two);
  /* a few -3 */

  return(make_permanent_integer_unchecked(i));
}


s7_pointer s7_make_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  /* in snd-test this is called about 40000000 times, primarily test 8/18/22 */

  if (n == 0.0)
    return(real_zero);

  new_cell(sc, x, T_REAL);
  set_real(x, n);

  return(x);
}


s7_pointer s7_make_mutable_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL | T_MUTABLE);
  set_real(x, n);
  return(x);
}


static s7_pointer make_permanent_real(s7_double n)
{
  s7_pointer x;
  int nlen = 0;
  char *str;

  x = (s7_pointer)calloc(1, sizeof(s7_cell));
  set_type(x, T_IMMUTABLE | T_REAL);
  unheap(x);
  set_real(x, n);

  str = number_to_string_base_10(x, 0, float_format_precision, 'g', &nlen, USE_WRITE);
  set_print_name(x, str, nlen);
  return(x);
}


s7_pointer s7_make_complex(s7_scheme *sc, s7_double a, s7_double b)
{
  s7_pointer x;
  if (b == 0.0)
    {
      new_cell(sc, x, T_REAL);
      set_real(x, a);
    }
  else
    {
      new_cell(sc, x, T_COMPLEX);
      set_real_part(x, a);
      set_imag_part(x, b);
    }
  return(x);
}


s7_pointer s7_make_ratio(s7_scheme *sc, s7_int a, s7_int b)
{
  s7_pointer x;
  s7_int divisor;

  if (b == 0)
    return(division_by_zero_error(sc, make_string_wrapper(sc, "make-ratio"), set_elist_2(sc, make_integer(sc, a), small_int(0))));
  if (a == 0)
    return(small_int(0));
  if (b == 1)
    return(make_integer(sc, a));

#if (!WITH_GMP)
  if (b == s7_int_min)
    {
      if (a == b)
	return(small_int(1));

      /* we've got a problem... This should not trigger an error during reading -- we might have the
       *   ratio on a switch with-bignums or whatever, so its mere occurrence is just an annoyance.
       *   We'll try to do something...
       */
      if (a & 1)
	{
	  if (a == 1)
	    return(real_NaN);
	  /* not an error here? we can't get this in the ratio reader, I think, because the denominator is negative */
	  b = b + 1;
	  /* so (/ -1 most-negative-fixnum) -> 1/9223372036854775807 -- not ideal, but ... */
	}
      else
	{
	  a /= 2;
	  b /= 2;
	}
    }
#endif

  if (b < 0)
    {
      a = -a;
      b = -b;
    }
  divisor = c_gcd(a, b);
  if (divisor != 1)
    {
      a /= divisor;
      b /= divisor;
    }
  if (b == 1)
    return(make_integer(sc, a));

  new_cell(sc, x, T_RATIO);
  numerator(x) = a;
  denominator(x) = b;

  return(x);
}
/* in fc19 as a guest running in virtualbox on OSX, the line  a /= divisor can abort with an arithmetic exception (SIGFPE)
 *    if leastfix/mostfix -- apparently this is a bug in virtualbox.
 */


#define WITH_OVERFLOW_ERROR true
#define WITHOUT_OVERFLOW_ERROR false

#if (!WITH_PURE_S7)
static s7_pointer exact_to_inexact(s7_scheme *sc, s7_pointer x)
{
  /* this is tricky because a big int can mess up when turned into a double:
   *   (truncate (exact->inexact most-positive-fixnum)) -> -9223372036854775808
   */
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, (s7_double)(integer(x))));
    case T_RATIO:   return(make_real(sc, (s7_double)(fraction(x))));
    case T_REAL:                  
    case T_COMPLEX: return(x); /* apparently (exact->inexact 1+i) is not an error */
    default: 
      method_or_bust_with_type(sc, x, sc->exact_to_inexact_symbol, list_1(sc, x), a_number_string, 0);
    }
}

static s7_pointer inexact_to_exact(s7_scheme *sc, s7_pointer x, bool with_error)
{
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(x);

    case T_REAL:
      {
	s7_int numer = 0, denom = 1;
	s7_double val;

	val = s7_real(x);
	if ((is_inf(val)) || (is_NaN(val)))
	  {
	    if (with_error)
	      return(simple_wrong_type_argument_with_type(sc, sc->inexact_to_exact_symbol, x, a_normal_real_string));
	    return(sc->nil);
	  }

	if ((val > s7_int_max) ||
	    (val < s7_int_min))
	  {
	    if (with_error)
	      return(simple_out_of_range(sc, sc->inexact_to_exact_symbol, x, its_too_large_string));
	    return(sc->nil);
	  }

	if (c_rationalize(val, sc->default_rationalize_error, &numer, &denom))
	  return(s7_make_ratio(sc, numer, denom));
      }

    default:
      if (with_error)
	method_or_bust(sc, x, sc->inexact_to_exact_symbol, list_1(sc, x), T_REAL, 0);
      return(sc->nil);
    }
  return(x);
}
#endif

s7_double s7_number_to_real_with_caller(s7_scheme *sc, s7_pointer x, const char *caller)
{
  if (is_t_real(x))
    return(real(x));
  /* this is nearly always the case in current usage, so by avoiding the "switch" we can go twice as fast */

  switch (type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return((s7_double)numerator(x) / (s7_double)denominator(x));
    case T_REAL:        return(real(x));
#if WITH_GMP
    case T_BIG_INTEGER: return((s7_double)big_integer_to_s7_int(big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long double)big_integer_to_s7_int(mpq_numref(big_ratio(x))) / 
					   (long double)big_integer_to_s7_int(mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), GMP_RNDN));
#endif
    }
  s7_wrong_type_arg_error(sc, caller, 0, x, "a real number");
  return(0.0);
}


s7_double s7_number_to_real(s7_scheme *sc, s7_pointer x)
{
  return(s7_number_to_real_with_caller(sc, x, "s7_number_to_real"));
}


s7_int s7_number_to_integer_with_caller(s7_scheme *sc, s7_pointer x, const char *caller) /* currently unused */
{
  if (type(x) != T_INTEGER)
    s7_wrong_type_arg_error(sc, caller, 0, x, "an integer");
  return(integer(x));
}

s7_int s7_number_to_integer(s7_scheme *sc, s7_pointer x)                                 /* currently unused */
{
  return(s7_number_to_integer_with_caller(sc, x, "s7_number_to_integer"));
}


s7_int s7_numerator(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x));
    case T_RATIO:       return(numerator(x));
#if WITH_GMP
    case T_BIG_INTEGER: return(big_integer_to_s7_int(big_integer(x)));
    case T_BIG_RATIO:   return(big_integer_to_s7_int(mpq_numref(big_ratio(x))));
#endif
    }
  return(0);
}


s7_int s7_denominator(s7_pointer x)
{
  switch (type(x))
    {
    case T_RATIO:     return(denominator(x));
#if WITH_GMP
    case T_BIG_RATIO: return(big_integer_to_s7_int(mpq_denref(big_ratio(x))));
#endif
    }
  return(1);
}


s7_int s7_integer(s7_pointer p)
{
#if WITH_GMP
  if (is_t_big_integer(p))
    return(big_integer_to_s7_int(big_integer(p)));
#endif
  return(integer(p));
}


s7_double s7_real(s7_pointer p)
{
#if WITH_GMP
  if (is_t_big_real(p))
    return((s7_double)mpfr_get_d(big_real(p), GMP_RNDN));
#endif
  return(real(p));
}


#if (!WITH_GMP)
static s7_complex s7_to_c_complex(s7_pointer p)
{
#if HAVE_COMPLEX_NUMBERS
  return(CMPLX(s7_real_part(p), s7_imag_part(p)));
#else
  return(0.0);
#endif
}


static s7_pointer s7_from_c_complex(s7_scheme *sc, s7_complex z)
{
  return(s7_make_complex(sc, creal(z), cimag(z)));
}
#endif


#if ((!WITH_PURE_S7) || (!HAVE_OVERFLOW_CHECKS))
static int integer_length(s7_int a)
{
  static const int bits[256] =
    {0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
     6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
     7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
     7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
     8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
     8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
     8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
     8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8};

  #define I_8 256LL
  #define I_16 65536LL
  #define I_24 16777216LL
  #define I_32 4294967296LL
  #define I_40 1099511627776LL
  #define I_48 281474976710656LL
  #define I_56 72057594037927936LL

  /* a might be most-negative-fixnum! in Clisp: (integer-length -9223372036854775808) -> 63
   */
  if (a < 0)
    {
      if (a == s7_int_min) return(63);
      a = -a;
    }
  if (a < I_8) return(bits[a]);
  if (a < I_16) return(8 + bits[a >> 8]);
  if (a < I_24) return(16 + bits[a >> 16]);
  if (a < I_32) return(24 + bits[a >> 24]);
  if (a < I_40) return(32 + bits[a >> 32]);
  if (a < I_48) return(40 + bits[a >> 40]);
  if (a < I_56) return(48 + bits[a >> 48]);
  return(56 + bits[a >> 56]);
}
#endif

static int s7_int32_max = 0, s7_int32_min = 0, s7_int_bits = 0, s7_int_digits = 0; /* initialized later */
static int s7_int_digits_by_radix[17];


#if (!WITH_GMP)
static s7_pointer s7_negate(s7_scheme *sc, s7_pointer p)     /* can't use "negate" because it confuses C++! */
{
  switch (type(p))
    {
    case T_INTEGER: return(make_integer(sc, -integer(p)));
    case T_RATIO:   return(s7_make_ratio(sc, -numerator(p), denominator(p)));
    case T_REAL:    return(make_real(sc, -real(p)));
    default:        return(s7_make_complex(sc, -real_part(p), -imag_part(p)));
    }
}
#endif


static s7_pointer s7_invert(s7_scheme *sc, s7_pointer p)      /* s7_ to be consistent... */
{
  switch (type(p))
    {
    case T_INTEGER:
      return(s7_make_ratio(sc, 1, integer(p)));      /* a already checked, not 0 */

    case T_RATIO:
      return(s7_make_ratio(sc, denominator(p), numerator(p)));

    case T_REAL:
      return(make_real(sc, 1.0 / real(p)));

    case T_COMPLEX:
      {
	s7_double r2, i2, den;
	r2 = real_part(p);
	i2 = imag_part(p);
	den = (r2 * r2 + i2 * i2);
	return(s7_make_complex(sc, r2 / den, -i2 / den));
      }

    default:
      return(wrong_type_argument_with_type(sc, sc->divide_symbol, 1, p, a_number_string));
    }
}


static s7_pointer subtract_ratios(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  s7_int d1, d2, n1, n2;
  d1 = number_to_denominator(x);
  n1 = number_to_numerator(x);
  d2 = number_to_denominator(y);
  n2 = number_to_numerator(y);

  if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int case */
    return(s7_make_ratio(sc, n1 - n2, d1));

#if (!WITH_GMP)
#if HAVE_OVERFLOW_CHECKS
  {
    s7_int n1d2, n2d1, d1d2, dn;
    if ((multiply_overflow(d1, d2, &d1d2)) ||
	(multiply_overflow(n1, d2, &n1d2)) ||
	(multiply_overflow(n2, d1, &n2d1)) ||
	(subtract_overflow(n1d2, n2d1, &dn)))
      return(make_real(sc, ((long double)n1 / (long double)d1) - ((long double)n2 / (long double)d2)));
    return(s7_make_ratio(sc, dn, d1d2));
  }
#else
  if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
      (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
      (n1 < s7_int32_min) || (n2 < s7_int32_min))
    {
      int d1bits, d2bits;
      d1bits = integer_length(d1);
      d2bits = integer_length(d2);
      if (((d1bits + d2bits) > s7_int_bits) ||
	  ((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
	  ((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
	return(make_real(sc, ((long double)n1 / (long double)d1) - ((long double)n2 / (long double)d2)));
      return(s7_make_ratio(sc, n1 * d2 - n2 * d1, d1 * d2));
    }
#endif
#endif
  return(s7_make_ratio(sc, n1 * d2 - n2 * d1, d1 * d2));
}


static bool s7_is_negative(s7_pointer obj)
{
  switch (type(obj))
    {
    case T_INTEGER:     return(integer(obj) < 0);
    case T_RATIO:       return(numerator(obj) < 0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(obj), 0) < 0);
    case T_BIG_RATIO:   return(mpq_cmp_ui(big_ratio(obj), 0, 1) < 0);
    case T_BIG_REAL:    return(mpfr_cmp_ui(big_real(obj), 0) < 0);
#endif
    default:            return(real(obj) < 0);
    }
}


static bool s7_is_positive(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x) > 0);
    case T_RATIO:       return(numerator(x) > 0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(x), 0) > 0);
    case T_BIG_RATIO:   return(mpq_cmp_ui(big_ratio(x), 0, 1) > 0);
    case T_BIG_REAL:    return(mpfr_cmp_ui(big_real(x), 0) > 0);
#endif
    default:            return(real(x) > 0.0);
    }
}


static bool s7_is_zero(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x) == 0);
    case T_REAL:        return(real(x) == 0.0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(x), 0) == 0);
    case T_BIG_REAL:    return(mpfr_zero_p(big_real(x)));
#endif
    default:            return(false); /* ratios and complex numbers here are already collapsed into integers and reals */
    }
}


static bool s7_is_one(s7_pointer x)
{
    return(((is_integer(x)) && (integer(x) == 1)) ||
	   ((is_t_real(x)) && (real(x) == 1.0)));
}


/* optimize exponents */
#define MAX_POW 32
static double pepow[17][MAX_POW], mepow[17][MAX_POW];

static void init_pows(void)
{
  int i, j;
  for (i = 2; i < 17; i++)        /* radix between 2 and 16 */
    for (j = 0; j < MAX_POW; j++) /* saved exponent between 0 and +/- MAX_POW */
      {
	pepow[i][j] = pow((double)i, (double)j);
	mepow[i][j] = pow((double)i, (double)(-j));
      }
}

static double ipow(int x, int y)
{
  if ((y < MAX_POW) && (y > (-MAX_POW)))
    {
      if (y >= 0)
	return(pepow[x][y]);
      return(mepow[x][-y]);
    }
  return(pow((double)x, (double)y));
}


static int s7_int_to_string(char *p, s7_int n, int radix, int width)
{
  static const char dignum[] = "0123456789abcdef";
  int i, len, start, end;
  bool sign;
  s7_int pown;

  if ((radix < 2) || (radix > 16))
    return(0);

  if (n == s7_int_min) /* can't negate this, so do it by hand */
    {
      static const char *mnfs[17] = {"","",
	"-1000000000000000000000000000000000000000000000000000000000000000", "-2021110011022210012102010021220101220222",
	"-20000000000000000000000000000000", "-1104332401304422434310311213", "-1540241003031030222122212",
	"-22341010611245052052301", "-1000000000000000000000", "-67404283172107811828",	"-9223372036854775808",
	"-1728002635214590698",	"-41a792678515120368", "-10b269549075433c38", "-4340724c6c71dc7a8", "-160e2ad3246366808", "-8000000000000000"};

      len = safe_strlen(mnfs[radix]);
      if (width > len)
	{
	  start = width - len - 1;
	  memset((void *)p, (int)' ', start);
	}
      else start = 0;
      for (i = 0; i < len; i++)
	p[start + i] = mnfs[radix][i];
      p[len + start] = '\0';
      return(len + start);
    }

  sign = (n < 0);
  if (sign) n = -n;

  /* the previous version that counted up to n, rather than dividing down below n, as here,
   *   could be confused by large ints on 64 bit machines
   */
  pown = n;
  for (i = 1; i < 100; i++)
    {
      if (pown < radix)
	break;
      pown /= (s7_int)radix;
    }
  len = i - 1;
  if (sign) len++;
  end = 0;
  if (width > len)                  /* (format #f "~10B" 123) */
    {
      start = width - len - 1;
      end += start;
      memset((void *)p, (int)' ', start);
    }
  else
    {
      start = 0;
      end = 0;
    }

  if (sign)
    {
      p[start] = '-';
      end++;
    }

  for (i = start + len; i >= end; i--)
    {
      p[i] = dignum[n % radix];
      n /= radix;
    }
  p[len + start + 1] = '\0';
  return(len + start + 1);
}


static char *integer_to_string_base_10_no_width(s7_pointer obj, int *nlen) /* do not free the returned string */
{
  long long int num;
  char *p, *op;
  bool sign;
  static char int_to_str[INT_TO_STR_SIZE];

  if (has_print_name(obj))
    {
      (*nlen) = print_name_length(obj);
      return((char *)print_name(obj));
    }
  /* (*nlen) = snprintf(int_to_str, INT_TO_STR_SIZE, "%lld", (long long int)integer(obj));
   *  but that is very slow -- the following code is 6 times faster
   */
  num = (long long int)integer(obj);
  if (num == s7_int_min)
    {
      (*nlen) = 20;
      return((char *)"-9223372036854775808");
    }
  p = (char *)(int_to_str + INT_TO_STR_SIZE - 1);
  op = p;
  *p-- = '\0';

  sign = (num < 0);
  if (sign) num = -num;  /* we need a positive index below */
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  if (sign)
    {
      *p = '-';
      (*nlen) = op - p;
      return(p);
    }

  (*nlen) = op - p - 1;
  return(++p);
}


#define BASE_10 10
static int num_to_str_size = -1;
static char *num_to_str = NULL;
static const char *float_format_g = NULL;

static char *floatify(char *str, int *nlen)
{
  if ((!strchr(str, 'e')) &&
      (!strchr(str, '.')))
    {
      /* this assumes there is room in str for 2 more chars */
      int len;
      len = *nlen;
      str[len]='.';
      str[len + 1]='0';
      str[len + 2]='\0';
      (*nlen) = len + 2;
    }
  return(str);
}

static char *number_to_string_base_10(s7_pointer obj, int width, int precision, char float_choice, int *nlen, use_write_t choice) /* don't free result */
{
  /* the rest of s7 assumes nlen is set to the correct length 
   *   a tricky case: (format #f "~f" 1e308) -- tries to print 308 digits! so 256 as default len is too small.
   *   but then even worse: (format #f "~F" 1e308+1e308i)!
   */
  int len;
  len = 1024;
  if (width > len) len = 2 * width;
  if (len > num_to_str_size)
    {
      if (!num_to_str)
	num_to_str = (char *)malloc(len * sizeof(char));
      else num_to_str = (char *)realloc(num_to_str, len * sizeof(char));
      num_to_str_size = len;
    }

  /* bignums can't happen here */
  switch (type(obj))
    {
    case T_INTEGER:
      if (width == 0)
	return(integer_to_string_base_10_no_width(obj, nlen));
      (*nlen) = snprintf(num_to_str, num_to_str_size, "%*lld", width, (long long int)integer(obj));
      break;

    case T_RATIO:
      len = snprintf(num_to_str, num_to_str_size, "%lld/%lld", (long long int)numerator(obj), (long long int)denominator(obj));
      if (width > len)
	{
	  int spaces;
	  if (width >= num_to_str_size)
	    {
	      num_to_str_size = width + 1;
	      num_to_str = (char *)realloc(num_to_str, num_to_str_size * sizeof(char));
	    }
	  spaces = width - len;
	  num_to_str[width] = '\0';
	  memmove((void *)(num_to_str + spaces), (void *)num_to_str, len);
	  memset((void *)num_to_str, (int)' ', spaces);
	  (*nlen) = width;
	}
      else (*nlen) = len;
      break;

    case T_REAL:
      {
	const char *frmt;
	if (sizeof(double) >= sizeof(s7_double))
	  frmt = (float_choice == 'g') ? "%*.*g" : ((float_choice == 'f') ? "%*.*f" : "%*.*e");
	else frmt = (float_choice == 'g') ? "%*.*Lg" : ((float_choice == 'f') ? "%*.*Lf" : "%*.*Le");

	len = snprintf(num_to_str, num_to_str_size - 4, frmt, width, precision, s7_real(obj)); /* -4 for floatify */
	(*nlen) = len;
	floatify(num_to_str, nlen);
      }
      break;

    default:
      {
	if ((choice == USE_READABLE_WRITE) &&
	    ((is_NaN(real_part(obj))) || (is_NaN(imag_part(obj))) || ((is_inf(real_part(obj))) || (is_inf(imag_part(obj))))))
	  {
	    char rbuf[128], ibuf[128];
	    char *rp, *ip;
	    if (is_NaN(real_part(obj)))
	      rp = (char *)"nan.0";
	    else
	      {
		if (is_inf(real_part(obj)))
		  {
		    if (real_part(obj) < 0.0)
		      rp = (char *)"-inf.0";
		    else rp = (char *)"inf.0";
		  }
		else
		  {
		    snprintf(rbuf, 128, float_format_g, precision, real_part(obj));
		    rp = rbuf;
		  }
	      }
	    if (is_NaN(imag_part(obj)))
	      ip = (char *)"nan.0";
	    else
	      {
		if (is_inf(imag_part(obj)))
		  {
		    if (imag_part(obj) < 0.0)
		      ip = (char *)"-inf.0";
		    else ip = (char *)"inf.0";
		  }
		else
		  {
		    snprintf(ibuf, 128, float_format_g, precision, imag_part(obj));
		    ip = ibuf;
		  }
	      }
	    len = snprintf(num_to_str, num_to_str_size, "(complex %s %s)", rp, ip);
	  }
	else
	  {
	    const char *frmt;
	    if (sizeof(double) >= sizeof(s7_double))
	      {
		if (imag_part(obj) >= 0.0)
		  frmt = (float_choice == 'g') ? "%.*g+%.*gi" : ((float_choice == 'f') ? "%.*f+%.*fi" : "%.*e+%.*ei");
		else frmt = (float_choice == 'g') ? "%.*g%.*gi" : ((float_choice == 'f') ? "%.*f%.*fi" :"%.*e%.*ei"); /* minus sign comes with the imag_part */
	      }
	    else
	      {
		if (imag_part(obj) >= 0.0)
		  frmt = (float_choice == 'g') ? "%.*Lg+%.*Lgi" : ((float_choice == 'f') ? "%.*Lf+%.*Lfi" : "%.*Le+%.*Lei");
		else frmt = (float_choice == 'g') ? "%.*Lg%.*Lgi" : ((float_choice == 'f') ? "%.*Lf%.*Lfi" : "%.*Le%.*Lei");
	      }

	    len = snprintf(num_to_str, num_to_str_size, frmt, precision, real_part(obj), precision, imag_part(obj));
	  }

	if (width > len)  /* (format #f "~20g" 1+i) */
	  {
	    int spaces;
	    if (width >= num_to_str_size)
	      {
		num_to_str_size = width + 1;
		num_to_str = (char *)realloc(num_to_str, num_to_str_size * sizeof(char));
	      }
	    spaces = width - len;
	    num_to_str[width] = '\0';
	    memmove((void *)(num_to_str + spaces), (void *)num_to_str, len);
	    memset((void *)num_to_str, (int)' ', spaces);
	    (*nlen) = width;
	  }
	else (*nlen) = len;
      }
      break;
    }
  return(num_to_str);
}


static char *number_to_string_with_radix(s7_scheme *sc, s7_pointer obj, int radix, int width, int precision, char float_choice, int *nlen)
{
  /* the rest of s7 assumes nlen is set to the correct length */
  char *p;
  int len, str_len;

#if WITH_GMP
  if (s7_is_bignum(obj))
    return(big_number_to_string_with_radix(obj, radix, width, nlen, USE_WRITE));
  /* this ignores precision because it's way too hard to get the mpfr string to look like
   *   C's output -- we either have to call mpfr_get_str twice (the first time just to
   *   find out what the exponent is and how long the string actually is), or we have
   *   to do messy string manipulations.  So (format #f "",3F" pi) ignores the "3" and
   *   prints the full string.
   */
#endif

  if (radix == 10)
    {
      p = number_to_string_base_10(obj, width, precision, float_choice, nlen, USE_WRITE);
      return(copy_string_with_length(p, *nlen));
    }

  switch (type(obj))
    {
    case T_INTEGER:
      p = (char *)malloc((128 + width) * sizeof(char));
      *nlen = s7_int_to_string(p, s7_integer(obj), radix, width);
      return(p);

    case T_RATIO:
      {
	char n[128], d[128];
	s7_int_to_string(n, numerator(obj), radix, 0);
	s7_int_to_string(d, denominator(obj), radix, 0);
	p = (char *)malloc(256 * sizeof(char));
	len = snprintf(p, 256, "%s/%s", n, d);
	str_len = 256;
      }
      break;

    case T_REAL:
      {
	int i;
	s7_int int_part;
	s7_double x, frac_part, min_frac, base;
	bool sign = false;
	char n[128], d[256];

	x = s7_real(obj);

	if (is_NaN(x))
	  return(copy_string_with_length("nan.0", *nlen = 5));
	if (is_inf(x))
	  {
	    if (x < 0.0)
	      return(copy_string_with_length("-inf.0", *nlen = 6));
	    return(copy_string_with_length("inf.0", *nlen = 5));
	  }

	if (x < 0.0)
	  {
	    sign = true;
	    x = -x;
	  }

	if (x > 1.0e18) /* i.e. close to or greater than most-positive-fixnum (9.22e18), so the code below is unlikely to work, (format #f "~X" 1e19) */
	  {
	    int ep;
	    char *p1;
	    s7_pointer r;

	    len = 0;
	    ep = (int)floor(log(x) / log((double)radix));
	    r = make_real(sc, x / pow((double)radix, (double)ep)); /* divide it down to one digit, then the fractional part */
	    p1 = number_to_string_with_radix(sc, r, radix, width, precision, float_choice, &len);
	    p = (char *)malloc((len + 8) * sizeof(char));
	    (*nlen) = snprintf(p, len + 8, "%s%se%d", (sign) ? "-" : "", p1, ep);
	    free(p1);
	    return(p);
	  }

	int_part = (s7_int)floor(x);
	frac_part = x - int_part;
	s7_int_to_string(n, int_part, radix, 0);
	min_frac = (s7_double)ipow(radix, -precision);

	/* doesn't this assume precision < 128/256 and that we can fit in 256 digits (1e308)? */

	for (i = 0, base = radix; (i < precision) && (frac_part > min_frac); i++, base *= radix)
	  {
	    s7_int ipart;
	    ipart = (s7_int)(frac_part * base);
	    if (ipart >= radix)         /* rounding confusion */
	      ipart = radix - 1;
	    frac_part -= (ipart / base);
	    if (ipart < 10)
	      d[i] = (char)('0' + ipart);
	    else d[i] = (char)('a' + ipart -  10);
	  }
	if (i == 0)
	  d[i++] = '0';
	d[i] = '\0';
	p = (char *)malloc(256 * sizeof(char));
	len = snprintf(p, 256, "%s%s.%s", (sign) ? "-" : "", n, d);
	str_len = 256;
      }
      break;

    default:
      {
	char *n, *d;
	p = (char *)malloc(512 * sizeof(char));
	n = number_to_string_with_radix(sc, make_real(sc, real_part(obj)), radix, 0, precision, float_choice, &len);
	d = number_to_string_with_radix(sc, make_real(sc, imag_part(obj)), radix, 0, precision, float_choice, &len);
	len = snprintf(p, 512, "%s%s%si", n, (imag_part(obj) < 0.0) ? "" : "+", d);
	str_len = 512;
	free(n);
	free(d);
      }
      break;
    }

  if (width > len)
    {
      int spaces;
      if (width >= str_len)
	{
	  str_len = width + 1;
	  p = (char *)realloc(p, str_len * sizeof(char));
	}
      spaces = width - len;
      p[width] = '\0';
      memmove((void *)(p + spaces), (void *)p, len);
      memset((void *)p, (int)' ', spaces);
      (*nlen) = width;
    }
  else (*nlen) = len;
  return(p);
}


char *s7_number_to_string(s7_scheme *sc, s7_pointer obj, int radix)
{
  int nlen = 0;
  return(number_to_string_with_radix(sc, obj, radix, 0, 20, 'g', &nlen));
  /* (log top 10) so we get all the digits in base 10 (??) */
}


static void prepare_temporary_string(s7_scheme *sc, int len, int which)
{
  s7_pointer p;
  p = sc->tmp_strs[which];
  if (len > string_temp_true_length(p))
    {
      string_value(p) = (char *)realloc(string_value(p), len * sizeof(char));
      string_temp_true_length(p) = len;
    }
}

static s7_pointer g_number_to_string_1(s7_scheme *sc, s7_pointer args, bool temporary)
{
  #define H_number_to_string "(number->string num (radix 10)) converts the number num into a string."
  #define Q_number_to_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_number_symbol, sc->is_integer_symbol)

  s7_int radix = 10;
  int size, nlen = 0;
  char *res;
  s7_pointer x;

  x = car(args);
  if (!s7_is_number(x))
    method_or_bust_with_type(sc, x, sc->number_to_string_symbol, args, a_number_string, 1);

  if (is_pair(cdr(args)))
    {
      s7_pointer y;
      y = cadr(args);
      if (s7_is_integer(y))
	radix = s7_integer(y);
      else method_or_bust(sc, y, sc->number_to_string_symbol, args, T_INTEGER, 2);
      if ((radix < 2) || (radix > 16))
	return(out_of_range(sc, sc->number_to_string_symbol, small_int(2), y, a_valid_radix_string));
    }

#if WITH_GMP
  if (s7_is_bignum(x))
    {
      res = big_number_to_string_with_radix(x, radix, 0, &nlen, USE_WRITE);
      return(make_string_uncopied_with_length(sc, res, nlen));
    }
#endif

  size = float_format_precision;
  if (!is_rational(x))
    {
      /* if size = 20, (number->string .1) gives "0.10000000000000000555", but if it's less than 20,
       *    large numbers (or very small numbers) mess up the less significant digits.
       */
      if (radix == 10)
	{
	  if (is_real(x))
	    {
	      s7_double val;
	      val = fabs(s7_real(x));
	      if ((val > (s7_int32_max / 4)) || (val < 1.0e-6))
		size += 4;
	    }
	  else
	    {
	      s7_double rl;
	      rl = fabs(s7_real_part(x));
	      if ((rl > (s7_int32_max / 4)) || (rl < 1.0e-6))
		{
		  s7_double im;
		  im = fabs(s7_imag_part(x));
		  if ((im > (s7_int32_max / 4)) || (im < 1.0e-6))
		    size += 4;
		}
	    }
	}
    }
  if (radix != 10)
    {
      res = number_to_string_with_radix(sc, x, radix, 0, size, 'g', &nlen);
      return(make_string_uncopied_with_length(sc, res, nlen));
    }
  res = number_to_string_base_10(x, 0, size, 'g', &nlen, USE_WRITE);
  if (temporary)
    {
      s7_pointer p;
      prepare_temporary_string(sc, nlen + 1, 1);
      p = sc->tmp_strs[1];
      string_length(p) = nlen;
      memcpy((void *)(string_value(p)), (void *)res, nlen);
      string_value(p)[nlen] = 0;
      return(p);
    }
  return(s7_make_string_with_length(sc, res, nlen));
}

static s7_pointer g_number_to_string(s7_scheme *sc, s7_pointer args)
{
  return(g_number_to_string_1(sc, args, false));
}

static s7_pointer number_to_string_temp;
static s7_pointer g_number_to_string_temp(s7_scheme *sc, s7_pointer args)
{
  return(g_number_to_string_1(sc, args, true));
}

static s7_pointer number_to_string_pf_temp(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t f;
  s7_pointer x;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(g_number_to_string_1(sc, set_plist_1(sc, x), true));
}

static s7_pointer number_to_string_pf_s_temp(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer x;
  (*p)++; x = slot_value(**p); (*p)++;
  return(g_number_to_string_1(sc, set_plist_1(sc, x), true));
}

static s7_pointer c_number_to_string(s7_scheme *sc, s7_pointer n) {return(g_number_to_string_1(sc, set_plist_1(sc, n), false));}
PF_TO_PF(number_to_string, c_number_to_string)


#define CTABLE_SIZE 256
static bool *exponent_table, *slashify_table, *char_ok_in_a_name, *white_space, *number_table, *symbol_slashify_table;
static int *digits;

static void init_ctables(void)
{
  int i;

  exponent_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  slashify_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  symbol_slashify_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  char_ok_in_a_name = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  white_space = (bool *)calloc(CTABLE_SIZE + 1, sizeof(bool));
  white_space++;      /* leave white_space[-1] false for white_space[EOF] */
  number_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));

  for (i = 1; i < CTABLE_SIZE; i++)
    char_ok_in_a_name[i] = true;
  char_ok_in_a_name[0] = false;
  char_ok_in_a_name[(unsigned char)'('] = false;  /* idiotic cast is for C++'s benefit */
  char_ok_in_a_name[(unsigned char)')'] = false;
  char_ok_in_a_name[(unsigned char)';'] = false;
  char_ok_in_a_name[(unsigned char)'\t'] = false;
  char_ok_in_a_name[(unsigned char)'\n'] = false;
  char_ok_in_a_name[(unsigned char)'\r'] = false;
  char_ok_in_a_name[(unsigned char)' '] = false;
  char_ok_in_a_name[(unsigned char)'"'] = false;
  /* what about stuff like vertical tab?  or comma? */

  for (i = 0; i < CTABLE_SIZE; i++)
    white_space[i] = false;
  white_space[(unsigned char)'\t'] = true;
  white_space[(unsigned char)'\n'] = true;
  white_space[(unsigned char)'\r'] = true;
  white_space[(unsigned char)'\f'] = true;
  white_space[(unsigned char)'\v'] = true;
  white_space[(unsigned char)' '] = true;
  white_space[(unsigned char)'\205'] = true; /* 133 */
  white_space[(unsigned char)'\240'] = true; /* 160 */

  /* surely only 'e' is needed... */
  exponent_table[(unsigned char)'e'] = true; exponent_table[(unsigned char)'E'] = true;
  exponent_table[(unsigned char)'@'] = true;
#if WITH_EXTRA_EXPONENT_MARKERS
  exponent_table[(unsigned char)'s'] = true; exponent_table[(unsigned char)'S'] = true;
  exponent_table[(unsigned char)'f'] = true; exponent_table[(unsigned char)'F'] = true;
  exponent_table[(unsigned char)'d'] = true; exponent_table[(unsigned char)'D'] = true;
  exponent_table[(unsigned char)'l'] = true; exponent_table[(unsigned char)'L'] = true;
#endif

  for (i = 0; i < 32; i++)
    slashify_table[i] = true;
  for (i = 127; i < 160; i++)
    slashify_table[i] = true;
  slashify_table[(unsigned char)'\\'] = true;
  slashify_table[(unsigned char)'"'] = true;
  slashify_table[(unsigned char)'\n'] = false;

  for (i = 0; i < CTABLE_SIZE; i++)
    symbol_slashify_table[i] = ((slashify_table[i]) || (!char_ok_in_a_name[i]));

  digits = (int *)calloc(CTABLE_SIZE, sizeof(int));
  for (i = 0; i < CTABLE_SIZE; i++)
    digits[i] = 256;

  digits[(unsigned char)'0'] = 0; digits[(unsigned char)'1'] = 1; digits[(unsigned char)'2'] = 2; digits[(unsigned char)'3'] = 3; digits[(unsigned char)'4'] = 4;
  digits[(unsigned char)'5'] = 5; digits[(unsigned char)'6'] = 6; digits[(unsigned char)'7'] = 7; digits[(unsigned char)'8'] = 8; digits[(unsigned char)'9'] = 9;
  digits[(unsigned char)'a'] = 10; digits[(unsigned char)'A'] = 10;
  digits[(unsigned char)'b'] = 11; digits[(unsigned char)'B'] = 11;
  digits[(unsigned char)'c'] = 12; digits[(unsigned char)'C'] = 12;
  digits[(unsigned char)'d'] = 13; digits[(unsigned char)'D'] = 13;
  digits[(unsigned char)'e'] = 14; digits[(unsigned char)'E'] = 14;
  digits[(unsigned char)'f'] = 15; digits[(unsigned char)'F'] = 15;

  for (i = 0; i < CTABLE_SIZE; i++)
    number_table[i] = false;
  number_table[(unsigned char)'0'] = true;
  number_table[(unsigned char)'1'] = true;
  number_table[(unsigned char)'2'] = true;
  number_table[(unsigned char)'3'] = true;
  number_table[(unsigned char)'4'] = true;
  number_table[(unsigned char)'5'] = true;
  number_table[(unsigned char)'6'] = true;
  number_table[(unsigned char)'7'] = true;
  number_table[(unsigned char)'8'] = true;
  number_table[(unsigned char)'9'] = true;
  number_table[(unsigned char)'.'] = true;
  number_table[(unsigned char)'+'] = true;
  number_table[(unsigned char)'-'] = true;
  number_table[(unsigned char)'#'] = true;
}


#define is_white_space(C) white_space[C]
  /* this is much faster than C's isspace, and does not depend on the current locale.
   * if c == EOF (-1), it indexes into the empty (0) slot we preallocated below white_space
   */


static s7_pointer check_sharp_readers(s7_scheme *sc, const char *name)
{
  s7_pointer reader, value, args;
  bool need_loader_port;
  value = sc->F;
  args = sc->F;

  /* *#reader* is assumed to be an alist of (char . proc)
   *    where each proc takes one argument, the string from just beyond the "#" to the next delimiter.
   *    The procedure can call read-char to read ahead in the current-input-port.
   *    If it returns anything other than #f, that is the value of the sharp expression.
   *    Since #f means "nothing found", it is tricky to handle #F: 
   *       (cons #\F (lambda (str) (and (string=? str "F") (list 'not #t))))
   * This search happens after #|, #t, and #f (and #nD for multivectors?). #! has a fallback.
   */

  need_loader_port = is_loader_port(sc->input_port);
  if (need_loader_port)
    clear_loader_port(sc->input_port);

  /* normally read* can't read from sc->input_port if it is in use by the loader,
   *   but here we are deliberately making that possible.
   */
  for (reader = slot_value(sc->sharp_readers); is_not_null(reader); reader = cdr(reader))
    {
      if (name[0] == s7_character(caar(reader)))
	{
	  if (args == sc->F)
	    args = list_1(sc, s7_make_string(sc, name));
	  /* args is GC protected by s7_apply_function?? (placed on the stack) */
	  value = s7_apply_function(sc, cdar(reader), args); /* this is much less error-safe than s7_call */
	  if (value != sc->F)
	    break;
	}
    }
  if (need_loader_port)
    set_loader_port(sc->input_port);
  return(value);
}


static s7_pointer g_sharp_readers_set(s7_scheme *sc, s7_pointer args)
{
  /* new value must be either () or a proper list of conses (char . func) */
  if (is_null(cadr(args))) return(cadr(args));
  if (is_pair(cadr(args)))
    {
      s7_pointer x;
      for (x = cadr(args); is_pair(x); x = cdr(x))
	{
	  if ((!is_pair(car(x))) ||
	      (!s7_is_character(caar(x))) ||
	      (!s7_is_procedure(cdar(x))))
	    return(sc->error_symbol);
	}
      if (is_null(x))
	return(cadr(args));
    }
  return(sc->error_symbol);
}


static bool is_abnormal(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(false);

    case T_REAL:
      return(is_inf(real(x)) ||
	     is_NaN(real(x)));

    case T_COMPLEX:
      return(((is_inf(s7_real_part(x)))  ||
	      (is_inf(s7_imag_part(x)))  ||
	      (is_NaN(s7_real_part(x))) ||
	      (is_NaN(s7_imag_part(x)))));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(false);

    case T_BIG_REAL:
      return((is_inf(s7_real_part(x))) ||
	     (is_NaN(s7_real_part(x))));

    case T_BIG_COMPLEX:
      return((is_inf(s7_real_part(x))) ||
	     (is_inf(s7_imag_part(x))) ||
	     (is_NaN(s7_real_part(x))) ||
	     (is_NaN(s7_imag_part(x))));
#endif

    default:
      return(true);
    }
}

static s7_pointer unknown_sharp_constant(s7_scheme *sc, char *name)
{
  /* check *read-error-hook* */
  if (hook_has_functions(sc->read_error_hook))
    {
      s7_pointer result;
      result = s7_call(sc, sc->read_error_hook, list_2(sc, sc->T, make_string_wrapper(sc, name)));
      if (result != sc->unspecified)
	return(result);
    }
  return(sc->nil);
}

#define NESTED_SHARP false
#define UNNESTED_SHARP true

#define SYMBOL_OK true
#define NO_SYMBOLS false

static s7_pointer make_sharp_constant(s7_scheme *sc, char *name, bool at_top, int radix, bool with_error)
{
  /* name is the stuff after the '#', return sc->nil if not a recognized #... entity */
  int len;
  s7_pointer x;

  if ((name[0] == 't') &&
      ((name[1] == '\0') || (strings_are_equal(name, "true"))))
    return(sc->T);

  if ((name[0] == 'f') &&
      ((name[1] == '\0') || (strings_are_equal(name, "false"))))
    return(sc->F);

  if (is_not_null(slot_value(sc->sharp_readers)))
    {
      x = check_sharp_readers(sc, name);
      if (x != sc->F)
	return(x);
    }

  len = safe_strlen5(name); /* just count up to 5 */
  if (len < 2)
    return(unknown_sharp_constant(sc, name));

  switch (name[0])
    {
      /* -------- #< ... > -------- */
    case '<':
      if (strings_are_equal(name, "<unspecified>"))
	return(sc->unspecified);

      if (strings_are_equal(name, "<undefined>"))
	return(sc->undefined);

      if (strings_are_equal(name, "<eof>"))
	return(sc->eof_object);

      return(unknown_sharp_constant(sc, name));


      /* -------- #o #d #x #b -------- */
    case 'o':   /* #o (octal) */
    case 'd':   /* #d (decimal) */
    case 'x':   /* #x (hex) */
    case 'b':   /* #b (binary) */
      {
	int num_at = 1;
#if (!WITH_PURE_S7)
	bool to_inexact = false, to_exact = false;

	if (name[1] == '#')
	  {
	    if (!at_top)
	      return(unknown_sharp_constant(sc, name));
	    if ((len > 2) && ((name[2] == 'e') || (name[2] == 'i'))) /* r6rs includes caps here */
	      {
		if ((len > 3) && (name[3] == '#'))
		  return(unknown_sharp_constant(sc, name));
		to_inexact = (name[2] == 'i');
		to_exact = (name[2] == 'e');
		num_at = 3;
	      }
	    else return(unknown_sharp_constant(sc, name));
	  }
#endif
	/* the #b or whatever overrides any radix passed in earlier */
	x = make_atom(sc, (char *)(name + num_at), (name[0] == 'o') ? 8 : ((name[0] == 'x') ? 16 : ((name[0] == 'b') ? 2 : 10)), NO_SYMBOLS, with_error);

	/* #x#i1 apparently makes sense, so #x1.0 should also be accepted.
	 * here we can get #b#e0/0 or #b#e+1/0 etc.
	 * surely if #e1+i is an error (or #f), and #e#x1+i is an error,
	 *   #x#e1+i should also be an error, but #e1+0i is not an error I guess since there actually isn't any imaginary part
	 */
	if (is_abnormal(x))
	  return(unknown_sharp_constant(sc, name));

#if (!WITH_PURE_S7)
	if ((!to_exact) && (!to_inexact))
	  return(x);

	if ((s7_imag_part(x) != 0.0) && (to_exact))  /* #x#e1+i */
	  return(unknown_sharp_constant(sc, name));

#if WITH_GMP
	if (s7_is_bignum(x))
	  {
	    if (to_exact)
	      return(big_inexact_to_exact(sc, set_plist_1(sc, x)));
	    return(big_exact_to_inexact(sc, set_plist_1(sc, x)));
	  }
#endif
	if (to_exact)
	  return(inexact_to_exact(sc, x, with_error));
	return(exact_to_inexact(sc, x));
#else
	return(x);
#endif
      }
      break;

#if (!WITH_PURE_S7)
      /* -------- #i -------- */
    case 'i':   /* #i<num> = ->inexact (see token for table of choices here) */
      if (name[1] == '#')
	{
	  /* there are special cases here: "#e0/0" or "#e#b0/0" -- all infs are complex:
	   *    #i1/0=nan.0 but #i1/0+i=inf+1i so e->i is a no-op but i->e is not
	   *
	   * even trickier: a *#reader* like #t<num> could be used as #e#t13.25 so make_sharp_constant
	   *   needs to be willing to call the readers even when not at_top (i.e. when NESTED_SHARP).
	   */

	  if ((name[2] == 'e') ||                        /* #i#e1 -- assume these aren't redefinable? */
	      (name[2] == 'i'))
	    return(unknown_sharp_constant(sc, name));

	  x = make_sharp_constant(sc, (char *)(name + 2), NESTED_SHARP, radix, with_error);
	  if (s7_is_number(x))
	    {
	      if (is_abnormal(x))
		return(unknown_sharp_constant(sc, name));
#if WITH_GMP
	      if (s7_is_bignum(x))                        /* (string->number "#b#e-11e+111") */
		return(big_exact_to_inexact(sc, set_plist_1(sc, x)));
#endif
	      return(exact_to_inexact(sc, x));
	    }
	  return(unknown_sharp_constant(sc, name));
	}
      x = make_atom(sc, (char *)(name + 1), radix, NO_SYMBOLS, with_error);
      if (!s7_is_number(x))  /* not is_abnormal(x) -- #i0/0 -> nan etc */
	return(unknown_sharp_constant(sc, name));
#if WITH_GMP
      if (s7_is_bignum(x))
	return(big_exact_to_inexact(sc, set_plist_1(sc, x)));
#endif
      return(exact_to_inexact(sc, x));


      /* -------- #e -------- */
    case 'e':   /* #e<num> = ->exact */
      if (name[1] == '#')
	{
	  if ((name[2] == 'e') ||                        /* #e#e1 */
	      (name[2] == 'i'))
	    return(unknown_sharp_constant(sc, name));

	  x = make_sharp_constant(sc, (char *)(name + 2), NESTED_SHARP, radix, with_error);
	  if (s7_is_number(x))
	    {
	      if (is_abnormal(x))                        /* (string->number "#e#b0/0") */
		return(unknown_sharp_constant(sc, name));
	      if (!s7_is_real(x))                        /* (string->number "#e#b1+i") */
		return(unknown_sharp_constant(sc, name));
#if WITH_GMP
	      return(big_inexact_to_exact(sc, set_plist_1(sc, x)));
#endif
	      return(inexact_to_exact(sc, x, with_error));
	    }
	  return(unknown_sharp_constant(sc, name));
	}

      x = make_atom(sc, (char *)(name + 1), radix, NO_SYMBOLS, with_error);
#if WITH_GMP
      /* #e1e310 is a simple case */
      if (s7_is_bignum(x))
	return(big_inexact_to_exact(sc, set_plist_1(sc, x)));
#endif
      if (is_abnormal(x))                                /* (string->number "#e0/0") */
	return(unknown_sharp_constant(sc, name));
      if (!s7_is_real(x))                                /* (string->number "#e1+i") */
	return(unknown_sharp_constant(sc, name));

#if WITH_GMP
      /* there are non-big floats that are greater than most-positive-fixnum:
       *    :(> .1e20 most-positive-fixnum) -> #t
       *    :(bignum? .1e20) -> #f
       * so we have to check that, not just is it a bignum.
       */
      return(big_inexact_to_exact(sc, set_plist_1(sc, x)));
#endif
      return(inexact_to_exact(sc, x, with_error));
#endif /* !WITH_PURE_S7 */


      /* -------- #_... -------- */
    case '_':
      {
	s7_pointer sym;
	sym = make_symbol(sc, (char *)(name + 1));
	if (is_slot(initial_slot(sym)))
	  return(slot_value(initial_slot(sym)));
	return(s7_error(sc, sc->syntax_error_symbol, set_elist_2(sc, make_string_wrapper(sc, "#~A is undefined"), make_string_wrapper(sc, name))));
	/* return(sc->undefined); */
      }


      /* -------- #\... -------- */
    case '\\':
      if (name[2] == 0)                             /* the most common case: #\a */
	return(chars[(unsigned char)(name[1])]);
      /* not unsigned int here!  (unsigned int)255 (as a char) returns -1!! */
      switch (name[1])
	{
	case 'n':
	  if ((strings_are_equal(name + 1, "null")) ||
	      (strings_are_equal(name + 1, "nul")))
	    return(chars[0]);

	  if (strings_are_equal(name + 1, "newline"))
	    return(chars[(unsigned char)'\n']);
	  break;

	case 's':
	  if (strings_are_equal(name + 1, "space"))
	    return(chars[(unsigned char)' ']);
	  break;

	case 'r':
	  if (strings_are_equal(name + 1, "return"))
	    return(chars[(unsigned char)'\r']);
	  break;

	case 'l':
	  if (strings_are_equal(name + 1, "linefeed"))
	    return(chars[(unsigned char)'\n']);
	  break;

	case 't':
	  if (strings_are_equal(name + 1, "tab"))
	    return(chars[(unsigned char)'\t']);
	  break;

	case 'a':
	  /* the next 4 are for r7rs */
	  if (strings_are_equal(name + 1, "alarm"))
	    return(chars[7]);
	  break;

	case 'b':
	  if (strings_are_equal(name + 1, "backspace"))
	    return(chars[8]);
	  break;

	case 'e':
	  if (strings_are_equal(name + 1, "escape"))
	    return(chars[0x1b]);
	  break;

	case 'd':
	  if (strings_are_equal(name + 1, "delete"))
	    return(chars[0x7f]);
	  break;

	case 'x':
	  /* #\x is just x, but apparently #\x<num> is int->char? #\x65 -> #\e -- Guile doesn't have this
	   *
	   * r7rs has 2/3/4-byte "characters" of the form #\xcebb but this is not compatible with
	   *   make-string, string-length, and so on.  We'd either have to have 2-byte chars
	   *   so (string-length (make-string 3 #\xcebb)) = 3, or accept 6 here for number of chars.
	   *   Then substring and string-set! and so on have to use utf8 encoding throughout or
	   *   risk changing the string length unexpectedly.
	   */
	  {
	    /* sscanf here misses errors like #\x1.4, but make_atom misses #\x6/3,
	     *   #\x#b0, #\x#e0.0, #\x-0, #\x#e0e100 etc, so we have to do it at
	     *   an even lower level.
	     * another problem: #\xbdca2cbec overflows so lval is -593310740 -> segfault unless caught
	     */
	    bool happy = true;
	    char *tmp;
	    int lval = 0;

	    tmp = (char *)(name + 2);
	    while ((*tmp) && (happy) && (lval >= 0))
	      {
		int dig;
		dig = digits[(int)(*tmp++)];
		if (dig < 16)
		  lval = dig + (lval * 16);
		else happy = false;
	      }
	    if ((happy) &&
		(lval < 256) &&
		(lval >= 0))
	      return(chars[lval]);
	  }
	  break;
	}
    }
  return(unknown_sharp_constant(sc, name));
}


static s7_int string_to_integer(const char *str, int radix, bool *overflow)
{
  bool negative = false;
  s7_int lval = 0;
  int dig;
  char *tmp = (char *)str;
  char *tmp1;

  if (str[0] == '+')
    tmp++;
  else
    {
      if (str[0] == '-')
	{
	  negative = true;
	  tmp++;
	}
    }
  while (*tmp == '0') {tmp++;};
  tmp1 = tmp;

 if (radix == 10)
    {
      while (true)
	{
	  dig = digits[(unsigned char)(*tmp++)];
	  if (dig > 9) break;
#if HAVE_OVERFLOW_CHECKS
	  if (multiply_overflow(lval, (s7_int)10, &lval)) break;
	  if (add_overflow(lval, (s7_int)dig, &lval)) break;
#else
	  lval = dig + (lval * 10);
	  dig = digits[(unsigned char)(*tmp++)];
	  if (dig > 9) break;
	  lval = dig + (lval * 10);
#endif
	}
    }
  else
    {
      while (true)
	{
	  dig = digits[(unsigned char)(*tmp++)];
	  if (dig >= radix) break;
#if HAVE_OVERFLOW_CHECKS
	  if (multiply_overflow(lval, (s7_int)radix, &lval)) break;
	  if (add_overflow(lval, (s7_int)dig, &lval)) break;
#else
	  lval = dig + (lval * radix);
	  dig = digits[(unsigned char)(*tmp++)];
	  if (dig >= radix) break;
	  lval = dig + (lval * radix);
#endif
	}
    }

#if WITH_GMP
  (*overflow) = ((lval > s7_int32_max) ||
		 ((tmp - tmp1) > s7_int_digits_by_radix[radix]));
  /* this tells the string->number readers to create a bignum.  We need to be very
   *    conservative here to catch contexts such as (/ 1/524288 19073486328125)
   */
#else
  if ((tmp - tmp1 - 2) > s7_int_digits_by_radix[radix])
    {
      /* I can't decide what to do with these non-gmp overflows.  Perhaps NAN in all cases?
       *     overflow: 9223372036854775810 -> -9223372036854775806 -- this is not caught currently
       */
      (*overflow) = true;
      if (negative)
	return(s7_int_min);       /* or INFINITY? */
      return(s7_int_max);         /* 0/100000000000000000000000000000000000000000000000000000000000000000000 */
    }
#endif

  if (negative)
    return(-lval);
  return(lval);
}


/*  9223372036854775807                9223372036854775807
 * -9223372036854775808               -9223372036854775808
 * 0000000000000000000000000001.0     1.0
 * 1.0000000000000000000000000000     1.0
 * 1000000000000000000000000000.0e-40 1.0e-12
 * 0.0000000000000000000000000001e40  1.0e12
 * 1.0e00000000000000000001           10.0
 */

static s7_double string_to_double_with_radix(const char *ur_str, int radix, bool *overflow)
{
  /* strtod follows LANG which is not what we want (only "." is decimal point in Scheme).
   *   To overcome LANG in strtod would require screwing around with setlocale which never works.
   *   So we use our own code -- according to valgrind, this function is much faster than strtod.
   *
   * comma as decimal point causes ambiguities: `(+ ,1 2) etc
   */

  int i, sign = 1, frac_len, int_len, dig, max_len, exponent = 0;
  long long int int_part = 0, frac_part = 0;
  char *str;
  char *ipart, *fpart;
  s7_double dval = 0.0;

  /* there's an ambiguity in number notation here if we allow "1e1" or "1.e1" in base 16 (or 15) -- is e a digit or an exponent marker?
   *   but 1e+1, for example disambiguates it -- kind of messy! -- the scheme spec says "e" can only occur in base 10.
   *   mpfr says "e" as exponent only in bases <= 10 -- else use '@' which works in any base.  This can only cause confusion
   *   in scheme, unfortunately, due to the idiotic scheme polar notation.  But we accept "s" and "l" as exponent markers
   *   so, perhaps for radix > 10, the exponent, if any, has to use one of S s L l?  Not "l"!  And "s" originally meant "short".
   *
   * '@' can now be used as the exponent marker (26-Mar-12).
   * Another slight ambiguity: 1+1/2i is parsed as 1 + 0.5i, not 1+1/(2i), or (1+1)/(2i) or (1+1/2)i etc
   */

  max_len = s7_int_digits_by_radix[radix];
  str = (char *)ur_str;

  if (*str == '+')
    str++;
  else
    {
      if (*str == '-')
	{
	  str++;
	  sign = -1;
	}
    }
  while (*str == '0') {str++;};

  ipart = str;
  while (digits[(int)(*str)] < radix) str++;
  int_len = str - ipart;

  if (*str == '.') str++;
  fpart = str;
  while (digits[(int)(*str)] < radix) str++;
  frac_len = str - fpart;

  if ((*str) && (exponent_table[(unsigned char)(*str)]))
    {
      int exp_negative = false;
      str++;
      if (*str == '+')
	str++;
      else
	{
	  if (*str == '-')
	    {
	      str++;
	      exp_negative = true;
	    }
	}
      while ((dig = digits[(int)(*str++)]) < 10) /* exponent itself is always base 10 */
	{
#if HAVE_OVERFLOW_CHECKS
	  if ((int_multiply_overflow(exponent, 10, &exponent)) ||
	      (int_add_overflow(exponent, dig, &exponent)))
	    {
	      exponent = 1000000; /* see below */
	      break;
	    }
#else
	  exponent = dig + (exponent * 10);
#endif
	}
#if (!defined(__GNUC__)) || (__GNUC__ < 5)
      if (exponent < 0)         /* we overflowed, so make sure we notice it below (need to check for 0.0e... first) (Brian Damgaard) */
	exponent = 1000000;     /*   see below for examples -- this number needs to be very big but not too big for add */
#endif
      if (exp_negative)
	exponent = -exponent;

      /*           2e12341234123123123123213123123123 -> 0.0
       * but exp len is not the decider: 2e00000000000000000000000000000000000000001 -> 20.0
       * first zero: 2e123412341231231231231
       * then:     2e12341234123123123123123123 -> inf
       * then:     2e123412341231231231231231231231231231 -> 0.0
       *           2e-123412341231231231231 -> inf
       * but:      0e123412341231231231231231231231231231
       */
    }

#if WITH_GMP
  /* 9007199254740995.0 */
  if (int_len + frac_len >= max_len)
    {
      (*overflow) = true;
      return(0.0);
    }
#endif

  str = ipart;
  if ((int_len + exponent) > max_len)
    {
      /*  12341234.56789e12                   12341234567889999872.0              1.234123456789e+19
       * -1234567890123456789.0              -1234567890123456768.0              -1.2345678901235e+18
       *  12345678901234567890.0              12345678901234567168.0              1.2345678901235e+19
       *  123.456e30                          123456000000000012741097792995328.0 1.23456e+32
       *  12345678901234567890.0e12           12345678901234569054409354903552.0  1.2345678901235e+31
       *  1.234567890123456789012e30          1234567890123456849145940148224.0   1.2345678901235e+30
       *  1e20                                100000000000000000000.0             1e+20
       *  1234567890123456789.0               1234567890123456768.0               1.2345678901235e+18
       *  123.456e16                          1234560000000000000.0               1.23456e+18
       *  98765432101234567890987654321.0e-5  987654321012345728401408.0          9.8765432101235e+23
       *  98765432101234567890987654321.0e-10 9876543210123456512.0               9.8765432101235e+18
       *  0.00000000000000001234e20           1234.0
       *  0.000000000000000000000000001234e30 1234.0
       *  0.0000000000000000000000000000000000001234e40 1234.0
       *  0.000000000012345678909876543210e15 12345.678909877
       *  0e1000                              0.0
       */

      for (i = 0; i < max_len; i++)
	{
	  dig = digits[(int)(*str++)];
	  if (dig < radix)
	    int_part = dig + (int_part * radix);
	  else break;
	}

      /* if the exponent is huge, check for 0 int_part and frac_part before complaining (0e1000 or 0.0e1000)
       */
      if ((int_part == 0) &&
	  (exponent > max_len))
	{
	  /* if frac_part is also 0, return 0.0 */
	  if (frac_len == 0)
	    return(0.0);

	  str = fpart;
	  while ((dig = digits[(int)(*str++)]) < radix)
	    frac_part = dig + (frac_part * radix);
	  if (frac_part == 0)
	    return(0.0);

#if WITH_GMP
	  (*overflow) = true;
#endif
	}

#if WITH_GMP
      (*overflow) = ((int_part > 0) || (exponent > 20));    /* .1e310 is a tricky case */
#endif

      if (int_part != 0) /* 0.<310 zeros here>1e310 for example --
			  *   pow (via ipow) thinks it has to be too big, returns Nan,
			  *   then Nan * 0 -> Nan and the NaN propagates
			  */
	{
	  if (int_len <= max_len)
	    dval = int_part * ipow(radix, exponent);
	  else dval = int_part * ipow(radix, exponent + int_len - max_len);
	}
      else dval = 0.0;

      /* shift by exponent, but if int_len > max_len then we assumed (see below) int_len - max_len 0's on the left */
      /*   using int_to_int or table lookups here instead of pow did not make any difference in speed */

      if (int_len < max_len)
	{
	  int k, flen;
	  str = fpart;

	  for (k = 0; (frac_len > 0) && (k < exponent); k += max_len)
	    {
	      if (frac_len > max_len) flen = max_len; else flen = frac_len;
	      frac_len -= max_len;

	      frac_part = 0;
	      for (i = 0; i < flen; i++)
		frac_part = digits[(int)(*str++)] + (frac_part * radix);

	      if (frac_part != 0)                                /* same pow->NaN problem as above can occur here */
		dval += frac_part * ipow(radix, exponent - flen - k);
	    }
	}
      else
	{
	  /* some of the fraction is in the integer part before the negative exponent shifts it over */
	  if (int_len > max_len)
	    {
	      int ilen;
	      /* str should be at the last digit we read */
	      ilen = int_len - max_len;                          /* we read these above */
	      if (ilen > max_len)
		ilen = max_len;

	      for (i = 0; i < ilen; i++)
		frac_part = digits[(int)(*str++)] + (frac_part * radix);

	      dval += frac_part * ipow(radix, exponent - ilen);
	    }
	}

      return(sign * dval);
    }

  /* int_len + exponent <= max_len */

  if (int_len <= max_len)
    {
      int int_exponent;

      /* a better algorithm (since the inaccuracies are in the radix^exponent portion):
       *   strip off leading zeros and possible sign,
       *   strip off digits beyond max_len, then remove any trailing zeros.
       *     (maybe fiddle with the lowest order digit here for rounding, but I doubt it matters)
       *   read digits until end of number or max_len reached, ignoring the decimal point
       *   get exponent and use it and decimal point location to position the current result integer
       * this always combines the same integer and the same exponent no matter how the number is expressed.
       */

      int_exponent = exponent;
      if (int_len > 0)
	{
	  char *iend;
	  iend = (char *)(str + int_len - 1);
	  while ((*iend == '0') && (iend != str)) {iend--; int_exponent++;}

	  while (str <= iend)
	    int_part = digits[(int)(*str++)] + (int_part * radix);
	}
      if (int_exponent != 0)
	dval = int_part * ipow(radix, int_exponent);
      else dval = (s7_double)int_part;
    }
  else
    {
      int len, flen;
      long long int frpart = 0;

      /* 98765432101234567890987654321.0e-20    987654321.012346
       * 98765432101234567890987654321.0e-29    0.98765432101235
       * 98765432101234567890987654321.0e-30    0.098765432101235
       * 98765432101234567890987654321.0e-28    9.8765432101235
       */

      len = int_len + exponent;
      for (i = 0; i < len; i++)
	int_part = digits[(int)(*str++)] + (int_part * radix);

      flen = -exponent;
      if (flen > max_len)
	flen = max_len;

      for (i = 0; i < flen; i++)
	frpart = digits[(int)(*str++)] + (frpart * radix);

      if (len <= 0)
	dval = int_part + frpart * ipow(radix, len - flen);
      else dval = int_part + frpart * ipow(radix, -flen);
    }

  if (frac_len > 0)
    {
      str = fpart;
      if (frac_len <= max_len)
	{
	  /* splitting out base 10 case saves very little here */
	  /* this ignores trailing zeros, so that 0.3 equals 0.300 */
	  char *fend;

	  fend = (char *)(str + frac_len - 1);
	  while ((*fend == '0') && (fend != str)) {fend--; frac_len--;} /* (= .6 0.6000) */

	  while (str <= fend)
	    frac_part = digits[(int)(*str++)] + (frac_part * radix);
	  dval += frac_part * ipow(radix, exponent - frac_len);

	  /* fprintf(stderr, "frac: %lld, exp: (%d %d) %.20f, val: %.20f\n", frac_part, exponent, frac_len, ipow(radix, exponent - frac_len), dval);
	   * 0.6:    frac:    6, exp: 0.10000000000000000555, val: 0.60000000000000008882
	   * 0.60:   frac:   60, exp: 0.01000000000000000021, val: 0.59999999999999997780
	   * 0.6000: frac: 6000, exp: 0.00010000000000000000, val: 0.59999999999999997780
	   * :(= 0.6 0.60)
	   * #f
	   * :(= #i3/5 0.6)
	   * #f
	   * so (string->number (number->string num)) == num only if both num's are the same text (or you get lucky)
	   * :(= 0.6 6e-1) ; but not 60e-2
	   * #t
	   *
	   * to fix the 0.60 case, we need to ignore trailing post-dot zeros.
	   */
	}
      else
	{
	  if (exponent <= 0)
	    {
	      for (i = 0; i < max_len; i++)
		frac_part = digits[(int)(*str++)] + (frac_part * radix);

	      dval += frac_part * ipow(radix, exponent - max_len);
	    }
	  else
	    {
	      /* 1.0123456789876543210e1         10.12345678987654373771
	       * 1.0123456789876543210e10        10123456789.87654304504394531250
	       * 0.000000010000000000000000e10   100.0
	       * 0.000000010000000000000000000000000000000000000e10 100.0
	       * 0.000000012222222222222222222222222222222222222e10 122.22222222222222
	       * 0.000000012222222222222222222222222222222222222e17 1222222222.222222
	       */

	      int_part = 0;
	      for (i = 0; i < exponent; i++)
		int_part = digits[(int)(*str++)] + (int_part * radix);

	      frac_len -= exponent;
	      if (frac_len > max_len)
		frac_len = max_len;

	      for (i = 0; i < frac_len; i++)
		frac_part = digits[(int)(*str++)] + (frac_part * radix);

	      dval += int_part + frac_part * ipow(radix, -frac_len);
	    }
	}
    }

#if WITH_GMP
  if ((int_part == 0) &&
      (frac_part == 0))
    return(0.0);
  (*overflow) = ((frac_len - exponent) > max_len);
#endif

  return(sign * dval);
}


static s7_pointer make_atom(s7_scheme *sc, char *q, int radix, bool want_symbol, bool with_error)
{
  /* make symbol or number from string */
  #define IS_DIGIT(Chr, Rad) (digits[(unsigned char)Chr] < Rad)

  char c, *p;
  bool has_dec_point1 = false;

  p = q;
  c = *p++;

  /* a number starts with + - . or digit, but so does 1+ for example */

  switch (c)
    {
    case '#':
      return(make_sharp_constant(sc, p, UNNESTED_SHARP, radix, with_error)); /* make_sharp_constant expects the '#' to be removed */

    case '+':
    case '-':
      c = *p++;
      if (c == '.')
	{
	  has_dec_point1 = true;
	  c = *p++;
	}
      if ((!c) || (!IS_DIGIT(c, radix)))
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      break;

    case '.':
      has_dec_point1 = true;
      c = *p++;

      if ((!c) || (!IS_DIGIT(c, radix)))
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      break;

    case '0':        /* these two are always digits */
    case '1':
      break;

    default:
      if (!IS_DIGIT(c, radix))
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      break;
    }

  /* now it's possibly a number -- the first character(s) could be part of a number in the current radix */
  {
    char *slash1 = NULL, *slash2 = NULL, *plus = NULL, *ex1 = NULL, *ex2 = NULL;
    bool has_i = false, has_dec_point2 = false;
    int has_plus_or_minus = 0, current_radix;

#if (!WITH_GMP)
    bool overflow = false;
#endif
    current_radix = radix;  /* current_radix is 10 for the exponent portions, but radix for all the rest */

    for ( ; (c = *p) != 0; ++p)
      {
	/* what about embedded null? (string->number (string #\1 (integer->char 0) #\0))
	 *   currently we stop and return 1, but Guile returns #f
	 */
	if (!IS_DIGIT(c, current_radix))         /* moving this inside the switch statement was much slower */
	  {
	    current_radix = radix;

	    switch (c)
	      {
		/* -------- decimal point -------- */
	      case '.':
		if ((!IS_DIGIT(p[1], current_radix)) &&
		    (!IS_DIGIT(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    if ((has_dec_point1) || (slash1))
		      return((want_symbol) ? make_symbol(sc, q) : sc->F);
		    has_dec_point1 = true;
		  }
		else
		  {
		    if ((has_dec_point2) || (slash2))
		      return((want_symbol) ? make_symbol(sc, q) : sc->F);
		    has_dec_point2 = true;
		  }
		continue;


		/* -------- exponent marker -------- */
#if WITH_EXTRA_EXPONENT_MARKERS
		/* 1st 3d-perspective 0.0f 128.0f 3d 1s -- in 2 million lines of public scheme code, not one actual use! */
	      case 's': case 'S':
	      case 'd': case 'D':
	      case 'f': case 'F':
	      case 'l': case 'L':
#endif
	      case 'e': case 'E':
		if (current_radix > 10)
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);
		/* see note above */
		/* fall through -- if '@' used, radices>10 are ok */

	      case '@':
		current_radix = 10;

		if (((ex1) ||
		     (slash1)) &&
		    (has_plus_or_minus == 0)) /* ee */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (((ex2) ||
		     (slash2)) &&
		    (has_plus_or_minus != 0)) /* 1+1.0ee */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if ((!IS_DIGIT(p[-1], radix)) && /* was current_radix but that's always 10! */
		    (p[-1] != '.'))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    ex1 = p;
		    has_dec_point1 = true; /* decimal point illegal from now on */
		  }
		else
		  {
		    ex2 = p;
		    has_dec_point2 = true;
		  }
		p++;
		if ((*p == '-') || (*p == '+')) p++;
		if (IS_DIGIT(*p, current_radix))
		  continue;
		break;


		/* -------- internal + or - -------- */
	      case '+':
	      case '-':
		if (has_plus_or_minus != 0) /* already have the separator */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (c == '+') has_plus_or_minus = 1; else has_plus_or_minus = -1;
		plus = (char *)(p + 1);
		continue;

		/* ratio marker */
	      case '/':
		if ((has_plus_or_minus == 0) &&
		    ((ex1) ||
		     (slash1) ||
		     (has_dec_point1)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if ((has_plus_or_minus != 0) &&
		    ((ex2) ||
		     (slash2) ||
		     (has_dec_point2)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  slash1 = (char *)(p + 1);
		else slash2 = (char *)(p + 1);

		if ((!IS_DIGIT(p[1], current_radix)) ||
		    (!IS_DIGIT(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		continue;


		/* -------- i for the imaginary part -------- */
	      case 'i':
		if ((has_plus_or_minus != 0) &&
		    (!has_i))
		  {
		    has_i = true;
		    continue;
		  }
		break;

	      default:
		break;
	      }
	    return((want_symbol) ? make_symbol(sc, q) : sc->F);
	  }
      }

    if ((has_plus_or_minus != 0) &&        /* that is, we have an internal + or - */
	(!has_i))                          /*   but no i for the imaginary part */
      return((want_symbol) ? make_symbol(sc, q) : sc->F);

    if (has_i)
      {
#if (!WITH_GMP)
	s7_double rl = 0.0, im = 0.0;
#else
	char e1 = 0, e2 = 0;
#endif
	s7_pointer result;
	int len;
	char ql1, pl1;

	len = safe_strlen(q);

	if (q[len - 1] != 'i')
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);

	/* save original string */
	ql1 = q[len - 1];
	pl1 = (*(plus - 1));
#if WITH_GMP
	if (ex1) {e1 = *ex1; (*ex1) = '@';} /* for mpfr */
	if (ex2) {e2 = *ex2; (*ex2) = '@';}
#endif

	/* look for cases like 1+i */
	if ((q[len - 2] == '+') || (q[len - 2] == '-'))
	  q[len - 1] = '1';
	else q[len - 1] = '\0'; /* remove 'i' */

	(*((char *)(plus - 1))) = '\0';

	/* there is a slight inconsistency here:
	   1/0      -> nan.0
           1/0+0i   -> inf.0 (0/1+0i is 0.0)
	   #i1/0+0i -> inf.0
	   0/0      -> nan.0
	   0/0+0i   -> -nan.0
	*/

#if (!WITH_GMP)
	if ((has_dec_point1) ||
	    (ex1))
	  {
	    /* (string->number "1100.1+0.11i" 2) -- need to split into 2 honest reals before passing to non-base-10 str->dbl */
	    rl = string_to_double_with_radix(q, radix, &overflow);
	  }
	else
	  {
	    if (slash1)
	      {
		/* here the overflow could be innocuous if it's in the denominator and the numerator is 0
		 *    0/100000000000000000000000000000000000000-0i
		 */
		s7_int num, den;
		num = string_to_integer(q, radix, &overflow);
		den = string_to_integer(slash1, radix, &overflow);
		if (den == 0)
		  rl = NAN;
		else
		  {
		    if (num == 0)
		      {
			rl = 0.0;
			overflow = false;
		      }
		    else rl = (s7_double)num / (s7_double)den;
		  }
	      }
	    else rl = (s7_double)string_to_integer(q, radix, &overflow);
	    if (overflow) return(real_NaN);
	  }
	if (rl == -0.0) rl = 0.0;

	if ((has_dec_point2) ||
	    (ex2))
	  im = string_to_double_with_radix(plus, radix, &overflow);
	else
	  {
	    if (slash2)
	      {
		/* same as above: 0-0/100000000000000000000000000000000000000i
		 */
		s7_int num, den;
		num = string_to_integer(plus, radix, &overflow);
		den = string_to_integer(slash2, radix, &overflow);
		if (den == 0)
		  im = NAN;
		else
		  {
		    if (num == 0)
		      {
			im = 0.0;
			overflow = false;
		      }
		    else im = (s7_double)num / (s7_double)den;
		  }
	      }
	    else im = (s7_double)string_to_integer(plus, radix, &overflow);
	    if (overflow) return(real_NaN);
	  }
	if ((has_plus_or_minus == -1) &&
	    (im != 0.0))
	  im = -im;
	result = s7_make_complex(sc, rl, im);
#else
	result = string_to_either_complex(sc, q, slash1, ex1, has_dec_point1, plus, slash2, ex2, has_dec_point2, radix, has_plus_or_minus);
#endif

	/* restore original string */
	q[len - 1] = ql1;
	(*((char *)(plus - 1))) = pl1;
#if WITH_GMP
	if (ex1) (*ex1) = e1;
	if (ex2) (*ex2) = e2;
#endif

	return(result);
      }

    /* not complex */
    if ((has_dec_point1) ||
	(ex1))
      {
	s7_pointer result;

	if (slash1)  /* not complex, so slash and "." is not a number */
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);

#if (!WITH_GMP)
	result = make_real(sc, string_to_double_with_radix(q, radix, &overflow));
#else
	{
	  char old_e = 0;
	  if (ex1)
	    {
	      old_e = (*ex1);
	      (*ex1) = '@';
	    }
	  result = string_to_either_real(sc, q, radix);
	  if (ex1)
	    (*ex1) = old_e;
	}
#endif
	return(result);
      }

    /* not real */
    if (slash1)
#if (!WITH_GMP)
      {
	s7_int n, d;

	n = string_to_integer(q, radix, &overflow);
	d = string_to_integer(slash1, radix, &overflow);

	if ((n == 0) && (d != 0))                        /* 0/100000000000000000000000000000000000000 */
	  return(small_int(0));
	if ((d == 0) || (overflow))
	  return(real_NaN);
	/* it would be neat to return 1 from 10000000000000000000000000000/10000000000000000000000000000
	 *   but q is the entire number ('/' included) and slash1 is the stuff after the '/', and every
	 *   big number comes through here, so there's no clean and safe way to check that q == slash1.
	 */
	return(s7_make_ratio(sc, n, d));
      }
#else
    return(string_to_either_ratio(sc, q, slash1, radix));
#endif

    /* integer */
#if (!WITH_GMP)
    {
      s7_int x;
      x = string_to_integer(q, radix, &overflow);
      if (overflow)
	return((q[0] == '-') ? real_minus_infinity : real_infinity);
      return(make_integer(sc, x));
    }
#else
    return(string_to_either_integer(sc, q, radix));
#endif
  }
}


static s7_pointer s7_string_to_number(s7_scheme *sc, char *str, int radix)
{
  s7_pointer x;
  x = make_atom(sc, str, radix, NO_SYMBOLS, WITHOUT_OVERFLOW_ERROR);
  if (s7_is_number(x))  /* only needed because str might start with '#' and not be a number (#t for example) */
    return(x);
  return(sc->F);
}


static s7_pointer g_string_to_number_1(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  #define H_string_to_number "(string->number str (radix 10)) converts str into a number. \
If str does not represent a number, string->number returns #f.  If 'str' has an embedded radix, \
the 'radix' it is ignored: (string->number \"#x11\" 2) -> 17 not 3."
  #define Q_string_to_number s7_make_signature(sc, 3, s7_make_signature(sc, 2, sc->is_number_symbol, sc->is_boolean_symbol), sc->is_string_symbol, sc->is_integer_symbol)

  s7_int radix = 0;
  char *str;

  if (!is_string(car(args)))
    method_or_bust(sc, car(args), caller, args, T_STRING, 1);

  if (is_pair(cdr(args)))
    {
      s7_pointer rad, p;
      rad = cadr(args);
      if (!s7_is_integer(rad))
	{
	  if (!s7_is_integer(p = check_values(sc, rad, cdr(args))))
	    method_or_bust(sc, rad, caller, args, T_INTEGER, 2);
	  rad = p;
	}
      radix = s7_integer(rad);
      if ((radix < 2) ||              /* what about negative int as base (Knuth), reals such as phi, and some complex like -1+i */
	  (radix > 16))               /* the only problem here is printing the number; perhaps put each digit in "()" in base 10: (123)(0)(34) */
	return(out_of_range(sc, caller, small_int(2), rad, a_valid_radix_string));
    }
  else radix = 10;

  str = (char *)string_value(car(args));
  if ((!str) || (!(*str)))
    return(sc->F);

  switch (str[0])
    {
    case 'n':
      if (safe_strcmp(str, "nan.0"))
	return(real_NaN);
      break;

    case 'i':
      if (safe_strcmp(str, "inf.0"))
	return(real_infinity);
      break;

    case '-':
      if ((str[1] == 'i') && (safe_strcmp((const char *)(str + 1), "inf.0")))
	return(real_minus_infinity);
      break;

    case '+':
      if ((str[1] == 'i') && (safe_strcmp((const char *)(str + 1), "inf.0")))
	return(real_infinity);
      break;
    }
  return(s7_string_to_number(sc, str, radix));
}


static s7_pointer g_string_to_number(s7_scheme *sc, s7_pointer args)
{
  return(g_string_to_number_1(sc, args, sc->string_to_number_symbol));
}

static s7_pointer c_string_to_number(s7_scheme *sc, s7_pointer n)
{
  return(g_string_to_number_1(sc, set_plist_1(sc, n), sc->string_to_number_symbol));
}

PF_TO_PF(string_to_number, c_string_to_number)


static bool numbers_are_eqv(s7_pointer a, s7_pointer b)
{
  if (type(a) != type(b)) /* (eqv? 1 1.0) -> #f! */
    return(false);

  switch (type(a))
    {
    case T_INTEGER:
      return((integer(a) == integer(b)));

    case T_RATIO:
      return((numerator(a) == numerator(b)) &&
	     (denominator(a) == denominator(b)));

    case T_REAL:
      if (is_NaN(real(a)))
	return(false);
      return(real(a) == real(b));

    case T_COMPLEX:
      if ((is_NaN(real_part(a))) ||
	  (is_NaN(imag_part(a))))
	return(false);
      return((real_part(a) == real_part(b)) &&
	     (imag_part(a) == imag_part(b)));

    default:
#if WITH_GMP
      if ((is_big_number(a)) || (is_big_number(b))) /* this can happen if (member bignum ...) -> memv */
	return(big_numbers_are_eqv(a, b));
#endif
      break;
    }
  return(false);
}


static bool is_rational_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_rational(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_rational_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}


/* -------------------------------- abs -------------------------------- */
#if (!WITH_GMP)
static s7_pointer g_abs(s7_scheme *sc, s7_pointer args)
{
  #define H_abs "(abs x) returns the absolute value of the real number x"
  #define Q_abs s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_real_symbol)  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) < 0)
	{
	  if (integer(x) == s7_int_min)
	    return(make_integer(sc, s7_int_max));
	  return(make_integer(sc, -integer(x)));
	}
      return(x);

    case T_RATIO:
      if (numerator(x) < 0)
	{
	  if (numerator(x) == s7_int_min)
	    return(s7_make_ratio(sc, s7_int_max, denominator(x)));
	  return(s7_make_ratio(sc, -numerator(x), denominator(x)));
	}
      return(x);

    case T_REAL:
      if (is_NaN(real(x)))                  /* (abs -nan.0) -> nan.0, not -nan.0 */
	return(real_NaN);
      if (real(x) < 0.0)
	return(make_real(sc, -real(x)));
      return(x);

    default:
      method_or_bust(sc, x, sc->abs_symbol, args, T_REAL, 0);
    }
}

static s7_int c_abs_i(s7_scheme *sc, s7_int arg) {return((arg < 0) ? (-arg) : arg);}
IF_TO_IF(abs, c_abs_i)

static s7_double c_abs_r(s7_scheme *sc, s7_double arg) {return((arg < 0.0) ? (-arg) : arg);} 
DIRECT_RF_TO_RF(fabs)


/* -------------------------------- magnitude -------------------------------- */

static double my_hypot(double x, double y)
{
  /* according to callgrind, this is much faster than libc's hypot */
  if (x == 0.0) return(fabs(y));
  if (y == 0.0) return(fabs(x));
  if (x == y) return(1.414213562373095 * fabs(x));
  if ((is_NaN(x)) || (is_NaN(y))) return(NAN);
  return(sqrt(x * x + y * y));
}

static s7_pointer g_magnitude(s7_scheme *sc, s7_pointer args)
{
  #define H_magnitude "(magnitude z) returns the magnitude of z"
  #define Q_magnitude s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  s7_pointer x;
  x = car(args);

  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == s7_int_min)
	return(make_integer(sc, s7_int_max));
      /* (magnitude -9223372036854775808) -> -9223372036854775808
       *   same thing happens in abs, lcm and gcd: (gcd -9223372036854775808) -> -9223372036854775808
       */
      if (integer(x) < 0)
        return(make_integer(sc, -integer(x)));
      return(x);

    case T_RATIO:
      if (numerator(x) < 0)
	return(s7_make_ratio(sc, -numerator(x), denominator(x)));
      return(x);

    case T_REAL:
      if (is_NaN(real(x)))                 /* (magnitude -nan.0) -> nan.0, not -nan.0 */
	return(real_NaN);
      if (real(x) < 0.0)
	return(make_real(sc, -real(x)));
      return(x);

    case T_COMPLEX:
      return(make_real(sc, my_hypot(imag_part(x), real_part(x))));

    default:
      method_or_bust_with_type(sc, x, sc->magnitude_symbol, args, a_number_string, 0);
    }
}

IF_TO_IF(magnitude, c_abs_i)
RF_TO_RF(magnitude, c_abs_r)



/* -------------------------------- rationalize -------------------------------- */
static s7_pointer g_rationalize(s7_scheme *sc, s7_pointer args)
{
  #define H_rationalize "(rationalize x err) returns the ratio with lowest denominator within err of x"
  #define Q_rationalize s7_make_signature(sc, 3, sc->is_rational_symbol, sc->is_real_symbol, sc->is_real_symbol)
  s7_double err;
  s7_pointer x;

  x = car(args);
  if (!s7_is_real(x))
    method_or_bust(sc, x, sc->rationalize_symbol, args, T_REAL, 1);

  if (is_not_null(cdr(args)))
    {
      s7_pointer ex;
      ex = cadr(args);
      if (!s7_is_real(ex))
	method_or_bust(sc, ex, sc->rationalize_symbol, args, T_REAL, 2);

      err = real_to_double(sc, ex, "rationalize");
      if (is_NaN(err))
	return(out_of_range(sc, sc->rationalize_symbol, small_int(2), cadr(args), its_nan_string));
      if (err < 0.0) err = -err;
    }
  else err = sc->default_rationalize_error;

  switch (type(x))
    {
    case T_INTEGER:
      {
	s7_int a, b, pa;
	if (err < 1.0) return(x);
	a = s7_integer(x);
	if (a < 0) pa = -a; else pa = a;
	if (err >= pa) return(small_int(0));
	b = (s7_int)err;
	pa -= b;
	if (a < 0)
	  return(make_integer(sc, -pa));
	return(make_integer(sc, pa));
      }

    case T_RATIO:
      if (err == 0.0)
	return(x);

    case T_REAL:
      {
	s7_double rat;
	s7_int numer = 0, denom = 1;

	rat = real_to_double(sc, x, "rationalize");

	if ((is_NaN(rat)) || (is_inf(rat)))
	  return(wrong_type_argument_with_type(sc, sc->rationalize_symbol, 1, x, a_normal_real_string));

	if (err >= fabs(rat))
	  return(small_int(0));

	if ((rat > 9.2233720368548e+18) || (rat < -9.2233720368548e+18))
	  return(out_of_range(sc, sc->rationalize_symbol, small_int(1), x, its_too_large_string));

	if ((fabs(rat) + fabs(err)) < 1.0e-18)
	  err = 1.0e-18;
	/* (/ 1.0 most-positive-fixnum) is 1.0842021e-19, so if we let err be less than that,
	 * (rationalize 1e-19 1e-20) hangs, but this only affects the initial ceiling, I believe.
	 */

	if (fabs(rat) < fabs(err))
	  return(small_int(0));

	if (c_rationalize(rat, err, &numer, &denom))
	  return(s7_make_ratio(sc, numer, denom));

	return(sc->F);
      }
    }
  return(sc->F); /* make compiler happy */
}

static s7_pointer c_rats(s7_scheme *sc, s7_pointer x) {return(g_rationalize(sc, set_plist_1(sc, x)));}
PF_TO_PF(rationalize, c_rats)


/* -------------------------------- angle -------------------------------- */
static s7_pointer g_angle(s7_scheme *sc, s7_pointer args)
{
  #define H_angle "(angle z) returns the angle of z"
  #define Q_angle s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  s7_pointer x;
  /* (angle inf+infi) -> 0.78539816339745 ? 
   *   I think this should be -pi < ang <= pi
   */

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) < 0)
	return(real_pi);
      return(small_int(0));

    case T_RATIO:
      if (numerator(x) < 0)
	return(real_pi);
      return(small_int(0));

    case T_REAL:
      if (is_NaN(real(x))) return(x);
      if (real(x) < 0.0)
	return(real_pi);
      return(real_zero);

    case T_COMPLEX:
      return(make_real(sc, atan2(imag_part(x), real_part(x))));

    default:
      method_or_bust_with_type(sc, x, sc->angle_symbol, args, a_number_string, 0);
    }
}


/* -------------------------------- make-polar -------------------------------- */
#if (!WITH_PURE_S7)
static s7_pointer g_make_polar(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  s7_double ang, mag;
  #define H_make_polar "(make-polar mag ang) returns a complex number with magnitude mag and angle ang"
  #define Q_make_polar s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_real_symbol, sc->is_real_symbol)

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) == 0) return(x);            /* (make-polar 0 1) -> 0 */
	  if (integer(y) == 0) return(x);            /* (make-polar 1 0) -> 1 */
	  mag = (s7_double)integer(x);
	  ang = (s7_double)integer(y);
	  break;

	case T_RATIO:
	  if (integer(x) == 0) return(x);
	  mag = (s7_double)integer(x);
	  ang = (s7_double)fraction(y);
	  break;

	case T_REAL:
	  ang = real(y);
	  if (ang == 0.0) return(x);
	  if (is_NaN(ang)) return(y);
	  if (is_inf(ang)) return(real_NaN);
	  if ((ang == M_PI) || (ang == -M_PI)) return(make_integer(sc, -integer(x)));
	  mag = (s7_double)integer(x);
	  break;

	default:
	  method_or_bust(sc, y, sc->make_polar_symbol, args, T_REAL, 2);
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  mag = (s7_double)fraction(x);
	  ang = (s7_double)integer(y);
	  break;

	case T_RATIO:
	  mag = (s7_double)fraction(x);
	  ang = (s7_double)fraction(y);
	  break;

	case T_REAL:
	  ang = real(y);
	  if (ang == 0.0) return(x);
	  if (is_NaN(ang)) return(y);
	  if (is_inf(ang)) return(real_NaN);
	  if ((ang == M_PI) || (ang == -M_PI)) return(s7_make_ratio(sc, -numerator(x), denominator(x)));
	  mag = (s7_double)fraction(x);
	  break;

	default:
	  method_or_bust(sc, y, sc->make_polar_symbol, args, T_REAL, 2);
	}
      break;

    case T_REAL:
      mag = real(x);
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(mag)) return(x);
	  if (integer(y) == 0) return(x);
	  ang = (s7_double)integer(y);
	  break;

	case T_RATIO:
	  if (is_NaN(mag)) return(x);
	  ang = (s7_double)fraction(y);
	  break;

	case T_REAL:
	  if (is_NaN(mag)) return(x);
	  ang = real(y);
	  if (ang == 0.0) return(x);
	  if (is_NaN(ang)) return(y);
	  if (is_inf(ang)) return(real_NaN);
	  break;

	default:
	  method_or_bust(sc, y, sc->make_polar_symbol, args, T_REAL, 2);
	}
      break;

    default:
      method_or_bust(sc, x, sc->make_polar_symbol, args, T_REAL, 1);
    }

  return(s7_make_complex(sc, mag * cos(ang), mag * sin(ang)));

  /* since sin is inaccurate for large arguments, so is make-polar:
   *    (make-polar 1.0 1e40) -> -0.76267273202438+0.64678458842683i, not 8.218988919070239214448025364432557517335E-1-5.696334009536363273080341815735687231337E-1i
   */
}

static s7_pointer c_make_polar_2(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_make_polar(sc, set_plist_2(sc, x, y)));}
PF2_TO_PF(make_polar, c_make_polar_2)
#endif


/* -------------------------------- complex -------------------------------- */
static s7_pointer g_complex(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  #define H_complex "(complex x1 x2) returns a complex number with real-part x1 and imaginary-part x2"
  #define Q_complex s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_real_symbol, sc->is_real_symbol)

  x = car(args);
  y = cadr(args);

  switch (type(y))
    {
    case T_INTEGER:
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  return(s7_make_complex(sc, (s7_double)integer(x), (s7_double)integer(y)));

	case T_RATIO:
	  if (integer(y) == 0) return(x);
	  return(s7_make_complex(sc, (s7_double)fraction(x), (s7_double)integer(y)));

	case T_REAL:
	  if (integer(y) == 0) return(x);
	  return(s7_make_complex(sc, real(x), (s7_double)integer(y)));

	default:
	  method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1);
	}

    case T_RATIO:
      switch (type(x))
	{
	case T_INTEGER: return(s7_make_complex(sc, (s7_double)integer(x), (s7_double)fraction(y)));
	case T_RATIO:   return(s7_make_complex(sc, (s7_double)fraction(x), (s7_double)fraction(y)));
	case T_REAL:    return(s7_make_complex(sc, real(x), (s7_double)fraction(y)));
	default:
	  method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1);
	}

    case T_REAL:
      switch (type(x))
	{
	case T_INTEGER:
	  if (real(y) == 0.0) return(x);
	  return(s7_make_complex(sc, (s7_double)integer(x), real(y)));

	case T_RATIO:
	  if (real(y) == 0.0) return(x);
	  return(s7_make_complex(sc, (s7_double)fraction(x), real(y)));

	case T_REAL:
	  if (real(y) == 0.0) return(x);
	  return(s7_make_complex(sc, real(x), real(y)));

	default:
	  method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1);
	}

    default:
      method_or_bust(sc, (is_let(x)) ? x : y, sc->complex_symbol, args, T_REAL, 2);
    }
}

static s7_pointer c_make_complex_2(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_complex(sc, set_plist_2(sc, x, y)));}
PF2_TO_PF(make_complex, c_make_complex_2)


/* -------------------------------- exp -------------------------------- */
static s7_pointer g_exp(s7_scheme *sc, s7_pointer args)
{
  #define H_exp "(exp z) returns e^z, (exp 1) is 2.718281828459"
  #define Q_exp pcl_n  

  s7_pointer x;

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(1));                       /* (exp 0) -> 1 */
      return(make_real(sc, exp((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, exp((s7_double)fraction(x))));

    case T_REAL:
      return(make_real(sc, exp(real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, cexp(as_c_complex(x))));
      /* this is inaccurate for large arguments:
       *   (exp 0+1e20i) -> -0.66491178990701-0.74692189125949i, not 7.639704044417283004001468027378811228331E-1-6.45251285265780844205811711312523007406E-1i
       */
#else
      return(out_of_range(sc, sc->exp_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->exp_symbol, args, a_number_string, 0);
    }
}

DIRECT_RF_TO_RF(exp)


/* -------------------------------- log -------------------------------- */

#if __cplusplus
#define LOG_2 1.4426950408889634074
#else
#define LOG_2 1.4426950408889634073599246810018921L /* (/ (log 2.0)) */
#endif

static s7_pointer g_log(s7_scheme *sc, s7_pointer args)
{
  #define H_log "(log z1 (z2 e)) returns log(z1) / log(z2) where z2 (the base) defaults to e: (log 8 2) = 3"
  #define Q_log pcl_n

  s7_pointer x;
  x = car(args);
  if (!s7_is_number(x))
    method_or_bust_with_type(sc, x, sc->log_symbol, args, a_number_string, 1);

  if (is_pair(cdr(args)))
    {
      s7_pointer y;

      y = cadr(args);
      if (!(s7_is_number(y)))
	method_or_bust_with_type(sc, y, sc->log_symbol, args, a_number_string, 2);

      if (y == small_int(2))
	{
	  /* (define (2^n? x) (and (not (zero? x)) (zero? (logand x (- x 1))))) */
	  if (is_integer(x))
	    {
	      s7_int ix;
	      ix = s7_integer(x);
	      if (ix > 0)
		{
		  s7_double fx;
#if (__ANDROID__) || (MS_WINDOWS) || ((__GNUC__) && ((__GNUC__ < 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ <= 4))))
		  /* just a guess -- log2 gets a warning in gcc 4.3.2, but not in 4.4.4 */
		  fx = log((double)ix) / log(2.0);
#else
		  fx = log2((double)ix);
#endif
		  /* (s7_int)fx rounds (log 8 2) to 2 in FreeBSD! */
#if ((__GNUC__) && ((__GNUC__ < 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ < 4))))
		  return(make_real(sc, fx));
#else
		  if ((ix & (ix - 1)) == 0)
		    return(make_integer(sc, (s7_int)s7_round(fx)));
		  return(make_real(sc, fx));
#endif
		}
	    }
	  if ((s7_is_real(x)) &&
	      (s7_is_positive(x)))
	    return(make_real(sc, log(real_to_double(sc, x, "log")) * LOG_2));
	  return(s7_from_c_complex(sc, clog(s7_to_c_complex(x)) * LOG_2));
	}

      if ((x == small_int(1)) && (y == small_int(1)))  /* (log 1 1) -> 0 (this is NaN in the bignum case) */
	return(small_int(0));

      /* (log 1 0) must be 0 since everyone says (expt 0 0) is 1 */
      if (s7_is_zero(y))
	{
	  if ((y == small_int(0)) &&
	      (x == small_int(1)))
	    return(y);
	  return(out_of_range(sc, sc->log_symbol, small_int(2), y, make_string_wrapper(sc, "can't be 0")));
	}

      if (s7_is_one(y))          /* this used to raise an error, but the bignum case is simpler if we return inf */
	{
	  if (s7_is_one(x))      /* but (log 1.0 1.0) -> 0.0 */
	    return(real_zero);
	  return(real_infinity); /* currently (log 1/0 1) is inf? */
	}

      if ((s7_is_real(x)) &&
	  (s7_is_real(y)) &&
	  (s7_is_positive(x)) &&
	  (s7_is_positive(y)))
	{
	  if ((s7_is_rational(x)) &&
	      (s7_is_rational(y)))
	    {
	      s7_double res;
	      s7_int ires;
	      res = log(rational_to_double(sc, x)) / log(rational_to_double(sc, y));
	      ires = (s7_int)res;
	      if (res - ires == 0.0)
		return(make_integer(sc, ires));   /* (log 8 2) -> 3 or (log 1/8 2) -> -3 */
	      return(make_real(sc, res));         /* perhaps use rationalize here? (log 2 8) -> 1/3 */
	    }
	  return(make_real(sc, log(real_to_double(sc, x, "log")) / log(real_to_double(sc, y, "log"))));
	}
      return(s7_from_c_complex(sc, clog(s7_to_c_complex(x)) / clog(s7_to_c_complex(y))));
    }

  if (s7_is_real(x))
    {
      if (s7_is_positive(x))
	return(make_real(sc, log(real_to_double(sc, x, "log"))));
      return(s7_make_complex(sc, log(-real_to_double(sc, x, "log")), M_PI));
    }
  return(s7_from_c_complex(sc, clog(s7_to_c_complex(x))));
}


/* -------------------------------- sin -------------------------------- */
static s7_pointer g_sin(s7_scheme *sc, s7_pointer args)
{
  #define H_sin "(sin z) returns sin(z)"
  #define Q_sin pcl_n  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_REAL:
      return(make_real(sc, sin(real(x))));

    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                      /* (sin 0) -> 0 */
      return(make_real(sc, sin((s7_double)integer(x))));

    case T_RATIO:
      return(make_real(sc, sin((s7_double)(fraction(x)))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, csin(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->sin_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->sin_symbol, args, a_number_string, 0);
    }

  /* sin is totally inaccurate over about 1e18.  There's a way to get true results,
   *   but it involves fancy "range reduction" techniques.
   *   This means that lots of things are inaccurate:
   * (sin (remainder 1e22 (* 2 pi)))
   * -0.57876806033477
   * but it should be -8.522008497671888065747423101326159661908E-1
   * ---
   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22 !!
   *   it should be 5.263007914620499494429139986095833592117E0
   */
}

DIRECT_RF_TO_RF(sin)


/* -------------------------------- cos -------------------------------- */
static s7_pointer g_cos(s7_scheme *sc, s7_pointer args)
{
  #define H_cos "(cos z) returns cos(z)"
  #define Q_cos pcl_n  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_REAL:
      return(make_real(sc, cos(real(x))));

    case T_INTEGER:
      if (integer(x) == 0) return(small_int(1));                     /* (cos 0) -> 1 */
      return(make_real(sc, cos((s7_double)integer(x))));

    case T_RATIO:
      return(make_real(sc, cos((s7_double)(fraction(x)))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, ccos(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->cos_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->cos_symbol, args, a_number_string, 0);
    }
}

DIRECT_RF_TO_RF(cos)


/* -------------------------------- tan -------------------------------- */
static s7_pointer g_tan(s7_scheme *sc, s7_pointer args)
{
  #define H_tan "(tan z) returns tan(z)"
  #define Q_tan pcl_n  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_REAL:
      return(make_real(sc, tan(real(x))));

    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                      /* (tan 0) -> 0 */
      return(make_real(sc, tan((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, tan((s7_double)(fraction(x)))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      if (imag_part(x) > 350.0)
	return(s7_make_complex(sc, 0.0, 1.0));
      if (imag_part(x) < -350.0)
	return(s7_make_complex(sc, 0.0, -1.0));
      return(s7_from_c_complex(sc, ctan(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->tan_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->tan_symbol, args, a_number_string, 0);
    }
}

DIRECT_RF_TO_RF(tan)


/* -------------------------------- asin -------------------------------- */
static s7_pointer c_asin(s7_scheme *sc, s7_double x)
{
  s7_double absx, recip;
  s7_complex result;

  absx = fabs(x);
  if (absx <= 1.0)
    return(make_real(sc, asin(x)));

  /* otherwise use maxima code: */
  recip = 1.0 / absx;
  result = (M_PI / 2.0) - (_Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip)))));
  if (x < 0.0)
    return(s7_from_c_complex(sc, -result));
  return(s7_from_c_complex(sc, result));
}

static s7_pointer g_asin_1(s7_scheme *sc, s7_pointer n)
{
  switch (type(n))
    {
    case T_INTEGER:
      if (integer(n) == 0) return(small_int(0));                    /* (asin 0) -> 0 */
      /* in netBSD, (asin 2) returns 0.25383842987008+0.25383842987008i according to Peter Bex */
      return(c_asin(sc, (s7_double)integer(n)));

    case T_RATIO:
      return(c_asin(sc, (s7_double)numerator(n) / (s7_double)denominator(n)));

    case T_REAL:
      return(c_asin(sc, real(n)));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      /* if either real or imag part is very large, use explicit formula, not casin */
      /*   this code taken from sbcl's src/code/irrat.lisp */
      /* break is around x+70000000i */

      if ((fabs(real_part(n)) > 1.0e7) ||
	  (fabs(imag_part(n)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z;
	  z = as_c_complex(n);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);
	  return(s7_make_complex(sc, atan(real_part(n) / creal(sq1mz * sq1pz)), asinh(cimag(sq1pz * conj(sq1mz)))));
	}
      return(s7_from_c_complex(sc, casin(as_c_complex(n))));
#else
      return(out_of_range(sc, sc->asin_symbol, small_int(1), n, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, n, sc->asin_symbol, list_1(sc, n), a_number_string, 0);
    }
}

static s7_pointer g_asin(s7_scheme *sc, s7_pointer args)
{
  #define H_asin "(asin z) returns asin(z); (sin (asin x)) = x"
  #define Q_asin pcl_n  

  return(g_asin_1(sc, car(args)));
}

R_P_F_TO_PF(asin, c_asin, g_asin_1, g_asin_1)
/* g_asin_1 is safe for the gf case because it won't trigger the GC before it is done with its argument */


/* -------------------------------- acos -------------------------------- */
static s7_pointer c_acos(s7_scheme *sc, s7_double x)
{
  s7_double absx, recip;
  s7_complex result;

  absx = fabs(x);
  if (absx <= 1.0)
    return(make_real(sc, acos(x)));

  /* else follow maxima again: */
  recip = 1.0 / absx;
  if (x > 0.0)
    result = _Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  else result = M_PI - _Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  return(s7_from_c_complex(sc, result));
}

static s7_pointer g_acos_1(s7_scheme *sc, s7_pointer n)
{
  switch (type(n))
    {
    case T_INTEGER:
      if (integer(n) == 1) return(small_int(0));
      return(c_acos(sc, (s7_double)integer(n)));

    case T_RATIO:
      return(c_acos(sc, (s7_double)numerator(n) / (s7_double)denominator(n)));

    case T_REAL:
      return(c_acos(sc, real(n)));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      /* if either real or imag part is very large, use explicit formula, not cacos */
      /*   this code taken from sbcl's src/code/irrat.lisp */

      if ((fabs(real_part(n)) > 1.0e7) ||
	  (fabs(imag_part(n)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z;
	  z = as_c_complex(n);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);
	  return(s7_make_complex(sc, 2.0 * atan(creal(sq1mz) / creal(sq1pz)), asinh(cimag(sq1mz * conj(sq1pz)))));
	}
      return(s7_from_c_complex(sc, cacos(s7_to_c_complex(n))));
#else
      return(out_of_range(sc, sc->acos_symbol, small_int(1), n, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, n, sc->acos_symbol, list_1(sc, n), a_number_string, 0);
    }
}

static s7_pointer g_acos(s7_scheme *sc, s7_pointer args)
{
  #define H_acos "(acos z) returns acos(z); (cos (acos 1)) = 1"
  #define Q_acos pcl_n  
  return(g_acos_1(sc, car(args)));
}

R_P_F_TO_PF(acos, c_acos, g_acos_1, g_acos_1)


/* -------------------------------- atan -------------------------------- */

static s7_double c_atan(s7_scheme *sc, s7_double x, s7_double y)
{
  return(atan2(x, y));
}

static s7_pointer g_atan(s7_scheme *sc, s7_pointer args)
{
  #define H_atan "(atan z) returns atan(z), (atan y x) returns atan(y/x)"
  #define Q_atan s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_number_symbol, sc->is_real_symbol)
  /* actually if there are two args, both should be real, but how to express that in the signature? */
  s7_pointer x, y;
  s7_double x1, x2;

  /* currently (atan inf.0 inf.0) -> 0.78539816339745, and (atan inf.0 -inf.0) -> 2.3561944901923 (etc) */

  x = car(args);
  if (!is_pair(cdr(args)))
    {
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == 0) return(small_int(0));                /* (atan 0) -> 0 */

	case T_RATIO:
	case T_REAL:
	  return(make_real(sc, atan(real_to_double(sc, x, "atan"))));

	case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
	  return(s7_from_c_complex(sc, catan(as_c_complex(x))));
#else
	  return(out_of_range(sc, sc->atan_symbol, small_int(1), x, no_complex_numbers_string));
#endif

	default:
	  method_or_bust_with_type(sc, x, sc->atan_symbol, args, a_number_string, 0);
	}
    }

  if (!s7_is_real(x))
    method_or_bust(sc, x, sc->atan_symbol, args, T_REAL, 1);

  y = cadr(args);
  if (!s7_is_real(y))
    method_or_bust(sc, y, sc->atan_symbol, args, T_REAL, 2);
  
  x1 = real_to_double(sc, x, "atan");
  x2 = real_to_double(sc, y, "atan");
  return(make_real(sc, atan2(x1, x2)));
}

RF2_TO_RF(atan, c_atan)


/* -------------------------------- sinh -------------------------------- */
static s7_pointer g_sinh(s7_scheme *sc, s7_pointer args)
{
  #define H_sinh "(sinh z) returns sinh(z)"
  #define Q_sinh pcl_n  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                    /* (sinh 0) -> 0 */

    case T_REAL:
    case T_RATIO:
      return(make_real(sc, sinh(real_to_double(sc, x, "sinh"))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, csinh(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->sinh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->sinh_symbol, args, a_number_string, 0);
    }
}

DIRECT_RF_TO_RF(sinh)


/* -------------------------------- cosh -------------------------------- */
static s7_pointer g_cosh(s7_scheme *sc, s7_pointer args)
{
  #define H_cosh "(cosh z) returns cosh(z)"
  #define Q_cosh pcl_n  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(1));                   /* (cosh 0) -> 1 */

    case T_REAL:
    case T_RATIO:
      /* this is not completely correct when optimization kicks in.
       * :(define (hi) (do ((i 0 (+ i 1))) ((= i 1)) (display (cosh i))))
       * hi
       * :(hi)
       * 1.0()
       * :(cosh 0)
       * 1
       */
      return(make_real(sc, cosh(real_to_double(sc, x, "cosh"))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, ccosh(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->cosh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->cosh_symbol, args, a_number_string, 0);
    }
}

DIRECT_RF_TO_RF(cosh)


/* -------------------------------- tanh -------------------------------- */
static s7_pointer g_tanh(s7_scheme *sc, s7_pointer args)
{
  #define H_tanh "(tanh z) returns tanh(z)"
  #define Q_tanh pcl_n  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));  /* (tanh 0) -> 0 */

    case T_REAL:
    case T_RATIO:
      return(make_real(sc, tanh(real_to_double(sc, x, "tanh"))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      if (real_part(x) > 350.0)
	return(real_one);                         /* closer than 0.0 which is what ctanh is about to return! */
      if (real_part(x) < -350.0)
	return(make_real(sc, -1.0));              /* closer than ctanh's -0.0 */
      return(s7_from_c_complex(sc, ctanh(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->tanh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->tanh_symbol, args, a_number_string, 0);
    }
}

DIRECT_RF_TO_RF(tanh)


/* -------------------------------- asinh -------------------------------- */
static s7_pointer c_asinh_1(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));
      return(make_real(sc, asinh((s7_double)integer(x))));

    case T_RATIO:
      return(make_real(sc, asinh((s7_double)numerator(x) / (s7_double)denominator(x))));

    case T_REAL:
      return(make_real(sc, asinh(real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(s7_from_c_complex(sc, casinh_1(as_c_complex(x))));
  #else
      return(s7_from_c_complex(sc, casinh(as_c_complex(x))));
  #endif
#else
      return(out_of_range(sc, sc->asinh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->asinh_symbol, list_1(sc, x), a_number_string, 0);
    }
}

static s7_pointer g_asinh(s7_scheme *sc, s7_pointer args)
{
  #define H_asinh "(asinh z) returns asinh(z)"
  #define Q_asinh pcl_n  

  return(c_asinh_1(sc, car(args)));
}

static s7_pointer c_asinh(s7_scheme *sc, s7_double x)
{
  return(make_real(sc, asinh(x)));
}

R_P_F_TO_PF(asinh, c_asinh, c_asinh_1, c_asinh_1)


/* -------------------------------- acosh -------------------------------- */
static s7_pointer c_acosh_1(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 1) return(small_int(0));

    case T_REAL:
    case T_RATIO:
      {
	double x1;
	x1 = real_to_double(sc, x, "acosh");
	if (x1 >= 1.0)
	  return(make_real(sc, acosh(x1)));
      }

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #ifdef __OpenBSD__
      return(s7_from_c_complex(sc, cacosh_1(s7_to_c_complex(x))));
  #else
      return(s7_from_c_complex(sc, cacosh(s7_to_c_complex(x)))); /* not as_c_complex because x might not be complex */
  #endif
#else
      /* since we can fall through to this branch, we need a better error message than "must be a number, not 0.0" */
      return(out_of_range(sc, sc->acosh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->acosh_symbol, list_1(sc, x), a_number_string, 0);
    }
}

static s7_pointer g_acosh(s7_scheme *sc, s7_pointer args)
{
  #define H_acosh "(acosh z) returns acosh(z)"
  #define Q_acosh pcl_n  
  return(c_acosh_1(sc, car(args)));
}

static s7_pointer c_acosh(s7_scheme *sc, s7_double x)
{
  if (x >= 1.0)
    return(make_real(sc, acosh(x)));
  return(c_acosh_1(sc, set_plist_1(sc, make_real(sc, x))));
}

R_P_F_TO_PF(acosh, c_acosh, c_acosh_1, c_acosh_1)


/* -------------------------------- atanh -------------------------------- */
static s7_pointer c_atanh_1(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                    /* (atanh 0) -> 0 */

    case T_REAL:
    case T_RATIO:
      {
	double x1;
	x1 = real_to_double(sc, x, "atanh");
	if (fabs(x1) < 1.0)
	  return(make_real(sc, atanh(x1)));
      }

      /* if we can't distinguish x from 1.0 even with long doubles, we'll get inf.0:
       *    (atanh 9223372036854775/9223372036854776) -> 18.714973875119
       *    (atanh 92233720368547758/92233720368547757) -> inf.0
       */
    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(s7_from_c_complex(sc, catanh_1(s7_to_c_complex(x))));
  #else
      return(s7_from_c_complex(sc, catanh(s7_to_c_complex(x))));
  #endif
#else
      return(out_of_range(sc, sc->atanh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->atanh_symbol, list_1(sc, x), a_number_string, 0);
    }
}

static s7_pointer g_atanh(s7_scheme *sc, s7_pointer args)
{
  #define H_atanh "(atanh z) returns atanh(z)"
  #define Q_atanh pcl_n  
  return(c_atanh_1(sc, car(args)));
}

static s7_pointer c_atanh(s7_scheme *sc, s7_double x)
{
  if (fabs(x) < 1.0)
    return(make_real(sc, atanh(x)));
  return(c_atanh_1(sc, set_plist_1(sc, make_real(sc, x))));
}

R_P_F_TO_PF(atanh, c_atanh, c_atanh_1, c_atanh_1)


/* -------------------------------- sqrt -------------------------------- */
static s7_pointer g_sqrt(s7_scheme *sc, s7_pointer args)
{
  #define H_sqrt "(sqrt z) returns the square root of z"
  #define Q_sqrt pcl_n  

  s7_pointer n;
  s7_double sqx;

  n = car(args);
  switch (type(n))
    {
    case T_INTEGER:
      if (integer(n) >= 0)
	{
	  s7_int ix;
	  sqx = sqrt((s7_double)integer(n));
	  ix = (s7_int)sqx;
	  if ((ix * ix) == integer(n))
	    return(make_integer(sc, ix));
	  return(make_real(sc, sqx));
	  /* Mark Weaver notes that
	   *     (zero? (- (sqrt 9007199136250226) 94906265.0)) -> #t
	   * but (* 94906265 94906265) -> 9007199136250225 -- oops
	   * at least we return a real here, not an incorrect integer and
	   *     (sqrt 9007199136250225) -> 94906265
	   */
	}
      sqx = (s7_double)integer(n); /* we're trying to protect against (sqrt -9223372036854775808) where we can't negate the integer argument */
      return(s7_make_complex(sc, 0.0, sqrt((s7_double)(-sqx))));

    case T_RATIO:
      sqx = (s7_double)fraction(n);
      if (sqx > 0.0) /* else it's complex, so it can't be a ratio */
	{
	  s7_int nm = 0, dn = 1;
	  if (c_rationalize(sqx, 1.0e-16, &nm, &dn)) /* 1e-16 so that (sqrt 1/1099511627776) returns 1/1048576 */
	    {
#if HAVE_OVERFLOW_CHECKS
	      s7_int nm2, dn2;
	      if ((multiply_overflow(nm, nm, &nm2)) ||
		  (multiply_overflow(dn, dn, &dn2)))
		return(make_real(sc, sqrt(sqx)));
	      if ((nm2 == numerator(n)) &&
		  (dn2 == denominator(n)))
		return(s7_make_ratio(sc, nm, dn));
#else
	      if ((nm * nm == numerator(n)) &&
		  (dn * dn == denominator(n)))
		return(s7_make_ratio(sc, nm, dn));
#endif
	    }
	  return(make_real(sc, sqrt(sqx)));
	}
      return(s7_make_complex(sc, 0.0, sqrt(-sqx)));

    case T_REAL:
      if (is_NaN(real(n)))
	return(real_NaN);
      if (real(n) >= 0.0)
	return(make_real(sc, sqrt(real(n))));
      return(s7_make_complex(sc, 0.0, sqrt(-real(n))));

    case T_COMPLEX:
      /* (* inf.0 (sqrt -1)) -> -nan+infi, but (sqrt -inf.0) -> 0+infi */
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, csqrt(as_c_complex(n))));
#else
      return(out_of_range(sc, sc->sqrt_symbol, small_int(1), n, no_complex_numbers_string));
#endif

    default:
      method_or_bust_with_type(sc, n, sc->sqrt_symbol, args, a_number_string, 0);
    }
}


/* -------------------------------- expt -------------------------------- */

static s7_int int_to_int(s7_int x, s7_int n)
{
  /* from GSL */
  s7_int value = 1;
  do {
    if (n & 1) value *= x;
    n >>= 1;
#if HAVE_OVERFLOW_CHECKS
    if (multiply_overflow(x, x, &x))
      break;
#else
    x *= x;
#endif
  } while (n);
  return(value);
}


static const long long int nth_roots[63] = {
  S7_LLONG_MAX, S7_LLONG_MAX, 3037000499LL, 2097151, 55108, 6208, 1448, 511, 234, 127, 78, 52, 38, 28, 22,
  18, 15, 13, 11, 9, 8, 7, 7, 6, 6, 5, 5, 5, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2};

static const long int_nth_roots[31] = {
  S7_LONG_MAX, S7_LONG_MAX, 46340, 1290, 215, 73, 35, 21, 14, 10, 8, 7, 5, 5, 4, 4, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2};

static bool int_pow_ok(s7_int x, s7_int y)
{
  if (s7_int_bits > 31)
    return((y < 63) &&
	   (nth_roots[y] >= s7_int_abs(x)));
  return((y < 31) &&
	 (int_nth_roots[y] >= s7_int_abs(x)));
}


static s7_pointer g_expt(s7_scheme *sc, s7_pointer args)
{
  #define H_expt "(expt z1 z2) returns z1^z2"
  #define Q_expt pcl_n
  s7_pointer n, pw;

  n = car(args);
  if (!s7_is_number(n))
    method_or_bust_with_type(sc, n, sc->expt_symbol, args, a_number_string, 1);

  pw = cadr(args);
  if (!s7_is_number(pw))
    method_or_bust_with_type(sc, pw, sc->expt_symbol, args, a_number_string, 2);

  /* this provides more than 2 args to expt:
   *  if (is_not_null(cddr(args)))
   *    return(g_expt(sc, list_2(sc, car(args), g_expt(sc, cdr(args)))));
   *
   * but it's unusual in scheme to process args in reverse order, and the
   * syntax by itself is ambiguous (does (expt 2 2 3) = 256 or 64?)
   */

  if (s7_is_zero(n))
    {
      if (s7_is_zero(pw))
	{
	  if ((s7_is_integer(n)) && (s7_is_integer(pw)))       /* (expt 0 0) -> 1 */
	    return(small_int(1));
	  return(real_zero);                                   /* (expt 0.0 0) -> 0.0 */
	}

      if (s7_is_real(pw))
	{
	  if (s7_is_negative(pw))                              /* (expt 0 -1) */
	    return(division_by_zero_error(sc, sc->expt_symbol, args));
	  /* (Clisp gives divide-by-zero error here, Guile returns inf.0) */

	  if ((!s7_is_rational(pw)) &&                         /* (expt 0 most-positive-fixnum) */
	      (is_NaN(s7_real(pw))))                           /* (expt 0 +nan.0) */
	    return(pw);
	}
      else
	{                                                      /* (expt 0 a+bi) */
	  if (real_part(pw) < 0.0)                             /* (expt 0 -1+i) */
	    return(division_by_zero_error(sc, sc->expt_symbol, args));
	  if ((is_NaN(real_part(pw))) ||                       /* (expt 0 0+1/0i) */
	      (is_NaN(imag_part(pw))))
	    return(real_NaN);
	}

      if ((s7_is_integer(n)) && (s7_is_integer(pw)))           /* pw != 0, (expt 0 2312) */
	return(small_int(0));
      return(real_zero);                                       /* (expt 0.0 123123) */
    }

  if (s7_is_one(pw))
    {
      if (s7_is_integer(pw))
	return(n);
      if (is_rational(n))
	return(make_real(sc, rational_to_double(sc, n)));
      return(n);
    }

  if (is_t_integer(pw))
    {
      s7_int y;
      y = integer(pw);
      if (y == 0)
	{
	  if (is_rational(n))                                 /* (expt 3 0) */
	    return(small_int(1));
	  if ((is_NaN(s7_real_part(n))) ||                    /* (expt 1/0 0) -> NaN */
	      (is_NaN(s7_imag_part(n))))                      /* (expt (complex 0 1/0) 0) -> NaN */
	    return(n);
	  return(real_one);                                   /* (expt 3.0 0) */
	}

      switch (type(n))
	{
	case T_INTEGER:
	  {
	    s7_int x;
	    x = s7_integer(n);
	    if (x == 1)                                       /* (expt 1 y) */
	      return(n);

	    if (x == -1)
	      {
		if (y == s7_int_min)                        /* (expt -1 most-negative-fixnum) */
		  return(small_int(1));

		if (s7_int_abs(y) & 1)                        /* (expt -1 odd-int) */
		  return(n);
		return(small_int(1));                         /* (expt -1 even-int) */
	      }

	    if (y == s7_int_min)                            /* (expt x most-negative-fixnum) */
	      return(small_int(0));
	    if (x == s7_int_min)                            /* (expt most-negative-fixnum y) */
	      return(make_real(sc, pow((double)x, (double)y)));

	    if (int_pow_ok(x, s7_int_abs(y)))
	      {
		if (y > 0)
		  return(make_integer(sc, int_to_int(x, y)));
		return(s7_make_ratio(sc, 1, int_to_int(x, -y)));
	      }
	  }
	  break;

	case T_RATIO:
	  {
	    s7_int nm, dn;

	    nm = numerator(n);
	    dn = denominator(n);

	    if (y == s7_int_min)
	      {
		if (s7_int_abs(nm) > dn)
		  return(small_int(0));              /* (expt 4/3 most-negative-fixnum) -> 0? */
		return(real_infinity);               /* (expt 3/4 most-negative-fixnum) -> inf? */
	      }

	    if ((int_pow_ok(nm, s7_int_abs(y))) &&
		(int_pow_ok(dn, s7_int_abs(y))))
	      {
		if (y > 0)
		  return(s7_make_ratio(sc, int_to_int(nm, y), int_to_int(dn, y)));
		return(s7_make_ratio(sc, int_to_int(dn, -y), int_to_int(nm, -y)));
	      }
	  }
	  break;
	  /* occasionally int^rat can be int but it happens so infrequently it's probably not worth checking
	   *  one possibly easy case: (expt 1 1/2) -> 1 (-1?) etc
	   */

	case T_REAL:
	  /* (expt -1.0 most-positive-fixnum) should be -1.0
	   * (expt -1.0 (+ (expt 2 53) 1)) -> -1.0
	   * (expt -1.0 (- 1 (expt 2 54))) -> -1.0
	   */
	  if (real(n) == -1.0)
	    {
	      if (y == s7_int_min)
		return(real_one);

	      if (s7_int_abs(y) & 1)
		return(n);
	      return(real_one);
	    }
	  break;

	case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
	  if ((s7_real_part(n) == 0.0) &&
	      ((s7_imag_part(n) == 1.0) ||
	       (s7_imag_part(n) == -1.0)))
	    {
	      bool yp, np;
	      yp = (y > 0);
	      np = (s7_imag_part(n) > 0.0);
	      switch (s7_int_abs(y) % 4)
		{
		case 0: return(real_one);
		case 1: return(s7_make_complex(sc, 0.0, (yp == np) ? 1.0 : -1.0));
		case 2: return(make_real(sc, -1.0));
		case 3: return(s7_make_complex(sc, 0.0, (yp == np) ? -1.0 : 1.0));
		}
	    }
#else
	  return(out_of_range(sc, sc->expt_symbol, small_int(2), n, no_complex_numbers_string));
#endif
	  break;
	}
    }

  if ((s7_is_real(n)) &&
      (s7_is_real(pw)))
    {
      s7_double x, y;

      if ((is_t_ratio(pw)) &&
	  (numerator(pw) == 1))
	{
	  if (denominator(pw) == 2)
	    return(g_sqrt(sc, args));
	  if (denominator(pw) == 3)
	    return(make_real(sc, cbrt(real_to_double(sc, n, "expt")))); /* (expt 27 1/3) should be 3, not 3.0... */

	  /* but: (expt 512/729 1/3) -> 0.88888888888889
	   */
	  /* and 4 -> sqrt(sqrt...) etc? */
	}

      x = real_to_double(sc, n, "expt");
      y = real_to_double(sc, pw, "expt");

      if (is_NaN(x)) return(n);
      if (is_NaN(y)) return(pw);
      if (y == 0.0) return(real_one);

      if (x > 0.0)
	return(make_real(sc, pow(x, y)));
      /* tricky cases abound here: (expt -1 1/9223372036854775807)
       */
    }

  /* (expt 0+i 1e+16) = 0.98156860153485-0.19111012657867i ?
   * (expt 0+i 1+1/0i) = 0.0 ??
   */
  return(s7_from_c_complex(sc, cpow(s7_to_c_complex(n), s7_to_c_complex(pw))));
}


#if (!WITH_GMP)
static s7_pointer c_expt_i(s7_scheme *sc, s7_int x, s7_int y)
{
  if (y == 0) return(small_int(1));
  if (y == 1) return(make_integer(sc, x));
  return(g_expt(sc, set_plist_2(sc, make_integer(sc, x), make_integer(sc, y))));
}

static s7_pointer c_expt_r(s7_scheme *sc, s7_double x, s7_double y)
{
  if (y > 0.0)
    return(make_real(sc, pow(x, y)));
  return(g_expt(sc, set_plist_2(sc, make_real(sc, x), make_real(sc, y))));
}

static s7_pointer c_expt_2(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  return(g_expt(sc, set_plist_2(sc, x, y)));
}

XF2_TO_PF(expt, c_expt_i, c_expt_r, c_expt_2)
#endif


/* -------------------------------- lcm -------------------------------- */
static s7_pointer g_lcm(s7_scheme *sc, s7_pointer args)
{
  #define H_lcm "(lcm ...) returns the least common multiple of its rational arguments"
  #define Q_lcm pcl_f

  s7_int n = 1, d = 0;
  s7_pointer p;

  if (!is_pair(args))
    return(small_int(1));

  if (!is_pair(cdr(args)))
    {
      if (!is_rational(car(args)))
	method_or_bust_with_type(sc, car(args), sc->lcm_symbol, args, a_rational_string, 1);
      return(g_abs(sc, args));
    }

  for (p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer x;
      s7_int b;
      x = car(p);
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == 0)
	    n = 0;
	  else 
	    {
	      b = integer(x);
	      if (b < 0) b = -b;
	      n = (n / c_gcd(n, b)) * b;
	    }
	  if (d != 0) d = 1;
	  break;

	case T_RATIO:
	  b = numerator(x);
	  if (b < 0) b = -b;
	  n = (n / c_gcd(n, b)) * b;
	  if (d == 0)
	    {
	      if (p == args)
		d = s7_denominator(x);
	      else d = 1;
	    }
	  else d = c_gcd(d, s7_denominator(x));
	  break;

	default:
	  method_or_bust_with_type(sc, x, sc->lcm_symbol, cons(sc, (d <= 1) ? make_integer(sc, n) : s7_make_ratio(sc, n, d), p), a_rational_string, position_of(p, args));
	}
      if (n < 0) return(simple_out_of_range(sc, sc->lcm_symbol, args, result_is_too_large_string));
      if (n == 0)
	{
	  for (p = cdr(p); is_pair(p); p = cdr(p))
	    if (!is_rational_via_method(sc, car(p)))
	      return(wrong_type_argument_with_type(sc, sc->lcm_symbol, position_of(p, args), x, a_rational_string));
	  return(small_int(0));
	}
    }

  if (d <= 1)
    return(make_integer(sc, n));
  return(s7_make_ratio(sc, n, d));
}

static s7_int c_lcm(s7_scheme *sc, s7_int a, s7_int b)
{
  if ((a == 0) || (b == 0)) return(0);
  if (a < 0) a = -a;
  if (b < 0) b = -b;
  return((a / c_gcd(a, b)) * b);
}

IF2_TO_IF(lcm, c_lcm)


/* -------------------------------- gcd -------------------------------- */
static s7_pointer g_gcd(s7_scheme *sc, s7_pointer args)
{
  #define H_gcd "(gcd ...) returns the greatest common divisor of its rational arguments"
  #define Q_gcd pcl_f
  s7_int n = 0, d = 1;
  s7_pointer p;

  if (!is_pair(args))
    return(small_int(0));

  if (!is_pair(cdr(args)))
    {
      if (!is_rational(car(args)))
	method_or_bust_with_type(sc, car(args), sc->gcd_symbol, args, a_rational_string, 1);
      return(g_abs(sc, args));
    }

  for (p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer x;
      s7_int b;
      x = car(p);
      switch (type(x))
	{
	case T_INTEGER:
	  n = c_gcd(n, integer(x));
	  break;

	case T_RATIO:
	  n = c_gcd(n, s7_numerator(x));
	  b = s7_denominator(x);
	  if (b < 0) b = -b;
	  d = (d / c_gcd(d, b)) * b;
	  if (d < 0) return(simple_out_of_range(sc, sc->gcd_symbol, args, result_is_too_large_string));
	  break;

	default:
	  method_or_bust_with_type(sc, x, sc->gcd_symbol, cons(sc, (d <= 1) ? make_integer(sc, n) : s7_make_ratio(sc, n, d), p), a_rational_string, position_of(p, args));
	}
      if (n < 0) return(simple_out_of_range(sc, sc->gcd_symbol, args, result_is_too_large_string));
    }

  if (d <= 1)
    return(make_integer(sc, n));
  return(s7_make_ratio(sc, n, d));
}

static s7_int c_gcd_1(s7_scheme *sc, s7_int a, s7_int b) {return(c_gcd(a, b));}

IF2_TO_IF(gcd, c_gcd_1)


static s7_pointer s7_truncate(s7_scheme *sc, s7_pointer caller, s7_double xf)   /* can't use "truncate" -- it's in unistd.h */
{
  if ((xf > s7_int_max) ||
      (xf < s7_int_min))
    return(simple_out_of_range(sc, caller, make_real(sc, xf), its_too_large_string));

  if (xf > 0.0)
    return(make_integer(sc, (s7_int)floor(xf)));
  return(make_integer(sc, (s7_int)ceil(xf)));
}

static s7_int c_quo_int(s7_scheme *sc, s7_int x, s7_int y)
{
  if (y == 0)
    division_by_zero_error(sc, sc->quotient_symbol, set_elist_2(sc, make_integer(sc, x), make_integer(sc, y)));
  if ((y == -1) && (x == s7_int_min))   /* (quotient most-negative-fixnum -1) */
    simple_out_of_range(sc, sc->quotient_symbol, set_elist_2(sc, make_integer(sc, x), make_integer(sc, y)), its_too_large_string);
  return(x / y);
}

static s7_double c_quo_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_double xf;

  if (y == 0.0)
    division_by_zero_error(sc, sc->quotient_symbol, set_elist_2(sc, make_real(sc, x), make_real(sc, y)));
  if ((is_inf(y)) || (is_NaN(y)))
    wrong_type_argument_with_type(sc, sc->quotient_symbol, 2, make_real(sc, y), a_normal_real_string);

  xf = x / y;
  if ((xf > s7_int_max) ||
      (xf < s7_int_min))
    simple_out_of_range(sc, sc->quotient_symbol, make_real(sc, xf), its_too_large_string);

  if (xf > 0.0)
    return(floor(xf));
  return(ceil(xf));
}

static s7_pointer g_quotient(s7_scheme *sc, s7_pointer args)
{
  #define H_quotient "(quotient x1 x2) returns the integer quotient of x1 and x2; (quotient 4 3) = 1"
  #define Q_quotient pcl_r
  /* (define (quo x1 x2) (truncate (/ x1 x2))) ; slib
   */
  s7_pointer x, y;
  s7_int d1, d2, n1, n2;

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, c_quo_int(sc, integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  goto RATIO_QUO_RATIO;

	case T_REAL:
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(wrong_type_argument_with_type(sc, sc->quotient_symbol, 2, y, a_normal_real_string));
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)integer(x) / real(y)));

	default:
	  method_or_bust(sc, y, sc->quotient_symbol, args, T_REAL, 2);
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);
	  d2 = 1;
	  goto RATIO_QUO_RATIO;
	  /* this can lose:
	   *   (quotient 1 2305843009213693952/4611686018427387903) -> 2, not 1
	   *   (quotient 21053343141/6701487259 3587785776203/1142027682075) -> 1, not 0
	   */

	case T_RATIO:
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = numerator(y);
	  d2 = denominator(y);
	RATIO_QUO_RATIO:
	  if (d1 == d2)
	    return(make_integer(sc, n1 / n2));              /* (quotient 3/9223372036854775807 1/9223372036854775807) */
	  if (n1 == n2)
	    return(make_integer(sc, d2 / d1));              /* (quotient 9223372036854775807/2 9223372036854775807/8) */
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int n1d2, n2d1;
	    if ((multiply_overflow(n1, d2, &n1d2)) ||
		(multiply_overflow(n2, d1, &n2d1)))
	      return(s7_truncate(sc, sc->quotient_symbol, ((long double)n1 / (long double)n2) * ((long double)d2 / (long double)d1)));
	    return(make_integer(sc, n1d2 / n2d1));
	  }
#else
	  if ((integer_length(n1) + integer_length(d2) >= s7_int_bits) ||
	      (integer_length(n2) + integer_length(d1) >= s7_int_bits))
	    return(s7_truncate(sc, sc->quotient_symbol, ((long double)n1 / (long double)n2) * ((long double)d2 / (long double)d1)));
	  return(make_integer(sc, (n1 * d2) / (n2 * d1)));
#endif

	case T_REAL:
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(wrong_type_argument_with_type(sc, sc->quotient_symbol, 2, y, a_normal_real_string));
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)fraction(x) / real(y)));

	default:
	  method_or_bust(sc, y, sc->quotient_symbol, args, T_REAL, 2);
	}

    case T_REAL:
      if ((is_inf(real(x))) || (is_NaN(real(x))))
	return(wrong_type_argument_with_type(sc, sc->quotient_symbol, 1, x, a_normal_real_string));

      /* if infs allowed we need to return infs/nans, else:
       *    (quotient inf.0 1e-309) -> -9223372036854775808
       *    (quotient inf.0 inf.0) -> -9223372036854775808
       */

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  return(s7_truncate(sc, sc->quotient_symbol, real(x) / (s7_double)integer(y)));

	case T_RATIO:
	  return(s7_truncate(sc, sc->quotient_symbol, real(x) / (s7_double)fraction(y)));

	case T_REAL:
	  return(make_real(sc, c_quo_dbl(sc, real(x), real(y))));

	default:
	  method_or_bust(sc, y, sc->quotient_symbol, args, T_REAL, 2);
	}

    default:
      method_or_bust(sc, x, sc->quotient_symbol, args, T_REAL, 2);
    }
}


IF2_TO_IF(quotient, c_quo_int)
RF2_TO_RF(quotient, c_quo_dbl)


static s7_int c_rem_int(s7_scheme *sc, s7_int x, s7_int y)
{
  if (y == 0)
    division_by_zero_error(sc, sc->remainder_symbol, set_elist_2(sc, make_integer(sc, x), make_integer(sc, y)));
  if ((y == 1) || (y == -1))   /* (remainder most-negative-fixnum -1) will segfault with arithmetic exception */
    return(0);
  return(x % y);
}

static s7_double c_rem_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_int quo;
  s7_double pre_quo;
  if (y == 0.0)
    division_by_zero_error(sc, sc->remainder_symbol, set_elist_2(sc, make_real(sc, x), make_real(sc, y)));
  if ((is_inf(y)) || (is_NaN(y)))
    wrong_type_argument_with_type(sc, sc->remainder_symbol, 2, set_elist_1(sc, make_real(sc, y)), a_normal_real_string);

  pre_quo = x / y;
  if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
    simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, make_real(sc, x), make_real(sc, y)), its_too_large_string);
  if (pre_quo > 0.0) 
    quo = (s7_int)floor(pre_quo); 
  else quo = (s7_int)ceil(pre_quo);
  return(x - (y * quo));
}

static s7_pointer g_remainder(s7_scheme *sc, s7_pointer args)
{
  #define H_remainder "(remainder x1 x2) returns the remainder of x1/x2; (remainder 10 3) = 1"
  #define Q_remainder pcl_r
  /* (define (rem x1 x2) (- x1 (* x2 (quo x1 x2)))) ; slib, if x2 is an integer (- x1 (truncate x1 x2)), fractional part: (remainder x 1) */

  s7_pointer x, y;
  s7_int quo, d1, d2, n1, n2;
  s7_double pre_quo;

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, c_rem_int(sc, integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  goto RATIO_REM_RATIO;

	case T_REAL:
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->remainder_symbol, args));
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(wrong_type_argument_with_type(sc, sc->remainder_symbol, 2, y, a_normal_real_string));

	  pre_quo = (s7_double)integer(x) / real(y);
	  if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	    return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	  if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	  return(make_real(sc, integer(x) - real(y) * quo));

	default:
	  method_or_bust(sc, y, sc->remainder_symbol, args, T_REAL, 2);
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  n2 = integer(y);
	  if (n2 == 0)
	    return(division_by_zero_error(sc, sc->remainder_symbol, args));
	  n1 = numerator(x);
	  d1 = denominator(x);
	  d2 = 1;
	  goto RATIO_REM_RATIO;

	case T_RATIO:
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = numerator(y);
	  d2 = denominator(y);
	RATIO_REM_RATIO:
	  if (d1 == d2)
	    quo = (s7_int)(n1 / n2);
	  else
	    {
	      if (n1 == n2)
		quo = (s7_int)(d2 / d1);
	      else
		{
#if HAVE_OVERFLOW_CHECKS
		  s7_int n1d2, n2d1;
		  if ((multiply_overflow(n1, d2, &n1d2)) ||
		      (multiply_overflow(n2, d1, &n2d1)))
		    {
		      pre_quo = ((long double)n1 / (long double)n2) * ((long double)d2 / (long double)d1);
		      if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
			return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
		      if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
		    }
		  else quo = n1d2 / n2d1;
#else
		  if ((integer_length(n1) + integer_length(d2) >= s7_int_bits) ||
		      (integer_length(n2) + integer_length(d1) >= s7_int_bits))
		    {
		      pre_quo = ((long double)n1 / (long double)n2) * ((long double)d2 / (long double)d1);
		      if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
			return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
		      if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
		    }
		  else quo = (n1 * d2) / (n2 * d1);
#endif
		}
	    }
	  if (quo == 0)
	    return(x);

#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn, nq;
	    if (!multiply_overflow(n2, quo, &nq))
	      {
		if ((d1 == d2) &&
		    (!subtract_overflow(n1, nq, &dn)))
		  return(s7_make_ratio(sc, dn, d1));

		if ((!multiply_overflow(n1, d2, &dn)) &&
		    (!multiply_overflow(nq, d1, &nq)) &&
		    (!subtract_overflow(dn, nq, &nq)) &&
		    (!multiply_overflow(d1, d2, &d1)))
		  return(s7_make_ratio(sc, nq, d1));
	      }
	  }
#else
	  if ((d1 == d2) &&
	      ((integer_length(n2) + integer_length(quo)) < s7_int_bits))
	    return(s7_make_ratio(sc, n1 - n2 * quo, d1));

	  if ((integer_length(n1) + integer_length(d2) < s7_int_bits) &&
	      (integer_length(d1) + integer_length(d2) < s7_int_bits) &&
	      (integer_length(n2) + integer_length(d1) + integer_length(quo) < s7_int_bits))
	    return(s7_make_ratio(sc, n1 * d2 - n2 * d1 * quo, d1 * d2));
#endif
	  return(simple_out_of_range(sc, sc->remainder_symbol, args, make_string_wrapper(sc, "intermediate (a/b) is too large")));

	case T_REAL:
	  {
	    s7_double frac;
	    if (real(y) == 0.0)
	      return(division_by_zero_error(sc, sc->remainder_symbol, args));
	    if ((is_inf(real(y))) || (is_NaN(real(y))))
	      return(wrong_type_argument_with_type(sc, sc->remainder_symbol, 2, y, a_normal_real_string));
	    frac = (s7_double)fraction(x);
	    pre_quo = frac / real(y);
	    if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	      return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	    if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	    return(make_real(sc, frac - real(y) * quo));
	  }

	default:
	  method_or_bust(sc, y, sc->remainder_symbol, args, T_REAL, 2);
	}

    case T_REAL:
      if ((is_inf(real(x))) || (is_NaN(real(x))))
	return(wrong_type_argument_with_type(sc, sc->remainder_symbol, 1, x, a_normal_real_string));

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->remainder_symbol, args));
	  pre_quo = real(x) / (s7_double)integer(y);
	  if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	    return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	  if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	  return(make_real(sc, real(x) - integer(y) * quo));
	  /* but... (remainder 1e+18 9223372036854775807) -> 1e+18 */

	case T_RATIO:
	  {
	    /* bad cases here start around 1e16: (remainder 1e15 3/13) -> 0.0 with loss of digits earlier
	     *   would long double help?
	     */
	    s7_double frac;
	    frac = (s7_double)fraction(y);
	    pre_quo = real(x) / frac;
	    if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	      return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	    if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	    return(make_real(sc, real(x) - frac * quo));
	  }

	case T_REAL:
	  return(make_real(sc, c_rem_dbl(sc, real(x), real(y))));

	  /* see under sin -- this calculation is completely bogus if "a" is large
	   * (quotient 1e22 (* 2 pi)) -> -9223372036854775808 -- should this return arithmetic-overflow?
	   *          but it should be 1591549430918953357688,
	   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22
	   * -- the "remainder" is greater than the original argument!
	   * Clisp gives 0.0 here, as does sbcl
	   * currently s7 throws an error (out-of-range).
	   */

	default:
	  method_or_bust(sc, y, sc->remainder_symbol, args, T_REAL, 2);
	}

    default:
      method_or_bust(sc, x, sc->remainder_symbol, args, T_REAL, 1);
    }
}

IF2_TO_IF(remainder, c_rem_int)
RF2_TO_RF(remainder, c_rem_dbl)


/* -------------------------------- floor -------------------------------- */

#define REAL_TO_INT_LIMIT 9.2233727815085e+18
/* unfortunately, this limit is only a max in a sense: (ceiling 9223372036854770.9) => 9223372036854770
 *    see s7test for more examples
 */

static s7_pointer g_floor(s7_scheme *sc, s7_pointer args)
{
  #define H_floor "(floor x) returns the integer closest to x toward -inf"
  #define Q_floor s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      {
	s7_int val;
	val = numerator(x) / denominator(x);
	/* C "/" truncates? -- C spec says "truncation toward 0" */
	/* we're avoiding "floor" here because the int->double conversion introduces inaccuracies for big numbers */
	if (numerator(x) < 0) /* not "val" because it might be truncated to 0 */
	  return(make_integer(sc, val - 1));
	return(make_integer(sc, val));
      }

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->floor_symbol, x, its_nan_string));
	if (fabs(z) > REAL_TO_INT_LIMIT)
	  return(simple_out_of_range(sc, sc->floor_symbol, x, its_too_large_string));
	return(make_integer(sc, (s7_int)floor(z)));
	/* floor here rounds down, whereas a straight int<=real coercion apparently rounds towards 0 */
      }

    case T_COMPLEX:
    default:
      method_or_bust(sc, x, sc->floor_symbol, args, T_REAL, 0);
    }
}

static s7_int c_floor(s7_scheme *sc, s7_double x) {return((s7_int)floor(x));}
RF_TO_IF(floor, c_floor)


/* -------------------------------- ceiling -------------------------------- */
static s7_pointer g_ceiling(s7_scheme *sc, s7_pointer args)
{
  #define H_ceiling "(ceiling x) returns the integer closest to x toward inf"
  #define Q_ceiling s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      {
	s7_int val;
	val = numerator(x) / denominator(x);
	if (numerator(x) < 0)
	  return(make_integer(sc, val));
	return(make_integer(sc, val + 1));
      }

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->ceiling_symbol, x, its_nan_string));
	if ((is_inf(z)) ||
	    (z > REAL_TO_INT_LIMIT) ||
	    (z < -REAL_TO_INT_LIMIT))
	  return(simple_out_of_range(sc, sc->ceiling_symbol, x, its_too_large_string));
	return(make_integer(sc, (s7_int)ceil(real(x))));
      }

    case T_COMPLEX:
    default:
      method_or_bust(sc, x, sc->ceiling_symbol, args, T_REAL, 0);
    }
}

static s7_int c_ceiling(s7_scheme *sc, s7_double x) {return((s7_int)ceil(x));}
RF_TO_IF(ceiling, c_ceiling)


/* -------------------------------- truncate -------------------------------- */
static s7_pointer g_truncate(s7_scheme *sc, s7_pointer args)
{
  #define H_truncate "(truncate x) returns the integer closest to x toward 0"
  #define Q_truncate s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      return(make_integer(sc, (s7_int)(numerator(x) / denominator(x)))); /* C "/" already truncates */

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->truncate_symbol, x, its_nan_string));
	if (is_inf(z))
	  return(simple_out_of_range(sc, sc->truncate_symbol, x, its_infinite_string));
	return(s7_truncate(sc, sc->truncate_symbol, real(x)));
      }

    case T_COMPLEX:
    default:
      method_or_bust(sc, x, sc->truncate_symbol, args, T_REAL, 0);
    }
}

static s7_int c_trunc(s7_scheme *sc, s7_double x)
{
  if ((x > s7_int_max) || (x < s7_int_min))
    simple_out_of_range(sc, sc->truncate_symbol, make_real(sc, x), its_too_large_string);
  if (x > 0.0)
    return((s7_int)floor(x));
  return((s7_int)ceil(x));
}

RF_TO_IF(truncate, c_trunc)


/* -------------------------------- round -------------------------------- */
static s7_double round_per_R5RS(s7_double x)
{
  s7_double fl, ce, dfl, dce;

  fl = floor(x);
  ce = ceil(x);
  dfl = x - fl;
  dce = ce - x;

  if (dfl > dce) return(ce);
  if (dfl < dce) return(fl);
  if (fmod(fl, 2.0) == 0.0) return(fl);
  return(ce);
}

static s7_pointer g_round(s7_scheme *sc, s7_pointer args)
{
  #define H_round "(round x) returns the integer closest to x"
  #define Q_round s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      {
	s7_int truncated, remains;
	long double frac;

	truncated = numerator(x) / denominator(x);
	remains = numerator(x) % denominator(x);
	frac = s7_fabsl((long double)remains / (long double)denominator(x));

	if ((frac > 0.5) ||
	    ((frac == 0.5) &&
	     (truncated % 2 != 0)))
	  {
	    if (numerator(x) < 0)
	      return(make_integer(sc, truncated - 1));
	    return(make_integer(sc, truncated + 1));
	  }
	return(make_integer(sc, truncated));
      }

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->round_symbol, x, its_nan_string));
	if ((is_inf(z)) ||
	    (z > REAL_TO_INT_LIMIT) ||
	    (z < -REAL_TO_INT_LIMIT))
	  return(simple_out_of_range(sc, sc->round_symbol, x, its_too_large_string));
	return(make_integer(sc, (s7_int)round_per_R5RS(z)));
      }

    case T_COMPLEX:
    default:
      method_or_bust(sc, x, sc->round_symbol, args, T_REAL, 0);
    }
}

static s7_int c_round(s7_scheme *sc, s7_double x) {return((s7_int)round_per_R5RS(x));}
RF_TO_IF(round, c_round)


static s7_int c_mod(s7_scheme *sc, s7_int x, s7_int y)
{
  s7_int z;
  /* if (y == 0) return(x); */ /* else arithmetic exception, but we're checking for this elsewhere */
  z = x % y;
  if (((y < 0) && (z > 0)) ||
      ((y > 0) && (z < 0)))
    return(z + y);
  return(z);
}

static s7_pointer g_modulo(s7_scheme *sc, s7_pointer args)
{
  #define H_modulo "(modulo x1 x2) returns x1 mod x2; (modulo 4 3) = 1.  The arguments can be real numbers."
  #define Q_modulo pcl_r
  /* (define (mod x1 x2) (- x1 (* x2 (floor (/ x1 x2))))) from slib
   * (mod x 0) = x according to "Concrete Mathematics"
   */
  s7_pointer x, y;
  s7_double a, b;
  s7_int n1, n2, d1, d2;

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(x);
	  if ((integer(y) == 1) || (integer(y) == -1))
	    return(small_int(0));
	  /* (modulo most-negative-fixnum -1) will segfault with arithmetic exception */
	  return(make_integer(sc, c_mod(sc, integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  goto RATIO_MOD_RATIO;

	case T_REAL:
	  b = real(y);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  if (is_inf(b)) return(real_NaN);
	  a = (s7_double)integer(x);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	default:
	  method_or_bust(sc, y, sc->modulo_symbol, args, T_REAL, 2);
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);

	  if ((n2 > 0) && (n1 > 0) && (n2 > n1)) return(x);
	  if ((n2 < 0) && (n1 < 0) && (n2 < n1)) return(x);

	  if (n2 == s7_int_min)
	    return(simple_out_of_range(sc, sc->modulo_symbol, y, make_string_wrapper(sc, "intermediate (a/b) is too large")));
	  /* the problem here is that (modulo 3/2 most-negative-fixnum)
	   * will segfault with signal SIGFPE, Arithmetic exception, so try to trap it.
	   */

	  d2 = 1;
	  goto RATIO_MOD_RATIO;

	case T_RATIO:
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = numerator(y); /* can't be 0 */
	  d2 = denominator(y);
	  if (d1 == d2)
	    return(s7_make_ratio(sc, c_mod(sc, n1, n2), d1));

	RATIO_MOD_RATIO:

	  if ((n1 == n2) &&
	      (d1 > d2))
	    return(x);                 /* signs match so this should be ok */
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int n2d1, n1d2, d1d2, fl;
	    if (!multiply_overflow(n2, d1, &n2d1))
	      {
		if (n2d1 == 1)
		  return(small_int(0));
		
		if (!multiply_overflow(n1, d2, &n1d2))
		  {
		    /* can't use "floor" here (int->float ruins everything) */
		    fl = (s7_int)(n1d2 / n2d1);
		    if (((n1 < 0) && (n2 > 0)) ||
			((n1 > 0) && (n2 < 0)))
		      fl -= 1;

		    if (fl == 0)
		      return(x);

		    if ((!multiply_overflow(d1, d2, &d1d2)) &&
			(!multiply_overflow(fl, n2d1, &fl)) &&
			(!subtract_overflow(n1d2, fl, &fl)))
		      return(s7_make_ratio(sc, fl, d1d2));
		  }
	      }
	  }
#else
	  if ((integer_length(n1) + integer_length(d2) < s7_int_bits) &&
	      (integer_length(n2) + integer_length(d1) < s7_int_bits) &&
	      (integer_length(d1) + integer_length(d2) < s7_int_bits))
	    {
	      s7_int n1d2, n2d1, fl;
	      n1d2 = n1 * d2;
	      n2d1 = n2 * d1;

	      if (n2d1 == 1)
		return(small_int(0));

	      /* can't use "floor" here (int->float ruins everything) */
	      fl = (s7_int)(n1d2 / n2d1);
	      if (((n1 < 0) && (n2 > 0)) ||
		  ((n1 > 0) && (n2 < 0)))
		fl -= 1;

	      if (fl == 0)
		return(x);

	      if (integer_length(n2d1) + integer_length(fl) < s7_int_bits)
		return(s7_make_ratio(sc, n1d2 - (n2d1 * fl), d1 * d2));
	    }
#endif

	  /* there are cases here we might want to catch:
	   *    (modulo 9223372036 1/9223372036) -> error, not 0?
	   *    (modulo 1 1/9223372036854775807) -> error, not 0?
	   */
	  return(simple_out_of_range(sc, sc->modulo_symbol, x, make_string_wrapper(sc, "intermediate (a/b) is too large")));

	case T_REAL:
	  b = real(y);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  if (is_inf(b)) return(real_NaN);
	  a = fraction(x);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	default:
	  method_or_bust(sc, y, sc->modulo_symbol, args, T_REAL, 2);
	}

    case T_REAL:
      a = real(x);

      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(a)) return(x);
	  if (is_inf(a)) return(real_NaN);
	  if (integer(y) == 0) return(x);
	  b = (s7_double)integer(y);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	case T_RATIO:
	  if (is_NaN(a)) return(x);
	  if (is_inf(a)) return(real_NaN);
	  b = fraction(y);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	case T_REAL:
	  if (is_NaN(a)) return(x);
	  if (is_inf(a)) return(real_NaN);
	  b = real(y);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  if (is_inf(b)) return(real_NaN);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	default:
	  method_or_bust(sc, y, sc->modulo_symbol, args, T_REAL, 2);
	}

    default:
      method_or_bust(sc, x, sc->modulo_symbol, args, T_REAL, 1);
    }
}

IF2_TO_IF(modulo, c_mod)
static s7_double c_mod_r(s7_scheme *sc, s7_double x, s7_double y) {return(x - y * (s7_int)floor(x / y));}
RF2_TO_RF(modulo, c_mod_r)

static s7_pointer mod_si;
static s7_pointer g_mod_si(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int y;

  x = find_symbol_checked(sc, car(args));
  y = integer(cadr(args));

  if (is_integer(x))
    {
      s7_int z;
      /* here we know y is positive */
      z = integer(x) % y;
      if (z < 0)
	return(make_integer(sc, z + y));
      return(make_integer(sc, z));
    }

  if (is_t_real(x))
    {
      s7_double a, b;
      a = real(x);
      if (is_NaN(a)) return(x);
      if (is_inf(a)) return(real_NaN);
      b = (s7_double)y;
      return(make_real(sc, a - b * (s7_int)floor(a / b)));
    }

  if (s7_is_ratio(x))
    return(g_modulo(sc, set_plist_2(sc, x, cadr(args))));

  method_or_bust(sc, x, sc->modulo_symbol, list_2(sc, x, cadr(args)), T_REAL, 1);
}

static s7_pointer g_is_zero(s7_scheme *sc, s7_pointer args);
static s7_pointer mod_si_is_zero;
static s7_pointer g_mod_si_is_zero(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int y;

  /* car is (modulo symbol integer), cadr is 0 or not present (if zero?) */
  x = find_symbol_checked(sc, cadar(args));
  y = integer(caddar(args));

  if (is_integer(x))
    return(make_boolean(sc, (integer(x) % y) == 0));

  if (is_t_real(x))
    return(make_boolean(sc, (fmod(real(x), (s7_double)y) == 0.0)));

  if (s7_is_ratio(x))
    return(sc->F);

  {
    s7_pointer func;
    if ((func = find_method(sc, find_let(sc, x), sc->modulo_symbol)) != sc->undefined)
      return(g_is_zero(sc, set_plist_1(sc, s7_apply_function(sc, func, list_2(sc, x, caddar(args))))));
  }
  return(wrong_type_argument(sc, sc->modulo_symbol, 1, x, T_REAL));
}
#endif
/* !WITH_GMP */


static int reduce_fraction(s7_scheme *sc, s7_int *numer, s7_int *denom)
{
  /* we're assuming in several places that we have a normal s7 rational after returning,
   *    so the denominator needs to be positive.
   */
  s7_int divisor;

  if (*numer == 0)
    {
      *denom = 1;
      return(T_INTEGER);
    }
  if (*denom < 0)
    {
      if (*denom == *numer)
	{
	  *denom = 1;
	  *numer = 1;
	  return(T_INTEGER);
	}
      if (*denom == s7_int_min)
	{
	  if (*numer & 1)
	    return(T_RATIO);
	  *denom /= 2;
	  *numer /= 2;
	}
      else
	{
	  if (*numer == s7_int_min)
	    {
	      if (*denom & 1)
		return(T_RATIO);
	      *denom /= 2;
	      *numer /= 2;
	    }
	}
      *denom = -*denom; 
      *numer = -*numer;
    }
  divisor = c_gcd(*numer, *denom);
  if (divisor != 1)
    {
      *numer /= divisor;
      *denom /= divisor;
    }
  if (*denom == 1)
    return(T_INTEGER);
  return(T_RATIO);
}



/* ---------------------------------------- add ---------------------------------------- */

static s7_pointer g_add(s7_scheme *sc, s7_pointer args)
{
  #define H_add "(+ ...) adds its arguments"
  #define Q_add pcl_n
  s7_pointer x, p;
  s7_int num_a, den_a, dn;
  s7_double rl_a, im_a;

#if (!WITH_GMP)
  if (is_null(args))
    return(small_int(0));
#endif

  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	method_or_bust_with_type(sc, x, sc->add_symbol, args, a_number_string, 0);
      return(x);
    }

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);

    ADD_INTEGERS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_add(sc, cons(sc, s7_int_to_big_integer(sc, num_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  if (add_overflow(num_a, integer(x), &den_a))
	    {
	      rl_a = (s7_double)num_a + (s7_double)integer(x);
	      if (is_null(p)) return(make_real(sc, rl_a));
	      goto ADD_REALS;
	    }
#else	    
	  den_a = num_a + integer(x);
	  if (den_a < 0)
	    {
	      if ((num_a > 0) && (integer(x) > 0))
		{
		  rl_a = (s7_double)num_a + (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));
		  goto ADD_REALS;
		}
	    }
	  else
	    {
	      if ((num_a < 0) && (integer(x) < 0))
		{
		  rl_a = (s7_double)num_a + (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));

		  /* this is not ideal!  piano.scm has its own noise generator that wants integer
		   *    arithmetic to overflow as an integer.  Perhaps 'safety==0 would not check
		   *    anywhere?
		   */
		  goto ADD_REALS;
		}
	    }
#endif
	  if (is_null(p)) return(make_integer(sc, den_a));
	  num_a = den_a;
	  /* (+ 4611686018427387904 4611686018427387904) -> -9223372036854775808
	   * (+ most-positive-fixnum most-positive-fixnum) -> -2
	   * (+ most-negative-fixnum most-negative-fixnum) -> 0
	   * can't check result - arg: (- 0 most-negative-fixnum) -> most-negative-fixnum
	   */
	  goto ADD_INTEGERS;

	case T_RATIO:
	  den_a = denominator(x);
#if HAVE_OVERFLOW_CHECKS
	  if ((multiply_overflow(den_a, num_a, &dn)) || 
	      (add_overflow(dn, numerator(x), &dn)))
#else
	  if ((integer_length(num_a) + integer_length(den_a) + integer_length(numerator(x))) < s7_int_bits)
	    dn = numerator(x) + (num_a * den_a);
	  else
#endif
	    {
	      if (is_null(p))
		{
		  if (num_a == 0)                /* (+ 0 1/9223372036854775807) */
		    return(x);
		  return(make_real(sc, num_a + fraction(x)));
		}
	      rl_a = (s7_double)num_a + fraction(x);
	      goto ADD_REALS;
	    }
	  if (is_null(p)) return(s7_make_ratio(sc, dn, den_a));
	  num_a = dn;

	  /* overflow examples:
	   *   (+ 100000 1/142857142857140) -> -832205957599110323/28571428571428
	   *   (+ 4611686018427387904 3/4) -> 3/4
	   * see s7test for more
	   */
	  goto ADD_RATIOS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, num_a + real(x)));
	  rl_a = (s7_double)num_a + real(x);
	  goto ADD_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, num_a + real_part(x), imag_part(x)));
	  rl_a = (s7_double)num_a + real_part(x);
	  im_a = imag_part(x);
	  goto ADD_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->add_symbol, cons(sc, s7_make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
    ADD_RATIOS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (den_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_add(sc, cons(sc, s7_ratio_to_big_ratio(sc, num_a, den_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  if ((multiply_overflow(den_a, integer(x), &dn)) || 
	      (add_overflow(dn, num_a, &dn)))
#else
	  if ((integer_length(integer(x)) + integer_length(den_a) + integer_length(num_a)) < s7_int_bits)
	    dn = num_a + (integer(x) * den_a);
	  else
#endif
	    {
	      /* (+ 3/4 4611686018427387904) -> 3/4
	       * (+ 1/17179869184 1073741824) -> 1/17179869184
	       * (+ 1/8589934592 1073741824) -> -9223372036854775807/8589934592
	       */
	      if (is_null(p))
		return(make_real(sc, (s7_double)integer(x) + ((long double)num_a / (long double)den_a)));
	      rl_a = (s7_double)integer(x) + ((long double)num_a / (long double)den_a);
	      goto ADD_REALS;
	    }
	  if (is_null(p)) return(s7_make_ratio(sc, dn, den_a));
	  num_a = dn;
	  if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto ADD_INTEGERS;
	  goto ADD_RATIOS;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = den_a;
	    n1 = num_a;
	    d2 = denominator(x);
	    n2 = numerator(x);
	    if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int case */
	      {
		if (is_null(p))
		  return(s7_make_ratio(sc, n1 + n2, d1));
		num_a += n2;                  /* d1 can't be zero */
	      }
	    else
	      {
#if (!WITH_GMP)
#if HAVE_OVERFLOW_CHECKS
		s7_int n1d2, n2d1;
		if ((multiply_overflow(d1, d2, &den_a)) ||
		    (multiply_overflow(n1, d2, &n1d2)) ||
		    (multiply_overflow(n2, d1, &n2d1)) ||
		    (add_overflow(n1d2, n2d1, &num_a)))
		  {
		    if (is_null(p))
		      return(make_real(sc, ((long double)n1 / (long double)d1) + ((long double)n2 / (long double)d2)));
		    rl_a = ((long double)n1 / (long double)d1) + ((long double)n2 / (long double)d2);
		    goto ADD_REALS;
		  }
#else
		if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		    (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
		    (n1 < s7_int32_min) || (n2 < s7_int32_min))
		  {
		    int d1bits, d2bits;
		    d1bits = integer_length(d1);
		    d2bits = integer_length(d2);
		    if (((d1bits + d2bits) > s7_int_bits) ||
			((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
			((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
		      {
			if (is_null(p))
			  return(make_real(sc, ((long double)n1 / (long double)d1) + ((long double)n2 / (long double)d2)));
			rl_a = ((long double)n1 / (long double)d1) + ((long double)n2 / (long double)d2);
			/* this can lose:
			 *   (+ 1 1/9223372036854775807 -1) -> 0.0 not 1/9223372036854775807
			 */
			goto ADD_REALS;
		      }
		  }
		num_a = n1 * d2 + n2 * d1;
		den_a = d1 * d2;
#endif
#else
		num_a = n1 * d2 + n2 * d1;
		den_a = d1 * d2;
#endif
		if (is_null(p))
		  return(s7_make_ratio(sc, num_a, den_a));
	      }
	    /* (+ 1/100 99/100 (- most-positive-fixnum 2)) should not be converted to real
	     */
	    if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto ADD_INTEGERS;
	  goto ADD_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, ((long double)num_a / (long double)den_a) + real(x)));
	  rl_a = ((long double)num_a / (long double)den_a) + real(x);
	  goto ADD_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, ((long double)num_a / (long double)den_a) + real_part(x), imag_part(x)));
	  rl_a = ((long double)num_a / (long double)den_a) + real_part(x);
	  im_a = imag_part(x);
	  goto ADD_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->add_symbol, cons(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_REAL:
      rl_a = real(x);

    ADD_REALS:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(make_real(sc, rl_a + integer(x)));
	  rl_a += (s7_double)integer(x);
	  goto ADD_REALS;

	case T_RATIO:
	  if (is_null(p)) return(make_real(sc, rl_a + fraction(x)));
	  rl_a += (s7_double)fraction(x);
	  goto ADD_REALS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, rl_a + real(x)));
	  rl_a += real(x);
	  goto ADD_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + real_part(x), imag_part(x)));
	  rl_a += real_part(x);
	  im_a = imag_part(x);
	  goto ADD_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->add_symbol, cons(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);

    ADD_COMPLEX:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + integer(x), im_a));
	  rl_a += (s7_double)integer(x);
	  goto ADD_COMPLEX;

	case T_RATIO:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + fraction(x), im_a));
	  rl_a += (s7_double)fraction(x);
	  goto ADD_COMPLEX;

	case T_REAL:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + real(x), im_a));
	  rl_a += real(x);
	  goto ADD_COMPLEX;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + real_part(x), im_a + imag_part(x)));
	  rl_a += real_part(x);
	  im_a += imag_part(x);
	  if (im_a == 0.0)
	    goto ADD_REALS;
	  goto ADD_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->add_symbol, cons(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    default:
      method_or_bust_with_type(sc, x, sc->add_symbol, args, a_number_string, 1);
    }
}


static s7_pointer add_2, add_1s, add_s1, add_cs1, add_si, add_sf, add_fs;

static s7_pointer add_ratios(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  s7_int d1, d2, n1, n2;
  d1 = number_to_denominator(x);
  n1 = number_to_numerator(x);
  d2 = number_to_denominator(y);
  n2 = number_to_numerator(y);

  if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int case */
    return(s7_make_ratio(sc, n1 + n2, d1));

#if HAVE_OVERFLOW_CHECKS
  {
    s7_int n1d2, n2d1, d1d2, dn;
    if ((multiply_overflow(d1, d2, &d1d2)) ||
	(multiply_overflow(n1, d2, &n1d2)) ||
	(multiply_overflow(n2, d1, &n2d1)) ||
	(add_overflow(n1d2, n2d1, &dn)))
      return(make_real(sc, ((long double)n1 / (long double)d1) + ((long double)n2 / (long double)d2)));
    return(s7_make_ratio(sc, dn, d1d2));
  }
#else
  if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
      (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
      (n1 < s7_int32_min) || (n2 < s7_int32_min))
    {
      int d1bits, d2bits;
      d1bits = integer_length(d1);
      d2bits = integer_length(d2);
      if (((d1bits + d2bits) > s7_int_bits) ||
	  ((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
	  ((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
	return(make_real(sc, ((long double)n1 / (long double)d1) + ((long double)n2 / (long double)d2)));
    }
  return(s7_make_ratio(sc, n1 * d2 + n2 * d1, d1 * d2));
#endif
}


static s7_pointer g_add_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  x = car(args);
  y = cadr(args);

  if (type(x) == type(y))
    {
      if (is_t_real(x))
	return(make_real(sc, real(x) + real(y)));
      else
	{
	  switch (type(x))
	    {
#if HAVE_OVERFLOW_CHECKS
	    case T_INTEGER:
	      {
		s7_int val;
		if (add_overflow(integer(x), integer(y), &val))
		  return(make_real(sc, (double)integer(x) + (double)integer(y)));
		return(make_integer(sc, val));
	      }
#else
	    case T_INTEGER: return(make_integer(sc, integer(x) + integer(y)));
#endif
	    case T_RATIO:   return(add_ratios(sc, x, y));
	    case T_REAL:    return(make_real(sc, real(x) + real(y)));
	    case T_COMPLEX: return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
	    default:
	      if (!is_number(x))
		method_or_bust_with_type(sc, x, sc->add_symbol, args, a_number_string, 1);
	      method_or_bust_with_type(sc, y, sc->add_symbol, args, a_number_string, 2);
	    }
	}
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_integer(sc, integer(x) + integer(y)));
	case T_RATIO:   return(add_ratios(sc, x, y));
	case T_REAL:    return(make_real(sc, integer(x) + real(y)));
	case T_COMPLEX: return(make_complex(sc, integer(x) + real_part(y), imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->add_symbol, args, a_number_string, 2);
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:   return(add_ratios(sc, x, y));
	case T_REAL:    return(make_real(sc, fraction(x) + real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, fraction(x) + real_part(y), imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->add_symbol, args, a_number_string, 2);
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_real(sc, real(x) + integer(y)));
	case T_RATIO:   return(make_real(sc, real(x) + fraction(y)));
	case T_REAL:    return(make_real(sc, real(x) + real(y)));
	case T_COMPLEX: return(make_complex(sc, real(x) + real_part(y), imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->add_symbol, args, a_number_string, 2);
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER: return(s7_make_complex(sc, real_part(x) + integer(y), imag_part(x)));
	case T_RATIO:   return(s7_make_complex(sc, real_part(x) + fraction(y), imag_part(x)));
	case T_REAL:    return(s7_make_complex(sc, real_part(x) + real(y), imag_part(x)));
	case T_COMPLEX: return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->add_symbol, args, a_number_string, 2);
	}

    default:
      method_or_bust_with_type(sc, x, sc->add_symbol, args, a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_add_s1_1(s7_scheme *sc, s7_pointer x, s7_pointer args)
{
  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (add_overflow(integer(x), 1, &val))
	  return(make_real(sc, (double)integer(x) + 1.0));
	return(make_integer(sc, val));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) + 1));
#endif
    case T_RATIO:   return(add_ratios(sc, x, small_int(1)));
    case T_REAL:    return(make_real(sc, real(x) + 1.0));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + 1.0, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->add_symbol, cons(sc, x, cdr(args)), a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_add_s1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = car(args);
  if (is_t_integer(x))
    return(make_integer(sc, integer(x) + 1));
  return(g_add_s1_1(sc, x, args));
}

static s7_pointer c_add_s1(s7_scheme *sc, s7_pointer x)
{
  if (is_t_integer(x))
    return(make_integer(sc, integer(x) + 1));
  return(g_add_s1_1(sc, x, set_plist_1(sc, x)));
}

static s7_pointer g_add_cs1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = find_symbol_checked(sc, car(args));
  if (is_integer(x))
    return(make_integer(sc, integer(x) + 1));
  return(g_add_s1_1(sc, x, args));
}

static s7_pointer g_add_1s(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;

  x = cadr(args);
  if (is_integer(x))
    return(make_integer(sc, integer(x) + 1));

  switch (type(x))
    {
    case T_INTEGER: return(make_integer(sc, integer(x) + 1));
    case T_RATIO:   return(add_ratios(sc, x, small_int(1)));
    case T_REAL:    return(make_real(sc, real(x) + 1.0));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + 1.0, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->add_symbol, args, a_number_string, 2);
    }
  return(x);
}

static s7_pointer g_add_si(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int n;

  x = find_symbol_checked(sc, car(args));
  n = integer(cadr(args));
  if (is_integer(x))
#if HAVE_OVERFLOW_CHECKS
    {
      s7_int val;
      if (add_overflow(integer(x), n, &val))
	return(make_real(sc, (double)integer(x) + (double)n));
      return(make_integer(sc, val));
    }
#else
    return(make_integer(sc, integer(x) + n));
#endif
  switch (type(x))
    {
    case T_INTEGER: return(make_integer(sc, integer(x) + n));
    case T_RATIO:   return(add_ratios(sc, x, cadr(args)));
    case T_REAL:    return(make_real(sc, real(x) + n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + n, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->add_symbol, list_2(sc, x, cadr(args)), a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_add_sf(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = find_symbol_checked(sc, car(args));
  n = real(cadr(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) + n));
    case T_RATIO:   return(make_real(sc, fraction(x) + n));
    case T_REAL:    return(make_real(sc, real(x) + n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + n, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->add_symbol, list_2(sc, x, cadr(args)), a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_add_fs(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = find_symbol_checked(sc, cadr(args));
  n = real(car(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) + n));
    case T_RATIO:   return(make_real(sc, fraction(x) + n));
    case T_REAL:    return(make_real(sc, real(x) + n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + n, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->add_symbol, list_2(sc, x, car(args)), a_number_string, 2);
    }
  return(x);
}

static s7_pointer add_f_sf;
static s7_pointer g_add_f_sf(s7_scheme *sc, s7_pointer args)
{
  /* (+ x (* s y)) */
  s7_pointer vargs, s;
  s7_double x, y;

  x = real(car(args));
  vargs = cdadr(args);
  s = find_symbol_checked(sc, car(vargs));
  y = real(cadr(vargs));

  if (is_t_real(s))
    return(make_real(sc, x + (real(s) * y)));

  switch (type(s))
    {
    case T_INTEGER: return(make_real(sc, x + (integer(s) * y)));
    case T_RATIO:   return(make_real(sc, x + (fraction(s) * y)));
    case T_REAL:    return(make_real(sc, x + real(s) * y));
    case T_COMPLEX: return(s7_make_complex(sc, x + (real_part(s) * y), imag_part(s) * y));
    default:
      {
	s7_pointer func;
	if ((func = find_method(sc, find_let(sc, s), sc->multiply_symbol)) != sc->undefined)
	  return(g_add_2(sc, set_plist_2(sc, car(args), s7_apply_function(sc, func, list_2(sc, s, cadr(vargs))))));
	return(wrong_type_argument_with_type(sc, sc->multiply_symbol, 1, s, a_number_string));
      }
    }
  return(s);
}


static s7_pointer add_ss_1ss_1(s7_scheme *sc, s7_pointer s1, s7_pointer s2, s7_pointer s3)
{
  s7_double r1, r2, r3, loc, i1, i2, i3, is1;
  if ((is_t_real(s1)) &&
      (is_t_real(s2)) &&
      (is_t_real(s3)))
    return(make_real(sc, (real(s1) * real(s2))  + ((1.0 - real(s1)) * real(s3))));

  if ((is_real(s1)) &&
      (is_real(s2)) &&
      (is_real(s3)))
    {
      r1 = real_to_double(sc, s1, "*");
      r2 = real_to_double(sc, s2, "*");
      r3 = real_to_double(sc, s3, "*");
      return(make_real(sc, (r1 * r2)  + ((1.0 - r1) * r3)));
    }

  r1 = s7_real_part(s1);
  loc = 1.0 - r1;
  r2 = s7_real_part(s2);
  r3 = s7_real_part(s3);
  i1 = s7_imag_part(s1);
  is1 = -i1;
  i2 = s7_imag_part(s2);
  i3 = s7_imag_part(s3);
  return(s7_make_complex(sc,
			 (r1 * r2 - i1 * i2) + (loc * r3 - is1 * i3),
			 (r1 * i2 + r2 * i1) + (loc * i3 + r3 * is1)));
  /* (let ()
   *   (define (hi a b c) (+ (* a b) (* (- 1.0 a) c)))
   *   (define (hi1 a b c) (+ (* b a) (* c (- 1 a))))
   *   (define (ho a b c) (list (hi a b c) (hi1 a b c)))
   *   (ho 1.4 2.5+i 3.1))
   */
}

static s7_pointer add_ss_1ss;
static s7_pointer g_add_ss_1ss(s7_scheme *sc, s7_pointer args)
{
  /* (+ (* s1 s2) (* (- 1.0 s1) s3)) */
  s7_pointer s1, s2, s3;
  s1 = find_symbol_checked(sc, cadr(car(args)));
  s2 = find_symbol_checked(sc, opt_sym1(args)); /* caddr(car(args))) */
  s3 = find_symbol_checked(sc, opt_sym2(args)); /* caddr(cadr(args))) */

  return(add_ss_1ss_1(sc, s1, s2, s3));
}


#if (!WITH_GMP)
static s7_double add_rf_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_rf_t r1, r2;
  s7_double x, y;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x + y);
}

static s7_double add_rf_rx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_rf_t r1;
  s1 = **p; (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  return(r1(sc, p) + real_to_double(sc, s1, "+"));
}

static s7_double add_rf_sx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_rf_t r1;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  return(r1(sc, p) + real_to_double(sc, s1, "+"));
}

static s7_double add_rf_ss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s7_double x1;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "+");
  s2 = slot_value(**p); (*p)++;
  return(x1 + real_to_double(sc, s2, "+"));
}

static s7_double add_rf_rs(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s7_double x1;
  s1 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  x1 = real_to_double(sc, c1, "+");
  return(x1 + real_to_double(sc, s1, "+"));
}


static s7_double add_rf_xxx(s7_scheme *sc, s7_pointer **p)
{
  s7_rf_t r1, r2, r3;
  s7_double x, y, z;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  r3 = (s7_rf_t)(**p); (*p)++;
  z = r3(sc, p);
  return(x + y + z);
}

static s7_double add_rf_rxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1;
  s7_rf_t r1, r2;
  s7_double x, y;
  c1 = **p; (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x + y + real_to_double(sc, c1, "+"));
}

static s7_double add_rf_sxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_rf_t r1, r2;
  s7_double x, y;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x + y + real_to_double(sc, s1, "+"));
}

static s7_double add_rf_rsx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s7_rf_t r1;
  s7_double x, x1, x2;
  s1 = slot_value(**p); (*p)++;
  x2 = real_to_double(sc, s1, "+");
  c1 = **p; (*p)++;
  x1 = real_to_double(sc, c1, "+");
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x + x1 + x2);
}

static s7_double add_rf_ssx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s7_rf_t r1;
  s7_double x, x1;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "+");
  s2 = slot_value(**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x + x1 + real_to_double(sc, s2, "+"));
}

static s7_double add_rf_sss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2, s3;
  s7_double x1, x2;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "+");
  s2 = slot_value(**p); (*p)++;
  x2 = real_to_double(sc, s2, "+");
  s3 = slot_value(**p); (*p)++;
  return(x1 + x2 + real_to_double(sc, s3, "+"));
}

static s7_double add_rf_rss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1, s2;
  s7_double x1, x2;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "+");
  s2 = slot_value(**p); (*p)++;
  x2 = real_to_double(sc, s2, "+");
  c1 = **p; (*p)++;
  return(real_to_double(sc, c1, "+") + x1 + x2);
}

static s7_rf_t add_rf_1(s7_scheme *sc, s7_pointer expr, int len)
{
  if (len == 3)
    return(com_rf_2(sc, expr, add_r_ops));
  if (len == 4)
    return(com_rf_3(sc, expr, add_r_ops));

  if (len > 4)
    {
      s7_rf_t rf;
      ptr_int loc;
      int first_len;
      xf_t *rc;

      first_len = (int)(len / 2);
      xf_init(2);
      xf_save_loc(loc);
      rf = add_rf_1(sc, expr, first_len + 1);
      if (rf)
	{
	  int i;
	  s7_pointer p;
	  xf_store_at(loc, (s7_pointer)rf);
	  xf_save_loc(loc);
	  for (i = 0, p = expr; i < first_len; i++, p = cdr(p));
	  rf = add_rf_1(sc, p, len - first_len);
	  if (rf)
	    {
	      xf_store_at(loc, (s7_pointer)rf);
	      return(add_rf_xx);
	    }
	  else return(NULL);
	}
      else return(NULL);
    }
  return(NULL);
}

static s7_rf_t add_rf(s7_scheme *sc, s7_pointer expr)
{
  return(add_rf_1(sc, expr, s7_list_length(sc, expr)));
}


static s7_int add_if_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_if_t r1, r2;
  s7_int x, y;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x + y);
}

static s7_int add_if_rx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_if_t r1;
  s1 = **p; (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  return(r1(sc, p) + integer(s1));
}

static s7_int add_if_sx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_if_t r1;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  return(r1(sc, p) + integer(s1));
}

static s7_int add_if_ss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  return(integer(s1) + integer(s2));
}

static s7_int add_if_rs(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s1 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  return(integer(c1) + integer(s1));
}


static s7_int add_if_xxx(s7_scheme *sc, s7_pointer **p)
{
  s7_if_t r1, r2, r3;
  s7_int x, y, z;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  r3 = (s7_if_t)(**p); (*p)++;
  z = r3(sc, p);
  return(x + y + z);
}

static s7_int add_if_rxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1;
  s7_if_t r1, r2;
  s7_int x, y;
  c1 = **p; (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x + y + integer(c1));
}

static s7_int add_if_sxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_if_t r1, r2;
  s7_int x, y;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x + y + integer(s1));
}

static s7_int add_if_rsx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s7_if_t r1;
  s7_int x;
  s1 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x + integer(c1) + integer(s1));
}

static s7_int add_if_ssx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s7_if_t r1;
  s7_int x;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x + integer(s1) + integer(s2));
}

static s7_int add_if_sss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2, s3;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  s3 = slot_value(**p); (*p)++;
  return(integer(s1) + integer(s2) + integer(s3));
}

static s7_int add_if_rss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1, s2;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  return(integer(c1) + integer(s1) + integer(s2));
}

static s7_if_t add_if_1(s7_scheme *sc, s7_pointer expr, int len)
{
  if (len == 3)
    return(com_if_2(sc, expr, add_i_ops));
  if (len == 4)
    return(com_if_3(sc, expr, add_i_ops));

  if (len > 4)
    {
      s7_if_t xf;
      ptr_int loc;
      int first_len;
      xf_t *rc;

      xf_init(2);
      xf_save_loc(loc);
      first_len = (int)(len / 2);
      xf = add_if_1(sc, expr, first_len + 1);
      if (xf)
	{
	  int i;
	  s7_pointer p;
	  xf_store_at(loc, (s7_pointer)xf);
	  xf_save_loc(loc);
	  for (i = 0, p = expr; i < first_len; i++, p = cdr(p));
	  xf = add_if_1(sc, p, len - first_len);
	  if (xf)
	    {
	      xf_store_at(loc, (s7_pointer)xf);
	      return(add_if_xx);
	    }
	  else return(NULL);
	}
      else return(NULL);
    }
  return(NULL);
}

static s7_if_t add_if(s7_scheme *sc, s7_pointer expr)
{
  return(add_if_1(sc, expr, s7_list_length(sc, expr)));
}


static void init_add_ops(void)
{
  add_r_ops = (rf_ops *)calloc(1, sizeof(rf_ops));
  add_r_ops->r = rf_c;
  add_r_ops->s = rf_s;

  add_r_ops->rs = add_rf_rs;
  add_r_ops->rp = add_rf_rx;
  add_r_ops->sp = add_rf_sx;
  add_r_ops->ss = add_rf_ss;
  add_r_ops->pp = add_rf_xx;

  add_r_ops->rss = add_rf_rss;
  add_r_ops->rsp = add_rf_rsx;
  add_r_ops->rpp = add_rf_rxx;
  add_r_ops->sss = add_rf_sss;
  add_r_ops->ssp = add_rf_ssx;
  add_r_ops->spp = add_rf_sxx;
  add_r_ops->ppp = add_rf_xxx;

  add_i_ops = (if_ops *)calloc(1, sizeof(if_ops));
  add_i_ops->r = if_c;
  add_i_ops->s = if_s;

  add_i_ops->rs = add_if_rs;
  add_i_ops->rp = add_if_rx;
  add_i_ops->sp = add_if_sx;
  add_i_ops->ss = add_if_ss;
  add_i_ops->pp = add_if_xx;

  add_i_ops->rss = add_if_rss;
  add_i_ops->rsp = add_if_rsx;
  add_i_ops->rpp = add_if_rxx;
  add_i_ops->sss = add_if_sss;
  add_i_ops->ssp = add_if_ssx;
  add_i_ops->spp = add_if_sxx;
  add_i_ops->ppp = add_if_xxx;
}

#if WITH_ADD_PF
static s7_pointer c_add_pf2(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t pf;
  s7_pointer x, y;
  pf = (s7_pf_t)(**p); (*p)++;
  x = pf(sc, p);
  xf_push(sc, x);
  pf = (s7_pf_t)(**p); (*p)++;
  y = pf(sc, p);
  x = g_add_2(sc, set_plist_2(sc, x, y));
  xf_pop(sc);
  return(x);
}

static s7_pf_t add_pf(s7_scheme *sc, s7_pointer expr)
{
  int len;
  len = s7_list_length(sc, expr);
  if (len == 3)
    {
      if ((s7_arg_to_pf(sc, cadr(expr))) &&
	  (s7_arg_to_pf(sc, caddr(expr))))
	return(c_add_pf2);
    }
  return(NULL);
}
#endif

#endif


/* ---------------------------------------- subtract ---------------------------------------- */

static s7_pointer g_subtract(s7_scheme *sc, s7_pointer args)
{
  #define H_subtract "(- x1 ...) subtracts its trailing arguments from the first, or negates the first if only one it is given"
  #define Q_subtract pcl_n

  s7_pointer x, p;
  s7_int num_a, den_a;
  s7_double rl_a, im_a;

  x = car(args);
  p = cdr(args);

#if (!WITH_GMP)
  if (is_null(p))
    {
      if (!is_number(x))
	method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 0);
      return(s7_negate(sc, x));
    }
#endif

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);

    SUBTRACT_INTEGERS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_subtract(sc, cons(sc, s7_int_to_big_integer(sc, num_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  if (subtract_overflow(num_a, integer(x), &den_a))
	    {
	      rl_a = (s7_double)num_a - (s7_double)integer(x);
	      if (is_null(p)) return(make_real(sc, rl_a));
	      goto SUBTRACT_REALS;
	    }
#else
	  den_a = num_a - integer(x);
	  if (den_a < 0)
	    {
	      if ((num_a > 0) && (integer(x) < 0))
		{
		  rl_a = (s7_double)num_a - (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));
		  goto SUBTRACT_REALS;
		}
	      /* (- most-positive-fixnum most-negative-fixnum) -> -1 (1.8446744073709551615E19)
	       */
	    }
	  else
	    {
	      if ((num_a < 0) && (integer(x) > 0))
		{
		  rl_a = (s7_double)num_a - (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));
		  goto SUBTRACT_REALS;
		}
	      /* (- most-negative-fixnum most-positive-fixnum) -> 1 (-1.8446744073709551615E19)
	       */
	    }
#endif
	  if (is_null(p)) return(make_integer(sc, den_a));
	  num_a = den_a;
	  goto SUBTRACT_INTEGERS;

	case T_RATIO:
	  {
	    s7_int dn;
	    den_a = denominator(x);
#if HAVE_OVERFLOW_CHECKS
	    if ((multiply_overflow(num_a, den_a, &dn)) ||
		(subtract_overflow(dn, numerator(x), &dn)))
	      {
		if (is_null(p)) return(make_real(sc, num_a - fraction(x)));
		rl_a = (s7_double)num_a - fraction(x);
		goto SUBTRACT_REALS;
	      }
#else
	    if ((integer_length(num_a) + integer_length(den_a) + integer_length(numerator(x))) > s7_int_bits)
	      {
		if (is_null(p)) return(make_real(sc, num_a - fraction(x)));
		rl_a = (s7_double)num_a - fraction(x);
		goto SUBTRACT_REALS;
	      }
	    dn = (num_a * den_a) - numerator(x);
#endif
	    if (is_null(p)) return(s7_make_ratio(sc, dn, den_a));
	    num_a = dn;
	    if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	      goto SUBTRACT_INTEGERS;
	    goto SUBTRACT_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, num_a - real(x)));
	  rl_a = (s7_double)num_a - real(x);
	  goto SUBTRACT_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, num_a - real_part(x), -imag_part(x)));
	  rl_a = (s7_double)num_a - real_part(x);
	  im_a = -imag_part(x);
	  goto SUBTRACT_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->subtract_symbol, cons(sc, s7_make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
    SUBTRACT_RATIOS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (den_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_subtract(sc, cons(sc, s7_ratio_to_big_ratio(sc, num_a, den_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int di;
	    if ((multiply_overflow(den_a, integer(x), &di)) ||
		(subtract_overflow(num_a, di, &di)))
	      {
		if (is_null(p)) return(make_real(sc, ((long double)num_a / (long double)den_a) - integer(x)));
		rl_a = ((long double)num_a / (long double)den_a) - integer(x);
		goto SUBTRACT_REALS;
	      }
	    if (is_null(p)) return(s7_make_ratio(sc, di, den_a));
	    num_a = di;
	  }
#else
	  if ((integer_length(integer(x)) + integer_length(num_a) + integer_length(den_a)) > s7_int_bits)
	    {
	      if (is_null(p)) return(make_real(sc, ((long double)num_a / (long double)den_a) - integer(x)));
	      rl_a = ((long double)num_a / (long double)den_a) - integer(x);
	      goto SUBTRACT_REALS;
	    }
	  if (is_null(p)) return(s7_make_ratio(sc, num_a - (den_a * integer(x)), den_a));
	  num_a -= (den_a * integer(x));
#endif
	  if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto SUBTRACT_INTEGERS;
	  goto SUBTRACT_RATIOS;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = den_a;
	    n1 = num_a;
	    d2 = denominator(x);
	    n2 = numerator(x);
	    if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int case */
	      {
		if (is_null(p))
		  return(s7_make_ratio(sc, n1 - n2, d1));
		num_a -= n2;                  /* d1 can't be zero */
	      }
	    else
	      {
#if (!WITH_GMP)
#if HAVE_OVERFLOW_CHECKS
		s7_int n1d2, n2d1;
		if ((multiply_overflow(d1, d2, &den_a)) ||
		    (multiply_overflow(n1, d2, &n1d2)) ||
		    (multiply_overflow(n2, d1, &n2d1)) ||
		    (subtract_overflow(n1d2, n2d1, &num_a)))
		  {
		    if (is_null(p))
		      return(make_real(sc, ((long double)n1 / (long double)d1) - ((long double)n2 / (long double)d2)));
		    rl_a = ((long double)n1 / (long double)d1) - ((long double)n2 / (long double)d2);
		    goto SUBTRACT_REALS;
		  }
#else
		if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		    (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
		    (n1 < s7_int32_min) || (n2 < s7_int32_min))
		  {
		    int d1bits, d2bits;
		    d1bits = integer_length(d1);
		    d2bits = integer_length(d2);
		    if (((d1bits + d2bits) > s7_int_bits) ||
			((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
			((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
		      {
			if (is_null(p))
			  return(make_real(sc, ((long double)n1 / (long double)d1) - ((long double)n2 / (long double)d2)));
			rl_a = ((long double)n1 / (long double)d1) - ((long double)n2 / (long double)d2);
			goto SUBTRACT_REALS;
		      }
		  }
		num_a = n1 * d2 - n2 * d1;
		den_a = d1 * d2;
#endif
#else
		num_a = n1 * d2 - n2 * d1;
		den_a = d1 * d2;
#endif
		if (is_null(p))
		  return(s7_make_ratio(sc, num_a, den_a));
	      }
	    if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto SUBTRACT_INTEGERS;
	  goto SUBTRACT_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, ((long double)num_a / (long double)den_a) - real(x)));
	  rl_a = ((long double)num_a / (long double)den_a) - real(x);
	  goto SUBTRACT_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, ((long double)num_a / (long double)den_a) - real_part(x), -imag_part(x)));
	  rl_a = ((long double)num_a / (long double)den_a) - real_part(x);
	  im_a = -imag_part(x);
	  goto SUBTRACT_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->subtract_symbol, cons(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_REAL:
      rl_a = real(x);

    SUBTRACT_REALS:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(make_real(sc, rl_a - integer(x)));
	  rl_a -= (s7_double)integer(x);
	  goto SUBTRACT_REALS;

	case T_RATIO:
	  if (is_null(p)) return(make_real(sc, rl_a - fraction(x)));
	  rl_a -= (s7_double)fraction(x);
	  goto SUBTRACT_REALS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, rl_a - real(x)));
	  rl_a -= real(x);
	  goto SUBTRACT_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - real_part(x), -imag_part(x)));
	  rl_a -= real_part(x);
	  im_a = -imag_part(x);
	  goto SUBTRACT_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->subtract_symbol, cons(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);

    SUBTRACT_COMPLEX:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - integer(x), im_a));
	  rl_a -= (s7_double)integer(x);
	  goto SUBTRACT_COMPLEX;

	case T_RATIO:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - fraction(x), im_a));
	  rl_a -= (s7_double)fraction(x);
	  goto SUBTRACT_COMPLEX;

	case T_REAL:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - real(x), im_a));
	  rl_a -= real(x);
	  goto SUBTRACT_COMPLEX;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - real_part(x), im_a - imag_part(x)));
	  rl_a -= real_part(x);
	  im_a -= imag_part(x);
	  if (im_a == 0.0)
	    goto SUBTRACT_REALS;
	  goto SUBTRACT_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->subtract_symbol, cons(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1);
    }
}


static s7_pointer subtract_1, subtract_s1, subtract_cs1, subtract_2, subtract_csn;
static s7_pointer g_subtract_1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p;

  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
      if (integer(p) == s7_int_min)
#if WITH_GMP
	return(big_negate(sc, set_plist_1(sc, promote_number(sc, T_BIG_INTEGER, p))));
#else
        return(make_integer(sc, s7_int_max));
#endif
      return(make_integer(sc, -integer(p)));

    case T_RATIO:
      return(s7_make_ratio(sc, -numerator(p), denominator(p)));

    case T_REAL:
      return(make_real(sc, -real(p)));

    case T_COMPLEX:
      return(s7_make_complex(sc, -real_part(p), -imag_part(p)));

    default:
      method_or_bust_with_type(sc, p, sc->subtract_symbol, args, a_number_string, 1);
    }
}

static s7_pointer g_subtract_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;

  x = car(args);
  y = cadr(args);

  if (type(x) == type(y))
    {
      if (is_t_real(x))
	return(make_real(sc, real(x) - real(y)));
      else
	{
	  switch (type(x))
	    {
#if HAVE_OVERFLOW_CHECKS
	    case T_INTEGER:
	      {
		s7_int val;
		if (subtract_overflow(integer(x), integer(y), &val))
		  return(make_real(sc, (double)integer(x) - (double)integer(y)));
		return(make_integer(sc, val));
	      }
#else
	    case T_INTEGER: return(make_integer(sc, integer(x) - integer(y)));
#endif
	    case T_RATIO:   return(g_subtract(sc, args));
	    case T_REAL:    return(make_real(sc, real(x) - real(y)));
	    case T_COMPLEX: return(make_complex(sc, real_part(x) - real_part(y), imag_part(x) - imag_part(y)));
	    default:
	      if (!is_number(x))
		method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1);
	      method_or_bust_with_type(sc, y, sc->subtract_symbol, args, a_number_string, 2);
	    }
	}
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_integer(sc, integer(x) - integer(y)));
	case T_RATIO:   return(g_subtract(sc, args));
	case T_REAL:    return(make_real(sc, integer(x) - real(y)));
	case T_COMPLEX: return(make_complex(sc, integer(x) - real_part(y), -imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->subtract_symbol, args, a_number_string, 2);
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:   return(g_subtract(sc, args));
	case T_REAL:    return(make_real(sc, fraction(x) - real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, fraction(x) - real_part(y), -imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->subtract_symbol, args, a_number_string, 2);
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_real(sc, real(x) - integer(y)));
	case T_RATIO:   return(make_real(sc, real(x) - fraction(y)));
	case T_REAL:    return(make_real(sc, real(x) - real(y)));
	case T_COMPLEX: return(make_complex(sc, real(x) - real_part(y), -imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->subtract_symbol, args, a_number_string, 2);
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER: return(s7_make_complex(sc, real_part(x) - integer(y), imag_part(x)));
	case T_RATIO:   return(s7_make_complex(sc, real_part(x) - fraction(y), imag_part(x)));
	case T_REAL:    return(s7_make_complex(sc, real_part(x) - real(y), imag_part(x)));
	case T_COMPLEX: return(make_complex(sc, real_part(x) - real_part(y), imag_part(x) - imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->subtract_symbol, args, a_number_string, 2);
	}

    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1);
    }
  return(x);
}


static s7_pointer g_subtract_cs1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = find_symbol_checked(sc, car(args));
  if (is_integer(x))
    return(make_integer(sc, integer(x) - 1));

  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (subtract_overflow(integer(x), 1, &val))
	  return(make_real(sc, (double)integer(x) - 1.0));
	return(make_integer(sc, val));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) - 1));
#endif
    case T_RATIO:   return(subtract_ratios(sc, x, small_int(1)));
    case T_REAL:    return(make_real(sc, real(x) - 1.0));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - 1.0, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, small_int(1)), a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_subtract_s1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = car(args);
  /* this one seems to hit reals as often as integers */
  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (subtract_overflow(integer(x), 1, &val))
	  return(make_real(sc, (double)integer(x) - 1.0));
	return(make_integer(sc, val));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) - 1));
#endif
    case T_RATIO:   return(subtract_ratios(sc, x, small_int(1)));
    case T_REAL:    return(make_real(sc, real(x) - 1.0));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - 1.0, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_subtract_csn(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int n;

  x = find_symbol_checked(sc, car(args));
  n = s7_integer(cadr(args));
  if (is_integer(x))
    return(make_integer(sc, integer(x) - n));

  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (subtract_overflow(integer(x), n, &val))
	  return(make_real(sc, (double)integer(x) - (double)n));
	return(make_integer(sc, val));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) - n));
#endif
    case T_RATIO:   return(subtract_ratios(sc, x, cadr(args)));
    case T_REAL:    return(make_real(sc, real(x) - n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - n, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, cadr(args)), a_number_string, 1);
    }
  return(x);
}

static s7_pointer subtract_sf;
static s7_pointer g_subtract_sf(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = find_symbol_checked(sc, car(args));
  n = real(cadr(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) - n));
    case T_RATIO:   return(make_real(sc, fraction(x) - n));
    case T_REAL:    return(make_real(sc, real(x) - n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - n, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, cadr(args)), a_number_string, 1);
    }
  return(x);
}

static s7_pointer subtract_2f;
static s7_pointer g_subtract_2f(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = car(args);
  n = real(cadr(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) - n));
    case T_RATIO:   return(make_real(sc, fraction(x) - n));
    case T_REAL:    return(make_real(sc, real(x) - n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - n, imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1);
    }
  return(x);
}

static s7_pointer subtract_fs;
static s7_pointer g_subtract_fs(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = find_symbol_checked(sc, cadr(args));
  n = real(car(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, n - integer(x)));
    case T_RATIO:   return(make_real(sc, n - fraction(x)));
    case T_REAL:    return(make_real(sc, n - real(x)));
    case T_COMPLEX: return(s7_make_complex(sc, n - real_part(x), -imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, car(args), x), a_number_string, 2);
    }
  return(x);
}

static s7_pointer subtract_f_sqr;
static s7_pointer g_subtract_f_sqr(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double y;

  y = real(car(args));
  x = find_symbol_checked(sc, cadr(cadr(args)));
  if (is_t_real(x))
    return(make_real(sc, y - (real(x) * real(x))));

  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, y - (integer(x) * integer(x))));
    case T_RATIO:   return(make_real(sc, y - (fraction(x) * fraction(x))));
    case T_REAL:    return(make_real(sc, y - (real(x) * real(x))));
    case T_COMPLEX: return(s7_make_complex(sc, y - real_part(x) * real_part(x) + imag_part(x) * imag_part(x), 2.0 * real_part(x) * imag_part(x)));
    default:
      /* complicated -- look for * method, if any get (* x x), then go to g_subtract_2 with that and the original y
       *   can't use check_method here because it returns from the caller.
       */
      {
        s7_pointer func;
	if ((func = find_method(sc, find_let(sc, x), sc->multiply_symbol)) != sc->undefined)
	  return(g_subtract_2(sc, set_plist_2(sc, car(args), s7_apply_function(sc, func, list_2(sc, x, x)))));
	return(wrong_type_argument_with_type(sc, sc->multiply_symbol, 1, x, a_number_string));
      }
    }
  return(x);
}

#if (!WITH_GMP)
/* (define (hi) (- (random 100) 50)) (define (ho) (- (random 1.0) 0.5)) */
static s7_pointer sub_random_ic, sub_random_rc;
static s7_pointer g_sub_random_ic(s7_scheme *sc, s7_pointer args)
{
  return(make_integer(sc, ((s7_int)(integer(cadar(args)) * next_random(sc->default_rng))) - integer(cadr(args))));
}

static s7_pointer g_sub_random_rc(s7_scheme *sc, s7_pointer args)
{
  return(make_real(sc, real(cadar(args)) * next_random(sc->default_rng) - real(cadr(args))));
}


static s7_int negate_if_c(s7_scheme *sc, s7_pointer **p) {s7_pointer x; x = (**p); (*p)++; return(-integer(x));}
static s7_int negate_if_s(s7_scheme *sc, s7_pointer **p) {s7_pointer x; x = slot_value(**p); (*p)++; return(-integer(x));}
static s7_int negate_if_p(s7_scheme *sc, s7_pointer **p) {s7_if_t f; f = (s7_if_t)(**p); (*p)++; return(f(sc, p));}

static s7_int sub_if_cc(s7_scheme *sc, s7_pointer **p) {s7_pointer x, y; x = (**p); (*p)++; y = (**p); (*p)++; return(integer(x) - integer(y));}
static s7_int sub_if_cs(s7_scheme *sc, s7_pointer **p) {s7_pointer x, y; x = (**p); (*p)++; y = slot_value(**p); (*p)++; return(integer(x) - integer(y));}
static s7_int sub_if_ss(s7_scheme *sc, s7_pointer **p) {s7_pointer x, y; x = slot_value(**p); (*p)++; y = slot_value(**p); (*p)++; return(integer(x) - integer(y));}
static s7_int sub_if_sc(s7_scheme *sc, s7_pointer **p) {s7_pointer x, y; x = slot_value(**p); (*p)++; y = (**p); (*p)++; return(integer(x) - integer(y));}

static s7_int sub_if_cp(s7_scheme *sc, s7_pointer **p) 
{
  s7_if_t xf;
  s7_pointer x; 
  x = (**p); (*p)++; 
  xf = (s7_if_t)(**p); (*p)++; 
  return(integer(x) - xf(sc, p));
}

static s7_int sub_if_pc(s7_scheme *sc, s7_pointer **p) 
{
  s7_if_t xf; 
  s7_int x; 
  s7_pointer y; 
  xf = (s7_if_t)(**p); (*p)++; x = xf(sc,p); 
  y = (**p); (*p)++; 
  return(x - integer(y));
}

static s7_int sub_if_sp(s7_scheme *sc, s7_pointer **p) 
{
  s7_if_t xf; 
  s7_pointer x; 
  x = slot_value(**p); (*p)++; 
  xf = (s7_if_t)(**p); (*p)++; 
  return(integer(x) - xf(sc, p));
}

static s7_int sub_if_ps(s7_scheme *sc, s7_pointer **p) 
{
  s7_if_t xf; 
  s7_int x; 
  s7_pointer y; 
  xf = (s7_if_t)(**p); (*p)++; x = xf(sc,p); 
  y = slot_value(**p); (*p)++;
  return(x - integer(y));
}

static s7_int sub_if_pp(s7_scheme *sc, s7_pointer **p) 
{
  s7_if_t xf; 
  s7_int x, y;
  xf = (s7_if_t)(**p); (*p)++; x = xf(sc,p); 
  xf = (s7_if_t)(**p); (*p)++; y = xf(sc,p); 
  return(x - y);
}


static s7_if_t subtract_if(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer a1, a2, slot;
  xf_t *rc;
  if (!is_pair(cdr(expr))) return(NULL);
  
  xf_init(2);
  a1 = cadr(expr);
  if (is_null(cddr(expr)))
    {
      if (is_t_integer(a1))
	{
	  xf_store(a1);
	  return(negate_if_c);
	}
      if (is_symbol(a1))
	{
	  s7_pointer s1;
	  s1 = s7_slot(sc, a1);
	  if ((!is_slot(s1)) || (is_unsafe_stepper(s1)) || (!is_t_integer(slot_value(s1)))) return(NULL);
	  xf_store(s1);
	  return(negate_if_s);
	}
      if ((is_pair(a1)) &&
	  (s7_arg_to_if(sc, a1)))
	return(negate_if_p);
      return(NULL);
    }
  
  a2 = caddr(expr);
  if (is_null(cdddr(expr)))
    {
      if (is_t_integer(a1))
	{
	  xf_store(a1);
	  if (is_t_integer(a2))
	    {
	      xf_store(a2);
	      return(sub_if_cc);
	    }
	  if (is_symbol(a2))
	    {
	      slot = s7_slot(sc, a2);
	      if ((!slot) || (!is_t_integer(slot_value(slot))) || (is_unsafe_stepper(slot))) return(NULL);
	      xf_store(slot);
	      return(sub_if_cs);
	    }
	  if ((is_pair(a2)) &&
	      (s7_arg_to_if(sc, a2)))
	    return(sub_if_cp);
	  return(NULL);
	}
      if (is_symbol(a1))
	{
	  slot = s7_slot(sc, a1);
	  if ((!slot) || (!is_t_integer(slot_value(slot))) || (is_unsafe_stepper(slot))) return(NULL);
	  xf_store(slot);
	  if (is_t_integer(a2))
	    {
	      xf_store(a2);
	      return(sub_if_sc);
	    }
	  if (is_symbol(a2))
	    {
	      slot = s7_slot(sc, a2);
	      if ((!slot) || (!is_t_integer(slot_value(slot))) || (is_unsafe_stepper(slot))) return(NULL);
	      xf_store(slot);
	      return(sub_if_ss);
	    }
	  if ((is_pair(a2)) &&
	      (s7_arg_to_if(sc, a2)))
	    return(sub_if_sp);
	  return(NULL);
	}
      if (is_pair(a1) &&
	  (s7_arg_to_if(sc, a1)))
	{
	  if (is_t_integer(a2))
	    {
	      xf_store(a2);
	      return(sub_if_pc);
	    }
	  if (is_symbol(a2))
	    {
	      slot = s7_slot(sc, a2);
	      if ((!slot) || (!is_t_integer(slot_value(slot))) || (is_unsafe_stepper(slot))) return(NULL);
	      xf_store(slot);
	      return(sub_if_ps);
	    }
	  if ((is_pair(a2)) &&
	      (s7_arg_to_if(sc, a2)))
	    return(sub_if_pp);
	}
      return(NULL);
    }
  
  {
    s7_if_t xf, res;
    ptr_int loc;
    
    if (is_t_integer(a1))
      {
	xf_store(a1);
	res = sub_if_cp;
      }
    else
      {
	if (is_symbol(a1))
	  {
	    slot = s7_slot(sc, a1);
	    if ((!slot) || (!is_t_integer(slot_value(slot))) || (is_unsafe_stepper(slot))) return(NULL);
	    xf_store(slot);
	    res = sub_if_sp;
	  }
	else 
	  {
	    if ((!is_pair(a1)) || (!s7_arg_to_if(sc, a1))) return(NULL);
	    res = sub_if_pp;
	  }
      }
    
    xf_save_loc(loc);
    xf = add_if(sc, cdr(expr));
    if (xf)
      {
	xf_store_at(loc, (s7_pointer)xf);
	return(res);
      }
  }
  return(NULL);
}


static s7_double negate_rf_c(s7_scheme *sc, s7_pointer **p) {s7_pointer x; x = (**p); (*p)++; return(-(real_to_double(sc, x, "-")));}
static s7_double negate_rf_s(s7_scheme *sc, s7_pointer **p) {s7_pointer x; x = slot_value(**p); (*p)++; return(-(real_to_double(sc, x, "-")));}
static s7_double negate_rf_p(s7_scheme *sc, s7_pointer **p) {s7_rf_t f; f = (s7_rf_t)(**p); (*p)++; return(f(sc, p));}

static s7_double sub_rf_cc(s7_scheme *sc, s7_pointer **p) 
{
  s7_pointer x, y; 
  x = (**p); (*p)++; 
  y = (**p); (*p)++; 
  return(real(x) - real_to_double(sc, y, "-"));
}

static s7_double sub_rf_cs(s7_scheme *sc, s7_pointer **p) 
{
  s7_pointer x, y; 
  x = (**p); (*p)++;
  y = slot_value(**p); (*p)++; 
  return(real(x) - real_to_double(sc, y, "-"));
}

static s7_double sub_rf_ss(s7_scheme *sc, s7_pointer **p) 
{
  s7_pointer x, y; 
  s7_double x1;
  x = slot_value(**p); (*p)++; 
  y = slot_value(**p); (*p)++; 
  x1 = real_to_double(sc, x, "-");
  return(x1 - real_to_double(sc, y, "-"));
}

static s7_double sub_rf_sc(s7_scheme *sc, s7_pointer **p) 
{
  s7_pointer x, y; 
  x = slot_value(**p); (*p)++; 
  y = (**p); (*p)++; 
  return(real_to_double(sc, x, "-") - real(y));
}

static s7_double sub_rf_cp(s7_scheme *sc, s7_pointer **p) 
{
  s7_rf_t rf;
  s7_pointer x; 
  x = (**p); (*p)++; 
  rf = (s7_rf_t)(**p); (*p)++; 
  return(real_to_double(sc, x, "-") - rf(sc, p));
}

static s7_double sub_rf_pc(s7_scheme *sc, s7_pointer **p) 
{
  s7_rf_t rf; 
  s7_double x; 
  s7_pointer y; 
  rf = (s7_rf_t)(**p); (*p)++; x = rf(sc,p); 
  y = (**p); (*p)++; 
  return(x - real_to_double(sc, y, "-"));
}

static s7_double sub_rf_sp(s7_scheme *sc, s7_pointer **p) 
{
  s7_rf_t rf; 
  s7_pointer x; 
  x = slot_value(**p); (*p)++; 
  rf = (s7_rf_t)(**p); (*p)++; 
  return(real_to_double(sc, x, "-") - rf(sc, p));
}

static s7_double sub_rf_ps(s7_scheme *sc, s7_pointer **p) 
{
  s7_rf_t rf; 
  s7_double x; 
  s7_pointer y; 
  rf = (s7_rf_t)(**p); (*p)++; x = rf(sc,p); 
  y = slot_value(**p); (*p)++;
  return(x - real_to_double(sc, y, "-"));
}

static s7_double sub_rf_pp(s7_scheme *sc, s7_pointer **p) 
{
  s7_rf_t rf; 
  s7_double x, y;
  rf = (s7_rf_t)(**p); (*p)++; x = rf(sc,p); 
  rf = (s7_rf_t)(**p); (*p)++; y = rf(sc,p); 
  return(x - y);
}

static s7_rf_t subtract_rf(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer a1, a2, slot1, slot2;
  xf_t *rc;
  if (!is_pair(cdr(expr))) return(NULL);
  
  xf_init(2);
  a1 = cadr(expr);
  if (is_null(cddr(expr)))
    {
      if (is_t_real(a1))
	{
	  xf_store(a1);
	  return(negate_rf_c);
	}
      if (is_symbol(a1))
	{
	  slot1 = s7_slot(sc, a1);
	  if ((!is_slot(slot1)) || (is_unsafe_stepper(slot1)) || (!(is_real(slot_value(slot1))))) return(NULL);
	  xf_store(slot1);
	  return(negate_rf_s);
	}
      if ((is_pair(a1)) &&
	  (s7_arg_to_if(sc, a1)))
	return(negate_rf_p);
      return(NULL);
    }
  
  a2 = caddr(expr);
  if (is_null(cdddr(expr)))
    {
      if (is_t_real(a1))
	{
	  xf_store(a1);
	  if (is_real(a2))
	    {
	      xf_store(a2);
	      return(sub_rf_cc);
	    }
	  if (is_symbol(a2))
	    {
	      slot2 = s7_slot(sc, a2);
	      if ((!slot2) || (!is_real(slot_value(slot2))) || (is_unsafe_stepper(slot2))) return(NULL);
	      xf_store(slot2);
	      return(sub_rf_cs);
	    }
	  if ((is_pair(a2)) &&
	      (s7_arg_to_if(sc, a2)))
	    return(sub_rf_cp);
	  return(NULL);
	}
      if (is_symbol(a1))
	{
	  slot1 = s7_slot(sc, a1);
	  if ((!slot1) || (!is_real(slot_value(slot1))) || (is_unsafe_stepper(slot1))) return(NULL);
	  xf_store(slot1);
	  if (is_t_real(a2))
	    {
	      xf_store(a2);
	      return(sub_rf_sc);
	    }
	  if (is_symbol(a2))
	    {
	      slot2 = s7_slot(sc, a2);
	      if ((!slot2) || (!is_real(slot_value(slot2))) || (is_unsafe_stepper(slot2))) return(NULL);
	      if ((!is_t_real(slot_value(slot1))) && (!is_t_real(slot_value(slot2)))) return(NULL);
	      xf_store(slot2);
	      return(sub_rf_ss);
	    }
	  if ((is_pair(a2)) &&
	      (s7_arg_to_rf(sc, a2)))
	    return(sub_rf_sp);
	  return(NULL);
	}
      if (is_pair(a1) &&
	  (s7_arg_to_rf(sc, a1)))
	{
	  if (is_real(a2))
	    {
	      xf_store(a2);
	      return(sub_rf_pc);
	    }
	  if (is_symbol(a2))
	    {
	      slot2 = s7_slot(sc, a2);
	      if ((!slot2) || (!is_real(slot_value(slot2))) || (is_unsafe_stepper(slot2))) return(NULL);
	      xf_store(slot2);
	      return(sub_rf_ps);
	    }
	  if ((is_pair(a2)) &&
	      (s7_arg_to_rf(sc, a2)))
	    return(sub_rf_pp);
	}
      return(NULL);
    }
  
  {
    s7_rf_t rf, res;
    ptr_int loc;
    
    if (is_real(a1))
      {
	xf_store(a1);
	res = sub_rf_cp;
      }
    else
      {
	if (is_symbol(a1))
	  {
	    slot1 = s7_slot(sc, a1);
	    if ((!slot1) || (!is_t_integer(slot_value(slot1))) || (is_unsafe_stepper(slot1))) return(NULL);
	    xf_store(slot1);
	    res = sub_rf_sp;
	  }
	else 
	  {
	    if ((!is_pair(a1)) || (!s7_arg_to_rf(sc, a1))) return(NULL);
	    res = sub_rf_pp;
	  }
      }
    
    xf_save_loc(loc);
    rf = add_rf(sc, cdr(expr));
    if (rf)
      {
	xf_store_at(loc, (s7_pointer)rf);
	return(res);
      }
  }
  return(NULL);
}

#if WITH_ADD_PF
static s7_pointer c_subtract_pf2(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t pf;
  s7_pointer x, y;
  pf = (s7_pf_t)(**p); (*p)++;
  x = pf(sc, p);
  xf_push(sc, x);
  pf = (s7_pf_t)(**p); (*p)++;
  y = pf(sc, p);
  x = g_subtract_2(sc, set_plist_2(sc, x, y));
  xf_pop(sc);
  return(x);
}

static s7_pf_t subtract_pf(s7_scheme *sc, s7_pointer expr)
{
  int len;
  len = s7_list_length(sc, expr);
  if (len == 3)
    {
      if ((s7_arg_to_pf(sc, cadr(expr))) &&
	  (s7_arg_to_pf(sc, caddr(expr))))
	return(c_subtract_pf2);
    }
  return(NULL);
}
#endif
#endif


/* ---------------------------------------- multiply ---------------------------------------- */

static s7_pointer g_multiply(s7_scheme *sc, s7_pointer args)
{
  #define H_multiply "(* ...) multiplies its arguments"
  #define Q_multiply pcl_n

  s7_pointer x, p;
  s7_int num_a, den_a;
  s7_double rl_a, im_a;

#if (!WITH_GMP)
  if (is_null(args))
    return(small_int(1));
#endif

  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	method_or_bust_with_type(sc, x, sc->multiply_symbol, args, a_number_string, 0);
      return(x);
    }

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);

    MULTIPLY_INTEGERS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_multiply(sc, cons(sc, s7_int_to_big_integer(sc, num_a), p)));
#endif
      x = car(p);
      p = cdr(p);
      switch (type(x))
	{
	case T_INTEGER:
#if WITH_GMP
	  if ((integer(x) > s7_int32_max) ||
	      (integer(x) < s7_int32_min))
	    return(big_multiply(sc, cons(sc, s7_int_to_big_integer(sc, num_a), cons(sc, x, p))));
#endif

#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(num_a, integer(x), &dn))
	      {
		if (is_null(p)) return(make_real(sc, (s7_double)num_a * (s7_double)integer(x)));
		rl_a = (s7_double)num_a * (s7_double)integer(x);
		goto MULTIPLY_REALS;
	      }
	    num_a = dn;
	  }
#else
	  /* perhaps put all the math-safety stuff on the 'safety switch?
	   *    (* 256 17179869184 4194304) -> 0 which is annoying
	   *    (* 134217728 137438953472) -> 0
	   */
	  if ((integer_length(num_a) + integer_length(integer(x))) >= s7_int_bits)
	    {
	      if (is_null(p)) return(make_real(sc, (s7_double)num_a * (s7_double)integer(x)));
	      rl_a = (s7_double)num_a * (s7_double)integer(x);
	      goto MULTIPLY_REALS;
	    }
	  num_a *= integer(x);
#endif
	  if (is_null(p)) return(make_integer(sc, num_a));
	  goto MULTIPLY_INTEGERS;

	case T_RATIO:
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(numerator(x), num_a, &dn))
	      {
		if (is_null(p))
		  return(make_real(sc, (s7_double)num_a * fraction(x)));
		rl_a = (s7_double)num_a * fraction(x);
		goto MULTIPLY_REALS;
	      }
	    num_a = dn;
	  }
#else
	  if ((integer_length(num_a) + integer_length(numerator(x))) >= s7_int_bits)
	    {
	      if (is_null(p))
		return(make_real(sc, (s7_double)num_a * fraction(x)));
	      rl_a = (s7_double)num_a * fraction(x);
	      goto MULTIPLY_REALS;
	    }
	  num_a *= numerator(x);
#endif
	  den_a = denominator(x);
	  if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	  if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto MULTIPLY_INTEGERS;
	  goto MULTIPLY_RATIOS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, num_a * real(x)));
	  rl_a = num_a * real(x);
	  goto MULTIPLY_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, num_a * real_part(x), num_a * imag_part(x)));
	  rl_a = num_a * real_part(x);
	  im_a = num_a * imag_part(x);
	  goto MULTIPLY_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->multiply_symbol, cons(sc, s7_make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
    MULTIPLY_RATIOS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (den_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_multiply(sc, cons(sc, s7_ratio_to_big_ratio(sc, num_a, den_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  /* as in +, this can overflow:
	   *   (* 8 -9223372036854775807 8) -> 64
	   *   (* 3/4 -9223372036854775807 8) -> 6
	   *   (* 8 -9223372036854775808 8) -> 0
	   *   (* -1 9223372036854775806 8) -> 16
	   *   (* -9223372036854775808 8 1e+308) -> 0.0
	   */
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(integer(x), num_a, &dn))
	      {
		if (is_null(p))
		  return(make_real(sc, ((s7_double)integer(x) / (s7_double)den_a) * (s7_double)num_a));
		rl_a = ((s7_double)integer(x) / (s7_double)den_a) * (s7_double)num_a;
		goto MULTIPLY_REALS;
	      }
	    num_a = dn;
	  }
#else
	  if ((integer_length(num_a) + integer_length(integer(x))) >= s7_int_bits)
	    {
	      if (is_null(p))
		return(make_real(sc, ((s7_double)integer(x) / (s7_double)den_a) * (s7_double)num_a));
	      rl_a = ((s7_double)integer(x) / (s7_double)den_a) * (s7_double)num_a;
	      goto MULTIPLY_REALS;
	    }
	  num_a *= integer(x);
#endif
	  if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	  if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto MULTIPLY_INTEGERS;
	  goto MULTIPLY_RATIOS;

	case T_RATIO:
	  {
#if (!WITH_GMP)
	    s7_int d1, n1;
#endif
	    s7_int d2, n2;
	    d2 = denominator(x);
	    n2 = numerator(x);
#if (!WITH_GMP)
	    d1 = den_a;
	    n1 = num_a;
#if HAVE_OVERFLOW_CHECKS
	    if ((multiply_overflow(n1, n2, &num_a)) ||
		(multiply_overflow(d1, d2, &den_a)))
	      {
		if (is_null(p))
		  return(make_real(sc, ((long double)n1 / (long double)d1) * ((long double)n2 / (long double)d2)));
		rl_a = ((long double)n1 / (long double)d1) * ((long double)n2 / (long double)d2);
		goto MULTIPLY_REALS;
	      }
#else
	    if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		(n1 > s7_int32_max) || (n2 > s7_int32_max) ||     /*    (* 1/524288 1/19073486328125) for example */
		(n1 < s7_int32_min) || (n2 < s7_int32_min))
	      {
		if ((integer_length(d1) + integer_length(d2) > s7_int_bits) ||
		    (integer_length(n1) + integer_length(n2) > s7_int_bits))
		  {
		    if (is_null(p))
		      return(make_real(sc, ((long double)n1 / (long double)d1) * ((long double)n2 / (long double)d2)));
		    rl_a = ((long double)n1 / (long double)d1) * ((long double)n2 / (long double)d2);
		    goto MULTIPLY_REALS;
		  }
	      }
	    num_a *= n2;
	    den_a *= d2;
#endif
#else
	    num_a *= n2;
	    den_a *= d2;
#endif
	    if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	    if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	      goto MULTIPLY_INTEGERS;
	    goto MULTIPLY_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, ((long double)num_a / (long double)den_a) * real(x)));
	  rl_a = ((long double)num_a / (long double)den_a) * real(x);
	  goto MULTIPLY_REALS;

	case T_COMPLEX:
	  {
	    s7_double frac;
	    frac = ((long double)num_a / (long double)den_a);
	    if (is_null(p)) return(s7_make_complex(sc, frac * real_part(x), frac * imag_part(x)));
	    rl_a = frac * real_part(x);
	    im_a = frac * imag_part(x);
	    goto MULTIPLY_COMPLEX;
	  }

	default:
	  method_or_bust_with_type(sc, x, sc->multiply_symbol, cons(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_REAL:
      rl_a = real(x);

    MULTIPLY_REALS:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(make_real(sc, rl_a * integer(x)));
	  rl_a *= integer(x);
	  goto MULTIPLY_REALS;

	case T_RATIO:
	  if (is_null(p)) return(make_real(sc, rl_a * fraction(x)));
	  rl_a *= (s7_double)fraction(x);
	  goto MULTIPLY_REALS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, rl_a * real(x)));
	  rl_a *= real(x);
	  goto MULTIPLY_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a * real_part(x), rl_a * imag_part(x)));
	  im_a = rl_a * imag_part(x);
	  rl_a *= real_part(x);
	  goto MULTIPLY_COMPLEX;

	default:
	  method_or_bust_with_type(sc, x, sc->multiply_symbol, cons(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);

    MULTIPLY_COMPLEX:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a * integer(x), im_a * integer(x)));
	  rl_a *= integer(x);
	  im_a *= integer(x);
	  goto MULTIPLY_COMPLEX;

	case T_RATIO:
	  {
	    s7_double frac;
	    frac = fraction(x);
	    if (is_null(p)) return(s7_make_complex(sc, rl_a * frac, im_a * frac));
	    rl_a *= frac;
	    im_a *= frac;
	    goto MULTIPLY_COMPLEX;
	  }

	case T_REAL:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a * real(x), im_a * real(x)));
	  rl_a *= real(x);
	  im_a *= real(x);
	  goto MULTIPLY_COMPLEX;

	case T_COMPLEX:
	  {
	    s7_double r1, r2, i1, i2;
	    r1 = rl_a;
	    i1 = im_a;
	    r2 = real_part(x);
	    i2 = imag_part(x);
	    if (is_null(p))
	      return(s7_make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	    rl_a = r1 * r2 - i1 * i2;
	    im_a = r1 * i2 + r2 * i1;
	    if (im_a == 0.0)
	      goto MULTIPLY_REALS;
	    goto MULTIPLY_COMPLEX;
	  }

	default:
	  method_or_bust_with_type(sc, x, sc->multiply_symbol, cons(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    default:
      method_or_bust_with_type(sc, x, sc->multiply_symbol, args, a_number_string, 1);
    }
}

#if (!WITH_GMP)
static s7_pointer multiply_2, multiply_fs, multiply_sf, multiply_is, multiply_si;

static s7_pointer g_multiply_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  x = car(args);
  y = cadr(args);

  if (type(x) == type(y))
    {
      if (is_t_real(x))
	return(make_real(sc, real(x) * real(y)));
      else
	{
	  switch (type(x))
	    {
#if HAVE_OVERFLOW_CHECKS
	    case T_INTEGER:
	      {
		s7_int n;
		if (multiply_overflow(integer(x), integer(y), &n))
		  return(make_real(sc, ((s7_double)integer(x)) * ((s7_double)integer(y))));
		return(make_integer(sc, n));
	      }
#else
	    case T_INTEGER: return(make_integer(sc, integer(x) * integer(y)));
#endif
	    case T_RATIO:   return(g_multiply(sc, args));
	    case T_REAL:    return(make_real(sc, real(x) * real(y)));
	    case T_COMPLEX:
	      {
		s7_double r1, r2, i1, i2;
		r1 = real_part(x);
		r2 = real_part(y);
		i1 = imag_part(x);
		i2 = imag_part(y);
		return(s7_make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	      }
	    default:
	      if (!is_number(x))
		method_or_bust_with_type(sc, x, sc->multiply_symbol, args, a_number_string, 1);
	      method_or_bust_with_type(sc, y, sc->multiply_symbol, args, a_number_string, 2);
	    }
	}
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_integer(sc, integer(x) * integer(y)));
	case T_RATIO:   return(g_multiply(sc, args));
	case T_REAL:    return(make_real(sc, integer(x) * real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, integer(x) * real_part(y), integer(x) * imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->multiply_symbol, args, a_number_string, 2);
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:    return(g_multiply(sc, args));
	case T_REAL:     return(make_real(sc, fraction(x) * real(y)));
	case T_COMPLEX:
	  {
	    s7_double frac;
	    frac = fraction(x);
	    return(s7_make_complex(sc, frac * real_part(y), frac * imag_part(y)));
	  }
	default:
	  method_or_bust_with_type(sc, y, sc->multiply_symbol, args, a_number_string, 2);
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_real(sc, real(x) * integer(y)));
	case T_RATIO:   return(make_real(sc, real(x) * fraction(y)));
	case T_REAL:    return(make_real(sc, real(x) * real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, real(x) * real_part(y), real(x) * imag_part(y)));
	default:
	  method_or_bust_with_type(sc, y, sc->multiply_symbol, args, a_number_string, 2);
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER: return(s7_make_complex(sc, real_part(x) * integer(y), imag_part(x) * integer(y)));
	case T_RATIO:
	  {
	    s7_double frac;
	    frac = fraction(y);
	    return(s7_make_complex(sc, real_part(x) * frac, imag_part(x) * frac));
	  }
	case T_REAL:    return(s7_make_complex(sc, real_part(x) * real(y), imag_part(x) * real(y)));
	case T_COMPLEX:
	  {
	    s7_double r1, r2, i1, i2;
	    r1 = real_part(x);
	    r2 = real_part(y);
	    i1 = imag_part(x);
	    i2 = imag_part(y);
	    return(s7_make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	  }
	default:
	  method_or_bust_with_type(sc, y, sc->multiply_symbol, args, a_number_string, 2);
	}

    default:
      method_or_bust_with_type(sc, x, sc->multiply_symbol, args, a_number_string, 1);
    }
  return(x);
}

/* all of these mess up if overflows occur
 *  (let () (define (f x) (* x 9223372036854775806)) (f -63)) -> -9223372036854775682, but (* -63 9223372036854775806) -> -5.810724383218509e+20
 *  how to catch this?  (affects * - +)
 */

static s7_pointer g_multiply_si(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int n;

  x = find_symbol_checked(sc, car(args));
  n = integer(cadr(args));

  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER: 
      {
	s7_int val;
	if (multiply_overflow(integer(x), n, &val))
	  return(make_real(sc, (double)integer(x) * (double)n));
	return(make_integer(sc, val));
      }
    case T_RATIO:
      {
	s7_int val;
	if (multiply_overflow(numerator(x), n, &val))
	  return(make_real(sc, fraction(x) * (double)n));
	return(s7_make_ratio(sc, val, denominator(x)));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) * n));
    case T_RATIO:   return(s7_make_ratio(sc, numerator(x) * n, denominator(x)));
#endif
    case T_REAL:    return(make_real(sc, real(x) * n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * n, imag_part(x) * n));
    default:
      method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, x, cadr(args)), a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_multiply_is(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int n;

  x = find_symbol_checked(sc, cadr(args));
  n = integer(car(args));

  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER: 
      {
	s7_int val;
	if (multiply_overflow(integer(x), n, &val))
	  return(make_real(sc, (double)integer(x) * (double)n));
	return(make_integer(sc, val));
      }
    case T_RATIO:
      {
	s7_int val;
	if (multiply_overflow(numerator(x), n, &val))
	  return(make_real(sc, fraction(x) * (double)n));
	return(s7_make_ratio(sc, val, denominator(x)));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) * n));
    case T_RATIO:   return(s7_make_ratio(sc, numerator(x) * n, denominator(x)));
#endif
    case T_REAL:    return(make_real(sc, real(x) * n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * n, imag_part(x) * n));
    default:
      method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, car(args), x), a_number_string, 2);
    }
  return(x);
}

static s7_pointer g_multiply_fs(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double scl;

  scl = real(car(args));
  x = find_symbol_checked(sc, cadr(args));

  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) * scl));
    case T_RATIO:   return(make_real(sc, numerator(x) * scl / denominator(x)));
    case T_REAL:    return(make_real(sc, real(x) * scl));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * scl, imag_part(x) * scl));
    default:
      method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, car(args), x), a_number_string, 1);
    }
  return(x);
}

static s7_pointer g_multiply_sf(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double scl;

  scl = real(cadr(args));
  x = find_symbol_checked(sc, car(args));

  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) * scl));
    case T_RATIO:   return(make_real(sc, numerator(x) * scl / denominator(x)));
    case T_REAL:    return(make_real(sc, real(x) * scl));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * scl, imag_part(x) * scl));
    default:
      method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, x, cadr(args)), a_number_string, 2);
    }
  return(x);
}

static s7_pointer sqr_ss;
static s7_pointer g_sqr_ss(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = find_symbol_checked(sc, car(args));

  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER: 
      {
	s7_int val;
	if (multiply_overflow(integer(x), integer(x), &val))
	  return(make_real(sc, (double)integer(x) * (double)integer(x)));
	return(make_integer(sc, val));
      }
    case T_RATIO:
      {
	s7_int num, den;
	if ((multiply_overflow(numerator(x), numerator(x), &num)) ||
	    (multiply_overflow(denominator(x), denominator(x), &den)))
	  return(make_real(sc, fraction(x) * fraction(x)));
	return(s7_make_ratio(sc, num, den));
      }
#else
    case T_INTEGER: return(s7_make_integer(sc, integer(x) * integer(x)));
    case T_RATIO:   return(s7_make_ratio(sc, numerator(x) * numerator(x), denominator(x) * denominator(x)));
#endif
    case T_REAL:    return(make_real(sc, real(x) * real(x)));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * real_part(x) - imag_part(x) * imag_part(x), 2.0 * real_part(x) * imag_part(x)));
    default:
      method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, x, x), a_number_string, 1);
    }
  return(x);
}

static s7_pointer mul_1ss;
static s7_pointer g_mul_1ss(s7_scheme *sc, s7_pointer args)
{
  /* (* (- 1.0 x) y) */
  s7_pointer x, y;

  x = find_symbol_checked(sc, caddr(car(args)));
  y = find_symbol_checked(sc, cadr(args));

  if ((is_t_real(x)) &&
      (is_t_real(y)))
    return(make_real(sc, real(y) * (1.0 - real(x))));

  if ((is_real(x)) &&
      (is_real(y)))
    {
      s7_double x1;
      x1 = real_to_double(sc, y, "*");
      return(make_real(sc, x1 * (1.0 - real_to_double(sc, x, "*"))));
    }
  else
    {
      s7_double r1, r2, i1, i2;
      if (!is_number(x))
	{
	  s7_pointer func;
	  if ((func = find_method(sc, find_let(sc, x), sc->subtract_symbol)) != sc->undefined)
	    return(g_multiply_2(sc, set_plist_2(sc, s7_apply_function(sc, func, list_2(sc, real_one, x)), y)));
	  return(wrong_type_argument_with_type(sc, sc->subtract_symbol, 2, x, a_number_string));
	}
      if (!is_number(y))
	{
	  s7_pointer func;
	  if ((func = find_method(sc, find_let(sc, y), sc->multiply_symbol)) != sc->undefined)
	    return(s7_apply_function(sc, func, list_2(sc, g_subtract(sc, list_2(sc, real_one, x)), y)));
	  return(wrong_type_argument_with_type(sc, sc->multiply_symbol, 2, y, a_number_string));
	}

      r1 = 1.0 - s7_real_part(x);
      r2 = s7_real_part(y);
      i1 = -s7_imag_part(x);
      i2 = s7_imag_part(y);
      return(s7_make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
    }
}

static s7_pointer multiply_cs_cos;
static s7_pointer g_multiply_cs_cos(s7_scheme *sc, s7_pointer args)
{
  /* ([*] -2.0 r (cos x)) */
  s7_pointer r, x;

  r = find_symbol_checked(sc, cadr(args));
  x = find_symbol_checked(sc, cadr(caddr(args)));

  if ((is_t_real(r)) &&
      (is_t_real(x)))
    return(make_real(sc, real(car(args)) * real(r) * cos(real(x))));

  if ((is_real(r)) &&
      (is_real(x)))
    return(make_real(sc, real(car(args)) * real_to_double(sc, r, "*") * cos(real_to_double(sc, x, "*"))));
  return(g_multiply(sc, set_plist_3(sc, car(args), r, g_cos(sc, set_plist_1(sc, x)))));
}

static s7_pointer mul_s_sin_s, mul_s_cos_s;
static s7_pointer g_mul_s_sin_s(s7_scheme *sc, s7_pointer args)
{
  /* (* s (sin s)) */
  s7_pointer x, y;

  x = find_symbol_checked(sc, car(args));
  y = find_symbol_checked(sc, cadadr(args));

  if ((is_real(x)) && (is_real(y)))
    return(make_real(sc, real_to_double(sc, x, "*") * sin(real_to_double(sc, y, "sin"))));

  return(g_multiply(sc, set_plist_2(sc, x, g_sin(sc, set_plist_1(sc, y)))));
}

static s7_pointer g_mul_s_cos_s(s7_scheme *sc, s7_pointer args)
{
  /* (* s (cos s)) */
  s7_pointer x, y;

  x = find_symbol_checked(sc, car(args));
  y = find_symbol_checked(sc, cadadr(args));

  if ((is_real(x)) && (is_real(y)))
    return(make_real(sc, real_to_double(sc, x, "*") * cos(real_to_double(sc, y, "cos"))));

  return(g_multiply(sc, set_plist_2(sc, x, g_cos(sc, set_plist_1(sc, y)))));
}


static s7_double multiply_rf_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_rf_t r1, r2;
  s7_double x, y;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x * y);
}

static s7_double multiply_rf_rx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1;
  s7_rf_t r1;
  s7_double x;
  c1 = **p; (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * real_to_double(sc, c1, "*"));
}

static s7_double multiply_rf_sx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_rf_t r1;
  s7_double x;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * real_to_double(sc, s1, "*"));
}

static s7_double multiply_rf_ss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s7_double x1;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "*");
  s2 = slot_value(**p); (*p)++;
  return(x1 * real_to_double(sc, s2, "*"));
}

static s7_double multiply_rf_rs(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s7_double x1;
  s1 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  x1 = real_to_double(sc, c1, "*");
  return(x1 * real_to_double(sc, s1, "*"));
}


static s7_double multiply_rf_xxx(s7_scheme *sc, s7_pointer **p)
{
  s7_rf_t r1, r2, r3;
  s7_double x, y, z;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  r3 = (s7_rf_t)(**p); (*p)++;
  z = r3(sc, p);
  return(x * y * z);
}

static s7_double multiply_rf_rxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1;
  s7_rf_t r1, r2;
  s7_double x, y;
  c1 = **p; (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x * y * real_to_double(sc, c1, "*"));
}

static s7_double multiply_rf_sxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_rf_t r1, r2;
  s7_double x, y;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_rf_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x * y * real_to_double(sc, s1, "*"));
}

static s7_double multiply_rf_rsx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s7_rf_t r1;
  s7_double x, x1;
  s1 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  x1 = real_to_double(sc, c1, "*");
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * x1 * real_to_double(sc, s1, "*"));
}

static s7_double multiply_rf_ssx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s7_rf_t r1;
  s7_double x, x1;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "*");
  s2 = slot_value(**p); (*p)++;
  r1 = (s7_rf_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * x1 * real_to_double(sc, s2, "*"));
}

static s7_double multiply_rf_sss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2, s3;
  s7_double x1, x2, x3;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "*");
  s2 = slot_value(**p); (*p)++;
  x2 = real_to_double(sc, s2, "*");
  s3 = slot_value(**p); (*p)++;
  x3 = real_to_double(sc, s3, "*");
  return(x1 * x2 * x3);
}

static s7_double multiply_rf_rss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1, s2;
  s7_double x1, x2, x3;
  s1 = slot_value(**p); (*p)++;
  x1 = real_to_double(sc, s1, "*");
  s2 = slot_value(**p); (*p)++;
  x2 = real_to_double(sc, s2, "*");
  c1 = **p; (*p)++;
  x3 = real_to_double(sc, c1, "*");
  return(x1 * x2 * x3);
}

static s7_rf_t multiply_rf_1(s7_scheme *sc, s7_pointer expr, int len)
{
  if (len == 3)
    return(com_rf_2(sc, expr, multiply_r_ops));
  if (len == 4)
    return(com_rf_3(sc, expr, multiply_r_ops));

  if (len > 4)
    {
      s7_rf_t rf;
      ptr_int loc;
      xf_t *rc;
      int first_len;

      xf_init(2);
      first_len = (int)(len / 2);
      xf_save_loc(loc);
      rf = multiply_rf_1(sc, expr, first_len + 1);
      if (rf)
	{
	  int i;
	  s7_pointer p;
	  xf_store_at(loc, (s7_pointer)rf);
	  xf_save_loc(loc);
	  for (i = 0, p = expr; i < first_len; i++, p = cdr(p));
	  rf = multiply_rf_1(sc, p, len - first_len);
	  if (rf)
	    {
	      xf_store_at(loc, (s7_pointer)rf);
	      return(multiply_rf_xx);
	    }
	  else return(NULL);
	}
      else return(NULL);
    }
  return(NULL);
}

static s7_rf_t multiply_rf(s7_scheme *sc, s7_pointer expr)
{
  return(multiply_rf_1(sc, expr, s7_list_length(sc, expr)));
}


static s7_int multiply_if_xx(s7_scheme *sc, s7_pointer **p)
{
  s7_if_t r1, r2;
  s7_int x, y;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x * y);
}

static s7_int multiply_if_rx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1;
  s7_if_t r1;
  s7_int x;
  c1 = **p; (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * integer(c1));
}

static s7_int multiply_if_sx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_if_t r1;
  s7_int x;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * integer(s1));
}

static s7_int multiply_if_ss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  return(integer(s1) * integer(s2));
}

static s7_int multiply_if_rs(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s1 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  return(integer(c1) * integer(s1));
}


static s7_int multiply_if_xxx(s7_scheme *sc, s7_pointer **p)
{
  s7_if_t r1, r2, r3;
  s7_int x, y, z;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  r3 = (s7_if_t)(**p); (*p)++;
  z = r3(sc, p);
  return(x * y * z);
}

static s7_int multiply_if_rxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1;
  s7_if_t r1, r2;
  s7_int x, y;
  c1 = **p; (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x * y * integer(c1));
}

static s7_int multiply_if_sxx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1;
  s7_if_t r1, r2;
  s7_int x, y;
  s1 = slot_value(**p); (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  r2 = (s7_if_t)(**p); (*p)++;
  y = r2(sc, p);
  return(x * y * integer(s1));
}

static s7_int multiply_if_rsx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1;
  s7_if_t r1;
  s7_int x;
  s1 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * integer(c1) * integer(s1));
}

static s7_int multiply_if_ssx(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2;
  s7_if_t r1;
  s7_int x;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  r1 = (s7_if_t)(**p); (*p)++;
  x = r1(sc, p);
  return(x * integer(s1) * integer(s2));
}

static s7_int multiply_if_sss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer s1, s2, s3;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  s3 = slot_value(**p); (*p)++;
  return(integer(s1) * integer(s2) * integer(s3));
}

static s7_int multiply_if_rss(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer c1, s1, s2;
  s1 = slot_value(**p); (*p)++;
  s2 = slot_value(**p); (*p)++;
  c1 = **p; (*p)++;
  return(integer(c1) * integer(s1) * integer(s2));
}


static s7_if_t multiply_if_1(s7_scheme *sc, s7_pointer expr, int len)
{
  if (len == 3)
    return(com_if_2(sc, expr, multiply_i_ops));
  if (len == 4)
    return(com_if_3(sc, expr, multiply_i_ops));

  if (len > 4)
    {
      s7_if_t xf;
      xf_t *rc;
      ptr_int loc;
      int first_len;

      xf_init(2);
      first_len = (int)(len / 2);
      xf_save_loc(loc);
      xf = multiply_if_1(sc, expr, first_len + 1);
      if (xf)
	{
	  int i;
	  s7_pointer p;
	  xf_store_at(loc, (s7_pointer)xf);
	  xf_save_loc(loc);
	  for (i = 0, p = expr; i < first_len; i++, p = cdr(p));
	  xf = multiply_if_1(sc, p, len - first_len);
	  if (xf)
	    {
	      xf_store_at(loc, (s7_pointer)xf);
	      return(multiply_if_xx);
	    }
	  else return(NULL);
	}
      else return(NULL);
    }
  return(NULL);
}

static s7_if_t multiply_if(s7_scheme *sc, s7_pointer expr)
{
  return(multiply_if_1(sc, expr, s7_list_length(sc, expr)));
}


static void init_multiply_ops(void)
{
  multiply_r_ops = (rf_ops *)calloc(1, sizeof(rf_ops));
  multiply_r_ops->r = rf_c;
  multiply_r_ops->s = rf_s;

  multiply_r_ops->rs = multiply_rf_rs;
  multiply_r_ops->rp = multiply_rf_rx;
  multiply_r_ops->sp = multiply_rf_sx;
  multiply_r_ops->ss = multiply_rf_ss;
  multiply_r_ops->pp = multiply_rf_xx;

  multiply_r_ops->rss = multiply_rf_rss;
  multiply_r_ops->rsp = multiply_rf_rsx;
  multiply_r_ops->rpp = multiply_rf_rxx;
  multiply_r_ops->sss = multiply_rf_sss;
  multiply_r_ops->ssp = multiply_rf_ssx;
  multiply_r_ops->spp = multiply_rf_sxx;
  multiply_r_ops->ppp = multiply_rf_xxx;

  multiply_i_ops = (if_ops *)calloc(1, sizeof(if_ops));
  multiply_i_ops->r = if_c;
  multiply_i_ops->s = if_s;

  multiply_i_ops->rs = multiply_if_rs;
  multiply_i_ops->rp = multiply_if_rx;
  multiply_i_ops->sp = multiply_if_sx;
  multiply_i_ops->ss = multiply_if_ss;
  multiply_i_ops->pp = multiply_if_xx;

  multiply_i_ops->rss = multiply_if_rss;
  multiply_i_ops->rsp = multiply_if_rsx;
  multiply_i_ops->rpp = multiply_if_rxx;
  multiply_i_ops->sss = multiply_if_sss;
  multiply_i_ops->ssp = multiply_if_ssx;
  multiply_i_ops->spp = multiply_if_sxx;
  multiply_i_ops->ppp = multiply_if_xxx;
}

#if WITH_ADD_PF
static s7_pointer c_mul_pf2(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t pf;
  s7_pointer x, y;
  pf = (s7_pf_t)(**p); (*p)++;
  x = pf(sc, p);
  xf_push(sc, x);
  pf = (s7_pf_t)(**p); (*p)++;
  y = pf(sc, p);
  x = g_multiply_2(sc, set_plist_2(sc, x, y));
  xf_pop(sc);
  return(x);
}

static s7_pf_t multiply_pf(s7_scheme *sc, s7_pointer expr)
{
  int len;
  len = s7_list_length(sc, expr);
  if (len == 3)
    {
      if ((s7_arg_to_pf(sc, cadr(expr))) &&
	  (s7_arg_to_pf(sc, caddr(expr))))
	return(c_mul_pf2);
    }
  return(NULL);
}
#endif

#endif /* with-gmp */



/* ---------------------------------------- divide ---------------------------------------- */

static bool is_number_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_number(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_number_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}

static s7_pointer g_divide(s7_scheme *sc, s7_pointer args)
{
  #define H_divide "(/ x1 ...) divides its first argument by the rest, or inverts the first if there is only one argument"
  #define Q_divide pcl_n

  s7_pointer x, p;
  s7_int num_a, den_a;
  s7_double rl_a, im_a;

  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	method_or_bust_with_type(sc, x, sc->divide_symbol, args, a_number_string, 0);
      if (s7_is_zero(x))
	return(division_by_zero_error(sc, sc->divide_symbol, args));
      return(s7_invert(sc, x));
    }

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);
      if (num_a == 0)
	{
	  bool return_nan = false, return_real_zero = false;
	  for (; is_pair(p); p = cdr(p))
	    {
	      s7_pointer n;
	      n = car(p);
	      if (!s7_is_number(n))
		{
		  n = check_values(sc, n, p);
		  if (!s7_is_number(n))
		    return(wrong_type_argument_with_type(sc, sc->divide_symbol, position_of(p, args), n, a_number_string));
		}
	      if (s7_is_zero(n))
		return(division_by_zero_error(sc, sc->divide_symbol, args));
	      if (type(n) > T_RATIO)
		{
		  return_real_zero = true;
		  if (is_NaN(s7_real_part(n)))
		    return_nan = true;
		}
	    }
	  if (return_nan)
	    return(real_NaN);
	  if (return_real_zero)
	    return(real_zero);
	  return(small_int(0));
	}

    DIVIDE_INTEGERS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_divide(sc, cons(sc, s7_int_to_big_integer(sc, num_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == 0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));

	  /* to be consistent, I suppose we should search first for NaNs in the divisor list.
	   *   (* 0 0/0) is NaN, so (/ 1 0 0/0) should equal (/ 1 0/0) = NaN.  But the whole
	   *   thing is ridiculous.
	   */
	  if (is_null(p))
	    return(s7_make_ratio(sc, num_a, integer(x)));

	  den_a = integer(x);
	  if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto DIVIDE_INTEGERS;
	  goto DIVIDE_RATIOS;

	case T_RATIO:
	  den_a = denominator(x);
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(num_a, den_a, &dn))
	      {
		if (is_null(p)) return(make_real(sc, num_a * inverted_fraction(x)));
		rl_a = (s7_double)num_a * inverted_fraction(x);
		goto DIVIDE_REALS;
	      }
	    num_a = dn;
	  }
#else
	  if ((integer_length(num_a) + integer_length(den_a)) > s7_int_bits)
	    {
	      if (is_null(p)) return(make_real(sc, num_a * inverted_fraction(x)));
	      rl_a = (s7_double)num_a * inverted_fraction(x);
	      goto DIVIDE_REALS;
	    }
	  num_a *= den_a;
#endif
	  den_a = numerator(x);
	  if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	  if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto DIVIDE_INTEGERS;
	  goto DIVIDE_RATIOS;

	case T_REAL:
	  rl_a = (s7_double)num_a;
	  if (real(x) == 0.0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
	  if (is_null(p)) return(make_real(sc, rl_a / real(x)));
	  rl_a /= real(x);
	  goto DIVIDE_REALS;

	case T_COMPLEX:
	  {
	    s7_double i2, r2, den;
	    rl_a = (s7_double)num_a;
	    r2 = real_part(x);
	    i2 = imag_part(x);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    /* we could avoid the squaring (see Knuth II p613 16)
	     *    not a big deal: (/ 1.0e308+1.0e308i 2.0e308+2.0e308i) => nan
	     *    (gmp case is ok here)
	     */
	    if (is_null(p))
	      return(s7_make_complex(sc, rl_a * r2 * den, -(rl_a * i2 * den)));
	    im_a = -rl_a * i2 * den;
	    rl_a *= r2 * den;
	    goto DIVIDE_COMPLEX;
	  }

	default:
	  method_or_bust_with_type(sc, x, sc->divide_symbol, cons(sc, s7_make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
    DIVIDE_RATIOS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (den_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_divide(sc, cons(sc, s7_ratio_to_big_ratio(sc, num_a, den_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == 0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(den_a, integer(x), &dn))
	      {
		if (is_null(p)) return(make_real(sc, (long double)num_a / ((long double)den_a * (s7_double)integer(x))));
		rl_a = (long double)num_a / ((long double)den_a * (s7_double)integer(x));
		goto DIVIDE_REALS;
	      }
	    den_a = dn;
	  }
#else
	  if ((integer_length(integer(x)) + integer_length(den_a)) > s7_int_bits)
	    {
	      if (is_null(p)) return(make_real(sc, (long double)num_a / ((long double)den_a * (s7_double)integer(x))));
	      rl_a = (long double)num_a / ((long double)den_a * (s7_double)integer(x));
	      goto DIVIDE_REALS;
	    }
	  den_a *= integer(x);
#endif
	  if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	  if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	    goto DIVIDE_INTEGERS;
	  goto DIVIDE_RATIOS;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = den_a;
	    n1 = num_a;
	    d2 = denominator(x);
	    n2 = numerator(x);
	    if (d1 == d2)
	      {
		if (is_null(p))
		  return(s7_make_ratio(sc, n1, n2));
		den_a = n2;
	      }
	    else
	      {
#if (!WITH_GMP)
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &d1)))
		  {
		    s7_double r1, r2;
		    r1 = ((long double)num_a / (long double)den_a);
		    r2 = inverted_fraction(x);
		    if (is_null(p)) return(make_real(sc, r1 * r2));
		    rl_a = r1 * r2;
		    goto DIVIDE_REALS;
		  }
		num_a = n1;
		den_a = d1;
#else
		if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		    (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
		    (n1 < s7_int32_min) || (n2 < s7_int32_min))
		  {
		    if ((integer_length(d1) + integer_length(n2) > s7_int_bits) ||
			(integer_length(d2) + integer_length(n1) > s7_int_bits))
		      {
			s7_double r1, r2;
			r1 = ((long double)num_a / (long double)den_a);
			r2 = inverted_fraction(x);
			if (is_null(p)) return(make_real(sc, r1 * r2));
			rl_a = r1 * r2;
			goto DIVIDE_REALS;
		      }
		  }
		num_a *= d2;
		den_a *= n2;
#endif
#else
		num_a *= d2;
		den_a *= n2;
#endif
		if (is_null(p))
		  return(s7_make_ratio(sc, num_a, den_a));
	      }
	    if (reduce_fraction(sc, &num_a, &den_a) == T_INTEGER)
	      goto DIVIDE_INTEGERS;
	    goto DIVIDE_RATIOS;
	  }

	case T_REAL:
	  {
	    s7_double r1;
	    if (real(x) == 0.0)
	      return(division_by_zero_error(sc, sc->divide_symbol, args));
	    r1 = ((long double)num_a / (long double)den_a);
	    if (is_null(p)) return(make_real(sc, r1 / real(x)));
	    rl_a = r1 / real(x);
	    goto DIVIDE_REALS;
	  }

	case T_COMPLEX:
	  {
	    s7_double den, i2, r2;
	    rl_a = ((long double)num_a / (long double)den_a);
	    r2 = real_part(x);
	    i2 = imag_part(x);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    if (is_null(p))
	      return(s7_make_complex(sc, rl_a * r2 * den, -rl_a * i2 * den));
	    im_a = -rl_a * i2 * den;
	    rl_a *= r2 * den;
	    goto DIVIDE_COMPLEX;
	  }

	default:
	  method_or_bust_with_type(sc, x, sc->divide_symbol, cons(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_REAL:
      rl_a = real(x);
      if (rl_a == 0)
	{
	  bool return_nan = false;
	  for (; is_pair(p); p = cdr(p))
	    {
	      s7_pointer n;
	      n = car(p);
	      if (!s7_is_number(n))
		{
		  n = check_values(sc, n, p);
		  if (!s7_is_number(n))
		    return(wrong_type_argument_with_type(sc, sc->divide_symbol, position_of(p, args), n, a_number_string));
		}
	      if (s7_is_zero(n))
		return(division_by_zero_error(sc, sc->divide_symbol, args));
	      if ((is_t_real(n)) &&
		  (is_NaN(real(n))))
		return_nan = true;
	    }
	  if (return_nan)
	    return(real_NaN);
	  return(real_zero);
	}

    DIVIDE_REALS:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == 0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
	  if (is_null(p)) return(make_real(sc, rl_a / integer(x)));
	  rl_a /= (s7_double)integer(x);
	  goto DIVIDE_REALS;

	case T_RATIO:
	  if (is_null(p)) return(make_real(sc, rl_a * inverted_fraction(x)));
	  rl_a *= (s7_double)inverted_fraction(x);
	  goto DIVIDE_REALS;

	case T_REAL:
	  if (real(x) == 0.0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
	  if (is_null(p)) return(make_real(sc, rl_a / real(x)));
	  rl_a /= real(x);
	  goto DIVIDE_REALS;

	case T_COMPLEX:
	  {
	    s7_double den, r2, i2;
	    r2 = real_part(x);
	    i2 = imag_part(x);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    if (is_null(p))
	      return(s7_make_complex(sc, rl_a * r2 * den, -rl_a * i2 * den));
	    im_a = -rl_a * i2 * den;
	    rl_a *= r2 * den;
	    goto DIVIDE_COMPLEX;
	  }

	default:
	  method_or_bust_with_type(sc, x, sc->divide_symbol, cons(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);

    DIVIDE_COMPLEX:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  {
	    s7_double r1;
	    if (integer(x) == 0)
	      return(division_by_zero_error(sc, sc->divide_symbol, args));
	    r1 = 1.0 / (s7_double)integer(x);
	    if (is_null(p)) return(s7_make_complex(sc, rl_a * r1, im_a * r1));
	    rl_a *= r1;
	    im_a *= r1;
	    goto DIVIDE_COMPLEX;
	  }

	case T_RATIO:
	  {
	    s7_double frac;
	    frac = inverted_fraction(x);
	    if (is_null(p)) return(s7_make_complex(sc, rl_a * frac, im_a * frac));
	    rl_a *= frac;
	    im_a *= frac;
	    goto DIVIDE_COMPLEX;
	  }

	case T_REAL:
	  {
	    s7_double r1;
	    if (real(x) == 0.0)
	      return(division_by_zero_error(sc, sc->divide_symbol, args));
	    r1 = 1.0 / real(x);
	    if (is_null(p)) return(s7_make_complex(sc, rl_a * r1, im_a * r1));
	    rl_a *= r1;
	    im_a *= r1;
	    goto DIVIDE_COMPLEX;
	  }

	case T_COMPLEX:
	  {
	    s7_double r1, r2, i1, i2, den;
	    r1 = rl_a;
	    i1 = im_a;
	    r2 = real_part(x);
	    i2 = imag_part(x);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    if (is_null(p))
	      return(s7_make_complex(sc, (r1 * r2 + i1 * i2) * den, (r2 * i1 - r1 * i2) * den));
	    rl_a = (r1 * r2 + i1 * i2) * den;
	    im_a = (r2 * i1 - r1 * i2) * den;
	    goto DIVIDE_COMPLEX;
	  }

	default:
	  method_or_bust_with_type(sc, x, sc->divide_symbol, cons(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	}
      break;

    default:
      method_or_bust_with_type(sc, x, sc->divide_symbol, args, a_number_string, 1);
    }
}


#if (!WITH_GMP)
static s7_pointer invert_1;

static s7_pointer g_invert_1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
      if (integer(p) != 0)
	return(s7_make_ratio(sc, 1, integer(p)));      /* a already checked, not 0 */
      return(division_by_zero_error(sc, sc->divide_symbol, args));

    case T_RATIO:
      return(s7_make_ratio(sc, denominator(p), numerator(p)));

    case T_REAL:
      if (real(p) != 0.0)
	return(make_real(sc, 1.0 / real(p)));
      return(division_by_zero_error(sc, sc->divide_symbol, args));

    case T_COMPLEX:
      {
	s7_double r2, i2, den;
	r2 = real_part(p);
	i2 = imag_part(p);
	den = (r2 * r2 + i2 * i2);
	return(s7_make_complex(sc, r2 / den, -i2 / den));
      }

    default:
      method_or_bust_with_type(sc, p, sc->divide_symbol, args, a_number_string, 1);
    }
}


static s7_pointer divide_1r;
static s7_pointer g_divide_1r(s7_scheme *sc, s7_pointer args)
{
  if (s7_is_real(cadr(args)))
    {
      s7_double rl;
      rl = real_to_double(sc, cadr(args), "/");
      if (rl == 0.0)
	return(division_by_zero_error(sc, sc->divide_symbol, args));
      return(make_real(sc, 1.0 / rl));
    }
  return(g_divide(sc, args));
}


static s7_double c_dbl_invert(s7_scheme *sc, s7_double x)
{
  if (x == 0.0) division_by_zero_error(sc, sc->divide_symbol, set_elist_1(sc, real_zero));
  return(1.0 / x);
}

static s7_double c_dbl_divide_2(s7_scheme *sc, s7_double x, s7_double y)
{
  if (y == 0.0) division_by_zero_error(sc, sc->divide_symbol, set_elist_2(sc, make_real(sc, x), real_zero));
  return(x / y);
}

static s7_double c_dbl_divide_3(s7_scheme *sc, s7_double x, s7_double y, s7_double z)
{
  s7_double d;
  d = y * z;
  if (d == 0.0) division_by_zero_error(sc, sc->divide_symbol, set_elist_3(sc, make_real(sc, x), make_real(sc, y), make_real(sc, z)));
  return(x / d);
}

RF_3_TO_RF(divide, c_dbl_invert, c_dbl_divide_2, c_dbl_divide_3)
#endif


/* ---------------------------------------- max/min ---------------------------------------- */

static bool is_real_via_method_1(s7_scheme *sc, s7_pointer p)
{
  s7_pointer f;
  f = find_method(sc, find_let(sc, p), sc->is_real_symbol);
  if (f != sc->undefined)
    return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
  return(false);
}

#define is_real_via_method(sc, p) ((s7_is_real(p)) || ((has_methods(p)) && (is_real_via_method_1(sc, p))))


static s7_pointer g_max(s7_scheme *sc, s7_pointer args)
{
  #define H_max "(max ...) returns the maximum of its arguments"
  #define Q_max pcl_r

  s7_pointer x, y, p;
  s7_int num_a, num_b, den_a, den_b;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    MAX_INTEGERS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);
      /* fprintf(stderr, "%d: %s %s\n", __LINE__, DISPLAY(x), DISPLAY(y)); */

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) < integer(y)) x = y;
	  goto MAX_INTEGERS;

	case T_RATIO:
	  num_a = integer(x);
	  den_a = 1;
	  num_b = numerator(y);
	  den_b = denominator(y);
	  goto RATIO_MAX_RATIO;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (integer(x) < real(y))
	    {
	      x = y;
	      goto MAX_REALS;
	    }
	  goto MAX_INTEGERS;

	default:
	  method_or_bust(sc, y, sc->max_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_RATIO:
    MAX_RATIOS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);
      /* fprintf(stderr, "%d: %s %s\n", __LINE__, DISPLAY(x), DISPLAY(y)); */

      switch (type(y))
	{
	case T_INTEGER:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = integer(y);
	  den_b = 1;
	  goto RATIO_MAX_RATIO;

	case T_RATIO:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = numerator(y);
	  den_b = denominator(y);

	RATIO_MAX_RATIO:
	  /* there are tricky cases here where long ints outrun doubles:
	   *   (max 92233720368547758/9223372036854775807 92233720368547757/9223372036854775807)
	   * which should be 92233720368547758/9223372036854775807) but first the fraction gets reduced
	   * to 13176245766935394/1317624576693539401, so we fall into the double comparison, and
	   * there we should be comparing
	   *    9.999999999999999992410584792601468961145E-3 and
	   *    9.999999999999999883990367544051025548645E-3
	   * but if using doubles we get
	   *    0.010000000000000000208166817117 and
	   *    0.010000000000000000208166817117
	   * that is, we can't distinguish these two fractions once they're coerced to doubles.
	   *
	   * Even long doubles fail in innocuous-looking cases:
	   *     (min 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
	   *     (max 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
	   *
	   * Another consequence: outside gmp, we can't handle cases like
	   *    (max 9223372036854776/9223372036854775807 #i9223372036854775/9223372036854775000)
	   *    (max #i9223372036854776/9223372036854775807 9223372036854775/9223372036854775000)
	   * I guess if the user is using "inexact" numbers (#i...), he accepts their inexactness.
	   */
	  
	  if ((num_a < 0) && (num_b >= 0)) /* x < 0, y >= 0 -> y */
	    x = y;
	  else
	    {
	      if ((num_a < 0) || (num_b >= 0))
		{
		  if (den_a == den_b)
		    {
		      if (num_a < num_b)
			x = y;
		    }
		  else
		    {
		      if (num_a == num_b)
			{
			  if (((num_a >= 0) &&
			       (den_a > den_b)) ||
			      ((num_a < 0) &&
			       (den_a < den_b)))
			    x = y;
			}
		      else
			{
			  s7_int vala, valb;
			  vala = num_a / den_a;
			  valb = num_b / den_b;
			  /* fprintf(stderr, "val: %lld %lld %d %d\n", vala, valb, -1/2, 0); */
			  
			  if (!((vala > valb) ||
				((vala == valb) && (is_t_integer(y)))))
			    {
			      if ((valb > vala) ||
				  ((vala == valb) && (is_t_integer(x))) ||
				  /* sigh -- both are ratios and the int parts are equal */
				  (((long double)(num_a % den_a) / (long double)den_a) <= ((long double)(num_b % den_b) / (long double)den_b)))
				x = y;
			    }
			}
		    }
		}
	    }
	  if (is_t_ratio(x))
	    goto MAX_RATIOS;
	  goto MAX_INTEGERS;

	case T_REAL:
	  /* (max 3/4 nan.0) should probably return NaN */
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }

	  if (fraction(x) < real(y))
	    {
	      x = y;
	      goto MAX_REALS;
	    }
	  goto MAX_RATIOS;

	default:
	  method_or_bust(sc, y, sc->max_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_REAL:
      if (is_NaN(real(x)))
	{
	  for (; is_not_null(p); p = cdr(p))
	    if (!is_real_via_method(sc, car(p)))
	      return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	  return(x);
	}

    MAX_REALS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) < integer(y))
	    {
	      x = y;
	      goto MAX_INTEGERS;
	    }
	  goto MAX_REALS;

	case T_RATIO:
	  if (real(x) < fraction(y))
	    {
	      x = y;
	      goto MAX_RATIOS;
	    }
	  goto MAX_REALS;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (real(x) < real(y)) x = y;
	  goto MAX_REALS;

	default:
	  method_or_bust(sc, y, sc->max_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}

    default:
      method_or_bust(sc, x, sc->max_symbol, cons(sc, x, p), T_REAL, 1);
    }
}

#if (!WITH_GMP)
static s7_pointer max_f2;
static s7_pointer g_max_f2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  x = car(args);
  y = cadr(args);
  if (is_t_real(y))
    return((real(x) >= real(y)) ? x : y);
  if (is_real(y))
    return((real(x) >= real_to_double(sc, y, "max")) ? x : y);
  method_or_bust(sc, y, sc->max_symbol, args, T_REAL, 2);
}
#endif

static s7_pointer g_min(s7_scheme *sc, s7_pointer args)
{
  #define H_min "(min ...) returns the minimum of its arguments"
  #define Q_min pcl_r

  s7_pointer x, y, p;
  s7_int num_a, num_b, den_a, den_b;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    MIN_INTEGERS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) > integer(y)) x = y;
	  goto MIN_INTEGERS;

	case T_RATIO:
	  num_a = integer(x);
	  den_a = 1;
	  num_b = numerator(y);
	  den_b = denominator(y);
	  goto RATIO_MIN_RATIO;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (integer(x) > real(y))
	    {
	      x = y;
	      goto MIN_REALS;
	    }
	  goto MIN_INTEGERS;

	default:
	  method_or_bust(sc, y, sc->min_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_RATIO:
    MIN_RATIOS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = integer(y);
	  den_b = 1;
	  goto RATIO_MIN_RATIO;

	case T_RATIO:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = numerator(y);
	  den_b = denominator(y);

	RATIO_MIN_RATIO:
	  if ((num_a >= 0) && (num_b < 0))
	    x = y;
	  else
	    {
	      if ((num_a >= 0) || (num_b < 0))
		{
		  if (den_a == den_b)
		    {
		      if (num_a > num_b)
			x = y;
		    }
		  else
		    {
		      if (num_a == num_b)
			{
			  if (((num_a >= 0) &&
			       (den_a < den_b)) ||
			      ((num_a < 0) &&
			       (den_a > den_b)))
			    x = y;
			}
		      else
			{
			  s7_int vala, valb;
			  vala = num_a / den_a;
			  valb = num_b / den_b;
			  
			  if (!((vala < valb) ||
				((vala == valb) && (is_t_integer(x)))))
			    {
			      if ((valb < vala) ||
				  ((vala == valb) && (is_t_integer(y))) ||
				  (((long double)(num_a % den_a) / (long double)den_a) >= ((long double)(num_b % den_b) / (long double)den_b)))
				x = y;
			    }
			}
		    }
		}
	    }
	  if (is_t_ratio(x))
	    goto MIN_RATIOS;
	  goto MIN_INTEGERS;

	case T_REAL:
	  /* (min 3/4 nan.0) should probably return NaN */
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (fraction(x) > real(y))
	    {
	      x = y;
	      goto MIN_REALS;
	    }
	  goto MIN_RATIOS;

	default:
	  method_or_bust(sc, y, sc->min_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_REAL:
      if (is_NaN(real(x)))
	{
	  for (; is_not_null(p); p = cdr(p))
	    if (!is_real_via_method(sc, car(p)))
	      return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	  return(x);
	}

    MIN_REALS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) > integer(y))
	    {
	      x = y;
	      goto MIN_INTEGERS;
	    }
	  goto MIN_REALS;

	case T_RATIO:
	  if (real(x) > fraction(y))
	    {
	      x = y;
	      goto MIN_RATIOS;
	    }
	  goto MIN_REALS;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (real(x) > real(y)) x = y;
	  goto MIN_REALS;

	default:
	  method_or_bust(sc, y, sc->min_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}

    default:
      method_or_bust(sc, x, sc->min_symbol, cons(sc, x, p), T_REAL, 1);
    }
}

#if (!WITH_GMP)
static s7_pointer min_f2;
static s7_pointer g_min_f2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  x = car(args);
  y = cadr(args);
  if (is_t_real(y))
    return((real(x) <= real(y)) ? x : y);
  if (is_real(y))
    return((real(x) <= real_to_double(sc, y, "min")) ? x : y);
  method_or_bust(sc, y, sc->min_symbol, args, T_REAL, 2);
}

static s7_int c_max_i1(s7_scheme *sc, s7_int x) {return(x);}
static s7_int c_max_i2(s7_scheme *sc, s7_int x, s7_int y) {return((x >= y) ? x : y);}
static s7_int c_max_i3(s7_scheme *sc, s7_int x, s7_int y, s7_int z) {return(((x >= y) ? ((x >= z) ? x : z) : ((y >= z) ? y : z)));}
IF_3_TO_IF(max, c_max_i1, c_max_i2, c_max_i3)

static s7_int c_min_i1(s7_scheme *sc, s7_int x) {return(x);}
static s7_int c_min_i2(s7_scheme *sc, s7_int x, s7_int y) {return((x <= y) ? x : y);}
static s7_int c_min_i3(s7_scheme *sc, s7_int x, s7_int y, s7_int z) {return(((x <= y) ? ((x <= z) ? x : z) : ((y <= z) ? y : z)));}
IF_3_TO_IF(min, c_min_i1, c_min_i2, c_min_i3)

static s7_double c_max_r1(s7_scheme *sc, s7_double x) {return(x);}
static s7_double c_max_r2(s7_scheme *sc, s7_double x, s7_double y) {return((x >= y) ? x : y);}
static s7_double c_max_r3(s7_scheme *sc, s7_double x, s7_double y, s7_double z) {return(((x >= y) ? ((x >= z) ? x : z) : ((y >= z) ? y : z)));}
RF_3_TO_RF(max, c_max_r1, c_max_r2, c_max_r3)

static s7_double c_min_r1(s7_scheme *sc, s7_double x) {return(x);}
static s7_double c_min_r2(s7_scheme *sc, s7_double x, s7_double y) {return((x <= y) ? x : y);}
static s7_double c_min_r3(s7_scheme *sc, s7_double x, s7_double y, s7_double z) {return(((x <= y) ? ((x <= z) ? x : z) : ((y <= z) ? y : z)));}
RF_3_TO_RF(min, c_min_r1, c_min_r2, c_min_r3)
#endif



/* ---------------------------------------- = > < >= <= ---------------------------------------- */

static s7_pointer g_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_equal "(= z1 ...) returns #t if all its arguments are equal"
  #define Q_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_number_symbol)
  s7_pointer x, p;
  s7_int num_a, den_a;
  s7_double rl_a, im_a;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	      if (num_a != integer(x)) goto NOT_EQUAL;
	      break;

	    case T_RATIO:
	    case T_COMPLEX:
	      goto NOT_EQUAL;

	    case T_REAL:
	      if (num_a != real(x)) goto NOT_EQUAL;
	      break;

	    default:
	      method_or_bust_with_type(sc, x, sc->eq_symbol, cons(sc, make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
      rl_a = 0.0;
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	    case T_COMPLEX:
	      goto NOT_EQUAL;

	    case T_RATIO:
	      if ((num_a != numerator(x)) || (den_a != denominator(x)))	goto NOT_EQUAL; /* hidden cast here */
	      break;

	    case T_REAL:
	      if (rl_a == 0.0)
		rl_a = ((long double)num_a) / ((long double)den_a);
	      if (rl_a != real(x)) goto NOT_EQUAL;
	      break;

	    default:
	      method_or_bust_with_type(sc, x, sc->eq_symbol, cons(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    case T_REAL:
      rl_a = real(x);
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	      if (rl_a != integer(x)) goto NOT_EQUAL;
	      break;

	    case T_RATIO:
	      if (rl_a != (double)fraction(x)) goto NOT_EQUAL;
	      /* the cast to double is needed because rl_a is s7_double and we want (= ratio real) to be the same as (= real ratio):
	       *   (= 1.0 9223372036854775807/9223372036854775806)
	       *   (= 9223372036854775807/9223372036854775806 1.0)
	       */
	      break;

	    case T_REAL:
	      if (rl_a != real(x)) goto NOT_EQUAL;
	      break;

	    case T_COMPLEX:
	      goto NOT_EQUAL;

	    default:
	      method_or_bust_with_type(sc, x, sc->eq_symbol, cons(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	    case T_RATIO:
	    case T_REAL:
	      goto NOT_EQUAL;
	      break;

	    case T_COMPLEX:
	      if ((rl_a != real_part(x)) || (im_a != imag_part(x)))
		goto NOT_EQUAL;
	      break;

	    default:
	      method_or_bust_with_type(sc, x, sc->eq_symbol, cons(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1);
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    default:
      method_or_bust_with_type(sc, x, sc->eq_symbol, args, a_number_string, 1);
    }

 NOT_EQUAL:
  for (; is_pair(p); p = cdr(p))
    if (!is_number_via_method(sc, car(p)))
      return(wrong_type_argument_with_type(sc, sc->eq_symbol, position_of(p, args), car(p), a_number_string));

  return(sc->F);
}


static s7_pointer equal_s_ic, equal_2;
static s7_pointer g_equal_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer val;

  val = find_symbol_checked(sc, car(args));
  y = s7_integer(cadr(args));
  if (is_integer(val))
    return(make_boolean(sc, integer(val) == y));

  switch (type(val))
    {
    case T_INTEGER: return(make_boolean(sc, integer(val) == y));
    case T_RATIO:   return(sc->F);
    case T_REAL:    return(make_boolean(sc, real(val) == y));
    case T_COMPLEX: return(sc->F);
    default:
      method_or_bust_with_type(sc, val, sc->eq_symbol, list_2(sc, val, cadr(args)), a_number_string, 1);
    }
  return(sc->T);
}

static s7_int object_length_to_int(s7_scheme *sc, s7_pointer obj);
#if (!WITH_GMP)
static s7_pointer equal_length_ic;
static s7_pointer g_equal_length_ic(s7_scheme *sc, s7_pointer args)
{
  /* avoid make_integer (and telescope opts), we get here with car=length expr, cadr=int */
  s7_int ilen;
  s7_pointer val;

  val = find_symbol_checked(sc, cadar(args));
  ilen = s7_integer(cadr(args));

  switch (type(val))
    {
    case T_PAIR:         return(make_boolean(sc, s7_list_length(sc, val) == ilen));
    case T_NIL:          return(make_boolean(sc, ilen == 0));
    case T_STRING:       return(make_boolean(sc, string_length(val) == ilen));
    case T_HASH_TABLE:   return(make_boolean(sc, (hash_table_mask(val) + 1) == ilen));
    case T_ITERATOR:     return(make_boolean(sc, iterator_length(val) == ilen));
    case T_C_OBJECT:     return(make_boolean(sc, object_length_to_int(sc, val) == ilen));
    case T_LET:          return(make_boolean(sc, let_length(sc, val) == ilen));
    case T_INT_VECTOR:
    case T_FLOAT_VECTOR:
    case T_VECTOR:       return(make_boolean(sc, vector_length(val) == ilen));
    case T_CLOSURE:
    case T_CLOSURE_STAR: if (has_methods(val)) return(make_boolean(sc, closure_length(sc, val) == ilen));
    default:             return(simple_wrong_type_argument_with_type(sc, sc->length_symbol, val, a_sequence_string));
      /* here we already lost because we checked for the length above */
    }
  return(sc->F);
}
#endif


static s7_pointer c_equal_2_1(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) == integer(y)));
	case T_RATIO:   return(sc->F);
	case T_REAL:    return(make_boolean(sc, integer(x) == real(y)));
	case T_COMPLEX: return(sc->F);
	default:
	  method_or_bust_with_type(sc, y, sc->eq_symbol, list_2(sc, x, y), a_number_string, 2);
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(sc->F);
	case T_RATIO:   return(make_boolean(sc, (numerator(x) == numerator(y)) && (denominator(x) == denominator(y))));
	case T_REAL:    return(make_boolean(sc, fraction(x) == real(y)));            /* this could avoid the divide via numerator == denominator * x */
	case T_COMPLEX: return(sc->F);
	default:
	  method_or_bust_with_type(sc, y, sc->eq_symbol, list_2(sc, x, y), a_number_string, 2);
	}
      break;

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_boolean(sc, real(x) == integer(y)));
	case T_RATIO:   return(make_boolean(sc, real(x) == fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) == real(y)));
	case T_COMPLEX: return(sc->F);
	default:
	  method_or_bust_with_type(sc, y, sc->eq_symbol, list_2(sc, x, y), a_number_string, 2);
	}
      break;

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:
	case T_REAL:
	  return(sc->F);

#if (!MS_WINDOWS)
	case T_COMPLEX:
	  return(make_boolean(sc, (real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y))));
#else
	case T_COMPLEX:
	  if ((real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y))) return(sc->T); else return(sc->F);
#endif
	default:
	  method_or_bust_with_type(sc, y, sc->eq_symbol, list_2(sc, x, y), a_number_string, 2);
	}
      break;

    default:
      method_or_bust_with_type(sc, x, sc->eq_symbol, list_2(sc, x, y), a_number_string, 1);
    }
  return(sc->F);
}


static s7_pointer c_equal_2(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      if (is_integer(x))
	return(make_boolean(sc, integer(x) == integer(y)));
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) == integer(y)));
	case T_RATIO:   return(make_boolean(sc, (numerator(x) == numerator(y)) && (denominator(x) == denominator(y))));
	case T_REAL:    return(make_boolean(sc, real(x) == real(y)));
	case T_COMPLEX: return(make_boolean(sc, (real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y))));
	}
    }
#endif
  return(c_equal_2_1(sc, x, y));
}


static s7_pointer g_equal_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;

  x = car(args);
  y = cadr(args);

#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      if (is_integer(x))
	return(make_boolean(sc, integer(x) == integer(y)));
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) == integer(y)));
	case T_RATIO:   return(make_boolean(sc, (numerator(x) == numerator(y)) && (denominator(x) == denominator(y))));
	case T_REAL:    return(make_boolean(sc, real(x) == real(y)));
	case T_COMPLEX: return(make_boolean(sc, (real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y))));
	}
    }
#endif
  return(c_equal_2_1(sc, x, y));
}

#if (!WITH_GMP)
static s7_pointer equal_i2(s7_scheme *sc, s7_pointer **p)
{
  s7_if_t f;
  s7_int x, y;
  f = (s7_if_t)(**p); (*p)++;	x = f(sc, p);
  f = (s7_if_t)(**p); (*p)++;	y = f(sc, p);
  return(make_boolean(sc, x == y));
}

static s7_pointer equal_i2_ic(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer x, y;
  (*p)++;
  x = slot_value(**p); (*p) += 2;
  y = (**p); (*p)++;
  if (!is_integer(x)) 
    return(c_equal_2_1(sc, x, y));
  return(make_boolean(sc, integer(x) == integer(y)));
}

static s7_pointer equal_i2_ii(s7_scheme *sc, s7_pointer **p)
{
  s7_pointer x, y;
  (*p)++;
  x = slot_value(**p); (*p) += 2;
  y = slot_value(**p); (*p)++;
  if (!is_integer(x)) 
    return(c_equal_2_1(sc, x, y));
  return(make_boolean(sc, integer(x) == integer(y)));
}

static s7_pointer equal_r2(s7_scheme *sc, s7_pointer **p)
{
  s7_rf_t f;
  s7_double x, y;
  f = (s7_rf_t)(**p); (*p)++;	x = f(sc, p);
  f = (s7_rf_t)(**p); (*p)++;	y = f(sc, p);
  return(make_boolean(sc, x == y));
}

static s7_pointer equal_p2(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t f;
  s7_pointer x, y;
  f = (s7_pf_t)(**p); (*p)++;	x = f(sc, p);
  f = (s7_pf_t)(**p); (*p)++;	y = f(sc, p);
  return(c_equal_2(sc, x, y));
}

static s7_pf_t equal_pf(s7_scheme *sc, s7_pointer expr)
{
  if ((is_pair(cdr(expr))) && (is_pair(cddr(expr))) && (is_null(cdddr(expr))))
    {
      ptr_int loc;
      s7_pointer a1, a2;
      a1 = cadr(expr);
      a2 = caddr(expr);
      loc = rc_loc(sc);
      if ((s7_arg_to_if(sc, cadr(expr))) && (s7_arg_to_if(sc, caddr(expr))))
	{
	  if (is_symbol(a1))
	    {
	      if (is_integer(a2)) return(equal_i2_ic);
	      if (is_symbol(a2)) return(equal_i2_ii);
	    }
	  return(equal_i2);
	}
      sc->cur_rf->cur = rc_go(sc, loc);
      if ((s7_arg_to_rf(sc, cadr(expr))) && (s7_arg_to_rf(sc, caddr(expr)))) return(equal_r2);
      sc->cur_rf->cur = rc_go(sc, loc);	
      if ((s7_arg_to_pf(sc, cadr(expr))) && (s7_arg_to_pf(sc, caddr(expr)))) return(equal_p2);
    }
  return(NULL);
}


static s7_pointer g_less(s7_scheme *sc, s7_pointer args)
{
  #define H_less "(< x1 ...) returns #t if its arguments are in increasing order"
  #define Q_less s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x, y, p;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_LESS:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) >= integer(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LESS;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) >= 0) && (numerator(y) < 0)) goto NOT_LESS;  /* (< 1 -1/2), ratio numerator can't be 0 */
	  if ((integer(x) <= 0) && (numerator(y) > 0))                 /* (< 0 1/2) */
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_LESS;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) >= numerator(y)) goto NOT_LESS;
	    }
	  else
	    {
	      if (integer(x) >= fraction(y)) goto NOT_LESS;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LESS;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LESS;
	  if (integer(x) >= real(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LESS;

	default:
	  method_or_bust(sc, y, sc->lt_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_RATIO:
    RATIO_LESS:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) > 0) && (integer(y) <= 0)) goto NOT_LESS;
	  if ((numerator(x) < 0) && (integer(y) >= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_LESS;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) >= (integer(y) * denominator(x))) goto NOT_LESS;
	    }
	  else
	    {
	      if (fraction(x) >= integer(y)) goto NOT_LESS;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LESS;

	case T_RATIO:
	  /* conversion to real and >= is not safe here (see comment under g_greater) */
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 >= n2) goto NOT_LESS;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) >= fraction(y)) goto NOT_LESS;
		  }
		else
		  {
		    if (n1 >= n2) goto NOT_LESS;
		  }
#else
		if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		    (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
		    (n1 < s7_int32_min) || (n2 < s7_int32_min))
		  {
		    int d1bits, d2bits;
		    d1bits = integer_length(d1);
		    d2bits = integer_length(d2);
		    if (((d1bits + d2bits) > s7_int_bits) ||
			((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
			((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
		      {
			if (fraction(x) >= fraction(y)) goto NOT_LESS;

			/* (< 21053343141/6701487259 3587785776203/1142027682075) -> #f because even long doubles aren't enough here
			 * (= 21053343141/6701487259 3587785776203/1142027682075) is #f because it checks the actual ints and
			 * (> 21053343141/6701487259 3587785776203/1142027682075) is #f just like the < case.
			 * similarly
			 * (min 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
			 * (max 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
			 *
			 * if we print the long double results as integers, both are -3958705157555305931
			 *    so there's not a lot I can do in the non-gmp case.
			 */
		      }
		    else
		      {
			if ((n1 * d2) >= (n2 * d1)) goto NOT_LESS;
		      }
		  }
		else
		  {
		    if ((n1 * d2) >=  (n2 * d1)) goto NOT_LESS;
		  }
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LESS;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LESS;
	  if (fraction(x) >= real(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LESS;

	default:
	  method_or_bust(sc, y, sc->lt_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_REAL:
      if (is_NaN(real(x))) goto NOT_LESS;

    REAL_LESS:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) >= integer(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LESS;

	case T_RATIO:
	  if (real(x) >= fraction(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LESS;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LESS;
	  if (real(x) >= real(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LESS;

	default:
	  method_or_bust(sc, y, sc->lt_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}

    default:
      method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1);
    }

 NOT_LESS:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->lt_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);
}


static s7_pointer g_less_or_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_less_or_equal "(<= x1 ...) returns #t if its arguments are in increasing order"
  #define Q_less_or_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x, y, p;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_LEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) > integer(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LEQ;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) >= 0) && (numerator(y) < 0)) goto NOT_LEQ;  /* (< 1 -1/2), ratio numerator can't be 0 */
	  if ((integer(x) <= 0) && (numerator(y) > 0))                 /* (< 0 1/2) */
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_LEQ;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) > numerator(y)) goto NOT_LEQ;
	    }
	  else
	    {
	      if (integer(x) > fraction(y)) goto NOT_LEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LEQ;
	  if (integer(x) > real(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LEQ;

	default:
	  method_or_bust(sc, y, sc->leq_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_RATIO:
    RATIO_LEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) > 0) && (integer(y) <= 0)) goto NOT_LEQ;
	  if ((numerator(x) < 0) && (integer(y) >= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_LEQ;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) > (integer(y) * denominator(x))) goto NOT_LEQ;
	    }
	  else
	    {
	      if (fraction(x) > integer(y)) goto NOT_LEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LEQ;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 > n2) goto NOT_LEQ;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) > fraction(y)) goto NOT_LEQ;
		  }
		else
		  {
		    if (n1 > n2) goto NOT_LEQ;
		  }
#else
		if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		    (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
		    (n1 < s7_int32_min) || (n2 < s7_int32_min))
		  {
		    int d1bits, d2bits;
		    d1bits = integer_length(d1);
		    d2bits = integer_length(d2);
		    if (((d1bits + d2bits) > s7_int_bits) ||
			((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
			((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
		      {
			if (fraction(x) > fraction(y)) goto NOT_LEQ;
		      }
		    else
		      {
			if ((n1 * d2) > (n2 * d1)) goto NOT_LEQ;
		      }
		  }
		else
		  {
		    if ((n1 * d2) >  (n2 * d1)) goto NOT_LEQ;
		  }
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LEQ;
	  if (fraction(x) > real(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LEQ;

	default:
	  method_or_bust(sc, y, sc->leq_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_REAL:
      if (is_NaN(real(x))) goto NOT_LEQ;

    REAL_LEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) > integer(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LEQ;

	case T_RATIO:
	  if (real(x) > fraction(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LEQ;
	  if (real(x) > real(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LEQ;

	default:
	  method_or_bust(sc, y, sc->leq_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}

    default:
      method_or_bust(sc, x, sc->leq_symbol, args, T_REAL, 1);
    }

 NOT_LEQ:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->leq_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);
}


static s7_pointer g_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_greater "(> x1 ...) returns #t if its arguments are in decreasing order"
  #define Q_greater s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x, y, p;
  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_GREATER:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) <= integer(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GREATER;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) <= 0) && (numerator(y) > 0)) goto NOT_GREATER;
	  if ((integer(x) >= 0) && (numerator(y) < 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_GREATER;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) <= numerator(y)) goto NOT_GREATER;
	    }
	  else
	    {
	      if (integer(x) <= fraction(y)) goto NOT_GREATER;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GREATER;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GREATER;
	  if (integer(x) <= real(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GREATER;

	default:
	  method_or_bust(sc, y, sc->gt_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_RATIO:
    RATIO_GREATER:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) < 0) && (integer(y) >= 0)) goto NOT_GREATER;
	  if ((numerator(x) > 0) && (integer(y) <= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_GREATER;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) <= (integer(y) * denominator(x))) goto NOT_GREATER;
	    }
	  else
	    {
	      if (fraction(x) <= integer(y)) goto NOT_GREATER;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GREATER;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 <= n2) goto NOT_GREATER;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) <= fraction(y)) goto NOT_GREATER;
		  }
		else
		  {
		    if (n1 <= n2) goto NOT_GREATER;
		  }
#else
		if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		    (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
		    (n1 < s7_int32_min) || (n2 < s7_int32_min))
		  {
		    int d1bits, d2bits;
		    d1bits = integer_length(d1);
		    d2bits = integer_length(d2);
		    if (((d1bits + d2bits) > s7_int_bits) ||
			((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
			((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
		      {
			if (fraction(x) <= fraction(y)) goto NOT_GREATER;

			/* (< 21053343141/6701487259 3587785776203/1142027682075) -> #f because even long doubles aren't enough here
			 * (= 21053343141/6701487259 3587785776203/1142027682075) is #f because it checks the actual ints and
			 * (> 21053343141/6701487259 3587785776203/1142027682075) is #f just like the < case.
			 * similarly
			 * (min 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
			 * (max 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
			 *
			 * if we print the long double results as integers, both are -3958705157555305931
			 *    so there's not a lot I can do in the non-gmp case.
			 */
		      }
		    else
		      {
			if ((n1 * d2) <= (n2 * d1)) goto NOT_GREATER;
		      }
		  }
		else
		  {
		    if ((n1 * d2) <=  (n2 * d1)) goto NOT_GREATER;
		  }
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GREATER;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GREATER;
	  if (fraction(x) <= real(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GREATER;

	default:
	  method_or_bust(sc, y, sc->gt_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_REAL:
      if (is_NaN(real(x))) goto NOT_GREATER;

    REAL_GREATER:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) <= integer(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GREATER;

	case T_RATIO:
	  if (real(x) <= fraction(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GREATER;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GREATER;
	  if (real(x) <= real(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GREATER;

	default:
	  method_or_bust(sc, y, sc->gt_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}

    default:
      method_or_bust(sc, x, sc->gt_symbol, args, T_REAL, 1);
    }

 NOT_GREATER:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->gt_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);
}


static s7_pointer g_greater_or_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_greater_or_equal "(>= x1 ...) returns #t if its arguments are in decreasing order"
  #define Q_greater_or_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)
  /* (>= 1+i 1+i) is an error which seems unfortunate */
  s7_pointer x, y, p;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_GEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) < integer(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GEQ;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) <= 0) && (numerator(y) > 0)) goto NOT_GEQ;
	  if ((integer(x) >= 0) && (numerator(y) < 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_GEQ;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) < numerator(y)) goto NOT_GEQ;
	    }
	  else
	    {
	      if (integer(x) < fraction(y)) goto NOT_GEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GEQ;
	  if (integer(x) < real(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GEQ;

	default:
	  method_or_bust(sc, y, sc->geq_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_RATIO:
    RATIO_GEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) < 0) && (integer(y) >= 0)) goto NOT_GEQ;
	  if ((numerator(x) > 0) && (integer(y) <= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_GEQ;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) < (integer(y) * denominator(x))) goto NOT_GEQ;
	    }
	  else
	    {
	      if (fraction(x) < integer(y)) goto NOT_GEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GEQ;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 < n2) goto NOT_GEQ;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) < fraction(y)) goto NOT_GEQ;
		  }
		else
		  {
		    if (n1 < n2) goto NOT_GEQ;
		  }
#else
		if ((d1 > s7_int32_max) || (d2 > s7_int32_max) ||     /* before counting bits, check that overflow is possible */
		    (n1 > s7_int32_max) || (n2 > s7_int32_max) ||
		    (n1 < s7_int32_min) || (n2 < s7_int32_min))
		  {
		    int d1bits, d2bits;
		    d1bits = integer_length(d1);
		    d2bits = integer_length(d2);
		    if (((d1bits + d2bits) > s7_int_bits) ||
			((d1bits + integer_length(n2)) > (s7_int_bits - 1)) ||
			((d2bits + integer_length(n1)) > (s7_int_bits - 1)))
		      {
			if (fraction(x) < fraction(y)) goto NOT_GEQ;
		      }
		    else
		      {
			if ((n1 * d2) < (n2 * d1)) goto NOT_GEQ;
		      }
		  }
		else
		  {
		    if ((n1 * d2) <  (n2 * d1)) goto NOT_GEQ;
		  }
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GEQ;
	  if (fraction(x) < real(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GEQ;

	default:
	  method_or_bust(sc, y, sc->geq_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}


    case T_REAL:
      if (is_NaN(real(x))) goto NOT_GEQ;

    REAL_GEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) < integer(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GEQ;

	case T_RATIO:
	  if (real(x) < fraction(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GEQ;
	  if (real(x) < real(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GEQ;

	default:
	  method_or_bust(sc, y, sc->geq_symbol, cons(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1);
	}

    default:
      method_or_bust(sc, x, sc->geq_symbol, args, T_REAL, 1);
    }

 NOT_GEQ:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->geq_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);

}


static s7_pointer less_s_ic, less_s0;
static s7_pointer g_less_s0(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = car(args);
  if (is_integer(x))
    return(make_boolean(sc, integer(x) < 0));
  if (is_real(x))
    return(make_boolean(sc, s7_is_negative(x)));
  method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1);
}

static s7_pointer g_less_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = integer(cadr(args));
  if (is_integer(x))
    return(make_boolean(sc, integer(x) < y));

  switch (type(x))
    {
    case T_INTEGER:
      return(make_boolean(sc, integer(x) < y));

    case T_RATIO:
      if ((y >= 0) && (numerator(x) < 0))
	return(sc->T);
      if ((y <= 0) && (numerator(x) > 0))
	return(sc->F);
      if (denominator(x) < s7_int32_max)
	return(make_boolean(sc, (numerator(x) < (y * denominator(x)))));
      return(make_boolean(sc, fraction(x) < y));

    case T_REAL:
      return(make_boolean(sc, real(x) < y));

    case T_COMPLEX:
    default:
      method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1);
    }
  return(sc->T);
}

static s7_pointer less_length_ic;
static s7_pointer g_less_length_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int ilen;
  s7_pointer val;

  val = find_symbol_checked(sc, cadar(args));
  ilen = s7_integer(cadr(args));

  switch (type(val))
    {
    case T_PAIR:         return(make_boolean(sc, s7_list_length(sc, val) < ilen));
    case T_NIL:          return(make_boolean(sc, ilen > 0));
    case T_STRING:       return(make_boolean(sc, string_length(val) < ilen));
    case T_HASH_TABLE:   return(make_boolean(sc, hash_table_mask(val) < ilen)); /* was <=? -- changed 15-Dec-15 */
    case T_ITERATOR:     return(make_boolean(sc, iterator_length(val) < ilen));
    case T_C_OBJECT:     return(make_boolean(sc, object_length_to_int(sc, val) < ilen));
    case T_LET:          return(make_boolean(sc, let_length(sc, val) < ilen));  /* this works because let_length handles the length method itself! */
    case T_INT_VECTOR:
    case T_FLOAT_VECTOR:
    case T_VECTOR:       return(make_boolean(sc, vector_length(val) < ilen));
    case T_CLOSURE:
    case T_CLOSURE_STAR: if (has_methods(val)) return(make_boolean(sc, closure_length(sc, val) < ilen));
    default:             return(simple_wrong_type_argument_with_type(sc, sc->length_symbol, val, a_sequence_string)); /* no check method here because we checked above */
    }
  return(sc->F);
}

static s7_pointer c_less_2_1(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_boolean(sc, integer(x) < integer(y)));

	case T_RATIO:
	  return(g_less(sc, list_2(sc, x, y)));

	case T_REAL:
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, integer(x) < real(y)));

	default:
	  method_or_bust(sc, y, sc->lt_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    case T_RATIO:
      return(g_less(sc, list_2(sc, x, y)));

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) < integer(y)));

	case T_RATIO:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) < fraction(y)));

	case T_REAL:
	  if (is_NaN(real(x))) return(sc->F);
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, real(x) < real(y)));

	default:
	  method_or_bust(sc, y, sc->lt_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    default:
      method_or_bust(sc, x, sc->lt_symbol, list_2(sc, x, y), T_REAL, 1);
    }
  return(sc->T);
}

static s7_pointer c_less_2(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) < integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) < fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) < real(y)));
	}
    }
#endif
  return(c_less_2_1(sc, x, y));
}

static s7_pointer less_2;
static s7_pointer g_less_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;

  x = car(args);
  y = cadr(args);

#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) < integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) < fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) < real(y)));
	}
    }
#endif
  return(c_less_2_1(sc, x, y));
}

static s7_pointer c_less_i(s7_scheme *sc, s7_int x, s7_int y) {return(make_boolean(sc, x < y));}
static s7_pointer c_less_r(s7_scheme *sc, s7_double x, s7_double y) {return(make_boolean(sc, x < y));}
XF2_TO_PF(less, c_less_i, c_less_r, c_less_2)


static s7_pointer leq_s_ic;
static s7_pointer g_leq_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = s7_integer(cadr(args));

  switch (type(x))
    {
    case T_INTEGER:
      return(make_boolean(sc, integer(x) <= y));

    case T_RATIO:
      if ((y >= 0) && (numerator(x) <= 0))
	return(sc->T);
      if ((y <= 0) && (numerator(x) > 0))
	return(sc->F);
      if (denominator(x) < s7_int32_max)
	return(make_boolean(sc, (numerator(x) <= (y * denominator(x)))));
      return(make_boolean(sc, fraction(x) <= y));

    case T_REAL:
      return(make_boolean(sc, real(x) <= y));

    default:
      method_or_bust(sc, x, sc->leq_symbol, args, T_REAL, 1);
    }
  return(sc->T);
}


static s7_pointer c_leq_2_1(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_boolean(sc, integer(x) <= integer(y)));

	case T_RATIO:
	  return(g_less_or_equal(sc, list_2(sc, x, y)));

	case T_REAL:
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, integer(x) <= real(y)));

	default:
	  method_or_bust(sc, y, sc->leq_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    case T_RATIO:
      return(g_less_or_equal(sc, list_2(sc, x, y)));

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) <= integer(y)));

	case T_RATIO:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) <= fraction(y)));

	case T_REAL:
	  if (is_NaN(real(x))) return(sc->F);
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, real(x) <= real(y)));

	default:
	  method_or_bust(sc, y, sc->leq_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    default:
      method_or_bust(sc, x, sc->leq_symbol, list_2(sc, x, y), T_REAL, 1);
    }
  return(sc->T);
}

static s7_pointer c_leq_2(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) <= integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) <= fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) <= real(y)));
	}
    }
#endif
  return(c_leq_2_1(sc, x, y));
}

static s7_pointer leq_2;
static s7_pointer g_leq_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;

  x = car(args);
  y = cadr(args);

#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) <= integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) <= fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) <= real(y)));
	}
    }
#endif
  return(c_leq_2_1(sc, x, y));
}

static s7_pointer c_leq_i(s7_scheme *sc, s7_int x, s7_int y) {return(make_boolean(sc, x <= y));}
static s7_pointer c_leq_r(s7_scheme *sc, s7_double x, s7_double y) {return(make_boolean(sc, x <= y));}
XF2_TO_PF(leq, c_leq_i, c_leq_r, c_leq_2)


static s7_pointer greater_s_ic, greater_s_fc;
static s7_pointer g_greater_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = integer(cadr(args));
  switch (type(x))
    {
    case T_INTEGER:
      return(make_boolean(sc, integer(x) > y));

    case T_RATIO:
      if (denominator(x) < s7_int32_max)               /* y has already been checked for range */
	return(make_boolean(sc, (numerator(x) > (y * denominator(x)))));
      return(make_boolean(sc, fraction(x) > y));

    case T_REAL:
      return(make_boolean(sc, real(x) > y));

    default:
      method_or_bust_with_type(sc, x, sc->gt_symbol, args, a_number_string, 1);
    }
  return(sc->T);
}

static s7_pointer g_greater_s_fc(s7_scheme *sc, s7_pointer args)
{
  s7_double y;
  s7_pointer x;

  x = car(args);
  y = real(cadr(args));

  if (is_t_real(x))
    return(make_boolean(sc, real(x) > y));

  switch (type(x))
    {
    case T_INTEGER:
      return(make_boolean(sc, integer(x) > y));

    case T_RATIO:
      /* (> 9223372036854775807/9223372036854775806 1.0) */
      if (denominator(x) < s7_int32_max) /* y range check was handled in greater_chooser */
	return(make_boolean(sc, (numerator(x) > (y * denominator(x)))));
      return(make_boolean(sc, fraction(x) > y));

    case T_REAL:
      return(make_boolean(sc, real(x) > y));

    default:
      method_or_bust_with_type(sc, x, sc->gt_symbol, args, a_number_string, 1);
    }
  return(sc->T);
}


static s7_pointer c_greater_2_1(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_boolean(sc, integer(x) > integer(y)));

	case T_RATIO:
	  return(g_greater(sc, list_2(sc, x, y)));

	case T_REAL:
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, integer(x) > real(y)));

	default:
	  method_or_bust(sc, y, sc->gt_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    case T_RATIO:
      return(g_greater(sc, list_2(sc, x, y)));

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) > integer(y)));

	case T_RATIO:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) > fraction(y)));

	case T_REAL:
	  if (is_NaN(real(x))) return(sc->F);
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, real(x) > real(y)));

	default:
	  method_or_bust(sc, y, sc->gt_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    default:
      method_or_bust(sc, x, sc->gt_symbol, list_2(sc, x, y), T_REAL, 1);
    }
  return(sc->T);
}

static s7_pointer c_greater_2(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) > integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) > fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) > real(y)));
	}
    }
#endif
  return(c_greater_2_1(sc, x, y));
}

static s7_pointer greater_2;
static s7_pointer g_greater_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;

  x = car(args);
  y = cadr(args);

#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) > integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) > fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) > real(y)));
	}
    }
#endif
  return(c_greater_2_1(sc, x, y));
}

static s7_pointer c_gt_i(s7_scheme *sc, s7_int x, s7_int y) {return(make_boolean(sc, x > y));}
static s7_pointer c_gt_r(s7_scheme *sc, s7_double x, s7_double y) {return(make_boolean(sc, x > y));}
XF2_TO_PF(gt, c_gt_i, c_gt_r, c_greater_2)


static s7_pointer greater_2_f;
static s7_pointer g_greater_2_f(s7_scheme *sc, s7_pointer args)
{
  return(make_boolean(sc, real(car(args)) > real(cadr(args))));
}


static s7_pointer c_geq_2_1(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_boolean(sc, integer(x) >= integer(y)));

	case T_RATIO:
	  return(g_greater_or_equal(sc, list_2(sc, x, y)));

	case T_REAL:
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, integer(x) >= real(y)));

	default:
	  method_or_bust(sc, y, sc->geq_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    case T_RATIO:
      return(g_greater_or_equal(sc, list_2(sc, x, y)));

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) >= integer(y)));

	case T_RATIO:
	  if (is_NaN(real(x))) return(sc->F);
	  return(make_boolean(sc, real(x) >= fraction(y)));

	case T_REAL:
	  if (is_NaN(real(x))) return(sc->F);
	  if (is_NaN(real(y))) return(sc->F);
	  return(make_boolean(sc, real(x) >= real(y)));

	default:
	  method_or_bust(sc, y, sc->geq_symbol, list_2(sc, x, y), T_REAL, 2);
	}
      break;

    default:
      method_or_bust(sc, x, sc->geq_symbol, list_2(sc, x, y), T_REAL, 1);
    }
  return(sc->T);
}

static s7_pointer c_geq_2(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) >= integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) >= fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) >= real(y)));
	}
    }
#endif
  return(c_geq_2_1(sc, x, y));
}
#endif

static s7_pointer geq_2 = NULL;

#if (!WITH_GMP)
static s7_pointer g_geq_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;

  x = car(args);
  y = cadr(args);

#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      if (is_integer(x))
	return(make_boolean(sc, integer(x) >= integer(y)));
      switch (type(x))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) >= integer(y)));
	case T_RATIO:   return(make_boolean(sc, fraction(x) >= fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) >= real(y)));
	}
    }
#endif
  return(c_geq_2_1(sc, x, y));
}

static s7_pointer c_geq_i(s7_scheme *sc, s7_int x, s7_int y) {return(make_boolean(sc, x >= y));}
static s7_pointer c_geq_r(s7_scheme *sc, s7_double x, s7_double y) {return(make_boolean(sc, x >= y));}
XF2_TO_PF(geq, c_geq_i, c_geq_r, c_geq_2)


static s7_pointer geq_s_fc;
static s7_pointer g_geq_s_fc(s7_scheme *sc, s7_pointer args)
{
  s7_double y;
  s7_pointer x;

  x = car(args);
  y = real(cadr(args));

  if (is_t_real(x))
    return(make_boolean(sc, real(x) >= y));
  return(g_geq_2(sc, args));
}


static s7_pointer geq_length_ic;
static s7_pointer g_geq_length_ic(s7_scheme *sc, s7_pointer args)
{
  return(make_boolean(sc, is_false(sc, g_less_length_ic(sc, args))));
}


static s7_pointer geq_s_ic;
static s7_pointer g_geq_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = s7_integer(cadr(args));

  switch (type(x))
    {
    case T_INTEGER:
      return(make_boolean(sc, integer(x) >= y));

    case T_RATIO:
      if ((y >= 0) && (numerator(x) < 0))
	return(sc->F);
      if ((y <= 0) && (numerator(x) >= 0))
	return(sc->T);
      if ((y < s7_int32_max) &&
	  (y > s7_int32_min) &&
	  (denominator(x) < s7_int32_max))
	return(make_boolean(sc, (numerator(x) >= (y * denominator(x)))));
      return(make_boolean(sc, fraction(x) >= y));

    case T_REAL:
      return(make_boolean(sc, real(x) >= y));

    default:
      method_or_bust(sc, x, sc->geq_symbol, args, T_REAL, 1);
    }
  return(sc->T);
}
#endif
/* end (!WITH_GMP) */


/* ---------------------------------------- real-part imag-part ---------------------------------------- */

s7_double s7_real_part(s7_pointer x)
{
  switch(type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return(fraction(x));
    case T_REAL:        return(real(x));
    case T_COMPLEX:     return(real_part(x));
#if WITH_GMP
    case T_BIG_INTEGER: return((s7_double)big_integer_to_s7_int(big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long double)big_integer_to_s7_int(mpq_numref(big_ratio(x))) / (long double)big_integer_to_s7_int(mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), GMP_RNDN));
    case T_BIG_COMPLEX: return((s7_double)mpfr_get_d(mpc_realref(big_complex(x)), GMP_RNDN));
#endif
    }
  return(0.0);
}


s7_double s7_imag_part(s7_pointer x)
{
  switch (type(x))
    {
    case T_COMPLEX:     return(imag_part(x));
#if WITH_GMP
    case T_BIG_COMPLEX: return((s7_double)mpfr_get_d(mpc_imagref(big_complex(x)), GMP_RNDN));
#endif
    }
  return(0.0);
}

static s7_pointer g_real_part(s7_scheme *sc, s7_pointer args)
{
  #define H_real_part "(real-part num) returns the real part of num"
  #define Q_real_part s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)  

  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
    case T_RATIO:
    case T_REAL:
      return(p);

    case T_COMPLEX:
      return(make_real(sc, real_part(p)));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
    case T_BIG_REAL:
      return(p);

    case T_BIG_COMPLEX:
      {
	s7_pointer x;

	new_cell(sc, x, T_BIG_REAL);
	add_bigreal(sc, x);
	mpfr_init(big_real(x));
	mpc_real(big_real(x), big_complex(p), GMP_RNDN);

	return(x);
      }
#endif

    default:
      method_or_bust_with_type(sc, p, sc->real_part_symbol, args, a_number_string, 0);
    }
}

#if (!WITH_GMP)
static s7_double c_real_part(s7_scheme *sc, s7_pointer x) {return(real(g_real_part(sc, set_plist_1(sc, x))));}
PF_TO_RF(real_part, c_real_part)
#endif


static s7_pointer g_imag_part(s7_scheme *sc, s7_pointer args)
{
  #define H_imag_part "(imag-part num) returns the imaginary part of num"
  #define Q_imag_part s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)  
  s7_pointer p;
  /* currently (imag-part nan.0) -> 0.0 ? it's true but maybe confusing */

  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
    case T_RATIO:
      return(small_int(0));

    case T_REAL:
      return(real_zero);

    case T_COMPLEX:
      return(make_real(sc, imag_part(p)));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(small_int(0));

    case T_BIG_REAL:
      return(real_zero);

    case T_BIG_COMPLEX:
      {
	s7_pointer x;
	new_cell(sc, x, T_BIG_REAL);
	add_bigreal(sc, x);
	mpfr_init(big_real(x));
	mpc_imag(big_real(x), big_complex(p), GMP_RNDN);

	return(x);
      }
#endif

    default:
      method_or_bust_with_type(sc, p, sc->imag_part_symbol, args, a_number_string, 0);
    }
}

#if (!WITH_GMP)
static s7_double c_imag_part(s7_scheme *sc, s7_pointer x) {return(real(g_imag_part(sc, set_plist_1(sc, x))));}
PF_TO_RF(imag_part, c_imag_part)
#endif


/* ---------------------------------------- numerator denominator ---------------------------------------- */

static s7_pointer g_numerator(s7_scheme *sc, s7_pointer args)
{
  #define H_numerator "(numerator rat) returns the numerator of the rational number rat"
  #define Q_numerator s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_rational_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_RATIO:       return(make_integer(sc, numerator(x)));
    case T_INTEGER:     return(x);
#if WITH_GMP
    case T_BIG_INTEGER: return(x);
    case T_BIG_RATIO:   return(mpz_to_big_integer(sc, mpq_numref(big_ratio(x))));
#endif
    default:            method_or_bust_with_type(sc, x, sc->numerator_symbol, args, a_rational_string, 0);
    }
}

#if (!WITH_GMP)
static s7_int c_numerator(s7_scheme *sc, s7_pointer x) {return(s7_numerator(x));}
PF_TO_IF(numerator, c_numerator)
#endif


static s7_pointer g_denominator(s7_scheme *sc, s7_pointer args)
{
  #define H_denominator "(denominator rat) returns the denominator of the rational number rat"
  #define Q_denominator s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_rational_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_RATIO:       return(make_integer(sc, denominator(x)));
    case T_INTEGER:     return(small_int(1));
#if WITH_GMP
    case T_BIG_INTEGER: return(small_int(1));
    case T_BIG_RATIO:   return(mpz_to_big_integer(sc, mpq_denref(big_ratio(x))));
#endif
    default:            method_or_bust_with_type(sc, x, sc->denominator_symbol, args, a_rational_string, 0);
    }
}

#if (!WITH_GMP)
static s7_int c_denominator(s7_scheme *sc, s7_pointer x) {return(s7_denominator(x));}
PF_TO_IF(denominator, c_denominator)
#endif


/* ---------------------------------------- nan? infinite? ---------------------------------------- */

static s7_pointer g_is_nan(s7_scheme *sc, s7_pointer args)
{
  #define H_is_nan "(nan? obj) returns #t if obj is a NaN"
  #define Q_is_nan pl_bn  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(sc->F);

    case T_REAL:
      return(make_boolean(sc, is_NaN(real(x))));

    case T_COMPLEX:
      return(make_boolean(sc, (is_NaN(real_part(x))) || (is_NaN(imag_part(x)))));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(sc->F);

    case T_BIG_REAL:
      return(make_boolean(sc, is_NaN(s7_real_part(x))));

    case T_BIG_COMPLEX:
      return(make_boolean(sc, (is_NaN(s7_real_part(x))) || (is_NaN(s7_imag_part(x)))));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->is_nan_symbol, list_1(sc, x), a_number_string, 0);
    }
}

#if (!WITH_GMP)
static s7_pointer c_is_nan(s7_scheme *sc, s7_double x) {return((is_NaN(x)) ? sc->T : sc->F);}
RF_TO_PF(is_nan, c_is_nan)
#endif


static s7_pointer g_is_infinite(s7_scheme *sc, s7_pointer args)
{
  #define H_is_infinite "(infinite? obj) returns #t if obj is an infinite real"
  #define Q_is_infinite pl_bn  

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(sc->F);

    case T_REAL:
      return(make_boolean(sc, is_inf(real(x))));

    case T_COMPLEX:
      return(make_boolean(sc, (is_inf(real_part(x))) || (is_inf(imag_part(x)))));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(sc->F);

    case T_BIG_REAL:
      return(make_boolean(sc, mpfr_inf_p(big_real(x)) != 0));

    case T_BIG_COMPLEX:
      return(make_boolean(sc,
			  (mpfr_inf_p(big_real(g_real_part(sc, list_1(sc, x)))) != 0) ||
			  (mpfr_inf_p(big_real(g_imag_part(sc, list_1(sc, x)))) != 0)));
#endif

    default:
      method_or_bust_with_type(sc, x, sc->is_infinite_symbol, list_1(sc, x), a_number_string, 0);
    }
}

#if (!WITH_GMP)
static s7_pointer c_is_infinite(s7_scheme *sc, s7_double x) {return((is_inf(x)) ? sc->T : sc->F);}
RF_TO_PF(is_infinite, c_is_infinite)
#endif


/* ---------------------------------------- number? complex? integer? rational? real?  ---------------------------------------- */

static s7_pointer g_is_number(s7_scheme *sc, s7_pointer args)
{
  #define H_is_number "(number? obj) returns #t if obj is a number"
  #define Q_is_number pl_bt  
  check_boolean_method(sc, s7_is_number, sc->is_number_symbol, args); /* we need the s7_* versions here for the GMP case */
}


static s7_pointer g_is_integer(s7_scheme *sc, s7_pointer args)
{
  #define H_is_integer "(integer? obj) returns #t if obj is an integer"
  #define Q_is_integer pl_bt  
  check_boolean_method(sc, s7_is_integer, sc->is_integer_symbol, args);
}


static s7_pointer g_is_real(s7_scheme *sc, s7_pointer args)
{
  #define H_is_real "(real? obj) returns #t if obj is a real number"
  #define Q_is_real pl_bt  
  check_boolean_method(sc, s7_is_real, sc->is_real_symbol, args);
}


static s7_pointer g_is_complex(s7_scheme *sc, s7_pointer args)
{
  #define H_is_complex "(complex? obj) returns #t if obj is a number"
  #define Q_is_complex pl_bt  
  check_boolean_method(sc, s7_is_number, sc->is_complex_symbol, args);
}


static s7_pointer g_is_rational(s7_scheme *sc, s7_pointer args)
{
  #define H_is_rational "(rational? obj) returns #t if obj is a rational number (either an integer or a ratio)"
  #define Q_is_rational pl_bt  
  check_boolean_method(sc, s7_is_rational, sc->is_rational_symbol, args);
  /* in the non-gmp case, (rational? 455702434782048082459/86885567283849955830) -> #f, not #t
   *  and similarly for exact? etc.
   */
}


/* ---------------------------------------- even? odd?---------------------------------------- */

static s7_pointer g_is_even(s7_scheme *sc, s7_pointer args)
{
  #define H_is_even "(even? int) returns #t if the integer int is even"
  #define Q_is_even s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_integer_symbol)

  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:     return(make_boolean(sc, ((integer(p) & 1) == 0)));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, mpz_even_p(big_integer(p))));
#endif
    default:            method_or_bust(sc, p, sc->is_even_symbol, list_1(sc, p), T_INTEGER, 0);
    }
}

#if (!WITH_GMP)
static s7_pointer c_is_even(s7_scheme *sc, s7_int arg) {return(((arg & 1) == 0) ? sc->T : sc->F);}
IF_TO_PF(is_even, c_is_even)
#endif


static s7_pointer g_is_odd(s7_scheme *sc, s7_pointer args)
{
  #define H_is_odd "(odd? int) returns #t if the integer int is odd"
  #define Q_is_odd s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_integer_symbol)

  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:     return(make_boolean(sc, ((integer(p) & 1) == 1)));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, mpz_odd_p(big_integer(p))));
#endif
    default:            method_or_bust(sc, p, sc->is_odd_symbol, list_1(sc, p), T_INTEGER, 0);
    }
}

#if (!WITH_GMP)
static s7_pointer c_is_odd(s7_scheme *sc, s7_int arg) {return(((arg & 1) == 0) ? sc->F : sc->T);}
IF_TO_PF(is_odd, c_is_odd)
#endif


/* ---------------------------------------- zero? ---------------------------------------- */
static s7_pointer c_is_zero(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(make_boolean(sc, integer(x) == 0));
    case T_REAL:        return(make_boolean(sc, real(x) == 0.0));
    case T_RATIO:
    case T_COMPLEX:     return(sc->F);      /* ratios and complex numbers are already collapsed into integers and reals */
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, mpz_cmp_ui(big_integer(x), 0) == 0));
    case T_BIG_REAL:    return(make_boolean(sc, mpfr_zero_p(big_real(x))));
    case T_BIG_RATIO:
    case T_BIG_COMPLEX: return(sc->F);
#endif
    default:
      method_or_bust_with_type(sc, x, sc->is_zero_symbol, list_1(sc, x), a_number_string, 0);
    }
}

static s7_pointer g_is_zero(s7_scheme *sc, s7_pointer args)
{
  #define H_is_zero "(zero? num) returns #t if the number num is zero"
  #define Q_is_zero pl_bn

  return(c_is_zero(sc, car(args)));
}

static s7_pointer c_is_zero_i(s7_scheme *sc, s7_int x) {return(make_boolean(sc, x == 0));}
static s7_pointer c_is_zero_r(s7_scheme *sc, s7_double x) {return(make_boolean(sc, x == 0.0));}
XF_TO_PF(is_zero, c_is_zero_i, c_is_zero_r, c_is_zero)


/* -------------------------------- positive? -------------------------------- */
static s7_pointer c_is_positive(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(make_boolean(sc, integer(x) > 0));
    case T_RATIO:       return(make_boolean(sc, numerator(x) > 0));
    case T_REAL:        return(make_boolean(sc, real(x) > 0.0));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, (mpz_cmp_ui(big_integer(x), 0) > 0)));
    case T_BIG_RATIO:   return(make_boolean(sc, (mpq_cmp_ui(big_ratio(x), 0, 1) > 0)));
    case T_BIG_REAL:    return(make_boolean(sc, (mpfr_cmp_ui(big_real(x), 0) > 0)));
#endif
    default:
      method_or_bust(sc, x, sc->is_positive_symbol, list_1(sc, x), T_REAL, 0);
    }
}

static s7_pointer g_is_positive(s7_scheme *sc, s7_pointer args)
{
  #define H_is_positive "(positive? num) returns #t if the real number num is positive (greater than 0)"
  #define Q_is_positive s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  return(c_is_positive(sc, car(args)));
}

static s7_pointer c_is_positive_i(s7_scheme *sc, s7_int x) {return(make_boolean(sc, x > 0));}
static s7_pointer c_is_positive_r(s7_scheme *sc, s7_double x) {return(make_boolean(sc, x > 0.0));}
XF_TO_PF(is_positive, c_is_positive_i, c_is_positive_r, c_is_positive)


/* -------------------------------- negative? -------------------------------- */
static s7_pointer c_is_negative(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(make_boolean(sc, integer(x) < 0));
    case T_RATIO:       return(make_boolean(sc, numerator(x) < 0));
    case T_REAL:        return(make_boolean(sc, real(x) < 0.0));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, (mpz_cmp_ui(big_integer(x), 0) < 0)));
    case T_BIG_RATIO:   return(make_boolean(sc, (mpq_cmp_ui(big_ratio(x), 0, 1) < 0)));
    case T_BIG_REAL:    return(make_boolean(sc, (mpfr_cmp_ui(big_real(x), 0) < 0)));
#endif
    default:
      method_or_bust(sc, x, sc->is_negative_symbol, list_1(sc, x), T_REAL, 0);
    }
}

static s7_pointer g_is_negative(s7_scheme *sc, s7_pointer args)
{
  #define H_is_negative "(negative? num) returns #t if the real number num is negative (less than 0)"
  #define Q_is_negative s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  return(c_is_negative(sc, car(args)));
}

static s7_pointer c_is_negative_i(s7_scheme *sc, s7_int x) {return(make_boolean(sc, x < 0));}
static s7_pointer c_is_negative_r(s7_scheme *sc, s7_double x) {return(make_boolean(sc, x < 0.0));}
XF_TO_PF(is_negative, c_is_negative_i, c_is_negative_r, c_is_negative)


bool s7_is_ulong(s7_pointer arg)
{
  return(is_integer(arg));
}


unsigned long s7_ulong(s7_pointer p)
{
  return((_NFre(p))->object.number.ul_value);
}


s7_pointer s7_make_ulong(s7_scheme *sc, unsigned long n)
{
  s7_pointer x;
  new_cell(sc, x, T_INTEGER);
  x->object.number.ul_value = n;
  return(x);
}


bool s7_is_ulong_long(s7_pointer arg)
{
  return(is_integer(arg));
}


unsigned long long s7_ulong_long(s7_pointer p)
{
  return((_NFre(p))->object.number.ull_value);
}


s7_pointer s7_make_ulong_long(s7_scheme *sc, unsigned long long n)
{
  s7_pointer x;
  new_cell(sc, x, T_INTEGER);
  x->object.number.ull_value = n;
  return(x);
}


#if (!WITH_PURE_S7)
#if (!WITH_GMP)
/* ---------------------------------------- exact<->inexact exact? inexact? ---------------------------------------- */

static s7_pointer g_exact_to_inexact(s7_scheme *sc, s7_pointer args)
{
  #define H_exact_to_inexact "(exact->inexact num) converts num to an inexact number; (exact->inexact 3/2) = 1.5"
  #define Q_exact_to_inexact pcl_r
  return(exact_to_inexact(sc, car(args)));
}


static s7_pointer g_inexact_to_exact(s7_scheme *sc, s7_pointer args)
{
  #define H_inexact_to_exact "(inexact->exact num) converts num to an exact number; (inexact->exact 1.5) = 3/2"
  #define Q_inexact_to_exact s7_make_signature(sc, 2, sc->is_rational_symbol, sc->is_real_symbol)
  return(inexact_to_exact(sc, car(args), WITH_OVERFLOW_ERROR));
}
#endif
/* (!WITH_GMP) */


static s7_pointer g_is_exact(s7_scheme *sc, s7_pointer args)
{
  #define H_is_exact "(exact? num) returns #t if num is exact (an integer or a ratio)"
  #define Q_is_exact pl_bn

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:       return(sc->T);
    case T_REAL:
    case T_COMPLEX:     return(sc->F);
#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:   return(sc->T);
    case T_BIG_REAL:
    case T_BIG_COMPLEX: return(sc->F);
#endif
    default:
      method_or_bust_with_type(sc, x, sc->is_exact_symbol, args, a_number_string, 0);
    }
}


static s7_pointer g_is_inexact(s7_scheme *sc, s7_pointer args)
{
  #define H_is_inexact "(inexact? num) returns #t if num is inexact (neither an integer nor a ratio)"
  #define Q_is_inexact pl_bn

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:       return(sc->F);
    case T_REAL:
    case T_COMPLEX:     return(sc->T);
#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:   return(sc->F);
    case T_BIG_REAL:
    case T_BIG_COMPLEX: return(sc->T);
#endif
    default:
      method_or_bust_with_type(sc, x, sc->is_inexact_symbol, args, a_number_string, 0);
    }
}


/* ---------------------------------------- integer-length, integer-decode-float ---------------------------------------- */

static s7_pointer g_integer_length(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_length "(integer-length arg) returns the number of bits required to represent the integer 'arg': (ceiling (log (abs arg) 2))"
  #define Q_integer_length pcl_i

  s7_int x;
  s7_pointer p;

  p = car(args);
  if (!s7_is_integer(p))
    method_or_bust(sc, p, sc->integer_length_symbol, args, T_INTEGER, 0);


  x = s7_integer(p);
  if (x < 0)
    return(make_integer(sc, integer_length(-(x + 1))));
  return(make_integer(sc, integer_length(x)));
}

#if (!WITH_GMP)
static s7_int c_integer_length(s7_scheme *sc, s7_int arg) {return((arg < 0) ? integer_length(-(arg + 1)) : integer_length(arg));}
IF_TO_IF(integer_length, c_integer_length)
#endif
#endif /* !pure s7 */


static s7_pointer g_integer_decode_float(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_decode_float "(integer-decode-float x) returns a list containing the significand, exponent, and \
sign of 'x' (1 = positive, -1 = negative).  (integer-decode-float 0.0): (0 0 1)"
  #define Q_integer_decode_float s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_float_symbol)

  /* no matter what s7_double is, integer-decode-float acts as if x is a C double */

  typedef struct decode_float_t {
    union {
      long long int ix;
      double fx;
    } value;
  } decode_float_t;

 decode_float_t num;
  s7_pointer x;
  x = car(args);

  switch (type(x))
    {
    case T_REAL:
      num.value.fx = (double)real(x);
      break;

#if WITH_GMP
    case T_BIG_REAL:
      num.value.fx = (double)real_to_double(sc, x, "integer-decode-float");
      break;
#endif

    default:
      method_or_bust_with_type(sc, x, sc->integer_decode_float_symbol, args, make_string_wrapper(sc, "a non-rational real"), 0);
    }

  if (num.value.fx == 0.0)
    return(list_3(sc, small_int(0), small_int(0), small_int(1)));

  return(list_3(sc,
		make_integer(sc, (s7_int)((num.value.ix & 0xfffffffffffffLL) | 0x10000000000000LL)),
		make_integer(sc, (s7_int)(((num.value.ix & 0x7fffffffffffffffLL) >> 52) - 1023 - 52)),
		make_integer(sc, ((num.value.ix & 0x8000000000000000LL) != 0) ? -1 : 1)));
}


/* -------------------------------- logior -------------------------------- */
static s7_pointer g_logior(s7_scheme *sc, s7_pointer args)
{
  #define H_logior "(logior int ...) returns the bitwise OR of its integer arguments (the bits that are on in any of the arguments)"
  #define Q_logior pcl_i
  s7_int result = 0;
  s7_pointer x;

  for (x = args; is_not_null(x); x = cdr(x))
    {
      if (!s7_is_integer(car(x)))
	method_or_bust(sc, car(x), sc->logior_symbol, cons(sc, make_integer(sc, result), x), T_INTEGER, position_of(x, args));
      result |= s7_integer(car(x));
    }
  return(make_integer(sc, result));
}

#if (!WITH_GMP)
static s7_int c_logior(s7_scheme *sc, s7_int x, s7_int y) {return(x | y);}
IF2_TO_IF(logior, c_logior)
#endif


/* -------------------------------- logxor -------------------------------- */
static s7_pointer g_logxor(s7_scheme *sc, s7_pointer args)
{
  #define H_logxor "(logxor int ...) returns the bitwise XOR of its integer arguments (the bits that are on in an odd number of the arguments)"
  #define Q_logxor pcl_i
  s7_int result = 0;
  s7_pointer x;

  for (x = args; is_not_null(x); x = cdr(x))
    {
      if (!s7_is_integer(car(x)))
	method_or_bust(sc, car(x), sc->logxor_symbol, cons(sc, make_integer(sc, result), x), T_INTEGER, position_of(x, args));
      result ^= s7_integer(car(x));
    }
  return(make_integer(sc, result));
}

#if (!WITH_GMP)
static s7_int c_logxor(s7_scheme *sc, s7_int x, s7_int y) {return(x ^ y);}
IF2_TO_IF(logxor, c_logxor)
#endif


/* -------------------------------- logand -------------------------------- */
static s7_pointer g_logand(s7_scheme *sc, s7_pointer args)
{
  #define H_logand "(logand int ...) returns the bitwise AND of its integer arguments (the bits that are on in every argument)"
  #define Q_logand pcl_i
  s7_int result = -1;
  s7_pointer x;

  for (x = args; is_not_null(x); x = cdr(x))
    {
      if (!s7_is_integer(car(x)))
	method_or_bust(sc, car(x), sc->logand_symbol, cons(sc, make_integer(sc, result), x), T_INTEGER, position_of(x, args));
      result &= s7_integer(car(x));
    }
  return(make_integer(sc, result));
}

#if (!WITH_GMP)
static s7_int c_logand(s7_scheme *sc, s7_int x, s7_int y) {return(x & y);}
IF2_TO_IF(logand, c_logand)
#endif


/* -------------------------------- lognot -------------------------------- */

static s7_pointer g_lognot(s7_scheme *sc, s7_pointer args)
{
  #define H_lognot "(lognot num) returns the bitwise negation (the complement, the bits that are not on) in num: (lognot 0) -> -1"
  #define Q_lognot pcl_i
  if (!s7_is_integer(car(args)))
    method_or_bust(sc, car(args), sc->lognot_symbol, args, T_INTEGER, 0);
  return(make_integer(sc, ~s7_integer(car(args))));
}

#if (!WITH_GMP)
static s7_int c_lognot(s7_scheme *sc, s7_int arg) {return(~arg);}
IF_TO_IF(lognot, c_lognot)
#endif


/* -------------------------------- logbit? -------------------------------- */
/* logbit?  CL is (logbitp index int) using 2^index, but that order strikes me as backwards
 *   at least gmp got the arg order right!
 */

static s7_pointer g_logbit(s7_scheme *sc, s7_pointer args)
{
  #define H_logbit "(logbit? int index) returns #t if the index-th bit is on in int, otherwise #f. The argument \
order here follows gmp, and is the opposite of the CL convention.  (logbit? int bit) is the same as (not (zero? (logand int (ash 1 bit))))."
  #define Q_logbit s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_integer_symbol)

  s7_pointer x, y;
  s7_int index;      /* index in gmp is mp_bitcnt which is an unsigned long int */

  x = car(args);
  y = cadr(args);

  if (!s7_is_integer(x))
    method_or_bust(sc, x, sc->logbit_symbol, args, T_INTEGER, 1);
  if (!s7_is_integer(y))
    method_or_bust(sc, y, sc->logbit_symbol, args, T_INTEGER, 2);

  index = s7_integer(y);
  if (index < 0)
    return(out_of_range(sc, sc->logbit_symbol, small_int(2), y, its_negative_string));

#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, (mpz_tstbit(big_integer(x), index) != 0)));
#endif

  if (index >= s7_int_bits)           /* not sure about the >: (logbit? -1 64) ?? */
    return(make_boolean(sc, integer(x) < 0));

  /* :(zero? (logand most-positive-fixnum (ash 1 63)))
   *   -> ash argument 2, 63, is out of range (shift is too large)
   *   so logbit? has a wider range than the logand/ash shuffle above.
   */

  /* all these long long ints are necessary, else C turns it into an int, gets confused about signs etc */
  return(make_boolean(sc, ((((long long int)(1LL << (long long int)index)) & (long long int)integer(x)) != 0)));
}

/* -------------------------------- ash -------------------------------- */
static s7_int c_ash(s7_scheme *sc, s7_int arg1, s7_int arg2)
{
  if (arg1 == 0) return(0);

  if (arg2 >= s7_int_bits)
    out_of_range(sc, sc->ash_symbol, small_int(2), make_integer(sc, arg2), its_too_large_string);

  if (arg2 < -s7_int_bits)
    {
      if (arg1 < 0)                      /* (ash -31 -100) */
	return(-1);
      return(0);
    }

  /* I can't see any point in protecting this: (ash 9223372036854775807 1) -> -2, but anyone using ash must know something about bits */
  if (arg2 >= 0)
    {
      if (arg1 < 0)
	{
	  unsigned long long int z;
	  z = (unsigned long long int)arg1;
	  return((s7_int)(z << arg2));
	}
      return(arg1 << arg2);
    }
  return(arg1 >> -arg2);
}

static s7_pointer g_ash(s7_scheme *sc, s7_pointer args)
{
  #define H_ash "(ash i1 i2) returns i1 shifted right or left i2 times, i1 << i2, (ash 1 3) -> 8, (ash 8 -3) -> 1"
  #define Q_ash pcl_i
  s7_pointer x, y;

  x = car(args);
  if (!s7_is_integer(x))
    method_or_bust(sc, x, sc->ash_symbol, args, T_INTEGER, 1);

  y = cadr(args);
  if (!s7_is_integer(y))
    method_or_bust(sc, y, sc->ash_symbol, args, T_INTEGER, 2);

  return(make_integer(sc, c_ash(sc, s7_integer(x), s7_integer(y))));
}

#if (!WITH_GMP)
IF2_TO_IF(ash, c_ash)
#endif


/* ---------------------------------------- random ---------------------------------------- */

/* random numbers.  The simple version used in clm.c is probably adequate,
 *   but here I'll use Marsaglia's MWC algorithm.
 *     (random num) -> a number (0..num), if num == 0 return 0, use global default state
 *     (random num state) -> same but use this state
 *     (random-state seed) -> make a new state
 *   to save the current seed, use copy
 *   to save it across load, random-state->list and list->random-state.
 *   random-state? returns #t if its arg is one of these guys
 */

#if (!WITH_GMP)
s7_pointer s7_random_state(s7_scheme *sc, s7_pointer args)
{
  #define H_random_state "(random-state seed (carry plausible-default)) returns a new random number state initialized with 'seed'. \
Pass this as the second argument to 'random' to get a repeatable random number sequence:\n\
    (let ((seed (random-state 1234))) (random 1.0 seed))"
  #define Q_random_state s7_make_circular_signature(sc, 1, 2, sc->is_random_state_symbol, sc->is_integer_symbol)

  s7_pointer r1, r2, p;
  s7_int i1, i2;

  r1 = car(args);
  if (!s7_is_integer(r1))
    method_or_bust(sc, r1, sc->random_state_symbol, args, T_INTEGER, 1);
  i1 = s7_integer(r1);
  if (i1 < 0)
    return(out_of_range(sc, sc->random_state_symbol, small_int(1), r1, its_negative_string));

  if (is_null(cdr(args)))
    {
      new_cell(sc, p, T_RANDOM_STATE);
      random_seed(p) = (unsigned long long int)i1;
      random_carry(p) = 1675393560;                          /* should this be dependent on the seed? */
      return(p);
    }

  r2 = cadr(args);
  if (!s7_is_integer(r2))
    method_or_bust(sc, r2, sc->random_state_symbol, args, T_INTEGER, 2);
  i2 = s7_integer(r2);
  if (i2 < 0)
    return(out_of_range(sc, sc->random_state_symbol, small_int(2), r2, its_negative_string));

  new_cell(sc, p, T_RANDOM_STATE);
  random_seed(p) = (unsigned long long int)i1;
  random_carry(p) = (unsigned long long int)i2;
  return(p);
}

#define g_random_state s7_random_state

static s7_pointer c_random_state(s7_scheme *sc, s7_pointer x) {return(s7_random_state(sc, set_plist_1(sc, x)));}
PF_TO_PF(random_state, c_random_state)
#endif

static s7_pointer rng_copy(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  return(sc->F); /* I can't find a way to copy a gmp random generator */
#else
  s7_pointer obj;
  obj = car(args);
  if (is_random_state(obj))
    {
      s7_pointer new_r;
      new_cell(sc, new_r, T_RANDOM_STATE);
      random_seed(new_r) = random_seed(obj);
      random_carry(new_r) = random_carry(obj);
      return(new_r);
    }
  return(sc->F);
#endif
}


static s7_pointer g_is_random_state(s7_scheme *sc, s7_pointer args)
{
  #define H_is_random_state "(random-state? obj) returns #t if obj is a random-state object (from random-state)."
  #define Q_is_random_state pl_bt
  check_boolean_method(sc, is_random_state, sc->is_random_state_symbol, args);
}

s7_pointer s7_random_state_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_random_state_to_list "(random-state->list r) returns the random state object as a list.\
You can later apply random-state to this list to continue a random number sequence from any point."
  #define Q_random_state_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_random_state_symbol)

#if WITH_GMP
  if ((is_pair(args)) &&
      (!is_random_state(car(args))))
    method_or_bust_with_type(sc, car(args), sc->random_state_to_list_symbol, args, a_random_state_object_string, 1);
  return(sc->nil);
#else
  s7_pointer r;
  if (is_null(args))
    r = sc->default_rng;
  else
    {
      r = car(args);
      if (!is_random_state(r))
	method_or_bust_with_type(sc, r, sc->random_state_to_list_symbol, args, a_random_state_object_string, 1);
    }
  return(list_2(sc, make_integer(sc, random_seed(r)), make_integer(sc, random_carry(r))));
#endif
}

#define g_random_state_to_list s7_random_state_to_list

s7_pointer c_random_state_to_list(s7_scheme *sc, s7_pointer x) {return(s7_random_state_to_list(sc, set_plist_1(sc, x)));}
PF_TO_PF(random_state_to_list, c_random_state_to_list)


void s7_set_default_random_state(s7_scheme *sc, s7_int seed, s7_int carry)
{
#if (!WITH_GMP)
  s7_pointer p;
  new_cell(sc, p, T_RANDOM_STATE);
  random_seed(p) = (unsigned long long int)seed;
  random_carry(p) = (unsigned long long int)carry;
  sc->default_rng = p;
#endif
}

#if (!WITH_GMP)
/* -------------------------------- random -------------------------------- */

static double next_random(s7_pointer r)
{
  /* The multiply-with-carry generator for 32-bit integers:
   *        x(n)=a*x(n-1) + carry mod 2^32
   * Choose multiplier a from this list:
   *   1791398085 1929682203 1683268614 1965537969 1675393560
   *   1967773755 1517746329 1447497129 1655692410 1606218150
   *   2051013963 1075433238 1557985959 1781943330 1893513180
   *   1631296680 2131995753 2083801278 1873196400 1554115554
   * ( or any 'a' for which both a*2^32-1 and a*2^31-1 are prime)
   */
  double result;
  unsigned long long int temp;
  #define RAN_MULT 2131995753UL

  temp = random_seed(r) * RAN_MULT + random_carry(r);
  random_seed(r) = (temp & 0xffffffffUL);
  random_carry(r) = (temp >> 32);
  result = (double)((unsigned int)(random_seed(r))) / 4294967295.5;
  /* divisor was 2^32-1 = 4294967295.0, but somehow this can round up once in a billion tries?
   *   do we want the double just less than 2^32?
   */

  /* (let ((mx 0) (mn 1000)) (do ((i 0 (+ i 1))) ((= i 10000)) (let ((val (random 123))) (set! mx (max mx val)) (set! mn (min mn val)))) (list mn mx)) */
  return(result);
}


s7_double s7_random(s7_scheme *sc, s7_pointer state)
{
  if (!state)
    return(next_random(sc->default_rng));
  return(next_random(state));
}


static s7_pointer g_random(s7_scheme *sc, s7_pointer args)
{
  #define H_random "(random num (state #f)) returns a random number between 0 and num (0 if num=0)."
  #define Q_random s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_number_symbol, sc->is_random_state_symbol)
  s7_pointer r, num;

  num = car(args);
  if (!s7_is_number(num))
    method_or_bust_with_type(sc, num, sc->random_symbol, args, a_number_string, 1);

  if (is_not_null(cdr(args)))
    {
      r = cadr(args);
      if (!is_random_state(r))
	method_or_bust_with_type(sc, r, sc->random_symbol, args, a_random_state_object_string, 2);
    }
  else r = sc->default_rng;

  switch (type(num))
    {
    case T_INTEGER:
      return(make_integer(sc, (s7_int)(integer(num) * next_random(r))));

    case T_RATIO:
      {
	s7_double x, error;
	s7_int numer = 0, denom = 1;
	/* the error here needs to take the size of the fraction into account.  Otherwise, if
	 *    error is (say) 1e-6 and the fraction is (say) 9000000/9223372036854775807,
	 *    c_rationalize will always return 0.  But even that isn't foolproof:
	 *    (random 1/562949953421312) -> 1/376367230475000
	 */
	x = fraction(num);
	if ((x < 1.0e-10) && (x > -1.0e-10))
	  {
	    /* 1e-12 is not tight enough:
	     *    (random 1/2251799813685248) -> 1/2250240579436280
	     *    (random -1/4503599627370496) -> -1/4492889778435526
	     *    (random 1/140737488355328) -> 1/140730223985746
	     *    (random -1/35184372088832) -> -1/35183145492420
	     *    (random -1/70368744177664) -> -1/70366866392738
	     *    (random 1/4398046511104) -> 1/4398033095756
	     *    (random 1/137438953472) -> 1/137438941127
	     */
	    if (numerator(num) < -10)
	      numer = -(s7_int)(floor(-numerator(num) * next_random(r)));
	    else
	      {
		if (numerator(num) > 10)
		  numer = (s7_int)floor(numerator(num) * next_random(r));
		else
		  {
		    long long int diff;
		    numer = numerator(num);
		    diff = s7_int_max - denominator(num);
		    if (diff < 100)
		      return(s7_make_ratio(sc, numer, denominator(num)));
		    denom = denominator(num) + (s7_int)floor(diff * next_random(r));
		    return(s7_make_ratio(sc, numer, denom));
		  }
	      }
	    return(s7_make_ratio(sc, numer, denominator(num)));
	  }
	if ((x < 1e-6) && (x > -1e-6))
	  error = 1e-18;
	else error = 1e-12;
	c_rationalize(x * next_random(r), error, &numer, &denom);
	return(s7_make_ratio(sc, numer, denom));
      }

    case T_REAL:
      return(make_real(sc, real(num) * next_random(r)));

    case T_COMPLEX:
      return(s7_make_complex(sc, real_part(num) * next_random(r), imag_part(num) * next_random(r)));
    }
  return(sc->F);
}

static s7_int c_random_i(s7_scheme *sc, s7_int arg) {return((s7_int)(arg * next_random(sc->default_rng)));} /* not round! */
IF_TO_IF(random, c_random_i)
static s7_double c_random_r(s7_scheme *sc, s7_double arg) {return(arg * next_random(sc->default_rng));}
RF_TO_RF(random, c_random_r)

static s7_pointer random_ic, random_rc, random_i;

static s7_pointer g_random_ic(s7_scheme *sc, s7_pointer args)
{
  return(make_integer(sc, (s7_int)(integer(car(args)) * next_random(sc->default_rng))));
}

static s7_pointer g_random_i(s7_scheme *sc, s7_pointer args)
{
  return(make_integer(sc, (s7_int)(integer(slot_value(global_slot(car(args)))) * next_random(sc->default_rng))));
}

static s7_pointer g_random_rc(s7_scheme *sc, s7_pointer args)
{
  return(make_real(sc, real(car(args)) * next_random(sc->default_rng)));
}

static s7_pointer random_chooser(s7_scheme *sc, s7_pointer f, int args, s7_pointer expr)
{
  if (args == 1)
    {
      s7_pointer arg1;
      arg1 = cadr(expr);
      if (s7_is_integer(arg1))
	{
	  set_optimize_op(expr, HOP_SAFE_C_C);
	  return(random_ic);
	}
      if ((is_real(arg1)) &&
	  (!is_rational(arg1)))
	{
	  set_optimize_op(expr, HOP_SAFE_C_C);
	  return(random_rc);
	}
      if ((is_symbol(arg1)) &&
	  (is_immutable_symbol(arg1)) &&
	  (is_global(arg1)) &&
	  (is_integer(slot_value(global_slot(arg1)))))
	{
	  set_optimize_op(expr, HOP_SAFE_C_C);
	  return(random_i);
	}
    }
  return(f);
}
#endif /* gmp */



/* -------------------------------- characters -------------------------------- */

#define NUM_CHARS 256

static s7_pointer g_char_to_integer(s7_scheme *sc, s7_pointer args)
{
  #define H_char_to_integer "(char->integer c) converts the character c to an integer"
  #define Q_char_to_integer s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_char_symbol)

  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_to_integer_symbol, args, T_CHARACTER, 0);
  return(small_int(character(car(args))));
}

#define int_method_or_bust(Sc, Obj, Method, Args, Type, Num)		\
  {									\
    s7_pointer func;							\
    if ((has_methods(Obj)) && ((func = find_method(Sc, find_let(Sc, Obj), Method)) != Sc->undefined)) \
      return(integer(s7_apply_function(Sc, func, Args)));		\
    if (Num == 0) simple_wrong_type_argument(Sc, Method, Obj, Type);	\
    wrong_type_argument(Sc, Method, Num, Obj, Type);			\
  }

static s7_int c_char_to_integer(s7_scheme *sc, s7_pointer p)
{
  if (!s7_is_character(p))
    int_method_or_bust(sc, p, sc->char_to_integer_symbol, set_plist_1(sc, p), T_CHARACTER, 0);
  return(character(p));
}

PF_TO_IF(char_to_integer, c_char_to_integer)


static s7_pointer c_int_to_char(s7_scheme *sc, s7_int ind)
{
  if ((ind < 0) || (ind >= NUM_CHARS))
    return(simple_wrong_type_argument_with_type(sc, sc->integer_to_char_symbol, make_integer(sc, ind), 
						make_string_wrapper(sc, "an integer that can represent a character")));
  return(s7_make_character(sc, (unsigned char)ind));
}

static s7_pointer c_integer_to_char(s7_scheme *sc, s7_pointer x)
{
  s7_int ind;
  if (!s7_is_integer(x))
    method_or_bust(sc, x, sc->integer_to_char_symbol, list_1(sc, x), T_INTEGER, 0);
  ind = s7_integer(x);
  if ((ind < 0) || (ind >= NUM_CHARS))
    return(simple_wrong_type_argument_with_type(sc, sc->integer_to_char_symbol, x, make_string_wrapper(sc, "an integer that can represent a character")));
  return(s7_make_character(sc, (unsigned char)ind));
}

static s7_pointer g_integer_to_char(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_to_char "(integer->char i) converts the non-negative integer i to a character"
  #define Q_integer_to_char s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_integer_symbol)
  return(c_integer_to_char(sc, car(args)));
}

IF_TO_PF(integer_to_char, c_int_to_char)


static unsigned char uppers[256], lowers[256];
static void init_uppers(void)
{
  int i;
  for (i = 0; i < 256; i++)
    {
      uppers[i] = (unsigned char)toupper(i);
      lowers[i] = (unsigned char)tolower(i);
    }
}

static s7_pointer c_char_upcase(s7_scheme *sc, s7_pointer arg)
{
  if (!s7_is_character(arg))
    method_or_bust(sc, arg, sc->char_upcase_symbol, set_plist_1(sc, arg), T_CHARACTER, 0);
  return(s7_make_character(sc, upper_character(arg)));
}

static s7_pointer g_char_upcase(s7_scheme *sc, s7_pointer args)
{
  #define H_char_upcase "(char-upcase c) converts the character c to upper case"
  #define Q_char_upcase pcl_c
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_upcase_symbol, args, T_CHARACTER, 0);
  return(s7_make_character(sc, upper_character(car(args))));
}

PF_TO_PF(char_upcase, c_char_upcase)


static s7_pointer c_char_downcase(s7_scheme *sc, s7_pointer arg)
{
  if (!s7_is_character(arg))
    method_or_bust(sc, arg, sc->char_downcase_symbol, set_plist_1(sc, arg), T_CHARACTER, 0);
  return(s7_make_character(sc, lowers[(int)character(arg)]));
}

static s7_pointer g_char_downcase(s7_scheme *sc, s7_pointer args)
{
  #define H_char_downcase "(char-downcase c) converts the character c to lower case"
  #define Q_char_downcase pcl_c
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_downcase_symbol, args, T_CHARACTER, 0);
  return(s7_make_character(sc, lowers[character(car(args))]));
}

PF_TO_PF(char_downcase, c_char_downcase)


static s7_pointer c_is_char_alphabetic(s7_scheme *sc, s7_pointer arg)
{
  if (!s7_is_character(arg))
    method_or_bust(sc, arg, sc->is_char_alphabetic_symbol, set_plist_1(sc, arg), T_CHARACTER, 0);
  return(make_boolean(sc, is_char_alphabetic(arg)));
}

static s7_pointer g_is_char_alphabetic(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_alphabetic "(char-alphabetic? c) returns #t if the character c is alphabetic"
  #define Q_is_char_alphabetic pl_bc
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->is_char_alphabetic_symbol, args, T_CHARACTER, 0);
  return(make_boolean(sc, is_char_alphabetic(car(args))));

  /* isalpha returns #t for (integer->char 226) and others in that range */
}

PF_TO_PF(is_char_alphabetic, c_is_char_alphabetic)


static s7_pointer c_is_char_numeric(s7_scheme *sc, s7_pointer arg)
{
  if (!s7_is_character(arg))
    method_or_bust(sc, arg, sc->is_char_numeric_symbol, set_plist_1(sc, arg), T_CHARACTER, 0);
  return(make_boolean(sc, is_char_numeric(arg)));
}

static s7_pointer g_is_char_numeric(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_numeric "(char-numeric? c) returns #t if the character c is a digit"
  #define Q_is_char_numeric pl_bc
  return(c_is_char_numeric(sc, car(args)));
}

PF_TO_PF(is_char_numeric, c_is_char_numeric)


static s7_pointer c_is_char_whitespace(s7_scheme *sc, s7_pointer arg)
{
  if (!s7_is_character(arg))
    method_or_bust(sc, arg, sc->is_char_whitespace_symbol, set_plist_1(sc, arg), T_CHARACTER, 0);
  return(make_boolean(sc, is_char_whitespace(arg)));
}

static s7_pointer g_is_char_whitespace(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_whitespace "(char-whitespace? c) returns #t if the character c is non-printing character"
  #define Q_is_char_whitespace pl_bc
  return(c_is_char_whitespace(sc, car(args)));
}

PF_TO_PF(is_char_whitespace, c_is_char_whitespace)


static s7_pointer c_is_char_upper_case(s7_scheme *sc, s7_pointer arg)
{
  if (!s7_is_character(arg))
    method_or_bust(sc, arg, sc->is_char_upper_case_symbol, set_plist_1(sc, arg), T_CHARACTER, 0);
  return(make_boolean(sc, is_char_uppercase(arg)));
}

static s7_pointer g_is_char_upper_case(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_upper_case "(char-upper-case? c) returns #t if the character c is in upper case"
  #define Q_is_char_upper_case pl_bc
  return(c_is_char_upper_case(sc, car(args)));
}

PF_TO_PF(is_char_upper_case, c_is_char_upper_case)


static s7_pointer c_is_char_lower_case(s7_scheme *sc, s7_pointer arg)
{
  if (!s7_is_character(arg))
    method_or_bust(sc, arg, sc->is_char_lower_case_symbol, set_plist_1(sc, arg), T_CHARACTER, 0);
  return(make_boolean(sc, is_char_lowercase(arg)));
}

static s7_pointer g_is_char_lower_case(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_lower_case "(char-lower-case? c) returns #t if the character c is in lower case"
  #define Q_is_char_lower_case pl_bc
  return(c_is_char_lower_case(sc, car(args)));
}

PF_TO_PF(is_char_lower_case, c_is_char_lower_case)



static s7_pointer g_is_char(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char "(char? obj) returns #t if obj is a character"
  #define Q_is_char pl_bt
  check_boolean_method(sc, s7_is_character, sc->is_char_symbol, args);
}


s7_pointer s7_make_character(s7_scheme *sc, unsigned int c)
{
  return(chars[c]);
}


bool s7_is_character(s7_pointer p)
{
  return(type(p) == T_CHARACTER);
}


char s7_character(s7_pointer p)
{
  return(character(p));
}


static int charcmp(unsigned char c1, unsigned char c2)
{
  return((c1 == c2) ? 0 : (c1 < c2) ? -1 : 1);
  /* not tolower here -- the single case is apparently supposed to be upper case
   *   this matters in a case like (char-ci<? #\_ #\e) which Guile and Gauche say is #f
   *   although (char<? #\_ #\e) is #t -- the spec does not say how to interpret this!
   */
}


static bool is_character_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_character(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_char_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}


static s7_pointer g_char_cmp(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sym, args, T_CHARACTER, 1);

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args));

      if (charcmp(character(y), character(car(x))) != val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_char_cmp_not(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sym, args, T_CHARACTER, 1);

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args));

      if (charcmp(character(y), character(car(x))) == val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_chars_are_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_equal "(char=? char ...) returns #t if all the character arguments are equal"
  #define Q_chars_are_equal pcl_bc

  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_eq_symbol, args, T_CHARACTER, 1);

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	method_or_bust(sc, car(x), sc->char_eq_symbol, cons(sc, y, x), T_CHARACTER, position_of(x, args));

      if (car(x) != y)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sc->char_eq_symbol, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
    }
  return(sc->T);
}


static s7_pointer g_chars_are_less(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_less "(char<? char ...) returns #t if all the character arguments are increasing"
  #define Q_chars_are_less pcl_bc

  return(g_char_cmp(sc, args, -1, sc->char_lt_symbol));
}


static s7_pointer g_chars_are_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_greater "(char>? char ...) returns #t if all the character arguments are decreasing"
  #define Q_chars_are_greater pcl_bc

  return(g_char_cmp(sc, args, 1, sc->char_gt_symbol));
}


static s7_pointer g_chars_are_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_geq "(char>=? char ...) returns #t if all the character arguments are equal or decreasing"
  #define Q_chars_are_geq pcl_bc

  return(g_char_cmp_not(sc, args, -1, sc->char_geq_symbol));
}


static s7_pointer g_chars_are_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_leq "(char<=? char ...) returns #t if all the character arguments are equal or increasing"
  #define Q_chars_are_leq pcl_bc

  return(g_char_cmp_not(sc, args, 1, sc->char_leq_symbol));
}

static s7_pointer simple_char_eq;
static s7_pointer g_simple_char_eq(s7_scheme *sc, s7_pointer args)
{
  return(make_boolean(sc, character(car(args)) == character(cadr(args))));
}

static s7_pointer c_char_eq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_eq_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_eq_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, x == y));
}

static s7_pointer s7_procedure_signature(s7_scheme *sc, s7_pointer x);
static bool char_check(s7_scheme *sc, s7_pointer obj)
{
  if (s7_is_character(obj)) return(true);
  if ((is_pair(obj)) && (is_symbol(car(obj))))
    {
      s7_pointer sig;
      sig = s7_procedure_signature(sc, s7_symbol_value(sc, car(obj)));
      return((sig) && (is_pair(sig)) && (car(sig) == sc->is_char_symbol));
    }
  return(false);
}

PF2_TO_PF_X(char_eq, char_check, c_char_eq, c_is_eq)


static s7_pointer char_equal_s_ic, char_equal_2;
static s7_pointer g_char_equal_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_pointer c;
  c = find_symbol_checked(sc, car(args));
  if (c == cadr(args))
    return(sc->T);
  if (s7_is_character(c))
    return(sc->F);
  method_or_bust(sc, c, sc->char_eq_symbol, list_2(sc, c, cadr(args)), T_CHARACTER, 1);
}

static s7_pointer g_char_equal_2(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_eq_symbol, args, T_CHARACTER, 1);
  if (car(args) == cadr(args))
    return(sc->T);
  if (!s7_is_character(cadr(args)))
    method_or_bust(sc, cadr(args), sc->char_eq_symbol, args, T_CHARACTER, 2);
  return(sc->F);
}


static s7_pointer char_less_s_ic, char_less_2;
static s7_pointer g_char_less_s_ic(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_lt_symbol, args, T_CHARACTER, 1);
  return(make_boolean(sc, character(car(args)) < character(cadr(args))));
}

static s7_pointer g_char_less_2(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_lt_symbol, args, T_CHARACTER, 1);
  if (!s7_is_character(cadr(args)))
    method_or_bust(sc, cadr(args), sc->char_lt_symbol, args, T_CHARACTER, 2);
  return(make_boolean(sc, character(car(args)) < character(cadr(args))));
}

static s7_pointer c_char_lt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_lt_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_lt_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, character(x) < character(y)));
}

static s7_pointer c_clt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  return(make_boolean(sc, character(x) < character(y)));
}

PF2_TO_PF_X(char_lt, char_check, c_char_lt, c_clt)


static s7_pointer char_greater_s_ic, char_greater_2;
static s7_pointer g_char_greater_s_ic(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_gt_symbol, args, T_CHARACTER, 1);
  return(make_boolean(sc, character(car(args)) > character(cadr(args))));
}

static s7_pointer g_char_greater_2(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    method_or_bust(sc, car(args), sc->char_gt_symbol, args, T_CHARACTER, 1);
  if (!s7_is_character(cadr(args)))
    method_or_bust(sc, cadr(args), sc->char_gt_symbol, args, T_CHARACTER, 2);
  return(make_boolean(sc, character(car(args)) > character(cadr(args))));
}

static s7_pointer c_char_gt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_gt_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_gt_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, character(x) > character(y)));
}

static s7_pointer c_cgt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  return(make_boolean(sc, character(x) > character(y)));
}

PF2_TO_PF_X(char_gt, char_check, c_char_gt, c_cgt)


static s7_pointer c_char_geq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_geq_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_geq_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, character(x) >= character(y)));
}

PF2_TO_PF(char_geq, c_char_geq)


static s7_pointer c_char_leq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_leq_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_leq_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, character(x) <= character(y)));
}

PF2_TO_PF(char_leq, c_char_leq)


#if (!WITH_PURE_S7)
static s7_pointer g_char_cmp_ci(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sym, args, T_CHARACTER, 1);

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args));
      if (charcmp(upper_character(y), upper_character(car(x))) != val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_char_cmp_ci_not(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sym, args, T_CHARACTER, 1);
  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args));
      if (charcmp(upper_character(y), upper_character(car(x))) == val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_chars_are_ci_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_equal "(char-ci=? char ...) returns #t if all the character arguments are equal, ignoring case"
  #define Q_chars_are_ci_equal pcl_bc

  return(g_char_cmp_ci(sc, args, 0, sc->char_ci_eq_symbol));
}

static s7_pointer c_char_ci_eq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_ci_eq_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_ci_eq_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, upper_character(x) == upper_character(y)));
}

PF2_TO_PF(char_ci_eq, c_char_ci_eq)


static s7_pointer g_chars_are_ci_less(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_less "(char-ci<? char ...) returns #t if all the character arguments are increasing, ignoring case"
  #define Q_chars_are_ci_less pcl_bc

  return(g_char_cmp_ci(sc, args, -1, sc->char_ci_lt_symbol));
}

static s7_pointer c_char_ci_lt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_ci_lt_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_ci_lt_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, upper_character(x) < upper_character(y)));
}

PF2_TO_PF(char_ci_lt, c_char_ci_lt)


static s7_pointer g_chars_are_ci_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_greater "(char-ci>? char ...) returns #t if all the character arguments are decreasing, ignoring case"
  #define Q_chars_are_ci_greater pcl_bc

  return(g_char_cmp_ci(sc, args, 1, sc->char_ci_gt_symbol));
}

static s7_pointer c_char_ci_gt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_ci_gt_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_ci_gt_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, upper_character(x) > upper_character(y)));
}

PF2_TO_PF(char_ci_gt, c_char_ci_gt)


static s7_pointer g_chars_are_ci_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_geq "(char-ci>=? char ...) returns #t if all the character arguments are equal or decreasing, ignoring case"
  #define Q_chars_are_ci_geq pcl_bc

  return(g_char_cmp_ci_not(sc, args, -1, sc->char_ci_geq_symbol));
}

static s7_pointer c_char_ci_geq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_ci_geq_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_ci_geq_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, upper_character(x) >= upper_character(y)));
}

PF2_TO_PF(char_ci_geq, c_char_ci_geq)


static s7_pointer g_chars_are_ci_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_leq "(char-ci<=? char ...) returns #t if all the character arguments are equal or increasing, ignoring case"
  #define Q_chars_are_ci_leq pcl_bc

  return(g_char_cmp_ci_not(sc, args, 1, sc->char_ci_leq_symbol));
}

static s7_pointer c_char_ci_leq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!s7_is_character(x))
    method_or_bust(sc, x, sc->char_ci_leq_symbol, list_2(sc, x, y), T_CHARACTER, 1);
  if (!s7_is_character(y))
    method_or_bust(sc, y, sc->char_ci_leq_symbol, list_2(sc, x, y), T_CHARACTER, 2);
  return(make_boolean(sc, upper_character(x) <= upper_character(y)));
}

PF2_TO_PF(char_ci_leq, c_char_ci_leq)
#endif /* not pure s7 */


static s7_pointer g_char_position(s7_scheme *sc, s7_pointer args)
{
  #define H_char_position "(char-position char-or-str str (start 0)) returns the position of the first occurrence of char in str, or #f"
  #define Q_char_position s7_make_signature(sc, 4, s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_boolean_symbol), s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_string_symbol), sc->is_string_symbol, sc->is_integer_symbol)

  const char *porig, *pset;
  s7_int start, pos, len; /* not "int" because start arg might be most-negative-fixnum */
  s7_pointer arg1, arg2;

  arg1 = car(args);
  if ((!s7_is_character(arg1)) &&
      (!is_string(arg1)))
    method_or_bust(sc, arg1, sc->char_position_symbol, args, T_CHARACTER, 1);

  arg2 = cadr(args);
  if (!is_string(arg2))
    method_or_bust(sc, arg2, sc->char_position_symbol, args, T_STRING, 2);

  porig = string_value(arg2);
  len = string_length(arg2);

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3;
      arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	{
	  s7_pointer p;
	  if (!s7_is_integer(p = check_values(sc, arg3, cddr(args))))
	    method_or_bust(sc, arg3, sc->char_position_symbol, args, T_INTEGER, 3);
	  arg3 = p;
	}
      start = s7_integer(arg3);
      if (start < 0)
	return(wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, arg3, a_non_negative_integer_string));
    }
  else start = 0;
  if (start >= len) return(sc->F);

  if (s7_is_character(arg1))
    {
      char c;
      const char *p;
      c = character(arg1);
      p = strchr((const char *)(porig + start), (int)c); /* use strchrnul in Gnu C to catch embedded null case */
      if (p)
	return(make_integer(sc, p - porig));
      return(sc->F);
    }

  if (string_length(arg1) == 0)
    return(sc->F);
  pset = string_value(arg1);

  pos = strcspn((const char *)(porig + start), (const char *)pset);
  if ((pos + start) < len)
    return(make_integer(sc, pos + start));

  /* but if the string has an embedded null, we can get erroneous results here --
   *   perhaps check for null at pos+start?  What about a searched-for string that
   *   also has embedded nulls?
   *
   * The embedded nulls are for byte-vector usages, where presumably you're not talking
   *   about chars and strings, so I think I'll ignore these cases.  In unicode, you'd
   *   want to use unicode-aware searchers, so that also is irrelevant.
   */
  return(sc->F);
}

static s7_pointer c_char_position_ppi(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_int z) {return(g_char_position(sc, set_plist_3(sc, x, y, make_integer(sc, z))));}
static s7_pointer c_char_position_pp(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_char_position(sc, set_plist_2(sc, x, y)));}
PPIF_TO_PF(char_position, c_char_position_pp, c_char_position_ppi)


static s7_pointer char_position_csi;
static s7_pointer g_char_position_csi(s7_scheme *sc, s7_pointer args)
{
  /* assume char arg1, no end */
  const char *porig, *p;
  char c;
  s7_pointer arg2;
  s7_int start, len;

  c = character(car(args));
  arg2 = cadr(args);

  if (!is_string(arg2))
    return(g_char_position(sc, args));

  len = string_length(arg2); /* can't return #f here if len==0 -- need start error check first */
  porig = string_value(arg2);

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3;
      arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	return(g_char_position(sc, args));
      start = s7_integer(arg3);
      if (start < 0)
	return(wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, arg3, a_non_negative_integer_string));
      if (start >= len) return(sc->F);
    }
  else start = 0;

  if (len == 0) return(sc->F);
  p = strchr((const char *)(porig + start), (int)c);
  if (p)
    return(make_integer(sc, p - porig));
  return(sc->F);
}


static s7_pointer g_string_position(s7_scheme *sc, s7_pointer args)
{
  #define H_string_position "(string-position str1 str2 (start 0)) returns the starting position of str1 in str2 or #f"
  #define Q_string_position s7_make_signature(sc, 4, s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_boolean_symbol), sc->is_string_symbol, sc->is_string_symbol, sc->is_integer_symbol)
  const char *s1, *s2, *p2;
  s7_int start = 0;
  s7_pointer s1p, s2p;

  s1p = car(args);
  if (!is_string(s1p))
    method_or_bust(sc, s1p, sc->string_position_symbol, args, T_STRING, 1);

  s2p = cadr(args);
  if (!is_string(s2p))
    method_or_bust(sc, s2p, sc->string_position_symbol, args, T_STRING, 2);

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3;
      arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	{
	  s7_pointer p;
	  if (!s7_is_integer(p = check_values(sc, arg3, cddr(args))))
	    method_or_bust(sc, arg3, sc->string_position_symbol, args, T_INTEGER, 3);
	  arg3 = p;
	}
      start = s7_integer(arg3);
      if (start < 0)
	return(wrong_type_argument_with_type(sc, sc->string_position_symbol, 3, arg3, a_non_negative_integer_string));
    }

  if (string_length(s1p) == 0)
    return(sc->F);
  s1 = string_value(s1p);
  s2 = string_value(s2p);
  if (start >= string_length(s2p))
    return(sc->F);

  p2 = strstr((const char *)(s2 + start), s1);
  if (!p2) return(sc->F);
  return(make_integer(sc, p2 - s2));
}

static s7_pointer c_string_position_ppi(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_int z) {return(g_string_position(sc, set_plist_3(sc, x, y, make_integer(sc, z))));}
static s7_pointer c_string_position_pp(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_string_position(sc, set_plist_2(sc, x, y)));}
PPIF_TO_PF(string_position, c_string_position_pp, c_string_position_ppi)



/* -------------------------------- strings -------------------------------- */

s7_pointer s7_make_string_with_length(s7_scheme *sc, const char *str, int len)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING | T_SAFE_PROCEDURE);
  string_value(x) = (char *)malloc((len + 1) * sizeof(char));
  if (len != 0)                                             /* memcpy can segfault if string_value(x) is NULL */
    memcpy((void *)string_value(x), (void *)str, len);
  string_value(x)[len] = 0;
  string_length(x) = len;
  string_hash(x) = 0;
  string_needs_free(x) = true;
  Add_String(x);
  return(x);
}


static s7_pointer make_string_uncopied_with_length(s7_scheme *sc, char *str, int len)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING | T_SAFE_PROCEDURE);
  string_value(x) = str;
  string_length(x) = len;
  string_hash(x) = 0;
  string_needs_free(x) = true;
  add_string(sc, x);
  return(x);
}


static s7_pointer make_string_wrapper_with_length(s7_scheme *sc, const char *str, int len)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING | T_IMMUTABLE | T_SAFE_PROCEDURE);
  string_value(x) = (char *)str;
  string_length(x) = len;
  string_hash(x) = 0;
  string_needs_free(x) = false;
  return(x);
}

static s7_pointer make_string_wrapper(s7_scheme *sc, const char *str)
{
  return(make_string_wrapper_with_length(sc, str, safe_strlen(str)));
}

static s7_pointer make_empty_string(s7_scheme *sc, int len, char fill)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING);
  string_value(x) = (char *)malloc((len + 1) * sizeof(char));
  if (fill != 0)
    memset((void *)(string_value(x)), fill, len);
  string_value(x)[len] = 0;
  string_hash(x) = 0;
  string_length(x) = len;
  string_needs_free(x) = true;
  add_string(sc, x);
  return(x);
}


s7_pointer s7_make_string(s7_scheme *sc, const char *str)
{
  if (str)
    return(s7_make_string_with_length(sc, str, safe_strlen(str)));
  return(make_empty_string(sc, 0, 0));
}


static char *make_permanent_string(const char *str)
{
  char *x;
  int len;
  len = safe_strlen(str);
  x = (char *)malloc((len + 1) * sizeof(char));
  memcpy((void *)x, (void *)str, len);
  x[len] = 0;
  return(x);
}


s7_pointer s7_make_permanent_string(const char *str)
{
  /* for the symbol table which is never GC'd */
  s7_pointer x;
  x = alloc_pointer();
  unheap(x);
  set_type(x, T_STRING | T_IMMUTABLE);
  if (str)
    {
      unsigned int len;
      len = safe_strlen(str);
      string_length(x) = len;
      string_value(x) = (char *)malloc((len + 1) * sizeof(char));
      memcpy((void *)string_value(x), (void *)str, len);
      string_value(x)[len] = 0;
    }
  else
    {
      string_value(x) = NULL;
      string_length(x) = 0;
    }
  string_hash(x) = 0;
  string_needs_free(x) = false;
  return(x);
}


static s7_pointer make_temporary_string(s7_scheme *sc, const char *str, int len)
{
  s7_pointer p;
  p = sc->tmp_strs[0];
  prepare_temporary_string(sc, len + 1, 0);
  string_length(p) = len;
  if (len > 0)
    memmove((void *)(string_value(p)), (void *)str, len); /* not memcpy because str might be a temp string (i.e. sc->tmp_str_chars -> itself) */
  string_value(p)[len] = 0;
  return(p);
}


bool s7_is_string(s7_pointer p)
{
  return(is_string(p));
}


const char *s7_string(s7_pointer p)
{
  return(string_value(p));
}


static s7_pointer g_is_string(s7_scheme *sc, s7_pointer args)
{
  #define H_is_string "(string? obj) returns #t if obj is a string"
  #define Q_is_string pl_bt

  check_boolean_method(sc, is_string, sc->is_string_symbol, args);
}


/* -------------------------------- make-string -------------------------------- */
static s7_pointer g_make_string(s7_scheme *sc, s7_pointer args)
{
  #define H_make_string "(make-string len (val #\\space)) makes a string of length len filled with the character val (default: space)"
  #define Q_make_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_integer_symbol, sc->is_char_symbol)

  s7_pointer n;
  s7_int len;
  char fill = ' ';

  n = car(args);
  if (!s7_is_integer(n))
    {
      check_two_methods(sc, n, sc->make_string_symbol, sc->make_byte_vector_symbol, args);
      return(wrong_type_argument(sc, sc->make_string_symbol, 1, n, T_INTEGER));
    }

  len = s7_integer(n);
  if ((len < 0) || (len > sc->max_string_length))
    return(out_of_range(sc, sc->make_string_symbol, small_int(1), n, (len < 0) ? its_negative_string : its_too_large_string));

  if (is_not_null(cdr(args)))
    {
      if (!s7_is_character(cadr(args)))
	method_or_bust(sc, cadr(args), sc->make_string_symbol, args, T_CHARACTER, 2);
      fill = s7_character(cadr(args));
    }
  n = make_empty_string(sc, (int)len, fill);
  if (fill == '\0')
    memset((void *)string_value(n), 0, (int)len);
  return(n);
}

static s7_pointer c_make_string(s7_scheme *sc, s7_int len) {return(make_empty_string(sc, (int)len, ' '));}
IF_TO_PF(make_string, c_make_string)


#if (!WITH_PURE_S7)
static s7_pointer g_string_length(s7_scheme *sc, s7_pointer args)
{
  #define H_string_length "(string-length str) returns the length of the string str"
  #define Q_string_length s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_string_symbol)
  s7_pointer p;
  p = car(args);
  if (!is_string(p))
    method_or_bust(sc, p, sc->string_length_symbol, args, T_STRING, 0);
  return(make_integer(sc, string_length(p)));
}

static s7_int c_string_length(s7_scheme *sc, s7_pointer p)
{
  if (!is_string(p))
    int_method_or_bust(sc, p, sc->string_length_symbol, set_plist_1(sc, p), T_STRING, 0);
  return(string_length(p));
}

PF_TO_IF(string_length, c_string_length)
#endif


/* -------------------------------- string-up|downcase -------------------------------- */

static s7_pointer c_string_downcase(s7_scheme *sc, s7_pointer p)
{
  s7_pointer newstr;
  int i, len;
  unsigned char *nstr, *ostr;

  sc->temp3 = p;
  if (!is_string(p))
    method_or_bust(sc, p, sc->string_downcase_symbol, list_1(sc, p), T_STRING, 0);

  len = string_length(p);
  newstr = make_empty_string(sc, len, 0);

  ostr = (unsigned char *)string_value(p);
  nstr = (unsigned char *)string_value(newstr);
  for (i = 0; i < len; i++)
    nstr[i] = lowers[(int)ostr[i]];

  return(newstr);
}

static s7_pointer g_string_downcase(s7_scheme *sc, s7_pointer args)
{
  #define H_string_downcase "(string-downcase str) returns the lower case version of str."
  #define Q_string_downcase pcl_s
  return(c_string_downcase(sc, car(args)));
}

PF_TO_PF(string_downcase, c_string_downcase)


static s7_pointer c_string_upcase(s7_scheme *sc, s7_pointer p)
{
  s7_pointer newstr;
  int i, len;
  unsigned char *nstr, *ostr;

  sc->temp3 = p;
  if (!is_string(p))
    method_or_bust(sc, p, sc->string_upcase_symbol, list_1(sc, p), T_STRING, 0);

  len = string_length(p);
  newstr = make_empty_string(sc, len, 0);

  ostr = (unsigned char *)string_value(p);
  nstr = (unsigned char *)string_value(newstr);
  for (i = 0; i < len; i++)
    nstr[i] = uppers[(int)ostr[i]];

  return(newstr);
}

static s7_pointer g_string_upcase(s7_scheme *sc, s7_pointer args)
{
  #define H_string_upcase "(string-upcase str) returns the upper case version of str."
  #define Q_string_upcase pcl_s
  return(c_string_upcase(sc, car(args)));
}

PF_TO_PF(string_upcase, c_string_upcase)


unsigned int s7_string_length(s7_pointer str)
{
  return(string_length(str));
}


/* -------------------------------- string-ref -------------------------------- */
static s7_pointer string_ref_1(s7_scheme *sc, s7_pointer strng, s7_pointer index)
{
  /* every use of this has already checked for the byte-vector case */
  char *str;
  s7_int ind;

  if (!s7_is_integer(index))
    {
      s7_pointer p;
      if (!s7_is_integer(p = check_values(sc, index, cons(sc, index, sc->nil))))
	method_or_bust(sc, index, sc->string_ref_symbol, list_2(sc, strng, index), T_INTEGER, 2);
      index = p;
    }
  ind = s7_integer(index);
  if (ind < 0)
    return(wrong_type_argument_with_type(sc, sc->string_ref_symbol, 2, index, a_non_negative_integer_string));
  if (ind >= string_length(strng))
    return(out_of_range(sc, sc->string_ref_symbol, small_int(2), index, its_too_large_string));

  str = string_value(strng);
  return(s7_make_character(sc, ((unsigned char *)str)[ind]));
}


static s7_pointer g_string_ref_2(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  s7_pointer strng, index, p;
  char *str;
  s7_int ind;

  strng = car(args);
  if (!is_string(strng))
    method_or_bust(sc, strng, caller, args, T_STRING, 1);

  index = cadr(args);
  if (!s7_is_integer(index))
    {
      if (!s7_is_integer(p = check_values(sc, index, cdr(args))))
	method_or_bust(sc, index, caller, args, T_INTEGER, 2);
      index = p;
    }
  ind = s7_integer(index);
  if (ind < 0)
    return(wrong_type_argument_with_type(sc, caller, 2, index, a_non_negative_integer_string));
  if (ind >= string_length(strng))
    return(out_of_range(sc, caller, small_int(2), index, its_too_large_string));

  str = string_value(strng);
  if (is_byte_vector(strng))
    return(small_int((unsigned char)(str[ind])));
  return(s7_make_character(sc, ((unsigned char *)str)[ind]));
}

static s7_pointer g_string_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_string_ref "(string-ref str index) returns the character at the index-th element of the string str"
  #define Q_string_ref s7_make_signature(sc, 3, sc->is_char_symbol, sc->is_string_symbol, sc->is_integer_symbol)
  return(g_string_ref_2(sc, args, sc->string_ref_symbol));
}

static s7_pointer g_byte_vector_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_byte_vector_ref "(byte-vector-ref vect index) returns the byte at the index-th element of the byte-vector vect"
  #define Q_byte_vector_ref s7_make_signature(sc, 3, sc->is_integer_symbol, sc->is_byte_vector_symbol, sc->is_integer_symbol)
  return(g_string_ref_2(sc, args, sc->byte_vector_ref_symbol));
}

static s7_pointer c_string_ref(s7_scheme *sc, s7_pointer str, s7_int ind)
{
  if (!is_string(str))
    method_or_bust(sc, str, sc->string_ref_symbol, list_2(sc, str, make_integer(sc, ind)), T_STRING, 1);
  if (ind < 0)
    return(wrong_type_argument_with_type(sc, sc->string_ref_symbol, 2, make_integer(sc, ind), a_non_negative_integer_string));
  if (ind >= string_length(str))
    return(out_of_range(sc, sc->string_ref_symbol, small_int(2), make_integer(sc, ind), its_too_large_string));
  if (is_byte_vector(str))
    return(small_int(((unsigned char *)string_value(str))[ind]));
  return(s7_make_character(sc, ((unsigned char *)string_value(str))[ind]));
}

PIF_TO_PF(string_ref, c_string_ref)


/* -------------------------------- string-set! -------------------------------- */
static s7_pointer g_string_set_2(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  s7_pointer x, c, index;
  char *str;
  s7_int ind;

  x = car(args);
  if (!is_string(x))
    method_or_bust(sc, x, caller, args, T_STRING, 1);

  index = cadr(args);
  if (!s7_is_integer(index))
    {
      s7_pointer p;
      if (!s7_is_integer(p = check_values(sc, index, cdr(args))))
	method_or_bust(sc, index, caller, args, T_INTEGER, 2);
      index = p;
    }
  ind = s7_integer(index);
  if (ind < 0)
    return(wrong_type_argument_with_type(sc, caller, 2, index, a_non_negative_integer_string));
  if (ind >= string_length(x))
    return(out_of_range(sc, caller, small_int(2), index, its_too_large_string));
  str = string_value(_TSet(x));

  c = caddr(args);
  if (!s7_is_character(c))
    {
      if ((is_byte_vector(x)) &&
	  (s7_is_integer(c)))
	{
	  s7_int ic;  /* not int here! */
	  ic = s7_integer(c);
	  if ((ic < 0) || (ic > 255))
	    return(wrong_type_argument_with_type(sc, caller, 3, c, an_unsigned_byte_string));
	  str[ind] = (char)ic;
	  return(c);
	}
      method_or_bust(sc, c, caller, list_3(sc, x, index, c), T_CHARACTER, 3);
    }
  str[ind] = (char)s7_character(c);
  return(c);
}

static s7_pointer g_string_set(s7_scheme *sc, s7_pointer args)
{
  #define H_string_set "(string-set! str index chr) sets the index-th element of the string str to the character chr"
  #define Q_string_set s7_make_signature(sc, 4, sc->is_char_symbol, sc->is_string_symbol, sc->is_integer_symbol, sc->is_char_symbol)
  return(g_string_set_2(sc, args, sc->string_set_symbol));
}

static s7_pointer g_byte_vector_set(s7_scheme *sc, s7_pointer args)
{
  #define H_byte_vector_set "(byte-vector-set! vect index byte) sets the index-th element of the byte-vector vect to the integer byte"
  #define Q_byte_vector_set s7_make_signature(sc, 4, sc->is_integer_symbol, sc->is_byte_vector_symbol, sc->is_integer_symbol, sc->is_integer_symbol)
  return(g_string_set_2(sc, args, sc->byte_vector_set_symbol));
}

static int c_string_tester(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer a1;
  a1 = cadr(expr);
  if (is_symbol(a1))
    {
      s7_pointer table;
      table = s7_slot(sc, a1);
      if ((is_slot(table)) && 
	  ((is_immutable_symbol(a1)) || (!is_stepper(table))) &&
	  (is_string(slot_value(table))))
	{
	  s7_pointer a2;
	  s7_xf_store(sc, slot_value(table));
	  a2 = caddr(expr);
	  if (is_symbol(a2))
	    {
	      s7_pointer slot;
	      slot = s7_slot(sc, a2);
	      if ((is_slot(slot)) && 
		  (is_integer(slot_value(slot))))
		{
		  s7_xf_store(sc, slot);
		  return(TEST_SS);
		}
	    }
	  else
	    {
	      if (s7_arg_to_if(sc, a1))
		return(TEST_SI);
	    }
	  return(TEST_SQ);
	}
    }
  return(TEST_NO_S);
}

static s7_pointer c_string_set_s(s7_scheme *sc, s7_pointer vec, s7_int index, s7_pointer val)
{
  if ((index < 0) ||
      (index >= string_length(vec)))
    return(out_of_range(sc, sc->string_set_symbol, small_int(2), make_integer(sc, index), (index < 0) ? its_negative_string : its_too_large_string));

  if (!s7_is_character(val))
    {
      if ((is_byte_vector(vec)) &&
	  (s7_is_integer(val)))
	{
	  s7_int ic;  /* not int here! */
	  ic = s7_integer(val);
	  if ((ic < 0) || (ic > 255))
	    return(wrong_type_argument_with_type(sc, sc->string_set_symbol, 3, val, an_unsigned_byte_string));
	  string_value(vec)[index] = (char)ic;
	  return(val);
	}
      method_or_bust(sc, val, sc->string_set_symbol, list_3(sc, vec, make_integer(sc, index), val), T_CHARACTER, 3);
    }
  string_value(vec)[index] = (char)character(val);
  return(val);
}

static s7_pointer c_string_set(s7_scheme *sc, s7_pointer vec, s7_int index, s7_pointer val)
{
  if (!s7_is_string(vec))
    method_or_bust(sc, vec, sc->string_set_symbol, set_plist_3(sc, vec, make_integer(sc, index), val), T_STRING, 1);
  return(c_string_set_s(sc, vec, index, val));
}

PIPF_TO_PF(string_set, c_string_set_s, c_string_set, c_string_tester)


/* -------------------------------- string-append -------------------------------- */
static s7_pointer g_string_append_1(s7_scheme *sc, s7_pointer args, bool use_temp)
{
  int len = 0;
  s7_pointer x, newstr;
  char *pos;

  if (is_null(args))
    return(s7_make_string_with_length(sc, "", 0));

  /* get length for new string */
  for (x = args; is_not_null(x); x = cdr(x))
    {
      s7_pointer p;
      p = car(x);
      if (!is_string(p))
	{
	  /* look for string-append and if found, cobble up a plausible intermediate call */
	  if (has_methods(p))
	    {
	      s7_pointer func;
	      func = find_method(sc, find_let(sc, p), sc->string_append_symbol);
	      if (func != sc->undefined)
		{
		  s7_pointer y;
		  if (len == 0)
		    return(s7_apply_function(sc, func, args));
		  newstr = make_empty_string(sc, len, 0);
		  for (pos = string_value(newstr), y = args; y != x; pos += string_length(car(y)), y = cdr(y))
		    memcpy(pos, string_value(car(y)), string_length(car(y)));
		  return(s7_apply_function(sc, func, cons(sc, newstr, x)));
		}
	    }
	  return(wrong_type_argument(sc, sc->string_append_symbol, position_of(x, args), p, T_STRING));
	}
      len += string_length(p);
    }

  if (use_temp)
    {
      newstr = sc->tmp_strs[0];
      prepare_temporary_string(sc, len + 1, 0);
      string_length(newstr) = len;
      string_value(newstr)[len] = 0;
    }
  else
    {
      /* store the contents of the argument strings into the new string */
      newstr = make_empty_string(sc, len, 0);
    }
  for (pos = string_value(newstr), x = args; is_not_null(x); pos += string_length(car(x)), x = cdr(x))
    memcpy(pos, string_value(car(x)), string_length(car(x)));

  if (is_byte_vector(car(args)))
    set_byte_vector(newstr);

  return(newstr);
}

static s7_pointer g_string_append(s7_scheme *sc, s7_pointer args)
{
  #define H_string_append "(string-append str1 ...) appends all its string arguments into one string"
  #define Q_string_append pcl_s
  return(g_string_append_1(sc, args, false));
}

static s7_pointer string_append_to_temp;
static s7_pointer g_string_append_to_temp(s7_scheme *sc, s7_pointer args)
{
  return(g_string_append_1(sc, args, true));
}


#if (!WITH_PURE_S7)
static s7_pointer g_string_copy(s7_scheme *sc, s7_pointer args)
{
  #define H_string_copy "(string-copy str) returns a copy of its string argument"
  #define Q_string_copy s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_string_symbol)
  s7_pointer p;
  p = car(args);
  if (!is_string(p))
    method_or_bust(sc, p, sc->string_copy_symbol, args, T_STRING, 1);
  return(s7_make_string_with_length(sc, string_value(p), string_length(p)));
}
#endif


/* -------------------------------- substring -------------------------------- */
static s7_pointer start_and_end(s7_scheme *sc, s7_pointer caller, s7_pointer fallback,
				s7_pointer start_and_end_args, s7_pointer args, int position, s7_int *start, s7_int *end)
{
  /* we assume that *start=0 and *end=length, that end is "exclusive"
   *   return true if the start/end points are not changed.
   */
  s7_pointer pstart, pend, p;
  s7_int index;

#if DEBUGGING
  if (is_null(start_and_end_args))
    {
      fprintf(stderr, "start_and_end args is null\n");
      return(sc->gc_nil);
    }
#endif

  pstart = car(start_and_end_args);
  if (!s7_is_integer(pstart))
    {
      if (!s7_is_integer(p = check_values(sc, pstart, start_and_end_args)))
	{
	  check_two_methods(sc, pstart, caller, fallback, args);
	  return(wrong_type_argument(sc, caller, position, pstart, T_INTEGER));
	}
      else pstart = p;
    }

  index = s7_integer(pstart);
  if ((index < 0) ||
      (index > *end)) /* *end == length here */
    return(out_of_range(sc, caller, small_int(position), pstart, (index < 0) ? its_negative_string : its_too_large_string));
  *start = index;

  if (is_null(cdr(start_and_end_args)))
    return(sc->gc_nil);

  pend = cadr(start_and_end_args);
  if (!s7_is_integer(pend))
    {
      if (!s7_is_integer(p = check_values(sc, pend, cdr(start_and_end_args))))
	{
	  check_two_methods(sc, pend, caller, fallback,
			    (position == 2) ? list_3(sc, car(args), pstart, pend) : list_4(sc, car(args), cadr(args), pstart, pend));
	  return(wrong_type_argument(sc, caller, position + 1, pend, T_INTEGER));
	}
      else pend = p;
    }
  index = s7_integer(pend);
  if ((index < *start) ||
      (index > *end))
    return(out_of_range(sc, caller, small_int(position + 1), pend, (index < *start) ? its_too_small_string : its_too_large_string));
  *end = index;
  return(sc->gc_nil);
}


static s7_pointer g_substring(s7_scheme *sc, s7_pointer args)
{
  #define H_substring "(substring str start (end (length str))) returns the portion of the string str between start and \
end: (substring \"01234\" 1 2) -> \"1\""
  #define Q_substring s7_make_circular_signature(sc, 2, 3, sc->is_string_symbol, sc->is_string_symbol, sc->is_integer_symbol)

  s7_pointer x, str;
  s7_int start = 0, end;
  int len;
  char *s;

  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->substring_symbol, args, T_STRING, 1);

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      x = start_and_end(sc, sc->substring_symbol, NULL, cdr(args), args, 2, &start, &end);
      if (x != sc->gc_nil) return(x);
    }
  s = string_value(str);
  len = (int)(end - start);
  x = s7_make_string_with_length(sc, (char *)(s + start), len);
  string_value(x)[len] = 0;
  return(x);
}


static s7_pointer substring_to_temp;
static s7_pointer g_substring_to_temp(s7_scheme *sc, s7_pointer args)
{
  s7_pointer str;
  s7_int start = 0, end;

  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->substring_symbol, args, T_STRING, 1);

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      s7_pointer x;
      x = start_and_end(sc, sc->substring_symbol, NULL, cdr(args), args, 2, &start, &end);
      if (x != sc->gc_nil) return(x);
    }
  return(make_temporary_string(sc, (const char *)(string_value(str) + start), (int)(end - start)));
}


/* -------------------------------- object->string -------------------------------- */
static use_write_t write_choice(s7_scheme *sc, s7_pointer arg)
{
  if (arg == sc->F) return(USE_DISPLAY);
  if (arg == sc->T) return(USE_WRITE);
  if (arg == sc->key_readable_symbol) return(USE_READABLE_WRITE);
  return(USE_WRITE_WRONG);
}

#define DONT_USE_DISPLAY(Choice) ((Choice == USE_DISPLAY) ? USE_WRITE : Choice)

static char *s7_object_to_c_string_1(s7_scheme *sc, s7_pointer obj, use_write_t use_write, int *nlen);

static s7_pointer g_object_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_object_to_string "(object->string obj (write #t)) returns a string representation of obj."
  #define Q_object_to_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->T, s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_keyword_symbol))

  use_write_t choice;
  char *str;
  s7_pointer obj;
  int len = 0;

  if (is_not_null(cdr(args)))
    {
      choice = write_choice(sc, cadr(args));
      if (choice == USE_WRITE_WRONG)
	method_or_bust(sc, cadr(args), sc->object_to_string_symbol, args, T_BOOLEAN, 2);
    }
  else choice = USE_WRITE;
  /* can't use s7_object_to_string here anymore because it assumes use_write arg is a boolean */

  obj = car(args);
  check_method(sc, obj, sc->object_to_string_symbol, args);
  str = s7_object_to_c_string_1(sc, obj, choice, &len);
  if (str)
    return(make_string_uncopied_with_length(sc, str, len));
  return(s7_make_string_with_length(sc, "", 0));
}

static s7_pointer c_object_to_string(s7_scheme *sc, s7_pointer x) {return(g_object_to_string(sc, set_plist_1(sc, x)));}
PF_TO_PF(object_to_string, c_object_to_string)


/* -------------------------------- string comparisons -------------------------------- */
static int scheme_strcmp(s7_pointer s1, s7_pointer s2)
{
  /* tricky here because str[i] must be treated as unsigned
   *   (string<? (string (integer->char #xf0)) (string (integer->char #x70)))
   * also null or lack thereof does not say anything about the string end
   *   so we have to go by its length.
   */
  int i, len, len1, len2;
  char *str1, *str2;

  len1 = string_length(s1);
  len2 = string_length(s2);
  if (len1 > len2)
    len = len2;
  else len = len1;

  str1 = string_value(s1);
  str2 = string_value(s2);

  for (i = 0; i < len; i++)
    if ((unsigned char)(str1[i]) < (unsigned char )(str2[i]))
      return(-1);
    else
      {
	if ((unsigned char)(str1[i]) > (unsigned char)(str2[i]))
	  return(1);
      }

  if (len1 < len2)
    return(-1);
  if (len1 > len2)
    return(1);
  return(0);
}


static bool is_string_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_string(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_string_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}

static s7_pointer g_string_cmp(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    method_or_bust(sc, y, sym, args, T_STRING, 1);

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args));
      if (scheme_strcmp(y, car(x)) != val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_string_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_string_cmp_not(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    method_or_bust(sc, y, sym, args, T_STRING, 1);

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args));
      if (scheme_strcmp(y, car(x)) == val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_string_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static bool scheme_strings_are_equal(s7_pointer x, s7_pointer y)
{
  return((string_length(x) == string_length(y)) &&
	 (strings_are_equal_with_length(string_value(x), string_value(y), string_length(x))));
}


static s7_pointer g_strings_are_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_equal "(string=? str ...) returns #t if all the string arguments are equal"
  #define Q_strings_are_equal pcl_bs

  /* C-based check stops at null, but we can have embedded nulls.
   *   (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2))
   */
  s7_pointer x, y;
  bool happy = true;

  y = car(args);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_eq_symbol, args, T_STRING, 1);

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      s7_pointer p;
      p = car(x);
      if (y != p)
	{
	  if (!is_string(p))
	    method_or_bust(sc, p, sc->string_eq_symbol, cons(sc, y, x), T_STRING, position_of(x, args));
	  if (happy)
	    happy = scheme_strings_are_equal(p, y);
	}
    }
  if (!happy)
    return(sc->F);
  return(sc->T);
}

static s7_pointer c_string_eq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{ 
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_eq_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_eq_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, ((string_length(x) == string_length(y)) &&
			   (strings_are_equal_with_length(string_value(x), string_value(y), string_length(x))))));
}

PF2_TO_PF(string_eq, c_string_eq)


static s7_pointer g_strings_are_less(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_less "(string<? str ...) returns #t if all the string arguments are increasing"
  #define Q_strings_are_less pcl_bs

  return(g_string_cmp(sc, args, -1, sc->string_lt_symbol));
}

static s7_pointer c_string_lt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_lt_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_lt_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcmp(x, y) == -1));
}

PF2_TO_PF(string_lt, c_string_lt)


static s7_pointer g_strings_are_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_greater "(string>? str ...) returns #t if all the string arguments are decreasing"
  #define Q_strings_are_greater pcl_bs

  return(g_string_cmp(sc, args, 1, sc->string_gt_symbol));
}

static s7_pointer c_string_gt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_gt_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_gt_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcmp(x, y) == 1));
}

PF2_TO_PF(string_gt, c_string_gt)


static s7_pointer g_strings_are_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_geq "(string>=? str ...) returns #t if all the string arguments are equal or decreasing"
  #define Q_strings_are_geq pcl_bs

  return(g_string_cmp_not(sc, args, -1, sc->string_geq_symbol));
}

static s7_pointer c_string_geq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_geq_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_geq_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcmp(x, y) != -1));
}

PF2_TO_PF(string_geq, c_string_geq)


static s7_pointer g_strings_are_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_leq "(string<=? str ...) returns #t if all the string arguments are equal or increasing"
  #define Q_strings_are_leq pcl_bs

  return(g_string_cmp_not(sc, args, 1, sc->string_leq_symbol));
}

static s7_pointer c_string_leq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_leq_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_leq_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcmp(x, y) != 1));
}

PF2_TO_PF(string_leq, c_string_leq)


static s7_pointer string_equal_s_ic, string_equal_2;
static s7_pointer g_string_equal_s_ic(s7_scheme *sc, s7_pointer args)
{
  if (!is_string(car(args)))
    method_or_bust(sc, car(args), sc->string_eq_symbol, args, T_STRING, 1);
  return(make_boolean(sc, scheme_strings_are_equal(car(args), cadr(args))));
}

static s7_pointer g_string_equal_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_string(car(args)))
    method_or_bust(sc, car(args), sc->string_eq_symbol, args, T_STRING, 1);
  if (!is_string(cadr(args)))
    method_or_bust(sc, cadr(args), sc->string_eq_symbol, args, T_STRING, 2);
  return(make_boolean(sc, scheme_strings_are_equal(car(args), cadr(args))));
}


static s7_pointer string_less_2;
static s7_pointer g_string_less_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_string(car(args)))
    method_or_bust(sc, car(args), sc->string_lt_symbol, args, T_STRING, 1);
  if (!is_string(cadr(args)))
    method_or_bust(sc, cadr(args), sc->string_lt_symbol, args, T_STRING, 2);
  return(make_boolean(sc, scheme_strcmp(car(args), cadr(args)) == -1));
}


static s7_pointer string_greater_2;
static s7_pointer g_string_greater_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_string(car(args)))
    method_or_bust(sc, car(args), sc->string_gt_symbol, args, T_STRING, 1);
  if (!is_string(cadr(args)))
    method_or_bust(sc, cadr(args), sc->string_gt_symbol, args, T_STRING, 2);
  return(make_boolean(sc, scheme_strcmp(car(args), cadr(args)) == 1));
}


#if (!WITH_PURE_S7)

static int scheme_strcasecmp(s7_pointer s1, s7_pointer s2)
{
  /* same as scheme_strcmp -- watch out for unwanted sign! and lack of trailing null (length sets string end).
   */
  int i, len, len1, len2;
  unsigned char *str1, *str2;

  len1 = string_length(s1);
  len2 = string_length(s2);
  if (len1 > len2)
    len = len2;
  else len = len1;

  str1 = (unsigned char *)string_value(s1);
  str2 = (unsigned char *)string_value(s2);

  for (i = 0; i < len; i++)
    if (uppers[(int)str1[i]] < uppers[(int)str2[i]])
      return(-1);
    else
      {
	if (uppers[(int)str1[i]] > uppers[(int)str2[i]])
	  return(1);
      }

  if (len1 < len2)
    return(-1);
  if (len1 > len2)
    return(1);
  return(0);
}


static bool scheme_strequal_ci(s7_pointer s1, s7_pointer s2)
{
  /* same as scheme_strcmp -- watch out for unwanted sign! */
  int i, len, len2;
  unsigned char *str1, *str2;

  len = string_length(s1);
  len2 = string_length(s2);
  if (len != len2)
    return(false);

  str1 = (unsigned char *)string_value(s1);
  str2 = (unsigned char *)string_value(s2);

  for (i = 0; i < len; i++)
    if (uppers[(int)str1[i]] != uppers[(int)str2[i]])
      return(false);
  return(true);
}


static s7_pointer g_string_ci_cmp(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    method_or_bust(sc, y, sym, args, T_STRING, 1);

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args));
      if (val == 0)
	{
	  if (!scheme_strequal_ci(y, car(x)))
	    {
	      for (y = cdr(x); is_pair(y); y = cdr(y))
		if (!is_string_via_method(sc, car(y)))
		  return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
	      return(sc->F);
	    }
	}
      else
	{
	  if (scheme_strcasecmp(y, car(x)) != val)
	    {
	      for (y = cdr(x); is_pair(y); y = cdr(y))
		if (!is_string_via_method(sc, car(y)))
		  return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
	      return(sc->F);
	    }
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_string_ci_cmp_not(s7_scheme *sc, s7_pointer args, int val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    method_or_bust(sc, y, sym, args, T_STRING, 1);

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args));
      if (scheme_strcasecmp(y, car(x)) == val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_string_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_strings_are_ci_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_equal "(string-ci=? str ...) returns #t if all the string arguments are equal, ignoring case"
  #define Q_strings_are_ci_equal pcl_bs
  return(g_string_ci_cmp(sc, args, 0, sc->string_ci_eq_symbol));
}

static s7_pointer c_string_ci_eq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_ci_eq_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_ci_eq_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcasecmp(x, y) == 0));
}

PF2_TO_PF(string_ci_eq, c_string_ci_eq)


static s7_pointer g_strings_are_ci_less(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_less "(string-ci<? str ...) returns #t if all the string arguments are increasing, ignoring case"
  #define Q_strings_are_ci_less pcl_bs
  return(g_string_ci_cmp(sc, args, -1, sc->string_ci_lt_symbol));
}

static s7_pointer c_string_ci_lt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_ci_lt_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_ci_lt_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcasecmp(x, y) == -1));
}

PF2_TO_PF(string_ci_lt, c_string_ci_lt)


static s7_pointer g_strings_are_ci_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_greater "(string-ci>? str ...) returns #t if all the string arguments are decreasing, ignoring case"
  #define Q_strings_are_ci_greater pcl_bs
  return(g_string_ci_cmp(sc, args, 1, sc->string_ci_gt_symbol));
}

static s7_pointer c_string_ci_gt(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_ci_gt_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_ci_gt_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcasecmp(x, y) == 1));
}

PF2_TO_PF(string_ci_gt, c_string_ci_gt)


static s7_pointer g_strings_are_ci_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_geq "(string-ci>=? str ...) returns #t if all the string arguments are equal or decreasing, ignoring case"
  #define Q_strings_are_ci_geq pcl_bs
  return(g_string_ci_cmp_not(sc, args, -1, sc->string_ci_geq_symbol));
}

static s7_pointer c_string_ci_geq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_ci_geq_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_ci_geq_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcasecmp(x, y) != -1));
}

PF2_TO_PF(string_ci_geq, c_string_ci_geq)


static s7_pointer g_strings_are_ci_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_leq "(string-ci<=? str ...) returns #t if all the string arguments are equal or increasing, ignoring case"
  #define Q_strings_are_ci_leq pcl_bs
  return(g_string_ci_cmp_not(sc, args, 1, sc->string_ci_leq_symbol));
}

static s7_pointer c_string_ci_leq(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (!is_string(x))
    method_or_bust(sc, x, sc->string_ci_leq_symbol, list_2(sc, x, y), T_STRING, 1);
  if (!is_string(y))
    method_or_bust(sc, y, sc->string_ci_leq_symbol, list_2(sc, x, y), T_STRING, 2);
  return(make_boolean(sc, scheme_strcasecmp(x, y) != 1));
}

PF2_TO_PF(string_ci_leq, c_string_ci_leq)
#endif /* pure s7 */


static s7_pointer g_string_fill(s7_scheme *sc, s7_pointer args)
{
  #define H_string_fill "(string-fill! str chr start end) fills the string str with the character chr"
  #define Q_string_fill s7_make_circular_signature(sc, 3, 4, s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_integer_symbol), sc->is_string_symbol, sc->is_char_symbol, sc->is_integer_symbol)

  s7_pointer x, chr;
  s7_int start = 0, end, byte = 0;
  x = car(args);

  if (!is_string(x))
    method_or_bust(sc, x, sc->string_fill_symbol, args, T_STRING, 1); /* not two methods here */

  chr = cadr(args);
  if (!is_byte_vector(x))
    {
      if (!s7_is_character(chr))
	{
	  check_two_methods(sc, chr, sc->string_fill_symbol, sc->fill_symbol, args);
	  return(wrong_type_argument(sc, sc->string_fill_symbol, 2, chr, T_CHARACTER));
	}
    }
  else
    {
      if (!is_integer(chr))
	{
	  check_two_methods(sc, chr, sc->string_fill_symbol, sc->fill_symbol, args);
	  return(wrong_type_argument(sc, sc->fill_symbol, 2, chr, T_INTEGER));
	}
      byte = integer(chr);
      if ((byte < 0) || (byte > 255))
	return(simple_wrong_type_argument_with_type(sc, sc->string_fill_symbol, chr, an_unsigned_byte_string));
    }

  end = string_length(x);
  if (!is_null(cddr(args)))
    {
      s7_pointer p;
      p = start_and_end(sc, sc->string_fill_symbol, sc->fill_symbol, cddr(args), args, 3, &start, &end);
      if (p != sc->gc_nil) return(p);
      if (start == end) return(chr);
    }
  if (end == 0) return(chr);

  if (!is_byte_vector(x))
    memset((void *)(string_value(x) + start), (int)character(chr), end - start);
  else memset((void *)(string_value(x) + start), (int)byte, end - start);

  return(chr);
}

#if (!WITH_PURE_S7)
static s7_pointer c_string_fill(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_string_fill(sc, set_plist_2(sc, x, y)));}
PF2_TO_PF(string_fill, c_string_fill)
#endif


static s7_pointer g_string_1(s7_scheme *sc, s7_pointer args, s7_pointer sym)
{
  int i, len;
  s7_pointer x, newstr;
  char *str;

  /* get length for new string and check arg types */
  for (len = 0, x = args; is_not_null(x); len++, x = cdr(x))
    {
      s7_pointer p;
      p = car(x);
      if (!s7_is_character(p))
	{
	  if (has_methods(p))
	    {
	      s7_pointer func;
	      func = find_method(sc, find_let(sc, p), sym);
	      if (func != sc->undefined)
		{
		  s7_pointer y;
		  if (len == 0)
		    return(s7_apply_function(sc, func, args));
		  newstr = make_empty_string(sc, len, 0);
		  str = string_value(newstr);
		  for (i = 0, y = args; y != x; i++, y = cdr(y))
		    str[i] = character(car(y));
		  return(g_string_append(sc, set_plist_2(sc, newstr, s7_apply_function(sc, func, x))));
		}
	    }
	  return(wrong_type_argument(sc, sym, len + 1, car(x), T_CHARACTER));
	}
    }
  newstr = make_empty_string(sc, len, 0);
  str = string_value(newstr);
  for (i = 0, x = args; is_not_null(x); i++, x = cdr(x))
    str[i] = character(car(x));

  return(newstr);
}


static s7_pointer g_string(s7_scheme *sc, s7_pointer args)
{
  #define H_string "(string chr...) appends all its character arguments into one string"
  #define Q_string s7_make_circular_signature(sc, 1, 2, sc->is_string_symbol, sc->is_char_symbol)

  if (is_null(args))                                /* (string) but not (string ()) */
    return(s7_make_string_with_length(sc, "", 0));
  return(g_string_1(sc, args, sc->string_symbol));
}

#if (!WITH_PURE_S7)
static s7_pointer g_list_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_list_to_string "(list->string lst) appends all the list's characters into one string; (apply string lst)"
  #define Q_list_to_string s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_proper_list_symbol)

  if (is_null(car(args)))
    return(s7_make_string_with_length(sc, "", 0));

  if (!is_proper_list(sc, car(args)))
    method_or_bust_with_type(sc, car(args), sc->list_to_string_symbol, args, make_string_wrapper(sc, "a (proper, non-circular) list of characters"), 0);
  return(g_string_1(sc, car(args), sc->list_to_string_symbol));
}
#endif

static s7_pointer s7_string_to_list(s7_scheme *sc, const char *str, int len)
{
  int i;
  s7_pointer result;

  if (len == 0) 
    return(sc->nil);
  if (len >= (sc->free_heap_top - sc->free_heap))
    {
      gc(sc);
      while (len >= (sc->free_heap_top - sc->free_heap))
	resize_heap(sc);
    }

  sc->v = sc->nil;
  for (i = len - 1; i >= 0; i--)
    sc->v = cons_unchecked(sc, s7_make_character(sc, ((unsigned char)str[i])), sc->v);
  result = sc->v;
  sc->v = sc->nil;
  return(result);
}

#if (!WITH_PURE_S7)
static s7_pointer g_string_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_list "(string->list str start end) returns the elements of the string str in a list; (map values str)"
  #define Q_string_to_list s7_make_circular_signature(sc, 2, 3, sc->is_proper_list_symbol, sc->is_string_symbol, sc->is_integer_symbol)

  s7_int i, start = 0, end;
  s7_pointer p, str;

  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->string_to_list_symbol, args, T_STRING, 0);

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      p = start_and_end(sc, sc->string_to_list_symbol, NULL, cdr(args), args, 2, &start, &end);
      if (p != sc->gc_nil) return(p);
      if (start == end) return(sc->nil);
    }
  else
    {
      if (end == 0) return(sc->nil);
    }
  if ((start == 0) && (end == string_length(str)))
    return(s7_string_to_list(sc, string_value(str), string_length(str)));

  sc->w = sc->nil;
  for (i = end - 1; i >= start; i--)
    sc->w = cons(sc, s7_make_character(sc, ((unsigned char)string_value(str)[i])), sc->w);

  p = sc->w;
  sc->w = sc->nil;
  return(p);
}

static s7_pointer c_string_to_list(s7_scheme *sc, s7_pointer x) {return(g_string_to_list(sc, set_plist_1(sc, x)));}
PF_TO_PF(string_to_list, c_string_to_list)
#endif


/* -------------------------------- byte_vectors --------------------------------
 *
 * these are just strings with the T_BYTE_VECTOR bit set.
 */

static bool s7_is_byte_vector(s7_pointer b) {return((is_string(b)) && (is_byte_vector(b)));}

static s7_pointer g_is_byte_vector(s7_scheme *sc, s7_pointer args)
{
  #define H_is_byte_vector "(byte-vector? obj) returns #t if obj is a byte-vector"
  #define Q_is_byte_vector pl_bt

  check_boolean_method(sc, s7_is_byte_vector, sc->is_byte_vector_symbol, args);
}


static s7_pointer g_string_to_byte_vector(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_byte_vector "(string->byte-vector obj) turns a string into a byte-vector."
  #define Q_string_to_byte_vector s7_make_signature(sc, 2, sc->is_byte_vector_symbol, sc->is_string_symbol)
  s7_pointer str;
  str = car(args);
  if (is_integer(str))
    str = s7_make_string_with_length(sc, (const char *)(&(integer(str))), sizeof(s7_int));
  else
    {
      if (!is_string(str))
	method_or_bust(sc, str, sc->string_to_byte_vector_symbol, set_plist_1(sc, str), T_STRING, 1);
    }
  set_byte_vector(str);
  return(str);
}

static s7_pointer c_string_to_byte_vector(s7_scheme *sc, s7_pointer str) {return(g_string_to_byte_vector(sc, set_plist_1(sc, str)));}

PF_TO_PF(string_to_byte_vector, c_string_to_byte_vector)


static s7_pointer g_make_byte_vector(s7_scheme *sc, s7_pointer args)
{
  #define H_make_byte_vector "(make-byte-vector len (byte 0)) makes a byte-vector of length len filled with byte."
  #define Q_make_byte_vector s7_make_circular_signature(sc, 1, 2, sc->is_byte_vector_symbol, sc->is_integer_symbol)

  s7_pointer str;
  if (is_null(cdr(args)))
    {
      str = g_make_string(sc, args);
      if (is_string(str))
	memclr((void *)(string_value(str)), string_length(str));
    }
  else
    {
      s7_pointer len, byte;
      s7_int b;
      len = car(args);
      if (!is_integer(len))
	method_or_bust(sc, len, sc->make_byte_vector_symbol, args, T_INTEGER, 1);

      byte = cadr(args);
      if (!s7_is_integer(byte))
	method_or_bust(sc, byte, sc->make_byte_vector_symbol, args, T_INTEGER, 2);

      b = s7_integer(byte);
      if ((b < 0) || (b > 255))
	return(simple_wrong_type_argument_with_type(sc, sc->make_byte_vector_symbol, byte, an_unsigned_byte_string));
      str = g_make_string(sc, set_plist_2(sc, len, chars[b]));
    }
  set_byte_vector(str);
  return(str);
}


static s7_pointer g_byte_vector(s7_scheme *sc, s7_pointer args)
{
  #define H_byte_vector "(byte-vector ...) returns a byte-vector whose elements are the arguments"
  #define Q_byte_vector s7_make_circular_signature(sc, 1, 2, sc->is_byte_vector_symbol, sc->is_integer_symbol)

  s7_int i, len;
  s7_pointer vec, x;
  char *str;

  len = s7_list_length(sc, args);
  vec = make_empty_string(sc, len, 0);
  str = string_value(vec);

  for (i = 0, x = args; is_pair(x); i++, x = cdr(x))
    {
      s7_pointer byte;
      s7_int b;
      byte = car(x);
      if (!s7_is_integer(byte))
	{
	  if (has_methods(byte))
	    {
	      s7_pointer func;
	      func = find_method(sc, find_let(sc, byte), sc->byte_vector_symbol);
	      if (func != sc->undefined)
		{
		  if (i == 0)
		    return(s7_apply_function(sc, func, args));
		  string_length(vec) = i;
		  vec = g_string_append(sc, set_plist_2(sc, vec, s7_apply_function(sc, func, x)));
		  set_byte_vector(vec);
		  return(vec);
		}
	    }
	  return(wrong_type_argument(sc, sc->byte_vector_symbol, i + 1, byte, T_INTEGER));
	}
      b = s7_integer(byte);
      if ((b < 0) || (b > 255))
	return(simple_wrong_type_argument_with_type(sc, sc->byte_vector_symbol, byte, an_unsigned_byte_string));
      str[i] = (unsigned char)b;
    }
  set_byte_vector(vec);
  return(vec);
}

static s7_pointer byte_vector_to_list(s7_scheme *sc, const char *str, int len)
{
  int i;
  s7_pointer p;
  if (len == 0) return(sc->nil);
  sc->w = sc->nil;
  for (i = len - 1; i >= 0; i--)
    sc->w = cons(sc, small_int((unsigned int)((unsigned char)(str[i]))), sc->w); /* extra cast is not redundant! */
  p = sc->w;
  sc->w = sc->nil;
  return(p);
}



/* -------------------------------- ports --------------------------------
 *
 * originally nil served as stdin and friends, but that made it impossible to catch an error
 *   like (read-line (current-output-port)) when the latter was stdout.  So we now have
 *   the built-in constant ports *stdin*, *stdout*, and *stderr*.  Some way is needed to
 *   refer to these directly so that (read-line *stdin*) for example can insist on reading
 *   from the terminal, or whatever stdin is.
 */

static s7_pointer g_is_port_closed(s7_scheme *sc, s7_pointer args)
{
  #define H_is_port_closed "(port-closed? p) returns #t if the port p is closed."
  #define Q_is_port_closed pl_bt
  s7_pointer x;

  x = car(args);
  if ((is_input_port(x)) || (is_output_port(x)))
    return(make_boolean(sc, port_is_closed(x)));

  method_or_bust_with_type(sc, x, sc->is_port_closed_symbol, args, make_string_wrapper(sc, "a port"), 0);
}


static s7_pointer c_port_line_number(s7_scheme *sc, s7_pointer x)
{
  if ((!(is_input_port(x))) ||
      (port_is_closed(x)))
    method_or_bust_with_type(sc, x, sc->port_line_number_symbol, list_1(sc, x), an_input_port_string, 0);
  return(make_integer(sc, port_line_number(x)));
}

static s7_pointer g_port_line_number(s7_scheme *sc, s7_pointer args)
{
  #define H_port_line_number "(port-line-number input-file-port) returns the current read line number of port"
  #define Q_port_line_number s7_make_signature(sc, 2, sc->is_integer_symbol, s7_make_signature(sc, 2, sc->is_input_port_symbol, sc->is_null_symbol))

  if ((is_null(args)) || (is_null(car(args))))
    return(c_port_line_number(sc, sc->input_port));
  return(c_port_line_number(sc, car(args)));
}

PF_TO_PF(port_line_number, c_port_line_number)

int s7_port_line_number(s7_pointer p)
{
  if (is_input_port(p))
    return(port_line_number(p));
  return(0);
}

static s7_pointer g_set_port_line_number(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p, line;

  if ((is_null(car(args))) || 
      ((is_null(cdr(args))) && (is_integer(car(args)))))
    p = sc->input_port;
  else 
    {
      p = car(args);
      if (!(is_input_port(p)))
	return(s7_wrong_type_arg_error(sc, "set! port-line-number", 1, p, "an input port"));
    }

  line = (is_null(cdr(args)) ? car(args) : cadr(args));
  if (!is_integer(line))
    return(s7_wrong_type_arg_error(sc, "set! port-line-number", 2, line, "an integer"));
  port_line_number(p) = integer(line);
  return(line);
}


const char *s7_port_filename(s7_pointer x)
{
  if (((is_input_port(x)) ||
       (is_output_port(x))) &&
      (!port_is_closed(x)))
    return(port_filename(x));
  return(NULL);
}


static s7_pointer c_port_filename(s7_scheme *sc, s7_pointer x)
{
  if (((is_input_port(x)) ||
       (is_output_port(x))) &&
      (!port_is_closed(x)))
    {
      if (port_filename(x))
	return(make_string_wrapper_with_length(sc, port_filename(x), port_filename_length(x)));
      return(s7_make_string_with_length(sc, "", 0));
      /* otherwise (eval-string (port-filename)) and (string->symbol (port-filename)) segfault */
    }
  method_or_bust_with_type(sc, x, sc->port_filename_symbol, list_1(sc, x), an_open_port_string, 0);
}

static s7_pointer g_port_filename(s7_scheme *sc, s7_pointer args)
{
  #define H_port_filename "(port-filename file-port) returns the filename associated with port"
  #define Q_port_filename s7_make_signature(sc, 2, sc->is_string_symbol, sc->T)

  if (is_null(args))
    return(c_port_filename(sc, sc->input_port));
  return(c_port_filename(sc, car(args)));
}

PF_TO_PF(port_filename, c_port_filename)


bool s7_is_input_port(s7_scheme *sc, s7_pointer p)
{
  return(is_input_port(p));
}


static s7_pointer g_is_input_port(s7_scheme *sc, s7_pointer args)
{
  #define H_is_input_port "(input-port? p) returns #t if p is an input port"
  #define Q_is_input_port pl_bt
  check_boolean_method(sc, is_input_port, sc->is_input_port_symbol, args);
}


bool s7_is_output_port(s7_scheme *sc, s7_pointer p)
{
  return(is_output_port(p));
}


static s7_pointer g_is_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_is_output_port "(output-port? p) returns #t if p is an output port"
  #define Q_is_output_port pl_bt
  check_boolean_method(sc, is_output_port, sc->is_output_port_symbol, args);
}


s7_pointer s7_current_input_port(s7_scheme *sc)
{
  return(sc->input_port);
}


static s7_pointer g_current_input_port(s7_scheme *sc, s7_pointer args)
{
  #define H_current_input_port "(current-input-port) returns the current input port"
  #define Q_current_input_port s7_make_signature(sc, 1, sc->is_input_port_symbol)
  return(sc->input_port);
}

#if (!WITH_PURE_S7)
static s7_pointer g_set_current_input_port(s7_scheme *sc, s7_pointer args)
{
  #define H_set_current_input_port "(set-current-input-port port) sets the current-input port to port and returns the previous value of the input port"
  #define Q_set_current_input_port s7_make_signature(sc, 2, sc->is_input_port_symbol, sc->is_input_port_symbol)

  s7_pointer old_port, port;
  old_port = sc->input_port;
  port = car(args);
  if ((is_input_port(port)) &&
      (!port_is_closed(port)))
    sc->input_port = port;
  else
    {
      check_method(sc, port, s7_make_symbol(sc, "set-current-input-port"), args);
      return(s7_wrong_type_arg_error(sc, "set-current-input-port", 0, port, "an open input port"));
    }
  return(old_port);
}
#endif

s7_pointer s7_set_current_input_port(s7_scheme *sc, s7_pointer port)
{
  s7_pointer old_port;
  old_port = sc->input_port;
  sc->input_port = port;
  return(old_port);
}


s7_pointer s7_current_output_port(s7_scheme *sc)
{
  return(sc->output_port);
}


s7_pointer s7_set_current_output_port(s7_scheme *sc, s7_pointer port)
{
  s7_pointer old_port;
  old_port = sc->output_port;
  sc->output_port = port;
  return(old_port);
}


static s7_pointer g_current_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_current_output_port "(current-output-port) returns the current output port"
  #define Q_current_output_port s7_make_signature(sc, 1, sc->is_output_port_symbol)
  return(sc->output_port);
}

#if (!WITH_PURE_S7)
static s7_pointer g_set_current_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_set_current_output_port "(set-current-output-port port) sets the current-output port to port and returns the previous value of the output port"
  #define Q_set_current_output_port s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->is_output_port_symbol)

  s7_pointer old_port, port;
  old_port = sc->output_port;
  port = car(args);
  if (((is_output_port(port)) &&
       (!port_is_closed(port))) ||
      (port == sc->F))
    sc->output_port = port;
  else
    {
      check_method(sc, port, s7_make_symbol(sc, "set-current-output-port"), args);
      return(s7_wrong_type_arg_error(sc, "set-current-output-port", 0, port, "an open output port"));
    }
  return(old_port);
}
#endif

s7_pointer s7_current_error_port(s7_scheme *sc)
{
  return(sc->error_port);
}


s7_pointer s7_set_current_error_port(s7_scheme *sc, s7_pointer port)
{
  s7_pointer old_port;
  old_port = sc->error_port;
  sc->error_port = port;
  return(old_port);
}


static s7_pointer g_current_error_port(s7_scheme *sc, s7_pointer args)
{
  #define H_current_error_port "(current-error-port) returns the current error port"
  #define Q_current_error_port s7_make_signature(sc, 1, sc->is_output_port_symbol)
  return(sc->error_port);
}


static s7_pointer g_set_current_error_port(s7_scheme *sc, s7_pointer args)
{
  #define H_set_current_error_port "(set-current-error-port port) sets the current-error port to port and returns the previous value of the error port"
  #define Q_set_current_error_port s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->is_output_port_symbol)
  s7_pointer old_port, port;

  old_port = sc->error_port;
  port = car(args);
  if (((is_output_port(port)) &&
       (!port_is_closed(port))) ||
      (port == sc->F))
    sc->error_port = port;
  else
    {
      check_method(sc, port, s7_make_symbol(sc, "set-current-error-port"), args);
      return(s7_wrong_type_arg_error(sc, "set-current-error-port", 0, port, "an open output port"));
    }
  return(old_port);
}


#if (!WITH_PURE_S7)
static s7_pointer g_is_char_ready(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_ready "(char-ready? (port (current-input-port))) returns #t if a character is ready for input on the given port"
  #define Q_is_char_ready s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_input_port_symbol)
  if (is_not_null(args))
    {
      s7_pointer pt = car(args);
      if (!is_input_port(pt))
	method_or_bust_with_type(sc, pt, sc->is_char_ready_symbol, args, an_input_port_string, 0);
      if (port_is_closed(pt))
	return(simple_wrong_type_argument_with_type(sc, sc->is_char_ready_symbol, pt, an_open_port_string));

      if (is_function_port(pt))
	return((*(port_input_function(pt)))(sc, S7_IS_CHAR_READY, pt));
      return(make_boolean(sc, is_string_port(pt)));
    }
  return(make_boolean(sc, (is_input_port(sc->input_port)) && (is_string_port(sc->input_port))));
}
#endif


static s7_pointer g_is_eof_object(s7_scheme *sc, s7_pointer args)
{
  #define H_is_eof_object "(eof-object? val) returns #t if val is the end-of-file object"
  #define Q_is_eof_object pl_bt
  check_boolean_method(sc, is_eof, sc->is_eof_object_symbol, args);
}


static int closed_port_read_char(s7_scheme *sc, s7_pointer port);
static s7_pointer closed_port_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied);
static void closed_port_write_char(s7_scheme *sc, int c, s7_pointer port);
static void closed_port_write_string(s7_scheme *sc, const char *str, int len, s7_pointer port);
static void closed_port_display(s7_scheme *sc, const char *s, s7_pointer port);

void s7_close_input_port(s7_scheme *sc, s7_pointer p)
{
#if DEBUGGING
  if (!is_input_port(p))
    fprintf(stderr, "s7_close_input_port: %s\n", DISPLAY(p));
#endif
  if ((is_immutable_port(p)) ||
      ((is_input_port(p)) && (port_is_closed(p))))
    return;

  if (port_filename(p))
    {
      free(port_filename(p));
      port_filename(p) = NULL;
    }

  if (is_file_port(p))
    {
      if (port_file(p))
	{
	  fclose(port_file(p));
	  port_file(p) = NULL;
	}
    }
  else
    {
      if ((is_string_port(p)) &&
	  (port_gc_loc(p) != -1))
	s7_gc_unprotect_at(sc, port_gc_loc(p));
    }
  if (port_needs_free(p))
    {
      if (port_data(p))
	{
	  free(port_data(p));
	  port_data(p) = NULL;
	  port_data_size(p) = 0;
	}
      port_needs_free(p) = false;
    }

  port_read_character(p) = closed_port_read_char;
  port_read_line(p) = closed_port_read_line;
  port_write_character(p) = closed_port_write_char;
  port_write_string(p) = closed_port_write_string;
  port_display(p) = closed_port_display;
  port_is_closed(p) = true;
}


static s7_pointer c_close_input_port(s7_scheme *sc, s7_pointer pt)
{
  if (!is_input_port(pt))
    method_or_bust_with_type(sc, pt, sc->close_input_port_symbol, set_plist_1(sc, pt), an_input_port_string, 0);
  if (!is_immutable_port(pt))
    s7_close_input_port(sc, pt);
  return(sc->unspecified);
}

static s7_pointer g_close_input_port(s7_scheme *sc, s7_pointer args)
{
  #define H_close_input_port "(close-input-port port) closes the port"
  #define Q_close_input_port s7_make_signature(sc, 2, sc->is_unspecified_symbol, sc->is_input_port_symbol)
  return(c_close_input_port(sc, car(args)));
}

PF_TO_PF(close_input_port, c_close_input_port)


void s7_flush_output_port(s7_scheme *sc, s7_pointer p)
{
  if ((!is_output_port(p)) ||
      (!is_file_port(p)) ||
      (port_is_closed(p)) ||
      (p == sc->F))
    return;

  if (port_file(p))
    {
      if (port_position(p) > 0)
	{
	  if (fwrite((void *)(port_data(p)), 1, port_position(p), port_file(p)) != port_position(p))
	    s7_warn(sc, 64, "fwrite trouble in flush-output-port\n");
	  port_position(p) = 0;
	}
      fflush(port_file(p));
    }
}


static s7_pointer g_flush_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_flush_output_port "(flush-output-port port) flushes the port"
  #define Q_flush_output_port s7_make_signature(sc, 2, sc->T, sc->is_output_port_symbol)
  s7_pointer pt;

  if (is_null(args))
    pt = sc->output_port;
  else pt = car(args);

  if (!is_output_port(pt))
    {
      if (pt == sc->F) return(pt);
      method_or_bust_with_type(sc, pt, sc->flush_output_port_symbol, args, an_output_port_string, 0);
    }
  s7_flush_output_port(sc, pt);
  return(pt);
}

static s7_pointer c_flush_output_port(s7_scheme *sc) {return(g_flush_output_port(sc, sc->nil));}
PF_0(flush_output_port, c_flush_output_port)

static void close_output_port(s7_scheme *sc, s7_pointer p)
{
  if (is_file_port(p))
    {
      if (port_filename(p)) /* only a file (output) port has a filename */
	{
	  free(port_filename(p));
	  port_filename(p) = NULL;
	  port_filename_length(p) = 0;
	}

      if (port_file(p))
	{
	  if (port_position(p) > 0)
	    {
	      if (fwrite((void *)(port_data(p)), 1, port_position(p), port_file(p)) != port_position(p))
		s7_warn(sc, 64, "fwrite trouble in close-output-port\n");
	      port_position(p) = 0;
	    }
	  free(port_data(p));
	  fflush(port_file(p));
	  fclose(port_file(p));
	  port_file(p) = NULL;
	}
    }
  else
    {
      if ((is_string_port(p)) &&
	  (port_data(p)))
	{
	  free(port_data(p));
	  port_data(p) = NULL;
	  port_data_size(p) = 0;
	  port_needs_free(p) = false;
	}
    }
  port_read_character(p) = closed_port_read_char;
  port_read_line(p) = closed_port_read_line;
  port_write_character(p) = closed_port_write_char;
  port_write_string(p) = closed_port_write_string;
  port_display(p) = closed_port_display;
  port_is_closed(p) = true;
}

void s7_close_output_port(s7_scheme *sc, s7_pointer p)
{
  if ((is_immutable_port(p)) ||
      ((is_output_port(p)) && (port_is_closed(p))) ||
      (p == sc->F))
    return;
  close_output_port(sc, p);
}


static s7_pointer c_close_output_port(s7_scheme *sc, s7_pointer pt)
{
  if (!is_output_port(pt))
    {
      if (pt == sc->F) return(sc->unspecified);
      method_or_bust_with_type(sc, pt, sc->close_output_port_symbol, set_plist_1(sc, pt), an_output_port_string, 0);
    }
  if (!(is_immutable_port(pt)))
    s7_close_output_port(sc, pt);
  return(sc->unspecified);
}

static s7_pointer g_close_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_close_output_port "(close-output-port port) closes the port"
  #define Q_close_output_port s7_make_signature(sc, 2, sc->is_unspecified_symbol, sc->is_output_port_symbol)
  return(c_close_output_port(sc, car(args)));
}

PF_TO_PF(close_output_port, c_close_output_port)


/* -------- read character functions -------- */

static int file_read_char(s7_scheme *sc, s7_pointer port)
{
  return(fgetc(port_file(port)));
}


static int function_read_char(s7_scheme *sc, s7_pointer port)
{
  return(character((*(port_input_function(port)))(sc, S7_READ_CHAR, port)));
}


static int string_read_char(s7_scheme *sc, s7_pointer port)
{
  if (port_data_size(port) <= port_position(port)) /* port_string_length is 0 if no port string */
    return(EOF);
  return((unsigned char)port_data(port)[port_position(port)++]);
}


static int output_read_char(s7_scheme *sc, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->read_char_symbol, port, an_input_port_string);
  return(0);
}


static int closed_port_read_char(s7_scheme *sc, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->read_char_symbol, port, an_open_port_string);
  return(0);
}



/* -------- read line functions -------- */

static s7_pointer output_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  return(simple_wrong_type_argument_with_type(sc, sc->read_line_symbol, port, an_input_port_string));
}


static s7_pointer closed_port_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  return(simple_wrong_type_argument_with_type(sc, sc->read_line_symbol, port, an_open_port_string));
}


static s7_pointer function_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  return((*(port_input_function(port)))(sc, S7_READ_LINE, port));
}


static s7_pointer stdin_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  if (!sc->read_line_buf)
    {
      sc->read_line_buf_size = 1024;
      sc->read_line_buf = (char *)malloc(sc->read_line_buf_size * sizeof(char));
    }

  if (fgets(sc->read_line_buf, sc->read_line_buf_size, stdin))
    return(s7_make_string(sc, sc->read_line_buf)); /* fgets adds the trailing '\0' */
  return(s7_make_string_with_length(sc, NULL, 0));
}


static s7_pointer file_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  char *buf;
  int read_size, previous_size = 0;

  if (!sc->read_line_buf)
    {
      sc->read_line_buf_size = 1024;
      sc->read_line_buf = (char *)malloc(sc->read_line_buf_size * sizeof(char));
    }

  buf = sc->read_line_buf;
  read_size = sc->read_line_buf_size;

  while (true)
    {
      char *p, *rtn;
      size_t len;

      p = fgets(buf, read_size, port_file(port));
      if (!p)
	return(sc->eof_object);

      rtn = strchr(buf, (int)'\n');
      if (rtn)
	{
	  port_line_number(port)++;
	  return(s7_make_string_with_length(sc, sc->read_line_buf, (with_eol) ? (previous_size + rtn - p + 1) : (previous_size + rtn - p)));
	}
      /* if no newline, then either at eof or need bigger buffer */
      len = strlen(sc->read_line_buf);

      if ((len + 1) < sc->read_line_buf_size)
	return(s7_make_string_with_length(sc, sc->read_line_buf, len));

      previous_size = sc->read_line_buf_size;
      sc->read_line_buf_size *= 2;
      sc->read_line_buf = (char *)realloc(sc->read_line_buf, sc->read_line_buf_size * sizeof(char));
      read_size = previous_size;
      previous_size -= 1;
      buf = (char *)(sc->read_line_buf + previous_size);
    }
  return(sc->eof_object);
}


static s7_pointer string_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  unsigned int i, port_start;
  unsigned char *port_str, *cur, *start;

  port_start = port_position(port);
  port_str = port_data(port);
  start = (unsigned char *)(port_str + port_start);

  cur = (unsigned char *)strchr((const char *)start, (int)'\n'); /* this can run off the end making valgrind unhappy, but I think it's innocuous */
  if (cur)
      {
	port_line_number(port)++;
	i = cur - port_str;
	port_position(port) = i + 1;
	if (copied)
	  return(s7_make_string_with_length(sc, (const char *)start, ((with_eol) ? i + 1 : i) - port_start));
	return(make_string_wrapper_with_length(sc, (char *)start, ((with_eol) ? i + 1 : i) - port_start));
      }
  i = port_data_size(port);
  port_position(port) = i;
  if (i <= port_start)         /* the < part can happen -- if not caught we try to create a string of length -1 -> segfault */
    return(sc->eof_object);

  if (copied)
    return(s7_make_string_with_length(sc, (const char *)start, i - port_start));
  return(make_string_wrapper_with_length(sc, (char *)start, i - port_start));
}


/* -------- write character functions -------- */

static void resize_port_data(s7_pointer pt, int new_size)
{
  int loc;
  loc = port_data_size(pt);
  port_data_size(pt) = new_size;
  port_data(pt) = (unsigned char *)realloc(port_data(pt), new_size * sizeof(unsigned char));
  memclr((void *)(port_data(pt) + loc), new_size - loc);
}

static void string_write_char(s7_scheme *sc, int c, s7_pointer pt)
{
  if (port_position(pt) >= port_data_size(pt))
    resize_port_data(pt, port_data_size(pt) * 2);
  port_data(pt)[port_position(pt)++] = c;
}

static void stdout_write_char(s7_scheme *sc, int c, s7_pointer port)
{
  fputc(c, stdout);
}

static void stderr_write_char(s7_scheme *sc, int c, s7_pointer port)
{
  fputc(c, stderr);
}

static void function_write_char(s7_scheme *sc, int c, s7_pointer port)
{
  (*(port_output_function(port)))(sc, c, port);
}


#define PORT_DATA_SIZE 256
static void file_write_char(s7_scheme *sc, int c, s7_pointer port)
{
  if (port_position(port) == PORT_DATA_SIZE)
    {
      if (fwrite((void *)(port_data(port)), 1, PORT_DATA_SIZE, port_file(port)) != PORT_DATA_SIZE)
	s7_warn(sc, 64, "fwrite trouble during write-char\n");
      port_position(port) = 0;
    }
  port_data(port)[port_position(port)++] = (unsigned char)c;
}


static void input_write_char(s7_scheme *sc, int c, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_char_symbol, port, an_output_port_string);
}


static void closed_port_write_char(s7_scheme *sc, int c, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_char_symbol, port, an_open_port_string);
}



/* -------- write string functions -------- */

static void input_write_string(s7_scheme *sc, const char *str, int len, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_symbol, port, an_output_port_string);
}


static void closed_port_write_string(s7_scheme *sc, const char *str, int len, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_symbol, port, an_open_port_string);
}


static void input_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_symbol, port, an_output_port_string);
}

static void closed_port_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_symbol, port, an_open_port_string);
}

static void stdout_write_string(s7_scheme *sc, const char *str, int len, s7_pointer port)
{
  if (str[len] == '\0')
    fputs(str, stdout);
  else
    {
      int i;
      for (i = 0; i < len; i++)
	fputc(str[i], stdout);
    }
}

static void stderr_write_string(s7_scheme *sc, const char *str, int len, s7_pointer port)
{
  if (str[len] == '\0')
    fputs(str, stderr);
  else
    {
      int i;
      for (i = 0; i < len; i++)
	fputc(str[i], stderr);
    }
}

static void string_write_string(s7_scheme *sc, const char *str, int len, s7_pointer pt)
{
  int new_len;  /* len is known to be non-zero */

  new_len = port_position(pt) + len;
  if (new_len >= (int)port_data_size(pt))
    resize_port_data(pt, new_len * 2);

  memcpy((void *)(port_data(pt) + port_position(pt)), (void *)str, len);
  /* memcpy is much faster than the equivalent while loop */
  port_position(pt) = new_len;
}


static s7_pointer write_string_chooser(s7_scheme *sc, s7_pointer f, int args, s7_pointer expr)
{
  check_for_substring_temp(sc, expr);
  return(f);
}


static void file_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s)
    {
      if (port_position(port) > 0)
	{
	  if (fwrite((void *)(port_data(port)), 1, port_position(port), port_file(port)) != port_position(port))
	    s7_warn(sc, 64, "fwrite trouble in display\n");
	  port_position(port) = 0;
	}
      if (fputs(s, port_file(port)) == EOF)
	s7_warn(sc, 64, "write to %s: %s\n", port_filename(port), strerror(errno));
    }
}

static void file_write_string(s7_scheme *sc, const char *str, int len, s7_pointer pt)
{
  int new_len;
  new_len = port_position(pt) + len;
  if (new_len >= PORT_DATA_SIZE)
    {
      if (port_position(pt) > 0)
	{
	  if (fwrite((void *)(port_data(pt)), 1, port_position(pt), port_file(pt)) != port_position(pt))
	    s7_warn(sc, 64, "fwrite trouble in write-string\n");
	  port_position(pt) = 0;
	}
      if (fwrite((void *)str, 1, len, port_file(pt)) != (size_t)len)
	s7_warn(sc, 64, "fwrite trouble in write-string\n");
    }
  else
    {
      memcpy((void *)(port_data(pt) + port_position(pt)), (void *)str, len);
      port_position(pt) = new_len;
    }
}

static void string_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s)
    string_write_string(sc, s, safe_strlen(s), port);
}


static void function_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s)
    {
      for (; *s; s++)
	(*(port_output_function(port)))(sc, *s, port);
    }
}

static void function_write_string(s7_scheme *sc, const char *str, int len, s7_pointer pt)
{
  int i;
  for (i = 0; i < len; i++)
    (*(port_output_function(pt)))(sc, str[i], pt);
}

static void stdout_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s) fputs(s, stdout);
}


static void stderr_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s) fputs(s, stderr);
}


static s7_pointer g_write_string(s7_scheme *sc, s7_pointer args)
{
  #define H_write_string "(write-string str port start end) writes str to port."
  #define Q_write_string s7_make_circular_signature(sc, 3, 4, sc->is_string_symbol, sc->is_string_symbol, sc->is_output_port_symbol, sc->is_integer_symbol)
  s7_pointer str, port;
  s7_int start = 0, end;

  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->write_string_symbol, args, T_STRING, 1);

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      s7_pointer inds;
      port = cadr(args);
      inds = cddr(args);
      if (!is_null(inds))
	{
	  s7_pointer p;
	  p = start_and_end(sc, sc->write_string_symbol, NULL, inds, args, 3, &start, &end);
	  if (p != sc->gc_nil) return(p);
	}
    }
  else port = sc->output_port;
  if (!is_output_port(port))
    {
      if (port == sc->F)
	{
	  s7_pointer x;
	  int len;
	  if ((start == 0) && (end == string_length(str)))
	    return(str);
	  len = (int)(end - start);
	  x = s7_make_string_with_length(sc, (char *)(string_value(str) + start), len);
	  string_value(x)[len] = 0;
	  return(x);
	}
      method_or_bust_with_type(sc, port, sc->write_string_symbol, args, an_output_port_string, 2);
    }

  if (start == 0)
    port_write_string(port)(sc, string_value(str), end, port);
  else port_write_string(port)(sc, (char *)(string_value(str) + start), (end - start), port);
  return(str);
}

static s7_pointer c_write_string(s7_scheme *sc, s7_pointer x) {return(g_write_string(sc, set_plist_1(sc, x)));}
PF_TO_PF(write_string, c_write_string)



/* -------- skip to newline readers -------- */

static token_t file_read_semicolon(s7_scheme *sc, s7_pointer pt)
{
  int c;
  do (c = fgetc(port_file(pt))); while ((c != '\n') && (c != EOF));
  port_line_number(pt)++;
  if (c == EOF)
    return(TOKEN_EOF);
  return(token(sc));
}


static token_t string_read_semicolon(s7_scheme *sc, s7_pointer pt)
{
  const char *orig_str, *str;
  str = (const char *)(port_data(pt) + port_position(pt));
  orig_str = strchr(str, (int)'\n');
  if (!orig_str)
    {
      port_position(pt) = port_data_size(pt);
      return(TOKEN_EOF);
    }
  port_position(pt) += (orig_str - str + 1); /* + 1 because strchr leaves orig_str pointing at the newline */
  port_line_number(pt)++;
  return(token(sc));
}


/* -------- white space readers -------- */

static int file_read_white_space(s7_scheme *sc, s7_pointer port)
{
  int c;
  while (is_white_space(c = fgetc(port_file(port))))
    if (c == '\n')
      port_line_number(port)++;
  return(c);
}


static int terminated_string_read_white_space(s7_scheme *sc, s7_pointer pt)
{
  const unsigned char *str;
  unsigned char c;
  /* here we know we have null termination and white_space[#\null] is false.
   */
  str = (const unsigned char *)(port_data(pt) + port_position(pt));

  while (white_space[c = *str++]) /* (let ((a 1)) a) -- 255 is not -1 = EOF */
    if (c == '\n')
      port_line_number(pt)++;
  if (c)
    port_position(pt) = str - port_data(pt);
  else port_position(pt) = port_data_size(pt);
  return((int)c);
}


/* name (alphanumeric token) readers */

static void resize_strbuf(s7_scheme *sc, unsigned int needed_size)
{
  unsigned int i, old_size;
  old_size = sc->strbuf_size;
  while (sc->strbuf_size <= needed_size) sc->strbuf_size *= 2;
  sc->strbuf = (char *)realloc(sc->strbuf, sc->strbuf_size * sizeof(char));
  for (i = old_size; i < sc->strbuf_size; i++) sc->strbuf[i] = '\0';
}


static s7_pointer file_read_name_or_sharp(s7_scheme *sc, s7_pointer pt, bool atom_case)
{
  int c;
  unsigned int i = 1;
  /* sc->strbuf[0] has the first char of the string we're reading */

  do {
    c = fgetc(port_file(pt)); /* might return EOF */
    if (c == '\n')
      port_line_number(pt)++;

    sc->strbuf[i++] = c;
    if (i >= sc->strbuf_size)
      resize_strbuf(sc, i);
  } while ((c != EOF) && (char_ok_in_a_name[c]));

  if ((i == 2) &&
      (sc->strbuf[0] == '\\'))
    sc->strbuf[2] = '\0';
  else
    {
      if (c != EOF)
	{
	  if (c == '\n')
	    port_line_number(pt)--;
	  ungetc(c, port_file(pt));
	}
      sc->strbuf[i - 1] = '\0';
    }

  if (atom_case)
    return(make_atom(sc, sc->strbuf, BASE_10, SYMBOL_OK, WITH_OVERFLOW_ERROR));

  return(make_sharp_constant(sc, sc->strbuf, UNNESTED_SHARP, BASE_10, WITH_OVERFLOW_ERROR));
}

static s7_pointer file_read_name(s7_scheme *sc, s7_pointer pt)
{
  return(file_read_name_or_sharp(sc, pt, true));
}

static s7_pointer file_read_sharp(s7_scheme *sc, s7_pointer pt)
{
  return(file_read_name_or_sharp(sc, pt, false));
}


static s7_pointer string_read_name_no_free(s7_scheme *sc, s7_pointer pt)
{
  /* sc->strbuf[0] has the first char of the string we're reading */
  unsigned int k;
  char *str, *orig_str;

  str = (char *)(port_data(pt) + port_position(pt));

  if (!char_ok_in_a_name[(unsigned char)*str])
    {
      s7_pointer result;
      result = sc->singletons[(unsigned char)(sc->strbuf[0])];
      if (!result)
	{
	  sc->strbuf[1] = '\0';
	  result = make_symbol_with_length(sc, sc->strbuf, 1);
	  sc->singletons[(unsigned char)(sc->strbuf[0])] = result;
	}
      return(result);
    }

  orig_str = (char *)(str - 1);
  str++;
  while (char_ok_in_a_name[(unsigned char)(*str)]) {str++;}
  k = str - orig_str;
  if (*str != 0)
    port_position(pt) += (k - 1);
  else port_position(pt) = port_data_size(pt);

  /* this is equivalent to:
   *    str = strpbrk(str, "(); \"\t\r\n");
   *    if (!str)
   *      {
   *        k = strlen(orig_str);
   *        str = (char *)(orig_str + k);
   *      }
   *    else k = str - orig_str;
   * but slightly faster.
   */

  if (!number_table[(unsigned char)(*orig_str)])
    return(make_symbol_with_length(sc, orig_str, k));

  /* eval_c_string string is a constant so we can't set and unset the token's end char */
  if ((k + 1) >= sc->strbuf_size)
    resize_strbuf(sc, k + 1);

  memcpy((void *)(sc->strbuf), (void *)orig_str, k);
  sc->strbuf[k] = '\0';
  return(make_atom(sc, sc->strbuf, BASE_10, SYMBOL_OK, WITH_OVERFLOW_ERROR));
}


static s7_pointer string_read_sharp(s7_scheme *sc, s7_pointer pt)
{
  /* sc->strbuf[0] has the first char of the string we're reading.
   *   since a *#readers* function might want to get further input, we can't mess with the input even when it is otherwise safe
   */
  unsigned int k;
  char *orig_str, *str;

  str = (char *)(port_data(pt) + port_position(pt));

  if (!char_ok_in_a_name[(unsigned char)*str])
    {
      if (sc->strbuf[0] == 'f')
	return(sc->F);
      if (sc->strbuf[0] == 't')
	return(sc->T);
      if (sc->strbuf[0] == '\\')
	{
	  /* must be from #\( and friends -- a character that happens to be not ok-in-a-name */
	  sc->strbuf[1] = str[0];
	  sc->strbuf[2] = '\0';
	  port_position(pt)++;
	}
      else sc->strbuf[1] = '\0';
      return(make_sharp_constant(sc, sc->strbuf, UNNESTED_SHARP, BASE_10, WITH_OVERFLOW_ERROR));
    }

  orig_str = (char *)(str - 1);
  str++;
  while (char_ok_in_a_name[(unsigned char)(*str)]) {str++;}
  k = str - orig_str;
  if (*str != 0)
    port_position(pt) += (k - 1);
  else port_position(pt) += k;

  if ((k + 1) >= sc->strbuf_size)
    resize_strbuf(sc, k + 1);

  memcpy((void *)(sc->strbuf), (void *)orig_str, k);
  sc->strbuf[k] = '\0';
  return(make_sharp_constant(sc, sc->strbuf, UNNESTED_SHARP, BASE_10, WITH_OVERFLOW_ERROR));
}


static s7_pointer string_read_name(s7_scheme *sc, s7_pointer pt)
{
  /* port_string was allocated (and read from a file) so we can mess with it directly */
  s7_pointer result;
  unsigned int k;
  char *orig_str, *str;
  char endc;

  str = (char *)(port_data(pt) + port_position(pt));
  if (!char_ok_in_a_name[(unsigned char)*str])
    {
      s7_pointer result;
      result = sc->singletons[(unsigned char)(sc->strbuf[0])];
      if (!result)
	{
	  sc->strbuf[1] = '\0';
	  result = make_symbol_with_length(sc, sc->strbuf, 1);
	  sc->singletons[(unsigned char)(sc->strbuf[0])] = result;
	}
      return(result);
    }

  orig_str = (char *)(str - 1);
  str++;
  while (char_ok_in_a_name[(unsigned char)(*str)]) {str++;}
  k = str - orig_str;
  if (*str != 0)
    port_position(pt) += (k - 1);
  else port_position(pt) = port_data_size(pt);

  if (!number_table[(unsigned char)(*orig_str)])
    return(make_symbol_with_length(sc, orig_str, k));

  endc = (*str);
  (*str) = '\0';
  result = make_atom(sc, orig_str, BASE_10, SYMBOL_OK, WITH_OVERFLOW_ERROR);
  (*str) = endc;
  return(result);
}


static s7_pointer read_file(s7_scheme *sc, FILE *fp, const char *name, long max_size, const char *caller)
{
  s7_pointer port;
#ifndef _MSC_VER
  long size;
#endif
  unsigned int port_loc;

  new_cell(sc, port, T_INPUT_PORT);
  port_loc = s7_gc_protect(sc, port);
  port_port(port) = alloc_port(sc);
  port_is_closed(port) = false;
  port_original_input_string(port) = sc->nil;
  port_write_character(port) = input_write_char;
  port_write_string(port) = input_write_string;

  /* if we're constantly opening files, and each open saves the file name in permanent
   *   memory, we gradually core-up.
   */
  port_filename_length(port) = safe_strlen(name);
  port_filename(port) = copy_string_with_length(name, port_filename_length(port));
  port_line_number(port) = 1;  /* first line is numbered 1 */
  add_input_port(sc, port);

#ifndef _MSC_VER
  /* this doesn't work in MS C */
  fseek(fp, 0, SEEK_END);
  size = ftell(fp);
  rewind(fp);

  /* pseudo files (under /proc for example) have size=0, but we can read them, so don't assume a 0 length file is empty
   */

  if ((size > 0) &&                /* if (size != 0) we get (open-input-file "/dev/tty") -> (open "/dev/tty") read 0 bytes of an expected -1? */
      ((max_size < 0) || (size < max_size)))
    {
      size_t bytes;
      unsigned char *content;

      content = (unsigned char *)malloc((size + 2) * sizeof(unsigned char));
      bytes = fread(content, sizeof(unsigned char), size, fp);
      if (bytes != (size_t)size)
	{
	  char tmp[256];
	  int len;
	  len = snprintf(tmp, 256, "(%s \"%s\") read %ld bytes of an expected %ld?", caller, name, (long)bytes, size);
	  port_write_string(sc->output_port)(sc, tmp, len, sc->output_port);
	  size = bytes;
	}
      content[size] = '\0';
      content[size + 1] = '\0';
      fclose(fp);

      port_type(port) = STRING_PORT;
      port_data(port) = content;
      port_data_size(port) = size;
      port_position(port) = 0;
      port_needs_free(port) = true;
      port_gc_loc(port) = -1;
      port_read_character(port) = string_read_char;
      port_read_line(port) = string_read_line;
      port_display(port) = input_display;
      port_read_semicolon(port) = string_read_semicolon;
      port_read_white_space(port) = terminated_string_read_white_space;
      port_read_name(port) = string_read_name;
      port_read_sharp(port) = string_read_sharp;
    }
  else
    {
      port_file(port) = fp;
      port_type(port) = FILE_PORT;
      port_needs_free(port) = false;
      port_read_character(port) = file_read_char;
      port_read_line(port) = file_read_line;
      port_display(port) = input_display;
      port_read_semicolon(port) = file_read_semicolon;
      port_read_white_space(port) = file_read_white_space;
      port_read_name(port) = file_read_name;
      port_read_sharp(port) = file_read_sharp; /* was string_read_sharp?? */
    }
#else
  /* _stat64 is no better than the fseek/ftell route, and
   *    GetFileSizeEx and friends requires Windows.h which makes hash of everything else.
   *    fread until done takes too long on big files, so use a file port
   */
  port_file(port) = fp;
  port_type(port) = FILE_PORT;
  port_needs_free(port) = false;
  port_read_character(port) = file_read_char;
  port_read_line(port) = file_read_line;
  port_display(port) = input_display;
  port_read_semicolon(port) = file_read_semicolon;
  port_read_white_space(port) = file_read_white_space;
  port_read_name(port) = file_read_name;
  port_read_sharp(port) = file_read_sharp;
#endif

  s7_gc_unprotect_at(sc, port_loc);
  return(port);
}


static s7_pointer make_input_file(s7_scheme *sc, const char *name, FILE *fp)
{
  #define MAX_SIZE_FOR_STRING_PORT 5000000
  return(read_file(sc, fp, name, MAX_SIZE_FOR_STRING_PORT, "open"));
}

#if (!MS_WINDOWS)
#include <sys/stat.h>
#endif

static bool is_directory(const char *filename)
{
#if (!MS_WINDOWS)
  #ifdef S_ISDIR
    struct stat statbuf;
    return((stat(filename, &statbuf) >= 0) &&
	   (S_ISDIR(statbuf.st_mode)));
  #endif
#endif
  return(false);
}


static s7_pointer open_input_file_1(s7_scheme *sc, const char *name, const char *mode, const char *caller)
{
  FILE *fp;
  /* see if we can open this file before allocating a port */

  if (is_directory(name))
    return(file_error(sc, caller, "is a directory", name));

  errno = 0;
  fp = fopen(name, mode);
  if (!fp)
    {
#if (!MS_WINDOWS)
      if (errno == EINVAL)
	return(file_error(sc, caller, "invalid mode", mode));
  #if WITH_GCC
      /* catch one special case, "~/..." */
      if ((name[0] == '~') &&
	  (name[1] == '/'))
	{
	  char *home;
	  home = getenv("HOME");
	  if (home)
	    {
	      char *filename;
	      int len;
	      len = safe_strlen(name) + safe_strlen(home) + 1;
	      tmpbuf_malloc(filename, len);
	      snprintf(filename, len, "%s%s", home, (char *)(name + 1));
	      fp = fopen(filename, "r");
	      tmpbuf_free(filename, len);
	      if (fp)
		return(make_input_file(sc, name, fp));
	    }
	}
  #endif
#endif
      return(file_error(sc, caller, strerror(errno), name));
    }
  return(make_input_file(sc, name, fp));
}


s7_pointer s7_open_input_file(s7_scheme *sc, const char *name, const char *mode)
{
  return(open_input_file_1(sc, name, mode, "open-input-file"));
}


static s7_pointer g_open_input_file(s7_scheme *sc, s7_pointer args)
{
  #define H_open_input_file "(open-input-file filename (mode \"r\")) opens filename for reading"
  #define Q_open_input_file s7_make_signature(sc, 3, sc->is_input_port_symbol, sc->is_string_symbol, sc->is_string_symbol)
  s7_pointer name = car(args);

  if (!is_string(name))
    method_or_bust(sc, name, sc->open_input_file_symbol, args, T_STRING, 1);
  /* what if the file name is a byte-vector? currently we accept it */

  if (is_pair(cdr(args)))
    {
      s7_pointer mode;
      mode = cadr(args);
      if (!is_string(mode))
	method_or_bust_with_type(sc, mode, sc->open_input_file_symbol, args, make_string_wrapper(sc, "a string (a mode such as \"r\")"), 2);
      /* since scheme allows embedded nulls, dumb stuff is accepted here: (open-input-file file "a\x00b") -- should this be an error? */
      return(open_input_file_1(sc, string_value(name), string_value(mode), "open-input-file"));
    }
  return(open_input_file_1(sc, string_value(name), "r", "open-input-file"));
}


static void make_standard_ports(s7_scheme *sc)
{
  s7_pointer x;

  /* standard output */
  x = alloc_pointer();
  unheap(x);
  set_type(x, T_OUTPUT_PORT | T_IMMUTABLE);
  port_port(x) = (port_t *)calloc(1, sizeof(port_t));
  port_type(x) = FILE_PORT;
  port_data(x) = NULL;
  port_is_closed(x) = false;
  port_filename_length(x) = 8;
  port_filename(x) = copy_string_with_length("*stdout*", 8);
  port_file_number(x) = remember_file_name(sc, port_filename(x)); /* these numbers need to be correct for the evaluator (__FUNC__ data) */
  port_line_number(x) = 0;
  port_file(x) = stdout;
  port_needs_free(x) = false;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = stdout_display;
  port_write_character(x) = stdout_write_char;
  port_write_string(x) = stdout_write_string;
  sc->standard_output = x;

  /* standard error */
  x = alloc_pointer();
  unheap(x);
  set_type(x, T_OUTPUT_PORT | T_IMMUTABLE);
  port_port(x) = (port_t *)calloc(1, sizeof(port_t));
  port_type(x) = FILE_PORT;
  port_data(x) = NULL;
  port_is_closed(x) = false;
  port_filename_length(x) = 8;
  port_filename(x) = copy_string_with_length("*stderr*", 8);
  port_file_number(x) = remember_file_name(sc, port_filename(x));
  port_line_number(x) = 0;
  port_file(x) = stderr;
  port_needs_free(x) = false;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = stderr_display;
  port_write_character(x) = stderr_write_char;
  port_write_string(x) = stderr_write_string;
  sc->standard_error = x;

  /* standard input */
  x = alloc_pointer();
  unheap(x);
  set_type(x, T_INPUT_PORT | T_IMMUTABLE);
  port_port(x) = (port_t *)calloc(1, sizeof(port_t));
  port_type(x) = FILE_PORT;
  port_is_closed(x) = false;
  port_original_input_string(x) = sc->nil;
  port_filename_length(x) = 7;
  port_filename(x) = copy_string_with_length("*stdin*", 7);
  port_file_number(x) = remember_file_name(sc, port_filename(x));
  port_line_number(x) = 0;
  port_file(x) = stdin;
  port_needs_free(x) = false;
  port_read_character(x) = file_read_char;
  port_read_line(x) = stdin_read_line;
  port_display(x) = input_display;
  port_read_semicolon(x) = file_read_semicolon;
  port_read_white_space(x) = file_read_white_space;
  port_read_name(x) = file_read_name;
  port_read_sharp(x) = file_read_sharp;
  port_write_character(x) = input_write_char;
  port_write_string(x) = input_write_string;
  sc->standard_input = x;

  s7_define_constant(sc, "*stdin*", sc->standard_input);
  s7_define_constant(sc, "*stdout*", sc->standard_output);
  s7_define_constant(sc, "*stderr*", sc->standard_error);

  sc->input_port = sc->standard_input;
  sc->output_port = sc->standard_output;
  sc->error_port = sc->standard_error;
  sc->current_file = NULL;
  sc->current_line = -1;
}


s7_pointer s7_open_output_file(s7_scheme *sc, const char *name, const char *mode)
{
  FILE *fp;
  s7_pointer x;
  /* see if we can open this file before allocating a port */

  errno = 0;
  fp = fopen(name, mode);
  if (!fp)
    {
#if (!MS_WINDOWS)
      if (errno == EINVAL)
	return(file_error(sc, "open-output-file", "invalid mode", mode));
#endif
      return(file_error(sc, "open-output-file", strerror(errno), name));
    }

  new_cell(sc, x, T_OUTPUT_PORT);
  port_port(x) = alloc_port(sc);
  port_type(x) = FILE_PORT;
  port_is_closed(x) = false;
  port_filename_length(x) = safe_strlen(name);
  port_filename(x) = copy_string_with_length(name, port_filename_length(x));
  port_line_number(x) = 1;
  port_file(x) = fp;
  port_needs_free(x) = false;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = file_display;
  port_write_character(x) = file_write_char;
  port_write_string(x) = file_write_string;
  port_position(x) = 0;
  port_data_size(x) = PORT_DATA_SIZE;
  port_data(x) = (unsigned char *)malloc(PORT_DATA_SIZE); /* was +8? */
  add_output_port(sc, x);
  return(x);
}


static s7_pointer g_open_output_file(s7_scheme *sc, s7_pointer args)
{
  #define H_open_output_file "(open-output-file filename (mode \"w\")) opens filename for writing"
  #define Q_open_output_file s7_make_signature(sc, 3, sc->is_output_port_symbol, sc->is_string_symbol, sc->is_string_symbol)
  s7_pointer name = car(args);

  if (!is_string(name))
    method_or_bust(sc, name, sc->open_output_file_symbol, args, T_STRING, 1);

  if (is_pair(cdr(args)))
    {
      if (!is_string(cadr(args)))
	method_or_bust_with_type(sc, cadr(args), sc->open_output_file_symbol, args, make_string_wrapper(sc, "a string (a mode such as \"w\")"), 2);
      return(s7_open_output_file(sc, string_value(name), string_value(cadr(args))));
    }
  return(s7_open_output_file(sc, string_value(name), "w"));
}


static s7_pointer open_input_string(s7_scheme *sc, const char *input_string, int len)
{
  s7_pointer x;
  new_cell(sc, x, T_INPUT_PORT);
  port_port(x) = alloc_port(sc);
  port_type(x) = STRING_PORT;
  port_is_closed(x) = false;
  port_original_input_string(x) = sc->nil;
  port_data(x) = (unsigned char *)input_string;
  port_data_size(x) = len;
  port_position(x) = 0;
  port_filename_length(x) = 0;
  port_filename(x) = NULL;
  port_file_number(x) = 0; /* unsigned int */
  port_line_number(x) = 0;
  port_needs_free(x) = false;
  port_gc_loc(x) = -1;
  port_read_character(x) = string_read_char;
  port_read_line(x) = string_read_line;
  port_display(x) = input_display;
  port_read_semicolon(x) = string_read_semicolon;
#if DEBUGGING
  if (input_string[len] != '\0')
    fprintf(stderr, "read_white_space string is not terminated: %s", input_string);
#endif
  port_read_white_space(x) = terminated_string_read_white_space;
  port_read_name(x) = string_read_name_no_free;
  port_read_sharp(x) = string_read_sharp;
  port_write_character(x) = input_write_char;
  port_write_string(x) = input_write_string;
  add_input_port(sc, x);
  return(x);
}


static s7_pointer open_and_protect_input_string(s7_scheme *sc, s7_pointer str)
{
  s7_pointer p;
  p = open_input_string(sc, string_value(str), string_length(str));
  port_gc_loc(p) = s7_gc_protect(sc, str);
  return(p);
}


s7_pointer s7_open_input_string(s7_scheme *sc, const char *input_string)
{
  return(open_input_string(sc, input_string, safe_strlen(input_string)));
}


static s7_pointer g_open_input_string(s7_scheme *sc, s7_pointer args)
{
  #define H_open_input_string "(open-input-string str) opens an input port reading str"
  #define Q_open_input_string s7_make_signature(sc, 2, sc->is_input_port_symbol, sc->is_string_symbol)
  s7_pointer input_string, port;

  input_string = car(args);
  if (!is_string(input_string))
    method_or_bust(sc, input_string, sc->open_input_string_symbol, args, T_STRING, 0);
  port = open_and_protect_input_string(sc, input_string);
  return(port);
}


#define FORMAT_PORT_LENGTH 128
/* the large majority (> 99% in my tests) of the output strings have less than 128 chars when the port is finally closed
 *   256 is slightly slower (the calloc time below dominates the realloc time in string_write_string)
 *   64 is much slower (realloc dominates)
 */

static s7_pointer open_output_string(s7_scheme *sc, int len)
{
  s7_pointer x;
  new_cell(sc, x, T_OUTPUT_PORT);
  port_port(x) = alloc_port(sc);
  port_type(x) = STRING_PORT;
  port_is_closed(x) = false;
  port_data_size(x) = len;
  port_data(x) = (unsigned char *)malloc(len * sizeof(unsigned char)); /* was +8? */
  port_data(x)[0] = '\0';   /* in case s7_get_output_string before any output */
  port_position(x) = 0;
  port_needs_free(x) = true;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = string_display;
  port_write_character(x) = string_write_char;
  port_write_string(x) = string_write_string;
  add_output_port(sc, x);
  return(x);
}

s7_pointer s7_open_output_string(s7_scheme *sc)
{
  return(open_output_string(sc, sc->initial_string_port_length));
}


static s7_pointer g_open_output_string(s7_scheme *sc, s7_pointer args)
{
  #define H_open_output_string "(open-output-string) opens an output string port"
  #define Q_open_output_string s7_make_signature(sc, 1, sc->is_output_port_symbol)
  return(s7_open_output_string(sc));
}


const char *s7_get_output_string(s7_scheme *sc, s7_pointer p)
{
  port_data(p)[port_position(p)] = '\0';
  return((const char *)port_data(p));
}


static s7_pointer g_get_output_string(s7_scheme *sc, s7_pointer args)
{
  #define H_get_output_string "(get-output-string port clear-port) returns the output accumulated in port.  \
If the optional 'clear-port' is #t, the current string is flushed."
  #define Q_get_output_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_output_port_symbol, sc->is_boolean_symbol)

  s7_pointer p, result;
  bool clear_port = false;

  if (is_pair(cdr(args)))
    {
      p = cadr(args);
      if (!s7_is_boolean(p))
	return(wrong_type_argument(sc, sc->get_output_string_symbol, 2, p, T_BOOLEAN));
      clear_port = (p == sc->T);
    }
  p = car(args);
  if ((!is_output_port(p)) ||
      (!is_string_port(p)))
    {
      if (p == sc->F) return(make_empty_string(sc, 0, 0));
      method_or_bust_with_type(sc, p, sc->get_output_string_symbol, args, make_string_wrapper(sc, "an output string port"), 0);
    }
  if (port_is_closed(p))
    return(simple_wrong_type_argument_with_type(sc, sc->get_output_string_symbol, p, make_string_wrapper(sc, "an active (open) string port")));

  result = s7_make_string_with_length(sc, (const char *)port_data(p), port_position(p));
  if (clear_port)
    {
      port_position(p) = 0;
      port_data(p)[0] = '\0';
    }
  return(result);
}


s7_pointer s7_open_input_function(s7_scheme *sc, s7_pointer (*function)(s7_scheme *sc, s7_read_t read_choice, s7_pointer port))
{
  s7_pointer x;
  new_cell(sc, x, T_INPUT_PORT);
  port_port(x) = alloc_port(sc);
  port_type(x) = FUNCTION_PORT;
  port_is_closed(x) = false;
  port_original_input_string(x) = sc->nil;
  port_needs_free(x) = false;
  port_input_function(x) = function;
  port_read_character(x) = function_read_char;
  port_read_line(x) = function_read_line;
  port_display(x) = input_display;
  port_write_character(x) = input_write_char;
  port_write_string(x) = input_write_string;
  add_input_port(sc, x);
  return(x);
}


s7_pointer s7_open_output_function(s7_scheme *sc, void (*function)(s7_scheme *sc, unsigned char c, s7_pointer port))
{
  s7_pointer x;
  new_cell(sc, x, T_OUTPUT_PORT);
  port_port(x) = alloc_port(sc);
  port_type(x) = FUNCTION_PORT;
  port_data(x) = NULL;
  port_is_closed(x) = false;
  port_needs_free(x) = false;
  port_output_function(x) = function;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = function_display;
  port_write_character(x) = function_write_char;
  port_write_string(x) = function_write_string;
  add_output_port(sc, x);
  return(x);
}


static void push_input_port(s7_scheme *sc, s7_pointer new_port)
{
  sc->temp6 = sc->input_port;
  sc->input_port = new_port;
  sc->input_port_stack = cons(sc, sc->temp6, sc->input_port_stack);
  sc->temp6 = sc->nil;
}


static void pop_input_port(s7_scheme *sc)
{
  if (is_pair(sc->input_port_stack))
    {
      s7_pointer nxt;
      sc->input_port = car(sc->input_port_stack);
      nxt = cdr(sc->input_port_stack);
      /* is this safe? */
      free_cell(sc, sc->input_port_stack);
      sc->input_port_stack = nxt;
    }
  else sc->input_port = sc->standard_input;
}


static int inchar(s7_pointer pt)
{
  int c;
  if (is_file_port(pt))
    c = fgetc(port_file(pt)); /* not unsigned char! -- could be EOF */
  else
    {
      if (port_data_size(pt) <= port_position(pt))
	return(EOF);
      c = (unsigned char)port_data(pt)[port_position(pt)++];
    }

  if (c == '\n')
    port_line_number(pt)++;

  return(c);
}


static void backchar(char c, s7_pointer pt)
{
  if (c == '\n')
    port_line_number(pt)--;

  if (is_file_port(pt))
    ungetc(c, port_file(pt));
  else
    {
      if (port_position(pt) > 0)
	port_position(pt)--;
    }
}


int s7_read_char(s7_scheme *sc, s7_pointer port)
{
  /* needs to be int return value so EOF=-1, but not 255 */
  return(port_read_character(port)(sc, port));
}


int s7_peek_char(s7_scheme *sc, s7_pointer port)
{
  int c;              /* needs to be an int so EOF=-1, but not 255 */
  c = port_read_character(port)(sc, port);
  if (c != EOF)
    backchar(c, port);
  return(c);
}


void s7_write_char(s7_scheme *sc, int c, s7_pointer pt)
{
  if (pt != sc->F)
    port_write_character(pt)(sc, c, pt);
}


static s7_pointer input_port_if_not_loading(s7_scheme *sc)
{
  s7_pointer port;
  port = sc->input_port;
  if (is_loader_port(port)) /* this flag is turned off by the reader macros, so we aren't in that context */
    {
      int c;
      c = port_read_white_space(port)(sc, port);
      if (c > 0)            /* we can get either EOF or NULL at the end */
	{
	  backchar(c, port);
	  return(NULL);
	}
      return(sc->standard_input);
    }
  return(port);
}

static s7_pointer g_read_char(s7_scheme *sc, s7_pointer args)
{
  #define H_read_char "(read-char (port (current-input-port))) returns the next character in the input port"
  #define Q_read_char s7_make_signature(sc, 2, s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol)
  s7_pointer port;

  if (is_not_null(args))
    port = car(args);
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(sc->eof_object);
    }
  if (!is_input_port(port))
    method_or_bust_with_type(sc, port, sc->read_char_symbol, args, an_input_port_string, 0);
  return(chars[port_read_character(port)(sc, port)]);
}


static s7_pointer read_char_0, read_char_1;
static s7_pointer g_read_char_0(s7_scheme *sc, s7_pointer args)
{
  s7_pointer port;
  port = input_port_if_not_loading(sc);
  if (port)
    return(chars[port_read_character(port)(sc, port)]);
  return(sc->eof_object);
}


static s7_pointer g_read_char_1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer port;
  port = car(args);
  if (!is_input_port(port))
    method_or_bust_with_type(sc, port, sc->read_char_symbol, args, an_input_port_string, 0);
  return(chars[port_read_character(port)(sc, port)]);
}

static s7_pointer c_read_char(s7_scheme *sc)
{
  int c;
  s7_pointer port;
  port = input_port_if_not_loading(sc);
  if (!port) return(sc->eof_object);
  c = port_read_character(port)(sc, port);
  if (c == EOF)
    return(sc->eof_object);
  return(chars[c]);
}

PF_0(read_char, c_read_char)


static s7_pointer read_char_chooser(s7_scheme *sc, s7_pointer f, int args, s7_pointer expr)
{
  if (args == 0)
    return(read_char_0);
  if (args == 1)
    return(read_char_1);
  return(f);
}


static s7_pointer g_write_char(s7_scheme *sc, s7_pointer args)
{
  #define H_write_char "(write-char char (port (current-output-port))) writes char to the output port"
  #define Q_write_char s7_make_signature(sc, 3, sc->is_char_symbol, sc->is_char_symbol, sc->is_output_port_symbol)
  s7_pointer port, chr;

  chr = car(args);
  if (!s7_is_character(chr))
    method_or_bust(sc, chr, sc->write_char_symbol, args, T_CHARACTER, 1);

  if (is_pair(cdr(args)))
    port = cadr(args);
  else port = sc->output_port;
  if (port == sc->F) return(chr);
  if (!is_output_port(port))
    method_or_bust_with_type(sc, port, sc->write_char_symbol, args, an_output_port_string, 2);

  port_write_character(port)(sc, s7_character(chr), port);
  return(chr);
}

static s7_pointer c_write_char(s7_scheme *sc, s7_pointer chr)
{
  if (!s7_is_character(chr))
    method_or_bust(sc, chr, sc->write_char_symbol, set_plist_1(sc, chr), T_CHARACTER, 1);
  if (sc->output_port == sc->F) return(chr);
  port_write_character(sc->output_port)(sc, s7_character(chr), sc->output_port);
  return(chr);
}

static s7_pointer write_char_1;
static s7_pointer g_write_char_1(s7_scheme *sc, s7_pointer args) {return(c_write_char(sc, car(args)));}

PF_TO_PF(write_char, c_write_char)


static s7_pointer write_char_chooser(s7_scheme *sc, s7_pointer f, int args, s7_pointer expr)
{
  if (args == 1)
    return(write_char_1);
  return(f);
}

/* (with-output-to-string (lambda () (write-char #\space))) -> " "
 * (with-output-to-string (lambda () (write #\space))) -> "#\\space"
 * (with-output-to-string (lambda () (display #\space))) -> " "
 * is this correct?  It's what Guile does.  write-char is actually display-char.
 */


static s7_pointer g_peek_char(s7_scheme *sc, s7_pointer args)
{
  #define H_peek_char "(peek-char (port (current-input-port))) returns the next character in the input port, but does not remove it from the input stream"
  #define Q_peek_char s7_make_signature(sc, 2, s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol)
  s7_pointer port;

  if (is_not_null(args))
    port = car(args);
  else port = sc->input_port;

  if (!is_input_port(port))
    method_or_bust_with_type(sc, port, sc->peek_char_symbol, args, an_input_port_string, 0);
  if (port_is_closed(port))
    return(simple_wrong_type_argument_with_type(sc, sc->peek_char_symbol, port, an_open_port_string));

  if (is_function_port(port))
    return((*(port_input_function(port)))(sc, S7_PEEK_CHAR, port));
  return(chars[s7_peek_char(sc, port)]);
}

static s7_pointer c_peek_char(s7_scheme *sc) {return(chars[s7_peek_char(sc, sc->input_port)]);}
PF_0(peek_char, c_peek_char)


static s7_pointer g_read_byte(s7_scheme *sc, s7_pointer args)
{
  #define H_read_byte "(read-byte (port (current-input-port))): reads a byte from the input port"
  #define Q_read_byte s7_make_signature(sc, 2, s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol)
  s7_pointer port;
  int c;

  if (is_not_null(args))
    port = car(args);
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(sc->eof_object);
    }
  if (!is_input_port(port))
    method_or_bust_with_type(sc, port, sc->read_byte_symbol, args, an_input_port_string, 0);

  c = port_read_character(port)(sc, port);
  if (c == EOF)
    return(sc->eof_object);
  return(small_int(c));
}

static s7_pointer c_read_byte(s7_scheme *sc)
{
  int c;
  s7_pointer port;
  port = input_port_if_not_loading(sc);
  if (!port) return(sc->eof_object);
  c = port_read_character(port)(sc, port);
  if (c == EOF)
    return(sc->eof_object);
  return(small_int(c));
}

PF_0(read_byte, c_read_byte)


static s7_pointer g_write_byte(s7_scheme *sc, s7_pointer args)
{
  #define H_write_byte "(write-byte byte (port (current-output-port))): writes byte to the output port"
  #define Q_write_byte s7_make_signature(sc, 3, sc->is_integer_symbol, sc->is_integer_symbol, sc->is_output_port_symbol)
  s7_pointer port, b;
  s7_int val;

  b = car(args);
  if (!s7_is_integer(b))
    method_or_bust(sc, car(args), sc->write_byte_symbol, args, T_INTEGER, 1);

  val = s7_integer(b);
  if ((val < 0) || (val > 255)) /* need to check this before port==#f, else (write-byte most-positive-fixnum #f) is not an error */
    return(wrong_type_argument_with_type(sc, sc->write_byte_symbol, 1, b, an_unsigned_byte_string));

  if (is_pair(cdr(args)))
    port = cadr(args);
  else port = sc->output_port;

  if (!is_output_port(port))
    {
      if (port == sc->F) return(car(args));
      method_or_bust_with_type(sc, port, sc->write_byte_symbol, args, an_output_port_string, 0);
    }

  s7_write_char(sc, (int)(s7_integer(b)), port);
  return(b);
}

static s7_int c_write_byte(s7_scheme *sc, s7_int x)
{
  if ((x < 0) || (x > 255))
    wrong_type_argument_with_type(sc, sc->write_byte_symbol, 1, make_integer(sc, x), an_unsigned_byte_string);
  s7_write_char(sc, (int)x, sc->output_port);
  return(x);
}

IF_TO_IF(write_byte, c_write_byte)


static s7_pointer g_read_line(s7_scheme *sc, s7_pointer args)
{
  #define H_read_line "(read-line port (with-eol #f)) returns the next line from port, or #<eof>.\
If 'with-eol' is not #f, read-line includes the trailing end-of-line character."
  #define Q_read_line s7_make_signature(sc, 3, s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol, sc->is_boolean_symbol)

  s7_pointer port;
  bool with_eol = false;

  if (is_not_null(args))
    {
      port = car(args);
      if (!is_input_port(port))
	method_or_bust_with_type(sc, port, sc->read_line_symbol, args, an_input_port_string, 1);

      if (is_not_null(cdr(args)))
	with_eol = (cadr(args) != sc->F);
    }
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(sc->eof_object);
    }
  return(port_read_line(port)(sc, port, with_eol, true));
}

static s7_pointer c_read_line(s7_scheme *sc) {return(g_read_line(sc, sc->nil));}
PF_0(read_line, c_read_line)


static s7_pointer read_line_uncopied;
static s7_pointer g_read_line_uncopied(s7_scheme *sc, s7_pointer args)
{
  s7_pointer port;
  bool with_eol = false;
  port = car(args);
  if (!is_input_port(port))
    return(g_read_line(sc, args));
  if (is_not_null(cdr(args)))
    with_eol = (cadr(args) != sc->F);
  return(port_read_line(port)(sc, port, with_eol, false));
}


static s7_pointer c_read_string(s7_scheme *sc, s7_int chars, s7_pointer port)
{
  s7_pointer s;
  s7_int i;
  unsigned char *str;

  if (chars < 0)
    return(wrong_type_argument_with_type(sc, sc->read_string_symbol, 1, make_integer(sc, chars), a_non_negative_integer_string));
  if (chars > sc->max_string_length)
    return(out_of_range(sc, sc->read_string_symbol, small_int(1), make_integer(sc, chars), its_too_large_string));

  if (!port) return(sc->eof_object);
  if (!is_input_port(port))
    method_or_bust_with_type(sc, port, sc->read_string_symbol, list_2(sc, make_integer(sc, chars), port), an_input_port_string, 2);

  if (chars == 0)
    return(make_empty_string(sc, 0, 0));

  s = make_empty_string(sc, chars, 0);
  str = (unsigned char *)string_value(s);
  for (i = 0; i < chars; i++)
    {
      int c;
      c = port_read_character(port)(sc, port);
      if (c == EOF)
	{
	  if (i == 0)
	    return(sc->eof_object);
	  string_length(s) = i;
	  return(s);
	}
      str[i] = (unsigned char)c;
    }
  return(s);
}

static s7_pointer g_read_string(s7_scheme *sc, s7_pointer args)
{
  /* read-chars would be a better name -- read-string could mean CL-style read-from-string (like eval-string) 
   *   similarly read-bytes could return a byte-vector (rather than r7rs's read-bytevector)
   *   and write-string -> write-chars, write-bytevector -> write-bytes
   */
  #define H_read_string "(read-string k port) reads k characters from port into a new string and returns it."
  #define Q_read_string s7_make_signature(sc, 3, s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_eof_object_symbol), sc->is_integer_symbol, sc->is_input_port_symbol)
  s7_pointer k, port;

  k = car(args);
  if (!s7_is_integer(k))
    method_or_bust(sc, k, sc->read_string_symbol, args, T_INTEGER, 1);

  if (!is_null(cdr(args)))
    port = cadr(args);
  else port = input_port_if_not_loading(sc); /* port checked (for NULL) in c_read_string */
  return(c_read_string(sc, s7_integer(k), port));
}

static s7_pointer c_read_string_1(s7_scheme *sc, s7_int chars) {return(c_read_string(sc, chars, input_port_if_not_loading(sc)));}
IF_TO_PF(read_string, c_read_string_1)

#define declare_jump_info() bool old_longjmp; int old_jump_loc, jump_loc; jmp_buf old_goto_start

#define store_jump_info(Sc)						\
  do {									\
      old_longjmp = Sc->longjmp_ok;					\
      old_jump_loc = Sc->setjmp_loc;					\
      memcpy((void *)old_goto_start, (void *)(Sc->goto_start), sizeof(jmp_buf));\
  } while (0)

#define restore_jump_info(Sc)						\
  do {									\
      Sc->longjmp_ok = old_longjmp;					\
      Sc->setjmp_loc = old_jump_loc;					\
      memcpy((void *)(Sc->goto_start), (void *)old_goto_start, sizeof(jmp_buf));\
      if ((jump_loc == ERROR_JUMP) &&\
          (sc->longjmp_ok))\
        longjmp(sc->goto_start, ERROR_JUMP);\
  } while (0)

#define set_jump_info(Sc, Tag)			\
  do {						\
      sc->longjmp_ok = true;			\
      sc->setjmp_loc = Tag;				\
      jump_loc = setjmp(sc->goto_start);		\
  } while (0)


s7_pointer s7_read(s7_scheme *sc, s7_pointer port)
{
  if (is_input_port(port))
    {
      s7_pointer old_envir;
      declare_jump_info();

      old_envir = sc->envir;
      sc->envir = sc->nil;
      push_input_port(sc, port);

      store_jump_info(sc);
      set_jump_info(sc, READ_SET_JUMP);
      if (jump_loc != NO_JUMP)
	{
	  if (jump_loc != ERROR_JUMP)
	    eval(sc, sc->op);
	}
      else 
	{
	  push_stack(sc, OP_BARRIER, port, sc->nil);
	  push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);

	  eval(sc, OP_READ_INTERNAL);

	  if (sc->tok == TOKEN_EOF)
	    sc->value = sc->eof_object;

	  if ((sc->op == OP_EVAL_DONE) &&
	      (stack_op(sc->stack, s7_stack_top(sc) - 1) == OP_BARRIER))
	    pop_stack(sc); 
	}
      pop_input_port(sc);
      sc->envir = old_envir;

      restore_jump_info(sc);
      return(sc->value);
    }
  return(simple_wrong_type_argument_with_type(sc, sc->read_symbol, port, an_input_port_string));
}


static s7_pointer g_read(s7_scheme *sc, s7_pointer args)
{
  /* would it be useful to add an environment arg here?  (just set sc->envir at the end?)
   *    except for expansions, nothing is evaluated at read time, unless...
   *    say we set up a dot reader:
   *        (set! *#readers* (cons (cons #\. (lambda (str) (if (string=? str ".") (eval (read)) #f))) *#readers*))
   *    then
   *        (call-with-input-string "(+ 1 #.(+ 1 hiho))" (lambda (p) (read p)))
   *    evaluates hiho in the rootlet, but how to pass the env to the inner eval or read?
   * (eval, eval-string and load already have an env arg)
   */
  #define H_read "(read (port (current-input-port))) returns the next object in the input port, or #<eof> at the end"
  #define Q_read s7_make_signature(sc, 2, sc->T, sc->is_input_port_symbol)
  s7_pointer port;

  if (is_not_null(args))
    port = car(args);
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(sc->eof_object);
    }

  if (!is_input_port(port))
    method_or_bust_with_type(sc, port, sc->read_symbol, args, an_input_port_string, 0);

  if (is_function_port(port))
    return((*(port_input_function(port)))(sc, S7_READ, port));

  if ((is_string_port(port)) &&
      (port_data_size(port) <= port_position(port)))
    return(sc->eof_object);

  push_input_port(sc, port);
  push_stack(sc, OP_READ_DONE, sc->nil, sc->nil); /* this stops the internal read process so we only get one form */
  push_stack(sc, OP_READ_INTERNAL, sc->nil, sc->nil);

  return(port);
}

static s7_pointer c_read(s7_scheme *sc) {return(g_read(sc, sc->nil));}
PF_0(read, c_read)


/* -------------------------------- load -------------------------------- */

static FILE *search_load_path(s7_scheme *sc, const char *name)
{
  int i, len;
  s7_pointer lst;

  lst = s7_load_path(sc);
  len = s7_list_length(sc, lst);
  for (i = 0; i < len; i++)
    {
      const char *new_dir;
      new_dir = string_value(s7_list_ref(sc, lst, i));
      if (new_dir)
	{
	  FILE *fp;
	  snprintf(sc->tmpbuf, TMPBUF_SIZE, "%s/%s", new_dir, name);
	  fp = fopen(sc->tmpbuf, "r");
	  if (fp) return(fp);
	}
    }
  return(NULL);
}


s7_pointer s7_load_with_environment(s7_scheme *sc, const char *filename, s7_pointer e)
{
  s7_pointer port;
  FILE *fp;
  char *new_filename = NULL;
  declare_jump_info();

  fp = fopen(filename, "r");
  if (!fp)
    {
      fp = search_load_path(sc, filename);
      if (fp) 
	new_filename = copy_string(sc->tmpbuf); /* (require libc.scm) for example needs the directory for cload in some cases */
    }
  if (!fp)
    return(file_error(sc, "load", "can't open", filename));

  if (hook_has_functions(sc->load_hook))
    s7_call(sc, sc->load_hook, list_1(sc, sc->temp4 = s7_make_string(sc, filename)));

  port = read_file(sc, fp, (new_filename) ? (const char *)new_filename : filename, -1, "load");   /* -1 means always read its contents into a local string */
  port_file_number(port) = remember_file_name(sc, filename);
  if (new_filename) free(new_filename);
  set_loader_port(port);
  push_input_port(sc, port);

  /* it's possible to call this recursively (s7_load is Xen_load_file which can be invoked via s7_call)
   *   but in that case, we actually want it to behave like g_load and continue the evaluation upon completion
   */
  sc->envir = e;
  push_stack(sc, OP_LOAD_RETURN_IF_EOF, port, sc->code);

  store_jump_info(sc);
  set_jump_info(sc, LOAD_SET_JUMP);
  if (jump_loc != NO_JUMP)
    {
      if (jump_loc != ERROR_JUMP)
	eval(sc, sc->op);
    }
  else eval(sc, OP_READ_INTERNAL);

  pop_input_port(sc);
  if (is_input_port(port))
    s7_close_input_port(sc, port);

  restore_jump_info(sc);
  if (is_multiple_value(sc->value))
    sc->value = splice_in_values(sc, multiple_value(sc->value));
 return(sc->value);
}


s7_pointer s7_load(s7_scheme *sc, const char *filename)
{
  return(s7_load_with_environment(sc, filename, sc->nil));
}


#if WITH_C_LOADER
#include <dlfcn.h>

static char *full_filename(const char *filename)
{
  int len;
  char *pwd, *rtn;
  pwd = getcwd(NULL, 0); /* docs say this means it will return a new string of the right size */
  len = safe_strlen(pwd) + safe_strlen(filename) + 8;
  rtn = (char *)malloc(len * sizeof(char));
  if (pwd)
    {
      snprintf(rtn, len, "%s/%s", pwd, filename);
      free(pwd);
    }
  else snprintf(rtn, len, "%s", filename);
  return(rtn);
}
#endif


static s7_pointer g_load(s7_scheme *sc, s7_pointer args)
{
  #define H_load "(load file (env (rootlet))) loads the scheme file 'file'. The 'env' argument \
defaults to the rootlet.  To load into the current environment instead, pass (curlet)."
  #define Q_load s7_make_signature(sc, 3, sc->values_symbol, sc->is_string_symbol, sc->is_let_symbol)

  FILE *fp = NULL;
  s7_pointer name, port;
  const char *fname;

  name = car(args);
  if (!is_string(name))
    method_or_bust(sc, name, sc->load_symbol, args, T_STRING, 1);

  if (is_not_null(cdr(args)))
    {
      s7_pointer e;
      e = cadr(args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->load_symbol, 2, e, a_let_string));
      if (e == sc->rootlet)
	sc->envir = sc->nil;
      else sc->envir = e;
    }
  else sc->envir = sc->nil;

  fname = string_value(name);
  if ((!fname) || (!(*fname)))                 /* fopen("", "r") returns a file pointer?? */
    return(s7_error(sc, sc->out_of_range_symbol, set_elist_2(sc, make_string_wrapper(sc, "load's first argument, ~S, should be a filename"), name)));

  if (is_directory(fname))
    return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, make_string_wrapper(sc, "load argument, ~S, is a directory"), name)));

#if WITH_C_LOADER
  /* if fname ends in .so, try loading it as a c shared object
   *   (load "/home/bil/cl/m_j0.so" (inlet (cons 'init_func 'init_m_j0)))
   */
  {
    int fname_len;

    fname_len = safe_strlen(fname);
    if ((fname_len > 3) &&
	(is_pair(cdr(args))) &&
	(local_strcmp((const char *)(fname + (fname_len - 3)), ".so")))
      {
	s7_pointer init;

	init = let_ref_1(sc, sc->envir, s7_make_symbol(sc, "init_func"));
	if (is_symbol(init))
	  {
	    void *library;
	    char *pwd_name = NULL;

	    if (fname[0] != '/')
	      pwd_name = full_filename(fname); /* this is necessary, at least in Linux -- we can't blithely dlopen whatever is passed to us */
	    library = dlopen((pwd_name) ? pwd_name : fname, RTLD_NOW);
	    if (library)
	      {
		const char *init_name = NULL;
		void *init_func;

		init_name = symbol_name(init);
		init_func = dlsym(library, init_name);
		if (init_func)
		  {
		    typedef void *(*dl_func)(s7_scheme *sc);
		    ((dl_func)init_func)(sc);
		    if (pwd_name) free(pwd_name);
		    return(sc->T);
		  }
		else
		  {
		    s7_warn(sc, 512, "loaded %s, but can't find %s (%s)?\n", fname, init_name, dlerror());
		    dlclose(library);
		  }
	      }
	    else s7_warn(sc, 512, "load %s failed: %s\n", (pwd_name) ? pwd_name : fname, dlerror());
	    if (pwd_name) free(pwd_name);
	  }
	else s7_warn(sc, 512, "can't load %s: no init function\n", fname);
	return(sc->F);
      }
  }
#endif

  fp = fopen(fname, "r");

#if WITH_GCC
  if (!fp)
    {
      /* catch one special case, "~/..." since it causes 99.9% of the "can't load ..." errors */
      if ((fname[0] == '~') &&
	  (fname[1] == '/'))
	{
	  char *home;
	  home = getenv("HOME");
	  if (home)
	    {
	      char *filename;
	      int len;
	      len = safe_strlen(fname) + safe_strlen(home) + 1;
	      tmpbuf_malloc(filename, len);
	      snprintf(filename, len, "%s%s", home, (char *)(fname + 1));
	      fp = fopen(filename, "r");
	      tmpbuf_free(filename, len);
	    }
	}
    }
#endif

  if (!fp)
    {
      fp = search_load_path(sc, fname);
      if (!fp)
	return(file_error(sc, "load", "can't open", fname));
    }

  port = read_file(sc, fp, fname, -1, "load");
  port_file_number(port) = remember_file_name(sc, fname);
  set_loader_port(port);
  push_input_port(sc, port);

  push_stack(sc, OP_LOAD_CLOSE_AND_POP_IF_EOF, sc->nil, sc->nil);  /* was pushing args and code, but I don't think they're used later */
  push_stack(sc, OP_READ_INTERNAL, sc->nil, sc->nil);

  /* now we've opened and moved to the file to be loaded, and set up the stack to return
   *   to where we were.  Call *load-hook* if it is a procedure.
   */

  if (hook_has_functions(sc->load_hook))
    s7_apply_function(sc, sc->load_hook, list_1(sc, sc->temp4 = s7_make_string(sc, fname)));

  return(sc->unspecified);
}


s7_pointer s7_load_path(s7_scheme *sc)
{
  return(s7_symbol_value(sc, sc->load_path_symbol));
}


s7_pointer s7_add_to_load_path(s7_scheme *sc, const char *dir)
{
  s7_symbol_set_value(sc,
		      sc->load_path_symbol,
		      cons(sc,
			   s7_make_string(sc, dir),
			   s7_symbol_value(sc, sc->load_path_symbol)));
  return(s7_symbol_value(sc, sc->load_path_symbol));
}


static s7_pointer g_load_path_set(s7_scheme *sc, s7_pointer args)
{
  /* new value must be either () or a proper list of strings */
  if (is_null(cadr(args))) return(cadr(args));
  if (is_pair(cadr(args)))
    {
      s7_pointer x;
      for (x = cadr(args); is_pair(x); x = cdr(x))
	if (!is_string(car(x)))
	  return(sc->error_symbol);
      if (is_null(x))
	return(cadr(args));
    }
  return(sc->error_symbol);
}

static s7_pointer g_cload_directory_set(s7_scheme *sc, s7_pointer args)
{
  s7_pointer cl_dir;
  cl_dir = cadr(args);
  if (!is_string(cl_dir))
    return(sc->error_symbol);
  s7_symbol_set_value(sc, sc->cload_directory_symbol, cl_dir);
  if (safe_strlen(string_value(cl_dir)) > 0)
    s7_add_to_load_path(sc, (const char *)(string_value(cl_dir)));
  return(cl_dir);
}


/* ---------------- autoload ---------------- */

void s7_autoload_set_names(s7_scheme *sc, const char **names, int size)
{
  /* the idea here is that by sticking to string constants we can handle 90% of the work at compile-time,
   *   with less start-up memory.  Then eventually we'll add C libraries a la xg (gtk) as environments
   *   and every name in that library will come as an import once dlopen has picked up the library.
   *   So, hopefully, we can pre-declare as many names as we want from as many libraries as we want,
   *   without a bloated mess of a run-time image.  And new libraries are easy to accommodate --
   *   add the names to be auto-exported to this list with the name of the scheme file that cloads
   *   the library and exports the given name. So, we'll need a separate such file for each library?
   *
   * the environment variable could use the library base name in *: *libm* or *libgtk*
   *   (*libm* 'j0)
   * why not just predeclare these libraries?  The caller could import what he wants via require.
   * So the autoloader need only know which libraries, but this doesn't fit the current use of gtk in xg
   * In fact, we only need to see *libm* -> libm.so etc, but we still need the arg/return types of each function, etc
   * And libgtk is enormous -- seems too bad to tie-in everything via the FFI when we need less than 1% of it.
   * Perhaps each module as an environment within the main one: ((*libgtk* *gtkwidget*) 'gtk_widget_new)?
   * But that requires inside knowlege of the library, and changes without notice.
   *
   * Also we need to decide how to handle name collisions (by order of autoload lib setup)
   * And (lastly?) how to handle different library versions?
   *
   *
   * so autoload known libs here in s7 so we're indepentdent of snd
   *   (currently these are included in make-index.scm[line 575] -> snd-xref.c)
   * for each module, include an env in the lib env (*libgtk* 'gtkwidget.h) or whatever that has the names in that header
   * in autoload below, don't sort! -- just build a list of autoload tables and check each in order at autoload time (we want startup to be fast)
   * for versions, include wrapper macro at end of each c-define choice
   * in the xg case, there's no savings in delaying the defines
   *
   */

  if (!sc->autoload_names)
    {
      sc->autoload_names = (const char ***)calloc(INITIAL_AUTOLOAD_NAMES_SIZE, sizeof(const char **));
      sc->autoload_names_sizes = (int *)calloc(INITIAL_AUTOLOAD_NAMES_SIZE, sizeof(int));
      sc->autoloaded_already = (bool **)calloc(INITIAL_AUTOLOAD_NAMES_SIZE, sizeof(bool *));
      sc->autoload_names_top = INITIAL_AUTOLOAD_NAMES_SIZE;
      sc->autoload_names_loc = 0;
    }
  else
    {
      if (sc->autoload_names_loc >= sc->autoload_names_top)
	{
	  int i;
	  sc->autoload_names_top *= 2;
	  sc->autoload_names = (const char ***)realloc(sc->autoload_names, sc->autoload_names_top * sizeof(const char **));
	  sc->autoload_names_sizes = (int *)realloc(sc->autoload_names_sizes, sc->autoload_names_top * sizeof(int));
	  sc->autoloaded_already = (bool **)realloc(sc->autoloaded_already, sc->autoload_names_top * sizeof(bool *));
	  for (i = sc->autoload_names_loc; i < sc->autoload_names_top; i++)
	    {
	      sc->autoload_names[i] = NULL;
	      sc->autoload_names_sizes[i] = 0;
	      sc->autoloaded_already[i] = NULL;
	    }
	}
    }

  sc->autoload_names[sc->autoload_names_loc] = names;
  sc->autoload_names_sizes[sc->autoload_names_loc] = size;
  sc->autoloaded_already[sc->autoload_names_loc] = (bool *)calloc(size, sizeof(bool));
  sc->autoload_names_loc++;
}


static const char *find_autoload_name(s7_scheme *sc, s7_pointer symbol, bool *already_loaded, bool loading)
{
  int l = 0, pos = -1, lib, libs;
  const char *name, *this_name;

  name = symbol_name(symbol);
  libs = sc->autoload_names_loc;

  for (lib = 0; lib < libs; lib++)
    {
      const char **names;
      int u;
      u = sc->autoload_names_sizes[lib] - 1;
      names = sc->autoload_names[lib];

      while (true)
	{
	  int comp;
	  if (u < l) break;
	  pos = (l + u) / 2;
	  this_name = names[pos * 2];
	  comp = strcmp(this_name, name);
	  if (comp == 0)
	    {
	      *already_loaded = sc->autoloaded_already[lib][pos];
	      if (loading) sc->autoloaded_already[lib][pos] = true;
	      return(names[pos * 2 + 1]);             /* file name given func name */
	    }
	  if (comp < 0)
	    l = pos + 1;
	  else u = pos - 1;
	}
    }
  return(NULL);
}


s7_pointer s7_autoload(s7_scheme *sc, s7_pointer symbol, s7_pointer file_or_function)
{
  /* add '(symbol . file) to s7's autoload table */
  if (is_null(sc->autoload_table))
    sc->autoload_table = s7_make_hash_table(sc, sc->default_hash_table_length);
  s7_hash_table_set(sc, sc->autoload_table, symbol, file_or_function);
  return(file_or_function);
}


static s7_pointer g_autoload(s7_scheme *sc, s7_pointer args)
{
  #define H_autoload "(autoload symbol file-or-function) adds the symbol to its table of autoloadable symbols. \
If that symbol is encountered as an unbound variable, s7 either loads the file (following *load-path*), or calls \
the function.  The function takes one argument, the calling environment.  Presumably the symbol is defined \
in the file, or by the function."
  #define Q_autoload s7_make_signature(sc, 3, sc->T, sc->is_symbol_symbol, sc->T)

  s7_pointer sym, value;

  sym = car(args);
  if (is_string(sym))
    {
      if (string_length(sym) == 0)                   /* (autoload "" ...) */
	return(s7_wrong_type_arg_error(sc, "autoload", 1, sym, "a symbol-name or a symbol"));
      sym = make_symbol_with_length(sc, string_value(sym), string_length(sym));
    }
  if (!is_symbol(sym))
    {
      check_method(sc, sym, sc->autoload_symbol, args);
      return(s7_wrong_type_arg_error(sc, "autoload", 1, sym, "a string (symbol-name) or a symbol"));
    }
  if (is_keyword(sym))
    return(s7_wrong_type_arg_error(sc, "autoload", 1, sym, "a normal symbol (a keyword is never unbound)"));

  value = cadr(args);
  if (is_string(value))
    return(s7_autoload(sc, sym, value));
  if (((is_closure(value)) || (is_closure_star(value))) &&
      (s7_is_aritable(sc, value, 1)))
    return(s7_autoload(sc, sym, value));

  check_method(sc, value, sc->autoload_symbol, args);
  return(s7_wrong_type_arg_error(sc, "autoload", 2, value, "a string (file-name) or a thunk"));
}


static s7_pointer g_autoloader(s7_scheme *sc, s7_pointer args)
{
  #define H_autoloader "(*autoload* sym) returns the autoload info for the symbol sym, or #f."
  #define Q_autoloader s7_make_signature(sc, 2, sc->T, sc->is_symbol_symbol)
  s7_pointer sym;

  sym = car(args);
  if (!is_symbol(sym))
    {
      check_method(sc, sym, sc->autoloader_symbol, args);
      return(s7_wrong_type_arg_error(sc, "*autoload*", 1, sym, "a symbol"));
    }
  if (sc->autoload_names)
    {
      const char *file;
      bool loaded = false;
      file = find_autoload_name(sc, sym, &loaded, false);
      if (file)
	return(s7_make_string(sc, file));
    }
  if (is_hash_table(sc->autoload_table))
    return(s7_hash_table_ref(sc, sc->autoload_table, sym));

  return(sc->F);
}


static s7_pointer g_require(s7_scheme *sc, s7_pointer args)
{
  #define H_require "(require . symbols) loads each file associated with each symbol if it has not been loaded already.\
The symbols refer to the argument to \"provide\"."
  #define Q_require s7_make_circular_signature(sc, 1, 2, sc->T, sc->is_symbol_symbol)

  s7_pointer p;
  sc->temp5 = cons(sc, args, sc->temp5);
  for (p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer sym;
      if (is_symbol(car(p)))
	sym = car(p);
      else
	{
	  if ((is_pair(car(p))) && 
	      (caar(p) == sc->quote_symbol) &&
	      (is_symbol(cadar(p))))
	    sym = cadar(p);
	  else return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, make_string_wrapper(sc, "require: ~S is not a symbol"), car(p))));
	}
      if (!is_slot(find_symbol(sc, sym)))
	{
	  s7_pointer f;
	  f = g_autoloader(sc, list_1(sc, sym));
	  if (is_string(f))
	    s7_load_with_environment(sc, string_value(f), sc->envir);
	  else
	    {
	      sc->temp5 = sc->nil; 
	      return(s7_error(sc, make_symbol(sc, "autoload-error"), 
			      set_elist_2(sc, make_string_wrapper(sc, "require: no autoload info for ~S"), sym)));
	    }
	}
    }
  sc->temp5 = cdr(sc->temp5); /* in-coming value */
  return(sc->T);
}


/* -------------------------------- eval-string -------------------------------- */

s7_pointer s7_eval_c_string_with_environment(s7_scheme *sc, const char *str, s7_pointer e)
{
  s7_pointer code, port;
  port = s7_open_input_string(sc, str);
  code = s7_read(sc, port);
  s7_close_input_port(sc, port);
  return(s7_eval(sc, _NFre(code), e));
}


s7_pointer s7_eval_c_string(s7_scheme *sc, const char *str)
{
  return(s7_eval_c_string_with_environment(sc, str, sc->nil));
}

static s7_pointer g_eval_string(s7_scheme *sc, s7_pointer args)
{
  #define H_eval_string "(eval-string str (env (curlet))) returns the result of evaluating the string str as Scheme code"
  #define Q_eval_string s7_make_signature(sc, 3, sc->values_symbol, sc->is_string_symbol, sc->is_let_symbol)
  s7_pointer port, str;
  
  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->eval_string_symbol, args, T_STRING, 1);

  if (is_not_null(cdr(args)))
    {
      s7_pointer e;
      e = cadr(args);
      if (!is_let(e))
 	return(wrong_type_argument_with_type(sc, sc->eval_string_symbol, 2, e, a_let_string));
      if (e == sc->rootlet)
	sc->envir = sc->nil;
      else sc->envir = e;
    }

  port = open_and_protect_input_string(sc, str);
  push_input_port(sc, port);

  sc->temp3 = sc->args;
  push_stack(sc, OP_EVAL_STRING_1, args, sc->code); 
  push_stack(sc, OP_READ_INTERNAL, sc->nil, sc->nil);

  return(sc->F);
}

static s7_pointer eval_string_chooser(s7_scheme *sc, s7_pointer f, int args, s7_pointer expr)
{
  check_for_substring_temp(sc, expr);
  return(f);
}


static s7_pointer call_with_input(s7_scheme *sc, s7_pointer port, s7_pointer args)
{
  s7_pointer p;
  p = cadr(args);
  port_original_input_string(port) = car(args);
  push_stack(sc, OP_UNWIND_INPUT, sc->input_port, port);
  push_stack(sc, OP_APPLY, list_1(sc, port), p);
  return(sc->F);
}


/* -------------------------------- call-with-input-string -------------------------------- */

static s7_pointer g_call_with_input_string(s7_scheme *sc, s7_pointer args)
{
  s7_pointer str, proc;
  #define H_call_with_input_string "(call-with-input-string str proc) opens a string port for str and applies proc to it"
  #define Q_call_with_input_string pl_sf
  /* (call-with-input-string "44" (lambda (p) (+ 1 (read p)))) -> 45 */

  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->call_with_input_string_symbol, args, T_STRING, 1);

  proc = cadr(args);
  if (is_let(proc))
    check_method(sc, proc, sc->call_with_input_string_symbol, args);

  if (!s7_is_aritable(sc, proc, 1))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_string_symbol, 2, proc,
					 make_string_wrapper(sc, "a procedure of one argument (the port)")));

  if ((is_continuation(proc)) || (is_goto(proc)))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_string_symbol, 2, proc, a_normal_procedure_string));

  return(call_with_input(sc, open_and_protect_input_string(sc, str), args));
}

static s7_pointer c_call_with_input_string(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_call_with_input_string(sc, set_plist_2(sc, x, y)));}
PF2_TO_PF(call_with_input_string, c_call_with_input_string)


/* -------------------------------- call-with-input-file -------------------------------- */

static s7_pointer g_call_with_input_file(s7_scheme *sc, s7_pointer args)
{
  #define H_call_with_input_file "(call-with-input-file filename proc) opens filename and calls proc with the input port as its argument"
  #define Q_call_with_input_file pl_sf
  s7_pointer str, proc;

  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->call_with_input_file_symbol, args, T_STRING, 1);

  proc = cadr(args);
  if (!s7_is_aritable(sc, proc, 1))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_file_symbol, 2, proc,
					 make_string_wrapper(sc, "a procedure of one argument (the port)")));
  if ((is_continuation(proc)) || (is_goto(proc)))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_file_symbol, 2, proc, a_normal_procedure_string));

  return(call_with_input(sc, open_input_file_1(sc, string_value(str), "r", "call-with-input-file"), args));
}

static s7_pointer c_call_with_input_file(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(g_call_with_input_file(sc, set_plist_2(sc, x, y)));}
PF2_TO_PF(call_with_input_file, c_call_with_input_file)


static s7_pointer with_input(s7_scheme *sc, s7_pointer port, s7_pointer args)
{
  s7_pointer old_input_port, p;
  old_input_port = sc->input_port;
  sc->input_port = port;
  port_original_input_string(port) = car(args);
  push_stack(sc, OP_UNWIND_INPUT, old_input_port, port);
  p = cadr(args);
  push_stack(sc, OP_APPLY, sc->nil, p);
  return(sc->F);
}


/* -------------------------------- with-input-from-string -------------------------------- */

static s7_pointer g_with_input_from_string(s7_scheme *sc, s7_pointer args)
{
  #define H_with_input_from_string "(with-input-from-string str thunk) opens str as the temporary current-input-port and calls thunk"
  #define Q_with_input_from_string pl_sf
  s7_pointer str;

  str = car(args);
  if (!is_string(str))
    method_or_bust(sc, str, sc->with_input_from_string_symbol, args, T_STRING, 1);

  if (!is_thunk(sc, cadr(args)))
    method_or_bust_with_type(sc, cadr(args), sc->with_input_from_string_symbol, args, a_thunk_string, 2);

  /* since the arguments are evaluated before we get here, we can get some confusing situations:
   *   (with-input-from-string "#x2.1" (read))
   *   (read) -> whatever it can get from the current input port!
   *   ";with-input-from-string argument 2, #<eof>, is untyped but should be a thunk"
   */
  return(with_input(sc, open_and_protect_input_string(sc, str), args));
}

static s7_pointer c_with_input_from_string(s7_scheme *sc, s7_pointer x) {return(g_with_input_from_string(sc, set_plist_1(sc, x)));}
PF_TO_PF(with_input_from_string, c_with_input_from_string)


/* -------------------------------- with-input-from-file -------------------------------- */

static s7_pointer g_with_input_from_file(s7_scheme *sc, s7_pointer args)
{
  #define H_with_input_from_file "(with-input-from-file filename thunk) opens filename as the temporary current-input-port and calls thunk"
  #define Q_with_input_from_file pl_sf

  if (!is_string(car(args)))
    method_or_bust(sc, car(args), sc->with_input_from_file_symbol, args, T_STRING, 1);

  if (!is_thunk(sc, cadr(args)))
    method_or_bust_with_type(sc, cadr(args), sc->with_input_from_file_symbol, args, a_thunk_string, 2);

  return(with_input(sc, open_input_file_1(sc, string_value(car(args)), "r", "with-input-from-file"), args));
}

static s7_pointer c_with_input_from_file(s7_scheme *sc, s7_pointer x) {return(g_with_input_from_file(sc, set_plist_1(sc, x)));}
PF_TO_PF(with_input_from_file, c_with_input_from_file)



/* -------------------------------- iterators -------------------------------- */

static s7_pointer g_is_iterator(s7_scheme *sc, s7_pointer args)
{
  #define H_is_iterator "(iterator? obj) returns #t if obj is an iterator."
  #define Q_is_iterator pl_bt
  s7_pointer x;

  x = car(args);
  if (is_iterator(x)) return(sc->T);
  check_closure_for(sc, x, sc->is_iterator_symbol);
  check_boolean_method(sc, is_iterator, sc->is_iterator_symbol, args);
  return(sc->F);
}


static s7_pointer iterator_copy(s7_scheme *sc, s7_pointer p)
{
  /* fields are obj cur [loc|lcur] [len|slow|hcur] next */
  s7_pointer iter;
  new_cell(sc, iter, T_ITERATOR | T_SAFE_PROCEDURE);
  iterator_sequence(iter) = iterator_sequence(p);   /* obj */
  iterator_position(iter) = iterator_position(p);   /* loc|lcur (loc is s7_int) */
  iterator_length(iter) = iterator_length(p);       /* len|slow|hcur (len is s7_int) */
  iterator_current(iter) = iterator_current(p);     /* cur */
  iterator_next(iter) = iterator_next(p);           /* next */
  return(iter);
}


static s7_pointer iterator_finished(s7_scheme *sc, s7_pointer iterator)
{
  return(sc->ITERATOR_END);
}

static s7_pointer let_iterate(s7_scheme *sc, s7_pointer iterator)
{
  s7_pointer slot;
  slot = iterator_current_slot(iterator);
  if (is_slot(slot))
    {
      iterator_set_current_slot(iterator, next_slot(slot));
      if (iterator_let_cons(iterator))
	{
	  s7_pointer p;
	  p = iterator_let_cons(iterator);
	  set_car(p, slot_symbol(slot));
	  set_cdr(p, slot_value(slot));
	  return(p);
	}
      return(cons(sc, slot_symbol(slot), slot_value(slot)));
    }
  iterator_next(iterator) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer rootlet_iterate(s7_scheme *sc, s7_pointer iterator)
{
  s7_pointer slot;
  slot = iterator_current(iterator);
  if (is_slot(slot))
    {
      if (iterator_position(iterator) < sc->rootlet_entries)
	{
	  iterator_position(iterator)++;
	  iterator_current(iterator) = vector_element(sc->rootlet, iterator_position(iterator));
	}
      else iterator_current(iterator) = sc->nil;
      return(cons(sc, slot_symbol(slot), slot_value(slot)));
    }
  iterator_next(iterator) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer hash_table_iterate(s7_scheme *sc, s7_pointer iterator)
{
  s7_pointer table;
  int loc, len;
  hash_entry_t **elements;
  hash_entry_t *lst;

  lst = iterator_hash_current(iterator);
  if (lst)
    {
      iterator_hash_current(iterator) = lst->next;
      if (iterator_current(iterator))
	{
	  s7_pointer p;
	  p = iterator_current(iterator);
	  set_car(p, lst->key);
	  set_cdr(p, lst->value);
	  return(p);
	}
      return(cons(sc, lst->key, lst->value));
    }

  table = iterator_sequence(iterator); /* using iterator_length and hash_table_entries here was slightly slower */
  len = hash_table_mask(table) + 1;
  elements = hash_table_elements(table);

  for (loc = iterator_position(iterator) + 1; loc < len;  loc++)
    {
      hash_entry_t *x;
      x = elements[loc];
      if (x)
	{
	  iterator_position(iterator) = loc;
	  iterator_hash_current(iterator) = x->next;
	  if (iterator_current(iterator))
	    {
	      s7_pointer p;
	      p = iterator_current(iterator);
	      set_car(p, x->key);
	      set_cdr(p, x->value);
	      return(p);
	    }
	  return(cons(sc, x->key, x->value));
	}
    }
  iterator_next(iterator) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer string_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(s7_make_character(sc, (unsigned char)(string_value(iterator_sequence(obj))[iterator_position(obj)++])));
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer byte_vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(small_int((unsigned char)(string_value(iterator_sequence(obj))[iterator_position(obj)++])));
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer float_vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(make_real(sc, float_vector_element(iterator_sequence(obj), iterator_position(obj)++)));
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer int_vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(make_integer(sc, int_vector_element(iterator_sequence(obj), iterator_position(obj)++)));
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(vector_element(iterator_sequence(obj), iterator_position(obj)++));
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer closure_iterate(s7_scheme *sc, s7_pointer obj)
{
  s7_pointer result;
  result = s7_apply_function(sc, iterator_sequence(obj), sc->nil);
  if (result == sc->ITERATOR_END)
    iterator_next(obj) = iterator_finished;
  return(result);
}

static s7_pointer c_object_direct_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    {
      s7_pointer result, p;
      p = iterator_sequence(obj);
      result = c_object_cref(p)(sc, p, iterator_position(obj));
      iterator_position(obj)++;
      if (result == sc->ITERATOR_END)
	iterator_next(obj) = iterator_finished;
      return(result);
    }
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer c_object_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    {
      s7_pointer result, p, cur;
      p = iterator_sequence(obj);
      cur = iterator_current(obj);
      set_car(sc->z2_1, sc->x);
      set_car(sc->z2_2, sc->z); /* is this necessary? */
      set_car(cur, make_integer(sc, iterator_position(obj)));
      result = (*(c_object_ref(p)))(sc, p, cur);
      sc->x = car(sc->z2_1);
      sc->z = car(sc->z2_2);
      iterator_position(obj)++;
      if (result == sc->ITERATOR_END)
	iterator_next(obj) = iterator_finished;
      return(result);
    }
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}


static s7_pointer pair_iterate_1(s7_scheme *sc, s7_pointer obj);
static s7_pointer pair_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (is_pair(iterator_current(obj)))
    {
      s7_pointer result;
      result = car(iterator_current(obj));
      iterator_current(obj) = cdr(iterator_current(obj));
      if (iterator_current(obj) == iterator_slow(obj))
	{
	  iterator_next(obj) = iterator_finished;
	  return(result);
	}
      iterator_next(obj) = pair_iterate_1;
      return(result);
    }
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer pair_iterate_1(s7_scheme *sc, s7_pointer obj)
{
  if (is_pair(iterator_current(obj)))
    {
      s7_pointer result;
      result = car(iterator_current(obj));
      iterator_current(obj) = cdr(iterator_current(obj));
      if (iterator_current(obj) == iterator_slow(obj))
	{
	  iterator_next(obj) = iterator_finished;
	  return(result);
	}
      iterator_set_slow(obj, cdr(iterator_slow(obj)));
      iterator_next(obj) = pair_iterate;
      return(result);
    }
  iterator_next(obj) = iterator_finished;
  return(sc->ITERATOR_END);
}

static s7_pointer iterator_method(s7_scheme *sc, s7_pointer e)
{
  s7_pointer func;
  if ((has_methods(e)) && 
      ((func = find_method(sc, find_let(sc, e), sc->make_iterator_symbol)) != sc->undefined))
    {
      s7_pointer it;
      it = s7_apply_function(sc, func, list_1(sc, e));
      if (!is_iterator(it))
	return(s7_error(sc, sc->error_symbol, set_elist_2(sc, make_string_wrapper(sc, "make-iterator method must return an interator: ~S"), it)));
      return(it);
    }
  return(NULL);
}

s7_pointer s7_make_iterator(s7_scheme *sc, s7_pointer e)
{
  s7_pointer iter;

  new_cell(sc, iter, T_ITERATOR | T_SAFE_PROCEDURE);
  iterator_sequence(iter) = e;
  iterator_position(iter) = 0;

  switch (type(e))
    {
    case T_LET:
      if (e == sc->rootlet)
	{
	  iterator_current(iter) = vector_element(e, 0); /* unfortunately tricky -- let_iterate uses different fields */
	  iterator_next(iter) = rootlet_iterate;
	}
      else
	{
	  s7_pointer f;
	  sc->temp6 = iter;
	  f = iterator_method(sc, e);
	  sc->temp6 = sc->nil;
	  if (f) {free_cell(sc, iter); return(f);}
	  iterator_set_current_slot(iter, let_slots(e));
	  iterator_next(iter) = let_iterate;
	  iterator_let_cons(iter) = NULL;
	}
      break;

    case T_HASH_TABLE:
      iterator_hash_current(iter) = NULL;
      iterator_current(iter) = NULL;
      iterator_position(iter) = -1;
      iterator_next(iter) = hash_table_iterate;
      break;

    case T_STRING:
      iterator_length(iter) = string_length(e);
      if (is_byte_vector(e))
	iterator_next(iter) = byte_vector_iterate;
      else iterator_next(iter) = string_iterate;
      break;

    case T_VECTOR:
      iterator_length(iter) = vector_length(e);
      iterator_next(iter) = vector_iterate;
      break;

    case T_INT_VECTOR:
      iterator_length(iter) = vector_length(e);
      iterator_next(iter) = int_vector_iterate;
      break;

    case T_FLOAT_VECTOR:
      iterator_length(iter) = vector_length(e);
      iterator_next(iter) = float_vector_iterate;
      break;

    case T_PAIR:
      iterator_current(iter) = e;
      iterator_next(iter) = pair_iterate;
      iterator_set_slow(iter, e);
      break;

    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
    case T_CLOSURE: case T_CLOSURE_STAR:
      {
	s7_pointer p;
	p = cons(sc, e, sc->nil);
	if (g_is_iterator(sc, p) != sc->F)
	  {
	    set_car(p, small_int(0));
	    iterator_current(iter) = p;
	    set_mark_seq(iter);
	    iterator_next(iter) = closure_iterate;
	    if (has_methods(e))
	      iterator_length(iter) = closure_length(sc, e);
	    else iterator_length(iter) = s7_int_max;
	  }
	else 
	  {
	    free_cell(sc, iter);
	    return(simple_wrong_type_argument_with_type(sc, sc->make_iterator_symbol, e, make_string_wrapper(sc, "a closure/macro with an 'iterator local that is not #f")));
	  }
      }
      break;

    case T_C_OBJECT:
      iterator_length(iter) = object_length_to_int(sc, e);
      if (c_object_direct_ref(e))
	{
	  iterator_next(iter) = c_object_direct_iterate;
	  c_object_cref(e) = c_object_direct_ref(e);
	}
      else
	{
	  s7_pointer f;
	  sc->temp6 = iter;
	  f = iterator_method(sc, e);
	  sc->temp6 = sc->nil;
	  if (f) {free_cell(sc, iter); return(f);}
	  iterator_current(iter) = cons(sc, small_int(0), sc->nil);
	  set_mark_seq(iter);
	  iterator_next(iter) = c_object_iterate;
	}
      break;

    default:
      return(simple_wrong_type_argument_with_type(sc, sc->make_iterator_symbol, e, a_sequence_string));
    }
  return(iter);
}


static s7_pointer g_make_iterator(s7_scheme *sc, s7_pointer args)
{
  #define H_make_iterator "(make-iterator sequence) returns an iterator object that \
returns the next value in the sequence each time it is called.  When it reaches the end, it returns " ITERATOR_END_NAME "."
  #define Q_make_iterator s7_make_signature(sc, 3, sc->is_iterator_symbol, sc->is_sequence_symbol, sc->is_pair_symbol)
  
  s7_pointer seq;
  seq = car(args);

  if (is_pair(cdr(args)))
    {
      if (is_pair(cadr(args)))
	{
	  if (is_hash_table(seq))
	    {
	      s7_pointer iter;
	      iter = s7_make_iterator(sc, seq);
	      iterator_current(iter) = cadr(args);
	      set_mark_seq(iter);
	      return(iter);
	    }
	  if ((is_let(seq)) && (seq != sc->rootlet))
	    {
	      s7_pointer iter;
	      iter = s7_make_iterator(sc, seq);
	      iterator_let_cons(iter) = cadr(args);
	      set_mark_seq(iter);
	      return(iter);
	    }
	}
      else return(simple_wrong_type_argument(sc, sc->make_iterator_symbol, cadr(args), T_PAIR));
    }
  return(s7_make_iterator(sc, seq));
}

PF_TO_PF(make_iterator, s7_make_iterator)


static s7_pointer c_iterate(s7_scheme *sc, s7_pointer iter)
{
  if (!is_iterator(iter))
    method_or_bust(sc, iter, sc->iterate_symbol, list_1(sc, iter), T_ITERATOR, 0);
  return((iterator_next(iter))(sc, iter));
}

static s7_pointer g_iterate(s7_scheme *sc, s7_pointer args)
{
  #define H_iterate "(iterate obj) returns the next element from the iterator obj, or " ITERATOR_END_NAME "."
  #define Q_iterate s7_make_signature(sc, 2, sc->T, sc->is_iterator_symbol)

  s7_pointer iter;
  iter = car(args);
  if (!is_iterator(iter))
    method_or_bust(sc, iter, sc->iterate_symbol, args, T_ITERATOR, 0);
  return((iterator_next(iter))(sc, iter));
}

static s7_pointer iterate_pf_p(s7_scheme *sc, s7_pointer **p)
{
  s7_pf_t f;
  s7_pointer x;
  f = (s7_pf_t)(**p); (*p)++;	
  x = f(sc, p);
  return(c_iterate(sc, x));
}

static s7_pointer iterate_pf_s(s7_scheme *sc, s7_pointer **p)
{
  pf_pf_t f;
  s7_pointer x;
  x = (s7_pointer)(**p); (*p)++;	
  f = (pf_pf_t)(**p); (*p)++;	
  return(f(sc, x));
}

static s7_pf_t iterate_gf(s7_scheme *sc, s7_pointer expr)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))))
    {
      s7_pointer a1, obj;
      a1 = cadr(expr);
      if ((is_symbol(a1)) &&
	  (!s7_xf_is_stepper(sc, a1)) &&
	  (is_iterator(obj = s7_symbol_value(sc, a1))))
	{
	  s7_xf_store(sc, obj);
	  s7_xf_store(sc, (s7_pointer)iterator_next(obj));
	  return(iterate_pf_s);
	}
      if (s7_arg_to_pf(sc, a1))
	return(iterate_pf_p);
    }
  return(NULL);
}

static s7_pf_t iterate_pf(s7_scheme *sc, s7_pointer expr)
{
  if ((is_pair(cdr(expr))) && (is_null(cddr(expr))))
    {
      s7_pointer a1, obj;
      a1 = cadr(expr);
      if ((is_symbol(a1)) &&
	  (!s7_xf_is_stepper(sc, a1)) &&
	  (is_iterator(obj = s7_symbol_value(sc, a1))))
	{
	  s7_pointer seq;
	  seq = iterator_sequence(obj);
	  if ((type(seq) == T_VECTOR) || (is_string(seq)) || (is_pair(seq)))
	    {
	      s7_xf_store(sc, obj);
	      s7_xf_store(sc, (s7_pointer)iterator_next(obj));
	      return(iterate_pf_s);
	    }
	}
    }
  return(NULL);
}

s7_pointer s7_iterate(s7_scheme *sc, s7_pointer obj)
{
  return((iterator_next(obj))(sc, obj));
}

bool s7_is_iterator(s7_pointer obj)
{
  return(is_iterator(obj));
}

bool s7_iterator_is_at_end(s7_pointer obj)
{
  return(iterator_is_at_end(obj));
}


static s7_pointer g_iterator_sequence(s7_scheme *sc, s7_pointer args)
{
  #define H_iterator_sequence "(iterator-sequence iterator) returns the sequence that iterator is traversing."
  #define Q_iterator_sequence s7_make_signature(sc, 2, sc->is_sequence_symbol, sc->is_iterator_symbol)

  s7_pointer iter;

  iter = car(args);
  if (!is_iterator(iter))
    return(simple_wrong_type_argument(sc, sc->iterator_sequence_symbol, iter, T_ITERATOR));
  return(iterator_sequence(iter));
}

static s7_pointer c_iterator_sequence(s7_scheme *sc, s7_pointer iter)
{
  if (!is_iterator(iter))
    return(simple_wrong_type_argument(sc, sc->iterator_sequence_symbol, iter, T_ITERATOR));
  return(iterator_sequence(iter));
}

PF_TO_PF(iterator_sequence, c_iterator_sequence)


static s7_pointer g_iterator_is_at_end(s7_scheme *sc, s7_pointer args)
{
  #define H_iterator_is_at_end "(iterator-at-end? iter) returns #t if the iterator has reached the end of its sequence."
  #define Q_iterator_is_at_end s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_iterator_symbol)
  s7_pointer iter;

  iter = car(args);
  if (!is_iterator(iter))
    return(simple_wrong_type_argument(sc, sc->iterator_is_at_end_symbol, iter, T_ITERATOR));
  return(make_boolean(sc, iterator_is_at_end(iter)));
}



/* -------------------------------------------------------------------------------- */

#define INITIAL_SHARED_INFO_SIZE 8

static int shared_ref(shared_info *ci, s7_pointer p)
{
  /* from print after collecting refs, not called by equality check */
  int i;
  s7_pointer *objs;

  if (!is_collected(p)) return(0);

  objs = ci->objs;
  for (i = 0; i < ci->top; i++)
    if (objs[i] == p)
      {
	int val;
	val = ci->refs[i];
	if (val > 0)
	  ci->refs[i] = -ci->refs[i];
	return(val);
      }
  return(0);
}


static int peek_shared_ref(shared_info *ci, s7_pointer p)
{
  /* returns 0 if not found, otherwise the ref value for p */
  int i;
  s7_pointer *objs;
  objs = ci->objs;

  if (!is_collected(p)) return(0);
  for (i = 0; i < ci->top; i++)
    if (objs[i] == p) return(ci->refs[i]);

  return(0);
}


static void enlarge_shared_info(shared_info *ci)
{
  int i;
  ci->size *= 2;
  ci->objs = (s7_pointer *)realloc(ci->objs, ci->size * sizeof(s7_pointer));
  ci->refs = (int *)realloc(ci->refs, ci->size * sizeof(int));
  for (i = ci->top; i < ci->size; i++)
    {
      ci->refs[i] = 0;
      ci->objs[i] = NULL;
    }
}


static void add_equal_ref(shared_info *ci, s7_pointer x, s7_pointer y)
{
  /* assume neither x nor y is in the table, and that they should share a ref value,
   *   called only in equality check, not printer.
   */

  if ((ci->top + 2) >= ci->size)
    enlarge_shared_info(ci);

  set_collected(x);
  set_collected(y);

  ci->ref++;
  ci->objs[ci->top] = x;
  ci->refs[ci->top++] = ci->ref;
  ci->objs[ci->top] = y;
  ci->refs[ci->top++] = ci->ref;
}


static void add_shared_ref(shared_info *ci, s7_pointer x, int ref_x)
{
  /* called only in equality check, not printer */

  if (ci->top == ci->size)
    enlarge_shared_info(ci);

  set_collected(x);

  ci->objs[ci->top] = x;
  ci->refs[ci->top++] = ref_x;
}

static shared_info *collect_shared_info(s7_scheme *sc, shared_info *ci, s7_pointer top, bool stop_at_print_length, bool *cyclic);
static hash_entry_t *hash_equal(s7_scheme *sc, s7_pointer table, s7_pointer key);

static void collect_vector_info(s7_scheme *sc, shared_info *ci, s7_pointer top, bool stop_at_print_length, bool *cyclic)
{
  s7_int i, plen;

  if (stop_at_print_length)
    {
      plen = sc->print_length;
      if (plen > vector_length(top))
	plen = vector_length(top);
    }
  else plen = vector_length(top);

  for (i = 0; i < plen; i++)
    if (has_structure(vector_element(top, i)))
      collect_shared_info(sc, ci, vector_element(top, i), stop_at_print_length, cyclic);
}


static shared_info *collect_shared_info(s7_scheme *sc, shared_info *ci, s7_pointer top, bool stop_at_print_length, bool *cyclic)
{
  /* look for top in current list.
   *
   * As we collect objects (guaranteed to have structure) we set the collected bit.  If we ever
   *   encounter an object with that bit on, we've seen it before so we have a possible cycle.
   *   Once the collection pass is done, we run through our list, and clear all these bits.
   */
  if (is_shared(top))
    return(ci);

  if (is_collected(top))
    {
      s7_pointer *p, *objs_end;
      int i;
      *cyclic = true;
      objs_end = (s7_pointer *)(ci->objs + ci->top);

      for (p = ci->objs; p < objs_end; p++)
	if ((*p) == top)
	  {
	    i = (int)(p - ci->objs);
	    if (ci->refs[i] == 0)
	      {
		ci->has_hits = true;
		ci->refs[i] = ++ci->ref;  /* if found, set the ref number */
	      }
	    break;
	  }
    }
  else
    {
      /* top not seen before -- add it to the list */
      bool top_cyclic = false;
      set_collected(top);

      if (ci->top == ci->size)
	enlarge_shared_info(ci);
      ci->objs[ci->top++] = top;

      /* now search the rest of this structure */
      switch (type(top))
	{
	case T_PAIR:
	  if (has_structure(car(top)))
	    collect_shared_info(sc, ci, car(top), stop_at_print_length, &top_cyclic);
	  if (has_structure(cdr(top)))
	    collect_shared_info(sc, ci, cdr(top), stop_at_print_length, &top_cyclic);
	  break;

	case T_VECTOR:
	  collect_vector_info(sc, ci, top, stop_at_print_length, &top_cyclic);
	  break;

	case T_ITERATOR:
	  collect_shared_info(sc, ci, iterator_sequence(top), stop_at_print_length, &top_cyclic);
	  break;

	case T_HASH_TABLE:
	  if (hash_table_entries(top) > 0)
	    {
	      unsigned int i, len;
	      hash_entry_t **entries;
	      bool keys_safe;

	      keys_safe = ((hash_table_checker(top) != hash_equal) &&
			   (!hash_table_checker_locked(top)));
	      entries = hash_table_elements(top);
	      len = hash_table_mask(top) + 1;
	      for (i = 0; i < len; i++)
		{
		  hash_entry_t *p;
		  for (p = entries[i]; p; p = p->next)
		    {
		      if ((!keys_safe) &&
			  (has_structure(p->key)))
			collect_shared_info(sc, ci, p->key, stop_at_print_length, &top_cyclic);
		      if (has_structure(p->value))
			collect_shared_info(sc, ci, p->value, stop_at_print_length, &top_cyclic);
		    }
		}
	    }
	  break;

	case T_SLOT:
	  if (has_structure(slot_value(top)))
	    collect_shared_info(sc, ci, slot_value(top), stop_at_print_length, &top_cyclic);
	  break;

	case T_LET:
	  if (top == sc->rootlet)
	    collect_vector_info(sc, ci, top, stop_at_print_length, &top_cyclic);
	  else
	    {
	      s7_pointer p;
	      for (p = let_slots(top); is_slot(p); p = next_slot(p))
		if (has_structure(slot_value(p)))
		  collect_shared_info(sc, ci, slot_value(p), stop_at_print_length, &top_cyclic);
	    }
	  break;
	}
      if (!top_cyclic)
	set_shared(top);
      else *cyclic = true;
    }
  return(ci);
}


static shared_info *new_shared_info(s7_scheme *sc)
{
  shared_info *ci;
  if (!sc->circle_info)
    {
      ci = (shared_info *)calloc(1, sizeof(shared_info));
      ci->size = INITIAL_SHARED_INFO_SIZE;
      ci->objs = (s7_pointer *)malloc(ci->size * sizeof(s7_pointer));
      ci->refs = (int *)calloc(ci->size, sizeof(int));   /* finder expects 0 = unseen previously */
      sc->circle_info = ci;
    }
  else
    {
      int i;
      ci = sc->circle_info;
      memclr((void *)(ci->refs), ci->top * sizeof(int));
      for (i = 0; i < ci->top; i++)
	clear_collected_and_shared(ci->objs[i]);
    }
  ci->top = 0;
  ci->ref = 0;
  ci->has_hits = false;
  return(ci);
}


static shared_info *make_shared_info(s7_scheme *sc, s7_pointer top, bool stop_at_print_length)
{
  /* for the printer */
  shared_info *ci;
  int i, refs;
  s7_pointer *ci_objs;
  int *ci_refs;
  bool no_problem = true, cyclic = false;

  /* check for simple cases first */
  if (is_pair(top))
    {
      if (s7_list_length(sc, top) != 0) /* it is not circular at the top level (following cdr), so we can check each car(x) */
	{
	  s7_pointer x;
	  for (x = top; is_pair(x); x = cdr(x))
	    if (has_structure(car(x)))
	      {
		/* it can help a little in some cases to scan vectors here (and slots):
		 *   if no element has structure, it's ok (maybe also hash_table_entries == 0)
		 */
		no_problem = false;
		break;
	      }
	  if ((no_problem) &&
	      (!is_null(x)) &&
	      (has_structure(x)))
	    no_problem = false;

	  if (no_problem)
	    return(NULL);
	}
    }
  else
    {
      if (s7_is_vector(top))
	{
	  if (type(top) != T_VECTOR)
	    return(NULL);

	  for (i = 0; i < vector_length(top); i++)
	    if (has_structure(vector_element(top, i)))
	      {
		no_problem = false;
		break;
	      }
	  if (no_problem)
	    return(NULL);
	}
    }

  ci = new_shared_info(sc);

  /* collect all pointers associated with top */
  collect_shared_info(sc, ci, top, stop_at_print_length, &cyclic);

  for (i = 0; i < ci->top; i++)
    {
      s7_pointer p;
      p = ci->objs[i];
      clear_collected_and_shared(p);
    }
  if (!cyclic)
    return(NULL);

  if (!(ci->has_hits))
    return(NULL);

  ci_objs = ci->objs;
  ci_refs = ci->refs;

  /* find if any were referenced twice (once for just being there, so twice=shared)
   *   we know there's at least one such reference because has_hits is true.
   */
  for (i = 0, refs = 0; i < ci->top; i++)
    if (ci_refs[i] > 0)
      {
	set_collected(ci_objs[i]);
	if (i == refs)
	  refs++;
	else
	  {
	    ci_objs[refs] = ci_objs[i];
	    ci_refs[refs++] = ci_refs[i];
	    ci_refs[i] = 0;
	    ci_objs[i] = NULL;
	  }
      }
  ci->top = refs;
  return(ci);
}

/* -------------------------------- cyclic-sequences -------------------------------- */

static s7_pointer cyclic_sequences(s7_scheme *sc, s7_pointer obj, bool return_list)
{
  if (has_structure(obj))
    {
      shared_info *ci;
      ci = make_shared_info(sc, obj, false); /* false=don't stop at print length (vectors etc) */
      if (ci)
	{
	  if (return_list)
	    {
	      int i;
	      s7_pointer lst;
	      sc->w = sc->nil;
	      for (i = 0; i < ci->top; i++)
		sc->w = cons(sc, ci->objs[i], sc->w);
	      lst = sc->w;
	      sc->w = sc->nil;
	      return(lst);
	    }
	  else return(sc->T);
	}
    }
  return(sc->nil);
}

static s7_pointer g_cyclic_sequences(s7_scheme *sc, s7_pointer args)
{
  #define H_cyclic_sequences "(cyclic-sequences obj) returns a list of elements that are cyclic."
  #define Q_cyclic_sequences s7_make_signature(sc, 2, sc->is_proper_list_symbol, sc->T)
  return(cyclic_sequences(sc, car(args), true));
}

static int circular_list_entries(s7_pointer lst)
{
  int i;
  s7_pointer x;
  for (i = 1, x = cdr(lst); ; i++, x = cdr(x))
    {
      int j;
      s7_pointer y;
      for (y = lst, j = 0; j < i; y = cdr(y), j++)
	if (x == y)
	  return(i);
    }
}


static void object_to_port_with_circle_check(s7_scheme *sc, s7_pointer vr, s7_pointer port, use_write_t use_write, shared_info *ci);
static void object_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci);
static s7_pointer object_out(s7_scheme *sc, s7_pointer obj, s7_pointer strport, use_write_t choice);

static char *multivector_indices_to_string(s7_scheme *sc, s7_int index, s7_pointer vect, char *str, int cur_dim)
{
  s7_int size, ind;
  char buf[64];

  size = vector_dimension(vect, cur_dim);
  ind = index % size;
  if (cur_dim > 0)
    multivector_indices_to_string(sc, (index - ind) / size, vect, str, cur_dim - 1);

  snprintf(buf, 64, " %lld", ind);
#ifdef __OpenBSD__
  strlcat(str, buf, 128); /* 128=length of str */
#else
  strcat(str, buf);
#endif
  return(str);
}


static int multivector_to_port(s7_scheme *sc, s7_pointer vec, s7_pointer port,
			       int out_len, int flat_ref, int dimension, int dimensions, bool *last,
			       use_write_t use_write, shared_info *ci)
{
  int i;

  if (use_write != USE_READABLE_WRITE)
    {
      if (*last)
	port_write_string(port)(sc, " (", 2, port);
      else port_write_character(port)(sc, '(', port);
      (*last) = false;
    }

  for (i = 0; i < vector_dimension(vec, dimension); i++)
    {
      if (dimension == (dimensions - 1))
	{
	  if (flat_ref < out_len)
	    {
	      if (use_write == USE_READABLE_WRITE)
		{
		  int plen;
		  char buf[128];
		  char *indices;
		  /* need to translate flat_ref into a set of indices
		   */
		  tmpbuf_calloc(indices, 128);
		  plen = snprintf(buf, 128, "(set! ({v}%s) ", multivector_indices_to_string(sc, flat_ref, vec, indices, dimension));
		  port_write_string(port)(sc, buf, plen, port);
		  tmpbuf_free(indices, 128);
		}
	      object_to_port_with_circle_check(sc, vector_element(vec, flat_ref), port, DONT_USE_DISPLAY(use_write), ci);

	      if (use_write == USE_READABLE_WRITE)
		port_write_string(port)(sc, ") ", 2, port);
	      flat_ref++;
	    }
	  else
	    {
	      port_write_string(port)(sc, "...)", 4, port);
	      return(flat_ref);
	    }
	  if ((use_write != USE_READABLE_WRITE) &&
	      (i < (vector_dimension(vec, dimension) - 1)))
	    port_write_character(port)(sc, ' ', port);
	}
      else
	{
	  if (flat_ref < out_len)
	    flat_ref = multivector_to_port(sc, vec, port, out_len, flat_ref, dimension + 1, dimensions, last, DONT_USE_DISPLAY(use_write), ci);
	  else
	    {
	      port_write_string(port)(sc, "...)", 4, port);
	      return(flat_ref);
	    }
	}
    }
  if (use_write != USE_READABLE_WRITE)
    port_write_character(port)(sc, ')', port);
  (*last) = true;
  return(flat_ref);
}


static void vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  s7_int i, len;
  int plen;
  bool too_long = false;
  char buf[128];

  len = vector_length(vect);
  if (len == 0)
    {
      if (vector_rank(vect) > 1)
	{
	  plen = snprintf(buf, 32, "#%uD()", vector_ndims(vect));
	  port_write_string(port)(sc, buf, plen, port);
	}
      else port_write_string(port)(sc, "#()", 3, port);
      return;
    }

  if (use_write != USE_READABLE_WRITE)
    {
      plen = sc->print_length;
      if (plen <= 0)
	{
	  if (vector_rank(vect) > 1)
	    {
	      plen = snprintf(buf, 32, "#%uD(...)", vector_ndims(vect));
	      port_write_string(port)(sc, buf, plen, port);
	    }
	  else port_write_string(port)(sc, "#(...)", 6, port);
	  return;
	}

      if (len > plen)
	{
	  too_long = true;
	  len = plen;
	}
    }

  if (use_write == USE_READABLE_WRITE)
    {
      if ((ci) &&
	  (peek_shared_ref(ci, vect) != 0))
	{
	  port_write_string(port)(sc, "(let (({v} (make-vector ", 24, port);
	  if (vector_rank(vect) > 1)
	    {
	      unsigned int dim;
	      port_write_string(port)(sc, "'(", 2, port);
	      for (dim = 0; dim < vector_ndims(vect); dim++)
		{
		  plen = snprintf(buf, 128, "%lld ", vector_dimension(vect, dim));
		  port_write_string(port)(sc, buf, plen, port);
		}
	      port_write_string(port)(sc, ")))) ", 5, port);
	    }
	  else 
	    {
	      plen = snprintf(buf, 128, "%lld))) ", vector_length(vect));
	      port_write_string(port)(sc, buf, plen, port);
	    }
	  if (shared_ref(ci, vect) < 0)
	    {
	      plen = snprintf(buf, 128, "(set! {%d} {v}) ", -shared_ref(ci, vect));
	      port_write_string(port)(sc, buf, plen, port);
	    }
	  
	  if (vector_rank(vect) > 1)
	    {
	      bool last = false;
	      multivector_to_port(sc, vect, port, len, 0, 0, vector_ndims(vect), &last, use_write, ci);
	    }
	  else
	    {
	      for (i = 0; i < len; i++)
		{
		  port_write_string(port)(sc, "(set! ({v} ", 11, port);
		  plen = snprintf(buf, 128, "%lld) ", i);
		  port_write_string(port)(sc, buf, plen, port);
		  object_to_port_with_circle_check(sc, vector_element(vect, i), port, use_write, ci);
		  port_write_string(port)(sc, ") ", 2, port);
		}
	    }
	  port_write_string(port)(sc, "{v})", 4, port);
	}
      else /* simple readable case */
	{
	  if (vector_rank(vect) > 1)
	    port_write_string(port)(sc, "(make-shared-vector (vector", 27, port);
	  else port_write_string(port)(sc, "(vector", 7, port);

	  for (i = 0; i < len; i++)
	    {
	      port_write_character(port)(sc, ' ', port);
	      object_to_port_with_circle_check(sc, vector_element(vect, i), port, use_write, ci);
	    }
	  port_write_character(port)(sc, ')', port);

	  if (vector_rank(vect) > 1)
	    {
	      unsigned int dim;
	      port_write_string(port)(sc, " '(", 3, port);
	      for (dim = 0; dim < vector_ndims(vect) - 1; dim++)
		{
		  plen = snprintf(buf, 128, "%lld ", vector_dimension(vect, dim));
		  port_write_string(port)(sc, buf, plen, port);
		}
	      plen = snprintf(buf, 128, "%lld", vector_dimension(vect, dim));
	      port_write_string(port)(sc, buf, plen, port);
	      port_write_string(port)(sc, "))", 2, port);
	    }
	}
    }
  else
    {
      if (vector_rank(vect) > 1)
	{
	  bool last = false;
	  if (vector_ndims(vect) > 1)
	    {
	      plen = snprintf(buf, 32, "#%uD", vector_ndims(vect));
	      port_write_string(port)(sc, buf, plen, port);
	    }
	  else port_write_character(port)(sc, '#', port);
	  multivector_to_port(sc, vect, port, len, 0, 0, vector_ndims(vect), &last, use_write, ci);
	}
      else
	{
	  port_write_string(port)(sc, "#(", 2, port);
	  for (i = 0; i < len - 1; i++)
	    {
	      object_to_port_with_circle_check(sc, vector_element(vect, i), port, DONT_USE_DISPLAY(use_write), ci);
	      port_write_character(port)(sc, ' ', port);
	    }
	  object_to_port_with_circle_check(sc, vector_element(vect, i), port, DONT_USE_DISPLAY(use_write), ci);

	  if (too_long)
	    port_write_string(port)(sc, " ...)", 5, port);
	  else port_write_character(port)(sc, ')', port);
	}
    }
}

static bool string_needs_slashification(const char *str, int len)
{
  /* we have to go by len (str len) not *s==0 because s7 strings can have embedded nulls */
  unsigned char *p, *pend;
  pend = (unsigned char *)(str + len);
  for (p = (unsigned char *)str; p < pend; p++)
    if (slashify_table[*p])
      return(true);
  return(false);
}

#define IN_QUOTES true
#define NOT_IN_QUOTES false

static char *slashify_string(s7_scheme *sc, const char *p, int len, bool quoted, int *nlen) /* do not free result */
{
  int j = 0, cur_size, size;
  char *s;
  unsigned char *pcur, *pend;

  pend = (unsigned char *)(p + len);
  size = len + 256;
  if (size > sc->slash_str_size)
    {
      if (sc->slash_str) free(sc->slash_str);
      sc->slash_str_size = size;
      sc->slash_str = (char *)malloc(size);
    }
  else size = sc->slash_str_size;
  cur_size = size - 2;

  /* memset((void *)sc->slash_str, 0, size); */
  s = sc->slash_str;

  if (quoted) s[j++] = '"';

  /* what about the trailing nulls? Guile writes them out (as does s7 currently)
   *    but that is not ideal.  I'd like to use ~S for error messages, so that
   *    strings are clearly identified via the double-quotes, but this way of
   *    writing them is ugly:
   *
   *    :(let ((str (make-string 8 #\null))) (set! (str 0) #\a) str)
   *    "a\x00\x00\x00\x00\x00\x00\x00"
   *
   * but it would be misleading to omit them because:
   *
   *    :(let ((str (make-string 8 #\null))) (set! (str 0) #\a) (string-append str "bc"))
   *    "a\x00\x00\x00\x00\x00\x00\x00bc"
   */

  for (pcur = (unsigned char *)p; pcur < pend; pcur++)
    {
      if (slashify_table[*pcur])
	{
	  s[j++] = '\\';
	  switch (*pcur)
	    {
	    case '"':
	      s[j++] = '"';
	      break;

	    case '\\':
	      s[j++] = '\\';
	      break;

	    default:               /* this is the "\x01" stuff */
	      {
		unsigned int n;
		static char dignum[] = "0123456789abcdef";
		s[j++] = 'x';
		n = (unsigned int)(*pcur);
		if (n < 16)
		  s[j++] = '0';
		else s[j++] = dignum[(n / 16) % 16];
		s[j++] = dignum[n % 16];
	      }
	      break;
	    }
	}
      else s[j++] = *pcur;
      if (j >= cur_size) /* even with 256 extra, we can overflow (for example, inordinately many tabs in ALSA output) */
	{
	  /* int k; */
	  size *= 2;
	  sc->slash_str = (char *)realloc(sc->slash_str, size * sizeof(char));    
	  sc->slash_str_size = size;
	  cur_size = size - 2;
	  s = sc->slash_str;
	  /* for (k = j; k < size; k++) s[k] = 0; */
	}
    }
  if (quoted) s[j++] = '"';
  s[j] = '\0';
  (*nlen) = j;
  return(s);
}

static void output_port_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write)
{
  if ((obj == sc->standard_output) ||
      (obj == sc->standard_error))
    port_write_string(port)(sc, port_filename(obj), port_filename_length(obj), port);
  else
    {
      int nlen;
      if (use_write == USE_READABLE_WRITE)
	{
	  if (port_is_closed(obj))
	    port_write_string(port)(sc, "(let ((p (open-output-string))) (close-output-port p) p)", 56, port);
	  else 
	    {
	      char *str;
	      if (is_string_port(obj))
		{
		  port_write_string(port)(sc, "(let ((p (open-output-string)))", 31, port);
		  if (port_position(obj) > 0)
		    {
		      port_write_string(port)(sc, " (display ", 10, port);
		      str = slashify_string(sc, (const char *)port_data(obj), port_position(obj), IN_QUOTES, &nlen);
		      port_write_string(port)(sc, str, nlen, port);
		      port_write_string(port)(sc, " p)", 3, port);
		    }
		  port_write_string(port)(sc, " p)", 3, port);
		}
	      else 
		{
		  str = (char *)malloc(256 * sizeof(char));
		  nlen = snprintf(str, 256, "(open-output-file \"%s\" \"a\")", port_filename(obj));
		  port_write_string(port)(sc, str, nlen, port);
		  free(str);
		}
	    }
	}
      else
	{
	  if (is_string_port(obj))
	    port_write_string(port)(sc, "<output-string-port", 19, port);
	  else
	    {
	      if (is_file_port(obj))
		port_write_string(port)(sc, "<output-file-port", 17, port);
	      else port_write_string(port)(sc, "<output-function-port", 21, port);
	    }
	  if (port_is_closed(obj)) 
	    port_write_string(port)(sc, " (closed)>", 10, port);
	  else port_write_character(port)(sc, '>', port);
	}
    }
}

static void input_port_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write)
{
  if (obj == sc->standard_input)
    port_write_string(port)(sc, port_filename(obj), port_filename_length(obj), port);
  else
    {
      int nlen = 0;
      if (use_write == USE_READABLE_WRITE)
	{
	  if (port_is_closed(obj))
	    port_write_string(port)(sc, "(call-with-input-string \"\" (lambda (p) p))", 42, port);
	  else
	    {
	      if (is_function_port(obj))
		port_write_string(port)(sc, "#<function input port>", 22, port);
	      else
		{
		  char *str;
		  if (is_file_port(obj))
		    {
		      str = (char *)malloc(256 * sizeof(char));
		      nlen = snprintf(str, 256, "(open-input-file \"%s\")", port_filename(obj));
		      port_write_string(port)(sc, str, nlen, port);
		      free(str);
		    }
		  else
		    {
		      /* if the string is large, slashify_string is a problem. Usually this is actually
		       *   a file port where the contents were read in one (up to 5MB) gulp, so the
		       *   readable version could be: open file, read-char to the current port_position.
		       *   s7_port_filename(port) has the char* name if any.
		       */
		      int data_len;
		      data_len = port_data_size(obj) - port_position(obj);
		      if (data_len > 100)
			{
			  const char *filename;
			  filename = (const char *)s7_port_filename(obj);
			  if (filename)
			    {
			      #define DO_STR_LEN 1024
			      char *do_str;
			      int len;
			      do_str = (char *)malloc(DO_STR_LEN * sizeof(char));
			      if (port_position(obj) > 0)
				{
				  len = snprintf(do_str, DO_STR_LEN, "(let ((port (open-input-file \"%s\")))", filename);
				  port_write_string(port)(sc, do_str, len, port);
				  len = snprintf(do_str, DO_STR_LEN, " (do ((i 0 (+ i 1)) (c (read-char port) (read-char port))) ((= i %u) port)))", 
						 port_position(obj) - 1);
				}
			      else len = snprintf(do_str, DO_STR_LEN, "(open-input-file \"%s\")", filename);
			      port_write_string(port)(sc, do_str, len, port);
			      free(do_str);
			      return;
			    }
			}
		      port_write_string(port)(sc, "(open-input-string ", 19, port);
		      /* not port_write_string here because there might be embedded double-quotes */
		      str = slashify_string(sc, (const char *)(port_data(obj) + port_position(obj)), port_data_size(obj) - port_position(obj), IN_QUOTES, &nlen);
		      port_write_string(port)(sc, str, nlen, port);
		      port_write_character(port)(sc, ')', port);
		    }
		}
	    }
	}
      else
	{
	  if (is_string_port(obj))
	    port_write_string(port)(sc, "<input-string-port", 18, port);
	  else
	    {
	      if (is_file_port(obj))
		port_write_string(port)(sc, "<input-file-port", 16, port);
	      else port_write_string(port)(sc, "<input-function-port", 20, port);
	    }
	  if (port_is_closed(obj)) 
	    port_write_string(port)(sc, " (closed)>", 10, port);
	  else port_write_character(port)(sc, '>', port);
	}
    }
}

static bool symbol_needs_slashification(s7_pointer obj)
{
  unsigned char *p, *pend;
  const char *str;
  int len;
  str = symbol_name(obj);
  if (str[0] == '#')
    return(true);
  len = symbol_name_length(obj);
  pend = (unsigned char *)(str + len);
  for (p = (unsigned char *)str; p < pend; p++)
    if (symbol_slashify_table[*p])
      return(true);
  set_clean_symbol(obj);
  return(false);
}

static void symbol_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write)
{
  /* I think this is the only place we print a symbol's name
   *   but in the readable case, what about (symbol "1;3")? it actually seems ok!
   */
  if ((!is_clean_symbol(obj)) &&
      (symbol_needs_slashification(obj)))
    {
      int nlen = 0;
      char *str, *symstr;
      str = slashify_string(sc, symbol_name(obj), symbol_name_length(obj), NOT_IN_QUOTES, &nlen);
      nlen += 16;
      tmpbuf_malloc(symstr, nlen);
      nlen = snprintf(symstr, nlen, "(symbol \"%s\")", str);
      port_write_string(port)(sc, symstr, nlen, port);
      tmpbuf_free(symstr, nlen);
    }
  else
    {
      if ((use_write == USE_READABLE_WRITE) &&
	  (!is_keyword(obj)))
	port_write_character(port)(sc, '\'', port);
      if (is_string_port(port))
	{
	  int new_len;
	  new_len = port_position(port) + symbol_name_length(obj);
	  if (new_len >= (int)port_data_size(port))
	    resize_port_data(port, new_len * 2);
	  memcpy((void *)(port_data(port) + port_position(port)), (void *)symbol_name(obj), symbol_name_length(obj));
	  port_position(port) = new_len;
	}
      else port_write_string(port)(sc, symbol_name(obj), symbol_name_length(obj), port);
    }
}

static void string_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write)
{
  if (string_length(obj) > 0)
    {
      /* this used to check for length > 1<<24 -- is that still necessary?
       * since string_length is a scheme length, not C, this write can embed nulls from C's point of view
       */
      if (use_write == USE_DISPLAY)
	port_write_string(port)(sc, string_value(obj), string_length(obj), port);
      else
	{
	  if (!string_needs_slashification(string_value(obj), string_length(obj)))
	    {
	      port_write_character(port)(sc, '"', port);
	      port_write_string(port)(sc, string_value(obj), string_length(obj), port);
	      port_write_character(port)(sc, '"', port);
	    }
	  else
	    {
	      char *str;
	      int nlen = 0;
	      str = slashify_string(sc, string_value(obj), string_length(obj), IN_QUOTES, &nlen);
	      port_write_string(port)(sc, str, nlen, port);
	    }
	}
    }
  else
    {
      if (use_write != USE_DISPLAY)
	port_write_string(port)(sc, "\"\"", 2, port);
    }
}

static void byte_vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write)
{
  s7_int i, len;
  int plen;
  bool too_long = false;

  len = string_length(vect);
  if (use_write == USE_READABLE_WRITE)
    plen = len;
  else plen = sc->print_length;

  if (len == 0)
    port_write_string(port)(sc, "#u8()", 5, port);
  else
    {
      if (plen <= 0)
	port_write_string(port)(sc, "#u8(...)", 8, port);
      else
	{
	  unsigned int nlen;
	  char *p;
	  if (len > plen)
	    {
	      too_long = true;
	      len = plen;
	    }
	  port_write_string(port)(sc, "#u8(", 4, port);
	  for (i = 0; i < len - 1; i++)
	    {
	      p = pos_int_to_str((int)((unsigned char)string_value(vect)[i]), &nlen, ' ');
	      port_write_string(port)(sc, p, nlen - 1, port);
	    }
	  p = pos_int_to_str((int)((unsigned char)string_value(vect)[i]), &nlen, (too_long) ? '\0' : ')');
	  port_write_string(port)(sc, p, nlen - 1, port);

	  if (too_long)
	    port_write_string(port)(sc, " ...)", 5, port);
	}
    }
}


static void int_or_float_vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write)
{
  s7_int i, len;
  int plen;
  bool too_long = false;

  len = vector_length(vect);
  if (use_write == USE_READABLE_WRITE)
    plen = len;
  else plen = sc->print_length;

  if (len == 0)
    port_write_string(port)(sc, "#()", 3, port);
  else
    {
      if (plen <= 0)
	port_write_string(port)(sc, "#(...)", 6, port);
      else
	{
	  char buf[128];
	  if (len > plen)
	    {
	      too_long = true;
	      len = plen;
	    }
	  if (is_int_vector(vect))
	    {
	      if (vector_rank(vect) > 1)
		port_write_string(port)(sc, "(make-shared-vector (int-vector", 31, port);
	      else port_write_string(port)(sc, "(int-vector", 11, port);

	      if (!is_string_port(port))
		{
		  for (i = 0; i < len; i++)
		    {
		      plen = snprintf(buf, 128, " %lld", int_vector_element(vect, i));
		      port_write_string(port)(sc, buf, plen, port);
		    }
		}
	      else
		{
		  /* an experiment */
		  int new_len, next_len;
		  unsigned char *dbuf;
		  new_len = port_position(port);
		  next_len = port_data_size(port) - 128;
		  dbuf = port_data(port);

		  for (i = 0; i < len; i++)
		    {
		      if (new_len >= next_len)
			{
			  resize_port_data(port, port_data_size(port) * 2);
			  next_len = port_data_size(port) - 128;
			  dbuf = port_data(port);
			}
		      plen = snprintf((char *)(dbuf + new_len), 128, " %lld", int_vector_element(vect, i));
		      new_len += plen;
		    }
		  port_position(port) = new_len;
		}
	    }
	  else
	    {
	      if (vector_rank(vect) > 1)
		port_write_string(port)(sc, "(make-shared-vector (float-vector", 33, port);
	      else port_write_string(port)(sc, "(float-vector", 13, port);

	      for (i = 0; i < len; i++)
		{
		  port_write_character(port)(sc, ' ', port);
		  plen = snprintf(buf, 124, float_format_g, float_format_precision, float_vector_element(vect, i)); /* 124 so floatify has room */
		  floatify(buf, &plen);
		  port_write_string(port)(sc, buf, plen, port);
		}
	    }

	  if (too_long)
	    port_write_string(port)(sc, " ...)", 5, port);
	  else port_write_character(port)(sc, ')', port);

	  if (vector_rank(vect) > 1)
	    {
	      unsigned int dim;
	      port_write_string(port)(sc, " '(", 3, port);
	      for (dim = 0; dim < vector_ndims(vect) - 1; dim++)
		{
		  plen = snprintf(buf, 128, "%lld ", vector_dimension(vect, dim));
		  port_write_string(port)(sc, buf, plen, port);
		}
	      plen = snprintf(buf, 128, "%lld", vector_dimension(vect, dim));
	      port_write_string(port)(sc, buf, plen, port);
	      port_write_string(port)(sc, "))", 2, port);
	    }
	}
    }
}


static void list_to_port(s7_scheme *sc, s7_pointer lst, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  /* we need list_to_starboard... */
  s7_pointer x;
  int i, len, true_len;

  true_len = s7_list_length(sc, lst);
  if (true_len < 0)                    /* a dotted list -- handle cars, then final cdr */
    len = (-true_len + 1);
  else
    {
      if (true_len == 0)               /* either () or a circular list */
	{
	  if (is_not_null(lst))
	    len = circular_list_entries(lst);
	  else
	    {
	      port_write_string(port)(sc, "()", 2, port);
	      return;
	    }
	}
      else len = true_len;
    }

  if (((car(lst) == sc->quote_symbol) ||
       (car(lst) == sc->quote_unchecked_symbol)) && /* this can happen (see lint.scm) */
      (true_len == 2))
    {
      /* len == 1 is important, otherwise (list 'quote 1 2) -> '1 2 which looks weird
       *   or (object->string (apply . `''1)) -> "'quote 1"
       * so (quote x) = 'x but (quote x y z) should be left alone (if evaluated, it's an error)
       */
      port_write_character(port)(sc, '\'', port);
      object_to_port_with_circle_check(sc, cadr(lst), port, USE_WRITE, ci);
      return;
    }
  else port_write_character(port)(sc, '(', port);

  if (is_multiple_value(lst))
    port_write_string(port)(sc, "values ", 7, port);

  if (use_write == USE_READABLE_WRITE)
    {
      if (ci)
	{
	  int plen;
	  char buf[128];

	  port_write_string(port)(sc, "let (({lst} (make-list ", 23, port);
	  plen = snprintf(buf, 128, "%d))) ", len);
	  port_write_string(port)(sc, buf, plen, port);

	  if ((shared_ref(ci, lst) < 0))
	    {
	      plen = snprintf(buf, 128, "(set! {%d} {lst}) ", -shared_ref(ci, lst));
	      port_write_string(port)(sc, buf, plen, port);
	    }

	  port_write_string(port)(sc, "(let (({x} {lst})) ", 19, port);
	  for (i = 0, x = lst; (i < len) && (is_pair(x)); i++, x = cdr(x))
	    {
	      port_write_string(port)(sc, "(set-car! {x} ", 14, port);
	      object_to_port_with_circle_check(sc, car(x), port, use_write, ci);
	      port_write_string(port)(sc, ") ", 2, port);
	      if (i < len - 1)
		port_write_string(port)(sc, "(set! {x} (cdr {x})) ", 21, port);
	    }
	  if (!is_null(x))
	    {
	      port_write_string(port)(sc, "(set-cdr! {x} ", 14, port);
	      object_to_port_with_circle_check(sc, x, port, use_write, ci);
	      port_write_string(port)(sc, ") ", 2, port);
	    }
	  port_write_string(port)(sc, ") {lst})", 8, port);
	}
      else
	{
	  /* the easier cases: no circles or shared refs to patch up */
	  if (true_len > 0)
	    {
	      port_write_string(port)(sc, "list", 4, port);
	      for (x = lst; is_pair(x); x = cdr(x))
		{
		  port_write_character(port)(sc, ' ', port);
		  object_to_port_with_circle_check(sc, car(x), port, use_write, ci);
		}
	      port_write_character(port)(sc, ')', port);
	    }
	  else
	    {
	      port_write_string(port)(sc, "cons ", 5, port);
	      object_to_port_with_circle_check(sc, car(lst), port, use_write, ci);
	      for (x = cdr(lst); is_pair(x); x = cdr(x))
		{
		  port_write_character(port)(sc, ' ', port);
		  port_write_string(port)(sc, "(cons ", 6, port);
		  object_to_port_with_circle_check(sc, car(x), port, use_write, ci);
		}
	      port_write_character(port)(sc, ' ', port);
	      object_to_port_with_circle_check(sc, x, port, use_write, ci);
	      for (i = 1; i < len; i++)
		port_write_character(port)(sc, ')', port);
	    }
	}
    }
  else
    {
      if (ci)
	{
	  for (x = lst, i = 0; (is_pair(x)) && (i < len) && ((!ci) || (i == 0) || (peek_shared_ref(ci, x) == 0)); i++, x = cdr(x))
	    {
	      object_to_port_with_circle_check(sc, car(x), port, DONT_USE_DISPLAY(use_write), ci);
	      if (i < (len - 1))
		port_write_character(port)(sc, ' ', port);
	    }
	  if (is_not_null(x))
	    {
	      if ((true_len == 0) &&
		  (i == len))
		port_write_string(port)(sc, " . ", 3, port);
	      else port_write_string(port)(sc, ". ", 2, port);
	      object_to_port_with_circle_check(sc, x, port, DONT_USE_DISPLAY(use_write), ci);
	    }
	  port_write_character(port)(sc, ')', port);
	}
      else
	{
	  for (x = lst, i = 0; (is_pair(x)) && (i < len); i++, x = cdr(x))
	    {
	      object_to_port(sc, car(x), port, DONT_USE_DISPLAY(use_write), ci);
	      if (i < (len - 1))
		port_write_character(port)(sc, ' ', port);
	    }
	  if (is_not_null(x))
	    {
	      port_write_string(port)(sc, ". ", 2, port);
	      object_to_port(sc, x, port, DONT_USE_DISPLAY(use_write), ci);
	    }
	  port_write_character(port)(sc, ')', port);
	}
    }
}


static void hash_table_to_port(s7_scheme *sc, s7_pointer hash, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  int i, len;
  unsigned int gc_iter;
  bool too_long = false;
  s7_pointer iterator, p;

  /* if hash is a member of ci, just print its number
   * (let ((ht (hash-table '(a . 1)))) (hash-table-set! ht 'b ht))
   *
   * since equal? does not care about the hash-table lengths, we can ignore that complication in the :readable case
   */

  len = hash_table_entries(hash);
  if (len == 0)
    {
      port_write_string(port)(sc, "(hash-table)", 12, port);
      return;
    }

  if (use_write != USE_READABLE_WRITE)
    {
      s7_int plen;
      plen = sc->print_length;
      if (plen <= 0)
	{
	  port_write_string(port)(sc, "(hash-table ...)", 16, port);
	  return;
	}
      if (len > plen)
	{
	  too_long = true;
	  len = plen;
	}
    }

  iterator = s7_make_iterator(sc, hash);
  gc_iter = s7_gc_protect(sc, iterator);
  p = cons(sc, sc->F, sc->F);
  iterator_current(iterator) = p;
  set_mark_seq(iterator);

  if ((use_write == USE_READABLE_WRITE) &&
      (ci) &&
      (peek_shared_ref(ci, hash) != 0))
    {
      port_write_string(port)(sc, "(let (({ht} (make-hash-table)))", 31, port);
      if (shared_ref(ci, hash) < 0)
	{
	  int plen;
	  char buf[64];
	  plen = snprintf(buf, 64, "(set! {%d} {ht}) ", -shared_ref(ci, hash));
	  port_write_string(port)(sc, buf, plen, port);
	}
      for (i = 0; i < len; i++)
	{
	  s7_pointer key_val, key, val;

	  key_val = hash_table_iterate(sc, iterator);
	  key = car(key_val);
	  val = cdr(key_val);

	  port_write_string(port)(sc, " (set! ({ht} ", 13, port);
	  if (key == hash)
	    port_write_string(port)(sc, "{ht}", 4, port);
	  else object_to_port_with_circle_check(sc, key, port, USE_READABLE_WRITE, ci);
	  port_write_string(port)(sc, ") ", 2, port);
	  if (val == hash)
	    port_write_string(port)(sc, "{ht}", 4, port);
	  else object_to_port_with_circle_check(sc, val, port, USE_READABLE_WRITE, ci);
	  port_write_character(port)(sc, ')', port);
	}
      port_write_string(port)(sc, " {ht})", 6, port);
    }
  else
    {
      port_write_string(port)(sc, "(hash-table", 11, port);
      for (i = 0; i < len; i++)
	{
	  s7_pointer key_val;
	  if (use_write == USE_READABLE_WRITE)
	    port_write_character(port)(sc, ' ', port);
	  else port_write_string(port)(sc, " '", 2, port);
	  key_val = hash_table_iterate(sc, iterator);
	  object_to_port_with_circle_check(sc, key_val, port, DONT_USE_DISPLAY(use_write), ci);
	}

      if (too_long)
	port_write_string(port)(sc, " ...)", 5, port);
      else port_write_character(port)(sc, ')', port);
    }

  s7_gc_unprotect_at(sc, gc_iter);
}


static int slot_to_port_1(s7_scheme *sc, s7_pointer x, s7_pointer port, use_write_t use_write, shared_info *ci, int n)
{
  if (is_slot(x))
    {
      n = slot_to_port_1(sc, next_slot(x), port, use_write, ci, n);
      if (n <= sc->print_length)
	{
	  port_write_character(port)(sc, ' ', port);
	  object_to_port_with_circle_check(sc, x, port, use_write, ci);
	}
      if (n == (sc->print_length + 1))
	port_write_string(port)(sc, " ...", 4, port);
    }
  return(n + 1);
}

static void let_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  /* if outer env points to (say) method list, the object needs to specialize object->string itself */
  if (has_methods(obj))
    {
      s7_pointer print_func;
      print_func = find_method(sc, obj, sc->object_to_string_symbol);
      if (print_func != sc->undefined)
	{
	  s7_pointer p;
	  /* what needs to be protected here? for one, the function might not return a string! */

	  clear_has_methods(obj);
	  if (use_write == USE_WRITE)
	    p = s7_apply_function(sc, print_func, list_1(sc, obj));
	  else p = s7_apply_function(sc, print_func, list_2(sc, obj, (use_write == USE_DISPLAY) ? sc->F : sc->key_readable_symbol));
	  set_has_methods(obj);

	  if ((is_string(p)) && 
	      (string_length(p) > 0))
	    port_write_string(port)(sc, string_value(p), string_length(p), port);
	  return;
	}
    }
  if (obj == sc->rootlet)
    port_write_string(port)(sc, "(rootlet)", 9, port);
  else
    {
      if (sc->short_print)
	port_write_string(port)(sc, "#<let>", 6, port);
      else
	{
	  /* circles can happen here:
	   *    (let () (let ((b (curlet))) (curlet)))
	   *    #<let 'b #<let>>
	   * or (let ((b #f)) (set! b (curlet)) (curlet))
	   *    #1=#<let 'b #1#>
	   */
	  if ((use_write == USE_READABLE_WRITE) &&
	      (ci) &&
	      (peek_shared_ref(ci, obj) != 0))
	    {
	      s7_pointer x;
	      port_write_string(port)(sc, "(let (({e} (inlet))) ", 21, port);
	      if ((ci) &&
		  (shared_ref(ci, obj) < 0))
		{
		  int plen;
		  char buf[64];
		  plen = snprintf(buf, 64, "(set! {%d} {e}) ", -shared_ref(ci, obj));
		  port_write_string(port)(sc, buf, plen, port);
		}
	      
	      port_write_string(port)(sc, "(apply varlet {e} (reverse (list ", 33, port);
	      for (x = let_slots(obj); is_slot(x); x = next_slot(x))
		{
		  port_write_string(port)(sc, "(cons ", 6, port);
		  symbol_to_port(sc, slot_symbol(x), port, use_write);
		  port_write_character(port)(sc, ' ', port);
		  object_to_port_with_circle_check(sc, slot_value(x), port, use_write, ci);
		  port_write_character(port)(sc, ')', port);
		}
	      port_write_string(port)(sc, "))) {e})", 8, port);
	    }
	  else
	    {
	      port_write_string(port)(sc, "(inlet", 6, port);
	      slot_to_port_1(sc, let_slots(obj), port, use_write, ci, 0);
	      port_write_character(port)(sc, ')', port);
	    }
	}
    }
}


static void write_macro_readably(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  s7_pointer arglist, body, expr;

  body = closure_body(obj);
  arglist = closure_args(obj);

  port_write_string(port)(sc, "(define-", 8, port);
  port_write_string(port)(sc, ((is_macro(obj)) || (is_macro_star(obj))) ? "macro" : "bacro", 5, port);
  if ((is_macro_star(obj)) || (is_bacro_star(obj)))
    port_write_character(port)(sc, '*', port);
  port_write_string(port)(sc, " (_m_", 5, port);
  if (is_symbol(arglist))
    {
      port_write_string(port)(sc, " . ", 3, port);
      port_write_string(port)(sc, symbol_name(arglist), symbol_name_length(arglist), port);
    }
  else
    {
      if (is_pair(arglist))
	{
	  for (expr = arglist; is_pair(expr); expr = cdr(expr))
	    {
	      port_write_character(port)(sc, ' ', port);
	      object_to_port(sc, car(expr), port, USE_WRITE, NULL);
	    }
	  if (!is_null(expr))
	    {
	      port_write_string(port)(sc, " . ", 3, port);
	      object_to_port(sc, expr, port, USE_WRITE, NULL);
	    }
	}
    }
  port_write_string(port)(sc, ") ", 2, port);
  for (expr = body; is_pair(expr); expr = cdr(expr))
    object_to_port(sc, car(expr), port, USE_WRITE, NULL);
  port_write_character(port)(sc, ')', port);
}


static s7_pointer match_symbol(s7_scheme *sc, s7_pointer symbol, s7_pointer e)
{
  s7_pointer y, le;
  for (le = e; is_let(le) && (le != sc->rootlet); le = outlet(le))
    for (y = let_slots(le); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(y);
  return(NULL);
}

static bool slot_memq(s7_pointer symbol, s7_pointer symbols)
{
  s7_pointer x;
  for (x = symbols; is_pair(x); x = cdr(x))
    if (slot_symbol(car(x)) == symbol)
      return(true);
  return(false);
}

static bool arg_memq(s7_pointer symbol, s7_pointer args)
{
  s7_pointer x;
  for (x = args; is_pair(x); x = cdr(x))
    if ((car(x) == symbol) ||
	((is_pair(car(x))) &&
	 (caar(x) == symbol)))
      return(true);
  return(false);
}


static void collect_locals(s7_scheme *sc, s7_pointer body, s7_pointer e, s7_pointer args, unsigned int gc_loc)
{
  if (is_pair(body))
    {
      collect_locals(sc, car(body), e, args, gc_loc);
      collect_locals(sc, cdr(body), e, args, gc_loc);
    }
  else
    {
      if ((is_symbol(body)) &&
	  (!arg_memq(body, args)) &&
	  (!slot_memq(body, gc_protected_at(sc, gc_loc))))
	{
	  s7_pointer slot;
	  slot = match_symbol(sc, body, e);
	  if (slot)
	    gc_protected_at(sc, gc_loc) = cons(sc, slot, gc_protected_at(sc, gc_loc));
	}
    }
}



static s7_pointer find_closure(s7_scheme *sc, s7_pointer closure, s7_pointer cur_env)
{
  s7_pointer e, y;
  for (e = cur_env; is_let(e); e = outlet(e))
    {
      if ((is_function_env(e)) &&
	  (is_global(funclet_function(e))) &&         /* (define (f1) (lambda () 1)) shouldn't say the returned closure is named f1 */
	  (slot_value(global_slot(funclet_function(e))) == closure))
	return(funclet_function(e));

      for (y = let_slots(e); is_slot(y); y = next_slot(y))
	if (slot_value(y) == closure)
	  return(slot_symbol(y));
    }
  return(sc->nil);
}

static void write_closure_name(s7_scheme *sc, s7_pointer closure, s7_pointer port)
{
  s7_pointer x;
  x = find_closure(sc, closure, closure_let(closure));
  /* this can be confusing!  In some cases, the function is in its environment, and in other very similar-looking cases it isn't:
   * (let ((a (lambda () 1))) a)
   * #<lambda ()>
   * (letrec ((a (lambda () 1))) a)
   * a
   * (let () (define (a) 1) a)
   * a
   */
  if (is_symbol(x)) /* after find_closure */
    {
      port_write_string(port)(sc, symbol_name(x), symbol_name_length(x), port);
      return;
    }
  
  /* names like #<closure> and #<macro> are useless -- try to be a bit more informative */
  switch (type(closure))
    {
    case T_CLOSURE:      
      port_write_string(port)(sc, "#<lambda ", 9, port);  
      break;

    case T_CLOSURE_STAR: 
      port_write_string(port)(sc, "#<lambda* ", 10, port);  
      break;

    case T_MACRO:        
      if (is_expansion(closure)) 
	port_write_string(port)(sc, "#<expansion ", 12, port); 
      else port_write_string(port)(sc, "#<macro ", 8, port); 
      break;

    case T_MACRO_STAR:   
      port_write_string(port)(sc, "#<macro* ", 9, port);  
      break;
						   
    case T_BACRO:        
      port_write_string(port)(sc, "#<bacro ", 8, port);   
      break;

    case T_BACRO_STAR:   
      port_write_string(port)(sc, "#<bacro* ", 9, port); 
      break;
    }

  if (is_null(closure_args(closure)))
    port_write_string(port)(sc, "()>", 3, port);
  else
    {
      s7_pointer args;
      args = closure_args(closure);
      if (is_symbol(args))
	{
	  port_write_string(port)(sc, symbol_name(args), symbol_name_length(args), port);
	  port_write_character(port)(sc, '>', port);    /* (lambda a a) -> #<lambda a> */
	}
      else
	{
	  port_write_character(port)(sc, '(', port);
	  x = car(args);
	  if (is_pair(x)) x = car(x);
	  port_write_string(port)(sc, symbol_name(x), symbol_name_length(x), port);
	  if (!is_null(cdr(args)))
	    {
	      s7_pointer y;
	      port_write_character(port)(sc, ' ', port);
	      if (is_pair(cdr(args)))
		{
		  y = cadr(args);
		  if (is_pair(y)) 
		    y = car(y);
		  else
		    {
		      if (y == sc->key_rest_symbol)
			{
			  port_write_string(port)(sc, ":rest ", 6, port);
			  args = cdr(args);
			  y = cadr(args);
			  if (is_pair(y)) y = car(y);
			}
		    }
		}
	      else 
		{
		  port_write_string(port)(sc, ". ", 2, port);
		  y = cdr(args);
		}
	      port_write_string(port)(sc, symbol_name(y), symbol_name_length(y), port);
	      if ((is_pair(cdr(args))) &&
		  (!is_null(cddr(args))))
		port_write_string(port)(sc, " ...", 4, port);
	    }
	  port_write_string(port)(sc, ")>", 2, port);
	}
    }
}

static s7_pointer closure_name(s7_scheme *sc, s7_pointer closure)
{
  /* this is used by the error handlers to get the current function name
   */
  s7_pointer x;

  x = find_closure(sc, closure, sc->envir);
  if (is_symbol(x))
    return(x);

  if (is_pair(current_code(sc)))
    return(current_code(sc));

  return(closure); /* desperation -- the parameter list (caar here) will cause endless confusion in OP_APPLY errors! */
}


static void write_closure_readably_1(s7_scheme *sc, s7_pointer obj, s7_pointer arglist, s7_pointer body, s7_pointer port)
{
  s7_int old_print_length;
  s7_pointer p;

  if (type(obj) == T_CLOSURE_STAR)  
    port_write_string(port)(sc, "(lambda* ", 9, port);
  else port_write_string(port)(sc, "(lambda ", 8, port);

  if ((is_pair(arglist)) &&
      (allows_other_keys(arglist)))
    {
      sc->temp9 = s7_append(sc, arglist, cons(sc, sc->key_allow_other_keys_symbol, sc->nil));
      object_out(sc, sc->temp9, port, USE_WRITE);
      sc->temp9 = sc->nil;
    }
  else object_out(sc, arglist, port, USE_WRITE); /* here we just want the straight output (a b) not (list 'a 'b) */

  old_print_length = sc->print_length;
  sc->print_length = 1048576;
  for (p = body; is_pair(p); p = cdr(p))
    {
      port_write_character(port)(sc, ' ', port);
      object_out(sc, car(p), port, USE_WRITE);
    }
  port_write_character(port)(sc, ')', port);
  sc->print_length = old_print_length;
}

static void write_closure_readably(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  s7_pointer body, arglist, pe, local_slots, setter = NULL;
  unsigned int gc_loc;
  
  body = closure_body(obj);
  arglist = closure_args(obj);
  pe = closure_let(obj);               /* perhaps check for documentation? */

  gc_loc = s7_gc_protect(sc, sc->nil);
  collect_locals(sc, body, pe, arglist, gc_loc);   /* collect locals used only here */
  if (s7_is_dilambda(obj))
    {
      setter = closure_setter(obj);
      if ((!(has_closure_let(setter))) ||
	  (closure_let(setter) != pe))
	setter = NULL;
    }
  if (setter)
    collect_locals(sc, closure_body(setter), pe, closure_args(setter), gc_loc);
  local_slots = _TLst(gc_protected_at(sc, gc_loc)); /* possibly a list of slots */

  if (!is_null(local_slots))
    {
      s7_pointer x;
      port_write_string(port)(sc, "(let (", 6, port);
      for (x = local_slots; is_pair(x); x = cdr(x))
	{
	  s7_pointer slot;
	  slot = car(x);
	  port_write_character(port)(sc, '(', port);
	  port_write_string(port)(sc, symbol_name(slot_symbol(slot)), symbol_name_length(slot_symbol(slot)), port);
	  port_write_character(port)(sc, ' ', port);
	  object_out(sc, slot_value(slot), port, USE_WRITE);
	  if (is_null(cdr(x)))
	    port_write_character(port)(sc, ')', port);
	  else port_write_string(port)(sc, ") ", 2, port);
	}
      port_write_string(port)(sc, ") ", 2, port);
    }

  if (setter)
    port_write_string(port)(sc, "(dilambda ", 10, port);

  write_closure_readably_1(sc, obj, arglist, body, port);

  if (setter)
    {
      port_write_character(port)(sc, ' ', port);
      write_closure_readably_1(sc, setter, closure_args(setter), closure_body(setter), port);
      port_write_character(port)(sc, ')', port);
    }

  if (!is_null(local_slots))
    port_write_character(port)(sc, ')', port);
  s7_gc_unprotect_at(sc, gc_loc);
}


#if TRAP_SEGFAULT
#include <signal.h>
static sigjmp_buf senv; /* global here is not a problem -- it is used only to protect s7_is_valid */
static volatile sig_atomic_t can_jump = 0;
static void segv(int ignored) {if (can_jump) siglongjmp(senv, 1);}
#endif

bool s7_is_valid(s7_scheme *sc, s7_pointer arg)
{
  bool result = false;
  if (!arg) return(false);

#if TRAP_SEGFAULT
  if (sigsetjmp(senv, 1) == 0)
    {
      void (*old_segv)(int sig);
      can_jump = 1;
      old_segv = signal(SIGSEGV, segv);
#endif
      result = ((!is_free(arg)) &&
		(type(arg) < NUM_TYPES) &&
		(arg->hloc >= not_heap) &&
		((arg->hloc < 0) ||
		 ((arg->hloc < (int)sc->heap_size) && (sc->heap[arg->hloc] == arg))));

#if TRAP_SEGFAULT
      signal(SIGSEGV, old_segv);
    }
  else result = false;
  can_jump = 0;
#endif

  return(result);
}

enum {NO_ARTICLE, INDEFINITE_ARTICLE};

static char *describe_type_bits(s7_scheme *sc, s7_pointer obj)
{
  unsigned int full_typ;
  unsigned char typ;
  char *buf;

  buf = (char *)malloc(512 * sizeof(char));
  typ = unchecked_type(obj);
  full_typ = typeflag(obj);

  /* if debugging all of these bits are being watched, so we need some ugly subterfuges */
  snprintf(buf, 512, "type: %d (%s), flags: #x%x%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
	   typ,
	   type_name(sc, obj, NO_ARTICLE),
	   full_typ,
	   ((full_typ & T_PROCEDURE) != 0) ?             " procedure" : "",
	   ((full_typ & T_GC_MARK) != 0) ?               " gc-marked" : "",
	   ((full_typ & T_IMMUTABLE) != 0) ?             " immutable" : "",
	   ((full_typ & T_EXPANSION) != 0) ?             " expansion" : "",
	   ((full_typ & T_MULTIPLE_VALUE) != 0) ?        " values or matched" : "",
	   ((full_typ & T_KEYWORD) != 0) ?               " keyword" : "",
	   ((full_typ & T_DONT_EVAL_ARGS) != 0) ?        " dont-eval-args" : "",
	   ((full_typ & T_SYNTACTIC) != 0) ?             " syntactic" : "",
	   ((full_typ & T_OVERLAY) != 0) ?               " overlay" : "",
	   ((full_typ & T_CHECKED) != 0) ?               " checked" : "",
	   ((full_typ & T_UNSAFE) != 0) ?                ((is_symbol(obj)) ? " clean" : " unsafe") : "",
	   ((full_typ & T_OPTIMIZED) != 0) ?             " optimized" : "",
	   ((full_typ & T_SAFE_CLOSURE) != 0) ?          " safe-closure" : "",
	   ((full_typ & T_SAFE_PROCEDURE) != 0)  ?       " safe-procedure" : "",
	   ((full_typ & T_SETTER) != 0) ?                " setter" : "",
	   ((full_typ & T_COPY_ARGS) != 0) ?             " copy-args" : "",
	   ((full_typ & T_COLLECTED) != 0) ?             " collected" : "",
	   ((full_typ & T_SHARED) != 0) ?                " shared" : "",
	   ((full_typ & T_HAS_METHODS) != 0) ?           " has-methods" : "",
	   ((full_typ & T_GLOBAL) != 0) ?                ((is_pair(obj)) ? " unsafe-do" : " global") : "",
	   ((full_typ & T_SAFE_STEPPER) != 0) ?          ((is_let(obj)) ? " let-set!-fallback" : ((is_slot(obj)) ? " safe-stepper" : " print-name")) : "",
	   ((full_typ & T_LINE_NUMBER) != 0) ? 
	        ((is_pair(obj)) ? " line number" : ((is_input_port(obj)) ? " loader-port" : ((is_let(obj)) ? " with-let" : " has accessor"))) : "",
	   ((full_typ & T_MUTABLE) != 0) ? 
               ((is_string(obj)) ? " byte-vector" : ((is_let(obj)) ? " let-ref-fallback" : 
		   ((is_iterator(obj)) ? " mark-seq" : ((is_slot(obj)) ? " stepper" : " mutable")))) : "",
	   ((full_typ & T_GENSYM) != 0) ?             
               ((is_let(obj)) ? " function-env" : ((is_unspecified(obj)) ? " no-value" : ((is_pair(obj)) ? " list-in-use" :
	       ((is_closure_star(obj)) ? " simple-args" : ((is_string(obj)) ? " documented" : " gensym"))))) : "");
  return(buf);
}

#if DEBUGGING
static const char *check_name(int typ)
{
  if ((typ >= 0) && (typ < NUM_TYPES))
    {
      s7_pointer p;
      p = prepackaged_type_names[typ];
      if (is_string(p)) return(string_value(p));
      
      switch (typ)
	{
	case T_C_OBJECT:    return("a c-object");
	case T_INPUT_PORT:  return("an input port");
	case T_OUTPUT_PORT: return("an output port");
	}
    }
  return("unknown type!");
}

static s7_pointer check_seti(s7_scheme *sc, s7_pointer x, const char *func, int line)
{
  if (is_immutable(x)) 
    {
      fprintf(stderr, "%s%s[%d]: set! immutable %s: %s%s\n", BOLD_TEXT, func, line, type_name(sc, x, NO_ARTICLE), DISPLAY(x), UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(x);
}

static s7_pointer check_ref(s7_pointer p, int expected_type, const char *func, int line, const char *func1, const char *func2)
{
  int typ;
  typ = unchecked_type(p);
  if (typ != expected_type)
    {
      if ((!func1) || (typ != T_FREE))
	{
	  fprintf(stderr, "%s%s[%d]: not %s, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(expected_type), check_name(typ), typ, UNBOLD_TEXT);
	  if (stop_at_error) abort();
	}
      else
	{
	  if ((strcmp(func, func1) != 0) &&
	      ((!func2) || (strcmp(func, func2) != 0)))
	    {
	      fprintf(stderr, "%s%s[%d]: free cell, not %s%s\n", BOLD_TEXT, func, line, check_name(expected_type), UNBOLD_TEXT);
	      if (stop_at_error) abort();
	    }
	}
    }
  return(p);
}

static s7_pointer check_ref2(s7_pointer p, int expected_type, int other_type, const char *func, int line, const char *func1, const char *func2)
{
  int typ;
  typ = unchecked_type(p);
  if ((typ != expected_type) && (typ != other_type))
    return(check_ref(p, expected_type, func, line, func1, func2));
  return(p);
}

static s7_pointer check_ref3(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if ((typ != T_INPUT_PORT) && (typ != T_OUTPUT_PORT) && (typ != T_FREE))
    {
      fprintf(stderr, "%s%s[%d]: not a port, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_ref4(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if ((typ != T_VECTOR) && (typ != T_FLOAT_VECTOR) && (typ != T_INT_VECTOR) && (typ != T_FREE))
    {
      fprintf(stderr, "%s%s[%d]: not a vector, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_ref5(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if (!t_has_closure_let[typ])
    {
      fprintf(stderr, "%s%s[%d]: not a closure, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_ref6(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if ((typ < T_C_FUNCTION_STAR) && (typ != T_C_MACRO))
    {
      fprintf(stderr, "%s%s[%d]: not a c function, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_ref7(s7_pointer p, const char *func, int line)
{
  if ((!func) || (strcmp(func, "decribe_type_bits") != 0))
    {
      int typ;
      typ = unchecked_type(p);
      if ((typ < T_INTEGER) || (typ > T_COMPLEX))
	{
	  fprintf(stderr, "%s%s[%d]: not a number, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
	  if (stop_at_error) abort();
	}
    }
  return(p);
}

static s7_pointer check_ref8(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if ((!t_sequence_p[typ]) && (!t_structure_p[typ]) && (!is_any_closure(p))) /* closure calling itself an iterator?? */
    {
      fprintf(stderr, "%s%s[%d]: not a sequence or structure, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_ref9(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if ((typ != T_LET) && (typ != T_C_OBJECT) && (!is_any_closure(p)) && (!is_any_macro(p)))
    {
      fprintf(stderr, "%s%s[%d]: not a possible method holder, but %s (%d)%s\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_ref10(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if ((typ != T_PAIR) && (typ != T_NIL) && (typ != T_SYMBOL))
    {
      fprintf(stderr, "%s%s[%d]: arglist is %s (%d)%s?\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_ref11(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if ((typ < T_CLOSURE) && (typ != T_BOOLEAN)) /* actually #t is an error here */
    {
      fprintf(stderr, "%s%s[%d]: setter is %s (%d)%s?\n", BOLD_TEXT, func, line, check_name(typ), typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer check_nref(s7_pointer p, const char *func, int line)
{
  int typ;
  typ = unchecked_type(p);
  if (typ == T_FREE)
    {
      fprintf(stderr, "%s%s[%d]: attempt to use cleared type%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if ((typ < 0) || (typ >= NUM_TYPES))
    {
      fprintf(stderr, "%s%s[%d]: attempt to use messed up cell (type: %d)%s\n", BOLD_TEXT, func, line, typ, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static void print_gc_info(s7_pointer obj, int line)
{
  fprintf(stderr, "%s%p is free (line %d), current: %s[%d], previous: %s[%d],  gc call: %s[%d], clear: %d, alloc: %s[%d]%s\n",
	  BOLD_TEXT, 
	  obj, line,
	  obj->current_alloc_func, obj->current_alloc_line,
	  obj->previous_alloc_func, obj->previous_alloc_line,
	  obj->gc_func, obj->gc_line, obj->clear_line, obj->alloc_func, obj->alloc_line,
	  UNBOLD_TEXT);
  abort();
}

static void show_opt1_bits(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  fprintf(stderr, "%sopt1 %s[%d]: %p->%p %x%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s\n", BOLD_TEXT, func, line, p, p->object.cons.opt1, p->debugger_bits,
	  ((p->debugger_bits & E_SET) != 0) ? " e-set" : "",
	  ((p->debugger_bits & E_FAST) != 0) ? " fast" : "",
	  ((p->debugger_bits & E_CFUNC) != 0) ? " cfunc" : "",
	  ((p->debugger_bits & E_CLAUSE) != 0) ? " clause" : "",
	  ((p->debugger_bits & E_BACK) != 0) ? " back" : "",
	  ((p->debugger_bits & E_LAMBDA) != 0) ? " lambda" : "",
	  ((p->debugger_bits & E_SYM) != 0) ? " sym" : "",
	  ((p->debugger_bits & E_PAIR) != 0) ? " pair" : "",
	  ((p->debugger_bits & E_CON) != 0) ? " con" : "",
	  ((p->debugger_bits & E_GOTO) != 0) ? " goto" : "",
	  ((p->debugger_bits & E_VECTOR) != 0) ? " vector" : "",
	  ((p->debugger_bits & E_ANY) != 0) ? " any" : "",
	  ((p->debugger_bits & E_SLOT) != 0) ? " slot" : "",
	  ((p->debugger_bits & S_HASH) != 0) ? " raw-hash" : "",	
  UNBOLD_TEXT);
}

static s7_pointer opt1_1(s7_scheme *sc, s7_pointer p, unsigned int role, const char *func, int line)
{
  if ((!opt1_is_set(p)) ||
      ((!opt1_role_matches(p, role)) &&
       (role != E_ANY)))
    {
      show_opt1_bits(sc, p, func, line);
      if (stop_at_error) abort();
    }
  return(p->object.cons.opt1);
}

static s7_pointer set_opt1_1(s7_scheme *sc, s7_pointer p, s7_pointer x, unsigned int role, const char *func, int line)
{
  p->object.cons.opt1 = x;
  set_opt1_role(p, role);
  set_opt1_is_set(p);
  return(x);
}

static unsigned long long int s_hash_1(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  if ((!opt1_is_set(p)) ||
      (!opt1_role_matches(p, S_HASH)))
    {
      show_opt1_bits(sc, p, func, line);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.hash);
}

static void set_s_hash_1(s7_scheme *sc, s7_pointer p, unsigned long long int x, const char *func, int line)
{
  p->object.sym_cons.hash = x;
  set_opt1_role(p, S_HASH);
  set_opt1_is_set(p);
}

static void show_opt2_bits(s7_scheme *sc, s7_pointer p, const char *func, int line, unsigned int role)
{
  fprintf(stderr, "%s%s[%d]: opt2: %p->%p is %x%s%s%s%s%s%s%s%s%s but expects %x%s%s%s%s%s%s%s%s%s%s\n", 
	  BOLD_TEXT, func, line, p, p->object.cons.opt2, 

	  p->debugger_bits,
	  ((p->debugger_bits & F_SET) != 0) ? " f-set" : "",
	  ((p->debugger_bits & F_KEY) != 0) ? " key" : "",
	  ((p->debugger_bits & F_SLOW) != 0) ? " slow" : "",
	  ((p->debugger_bits & F_SYM) != 0) ? " sym" : "",
	  ((p->debugger_bits & F_PAIR) != 0) ? " pair" : "",
	  ((p->debugger_bits & F_CON) != 0) ? " con" : "",
	  ((p->debugger_bits & F_CALL) != 0) ? " call" : "",
	  ((p->debugger_bits & F_LAMBDA) != 0) ? " lambda" : "",
	  ((p->debugger_bits & S_NAME) != 0) ? " raw-name" : "",

	  role,
	  ((role & F_SET) != 0) ? " f-set" : "",
	  ((role & F_KEY) != 0) ? " key" : "",
	  ((role & F_SLOW) != 0) ? " slow" : "",
	  ((role & F_SYM) != 0) ? " sym" : "",
	  ((role & F_PAIR) != 0) ? " pair" : "",
	  ((role & F_CON) != 0) ? " con" : "",
	  ((role & F_CALL) != 0) ? " call" : "",
	  ((role & F_LAMBDA) != 0) ? " lambda" : "",
	  ((role & S_NAME) != 0) ? " raw-name" : "",

	  UNBOLD_TEXT);
}

static s7_pointer opt2_1(s7_scheme *sc, s7_pointer p, unsigned int role, const char *func, int line)
{
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, role)))
    {
      show_opt2_bits(sc, p, func, line, role);
      fprintf(stderr, "p: %s\n", DISPLAY(p));
      if (stop_at_error) abort();
    }
  return(p->object.cons.opt2);
}

static void set_opt2_1(s7_scheme *sc, s7_pointer p, s7_pointer x, unsigned int role, const char *func, int line)
{
  p->object.cons.opt2 = x;
  set_opt2_role(p, role);
  set_opt2_is_set(p);
}

static const char *s_name_1(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, S_NAME)))
    {
      show_opt2_bits(sc, p, func, line, (unsigned int)S_NAME);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.fstr);
}

static void set_s_name_1(s7_scheme *sc, s7_pointer p, const char *str, const char *func, int line)
{
  p->object.sym_cons.fstr = str;
  set_opt2_role(p, S_NAME);
  set_opt2_is_set(p);
}

static void show_opt3_bits(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  fprintf(stderr, "%s%s[%d]: opt3: %x%s%s%s%s%s%s%s%s%s\n", BOLD_TEXT, func, line, 
	  p->debugger_bits, 
	  ((p->debugger_bits & G_SET) != 0) ? " g-set" : "",
	  ((p->debugger_bits & G_ARGLEN) != 0) ? " arglen" : "",
	  ((p->debugger_bits & G_SYM) != 0) ? " sym" : "",
	  ((p->debugger_bits & G_AND) != 0) ? " and" : "",
	  ((p->debugger_bits & S_LINE) != 0) ? " line" : "",
	  ((p->debugger_bits & S_LEN) != 0) ? " len" : "",
	  ((p->debugger_bits & S_OP) != 0) ? " op" : "",
	  ((p->debugger_bits & S_SYNOP) != 0) ? " syn-op" : "",
	  UNBOLD_TEXT);
}

static s7_pointer opt3_1(s7_scheme *sc, s7_pointer p, unsigned int role, const char *func, int line)
{
  if ((!opt3_is_set(p)) || 
      (!opt3_role_matches(p, role)))
    {
      show_opt3_bits(sc, p, func, line);
      if (stop_at_error) abort();
    }
  return(p->object.cons.opt3);
}

static void set_opt3_1(s7_scheme *sc, s7_pointer p, s7_pointer x, unsigned int role, const char *func, int line)
{
  typeflag(p) &= ~(T_OPTIMIZED | T_LINE_NUMBER);
  p->object.cons.opt3 = x;
  set_opt3_is_set(p);
  set_opt3_role(p, role);
}

/* S_LINE */
static unsigned int s_line_1(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  if ((!opt3_is_set(p)) || 
      ((p->debugger_bits & S_LINE) == 0) ||
      (!has_line_number(p)))
    {
      show_opt3_bits(sc, p, func, line);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.line);
}

static void set_s_line_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line)
{
  p->object.sym_cons.line = x;
  (p)->debugger_bits = (S_LINE | (p->debugger_bits & ~S_LEN)); /* turn on line, cancel len */
  set_opt3_is_set(p);
}

/* S_LEN (collides with S_LINE) */
static unsigned int s_len_1(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  if ((!opt3_is_set(p)) || 
      ((p->debugger_bits & S_LEN) == 0) ||
      (has_line_number(p)))
    {
      show_opt3_bits(sc, p, func, line);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.line);
}

static void set_s_len_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line)
{
  typeflag(p) &= ~(T_LINE_NUMBER);
  p->object.sym_cons.line = x;
  (p)->debugger_bits = (S_LEN | (p->debugger_bits & ~(S_LINE)));
  set_opt3_is_set(p);
}

/* S_OP */
static unsigned int s_op_1(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  if ((!opt3_is_set(p)) || 
      ((p->debugger_bits & S_OP) == 0))
    {
      show_opt3_bits(sc, p, func, line);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.op);
}

static void set_s_op_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line)
{
  p->object.sym_cons.op = x;
  (p)->debugger_bits = (S_OP | (p->debugger_bits & ~(S_SYNOP)));
  set_opt3_is_set(p);
}

/* S_SYNOP (collides with S_OP) */
static unsigned int s_syn_op_1(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  if ((!opt3_is_set(p)) || 
      ((p->debugger_bits & S_SYNOP) == 0))
    {
      show_opt3_bits(sc, p, func, line);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.op);
}

static void set_s_syn_op_1(s7_scheme *sc, s7_pointer p, unsigned int x, const char *func, int line)
{
  p->object.sym_cons.op = x;
  (p)->debugger_bits = (S_SYNOP | (p->debugger_bits & ~(S_OP)));
  set_opt3_is_set(p);
}

static void print_debugging_state(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  /* show current state, current allocated state, and previous allocated state.
   */
  char *current_bits, *allocated_bits, *previous_bits, *str;
  int save_typeflag, len, nlen;
  const char *excl_name;

  if (is_free(obj))
    excl_name = "free cell!";
  else excl_name = "unknown object!";

  current_bits = describe_type_bits(sc, obj);
  save_typeflag = typeflag(obj);
  typeflag(obj) = obj->current_alloc_type;
  allocated_bits = describe_type_bits(sc, obj);
  typeflag(obj) = obj->previous_alloc_type;
  previous_bits = describe_type_bits(sc, obj);
  typeflag(obj) = save_typeflag;

  len = safe_strlen(excl_name) +
    safe_strlen(current_bits) + safe_strlen(allocated_bits) + safe_strlen(previous_bits) +
    safe_strlen(obj->previous_alloc_func) + safe_strlen(obj->current_alloc_func) + 512;
  tmpbuf_malloc(str, len);

  nlen = snprintf(str, len,
		  "\n<%s %s,\n  current: %s[%d] %s,\n  previous: %s[%d] %s\n  hloc: %d (%d uses), free: %s[%d], clear: %d, alloc: %s[%d]>",
		  excl_name, current_bits,
		  obj->current_alloc_func, obj->current_alloc_line, allocated_bits,
		  obj->previous_alloc_func, obj->previous_alloc_line, previous_bits,
		  heap_location(obj), obj->uses,
		  obj->gc_func, obj->gc_line, obj->clear_line, obj->alloc_func, obj->alloc_line);

  free(current_bits);
  free(allocated_bits);
  free(previous_bits);
  if (is_null(port))
    fprintf(stderr, "%p: %s\n", obj, str);
  else port_write_string(port)(sc, str, nlen, port);
  tmpbuf_free(str, len);
}

static s7_pointer check_null_sym(s7_scheme *sc, s7_pointer p, s7_pointer sym, int line, const char *func)
{
  if (!p)
    {
      fprintf(stderr, "%s%s[%d]: %s unbound%s\n", BOLD_TEXT, func, line, symbol_name(sym), UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}
#endif

static void iterator_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  if (use_write == USE_READABLE_WRITE)
    {
      if (iterator_is_at_end(obj))
	port_write_string(port)(sc, "(make-iterator #())", 19, port);
      else
	{
	  s7_pointer seq;
	  seq = iterator_sequence(obj);
	  if ((is_string(seq)) && (!is_byte_vector(seq)))
	    {
	      port_write_string(port)(sc, "(make-iterator \"", 16, port);
	      port_write_string(port)(sc, (char *)(string_value(seq) + iterator_position(obj)), string_length(seq) - iterator_position(obj), port);
	      port_write_string(port)(sc, "\")", 2, port);
	    }
	  else
	    {
	      if (iterator_position(obj) > 0)
		port_write_string(port)(sc, "(let ((iter (make-iterator ", 27, port);
	      else port_write_string(port)(sc, "(make-iterator ", 15, port);
	      object_to_port_with_circle_check(sc, iterator_sequence(obj), port, use_write, ci);
	      if (iterator_position(obj) > 0)
		{
		  int nlen;
		  char *str;
		  str = (char *)malloc(128 * sizeof(char));
		  nlen = snprintf(str, 128, "))) (do ((i 0 (+ i 1))) ((= i %lld) iter) (iterate iter)))", iterator_position(obj));
		  port_write_string(port)(sc, str, nlen, port);
		  free(str);
		}
	      else port_write_character(port)(sc, ')', port);
	    }
	}
    }
  else
    {
      const char *str;
      str = type_name(sc, iterator_sequence(obj), NO_ARTICLE);
      port_write_string(port)(sc, "#<iterator: ", 12, port);
      port_write_string(port)(sc, str, safe_strlen(str), port);
      port_write_character(port)(sc, '>', port);
    }
}

static void baffle_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  int nlen;
  char buf[64];
  nlen = snprintf(buf, 64, "#<baffle: %d>", baffle_key(obj));
  port_write_string(port)(sc, buf, nlen, port);
}

static void c_pointer_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write)
{
  int nlen;
  char buf[64];

  if (use_write == USE_READABLE_WRITE)
    nlen = snprintf(buf, 64, "(c-pointer " INT_FORMAT ")", (ptr_int)raw_pointer(obj));
  else nlen = snprintf(buf, 64, "#<c_pointer %p>", raw_pointer(obj));
  port_write_string(port)(sc, buf, nlen, port);
}

static void rng_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write)
{
  int nlen;
  char buf[128];
#if WITH_GMP
  if (use_write == USE_READABLE_WRITE)
    nlen = snprintf(buf, 128, "#<unprint-readable object>");
  else nlen = snprintf(buf, 128, "#<rng %p>", obj);
#else
  if (use_write == USE_READABLE_WRITE)
    nlen = snprintf(buf, 128, "(random-state %llu %llu)", random_seed(obj), random_carry(obj));
  else nlen = snprintf(buf, 128, "#<rng %llu %llu>", random_seed(obj), random_carry(obj));
#endif
  port_write_string(port)(sc, buf, nlen, port);
}

static void object_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  int nlen;
  char *str;
  switch (type(obj))
    {
    case T_FLOAT_VECTOR:
    case T_INT_VECTOR:
      int_or_float_vector_to_port(sc, obj, port, use_write);
      break;

    case T_VECTOR:
      vector_to_port(sc, obj, port, use_write, ci);
      break;

    case T_PAIR:
      list_to_port(sc, obj, port, use_write, ci);
      break;

    case T_HASH_TABLE:
      hash_table_to_port(sc, obj, port, use_write, ci);
      break;

    case T_ITERATOR:
      iterator_to_port(sc, obj, port, use_write, ci);
      break;

    case T_LET:
      let_to_port(sc, obj, port, use_write, ci);
      break;

    case T_UNIQUE:
      /* if file has #<eof> it causes read to return #<eof> -> end of read! what is readable version? '#<eof> or (begin #<eof>) as below
       * but this is silly -- to fool read, the #<eof> has to be all by itself at the top-level!
       * and the read of #<eof> does not affect the port, so if you know it's there, just ignore #<eof> and continue reading.
       */
      if ((use_write == USE_READABLE_WRITE) &&
	  (obj == sc->eof_object))
	port_write_string(port)(sc, "(begin #<eof>)", 14, port);
      else port_write_string(port)(sc, unique_name(obj), unique_name_length(obj), port);
      break;

    case T_BOOLEAN:
    case T_NIL:
    case T_UNSPECIFIED:
      port_write_string(port)(sc, unique_name(obj), unique_name_length(obj), port);
      break;

    case T_INPUT_PORT:
      input_port_to_port(sc, obj, port, use_write);
      break;

    case T_OUTPUT_PORT:
      output_port_to_port(sc, obj, port, use_write);
      break;

    case T_COUNTER:
      port_write_string(port)(sc, "#<counter>", 10, port);
      break;

    case T_BAFFLE:
      baffle_to_port(sc, obj, port);
      break;

    case T_INTEGER:
      if (has_print_name(obj))
	port_write_string(port)(sc, print_name(obj), print_name_length(obj), port);
      else
	{
	  nlen = 0;
	  str = integer_to_string_base_10_no_width(obj, &nlen);
	  if (nlen > 0)
	    {
	      set_print_name(obj, str, nlen);
	      port_write_string(port)(sc, str, nlen, port);
	    }
	  else port_display(port)(sc, str, port);
	}
      break;

    case T_REAL:
    case T_RATIO:
    case T_COMPLEX:
      if (has_print_name(obj))
	port_write_string(port)(sc, print_name(obj), print_name_length(obj), port);
      else
	{
	  nlen = 0;
	  str = number_to_string_base_10(obj, 0, float_format_precision, 'g', &nlen, use_write); /* was 14 */
	  set_print_name(obj, str, nlen);
	  port_write_string(port)(sc, str, nlen, port);
	}
      break;

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
    case T_BIG_REAL:
    case T_BIG_COMPLEX:
      nlen = 0;
      str = big_number_to_string_with_radix(obj, BASE_10, 0, &nlen, use_write);
      port_write_string(port)(sc, str, nlen, port);
      free(str);
      break;
#endif

    case T_SYMBOL:
      symbol_to_port(sc, obj, port, use_write);
      break;

    case T_SYNTAX:
      port_display(port)(sc, symbol_name(syntax_symbol(obj)), port);
      break;

    case T_STRING:
      if (is_byte_vector(obj))
	byte_vector_to_port(sc, obj, port, use_write);
      else string_to_port(sc, obj, port, use_write);
      break;

    case T_CHARACTER:
      if (use_write == USE_DISPLAY)
	port_write_character(port)(sc, character(obj), port);
      else port_write_string(port)(sc, character_name(obj), character_name_length(obj), port);
      break;

    case T_CLOSURE:
    case T_CLOSURE_STAR:
      if (has_methods(obj))
	{
	  /* look for object->string method else fallback on ordinary case.
	   * can't use recursion on closure_let here because then the fallback name is #<let>.
	   */
	  s7_pointer print_func;
	  print_func = find_method(sc, closure_let(obj), sc->object_to_string_symbol);
	  if (print_func != sc->undefined)
	    {
	      s7_pointer p;
	      p = s7_apply_function(sc, print_func, list_1(sc, obj));
	      if (string_length(p) > 0)
		port_write_string(port)(sc, string_value(p), string_length(p), port);
	      break;
	    }
	}
      if (use_write == USE_READABLE_WRITE)
	write_closure_readably(sc, obj, port);
      else write_closure_name(sc, obj, port);
      break;

    case T_MACRO:
    case T_MACRO_STAR:
    case T_BACRO:
    case T_BACRO_STAR:
      if (use_write == USE_READABLE_WRITE)
	write_macro_readably(sc, obj, port);
      else write_closure_name(sc, obj, port);
      break;

    case T_C_OPT_ARGS_FUNCTION:
    case T_C_RST_ARGS_FUNCTION:
    case T_C_ANY_ARGS_FUNCTION:
    case T_C_FUNCTION:
    case T_C_FUNCTION_STAR:
      port_write_string(port)(sc, c_function_name(obj), c_function_name_length(obj), port);
      break;

    case T_C_MACRO:
      port_write_string(port)(sc, c_macro_name(obj), c_macro_name_length(obj), port);
      break;

    case T_C_POINTER:
      c_pointer_to_port(sc, obj, port, use_write);
      break;

    case T_RANDOM_STATE:
      rng_to_port(sc, obj, port, use_write);
      break;

    case T_CONTINUATION:
      if (use_write == USE_READABLE_WRITE)
	port_write_string(port)(sc, "continuation", 12, port);
      else port_write_string(port)(sc, "#<continuation>", 15, port);
      break;

    case T_GOTO:
      if (use_write == USE_READABLE_WRITE)
	port_write_string(port)(sc, "goto", 4, port);
      else port_write_string(port)(sc, "#<goto>", 7, port);
      break;

    case T_CATCH:
      port_write_string(port)(sc, "#<catch>", 8, port);
      break;

    case T_DYNAMIC_WIND:
      /* this can happen because (*s7* 'stack) can involve dynamic-wind markers */
      port_write_string(port)(sc, "#<dynamic-wind>", 15, port);
      break;

    case T_C_OBJECT:
      if (use_write == USE_READABLE_WRITE)
	str = ((*(c_object_print_readably(obj)))(sc, c_object_value(obj)));
      else str = ((*(c_object_print(obj)))(sc, c_object_value(obj)));
      port_display(port)(sc, str, port);
      free(str);
      break;

    case T_SLOT:
      if (use_write != USE_READABLE_WRITE)
	port_write_character(port)(sc, '\'', port);
      symbol_to_port(sc, slot_symbol(obj), port, use_write);
      port_write_character(port)(sc, ' ', port);
      object_to_port_with_circle_check(sc, slot_value(obj), port, use_write, ci);
      break;

    default:
#if DEBUGGING
      print_debugging_state(sc, obj, port);
#else
      {
	char *str, *tmp;
	int len;
	tmp = describe_type_bits(sc, obj);
	len = 32 + safe_strlen(tmp);
	tmpbuf_malloc(str, len);
	if (is_free(obj))
	  nlen = snprintf(str, len, "<free cell! %s>", tmp);
	else nlen = snprintf(str, len, "<unknown object! %s>", tmp);
	free(tmp);
	port_write_string(port)(sc, str, nlen, port);
	tmpbuf_free(str, len);
      }
#endif
      break;
    }
}


static void object_to_port_with_circle_check(s7_scheme *sc, s7_pointer vr, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  if ((ci) &&
      (has_structure(vr)))
    {
      int ref;
      ref = shared_ref(ci, vr);
      if (ref != 0)
	{
	  char buf[32];
	  int nlen;
	  char *p;
	  unsigned int len;
	  if (ref > 0)
	    {
	      if (use_write == USE_READABLE_WRITE)
		{
		  nlen = snprintf(buf, 32, "(set! {%d} ", ref);
		  port_write_string(port)(sc, buf, nlen, port);
		  object_to_port(sc, vr, port, USE_READABLE_WRITE, ci);
		  port_write_character(port)(sc, ')', port);
		}
	      else
		{
		  p = pos_int_to_str((s7_int)ref, &len, '=');
		  *--p = '#';
		  port_write_string(port)(sc, p, len, port);
		  object_to_port(sc, vr, port, DONT_USE_DISPLAY(use_write), ci);
		}
	    }
	  else
	    {
	      if (use_write == USE_READABLE_WRITE)
		{
		  nlen = snprintf(buf, 32, "{%d}", -ref);
		  port_write_string(port)(sc, buf, nlen, port);
		}
	      else 
		{
		  p = pos_int_to_str((s7_int)(-ref), &len, '#');
		  *--p = '#';
		  port_write_string(port)(sc, p, len, port);
		}
	    }
	  return;
	}
    }
  object_to_port(sc, vr, port, use_write, ci);
}


static void setup_shared_reads(s7_scheme *sc, s7_pointer port, shared_info *ci)
{
  int i;
  char buf[64];

  port_write_string(port)(sc, "(let (", 6, port);
  for (i = 1; i <= ci->top; i++)
    {
      int len;
      len = snprintf(buf, 64, "({%d} #f)", i);
      port_write_string(port)(sc, buf, len, port);
    }
  port_write_string(port)(sc, ") ", 2, port);
}

static void finish_shared_reads(s7_scheme *sc, s7_pointer port, shared_info *ci)
{
  port_write_character(port)(sc, ')', port);
}

static s7_pointer object_out(s7_scheme *sc, s7_pointer obj, s7_pointer strport, use_write_t choice)
{
  if ((has_structure(obj)) &&
      (obj != sc->rootlet))
    {
      shared_info *ci;
      ci = make_shared_info(sc, obj, choice != USE_READABLE_WRITE);
      if (ci)
	{
	  if (choice == USE_READABLE_WRITE)
	    {
	      setup_shared_reads(sc, strport, ci);
	      object_to_port_with_circle_check(sc, obj, strport, choice, ci);
	      finish_shared_reads(sc, strport, ci);
	    }
	  else object_to_port_with_circle_check(sc, obj, strport, choice, ci);
	  return(obj);
	}
    }
  object_to_port(sc, obj, strport, choice, NULL);
  return(obj);
}
  

static s7_pointer format_ports = NULL;

static s7_pointer open_format_port(s7_scheme *sc)
{
  s7_pointer x;
  int len;

  if (format_ports)
    {
      x = format_ports;
      format_ports = (s7_pointer)(port_port(x)->next);
      port_position(x) = 0;
      port_data(x)[0] = '\0';
      return(x);
    }

  len = FORMAT_PORT_LENGTH;
  x = alloc_pointer();
  set_type(x, T_OUTPUT_PORT);
  port_port(x) = (port_t *)calloc(1, sizeof(port_t));
  port_type(x) = STRING_PORT;
  port_is_closed(x) = false;
  port_data_size(x) = len;
  port_data(x) = (unsigned char *)malloc(len * sizeof(unsigned char)); /* was +8 */
  port_data(x)[0] = '\0';
  port_position(x) = 0;
  port_needs_free(x) = false;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = string_display;
  port_write_character(x) = string_write_char;
  port_write_string(x) = string_write_string;
  return(x);
}

static void close_format_port(s7_scheme *sc, s7_pointer port)
{
  port_port(port)->next = (void *)format_ports;
  format_ports = port;
}


static char *s7_object_to_c_string_1(s7_scheme *sc, s7_pointer obj, use_write_t use_write, int *nlen)
{
  char *str;
  s7_pointer strport;

  strport = open_format_port(sc);
  object_out(sc, obj, strport, use_write);
  if (nlen) (*nlen) = port_position(strport);

  str = (char *)malloc((port_position(strport) + 1) * sizeof(char));
  memcpy((void *)str, (void *)port_data(strport), port_position(strport));
  str[port_position(strport)] = '\0';
  close_format_port(sc, strport);

  return(str); 
}


char *s7_object_to_c_string(s7_scheme *sc, s7_pointer obj)
{
  return(s7_object_to_c_string_1(sc, obj, USE_WRITE, NULL));
}


s7_pointer s7_object_to_string(s7_scheme *sc, s7_pointer obj, bool use_write) /* unavoidable backwards compatibility rigidity here */
{
  char *str;
  int len = 0;

  str = s7_object_to_c_string_1(sc, obj, (use_write) ? 