/*
 * Copyright (C) 2016 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2 or version 3 of the License.
 * See http://www.gnu.org/copyleft/lgpl.html the full text of the license.
 */

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <gio/gunixsocketaddress.h>
#include <libsoup/soup.h>
#include <json-glib/json-glib.h>

#include "snapd-client.h"
#include "snapd-app.h"
#include "snapd-plug.h"
#include "snapd-slot.h"

typedef struct
{
    GSocket *snapd_socket;
    SnapdAuthData *auth_data;
    GList *requests;
    GSource *read_source;
    GByteArray *buffer;
    gsize n_read;
} SnapdClientPrivate;

G_DEFINE_TYPE_WITH_PRIVATE (SnapdClient, snapd_client, G_TYPE_OBJECT)

G_DEFINE_QUARK (snapd-client-error-quark, snapd_client_error)

/* snapd API documentation is at https://github.com/snapcore/snapd/blob/master/docs/rest.md */

/* Default socket to connect to */
#define SNAPD_SOCKET "/run/snapd.socket"

/* Number of milliseconds to poll for status in asynchronous operations */
#define ASYNC_POLL_TIME 100
#define ASYNC_POLL_TIMEOUT 1000

// FIXME: Make multiple async requests work at the same time

typedef enum
{
    SNAPD_REQUEST_GET_SYSTEM_INFORMATION,
    SNAPD_REQUEST_LOGIN,
    SNAPD_REQUEST_GET_ICON,
    SNAPD_REQUEST_LIST,
    SNAPD_REQUEST_LIST_ONE,
    SNAPD_REQUEST_GET_INTERFACES,
    SNAPD_REQUEST_CONNECT_INTERFACE,
    SNAPD_REQUEST_DISCONNECT_INTERFACE,
    SNAPD_REQUEST_FIND,
    SNAPD_REQUEST_SIDELOAD_SNAP, // FIXME
    SNAPD_REQUEST_GET_PAYMENT_METHODS,
    SNAPD_REQUEST_BUY,
    SNAPD_REQUEST_INSTALL,
    SNAPD_REQUEST_REFRESH,
    SNAPD_REQUEST_REMOVE,
    SNAPD_REQUEST_ENABLE,
    SNAPD_REQUEST_DISABLE
} RequestType;

G_DECLARE_FINAL_TYPE (SnapdRequest, snapd_request, SNAPD, REQUEST, GObject)

struct _SnapdRequest
{
    GObject parent_instance;

    SnapdClient *client;
    SnapdAuthData *auth_data;

    RequestType request_type;

    GCancellable *cancellable;
    GAsyncReadyCallback ready_callback;
    gpointer ready_callback_data;

    SnapdProgressCallback progress_callback;
    gpointer progress_callback_data;

    gchar *change_id;
    guint poll_timer;
    guint timeout_timer;
    SnapdTask *main_task;
    GPtrArray *tasks;

    gboolean completed;
    GError *error;
    gboolean result;
    SnapdSystemInformation *system_information;
    GPtrArray *snaps;
    gchar *suggested_currency;
    SnapdSnap *snap;
    SnapdIcon *icon;
    SnapdAuthData *received_auth_data;
    GPtrArray *plugs;
    GPtrArray *slots;
    gboolean allows_automatic_payment;
    GPtrArray *methods;
};

static gboolean
complete_cb (gpointer user_data)
{
    SnapdRequest *request = user_data;
    SnapdClientPrivate *priv = snapd_client_get_instance_private (request->client);
  
    if (request->ready_callback != NULL)
        request->ready_callback (G_OBJECT (request->client), G_ASYNC_RESULT (request), request->ready_callback_data);

    priv->requests = g_list_remove (priv->requests, request);
    g_object_unref (request);

    return G_SOURCE_REMOVE;
}

static void
snapd_request_complete (SnapdRequest *request, GError *error)
{
    g_return_if_fail (!request->completed);

    request->completed = TRUE;
    if (error != NULL)
        g_propagate_error (&request->error, error);

    /* Do in main loop so user can't block our reading callback */
    g_idle_add (complete_cb, request);
}

static void
snapd_request_wait (SnapdRequest *request)
{
    while (!request->completed)
       g_main_context_iteration (g_main_context_default (), TRUE);
}

static gboolean
snapd_request_set_error (SnapdRequest *request, GError **error)
{
    if (g_cancellable_set_error_if_cancelled (request->cancellable, error))
        return TRUE;

    if (request->error != NULL) {
        g_propagate_error (error, request->error);
        return TRUE;
    }

    return FALSE;
}

static GObject *
snapd_request_get_source_object (GAsyncResult *result)
{
    return G_OBJECT (SNAPD_REQUEST (result)->client);
}

static void
snapd_request_async_result_init (GAsyncResultIface *iface)
{
    iface->get_source_object = snapd_request_get_source_object;
}

G_DEFINE_TYPE_WITH_CODE (SnapdRequest, snapd_request, G_TYPE_OBJECT,
                         G_IMPLEMENT_INTERFACE (G_TYPE_ASYNC_RESULT, snapd_request_async_result_init))

static void
snapd_request_finalize (GObject *object)
{
    SnapdRequest *request = SNAPD_REQUEST (object);

    g_clear_object (&request->auth_data);
    g_clear_object (&request->cancellable);
    g_free (request->change_id);
    if (request->poll_timer != 0)
        g_source_remove (request->poll_timer);
    if (request->timeout_timer != 0)
        g_source_remove (request->timeout_timer);
    g_clear_object (&request->main_task);
    g_clear_pointer (&request->tasks, g_ptr_array_unref);
    g_clear_object (&request->system_information);
    g_clear_pointer (&request->snaps, g_ptr_array_unref);
    g_free (request->suggested_currency);
    g_clear_object (&request->snap);
    g_clear_object (&request->icon);
    g_clear_object (&request->received_auth_data);
    g_clear_pointer (&request->plugs, g_ptr_array_unref);
    g_clear_pointer (&request->slots, g_ptr_array_unref);
    g_clear_pointer (&request->methods, g_ptr_array_unref);  
}

static void
snapd_request_class_init (SnapdRequestClass *klass)
{
   GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

   gobject_class->finalize = snapd_request_finalize;
}

static void
snapd_request_init (SnapdRequest *request)
{
}

static gchar *
builder_to_string (JsonBuilder *builder)
{
    g_autoptr(JsonNode) json_root = NULL;
    g_autoptr(JsonGenerator) json_generator = NULL;

    json_root = json_builder_get_root (builder);
    json_generator = json_generator_new ();
    json_generator_set_pretty (json_generator, TRUE);
    json_generator_set_root (json_generator, json_root);
    return json_generator_to_data (json_generator, NULL);
}

static void
send_request (SnapdRequest *request, gboolean authorize, const gchar *method, const gchar *path, const gchar *content_type, const gchar *content)
{
    SnapdClientPrivate *priv = snapd_client_get_instance_private (request->client);
    g_autoptr(SoupMessageHeaders) headers = NULL;
    g_autoptr(GString) request_data = NULL;
    SoupMessageHeadersIter iter;
    const char *name, *value;
    gssize n_written;
    g_autoptr(GError) local_error = NULL;

    // NOTE: Would love to use libsoup but it doesn't support unix sockets
    // https://bugzilla.gnome.org/show_bug.cgi?id=727563

    headers = soup_message_headers_new (SOUP_MESSAGE_HEADERS_REQUEST);
    soup_message_headers_append (headers, "Host", "");
    soup_message_headers_append (headers, "Connection", "keep-alive");
    if (content_type)
        soup_message_headers_set_content_type (headers, content_type, NULL);
    if (content)
        soup_message_headers_set_content_length (headers, strlen (content));
    if (authorize && request->auth_data != NULL) {
        g_autoptr(GString) authorization;
        gchar **discharges;
        gsize i;

        authorization = g_string_new ("");
        g_string_append_printf (authorization, "Macaroon root=\"%s\"", snapd_auth_data_get_macaroon (request->auth_data));
        discharges = snapd_auth_data_get_discharges (request->auth_data);
        for (i = 0; discharges[i] != NULL; i++)
            g_string_append_printf (authorization, ",discharge=\"%s\"", discharges[i]);
        soup_message_headers_append (headers, "Authorization", authorization->str);
    }

    request_data = g_string_new ("");
    g_string_append_printf (request_data, "%s %s HTTP/1.1\r\n", method, path);
    soup_message_headers_iter_init (&iter, headers);
    while (soup_message_headers_iter_next (&iter, &name, &value))
        g_string_append_printf (request_data, "%s: %s\r\n", name, value);
    g_string_append (request_data, "\r\n");
    if (content)
        g_string_append (request_data, content);

    /* send HTTP request */
    // FIXME: Check for short writes
    n_written = g_socket_send (priv->snapd_socket, request_data->str, request_data->len, request->cancellable, &local_error);
    if (n_written < 0) {
        GError *error = g_error_new (SNAPD_CLIENT_ERROR,
                                     SNAPD_CLIENT_ERROR_WRITE_ERROR,
                                     "Failed to write to snapd: %s",
                                     local_error->message);
        snapd_request_complete (request, error);
    }
}

static gboolean
get_bool (JsonObject *object, const gchar *name, gboolean default_value)
{
    JsonNode *node = json_object_get_member (object, name);
    if (node != NULL && json_node_get_value_type (node) == G_TYPE_BOOLEAN)
        return json_node_get_boolean (node);
    else
        return default_value;
}

static gint64
get_int (JsonObject *object, const gchar *name, gint64 default_value)
{
    JsonNode *node = json_object_get_member (object, name);
    if (node != NULL && json_node_get_value_type (node) == G_TYPE_INT64)
        return json_node_get_int (node);
    else
        return default_value;
}

static const gchar *
get_string (JsonObject *object, const gchar *name, const gchar *default_value)
{
    JsonNode *node = json_object_get_member (object, name);
    if (node != NULL && json_node_get_value_type (node) == G_TYPE_STRING)
        return json_node_get_string (node);
    else
        return default_value;
}

static JsonArray *
get_array (JsonObject *object, const gchar *name)
{
    JsonNode *node = json_object_get_member (object, name);
    if (node != NULL && json_node_get_value_type (node) == JSON_TYPE_ARRAY)
        return json_array_ref (json_node_get_array (node));
    else
        return json_array_new ();
}

static JsonObject *
get_object (JsonObject *object, const gchar *name)
{
    JsonNode *node = json_object_get_member (object, name);
    if (node != NULL && json_node_get_value_type (node) == JSON_TYPE_OBJECT)
        return json_node_get_object (node);
    else
        return NULL;
}

static gboolean
parse_date (const gchar *date_string, gint *year, gint *month, gint *day)
{
    /* Example: 2016-05-17 */
    if (strchr (date_string, '-') != NULL) {
        g_auto(GStrv) tokens = NULL;

        tokens = g_strsplit (date_string, "-", -1);
        if (g_strv_length (tokens) != 3)
            return FALSE;

        *year = atoi (tokens[0]);
        *month = atoi (tokens[1]);
        *day = atoi (tokens[2]);

        return TRUE;
    }
    /* Example: 20160517 */  
    else if (strlen (date_string) == 8) {
        // FIXME: Implement
        return FALSE;
    }
    else
        return FALSE;
}

static gboolean
parse_time (const gchar *time_string, gint *hour, gint *minute, gdouble *seconds)
{
    /* Example: 09:36:53.682 or 09:36:53 or 09:36 */
    if (strchr (time_string, ':') != NULL) {
        g_auto(GStrv) tokens = NULL;

        tokens = g_strsplit (time_string, ":", 3);
        *hour = atoi (tokens[0]);
        if (tokens[1] == NULL)
            return FALSE;
        *minute = atoi (tokens[1]);
        if (tokens[2] != NULL)
            *seconds = g_ascii_strtod (tokens[2], NULL);
        else
            *seconds = 0.0;

        return TRUE;
    }
    /* Example: 093653.682 or 093653 or 0936 */
    else {
        // FIXME: Implement
        return FALSE;
    }
}

static gboolean
is_timezone_prefix (gchar c)
{
    return c == '+' || c == '-' || c == 'Z';
}

static GDateTime *
get_date_time (JsonObject *object, const gchar *name)
{
    const gchar *value;
    g_auto(GStrv) tokens = NULL;
    g_autoptr(GTimeZone) timezone = NULL;
    gint year = 0, month = 0, day = 0, hour = 0, minute = 0;
    gdouble seconds = 0.0;

    value = get_string (object, name, NULL);
    if (value == NULL)
        return NULL;

    /* Example: 2016-05-17T09:36:53+12:00 */
    tokens = g_strsplit (value, "T", 2);
    if (!parse_date (tokens[0], &year, &month, &day))
        return NULL;
    if (tokens[1] != NULL) {
        gchar *timezone_start;

        /* Timezone is either Z (UTC) +hh:mm or -hh:mm */
        timezone_start = tokens[1];
        while (*timezone_start != '\0' && !is_timezone_prefix (*timezone_start))
            timezone_start++;
        if (*timezone_start != '\0')
            timezone = g_time_zone_new (timezone_start);

        /* Strip off timezone */
        *timezone_start = '\0';

        if (!parse_time (tokens[1], &hour, &minute, &seconds))
            return NULL;
    }
  
    if (timezone == NULL)
        timezone = g_time_zone_new_local ();

    return g_date_time_new (timezone, year, month, day, hour, minute, seconds);
}

static gboolean
parse_result (const gchar *content_type, const gchar *content, gsize content_length, JsonObject **response, gchar **change_id, GError **error)
{
    g_autoptr(JsonParser) parser = NULL;
    g_autoptr(GError) error_local = NULL;
    JsonObject *root;
    const gchar *type;

    if (content_type == NULL) {
        g_set_error_literal (error,
                             SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_PARSE_ERROR,
                             "snapd returned no content type");
        return FALSE;
    }
    if (g_strcmp0 (content_type, "application/json") != 0) {
        g_set_error (error,
                     SNAPD_CLIENT_ERROR,
                     SNAPD_CLIENT_ERROR_PARSE_ERROR,
                     "snapd returned unexpected content type %s", content_type);
        return FALSE;
    }

    parser = json_parser_new ();
    if (!json_parser_load_from_data (parser, content, content_length, &error_local)) {
        g_set_error (error,
                     SNAPD_CLIENT_ERROR,
                     SNAPD_CLIENT_ERROR_PARSE_ERROR,
                     "Unable to parse snapd response: %s",
                     error_local->message);
        return FALSE;
    }

    if (!JSON_NODE_HOLDS_OBJECT (json_parser_get_root (parser))) {
        g_set_error_literal (error,
                             SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_PARSE_ERROR,
                             "snapd response does is not a valid JSON object");
        return FALSE;
    }
    root = json_node_get_object (json_parser_get_root (parser));

    type = get_string (root, "type", NULL);
    if (g_strcmp0 (type, "error") == 0) {
        const gchar *kind, *message;
        gint64 status_code;
        JsonObject *result;

        result = get_object (root, "result");
        status_code = get_int (root, "status-code", 0);
        kind = result != NULL ? get_string (result, "kind", NULL) : NULL;
        message = result != NULL ? get_string (result, "message", NULL) : NULL;

        if (g_strcmp0 (kind, "login-required") == 0) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_LOGIN_REQUIRED,
                                 message);
            return FALSE;
        }
        else if (g_strcmp0 (kind, "invalid-auth-data") == 0) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_INVALID_AUTH_DATA,
                                 message);
            return FALSE;
        }
        else if (g_strcmp0 (kind, "two-factor-required") == 0) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_TWO_FACTOR_REQUIRED,
                                 message);
            return FALSE;
        }
        else if (g_strcmp0 (kind, "two-factor-failed") == 0) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_TWO_FACTOR_FAILED,
                                 message);
            return FALSE;
        }
        else if (status_code == SOUP_STATUS_BAD_REQUEST) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_BAD_REQUEST,
                                 message);
            return FALSE;
        }
        else {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_GENERAL_ERROR,
                                 message);
            return FALSE;
        }
    }
    else if (g_strcmp0 (type, "async") == 0) {
        if (change_id)
            *change_id = g_strdup (get_string (root, "change", NULL));
    }
    else if (g_strcmp0 (type, "sync") == 0) {
    }

    if (response != NULL)
        *response = json_object_ref (root);

    return TRUE;
}

static void
parse_get_system_information_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    JsonObject *result;
    g_autoptr(SnapdSystemInformation) system_information = NULL;
    JsonObject *os_release;
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    result = get_object (response, "result");
    if (result == NULL) {
        error = g_error_new (SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_READ_ERROR,
                             "No result returned");
        snapd_request_complete (request, error);
        return;
    }

    os_release = get_object (result, "os-release");
    system_information = g_object_new (SNAPD_TYPE_SYSTEM_INFORMATION,
                                       "on-classic", get_bool (result, "on-classic", FALSE),
                                       "os-id", os_release != NULL ? get_string (os_release, "id", NULL) : NULL,
                                       "os-version", os_release != NULL ? get_string (os_release, "version-id", NULL) : NULL,
                                       "series", get_string (result, "series", NULL),
                                       "version", get_string (result, "version", NULL),
                                       NULL);
    request->system_information = g_steal_pointer (&system_information);
    snapd_request_complete (request, NULL);
}

static SnapdSnap *
parse_snap (JsonObject *object, GError **error)
{
    const gchar *confinement_string;
    SnapdConfinement confinement = SNAPD_CONFINEMENT_UNKNOWN;
    const gchar *snap_type_string;
    SnapdConfinement snap_type = SNAPD_SNAP_TYPE_UNKNOWN;
    const gchar *snap_status_string;
    SnapdSnapStatus snap_status = SNAPD_SNAP_STATUS_UNKNOWN;
    g_autoptr(JsonArray) apps = NULL;
    g_autoptr(GDateTime) install_date = NULL;
    g_autoptr(JsonArray) prices = NULL;
    g_autoptr(GPtrArray) apps_array = NULL;
    g_autoptr(GPtrArray) prices_array = NULL;
    guint i;

    confinement_string = get_string (object, "confinement", "");
    if (strcmp (confinement_string, "strict") == 0)
        confinement = SNAPD_CONFINEMENT_STRICT;
    else if (strcmp (confinement_string, "devmode") == 0)
        confinement = SNAPD_CONFINEMENT_DEVMODE;

    snap_type_string = get_string (object, "type", "");
    if (strcmp (snap_type_string, "app") == 0)
        snap_type = SNAPD_SNAP_TYPE_APP;
    else if (strcmp (snap_type_string, "kernel") == 0)
        snap_type = SNAPD_SNAP_TYPE_KERNEL;
    else if (strcmp (snap_type_string, "gadget") == 0)
        snap_type = SNAPD_SNAP_TYPE_GADGET;
    else if (strcmp (snap_type_string, "os") == 0)
        snap_type = SNAPD_SNAP_TYPE_OS;

    snap_status_string = get_string (object, "status", "");
    if (strcmp (snap_status_string, "available") == 0)
        snap_status = SNAPD_SNAP_STATUS_AVAILABLE;
    else if (strcmp (snap_status_string, "priced") == 0)
        snap_status = SNAPD_SNAP_STATUS_PRICED;
    else if (strcmp (snap_status_string, "installed") == 0)
        snap_status = SNAPD_SNAP_STATUS_INSTALLED;
    else if (strcmp (snap_status_string, "active") == 0)
        snap_status = SNAPD_SNAP_STATUS_ACTIVE;

    apps = get_array (object, "apps");
    apps_array = g_ptr_array_new_with_free_func (g_object_unref);
    for (i = 0; i < json_array_get_length (apps); i++) {
        JsonNode *node = json_array_get_element (apps, i);
        JsonObject *a;
        g_autoptr(SnapdApp) app = NULL;

        if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected app type");
            return NULL;
        }

        a = json_node_get_object (node);
        app = g_object_new (SNAPD_TYPE_APP,
                            "name", get_string (a, "name", NULL),
                            NULL);
        g_ptr_array_add (apps_array, g_steal_pointer (&app));
    }

    install_date = get_date_time (object, "install-date");

    prices = get_array (object, "prices");
    prices_array = g_ptr_array_new_with_free_func (g_object_unref);
    for (i = 0; i < json_array_get_length (prices); i++) {
        JsonNode *node = json_array_get_element (apps, i);
        JsonObject *p;
        g_autoptr(SnapdPrice) price = NULL;

        if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected price type");
            return NULL;
        }

        p = json_node_get_object (node);
        price = g_object_new (SNAPD_TYPE_PRICE,
                              "amount", get_string (p, "price", NULL),
                              "currency", get_string (p, "currency", NULL),
                              NULL);
        g_ptr_array_add (prices_array, g_steal_pointer (&price));
    }

    return g_object_new (SNAPD_TYPE_SNAP,
                         "apps", apps_array,
                         "channel", get_string (object, "channel", NULL),
                         "confinement", confinement,
                         "description", get_string (object, "description", NULL),
                         "developer", get_string (object, "developer", NULL),
                         "devmode", get_bool (object, "devmode", FALSE),
                         "download-size", get_int (object, "download-size", 0),
                         "icon", get_string (object, "icon", NULL),
                         "id", get_string (object, "id", NULL),
                         "install-date", install_date,
                         "installed-size", get_int (object, "installed-size", 0),
                         "name", get_string (object, "name", NULL),
                         "prices", prices_array,
                         "private", get_bool (object, "private", FALSE),
                         "revision", get_string (object, "revision", NULL),
                         "snap-type", snap_type,
                         "status", snap_status,
                         "summary", get_string (object, "summary", NULL),
                         "trymode", get_bool (object, "trymode", FALSE),
                         "version", get_string (object, "version", NULL),
                         NULL);
}

static GPtrArray *
parse_snap_array (JsonArray *array, GError **error)
{
    g_autoptr(GPtrArray) snaps = NULL;
    guint i;

    snaps = g_ptr_array_new_with_free_func (g_object_unref);
    for (i = 0; i < json_array_get_length (array); i++) {
        JsonNode *node = json_array_get_element (array, i);
        SnapdSnap *snap;

        if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected snap type");
            return NULL;
        }

        snap = parse_snap (json_node_get_object (node), error);
        if (snap == NULL)
            return NULL;
        g_ptr_array_add (snaps, snap);
    }

    return g_steal_pointer (&snaps);
}

static void
parse_get_icon_response (SnapdRequest *request, guint code, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    const gchar *content_type;
    g_autoptr(SnapdIcon) icon = NULL;
    g_autoptr(GBytes) data = NULL;

    content_type = soup_message_headers_get_content_type (headers, NULL);
    if (g_strcmp0 (content_type, "application/json") == 0) {
        GError *error = NULL;

        if (!parse_result (content_type, content, content_length, NULL, NULL, &error)) {
            snapd_request_complete (request, error);
            return;
        }
    }

    if (code != SOUP_STATUS_OK) {
        GError *error = g_error_new (SNAPD_CLIENT_ERROR,
                                     SNAPD_CLIENT_ERROR_READ_ERROR,
                                     "Got response %u retrieving icon", code);
        snapd_request_complete (request, error);
    }

    data = g_bytes_new (content, content_length);
    icon = g_object_new (SNAPD_TYPE_ICON,
                         "mime-type", content_type,
                         "data", data,
                         NULL);

    request->icon = g_steal_pointer (&icon);
    snapd_request_complete (request, NULL);
}

static void
parse_list_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    g_autoptr(JsonArray) array = NULL;
    GPtrArray *snaps;
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    array = get_array (response, "result");
    snaps = parse_snap_array (array, &error);
    if (snaps == NULL) {
        snapd_request_complete (request, error);
        return;
    }

    request->snaps = g_steal_pointer (&snaps);
    snapd_request_complete (request, NULL);
}

static void
parse_list_one_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    JsonObject *result;
    g_autoptr(SnapdSnap) snap = NULL;
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    result = get_object (response, "result");
    if (result == NULL) {
        error = g_error_new (SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_READ_ERROR,
                             "No result returned");
        snapd_request_complete (request, error);
        return;
    }

    snap = parse_snap (result, &error);
    if (snap == NULL) {
        snapd_request_complete (request, error);
        return;
    }

    request->snap = g_steal_pointer (&snap);
    snapd_request_complete (request, NULL);
}

static GPtrArray *
get_connections (JsonObject *object, const gchar *name, GError **error)
{
    g_autoptr(JsonArray) array = NULL;
    GPtrArray *connections;
    guint i;

    connections = g_ptr_array_new_with_free_func (g_object_unref);
    array = get_array (object, "connections");
    for (i = 0; i < json_array_get_length (array); i++) {
        JsonNode *node = json_array_get_element (array, i);
        JsonObject *object;
        SnapdConnection *connection;

        if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
            g_set_error_literal (error,
                                 SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected connection type");
            return NULL;
        }

        object = json_node_get_object (node);
        connection = g_object_new (SNAPD_TYPE_CONNECTION,
                                   "name", get_string (object, name, NULL),
                                   "snap", get_string (object, "snap", NULL),
                                   NULL);
        g_ptr_array_add (connections, connection);
    }

    return connections;
}

static void
parse_get_interfaces_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    g_autoptr(GPtrArray) plug_array = NULL;
    g_autoptr(GPtrArray) slot_array = NULL;
    JsonObject *result;
    g_autoptr(JsonArray) plugs = NULL;
    g_autoptr(JsonArray) slots = NULL;
    guint i;
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    result = get_object (response, "result");
    if (result == NULL) {
        error = g_error_new (SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_READ_ERROR,
                             "No result returned");
        snapd_request_complete (request, error);
        return;
    }

    plugs = get_array (result, "plugs");
    plug_array = g_ptr_array_new_with_free_func (g_object_unref);
    for (i = 0; i < json_array_get_length (plugs); i++) {
        JsonNode *node = json_array_get_element (plugs, i);
        JsonObject *object;
        g_autoptr(GPtrArray) connections = NULL;
        g_autoptr(SnapdPlug) plug = NULL;

        if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
            error = g_error_new (SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected plug type");
            snapd_request_complete (request, error);
            return;
        }
        object = json_node_get_object (node);

        connections = get_connections (object, "slot", &error);
        if (connections == NULL) {
            snapd_request_complete (request, error);
            return;
        }

        plug = g_object_new (SNAPD_TYPE_PLUG,
                             "name", get_string (object, "plug", NULL),
                             "snap", get_string (object, "snap", NULL),
                             "interface", get_string (object, "interface", NULL),
                             "label", get_string (object, "label", NULL),
                             "connections", g_steal_pointer (&connections),
                             // FIXME: apps
                             // FIXME: attrs
                             NULL);
        g_ptr_array_add (plug_array, g_steal_pointer (&plug));
    }
    slots = get_array (result, "slots");
    slot_array = g_ptr_array_new_with_free_func (g_object_unref);
    for (i = 0; i < json_array_get_length (slots); i++) {
        JsonNode *node = json_array_get_element (slots, i);
        JsonObject *object;
        g_autoptr(GPtrArray) connections = NULL;
        g_autoptr(SnapdSlot) slot = NULL;

        if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
            error = g_error_new (SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected slot type");
            snapd_request_complete (request, error);
            return;
        }
        object = json_node_get_object (node);

        connections = get_connections (object, "plug", &error);
        if (connections == NULL) {
            snapd_request_complete (request, error);
            return;
        }

        slot = g_object_new (SNAPD_TYPE_SLOT,
                             "name", get_string (object, "slot", NULL),
                             "snap", get_string (object, "snap", NULL),
                             "interface", get_string (object, "interface", NULL),
                             "label", get_string (object, "label", NULL),
                             "connections", g_steal_pointer (&connections),
                             // FIXME: apps
                             // FIXME: attrs
                             NULL);
        g_ptr_array_add (slot_array, g_steal_pointer (&slot));
    }

    request->plugs = g_steal_pointer (&plug_array);
    request->slots = g_steal_pointer (&slot_array);
    request->result = TRUE;
    snapd_request_complete (request, NULL);
}

static gboolean
async_poll_cb (gpointer data)
{
    SnapdRequest *request = data;
    g_autofree gchar *path = NULL;

    path = g_strdup_printf ("/v2/changes/%s", request->change_id);
    send_request (request, TRUE, "GET", path, NULL, NULL);

    request->poll_timer = 0;
    return G_SOURCE_REMOVE;
}

static gboolean
async_timeout_cb (gpointer data)
{
    SnapdRequest *request = data;
    GError *error;

    request->timeout_timer = 0;

    error = g_error_new (SNAPD_CLIENT_ERROR,
                         SNAPD_CLIENT_ERROR_READ_ERROR,
                         "Timeout waiting for snapd");
    snapd_request_complete (request, error);

    return G_SOURCE_REMOVE;
}

static gboolean
times_equal (GDateTime *time1, GDateTime *time2)
{
    if (time1 == NULL || time2 == NULL)
        return time1 == time2;
    return g_date_time_equal (time1, time2);
}

static gboolean
tasks_equal (SnapdTask *task1, SnapdTask *task2)
{
    return g_strcmp0 (snapd_task_get_id (task1), snapd_task_get_id (task2)) == 0 &&
           g_strcmp0 (snapd_task_get_kind (task1), snapd_task_get_kind (task2)) == 0 &&
           g_strcmp0 (snapd_task_get_summary (task1), snapd_task_get_summary (task2)) == 0 &&
           g_strcmp0 (snapd_task_get_status (task1), snapd_task_get_status (task2)) == 0 &&
           !!snapd_task_get_ready (task1) == !!snapd_task_get_ready (task2) &&
           snapd_task_get_progress_done (task1) == snapd_task_get_progress_done (task2) &&
           snapd_task_get_progress_total (task1) == snapd_task_get_progress_total (task2) &&    
           times_equal (snapd_task_get_spawn_time (task1), snapd_task_get_spawn_time (task2)) &&
           times_equal (snapd_task_get_spawn_time (task1), snapd_task_get_spawn_time (task2));
}

static gboolean
progress_equal (SnapdTask *main_task1, GPtrArray *tasks1, SnapdTask *main_task2, GPtrArray *tasks2)
{
    if (tasks1 == NULL || tasks2 == NULL) {
        if (tasks1 != tasks2)
            return FALSE;
    }
    else {
        int i;

        if (tasks1->len != tasks2->len)
            return FALSE;
        for (i = 0; i < tasks1->len; i++) {
            SnapdTask *t1 = tasks1->pdata[i], *t2 = tasks2->pdata[i];
            if (!tasks_equal (t1, t2))
                return FALSE;
        }
    }

    if (main_task1 == NULL || main_task2 == NULL) {
        if (main_task1 != main_task2)
            return FALSE;
    }
    else {
        if (!tasks_equal (main_task1, main_task2))
            return FALSE;
    }

    return TRUE;
}

static void
parse_async_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    g_autofree gchar *change_id = NULL;
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, &change_id, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    /* Cancel any pending timeout */
    if (request->timeout_timer != 0)
        g_source_remove (request->timeout_timer);
    request->timeout_timer = 0;

    /* First run we expect the change ID, following runs are updates */
    if (request->change_id == NULL) {
         if (change_id == NULL) {
             g_error_new (SNAPD_CLIENT_ERROR,
                          SNAPD_CLIENT_ERROR_READ_ERROR,
                          "No async response received");
             snapd_request_complete (request, error);
             return;
         }

         request->change_id = g_strdup (change_id);
    }
    else {
        gboolean ready;
        JsonObject *result;

        if (change_id != NULL) {
             error = g_error_new (SNAPD_CLIENT_ERROR,
                                  SNAPD_CLIENT_ERROR_READ_ERROR,
                                  "Duplicate async response received");
             snapd_request_complete (request, error);
             return;
        }

        result = get_object (response, "result");
        if (result == NULL) {
            error = g_error_new (SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "No async result returned");
            snapd_request_complete (request, error);
            return;
        }

        if (g_strcmp0 (request->change_id, get_string (result, "id", NULL)) != 0) {
            error = g_error_new (SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected change ID returned");
            snapd_request_complete (request, error);
            return;
        }

        /* Update caller with progress */
        if (request->progress_callback != NULL) {
            g_autoptr(JsonArray) array = NULL;
            guint i;
            g_autoptr(GPtrArray) tasks = NULL;
            g_autoptr(SnapdTask) main_task = NULL;
            g_autoptr(GDateTime) main_spawn_time = NULL;
            g_autoptr(GDateTime) main_ready_time = NULL;

            array = get_array (result, "tasks");
            tasks = g_ptr_array_new_with_free_func (g_object_unref);
            for (i = 0; i < json_array_get_length (array); i++) {
                JsonNode *node = json_array_get_element (array, i);
                JsonObject *object, *progress;
                g_autoptr(GDateTime) spawn_time = NULL;
                g_autoptr(GDateTime) ready_time = NULL;
                g_autoptr(SnapdTask) t = NULL;

                if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
                    error = g_error_new (SNAPD_CLIENT_ERROR,
                                         SNAPD_CLIENT_ERROR_READ_ERROR,
                                         "Unexpected task type");
                    snapd_request_complete (request, error);
                    return;
                }
                object = json_node_get_object (node);
                progress = get_object (object, "progress");
                spawn_time = get_date_time (object, "spawn-time");
                ready_time = get_date_time (object, "ready-time");

                t = g_object_new (SNAPD_TYPE_TASK,
                                  "id", get_string (object, "id", NULL),
                                  "kind", get_string (object, "kind", NULL),
                                  "summary", get_string (object, "summary", NULL),
                                  "status", get_string (object, "status", NULL),
                                  "ready", get_bool (object, "ready", FALSE),
                                  "progress-done", progress != NULL ? get_int (progress, "done", 0) : 0,
                                  "progress-total", progress != NULL ? get_int (progress, "total", 0) : 0,
                                  "spawn-time", spawn_time,
                                  "ready-time", ready_time,
                                  NULL);
                g_ptr_array_add (tasks, g_steal_pointer (&t));
            }

            main_spawn_time = get_date_time (result, "spawn-time");
            main_ready_time = get_date_time (result, "ready-time");
            main_task = g_object_new (SNAPD_TYPE_TASK,
                                      "id", get_string (result, "id", NULL),
                                      "kind", get_string (result, "kind", NULL),
                                      "summary", get_string (result, "summary", NULL),
                                      "status", get_string (result, "status", NULL),
                                      "ready", get_bool (result, "ready", FALSE),
                                      "spawn-time", main_spawn_time,
                                      "ready-time", main_ready_time,
                                      NULL);

            if (!progress_equal (request->main_task, request->tasks, main_task, tasks)) {
                g_clear_object (&request->main_task);
                request->main_task = g_steal_pointer (&main_task);
                g_clear_pointer (&request->tasks, g_ptr_array_unref);
                request->tasks = g_steal_pointer (&tasks);
                request->progress_callback (request->client, request->main_task, request->tasks, request->progress_callback_data);
            }
        }

        ready = get_bool (result, "ready", FALSE);
        if (ready) {
            request->result = TRUE;
            snapd_request_complete (request, NULL);
            return;
        }
    }

    /* Poll for updates */
    if (request->poll_timer != 0)
        g_source_remove (request->poll_timer);
    request->poll_timer = g_timeout_add (ASYNC_POLL_TIME, async_poll_cb, request);
    request->timeout_timer = g_timeout_add (ASYNC_POLL_TIMEOUT, async_timeout_cb, request);
}

static void
parse_connect_interface_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    parse_async_response (request, headers, content, content_length);
}

static void
parse_disconnect_interface_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    parse_async_response (request, headers, content, content_length);
}

static void
parse_login_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    JsonObject *result;
    g_autoptr(JsonArray) discharges = NULL;
    g_autoptr(GPtrArray) discharge_array = NULL;
    guint i;
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    result = get_object (response, "result");
    if (result == NULL) {
        error = g_error_new (SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_READ_ERROR,
                             "No result returned");
        snapd_request_complete (request, error);
        return;
    }

    discharges = get_array (result, "discharges");
    discharge_array = g_ptr_array_new ();
    for (i = 0; i < json_array_get_length (discharges); i++) {
        JsonNode *node = json_array_get_element (discharges, i);

        if (json_node_get_value_type (node) != G_TYPE_STRING) {
            error = g_error_new (SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected discharge type");
            snapd_request_complete (request, error);
            return;
        }

        g_ptr_array_add (discharge_array, (gpointer) json_node_get_string (node));
    }
    g_ptr_array_add (discharge_array, NULL);
    request->received_auth_data = snapd_auth_data_new (get_string (result, "macaroon", NULL), (gchar **) discharge_array->pdata);
    snapd_request_complete (request, NULL);
}

static void
parse_find_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    g_autoptr(JsonArray) array = NULL;
    g_autoptr(GPtrArray) snaps = NULL;
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    array = get_array (response, "result");
    snaps = parse_snap_array (array, &error);
    if (snaps == NULL) {
        snapd_request_complete (request, error);
        return;
    }

    request->suggested_currency = g_strdup (get_string (response, "suggested-currency", NULL));

    request->snaps = g_steal_pointer (&snaps);
    snapd_request_complete (request, NULL);
}

static void
parse_get_payment_methods_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    g_autoptr(JsonObject) response = NULL;
    gboolean allows_automatic_payment;
    GPtrArray *payment_methods;
    GError *error = NULL;
    JsonObject *result;
    g_autoptr(JsonArray) methods = NULL;
    guint i;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, &response, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    result = get_object (response, "result");
    if (result == NULL) {
        error = g_error_new (SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_READ_ERROR,
                             "No result returned");
        snapd_request_complete (request, error);
        return;
    }

    allows_automatic_payment = get_bool (result, "allows-automatic-payment", FALSE);
    payment_methods = g_ptr_array_new_with_free_func (g_object_unref);
    methods = get_array (result, "methods");
    for (i = 0; i < json_array_get_length (methods); i++) {
        JsonNode *node = json_array_get_element (methods, i);
        JsonObject *object;
        g_autoptr(JsonArray) currencies = NULL;
        g_autoptr(GPtrArray) currencies_array = NULL;
        guint j;
        SnapdPaymentMethod *payment_method;

        if (json_node_get_value_type (node) != JSON_TYPE_OBJECT) {
            error = g_error_new (SNAPD_CLIENT_ERROR,
                                 SNAPD_CLIENT_ERROR_READ_ERROR,
                                 "Unexpected method type");
            snapd_request_complete (request, error);
            return;
        }
        object = json_node_get_object (node);

        currencies = get_array (object, "currencies");
        currencies_array = g_ptr_array_new ();
        for (j = 0; j < json_array_get_length (currencies); j++) {
            JsonNode *node = json_array_get_element (currencies, j);

            if (json_node_get_value_type (node) != G_TYPE_STRING) {
                error = g_error_new (SNAPD_CLIENT_ERROR,
                                     SNAPD_CLIENT_ERROR_READ_ERROR,
                                     "Unexpected currency type");
                snapd_request_complete (request, error);
                return;
            }

            g_ptr_array_add (currencies_array, (gchar *) json_node_get_string (node));
        }
        g_ptr_array_add (currencies_array, NULL);
        payment_method = g_object_new (SNAPD_TYPE_PAYMENT_METHOD,
                                       "backend-id", get_string (object, "backend-id", NULL),
                                       "currencies", (gchar **) currencies_array->pdata,
                                       "description", get_string (object, "description", NULL),
                                       "id", get_int (object, "id", 0),
                                       "preferred", get_bool (object, "preferred", FALSE),
                                       "requires-interaction", get_bool (object, "requires-interaction", FALSE),
                                       NULL);
        g_ptr_array_add (payment_methods, payment_method);
    }

    request->allows_automatic_payment = allows_automatic_payment;
    request->methods = g_steal_pointer (&payment_methods);
    snapd_request_complete (request, NULL);
}

static void
parse_buy_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    GError *error = NULL;

    if (!parse_result (soup_message_headers_get_content_type (headers, NULL), content, content_length, NULL, NULL, &error)) {
        snapd_request_complete (request, error);
        return;
    }

    request->result = TRUE;
    snapd_request_complete (request, NULL);
}

static void
parse_install_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    parse_async_response (request, headers, content, content_length);
}

static void
parse_refresh_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    parse_async_response (request, headers, content, content_length);
}

static void
parse_remove_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    parse_async_response (request, headers, content, content_length);
}

static void
parse_enable_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    parse_async_response (request, headers, content, content_length);
}

static void
parse_disable_response (SnapdRequest *request, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    parse_async_response (request, headers, content, content_length);
}

static SnapdRequest *
get_next_request (SnapdClient *client)
{
    SnapdClientPrivate *priv = snapd_client_get_instance_private (client);
    GList *link;

    for (link = priv->requests; link != NULL; link = link->next) {
        SnapdRequest *request = link->data;
        if (!request->completed)
            return request;
    }

    return NULL;
}

static void
parse_response (SnapdClient *client, guint code, SoupMessageHeaders *headers, const gchar *content, gsize content_length)
{
    SnapdRequest *request;
    GError *error = NULL;

    /* Match this response to the next uncompleted request */
    request = get_next_request (client);
    if (request == NULL) {
        g_warning ("Ignoring unexpected response");
        return;
    }

    switch (request->request_type)
    {
    case SNAPD_REQUEST_GET_SYSTEM_INFORMATION:
        parse_get_system_information_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_GET_ICON:
        parse_get_icon_response (request, code, headers, content, content_length);
        break;
    case SNAPD_REQUEST_LIST:
        parse_list_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_LIST_ONE:
        parse_list_one_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_GET_INTERFACES:
        parse_get_interfaces_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_CONNECT_INTERFACE:
        parse_connect_interface_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_DISCONNECT_INTERFACE:
        parse_disconnect_interface_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_LOGIN:
        parse_login_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_FIND:
        parse_find_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_GET_PAYMENT_METHODS:
        parse_get_payment_methods_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_BUY:
        parse_buy_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_INSTALL:
        parse_install_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_REFRESH:
        parse_refresh_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_REMOVE:
        parse_remove_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_ENABLE:
        parse_enable_response (request, headers, content, content_length);
        break;
    case SNAPD_REQUEST_DISABLE:
        parse_disable_response (request, headers, content, content_length);
        break;
    default:
        error = g_error_new (SNAPD_CLIENT_ERROR,
                             SNAPD_CLIENT_ERROR_GENERAL_ERROR,
                             "Unknown request");
        snapd_request_complete (request, error);
        break;
    }
}

static gboolean
read_data (SnapdClient *client,
           gsize size,
           GCancellable *cancellable)
{
    SnapdClientPrivate *priv = snapd_client_get_instance_private (client);
    gssize n_read;
    g_autoptr(GError) error_local = NULL;

    if (priv->n_read + size > priv->buffer->len)
        g_byte_array_set_size (priv->buffer, priv->n_read + size);
    n_read = g_socket_receive (priv->snapd_socket,
                               (gchar *) (priv->buffer->data + priv->n_read),
                               size,
                               cancellable,
                               &error_local);
    if (n_read < 0)
    {
        g_printerr ("read error\n");
        // FIXME: Cancel all requests
        //g_set_error (error,
        //             SNAPD_CLIENT_ERROR,
        //             SNAPD_CLIENT_ERROR_READ_ERROR,
        //             "Failed to read from snapd: %s",
        //             error_local->message);
        return FALSE;
    }

    priv->n_read += n_read;

    return TRUE;
}

/* Check if we have all HTTP chunks */
static gboolean
have_chunked_body (const gchar *body, gsize body_length)
{
    while (TRUE) {
        const gchar *chunk_start;
        gsize chunk_header_length, chunk_length;

        /* Read chunk header, stopping on zero length chunk */
        chunk_start = g_strstr_len (body, body_length, "\r\n");
        if (chunk_start == NULL)
            return FALSE;
        chunk_header_length = chunk_start - body + 2;
        chunk_length = strtoul (body, NULL, 16);
        if (chunk_length == 0)
            return TRUE;

        /* Check enough space for chunk body */
        if (chunk_header_length + chunk_length + strlen ("\r\n") > body_length)
            return FALSE;
        // FIXME: Validate that \r\n is on the end of a chunk?
        body += chunk_header_length + chunk_length;
        body_length -= chunk_header_length + chunk_length;
    }
}

/* If more than one HTTP chunk, re-order buffer to contain one chunk.
 * Assumes body is a valid chunked data block (as checked with have_chunked_body()) */
static void
compress_chunks (gchar *body, gsize body_length, gchar **combined_start, gsize *combined_length, gsize *total_length)
{
    gchar *chunk_start;

    /* Use first chunk as output */
    *combined_length = strtoul (body, NULL, 16);
    *combined_start = strstr (body, "\r\n") + 2;

    /* Copy any remaining chunks beside the first one */
    chunk_start = *combined_start + *combined_length + 2;
    while (TRUE) {
        gsize chunk_length;

        chunk_length = strtoul (chunk_start, NULL, 16);
        chunk_start = strstr (chunk_start, "\r\n") + 2;
        if (chunk_length == 0)
            break;

        /* Move this chunk on the end of the last one */
        memmove (*combined_start + *combined_length, chunk_start, chunk_length);
        *combined_length += chunk_length;

        chunk_start += chunk_length + 2;
    }

    *total_length = chunk_start - body;
}

static gboolean
read_from_snapd (SnapdClient *client,
                 GCancellable *cancellable, gboolean blocking)
{
    SnapdClientPrivate *priv = snapd_client_get_instance_private (client);
    gchar *body;
    gsize header_length;
    g_autoptr(SoupMessageHeaders) headers = NULL;
    guint code;
    g_autofree gchar *reason_phrase = NULL;
    gchar *combined_start;
    gsize content_length, combined_length;

    if (!read_data (client, 1024, cancellable))
        return G_SOURCE_REMOVE;

    while (TRUE) {
        /* Look for header divider */
        body = g_strstr_len ((gchar *) priv->buffer->data, priv->n_read, "\r\n\r\n");
        if (body == NULL)
            return G_SOURCE_CONTINUE;
        body += 4;
        header_length = body - (gchar *) priv->buffer->data;

        /* Parse headers */
        headers = soup_message_headers_new (SOUP_MESSAGE_HEADERS_RESPONSE);
        if (!soup_headers_parse_response ((gchar *) priv->buffer->data, header_length, headers,
                                          NULL, &code, &reason_phrase)) {
            // FIXME: Cancel all requests
            return G_SOURCE_REMOVE;
        }

        /* Read content and process content */
        switch (soup_message_headers_get_encoding (headers)) {
        case SOUP_ENCODING_EOF:
            while (!g_socket_is_closed (priv->snapd_socket)) {
                if (!blocking)
                    return G_SOURCE_CONTINUE;
                if (!read_data (client, 1024, cancellable))
                    return G_SOURCE_REMOVE;
            }

            content_length = priv->n_read - header_length;
            parse_response (client, code, headers, body, content_length);
            break;

        case SOUP_ENCODING_CHUNKED:
            // FIXME: Find a way to abort on error
            while (!have_chunked_body (body, priv->n_read - header_length)) {
                if (!blocking)
                    return G_SOURCE_CONTINUE;
                if (!read_data (client, 1024, cancellable))
                    return G_SOURCE_REMOVE;
            }

            compress_chunks (body, priv->n_read - header_length, &combined_start, &combined_length, &content_length);
            parse_response (client, code, headers, combined_start, combined_length);
            break;

        case SOUP_ENCODING_CONTENT_LENGTH:
            content_length = soup_message_headers_get_content_length (headers);
            while (priv->n_read < header_length + content_length) {
                if (!blocking)
                    return G_SOURCE_CONTINUE;
                if (!read_data (client, 1024, cancellable))
                    return G_SOURCE_REMOVE;
            }

            parse_response (client, code, headers, body, content_length);
            break;

        default:
            // FIXME
            return G_SOURCE_REMOVE;
        }

        /* Move remaining data to the start of the buffer */
        g_byte_array_remove_range (priv->buffer, 0, header_length + content_length);
        priv->n_read -= header_length + content_length;
    }
}

static gboolean
read_cb (GSocket *socket, GIOCondition condition, SnapdClient *client)
{
    return read_from_snapd (client, NULL, FALSE); // FIXME: Use Cancellable from first request?
}

/**
 * snapd_client_connect_sync:
 * @client: a #SnapdClient
 * @cancellable: (allow-none): a #GCancellable or %NULL
 * @error: a #GError or %NULL
 *
 * Connect to snapd.
 *
 * Returns: %TRUE if successfully connected to snapd.
 */
gboolean
snapd_client_connect_sync (SnapdClient *client,
                           GCancellable *cancellable, GError **error)
{
    SnapdClientPrivate *priv;
    g_autoptr(GSocketAddress) address = NULL;
    g_autoptr(GError) error_local = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);

    priv = snapd_client_get_instance_private (client);
    g_return_val_if_fail (priv->snapd_socket == NULL, FALSE);

    priv->snapd_socket = g_socket_new (G_SOCKET_FAMILY_UNIX,
                                       G_SOCKET_TYPE_STREAM,
                                       G_SOCKET_PROTOCOL_DEFAULT,
                                       &error_local);
    if (priv->snapd_socket == NULL) {
        g_set_error (error,
                     SNAPD_CLIENT_ERROR,
                     SNAPD_CLIENT_ERROR_CONNECTION_FAILED,
                     "Unable to open snapd socket: %s",
                     error_local->message);
        return FALSE;
    }
    address = g_unix_socket_address_new (SNAPD_SOCKET);
    if (!g_socket_connect (priv->snapd_socket, address, cancellable, &error_local)) {
        g_set_error (error,
                     SNAPD_CLIENT_ERROR,
                     SNAPD_CLIENT_ERROR_CONNECTION_FAILED,
                     "Unable to connect snapd socket: %s",
                     error_local->message);
        g_clear_object (&priv->snapd_socket);
        return FALSE;
    }

    priv->read_source = g_socket_create_source (priv->snapd_socket, G_IO_IN, NULL);
    g_source_set_callback (priv->read_source, (GSourceFunc) read_cb, client, NULL);
    g_source_attach (priv->read_source, NULL);

    return TRUE;
}

static SnapdRequest *
make_request (SnapdClient *client, RequestType request_type,
              SnapdProgressCallback progress_callback, gpointer progress_callback_data,
              GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdClientPrivate *priv = snapd_client_get_instance_private (client);
    SnapdRequest *request;

    request = g_object_new (snapd_request_get_type (), NULL);
    if (priv->auth_data != NULL)
        request->auth_data = g_object_ref (priv->auth_data);
    request->client = client;
    request->request_type = request_type;
    if (cancellable != NULL)
        request->cancellable = g_object_ref (cancellable);
    request->ready_callback = callback;
    request->ready_callback_data = user_data;
    request->progress_callback = progress_callback;
    request->progress_callback_data = progress_callback_data;
    priv->requests = g_list_append (priv->requests, request);

    return request;
}

static SnapdRequest *
make_login_request (SnapdClient *client,
                    const gchar *username, const gchar *password, const gchar *otp,
                    GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autoptr(JsonBuilder) builder = NULL;
    g_autofree gchar *data = NULL;

    request = make_request (client, SNAPD_REQUEST_LOGIN, NULL, NULL, cancellable, callback, user_data);

    builder = json_builder_new ();
    json_builder_begin_object (builder);
    json_builder_set_member_name (builder, "username");
    json_builder_add_string_value (builder, username);
    json_builder_set_member_name (builder, "password");
    json_builder_add_string_value (builder, password);
    if (otp != NULL) {
        json_builder_set_member_name (builder, "otp");
        json_builder_add_string_value (builder, otp);
    }
    json_builder_end_object (builder);
    data = builder_to_string (builder);

    send_request (request, FALSE, "POST", "/v2/login", "application/json", data);

    return request;
}

/**
 * snapd_client_login_sync:
 * @client: a #SnapdClient.
 * @username: usename to log in with.
 * @password: password to log in with.
 * @otp: (allow-none): response to one-time password challenge.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Log into snapd. If successful, the authorization data is updated.
 * This can be requested using snapd_client_get_auth_data().
 *
 * Returns: %TRUE if the login was successful.
 */
gboolean
snapd_client_login_sync (SnapdClient *client,
                         const gchar *username, const gchar *password, const gchar *otp,
                         GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (username != NULL, FALSE);
    g_return_val_if_fail (password != NULL, FALSE);

    request = g_object_ref (make_login_request (client, username, password, otp, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_login_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_login_async:
 * @client: a #SnapdClient.
 * @username: usename to log in with.
 * @password: password to log in with.
 * @otp: (allow-none): response to one-time password challenge.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_login_async (SnapdClient *client,
                          const gchar *username, const gchar *password, const gchar *otp,
                          GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_login_request (client, username, password, otp, cancellable, callback, user_data);
}

/**
 * snapd_client_login_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Complete a login request. If successful, the authorization data is updated.
 * This can be requested using snapd_client_get_auth_data().
 *
 * Returns: %TRUE if the login was successful.
 */
gboolean
snapd_client_login_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdClientPrivate *priv;
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    priv = snapd_client_get_instance_private (client);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_LOGIN, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;

    g_clear_object (&priv->auth_data);
    priv->auth_data = g_object_ref (request->received_auth_data);

    return TRUE;
}

/**
 * snapd_client_set_auth_data:
 * @client: a #SnapdClient.
 * @auth_data: (allow-none): a #SnapdAuthData or %NULL.
 *
 * Set the authorization data to use for requests.
 */
void
snapd_client_set_auth_data (SnapdClient *client, SnapdAuthData *auth_data)
{
    SnapdClientPrivate *priv;

    g_return_if_fail (SNAPD_IS_CLIENT (client));

    priv = snapd_client_get_instance_private (client);
    g_clear_object (&priv->auth_data);
    if (auth_data != NULL)
        priv->auth_data = g_object_ref (auth_data);
}

/**
 * snapd_client_get_auth_data:
 * @client: a #SnapdClient.
 *
 * Returns: (transfer none): the #SnapdAuthData that is used for requests or %NULL if none used.
 */
SnapdAuthData *
snapd_client_get_auth_data (SnapdClient *client)
{
    SnapdClientPrivate *priv;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
  
    priv = snapd_client_get_instance_private (client);

    return priv->auth_data;
}

static SnapdRequest *
make_get_system_information_request (SnapdClient *client,
                                     GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;

    request = make_request (client, SNAPD_REQUEST_GET_SYSTEM_INFORMATION, NULL, NULL, cancellable, callback, user_data);
    send_request (request, TRUE, "GET", "/v2/system-info", NULL, NULL);

    return request;
}

/**
 * snapd_client_get_system_information_sync:
 * @client: a #SnapdClient.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Request system information from snapd.
 * While this blocks, snapd is expected to return the information quickly.
 *
 * Returns: (transfer full): a #SnapdSystemInformation or %NULL on error.
 */
SnapdSystemInformation *
snapd_client_get_system_information_sync (SnapdClient *client,
                                          GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);

    request = g_object_ref (make_get_system_information_request (client, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_get_system_information_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_get_system_information_async:
 * @client: a #SnapdClient.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 *
 * Request system information asynchronously from snapd.
 */
void
snapd_client_get_system_information_async (SnapdClient *client,
                                           GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_get_system_information_request (client, cancellable, callback, user_data);
}

/**
 * snapd_client_get_system_information_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer full): a #SnapdSystemInformation or %NULL on error.
 */
SnapdSystemInformation *
snapd_client_get_system_information_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), NULL);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_GET_SYSTEM_INFORMATION, NULL);

    if (snapd_request_set_error (request, error))
        return NULL;
    return request->system_information != NULL ? g_object_ref (request->system_information) : NULL;
}

static SnapdRequest *
make_list_one_request (SnapdClient *client,
                       const gchar *name,
                       GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autofree gchar *escaped = NULL, *path = NULL;

    request = make_request (client, SNAPD_REQUEST_LIST_ONE, NULL, NULL, cancellable, callback, user_data);
    escaped = soup_uri_encode (name, NULL);
    path = g_strdup_printf ("/v2/snaps/%s", escaped);
    send_request (request, TRUE, "GET", path, NULL, NULL);

    return request;
}

/**
 * snapd_client_list_one_sync:
 * @client: a #SnapdClient.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer full): a #SnapdSnap or %NULL on error.
 */
SnapdSnap *
snapd_client_list_one_sync (SnapdClient *client,
                            const gchar *name,
                            GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);

    request = g_object_ref (make_list_one_request (client, name, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_list_one_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_list_one_async:
 * @client: a #SnapdClient.
 * @name: name of snap to get.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_list_one_async (SnapdClient *client,
                             const gchar *name,
                             GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_list_one_request (client, name, cancellable, callback, user_data);
}

/**
 * snapd_client_list_one_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer full): a #SnapdSnap or %NULL on error.
 */
SnapdSnap *
snapd_client_list_one_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), NULL);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_LIST_ONE, NULL);
  
    if (snapd_request_set_error (request, error))
        return NULL;
    return request->snap != NULL ? g_object_ref (request->snap) : NULL;
}

static SnapdRequest *
make_get_icon_request (SnapdClient *client,
                       const gchar *name,
                       GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autofree gchar *escaped = NULL, *path = NULL;

    request = make_request (client, SNAPD_REQUEST_GET_ICON, NULL, NULL, cancellable, callback, user_data);
    escaped = soup_uri_encode (name, NULL);
    path = g_strdup_printf ("/v2/icons/%s/icon", escaped);
    send_request (request, TRUE, "GET", path, NULL, NULL);

    return request;
}

/**
 * snapd_client_get_icon_sync:
 * @client: a #SnapdClient.
 * @name: name of snap to get icon for.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer full): a #SnapdIcon or %NULL on error.
 */
SnapdIcon *
snapd_client_get_icon_sync (SnapdClient *client,
                            const gchar *name,
                            GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);

    request = g_object_ref (make_get_icon_request (client, name, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_get_icon_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_get_icon_async:
 * @client: a #SnapdClient.
 * @name: name of snap to get icon for.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_get_icon_async (SnapdClient *client,
                             const gchar *name,
                             GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_get_icon_request (client, name, cancellable, callback, user_data);
}

/**
 * snapd_client_get_icon_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer full): a #SnapdIcon or %NULL on error.
 */
SnapdIcon *
snapd_client_get_icon_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), NULL);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_GET_ICON, NULL);

    if (snapd_request_set_error (request, error))
        return NULL;
    return request->icon != NULL ? g_object_ref (request->icon) : NULL;
}

static SnapdRequest *
make_list_request (SnapdClient *client,
                   GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;

    request = make_request (client, SNAPD_REQUEST_LIST, NULL, NULL, cancellable, callback, user_data);
    send_request (request, TRUE, "GET", "/v2/snaps", NULL, NULL);

    return request;
}

/**
 * snapd_client_list_sync:
 * @client: a #SnapdClient.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer container) (element-type SnapdSnap): an array of #SnapdSnap or %NULL on error.
 */
GPtrArray *
snapd_client_list_sync (SnapdClient *client,
                        GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);

    request = g_object_ref (make_list_request (client, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_list_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_list_async:
 * @client: a #SnapdClient.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_list_async (SnapdClient *client,
                         GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_list_request (client, cancellable, callback, user_data);
}

/**
 * snapd_client_list_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer container) (element-type SnapdSnap): an array of #SnapdSnap or %NULL on error.
 */
GPtrArray *
snapd_client_list_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), NULL);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_LIST, NULL);

    if (snapd_request_set_error (request, error))
        return NULL;
    return request->snaps != NULL ? g_ptr_array_ref (request->snaps) : NULL;
}

static SnapdRequest *
make_get_interfaces_request (SnapdClient *client,
                             GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;

    request = make_request (client, SNAPD_REQUEST_GET_INTERFACES, NULL, NULL, cancellable, callback, user_data);
    send_request (request, TRUE, "GET", "/v2/interfaces", NULL, NULL);

    return request;
}

/**
 * snapd_client_get_interfaces_sync:
 * @client: a #SnapdClient.
 * @plugs: (out) (allow-none) (transfer container) (element-type SnapdPlug): the location to store the plug array or %NULL.
 * @slots: (out) (allow-none) (transfer container) (element-type SnapdSlot): the location to store the slot array or %NULL.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_get_interfaces_sync (SnapdClient *client,
                                  GPtrArray **plugs, GPtrArray **slots,
                                  GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);

    request = g_object_ref (make_get_interfaces_request (client, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_get_interfaces_finish (client, G_ASYNC_RESULT (request), plugs, slots, error);
}

/**
 * snapd_client_get_interfaces_async:
 * @client: a #SnapdClient.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_get_interfaces_async (SnapdClient *client,
                                   GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_get_interfaces_request (client, cancellable, callback, user_data);
}

/**
 * snapd_client_get_interfaces_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @plugs: (out) (allow-none) (transfer container) (element-type SnapdPlug): the location to store the plug array or %NULL.
 * @slots: (out) (allow-none) (transfer container) (element-type SnapdSlot): the location to store the slot array or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_get_interfaces_finish (SnapdClient *client, GAsyncResult *result,
                                    GPtrArray **plugs, GPtrArray **slots,
                                    GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_GET_INTERFACES, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    if (plugs)
       *plugs = request->plugs != NULL ? g_ptr_array_ref (request->plugs) : NULL;
    if (slots)
       *slots = request->slots != NULL ? g_ptr_array_ref (request->slots) : NULL;
    return request->result;
}

static SnapdRequest *
make_interface_request (SnapdClient *client,
                        RequestType request_type,
                        const gchar *action,
                        const gchar *plug_snap, const gchar *plug_name,
                        const gchar *slot_snap, const gchar *slot_name,
                        SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                        GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autoptr(JsonBuilder) builder = NULL;
    g_autofree gchar *data = NULL;

    request = make_request (client, request_type, progress_callback, progress_callback_data, cancellable, callback, user_data);

    builder = json_builder_new ();
    json_builder_begin_object (builder);
    json_builder_set_member_name (builder, "action");
    json_builder_add_string_value (builder, action);
    json_builder_set_member_name (builder, "plugs");
    json_builder_begin_array (builder);
    json_builder_begin_object (builder);
    json_builder_set_member_name (builder, "snap");
    json_builder_add_string_value (builder, plug_snap);
    json_builder_set_member_name (builder, "plug");
    json_builder_add_string_value (builder, plug_name);
    json_builder_end_object (builder);
    json_builder_end_array (builder);
    json_builder_set_member_name (builder, "slots");
    json_builder_begin_array (builder);
    json_builder_begin_object (builder);
    json_builder_set_member_name (builder, "snap");
    json_builder_add_string_value (builder, slot_snap);
    json_builder_set_member_name (builder, "slot");
    json_builder_add_string_value (builder, slot_name);
    json_builder_end_object (builder);
    json_builder_end_array (builder);
    json_builder_end_object (builder);
    data = builder_to_string (builder);

    send_request (request, TRUE, "POST", "/v2/interfaces", "application/json", data);

    return request;
}

static SnapdRequest *
make_connect_interface_request (SnapdClient *client,
                                const gchar *plug_snap, const gchar *plug_name,
                                const gchar *slot_snap, const gchar *slot_name,
                                SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                                GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    return make_interface_request (client, SNAPD_REQUEST_CONNECT_INTERFACE,
                                   "connect",
                                   plug_snap, plug_name,
                                   slot_snap, slot_name,
                                   progress_callback, progress_callback_data,
                                   cancellable, callback, user_data);
}

/**
 * snapd_client_connect_interface_sync:
 * @client: a #SnapdClient.
 * @plug_snap: name of snap containing plug.
 * @plug_name: name of plug to connect.
 * @slot_snap: name of snap containing socket.
 * @slot_name: name of slot to connect.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 */
gboolean
snapd_client_connect_interface_sync (SnapdClient *client,
                                     const gchar *plug_snap, const gchar *plug_name,
                                     const gchar *slot_snap, const gchar *slot_name,
                                     SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                                     GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);

    request = g_object_ref (make_connect_interface_request (client, plug_snap, plug_name, slot_snap, slot_name, progress_callback, progress_callback_data, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_connect_interface_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_connect_interface_async:
 * @client: a #SnapdClient.
 * @plug_snap: name of snap containing plug.
 * @plug_name: name of plug to connect.
 * @slot_snap: name of snap containing socket.
 * @slot_name: name of slot to connect.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_connect_interface_async (SnapdClient *client,
                                      const gchar *plug_snap, const gchar *plug_name,
                                      const gchar *slot_snap, const gchar *slot_name,
                                      SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                                      GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_connect_interface_request (client, plug_snap, plug_name, slot_snap, slot_name, progress_callback, progress_callback_data, cancellable, callback, user_data);
}

/**
 * snapd_client_connect_interface_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_connect_interface_finish (SnapdClient *client,
                                       GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_CONNECT_INTERFACE, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

static SnapdRequest *
make_disconnect_interface_request (SnapdClient *client,
                                   const gchar *plug_snap, const gchar *plug_name,
                                   const gchar *slot_snap, const gchar *slot_name,
                                   SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                                   GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    return make_interface_request (client, SNAPD_REQUEST_DISCONNECT_INTERFACE,
                                   "disconnect",
                                   plug_snap, plug_name,
                                   slot_snap, slot_name,
                                   progress_callback, progress_callback_data,
                                   cancellable, callback, user_data);
}

/**
 * snapd_client_disconnect_interface_sync:
 * @client: a #SnapdClient.
 * @plug_snap: name of snap containing plug.
 * @plug_name: name of plug to disconnect.
 * @slot_snap: name of snap containing socket.
 * @slot_name: name of slot to disconnect.
 * @progress_callback: (allow-none) (scope call): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_disconnect_interface_sync (SnapdClient *client,
                                        const gchar *plug_snap, const gchar *plug_name,
                                        const gchar *slot_snap, const gchar *slot_name,
                                        SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                                        GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);

    request = g_object_ref (make_disconnect_interface_request (client, plug_snap, plug_name, slot_snap, slot_name, progress_callback, progress_callback_data, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_disconnect_interface_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_disconnect_interface_async:
 * @client: a #SnapdClient.
 * @plug_snap: name of snap containing plug.
 * @plug_name: name of plug to disconnect.
 * @slot_snap: name of snap containing socket.
 * @slot_name: name of slot to disconnect.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_disconnect_interface_async (SnapdClient *client,
                                         const gchar *plug_snap, const gchar *plug_name,
                                         const gchar *slot_snap, const gchar *slot_name,
                                         SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                                         GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_disconnect_interface_request (client, plug_snap, plug_name, slot_snap, slot_name, progress_callback, progress_callback_data, cancellable, callback, user_data);
}

/**
 * snapd_client_disconnect_interface_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_disconnect_interface_finish (SnapdClient *client,
                                          GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_DISCONNECT_INTERFACE, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

static SnapdRequest *
make_find_request (SnapdClient *client,
                   SnapdFindFlags flags, const gchar *query,
                   GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autoptr(GString) path = NULL;
    g_autofree gchar *escaped = NULL;

    request = make_request (client, SNAPD_REQUEST_FIND, NULL, NULL, cancellable, callback, user_data);
    path = g_string_new ("/v2/find");
    escaped = soup_uri_encode (query, NULL);

    if ((flags & SNAPD_FIND_FLAGS_MATCH_NAME) != 0)
        g_string_append_printf (path, "?name=%s", escaped);
    else
        g_string_append_printf (path, "?q=%s", escaped);

    if ((flags & SNAPD_FIND_FLAGS_SELECT_PRIVATE) != 0)
        g_string_append_printf (path, "&select=private");
    else if ((flags & SNAPD_FIND_FLAGS_SELECT_REFRESH) != 0)
        g_string_append_printf (path, "&select=refresh");

    send_request (request, TRUE, "GET", path->str, NULL, NULL);

    return request;
}

/**
 * snapd_client_find_sync:
 * @client: a #SnapdClient.
 * @flags: a set of #SnapdFindFlags to control how the find is performed.
 * @query: query string to send.
 * @suggested_currency: (allow-none): location to store the ISO 4217 currency that is suggested to purchase with.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer container) (element-type SnapdSnap): an array of #SnapdSnap or %NULL on error.
 */
GPtrArray *
snapd_client_find_sync (SnapdClient *client,
                        SnapdFindFlags flags, const gchar *query,
                        gchar **suggested_currency,
                        GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
    g_return_val_if_fail (query != NULL, NULL);

    request = g_object_ref (make_find_request (client, flags, query, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_find_finish (client, G_ASYNC_RESULT (request), suggested_currency, error);
}

/**
 * snapd_client_find_async:
 * @client: a #SnapdClient.
 * @flags: a set of #SnapdFindFlags to control how the find is performed.
 * @query: query string to send.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_find_async (SnapdClient *client,
                         SnapdFindFlags flags, const gchar *query,
                         GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    g_return_if_fail (query != NULL);
    make_find_request (client, flags, query, cancellable, callback, user_data);
}

/**
 * snapd_client_find_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @suggested_currency: (allow-none): location to store the ISO 4217 currency that is suggested to purchase with.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer container) (element-type SnapdSnap): an array of #SnapdSnap or %NULL on error.
 */
GPtrArray *
snapd_client_find_finish (SnapdClient *client, GAsyncResult *result, gchar **suggested_currency, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), NULL);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_FIND, NULL);

    if (snapd_request_set_error (request, error))
        return NULL;

    if (suggested_currency != NULL)
        *suggested_currency = g_steal_pointer (&request->suggested_currency);
    return g_steal_pointer (&request->snaps);
}

static gchar *
make_action_data (const gchar *action, const gchar *channel)
{
    g_autoptr(JsonBuilder) builder = NULL;

    builder = json_builder_new ();
    json_builder_begin_object (builder);
    json_builder_set_member_name (builder, "action");
    json_builder_add_string_value (builder, action);
    if (channel != NULL) {
        json_builder_set_member_name (builder, "channel");
        json_builder_add_string_value (builder, channel);
    }
    json_builder_end_object (builder);

    return builder_to_string (builder);
}

static SnapdRequest *
make_install_request (SnapdClient *client,
                      const gchar *name, const gchar *channel,
                      SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                      GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autofree gchar *data = NULL;
    g_autofree gchar *escaped = NULL, *path = NULL;

    request = make_request (client, SNAPD_REQUEST_INSTALL, progress_callback, progress_callback_data, cancellable, callback, user_data);
    data = make_action_data ("install", channel);
    escaped = soup_uri_encode (name, NULL);
    path = g_strdup_printf ("/v2/snaps/%s", escaped);
    send_request (request, TRUE, "POST", path, "application/json", data);

    return request;
}

/**
 * snapd_client_install_sync:
 * @client: a #SnapdClient.
 * @name: name of snap to install.
 * @channel: (allow-none): channel to install from or %NULL for default.
 * @progress_callback: (allow-none) (scope call): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_install_sync (SnapdClient *client,
                           const gchar *name, const gchar *channel,
                           SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                           GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (name != NULL, FALSE);

    request = g_object_ref (make_install_request (client, name, channel, progress_callback, progress_callback_data, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_install_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_install_async:
 * @client: a #SnapdClient.
 * @name: name of snap to install.
 * @channel: (allow-none): channel to install from or %NULL for default.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_install_async (SnapdClient *client,
                            const gchar *name, const gchar *channel,
                            SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                            GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    g_return_if_fail (name != NULL);
    make_install_request (client, name, channel, progress_callback, progress_callback_data, cancellable, callback, user_data);
}

/**
 * snapd_client_install_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_install_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_INSTALL, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

static SnapdRequest *
make_refresh_request (SnapdClient *client,
                      const gchar *name, const gchar *channel,
                      SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                      GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autofree gchar *data = NULL;
    g_autofree gchar *escaped = NULL, *path = NULL;

    request = make_request (client, SNAPD_REQUEST_REFRESH, progress_callback, progress_callback_data, cancellable, callback, user_data);
    data = make_action_data ("refresh", channel);
    escaped = soup_uri_encode (name, NULL);
    path = g_strdup_printf ("/v2/snaps/%s", escaped);
    send_request (request, TRUE, "POST", path, "application/json", data);

    return request;
}

/**
 * snapd_client_refresh_sync:
 * @client: a #SnapdClient.
 * @name: name of snap to refresh.
 * @channel: (allow-none): channel to refresh from or %NULL for default.
 * @progress_callback: (allow-none) (scope call): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_refresh_sync (SnapdClient *client,
                           const gchar *name, const gchar *channel,
                           SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                           GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (name != NULL, FALSE);

    request = g_object_ref (make_refresh_request (client, name, channel, progress_callback, progress_callback_data, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_refresh_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_refresh_async:
 * @client: a #SnapdClient.
 * @name: name of snap to refresh.
 * @channel: (allow-none): channel to refresh from or %NULL for default.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_refresh_async (SnapdClient *client,
                            const gchar *name, const gchar *channel,
                            SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                            GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    g_return_if_fail (name != NULL);
    make_refresh_request (client, name, channel, progress_callback, progress_callback_data, cancellable, callback, user_data);
}

/**
 * snapd_client_refresh_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_refresh_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_REFRESH, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

static SnapdRequest *
make_remove_request (SnapdClient *client,
                     const gchar *name,
                     SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                     GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autofree gchar *data = NULL;
    g_autofree gchar *escaped = NULL, *path = NULL;

    request = make_request (client, SNAPD_REQUEST_REMOVE, progress_callback, progress_callback_data, cancellable, callback, user_data);
    data = make_action_data ("remove", NULL);
    escaped = soup_uri_encode (name, NULL);
    path = g_strdup_printf ("/v2/snaps/%s", escaped);
    send_request (request, TRUE, "POST", path, "application/json", data);

    return request;
}

/**
 * snapd_client_remove_sync:
 * @client: a #SnapdClient.
 * @name: name of snap to remove.
 * @progress_callback: (allow-none) (scope call): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_remove_sync (SnapdClient *client,
                          const gchar *name,
                          SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                          GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (name != NULL, FALSE);

    request = g_object_ref (make_remove_request (client, name, progress_callback, progress_callback_data, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_remove_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_remove_async:
 * @client: a #SnapdClient.
 * @name: name of snap to remove.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_remove_async (SnapdClient *client,
                           const gchar *name,
                           SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                           GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    g_return_if_fail (name != NULL);

    make_remove_request (client, name, progress_callback, progress_callback_data, cancellable, callback, user_data);
}

/**
 * snapd_client_remove_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_remove_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_REMOVE, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

static SnapdRequest *
make_enable_request (SnapdClient *client,
                     const gchar *name,
                     SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                     GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autofree gchar *data = NULL;
    g_autofree gchar *escaped = NULL, *path = NULL;

    request = make_request (client, SNAPD_REQUEST_ENABLE, progress_callback, progress_callback_data, cancellable, callback, user_data);
    data = make_action_data ("enable", NULL);
    escaped = soup_uri_encode (name, NULL);
    path = g_strdup_printf ("/v2/snaps/%s", escaped);
    send_request (request, TRUE, "POST", path, "application/json", data);

    return request;
}

/**
 * snapd_client_enable_sync:
 * @client: a #SnapdClient.
 * @name: name of snap to enable.
 * @progress_callback: (allow-none) (scope call): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_enable_sync (SnapdClient *client,
                          const gchar *name,
                          SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                          GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (name != NULL, FALSE);

    request = g_object_ref (make_enable_request (client, name, progress_callback, progress_callback_data, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_enable_finish (client, G_ASYNC_RESULT (request), error);
}


/**
 * snapd_client_enable_async:
 * @client: a #SnapdClient.
 * @name: name of snap to enable.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_enable_async (SnapdClient *client,
                           const gchar *name,
                           SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                           GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    g_return_if_fail (name != NULL);

    make_enable_request (client, name, progress_callback, progress_callback_data, cancellable, callback, user_data);
}

/**
 * snapd_client_enable_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_enable_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_ENABLE, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

static SnapdRequest *
make_disable_request (SnapdClient *client,
                      const gchar *name,
                      SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                      GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autofree gchar *data = NULL;
    g_autofree gchar *escaped = NULL, *path = NULL;

    request = make_request (client, SNAPD_REQUEST_DISABLE, progress_callback, progress_callback_data, cancellable, callback, user_data);
    data = make_action_data ("disable", NULL);
    escaped = soup_uri_encode (name, NULL);
    path = g_strdup_printf ("/v2/snaps/%s", escaped);
    send_request (request, TRUE, "POST", path, "application/json", data);

    return request;
}

/**
 * snapd_client_disable_sync:
 * @client: a #SnapdClient.
 * @name: name of snap to disable.
 * @progress_callback: (allow-none) (scope call): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_disable_sync (SnapdClient *client,
                           const gchar *name,
                           SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                           GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (name != NULL, FALSE);

    request = g_object_ref (make_disable_request (client, name,
                                                  progress_callback, progress_callback_data,
                                                  cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_disable_finish (client, G_ASYNC_RESULT (request), error);
}


/**
 * snapd_client_disable_async:
 * @client: a #SnapdClient.
 * @name: name of snap to disable.
 * @progress_callback: (allow-none) (scope async): function to callback with progress.
 * @progress_callback_data: (closure): user data to pass to @progress_callback.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_disable_async (SnapdClient *client,
                            const gchar *name,
                            SnapdProgressCallback progress_callback, gpointer progress_callback_data,
                            GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    g_return_if_fail (name != NULL);

    make_disable_request (client, name,
                          progress_callback, progress_callback_data,
                          cancellable, callback, user_data);
}

/**
 * snapd_client_disable_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_disable_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_DISABLE, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

static SnapdRequest *
make_get_payment_methods_request (SnapdClient *client,
                                  GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;

    request = make_request (client, SNAPD_REQUEST_GET_PAYMENT_METHODS, NULL, NULL, cancellable, callback, user_data);
    send_request (request, TRUE, "GET", "/v2/buy/methods", NULL, NULL);

    return request;
}

/**
 * snapd_client_get_payment_methods_sync:
 * @client: a #SnapdClient.
 * @allows_automatic_payment: (allow-none): the location to store if automatic payments are allowed or %NULL.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer container) (element-type SnapdPaymentMethod): an array of #SnapdPaymentMethod or %NULL on error.
 */
GPtrArray *
snapd_client_get_payment_methods_sync (SnapdClient *client,
                                       gboolean *allows_automatic_payment,
                                       GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);

    request = g_object_ref (make_get_payment_methods_request (client, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_get_payment_methods_finish (client, G_ASYNC_RESULT (request), allows_automatic_payment, error);
}


/**
 * snapd_client_get_payment_methods_async:
 * @client: a #SnapdClient.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_get_payment_methods_async (SnapdClient *client,
                                        GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    make_get_payment_methods_request (client, cancellable, callback, user_data);
}

/**
 * snapd_client_get_payment_methods_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @allows_automatic_payment: (allow-none):
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: (transfer container) (element-type SnapdPaymentMethod): an array of #SnapdPaymentMethod or %NULL on error.
 */
GPtrArray *
snapd_client_get_payment_methods_finish (SnapdClient *client, GAsyncResult *result, gboolean *allows_automatic_payment, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), NULL);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), NULL);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_GET_PAYMENT_METHODS, FALSE);

    if (snapd_request_set_error (request, error))
        return NULL;
    if (allows_automatic_payment)
        *allows_automatic_payment = request->allows_automatic_payment;
    return request->methods != NULL ? g_ptr_array_ref (request->methods) : NULL;
}

static SnapdRequest *
make_buy_request (SnapdClient *client,
                  SnapdSnap *snap, SnapdPrice *price, SnapdPaymentMethod *payment_method,
                  GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    SnapdRequest *request;
    g_autoptr(JsonBuilder) builder = NULL;
    g_autofree gchar *data = NULL;

    request = make_request (client, SNAPD_REQUEST_BUY, NULL, NULL, cancellable, callback, user_data);

    builder = json_builder_new ();
    json_builder_begin_object (builder);
    json_builder_set_member_name (builder, "snap-id");
    json_builder_add_string_value (builder, snapd_snap_get_id (snap));
    json_builder_set_member_name (builder, "snap-name");
    json_builder_add_string_value (builder, snapd_snap_get_name (snap));
    json_builder_set_member_name (builder, "price");
    json_builder_add_string_value (builder, snapd_price_get_amount (price));
    json_builder_set_member_name (builder, "currency");
    json_builder_add_string_value (builder, snapd_price_get_currency (price));
    if (payment_method != NULL) {
        json_builder_set_member_name (builder, "backend-id");
        json_builder_add_string_value (builder, snapd_payment_method_get_backend_id (payment_method));
        json_builder_set_member_name (builder, "method-id");
        json_builder_add_int_value (builder, snapd_payment_method_get_id (payment_method));
    }
    json_builder_end_object (builder);
    data = builder_to_string (builder);

    send_request (request, TRUE, "GET", "/v2/buy/methods", "application/json", data);

    return request;
}

/**
 * snapd_client_buy_sync:
 * @client: a #SnapdClient.
 * @snap: snap to buy.
 * @price: price to pay.
 * @payment_method: payment method to use.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_buy_sync (SnapdClient *client,
                       SnapdSnap *snap, SnapdPrice *price, SnapdPaymentMethod *payment_method,
                       GCancellable *cancellable, GError **error)
{
    g_autoptr(SnapdRequest) request = NULL;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_SNAP (snap), FALSE);
    g_return_val_if_fail (SNAPD_IS_PRICE (price), FALSE);

    request = g_object_ref (make_buy_request (client, snap, price, payment_method, cancellable, NULL, NULL));
    snapd_request_wait (request);
    return snapd_client_buy_finish (client, G_ASYNC_RESULT (request), error);
}

/**
 * snapd_client_buy_async:
 * @client: a #SnapdClient.
 * @snap: snap to buy.
 * @price: price to pay.
 * @payment_method: payment method to use.
 * @cancellable: (allow-none): a #GCancellable or %NULL.
 * @callback: (scope async): a #GAsyncReadyCallback to call when the request is satisfied.
 * @user_data: (closure): the data to pass to callback function.
 */
void
snapd_client_buy_async (SnapdClient *client,
                        SnapdSnap *snap, SnapdPrice *price, SnapdPaymentMethod *payment_method,
                        GCancellable *cancellable, GAsyncReadyCallback callback, gpointer user_data)
{
    g_return_if_fail (SNAPD_IS_CLIENT (client));
    g_return_if_fail (SNAPD_IS_SNAP (snap));
    g_return_if_fail (SNAPD_IS_PRICE (price));

    make_buy_request (client, snap, price, payment_method, cancellable, callback, user_data);
}

/**
 * snapd_client_buy_finish:
 * @client: a #SnapdClient.
 * @result: a #GAsyncResult.
 * @error: (allow-none): #GError location to store the error occurring, or %NULL to ignore.
 *
 * Returns: %TRUE on success or %FALSE on error.
 */
gboolean
snapd_client_buy_finish (SnapdClient *client, GAsyncResult *result, GError **error)
{
    SnapdRequest *request;

    g_return_val_if_fail (SNAPD_IS_CLIENT (client), FALSE);
    g_return_val_if_fail (SNAPD_IS_REQUEST (result), FALSE);

    request = SNAPD_REQUEST (result);
    g_return_val_if_fail (request->request_type == SNAPD_REQUEST_BUY, FALSE);

    if (snapd_request_set_error (request, error))
        return FALSE;
    return request->result;
}

/**
 * snapd_client_new:
 *
 * Create a new client to talk to snapd.
 *
 * Returns: a new #SnapdClient
 **/
SnapdClient *
snapd_client_new (void)
{
    return g_object_new (SNAPD_TYPE_CLIENT, NULL);
}

static void
snapd_client_finalize (GObject *object)
{
    SnapdClientPrivate *priv = snapd_client_get_instance_private (SNAPD_CLIENT (object));

    g_clear_object (&priv->snapd_socket);
    g_clear_object (&priv->auth_data);  
    g_list_free_full (priv->requests, g_object_unref);
    priv->requests = NULL;
    g_clear_pointer (&priv->read_source, g_source_unref);
    g_byte_array_unref (priv->buffer);
    priv->buffer = NULL;
}

static void
snapd_client_class_init (SnapdClientClass *klass)
{
   GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

   gobject_class->finalize = snapd_client_finalize;
}

static void
snapd_client_init (SnapdClient *client)
{
    SnapdClientPrivate *priv = snapd_client_get_instance_private (client);

    priv->buffer = g_byte_array_new ();
}
