/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*  $Id: fg-smtp-info.c,v 1.2 2006/01/06 08:44:31 tkitame Exp $ 
 *
 * Copyright (c) 2005 VA Linux Systems Japan, K.K. All Rights Reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <sys/param.h>
#include <unistd.h>
#include <errno.h>

#include <glib.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include "fg-smtp-info.h"
#include "fg-protocol-xml.h"

#define BUFFSIZE	1024
#define SELECT_TRY_MAX 1

static void fg_smtp_info_class_init (FGSmtpInfoClass *klass);
static void fg_smtp_info_init (FGSmtpInfo *sp);
static void fg_smtp_info_finalize (GObject *object);

/* private */
#define FG_SMTP_INFO_GET_PRIVATE(o)       (G_TYPE_INSTANCE_GET_PRIVATE ((o), FG_TYPE_SMTP_INFO, FGSmtpInfoPrivate))
typedef struct _FGSmtpInfoPrivate FGSmtpInfoPrivate;

struct _FGSmtpInfoPrivate {
        gchar *addr;
        gulong pid;
        gboolean session;
        gchar *mailfrom;
        gchar *rcptto;
        gboolean mydestination;
        gboolean mynetworks;
        gint wait;
        gchar *message;

	/* still not impled */
	gint code;
};

/* Object Signal, maybe unused */
#if 0
typedef enum {
	/* Place Signal Types Here */
	SIGNAL_TYPE_EXAMPLE,
	LAST_SIGNAL
} FGSmtpInfoSignalType;

typedef struct {
	FGSmtpInfo *object;
} FGSmtpInfoSignal;

static guint fg_smtp_info_signals[LAST_SIGNAL] = { 0 };
#endif /* 0 */


static GObjectClass *parent_class = NULL;

GQuark
fg_smtp_info_error_quark (void)
{
	static GQuark quark;

	if (!quark)
		quark = g_quark_from_static_string ("fg_smtp_info_error");

	return quark;
}

GType
fg_smtp_info_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (FGSmtpInfoClass),
			NULL,
			NULL,
			(GClassInitFunc)fg_smtp_info_class_init,
			NULL,
			NULL,
			sizeof (FGSmtpInfo),
			0,
			(GInstanceInitFunc)fg_smtp_info_init,
		};

		type = g_type_register_static(G_TYPE_OBJECT, 
			"FGSmtpInfo", &our_info, 0);
	}

	return type;
}

static void
fg_smtp_info_class_init (FGSmtpInfoClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = fg_smtp_info_finalize;

	/* Create signals here:
	   fg_smtp_info_signals[SIGNAL_TYPE_EXAMPLE] = g_signal_new(...)
 	*/

        g_type_class_add_private (object_class, sizeof (FGSmtpInfoPrivate));
}

static void
fg_smtp_info_init (FGSmtpInfo *obj)
{
	FGSmtpInfoPrivate *priv = FG_SMTP_INFO_GET_PRIVATE (obj);

	/* Initialize private members, etc. */
	priv->addr = NULL;
        priv->pid = 0;
        priv->session = TRUE;
        priv->mailfrom = NULL;
        priv->rcptto = NULL;
        priv->mydestination = FALSE;
        priv->mynetworks = FALSE;
        priv->wait = 0;
        priv->message = NULL;

	priv->code = 250;
}

static void
fg_smtp_info_finalize (GObject *object)
{
	FGSmtpInfo *cobj;
	cobj = FG_SMTP_INFO (object);

	G_OBJECT_CLASS (parent_class)->finalize(object);
}

/* private */

static gchar *
_xpath_get_string (const char *path, const char *attr,
		  xmlXPathContextPtr ctx, gchar *def)
{
	gchar *ret = NULL;
	gchar xpath[256];
	xmlXPathObjectPtr xpathObj;

	if (attr)
		g_snprintf (xpath, sizeof (xpath), "%s/@%s", path, attr);
	else
		g_snprintf (xpath, sizeof (xpath), "%s/text()", path);

	xpathObj = xmlXPathEvalExpression (BAD_CAST xpath, ctx);

	if (xpathObj) {
		ret = (char *) xmlXPathCastToString (xpathObj);
		xmlXPathFreeObject(xpathObj);
		if (ret && *ret == 0) {
			g_free (ret);
			ret = NULL;
			if (def) ret = g_strdup (ret);
		}
	} else {
		if (def) ret = g_strdup (def);
	}
	return ret;
}

static gboolean
_xpath_get_bool (const char *path, const char *attr,
		xmlXPathContextPtr ctx, gboolean def)
{
	gboolean ret = def;
	gchar *tmp;
	gchar xpath[256];
	xmlXPathObjectPtr xpathObj;

	if (attr)
		g_snprintf (xpath, sizeof (xpath), "%s/@%s", path, attr);
	else
		g_snprintf (xpath, sizeof (xpath), "%s/text()", path);

	xpathObj = xmlXPathEvalExpression (BAD_CAST xpath, ctx);

	if (xpathObj) {
		tmp = (char *) xmlXPathCastToString (xpathObj);
		if (strcmp ("true", tmp) == 0 || strcmp ("1", tmp) == 0)
			ret = TRUE;
		else if (strcmp ("false", tmp) == 0 || strcmp ("0", tmp) == 0)
			ret = FALSE;
		g_free (tmp);
		xmlXPathFreeObject(xpathObj);
	}
	return ret;
}

static gint
_xpath_get_int (const char *path, const char *attr,
	       xmlXPathContextPtr ctx, gint def)
{
	gint ret = def;
	gchar *tmp;
	gchar xpath[256];
	xmlXPathObjectPtr xpathObj;

	if (attr)
		g_snprintf (xpath, sizeof (xpath), "%s/@%s", path, attr);
	else
		g_snprintf (xpath, sizeof (xpath), "%s/text()", path);

	xpathObj = xmlXPathEvalExpression (BAD_CAST xpath, ctx);

	if (xpathObj) {
		tmp = (char *) xmlXPathCastToString (xpathObj);
		if (tmp) {
			ret = atoi (tmp);
			g_free (tmp);
		}
		xmlXPathFreeObject(xpathObj);
	}
	return ret;
}

static gchar *
_fg_smtp_info_gen_xml (FGSmtpInfo *fsi, GError **error)
{
	FGSmtpInfoPrivate *priv;

	xmlNodePtr rootNode;
	xmlNodePtr nPtr;
	xmlDocPtr doc;
	xmlChar *xmlbuff;
	gchar tmpbuf[BUFSIZ];
	gint buffersize;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), NULL);	
	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	if (priv->addr == NULL) {
		g_set_error (error, FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_INVALID_VALUE,
			     "IP Address must be setted." );
		return NULL;
	}

	/* Make a XML Doc buffer */
	doc = xmlNewDoc (BAD_CAST "1.0");

	/* nPtr0 is Root */
	rootNode = xmlNewNode (NULL, BAD_CAST FG_PROTO_ELM_ROOT);
	xmlDocSetRootElement (doc, rootNode);	
	xmlNewProp (rootNode, BAD_CAST FG_PROTO_ATTR_SESSION,
		    BAD_CAST (priv->session ? "true" : "false"));	

	/* <addr mynetworks=""></addr>  */
	nPtr = xmlNewChild (rootNode, NULL, BAD_CAST FG_PROTO_ELM_ADDR,
			     BAD_CAST priv->addr);
	xmlNewProp (nPtr, BAD_CAST FG_PROTO_ATTR_MYNETWORKS,
		    BAD_CAST (priv->mynetworks ? "true" : "false"));	

	/* <pid></pid> */
	g_snprintf (tmpbuf, sizeof (tmpbuf), "%ld", priv->pid);
	xmlNewChild (rootNode, NULL, BAD_CAST FG_PROTO_ELM_PID,
		     BAD_CAST tmpbuf);

	/* <mailfrom></mailfrom> */
	if (priv->mailfrom && *(priv->mailfrom) != 0) {
		xmlNewChild (rootNode, NULL,
			     BAD_CAST FG_PROTO_ELM_MAILFROM,
			     BAD_CAST priv->mailfrom);
	}

	/* <rcptto mydestination=""></rcptto> */
	if (priv->rcptto && *(priv->rcptto) != 0) {
		nPtr = xmlNewChild (rootNode, NULL,
				    BAD_CAST FG_PROTO_ELM_RCPTTO,
				    BAD_CAST priv->rcptto);
		xmlNewProp (nPtr, BAD_CAST FG_PROTO_ATTR_MYDESTINATION,
			    BAD_CAST (priv->mydestination ? "true" : "false"));
	}

	if ((priv->message && *(priv->message) != 0) || priv->wait > 0) {
		/* server side only */
		nPtr = xmlNewChild (rootNode, NULL,
				    BAD_CAST FG_PROTO_ELM_ACTION,
				    BAD_CAST priv->message);
		g_snprintf (tmpbuf, sizeof (tmpbuf), "%d", priv->wait);
		xmlNewProp (nPtr, BAD_CAST FG_PROTO_ATTR_WAIT,
			    BAD_CAST tmpbuf);
		g_snprintf (tmpbuf, sizeof (tmpbuf), "%d", priv->code);
		xmlNewProp (nPtr, BAD_CAST FG_PROTO_ATTR_CODE,
			    BAD_CAST tmpbuf);
	}

	xmlDocDumpFormatMemory (doc, &xmlbuff, &buffersize, 1);

	/* no longer used after here  */
	xmlFreeDoc (doc);

	if (buffersize == 0) {
		g_set_error (error, FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_GET_XML_FAILED,
			     "Could not generate XML." );
		return NULL;
	}
	return (gchar *)xmlbuff;
}

static void
_fg_smtp_info_xml_check_error (xmlXPathContextPtr xpathCtx, GError **error)
{
	gint code = 0;
	gchar *domain = NULL;
	gchar *message = NULL;

	message = _xpath_get_string (FG_PROTO_ERROR, NULL, xpathCtx, NULL);

	if (message != NULL) {
		code = _xpath_get_int (FG_PROTO_ERROR, FG_PROTO_ATTR_ERROR_CODE,
				       xpathCtx, 0);
		domain = _xpath_get_string (FG_PROTO_ERROR, FG_PROTO_ATTR_ERROR_DOMAIN,
					    xpathCtx, NULL);
		g_set_error (error, FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_DAEMON_ERROR,
			     "%s(%d): %s", domain, code, message);
		g_free (domain);
		g_free (message);
	}
}

static void
_fg_smtp_info_parse_xml (FGSmtpInfo *fsi, const gchar *xmlbuff, gsize len, GError **error)
{
	FGSmtpInfoPrivate *priv;

	xmlDocPtr doc;
	xmlXPathContextPtr xpathCtx;
	GError *pe = NULL;

	g_return_if_fail (error == NULL || *error == NULL);	

	if (!xmlbuff) {
		g_set_error (error, FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_INVALID_XML,
			     "Invalid argument, XML is empty");
		return;
	}

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	doc = xmlReadMemory (xmlbuff, len, NULL, NULL, 0);
	if (!doc) {
		g_set_error (error,
			     FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_XML_READ_FAILED,
			     "Could not generate XML Document, maybe invalid XML");
		return;
	}

	xpathCtx = xmlXPathNewContext (doc);
	if (!xpathCtx) {
		g_set_error (error,
			     FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_XML_READ_FAILED,
			     "Could not get XPath context.");
		xmlFreeDoc(doc);
		return;
	}

	/* error check */
	_fg_smtp_info_xml_check_error (xpathCtx, &pe);
	if (pe != NULL) {
		g_propagate_error (error, pe);
		xmlXPathFreeContext (xpathCtx); 
		xmlFreeDoc (doc);
		return;
	}

	/* session start flag */
	priv->session = _xpath_get_bool (FG_PROTO_ROOT, FG_PROTO_ATTR_SESSION,
					 xpathCtx, FALSE);

	/* IP Address (required) */
	priv->addr = _xpath_get_string (FG_PROTO_ADDR, NULL, xpathCtx, NULL);
	if (! priv->addr) {
		g_set_error (error,
			     FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_XML_PARSE_FAILED,
			     "XML does not contain <addr> or contents.");
		xmlXPathFreeContext(xpathCtx); 
		xmlFreeDoc(doc);
		return;
	}

	/* mynetworks flag */
	priv->mynetworks = _xpath_get_bool (FG_PROTO_ADDR,
					    FG_PROTO_ATTR_MYNETWORKS,
					    xpathCtx, FALSE);

	/* PID (required) */
	priv->pid = _xpath_get_int (FG_PROTO_PID, NULL, xpathCtx, 0);
	if (priv->pid == 0) {
		g_set_error (error,
			     FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_XML_PARSE_FAILED,
			     "XML does not contain <pid> or contents.");
		xmlXPathFreeContext(xpathCtx); 
		xmlFreeDoc(doc);
		return;
	}

	/* mailfrom */
	priv->mailfrom = _xpath_get_string (FG_PROTO_MAILFROM, NULL,
					   xpathCtx, NULL);

	/* rcptto */
	priv->rcptto = _xpath_get_string (FG_PROTO_RCPTTO,
					  NULL, xpathCtx, NULL);

	priv->mydestination = _xpath_get_bool (FG_PROTO_RCPTTO,
					       FG_PROTO_ATTR_MYDESTINATION,
					       xpathCtx, FALSE);

	/* action @ wait */
	priv->wait = _xpath_get_int (FG_PROTO_ACTION, FG_PROTO_ATTR_WAIT,
				     xpathCtx, 0);

	/* action */
	priv->message = _xpath_get_string (FG_PROTO_ACTION, NULL,
					   xpathCtx, NULL);

	/* clean up */
	xmlXPathFreeContext(xpathCtx); 
	xmlFreeDoc(doc);

	return;
}

/* public */
FGSmtpInfo *
fg_smtp_info_new (gboolean session)
{
	FGSmtpInfo *obj;

	obj = FG_SMTP_INFO (g_object_new (FG_TYPE_SMTP_INFO, NULL));

	fg_smtp_info_set_session (obj, session);

	return obj;
}

FGSmtpInfo *
fg_smtp_info_new_from_xml (gboolean session, const gchar *xmlbuff, gsize len,
			   GError **error)
{
	FGSmtpInfo *fsi;
	GError *err = NULL;

	g_return_val_if_fail (error == NULL || *error == NULL, NULL);

	fsi = fg_smtp_info_new (session);

	if (!fsi) {
		g_set_error (error, FG_SMTP_INFO_ERROR,
			     FG_SMTP_INFO_ERROR_ALLOC_FAILED,
			     "Could not allocate memory for FGSmtpInfo");
		return NULL;
	}

	_fg_smtp_info_parse_xml (fsi, xmlbuff, len,  &err);

	if (err != NULL) {
		g_propagate_error (error, err);
		return NULL;
	} else
		return fsi;

}

/* malloced buffer */
gchar *
fg_smtp_info_get_xml (FGSmtpInfo *fsi, GError **error)
{
	gchar *xmlbuff;
	GError *tmperror = NULL;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), NULL);

	xmlbuff = _fg_smtp_info_gen_xml (fsi, &tmperror);

	if (tmperror != NULL) {
		g_propagate_error (error, tmperror);
		return NULL;
	}

	return xmlbuff;
}

void
fg_smtp_info_set_session (FGSmtpInfo *fsi, gboolean session)
{
	FGSmtpInfoPrivate *priv;

	g_return_if_fail (FG_IS_SMTP_INFO (fsi));

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	priv->session = session;
}


gint
fg_smtp_info_set_addr (FGSmtpInfo *fsi,
		       const gchar *addr, gboolean mynetworks)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), -1);
	g_return_val_if_fail (addr != NULL, -1);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	if (priv->addr) g_free (priv->addr);

	priv->addr = g_strdup (addr);
	priv->mynetworks = mynetworks;

	return 0;
}

void
fg_smtp_info_set_pid (FGSmtpInfo *fsi, gint pid)
{
	FGSmtpInfoPrivate *priv;

	g_return_if_fail (FG_IS_SMTP_INFO (fsi));

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	priv->pid = pid;
}

gint
fg_smtp_info_set_mailfrom (FGSmtpInfo *fsi, const gchar *mailfrom)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), -1);
	g_return_val_if_fail (mailfrom != NULL, -1);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	if (priv->mailfrom) g_free (priv->mailfrom);

	priv->mailfrom = g_strdup (mailfrom);

	return 0;

}

gint
fg_smtp_info_set_rcptto (FGSmtpInfo *fsi,
			 const gchar *rcptto, gboolean mydestination)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), -1);
	g_return_val_if_fail (rcptto != NULL, -1);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	if (priv->rcptto) g_free (priv->rcptto);

	priv->rcptto = g_strdup (rcptto);
	priv->mydestination = mydestination;

	return 0;
}

gint
fg_smtp_info_set_action (FGSmtpInfo *fsi, gchar *message, gint wait, gint code)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), -1);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	if (priv->message) g_free (priv->message);
	priv->message = NULL;

	if (message) priv->message = g_strdup (message);

	priv->wait = wait;
	priv->code = code;

	return 0;
}

gint
fg_smtp_info_set_action_message (FGSmtpInfo *fsi, gchar *message)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), -1);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	if (priv->message) g_free (priv->message);
	priv->message = NULL;

	if (message) priv->message = g_strdup (message);

	return 0;
}

void
fg_smtp_info_set_wait (FGSmtpInfo *fsi, gint wait)
{
	FGSmtpInfoPrivate *priv;

	g_return_if_fail (FG_IS_SMTP_INFO (fsi));

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	priv->wait = wait;
}

void
fg_smtp_info_free (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_if_fail (FG_IS_SMTP_INFO (fsi));

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	if (priv->addr) g_free (priv->addr);
	if (priv->mailfrom) g_free (priv->mailfrom);
	if (priv->rcptto) g_free (priv->rcptto);
	if (priv->message) g_free (priv->message);

	g_free (fsi);

	fsi = NULL;
}

/* get values */
gboolean
fg_smtp_info_is_session (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), FALSE);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->session;
}

gchar *
fg_smtp_info_get_addr (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), NULL);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->addr;
}

gint
fg_smtp_info_get_pid (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), -1);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->pid;
}

gchar *
fg_smtp_info_get_mailfrom (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), NULL);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->mailfrom ? priv->mailfrom : "";
}

gchar *
fg_smtp_info_get_rcptto (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), NULL);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->rcptto;
}

gboolean 
fg_smtp_info_is_mynetworks (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), FALSE);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->mynetworks;
}

gboolean
fg_smtp_info_is_mydestination (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), FALSE);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->mydestination;
}

gchar *
fg_smtp_info_get_action_message (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), NULL);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);
	if (priv->message && *(priv->message))
		return priv->message;
	else
		return NULL;
}

gint
fg_smtp_info_get_wait (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), 0);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->wait;
}

gint
fg_smtp_info_get_action_code (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_val_if_fail (FG_IS_SMTP_INFO (fsi), 0);

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	return priv->code;
}

void
fg_smtp_info_sleep_wait (FGSmtpInfo *fsi)
{
	FGSmtpInfoPrivate *priv;

	g_return_if_fail (FG_IS_SMTP_INFO (fsi));

	priv = FG_SMTP_INFO_GET_PRIVATE (fsi);

	sleep (priv->wait);
}

