/***************************************************************************
 * img_settings.cpp  -  Image Settings Handler
 *
 * Copyright (C) 2005 - 2008 Florian Richter
 ***************************************************************************/
/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.
   
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "../video/img_settings.h"
#include "../core/game_core.h"
#include "../video/gl_surface.h"
#include "../core/math/utilities.h"
#include "../core/math/size.h"

/* *** *** *** *** *** *** cImage_settings_data *** *** *** *** *** *** *** *** *** *** *** */

cImage_settings_data :: cImage_settings_data( void )
{
	m_base_settings = 0;

	m_int_x = 0;
	m_int_y = 0;
	m_col_rect.clear();
	m_width = 0;
	m_height = 0;
	m_rotation_x = 0;
	m_rotation_y = 0;
	m_rotation_z = 0;
	m_mipmap = 0;

	m_type = -1;
	m_ground_type = GROUND_NORMAL;
	m_obsolete = 0;
}

cImage_settings_data :: ~cImage_settings_data( void )
{

}

cSize_Float cImage_settings_data :: Get_Surface_Size( SDL_Surface *sdl_surface )
{
	if( !sdl_surface )
	{
		return cSize_Float();
	}

	// check if texture needs to get downscaled
	float new_w = static_cast<float>(Get_Power_of_2( sdl_surface->w ));
	float new_h = static_cast<float>(Get_Power_of_2( sdl_surface->h ));
	
	// todo : add check for maximum opengl texture size
	// if image settings dimension
	if( m_width > 0 && m_height > 0 )
	{
		/* downscale to fit best for the current resolution
		 * uses the first dimension which is not smaller as the default scale
		*/
		while( new_w * 0.5f > m_width * global_upscalex && new_h * 0.5f > m_height * global_upscaley )
		{
			new_w *= 0.5f;
			new_h *= 0.5f;
		}

		// if texture detail below low
		if( pVideo->texture_detail < 0.25f )
		{
			// half size only if texture size is high
			if( new_w * 0.8f > m_width * global_upscalex && new_h * 0.8f > m_height * global_upscaley )
			{
				new_w *= 0.5f;
				new_h *= 0.5f;
			}
		}
	}

	return cSize_Float( new_w, new_h);
}

void cImage_settings_data :: Apply( cGL_Surface *image )
{
	// empty image
	if( !image )
	{
		printf( "Error : surface for base %s does not exist\n", m_base.c_str() );
		return;
	}

	// ## int_x/int_y
	image->int_x = static_cast<float>(m_int_x);
	image->int_y = static_cast<float>(m_int_y);

	// ## width
	if( m_width > 0 )
	{
		image->start_w = static_cast<float>(m_width);
		image->w = image->start_w;
		image->col_w = image->w;
	}
	// ## height
	if( m_height > 0 )
	{
		image->start_h = static_cast<float>(m_height);
		image->h = image->start_h;
		image->col_h = image->h;
	}

	// ## collision rect
	if( m_col_rect.w > 0 && m_col_rect.h > 0 )
	{
		// position
		image->col_pos.x = m_col_rect.x;
		image->col_pos.y = m_col_rect.y;
		// dimension
		image->col_w = m_col_rect.w;
		image->col_h = m_col_rect.h;
	}

	// ## rotation x
	if( m_rotation_x != 0 )
	{
		image->base_rotx = static_cast<float>(m_rotation_x);

		//image->col_pos = image->col_pos.rotate3d( image->base_rotx, 1, 0, 0 );
		// mirror
		if( image->base_rotx == 180 )
		{
			image->col_pos.y = image->h - ( image->col_h + image->col_pos.y );
		}
	}
	// ## rotation y
	if( m_rotation_y != 0 )
	{
		image->base_roty = static_cast<float>(m_rotation_y);

		//image->col_pos = image->col_pos.rotate3d( image->base_roty, 0, 1, 0 );
		// mirror
		if( image->base_roty == 180 )
		{
			image->col_pos.x = image->w - ( image->col_w + image->col_pos.x );
		}
	}
	// ## rotation z
	if( m_rotation_z != 0 )
	{
		image->base_rotz = static_cast<float>(m_rotation_z);

		//image->col_pos = image->col_pos.rotate3d( image->base_rotz, 0, 0, 1 );

		if( image->base_rotz == 90 )
		{
			// rotate position
			GL_point pos( image->int_x, image->int_y );
			pos = pos.rotate( GL_point( image->w * 0.5f, image->h * 0.5f ), image->base_rotz );
			image->int_x = pos.y;
			image->int_y = pos.x - image->h;
			// rotate collision position
			float orig_x = image->col_pos.x;
			image->col_pos.x = image->h - ( image->col_h + image->col_pos.y );
			image->col_pos.y = orig_x;

			// switch width and height
			float orig_w = image->w;
			image->w = image->h;
			image->h = orig_w;
			// switch collision width and height
			float orig_col_w = image->col_w;
			image->col_w = image->col_h;
			image->col_h = orig_col_w;
		}
		// mirror
		else if( image->base_rotz == 180 )
		{
			image->col_pos.y = image->h - ( image->col_h + image->col_pos.y );
		}
		else if( image->base_rotz == 270 )
		{
			// rotate position
			GL_point pos( image->int_x, image->int_y );
			pos = pos.rotate( GL_point( image->w * 0.5f, image->h * 0.5f ), image->base_rotz );
			image->int_x = pos.y - image->w;
			image->int_y = pos.x;
			// rotate collision position
			float orig_x = image->col_pos.x;
			image->col_pos.x = image->col_pos.y;
			image->col_pos.y = orig_x;

			// switch width and height
			float orig_w = image->w;
			image->w = image->h;
			image->h = orig_w;
			// switch collision width and height
			float orig_col_w = image->col_w;
			image->col_w = image->col_h;
			image->col_h = orig_col_w;
		}
	}

	// ## editor_tags
	if( !m_editor_tags.empty() )
	{
		image->editor_tags = m_editor_tags;
	}

	// ## name
	if( !m_name.empty() )
	{
		image->name = m_name;

		// finds all "_" and replaces them with " "
		for( string::iterator itr = image->name.begin(); itr != image->name.end(); ++itr )
		{
			// change
			if( *itr == '_' )
			{
				*itr = ' ';
			}
		}
	}

	// ## type
	if( m_type > 0 )
	{
		image->type = m_type;
	}

	// ## ground type
	image->ground_type = m_ground_type;

	// ## obsolete
	image->obsolete = m_obsolete;
}

void cImage_settings_data :: Apply_Base( cImage_settings_data *base_settings_data )
{
	if( !base_settings_data->m_base.empty() )
	{
		m_base = base_settings_data->m_base;
		m_base_settings = base_settings_data->m_base_settings;
	}

	m_int_x = base_settings_data->m_int_x;
	m_int_y = base_settings_data->m_int_y;
	m_col_rect = base_settings_data->m_col_rect;
	m_width = base_settings_data->m_width;
	m_height = base_settings_data->m_height;
	m_rotation_x = base_settings_data->m_rotation_x;
	m_rotation_y = base_settings_data->m_rotation_y;
	m_rotation_z = base_settings_data->m_rotation_z;
	m_mipmap = base_settings_data->m_mipmap;
	m_editor_tags = base_settings_data->m_editor_tags;
	m_name = base_settings_data->m_name;
	m_type = base_settings_data->m_type;
	m_author = base_settings_data->m_author;

	// only set if this isn't obsolete
	if( !m_obsolete && base_settings_data->m_obsolete )
	{
		m_obsolete = 1;
	}
}

/* *** *** *** *** *** *** cImage_settings *** *** *** *** *** *** *** *** *** *** *** */

cImage_settings :: cImage_settings( void )
: cFile_parser()
{
	settings = NULL;
	load_base = 1;
}

cImage_settings :: ~cImage_settings( void )
{
	//
}

cImage_settings_data *cImage_settings :: Get( string &filename, bool load_base_settings /* = 1 */ )
{
	load_base = load_base_settings;
	settings = new cImage_settings_data();

	Parse( filename );
	return settings;
}

bool cImage_settings :: HandleMessage( string *parts, unsigned int count, unsigned int line )
{
	if( parts[0].compare( "base" ) == 0 )
	{
		if( count < 2 || count > 3 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2-3 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[2] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		settings->m_base = data_file.substr( 0, data_file.rfind( "/" ) + 1 ) + parts[1];

		// with settings option
		if( count == 3 && string_to_int( parts[2] ) )
		{
			settings->m_base_settings = 1;

			if( load_base )
			{
				string settings_file = settings->m_base;

				// if settings file exists
				while( !settings_file.empty() )
				{
					// if not already image settings based
					if( settings_file.rfind( ".settings" ) == string::npos )
					{
						settings_file.erase( settings_file.rfind( "." ) + 1 );
						settings_file.insert( settings_file.rfind( "." ) + 1, "settings" );
					}

					// not found
					if( !File_Exists( settings_file ) )
					{
						break;
					}

					// create new temporary parser
					cImage_settings *temp_parser = new cImage_settings();
					cImage_settings_data *base_settings = temp_parser->Get( settings_file );
					// finished loading base settings
					delete temp_parser;
					settings_file.clear();

					// handle
					if( base_settings )
					{
						// todo : apply settings in reverse order ( deepest settings should override first )
						settings->Apply_Base( base_settings );
						
						// if also based on settings
						if( !base_settings->m_base.empty() && base_settings->m_base_settings )
						{
							settings_file = base_settings->m_base;
						}
						
						delete base_settings;
					}
				}
			}
		}
	}
	else if( parts[0].compare( "int_x" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[1] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		settings->m_int_x = string_to_int( parts[1] );
	}
	else if( parts[0].compare( "int_y" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[1] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		settings->m_int_y = string_to_int( parts[1] );
	}
	else if( parts[0].compare( "col_rect" ) == 0 )
	{
		if( count != 5 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 5 parameters" );
			return 0; // error
		}

		for( unsigned int i = 1; i < 5; i++ )
		{
			if( !is_valid_number( parts[i] ) )
			{
				printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
				printf( "%s is not a valid integer value\n", parts[1].c_str() );
				return 0; // error
			}
		}

		// position and dimension
		settings->m_col_rect = GL_rect( static_cast<float>(string_to_int( parts[1] )), static_cast<float>(string_to_int( parts[2] )), static_cast<float>(string_to_int( parts[3] )), static_cast<float>(string_to_int( parts[4] )) );
	}
	else if( parts[0].compare( "width" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[1] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		settings->m_width = string_to_int( parts[1] );
	}
	else if( parts[0].compare( "height" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[1] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		settings->m_height = string_to_int( parts[1] );
	}
	else if( parts[0].compare( "rotation" ) == 0 )
	{
		if( count < 2 || count > 5 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2-5 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[1] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		// x
		settings->m_rotation_x = string_to_int( parts[1] );

		// y
		if( count > 2 )
		{
			if( !is_valid_number( parts[2] ) )
			{
				printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
				printf( "%s is not a valid integer value\n", parts[2].c_str() );
				return 0; // error
			}

			settings->m_rotation_y = string_to_int( parts[2] );
		}
		// z
		if( count > 3 )
		{
			if( !is_valid_number( parts[3] ) )
			{
				printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
				printf( "%s is not a valid integer value\n", parts[3].c_str() );
				return 0; // error
			}

			settings->m_rotation_z = string_to_int( parts[3] );
		}
	}
	else if( parts[0].compare( "mipmap" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[1] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		// if mipmaps enabled
		if( string_to_int( parts[1] ) )
		{
			settings->m_mipmap = 1;
		}
	}
	else if( parts[0].compare( "editor_tags" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		settings->m_editor_tags = parts[1];
	}
	else if( parts[0].compare( "name" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		settings->m_name = parts[1];
	}
	else if( parts[0].compare( "type" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		settings->m_type = Get_Sprite_Type_Id( parts[1] );
	}
	else if( parts[0].compare( "ground_type" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		settings->m_ground_type = Get_Ground_Type_Id( parts[1] );
	}
	else if( parts[0].compare( "author" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		settings->m_author = parts[1];
	}
	else if( parts[0].compare( "obsolete" ) == 0 )
	{
		if( count != 2 )
		{
			printf( "%s : line %d Error :\n", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "Error : %s %s\n", parts[0].c_str(), "needs 2 parameters" );
			return 0; // error
		}

		if( !is_valid_number( parts[1] ) )
		{
			printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
			printf( "%s is not a valid integer value\n", parts[1].c_str() );
			return 0; // error
		}

		// if tagged obsolete
		if( string_to_int( parts[1] ) )
		{
			settings->m_obsolete = 1;
		}
	}
	else
	{
		printf( "%s : line %d Error : ", Get_Filename( data_file, 0, 0 ).c_str(), line );
		printf( "Unknown Command : %s\n", parts[0].c_str() );
		return 0; // error
	}

	return 1;
}

/* *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** */

cImage_settings *pSettingsParser = NULL;
