/***************************************************************************
    smb4kbrowserwidget.h  -  Network browse widget class for Smb4K.
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KBROWSERWIDGET_H
#define SMB4KBROWSERWIDGET_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qstringlist.h>

// KDE includes
#include <klistview.h>
#include <kconfig.h>
#include <kactioncollection.h>
#include <kaction.h>

// application specific includes
#include "../core/smb4knetworkitems.h"
#include "smb4kbrowserwidgetitem.h"


/**
 * This is the browser widget class of Smb4K.
 */
class Smb4KBrowserWidgetItem;
class Smb4KBrowserWidget : public KListView
{
  Q_OBJECT

  friend class Smb4KBrowserWidgetItem;

  public:
    /**
     * The constructor.
     */
    Smb4KBrowserWidget( QWidget *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KBrowserWidget();

    /**
      * This function is called, when the user tries to open an item in the list view.
      * Takes the selected item and a bool as arguments. For more information
      * look at the class documentation of KListView.
      */
    void setOpen( QListViewItem *item, bool on );
    /**
      * This function reads the options for the browser widget.
      */
    void readOptions();
    /**
     * This function returns a pointer to the action collection of this class.
     * You should use it to include the action into the action collection of the
     * main action collection (e.g. KMainWindow::actionCollection()).
     *
     * The entries in this action collection are members of the group "BrowserWidget".
     *
     * @returns       a pointer to the action collection of this class
     */
    KActionCollection *action_collection() { return m_collection; }

  signals:
    /**
     * This signal is emitted if the user wants to search for a share.
     */
    void searchRequest();
    /**
     * This signal emits the item that is currently highlighted.
     *
     * @param item    the item that's currently highlighted.
     */
    void highlightedItem( QListViewItem *item );
    /**
     * This signal is emitted when the browser widget has been cleared.
     */
    void cleared();

  protected slots:
    /**
     * Is called, when a list view item is to be mounted.
     */
    void slotItemExecuted( QListViewItem *item );
    /**
     * Is called whenever a QListViewItem is expanded or collapsed.
     */
    void slotItemExpandedCollapsed( QListViewItem *item );
    /**
     * Is called, when the selection in the list view is changed. (Is
     * connected to the selectionChanged() signal.
     */
    void slotSelectionChanged( QListViewItem *item );
    /**
     * This slot is invoked by the rightButtonPressed() signal.
     */
    void slotRightButtonPressed( QListViewItem *item, const QPoint &point, int col );
    /**
     * Is called, if the user clicked somewhere in the list view.
     */
    void slotItemClicked( QListViewItem *item );
    /**
     * Takes the workgroups/masters list and puts it into the list
     * view.
     */
    void slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> &list );
    /**
     * Takes a list of the members of a workgroup and inserts them into the
     * list view as children of the workgroup item.
     */
    void slotMembers( const QValueList<Smb4KHostItem *> &list );
    /**
     * Takes the shares of a queried host and inserts them as children of
     * the host item.
     */
    void slotShares( const QValueList<Smb4KShareItem *> &list );
    /**
     * Mounts a share.
     */
    void slotMountShare();
    /**
     * Opens the askpass dialod.
     */
    void slotAskPass();
    /**
     * Opens the preview dialog.
     */
    void slotPreview();
    /**
     * Adds a bookmark to the list of bookmarks.
     */
    void slotAddBookmark();
    /**
     * This slot initializes a rescan of the network neighborhood.
     */
    void slotRescan();
    /**
     * Takes a reference to a Smb4KHostItem, extracts the IP address of the
     * host and adds it to the appropriate item in the KListView.
     *
     * @param item      Smb4KHostItem which carries the IP address
     */
    void slotAddIPAddress( Smb4KHostItem *item );
    /**
     * Opens the print dialog for a printer share.
     */
    void slotPrint();
    /**
     * This slot marks the mounted shares. It is called by the
     * Smb4KMounter::updated() signal.
     */
    void slotMarkShares();
    /**
     * This slot is called when the user want to search for a share.
     */
    void slotSearch();
    /**
     * This slot inserts a host item into the browser widget.
     *
     * @param item      the item that should be inserted into the browser widget
     */
    void slotInsertItem( Smb4KHostItem *item );
    /**
     * This slot aborts all actions of the core.
     */
    void slotAbort();

  private:
    /**
     * This value is TRUE, if the hidden shares are to be shown, otherwise
     * it is FALSE.
     */
    bool m_hidden;
    /**
     * This value is TRUE, if the IPC$ shares are to be shown, otherwise
     * it is FALSE.
     */
    bool m_ipc;
    /**
     * This value is TRUE, if the ADMIN$ shares are to be shown, otherwise
     * it is false.
     */
    bool m_admin;
    /**
     * This value is TRUE, if the printer shares are to be shown, otherwise
     * it is false.
     */
    bool m_printer;
    /**
     * This function shows a dialog, that asks for a login name, if the user
     * wants to log into a 'homes' share.
     */
    const QString specifyUser( const QString &host );
    /**
     * Enumeration for the columns in the list view.
     */
    enum Columns{ Network = 0, Type = 1, IP = 2, Comment = 3 };
    /**
     * This is the action collection of this class.
     */
    KActionCollection *m_collection;
    /**
     * This function initializes the actions that are provided by this class.
     */
    void initActions();
    /**
     * This function inserts a host item into the browser window.
     */
    void insertItem( Smb4KHostItem *item );
    /**
     * The preview action.
     */
    KAction *m_preview_action;
    /**
     * The askpass action.
     */
    KAction *m_askpass_action;
    /**
     * The print action.
     */
    KAction *m_print_action;
    /**
     * The mount action.
     */
    KAction *m_mount_action;
    /**
     * The bookmark action.
     */
    KAction *m_bookmark_action;
    /**
     * The rescan action.
     */
    KAction *m_rescan_action;
    /**
     * The abort action.
     */
    KAction *m_abort_action;
    /**
     * The search action.
     */
    KAction *m_search_action;
};

#endif
