/***************************************************************************
    smb4kscanner.cpp  -  The network scan core class of Smb4K.
                             -------------------
    begin                : Sam Mai 31 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qapplication.h>
#include <qmap.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kmessagebox.h>

// Application specific includes.
#include "smb4kscanner.h"
#include "smb4kauthinfo.h"
#include "smb4kglobal.h"

using namespace Smb4K_Global;


Smb4KScanner::Smb4KScanner( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_proc = new KProcess( this, "ScannerProcess" );
  m_proc->setUseShell( true );

  m_password_handler = new Smb4KPasswordHandler( this, "ScannerPasswordHandler" );

  m_state = Idle;
  m_workgroup = QString::null;
  m_host = QString::null;
  m_share = QString::null;
  m_path = QString::null;

  m_timer = new QTimer( this );
  m_timer->start( 50, false );

  m_working = false;

  m_queue.setAutoDelete( true );

  QString *input =  new QString( QString( "%1:" ).arg( Init ) );
  m_queue.enqueue( input );

  connect( m_proc,  SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this,    SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc,  SIGNAL( processExited( KProcess* ) ),
           this,    SLOT( slotProcessExited( KProcess * ) ) );
  connect( m_proc,  SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,    SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_timer, SIGNAL( timeout() ),
           this,    SLOT( start() ) );
}


Smb4KScanner::~Smb4KScanner()
{
  // Do not use the abort() function here, because this
  // could cause crashes due to the emitted killed()
  // signal.
  if ( m_proc->isRunning() )
  {
    m_proc->kill();
  }

  m_workgroup_list.clear();
}



/****************************************************************************
   This function initializes any action, that's done by the scanner.
****************************************************************************/

void Smb4KScanner::init()
{
  config()->setGroup( "Browse Options" );
  bool auth_for_browse_list = config()->readBoolEntry( "Auth For Browse List", false );
  QString method = config()->readEntry( "Browse List", "nmblookup" );
  QString host_to_query = config()->readEntry( "Query Host", QString::null );

  QString wins = getWINSServer();
  QString nmblookup_options = getNmblookupOptions();
  QString smbclient_options = getSmbclientOptions();

  QString command;

  // Use nmblookup or query host to obtain the browse list?
  if ( QString::compare( method, "nmblookup" ) == 0 )
  {
    command = QString( "nmblookup -M" );

    if ( !nmblookup_options.stripWhiteSpace().isEmpty() )
    {
      command.append( nmblookup_options );
    }

    command.append( " -- - | grep '<01>' | awk '{print $1}'" );

    // Query the WINS server, if available
    if ( wins.isEmpty() )
    {
      command.append( " | xargs nmblookup -A" );
    }
    else
    {
      command.append( QString( " | xargs nmblookup -R -U %1 -A" ).arg( wins ) );
    }

    *m_proc << command;

    startProcess( Groups );
  }
  else if ( QString::compare( method, "host" ) == 0 )
  {
    command = QString( "smbclient -d1" );

    if ( !smbclient_options.stripWhiteSpace().isEmpty() )
    {
      command.append( smbclient_options );
    }

    if ( auth_for_browse_list )
    {
      Smb4KAuthInfo *auth = m_password_handler->readAuth( QString::null, host_to_query, QString::null );

      if ( !auth->user().isEmpty() )
      {
        command.append( QString( " -U %1" ).arg( KProcess::quote( auth->user() ) ) );

        if ( !auth->password().isEmpty() )
        {
          m_proc->setEnvironment( "PASSWD", auth->password() );
        }
      }
      else
      {
        command.append( " -U %" );
      }

      delete auth;
    }
    else
    {
      command.append( " -U %" );
    }

    command.append( QString( " -L %1" ).arg( KProcess::quote( host_to_query ) ) );

    *m_proc << command;

    startProcess( QueryHost );
  }
}


/****************************************************************************
   This function rescans the network.
****************************************************************************/

void Smb4KScanner::rescan()
{
  m_queue.enqueue( new QString( QString( "%1:" ).arg( Init ) ) );
}


/****************************************************************************
   Aborts any process that is running.
****************************************************************************/

void Smb4KScanner::abort()
{
  if ( m_proc->isRunning() )
  {
    m_proc->kill();
    m_queue.clear();
    emit killed();
  }
}


/****************************************************************************
   Gets the preview of a share. (public part)
****************************************************************************/

void Smb4KScanner::getPreview( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path )
{
  m_queue.enqueue( new QString( QString( "%1:%2:%3:%4:%5:%6" ).arg( Preview ).arg( workgroup, host, ip ).arg( share, path ) ) );
}


/****************************************************************************
   Gets the preview of a share. (private part)
****************************************************************************/

void Smb4KScanner::preview( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path )
{
  QString smbclient_options = getSmbclientOptions();

  m_workgroup = workgroup;
  m_host = host;
  m_share = share;
  m_path = path;
  m_ip = ip;

  QString command;
  Smb4KAuthInfo *auth = m_password_handler->readAuth( workgroup, host, share );

  // Assemble command:
  command = QString( "smbclient //%1/%2 -d1 -W %3 -c 'ls" ).arg( KProcess::quote( host ) ).arg( KProcess::quote( share ) ).arg( KProcess::quote( workgroup ) );

  if ( !path.isEmpty() )
  {
    // To be able to display directories, that contain umlauts, we have to do
    // two things:
    // (1) replace all slashes by backslashes;
    // (2) convert the directory name to local 8 bit.
    QString proc_path = path;
    proc_path = proc_path.replace( QChar( '/' ), QChar( '\\' ) ).local8Bit();
    command.append( " \"" ).append( proc_path ).append( "*\"" );
  }

  command.append( "'" );

  if ( !ip.isEmpty() )
  {
    command.append( QString( " -I %1" ).arg( KProcess::quote( ip ) ) );
  }

  if ( !smbclient_options.stripWhiteSpace().isEmpty() )
  {
    command.append( smbclient_options );
  }

  if ( !auth->user().isEmpty() )
  {
    command.append( QString( " -U %1" ).arg( KProcess::quote( auth->user() ) ) );

    if ( !auth->password().isEmpty() )
    {
      m_proc->setEnvironment( "PASSWD", auth->password() );
    }
  }
  else
  {
    command.append( " -U guest%" );
  }

  delete auth;

  m_proc->operator<<( command );
  startProcess( Preview );
}


/****************************************************************************
   Scans for workgroup members. (public part)
****************************************************************************/

void Smb4KScanner::getWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip )
{
  m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Hosts ).arg( workgroup, master, ip ) ) );
}


/****************************************************************************
   Scans for workgroup members. (private part)
****************************************************************************/

void Smb4KScanner::scanForWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip )
{
  config()->setGroup( "Browse Options" );
  bool master_auth = config()->readBoolEntry( "Master Browser Auth", false );

  QString smbclient_options = getSmbclientOptions();

  m_workgroup = workgroup;
  m_host = master;
  m_ip = ip;

  // Scan for the host list.
  QString command = QString( "smbclient -d1" );

  if ( !smbclient_options.stripWhiteSpace().isEmpty() )
  {
    command.append( smbclient_options );
  }

  if ( master_auth )
  {
    Smb4KAuthInfo *auth = m_password_handler->readAuth( workgroup, master, QString::null );

    if ( !auth->user().isEmpty() )
    {
      command.append( QString( " -U %1" ).arg( KProcess::quote( auth->user() ) ) );

      if ( !auth->password().isEmpty() )
      {
        m_proc->setEnvironment( "PASSWD", auth->password() );
      }
    }
    else
    {
      command.append( " -U %" );
    }

    delete auth;
  }
  else
  {
    command.append( " -U %" );
  }

  if ( !ip.isEmpty() )
  {
    command.append( QString( " -I %1" ).arg( ip ) );
  }

  command.append( QString( " -W %1 -L %2" ).arg( KProcess::quote( workgroup ) ).arg( KProcess::quote( master ) ) );

  m_proc->operator<<( command );
  startProcess( Hosts );
}


/****************************************************************************
   Scans for shares on a selected host. (public part)
****************************************************************************/

void Smb4KScanner::getShares( const QString &workgroup, const QString &host, const QString &ip )
{
  m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Shares ).arg( workgroup, host, ip ) ) );
}


/****************************************************************************
   Scans for shares on a selected host. (private part)
****************************************************************************/

void Smb4KScanner::scanForShares( const QString &workgroup, const QString &host, const QString &ip )
{
  m_workgroup = workgroup;
  m_host = host;
  m_ip = ip;

  Smb4KAuthInfo *auth = m_password_handler->readAuth( workgroup, host, QString::null );

  QString command;

  config()->setGroup( "Programs" );

  if ( !config()->readEntry( "net", QString::null ).isEmpty() )
  {
    command.append( QString( "net %1 -w %2 -S %3" ).arg( getNetOptions( "share" ), KProcess::quote( workgroup ), KProcess::quote( host ) ) );

    if ( !auth->user().isEmpty() )
    {
      command.append( QString( " -U %1" ).arg( KProcess::quote( auth->user() ) ) );

      if ( !auth->password().isEmpty() )
      {
        m_proc->setEnvironment( "PASSWD", auth->password() );
      }
    }
    else
    {
      command.append( " -U guest" );
      m_proc->setEnvironment( "PASSWD", QString::null );
    }
  }
  else
  {
    QString smbclient_options = getSmbclientOptions();

    // Assemble the scan command.
    command.append( QString( "smbclient -d1 -W %1" ).arg( KProcess::quote( workgroup ) ) );

    command.append( " -L " ).append( KProcess::quote( host ) );

    if ( !smbclient_options.stripWhiteSpace().isEmpty() )
    {
      command.append( smbclient_options );
    }

    if ( !auth->user().isEmpty() )
    {
      command.append( QString( " -U %1" ).arg( KProcess::quote( auth->user() ) ) );

      if ( !auth->password().isEmpty() )
      {
        m_proc->setEnvironment( "PASSWD", auth->password() );
      }
    }
    else
    {
      command.append( " -U guest%" );
    }
  }

  if ( !ip.isEmpty() )
  {
    command.append( QString( " -I %1" ).arg( KProcess::quote( ip ) ) );
  }

  delete auth;

  *m_proc << command;

  startProcess( Shares );
}


/****************************************************************************
   Gets more info on a selected host. (public part)
****************************************************************************/

void Smb4KScanner::getInfo( const QString &workgroup, const QString &host, const QString &ip )
{
  m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Info ).arg( workgroup, host, ip ) ) );
}


/****************************************************************************
   Gets more info on a selected host. (private part)
****************************************************************************/

void Smb4KScanner::scanForInfo( const QString &workgroup, const QString &host, const QString &ip )
{
  QString smbclient_options = getSmbclientOptions();

  m_workgroup = workgroup;
  m_host = host;

  QString command = QString( "smbclient -d1 -U % -W %1 -L %2" ).arg( KProcess::quote( workgroup ) ).arg( KProcess::quote( host ) );

  if ( !ip.isEmpty() )
  {
    command.append( QString( " -I %1" ).arg( KProcess::quote( ip ) ) );
  }

  if ( !smbclient_options.stripWhiteSpace().isEmpty() )
  {
    command.append( smbclient_options );
  }

  m_proc->operator<<( command );

  startProcess( Info );
}


/****************************************************************************
   Searches for a host. (public part)
****************************************************************************/

void Smb4KScanner::makeSearch( const QString &host )
{
  m_queue.enqueue( new QString( QString( "%1:%2" ).arg( Search ).arg( host ) ) );
}


/****************************************************************************
   Searches for a host. (private part)
****************************************************************************/

void Smb4KScanner::searchForHost( const QString &host )
{
  config()->setGroup( "Browse Options" );
  QString search_method = config()->readEntry( "Network Search", "nmblookup" );

  if ( QString::compare( search_method, "smbclient" ) == 0 && host.stripWhiteSpace().contains( "." ) == 3 )
  {
    emit error( ERROR_IP_CANNOT_BE_USED, QString::null );
    m_working = false;
    emit running( SCANNER_STOP, m_working );
    return;
  }

  QString wins = getWINSServer();
  QString nmblookup_options = getNmblookupOptions();
  QString smbclient_options = getSmbclientOptions();

  m_host = host;

  QString command;

  if ( QString::compare( search_method, "nmblookup" ) == 0 )
  {
    command = QString( "nmblookup" );

    if ( !nmblookup_options.stripWhiteSpace().isEmpty() )
    {
      command.append( nmblookup_options );
    }

    if ( host.contains( '.', true ) != 3 )
    {
      if ( !wins.isEmpty() )
      {
        command.append( QString( " -R -U %1 %2 -S | grep '<00>' | sed -e 's/<00>.*//'" ).arg( wins ).arg( m_host ) );
      }
      else
      {
        command.append( QString( " %1 -S | grep '<00>' | sed -e 's/<00>.*//'" ).arg( m_host ) );
      }
    }
    else
    {
      if ( !wins.isEmpty() )
      {
        command.append( QString( " -R -U %1 %2 -A | grep '<00>' | sed -e 's/<00>.*//'" ).arg( wins ).arg( m_host ) );
      }
      else
      {
        command.append( QString( " %1 -A | grep '<00>' | sed -e 's/<00>.*//'" ).arg( m_host ) );
      }
    }
  }
  else
  {
    command = QString( "smbclient -d2 -U % -L %1" ).arg( m_host );

    if ( !smbclient_options.stripWhiteSpace().isEmpty() )
    {
      command.append( smbclient_options );
    }
  }

  m_proc->operator<<( command );

  startProcess( Search );
}


/****************************************************************************
   Starts the scanner.
****************************************************************************/

void Smb4KScanner::startProcess( int state )
{
  m_state = state;
  m_buffer = QString::null;
  QApplication::setOverrideCursor( waitCursor );
  m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/****************************************************************************
   Tell the program what it has to do.
****************************************************************************/

void Smb4KScanner::endProcess()
{
  switch ( m_state )
  {
    case Groups:
      processWorkgroups();
      break;
    case QueryHost:
      processQueryHost();
      break;
    case Hosts:
      processHosts();
      break;
    case Shares:
      processShares();
      break;
    case Info:
      processInfo();
      break;
    case Preview:
      processPreview();
      break;
    case Search:
      processSearch();
      break;
    default:
      break;
  }
  m_state = Idle;
  m_workgroup = QString::null;
  m_host = QString::null;
  m_ip = QString::null;
  m_share = QString::null;
  m_path = QString::null;
  QApplication::restoreOverrideCursor();
  m_proc->clearArguments();

  m_working = false;
  emit running( SCANNER_STOP, m_working );
}


/****************************************************************************
   Process the list of workgroups.
****************************************************************************/

void Smb4KScanner::processWorkgroups()
{
  QStringList list = QStringList::split( '\n', m_buffer, false );

  m_workgroup_list.clear();

  QString group, master, ip;

  for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
  {
    if ( (*it).stripWhiteSpace().startsWith( "Looking" ) )
    {
      ip = (*it).stripWhiteSpace().section( " ", -1, -1 );
      continue;
    }
    else if ( (*it).contains( "<00>" ) != 0 && (*it).contains( "<GROUP>" ) == 0 )
    {
      if ( group.isEmpty() && master.isEmpty() && !ip.isEmpty() )
        master = (*it).section( "<00>", 0, 0 ).stripWhiteSpace();

      continue;
    }
    else if ( (*it).contains( "<00>" ) != 0 && (*it).contains( "<GROUP>" ) != 0 )
    {
      if ( group.isEmpty() && !master.isEmpty() && !ip.isEmpty() )
      {
        group = (*it).left( (*it).find( "<00>" ) ).stripWhiteSpace();

        m_workgroup_list.append( new Smb4KWorkgroupItem( group, master, ip ) );

        group = QString::null;
        master = QString::null;
        ip = QString::null;
      }

      continue;
    }
  }

  emit workgroups( m_workgroup_list );
}


void Smb4KScanner::processQueryHost()
{
  config()->setGroup( "Browse Options" );
  QString host_to_query = config()->readEntry( "Query Host", QString::null );

  QStringList list = QStringList::split( '\n', m_buffer, false );

  m_workgroup_list.clear();

  if ( list.grep( "tree connect failed: NT_STATUS_ACCESS_DENIED" ).count() == 0 &&
       list.grep( "NT_STATUS_LOGON_FAILURE" ).count() == 0 &&
       list.grep( "Connection to "+host_to_query+" failed" ).count() == 0 )
  {
    // Get the line starting with Workgroup.
    uint index( 0 );

    while( index < list.count() )
    {
      if ( list[index++].stripWhiteSpace().startsWith( "Workgroup" ) )
        break;
      else
        continue;
    }

    index++;

    while ( index < list.count() )
    {
      QString line = list[index].stripWhiteSpace();
      QString master, workgroup;
      master = line.section( "   ", -1, -1 ).stripWhiteSpace();
      workgroup = line.section( "   ", 0, -2 ).stripWhiteSpace();

      if ( workgroup.isEmpty() )
      {
        if ( master.isEmpty() )
          break;
        else
          continue;
      }
      else
        m_workgroup_list.append( new Smb4KWorkgroupItem( workgroup, master ) );

      index++;
    }

    emit workgroups( m_workgroup_list );
  }
  else
  {
    emit workgroups( m_workgroup_list );

    if ( list.grep( "tree connect failed: NT_STATUS_ACCESS_DENIED" ).count() != 0 )
    {
      // We should find an entry for the workgroup here:
      QString workgroup = list.grep( "Domain" ).first().section( "Domain=[", 1, 1 ).section( "]", 0, 0 ).stripWhiteSpace();

      if ( !workgroup.isEmpty() )
      {
        if ( m_password_handler->askpass( workgroup, host_to_query, QString::null, Smb4KPasswordHandler::AccessDenied ) )
        {
          QString *input = new QString( QString( "%1:" ).arg( Init ) );
          m_queue.enqueue( input );
        }
      }
      else
      {
        emit error( ERROR_GETTING_BROWSELIST, list.grep( "NT_STATUS" ).first() );
      }
    }
    else if ( list.grep( "NT_STATUS_LOGON_FAILURE" ).count() != 0 )
    {
      emit error( ERROR_GETTING_BROWSELIST, m_buffer );
    }
    else if ( list.grep( "Connection to "+host_to_query+" failed" ).count() != 0 )
    {
      emit error( ERROR_GETTING_BROWSELIST, m_buffer );
    }
  }
}


/****************************************************************************
   Process the memeber list of a workgroup.
****************************************************************************/

void Smb4KScanner::processHosts()
{
  QStringList list = QStringList::split( '\n', m_buffer, false );

  if ( list.grep( "Connection to" ).count() != 0 )
  {
    QString errmsg = list.grep( "Connection to" ).first().stripWhiteSpace();
    emit error( ERROR_GETTING_MEMBERS, errmsg );
  }
  else if ( m_buffer.contains( "tree connect failed: NT_STATUS_ACCESS_DENIED" ) == 0 &&
            m_buffer.contains( "NT_STATUS_LOGON_FAILURE" ) != 0  )
  {
    if ( m_password_handler->askpass( m_workgroup, m_host, QString::null, Smb4KPasswordHandler::LogonFailure ) )
    {
      QString *input = new QString( QString( "%1:%2:%3" ).arg( Hosts ).arg( m_workgroup ).arg( m_host ) );
      m_queue.enqueue( input );
    }
  }
  else
  {
    // Get the begin of the host listing.
    uint index( 0 );
    for ( ; index < list.count(); index++ )
      if ( list[index].stripWhiteSpace().startsWith( "Server" ) )
        break;

    index += 2;

    // FIXME: Rewrite this section and implement the m_host_list list.

    // Process list of hosts:
    QValueList<Smb4KHostItem *> hosts;

    QString line;

    while ( index < list.count() )
    {
      line = list[index++].stripWhiteSpace();

      if ( line.startsWith( "Workgroup" ) )
      {
        break;
      }
      else if ( line.contains( "  " ) == 0 )
      {
        hosts.append( new Smb4KHostItem( m_workgroup, line ) );
      }
      else
      {
        hosts.append( new Smb4KHostItem( m_workgroup, line.section( "  ", 0, 0 ).stripWhiteSpace(), line.section( "  ", 1, -1 ).stripWhiteSpace() ) );
      }
    }

    // If the list is empty, put the master in.
    if ( hosts.isEmpty() )
    {
      hosts.append( new Smb4KHostItem( m_workgroup, m_host ) );
    }

    emit members( hosts );
  }
}


/****************************************************************************
   Process the share list of a host.
****************************************************************************/

void Smb4KScanner::processShares()
{
  // Error handling
  if ( m_buffer.contains( "NT_STATUS", true ) != 0 ||
       m_buffer.contains( "Connection to", true ) != 0 ||
       m_buffer.contains( "tree connect failed:", true ) != 0 ||
       m_buffer.contains( "The username or password was not correct.", true ) != 0 )
  {
    if ( m_buffer.contains( "NT_STATUS", true ) != 0 )
    {
      if ( m_buffer.contains( "NT_STATUS_ACCESS_DENIED", true ) != 0 )
      {
        if ( m_password_handler->askpass( m_workgroup, m_host, QString::null, Smb4KPasswordHandler::AccessDenied ) )
        {
          m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Shares ).arg( m_workgroup, m_host, m_ip ) ) );
        }
      }
      else if ( m_buffer.contains( "NT_STATUS_LOGON_FAILURE", true ) != 0 )
      {
        if ( m_password_handler->askpass( m_workgroup, m_host, QString::null, Smb4KPasswordHandler::LogonFailure ) )
        {
          m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Shares ).arg( m_workgroup, m_host, m_ip ) ) );
        }
      }
      else
      {
        emit error( ERROR_GETTING_SHARES, m_buffer );
      }
    }
    else if ( m_buffer.contains( "tree connect failed:", true ) != 0 ||
              m_buffer.contains( "The username or password was not correct.", true ) != 0 )
    {
      if ( m_password_handler->askpass( m_workgroup, m_host, QString::null, Smb4KPasswordHandler::AccessDenied ) )
      {
        m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Shares ).arg( m_workgroup, m_host, m_ip ) ) );
      }
    }
    else
    {
      emit error( ERROR_GETTING_SHARES, m_buffer );
    }

    return;
  }
  else if ( (m_buffer.contains( "Error returning browse list:", true ) != 0 &&
             m_buffer.contains( "NT_STATUS", true ) == 0) ||
            (m_buffer.contains( "Could not connect to server", true ) != 0 &&
             m_buffer.contains( "The username or password was not correct.", true ) == 0) ||
             m_buffer.contains( "Unable to find a suitable server" ) != 0 )
  {
    emit error( ERROR_GETTING_SHARES, m_buffer );

    return;
  }

  QStringList list = QStringList::split( '\n', m_buffer, false );

  QValueList<Smb4KShareItem *> share_list;

  config()->setGroup( "Programs" );

  if ( !config()->readEntry( "net", QString::null ).isEmpty() )
  {
    bool process = false;

    for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( (*it).startsWith( "---" ) )
      {
        process = true;
      }

      if ( process )
      {
        QString name, type, comment;

        if ( (*it).contains( " Disk     ", true ) != 0 )
        {
          name = (*it).section( " Disk     ", 0, 0 ).stripWhiteSpace();
          type = "Disk";
          comment = (*it).section( " Disk     ", 1, 1 ).stripWhiteSpace();
        }
        else if ( (*it).contains( " Print    ", true ) != 0 )
        {
          name = (*it).section( " Print    ", 0, 0 ).stripWhiteSpace();
          type = "Printer";
          comment = (*it).section( " Print    ", 1, 1 ).stripWhiteSpace();
        }
        else if ( (*it).contains( " IPC      ", true ) != 0 )
        {
          name = (*it).section( " IPC      ", 0, 0 ).stripWhiteSpace();
          type = "IPC";
          comment = (*it).section( " IPC      ", 1, 1 ).stripWhiteSpace();
        }
        else
        {
          continue;
        }

        share_list.append( new Smb4KShareItem( m_workgroup, m_host, name, type, comment ) );
      }
      else
      {
        continue;
      }
    }
  }
  else
  {
    bool process = false;

    for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( (*it).stripWhiteSpace().startsWith( "Sharename" ) )
      {
        ++it;
        process = true;
      }

      if ( process )
      {
        QString name, type, comment;

        if ( (*it).contains( " Disk   " ) )
        {
          name = (*it).section( " Disk   ", 0, 0 ).stripWhiteSpace();
          type = "Disk";
          comment = (*it).section( " Disk   ", 1, 1 ).stripWhiteSpace();
        }
        else if ( (*it).contains( " Printer   " ) )
        {
          name = (*it).section( " Printer  ", 0, 0 ).stripWhiteSpace();
          type = "Printer";
          comment = (*it).section( " Printer   ", 1, 1 ).stripWhiteSpace();
        }
        else if ( (*it).contains( "  IPC  " ) )
        {
          name = (*it).section( "  IPC  ", 0, 0 ).stripWhiteSpace();
          type = "IPC";
          comment = (*it).section( "  IPC  ", 1, 1 ).stripWhiteSpace();
        }
        else if ( (*it).stripWhiteSpace().startsWith( "Server" ) )
        {
          break;
        }
        else
        {
          continue;
        }

        share_list.append( new Smb4KShareItem( m_workgroup, m_host, name, type, comment ) );
      }
    }
  }

  emit shares( share_list );
}


/****************************************************************************
   Process the information about a host.
****************************************************************************/

void Smb4KScanner::processInfo()
{
  QStringList list = QStringList::split( '\n', m_buffer, false );

  Smb4KHostItem *host = new Smb4KHostItem( m_workgroup, m_host );

  for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    if ( (*it).stripWhiteSpace().startsWith( "Domain" ) || (*it).stripWhiteSpace().startsWith( "OS" ) )
    {
      // The OS string.
      host->setOSString( (*it).section( "OS=[", 1, 1 ).section( "]", 0, 0 ).stripWhiteSpace() );

      // The Server string.
      host->setServerString( (*it).section( "Server=[", 1, 1 ).section( "]", 0, 0 ).stripWhiteSpace() );

      break;
    }
    else if ( (*it).contains( "Connection to", true ) != 0 )
    {
      break;
    }
  }

  emit info( host );

  delete host;
}


/****************************************************************************
   Process the preview.
****************************************************************************/

void Smb4KScanner::processPreview()
{
  QStringList list = QStringList::split( '\n', m_buffer, false );

  QValueList<Smb4KPreviewItem *> preview;

  // Now do the rest:
  if ( list.grep( "NT_STATUS" ).count() != 0 || list.grep( "Connection to" ).count() != 0 )
  {
    if ( list.grep( "NT_STATUS" ).count() != 0  )
    {
      QString errmsg = list.grep( "NT_STATUS" ).first().stripWhiteSpace().section( " ", 0, 0 );

      // The error output of smbclient is a little bit inconsistent:
      if ( errmsg.contains( "NT_STATUS" ) == 0 )
      {
        errmsg = list.grep( "NT_STATUS" ).first().stripWhiteSpace().section( " ", -1, -1 );
      }

      if ( QString::compare( errmsg, "NT_STATUS_ACCESS_DENIED" ) == 0 ||
           QString::compare( errmsg, "NT_STATUS_LOGON_FAILURE" ) == 0 )
      {
        int state = Smb4KPasswordHandler::None;

        if ( QString::compare( errmsg, "NT_STATUS_ACCESS_DENIED" ) == 0 )
        {
          state = Smb4KPasswordHandler::AccessDenied;
        }
        else if ( QString::compare( errmsg, "NT_STATUS_LOGON_FAILURE" ) == 0 )
        {
          state = Smb4KPasswordHandler::LogonFailure;
        }

        if ( m_password_handler->askpass( m_workgroup, m_host, m_share, state ) )
        {
          QString *input = new QString( QString( "%1:%2:%3:%4:%5:%6" ).arg( Preview ).arg( m_workgroup ).arg( m_host ).arg( m_ip ).arg( m_share ).arg( m_path ) );
          m_queue.enqueue( input );
        }
      }
      else
      {
        emit error( ERROR_GETTING_PREVIEW, list.grep( "NT_STATUS" ).first().stripWhiteSpace() );
      }
    }
    else
    {
      emit error( ERROR_GETTING_PREVIEW, list.grep( "Connection to" ).first().stripWhiteSpace() );
    }
  }
  else if ( list.grep( "Error returning browse list:" ).count() != 0 && list.grep( "NT_STATUS" ).count() == 0 )
  {
    emit error( ERROR_GETTING_PREVIEW, list.grep( "Error returning browse list:" ).first().stripWhiteSpace() );
  }
  else
  {
    for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( (*it).stripWhiteSpace().startsWith( "Domain" ) ||
           (*it).stripWhiteSpace().startsWith( "OS" ) ||
           (*it).stripWhiteSpace().startsWith( "Anonymous" ) )
      {
        continue;
      }
      else if ( (*it).contains( "blocks of size" ) != 0 )
      {
        continue;
      }
      else
      {
        QString tmp = (*it).stripWhiteSpace().section( " ", 0, -9 ).stripWhiteSpace();

        QString item = tmp.section( "  ", 0, -2 ).stripWhiteSpace();

        bool isFile = true;

        if ( !item.isEmpty() && tmp.section( "  ", -1, -1 ).contains( "D" ) != 0 )
          isFile = false;
        else if ( item.isEmpty() )
          item = tmp;

        Smb4KPreviewItem *i = new Smb4KPreviewItem( m_host, m_share, m_path, item, isFile );

        preview.append( i );
      }
    }
  }

  emit previewResult( preview );
}


/****************************************************************************
   Process the search data.
****************************************************************************/

void Smb4KScanner::processSearch()
{
  config()->setGroup( "Browse Options" );
  QString search_method = config()->readEntry( "Network Search", "nmblookup" );

  QStringList data = QStringList::split( "\n", m_buffer.stripWhiteSpace(), false );

  if ( QString::compare( search_method, "nmblookup" ) == 0 )
  {
    if ( !data.isEmpty() )
    {
      // The last entry in the list is the workgroup:
      QString workgroup = data.last().stripWhiteSpace();
      QString host, ip;

      if ( m_host.contains( ".", true ) != 3 )
      {
        // The IP address is in the first entry:
        ip = data.first().stripWhiteSpace().section( " ", 0, 0 );
        // The host.
        host = m_host.upper();
      }
      else
      {
        ip = m_host;
        host = data[0].stripWhiteSpace();
      }

      emit searchResult( new Smb4KHostItem( workgroup, host, QString::null, ip ) );
    }
    else
      emit searchResult( new Smb4KHostItem() );
  }
  else
  {
    if ( data.count() > 1 && !data[1].isEmpty() )
    {
      if ( m_buffer.contains( QString( "Connection to %1 failed" ).arg( m_host ) ) != 0 )
      {
        emit searchResult( new Smb4KHostItem() );
      }
      else
      {
        QString workgroup = data.grep( "Domain" ).first().section( "Domain=[", 1, 1 ).section( "]", 0, 0 );
        QString ip = data.grep( "Got a positive name query" ).first().section( "(", 1, 1 ).section( ")", 0, 0 ).stripWhiteSpace();

        emit searchResult( new Smb4KHostItem( workgroup, m_host.upper(), QString::null, ip ) );
      }
    }
    else
      emit searchResult( new Smb4KHostItem() );
  }
}


/****************************************************************************
   Appends an item to the list of workgroups.
****************************************************************************/

void Smb4KScanner::appendWorkgroup( Smb4KWorkgroupItem *item )
{
  item->setPseudo();

  if ( getWorkgroup( item->workgroup() ) == 0 )
  {
    m_workgroup_list.append( item );
  }
}


/****************************************************************************
   Get a workgroup item out of the workgroup list.
****************************************************************************/

Smb4KWorkgroupItem *Smb4KScanner::getWorkgroup( const QString &workgroup )
{
  QValueListIterator<Smb4KWorkgroupItem *> it;

  for ( it = m_workgroup_list.begin(); it != m_workgroup_list.end(); ++it )
  {
    if ( QString::compare( (*it)->workgroup(), workgroup ) == 0 )
    {
      break;
    }
    else
    {
      continue;
    }
  }

  return it == m_workgroup_list.end() ? 0 : *it;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KScanner::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KScanner::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KScanner::slotReceivedStderr( KProcess *, char  *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KScanner::start()
{
  if ( !m_working && !m_queue.isEmpty() )
  {
    // Start processing...
    QString *item = m_queue.dequeue();

    int todo = item->section( ":", 0, 0 ).toInt();

    // Tell the program, that the scanner is running again.
    m_working = true;

    QString workgroup, host;

    switch ( todo )
    {
      case Init:
        emit running( SCANNER_INIT, m_working );
        init();
        break;
      case Hosts:
        emit running( SCANNER_OPENING_WORKGROUP, m_working );
        scanForWorkgroupMembers( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ) );
        break;
      case Shares:
        emit running( SCANNER_OPENING_HOST, m_working );
        workgroup = item->section( ":", 1, 1 );
        host = item->section( ":", 2, 2 );
        scanForShares( workgroup, host, item->section( ":", 3, 3) );
        break;
      case Info:
        emit running( SCANNER_RETRIEVING_INFO, m_working );
        workgroup = item->section( ":", 1, 1 );
        host = item->section( ":", 2, 2 );
        scanForInfo( workgroup, host, item->section( ":", 3, 3 ) );
        break;
      case Preview:
        emit running( SCANNER_RETRIEVING_PREVIEW, m_working );
        preview( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), item->section( ":", 4, 4 ), item->section( ":", 5, 5 ) );
        break;
      case Search:
        emit running( SCANNER_SEARCHING, m_working );
        searchForHost( item->section( ":", 1, 1 ) );
        break;
      default:
        break;
    }

    // FIXME: The browser window sends todo == Shares and todo == GetInfo
    // requests back-on-back. The latter are redundant, because the necessary
    // info is also provided by processShares(). In the following we will remove
    // these entries.
    if ( !m_queue.isEmpty() )
    {
      if ( m_queue.head()->startsWith( QString( "%1:" ).arg( Info ) ) && m_queue.head()->contains( workgroup ) && m_queue.head()->contains( host ) )
      {
        m_queue.remove();
      }
    }

    delete item;
  }
}


/****************************************************************************
   External slots.
****************************************************************************/

void Smb4KScanner::slotInit()
{
  rescan();
}


#include "smb4kscanner.moc"
