/***************************************************************************
    smb4ksambaoptionsinfo  -  This is a container class that carries
    various information of extra options for a specific host.
                             -------------------
    begin                : Mi Okt 18 2006
    copyright            : (C) 2006-2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <QHostAddress>
#include <QAbstractSocket>

// system includes
#include <unistd.h>
#include <sys/types.h>

// application specific includes
#include <smb4ksambaoptionsinfo.h>
#include <smb4khost.h>
#include <smb4kshare.h>


Smb4KSambaOptionsInfo::Smb4KSambaOptionsInfo( Smb4KHost *host )
: m_unc( host->name() ), m_remount( UndefinedRemount ), m_port( -1 ),
#ifndef __FreeBSD__
  m_write_access( UndefinedWriteAccess ),
#endif
  m_protocol( UndefinedProtocol ), m_kerberos( UndefinedKerberos ),
  m_uid( getuid() ), m_uid_set( false ), m_gid( getgid() ), m_gid_set( false ),
  m_workgroup( host->workgroup() ), m_ip( host->ip() ), m_profile( QString() )
{
}



Smb4KSambaOptionsInfo::Smb4KSambaOptionsInfo( Smb4KShare *share )
: m_unc( share->unc() ), m_remount( UndefinedRemount ), m_port( -1 ),
#ifndef __FreeBSD__
  m_write_access( UndefinedWriteAccess ),
#endif
  m_protocol( UndefinedProtocol ), m_kerberos( UndefinedKerberos ),
  m_uid( share->uid() ), m_uid_set( true ), m_gid( share->gid() ), m_gid_set( true ),
  m_workgroup( share->workgroup() ), m_ip( share->hostIP() ), m_profile( QString() )
{
}



Smb4KSambaOptionsInfo::Smb4KSambaOptionsInfo( const Smb4KSambaOptionsInfo &info )
: m_unc( info.unc() ), m_remount( info.remount() ), m_port( info.port() ),
#ifndef __FreeBSD__
  m_write_access( info.writeAccess() ),
#endif
  m_protocol( info.protocol() ), m_kerberos( info.useKerberos() ),
  m_uid( info.uid() ), m_uid_set( true ), m_gid( info.gid() ), m_gid_set( true ),
  m_workgroup( info.workgroup() ), m_ip( info.ip() ), m_profile( info.profile() )
{
}


Smb4KSambaOptionsInfo::Smb4KSambaOptionsInfo()
: m_unc( QString() ), m_remount( UndefinedRemount ), m_port( -1 ),
#ifndef __FreeBSD__
  m_write_access( UndefinedWriteAccess ),
#endif
  m_protocol( UndefinedProtocol ), m_kerberos( UndefinedKerberos ),
  m_uid( getuid() ), m_uid_set( false ), m_gid( getgid() ), m_gid_set( false ),
  m_workgroup( QString() ), m_ip( QString() ), m_profile( QString() )
{
}


Smb4KSambaOptionsInfo::~Smb4KSambaOptionsInfo()
{
}


void Smb4KSambaOptionsInfo::setRemount( Smb4KSambaOptionsInfo::Remount remount )
{
  m_remount = remount;
}


void Smb4KSambaOptionsInfo::setUNC( const QString &unc )
{
  m_unc = unc;
}


void Smb4KSambaOptionsInfo::setPort( int port )
{
  m_port = port;
}


void Smb4KSambaOptionsInfo::setProtocol( Smb4KSambaOptionsInfo::Protocol protocol )
{
  m_protocol = protocol;
}


void Smb4KSambaOptionsInfo::setUseKerberos( Smb4KSambaOptionsInfo::Kerberos kerberos )
{
  m_kerberos = kerberos;
}


int Smb4KSambaOptionsInfo::type()
{
  return (m_unc.count( "/" ) == 3 ? Share : Host);
}


void Smb4KSambaOptionsInfo::setUID( uid_t uid )
{
  m_uid = uid;

  switch ( type() )
  {
    case Host:
    {
      m_uid_set = false;

      break;
    }
    case Share:
    {
      m_uid_set = true;

      break;
    }
    default:
    {
      break;
    }
  }
}


void Smb4KSambaOptionsInfo::setGID( gid_t gid )
{
  m_gid = gid;

  switch ( type() )
  {
    case Host:
    {
      m_gid_set = false;

      break;
    }
    case Share:
    {
      m_gid_set = true;

      break;
    }
    default:
    {
      break;
    }
  }
}

#ifndef __FreeBSD__

void Smb4KSambaOptionsInfo::setWriteAccess( Smb4KSambaOptionsInfo::WriteAccess write_access )
{
  m_write_access = write_access;
}

#endif


void Smb4KSambaOptionsInfo::setHasCustomOptions( bool has_options )
{
  m_has_custom_options = has_options;
}


void Smb4KSambaOptionsInfo::setWorkgroup( const QString &workgroup )
{
  m_workgroup = workgroup;
}


void Smb4KSambaOptionsInfo::setIP( const QString &ip )
{
  QHostAddress ip_address( ip );

  if ( ip_address.protocol() == QAbstractSocket::UnknownNetworkLayerProtocol )
  {
    // The IP address is invalid.
    m_ip.clear();
  }
  else
  {
    // The IP address is OK.
    m_ip = ip;
  }
}


void Smb4KSambaOptionsInfo::update( Smb4KSambaOptionsInfo *info )
{
  // UNC, workgroup and IP address are not updated.

  m_remount      = info->remount();
  m_port         = info->port();
#ifndef __FreeBSD__
  m_write_access = info->writeAccess();
#endif
  m_protocol     = info->protocol();
  m_kerberos     = info->useKerberos();
  m_uid          = info->uid();
  m_gid          = info->gid();
}


QMap<QString,QString> Smb4KSambaOptionsInfo::entries()
{
  QMap<QString,QString> entries;

  switch ( m_remount )
  {
    case DoRemount:
    {
      entries.insert( "remount", "true" );

      break;
    }
    case NoRemount:
    {
      entries.insert( "remount", "false" );

      break;
    }
    case UndefinedRemount:
    {
      entries.insert( "remount", QString() );

      break;
    }
    default:
    {
      break;
    }
  }

  entries.insert( "port", m_port != -1 ?
                          QString( "%1" ).arg( m_port ) :
                          QString() );

  switch ( m_protocol )
  {
    case Automatic:
    {
      entries.insert( "protocol", "auto" );

      break;
    }
    case RPC:
    {
      entries.insert( "protocol", "rpc" );

      break;
    }
    case RAP:
    {
      entries.insert( "protocol", "rap" );

      break;
    }
    case ADS:
    {
      entries.insert( "protocol", "ads" );

      break;
    }
    case UndefinedProtocol:
    {
      entries.insert( "protocol", QString() );

      break;
    }
    default:
    {
      break;
    }
  }

#ifndef __FreeBSD__
  switch ( m_write_access )
  {
    case ReadWrite:
    {
      entries.insert( "write_access", "true" );

      break;
    }
    case ReadOnly:
    {
      entries.insert( "write_access", "false" );

      break;
    }
    case UndefinedWriteAccess:
    {
      entries.insert( "write_access", QString() );

      break;
    }
    default:
    {
      break;
    }
  }
#endif

  switch ( m_kerberos )
  {
    case UseKerberos:
    {
      entries.insert( "kerberos", "true" );

      break;
    }
    case NoKerberos:
    {
      entries.insert( "kerberos", "false" );

      break;
    }
    case UndefinedKerberos:
    {
      entries.insert( "kerberos", QString() );

      break;
    }
    default:
    {
      break;
    }
  }

  if ( type() == Host )
  {
    entries.insert( "uid", QString() );
  }
  else
  {
    if ( m_uid_set )
    {
      entries.insert( "uid", QString( "%1" ).arg( m_uid ) );
    }
    else
    {
      entries.insert( "uid", QString() );
    }
  }

  if ( type() == Host )
  {
    entries.insert( "gid", QString() );
  }
  else
  {
    if ( m_gid_set )
    {
      entries.insert( "gid", QString( "%1" ).arg( m_gid ) );
    }
    else
    {
      entries.insert( "gid", QString() );
    }
  }

  return entries;
}


void Smb4KSambaOptionsInfo::setProfile( const QString &name )
{
  m_profile = name;
}

