/*
    This file is the "straight-talking" counterpart to the sly ident daemon
    (slidentd).  slidentd was written by Sean Hunter <sean@uncarved.com> as a
    minimal RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slidentd.h"
#include "slid_fmt_xlong0.h"
#include <stralloc.h>
#include <byte.h>
#include <unistd.h>
#include <buffer.h>

int slid_running_tcpserver;
int slid_running_inetd;
int slid_use_syslog;

int
main()
{
	int server_portnum;
	int client_portnum;
	int uid;
	buffer *log_buffer;
	stralloc *remoteip;
	struct passwd *p;
	struct passwd *user;

	slid_use_syslog = SLIDENTD_USE_SYSLOG;

	/* Various automatic auditing tools warn on getenv
	   As you can see, the usage below is perfectly safe -
	   If the environment variable is set it acts like its 
	   running under tcpserver.  This has the effect of using 
	   the TCPREMOTEIP and TCPREMOTEPORT env vars, and changing 
	   the logging somewhat when not using syslog */
	if (getenv("TCPREMOTEIP")) {
		slid_running_tcpserver = 1;
		slid_running_inetd = 0;
	} else {
		slid_running_inetd = 1;
		slid_running_tcpserver = 0;
	}

	p = getpwnam(SLIDENTD_UNPRIVILEGED_USER);
	slid_open_tcptable();
	slid_setup_remoteinfo();
	slid_chroot();
	if (p != NULL) {
		slid_setuid(p);
	}

	slid_get_input(&server_portnum, &client_portnum);

	remoteip = slid_get_remote_ip();

	log_buffer = slid_get_log_buffer();
	slid_put_datestamp(log_buffer);
	buffer_puts(log_buffer, "Q [");
	buffer_puts(log_buffer, remoteip->s);
	buffer_puts(log_buffer, "] - [");
	buffer_putlong(log_buffer, (long) server_portnum);
	buffer_puts(log_buffer, ", ");
	buffer_putlong(log_buffer, (long) client_portnum);
	buffer_putsflush(log_buffer, "]\n");
	uid = slid_search_tcptable(server_portnum, client_portnum);

	buffer_putlong(buffer_1, server_portnum);
	buffer_puts(buffer_1, ", ");
	buffer_putlong(buffer_1, client_portnum);
	buffer_puts(buffer_1, ": USERID: OTHER: ");
	user = getpwuid((uid_t) uid);
	if (user) {
		buffer_puts(buffer_1, user->pw_name);
	} else {
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer,
			    "Unable to find matching password entry for uid ");
		buffer_putlong(log_buffer, (long) uid);
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}
	buffer_putsflush(buffer_1, "\r\n");

	slid_put_datestamp(log_buffer);
	buffer_puts(log_buffer, "A [");
	buffer_puts(log_buffer, remoteip->s);
	buffer_puts(log_buffer, "] - [");
	buffer_puts(log_buffer, user->pw_name);
	buffer_puts(log_buffer, "] for [");
	buffer_putlong(log_buffer, (long) server_portnum);
	buffer_puts(log_buffer, ", ");
	buffer_putlong(log_buffer, (long) client_portnum);
	buffer_puts(log_buffer, "] UID [");
	buffer_putlong(log_buffer, uid);
	buffer_putsflush(log_buffer, "]\n");

	slid_close_tcptable();
	return (0);
}
