%  vwutils.sl: this file is part of the VWhere SLgtk guilet {{{
%
%  Copyright (C) 2003-2009 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
% 
% }}}

% Front matter {{{ 

require("toolbox");
require("gtkplot");

public variable _vwhere_version = 10310;
public variable _vwhere_version_string = "1.3.10";
variable window_title = "VWhere "+ _vwhere_version_string;
% }}}

define echo_status() % {{{
{
   variable vwd, message;
   (vwd, message) = ();

   % Write to gprompt terminal output window if visible, else status bar
   variable prompt = vwd.prompter.prompt;
   if (not prompt.visible())
	() = gtk_statusbar_push(vwd.status, 1, message);
   else if (message != "")
	prompt.log(message);
} % }}}

% Expression evaluation {{{
define evaluate(vwd, e, namespace)
{
   variable ad = _auto_declare, err;
   variable e2 = sprintf("_auto_declare=1; %s; _auto_declare = %d;", e, ad);

   try (err) { eval(e2, namespace); }
   catch AnyError: {

	e = sprintf("could not evaluate expression: %s\n%s", e, err.descr);

	variable prompt = vwd.prompter.prompt;

	if (prompt.visible())
	   prompt.log(e);
	else
	   _error_dialog("", e);

	return 0;
   }

   return 1;
}
% }}}

define has_background(vwd) % {{{
{
   if (vwd.filtd == NULL) return 0;
   return (length(vwd.filtd.plotd.dsets) > 1);
} % }}}

() = evalfile("vwprefs", vw_private_ns);

%  Region Filters {{{

variable SHAPE_RECTANGLE = "RectangleFilter";
variable SHAPE_ELLIPSE   = "EllipseFilter";
variable SHAPE_POLYGON   = "PolygonFilter";

% Note that exclusion filtering could be implemented more concisely, entirely
% outside each shape filter as not(included), but that doesn't scale as well
% because the negation operator performs another array traversal.  To save
% that cost, each shape filter explicitly performs the exclude, internally.

private define apply_rectangle(reg, x, y) % {{{
{
   if (reg.exclude) {
	variable xin = (x < reg.x[0] or x > reg.x[1]);
	variable yin = (y < reg.y[0] or y > reg.y[1]);
	return (xin or yin);
   }
   else {
	xin = (x >= reg.x[0] and x <= reg.x[1]);
	yin = (y >= reg.y[0] and y <= reg.y[1]);
	return (xin and yin);
   }
} % }}}

private define apply_circle(reg, x, y, h, k, r) % {{{
{
   variable z = (x - h) ^ 2 + (y - k)^2;

   if (reg.exclude)
	return (z > r^2);

   return (z <= r^2);
} % }}}

private define apply_ellipse(reg, x, y) % {{{
{
   variable r1 = abs(reg.x[1] - reg.x[0]) / 2.0;
   variable r2 = abs(reg.y[1] - reg.y[0]) / 2.0;
   variable h = reg.x[0] + r1;
   variable k = reg.y[0] + r2;

   if (r1 == r2)
      return apply_circle(x, y, h, k, r1);

   variable xp = (x - h) / r1;
   variable yp = (y - k) / r2;

   if (reg.exclude)
	return ((xp^2 + yp^2) > 1.0);

   return ((xp^2 + yp^2) <= 1.0);
} % }}}

private define apply_polygon(reg, x, y) % {{{
{
   return _polygon_filter(x, y, reg.x, reg.y, reg.exclude);
} % }}}

private variable RegionList = struct {
	parent,
	head,
	tail,
	append,
	delete,
	set_bbox,
	widget2struct,
	toggle_exclude,
	translate_coords
};

variable Region = struct {
	type,			% rectangle, ellipse, etc
	widget,			% underlying Gtk widget visualizing the region
	x, y,			% points defining the region
	bbox,			% region bounding box (in dataspace coords)
	exclude,		% does this region include or exclude contents?
	apply,
	prev,			% previous region in list
	next			% next region in list
};

private define _region_append(list, region)	% O(1) insertion {{{
{
   if (list.head == NULL) {
	list.head = region;
	list.tail = region;
   }
   else {
	region.prev = list.tail;
	list.tail.next = region;
	list.tail = region;
   }
} % }}}

private define _region_delete(list, region)	% O(1) removal {{{
{
   variable canvas = list.parent.plotd.canvas;

   gtk_plot_canvas_cancel_action(canvas);
   !if (gtk_plot_canvas_remove_child(canvas, region.widget))
	return;

   if (region.prev != NULL)
	region.prev.next = region.next;
   else
	list.head = region.next;

   if (region.next == NULL) list.tail = region.prev;
} % }}}

private define _region_toggle_exclude(list, region) % {{{
{
   region.exclude = not gtk_plot_canvas_child_toggle_slashed(region.widget);
} % }}}

private define _region_set_bbox(list, region) % {{{
{
   % Define bounding box, in plot data space, for moving/scaling/filtering
   variable x1, y1, x2, y2;
   variable filtd = list.parent;
   variable plotd = filtd.plotd;

   () = gtk_plot_canvas_get_child_position(plotd.canvas, region.widget,
   							&x1, &y1, &x2, &y2);
   gtk_plot_canvas_get_pixel(plotd.canvas, x1, y1, &x1, &y1);
   gtk_plot_canvas_get_pixel(plotd.canvas, x2, y2, &x2, &y2);

   gtk_plot_get_point(plotd.plot, x1, y1, &x1, &y1);
   gtk_plot_get_point(plotd.plot, x2, y2, &x2, &y2);

   region.bbox = [x1, y1, x2, y2];
} % }}}

private define _region_widget2struct(list, widget) % {{{
{
   foreach(list.head) {
	variable region = ();
	if (__eqs_gtkopaque(widget, region.widget))
	   return region;
   }
   return NULL;
} % }}}

private define _region_translate_coords(this, reg) % {{{
{
   reg = @reg;

   switch(reg.type)
	{ case SHAPE_RECTANGLE or case SHAPE_ELLIPSE:
	   % Bounding box already specificies data-relative coords, so just
	   % ensure (x1,y1) is bottom left, (x2,y2) top right (data coords)
	   % This is done b/c in screen coords (pixel or canvas) y increases
	   % from top to bottom, but in data-relative coords it's the reverse.
	   reg.x = [ reg.bbox[0], reg.bbox[2] ];
	   reg.y = [ reg.bbox[3], reg.bbox[1] ];
	}
	{ case SHAPE_POLYGON: 

	   % Poly may have been moved/scaled since being laid, but vertices
	   % (poly.x, poly.y, in canvas space) are still relative to the
	   % initial pos, so recenter/scale them relative to current pos.
	   variable x = min(reg.x), x2 = max(reg.x), w = (x2 - x);
	   variable y = min(reg.y), y2 = max(reg.y), h = (y2 - y);

	   variable bbx1, bbx2, bby1, bby2;
	   variable plotd = this.parent.plotd;
	   variable canvas = plotd.canvas;

	   () = gtk_plot_canvas_get_child_position(canvas, reg.widget,
						&bbx1, &bby1, &bbx2, &bby2);

	   variable xscale = (bbx2 - bbx1) / w;
	   variable yscale = (bby2 - bby1) / h;
	   reg.x = bbx1 + (reg.x - x) * xscale;
	   reg.y = bby1 + (reg.y - y) * yscale;

	   % Finally, map the canvas-space coordinates to data space
	   variable i, nvertices = length(reg.x);
	   for (i=0; i < nvertices; i++) {
		gtk_plot_canvas_get_pixel(canvas, reg.x[i], reg.y[i], &x, &y);
		gtk_plot_get_point(plotd.plot, x, y, &x, &y);
		reg.x[i] = x;
		reg.y[i] = y;
	   }
	}

   return reg;
} % }}}

private define _region_new_append(list, type, x, y) % {{{
{
   variable filtd = list.parent;
   variable canvas = filtd.plotd.canvas;

   variable r = @Region;
   r.type     = type;
   r.exclude = (filtd.key_being_pressed == GDK_e);

   switch(type)
   { case SHAPE_RECTANGLE:

	 r.widget = gtk_plot_canvas_put_rectangle(canvas,x[0],y[0],x[1],y[1],
			GTK_PLOT_LINE_SOLID, 0, gdk_blue,
			NULL, GTK_PLOT_BORDER_LINE, r.exclude << 1);
	 r.apply = &apply_rectangle;
   }
   { case SHAPE_ELLIPSE:

	 r.widget = gtk_plot_canvas_put_ellipse(canvas,x[0],y[0],x[1],y[1],
			GTK_PLOT_LINE_SOLID,0,gdk_blue,NULL, r.exclude << 1);
	 r.apply = &apply_ellipse;
   }
   { case SHAPE_POLYGON:

	r.widget = _gtk_plot_canvas_put_polygon(canvas, x, y,
			GTK_PLOT_LINE_SOLID,0,gdk_blue,NULL, r.exclude << 1);
	r.apply = &apply_polygon;
	r.x = x;
	r.y = y;
   }

   _region_set_bbox(list, r);
   _region_append(list, r);
} % }}}

define region_list_new(parent) % {{{
{
   variable l = @RegionList;
   l.parent = parent;
   l.append = &_region_new_append;
   l.delete = &_region_delete;
   l.widget2struct = &_region_widget2struct;
   l.set_bbox = &_region_set_bbox;
   l.translate_coords = &_region_translate_coords;
   l.toggle_exclude = &_region_toggle_exclude;
   return l;
} % }}}

private variable cursors = [ % {{{
	gdk_cursor_new(GDK_LEFT_PTR),
	gdk_cursor_new(GDK_CROSSHAIR),
	gdk_cursor_new(GDK_WATCH)
];
private variable num_cursors = length(cursors); % }}}

define set_cursor(widget,cursor) % {{{
{
   variable window = gtk_widget_get_window(widget);
   if (window != NULL) {
	if (cursor < 1)
	   cursor = 1;
	else if (cursor > num_cursors)
	   cursor = num_cursors;
	gdk_window_set_cursor(window, cursors[cursor-1]);
   }
} % }}}

define transform_regions(vwd) % {{{
{
   if (vwd.plotwin == NULL) return;

   variable fd = vwd.filtd, pd = fd.plotd, canvas = pd.canvas;

   set_cursor(vwd.plotwin, 3);	% shouldn't this require
   set_cursor(vwd.panes, 3);	% only 1 function call?
   set_cursor(pd.canvas, 3);

   gtk_plot_canvas_freeze(canvas);		% avoid multiple redraws
   variable bbox, x1, y1, x2, y2;

   foreach(fd.regions.head) {

	variable reg = ();
	bbox = reg.bbox;
	(x1,y1) = _gtk_plot_transform(pd, bbox[0], bbox[1], -1);
	(x2,y2) = _gtk_plot_transform(pd, bbox[2], bbox[3], -1);

	gtk_plot_canvas_child_move_resize(canvas, reg.widget, x1, y1, x2, y2);
   }
   gtk_plot_canvas_thaw(canvas);
   _gtk_plot_redraw(pd);

   set_cursor(vwd.plotwin, 1);
   set_cursor(vwd.panes, 1);
   set_cursor(pd.canvas, 1);
} % }}}

define transform_axis(plotd, which, shift, zoom, func) % {{{
{
   % Display a shifted centroid 1/2, 1/4, 1/8, ... size of original view
   variable axis = plotd.dset.axes[which];

   shift = shift * axis.range;
   variable zoom_delta = (1 - zoom) * (axis.range / 2);

   (@func)(plotd, axis.min + shift + zoom_delta, axis.max + shift - zoom_delta);
} % }}}

define display_zoom(vwd) % {{{
{
   if (vwd.plotwin == NULL) return;
   gtk_label_set_text(vwd.zoom_area,sprintf("Zoom: %3.3f",1/vwd.filtd.zoom));
} % }}}

private define format_coord(value, range) % {{{
{
   if (range < 0.001)
       return sprintf("%5.3E", value);

   return sprintf("%5.3f",value);
} % }}}

define display_scaling(vwd) % {{{
{
   if (vwd.plotwin == NULL) return;

   variable pd = vwd.filtd.plotd;
   variable xb = vwd.xscale_button, xid = g_object_get_data(xb, "handler_id");
   variable yb = vwd.yscale_button, yid = g_object_get_data(yb, "handler_id");

   g_signal_handler_block(xb, xid);
   g_signal_handler_block(yb, yid);

   gtk_toggle_button_set_active(xb, gtk_plot_get_xscale(pd.plot));
   gtk_toggle_button_set_active(yb, gtk_plot_get_yscale(pd.plot));

   g_signal_handler_unblock(xb, xid);
   g_signal_handler_unblock(yb, yid);
} % }}}

define display_coords(vwd) % {{{
{
   variable x, y, pd = vwd.filtd.plotd;
   gtk_widget_get_pointer(pd.canvas, &x, &y);

   gtk_plot_get_point(pd.plot, x, y, &x, &y);

   x = format_coord(x, pd.dset.axes[0].range);
   y = format_coord(y, pd.dset.axes[1].range);

   gtk_label_set_text(vwd.coords_area, sprintf("( %s, %s )", x, y));
} % }}}

define set_viewpoint(vwd) % {{{
{
   % Transform the plot (and any regions) accordingly
   variable fd = vwd.filtd, pd = fd.plotd;
   transform_axis(pd, 0, fd.hshift, fd.zoom, &_gtk_plot_set_xrange);
   transform_axis(pd, 1, fd.vshift, fd.zoom, &_gtk_plot_set_yrange);
   transform_regions(vwd);
   display_zoom(vwd);
   display_scaling(vwd);
} % }}}

private variable VW_AND  = 0;
private variable VW_OR   = 1;

define apply_region_filters(vwd) % {{{
{
   variable nplots = length(vwd.filtds), result = NULL;

   foreach (vwd.filtds) {

	variable filtd = (), x = filtd.x , y = filtd.y;
	variable regions = filtd.regions, region;
	variable included = NULL, excluded = NULL;

	foreach region (regions.head) {

	   region = regions.translate_coords(region);
	   variable this_selection = region.apply(x, y);

	   if (region.exclude) {
		if (excluded == NULL)
		    excluded = this_selection;
		else
		    excluded = excluded and this_selection;
	   }
	   else {
		if (included == NULL)
		   included = this_selection;
		else
		   included = included or this_selection;
	   }
	}

	variable selection = included;

	if (selection == NULL)
	   selection = excluded;
	else if (excluded != NULL)
	   selection = selection and excluded;

	if (result == NULL) {
	    result = selection;			% conserve memory
	    continue;
	}

	if (selection == NULL)			% any contribution
	   continue;				% from this plot?

	switch(vwd.fprefs.combine.value)
	   { case VW_AND: result = (result and selection); }
	   { case VW_OR:  result = (result or selection); }
	   { error("Illegal method specified for combining plot filters"); }
   }

   return result;
} % }}}

% }}}

%  Printing {{{

private define print_cb(ctx, vwd)
{
   variable pd = vwd.filtd.plotd, fname;

   if (ctx.dest == 0)
	fname = _tmp_file_name("vwhere");
   else
	fname = ctx.fname;

   % Generate an encapsulated postscript file
   () = gtk_plot_canvas_export_ps(pd.canvas, fname, ctx.orient, TRUE, ctx.size);

   if (ctx.dest == 0) {
	() = system( sprintf("%s %s",ctx.cmd, fname));
	() = remove(fname);
   }
}

private define print(vwd)
{
   if (vwd.print_ctx == NULL) {
	vwd.print_ctx = @GtkPrintContext;
	vwd.print_ctx.title = "VWhere Print Dialog";
	vwd.print_ctx.fname = "vwhere.ps";
   }
   _print_dialog(vwd.print_ctx, &print_cb, vwd);
}
% }}}

%  Toolbox {{{

private variable MAX_ZOOM = 512.0;
define zoom_in(current_zoom)
{
   return max( [current_zoom / 2.0, 1/ MAX_ZOOM] );
}

define zoom_out(current_zoom)
{
   return min( [current_zoom * 2.0, MAX_ZOOM ] );
}

static define help(dummy)
{
   _info_window(window_title+" Help", _get_slgtk_doc_string("vwhere"));
}

variable TB_ZOOM_OUT	= "ZoomOut";
variable TB_ZOOM_IN	= "ZoomIn";
variable TB_ZOOM_FIT	= "ZoomToFit";
variable TB_NORMAL	= "NormalView";
variable TB_SH_LEFT	= "ScrollLeft";
variable TB_SH_RIGHT	= "ScrollRight";
variable TB_SH_UP	= "ScrollUp";
variable TB_SH_DOWN	= "ScrollDown";
variable TB_LINESTYLE	= "CurrentPlotLineStyles";
variable TB_PREFS	= "Preferences";
variable TB_PRINT	= "Print";
variable TB_HELP	= "Help";

define tb_choose(tb,vwd)
{
   variable fd = vwd.filtd, pd = fd.plotd;

   pd.stop_interactive_draw();

   variable shift = 0.1 * fd.zoom;

   % User may pan up to 400% in either direction, or zoom in/out up to 16x
   % Each zoom in takes centroid of half
   switch(tb.curr)
	{ case TB_SH_LEFT   : fd.hshift = max( [fd.hshift - shift, -4.0] ); }
	{ case TB_SH_RIGHT  : fd.hshift = min( [fd.hshift + shift,  4.0] ); }
	{ case TB_SH_UP     : fd.vshift = min( [fd.vshift + shift,  4.0] ); }
	{ case TB_SH_DOWN   : fd.vshift = max( [fd.vshift - shift, -4.0] ); }
	{ case TB_ZOOM_IN   : fd.zoom = zoom_in(fd.zoom); }
	{ case TB_ZOOM_OUT  : fd.zoom = zoom_out(fd.zoom); }
	{ case TB_NORMAL    :

		if (fd.zoom == 1) return;
	   	fd.zoom = 1.0; fd.hshift = 0.0; fd.vshift = 0.0;
	}
	{ case TB_ZOOM_FIT  : set_cursor(pd.plot, 2); return; }
	{ case TB_PREFS     : prefs_select_full(vwd); return; }
	{ case TB_PRINT	    : print(vwd); return; }
	{ case TB_LINESTYLE : prefs_select_mini(vwd); return; }
	{ case TB_HELP      : help(NULL); return; }
	{ set_cursor(pd.plot, 1); return; }

   set_viewpoint(vwd);
}

define make_toolbox(cbfunc,data)
{
   variable tbox = toolbox_new(GTK_ORIENTATION_HORIZONTAL);

   toolbox_add_button(tbox,SLGTK_STOCK_RECTANGLE,SHAPE_RECTANGLE, cbfunc, data);
   toolbox_add_button(tbox,SLGTK_STOCK_ELLIPSE,SHAPE_ELLIPSE, cbfunc, data);
   toolbox_add_button(tbox,SLGTK_STOCK_POLYGON,SHAPE_POLYGON, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_FIT, TB_ZOOM_FIT, cbfunc, data);
   toolbox_set_focus_default(tbox,FALSE);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_IN, TB_ZOOM_IN, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_OUT, TB_ZOOM_OUT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_ZOOM_100, TB_NORMAL, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_BACK, TB_SH_LEFT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_FORWARD, TB_SH_RIGHT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_UP, TB_SH_UP, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_GO_DOWN, TB_SH_DOWN, cbfunc, data);
   toolbox_add_button(tbox,SLGTK_STOCK_LINESEG, TB_LINESTYLE,
	 					cbfunc, data, "pressed");
   toolbox_add_button(tbox,GTK_STOCK_PREFERENCES, TB_PREFS, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_PRINT, TB_PRINT, cbfunc, data);
   toolbox_add_button(tbox,GTK_STOCK_DIALOG_QUESTION, TB_HELP, cbfunc, data);
   return tbox;
}
% }}}
