/* -*- mode: C; mode: fold; -*- */

/* This file is part of
 *
 * 	SLgtk:  S-Lang bindings for GTK+ widget set
 *
 * Copyright (C) 2003-2004 Massachusetts Institute of Technology 
 * Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
 *
 */

#include <string.h>
#include "gtk/gtk.h"

/* S-Lang structure handling {{{ */
SLang_CStruct_Field_Type GEnumValue_Layout [] =
{
  MAKE_CSTRUCT_FIELD(GEnumValue,value,"value",SLANG_INT_TYPE,0),
  MAKE_CSTRUCT_FIELD(GEnumValue,value_name,"value_name",SLANG_STRING_TYPE,0),
  MAKE_CSTRUCT_FIELD(GEnumValue,value_nick,"value_nick",SLANG_STRING_TYPE,0),
  SLANG_END_CSTRUCT_TABLE
};

SLang_CStruct_Field_Type GError_Layout [] =
{
  MAKE_CSTRUCT_FIELD(GError,domain,"domain",SLANG_INT_TYPE,0),
  MAKE_CSTRUCT_FIELD(GError,code,"code",SLANG_INT_TYPE,0),
  MAKE_CSTRUCT_FIELD(GError,message,"message",SLANG_STRING_TYPE,0),
  SLANG_END_CSTRUCT_TABLE
};

SLang_CStruct_Field_Type GParamSpec_Layout [] =
{
  MAKE_CSTRUCT_FIELD(GParamSpec,g_type_instance,"g_type_instance",
							SLANG_STRUCT_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,name,"name",SLANG_STRING_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,flags,"flags",SLANG_INT_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,value_type,"value_type",SLANG_ULONG_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,owner_type,"owner_type",SLANG_ULONG_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,_nick,"_nick",SLANG_STRING_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,_blurb,"_blurb",SLANG_STRING_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,qdata,"qdata", SLANG_DATATYPE_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,ref_count,"ref_count",SLANG_UINT_TYPE,0),
  MAKE_CSTRUCT_FIELD(GParamSpec,param_id,"param_id",SLANG_UINT_TYPE,0),
  SLANG_END_CSTRUCT_TABLE
};
/* }}} */

/* GValue handling {{{ */

GValue   _GValue_Initializer;

int push_g_value(const GValue *arg)
{

   switch (G_TYPE_FUNDAMENTAL(G_VALUE_TYPE(arg))) {

	case G_TYPE_CHAR:
	   return SLang_push_char(g_value_get_char(arg));
	break;

	case G_TYPE_UCHAR:
	   return SLang_push_uchar(g_value_get_uchar(arg));
	break;

	case G_TYPE_BOOLEAN:
	   return SLang_push_integer(g_value_get_boolean(arg));
	break;

	case G_TYPE_ENUM:
	case G_TYPE_FLAGS:
	case G_TYPE_INT:
	   return SLang_push_integer(g_value_get_int(arg));
	break;

	case G_TYPE_UINT:
	   return SLang_push_uinteger(g_value_get_uint(arg));
	break;

	case G_TYPE_LONG:
	   return SLang_push_long(g_value_get_long(arg));
	break;

	case G_TYPE_ULONG:
	   return SLang_push_ulong(g_value_get_ulong(arg));
	break;

	case G_TYPE_FLOAT:
	   return SLang_push_float(g_value_get_float(arg));
	break;

	case G_TYPE_DOUBLE:
	   return SLang_push_double(g_value_get_double(arg));
	break;

	case G_TYPE_STRING: 	/* !!! Is this correct? */
	   return SLang_push_string((char*)g_value_get_string(arg));
	break;

	case G_TYPE_OBJECT:
	   return SLang_push_opaque(GObject_Type, g_value_get_object(arg), 0);
	break;

	case G_TYPE_BOXED:
	   return push_boxed(arg);
	break;

	case G_TYPE_PARAM:
	   return SLang_push_opaque(GtkOpaque_Type,g_value_get_param(arg), 0);
	break;

	case G_TYPE_POINTER:
	   return SLang_push_opaque(GtkOpaque_Type,g_value_get_pointer(arg),0);
	break;

	/* !!! FIXME: several types still ignored */
	default:

	   if (slgtk_debug > 0) 
		fprintf(stderr,"WARNING: GValue type <%s> ignored "
			"(push_g_value)\n",G_VALUE_TYPE_NAME(arg));
	   return -1;
   }
}

static int push_g_values(const GValue *args, int nargs)
{
   int i;

   for (i = 0; i < nargs; i++) {
	if ( push_g_value((const GValue*)args+i) != 0)
	   return -1;
   }

   return 0;
}

int pop_g_value(GValue *arg)
{

  switch (G_TYPE_FUNDAMENTAL(G_VALUE_TYPE(arg))) {

    case G_TYPE_STRING:
	{
	void *v;
	if (pop_nullable (SLANG_STRING_TYPE, &v, NULL) == -1)
	   return -1;
	g_value_set_string(arg, (const char*)v);
	}
	break;

    case G_TYPE_CHAR:
	{
	char c;
	if (SLang_pop_char(&c) == -1)
	   return -1;
	g_value_set_char(arg,c);
	}
	break;

    case G_TYPE_UCHAR:
	{
	unsigned char uc;
	if (SLang_pop_uchar(&uc) == -1)
	   return -1;
	g_value_set_uchar(arg,uc);
	}
	break;

    case G_TYPE_BOOLEAN:
	{
	int b;
	if (SLang_pop_integer(&b) == -1)
	   return -1;
	g_value_set_boolean(arg, b);
	}
	break;

    case G_TYPE_ENUM:
    case G_TYPE_FLAGS:
    case G_TYPE_INT:
	{
	int i;
	if (SLang_pop_integer(&i) == -1)
	   return -1;
	g_value_set_int(arg, i);
	}
	break;

    case G_TYPE_UINT:
	{
	unsigned int ui;
	if (SLang_pop_uinteger(&ui) == -1)
	   return -1;
	g_value_set_uint(arg, ui);
	}
	break;

    case G_TYPE_LONG:
	{
	long l;
	if (SLang_pop_long(&l) == -1)
	   return -1;
	g_value_set_long(arg, l);
	}
	break;

    case G_TYPE_ULONG:
	{
	unsigned long ul;
	if (SLang_pop_ulong(&ul) == -1)
	   return -1;
	g_value_set_ulong(arg, ul);
	}
	break;

    case G_TYPE_FLOAT:
	{
	float f;
	if (SLang_pop_float(&f) == -1)
	   return -1;
	g_value_set_float(arg, f);
	}
	break;

    case G_TYPE_DOUBLE:
	{
	double d;
	if (SLang_pop_double(&d) == -1)
	   return -1;
	g_value_set_double(arg, d);
	}
	break;

    default:
	if (slgtk_debug > 0) 
	   fprintf(stderr,"WARNING: GValue type of <%s> ignored "
	      					"(g_value_pop_arg)\n",
						G_VALUE_TYPE_NAME(arg));
	return -1;
  }
  return 0;
}
/*  }}} */

/* Function callbacks {{{ */

struct _slGClosure {
  GClosure closure;
  SLang_Name_Type *function;
  SLang_Any_Type **args;	/* func args passd in from S-Lang scope */
  unsigned int  nargs;		/* how many of them? */
  void *swap_data;		/* for _swapped signal connection calls */
};
typedef struct _slGClosure slGClosure;

static void
slg_closure_marshal(GClosure *closure,
                	GValue *return_value,
			guint n_param_values,
			const GValue *param_values,
			gpointer invocation_hint,
			gpointer marshal_data)
{
   slGClosure *pcl;
   unsigned int i;

   (void) invocation_hint; (void) marshal_data;
   pcl = (slGClosure*)closure;

   if (SLang_start_arg_list() == -1)
	return;

   if (pcl->swap_data) {
      (void) push_g_values(param_values+1,n_param_values-1);
      /* !!! needs failure check */
      SLang_push_anytype(pcl->swap_data);
   }
   else
      (void) push_g_values(param_values, n_param_values);

   /* !!! needs failure check */
   for (i = 0; i < pcl->nargs; i++)
	(void)SLang_push_anytype(pcl->args[i]);

   if (SLang_end_arg_list() == -1) {
      SLdo_pop_n(n_param_values + pcl->nargs);
      return;
   }

   (void) SLexecute_function(pcl->function);

   /* Unrecoverable S-Lang errors should not cause main loop to hang */
   if (SLang_get_error() < 0) {
	error_terminate_main_loop(pcl->function->name);
	return;
   }

   if (SLang_get_error()) {
	SLang_restart(0);
	SLang_set_error(0);
	return;
   }

   if (return_value && G_IS_VALUE(return_value))
	if (pop_g_value(return_value) == -1 || SLang_get_error() < 0) {
	   char msg[192];
	   strcpy(msg,"could not pop expected return value from: ");
	   strncat(msg,pcl->function->name,
		(size_t) MIN(strlen(pcl->function->name),192-strlen(msg)-1));
	   error_terminate_main_loop(msg);
	}
}

static void
slg_closure_destroy(gpointer data, GClosure *cl)
{
   unsigned int i;
   slGClosure *pcl = (slGClosure*)cl;

   (void)data;

   for (i=0; i < pcl->nargs; i++)
	SLang_free_anytype(pcl->args[i]);

   SLfree((char*)pcl->args);

   if (pcl->swap_data != NULL)
	SLang_free_anytype(pcl->swap_data);

   SLang_free_function(pcl->function);
}

GClosure*
slg_closure_new(SLang_Name_Type *function, SLang_Any_Type **args,
      			unsigned int nargs, SLang_Any_Type *swap_data)
{
   GClosure *closure;

   closure = g_closure_new_simple( sizeof(slGClosure), NULL);
   g_closure_add_finalize_notifier(closure, NULL, slg_closure_destroy);
   g_closure_set_marshal(closure, slg_closure_marshal);
   ((slGClosure*)closure)->function = function;
   ((slGClosure*)closure)->swap_data = swap_data;
   ((slGClosure*)closure)->args = args;
   ((slGClosure*)closure)->nargs = nargs;

  return closure;
}

int
extract_slang_args(unsigned int nargs, SLang_Any_Type ***pargs)
{
   SLang_Any_Type **args;
   SLang_Any_Type *arg;
   unsigned int narg;

   if (nargs <= 0) {
	*pargs = NULL;
	return 0;
   }

   args = (SLang_Any_Type**) SLmalloc(SIZEOF_POINTER*nargs);
   narg = nargs;
   while (narg) {
	if (pop_anytype_or_null(&arg) == -1) {
	   while (nargs > narg)
		SLang_free_anytype(args[--nargs]);
	   SLfree((char*)args);
	   return -1;
	}
	args[--narg] = arg;

   }
   *pargs = args;
   return 0;
}

void free_slang_args(unsigned int nargs, SLang_Any_Type **args)
{
   while (nargs > 0)
	SLang_free_anytype(args[--nargs]);
   SLfree((char*)args);
}

static void
sig_conn_generic(gboolean after, const char *usage)
{
   GObject *obj ;
   Slirp_Opaque *obj_o = NULL;
   int  connid = -1;
   SLang_Any_Type **args;
   SLang_Name_Type *function;
   char *signame = NULL;
   SLang_Ref_Type *function_ref = NULL;
   unsigned int nargs;

   if (usage_err(3, usage))
	return;

   nargs = SLang_Num_Function_Args - 3;
   if (extract_slang_args(nargs,&args) == -1)
	goto cleanup;

   if ( -1 == SLang_pop_ref(&function_ref) ||
		-1 == SLang_pop_slstring(&signame) ||
		-1 == SLang_pop_opaque( GObject_Type, (void**)&obj, &obj_o))
	goto cleanup;

   if ( (function = SLang_get_fun_from_ref(function_ref)))
	connid = g_signal_connect_closure(obj, signame,
			slg_closure_new(function,args,nargs,NULL), after);

   cleanup:

   if (connid <= 0)
      SLang_verror(SL_INTRINSIC_ERROR,"could not connect signal");

   SLang_free_ref(function_ref);
   SLang_free_opaque(obj_o);
   SLang_free_slstring(signame);
   SLang_push_integer(connid);
}

static void slg_signal_connect(void)
{
   sig_conn_generic(0,"id = g_signal_connect(GObject,string,function_ref,...)");
}

static void slg_signal_connect_after(void)
{
   sig_conn_generic(1,
	 	"id = g_signal_connect_after(GObject,string,function_ref,...)");
}

static void 
slg_signal_connect_swapped(void)
{
   int  connid = -1;
   GObject *obj;
   Slirp_Opaque *obj_o = NULL;
   char *signame = NULL;
   SLang_Any_Type *data;
   SLang_Any_Type **args;
   SLang_Name_Type *function;
   SLang_Ref_Type *function_ref = NULL;
   unsigned int nargs;
   
   if (usage_err(4,
	"id = g_signal_connect_swapped(GObject,string,function_ref,data,...)"))
	return;

   nargs = SLang_Num_Function_Args - 4;
   if (extract_slang_args(nargs,&args) == -1)
	goto cleanup;

   if ( -1 == pop_anytype_or_null(&data)   ||
		-1 == SLang_pop_ref(&function_ref) ||
		-1 == SLang_pop_slstring(&signame) ||
		-1 == SLang_pop_opaque( GObject_Type, (void**)&obj, &obj_o))
	goto cleanup;


   if ( (function = SLang_get_fun_from_ref(function_ref)))
	connid = g_signal_connect_closure(obj, signame,
			slg_closure_new(function,args,nargs,data),FALSE);

   cleanup:

   if (connid <= 0)
      SLang_verror(SL_INTRINSIC_ERROR,"could not connect signal");

   SLang_free_ref(function_ref);
   SLang_free_opaque(obj_o);
   SLang_free_slstring(signame);
   SLang_push_integer(connid);
}

			/* !!! currently ignores sig hdlr args */
static void slg_signal_emit_by_name(Slirp_Opaque *obj, char *name)
{
   g_signal_emit_by_name(obj->instance,name);
}
/* }}} */

/* Wrappers {{{ */
static void slg_enum_get_value(Slirp_Opaque *o, gint *value)
{
   GEnumValue *gev = g_enum_get_value(o->instance,*value);
   if (gev != NULL) {
	(void)SLang_push_cstruct(gev,GEnumValue_Layout);
	return;
   }
   (void)SLang_push_null();
}

static void data_destroy(gpointer data)
{
   SLang_free_anytype((SLang_Any_Type*)data);
}

static void slg_object_set_data(void)
{
   GObject *obj;
   char *key  = NULL;
   Slirp_Opaque *obj_o = NULL;
   SLang_Any_Type *data = NULL;

   if (SLang_Num_Function_Args != 3) {
	SLang_verror (SL_USAGE_ERROR,
		"Usage: g_object_set_data(GObject,string,data);");
	return;
   }

   if (	SLang_pop_anytype(&data) == 0	&&
	SLang_pop_slstring(&key) == 0	&&
	SLang_pop_opaque( GObject_Type, (void**)&obj, &obj_o) == 0)
   
	/* This implicitly destroys data previously registered w/ same key */
	g_object_set_data_full(obj, key, (gpointer)data, data_destroy);
   else {

	SLang_verror (SL_INTRINSIC_ERROR,
		"Unable to validate arguments to: g_object_set_data");
	if (data) SLang_free_anytype(data);
   }

   SLang_free_opaque(obj_o);
   if (key) SLang_free_slstring(key);
}

static void slg_object_get_data(Slirp_Opaque *ot, char *key)
{
   (void)SLang_push_anytype(
      	(SLang_Any_Type*) g_object_get_data(G_OBJECT(ot->instance),key));
}

static void slg_param_spec_set_qdata(void)
{
   GParamSpec *pspec;
   Slirp_Opaque *pspec_o = NULL;
   SLang_Any_Type *data = NULL;
   GQuark quark;

   if (SLang_Num_Function_Args != 3) {
	SLang_verror (SL_USAGE_ERROR,
		"Usage: g_param_spec_set_qdata(GParamSpec, uint, Any_Type);");
	return;
   }

   if (	SLang_pop_anytype(&data) == 0	&&
	SLang_pop_uinteger(&quark) == 0 &&
	SLang_pop_opaque( GtkOpaque_Type, (void**)&pspec, &pspec_o) == 0)

	/* Implicitly destroys data previously registered / same quark */
	g_param_spec_set_qdata_full( pspec, quark,(gpointer)data,data_destroy);
   else {
  	 
	SLang_verror (SL_INTRINSIC_ERROR,
		"Unable to validate arguments to: g_param_spec_get_qdata");

	if (data) SLang_free_anytype(data);
   }

   SLang_free_opaque(pspec_o);
}

static void slg_param_spec_get_qdata(void)
{
   GQuark quark;
   GParamSpec *pspec;
   Slirp_Opaque *pspec_o = NULL;

   if (SLang_Num_Function_Args != 2) {
	SLang_verror (SL_USAGE_ERROR,
		"Usage: Any_Type = g_param_spec_get_qdata(GParamSpec,uint);");
	return;
   }

   if (	SLang_pop_uinteger(&quark) == 0 &&
	SLang_pop_opaque( GtkOpaque_Type, (void**)&pspec, &pspec_o) == 0)

	(void) SLang_push_anytype( (SLang_Any_Type*)
				g_param_spec_get_qdata( pspec, quark));
   else 

	SLang_verror (SL_INTRINSIC_ERROR,
		"Unable to validate arguments to: g_param_spec_get_qdata");

   SLang_free_opaque(pspec_o);
}

static void slg_param_spec_steal_qdata(void)
{
   GQuark quark;
   GParamSpec *pspec;
   Slirp_Opaque *pspec_o = NULL;

   if (SLang_Num_Function_Args != 2) {
	SLang_verror (SL_USAGE_ERROR,
		"Usage: Any_Type = g_param_spec_steal_qdata(GParamSpec,uint);");
	return;
   }

   if (	SLang_pop_uinteger(&quark) == 0 &&
	SLang_pop_opaque( GtkOpaque_Type, (void**)&pspec, &pspec_o) == 0)

	(void) SLang_push_anytype( (SLang_Any_Type*)
		g_param_spec_get_qdata(pspec, quark));
   else 

	SLang_verror (SL_INTRINSIC_ERROR,
		"Unable to validate arguments to: g_param_spec_steal_qdata");

   SLang_free_opaque(pspec_o);
}

static void slg_slist_next(void)
{
   GSList *rtn, *slist;
   Slirp_Opaque *slist_o = NULL;

   if (SLang_Num_Function_Args != 1) {
	SLang_verror (SL_USAGE_ERROR,
	   "Usage: GtkOpaque = g_slist_next(GSList);");
	return;
   }
   if ( SLang_pop_opaque(GtkOpaque_Type, (void**)&slist, &slist_o) == -1 ) {
	SLang_verror (SL_INTRINSIC_ERROR,
	   "Unable to validate arguments to: g_slist_next");
	return;
   }
   rtn = g_slist_next(slist);
   SLang_free_opaque(slist_o);
   (void) SLang_push_opaque(GtkOpaque_Type, rtn, 0);
}

static SLang_Intrin_Fun_Type Glib_GObject_Funcs[] =
{
  MAKE_INTRINSIC_0("g_signal_connect",slg_signal_connect,V),
  MAKE_INTRINSIC_0("g_signal_connect_after",slg_signal_connect_after,V),
  MAKE_INTRINSIC_0("g_signal_connect_swapped",slg_signal_connect_swapped,V),
  MAKE_INTRINSIC_2("g_signal_emit_by_name",slg_signal_emit_by_name,V,O,S),
  MAKE_INTRINSIC_0("g_object_set_data",slg_object_set_data,V),
  MAKE_INTRINSIC_2("g_object_get_data",slg_object_get_data,V,O,S),
  MAKE_INTRINSIC_0("g_param_spec_set_qdata", slg_param_spec_set_qdata,V),
  MAKE_INTRINSIC_0("g_param_spec_get_qdata", slg_param_spec_get_qdata,V),
  MAKE_INTRINSIC_0("g_param_spec_steal_qdata", slg_param_spec_steal_qdata,V),
  SLANG_END_INTRIN_FUN_TABLE
};

static SLang_Intrin_Fun_Type Glib_Opaque_Funcs[] =
{
  MAKE_INTRINSIC_2("g_enum_get_value",slg_enum_get_value,V,O,I),
  MAKE_INTRINSIC_0("g_slist_next",slg_slist_next,V),
  SLANG_END_INTRIN_FUN_TABLE
};
/* }}} */

/* Intrinsic variables {{{ */
static unsigned long uldummy;

static SLang_Intrin_Var_Type GType_Intrin_Vars [] =
{
  MAKE_VARIABLE("G_TYPE_PARAM_CHAR",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_UCHAR",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_BOOLEAN",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_INT",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_UINT",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_LONG",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_ULONG",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_INT64",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_UINT64",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_UNICHAR",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_ENUM",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_FLAGS",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_FLOAT",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_DOUBLE",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_STRING",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_PARAM",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_BOXED",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_POINTER",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_VALUE_ARRAY",&uldummy,UL,1),
  MAKE_VARIABLE("G_TYPE_PARAM_OBJECT",&uldummy,UL,1),

  SLANG_END_INTRIN_VAR_TABLE
};
/* }}} */

static int init_sl_glib(SLang_NameSpace_Type *ns)
{
   int i = -1;
   patch_ftable(Glib_GObject_Funcs, O, GObject_Type);
   patch_ftable(Glib_Opaque_Funcs, O, GtkOpaque_Type);

   while (GType_Intrin_Vars[++i].name != NULL)
	GType_Intrin_Vars[i].addr = &g_param_spec_types[i];

   if (-1 == SLns_add_intrin_var_table (ns, GType_Intrin_Vars, NULL))
	return -1;

   if (-1 == SLns_add_intrin_fun_table (ns, Glib_GObject_Funcs, "__GLIB__"))
	return -1;

   /* This initializer is needed because g_value_init() actually */
   /* requires that you initialize the GValue prior to callng it */
   memset( &_GValue_Initializer, 0, sizeof(GValue));

   return SLns_add_intrin_fun_table (ns, Glib_Opaque_Funcs, NULL);
}
