/* -*- mode: C; mode: fold; -*- */

/* This file is part of
 *
 * 	SLgtk:  S-Lang bindings for GTK+ widget set
 *
 * Copyright (C) 2003-2004 Massachusetts Institute of Technology 
 * Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
 *
 */

#include <gdk/gdk.h>

/* S-Lang structure handling {{{ */
SLang_CStruct_Field_Type GdkColor_Layout [] =
{
   MAKE_CSTRUCT_FIELD(GdkColor, pixel, "pixel", SLANG_LONG_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkColor, red, "red", SLANG_USHORT_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkColor, green, "green", SLANG_USHORT_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkColor, blue, "blue", SLANG_USHORT_TYPE, 0),
   SLANG_END_CSTRUCT_TABLE
};

SLang_CStruct_Field_Type GdkRectangle_Layout [] =
{
   MAKE_CSTRUCT_FIELD(GdkRectangle, x, "x", SLANG_INT_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkRectangle, y, "y", SLANG_INT_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkRectangle, width, "width", SLANG_INT_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkRectangle, height, "height", SLANG_INT_TYPE, 0),
   SLANG_END_CSTRUCT_TABLE
};

SLang_CStruct_Field_Type GdkPoint_Layout [] =
{
   MAKE_CSTRUCT_FIELD(GdkPoint, x, "x", SLANG_INT_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkPoint, y, "y", SLANG_INT_TYPE, 0),
   SLANG_END_CSTRUCT_TABLE
};

SLang_CStruct_Field_Type GdkCursor_Layout [] =
{
   MAKE_CSTRUCT_FIELD(GdkCursor, type, "x", SLANG_INT_TYPE, 0),
   MAKE_CSTRUCT_FIELD(GdkCursor, ref_count, "ref_count", SLANG_UINT_TYPE, 0),
   SLANG_END_CSTRUCT_TABLE
};

#if SLANG_VERSION < 20000
extern int SLang_pop_struct(SLang_Struct_Type **);
extern void SLang_free_struct(SLang_Struct_Type *);
#endif
/* }}} */

/* Event handling {{{ */
static int push_event(GdkEvent *event)
{
   #define MAXFIELDS 8
   int status, nfields = 3;
 
   char 	 **names = g_new(char*, MAXFIELDS);
   SLtype	 *types  = g_new(SLtype, MAXFIELDS);
   VOID_STAR     *values = g_new(void*, MAXFIELDS);
   SLang_MMT_Type *window;

   /* Temporary variables for field values */
   int x, y, width, height;
   guint32 time;
   guint button, keyval;
   GdkEventMotion *mevent;
   GdkEventExpose *expevent;
   GdkEventConfigure *cevent;
   SLang_Struct_Type *area = NULL;
   
   if ((window = create_opaque_mmt(GdkDrawable_Type,event->any.window,0))
								     == NULL)
	return -1;

   names[0]  = (char*)"type";
   names[1]  = (char*)"window";
   names[2]  = (char*)"send_event";

   types[0] = SLANG_INT_TYPE;
   types[1] = GdkDrawable_Type;
   types[2] = SLANG_CHAR_TYPE;

   values[0] = (VOID_STAR)&event->type;
   values[1] = (VOID_STAR)&window;
   values[2] = (VOID_STAR)&(event->any.send_event);

   switch(event->type) {

	case GDK_BUTTON_PRESS:
	case GDK_2BUTTON_PRESS:
	case GDK_3BUTTON_PRESS:
	case GDK_BUTTON_RELEASE:

	   button = event->button.button;
	   names[nfields]  = "button";
	   types[nfields]  = SLANG_UINT_TYPE;
	   values[nfields++] = (VOID_STAR)&button;

	   time = event->button.button;
	   names[nfields]  = "time";

#if SIZEOF_INT == 4
	   types[nfields]  = SLANG_UINT_TYPE;
#elif SIZEOF_SHORT == 4
	   types[nfields]  = SLANG_USHORT_TYPE;
#elif SIZEOF_LONG == 4
	   types[nfields]  = SLANG_ULONG_TYPE;
#else
	ERROR: cannot map guint32 to a 4-byte unsigned S-Lang type
#endif

	   values[nfields++] = (VOID_STAR)&time;

	   /* intentional fall through */

	case GDK_MOTION_NOTIFY:

	   mevent = (GdkEventMotion*)event;
	   x = (int) mevent->x;		 /* cast coords to integral values */
	   y = (int) mevent->y;
	   names[nfields] = "x"; names[nfields+1] = "y";
	   types[nfields] = types[nfields+1] = SLANG_INT_TYPE;
	   values[nfields] = (VOID_STAR) &x;
	   values[nfields+1] = (VOID_STAR)&y;
	   nfields += 2;
	   break;

	case GDK_CONFIGURE:

	   cevent = (GdkEventConfigure*)event;
	   x = cevent->x; y = cevent->y;
	   width = cevent->width; height = cevent->width;
	   names[nfields] = "x"; names[nfields+1] = "y";
	   names[nfields+2] = "width"; names[nfields+3] = "height";
	   types[nfields] = types[nfields+1] = 
			types[nfields+2] = types[nfields+3] = SLANG_INT_TYPE;
	   values[nfields] = (VOID_STAR) &x;
	   values[nfields+1] = (VOID_STAR)&y;
	   values[nfields+2] = (VOID_STAR) &width;
	   values[nfields+3] = (VOID_STAR)&height;
	   nfields += 4;

	   break;

	case GDK_KEY_PRESS:

	   keyval = event->key.keyval;
	   names[nfields] = "keyval";
	   types[nfields] = SLANG_UINT_TYPE;
	   values[nfields] = (VOID_STAR) &keyval;
	   nfields++;
	   break;

	case GDK_EXPOSE:

	   expevent = (GdkEventExpose*)event;
	   if (-1 == SLang_push_cstruct((VOID_STAR)&(expevent->area),
							GdkRectangle_Layout))
		return -1;

	   if (-1 == SLang_pop_struct(&area))
		return -1;

	   names[nfields] = "area";
	   types[nfields] = SLANG_STRUCT_TYPE;
	   values[nfields] = (VOID_STAR) &area;
	   nfields++;

	   break;

	default:
	   break;
   }

    status = SLstruct_create_struct(nfields,names,types,values);
    g_free(names); g_free(types); g_free(values);

    if (area != NULL)
	SLang_free_struct (area);		/* decrease ref count */

    return status;
}

int push_boxed(const GValue* gval)
{
   gpointer boxed = g_value_get_boxed(gval);
   GType gtype = G_VALUE_TYPE(gval);

   if (gtype == GDK_TYPE_EVENT)
	return push_event((GdkEvent*)boxed);

   if (gtype == GDK_TYPE_RECTANGLE)
	return SLang_push_cstruct((VOID_STAR)boxed,GdkRectangle_Layout);

   /* !!! FIXME: pushing boxed types as opaque "works," but is insufficient,
 	  as it prohibits the caller from inspecting, in  S-Lang scope, 
	  values of the fields within the underlying structure */

   return SLang_push_opaque(GtkOpaque_Type, boxed, 0);
}
/* }}} */

/* Wrappers {{{ */
static void slgdk_pixmap_create_from_xpm(void)
{
   GdkPixmap *pixmap;
   GdkBitmap *bitmap;
   GdkColor  color;
   void *gcp = &color;
   char *fname = NULL;
   GdkWindow *win;
   Slirp_Opaque *win_o = NULL;
   SLang_Ref_Type *bitmap_ref = NULL;

   if ( -1 == SLang_pop_slstring (&fname) 			||
	-1 == pop_nullable(0, &gcp, (void*)GdkColor_Layout)	||
	-1 == pop_nullable(SLANG_REF_TYPE,(void**)&bitmap_ref,(void**)&bitmap)||
   	-1 == SLang_pop_opaque(GdkDrawable_Type, (void**)&win, &win_o))
   {
	SLang_push_null();
	goto cleanup;
   }

   if (bitmap_ref == NULL)
      pixmap = gdk_pixmap_create_from_xpm(win, NULL, gcp, fname);
   else
      pixmap = gdk_pixmap_create_from_xpm(win, &bitmap, gcp, fname);


   if (pixmap != NULL) {

	if (bitmap_ref != NULL) {

	   SLang_MMT_Type *mmt = create_opaque_mmt(GdkDrawable_Type,bitmap,0);
	   /* No need to check mmt != NULL, since it's ok to assign */
	   (void)SLang_assign_to_ref(bitmap_ref, GdkDrawable_Type, &mmt);
	   SLang_free_ref(bitmap_ref);
	}
	SLang_push_opaque(GdkDrawable_Type, pixmap, 0);
   }
   else
	SLang_push_null();

   cleanup:
	if (fname) SLang_free_slstring(fname);
	if (gcp) SLang_free_cstruct(gcp, GdkColor_Layout);
        SLang_free_opaque(win_o);
}

static void
slgdk_color_parse(gchar *colorname)
{
   GdkColor color;

   if ( gdk_color_parse(colorname,&color) &&
	gdk_colormap_alloc_color(gdk_colormap_get_system(),&color,FALSE,TRUE)) {

	if ( SLang_push_cstruct((VOID_STAR)&color,GdkColor_Layout) == 0)
	   return;
   }
   (void) SLang_push_null();
   return;
}

static void slgdk_rectangle_new(int *x,int *y,int *width, int *height)
{
   GdkRectangle r;
   r.x = *x;
   r.y = *y;
   r.width = *width;
   r.height = *height;
   if (SLang_push_cstruct((VOID_STAR)&r,GdkRectangle_Layout) != 0)
	SLang_push_null();
}

static void slgdk_point_new(int *x,int *y)
{
   GdkPoint p;
   p.x = *x;
   p.y = *y;
   if (SLang_push_cstruct((VOID_STAR)&p,GdkPoint_Layout) != 0)
      SLang_push_null();
}

static void slgdk_region_arrays(void)
{
   int fillrule, *x, *y;
   unsigned int i;
   GdkRegion *reg = NULL;
   GdkPoint *points = NULL;
   SLang_Array_Type *slx = NULL;
   SLang_Array_Type *sly = NULL;

   if (usage_err(3,"reg = gdk_region_arrays(xpoints,ypoints,fillrule)") ||
	-1 == SLang_pop_integer(&fillrule)		   ||
	-1 == SLang_pop_array_of_type(&sly,SLANG_INT_TYPE) ||
	-1 == SLang_pop_array_of_type(&slx,SLANG_INT_TYPE) ||
	slx->num_elements != sly->num_elements		   ||
	slx->num_elements < 3)
   {
	SLang_verror(SL_INTRINSIC_ERROR,
			"error popping or validating region, check input");
	goto cleanup;
   }

   if ((points = g_new(GdkPoint,slx->num_elements)) == NULL) {
	SLang_verror(SL_INTRINSIC_ERROR,"out of memory");
	goto cleanup;
   }

   x = (int*)slx->data; y = (int*)sly->data; 
   for (i=0; i < slx->num_elements; i++) {
        points[i].x = x[i];
        points[i].y = y[i];
   }

   if ( (reg = gdk_region_polygon(points,(gint)i,fillrule)) == NULL)
	SLang_verror(SL_INTRINSIC_ERROR,"could not allocate GdkRegion");

   cleanup:

   g_free(points);
   SLang_free_array(sly);
   SLang_free_array(slx);

   (void) SLang_push_opaque(GdkRegion_Type, reg, 0);
}

static void slgdk_query_depths (void)
{
   gint* depths;
   gint num;
   SLang_Array_Type *sarr;

   if (usage_err(0,"Integer_Type[] = gdk_query_depths()"))
	return;

   gdk_query_depths(&depths, &num);

   if ( (sarr = SLang_create_array (SLANG_INT_TYPE, 1, NULL, &num, 1)) == NULL)
	SLang_verror(SLEI, "creating array in %s", "gdk_query_depths");
   else {
	while (num--)
	   ((int*)sarr->data)[num] = depths[num];
	SLang_push_array(sarr, 1);
   }
}
/* }}} */

static SLang_Intrin_Fun_Type Gdk_Funcs[] =
{
  MAKE_INTRINSIC_0("gdk_pixmap_create_from_xpm",slgdk_pixmap_create_from_xpm,V),
  MAKE_INTRINSIC_1("gdk_color_parse",slgdk_color_parse,V,S),
  MAKE_INTRINSIC_4("gdk_rectangle_new",slgdk_rectangle_new,V,I,I,I,I),
  MAKE_INTRINSIC_2("gdk_point_new",slgdk_point_new,V,I,I),
  MAKE_INTRINSIC_0("gdk_region_arrays",slgdk_region_arrays,V),
  MAKE_INTRINSIC_0("gdk_query_depths",slgdk_query_depths,V),

  SLANG_END_INTRIN_FUN_TABLE
};

static int init_sl_gdk(SLang_NameSpace_Type *ns)
{
   patch_ftable(Gdk_Funcs, O, GtkOpaque_Type);
   return SLns_add_intrin_fun_table (ns, Gdk_Funcs, "__GDK__");
}
