/*
 * The Sleuth Kit
 *
 * $Date: 2005/09/02 23:34:04 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2005 Brian Carrier.  All rights reserved
 *
 * mmstat - Get stats on the volume system / media management
 *
 *
 * This software is distributed under the Common Public License 1.0
 *
 */

#include "mm_tools.h"


extern char *progname;


void
usage()
{
    fprintf(stderr,
	    "%s [-i imgtype] [-o imgoffset] [-vV] [-t mmtype] image [images]\n",
	    progname);
    fprintf(stderr, "\t-t mmtype: The type of partition system\n");
    fprintf(stderr, "\t-i imgtype: The format of the image file\n");
    fprintf(stderr,
	    "\t-o imgoffset: Offset to the start of the volume that contains the partition system (in sectors)\n");
    fprintf(stderr, "\t-v: verbose output\n");
    fprintf(stderr, "\t-V: print the version\n");
    fprintf(stderr, "Supported partition systems:\n");
    mm_print_types(stderr);
    fprintf(stderr, "Supported image formats:\n");
    img_print_types(stderr);
    exit(1);
}

static void
print_stats(MM_INFO * mm)
{
    printf("%s\n", mm_get_type(mm->mmtype));
    //printf("Type: %s\n", mm->str_type);
    return;
}

int
main(int argc, char **argv)
{
    MM_INFO *mm;
    char *mmtype = NULL;
    int ch;
    DADDR_T sect_offset = 0;
    char *imgtype = NULL;
    IMG_INFO *img;

    progname = argv[0];

    while ((ch = getopt(argc, argv, "i:o:t:vV")) > 0) {
	char *cp;
	switch (ch) {
	case 'i':
	    imgtype = optarg;
	    break;

	    // @@@ WE may want this to support the @ syntax...
	case 'o':
	    sect_offset = strtoull(optarg, &cp, 0);
	    if (*cp || cp == optarg) {
		fprintf(stderr, "Bad sector offset\n");
		usage();
	    }

	    break;
	case 't':
	    mmtype = optarg;
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    print_version(stdout);
	    exit(0);
	case '?':
	default:
	    fprintf(stderr, "Unknown argument\n");
	    usage();
	}
    }

    /* We need at least one more argument */
    if (optind >= argc) {
	fprintf(stderr, "Missing image name\n");
	usage();
    }

    /* open the image - do not use an offset */
    img =
	img_open(imgtype, "0", argc - optind,
		 (const char **) &argv[optind]);

    /* process the partition tables */
    mm = mm_open(img, mmtype, sect_offset);
    if (mm == NULL) {
	if (mmtype == NULL)
	    printf("Error determining partition system type\n");
	else
	    fprintf(stderr, "Error processing image\n");
	exit(1);
    }

    print_stats(mm);

    mm->close(mm);
    img->close(img);
    exit(0);
}
