/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a generic Skrooge plugin for html reports.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skghtmlboardwidget.h"
#include "skgdocument.h"
#include "skgtraces.h"
#include "skgreport.h"
#include "skgmainpanel.h"

SKGHtmlBoardWidget::SKGHtmlBoardWidget(SKGDocument* iDocument, const QString& iTitle, const QString& iTemplate, const QStringList& iTablesRefreshing, bool iEnablePreviousMenu)
    : SKGWidget(iDocument), m_HtmlTemplate(iTemplate), m_TablesRefreshing(iTablesRefreshing), m_refreshNeeded(false), m_previousMonth(NULL), m_title(iTitle)
{
    SKGTRACEIN(10, "SKGHtmlBoardWidget::SKGHtmlBoardWidget");

    QVBoxLayout* verticalLayout_2 = new QVBoxLayout(this);
    verticalLayout_2->setContentsMargins(0, 0, 0, 0);
    verticalLayout_2->setObjectName(QString::fromUtf8("verticalLayout_2"));
    m_group = new QGroupBox(this);
    m_group->setObjectName(QString::fromUtf8("m_group"));
    QVBoxLayout* verticalLayout = new QVBoxLayout(m_group);
    verticalLayout->setContentsMargins(0, 0, 0, 0);
    verticalLayout->setObjectName(QString::fromUtf8("verticalLayout"));
    m_Text = new QLabel(m_group);
    m_Text->setObjectName(QString::fromUtf8("m_Text"));
    m_Text->setTextFormat(Qt::RichText);
    m_Text->setAlignment(Qt::AlignLeading | Qt::AlignLeft | Qt::AlignTop);
    m_Text->setTextInteractionFlags(Qt::TextBrowserInteraction);

    verticalLayout->addWidget(m_Text);
    verticalLayout_2->addWidget(m_group);

    //Create menu
    if (iEnablePreviousMenu) {
        setContextMenuPolicy(Qt::ActionsContextMenu);
        m_previousMonth = new KAction(i18nc("Report for the previous month", "Previous month"), this);
        m_previousMonth->setCheckable(true);
        m_previousMonth->setChecked(true);
        connect(m_previousMonth, SIGNAL(triggered(bool)), this, SLOT(dataModified()));
        addAction(m_previousMonth);
    }

    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString,int)), this, SLOT(dataModified(QString,int)), Qt::QueuedConnection);
    connect(SKGMainPanel::getMainPanel(), SIGNAL(currentPageChanged()), this, SLOT(pageChanged()), Qt::QueuedConnection);
    connect(m_Text, SIGNAL(linkActivated(QString)), SKGMainPanel::getMainPanel(), SLOT(openPage(QString)));
}

SKGHtmlBoardWidget::~SKGHtmlBoardWidget()
{
    SKGTRACEIN(10, "SKGHtmlBoardWidget::~SKGHtmlBoardWidget");
    m_previousMonth = NULL;
}

QString SKGHtmlBoardWidget::getState()
{
    SKGTRACEIN(10, "SKGHtmlBoardWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    //Get state
    root.setAttribute("previousMonth", m_previousMonth && m_previousMonth->isChecked() ? "Y" : "N");

    return doc.toString();
}

void SKGHtmlBoardWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGHtmlBoardWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    //Set state
    if (m_previousMonth) m_previousMonth->setChecked(root.attribute("previousMonth") != "N");

    dataModified("", 0);
}

void SKGHtmlBoardWidget::pageChanged()
{
    if (m_refreshNeeded) dataModified();
}

void SKGHtmlBoardWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN(10, "SKGHtmlBoardWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    if (m_TablesRefreshing.isEmpty() || m_TablesRefreshing.contains(iTableName) || iTableName.isEmpty()) {
        //Is this the current page
        SKGTabPage* page = SKGTabPage::parentTabPage(this);
        if (page != NULL && page != SKGMainPanel::getMainPanel()->currentPage()) {
            //No, we memorize that we have to cumpute later the report
            m_refreshNeeded = true;
            return;
        }
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

        //Computation of the report
        SKGReport* report = getDocument()->getReport();
        if (report) {
            //Compute parameters
            QDate month = QDate::currentDate();
            month = month.addDays(-month.day() + 1);
            if (m_previousMonth && m_previousMonth->isChecked()) month = month.addMonths(-1);
            QString monthS = month.toString("yyyy-MM");

            //Set title
            m_group->setTitle(m_title % (m_previousMonth ? " - " + monthS : ""));

            //Set month
            report->setMonth(monthS);

            QString html;
            SKGError err = SKGReport::getReportFormTemplate(report, m_HtmlTemplate, html);
            if (err) html = err.getFullMessage();
            m_Text->setText(html);
        }
        m_refreshNeeded = false;

        QApplication::restoreOverrideCursor();
    }

    //TODO: No widget if no account (must not be hardcoded)
    bool exist = false;
    getDocument()->existObjects("account", "", exist);
    if (parentWidget()) setVisible(exist);
}
#include "skghtmlboardwidget.h"
